/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the XM AppID packet utility functions for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "traffic_interface.h"
#include "xmappid_packet_util.h"
#include "_xmappid_packet_util.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/


/*****************************************************************************
*
*   XMAPPID_PACKET_UTIL_bParsePayloadFromPacket
*
*   This API is used to parse the payload from an XM AppID packet.  Currently,
*   this function verifies the sync byte and will parse out the AppID value.
*   It doesn't verify either CRC or the length value since we are assuming
*   that all XM AppID packets we get from the SXI framework will be complete,
*   valid packets.
*
*   When the function returns successfully, hPacket should just contain the
*   payload from the XM AppID packet
*
*   Inputs:
*       hPacket - A handle to a valid OSAL_BUFFER_HDL that contains
*           the AppID packet
*       ptAppID - A pointer to a XM_APPID value that can be used to return
*           the AppID value found in the header.  This can be NULL if the
*           caller doesn't care
*       ptSeqID - A pointer to a XM_SEQID value that can be used to return
*           the SeqID value found in the header.  This can be NULL if the
*           caller doesn't care
*
*   Outputs:
*       Returns TRUE if the payload was successfully found and the header/CRC
*       were parsed out.  returns FALSE if there was some sort of error with
*       the parsing.  If the function returns FALSE, the value pointed to
*       by ptAppID should be considered invalid.
*
*****************************************************************************/
BOOLEAN XMAPPID_PACKET_UTIL_bParsePayloadFromPacket(
    OSAL_BUFFER_HDL hPacket,
    XM_APPID *ptAppID,
    XM_SEQID *ptSeqID
        )
{
    UN8 un8Sync = 0,
        un8AppID = 0,
        un8SeqID=0;
    size_t tNum;

    do
    {
        // Read the sync byte
        tNum = OSAL.tBufferReadHead(hPacket,
            &un8Sync, XM_APPID_SYNC_BYTELEN);

        if (tNum != XM_APPID_SYNC_BYTELEN)
        {
            break;
        }

        // Verify the sync byte
        if (un8Sync != 0xF0)
        {
            // Bad packet
            break;
        }

        // Read the AppID
        tNum = OSAL.tBufferReadHead(hPacket,
                &un8AppID, XM_APPID_ID_BYTELEN);

        if (tNum != XM_APPID_ID_BYTELEN)
        {
            break;
        }

        // seek past length
        tNum = OSAL.tBufferSeekHead(hPacket,XM_APPID_LEN_BYTELEN);

        if (tNum != XM_APPID_LEN_BYTELEN)
        {
            break;
        }

        // Read the SeqID
        tNum = OSAL.tBufferReadHead(hPacket,
                &un8SeqID, XM_APPID_ID_BYTELEN);

        if (tNum != XM_APPID_ID_BYTELEN)
        {
            break;
        }

        // Seek past the rest of the header
        tNum = OSAL.tBufferSeekHead(hPacket,XM_APPID_HDR_CRC_BYTELEN);

        if (tNum != XM_APPID_HDR_CRC_BYTELEN)
        {
            break;
        }

        // Remove AppID CRC from the tail
        // We are guarrenteed by the framework that the CRC is good
        tNum = OSAL.tBufferSeekTail(hPacket, XM_APPID_CRC_BYTELEN);

        if (tNum != XM_APPID_CRC_BYTELEN)
        {
            break;
        }

        // We are all good, so return our AppId and SeqID
        //if needed and indicate success
        if (ptAppID != NULL)
        {
            *ptAppID = un8AppID;
        }

        if (ptSeqID != NULL)
        {
            *ptSeqID = un8SeqID;
        }

        return TRUE;

    } while (FALSE);


    // We errored out at some point, so indicate that we have an issue
    return FALSE;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
