/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Tabular weather protocol version 1 implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "_weather_pvn1.h"
#include "weather_interface.h"
#include "string_obj.h"
#include "string.h"
#include "location_obj.h"
#include "sms_obj.h"
#include "baudot.h"
#include "dataservice_mgr_obj.h"
#include "db_util.h"
#include "forecast_obj.h"
#include "ski_conditions_obj.h"
#include "weather_msg_obj.h"
#include "ds_util.h"
#include "sms.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tMinimumOTABufferByteSize
*
*   Calculate the minimum number of bytes needed for this service
*   based upon how it's going to be used
*
*****************************************************************************/
static size_t tMinimumOTABufferByteSize (
    BOOLEAN bDBUpdatesEnabled
        )
{
    // We'll always need this as a starting point
    size_t tByteSize = WEATHER1_PAYLOAD_MAX_SIZE * 
        WEATHER1_NUM_OF_QUEUED_PAYLOADS;

    if (TRUE == bDBUpdatesEnabled)
    {
        tByteSize += WEATHER1_RFD_META_PAYLOAD_MAX_SIZE
            + WEATHER1_RFD_BLOCK_PAYLOAD_MAX_SIZE;
    }

    return tByteSize;
}

/*****************************************************************************
*
*   hInit
*
*****************************************************************************/
static WEATHER_INTERFACE_OBJECT hInit(
    WEATHER_SERVICE_OBJECT hWeatherService,
    SMS_OBJECT hParent,
    BOOLEAN bDBUpdatesEnabled,
    BOOLEAN bClearPersistent,
    UN8 un8CurDBVersion
        )
{
    BOOLEAN bOwner = FALSE, bSuccess;
    WEATHER1_OBJECT_STRUCT *psObj = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        if (hWeatherService == WEATHER_SERVICE_INVALID_OBJECT)
        {
            break;
        }

        // Verify we own service handle
        bOwner = SMSO_bOwner(hParent);
        if (bOwner == FALSE)
        {
            break;
        }

        // Create an instance of this object
        psObj = (WEATHER1_OBJECT_STRUCT *)
            SMSO_hCreate(
                WEATHER1_OBJECT_NAME,
                sizeof(*psObj),
                hParent, FALSE);

        if (psObj == NULL)
        {
            break;
        }

        // Save the service handle
        psObj->hWeatherService = hWeatherService;

        //creating block pool for cached data
        eReturnCode = OSAL.eBlockPoolCreate(&psObj->hBlockPool,
            WEATHER1_OBJECT_NAME":BlockPool",
            WEATHER1_BLOCK_POOL_BUF_SIZE,
            WEATHER1_BLOCK_POOL_BUF_NUM,
            OSAL_BLOCK_POOL_OPTION_NONE
                );
        if ((eReturnCode != OSAL_SUCCESS) ||
            (psObj->hBlockPool == OSAL_INVALID_OBJECT_HDL))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to create block pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Request access to ISO 3309 CRC32
        eReturnCode = OSAL.eGetCRC(
            &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32);

        if (eReturnCode != OSAL_SUCCESS)
        {
            printf(WEATHER1_OBJECT_NAME": Cannot create CRC object.\n");
            break;
        }

        // Ensure this is cleared
        psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;

        if (bDBUpdatesEnabled == TRUE)
        {
            bSuccess = bInitRFD(psObj, (RFD_UPDATE_VERSION)un8CurDBVersion);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to init RFD");
                break;
            }
        }

        bSuccess = bBuildPersistentFolderPath(psObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to build persistent folder path");
            break;
        }

        bSuccess = bBuildCacheFilePath(psObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to build cache file path");
            break;
        }

        if (bClearPersistent == TRUE)
        {
            // Empty our cache and move on
            vDeleteCache(psObj);
        }
        else // read the cache
        {
            bSuccess = bReadCacheFile(psObj);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to read cache file"
                        );
                // still continue since stream data can arrive
            }
        }

        return (WEATHER_INTERFACE_OBJECT)psObj;
    } while (FALSE);

    vUninitObject(psObj);

    return WEATHER_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vUnInit
*
*****************************************************************************/
static void vUnInit(
    WEATHER_INTERFACE_OBJECT hInterface
        )
{
    do
    {
        BOOLEAN bOwner, bSuccess;
        WEATHER1_OBJECT_STRUCT *psObj = 
            (WEATHER1_OBJECT_STRUCT*)hInterface;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // check if cache file synchronization is required
        if (psObj->bCacheSyncNeeded == TRUE)
        {
            bSuccess = bWriteCacheFile(psObj);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to write cache file");
            }
        }

        vUninitObject(psObj);
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bEnableSkiReports
*
*****************************************************************************/
static BOOLEAN bEnableSkiReports(
    WEATHER_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    WEATHER1_OBJECT_STRUCT *psObj = (WEATHER1_OBJECT_STRUCT*)hInterface;

    do {
        if (hInterface == WEATHER_INTERFACE_INVALID_OBJECT)
        {
            break;
        }
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        psObj->bSkiEnabled = TRUE;
        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bDisableSkiReports
*
*****************************************************************************/
static BOOLEAN bDisableSkiReports(
    WEATHER_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    WEATHER1_OBJECT_STRUCT *psObj = (WEATHER1_OBJECT_STRUCT*)hInterface;

    do {
        if (hInterface == WEATHER_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        psObj->bSkiEnabled = FALSE;
        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessMessage
*
*****************************************************************************/
static BOOLEAN bProcessMessage(
    WEATHER_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    WEATHER1_OBJECT_STRUCT *psObj = (WEATHER1_OBJECT_STRUCT*)hInterface;
    WEATHER1_MSG_TYPE_ENUM eMsgType = WEATHER_INVALID_MSG;

    printf(WEATHER1_OBJECT_NAME": [%s] start with parameters "
        "hInterface = %p, phPayload = %p\n",
        __FUNCTION__, hInterface, phPayload
            );

    do
    {
        if (phPayload == NULL)
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // getting message type
        bResult = bGetMessageType(psObj, *phPayload, &eMsgType);

        if (bResult == FALSE)
        {
            break;
        }

        printf(WEATHER1_OBJECT_NAME": Message type is %u\n", eMsgType);

        switch (eMsgType)
        {
            case WEATHER_FORECAST_REPORT_MSG:
            {
                bResult = bProcessWeatherForecast(psObj, *phPayload);
            }
            break;

            case WEATHER_SKI_REPORT_MSG:
            {
                if (psObj->bSkiEnabled == TRUE)
                {
                    bResult = bProcessSkiReport(psObj, *phPayload);
                }
                else
                {
                    bResult = TRUE;
                }
            }
            break;

            case WEATHER_LOCATION_DB_UPDATE_MSG:
            case WEATHER_METADATA_UPDATE_MSG:
            {
                if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
                {
                    // let message be processed if RFD has been initialized
                    bResult = RFD_INTERFACE_bProcessPayload(psObj->hRFD, *phPayload);
                    *phPayload = OSAL_INVALID_BUFFER_HDL;
                }
                else
                {
                    // skip if RFD has not been initialized
                    bResult = TRUE;
                }
            }
            break;

            default:
            break;
        }

    } while (FALSE);

    printf(WEATHER1_OBJECT_NAME": [%s] end with result %d\n",
        __FUNCTION__, bResult
            );

    return bResult;
}

/*****************************************************************************
*
*   bProcessTimerCall
*
*****************************************************************************/
static BOOLEAN bProcessTimerCall(WEATHER_INTERFACE_OBJECT hInterface)
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    WEATHER1_OBJECT_STRUCT *psObj = (WEATHER1_OBJECT_STRUCT*)hInterface;
    UN32 un32Time = 0;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        eReturnCode = OSAL.eTimeGet(&un32Time);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to get OS time"
                    );
            break;
        }

        vCancelHashItems(psObj, un32Time);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   bIsTimerNeeded
*
*****************************************************************************/
static BOOLEAN bIsTimerNeeded(WEATHER_INTERFACE_OBJECT hInterface)
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    WEATHER1_OBJECT_STRUCT *psObj = (WEATHER1_OBJECT_STRUCT*)hInterface;

    do
    {
        UN8 i = 0;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        for(i = 0; i < WEATHER_FCST_TYPE_MAX; i++)
        {
            if (psObj->asCacheItem[i].un32TimestampUTCsec != 0)
            {
                bResult = TRUE;
                break;
            }
        }

        if (bResult == FALSE)
        {
            // in case if we have no data in array, cache file update
            // is not needed
            psObj->bCacheSyncNeeded = FALSE;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessWindData
*
*****************************************************************************/
static BOOLEAN bProcessWindData (
    WIND_UNPARSED_DATA tData,
    WIND_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 un8Temp = 0;
    UN8 i = 0;

    do {
        if (psData == NULL)
        {
            break;
        }
        psData->un8Speed = ((tData & 0x0F) * 5);

        un8Temp = (tData & 0xF0) >> 4;
        for (i = 0; i < sizeof(gasWindDataStruct)/sizeof(*gasWindDataStruct); i++)
        {
            if (un8Temp == gasWindDataStruct[i].tValue)
            {
                psData->un16Direction = gasWindDataStruct[i].un16Direction;
                bResult = TRUE;
                break;
            }
        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessWeatherEventData
*
*****************************************************************************/
static BOOLEAN bProcessWeatherEventData (
    WEATHER_EVENT_UNPARSED_DATA tData,
    WEATHER_EVENT_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 i = 0;

    do {
        if (psData == NULL)
        {
            break;
        }
        for (i = 0; i < sizeof(gasWeatherEventDataStruct)/sizeof(*gasWeatherEventDataStruct); i++)
        {
            if (tData == gasWeatherEventDataStruct[i].tValue)
            {
                psData->hDescription = STRING_hCreateConst(gasWeatherEventDataStruct[i].pcDescription,
                        strlen(gasWeatherEventDataStruct[i].pcDescription));
                psData->un8EventCode = gasWeatherEventDataStruct[i].un8EventCode;
                psData->un8IconCode = gasWeatherEventDataStruct[i].un8IconCode;
                bResult = TRUE;
                break;
            }

        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessUvInfoData
*
*****************************************************************************/
static BOOLEAN bProcessUvInfoData (
    UV_INFO_UNPARSED_DATA tData,
    UV_INFO_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 i = 0;

    do {
        if (psData == NULL)
        {
            break;
        }
        for (i = 0; i < sizeof(gasUvInfoDataStruct)/sizeof(*gasUvInfoDataStruct); i++)
        {
            if (tData == gasUvInfoDataStruct[i].tValue)
            {
                psData->un8Low = gasUvInfoDataStruct[i].un8Low;
                psData->un8High = gasUvInfoDataStruct[i].un8High;
                psData->eRisk = gasUvInfoDataStruct[i].eRisk;
                bResult = TRUE;
                break;
            }

        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessTrailsData
*
*****************************************************************************/
static BOOLEAN bProcessTrailsData (
    TRAILS_UNPARSED_DATA tData,
    TRAILS_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 i = 0;

    do {
        if (psData == NULL)
        {
            break;
        }
        for (i = 0; i < sizeof(gasTrailsDataStruct)/sizeof(*gasTrailsDataStruct); i++)
        {
            if (tData == gasTrailsDataStruct[i].tValue)
            {
                psData->un8Low = gasTrailsDataStruct[i].un8Low;
                psData->un8High = gasTrailsDataStruct[i].un8High;
                bResult = TRUE;
                break;
            }

        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessNewSnowData
*
*****************************************************************************/
static BOOLEAN bProcessNewSnowData (
    NEW_SNOW_UNPARSED_DATA tData,
    NEW_SNOW_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 i = 0;

    do {
        if (psData == NULL)
        {
            break;
        }
        for (i = 0; i < sizeof(gasNewSnowDataStruct)/sizeof(*gasNewSnowDataStruct); i++)
        {
            if (tData == gasNewSnowDataStruct[i].tValue)
            {
                psData->un8Low = gasNewSnowDataStruct[i].un8Low;
                psData->un8High = gasNewSnowDataStruct[i].un8High;
                bResult = TRUE;
                break;
            }

        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessPrecipitationData
*
*****************************************************************************/
static BOOLEAN bProcessPrecipitationData (
    PRECIP_UNPARSED_DATA tData,
    PRECIP_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 i = 0;
    UN8 un8Temp = tData & 0x1F;

    do {
        if (psData == NULL)
        {
            break;
        }

        psData->eType = (PRECIPITATION_TYPE_ENUM) ((tData & 0x60) >> 5);

        for (i = 0; i < sizeof(gasPrecipAmountDataStruct)/sizeof(*gasPrecipAmountDataStruct); i++)
        {
            if (un8Temp == gasPrecipAmountDataStruct[i].tValue)
            {
                psData->hLow = OSAL_FIXED.hCreate(
                        gasPrecipAmountDataStruct[i].un8LowWhole,
                        gasPrecipAmountDataStruct[i].un8LowFrac,
                        gasPrecipAmountDataStruct[i].un8LowPow);

                psData->hHigh = OSAL_FIXED.hCreate(
                        gasPrecipAmountDataStruct[i].un8HighWhole,
                        gasPrecipAmountDataStruct[i].un8HighFrac,
                        gasPrecipAmountDataStruct[i].un8HighPow);
                bResult = TRUE;
                break;
            }

        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessHumidityData
*
*****************************************************************************/
static BOOLEAN bProcessHumidityData (
    HUMIDITY_UNPARSED_DATA tData,
    HUMIDITY_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 i = 0;

    do {
        if (psData == NULL)
        {
            break;
        }

        for (i = 0; i < sizeof(gasHumidityDataStruct)/sizeof(*gasHumidityDataStruct); i++)
        {
            if (tData == gasHumidityDataStruct[i].tValue)
            {
                psData->un8Low = gasHumidityDataStruct[i].un8Low;
                psData->un8High = gasHumidityDataStruct[i].un8High;
                bResult = TRUE;
                break;
            }

        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessPollenData
*
*****************************************************************************/
static BOOLEAN bProcessPollenData (
    POLLEN_UNPARSED_DATA tData,
    POLLEN_DATA_STRUCT *psData
        )
{
    BOOLEAN bResult = FALSE;
    UN8 i = 0;

    do {
        if (psData == NULL)
        {
            break;
        }

        for (i = 0; i < sizeof(gasPollenDataStruct)/sizeof(*gasPollenDataStruct); i++)
        {
            if (tData == gasPollenDataStruct[i].tValue)
            {
                psData->eLevel = gasPollenDataStruct[i].eLevel;
                psData->un8Value = tData;
                bResult = TRUE;
                break;
            }

        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bLoadForecastData
*
*****************************************************************************/
static BOOLEAN bLoadForecastData (
    WEATHER_INTERFACE_OBJECT hInterface,
    SMS_OBJECT hParent,
    OSAL_OBJECT_HDL hWeatherMsgsList
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bOwner;
        WEATHER1_OBJECT_STRUCT *psObj = (WEATHER1_OBJECT_STRUCT*)hInterface;
        UN32 un32Items = 0;
        UN8 i = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        WEATHER1_PARSE_ITERATOR_ARG sIteratorArg;

        if ((hParent == SMS_INVALID_OBJECT) ||
            (hWeatherMsgsList == OSAL_INVALID_OBJECT_HDL))
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListItems(hWeatherMsgsList, &un32Items);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to get number of list items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (un32Items == 0)
        {
            //nothing to do, exit
            bResult = TRUE;
            break;
        }

        OSAL.bMemSet(&sIteratorArg, 0, sizeof(sIteratorArg));
        sIteratorArg.hObjectsList = hWeatherMsgsList;
        sIteratorArg.hParent = hParent;
        sIteratorArg.psObj = psObj;

        for(i = 0; i < WEATHER_FCST_TYPE_SKI; i++)
        {
            if (psObj->asCacheItem[i].un32TimestampUTCsec != 0)
            {
                sIteratorArg.eFcst = (WEATHER_FCST_TYPE_ENUM)i;
                sIteratorArg.hHashEntry = &psObj->asCacheItem[i];
                eReturnCode = OSAL.eBufferBlocksIterate(
                    psObj->ahPayload[i],
                    (OSAL_BUFFER_ITERATOR_HANDLER)eParseForecasts,
                    &sIteratorArg);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to parse forecast "
                        "data for fcst %u", i);
                    break;
                }
            }
        }

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bLoadSkiData
*
*****************************************************************************/
static BOOLEAN bLoadSkiData (
    WEATHER_INTERFACE_OBJECT hInterface,
    SMS_OBJECT hParent,
    OSAL_OBJECT_HDL hWeatherMsgsList
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bOwner = FALSE;
        WEATHER1_OBJECT_STRUCT *psObj = (WEATHER1_OBJECT_STRUCT*)hInterface;
        UN32 un32Items = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WEATHER1_PARSE_ITERATOR_ARG sIteratorArg;

        if ((hParent == SMS_INVALID_OBJECT) ||
            (hWeatherMsgsList == OSAL_INVALID_OBJECT_HDL))
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListItems(hWeatherMsgsList, &un32Items);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to get number of list items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }
        if (un32Items == 0)
        {
            //nothing to do, exit
            bResult = TRUE;
            break;
        }

        if (psObj->asCacheItem[WEATHER_FCST_TYPE_SKI].un32TimestampUTCsec == 0)
        {
            // nothing to read, but this is not an error
            bResult = TRUE;
            break;
        }

        OSAL.bMemSet(&sIteratorArg, 0, sizeof(sIteratorArg));
        sIteratorArg.psObj = psObj;
        sIteratorArg.hObjectsList = hWeatherMsgsList;
        sIteratorArg.hHashEntry = 
            &psObj->asCacheItem[WEATHER_FCST_TYPE_SKI];
        sIteratorArg.hParent = hParent;

        eReturnCode = OSAL.eBufferBlocksIterate(
            psObj->ahPayload[WEATHER_FCST_TYPE_SKI], 
            (OSAL_BUFFER_ITERATOR_HANDLER)eParseSkiReports, &sIteratorArg);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to parse ski data");
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*    bBuildPersistentFolderPath
*
*******************************************************************************/
static BOOLEAN bBuildPersistentFolderPath(
    WEATHER1_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        const char *pacSMSPath = 0;
        size_t tSMSPathLen = 0;
        size_t tPathLen = 0;

        pacSMSPath = SMS_pacGetPath();
        if (pacSMSPath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to get SMS Path"
                        );
            break;
        }

        tSMSPathLen = strlen(pacSMSPath);

        tPathLen = tSMSPathLen +
                     strlen(WEATHER1_PERSISTENT_FOLDER_NAME) + 2;

        // Now we know the size of the path, so we can now
        // allocate the proper amount of memory
        psObj->pacPersistentFolderPath =
            (char*)SMSO_hCreate(
            WEATHER1_OBJECT_NAME":PersistentPath",
            tPathLen + 1,
            (SMS_OBJECT)psObj,
            FALSE
                );

        // Ensure allocation succeeded
        if (psObj->pacPersistentFolderPath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to allocate "
                        "persistent path memory"
                        );
            break;
        }

        // Construct the full location database filename
        snprintf(psObj->pacPersistentFolderPath,
            tPathLen,
            "%s/%s",
            pacSMSPath,
            WEATHER1_PERSISTENT_FOLDER_NAME
                );

        printf(WEATHER1_OBJECT_NAME": persistent path created: %s\n",
            psObj->pacPersistentFolderPath
                );

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*    n16FindMsgByLOCID
*
*******************************************************************************/
static N16 n16FindMsgByLOCID(
    WEATHER_MSG_OBJECT hWeatherMsg,
    LOC_ID *ptID
        )
{
    N16 n16Result = N16_MIN;

    if ((hWeatherMsg != WEATHER_MSG_INVALID_OBJECT) && (ptID != NULL))
    {
        LOC_ID tCurID = LOC_INVALID_ID;

        tCurID = WEATHER_MSG_tGetLOCID(hWeatherMsg);
        if (tCurID == *ptID)
        {
            n16Result = 0;
        }
        else
        {
            n16Result = 1;
        }
    }

    return n16Result;
}

/*******************************************************************************
*
*    bAssignForecastData
*
*******************************************************************************/
static BOOLEAN bAssignForecastData(
    WEATHER1_OBJECT_STRUCT *psObj,
    SMS_OBJECT hParent,
    WEATHER_FORECAST_TYPE_ENUM eForecastType,
    FORECAST_DATA_STRUCT *psData,
    LOC_ID tID,
    OSAL_OBJECT_HDL hWeatherMsgsList
        )
{
    BOOLEAN bResult = FALSE;
    FORECAST_OBJECT hForecast = FORECAST_INVALID_OBJECT;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        WEATHER_MSG_OBJECT hWeatherMsg = WEATHER_MSG_INVALID_OBJECT;
        BOOLEAN bSuccess = FALSE;

        eReturnCode = OSAL.eLinkedListLinearSearch(hWeatherMsgsList, &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindMsgByLOCID, &tID
                );
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // there is no such location in our list
            bResult = TRUE;
            break;
        }
        else if ((eReturnCode != OSAL_SUCCESS) ||
            (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to search msg for loc id %d", tID
                    );
            break;
        }

        hWeatherMsg = (WEATHER_MSG_OBJECT)OSAL.pvLinkedListThis(hEntry);
        if (hWeatherMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to get weather msg from entry"
                    );
            break;
        }

        hForecast = FORECAST_hCreate(hParent, psData,
            psObj->asCacheItem[eForecastType].un32TimestampUTCsec
                );
        if (hForecast == FORECAST_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to create forecast object"
                    );
            break;
        }

        bSuccess = WEATHER_MSG_bSetForecast(hWeatherMsg, eForecastType,
            hForecast
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to set forecast object for weather msg"
                    );
            break;
        }
        //reset handle to avoid dealloc
        hForecast = FORECAST_INVALID_OBJECT;

        bResult = TRUE;
    } while (FALSE);

    if (hForecast != FORECAST_INVALID_OBJECT)
    {
        FORECAST_vDestroy(hForecast);
    }

    return bResult;
}

/*******************************************************************************
*
*    bAssignSkiData
*
*******************************************************************************/
static BOOLEAN bAssignSkiData(
    WEATHER1_OBJECT_STRUCT *psObj,
    SMS_OBJECT hParent,
    SKI_DATA_STRUCT *psData,
    LOC_ID tID,
    OSAL_OBJECT_HDL hWeatherMsgsList
        )
{
    BOOLEAN bResult = FALSE;
    SKI_CONDITIONS_OBJECT hSkiCond = SKI_CONDITIONS_INVALID_OBJECT;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        WEATHER_MSG_OBJECT hWeatherMsg = WEATHER_MSG_INVALID_OBJECT;
        BOOLEAN bSuccess = FALSE;

        eReturnCode = OSAL.eLinkedListLinearSearch(hWeatherMsgsList, &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindMsgByLOCID, &tID
                );
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // there is no such location in list
            bResult = TRUE;
            break;
        }
        else if ((eReturnCode != OSAL_SUCCESS) ||
            (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to search msg for loc id %d", tID
                    );
            break;
        }

        hWeatherMsg = OSAL.pvLinkedListThis(hEntry);
        if (hWeatherMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to get weather msg from entry"
                    );
            break;
        }

        hSkiCond = SKI_CONDITIONS_hCreate(hParent, psData,
            psObj->asCacheItem[WEATHER_FCST_TYPE_SKI].un32TimestampUTCsec
                );
        if (hSkiCond == SKI_CONDITIONS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to create ski conditions object"
                    );
            break;
        }

        bSuccess = WEATHER_MSG_bSetSkiCond(hWeatherMsg, hSkiCond);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME
                ": failed to set skicond object for weather msg"
                    );
            break;
        }
        //reset handle to avoid dealloc
        hSkiCond = SKI_CONDITIONS_INVALID_OBJECT;

        bResult = TRUE;
    } while (FALSE);

    if (hSkiCond != SKI_CONDITIONS_INVALID_OBJECT)
    {
        SKI_CONDITIONS_vDestroy(hSkiCond);
    }

    return bResult;
}

/*******************************************************************************
*
*        vUninitObject
*
*******************************************************************************/
static void vUninitObject(
    WEATHER1_OBJECT_STRUCT *psObj
        )
{
    UN8 un8Index;

    if (psObj == NULL)
    {
        return;
    }

    printf(WEATHER1_OBJECT_NAME": Releasing CRC.\n");

    if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eReleaseCRC(psObj->hCRC);
        psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
    }

    printf(WEATHER1_OBJECT_NAME": Disconnecting RFD\n");

    if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
    {
        RFD_INTERFACE_vDisconnect(psObj->hRFD);
        psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;
    }

    if (psObj->pacPersistentFolderPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacPersistentFolderPath);
        psObj->pacPersistentFolderPath = NULL;
    }

    if (psObj->pacCacheFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacCacheFilePath);
        psObj->pacCacheFilePath = NULL;
    }

    for (un8Index = 0; un8Index < 
        sizeof(psObj->ahPayload)/sizeof(psObj->ahPayload[0]); un8Index++)
    {
        if (psObj->ahPayload[un8Index] != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL.eBufferFree(psObj->ahPayload[un8Index]);
            psObj->ahPayload[un8Index] = OSAL_INVALID_BUFFER_HDL;
        }
    }

    if (psObj->hBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eBlockPoolDelete(psObj->hBlockPool);
        psObj->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    }

    SMSO_vDestroy((SMS_OBJECT)psObj);

    return;
}


/*******************************************************************************
*
*     bHashDataCompare
*
*******************************************************************************/
static BOOLEAN bHashDataCompare(
    WEATHER_HASH_ITEM_DATA_STRUCT *psHashData1,
    WEATHER_HASH_ITEM_DATA_STRUCT *psHashData2
        )
{
    BOOLEAN bResult = FALSE;

    if ((psHashData1->tCRC == psHashData2->tCRC) &&
        (psHashData1->un8FirstAreaID == psHashData2->un8FirstAreaID) &&
        (psHashData1->un8FirstStateID == psHashData2->un8FirstStateID) &&
        (psHashData1->tAUSize == psHashData2->tAUSize))
    {
        bResult = TRUE;
    }

    return bResult;
}


/*****************************************************************************
*
*   eRFDFileProcessor
*
*   The callback invoked by RFD when an update file has been fully recieved
*   and is ready for processing
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        )
{
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl = (WEATHER1_RFD_CTRL_STRUCT *)pvCallbackArg;
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;

    if (psRFDCtrl == NULL)
    {
        return RFD_PROCESS_RESULT_ERROR;
    }

    if (eProcessStatus == RFD_PROCESS_STATUS_BEGIN)
    {
        BOOLEAN bOk;

        // Initialize the RFD Ctrl object now
        bOk = bInitRFDCtrl(psRFDCtrl, hConnection, tProgressIndex);

        if (bOk == TRUE)
        {
            // Process the update header now
            bOk = bProcessRFDHeader(psRFDCtrl, psRFDFile);
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": Unable to process RFD file header");

            return RFD_PROCESS_RESULT_ERROR;
        }
    }
    else if (eProcessStatus == RFD_PROCESS_STATUS_STOP)
    {
        // Free associated memory if we're stopping
        vDestroyRFDCtrl(psRFDCtrl);

        return RFD_PROCESS_RESULT_INCOMPLETE;
    }

    // Process the file contents and update the database
    eResult = eProcessNextRFDEntry(
        psRFDCtrl, psRFDFile, tFileVersion);
    if (eResult != RFD_PROCESS_RESULT_INCOMPLETE)
    {
        // Close our connection to the DB
        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
            psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Are we done?
        if (eResult == RFD_PROCESS_RESULT_COMPLETE)
        {
            BOOLEAN bFinalized;

            // Finalize the new database
            bFinalized = bFinalizeDB(psRFDCtrl);
            if (bFinalized == FALSE)
            {
                // We failed to get the DB ready for use
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }

        // Did we experience an error?
        if (eResult  >= RFD_PROCESS_RESULT_ERROR)
        {
            // Delete database
            vDeleteDB(psRFDCtrl);
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   bProcessRFDHeader
*
*****************************************************************************/
static BOOLEAN bProcessRFDHeader (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psRFDFile
        )
{
    BOOLEAN bDataTransferred;
    size_t tBitsRead;

    do
    {
        // Fill the buffer with file data
        bDataTransferred = DATASERVICE_MGR_bFillBufferBlock(
            psRFDFile, psRFDCtrl->hBuffer);
        if (bDataTransferred == FALSE)
        {
            // Couldn't access the file
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": Unable to read file data to buffer");
            break;
        }

        // Read version
        psRFDCtrl->sLocation.un8Version = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer,
            &psRFDCtrl->sLocation.un8Version, 0,
            WEATHER1_LOCATION_VER_BITLEN);
        if (tBitsRead != WEATHER1_LOCATION_VER_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": cannot read version from buffer");
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   eProcessNextRFDEntry
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    RFD_PROCESS_RESULT_ENUM eResult;
    BOOLEAN bSuccess = TRUE, bInformManager = TRUE;

    // Did we already process this group?
    if (psRFDCtrl->tCurProgressIndex < psRFDCtrl->tStartProgressIndex)
    {
        bInformManager = FALSE;
    }

    // Process the next group
    eResult = eProcessUpdateGroup(psRFDCtrl, psFile, bInformManager);

    if (eResult != RFD_PROCESS_RESULT_ERROR)
    {
        // Increment our progress index
        psRFDCtrl->tCurProgressIndex++;

        if (psRFDCtrl->tCurProgressIndex > psRFDCtrl->tStartProgressIndex)
        {
            // Report our progress
            bSuccess = RFD_INTERFACE_bReportProgress(
                psRFDCtrl->hRFD, psRFDCtrl->tCurProgressIndex);
        }
    }
	else
	{
	    bSuccess = FALSE;
	}

    if (bSuccess == TRUE)
    {
        // Are we done?
        if (eResult == RFD_PROCESS_RESULT_COMPLETE)
        {
            bSuccess = GsWeatherMgr.bDBUpdateEnd(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->acBuffer,
                WEATHER_MAX_SQL_STRING_LENGTH,
                (UN8)tUpdateVersion);
        }
    }

    if (bSuccess == FALSE)
    {
        eResult = RFD_PROCESS_RESULT_ERROR;
    }

    return eResult;
}

/*****************************************************************************
*
*   eProcessUpdateGroup
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessUpdateGroup (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    BOOLEAN bInformManager
        )
{
    size_t tNumLeftToProcess = 50;
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_INCOMPLETE;
    BOOLEAN bOk;
    WEATHER1_LOCATION_UPDATE_TYPE_ENUM eUpdateType;
    UN8 un8Buffer;
    size_t tBitsRead;

    // Start a transaction for this group
    bOk = SQL_INTERFACE.bStartTransaction(psRFDCtrl->hDBConnection);
    if (bOk == FALSE)
    {
        return RFD_PROCESS_RESULT_ERROR;
    }

    // Process until we've finished the batch, or
    // until we're done, or until an error occurs
    while ((tNumLeftToProcess-- > 0) &&
           (eResult == RFD_PROCESS_RESULT_INCOMPLETE))
    {
        // Fill the next block if need be
        DATASERVICE_MGR_bFillBufferBlock(
            psFile, psRFDCtrl->hBuffer);

        // Read update type
        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8Buffer, 0,
            WEATHER1_LOCATION_UTYPE_BITLEN);
        if (tBitsRead != WEATHER1_LOCATION_UTYPE_BITLEN)
        {
            eResult = RFD_PROCESS_RESULT_ERROR;
            break;
        }

        // Convert to our enum
        eUpdateType = (WEATHER1_LOCATION_UPDATE_TYPE_ENUM)un8Buffer;

        switch (eUpdateType)
        {
            case WEATHER_MGR_LOCATION_DELETE:
            {
                bOk = bProcessLocationDelete(
                    psRFDCtrl, bInformManager);
            }
            break;

            case WEATHER_MGR_LOCATION_NEW:
            {
                bOk = bProcessLocationNew(
                    psRFDCtrl, bInformManager);
            }
            break;

            case WEATHER_MGR_LOCATION_MODIFY:
            {
                bOk = bProcessLocationModify(
                    psRFDCtrl, bInformManager);
            }
            break;

            case WEATHER_MGR_LOCATION_EOL:
            {
                bOk = TRUE;
                eResult = RFD_PROCESS_RESULT_COMPLETE;
            }
            break;

            default:
            {
                bOk = FALSE;
            }
            break;
        }

        if (bOk == FALSE)
        {
            eResult = RFD_PROCESS_RESULT_ERROR;
        }
    }

    // End the transaction for this group
    bOk = SQL_INTERFACE.bEndTransaction(psRFDCtrl->hDBConnection);
    if (bOk == FALSE)
    {
        eResult = RFD_PROCESS_RESULT_ERROR;
    }

    return eResult;
}

/*****************************************************************************
*
*   bBuildCacheFilePath
*
*****************************************************************************/
static BOOLEAN bBuildCacheFilePath (
    WEATHER1_OBJECT_STRUCT *psObj
        )
{
    size_t tPathLen;

    tPathLen = strlen(psObj->pacPersistentFolderPath) + 
        sizeof(WEATHER1_CACHE_FILE_NAME) + 1;

    psObj->pacCacheFilePath = (char *)SMSO_hCreate(
        WEATHER1_OBJECT_NAME":CacheFilePath", tPathLen,
        (SMS_OBJECT)psObj, FALSE);
    if (psObj->pacCacheFilePath == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WEATHER1_OBJECT_NAME": failed to allocate memory for "
            "file path");
    }
    else
    {
        snprintf(psObj->pacCacheFilePath, tPathLen, "%s/%s",
            psObj->pacPersistentFolderPath, WEATHER1_CACHE_FILE_NAME);

        printf(WEATHER1_OBJECT_NAME": File path generated %s\n", 
            psObj->pacCacheFilePath);

        return TRUE;
    }

    return FALSE;
}


/*****************************************************************************
*
*   vDeleteCache
*
*****************************************************************************/
static void vDeleteCache (
    WEATHER1_OBJECT_STRUCT *psObj
        )
{
    int iResult;

    iResult = remove(psObj->pacCacheFilePath);
    if (iResult != 0)
    {
        printf(WEATHER1_OBJECT_NAME": cannot remove file %s, %d\n",
            psObj->pacCacheFilePath, iResult);
    }

    return;
}

/*****************************************************************************
*
*   bCheckBufferCRC
*
*****************************************************************************/
static BOOLEAN bCheckBufferCRC(
    OSAL_OBJECT_HDL hCRC,
    OSAL_BUFFER_HDL hBuffer,
    OSAL_CRC_RESULT tCRC,
    size_t tBufferSize
        )
{
    do
    {
        OSAL_CRC_RESULT tCalculatedCRC;
        BOOLEAN bSuccess;
        size_t tSize = 0;

        // Initialize the CRC
        bSuccess = OSAL.bInitializeCRC(hCRC, &tCalculatedCRC);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Compute the CRC of the entire payload
        tCalculatedCRC = OSAL.tComputeCRC(hCRC, tCalculatedCRC, 
            hBuffer, 0, tBufferSize, &tSize);
        if (tBufferSize != tSize)
        {
            // We didn't process the entire payload
            break;
        }

        // Invert the CRC bits
        tCalculatedCRC ^= DS_UTIL_AU_CRC_INVERSION_VALUE;
        if (tCRC != tCalculatedCRC)
        {
            break;
        }

        return TRUE;
    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bReadCachedPayload
*
*****************************************************************************/
static BOOLEAN bReadCachedPayload(
    WEATHER1_OBJECT_STRUCT *psObj,
    WEATHER1_CACHE_FILE_STRUCT *psCacheItem,
    size_t tOffset,
    FILE *psCacheFile,
    OSAL_BUFFER_HDL hBuffer
        )
{
    OSAL_BUFFER_BLOCK_HDL hBlock;

    do
    {
        int iResult;
        UN8 *pun8Data;
        size_t tSize;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bSuccess;


        if (tOffset != 0)
        {
            // setting position to start of our current payload
            iResult = fseek(psCacheFile, tOffset, SEEK_CUR);
            if (iResult != 0)
            {
                printf(WEATHER1_OBJECT_NAME": seek to offset %u failed\n",
                    tOffset);
                break;
            }
        }

        // getting the block from the buffer for reading
        hBlock = OSAL.hBufferGetBlock(hBuffer, &pun8Data, &tSize);
        if (hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
        {
            puts(WEATHER1_OBJECT_NAME": failed to get the block");
            break;
        }

        //checking if block space is enough to store the payload
        if (tSize < psCacheItem->sHash.tAUSize)
        {
            printf(WEATHER1_OBJECT_NAME": block is too small; block size "
                "%u, AU size %u\n", tSize, psCacheItem->sHash.tAUSize);
            break;
        }

        // reading payload from the file to block
        tSize = fread(pun8Data, psCacheItem->sHash.tAUSize, 1, 
            psCacheFile);
        if (tSize != 1)
        {
            printf(WEATHER1_OBJECT_NAME": payload size %u reading failed\n",
                psCacheItem->sHash.tAUSize);
            break;
        }

        eReturnCode = OSAL.eBufferWriteBlock(hBlock, 
            psCacheItem->sHash.tAUSize);
        if (eReturnCode != OSAL_SUCCESS)
        {
            printf(WEATHER1_OBJECT_NAME": block size %u writing failed "
                "(%s)\n", psCacheItem->sHash.tAUSize,
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bSuccess = bCheckBufferCRC(psObj->hCRC, hBuffer, 
            psCacheItem->sHash.tCRC, psCacheItem->sHash.tAUSize);
        if (bSuccess == FALSE)
        {
            puts(WEATHER1_OBJECT_NAME": payload CRC check failed");
            break;
        }

        return TRUE;
    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bReadCacheFile
*
*****************************************************************************/
static BOOLEAN bReadCacheFile(
    WEATHER1_OBJECT_STRUCT *psObj
        )
{
    FILE *psCacheFile = NULL;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    UN8 un8Index;

    do
    {
        BOOLEAN bSuccess;
        size_t tCount;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32SchemaVer, un32CurrentTime;
        OSAL_CRC_RESULT tCacheItemsCRC;

        // open cache data file
        psCacheFile = fopen(psObj->pacCacheFilePath, "rb");
        if (psCacheFile == NULL)
        {
            //probably it was not created yet
            printf(WEATHER1_OBJECT_NAME": cannot open cache file %s\n",
                psObj->pacCacheFilePath);
            return TRUE;
        }

        // reading schema version
        tCount = fread(&un32SchemaVer, sizeof(un32SchemaVer), 1,
            psCacheFile);
        if (tCount != 1)
        {
            // read has been failed
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read cache file "
                "schema ver");
            break;
        }

        // verifying schema version
        if (un32SchemaVer != WEATHER1_CACHE_SCHEMA_VER)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": Cache file schema version is wrong: "
                "%u, but should be %u", un32SchemaVer, 
                WEATHER1_CACHE_SCHEMA_VER);
            break;
        }

        // reading WEATHER1_CACHE_FILE_STRUCTs to array from the file
        tCount = fread(psObj->asCacheItem, sizeof(psObj->asCacheItem),
            1, psCacheFile);
        if (tCount != 1)
        {
            // read has been failed
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read cache file items "
                "array");
            break;
        }

        // reading CRC32 from file for cache items array
        tCount = fread(&tCacheItemsCRC, sizeof(tCacheItemsCRC), 1, 
            psCacheFile);
        if (tCount != 1)
        {
            // read has been failed
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read cache file items "
                "CRC32");
            break;
        }

        // preparing buffer for CRC calculation

        // allocating the buffer
        hBuffer = OSAL.hBufferAllocate(psObj->hBlockPool, FALSE, FALSE,
            OSAL_BUFFER_ALLOCATE_OPTION_NONE);
        if (hBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to allocate buffer");
            break;
        }

        // copying loaded data to buffer starting from schema version
        tCount = OSAL.tBufferWriteTail(hBuffer, &un32SchemaVer, 
            sizeof(un32SchemaVer));
        if (tCount != sizeof(un32SchemaVer))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to copy data to buffer");
            break;
        }

        // copying cache items to buffer
        tCount = OSAL.tBufferWriteTail(hBuffer, psObj->asCacheItem, 
            sizeof(psObj->asCacheItem));
        if (tCount != sizeof(psObj->asCacheItem))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to copy data to buffer");
            break;
        }

        //verifying the CRC for data in the buffer
        bSuccess = bCheckBufferCRC(psObj->hCRC, hBuffer, tCacheItemsCRC, 
            sizeof(psObj->asCacheItem) + sizeof(un32SchemaVer));
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": cache file CRC verification failed");
            break;
        }

        // if we are here, CRC is correct. Lets remove the buffer then.
        eReturnCode = OSAL.eBufferFree(hBuffer);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to remove the buffer (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }
        // preventing double free
        hBuffer = OSAL_INVALID_BUFFER_HDL;

        // Now loading the stored payloads

        // Getting current time
        eReturnCode = OSAL.eTimeGet(&un32CurrentTime);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to get OS time (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // initializing offset counter
        tCount = 0;

        for(un8Index = 0; un8Index < WEATHER_FCST_TYPE_MAX; un8Index++)
        {
            if (psObj->asCacheItem[un8Index].un32TimestampUTCsec == 0)
            {
                // this item has not been placed to cache file, moving
                // to next
                continue;
            }
            // check if this forecast/ski report is still valid
            // using only minutes for comparison here
            // since time indication event provides minutes only
            // (12:00:00 for example), while actual time could be 12:00:10
            // and file could be stored right before (12:00:03 for example)
            else if (((un32CurrentTime / 60) - 
                (psObj->asCacheItem[un8Index].un32TimestampUTCsec / 60)) >
                WEATHER_MSG_TIMEOUT / 60)
            {
                // if timeouted, set timestamp to zero and 
                // increment the offset value
                psObj->asCacheItem[un8Index].un32TimestampUTCsec = 0;
                tCount += psObj->asCacheItem[un8Index].sHash.tAUSize;
            }
            else
            {
                // this payload is not expired. Lets allocate the buffer
                // and load the data from file
                psObj->ahPayload[un8Index] = OSAL.hBufferAllocate(
                    psObj->hBlockPool, FALSE, FALSE, 
                    OSAL_BUFFER_ALLOCATE_OPTION_NONE);
                if (psObj->ahPayload[un8Index] == OSAL_INVALID_BUFFER_HDL)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to allocate buffer");
                    bSuccess = FALSE;
                    break;
                }

                bSuccess = bReadCachedPayload(psObj, 
                    &psObj->asCacheItem[un8Index], tCount, psCacheFile, 
                    psObj->ahPayload[un8Index]);
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to read payload "
                        "from cache file");
                    break;
                }

                // reset the offset since position has been changed
                tCount = 0;
            }
        }

        // if error happened in cycle, lets break here to make cleanup
        // before an exit
        if (bSuccess == FALSE)
        {
            break;
        }

        fclose(psCacheFile);

        return TRUE;
    } while (FALSE);

    if (psCacheFile != NULL)
    {
        fclose(psCacheFile);

        // in case of any error, removing cache file
        vDeleteCache(psObj);
    }

    if (hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL.eBufferFree(hBuffer);
    }

    // clear cache index structure in case if reading was failed
    OSAL.bMemSet(psObj->asCacheItem, 0, sizeof(psObj->asCacheItem));

    // clear all the payloads as potentially corrupted
    for (un8Index = 0; un8Index < WEATHER_FCST_TYPE_MAX; un8Index++)
    {
        if (psObj->ahPayload[un8Index] != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL.eBufferFree(psObj->ahPayload[un8Index]);
            psObj->ahPayload[un8Index] = OSAL_INVALID_BUFFER_HDL;
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*   bWriteCacheFile
*
*****************************************************************************/
static BOOLEAN bWriteCacheFile(
    WEATHER1_OBJECT_STRUCT *psObj
        )
{
    FILE *psCacheFile = NULL;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;

    do
    {
        BOOLEAN bSuccess = TRUE;
        size_t tCount, tBytesCounter;
        OSAL_CRC_RESULT tCRC = 0;
        UN32 un32SchemaVer = WEATHER1_CACHE_SCHEMA_VER;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN8 un8Index = 0;

        // allocating buffer for our cache index
        hBuffer = OSAL.hBufferAllocate(psObj->hBlockPool, FALSE, FALSE,
            OSAL_BUFFER_ALLOCATE_OPTION_NONE);
        if (hBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": cannot allocate buffer.");
            break;
        }

        // writing cache schema version to buffer
        tCount = OSAL.tBufferWriteTail(hBuffer,
            &un32SchemaVer, sizeof(un32SchemaVer));
        if (tCount != sizeof(un32SchemaVer))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": Failed to put schema version "
                "to buffer");
            break;
        }

        // writing cache index to buffer
        tCount += OSAL.tBufferWriteTail(hBuffer, psObj->asCacheItem,
            sizeof(psObj->asCacheItem));
        printf(WEATHER1_OBJECT_NAME": Written %d bytes of cache index "
            "to buffer\n", tCount);
        if (tCount != sizeof(psObj->asCacheItem) + sizeof(un32SchemaVer))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to put cache index to "
                "buffer.");
            break;
        }

        // calculating CRC for our cache index
        // Initialize the CRC
        bSuccess = OSAL.bInitializeCRC(psObj->hCRC, &tCRC);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to initialize CRC.");
            break;
        }

        // Compute the CRC of the buffer
        tCRC = OSAL.tComputeCRC(psObj->hCRC, tCRC, hBuffer, 0,
            tCount, &tBytesCounter);
        // Did we process the entire payload?
        if (tBytesCounter != tCount)
        {
            // No!
            break;
        }

        tCRC ^= DS_UTIL_AU_CRC_INVERSION_VALUE;

        printf(WEATHER1_OBJECT_NAME": Computed index CRC 0x%x\n", tCRC);

        // create or recreate cache file
        psCacheFile = fopen(psObj->pacCacheFilePath, "wb+");
        if (psCacheFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": cannot open index file %s",
                psObj->pacCacheFilePath);
            break;
        }

        eReturnCode = OSAL.eBufferWriteToFile(hBuffer, TRUE, psCacheFile);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to write buffer to file (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        eReturnCode = OSAL.eBufferFree(hBuffer);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to free the buffer (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // reset buffer handle to avoid double free
        hBuffer = OSAL_INVALID_BUFFER_HDL;

        tCount = fwrite(&tCRC, sizeof(tCRC), 1, psCacheFile);
        if (tCount != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to write the CRC to file");
            break;
        }

        for (un8Index = 0; un8Index < WEATHER_FCST_TYPE_MAX; un8Index++)
        {
            if (psObj->asCacheItem[un8Index].un32TimestampUTCsec != 0)
            {
                // we have valid payload for this forecast/ski report
                // lets put this payload to file
                eReturnCode =  OSAL.eBufferWriteToFile(
                    psObj->ahPayload[un8Index], FALSE, psCacheFile);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to write buffer "
                        "to file (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));

                    // marking the writing process as failed
                    bSuccess = FALSE;
                    break;
                }
            }
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        fclose(psCacheFile);

        return TRUE;
    } while (FALSE);

    if (psCacheFile != NULL)
    {
        fclose(psCacheFile);

        // in case of any error, removing cache file
        // we do not want to keep the potentially corrupted data for
        // next run
        vDeleteCache(psObj);
    }

    if (hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL.eBufferFree(hBuffer);
    }

    return FALSE;
}

/*****************************************************************************
*
*   bUpdateCache
*
*****************************************************************************/
static BOOLEAN bUpdateCache(
    WEATHER1_OBJECT_STRUCT *psObj,
    WEATHER_FCST_TYPE_ENUM eFcst,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        psObj->asCacheItem[eFcst].sHash = psObj->sHash;

        eReturnCode = OSAL.eTimeGet(
            &(psObj->asCacheItem[eFcst].un32TimestampUTCsec));
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to get OS time"
                    );
            break;
        }

        if (psObj->ahPayload[eFcst] != OSAL_INVALID_BUFFER_HDL)
        {
            // updating payload in cache
            eReturnCode = OSAL.eBufferFree(psObj->ahPayload[eFcst]);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to free cache buffer (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }

        psObj->ahPayload[eFcst] = OSAL.hBufferAllocate(psObj->hBlockPool,
            FALSE, FALSE, OSAL_BUFFER_ALLOCATE_OPTION_NONE);
        if (psObj->ahPayload[eFcst] == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to allocate cache buffer");
            break;
        }

        eReturnCode = OSAL.eBufferAppend(psObj->ahPayload[eFcst],
            hPayload);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to copy payload to cache "
                "buffer (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessWeatherForecast
*
*****************************************************************************/
static BOOLEAN bProcessWeatherForecast (
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bResult = FALSE;

    printf(WEATHER1_OBJECT_NAME": [%s] start with parameters psObj = %p, "
           "hPayload = %p\n", __FUNCTION__, psObj, hPayload);

    do
    {
        WEATHER_FCST_TYPE_ENUM eFcst = WEATHER_FCST_TYPE_MAX;
        BOOLEAN bSuccess = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        bSuccess = bProcessHash(psObj, hPayload, FALSE, &eFcst);
        if (bSuccess == FALSE)
        {
            printf(WEATHER1_OBJECT_NAME "Hash parsing failed.\n");
            break;
        }

        // Since hash is processed correctly, cache sync is required
        psObj->bCacheSyncNeeded = TRUE;

        // check array if we already have hash item for this FCST type
        if (psObj->asCacheItem[eFcst].un32TimestampUTCsec != 0)
        {
            //looks like we have something in array,
            //lets check if message is the same
            bSuccess = bHashDataCompare(&psObj->asCacheItem[eFcst].sHash,
                &psObj->sHash
                    );
            if (bSuccess == TRUE)
            {
                eReturnCode = OSAL.eTimeGet(
                    &(psObj->asCacheItem[eFcst].un32TimestampUTCsec)
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to get OS time"
                            );
                    break;
                }

                GsWeatherMgr.bForecastHashRepeated(psObj->hWeatherService,
                    &psObj->asCacheItem[eFcst]
                        );
            }
            else
            {
                //this is new hash, need to replace data
                GsWeatherMgr.bForecastHashRemoved(psObj->hWeatherService,
                    &psObj->asCacheItem[eFcst]
                        );

                bSuccess = bUpdateCache(psObj, eFcst, hPayload);
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to update cache"
                            );
                }

                vParseWeatherMsgs(psObj, psObj->ahPayload[eFcst],
                    &psObj->asCacheItem[eFcst], eFcst
                        );
            }
        }
        else
        {
            bSuccess = bUpdateCache(psObj, eFcst, hPayload);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to update cache"
                        );
            }

            vParseWeatherMsgs(psObj, psObj->ahPayload[eFcst],
                &psObj->asCacheItem[eFcst], eFcst
                    );
        }

        bResult = TRUE;
    } while (FALSE);

    printf(WEATHER1_OBJECT_NAME": [%s] end result %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   bProcessSkiReport
*
*****************************************************************************/
static BOOLEAN bProcessSkiReport (
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bResult = FALSE;

    printf(WEATHER1_OBJECT_NAME": [%s] start with parameters psObj = %p, "
        "hPayload = %p\n", __FUNCTION__, psObj, hPayload);

    do
    {
        WEATHER_FCST_TYPE_ENUM eFcst = WEATHER_FCST_TYPE_MAX;
        BOOLEAN bSuccess = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        bSuccess = bProcessHash(psObj, hPayload, TRUE, &eFcst);
        if (bSuccess == FALSE)
        {
            printf(WEATHER1_OBJECT_NAME "Hash parsing failed.\n");
            break;
        }

        // Since hash is processed correctly, cache sync is required
        psObj->bCacheSyncNeeded = TRUE;

        // check array if we already have hash item for this FCST type
        if (psObj->asCacheItem[eFcst].un32TimestampUTCsec != 0)
        {
            //looks like we have something in array,
            //lets check if message is the same
            bSuccess = bHashDataCompare(&psObj->asCacheItem[eFcst].sHash,
                &psObj->sHash
                    );
            if (bSuccess == TRUE)
            {
                eReturnCode = OSAL.eTimeGet(
                    &(psObj->asCacheItem[eFcst].un32TimestampUTCsec)
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to get OS time"
                            );
                    break;
                }

                GsWeatherMgr.bSkiHashRepeated(psObj->hWeatherService,
                    &psObj->asCacheItem[eFcst]
                        );
            }
            else
            {
                //this is new hash, need to replace data
                GsWeatherMgr.bSkiHashRemoved(psObj->hWeatherService,
                    &psObj->asCacheItem[eFcst]
                        );

                bSuccess = bUpdateCache(psObj, eFcst, hPayload);
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WEATHER1_OBJECT_NAME": failed to update cache"
                            );
                    break;
                }
                vParseSkiMsgs(psObj, psObj->ahPayload[eFcst],
                    &psObj->asCacheItem[eFcst]);
            }
        }
        else
        {
            bSuccess = bUpdateCache(psObj, eFcst, hPayload);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WEATHER1_OBJECT_NAME": failed to update cache");
                break;
            }
            vParseSkiMsgs(psObj, psObj->ahPayload[eFcst],
                &psObj->asCacheItem[eFcst]);
        }

        bResult = TRUE;
    } while (FALSE);

    printf(WEATHER1_OBJECT_NAME": [%s] end result %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   vParseWeatherMsgs
*
*****************************************************************************/
static void vParseWeatherMsgs(
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    WEATHER_HASH_OBJECT hHashEntry,
    WEATHER_FCST_TYPE_ENUM eFcst
        )
{
    BOOLEAN bSuccess;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    WEATHER1_PARSE_ITERATOR_ARG sIteratorArg;

    bSuccess = GsWeatherMgr.bStartForecastProcessing(psObj->hWeatherService);
    if (bSuccess == FALSE)
    {
        return;
    }

    OSAL.bMemSet(&sIteratorArg, 0, sizeof(sIteratorArg));

    sIteratorArg.psObj = psObj;
    sIteratorArg.hHashEntry = hHashEntry;
    sIteratorArg.eFcst = eFcst;

    eReturnCode = OSAL.eBufferBlocksIterate(hPayload, 
        (OSAL_BUFFER_ITERATOR_HANDLER)eParseForecasts, &sIteratorArg);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
            WEATHER1_OBJECT_NAME": Failed to iterate forecast payload "
            "blocks (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
    }

    GsWeatherMgr.bEndForecastProcessing(psObj->hWeatherService);

    return;
}

/*****************************************************************************
*
*   eParseForecasts
*
*****************************************************************************/
OSAL_BUFFER_ITERATOR_RESULT_ENUM eParseForecasts(
    UN8 *pun8Source,
    size_t tDataSize,
    WEATHER1_PARSE_ITERATOR_ARG *psIteratorArg
        )
{
    LOC_ID tPayloadID;
    LOC_ID tPrevID = LOC_INVALID_ID;
    FORECAST_DATA_STRUCT sData;
    BOOLEAN bSuccess;

    // skipping PVN/CARID/FCST
    size_t tBitsOffset = WEATHER1_FORECAST_HEADER_SIZE,
        tBufferSizeInBits;

    tBufferSizeInBits = tDataSize * UN8_BITLEN;

    printf(WEATHER1_OBJECT_NAME": Parsing forecasts fcst type %d, block "
        "size %d\n", psIteratorArg->eFcst, tDataSize);

    do
    {
        tPayloadID = tGetPayloadID(pun8Source, tPrevID,
            tBufferSizeInBits, &tBitsOffset);
        if (tPayloadID == LOC_INVALID_ID)
        {
            break;
        }

        bSuccess = OSAL.bMemSet(&sData,0,sizeof(sData));
        if (bSuccess == FALSE)
        {
            // Epic fail!
            break;
        }

        bSuccess = bParseForecastData(pun8Source, &sData, 
            tBufferSizeInBits, &tBitsOffset);
        if (bSuccess == FALSE)
        {
            break;
        }

        sData.hHashEntry = psIteratorArg->hHashEntry;
        if (psIteratorArg->hObjectsList != OSAL_INVALID_OBJECT_HDL)
        {
            // objects list provided, do the direct assignment
            bSuccess = bAssignForecastData(psIteratorArg->psObj, 
                psIteratorArg->hParent,
                (WEATHER_FORECAST_TYPE_ENUM)psIteratorArg->eFcst, &sData,
                tPayloadID, psIteratorArg->hObjectsList);
        }
        else
        {
            bSuccess = GsWeatherMgr.bProcessForecastReport(
                psIteratorArg->psObj->hWeatherService, tPayloadID, 
                (WEATHER_FORECAST_TYPE_ENUM)psIteratorArg->eFcst, &sData);
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        tPrevID = tPayloadID;
    } while (TRUE);

    return OSAL_BUFFER_ITERATOR_RESULT_KEEP;
}

/*******************************************************************************
*
*   vParseSkiMsgs
*
*******************************************************************************/
static void vParseSkiMsgs(
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    WEATHER_HASH_OBJECT hHashEntry
        )
{
    BOOLEAN bSuccess;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    WEATHER1_PARSE_ITERATOR_ARG sIteratorArg;

    bSuccess = GsWeatherMgr.bStartSkiProcessing(psObj->hWeatherService);
    if (bSuccess == FALSE)
    {
        return;
    }

    OSAL.bMemSet(&sIteratorArg, 0, sizeof(sIteratorArg));
    sIteratorArg.psObj = psObj;
    sIteratorArg.hHashEntry = hHashEntry;

    eReturnCode = OSAL.eBufferBlocksIterate(hPayload, 
        (OSAL_BUFFER_ITERATOR_HANDLER)eParseSkiReports, &sIteratorArg);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
            WEATHER1_OBJECT_NAME": Failed to iterate ski payload "
            "blocks (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
    }

    GsWeatherMgr.bEndSkiProcessing(psObj->hWeatherService);

    return;
}

/*******************************************************************************
*
*   eParseSkiReports
*
*******************************************************************************/
static OSAL_BUFFER_ITERATOR_RESULT_ENUM eParseSkiReports(
    UN8 *pun8Source,
    size_t tDataSize,
    WEATHER1_PARSE_ITERATOR_ARG *psIteratorArg
        )
{
    LOC_ID tPayloadID;
    LOC_ID tPrevID = LOC_INVALID_ID;
    SKI_DATA_STRUCT sData;
    BOOLEAN bSuccess;

    // skipping PVN/CARID
    size_t tBitsOffset = WEATHER1_SKI_REPORT_HEADER_SIZE,
        tBufferSizeInBits;

    tBufferSizeInBits = tDataSize * UN8_BITLEN;

    do
    {
        tPayloadID = tGetPayloadID(pun8Source, tPrevID,
            tBufferSizeInBits, &tBitsOffset);
        if (tPayloadID == LOC_INVALID_ID)
        {
            break;
        }

        bSuccess = OSAL.bMemSet(&sData,0,sizeof(sData));
        if (bSuccess == FALSE)
        {
            // Epic fail!
            break;
        }

        bSuccess = bParseSkiData(pun8Source, &sData, tBufferSizeInBits, 
            &tBitsOffset);
        if (bSuccess == FALSE)
        {
            break;
        }
        sData.hHashEntry = psIteratorArg->hHashEntry;

        if (psIteratorArg->hObjectsList != OSAL_INVALID_OBJECT_HDL)
        {
            // objects list provided, lets try direct assignment
            bSuccess = bAssignSkiData(psIteratorArg->psObj, 
                psIteratorArg->hParent, &sData, tPayloadID, 
                psIteratorArg->hObjectsList);
        }
        else
        {
            bSuccess = GsWeatherMgr.bProcessSkiReport(
                psIteratorArg->psObj->hWeatherService, tPayloadID, &sData);
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        tPrevID = tPayloadID;
    } while (TRUE);

    return OSAL_BUFFER_ITERATOR_RESULT_KEEP;
}

/*****************************************************************************
*
*   tGetPayloadID
*
*****************************************************************************/
static LOC_ID tGetPayloadID(
    UN8 *pun8Source,
    LOC_ID tPrevID,
    size_t tBufferSizeInBits,
    size_t *ptBitsOffset
        )
{
    LOC_ID tResult = LOC_INVALID_ID;

    do
    {
        BOOLEAN bSuccess;
        UN8 un8Buffer, un8Aseq;
        STATE_ID tStateID = STATE_INVALID_ID;
        AREA_ID tAreaID = AREA_INVALID_ID;

        // Getting ASEQ field
        bSuccess = bReadUN8(WEATHER1_ASEQ_BITLEN, ptBitsOffset, 
            tBufferSizeInBits, pun8Source, &un8Aseq);
        if (bSuccess == FALSE)
        {
            break;
        }

        if ((un8Aseq == WEATHER1_ASEQ_NEW_STATE_FIRST_LOC) ||
            (un8Aseq == WEATHER1_ASEQ_NEW_STATE_NEW_LOC))
        {
            //Reading state
            bSuccess = bReadUN8(WEATHER1_STATE_BITLEN, ptBitsOffset, 
                tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bSuccess == FALSE)
            {
                break;
            }

            tStateID = un8Buffer;
        }

        if ((un8Aseq == WEATHER1_ASEQ_NEW_STATE_NEW_LOC) ||
            (un8Aseq == WEATHER1_ASEQ_SAME_STATE_NEW_LOC))
        {
            bSuccess = bReadUN8(WEATHER1_LOCID_BITLEN, ptBitsOffset, 
                tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bSuccess == FALSE)
            {
                break;
            }

            tAreaID = un8Buffer;
        }

        if (un8Aseq == WEATHER1_ASEQ_SAME_STATE_NEXT_LOC)
        {
            //just incrementing previous location
            tAreaID = GET_AREA_ID_FROM_LOC_ID(tPrevID) + 1;
            tStateID = GET_STATE_ID_FROM_LOC_ID(tPrevID);
        }

        if (un8Aseq == WEATHER1_ASEQ_NEW_STATE_FIRST_LOC)
        {
            //setting loc to 1 per protocol spec (this ASEQ value says that
            // we start from 1st location in this state)
            tAreaID = 1;
        }

        if (un8Aseq == WEATHER1_ASEQ_SAME_STATE_NEW_LOC)
        {
            tStateID = GET_STATE_ID_FROM_LOC_ID(tPrevID);
        }

        tResult = GET_LOC_ID_FROM_STATE_AREA(tStateID, tAreaID);
    } while (FALSE);

    return tResult;
}

/*****************************************************************************
*
*   vCancelHashItems
*
*****************************************************************************/
static void vCancelHashItems(
    WEATHER1_OBJECT_STRUCT *psObj,
    UN32 un32CurrentTime
        )
{
    UN8 i = 0;

    for(i = 0; i < WEATHER_FCST_TYPE_MAX; i++)
    {
        if (psObj->asCacheItem[i].un32TimestampUTCsec != 0)
        {
            if ((un32CurrentTime - psObj->asCacheItem[i].un32TimestampUTCsec) >
                WEATHER_MSG_TIMEOUT)
            {
                if (i == WEATHER_FCST_TYPE_SKI)
                {
                    GsWeatherMgr.bSkiHashRemoved(psObj->hWeatherService,
                        &psObj->asCacheItem[i]
                            );
                }
                else
                {
                    GsWeatherMgr.bForecastHashRemoved(psObj->hWeatherService,
                        &psObj->asCacheItem[i]
                            );
                }

                psObj->asCacheItem[i].un32TimestampUTCsec = 0;
            }
        }
    }

    return;
}


/*****************************************************************************
*
*   bProcessHash
*
******************************************************************************/
static BOOLEAN bProcessHash(
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN bSkiFlag,
    WEATHER_FCST_TYPE_ENUM *peFcst
        )
{
    UN8 un8Aseq = 0;
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;
    UN8 un8MsgType = 0;
    UN8 un8StateID = 0;
    UN8 un8LocID = 0;
    size_t tBitsOffset = WEATHER1_PVN_BITLEN + WEATHER1_CAROUSEL_BITLEN;

    do
    {
        psObj->sHash.tAUSize = OSAL.tBufferGetSize(hPayload);

        if (bSkiFlag != TRUE)
        {
            // Read at the FCST
            tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8MsgType, 0,
                WEATHER1_FCST_BITLEN, tBitsOffset
                    );
            *peFcst = (WEATHER_FCST_TYPE_ENUM)un8MsgType;
            if ((tBitsRead != WEATHER1_FCST_BITLEN) ||
                    (*peFcst > WEATHER1_FCST_MAX))
            {
                // Read failed -- the message is probably
                // garbled, although that is unlikely since
                // the CRC checked out. Don't do anything rash here.
                printf(WEATHER1_OBJECT_NAME": Unable to read FCST\n");
                break;
            }

            tBitsOffset += tBitsRead;
        }
        else
        {
            *peFcst = WEATHER_FCST_TYPE_SKI;
        }

        // Getting ASEQ field
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8Aseq, 0,
            WEATHER1_ASEQ_BITLEN, tBitsOffset
                );
        if ((tBitsRead != WEATHER1_ASEQ_BITLEN) ||
            ((un8Aseq != WEATHER1_ASEQ_NEW_STATE_FIRST_LOC) &&
            (un8Aseq != WEATHER1_ASEQ_NEW_STATE_NEW_LOC)))
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            printf(WEATHER1_OBJECT_NAME": Unable to read ASEQ\n");
            break;
        }

        tBitsOffset += tBitsRead;

        // Getting State ID
        tBitsRead = OSAL.tBufferPeekBits(hPayload,
            &un8StateID,
            0,
            WEATHER1_STATE_BITLEN,
            tBitsOffset
                );

        if (tBitsRead != WEATHER1_STATE_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            printf(WEATHER1_OBJECT_NAME": Unable to read State ID\n");
            break;
        }
        else
        {
            printf(WEATHER1_OBJECT_NAME": State ID is %u\n", un8StateID);
        }

        tBitsOffset += tBitsRead;

        psObj->sHash.un8FirstStateID = un8StateID;

        // Checking if we have location in message
        // Otherwise we will keep loc id as 1
        if (un8Aseq == WEATHER1_ASEQ_NEW_STATE_NEW_LOC)
        {
            // Getting LOC ID
            tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8LocID, 0,
                WEATHER1_LOCID_BITLEN, tBitsOffset
                    );
            if (tBitsRead != WEATHER1_LOCID_BITLEN)
            {
                // Read failed -- the message is probably
                // garbled, although that is unlikely since
                // the CRC checked out. Don't do anything rash here.
                printf(WEATHER1_OBJECT_NAME": Unable to read Loc ID\n");
                break;
            }
            psObj->sHash.un8FirstAreaID = un8LocID;
        }
        else
        {
            psObj->sHash.un8FirstAreaID = 1;
        }

        printf(WEATHER1_OBJECT_NAME": LOC ID is %u\n",
            psObj->sHash.un8FirstAreaID
                );

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*    bGetMessageType
*
*****************************************************************************/
static BOOLEAN bGetMessageType(
        WEATHER1_OBJECT_STRUCT *psObj,
        OSAL_BUFFER_HDL hPayload,
        WEATHER1_MSG_TYPE_ENUM *peMsgType
            )
{
    UN8 un8CarouselID = 0;
    PVN tPVN = (PVN)0;
    size_t tBitsRead;
    BOOLEAN bSuccess = FALSE, bTemp = FALSE;

    do {
        // Peek at the PVN
        tBitsRead = OSAL.tBufferPeekBits(
                hPayload, &tPVN, 0,
                WEATHER1_PVN_BITLEN, 0);

        if (tBitsRead != WEATHER1_PVN_BITLEN)
        {
            // Peek failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            printf(WEATHER1_OBJECT_NAME":  Unable to read PVN\n");
            break;
        }

        // Verify the PVN
        if (tPVN != WEATHER1_PVN)
        {
            printf(WEATHER1_OBJECT_NAME": Incorrect PVN - got %u, expected %u\n",
                    tPVN, WEATHER1_PVN);
            break;
        }

        // Peek at the Carousel Id
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8CarouselID, 0,
                WEATHER1_CAROUSEL_BITLEN, WEATHER1_PVN_BITLEN);

        if (tBitsRead != WEATHER1_CAROUSEL_BITLEN)
        {
            // Peek failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            printf(WEATHER1_OBJECT_NAME":  Unable to read Carousel Id\n");
            break;
        }

        // determine message type
        switch (un8CarouselID)
        {
        case WEATHER1_LOCATION_UPDATE_CAROUSEL_ID:
        {
            *peMsgType = WEATHER_LOCATION_DB_UPDATE_MSG;
        }
        break;
        case WEATHER1_METADATA_UPDATE_CAROUSEL_ID:
        {
            *peMsgType = WEATHER_METADATA_UPDATE_MSG;
        }
        break;
        case WEATHER1_REPORT_CAROUSEL_ID:
        {
            bTemp = DS_UTIL_bIsCRCValid(psObj->hCRC, hPayload, &psObj->sHash.tCRC);
            if (bTemp == TRUE)
            {
                *peMsgType = WEATHER_FORECAST_REPORT_MSG;
            }

            bTemp = DS_UTIL_bCutCRC(hPayload);
            if (bTemp == FALSE)
            {
                printf(WEATHER1_OBJECT_NAME": Failed to cut off CRC (%d)\n",
                    WEATHER1_REPORT_CAROUSEL_ID);
            }
        }
        break;

        case WEATHER1_SKI_CAROUSEL_ID:
        {
            bTemp = DS_UTIL_bIsCRCValid(psObj->hCRC, hPayload, &psObj->sHash.tCRC);
            if (bTemp == TRUE)
            {
                *peMsgType = WEATHER_SKI_REPORT_MSG;
            }

            bTemp = DS_UTIL_bCutCRC(hPayload);
            if (bTemp == FALSE)
            {
                printf(WEATHER1_OBJECT_NAME": Failed to cut off CRC (%d)\n",
                    WEATHER1_SKI_CAROUSEL_ID);
            }
        }
        break;

        default:
        {
            *peMsgType = WEATHER_INVALID_MSG;
            printf(WEATHER1_OBJECT_NAME": Unknown Car ID: %u\n", un8CarouselID);
        }
        break;
        }

        if (*peMsgType == WEATHER_INVALID_MSG)
        {
            bSuccess = FALSE;
        }
        else
        {
            bSuccess = TRUE;
        }
    } while (FALSE);

    return bSuccess;
}

/*******************************************************************************
*
*        bProcessLocationDelete
*
*******************************************************************************/
static BOOLEAN bProcessLocationDelete(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager
        )
{
    BOOLEAN bResult;

    do
    {
        bResult = bReadLocId(psRFDCtrl);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read loc id");
            break;
        }

        if (bInformManager == TRUE)
        {
            bResult = GsWeatherMgr.bDeleteLocation(
                psRFDCtrl->hDBConnection, &psRFDCtrl->acBuffer[0],
                WEATHER_MAX_SQL_STRING_LENGTH, &psRFDCtrl->sLocation);
        }

    } while (FALSE);

    vDestroyLocData(&psRFDCtrl->sLocation);

    return bResult;
}

/*******************************************************************************
*
*        bProcessLocationNew
*
*******************************************************************************/
static BOOLEAN bProcessLocationNew(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager
        )
{
    BOOLEAN bResult;

    do
    {
        bResult = bReadLocId(psRFDCtrl);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read loc id");
            break;
        }

        bResult = bReadLocInfo(psRFDCtrl);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read loc info");
            break;
        }

        if (bInformManager == TRUE)
        {
            bResult = GsWeatherMgr.bAddLocation(
                psRFDCtrl->hDBConnection,&psRFDCtrl->acBuffer[0],
                WEATHER_MAX_SQL_STRING_LENGTH, &psRFDCtrl->sLocation);
        }

    } while (FALSE);

    vDestroyLocData(&psRFDCtrl->sLocation);

    return bResult;
}

/*******************************************************************************
*
*        bProcessLocationModify
*
*******************************************************************************/
static BOOLEAN bProcessLocationModify(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager
        )
{
    BOOLEAN bResult;

    do
    {
        bResult = bReadLocId(psRFDCtrl);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read loc id"
                    );
            break;
        }

        bResult = bReadLocInfo(psRFDCtrl);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to read loc info"
                    );
            break;
        }

        if (bInformManager == TRUE)
        {
            bResult = GsWeatherMgr.bModifyLocation(
                psRFDCtrl->hDBConnection, &psRFDCtrl->acBuffer[0],
                WEATHER_MAX_SQL_STRING_LENGTH, &psRFDCtrl->sLocation);
        }

    } while (FALSE);

    vDestroyLocData(&psRFDCtrl->sLocation);

    return bResult;
}

/*****************************************************************************
*
*   bReadLocId
*
*****************************************************************************/
static BOOLEAN bReadLocId(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bResult = FALSE;
    UN8 un8Buffer = 0;
    size_t tBitsRead = 0;

    do {

        // Read state id
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8Buffer, 0,
            WEATHER1_LOCATION_STATE_BITLEN);

        if (tBitsRead != WEATHER1_LOCATION_STATE_BITLEN)
        {
            break;
        }

        psRFDCtrl->sLocation.tID = (((LOC_ID)un8Buffer) << 8);

        // Read loc id
        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8Buffer, 0,
            WEATHER1_LOCATION_LOCID_BITLEN);
        if (tBitsRead != WEATHER1_LOCATION_LOCID_BITLEN)
        {
            break;
        }

        psRFDCtrl->sLocation.tID |= un8Buffer;

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bReadLocInfo
*
*****************************************************************************/
static BOOLEAN bReadLocInfo(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bResult = FALSE, bTemp = FALSE;
    UN8 un8Buffer = 0;
    size_t tBitsRead = 0;
    UN32 un32Buffer = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WEATHER_LOCATION_ROW_STRUCT *psLoc = &psRFDCtrl->sLocation;

    do {
        // Read ski flag
        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8Buffer, 0,
            WEATHER1_LOCATION_SKI_BITLEN);
        if (tBitsRead != WEATHER1_LOCATION_SKI_BITLEN)
        {
            break;
        }

        psLoc->bSkiFlag = (BOOLEAN)un8Buffer;

        // Read loclat
        un32Buffer = 0;
        bTemp = OSAL.bBufferReadBitsToUN32(
            psRFDCtrl->hBuffer,
            &un32Buffer,
            WEATHER1_LOCATION_LOCLAT_BITLEN);

        if (bTemp == FALSE)
        {
            break;
        }

        psLoc->hLat =
            OSAL_FIXED.hCreateInMemory((N32)un32Buffer, 0,
                psLoc->atLatFixedData);

        if (psLoc->hLat == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eDivide(
            psLoc->hLat, psRFDCtrl->hDegDivsor, psLoc->hLat);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Read loclon
        un32Buffer = 0;
        bTemp = OSAL.bBufferReadBitsToUN32(
            psRFDCtrl->hBuffer,
            &un32Buffer,
            WEATHER1_LOCATION_LOCLON_BITLEN);
        if (bTemp == FALSE)
        {
            break;
        }

        psLoc->hLon =
            OSAL_FIXED.hCreateInMemory((N32)un32Buffer, 0,
                psLoc->atLonFixedData);

        if (psLoc->hLon == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eDivide(
            psLoc->hLon, psRFDCtrl->hDegDivsor, psLoc->hLon);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eSubtract(
            psRFDCtrl->hLonBase, psLoc->hLon, psLoc->hLon);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        psLoc->hAreaName =
            BAUDOT_hToString(
                SMS_INVALID_OBJECT,
                psRFDCtrl->hBuffer,
                BAUDOT_BEHAVIOR_PROCESS_TO_END,
                TRUE, TRUE,
                WEATHER1_LOCATION_LNAME_NUM_SYMBOLS, NULL);

        if (psLoc->hAreaName == STRING_INVALID_OBJECT)
        {
            break;
        }

        psLoc->hICAOName =
            BAUDOT_hToString(
                SMS_INVALID_OBJECT,
                psRFDCtrl->hBuffer,
                BAUDOT_BEHAVIOR_PROCESS_TO_END,
                TRUE, TRUE,
                WEATHER1_LOCATION_LCODE_NUM_SYMBOLS, NULL);

        if (psLoc->hICAOName == STRING_INVALID_OBJECT)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   vDestroyLocData
*
*****************************************************************************/
static void vDestroyLocData(
    WEATHER_LOCATION_ROW_STRUCT *psLoc
        )
{
    // Free any strings created
    if (psLoc->hAreaName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psLoc->hAreaName);
        psLoc->hAreaName = STRING_INVALID_OBJECT;
    }

    // Free any strings created
    if (psLoc->hICAOName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psLoc->hICAOName);
        psLoc->hICAOName = STRING_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bParseForecastData
*
*****************************************************************************/
static BOOLEAN bParseForecastData (
    UN8 *pun8Source,
    FORECAST_DATA_STRUCT *psData,
    size_t tBufferSizeInBits,
    size_t *ptBitsOffset
        )
{
    UN8 un8Buffer;
    BOOLEAN bResult = FALSE;

    do
    {
        // Getting event code
        bResult = bReadUN8(WEATHER1_FORECAST_EVENT_BITLEN, ptBitsOffset, 
            tBufferSizeInBits, pun8Source, &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }
        psData->tEvent = un8Buffer;

        // Getting current temperature flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading current
        // temperature
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_TEMP_BITLEN, ptBitsOffset,
                tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->n16Temp = (N16)((N8)un8Buffer);
            psData->un16Flags |= (1 << FORECAST_AVERAGE_TEMP_FLAG);
        }

        //Getting maximum temperature flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading maximum
        // temperature
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_TMAX_BITLEN, ptBitsOffset,
                tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->n16Tmax = (N16)((N8)un8Buffer);
            psData->un16Flags |= (1 << FORECAST_HIGH_TEMP_FLAG);
        }

        //Getting minimum temperature flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading minimum
        // temperature
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_TMIN_BITLEN, ptBitsOffset,
                tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->n16Tmin = (N16)((N8)un8Buffer);
            psData->un16Flags |= (1 << FORECAST_LOW_TEMP_FLAG);
        }

        //Getting precipitation chance flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading precipitation
        // chance
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_POP_BITLEN, ptBitsOffset,
                tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->un8Pop = gaun8ChanceOfPrecipitation[un8Buffer];
            psData->un16Flags |= (1 << FORECAST_PRECIP_CHANCE_FLAG);
        }

        //Getting precipitation flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading precipitation
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_PRECIP_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->tPrecip = un8Buffer;
            psData->un16Flags |= (1 << FORECAST_PRECIP_AMOUNT_FLAG);
        }

        //Getting wind flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading wind value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_WIND_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->tData = un8Buffer;
            psData->un16Flags |= (1 << FORECAST_WIND_SPEED_FLAG);
            psData->un16Flags |= (1 << FORECAST_WIND_DIRECTION_FLAG);
        }

        //Getting humidity flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading humidity value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_HUMID_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->tHumid = un8Buffer;
            psData->un16Flags |= (1 << FORECAST_HUMIDITY_FLAG);
        }

        //Getting cloud cover flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading cloud cover
        // value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_CLOUD_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            // Make sure to map the cloud cover values correctly
            if ((un8Buffer == WEATHER1_CLOUDS_RSVD1) ||
                (un8Buffer == WEATHER1_CLOUDS_RSVD2) ||
                (un8Buffer == WEATHER1_CLOUDS_RSVD3))
            {
                // All of these are reported as the same enumerated value
                un8Buffer = (UN8)FORECAST_CLOUD_COVER_NO_DATA;
            }

            psData->eCloudCover = (FORECAST_CLOUD_COVER_ENUM)un8Buffer;
            psData->un16Flags |= (1 << FORECAST_CLOUD_COVER_FLAG);
        }

        //Getting UV index flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading UV index
        // value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_UV_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->tUV = un8Buffer;
            psData->un16Flags |= (1 << FORECAST_UV_INDEX_FLAG);
        }

        //Getting air quality flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading air quality
        // value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_AIR_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            // Make sure to map the air quality values correctly
            if ((un8Buffer == WEATHER1_AIRQ_RESERVED) ||
                (un8Buffer == WEATHER1_AIRQ_NA))
            {
                // Both of these are reported as the same enumerated value
                un8Buffer = (UN8)AIR_QUALITY_NO_DATA;
            }

            psData->eAirQuality = (AIR_QUALITY_ENUM)un8Buffer;
            psData->un16Flags |= (1 << FORECAST_AIR_QUALITY_FLAG);
        }

        //Getting pollen count flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading pollen
        // count value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_POLLEN_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            psData->tPollen = un8Buffer;
            psData->un16Flags |= (1 << FORECAST_POLLEN_COUNT_FLAG);
        }

        //Getting extended data count flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading ext
        // data count value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_FORECAST_EXTCNT_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            // calculating the offset past ext data
            *ptBitsOffset +=
                ((un8Buffer + 1) * WEATHER1_FORECAST_EXTDATA_ITEM_BITLEN);
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bParseSkiData
*
*****************************************************************************/
static BOOLEAN bParseSkiData(
    UN8 *pun8Source,
    SKI_DATA_STRUCT *psData,
    size_t tBufferSizeInBits,
    size_t *ptBitsOffset
        )
{
    UN8 un8Buffer;
    BOOLEAN bResult = FALSE;

    do
    {
        //Getting operational status flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading
        // operational status
        if (un8Buffer)
        {
            bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits,
                pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            if (un8Buffer)
            {
                psData->eOpStat = WEATHER_FLAG_TRUE;
            }
            else
            {
                psData->eOpStat = WEATHER_FLAG_FALSE;
            }
        }
        else
        {
            psData->eOpStat = WEATHER_FLAG_NOT_AVAILABLE;
        }

        //Getting event flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading event
        // data
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_EVENT_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->tEvent = un8Buffer;
            psData->un16Flags |= (1 << SKI_EVENT_CODE_FLAG);
        }

        //Getting current temperature flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading current
        // temperature
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_TEMP_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->n16Temp = (N16)((N8)un8Buffer);
            psData->un16Flags |= (1 << SKI_CURRENT_TEMP_FLAG);
        }

        //Getting wind conditions flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading wind
        // conditions
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_SWIND_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->eWind = (SKI_WIND_CONDITION_ENUM)un8Buffer;
            psData->un16Flags |= (1 << SKI_WIND_COND_FLAG);
        }

        //Getting snow conditions flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading snow
        // conditions
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_SNOW_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->eSnow = (SKI_SNOW_CONDITIONS_ENUM)un8Buffer;
            psData->un16Flags |= (1 << SKI_SNOW_COND_FLAG);
        }


        //Getting minimum base depth flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading min
        // base depth
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_BMIN_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->un8BMin = un8Buffer;
            psData->un16Flags |= (1 << SKI_MIN_BASE_DEPTH_FLAG);
        }


        //Getting max base depth flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading max
        // base depth
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_BMAX_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->un8BMax = un8Buffer;
            psData->un16Flags |= (1 << SKI_MAX_BASE_DEPTH_FLAG);
        }


        //Getting new snow flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading new
        // snow value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_NEW_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->tNew = un8Buffer;
            psData->un16Flags |= (1 << SKI_NEW_SNOW_FLAG);
        }


        //Getting num of lifts flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading num
        // of lifts value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_LIFT_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->un8Lift = un8Buffer;
            psData->un16Flags |= (1 << SKI_NUM_OF_LIFTS_FLAG);
        }


        //Getting num of trails flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading num
        // of trails value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_TRAIL_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            psData->tTrail = un8Buffer;
            psData->un16Flags |= (1 << SKI_NUM_OF_TRAILS_FLAG);
        }

        //Getting snowmaking flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading
        // snowmaking value
        if (un8Buffer)
        {
            bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits,
                pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            if (un8Buffer)
            {
                psData->eMake = WEATHER_FLAG_TRUE;
            }
            else
            {
                psData->eMake = WEATHER_FLAG_FALSE;
            }
        }
        else
        {
            psData->eMake = WEATHER_FLAG_NOT_AVAILABLE;
        }


        //Getting grooming flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading
        // grooming value
        if (un8Buffer)
        {
            bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, 
                pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            if (un8Buffer)
            {
                psData->eGroom = WEATHER_FLAG_TRUE;
            }
            else
            {
                psData->eGroom = WEATHER_FLAG_FALSE;
            }
        }
        else
        {
            psData->eGroom = WEATHER_FLAG_NOT_AVAILABLE;
        }


        //Getting night skiing flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading
        // night skiing value
        if (un8Buffer)
        {
            bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits,
                pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            if (un8Buffer)
            {
                psData->eNight = WEATHER_FLAG_TRUE;
            }
            else
            {
                psData->eNight = WEATHER_FLAG_FALSE;
            }
        }
        else
        {
            psData->eNight = WEATHER_FLAG_NOT_AVAILABLE;
        }


        //Getting snowboarding flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits, pun8Source,
            &un8Buffer);
        if (bResult == FALSE)
        {
            break;
        }

        // checking flag, setting flag in our table and reading
        // snowboarding value
        if (un8Buffer)
        {
            bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits,
                pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }

            if (un8Buffer)
            {
                psData->eBoard = WEATHER_FLAG_TRUE;
            }
            else
            {
                psData->eBoard = WEATHER_FLAG_FALSE;
            }
        }
        else
        {
            psData->eBoard = WEATHER_FLAG_NOT_AVAILABLE;
        }

        //Getting extended data count flag
        bResult = bReadFlag(ptBitsOffset, tBufferSizeInBits,
            pun8Source, &un8Buffer);

        // checking flag, setting flag in our table and reading ext
        // data count value
        if (un8Buffer)
        {
            bResult = bReadUN8(WEATHER1_SKI_EXTCNT_BITLEN,
                ptBitsOffset, tBufferSizeInBits, pun8Source, &un8Buffer);
            if (bResult == FALSE)
            {
                break;
            }
            *ptBitsOffset += 
                (un8Buffer + 1) * WEATHER1_SKI_EXTDATA_ITEM_BITLEN;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bInitRFD
*
*****************************************************************************/
static BOOLEAN bInitRFD(
    WEATHER1_OBJECT_STRUCT *psObj,
    RFD_UPDATE_VERSION tCurrentVersion
        )
{
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl = NULL;

    do
    {

        // Create the RFD processor object
        psRFDCtrl = psCreateRFDCtrl(psObj);
        if (psRFDCtrl == NULL)
        {
            break;
        }

        // Connect to RFD
        psObj->hRFD = RFD_INTERFACE_hConnect (
            WEATHER1_RFD_CLIENT_ID, tCurrentVersion,
            WEATHER1_MAX_VERSION_BITLEN,
            eRFDFileProcessor, NULL, psRFDCtrl);
        if (psObj->hRFD == RFD_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        return TRUE;
    } while (FALSE);

    vDestroyRFDCtrl(psRFDCtrl);
    return FALSE;
}

/*****************************************************************************
*
*   psCreateRFDCtrl
*
*****************************************************************************/
static WEATHER1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    WEATHER1_OBJECT_STRUCT *psObj
        )
{
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl;

    // Create the RFD processor object
    psRFDCtrl = (WEATHER1_RFD_CTRL_STRUCT *)
        SMSO_hCreate(
            WEATHER1_OBJECT_NAME":RFDCtrl",
            sizeof(WEATHER1_RFD_CTRL_STRUCT),
            SMS_INVALID_OBJECT, FALSE);
    if (psRFDCtrl == NULL)
    {
        return NULL;
    }

    // Populate the structure with the minimum required
    psRFDCtrl->hWeatherService = psObj->hWeatherService;

    return psRFDCtrl;
}

/*****************************************************************************
*
*   bInitRFDCtrl
*
*****************************************************************************/
static BOOLEAN bInitRFDCtrl (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        )
{
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hCurDB = STRING_INVALID_OBJECT;

    do
    {
        // Store the RFD connection handle
        psRFDCtrl->hRFD = hRFD;

        // Initialize the structure
        psRFDCtrl->tStartProgressIndex = tStartingIndex;
        psRFDCtrl->tCurProgressIndex = 0;

        // Create FIXED constants that are used to work
        // with location updates
        psRFDCtrl->hDegDivsor =
            OSAL_FIXED.hCreateFromFixed(
                WEATHER1_LAT_LON_DIVSOR_VALUE, 0);
        if (psRFDCtrl->hDegDivsor == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        psRFDCtrl->hLonBase =
            OSAL_FIXED.hCreateFromFixed(
                WEATHER1_LON_BASE_VALUE, 0);
        if (psRFDCtrl->hLonBase == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Create our file reader buffer
        bSuccess = DATASERVICE_MGR_bCreateFileBuffer(
                &psRFDCtrl->hBlockPool,
                &psRFDCtrl->hBuffer,
                WEATHER_RFD_READ_BLOCK_SIZE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WEATHER1_OBJECT_NAME": failed to create file reader buffer");
            break;
        }

        // Find out where to place our working copy
        bSuccess = GsWeatherMgr.bRefDBBank(
            psRFDCtrl->hWeatherService, &hCurDB, &psRFDCtrl->hDBPath);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Are we starting a new update?
        if (tStartingIndex == 0)
        {
            // Yes!  Get a copy of the db
            bSuccess = DB_UTIL_bCopyDB(
                STRING.pacCStr(psRFDCtrl->hDBPath),
                STRING.pacCStr(hCurDB));
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Connect to our working database file now
        bSuccess = bConnectToDB(psRFDCtrl, FALSE);
        if (bSuccess == FALSE)
        {
            break;
        }

        if (tStartingIndex == 0)
        {
            bSuccess = GsWeatherMgr.bDBUpdateBegin(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->acBuffer,
                WEATHER_MAX_SQL_STRING_LENGTH);

            if (bSuccess == FALSE)
            {
                break;
            }
        }

    } while (FALSE);

    if (hCurDB != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hCurDB);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vDestroyRFDCtrl
*
*****************************************************************************/
static void vDestroyRFDCtrl (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl == NULL)
    {
        return;
    }

    psRFDCtrl->hWeatherService = WEATHER_SERVICE_INVALID_OBJECT;
    psRFDCtrl->hRFD = RFD_INTERFACE_INVALID_OBJECT;

    // Disconnect from the database
    if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psRFDCtrl->hDBPath);
        psRFDCtrl->hDBPath = STRING_INVALID_OBJECT;
    }

    psRFDCtrl->tStartProgressIndex = 0;
    psRFDCtrl->tCurProgressIndex = 0;

    if (psRFDCtrl->hDegDivsor != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psRFDCtrl->hDegDivsor);
        psRFDCtrl->hDegDivsor = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psRFDCtrl->hLonBase != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psRFDCtrl->hLonBase);
        psRFDCtrl->hLonBase = OSAL_FIXED_INVALID_OBJECT;
    }

    // Destroy the file buffer
    DATASERVICE_MGR_vDestroyFileBuffer(psRFDCtrl->hBlockPool, psRFDCtrl->hBuffer);
    psRFDCtrl->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    psRFDCtrl->hBuffer = OSAL_INVALID_BUFFER_HDL;

    SMSO_vDestroy((SMS_OBJECT)psRFDCtrl);

    return;
}

/*****************************************************************************
*
*   bConnectToDB
*
*****************************************************************************/
static BOOLEAN bConnectToDB (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        )
{
    SQL_INTERFACE_OPTIONS_MASK tOptions = SQL_INTERFACE_OPTIONS_NONE;
    BOOLEAN bConnected = FALSE;

    if (psRFDCtrl->hDBPath != NULL)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);

        if (bPerformIntegrityCheck == FALSE)
        {
            tOptions |= SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK;
        }

        // Connect to the database
        psRFDCtrl->hDBConnection =
            SQL_INTERFACE.hConnect(
                pcDBPath, tOptions,
                (DATASERVICE_ERROR_CODE_ENUM *)NULL);

        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            bConnected = TRUE;
        }
    }

    return bConnected;
}

/*****************************************************************************
*
*   bFinalizeDB
*
*****************************************************************************/
static BOOLEAN bFinalizeDB (
   WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bDBReady;

    bDBReady = bConnectToDB(psRFDCtrl, TRUE);
    if (bDBReady == TRUE)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    return bDBReady;
}

/*****************************************************************************
*
*   vDeleteDB
*
*****************************************************************************/
static void vDeleteDB (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);
        remove(pcDBPath);
    }
    return;
}

/*****************************************************************************
*
*   bReadUN8
*
*****************************************************************************/
static BOOLEAN bReadUN8(
    UN8 un8Length,
    size_t *ptPosition,
    size_t tSourceBitLength,
    UN8 *pun8Source,
    UN8 *pun8Dest
        )
{
    size_t tByteLength, tStartByteIndex = 0;

    if ((un8Length + *ptPosition) > tSourceBitLength)
    {
        return FALSE;
    }

    tByteLength = (*ptPosition + un8Length - 1) >> 3; // instead of /UN8_BITLEN

    if (*ptPosition != 0)
    {
        tStartByteIndex = *ptPosition >> 3; // instead of /UN8_BITLEN
    }

    if (tByteLength == tStartByteIndex)
    {
        // if we have all the bits in single byte
        // Here are three operations:
        // 1. Shift the byte to remove bits which are not needed
        // (determined by position % UN8_BITLEN changed to position & 7
        // to make it faster)
        // 2. Remove unneeded bits by 0xFF mask.
        // 3. Shift bits right to remove unneeded bits from right.
        *pun8Dest = (0xFF & ((pun8Source[tStartByteIndex]) << 
            (*ptPosition & 7))) >> (UN8_BITLEN - un8Length);
    }
    else
    {
        // if we have bits split in two bytes
        // Here are three operations:
        // 1. Shift the fist byte to remove bits which are not needed
        // (determined by position % UN8_BITLEN changed to position & 7
        // to make it faster)
        // 2. Remove unneeded bits by 0xFF mask.
        // 3. Shift bits right to remove unneeded bits from right.
        // 4. Take next byte from source and make right shift to number of 
        // left bits.
        // 5. Combine these bytes together using logical OR.
        *pun8Dest = ((0xFF & ((pun8Source[tStartByteIndex]) << 
            (*ptPosition & 7))) >> (UN8_BITLEN - un8Length)) | 
            (pun8Source[tStartByteIndex + 1] >> (UN8_BITLEN - 
            (((*ptPosition) + un8Length) & 7)));
    }

    (*ptPosition) += un8Length;

    return TRUE;
}

/*****************************************************************************
*
*   bReadFlag
*
*****************************************************************************/
static BOOLEAN bReadFlag(
    size_t *ptPosition,
    size_t tSourceBitLength,
    UN8 *pun8Source,
    UN8 *pun8Dest
        )
{
    size_t tIndex = 0;

    if (*ptPosition >= tSourceBitLength)
    {
        return FALSE;
    }

    if (*ptPosition != 0)
    {
        tIndex = *ptPosition >> 3; // instead of /UN8_BITLEN
    }

    // To take a single bit, we use left shift to make required bit most
    // left in the byte, then apply the mask 0x80 to remove all other bits
    // It requires less operations than right shift.
    *pun8Dest = ((pun8Source[tIndex]) << (*ptPosition & 7)) & 0x80;
    (*ptPosition)++;

    return TRUE;
}


