/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Traffic Cameras protocol version 1 implementation
 *  for the Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "string.h"
#include "dataservice_mgr_impl.h"
#include "location_obj.h"
#include "db_util.h"
#include "ds_util.h"

#include "trafcam_interface.h"
#include "_trafcam_pvn1.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   hInit
*
*****************************************************************************/
static TRAFCAM_INTERFACE_OBJECT hInit(
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService,
    SMS_OBJECT hParent
        )
{
    TRAFCAM_INTERFACE_OBJECT hResult = TRAFCAM_INTERFACE_INVALID_OBJECT;
    TRAFCAM1_OBJECT_STRUCT *psObj = (TRAFCAM1_OBJECT_STRUCT *)NULL;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        BOOLEAN bOwner = FALSE, bValid = FALSE;
        UN8 un8Iterator = 0;

        bValid = DATASERVICE_IMPL_bIsValid(
            (DATASERVICE_IMPL_HDL)hTrafficCamerasService);
        if (bValid == FALSE)
        {
            break;
        }

        // Verify we own parent handle
        bOwner = SMSO_bOwner(hParent);
        if (bOwner == FALSE)
        {
            break;
        }

        // Create an instance of this object
        psObj = (TRAFCAM1_OBJECT_STRUCT *)
            SMSO_hCreate(
                TRAFCAM1_OBJECT_NAME,
                sizeof(*psObj),
                hParent, FALSE);
        if (psObj == NULL)
        {
            break;
        }

        psObj->hTrafficCamerasService = hTrafficCamerasService;

        // Request access to ISO 3309 CRC32
        eReturnCode = OSAL.eGetCRC(&psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Cannot create CRC object (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Creating list which keeps groups data and filter
        eReturnCode = OSAL.eLinkedListCreate(&psObj->hMessagesData,
            TRAFCAM1_OBJECT_NAME":Messages",
            (OSAL_LL_COMPARE_HANDLER)n16CompareMarkets,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Cannot create messages list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Creating block pool for CRC calculation
        eReturnCode = OSAL.eBlockPoolCreate(&psObj->hCRCCalcBlockPool,
            TRAFCAM1_OBJECT_NAME":CRCPool", TRAFCAM1_CRC_CALC_BLOCK_SIZE,
            TRAFCAM1_CRC_CALC_BUFFER_SIZE / TRAFCAM1_CRC_CALC_BLOCK_SIZE,
            OSAL_BLOCK_POOL_OPTION_NONE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Cannot create block pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // initializing processing slots
        for (un8Iterator = 0; un8Iterator < TRAFCAM1_NUM_OF_DMIS; un8Iterator++)
        {
            psObj->asSlot[un8Iterator].tMarket = TRAFFIC_INVALID_MARKET;
        }

        hResult = (TRAFCAM_INTERFACE_OBJECT)psObj;
    } while (FALSE);

    if (hResult == TRAFCAM_INTERFACE_INVALID_OBJECT)
    {
        vUninitObject(psObj);
    }

    return hResult;
}

/*****************************************************************************
*
*   vUnInit
*
*****************************************************************************/
static void vUnInit(
    TRAFCAM_INTERFACE_OBJECT hInterface
        )
{
    TRAFCAM1_OBJECT_STRUCT *psObj = (TRAFCAM1_OBJECT_STRUCT *)hInterface;
    BOOLEAN bOwner = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    if (bOwner == TRUE)
    {
        vUninitObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   bProcessMessage
*
*****************************************************************************/
static BOOLEAN bProcessMessage(
    TRAFCAM_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bOwner = FALSE, bSuccess = FALSE;
        TRAFCAM1_OBJECT_STRUCT *psObj = (TRAFCAM1_OBJECT_STRUCT *)hInterface;
        TRAFCAM1_MARKET_DATA_STRUCT *psMarketData = NULL;
        TRAFCAM1_BSA_DATA_STRUCT *psBSAData = NULL;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        if ((phPayload == NULL) || (*phPayload == OSAL_INVALID_BUFFER_HDL))
        {
            break;
        }

        // check packet header data
        bSuccess = bCheckPacketHeader(*phPayload);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Packet header check failed"
                    );
            break;
        }

        bSuccess = bGetMarketData(psObj->hMessagesData, *phPayload,
            &psMarketData, &psBSAData);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to get Market and BSA data."
                    );
            break;
        }
        else if ((psBSAData == NULL) || (psMarketData == NULL))
        {
            printf(TRAFCAM1_OBJECT_NAME": Packet has been filtered out\n");
            bResult = TRUE;
            break;
        }

        if (psMarketData->psSlot == NULL)
        {
            printf(TRAFCAM1_OBJECT_NAME": This market has no slot\n");
            bResult = TRUE;
            break;
        }

        // Packet seems suitable so check its CRC now
        bSuccess = DS_UTIL_bIsCRCValid(psObj->hCRC, *phPayload, NULL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Packet verification failed"
                    );
            break;
        }

        // Cut off CRC
        bSuccess = DS_UTIL_bCutCRC(*phPayload);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": failed to remove CRC"
                    );
            break;
        }

        // process packet data
        bSuccess = bParsePacketData(psObj, phPayload, psMarketData, psBSAData);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Packet data parsing failed"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bSetFilter
*
*****************************************************************************/
static BOOLEAN bSetFilter (
    TRAFCAM_INTERFACE_OBJECT hInterface,
    OSAL_OBJECT_HDL hFilter
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bOwner = FALSE;
        TRAFCAM1_OBJECT_STRUCT *psObj =
            (TRAFCAM1_OBJECT_STRUCT *)hInterface;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM1_FILTER_ITERATOR_STRUCT sIterator;

        OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));

        if ((hInterface == TRAFCAM_INTERFACE_INVALID_OBJECT) ||
            (hFilter == OSAL_INVALID_OBJECT_HDL))
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
        if (bOwner == FALSE)
        {
            break;
        }

        // initializing iterator to iterate pvn filter
        sIterator.bResult = TRUE;
        sIterator.hEntriesList = hFilter;

        // iterating our filter for removal using the input filter
        eReturnCode = OSAL.eLinkedListIterate(psObj->hMessagesData,
            (OSAL_LL_ITERATOR_HANDLER)bIterateToRemove, &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to iterate filter list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // if iteration has been failed inside, skip here
        if (sIterator.bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to generate filter items to"
                " remove."
                    );
            break;
        }

        // reinit iterator to iterate input filter
        sIterator.hEntriesList = psObj->hMessagesData;
        sIterator.psObj = psObj;
        sIterator.bResult = TRUE;

        // iterating input filter to find new market-BSA items and add them
        // to pvn object filter
        eReturnCode = OSAL.eLinkedListIterate(hFilter,
            (OSAL_LL_ITERATOR_HANDLER)bIterateToAdd, &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to iterate filter list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        if (sIterator.bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to generate filter items to add."
                    );
            break;
        }

        // updating DMIs according to updated filter data
        bResult = bUpdateDMIs(psObj);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bReportExpiredHash
*
*****************************************************************************/
static BOOLEAN bReportExpiredHash (
    TRAFCAM_INTERFACE_OBJECT hInterface,
    TRAFCAM_IMAGE_HASH tHash
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        bSuccess = SMSO_bOwner((SMS_OBJECT)hInterface);
        if (bSuccess == FALSE)
        {
            break;
        }

        if (tHash == TRAFCAM_IMAGE_INVALID_HASH)
        {
            break;
        }

        // removing the group from the list since it was expired
        eReturnCode = OSAL.eLinkedListRemove(
            ((TRAFCAM1_GROUP_STRUCT *)tHash)->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to remove the entry (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        OSAL.vLinkedListMemoryFree((TRAFCAM1_GROUP_STRUCT *)tHash);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bUpdateDMIs
*
*****************************************************************************/
static BOOLEAN bUpdateDMIs(
    TRAFCAM1_OBJECT_STRUCT *psObj
        )
{
    DATASERVICE_DMI_CONFIG_STRUCT asDMIConfig[TRAFCAM1_NUM_OF_DMIS];
    UN8 un8DMIIndex = 0, un8Iterator = 0;
    BOOLEAN bResult = TRUE;

    for (un8Iterator = 0; un8Iterator < TRAFCAM1_NUM_OF_DMIS; un8Iterator++)
    {
        if (psObj->asSlot[un8Iterator].bChanged == TRUE)
        {
            // ok, it was changed
            psObj->asSlot[un8Iterator].bChanged = FALSE;
            if (psObj->asSlot[un8Iterator].un8RefCount == 0)
            {
                // ref count is zero, means that we need to disable the slot
                asDMIConfig[un8DMIIndex].bEnable = FALSE;
                asDMIConfig[un8DMIIndex].tDMI = TRAFCAM1_FIRST_DMI +
                    un8Iterator;
            }
            else
            {
                // ref count is more than zero, need to enable the slot
                asDMIConfig[un8DMIIndex].bEnable = TRUE;
                asDMIConfig[un8DMIIndex].tDMI = TRAFCAM1_FIRST_DMI +
                    un8Iterator;
            }
            un8DMIIndex++;
        }
    }

    // change the settings only if our DMI configuration has been updated
    if (un8DMIIndex > 0)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

        eReturnCode = DATASERVICE_IMPL_eManageDataStream(
            (DATASERVICE_IMPL_HDL)psObj->hTrafficCamerasService, asDMIConfig,
            un8DMIIndex);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            bResult = FALSE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   vUnRegisterSlot
*
*****************************************************************************/
static void vUnRegisterSlot(
    TRAFCAM1_PROCESS_SLOT_STRUCT *psSlot,
    TRAFFIC_MARKET tMarket
        )
{
    if (psSlot != NULL)
    {
        // cleaning slot data for unregistered market
        bCleanSlot(psSlot, tMarket);

        if (psSlot->un8RefCount > 0)
        {
            psSlot->un8RefCount--;

            // if reference count has been decreased to 0,
            // need to notify about slot status change
            if (psSlot->un8RefCount == 0)
            {
                psSlot->bChanged = TRUE;
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   psRegisterMarket
*
*****************************************************************************/
static TRAFCAM1_PROCESS_SLOT_STRUCT *psRegisterMarket(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket
        )
{
    TRAFCAM1_PROCESS_SLOT_STRUCT *psResult = NULL;

    do
    {
        N8 n8SlotNumber = TRAFCAM1_INVALID_SLOT;

        if (tMarket > (sizeof(an8MarketToSlot)/sizeof(an8MarketToSlot[0]) - 1))
        {
            // no slot available for this market
            break;
        }

        n8SlotNumber = an8MarketToSlot[tMarket];
        if (n8SlotNumber == TRAFCAM1_INVALID_SLOT)
        {
            // no slot available for this market
            break;
        }

        // if slot has no clients yet, let's set its status flag to changed
        if (psObj->asSlot[n8SlotNumber].un8RefCount == 0)
        {
            if (psObj->asSlot[n8SlotNumber].bChanged == FALSE)
            {
                psObj->asSlot[n8SlotNumber].bChanged = TRUE;
            }
            else
            {
                psObj->asSlot[n8SlotNumber].bChanged = FALSE;
            }
        }

        // increasing reference count for this slot
        psObj->asSlot[n8SlotNumber].un8RefCount++;
        psResult = &psObj->asSlot[n8SlotNumber];

    } while (FALSE);

    return psResult;
}

/*****************************************************************************
*
*   bIterateBSAToRemove
*
*****************************************************************************/
static BOOLEAN bIterateBSAToRemove(
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_FILTER_BSA_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM_BSA_FILTER_STRUCT *psFilterItem = NULL;

        if ((psBSAData == NULL) || (psIteratorArg == NULL))
        {
            break;
        }

        psFilterItem = (TRAFCAM_BSA_FILTER_STRUCT *)
            OSAL.pvLinkedListThis(psIteratorArg->hEntry);

        if (psIteratorArg->tMarket != psFilterItem->tMarket)
        {
            // we've completed current market section
            break;
        }

        if (psBSAData->tBSA == psFilterItem->tBSA)
        {
            // this BSA exists in both filters, skipping to next item
            bContinue = TRUE;
            break;
        }

        if (psBSAData->tBSA < psFilterItem->tBSA)
        {
            // looks like this BSA doesn't exist in filter
            eReturnCode = OSAL.eLinkedListRemove(psBSAData->hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to remove BSA entry (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
            vReleaseBSAData(psBSAData);
            bContinue = TRUE;
            break;

        }

        // taking next item from input filter
        psIteratorArg->hEntry =
            OSAL.hLinkedListNext(psIteratorArg->hEntry, (void **)&psFilterItem);

        while ((psIteratorArg->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
            (psFilterItem != NULL))
        {
            // if market is changed, then need to break and take next item from
            // pvn filter
            if (psFilterItem->tMarket != psIteratorArg->tMarket)
            {
                break;
            }

            // items are sorted by BSA in increasing order, so, if BSA is less
            // that means BSA doesn't exist in input filter and should be
            // removed from pvn filter
            if (psBSAData->tBSA < psFilterItem->tBSA)
            {
                eReturnCode = OSAL.eLinkedListRemove(psBSAData->hEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFCAM1_OBJECT_NAME": Failed to remove BSA entry (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                    break;
                }
                vReleaseBSAData(psBSAData);
                bContinue = TRUE;
                break;
            }

            // if BSA exists in input filter, moving to next filter item
            if (psBSAData->tBSA == psFilterItem->tBSA)
            {
                bContinue = TRUE;
                break;
            }

            // taking next entry from input filter
            psIteratorArg->hEntry =
                OSAL.hLinkedListNext(psIteratorArg->hEntry,
                (void **)&psFilterItem);
        }

    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   n16FindMarket
*
*****************************************************************************/
static N16 n16FindMarket(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFFIC_MARKET tMarket
        )
{
    N16 n16Result = N16_MIN;

    if (psFilterItem != NULL)
    {
        n16Result = COMPARE(psFilterItem->tMarket, tMarket);
    }

    return n16Result;
}

/*****************************************************************************
*
*   bIterateToRemove
*
*****************************************************************************/
static BOOLEAN bIterateToRemove(
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_FILTER_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM1_FILTER_BSA_ITERATOR_STRUCT sIterator;

        if ((psMarketData == NULL) || (psIteratorArg == NULL))
        {
            break;
        }

        // looking for the market in input filter
        eReturnCode = OSAL.eLinkedListLinearSearch(psIteratorArg->hEntriesList, 
            &hEntry, (OSAL_LL_COMPARE_HANDLER)n16FindMarket,
            (void *)(size_t)psMarketData->tMarket);
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // need to remove the whole market
            eReturnCode = OSAL.eLinkedListRemove(psMarketData->hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to remove market entry (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
            vReleaseMarketData(psMarketData);
            bContinue = TRUE;
            break;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to search for market (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            sIterator.bResult = FALSE;
            break;
        }

        OSAL.bMemSet(&sIterator, 0, sizeof(sIterator));
        sIterator.hEntry = hEntry;
        sIterator.hEntriesList = psIteratorArg->hEntriesList;

        // now looking for BSA items
        eReturnCode = OSAL.eLinkedListIterate(psMarketData->hBSAData,
            (OSAL_LL_ITERATOR_HANDLER)bIterateBSAToRemove, &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to iterate BSA list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            psIteratorArg->bResult = FALSE;
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   n16CompareBSAData
*
*****************************************************************************/
static N16 n16CompareBSAData(
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData1,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData2
        )
{
    N16 n16Result = N16_MIN;

    if ((psBSAData1 != NULL) && (psBSAData2 != NULL))
    {
        n16Result = COMPARE(psBSAData1->tBSA, psBSAData2->tBSA);
    }

    return n16Result;
}

/*****************************************************************************
*
*   psCreateMarketData
*
*****************************************************************************/
static TRAFCAM1_MARKET_DATA_STRUCT *psCreateMarketData(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hFilter,
    TRAFFIC_MARKET tMarket
        )
{
    TRAFCAM1_MARKET_DATA_STRUCT *psResult = NULL;
    BOOLEAN bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // creating new market item
        psResult =
            (TRAFCAM1_MARKET_DATA_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
            TRAFCAM1_OBJECT_NAME":MarketData", sizeof(*psResult),
            TRUE);
        if (psResult == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to allocate market data"
                    );
            break;
        }

        // initializing market data
        psResult->tMarket = tMarket;

        // registering market and receiving processing slot pointer
        psResult->psSlot = psRegisterMarket(psObj, tMarket);

        // creating list of BSAs for this market
        eReturnCode = OSAL.eLinkedListCreate(&psResult->hBSAData,
            TRAFCAM1_OBJECT_NAME":BSAsList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareBSAData,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
               );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to add market data (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // adding our market to pvn filter
        eReturnCode =
            OSAL.eLinkedListAdd(hFilter, &psResult->hEntry,
            psResult);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to add market data (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bSuccess = TRUE;
    } while (FALSE);

    if (bSuccess == FALSE)
    {
        vReleaseMarketData(psResult);
        psResult = NULL;
    }

    return psResult;
}

/*****************************************************************************
*
*   psCreateBSAData
*
*****************************************************************************/
static TRAFCAM1_BSA_DATA_STRUCT *psCreateBSAData(
    OSAL_OBJECT_HDL hBSAList,
    TRAFFIC_BSA tBSA
        )
{
    TRAFCAM1_BSA_DATA_STRUCT *psResult = NULL;
    BOOLEAN bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        psResult =
            (TRAFCAM1_BSA_DATA_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
            TRAFCAM1_OBJECT_NAME":BSA_Data", sizeof(*psResult), TRUE);
        if (psResult == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to allocate BSA data"
                    );
            break;
        }

        // initializing BSA data
        psResult->tBSA = tBSA;

        // creating list of groups for this BSA
        eReturnCode = OSAL.eLinkedListCreate(&psResult->hGroups,
            TRAFCAM1_OBJECT_NAME":GroupsList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareSIGs,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to create groups list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // adding our BSA to list of BSAs
        eReturnCode = OSAL.eLinkedListAdd(hBSAList,
            &psResult->hEntry, psResult);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to add BSA data to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bSuccess = TRUE;
    } while (FALSE);

    if (bSuccess == FALSE)
    {
        vReleaseBSAData(psResult);
        psResult = NULL;
    }

    return psResult;
}

/*****************************************************************************
*
*   bIterateToAdd
*
*****************************************************************************/
static BOOLEAN bIterateToAdd (
    TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter,
    TRAFCAM1_FILTER_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        TRAFCAM1_MARKET_DATA_STRUCT *psMarketData = NULL;
        TRAFCAM1_BSA_DATA_STRUCT sTempBSAData;
        TRAFCAM1_MARKET_DATA_STRUCT sTempMarketData;

        OSAL.bMemSet(&sTempBSAData, 0, sizeof(sTempBSAData));
        OSAL.bMemSet(&sTempMarketData, 0, sizeof(sTempMarketData));

        if ((psBSAFilter == NULL) || (psIteratorArg == NULL))
        {
            break;
        }

        // looking for this market in pvn filter
        sTempMarketData.tMarket = psBSAFilter->tMarket;
        eReturnCode = OSAL.eLinkedListSearch(psIteratorArg->hEntriesList,
            &hEntry, &sTempMarketData);
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // ok, need to add
            psMarketData = psCreateMarketData(psIteratorArg->psObj,
                psIteratorArg->hEntriesList, psBSAFilter->tMarket);
            if (psMarketData == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to create market"
                        );
                psIteratorArg->bResult = FALSE;
                break;
            }
        }
        else if (eReturnCode == OSAL_SUCCESS)
        {
            psMarketData =
                (TRAFCAM1_MARKET_DATA_STRUCT *)OSAL.pvLinkedListThis(hEntry);
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to search in messages list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            psIteratorArg->bResult = FALSE;
            break;
        }

        // searching for BSA in pvn filter
        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        sTempBSAData.tBSA = psBSAFilter->tBSA;
        eReturnCode = OSAL.eLinkedListSearch(psMarketData->hBSAData,
            &hEntry, &sTempBSAData);
        if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            TRAFCAM1_BSA_DATA_STRUCT *psBSAData = NULL;

            // adding new BSA to pvn filter
            psBSAData = psCreateBSAData(psMarketData->hBSAData,
                psBSAFilter->tBSA);
            if (psBSAData == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to create BSA data");
                psIteratorArg->bResult = FALSE;
                break;
            }
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to search for BSA data (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            psIteratorArg->bResult = FALSE;
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   bGetMarketData
*
*****************************************************************************/
static BOOLEAN bGetMarketData(
    OSAL_OBJECT_HDL hMarketsList,
    OSAL_BUFFER_HDL hPayload,
    TRAFCAM1_MARKET_DATA_STRUCT **ppsMarketData,
    TRAFCAM1_BSA_DATA_STRUCT **ppsBSAData
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        size_t tPeekedBits = 0;
        UN8 un8Buffer = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        TRAFCAM1_MARKET_DATA_STRUCT sTempMarketData;
        TRAFCAM1_BSA_DATA_STRUCT sTempBSAData;

        OSAL.bMemSet(&sTempMarketData, 0, sizeof(sTempMarketData));
        OSAL.bMemSet(&sTempBSAData, 0, sizeof(sTempBSAData));

        *ppsMarketData = NULL;
        *ppsBSAData = NULL;

        tPeekedBits = OSAL.tBufferPeekBits(hPayload, &un8Buffer, 0,
            TRAFCAM1_TABLE_BITLEN, TRAFCAM1_TABLE_OFFSET);
        if (tPeekedBits != TRAFCAM1_TABLE_BITLEN)
        {
            // cannot get table number
            break;
        }

        printf(TRAFCAM1_OBJECT_NAME":Packet market %u\n", un8Buffer);

        // searching for packet market in pvn filter
        sTempMarketData.tMarket = (TRAFFIC_MARKET)un8Buffer;
        eReturnCode = OSAL.eLinkedListSearch(hMarketsList, &hEntry, 
            &sTempMarketData);
        if (eReturnCode == OSAL_SUCCESS)
        {
            *ppsMarketData =
                (TRAFCAM1_MARKET_DATA_STRUCT *)OSAL.pvLinkedListThis(hEntry);
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            bResult = TRUE;
            break;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Search in Market LL failed (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        un8Buffer = 0;
        tPeekedBits = OSAL.tBufferPeekBits(hPayload, &un8Buffer, 0,
            TRAFCAM1_BSA_BITLEN, TRAFCAM1_BSA_OFFSET);
        if (tPeekedBits != TRAFCAM1_BSA_BITLEN)
        {
            // cannot get BSA number
            break;
        }

        printf(TRAFCAM1_OBJECT_NAME":Packet BSA %u\n", un8Buffer);

        // searching for packet BSA in pvn filter
        sTempBSAData.tBSA = (TRAFFIC_BSA)un8Buffer;
        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        eReturnCode = OSAL.eLinkedListSearch((*ppsMarketData)->hBSAData,
            &hEntry, &sTempBSAData);
        if (eReturnCode == OSAL_SUCCESS)
        {
            *ppsBSAData =
                (TRAFCAM1_BSA_DATA_STRUCT *)OSAL.pvLinkedListThis(hEntry);
        }
        else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Search in BSA LL failed (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   psCreateGroup
*
*****************************************************************************/
static TRAFCAM1_GROUP_STRUCT *psCreateGroup (
    OSAL_OBJECT_HDL hGroupsList,
    TRAFCAM1_GROUP_STRUCT *psGroup
        )
{
    TRAFCAM1_GROUP_STRUCT *psResult = NULL;

    psResult =
        (TRAFCAM1_GROUP_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
        TRAFCAM1_OBJECT_NAME"Group", sizeof(*psResult), TRUE);
    if (psResult == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM1_OBJECT_NAME": Failed to allocate memory for group."
                );
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // adding our new group to pvn filter
        *psResult = *psGroup;
        eReturnCode = OSAL.eLinkedListAdd(hGroupsList, &psResult->hEntry,
            psResult);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to add to linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            OSAL.vLinkedListMemoryFree(psResult);
            psResult = NULL;
        }
    }

    return psResult;
}

/*****************************************************************************
*
*   vReleaseGroupWithNotify
*
*****************************************************************************/
static void vReleaseGroupWithNotify(
    TRAFCAM1_GROUP_STRUCT *psGroup
        )
{
    if (psGroup != NULL)
    {
        GsTrafcamMgr.bReportRemovedImage(
            psGroup->psObj->hTrafficCamerasService,
            (TRAFCAM_IMAGE_HASH)psGroup);

        OSAL.vLinkedListMemoryFree(psGroup);
    }

    return;
}


/*****************************************************************************
*
*   bReadPacketData
*
*****************************************************************************/
static BOOLEAN bReadPacketData(
    OSAL_BUFFER_HDL hPayload,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        size_t tBitsRead = 0;
        BOOLEAN bSuccess = FALSE;

        bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
            &psPacketInfo->un16SIG, TRAFCAM1_SIG_BITLEN);
        if (bSuccess != TRUE)
        {
            break;
        }

        // reading the data
        tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &psPacketInfo->un8AUTOT,
            0, TRAFCAM1_AUTOT_BITLEN);
        if (tBitsRead != TRAFCAM1_AUTOT_BITLEN)
        {
            // error!
            break;
        }

        psPacketInfo->un8AUTOT++;
        if (psPacketInfo->un8AUTOT > TRAFCAM1_AUTOT_MAX)
        {
            // error!
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
            &psPacketInfo->un8AUCT, 0, TRAFCAM1_AUCT_BITLEN);
        if (tBitsRead != TRAFCAM1_AUCT_BITLEN)
        {
            // error!
            break;
        }

        if (psPacketInfo->un8AUCT >= psPacketInfo->un8AUTOT)
        {
            // error!
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bCleanSlot
*
*****************************************************************************/
static BOOLEAN bCleanSlot(
    TRAFCAM1_PROCESS_SLOT_STRUCT *psSlot,
    TRAFFIC_MARKET tMarket
        )
{
    BOOLEAN bResult = TRUE;

    // cleaning could happen in case if slot should be cleaned anyway
    //  (TRAFFIC_INVALID_MARKET used) due to an error or
    //  if specific market is not needed anymore (valid market number is used)
    if ((psSlot != NULL) && ((tMarket == TRAFFIC_INVALID_MARKET) ||
        (tMarket == psSlot->tMarket)))
    {
        if (psSlot->hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            DATASERVICE_IMPL_bFreeDataPayload(psSlot->hBuffer);
            psSlot->hBuffer = OSAL_INVALID_BUFFER_HDL;
        }

        psSlot->un16SIG = 0;
        psSlot->un8AUCT = 0;
        psSlot->un8BSA = 0;
        psSlot->tMarket = TRAFFIC_INVALID_MARKET;
    }

    return bResult;
}

/*****************************************************************************
*
*   n16CompareSIGs
*
*****************************************************************************/
static N16 n16CompareSIGs(
    TRAFCAM1_GROUP_STRUCT *psGroup1,
    TRAFCAM1_GROUP_STRUCT *psGroup2
        )
{
    N16 n16Result = N16_MIN;

    if ((psGroup1 != NULL) && (psGroup2 != NULL))
    {
        n16Result = COMPARE(psGroup1->un16SIG, psGroup2->un16SIG);
    }

    return n16Result;
}

/*****************************************************************************
*
*   psExistingGroup
*
*****************************************************************************/
static TRAFCAM1_GROUP_STRUCT *psExistingGroup (
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    TRAFCAM1_GROUP_STRUCT *psResult = NULL;
    TRAFCAM1_GROUP_STRUCT sGroup;

    // searching for the group with the same SIG
    sGroup.un16SIG = psPacketInfo->un16SIG;
    eReturnCode = OSAL.eLinkedListSearch(psBSAData->hGroups, &hEntry, &sGroup);
    if (eReturnCode == OSAL_SUCCESS)
    {
        psResult = (TRAFCAM1_GROUP_STRUCT *)OSAL.pvLinkedListThis(hEntry);
    }
    else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM1_OBJECT_NAME": Failed to search in groups list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }

    return psResult;
}

/*****************************************************************************
*
*   bCalculateCRC
*
*****************************************************************************/
static BOOLEAN bCalculateCRC(
    OSAL_OBJECT_HDL hCRC,
    OSAL_BUFFER_HDL hBuffer,
    OSAL_CRC_RESULT *ptCRC
        )
{
    BOOLEAN bResult = FALSE;

    // Initialize the CRC
    bResult = OSAL.bInitializeCRC(hCRC, ptCRC);
    if (bResult == TRUE)
    {
        size_t tBufferSize = 0, tBytesProcessed = 0;

        tBufferSize = OSAL.tBufferGetSize(hBuffer);

        // Compute the CRC of the entire payload
        *ptCRC = OSAL.tComputeCRC(
            hCRC, *ptCRC, hBuffer,
            0, tBufferSize, &tBytesProcessed
                );

        // Did we process the entire payload?
        if (tBytesProcessed != tBufferSize)
        {
            // No!
            bResult = FALSE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bReadImageInfo
*
*****************************************************************************/
static BOOLEAN bReadImageInfo(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TRAFCAM_IMAGE_INFO_STRUCT *psImageInfo,
    TRAFCAM1_GROUP_STRUCT *psGroup
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_BUFFER_HDL hTempBuffer = OSAL_INVALID_BUFFER_HDL;

    do
    {
        size_t tBitsRead = 0;
        size_t tBitsReadAll = 0;
        UN8 un8SOI = 0, un8Buffer = 0;
        BOOLEAN bSuccess = FALSE;
        N16 n16BitsToSeek = 0;
        N32 n32Buffer = 0;

        // allocating the buffer which will be used as temporary storage for
        // values used for CRC calculation
        hTempBuffer = OSAL.hBufferAllocate(psObj->hCRCCalcBlockPool, FALSE,
            FALSE, OSAL_BUFFER_ALLOCATE_OPTION_NONE);
        if (hTempBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to allocate buffer."
                    );
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8SOI, 0,
            TRAFCAM1_SOI_BITLEN);
        if (tBitsRead != TRAFCAM1_SOI_BITLEN)
        {
            // error!
            break;
        }

        tBitsReadAll += tBitsRead;

        bSuccess = OSAL.bBufferReadBitsToN32(hPayload, &n32Buffer,
            TRAFCAM1_LON_BITLEN);
        if (bSuccess == FALSE)
        {
            // error!
            break;
        }

        bSuccess = 
			(OSAL.tBufferWriteTail(hTempBuffer, &n32Buffer, sizeof(n32Buffer)) > 0) ? 
				TRUE : FALSE;
        if (bSuccess == FALSE)
        {
            // error!
            break;
        }

        tBitsReadAll += TRAFCAM1_LON_BITLEN;
        // adding minus sign to longitude value
        n32Buffer = -n32Buffer;

        psImageInfo->hLon = OSAL_FIXED.hCreateInMemory(
            n32Buffer, TRAFCAM1_LON_FRAC_BITS,
            &psImageInfo->atFixedData[0]
                );
        if (psImageInfo->hLon == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        n32Buffer = 0;
        bSuccess = OSAL.bBufferReadBitsToN32(hPayload, &n32Buffer,
            TRAFCAM1_LAT_BITLEN);
        if (bSuccess == FALSE)
        {
            // error!
            break;
        }

        bSuccess = 
			(OSAL.tBufferWriteTail(hTempBuffer, &n32Buffer, sizeof(n32Buffer)) > 0) ?
				TRUE : FALSE;
        if (bSuccess == FALSE)
        {
            // error!
            break;
        }

        tBitsReadAll += TRAFCAM1_LAT_BITLEN;

        psImageInfo->hLat = OSAL_FIXED.hCreateInMemory(
            n32Buffer, TRAFCAM1_LAT_FRAC_BITS,
            &psImageInfo->atFixedData[OSAL_FIXED_OBJECT_SIZE]);
        if (psImageInfo->hLat == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
            TRAFCAM1_FLAG_BITLEN);
        if (tBitsRead != TRAFCAM1_FLAG_BITLEN)
        {
            break;
        }

        tBitsReadAll += tBitsRead;
        if (un8Buffer == 0)
        {
            psImageInfo->tPosCode = TRAFFIC_INVALID_POS_CODE;
        }
        else
        {
            UN16 un16Buffer = 0;

            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload, &un16Buffer,
                TRAFCAM1_TMC_BITLEN);
            if (bSuccess == FALSE)
            {
                break;
            }

            bSuccess = 
				(OSAL.tBufferWriteTail(hTempBuffer, &un16Buffer, sizeof(un16Buffer)) > 0) ?
					TRUE : FALSE;
            if (bSuccess == FALSE)
            {
                // error!
                break;
            }

            tBitsReadAll += TRAFCAM1_TMC_BITLEN;
            psImageInfo->tPosCode = (TRAFFIC_POS_CODE)un16Buffer;

            un8Buffer = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
                TRAFCAM1_OFFSET_BITLEN);
            if (tBitsRead != TRAFCAM1_OFFSET_BITLEN)
            {
                break;
            }

            bSuccess = 
				(OSAL.tBufferWriteTail(hTempBuffer, &un8Buffer, sizeof(un8Buffer)) > 0) ?
					TRUE : FALSE;
            if (bSuccess == FALSE)
            {
                // error!
                break;
            }

            tBitsReadAll += tBitsRead;
            psImageInfo->tOffset = un8Buffer;

            un8Buffer = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
                TRAFCAM1_DIRECTION_BITLEN);
            if (tBitsRead != TRAFCAM1_DIRECTION_BITLEN)
            {
                break;
            }

            bSuccess = 
				(OSAL.tBufferWriteTail(hTempBuffer, &un8Buffer, sizeof(un8Buffer)) > 0) ?
					TRUE : FALSE;
            if (bSuccess == FALSE)
            {
                // error!
                break;
            }

            tBitsReadAll += tBitsRead;
            if (un8Buffer == TRAFCAM1_DIRECTION_NEGATIVE)
            {
                psImageInfo->tOffset = -(psImageInfo->tOffset);
            }
        }

        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
            TRAFCAM1_FLAG_BITLEN);
        if (tBitsRead != TRAFCAM1_FLAG_BITLEN)
        {
            break;
        }

        tBitsReadAll += tBitsRead;

        if (un8Buffer == 0)
        {
            psImageInfo->un16Direction = TRAFCAM_INVALID_DIRECTION;
        }
        else
        {
            un8Buffer = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
                TRAFCAM1_DIR_BITLEN);
            if (tBitsRead != TRAFCAM1_DIR_BITLEN)
            {
                break;
            }

            bSuccess = 
				(OSAL.tBufferWriteTail(hTempBuffer, &un8Buffer, sizeof(un8Buffer)) > 0) ?
					TRUE : FALSE;
            if (bSuccess == FALSE)
            {
                // error!
                break;
            }

            tBitsReadAll += tBitsRead;
            psImageInfo->un16Direction = un8Buffer * 45;
        }

        // reading F+DESC
        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
            TRAFCAM1_FLAG_BITLEN);
        if (tBitsRead != TRAFCAM1_FLAG_BITLEN)
        {
            break;
        }

        tBitsReadAll += tBitsRead;

        if (un8Buffer != 0)
        {
            char acDesc[TRAFCAM1_DESC_MAX_LENGTH];
            UN8 un8Index = 0;

            do
            {
                un8Buffer = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
                    TRAFCAM1_DESC_SYMBOL_BITLEN);
                if (tBitsRead != TRAFCAM1_DESC_SYMBOL_BITLEN)
                {
                    bSuccess = FALSE;
                    break;
                }

                bSuccess = 
					(OSAL.tBufferWriteTail(hTempBuffer, &un8Buffer, sizeof(un8Buffer)) > 0) ?
						TRUE : FALSE;
                if (bSuccess == FALSE)
                {
                    // error!
                    break;
                }

                tBitsReadAll += tBitsRead;
                acDesc[un8Index++] = un8Buffer;
            } while ((un8Buffer != 0) && (un8Index < TRAFCAM1_DESC_MAX_LENGTH));

            if (bSuccess == FALSE)
            {
                // error!
                break;
            }

            psImageInfo->hDescription = STRING_hCreate(SMS_INVALID_OBJECT,
                acDesc, un8Index - 1, 0);
            if (psImageInfo->hDescription == STRING_INVALID_OBJECT)
            {
                // error!
                break;
            }
        }

        // reading F+TIME
        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
            TRAFCAM1_FLAG_BITLEN);
        if (tBitsRead != TRAFCAM1_FLAG_BITLEN)
        {
            break;
        }

        tBitsReadAll += tBitsRead;

        if (un8Buffer != 0)
        {
            UN16 un16Buffer = 0;
            struct tm sTime;
            struct tm *psTime;
            UN32 un32UTCTime;
            TIME_T tUTCTime;
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload, &un16Buffer,
                TRAFCAM1_TIME_BITLEN);
            if (bSuccess != TRUE)
            {
                break;
            }

            tBitsReadAll += TRAFCAM1_TIME_BITLEN;

            eReturnCode = OSAL.eTimeGet(&un32UTCTime);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // error!
                break;
            }

            tUTCTime = un32UTCTime;
            psTime = OSAL.gmtime_r(&tUTCTime, &sTime);
            if (psTime == NULL)
            {
                //error!
                break;
            }

            sTime.tm_hour = un16Buffer / 60;
            if (sTime.tm_hour > 23)
            {
                // error !
                break;
            }

            sTime.tm_min = un16Buffer % 60;
            sTime.tm_sec = 0;
            psImageInfo->un32ProcessTimeStamp = (UN32)OSAL.mktime(&sTime);
        }

        un8Buffer = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(hPayload, &un8Buffer, 0,
            TRAFCAM1_TTL_BITLEN);
        if (tBitsRead != TRAFCAM1_TTL_BITLEN)
        {
            // error !
            break;
        }
        else if (un8Buffer > TRAFCAM1_TTL_MAX)
        {
            break;
        }
        else
        {
            // converting TTL minutes to seconds
            psImageInfo->un16TTL = un8Buffer * 60;
            printf(TRAFCAM1_OBJECT_NAME":Seconds to live - %u\n",
                psImageInfo->un16TTL);
        }

        tBitsReadAll += tBitsRead;

        // seeking to the start of image
        n16BitsToSeek = (un8SOI + 1) * 8 - (N16)tBitsReadAll;
        if (n16BitsToSeek > 0)
        {
            tBitsRead = OSAL.tBufferSeekHeadBits(hPayload, n16BitsToSeek);
            if (tBitsRead != (size_t)n16BitsToSeek)
            {
                break;
            }
        }

        // now we calculate CRC of our temporary buffer
        bSuccess = bCalculateCRC(psObj->hCRC, hTempBuffer, &psGroup->tCRC);
        if (bSuccess == FALSE)
        {
            // error!
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (hTempBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        eReturnCode = OSAL.eBufferFree(hTempBuffer);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to free buffer (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   n16FindGroupByCRC
*
*****************************************************************************/
static N16 n16FindGroupByCRC(
    TRAFCAM1_GROUP_STRUCT *psGroup,
    OSAL_CRC_RESULT *ptCRC
        )
{
    static N16 n16Result = N16_MIN;

    if ((psGroup != NULL) && (ptCRC != NULL))
    {
        if (psGroup->tCRC == *ptCRC)
        {
            n16Result = 0;
        }
        else
        {
            n16Result = 1;
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   bProcessNewGroup
*
*****************************************************************************/
static BOOLEAN bProcessNewGroup (
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        if (psPacketInfo->un8AUTOT == 0)
        {
            //error!
            break;
        }

        if (psPacketInfo->un8AUCT != 0)
        {
            // let's wait for start packet
            bResult = TRUE;
            break;
        }

        if (psPacketInfo->un8AUTOT == 1)
        {
            // process single packet group
            bResult = bCompleteGroup(psObj, phPayload, psMarketData,
                psBSAData, psPacketInfo->un16SIG);
            break;
        }

        // initializing our process slot with group data
        psMarketData->psSlot->hBuffer = *phPayload;
        *phPayload = OSAL_INVALID_BUFFER_HDL;

        psMarketData->psSlot->un16SIG = psPacketInfo->un16SIG;
        psMarketData->psSlot->un8AUCT = psPacketInfo->un8AUCT;
        psMarketData->psSlot->un8BSA = (UN8)psBSAData->tBSA;
        psMarketData->psSlot->tMarket = psMarketData->tMarket;

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bProcessCleanSlot
*
*****************************************************************************/
static BOOLEAN bProcessCleanSlot (
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        )
{
    BOOLEAN bResult = TRUE;
    TRAFCAM1_GROUP_STRUCT *psGroup = NULL;

    // looking for the group with the same parameters - market, BSA and SIG
    psGroup = psExistingGroup(psBSAData, psPacketInfo);
    if (psGroup == NULL)
    {
        // not found, this is new group
        bResult = bProcessNewGroup(psObj, phPayload, psMarketData,
            psBSAData, psPacketInfo);
    }
    else
    {
        printf(TRAFCAM1_OBJECT_NAME":Reporting repeated group\n");
        bResult = GsTrafcamMgr.bReportRepeatedImage(
            psObj->hTrafficCamerasService, (TRAFCAM_IMAGE_HASH)psGroup);
    }

    return bResult;
}

/*****************************************************************************
*
*   bCompleteGroup
*
*****************************************************************************/
static BOOLEAN bCompleteGroup(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    UN16 un16SIG
        )
{
    BOOLEAN bResult = FALSE;
    TRAFCAM_IMAGE_INFO_STRUCT sImageInfo;

    OSAL.bMemSet(&sImageInfo, 0, sizeof(sImageInfo));

    sImageInfo.tMarket = psMarketData->tMarket;
    sImageInfo.tBSA = psBSAData->tBSA;

    do
    {
        BOOLEAN bSuccess = FALSE;
        TRAFCAM1_GROUP_STRUCT sGroup, *psGroup = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        OSAL.bMemSet(&sGroup, 0, sizeof(sGroup));
        sGroup.un16SIG = un16SIG;
        sGroup.psObj = psObj;

        if (psMarketData->psSlot->hBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            // ok, this is single packet group
            psMarketData->psSlot->hBuffer = *phPayload;
            *phPayload = OSAL_INVALID_BUFFER_HDL;
            bSuccess = bReadImageInfo(psObj, psMarketData->psSlot->hBuffer,
                &sImageInfo, &sGroup);
        }
        else
        {
            // this is multi packet group so we append the payload to buffer
            // at the beginning
            eReturnCode = OSAL.eBufferAppend(psMarketData->psSlot->hBuffer,
                *phPayload);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to append the buffer (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            bSuccess = bReadImageInfo(psObj, psMarketData->psSlot->hBuffer,
                &sImageInfo, &sGroup);
        }

        if (bSuccess == FALSE)
        {
            // failed to read image info
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to read image info.");
            break;
        }

        // now looking for group with the same CRC, it could be just
        // update of the camera picture
        eReturnCode = OSAL.eLinkedListLinearSearch(psBSAData->hGroups, &hEntry, 
            (OSAL_LL_COMPARE_HANDLER)n16FindGroupByCRC, &sGroup.tCRC);
        if (eReturnCode == OSAL_SUCCESS)
        {
            psGroup = (TRAFCAM1_GROUP_STRUCT *)OSAL.pvLinkedListThis(hEntry);

            bSuccess = GsTrafcamMgr.bReportUpdatedImage(
                psObj->hTrafficCamerasService, (TRAFCAM_IMAGE_HASH)psGroup,
                psMarketData->psSlot->hBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to report image file to"
                    " manager");
                break;
            }
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // this group is new, need to create new entry
            psGroup = psCreateGroup(psBSAData->hGroups, &sGroup);
            if (psGroup == NULL)
            {
                break;
            }

            sImageInfo.tHash = (TRAFCAM_IMAGE_HASH)psGroup;

            bSuccess = GsTrafcamMgr.bReportNewImage(
                psObj->hTrafficCamerasService, &sImageInfo,
                psMarketData->psSlot->hBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to report image file to"
                    " manager");
                break;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to search in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

    } while (FALSE);

    if (sImageInfo.hDescription != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sImageInfo.hDescription);
    }

    // clean slot since group processing is completed and data is not needed
    // anymore
    bResult = bCleanSlot(psMarketData->psSlot, psMarketData->tMarket);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFCAM1_OBJECT_NAME": Failed to clean slot.");
    }

    return bResult;
}

/*****************************************************************************
*
*   bProcessContinuingGroup
*
*****************************************************************************/
static BOOLEAN bProcessContinuingGroup(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        )
{
    BOOLEAN bResult = FALSE;

    if (psPacketInfo->un8AUTOT == psPacketInfo->un8AUCT + 1)
    {
        // this packet is last in the group, lets put all the data to file
        // without copying the packet payload to the buffer
        bResult = bCompleteGroup(psObj, phPayload, psMarketData,
            psBSAData, psPacketInfo->un16SIG);
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // just next packet in the group
        // lets copy data from payload to buffer
        eReturnCode = OSAL.eBufferAppend(psMarketData->psSlot->hBuffer,
            *phPayload);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to append payload to buffer");
            bCleanSlot(psMarketData->psSlot, psMarketData->tMarket);
        }
        else
        {
            psMarketData->psSlot->un8AUCT = psPacketInfo->un8AUCT;
            bResult = TRUE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bProcessNonCleanSlot
*
*****************************************************************************/
static BOOLEAN bProcessNonCleanSlot(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        )
{
    BOOLEAN bResult = FALSE;

    // check if slot contains the same group
    if ((psMarketData->psSlot->un16SIG == psPacketInfo->un16SIG) &&
        (psBSAData->tBSA == psMarketData->psSlot->un8BSA) &&
        (psMarketData->tMarket == psMarketData->psSlot->tMarket))
    {
        // ok, check if sequence is continued
        if (psPacketInfo->un8AUCT == psMarketData->psSlot->un8AUCT + 1)
        {
            bResult = bProcessContinuingGroup(psObj, phPayload, psMarketData,
                psBSAData, psPacketInfo);
        }
        else // looks like sequence is broken for some reason
        {
            printf(TRAFCAM1_OBJECT_NAME":Sequence is broken, cleaning slot\n");
            bResult = bCleanSlot(psMarketData->psSlot, psMarketData->tMarket);
        }
    }
    else // looks like sequence is broken, need to clean the slot
    {
        printf(TRAFCAM1_OBJECT_NAME":Sequence is broken, cleaning slot\n");

        bResult = bCleanSlot(psMarketData->psSlot, TRAFFIC_INVALID_MARKET);
        if (bResult == TRUE)
        {
            bResult = bProcessCleanSlot(psObj, phPayload, psMarketData,
                psBSAData, psPacketInfo);
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bParsePacketData
*
*****************************************************************************/
static BOOLEAN bParsePacketData(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;
        size_t tBitsRead = 0;
        TRAFCAM1_PACKET_INFO_STRUCT sPacketInfo;

        // Table and BSA are always processed so just skipping those bits
        tBitsRead = OSAL.tBufferSeekHeadBits(*phPayload, 
            TRAFCAM1_BSA_OFFSET + TRAFCAM1_BSA_BITLEN);

        if (tBitsRead != (TRAFCAM1_BSA_OFFSET + TRAFCAM1_BSA_BITLEN))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Insufficient packet data.");
            break;
        }

        OSAL.bMemSet(&sPacketInfo, 0, sizeof(sPacketInfo));

        bSuccess = bReadPacketData(*phPayload, &sPacketInfo);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to read packet data."
                    );
            break;
        }

        printf(TRAFCAM1_OBJECT_NAME": Received data for slot %p,"
            " AUCT %u, AUTOT %u\n", psMarketData->psSlot, sPacketInfo.un8AUCT,
            sPacketInfo.un8AUTOT);

        if (psMarketData->psSlot->hBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            // ok, slot is clean
            bSuccess = bProcessCleanSlot(psObj, phPayload, psMarketData,
                psBSAData, &sPacketInfo);
        }
        else
        {
            bSuccess = bProcessNonCleanSlot(psObj, phPayload, psMarketData,
                psBSAData, &sPacketInfo);
        }

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to process packet data."
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bCheckPacketHeader
*
*****************************************************************************/
static BOOLEAN bCheckPacketHeader(
    OSAL_BUFFER_HDL hPayload
        )
{

    BOOLEAN bResult = FALSE;

    do
    {
        size_t tBitsRead = 0;
        PVN tPVN = 0;
        UN8 un8CarouselID = 0;

        // Peek the PVN
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &tPVN, 0,
            TRAFCAM1_PVN_BITLEN, 0);
        if (tBitsRead != TRAFCAM1_PVN_BITLEN)
        {
            // Peek failed -- the message is probably
            // garbled. Don't do anything rash here.
            printf(TRAFCAM1_OBJECT_NAME":  Unable to read PVN\n");
            break;
        }

        // Verify the PVN
        if (tPVN != TRAFCAM1_PVN)
        {
            printf(TRAFCAM1_OBJECT_NAME
                ": Incorrect PVN - got %u, expected %u\n",
                tPVN, TRAFCAM1_PVN);
            break;
        }

        // Peek the Carousel Id
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8CarouselID, 0,
            TRAFCAM1_CAROUSEL_BITLEN, TRAFCAM1_PVN_BITLEN);
        if (tBitsRead != TRAFCAM1_CAROUSEL_BITLEN)
        {
            // Peek failed -- the message is probably
            // garbled. Don't do anything rash here.
            printf(TRAFCAM1_OBJECT_NAME":  Unable to read Carousel Id\n");
            break;
        }

        // Verify the Carousel ID
        if (un8CarouselID != TRAFCAM1_CAROUSEL_ID)
        {
            printf(TRAFCAM1_OBJECT_NAME
                ": Incorrect Carousel ID - got %u, expected %u\n",
                un8CarouselID, TRAFCAM1_CAROUSEL_ID);
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   n16CompareMarkets
*
*****************************************************************************/
static N16 n16CompareMarkets (
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData1,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData2
        )
{
    N16 n16Result = N16_MIN;

    if ((psMarketData1 != NULL) && (psMarketData2 != NULL))
    {
        n16Result = COMPARE(psMarketData1->tMarket, psMarketData2->tMarket);
    }

    return n16Result;
}

/*****************************************************************************
*
*   vReleaseBSAData
*
*****************************************************************************/
static void vReleaseBSAData(
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData
        )
{
    if (psBSAData != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        if (psBSAData->hGroups != OSAL_INVALID_OBJECT_HDL)
        {
            // cleaning all the groups and notifying manager about groups
            // removal
            eReturnCode = OSAL.eLinkedListRemoveAll(psBSAData->hGroups,
                (OSAL_LL_RELEASE_HANDLER)vReleaseGroupWithNotify);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to cleanup groups list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            eReturnCode = OSAL.eLinkedListDelete(psBSAData->hGroups);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to delete groups list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            psBSAData->hGroups = OSAL_INVALID_OBJECT_HDL;
        }

        OSAL.vLinkedListMemoryFree(psBSAData);
    }

    return;
}

/*****************************************************************************
*
*   vReleaseMarketData
*
*****************************************************************************/
static void vReleaseMarketData(
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData
        )
{
    if (psMarketData != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        vUnRegisterSlot(psMarketData->psSlot, psMarketData->tMarket);
        psMarketData->psSlot = NULL;

        if (psMarketData->hBSAData != OSAL_INVALID_OBJECT_HDL)
        {
            // cleaning BSAs assigned to this market
            eReturnCode = OSAL.eLinkedListRemoveAll(psMarketData->hBSAData,
                (OSAL_LL_RELEASE_HANDLER)vReleaseBSAData);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to clean up BSA list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            eReturnCode = OSAL.eLinkedListDelete(psMarketData->hBSAData);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFCAM1_OBJECT_NAME": Failed to delete BSA list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            psMarketData->hBSAData = OSAL_INVALID_OBJECT_HDL;
        }

        OSAL.vLinkedListMemoryFree(psMarketData);
    }

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject(
    TRAFCAM1_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    UN8 un8Iterator = 0;

    if (psObj == NULL)
    {
        return;
    }

    // deletion of CRC block pool
    if (psObj->hCRCCalcBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eBlockPoolDelete(psObj->hCRCCalcBlockPool);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to delete block pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        psObj->hCRCCalcBlockPool = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eReleaseCRC(psObj->hCRC);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to release CRC (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
    }

    // freeing pvn filter and groups
    if (psObj->hMessagesData != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hMessagesData,
            (OSAL_LL_RELEASE_HANDLER)vReleaseMarketData);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to release CRC (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(psObj->hMessagesData);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFCAM1_OBJECT_NAME": Failed to delete messages list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        psObj->hMessagesData = OSAL_INVALID_OBJECT_HDL;
    }

    // cleaning processing slots
    for (un8Iterator = 0; un8Iterator < TRAFCAM1_NUM_OF_DMIS; un8Iterator++)
    {
        bCleanSlot(&(psObj->asSlot[un8Iterator]), TRAFFIC_INVALID_MARKET);
    }

    psObj->hTrafficCamerasService = TRAFFIC_CAMERAS_SERVICE_INVALID_OBJECT;

    SMSO_vDestroy((SMS_OBJECT)psObj);

    return;
}
