/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 *
 *
 *
 *
 * DESCRIPTION
 *
 *  This module contains the High-Band specific sports service functions
 *  for the Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <stdlib.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sti_api.h"
#include "sms_obj.h"
#include "baudot.h"
#include "string_obj.h"

#include "xmappid_packet_util.h"

#include "sports_service_highband.h"
#include "_sports_service_highband.h"
#include "sports_service_db_constants.h"

#include "sms_api_debug.h"

// For sports debug output
#include "sports_service_mgr_obj.h"


/*****************************************************************************
                             DEBUG FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   _bLLIteratorDumpLL
*    DEBUG ONLY FUNCTION - iterate Linked list and dump the contents of each node
*
*****************************************************************************/
#if SSDL_HB_TREF
static BOOLEAN _bLLIteratorDumpLL ( void *pvData, void *pvArg )
{
    TREF_INFO_STRUCT *TrefInfo = (TREF_INFO_STRUCT *)pvData;   // member in list

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
            "_bLLIteratorDumpLL:"
            "DUMP: TREF[%d] @ BitOffset=%d  RowId=%llu  DI_%08X c%d, Pid=%llu Cid=%llu\n",
            (UN32)TrefInfo->un8RowIndex,
            (UN32)TrefInfo->un16ByteOffset*8, // express in bits
            TrefInfo->n64PrimaryRowId,
            (UN32)TrefInfo->un32TdefKey,
            (UN32)TrefInfo->un8LabelId,
            TrefInfo->tParentInstanceId,
            TrefInfo->tChildInstanceId
            );

    return TRUE;    // keep going
}
#endif // SSDL_HB_TREF

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hInit
*
*   Creates and initializes the "highband" sports service interface object.
*
*   Inputs:
*       hSportsDataService - A handle to the central sports service
*           manager object handle
*   Output:
*       An object handle representing this instantiation of the
*       sports service interface (representing high-band support)
*
*****************************************************************************/
static SPORTS_SERVICE_INTERFACE_OBJECT hInit (
    SPORTS_SERVICE_OBJECT hSportsDataService,
    SMS_OBJECT hParent
        )
{
    SPORTS_SERVICE_INTERFACE_OBJECT hHighBand =
        SPORTS_SERVICE_INTERFACE_INVALID_OBJECT;
    BOOLEAN bOwner;
    BOOLEAN bOk;

    // Verify we own the interface owner
    bOwner = SMSO_bOwner(hParent);
    if (bOwner == TRUE)
    {
        do
        {
            SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj;
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Create an instance of this object
            psHighbandObj = (SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *)
                SMSO_hCreate(
                    SPORTS_SVC_HB_OBJ_NAME,
                    sizeof(SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT),
                    hParent, FALSE );

            if (psHighbandObj == NULL)
            {
                // Error!
            	break;
            }
            // Request access to ISO 3309 CRC32
            eReturnCode = OSAL.eGetCRC(&psHighbandObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32);

            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSO_vDestroy((SMS_OBJECT)psHighbandObj);
                break;
            }

            // Create the empty table definitions lookup list
            psHighbandObj->hTableDefs = (SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *)
            		SMSO_hCreate(SPORTS_SVC_HB_OBJ_NAME":TableDefs",
							(size_t) sizeof(SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT) *
										SPORTS_SERVICE_HIGHBAND_MAX_NUMBER_TABLE_DEFS,
										(SMS_OBJECT)psHighbandObj,
										FALSE
						);
            if(!psHighbandObj->hTableDefs)
            {
                // Error!
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_LIFE_CYCLE,
                    "hInit:"
                    "Unable to create empty TDEF struct\n");

                OSAL.eReleaseCRC(psHighbandObj->hCRC);
                SMSO_vDestroy((SMS_OBJECT)psHighbandObj);
                break;
            }

            // Save the service handle
            psHighbandObj->hSportsDataService = hSportsDataService;

            hHighBand = (SPORTS_SERVICE_INTERFACE_OBJECT)psHighbandObj;

            // Load the table def from the database...
            bOk = GsSportsServiceMgrIntf.bUnsafeLoadTdefs(
                    hSportsDataService,
                    hHighBand );

            // Failure is informational.  Loading tdefs makes gathering data
            // immediatly possible.  If it fails keep going.  We will try to
            // get missing tdefs from the broadcast.
            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_LIFE_CYCLE,
                                    "hInit:"
                                    "bUnsafeLoadTdefs failure\n");
            }

        } while (0);
    }

    return hHighBand;
}

/*****************************************************************************
*
*   vUnInit
*
*   Uninitializes and destroys the "highband" sports data interface object.
*   This API must only be called when already in the context of the
*   sports data service manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*
*****************************************************************************/
static void vUnInit (
    SPORTS_SERVICE_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;
    UN32 un32Cnt;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj =
            (SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *)hInterface;

        // Release the CRC
        OSAL.eReleaseCRC(psHighbandObj->hCRC);

        // Destroy Table def lookup table
        if (bClearAllTableDefs(psHighbandObj))
        {
        	// Free up the memory used for the table defs
        	SMSO_vDestroy((SMS_OBJECT)psHighbandObj->hTableDefs);
        	psHighbandObj->hTableDefs = NULL;
        }

        // Clean up au gathering
        for( un32Cnt=0; un32Cnt<CAROUSEL_ID_COUNT; un32Cnt++ )
        {
            vEndAuTransaction( &psHighbandObj->asAuGroup[un32Cnt] );
        }

        SMSO_vDestroy((SMS_OBJECT)psHighbandObj);
    }

    return;
}

/*****************************************************************************
*
*   bAddTdef
*
*   This API must only be called when already in the context of the
*   highband sports data object.
*
*****************************************************************************/
static BOOLEAN bUnsafeAddTdef (
        SPORTS_SERVICE_INTERFACE_OBJECT hInterface,
        SPORTS_SERVICE_TDEF_ADD_STRUCT *psTdefAdd
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *psTdef;
    SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj =
                (SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *)hInterface;

    if( (hInterface == OSAL_INVALID_OBJECT_HDL) ||
        (psHighbandObj->hTableDefs == NULL) )
    {
        return FALSE;
    }

    if( psTdefAdd->un32TdefId >= SPORTS_SERVICE_HIGHBAND_MAX_NUMBER_TABLE_DEFS )
    {
        return FALSE;
    }

    psTdef = &psHighbandObj->hTableDefs[psTdefAdd->un32TdefId];
    bOk = OSAL.bMemCpy( psTdef,
                        &psTdefAdd->sTdefInfo,
                        sizeof(psTdefAdd->sTdefInfo) );

    if( bOk )
    {
        psTdef->IsValid = TRUE;
    }

    return bOk;
}
/*****************************************************************************
*
*   bClearAllTableDefs
*
*   Clears out all of the in-memory table defs
*   This API must only be called when already in the context of the
*   highband sports data object.
*
*   Inputs:
*       psHighbandObj - The highband object handle
*
*****************************************************************************/
static BOOLEAN bClearAllTableDefs(
		SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj
		)
{
	BOOLEAN bOk=FALSE;
    N16 n16LoopControlVariable;
    OSAL_RETURN_CODE_ENUM eReturnCode;

	if ( (psHighbandObj != NULL) &&
	     (psHighbandObj->hTableDefs != NULL) )
	{
		// Go through each entry and see if it owns a linked list.
		// Empty each one and destroy it.
		for (n16LoopControlVariable = 0;
				n16LoopControlVariable < SPORTS_SERVICE_HIGHBAND_MAX_NUMBER_TABLE_DEFS;
				n16LoopControlVariable++)
		{
			if (psHighbandObj->hTableDefs[n16LoopControlVariable].IsValid)
			{
				psHighbandObj->hTableDefs[n16LoopControlVariable].IsValid = FALSE;
				psHighbandObj->hTableDefs[n16LoopControlVariable].un8TableVer = 0;
				psHighbandObj->hTableDefs[n16LoopControlVariable].un8TableClass = 0;
				psHighbandObj->hTableDefs[n16LoopControlVariable].un8LabelCount = 0;
				psHighbandObj->hTableDefs[n16LoopControlVariable].CRC = 0;

				if (psHighbandObj->hTableDefs[n16LoopControlVariable].hLabelInfoList
						!= OSAL_INVALID_OBJECT_HDL)
				{

					// Remove links from the list and destroy
					eReturnCode = OSAL.eLinkedListRemoveAll(
							psHighbandObj->hTableDefs[n16LoopControlVariable].hLabelInfoList,
							(OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy
						);
					if (eReturnCode == OSAL_SUCCESS)
					{
						// Destroy the list itself
						eReturnCode = OSAL.eLinkedListDelete(
								psHighbandObj->hTableDefs[n16LoopControlVariable].hLabelInfoList
							);
						if(eReturnCode == OSAL_SUCCESS)
						{
							psHighbandObj->hTableDefs[n16LoopControlVariable].hLabelInfoList
									= OSAL_INVALID_OBJECT_HDL;
						}
					}
				}
			}
		}
		bOk = TRUE;
	}
	return bOk;
}

/***************************************************************************
*
*   eProcessMessage
*
*   This object interface function is used by the sports service manager in order
*   to indicate that a new message has been received.
*
*   This API must only be called when already in the
*   context of the sports service manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*       *phPayload - A pointer to valid STI payload
*
*   Output:
*       DATASERVICE_ERROR_CODE_ENUM
*
***************************************************************************/

static DATASERVICE_ERROR_CODE_ENUM eProcessMessage (
    SPORTS_SERVICE_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    AU_INFO_STRUCT sAuInfo;
    size_t tPayloadSize;
    BOOLEAN bOk;
    DATASERVICE_ERROR_CODE_ENUM eResult = DATASERVICE_ERROR_CODE_NONE;
    size_t tBitsRead;
    AU_GROUP_STRUCT *psAuGroup;
    SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj =
        (SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *)hInterface;


    if( phPayload == NULL )
    {
        return eResult;
    }

    if( *phPayload == OSAL_INVALID_BUFFER_HDL )
    {
        return eResult;
    }

    // initialize AU Info struct
    OSAL.bMemSet(&sAuInfo, 0, sizeof(sAuInfo));

    // Calls SMSO_bValid in all possible #ifdef forks.
    bOk = SMSO_bOwner(hInterface);

    // Get size of au so we can reject spec violations.
    tPayloadSize = OSAL.tBufferGetSize( *phPayload );

    if( !bOk || (tPayloadSize>SPORTS_SERVICE_MAX_AU_SIZE_BYTES) )
    {
        GsSportsServiceMgrIntf.vReleasePayload( *phPayload );
        return eResult;
    }

    // Begin parsing bits.
    sAuInfo.hPayload = *phPayload;
    sAuInfo.tOffset = 0;

    do
    {
        // Validate the message (don't consume the CRC bytes though)
        // Saves crc and size of au in bits
        bOk = bVerifyAu(
            (SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *)hInterface,
            &sAuInfo );

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bProcessMessage:"
                    "Packet Invalid\n");
            eResult = DATASERVICE_ERROR_CODE_NONE;
            break;
        }

        // Extract the PVN
        sAuInfo.un8Pvn = 0;
        tBitsRead = OSAL.tBufferPeekBits(
            *phPayload,
            &sAuInfo.un8Pvn,
            0,
            SPORTS_SERVICE_PVN_BITLEN,
            sAuInfo.tOffset );

        sAuInfo.tOffset += tBitsRead;

        if( tBitsRead != SPORTS_SERVICE_PVN_BITLEN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bProcessMessage:"
                    "Unable to read PVN\n");

            eResult = DATASERVICE_ERROR_CODE_NONE;
            break;
        }

        // Verify the PVN
        if( sAuInfo.un8Pvn != SPORTS_SERVICE_PVN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bProcessMessage:"
                    "Incorrect PVN - got %u, expected %u\n",
                    (UN32)sAuInfo.un8Pvn, (UN32)SPORTS_SERVICE_PVN);

            eResult = DATASERVICE_ERROR_CODE_NONE;
            break;
        }

        // Extract the carousel id
        sAuInfo.un8CarouselId = 0;
        tBitsRead = OSAL.tBufferPeekBits(
            *phPayload,
            &sAuInfo.un8CarouselId,
            0,
            SPORTS_SERVICE_CAROUSEL_BITLEN,
            sAuInfo.tOffset );

        sAuInfo.tOffset += tBitsRead;

        if( tBitsRead != SPORTS_SERVICE_CAROUSEL_BITLEN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bProcessMessage:"
                    "Unable to read Carousel Id\n");

            eResult = DATASERVICE_ERROR_CODE_NONE;
            break;
        }

        // Skip the RFU field
        sAuInfo.tOffset += SPORTS_SERVICE_RFU1_BITLEN;

        // Extract the configuration version.
        sAuInfo.un8CfgVer = 0;
        tBitsRead = OSAL.tBufferPeekBits(
            *phPayload,
            &sAuInfo.un8CfgVer,
            0,
            SPORTS_SERVICE_CFGVER_BITLEN,
            sAuInfo.tOffset );

        sAuInfo.tOffset += tBitsRead;

        if( tBitsRead != SPORTS_SERVICE_CFGVER_BITLEN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bProcessMessage:"
                    "Unable to read Config Ver\n");

            eResult = DATASERVICE_ERROR_CODE_NONE;
            break;
        }

        // Each carousel processor is responsible for freeing the buffer
        switch( sAuInfo.un8CarouselId )
        {
            case CAROUSEL_ID_CONFIG:
            {
                psAuGroup = &psHighbandObj->asAuGroup[CAROUSEL_ID_CONFIG];
                eResult = eProcessConfigAu(
                        psHighbandObj,
                        psAuGroup,
                        &sAuInfo );
                break;
            }
            case CAROUSEL_ID_AFFILIATIONS:
            {
                psAuGroup =
                    &psHighbandObj->asAuGroup[CAROUSEL_ID_AFFILIATIONS];
                eResult = eProcessAffiliateAu(
                        psHighbandObj,
                        psAuGroup,
                        &sAuInfo );
                break;
            }
            case CAROUSEL_ID_TABLEDEF:
            {
                psAuGroup = &psHighbandObj->asAuGroup[CAROUSEL_ID_TABLEDEF];
                eResult = eProcessTableDefAu(
                        psHighbandObj,
                        psAuGroup,
                        &sAuInfo );
                break;
            }
            case CAROUSEL_ID_DATA0:
            {
                psAuGroup = &psHighbandObj->asAuGroup[CAROUSEL_ID_DATA0];
                eResult = eProcessDataAu(
                        psHighbandObj,
                        psAuGroup,
                        &sAuInfo );
                break;
            }
            case CAROUSEL_ID_DATA1:
            {
                psAuGroup = &psHighbandObj->asAuGroup[CAROUSEL_ID_DATA1];
                eResult = eProcessDataAu(
                        psHighbandObj,
                        psAuGroup,
                        &sAuInfo );
                break;
            }
            default:
            {
                // Don't make this a big deal
                eResult = DATASERVICE_ERROR_CODE_NONE;
                GsSportsServiceMgrIntf.vReleasePayload( *phPayload );

                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bProcessMessage:"
                    "Unknown/unhandled Car ID: %u\n",
                    (UN32)sAuInfo.un8CarouselId );
            }
        }

        return eResult;

    } while(0);

    GsSportsServiceMgrIntf.vReleasePayload( *phPayload );
    return eResult;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eProcessConfigAu
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eProcessConfigAu(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo
        )
{
    BOOLEAN bOk;
    DATASERVICE_ERROR_CODE_ENUM eResult;

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
            "eProcessConfigAu:S"
            " psAuGroup 0x%x psAuInfo 0x%x\n",
            (UN32)psAuGroup,
            (UN32)psAuInfo );

    // The au fields are next in the payload
    eResult = eProcessAu(
            psAuGroup,
            psAuInfo,
            bIsConfigAuValid );

    bOk = bIsAuGroupComplete( psAuGroup, SS_MAX_CONFIG_PAYLOAD_SIZE_BITS );

    if( bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                "eProcessConfigAu:Group Complete\n" );

        vPrintAuGroup( psAuGroup, SSDL_HB_AU_PROCESSING );

        bOk = bProcessConfigAuGroup(
                psHighbandObj,
                psAuGroup );

        vEndAuTransaction( psAuGroup );
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
            "eProcessConfigAu:E"
            " eResult %u bOk %u\n",
            (UN32)eResult,
            (UN32)bOk );

    return eResult;
}

/*****************************************************************************
*
*   eProcessAffiliateAu
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eProcessAffiliateAu(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo
        )
{
    BOOLEAN bOk;
    DATASERVICE_ERROR_CODE_ENUM eResult;
    size_t tBitsRead;

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
            "eProcessAffiliateAu:S"
            " psAuGroup 0x%x psAuInfo 0x%x\n",
            (UN32)psAuGroup,
            (UN32)psAuInfo );

    // Extract the affiliate table version
    psAuInfo->un8AdVer = 0;
    tBitsRead = OSAL.tBufferPeekBits(
        psAuInfo->hPayload,
        &psAuInfo->un8AdVer,
        0,
        SPORTS_SERVICE_ADVER_BITLEN,
        psAuInfo->tOffset );

    psAuInfo->tOffset += tBitsRead;

    if( tBitsRead != SPORTS_SERVICE_ADVER_BITLEN )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                "eProcessAffiliateAu:"
                "Unable to read ADVER\n");

        GsSportsServiceMgrIntf.vReleasePayload( psAuInfo->hPayload );
        vEndAuTransaction( psAuGroup );
        return DATASERVICE_ERROR_CODE_NONE;
    }

    // The au fields are next in the payload
    eResult = eProcessAu(
            psAuGroup,
            psAuInfo,
            bIsAffiliateAuValid );

    bOk = bIsAuGroupComplete( psAuGroup, SS_MAX_AFFILIATE_PAYLOAD_SIZE_BITS );

    if( bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                "eProcessAffiliateAu:Group Complete\n" );
        vPrintAuGroup( psAuGroup, SSDL_HB_AU_PROCESSING );

        bOk = bProcessAffiliationAuGroup(
                psHighbandObj,
                psAuGroup );

        vEndAuTransaction( psAuGroup );
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
             "eProcessAffiliateAu:E"
             " eResult %u bOk %u\n",
             (UN32)eResult,
             (UN32)bOk );

    return eResult;
}

/*****************************************************************************
*
*   eProcessTableDefAu
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eProcessTableDefAu(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo
        )
{
    DATASERVICE_ERROR_CODE_ENUM eResult = DATASERVICE_ERROR_CODE_NONE;
    BOOLEAN bOk;

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
              "eProcessTableDefAu:S"
              " psAuGroup 0x%x psAuInfo 0x%x\n",
              (UN32)psAuGroup,
              (UN32)psAuInfo );

    // Table defs are single au only the au fields are not present.
    // Make the single au into an au group for uniformity.
    psAuGroup->un16Count = 1;
    psAuGroup->un32Crc = psAuInfo->un32Crc;
    psAuGroup->un8CfgVer = psAuInfo->un8CfgVer;
    psAuGroup->un32AuMask = 1;
    psAuGroup->asAu[0] = *psAuInfo;

    bOk = bProcessTableDefAuGroup(
            psHighbandObj,
            psAuGroup );

    vEndAuTransaction( psAuGroup );

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
            "eProcessTableDefAu:E"
            "bOk %u eResult %u\n",
            (UN32)bOk,
            (UN32)eResult );

    return eResult;
}

/*****************************************************************************
*
*   eProcessDataAu
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eProcessDataAu(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo
        )
{
    BOOLEAN bOk, bParseOk;
    DATASERVICE_ERROR_CODE_ENUM eResult;
    size_t tBitsRead;

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
             "eProcessDataAu:S"
             " psAuGroup 0x%x psAuInfo 0x%x\n",
             (UN32)psAuGroup,
             (UN32)psAuInfo );

//    DEBUG_DumpBuffer(psAuInfo->hPayload);

    bOk = FALSE;
    do
    {
        // Extract the affiliate id
        psAuInfo->un16AfId = 0;
        bParseOk = OSAL.bBufferPeekBitsToUN16(
            psAuInfo->hPayload,
            &psAuInfo->un16AfId,
            SPORTS_SERVICE_AFID_BITLEN,
            psAuInfo->tOffset );

        psAuInfo->tOffset += SPORTS_SERVICE_AFID_BITLEN;

        if( !bParseOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                 "eProcessDataAu:"
                 "Unable to read AFID\n");
            break;
        }

        // Extract the epoch
        psAuInfo->un16Epoch = 0;
        bParseOk = OSAL.bBufferPeekBitsToUN16(
            psAuInfo->hPayload,
            &psAuInfo->un16Epoch,
            SPORTS_SERVICE_EPOCH_BITLEN,
            psAuInfo->tOffset );

        psAuInfo->tOffset += SPORTS_SERVICE_EPOCH_BITLEN;

        if( !bParseOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                 "eProcessDataAu:"
                 "Unable to read EPOCH\n");
            break;
        }

        // Extract the season status
        psAuInfo->un8Season = 0;
        tBitsRead = OSAL.tBufferPeekBits(
            psAuInfo->hPayload,
            &psAuInfo->un8Season,
            0,
            SPORTS_SERVICE_SEASON_BITLEN,
            psAuInfo->tOffset );

        psAuInfo->tOffset += tBitsRead;

        if( tBitsRead != SPORTS_SERVICE_SEASON_BITLEN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                 "eProcessDataAu:"
                 "Unable to read EPOCH\n");
            break;
        }
        bOk = TRUE;
    }
    while(0);

    if( !bOk )
    {
        GsSportsServiceMgrIntf.vReleasePayload( psAuInfo->hPayload );
        vEndAuTransaction( psAuGroup );

        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                 "eProcessDataAu:Error"
                 " bParseOk %u bOk %u\n",
                 (UN32)bParseOk,
                 (UN32)bOk );

        return DATASERVICE_ERROR_CODE_NONE;
    }

    // The au fields are next in the payload
    eResult = eProcessAu(
         psAuGroup,
         psAuInfo,
         bIsDataAuValid );

    bOk = bIsAuGroupComplete( psAuGroup, SS_MAX_DATA_PAYLOAD_SIZE_BITS );

    if( bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                 "eProcessDataAu:Group Complete\n" );
        vPrintAuGroup( psAuGroup, SSDL_HB_AU_PROCESSING );

        bOk = bProcessDataAuGroup(
                psHighbandObj,
                psAuGroup );

        vEndAuTransaction( psAuGroup );
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
             "eProcessDataAu:E"
             " eResult %u bOk %u\n",
             (UN32)eResult,
             (UN32)bOk );

    return eResult;
}

/*****************************************************************************
*
*   eProcessAu
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eProcessAu(
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo,
        SPORTS_SERVICE_AU_VALIDATE_FUNC tbIsAuValid
        )
{
    BOOLEAN bOk;

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
             "eProcessAu:S"
             " psAuGroup 0x%x psAuInfo 0x%x\n",
             (UN32)psAuGroup,
             (UN32)psAuInfo );

    do
    {
        bOk = bParseAuGroupInfo( psAuInfo );
        if( !bOk )
        {
            break;
        }

//        vPrintAuGroup( psAuGroup, SSDL_HB_AU_PROCESSING );
//        vPrintAuInfo( psAuInfo, SSDL_HB_AU_PROCESSING );

        // Range check the au sizes.
        if( psAuInfo->un8CtSizePresent &&
            ((psAuInfo->un16AuTot >= SPORTS_SERVICE_MAX_AU_GROUP_SIZE) ||
            (psAuInfo->un16AuCt > psAuInfo->un16AuTot)) )
        {
            break;
        }

        // No current transaction
        if( psAuGroup->un16Count == 0 )
        {
            bOk = bStartAuTransaction (
                    psAuInfo,
                    psAuGroup );
            if( !bOk )
            {
                break;
            }
        }
        // Have current transaction
        else
        {
            // The packet must:
            //      have same cfg
            //      have same auid
            //      have same total count
            //      be in range of total
            // of the current transaction. Otherwise start over.
            if( (psAuInfo->un8CfgVer != psAuGroup->un8CfgVer) ||
                (psAuInfo->un16AuId != psAuGroup->un16AuId) ||
                ((psAuInfo->un16AuTot+1) != psAuGroup->un16Count) ||
                (psAuInfo->un16AuCt >= psAuGroup->un16Count) ||
                (!tbIsAuValid(psAuGroup,psAuInfo)) )
            {
                // End current transaction
                vEndAuTransaction( psAuGroup );

                // Try to start a new transaction with the new au
                bOk = bStartAuTransaction (
                        psAuInfo,
                        psAuGroup );
                if( !bOk )
                {
                    break;
                }
            }
            else
            {
                // The au is a duplicate
                if( psAuGroup->un32AuMask & (1<<psAuInfo->un16AuCt) )
                {
                    // Bad news crc does not match.  Discard the
                    // transaction and try over.
                    if( psAuGroup->asAu[psAuInfo->un16AuCt].un32Crc !=
                        psAuInfo->un32Crc )
                    {
                        vEndAuTransaction( psAuGroup );

                        bOk = bStartAuTransaction (
                                psAuInfo,
                                psAuGroup );
                        if( !bOk )
                        {
                            break;
                        }
                    }
                }
                // au is new
                else
                {
                    psAuGroup->un32AuMask |= (1<<psAuInfo->un16AuCt);
                    psAuGroup->asAu[psAuInfo->un16AuCt] = *psAuInfo;
                }
            }
        }

        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                 "eProcessAu:E:Au Used"
                 " psAuGroup 0x%x psAuInfo 0x%x\n",
                 (UN32)psAuGroup,
                 (UN32)psAuInfo );

//        vPrintAuGroup( psAuGroup, SSDL_HB_AU_PROCESSING );
//        vPrintAuInfo( psAuInfo, SSDL_HB_AU_PROCESSING );

        return DATASERVICE_ERROR_CODE_NONE;
    }
    while(0);

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
             "eProcessAu:E:Au Dropped"
             " psAuGroup 0x%x psAuInfo 0x%x\n",
             (UN32)psAuGroup,
             (UN32)psAuInfo );

    vPrintAuGroup( psAuGroup, SSDL_HB_AU_PROCESSING );
    vPrintAuInfo( psAuInfo, SSDL_HB_AU_PROCESSING );

     // End any current transaction
    vEndAuTransaction( psAuGroup );

    // Free new payload
    GsSportsServiceMgrIntf.vReleasePayload( psAuInfo->hPayload );

    // Silently exit
    return DATASERVICE_ERROR_CODE_NONE;
}

/*****************************************************************************
*
*   bStartAuTransaction
*
*****************************************************************************/
static BOOLEAN bStartAuTransaction (
        AU_INFO_STRUCT *psAuInfo,
        AU_GROUP_STRUCT *psAuGroup
        )
{
    BOOLEAN bOk = TRUE;

    // Single au table
    if( !psAuInfo->un8CtSizePresent )
    {
        psAuGroup->un16Count = 1;
        psAuGroup->un32Crc = psAuInfo->un32Crc;
        psAuGroup->un8CfgVer = psAuInfo->un8CfgVer;
        psAuGroup->un32AuMask = 1;
        psAuGroup->un8AdVer = psAuInfo->un8AdVer;
        psAuGroup->un16AfId = psAuInfo->un16AfId;
        psAuGroup->un16Epoch = psAuInfo->un16Epoch;
        psAuGroup->un8Season = psAuInfo->un8Season;
        psAuGroup->asAu[0] = *psAuInfo;
    }
    // Multi au table
    else
    {
        // We can only start on the first au
        if(psAuInfo->un16AuCt == 0)
        {
            psAuGroup->un16Count = psAuInfo->un16AuTot + 1;
            psAuGroup->un8CfgVer = psAuInfo->un8CfgVer;
            psAuGroup->un32AuMask = 1;
            psAuGroup->un16AuId = psAuInfo->un16AuId;
            psAuGroup->un8AdVer = psAuInfo->un8AdVer;
            psAuGroup->un16AfId = psAuInfo->un16AfId;
            psAuGroup->un16Epoch = psAuInfo->un16Epoch;
            psAuGroup->un8Season = psAuInfo->un8Season;
            psAuGroup->asAu[0] = *psAuInfo;
        }
        else
        {
            bOk = FALSE;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   vEndAuTransaction
*
*****************************************************************************/
static void vEndAuTransaction( AU_GROUP_STRUCT *psAuGroup )
{
    UN32 un32Au;

    // Memory must have been stepped on.  Limit the range of the damage we are
    // about to do in addition.
    if( psAuGroup->un16Count > SPORTS_SERVICE_MAX_AU_GROUP_SIZE )
    {
        psAuGroup->un16Count = SPORTS_SERVICE_MAX_AU_GROUP_SIZE;
    }

    for( un32Au=0; un32Au<psAuGroup->un16Count; un32Au++ )
    {
        if( psAuGroup->un32AuMask & (1<<un32Au) )
        {
            GsSportsServiceMgrIntf.vReleasePayload(
                    psAuGroup->asAu[un32Au].hPayload );
        }
    }

    psAuGroup->un32AuMask = 0;
    psAuGroup->un16Count = 0;
}

/*****************************************************************************
*
*   bIsAuGroupComplete
*
*****************************************************************************/
static BOOLEAN bIsAuGroupComplete(
        AU_GROUP_STRUCT *psAuGroup,
        size_t tMaxPayloadSize )
{
    UN32 un32Au;
    N32 n32AuPayloadSize;
    size_t tTotalPayloadSize = 0;
    BOOLEAN bComplete = TRUE;
    BOOLEAN bDropAuGroup = FALSE;

    // No transaction is considered not complete.
    if( psAuGroup->un16Count == 0 )
    {
        return FALSE;
    }

    // Memory must have been stepped on.  Limit the range of the iteration
    if( psAuGroup->un16Count > SPORTS_SERVICE_MAX_AU_GROUP_SIZE )
    {
        psAuGroup->un16Count = SPORTS_SERVICE_MAX_AU_GROUP_SIZE;
    }

    for( un32Au=0; un32Au<psAuGroup->un16Count; un32Au++ )
    {
        if( !(psAuGroup->un32AuMask & (1<<un32Au)) )
        {
            bComplete = FALSE;
        }
        else
        {
            // Sum up the size of all au's we have.
            n32AuPayloadSize = psAuGroup->asAu[un32Au].tPayloadStartSizeBits -
                    psAuGroup->asAu[un32Au].tOffset -
                    (size_t)(SPORTS_SERVICE_CRC_BYTELEN * 8);

            if( n32AuPayloadSize < 0 )
            {
                bDropAuGroup = TRUE;
                break;
            }
            else
            {
                tTotalPayloadSize += n32AuPayloadSize;
            }

            if( tTotalPayloadSize > tMaxPayloadSize )
            {
                bDropAuGroup = TRUE;
                break;
            }
        }
    }

    // End the au transaction if we exceeded our size limit.
    if( bDropAuGroup )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE | SSDL_HB_TD_UPDATE
                     | SSDL_HB_AF_UPDATE | SSDL_HB_DI_UPDATE,
                     "bIsAuGroupComplete:"
                     " Payload size zero or max exceeded: Act: %u Max: %u\n",
                     tTotalPayloadSize,
                     tMaxPayloadSize );

        vEndAuTransaction( psAuGroup );
        return FALSE;
    }

    return bComplete;
}

/*****************************************************************************
*
*   bParseAuGroupInfo
*
*****************************************************************************/
static BOOLEAN bParseAuGroupInfo( AU_INFO_STRUCT *psAuInfo )
{
    BOOLEAN bOk;
    size_t tBitsRead;
    size_t tReadCount;
    UN32 un32Remainder;

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
             "bParseAuGroupInfo:S"
             " psAuInfo 0x%x\n",
             (UN32)psAuInfo );

    bOk = FALSE;
    do
    {
        // Parse the presence of multi au fields
        psAuInfo->un8CtSizePresent = 0;
        tBitsRead = OSAL.tBufferPeekBits(
                psAuInfo->hPayload,
                &psAuInfo->un8CtSizePresent,
                0,
                SPORTS_SERVICE_PRESENCE_FLAG_BITLEN,
                psAuInfo->tOffset );

        if( tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                 "bParseAuGroupInfo:"
                 "Unable to read CtSizePresent\n");
            break;
        }

        psAuInfo->tOffset += tBitsRead;

        // Read the multi au details.
        if( psAuInfo->un8CtSizePresent )
        {
            psAuInfo->un8CtSize = 0;
            tBitsRead = OSAL.tBufferPeekBits(
                    psAuInfo->hPayload,
                    &psAuInfo->un8CtSize,
                    0,
                    SPORTS_SERVICE_CTSIZE_BITLEN,
                    psAuInfo->tOffset );

            if( tBitsRead != SPORTS_SERVICE_CTSIZE_BITLEN )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                     "bParseAuGroupInfo:"
                     "Unable to read CtSize\n");
                break;
            }

            psAuInfo->tOffset += tBitsRead;

            tReadCount = psAuInfo->un8CtSize + 1;

            psAuInfo->un16AuTot = 0;
            bOk = OSAL.bBufferPeekBitsToUN16(
                    psAuInfo->hPayload,
                    &psAuInfo->un16AuTot,
                    tReadCount,
                    psAuInfo->tOffset );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                      "bParseAuGroupInfo:"
                      "Unable to read AuTot\n");
                break;
            }

            psAuInfo->tOffset += tReadCount;

            psAuInfo->un16AuCt = 0;
            bOk = OSAL.bBufferPeekBitsToUN16(
                    psAuInfo->hPayload,
                    &psAuInfo->un16AuCt,
                    tReadCount,
                    psAuInfo->tOffset );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                      "bParseAuGroupInfo:"
                      "Unable to read AuCt\n");
                break;
            }

            psAuInfo->tOffset += tReadCount;

            psAuInfo->un16AuId = 0;
            bOk = OSAL.bBufferPeekBitsToUN16(
                    psAuInfo->hPayload,
                    &psAuInfo->un16AuId,
                    SPORTS_SERVICE_AUAID_BITLEN,
                    psAuInfo->tOffset );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
                      "bParseAuGroupInfo:"
                      "Unable to read AuId\n");
                break;
            }

            psAuInfo->tOffset += SPORTS_SERVICE_AUAID_BITLEN;
        }

        // Byte align the offset
        un32Remainder = psAuInfo->tOffset % 8;
        if( un32Remainder )
        {
            psAuInfo->tOffset += (8 - un32Remainder);
        }

        bOk = TRUE;
    }
    while(0);

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AU_PROCESSING,
             "bParseAuGroupInfo:S"
             " bOk %u\n",
             (UN32)bOk );

    return bOk;
}

/*****************************************************************************
*
*   bIsConfigAuValid
*
*****************************************************************************/
static BOOLEAN bIsConfigAuValid(
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo
        )
{
    return TRUE;
}

/*****************************************************************************
*
*   bIsAffiliateAuValid
*
*****************************************************************************/
static BOOLEAN bIsAffiliateAuValid(
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo
        )
{
    return (psAuInfo->un8AdVer == psAuGroup->un8AdVer);
}

/*****************************************************************************
*
*   bIsDataAuValid
*
*****************************************************************************/
static BOOLEAN bIsDataAuValid(
        AU_GROUP_STRUCT *psAuGroup,
        AU_INFO_STRUCT *psAuInfo
        )
{
    return ( (psAuInfo->un16AfId == psAuGroup->un16AfId) &&
             (psAuInfo->un16Epoch == psAuGroup->un16Epoch) &&
             (psAuInfo->un8Season == psAuGroup->un8Season) );
}

/*****************************************************************************
*
*   vPrintAuInfo
*
*****************************************************************************/
static void vPrintAuInfo( AU_INFO_STRUCT *psAuInfo, UN32 un32Level )
{
    SPORTS_SERVICE_MGR_vPrintSilent( un32Level,
            "AuInfo:0x%x\n"
            "   un8Pvn %u\n"
            "   un8CarouselId %u\n"
            "   un8CfgVer %u\n"
            "   un8CtSizePresent %u\n"
            "   un8CtSize %u\n"
            "   un8AdVer %u\n"
            "   un8Season %u\n"
            "   un16AuTot %u\n"
            "   un16AuCt %u\n"
            "   un16AfId %u\n"
            "   un16Epoch %u\n"
            "   un16AuId %u\n"
            "   un32Crc %u\n"
            "   tOffset %d\n"
            "   hPayload 0x%x\n"
            "   tPayloadStartSizeBits %d\n",
            (UN32)psAuInfo,
            (UN32)psAuInfo->un8Pvn,
            (UN32)psAuInfo->un8CarouselId,
            (UN32)psAuInfo->un8CfgVer,
            (UN32)psAuInfo->un8CtSizePresent,
            (UN32)psAuInfo->un8CtSize,
            (UN32)psAuInfo->un8AdVer,
            (UN32)psAuInfo->un8Season,
            (UN32)psAuInfo->un16AuTot,
            (UN32)psAuInfo->un16AuCt,
            (UN32)psAuInfo->un16AfId,
            (UN32)psAuInfo->un16Epoch,
            (UN32)psAuInfo->un16AuId,
            psAuInfo->un32Crc,
            psAuInfo->tOffset,
            (UN32)psAuInfo->hPayload,
            psAuInfo->tPayloadStartSizeBits
    );

    return;
}

/*****************************************************************************
*
*   vPrintAuGroup
*
*****************************************************************************/
static void vPrintAuGroup( AU_GROUP_STRUCT *psAuGroup, UN32 un32Level )
{
    SPORTS_SERVICE_MGR_vPrintSilent( un32Level,
            "AuGroup:0x%x\n"
            "   un16AuId %u\n"
            "   un16Count %u\n"
            "   un32AuMask 0x%x\n"
            "   un16CfgVer %u\n"
            "   un8AdVer %u\n"
            "   un8Season %u\n"
            "   un16AfId %u\n"
            "   un16Epoch %u\n"
            "   un32Crc %u\n",
            (UN32)psAuGroup,
            (UN32)psAuGroup->un16AuId,
            (UN32)psAuGroup->un16Count,
            psAuGroup->un32AuMask,
            (UN32)psAuGroup->un8CfgVer,
            (UN32)psAuGroup->un8AdVer,
            (UN32)psAuGroup->un8Season,
            (UN32)psAuGroup->un16AfId,
            (UN32)psAuGroup->un16Epoch,
            psAuGroup->un32Crc
    );

    return;
}

/*****************************************************************************
*
*   bCalcAuGroupCrc
*
*****************************************************************************/
static BOOLEAN bCalcAuGroupCrc(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup )
{
    BOOLEAN bOk;
    AU_INFO_STRUCT *psAuInfo;
    UN32 un32Au;
    OSAL_CRC_RESULT tCRC;
    size_t tNumBytesToCRC, tBytesProcessed;

    // No transaction is an error.
    if( psAuGroup->un16Count == 0 )
    {
        return FALSE;
    }

    // Optimization, use the existing au crc if there is only one au.
    if( psAuGroup->un16Count == 1 )
    {
        psAuGroup->un32Crc = psAuGroup->asAu[0].un32Crc;
        return TRUE;
    }

    // Memory must have been stepped on.  Limit the range of the iteration
    if( psAuGroup->un16Count > SPORTS_SERVICE_MAX_AU_GROUP_SIZE )
    {
        psAuGroup->un16Count = SPORTS_SERVICE_MAX_AU_GROUP_SIZE;
    }

    // Initialize the CRC
    bOk = OSAL.bInitializeCRC( psHighbandObj->hCRC, &tCRC );
    if (!bOk )
    {
        SMSAPI_DEBUG_vPrintErrorFull(NULL, __LINE__,
            SPORTS_SVC_HB_OBJ_NAME" bCalcAuGroupCrc - CRC init failed\n");
        return FALSE;
    }

    for( un32Au=0; un32Au<psAuGroup->un16Count; un32Au++ )
    {
        if( !(psAuGroup->un32AuMask & (1<<un32Au)) )
        {
            return FALSE;
        }

        psAuInfo = &psAuGroup->asAu[un32Au];

        // This should always be positive since we ranges checked the size
        // minus the crc bytes in bVerifyAu()
        tNumBytesToCRC =
            (psAuInfo->tPayloadStartSizeBits / 8) - SPORTS_SERVICE_CRC_BYTELEN;

        // Compute the CRC
        tCRC = OSAL.tComputeCRC(
            psHighbandObj->hCRC,
            tCRC,
            psAuInfo->hPayload,
            0,
            tNumBytesToCRC,
            &tBytesProcessed );

        if( tBytesProcessed != tNumBytesToCRC )
        {
            SMSAPI_DEBUG_vPrintErrorFull(NULL, __LINE__,
                SPORTS_SVC_HB_OBJ_NAME
                " bCalcAuGroupCrc - unable to process entire payload for crc\n");
            return FALSE;
        }
    }

    // Invert the CRC bits (SX-9845-0180 section 3.3.3)
    tCRC ^= SPORTS_SERVICE_CRC_INVERSION_VALUE;

    psAuGroup->un32Crc = (UN32)tCRC;

    return TRUE;
}

/*****************************************************************************
*
*   bProcessTableDefAuGroup
*
*****************************************************************************/
static BOOLEAN bProcessTableDefAuGroup(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup
        )
{
    UN32 un32Status;
    BOOLEAN bOk;
    AU_INFO_STRUCT *psAuInfo = NULL;
    size_t tBitsRead;
    N32 un32Cnt = 0;
    UN16 un16TdefId=0;
    UN8 un8TdefVersion=0;
    UN8 un8TdefClass=0;
    UN8 un8TdefLabelCount=0;
    SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *psTdefInfo;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_OBJECT_HDL hLabelList;
    SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *psLabelInfo;

    // Nothing to do if the cfg version does not match
    bOk = GsSportsServiceMgrIntf.bIsCorrectServiceConfigVersion(
            psHighbandObj->hSportsDataService,
            psAuGroup->un8CfgVer );

    if( !bOk )
    {
        return TRUE;
    }

    do
    {
        bOk = bCalcAuGroupCrc(
                psHighbandObj,
                psAuGroup );

        if( !bOk )
        {
            un32Status = 0;
            break;
        }

        // Tdefs are single au only.
        psAuInfo = &psAuGroup->asAu[0];

        //DEBUG_DumpBuffer(psAuInfo->hPayload);

        // Seek to the start of the table def bits.
        tBitsRead = OSAL.tBufferSeekHeadBits(
                psAuInfo->hPayload,
                psAuInfo->tOffset );

        if( tBitsRead != psAuInfo->tOffset )
        {
            un32Status = 1;
            break;
        }

        // Read the Table ID from the preamble
        bOk = OSAL.bBufferReadBitsToUN16(
                psAuInfo->hPayload,
                &un16TdefId,
                SPORTS_SERVICE_TABLEID_BITLEN );

        if( !bOk )
        {
            un32Status = 2;
            break;
        }

        // Read the Table Version from the preamble
        tBitsRead = OSAL.tBufferReadHeadBits(
            psAuInfo->hPayload,
            &un8TdefVersion, 0,
            SPORTS_SERVICE_TABVER_BITLEN );

        if( tBitsRead != SPORTS_SERVICE_TABVER_BITLEN )
        {
            un32Status = 3;
            break;
        }

        // Read the Table Class from the preamble
        tBitsRead = OSAL.tBufferReadHeadBits(
                psAuInfo->hPayload,
                &un8TdefClass,
                0,
                SPORTS_SERVICE_TABCLASS_BITLEN );

        if( tBitsRead != SPORTS_SERVICE_TABCLASS_BITLEN )
        {
            un32Status = 4;
            break;
        }

        // Success
        un32Status = 5;
    }
    while(0);

    if( un32Status != 5 )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
            "bProcessTableDefAuGroup:"
            "First Stage Error: un32Status: \n", un32Status );

        return FALSE;
    }

    //
    // See if we already have the tdef
    //

    // Get the memory tdef entry in our array
    psTdefInfo = (SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *)
            &psHighbandObj->hTableDefs[un16TdefId];

    // If execution continues after this block we are keeping the new tdef.
    // The memory version of the tdef will be marked invalid and ready for
    // complete reuse.  Only upon successful processing will it become valid
    // again.  We may have been able to reuse an existing label list.  If so
    // we keep a reference to it indicating no need to create one.
    hLabelList = OSAL_INVALID_OBJECT_HDL;
    if( psTdefInfo->IsValid )
    {
        // The tdef is a duplicate of the current.  Nothing to do
        if( psTdefInfo->CRC == psAuGroup->un32Crc )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
                "bProcessTableDefAuGroup:"
                "Duplicate Tdef: TABID: %u  TABVER: %u  CRC: 0x%X\n",
                (UN32)un16TdefId,
                (UN32)un8TdefVersion,
                psAuGroup->un32Crc );

            //DEBUG_DumpList(psHighbandObj);
            return TRUE;
        }

        // We are going to use the new tdef.  Regardless of what happens the
        // old tdef no longer applies to the input.
        psTdefInfo->IsValid = FALSE;

        // See if we want to keep the new version.
        bOk = GsSportsServiceMgrIntf.bShouldKeepTdef(
                psHighbandObj->hSportsDataService,
                un16TdefId,
                un8TdefClass );

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
                            "bProcessTableDefAuGroup:"
                            "Unsupported Tdef class: TABID: %u  TABVER: %u"
                            " CRC: 0x%X Class: %u\n",
                            (UN32)un16TdefId,
                            (UN32)un8TdefVersion,
                            psAuGroup->un32Crc,
                            un8TdefClass );

            if( psTdefInfo->hLabelInfoList != OSAL_INVALID_OBJECT_HDL )
            {
                OSAL.eLinkedListRemoveAll(
                        psTdefInfo->hLabelInfoList,
                        (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

                OSAL.eLinkedListDelete(
                        psTdefInfo->hLabelInfoList );

                psTdefInfo->hLabelInfoList = OSAL_INVALID_OBJECT_HDL;
            }
            return TRUE;
        }

        // Try to reuse the label linked list
        if( psTdefInfo->hLabelInfoList != OSAL_INVALID_OBJECT_HDL )
        {
            eOsalCode = OSAL.eLinkedListRemoveAll(
                    psTdefInfo->hLabelInfoList,
                    (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

            if( eOsalCode == OSAL_SUCCESS )
            {
                hLabelList = psTdefInfo->hLabelInfoList;
            }
            else
            {
                // We do not know the state of the list; dump it
                (void)OSAL.eLinkedListDelete( psTdefInfo->hLabelInfoList );
            }

            // We either kept a reference to this or deleted it.  Invalidate
            // it so out side of this function is it either invalid or points
            // allocated memory.
            psTdefInfo->hLabelInfoList = OSAL_INVALID_OBJECT_HDL;
        }
    }
    else
    {
        // We do not have the tdef.  See if we want to keep it.
        bOk = GsSportsServiceMgrIntf.bShouldKeepTdef(
                psHighbandObj->hSportsDataService,
                un16TdefId,
                un8TdefClass );

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
                            "bProcessTableDefAuGroup:"
                            "Unsupported Tdef class: TABID: %u  TABVER: %u"
                            " CRC: 0x%X Class: %u\n",
                            (UN32)un16TdefId,
                            (UN32)un8TdefVersion,
                            psAuGroup->un32Crc,
                            un8TdefClass );
            return TRUE;
        }
    }

    //
    // We have invalidated the current tdef and are going to replace it.
    //

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
        "bProcessTableDefAuGroup:"
        "Gathering new tDef: TABID:%u TABVER:%u CRC:0x%X hLabelList:0x%X\n",
        (UN32)un16TdefId,
        (UN32)un8TdefVersion,
        psAuGroup->un32Crc,
        (UN32)hLabelList );

    // We need to allocate a list for labels
    if( hLabelList == OSAL_INVALID_OBJECT_HDL )
    {
        // Create a linked list for the label info and populate it
        eOsalCode = OSAL.eLinkedListCreate(
                &hLabelList,
                SPORTS_SVC_HB_OBJ_NAME":TDL",
                NULL,   // Labels must be added to the list in the order they are encountered
                OSAL_LL_OPTION_LINEAR );

        if( eOsalCode != OSAL_SUCCESS )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
                "bProcessTableDefAuGroup:"
                "Failed to allocate label list: TABID: %u  TABVER: %u  CRC: 0x%X\n",
                (UN32)un16TdefId,
                (UN32)un8TdefVersion,
                psAuGroup->un32Crc );

            return FALSE;
        }
    }

    //
    // We are going to save this tdef.  Get the rest of the info.
    //

    do
    {
        if(psAuInfo == NULL)
        {
            break;
        }

        // Skip the stuffing bits
        tBitsRead = OSAL.tBufferSeekHeadBits(
            psAuInfo->hPayload,
            SPORTS_SERVICE_TABSTUFF_BITLEN);

        if( tBitsRead != SPORTS_SERVICE_TABSTUFF_BITLEN )
        {
            un32Status = 1;
            break;
        }

        // we are now byte-aligned to the start of the table definitions
        // Create a new TdefInfo instance and point that to our array
        // of label definitions
        // Now start to parse the label info.  Get the label count first then iterate for each label
        tBitsRead = OSAL.tBufferReadHeadBits(
                psAuInfo->hPayload,
                &un8TdefLabelCount,
                0,
                SPORTS_SERVICE_LBNO_BITLEN );

        if( tBitsRead != SPORTS_SERVICE_LBNO_BITLEN )
        {
            un32Status = 2;
            break;
        }

        un8TdefLabelCount++;    // table def holds count-1

        for( un32Cnt=0; un32Cnt<un8TdefLabelCount; un32Cnt++)
        {
            // create a new LabelInfo object that we will add to the linked list
            psLabelInfo = (SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *)SMSO_hCreate(
                    SPORTS_SVC_HB_OBJ_NAME":LBL",
                    sizeof(SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT),
                    SMS_INVALID_OBJECT,
                    FALSE );

            if( psLabelInfo == NULL )
            {
                un32Status = 3;
                break;
            }

            // SMSO_hCreate does this for us.
            //OSAL.bMemSet(pLabelInfo, 0, sizeof(SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT));

            // populate the label info struct
            bOk = bParseLabelInfoToObject(
                    psAuInfo->hPayload,
                    psLabelInfo );

            if( !bOk )
            {
                un32Status = 4;
                break;
            }

            // Now add the label info to the linked list
            eOsalCode = OSAL.eLinkedListAdd(
                    hLabelList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    (void *)psLabelInfo );

            if( eOsalCode != OSAL_SUCCESS )
            {
                un32Status = 5;
                break;
            }
        }

        if( un32Cnt != un8TdefLabelCount )
        {
            break;
        }

        // Successful parsing if we get here.  We have a valid tdef in memory
        psTdefInfo->un8TableVer = un8TdefVersion;
        psTdefInfo->un8TableClass = un8TdefClass;
        psTdefInfo->un8LabelCount = un8TdefLabelCount;
        psTdefInfo->CRC = psAuGroup->un32Crc;
        psTdefInfo->hLabelInfoList = hLabelList;
        psTdefInfo->IsValid = TRUE;

        // Try and save it to the db.
        bOk = GsSportsServiceMgrIntf.bAddTableDefToDB(
                psHighbandObj->hSportsDataService,
                psTdefInfo,
                un16TdefId );

        if( !bOk )
        {
            un32Status = 6;
            break;
        }

        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
                "bProcessTableDefAuGroup:"
                "New Table Def: %u ver: %u  Class: %u  LblCnt: %u CRC: 0x%X\n",
                (UN32)un16TdefId,
                (UN32)un8TdefVersion,
                (UN32)un8TdefClass,
                (UN32)un8TdefLabelCount,
                psAuGroup->un32Crc );

//        DEBUG_DumpTableDef(psHighbandObj, un16TdefId);

        return TRUE;
    }
    while(0);

    //
    // Failure if we get here.
    //

    psTdefInfo->IsValid = FALSE;
    psTdefInfo->hLabelInfoList = OSAL_INVALID_OBJECT_HDL;
    (void) OSAL.eLinkedListRemoveAll(
                      hLabelList,
                      (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );
    (void) OSAL.eLinkedListDelete( hLabelList );

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
            "bProcessTableDefAuGroup:"
            "Second Stage Error: TableID: %u  TabVer: %u CRC: 0x%X"
            " un32Status %u un32Cnt %d\n",
            (UN32)un16TdefId,
            (UN32)un8TdefVersion,
            psAuGroup->un32Crc,
            un32Status,
            un32Cnt );

    return FALSE;
 }

/*****************************************************************************
*
*   bProcessConfigAuGroup
*
*****************************************************************************/
static BOOLEAN bProcessConfigAuGroup(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup
        )
{
    BOOLEAN bOk;
    UN32 un32Au;

    bOk = bCalcAuGroupCrc(
            psHighbandObj,
            psAuGroup );

    if( !bOk )
    {
        return TRUE;
    }

    // See if we are going to update.  True means no change.
    bOk = GsSportsServiceMgrIntf.bStartServiceConfigUpdate(
         (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
         psAuGroup->un8CfgVer,
         psAuGroup->un32Crc );

    if( bOk )
    {
        return TRUE;
    }

    // since the DB was just cleared of all TD and DI info we
    // must also reset our in-memory table defs
    bClearAllTableDefs(psHighbandObj);

    bOk = FALSE;
    for( un32Au=0; un32Au<psAuGroup->un16Count; un32Au++ )
    {
        bOk = bParseConfigAu(
                psHighbandObj,
                &psAuGroup->asAu[un32Au] );

        if( !bOk )
        {
            break;
        }
    }

    GsSportsServiceMgrIntf.bEndServiceConfigUpdate(
             (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
             bOk,
             psAuGroup->un8CfgVer,
             psAuGroup->un32Crc );

    return TRUE;
}

/*****************************************************************************
*
*   bParseConfigAu
*
*****************************************************************************/
static BOOLEAN bParseConfigAu(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_INFO_STRUCT *psAuInfo
        )
{
    size_t tBitsRead;
    UN8 un8SportCount = 0;
    UN8 un8TableCount;
    UN8 un8SportID;
    UN16 un16TableID;
    UN8 n,m;
    BOOLEAN bOk;
    STRING_OBJECT hSportNameString = STRING_INVALID_OBJECT;
    size_t tNumSymbolsFound;

    // Initialize to success.  This handles a zero sport config properly.
    bOk = TRUE;
    do{
//        DEBUG_DumpBuffer( psAuInfo->hPayload );

        // Seek to the start of the sport table bits.
        tBitsRead = OSAL.tBufferSeekHeadBits(
                psAuInfo->hPayload,
                psAuInfo->tOffset );

        if( tBitsRead != psAuInfo->tOffset )
        {
            bOk = FALSE;
            break;
        }

        // Read the Sport count from the preamble

        tBitsRead = OSAL.tBufferReadHeadBits(
                psAuInfo->hPayload,
                &un8SportCount,
                0,
                SPORTS_SERVICE_SPORTCOUNT_BITLEN );

        if( tBitsRead != SPORTS_SERVICE_SPORTCOUNT_BITLEN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                    "bParseConfigAu:"
                    "Unable to read SportCount\n");

            bOk = FALSE;
            break;
        }

        un8SportCount++;

        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                "bParseConfigAu:"
                "SPORTS CONFIG: %u sports available:\n",
                (UN32)un8SportCount );

        for (n=0; n < un8SportCount; n++)
        {
            // read the sports name
            tNumSymbolsFound = 0;
            hSportNameString = BAUDOT_hToString(
                SMS_INVALID_OBJECT,
                psAuInfo->hPayload,
                BAUDOT_BEHAVIOR_PROCESS_TO_END,
                TRUE, TRUE,
                SS_MAX_SPORT_NAME_SYMBOL_COUNT,
                &tNumSymbolsFound );

            if( hSportNameString == STRING_INVALID_OBJECT )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                        "bParseConfigAu:"
                        "Failed to read SPORTNAME string\n");
                bOk = FALSE;
                break;
            }

            if( tNumSymbolsFound > (size_t)SS_MAX_SPORT_NAME_SYMBOL_COUNT )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                            "bParseConfigAu:"
                            "Failed to read SPORTNAME string too many symbols:%u\n",
                            tNumSymbolsFound );

                STRING_vDestroy(hSportNameString);
                bOk = FALSE;
                break;
            }

			un8SportID = 0;

            // get the sport ID
            tBitsRead = OSAL.tBufferReadHeadBits(
                    psAuInfo->hPayload,
                    &un8SportID,
                    0,
                    SPORTS_SERVICE_SPORTID_BITLEN );

            if( tBitsRead != SPORTS_SERVICE_SPORTID_BITLEN )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                        "bParseConfigAu:"
                        "Unable to read Sport ID\n");
                STRING_vDestroy(hSportNameString);
                bOk = FALSE;
                break;
            }

            // create an entry for this sport ID and name
            SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_SC_UPDATE,
                    "%-5u %-33s",
                    (UN32)un8SportID,
                    STRING.pacCStr(hSportNameString));

            bOk = GsSportsServiceMgrIntf.bAddSportsNameToDB(
                        (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                        hSportNameString,
                        un8SportID);

            STRING_vDestroy(hSportNameString);

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                        "bParseConfigAu:"
                        "SC_SportsNames table write failed\n");
                break;
            }

            // get the number of tables associated with the sport
            // WE CAN SKIP THESE BITS - NOT NEEDED FOR SMS IMPLEMENTATION
            un8TableCount = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                    psAuInfo->hPayload,
                    &un8TableCount,
                    0,
                    SPORTS_SERVICE_TABLENO_BITLEN );

            if( tBitsRead != SPORTS_SERVICE_TABLENO_BITLEN )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                        "bParseConfigAu:"
                        "Unable to read sports table count\n");
                bOk = FALSE;
                break;
            }

            un8TableCount++;

            // for each table association
            // create an entry in the TD_Info table
            SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_SC_UPDATE,
                    " %-5u tables: [", (UN32)un8TableCount);

            for(m=0; m < un8TableCount; m++)
            {
				un16TableID = 0;

                // get the number of tables associated with the sport
                bOk = OSAL.bBufferReadBitsToUN16(
                        psAuInfo->hPayload,
                        &un16TableID,
                        SPORTS_SERVICE_TABLEID_BITLEN);

                if( !bOk )
                {
                    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                            "bParseConfigAu:"
                            "Unable to read sports table ID\n");
                    break;

                }

                SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_SC_UPDATE,
                        " %d", (UN32)un16TableID );
                //------------------- AddTableRef(un8SportID, un16TableID); we do not need to save these

                bOk = GsSportsServiceMgrIntf.bAddTdefSportMappingToDB(
                        (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                        un16TableID,
                        un8SportID );

                if( !bOk )
                {
                    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_SC_UPDATE,
                                        "bParseConfigAu:"
                                        "Unable to add tdef sport mapping to db\n");
                    break;
                }
            }

            // catch failures from inner for loop
            if( !bOk )
            {
                break;
            }

            SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_SC_UPDATE,
                    " ]\n");
        }
        // catch failures from outer for loop
        if( !bOk )
        {
            break;
        }

     } while (0);

     // succeed or quietly fail
     return bOk;
}

/*****************************************************************************
*
*   bProcessAffiliationAuGroup
*
*****************************************************************************/
static BOOLEAN bProcessAffiliationAuGroup(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup
        )
{
    BOOLEAN bOk;
    UN32 un32Au;

    // Nothing to do if the cfg version does not match
    bOk = GsSportsServiceMgrIntf.bIsCorrectServiceConfigVersion(
            psHighbandObj->hSportsDataService,
            psAuGroup->un8CfgVer );

    if( !bOk )
    {
        return TRUE;
    }

    bOk = bCalcAuGroupCrc(
            psHighbandObj,
            psAuGroup );

    if( !bOk )
    {
        return TRUE;
    }

    // check the affiliation version matches
    // if it doesn't, or the cRC fails, discard the current AF tables
    bOk = GsSportsServiceMgrIntf.bStartSportAffiliationUpdate(
            (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
            psAuGroup->un8AdVer,
            psAuGroup->un32Crc );

    if( bOk )
    {
        return TRUE;
    }

    bOk = FALSE;
    for( un32Au=0; un32Au<psAuGroup->un16Count; un32Au++ )
    {
        bOk = bParseAffiliationAu(
                psHighbandObj,
                &psAuGroup->asAu[un32Au] );

        if( !bOk )
        {
            break;
        }
    }

    GsSportsServiceMgrIntf.bEndSportAffiliationUpdate(
            (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
            bOk,
            psAuGroup->un8AdVer,
            psAuGroup->un32Crc );

    return TRUE;
}

/*****************************************************************************
*
*   bParseAffiliationAu
*
*****************************************************************************/
static BOOLEAN bParseAffiliationAu(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_INFO_STRUCT *psAuInfo
        )
{
    size_t tBitsRead;
    UN16 un16AffiliationCount = 0;
    UN8 un8SportID=0;
    UN16 un16AffiliateId=0;
    UN8 un8GDRef=0;
    UN16 n;
    BOOLEAN bPresBit=0;
    BOOLEAN bOk = TRUE;
    STRING_OBJECT hSportAffiliationString = STRING_INVALID_OBJECT;
    size_t tNumSymbolsFound;

    do
    {
        //DEBUG_DumpBuffer(psAuInfo->hPayload);

        // Seek to the start of the affiliation table bits.
        tBitsRead = OSAL.tBufferSeekHeadBits(
                psAuInfo->hPayload,
                psAuInfo->tOffset );

        if( tBitsRead != psAuInfo->tOffset )
        {
            bOk = FALSE;
            break;
        }

        // Read the Affiliation count from the preamble
        bOk = OSAL.bBufferReadBitsToUN16(
                            psAuInfo->hPayload,
                            &un16AffiliationCount,
                            SPORTS_SERVICE_ADNO_BITLEN );
        if (!bOk)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                    "bParseAffiliationAu:"
                    "Unable to read Affiliation Count\n");
            break;
        }
        un16AffiliationCount++; // def carries one less

        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                "bParseAffiliationAu:"
                "SPORTS CONFIG: %u Affiliations available:\n",
                (UN32)un16AffiliationCount );

        SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_AF_UPDATE,
                 "%-5s %-5s %-5s %-5s\n",
                 "AfId",
                 "SpId",
                 "GDRef",
                 "Name");

        bOk = TRUE;
        for (n=0; n < un16AffiliationCount; n++)
        {
            // read the affiliate name
            tNumSymbolsFound = 0;
            hSportAffiliationString = BAUDOT_hToString(
                SMS_INVALID_OBJECT,
                psAuInfo->hPayload,
                BAUDOT_BEHAVIOR_PROCESS_TO_END,
                TRUE, TRUE,
                SS_MAX_AFFILIATION_NAME_SYMBOL_COUNT,
                &tNumSymbolsFound );

            if( hSportAffiliationString == STRING_INVALID_OBJECT )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                        "bParseAffiliationAu:"
                        "Failed to read ADNAME string\n");
                bOk = FALSE;
                break;
            }

            if( tNumSymbolsFound > (size_t)SS_MAX_AFFILIATION_NAME_SYMBOL_COUNT )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                        "bParseAffiliationAu:"
                        "Failed to read ADNAME string too many symbols:%u\n",
                        tNumSymbolsFound );

                bOk = FALSE;
                break;
            }
            // get the Affiliate ID
            bOk = OSAL.bBufferReadBitsToUN16(
                                psAuInfo->hPayload,
                                &un16AffiliateId,
                                SPORTS_SERVICE_AFID_BITLEN );
            if (!bOk)
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                        "bParseAffiliationAu:"
                        "Unable to read Affiliate ID\n");
                break;
            }

            // Get the GDREF if present
            tBitsRead = OSAL.tBufferReadHeadBits(
                psAuInfo->hPayload,
                &bPresBit,
                0,
                SPORTS_SERVICE_PRESENCE_FLAG_BITLEN );

            if( tBitsRead == SPORTS_SERVICE_PRESENCE_FLAG_BITLEN )
            {
                // Is the GDREF present?
                if( bPresBit == 1 )
               {
                    // Read the un8GDRef
                    tBitsRead = OSAL.tBufferReadHeadBits(
                            psAuInfo->hPayload,
                            &un8GDRef,
                            0,
                            SPORTS_SERVICE_GDREF_BITLEN );

                    if(tBitsRead != SPORTS_SERVICE_GDREF_BITLEN )
                    {
                        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                                "bParseAffiliationAu:"
                                "Failed to read GDRef\n");
                        bOk = FALSE;
                        break;
                    }
               }
            }
            else
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                        "bParseAffiliationAu:"
                        "Failed to read GDRef presence\n");
                bOk = FALSE;
                break;
            }

            // get the sport ID
            tBitsRead = OSAL.tBufferReadHeadBits(
                    psAuInfo->hPayload,
                    &un8SportID,
                    0,
                    SPORTS_SERVICE_SPORTID_BITLEN );

            if( tBitsRead != SPORTS_SERVICE_SPORTID_BITLEN )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                        "bParseAffiliationAu:"
                        "Unable to read Sport ID\n");
                bOk = FALSE;
                break;    // quietly drop it on the floor
            }

            // Helpful debug info
            if (bPresBit)
            {
                SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_AF_UPDATE,
                        "%-5u %-5u %-5u %s\n",
                        (UN32)un16AffiliateId,
                        (UN32)un8SportID,
                        (UN32)un8GDRef,
                        STRING.pacCStr(hSportAffiliationString));
            }
            else
            {
                SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_AF_UPDATE,
                    "%-5u %-5u ""     "" %s\n",
                    (UN32)un16AffiliateId,
                    (UN32)un8SportID,
                    STRING.pacCStr(hSportAffiliationString));
            }

            // create an entry for this sport ID and name
            bOk = GsSportsServiceMgrIntf.bAddSportsAffiliationToDB(
                        (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                        hSportAffiliationString,
                        un16AffiliateId,
                        bPresBit,
                        un8GDRef,
                        un8SportID);

            STRING_vDestroy(hSportAffiliationString);
            hSportAffiliationString = STRING_INVALID_OBJECT;

            if (!bOk)
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_AF_UPDATE,
                        "bParseAffiliationAu:"
                        "Sport Affiliation table write failed.\n");
                break;
            }
        }

        if (!bOk)   // catch fallout from FOR loop
        {
            if( hSportAffiliationString != STRING_INVALID_OBJECT )
            {
                STRING_vDestroy(hSportAffiliationString);
            }
            break;
        }

    } while (0);

    return bOk;
}

/*****************************************************************************
*
*   bParseLabelInfoToObject
*
*   This function parses data from the current position in phPayload
*   and parses it in to a label info struct
*
*****************************************************************************/
static BOOLEAN bParseLabelInfoToObject(
		OSAL_BUFFER_HDL hPayload,
		SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *psLabelInfo
        )
{
	BOOLEAN bOk;
	size_t tBitsRead;
	STRING_OBJECT hScratchString = STRING_INVALID_OBJECT;
	size_t tNumSymbolsFound;
    UN8 un8ExtDataCnt = 0;

	// Label summary:
	// LBID(8), DTYPE(1), LBTYPE(8), [F+DMOD(1+3)], LPRIORITY(3), LBSIZE(12)
	// [F+LBTXT(1+VAR)], [F+EXTCNT(1+8), EXTDATA]
	//OSAL_ERROR_CANNOT_ADD_OBJECT

	//=====================
	// Read the LabelId
	//=====================
	tBitsRead = OSAL.tBufferReadHeadBits(
			hPayload, &(psLabelInfo->un8LabelId), 0,
			SPORTS_SERVICE_LABEL_LABELID_BITLEN);
	if (tBitsRead != SPORTS_SERVICE_LABEL_LABELID_BITLEN)
	{
	    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
	            "bParseLabelInfoToObject:"
	            "Unable to read LabelID\n");
		return FALSE;
	}

	//=====================
	// Read the Data Type
	//=====================
	tBitsRead = OSAL.tBufferReadHeadBits(
			hPayload, &(psLabelInfo->un8Dtype), 0,
			SPORTS_SERVICE_LABEL_DTYPE_BITLEN);
	if (tBitsRead != SPORTS_SERVICE_LABEL_DTYPE_BITLEN)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "Unable to read DTYPE\n");
		return FALSE;
	}

	//=====================
	// Read the Label Type
	//=====================
	tBitsRead = OSAL.tBufferReadHeadBits(
			hPayload, &(psLabelInfo->un8LbType), 0,
			SPORTS_SERVICE_LABEL_LBTYPE_BITLEN);
	if (tBitsRead != SPORTS_SERVICE_LABEL_LBTYPE_BITLEN)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "Unable to read Label Type\n");
		return FALSE;
	}

	//=====================
	// Read the DMOD if present
	//=====================
	tBitsRead = OSAL.tBufferReadHeadBits(
			hPayload, &(psLabelInfo->bIsDmodPresent), 0,
			SPORTS_SERVICE_PRESENCE_FLAG_BITLEN);
	if (tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "Unable to read DMOD presence\n");
		return FALSE;
	} else if (psLabelInfo->bIsDmodPresent)
	{
		tBitsRead = OSAL.tBufferReadHeadBits(
				hPayload, &(psLabelInfo->un8Dmod), 0,
				SPORTS_SERVICE_LABEL_DMOD_BITLEN);
		if (tBitsRead != SPORTS_SERVICE_LABEL_DMOD_BITLEN)
		{
	        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
	                "bParseLabelInfoToObject:"
	                "Unable to read Label Type\n");
			return FALSE;
		}
	}

	//=====================
	// Get Label Priority
	//=====================
	tBitsRead = OSAL.tBufferReadHeadBits(
			hPayload, &(psLabelInfo->un8LPriority), 0,
			SPORTS_SERVICE_LABEL_LPRIORITY_BITLEN);
	if (tBitsRead != SPORTS_SERVICE_LABEL_LPRIORITY_BITLEN)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "Unable to read Label Priority\n");
		return FALSE;
	}

	//=====================
	// Get Label Size
	//=====================
	bOk = OSAL.bBufferReadBitsToUN16(
			hPayload, &(psLabelInfo->un16LabelSize),
			SPORTS_SERVICE_LABEL_LBSIZE_BITLEN);
	if (!bOk)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "Unable to read Label Size\n");
		return FALSE;
	}
	psLabelInfo->un16LabelSize++;	// one less is stored

	//=====================
	// Read the LBTXT if present
	//=====================
	tBitsRead = OSAL.tBufferReadHeadBits(
			hPayload, &(psLabelInfo->bIsLabelTxtPresent), 0,
			SPORTS_SERVICE_PRESENCE_FLAG_BITLEN);
	if (tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "Unable to read Label Text presence\n");
		return FALSE;
	} else if (psLabelInfo->bIsLabelTxtPresent)
	{

		hScratchString = BAUDOT_hToString(
				SMS_INVALID_OBJECT,
				hPayload,
				BAUDOT_BEHAVIOR_PROCESS_TO_END,
                TRUE, TRUE,
                SS_MAX_LABEL_TEXT_SYMBOL_COUNT,
                &tNumSymbolsFound );

		if( hScratchString == STRING_INVALID_OBJECT )
		{
	        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
	                "bParseLabelInfoToObject:"
	                "Failed to create LBTXT string\n");
			return FALSE;
		}

		if( tNumSymbolsFound > (size_t)SS_MAX_LABEL_TEXT_SYMBOL_COUNT )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bParseLabelInfoToObject:"
                    "LBTXT has too many symbols:%u\n",
                    tNumSymbolsFound );
            STRING_vDestroy(hScratchString);
            return FALSE;
        }

		// copy the string in to our object's char buffer
		// if it fails or the data is truncated, continue anyway
		STRING.tCopyToCStr(
				hScratchString,
				&psLabelInfo->LabelText[0],
				sizeof(psLabelInfo->LabelText)
			);
        STRING_vDestroy(hScratchString);
    }

	//=====================
	// Read the EXTDATA if present
	//=====================
	tBitsRead = OSAL.tBufferReadHeadBits(
			hPayload, &(psLabelInfo->bIsExtDataPresent), 0,
			SPORTS_SERVICE_PRESENCE_FLAG_BITLEN);
	if (tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "Unable to read ExtData presence\n");
		return FALSE;
	} else if (psLabelInfo->bIsExtDataPresent)
	{
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                "bParseLabelInfoToObject:"
                "EXTDATA not yet handled\n");

		// TODO: We must retrieve the EXTDATA if it is there
		// For now, we just skip over the data
		tBitsRead = OSAL.tBufferReadHeadBits(
		            hPayload,
		            &un8ExtDataCnt,
		            0,
		            SPORTS_SERVICE_LABEL_EXTCNT_BITLEN);
        if (tBitsRead != SPORTS_SERVICE_LABEL_EXTCNT_BITLEN)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bParseLabelInfoToObject:"
                    "Unable to read ExtData byte count\n");
            return FALSE;
        }
		// skip over the EXTDATA bits
        tBitsRead = OSAL.tBufferSeekHeadBits( hPayload, (un8ExtDataCnt+1)*8 );
        if ( tBitsRead != (unsigned)((un8ExtDataCnt+1)*8) )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_PARSING,
                    "bParseLabelInfoToObject:"
                    "Unable to skip over ExtData\n");
            return FALSE;
        }
	}

	//
	// Validate the label based on spec constraints.
	//

	SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TD_UPDATE,
	                    "Label:"
	                    " LBID: %u"
	                    " DTYPE: %u"
	                    " LBTYPE: %u"
	                    " F+DEMOD: %u %u"
	                    " LPRIORITY: %u"
	                    " LBSIZE: %u"
	                    " F+LBTXT: %u %s"
	                    " F+EXTCNT: %u %u\n",
	                    (UN32)psLabelInfo->un8LabelId,
	                    (UN32)psLabelInfo->un8Dtype,
	                    (UN32)psLabelInfo->un8LbType,
	                    (UN32)psLabelInfo->bIsDmodPresent,
	                    (UN32)psLabelInfo->un8Dmod,
	                    (UN32)psLabelInfo->un8LPriority,
	                    (UN32)psLabelInfo->un16LabelSize-1,
	                    (UN32)psLabelInfo->bIsLabelTxtPresent,
	                    (psLabelInfo->bIsLabelTxtPresent) ? psLabelInfo->LabelText : "",
	                    (UN32)psLabelInfo->bIsExtDataPresent,
	                    un8ExtDataCnt
	                    );

	// Labels not known at the time of development cannot be validated.
	if( psLabelInfo->un8LbType >=
	    (sizeof(GsLabelVerificationTable) / sizeof(GsLabelVerificationTable[0])) )
	{
	    return TRUE;
	}

	// Labels that do not support dmod cannot have dmod set.
	if( (GsLabelVerificationTable[psLabelInfo->un8LbType].un8DMod == SS_LABEL_SUPPORT_DMOD_NO) &&
	        psLabelInfo->bIsDmodPresent )
	{
	    return FALSE;
	}

	// Validate data types.
	if( GsLabelVerificationTable[psLabelInfo->un8LbType].un8DType != psLabelInfo->un8Dtype )
	{
	    return FALSE;
	}

	//
	// Validate lbsize against the hard limits.
	//

	// Trefs are special.  They must be an exact size.
	if( psLabelInfo->un8LbType == SPORTS_LABEL_TYPE_TABLEREF )
	{
	    if( psLabelInfo->un16LabelSize !=
	        GsLabelVerificationTable[SPORTS_LABEL_TYPE_TABLEREF].un16LbSize )
	    {
	        return FALSE;
	    }
	}
	else
	{
        if( psLabelInfo->un16LabelSize >
            GsLabelVerificationTable[psLabelInfo->un8LbType].un16LbSize )
        {
            return FALSE;
        }
	}
	return TRUE;
}

/*****************************************************************************
*
*   bProcessDataAuGroup
*
*****************************************************************************/
static BOOLEAN bProcessDataAuGroup(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup
        )
{
    BOOLEAN bOk;
    UN8 un8Byte;
    UN32 un32Cnt;
    size_t tIntervalEnd;
    size_t tIntervalOffset;
    size_t tBitsRead;
    size_t tBytesRead;
    size_t tBytesWritten;
    size_t tBitsConsumed;
    N32 n32CopyCount;
    AU_TREF_MAPPING_STRUCT asAuTrefMap[SPORTS_SERVICE_MAX_AU_GROUP_SIZE];
    AU_INFO_STRUCT *psAuInfo;
    OSAL_BUFFER_HDL hPayload;

    // Nothing to do if the cfg version does not match
    bOk = GsSportsServiceMgrIntf.bIsCorrectServiceConfigVersion(
            psHighbandObj->hSportsDataService,
            psAuGroup->un8CfgVer );

    if( !bOk )
    {
        return TRUE;
    }

    bOk = bCalcAuGroupCrc(
            psHighbandObj,
            psAuGroup );

    if( !bOk )
    {
        return TRUE;
    }

    // Make a continuous payload using the first au as a start point.
    // Note that crc bytes of each au have already been removed by bVerifyAu.

    // Prep the first au.
    psAuInfo = &psAuGroup->asAu[0];
    hPayload = psAuInfo->hPayload;

    // Set up the first interval of the mapping.
    tIntervalEnd = psAuInfo->tPayloadStartSizeBits - (SPORTS_SERVICE_CRC_BYTELEN * 8);
    tIntervalOffset = 0;

    asAuTrefMap[0].tIntervalEnd = tIntervalEnd;
    asAuTrefMap[0].tOffset = tIntervalOffset;

    // Drop the au header and concatenate the remaining access units.
    for(un32Cnt = 0; un32Cnt<psAuGroup->un16Count; un32Cnt++)
    {
        if( un32Cnt == 0 )
        {
            continue;
        }

        psAuInfo = &psAuGroup->asAu[un32Cnt];

        // Drop the au header bits.
        tBitsRead = OSAL.tBufferSeekHeadBits(
                psAuInfo->hPayload,
                psAuInfo->tOffset );

        if( tBitsRead != psAuInfo->tOffset )
        {
            break;
        }

        // Copy the remaining bytes from this au.
        n32CopyCount = ((psAuInfo->tPayloadStartSizeBits - psAuInfo->tOffset) / 8) -
                       SPORTS_SERVICE_CRC_BYTELEN;

        while( n32CopyCount > 0)
        {
            tBytesRead = OSAL.tBufferReadHead(
                            psAuInfo->hPayload,
                            &un8Byte,
                            1 );

            if( tBytesRead != 1 )
            {
                break;
            }

            tBytesWritten = OSAL.tBufferWriteTail(
                        hPayload,
                        &un8Byte,
                        1 );

            if( tBytesWritten != tBytesRead )
            {
                break;
            }

            n32CopyCount--;
        }

        if( n32CopyCount != 0 )
        {
            break;
        }

        tIntervalEnd += psAuInfo->tPayloadStartSizeBits;
        tIntervalOffset +=
                (psAuInfo->tOffset + SPORTS_SERVICE_CRC_BYTELEN * 8);

        asAuTrefMap[un32Cnt].tIntervalEnd = tIntervalEnd;
        asAuTrefMap[un32Cnt].tOffset = tIntervalOffset;
    }

    if( un32Cnt == psAuGroup->un16Count )
    {
        UN8 un8ClassId;

        // Drop the au header bits.
        tBitsRead = OSAL.tBufferSeekHeadBits(
                psAuGroup->asAu[0].hPayload,
                psAuGroup->asAu[0].tOffset );

        if( tBitsRead != psAuGroup->asAu[0].tOffset )
        {
            return FALSE;
        }

//        DEBUG_DumpBuffer(psAuGroup->asAu[0].hPayload);

        tBitsConsumed = psAuGroup->asAu[0].tOffset;

        // process the primary table, which may require recursive processing
        // of referenced tables.  Because of the ref'd tables, we need to also
        // provide how many bits have been consumed from the start of the AU.
        // This relies on broadcast only ref'ing one layer deep - otherwise
        // we could get in to stack trouble
        bOk = bParseDataInstance(
             psHighbandObj,
             psAuGroup,
             asAuTrefMap,
             &tBitsConsumed,
             &un8ClassId,
             NULL,
             TRUE );

        return bOk;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bGetTrefAdjustment
*
*****************************************************************************/
static BOOLEAN bGetTrefAdjustment(
        AU_TREF_MAPPING_STRUCT *psTrefMap,
        UN32 un32MapEntryCount,
        size_t tTrefOffset,
        size_t *ptResult
        )
{
    BOOLEAN bReturn = FALSE;
    UN32 un32Count;

    for(un32Count=0; un32Count<un32MapEntryCount; un32Count++)
    {
        if( tTrefOffset < psTrefMap[un32Count].tIntervalEnd )
        {
            *ptResult = psTrefMap[un32Count].tOffset;
            bReturn = TRUE;
            break;
        }
    }
    return bReturn;
}

/*****************************************************************************
 * bParseDataInstance
 *
 * Data is byte-aligned at this point, with phPayload pointing
 * to the start of the Table Instance Header.  This function may be called
 * recursively when handling the primary table when a TREF field
 * is encountered.  The count of bits consumed must be updated and returned.
 * This allows us to track how far we are from the start of the AU which is
 * required for handling the TREFs.
 * Returns false on error, at which time the un16BitsConsumed value is
 * non-determined.
 *
 *****************************************************************************/
static BOOLEAN bParseDataInstance(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        AU_GROUP_STRUCT *psAuGroup,
        AU_TREF_MAPPING_STRUCT *psTrefMap,
        size_t *ptBitsConsumed,
        UN8 *pun8ClassId,
        TABLE_ID *ptInstanceId,
        BOOLEAN bIsPrimaryTable
        )
{
    // Label summary:
    // TABLEID(10), TABLEVER(8),
    // [F+ITITLE(1+VAR)]
    // Then overrides for each label in the def:
    // F+LBID(1+10), LBSIZE(5)
    // ENTRYNO(8)
    // Then repeats for ENTRYNO wrt Table def:
    // F+LABELn
    SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *hMyTdef = NULL;
    OSAL_OBJECT_HDL hTrefList = OSAL_INVALID_OBJECT_HDL;
    SPORTS_SERVICE_DATA_INSTANCE_STRUCT DataInstanceInfo;
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT sTrans;
    OSAL_LINKED_LIST_ENTRY hLLEntry;
    AU_INFO_STRUCT *psAuInfo;
    BOOLEAN bIsLabelPresent=0;
    UN16 un16ActualLabelSize;
    size_t tBitsRead;
    UN8 n;
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode = OSAL_SUCCESS;
//    TREF_INFO_STRUCT *TRefInfo = OSAL_INVALID_OBJECT_HDL;


    // pull in what we know so far and initialize the instance struct
    OSAL.bMemSet(
            &DataInstanceInfo,
            0,
            sizeof(DataInstanceInfo)
        );
    DataInstanceInfo.un16AffiliateId = psAuGroup->un16AfId;
    DataInstanceInfo.un16Epoch = psAuGroup->un16Epoch;
    DataInstanceInfo.bInSeason = (BOOLEAN)psAuGroup->un8Season;
    DataInstanceInfo.un32Crc = psAuGroup->un32Crc;

    psAuInfo = &psAuGroup->asAu[0];

    //
    // Parse and evaluate the table header fields.
    //

    bOk = bParseTableHeader(
            psHighbandObj,
            psAuInfo->hPayload,
            &DataInstanceInfo,
            ptBitsConsumed );

    if( !bOk )
    {
        return FALSE;
    }

    // Make sure we have the tdef.
    if( !psHighbandObj->hTableDefs[DataInstanceInfo.un16TableId].IsValid )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseDataInstance:"
                "No table def for Instance with Table ID %u Ver %u\n",
                (UN32)DataInstanceInfo.un16TableId,
                DataInstanceInfo.un8TableVer );

        // causes TREFs to generate an error and Rollback primary table
        // Primary tables will release rollback point
        return bIsPrimaryTable;
    }

    // Make sure the tdef version is correct.
    if( DataInstanceInfo.un8TableVer !=
        psHighbandObj->hTableDefs[DataInstanceInfo.un16TableId].un8TableVer )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseDataInstance:"
                "Version mismatch for table def for data Instance with Table ID %u - need ver %u\n",
                (UN32)DataInstanceInfo.un16TableId,
                (UN32)DataInstanceInfo.un8TableVer );
        return FALSE;
    }

    // Get a reference to the table def used by this instance.
    hMyTdef = &(psHighbandObj->hTableDefs[DataInstanceInfo.un16TableId]);
    DataInstanceInfo.un32ColumnCount = hMyTdef->un8LabelCount;

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
            "bParseDataInstance:"
            "Process table: DI_%04X%04X\n",
            (UN32)DataInstanceInfo.un16TableId,
            (UN32)DataInstanceInfo.un8TableVer );

    //
    // Parse label overrides.
    //

    bOk = bParseLabelOverrides(
            psHighbandObj,
            psAuInfo->hPayload,
            ptBitsConsumed );

    if( !bOk )
    {
        return FALSE;
    }

    //====================================================
    // Find the number of entries (rows) in this instance
    //====================================================
    tBitsRead = OSAL.tBufferReadHeadBits(
            psAuInfo->hPayload,
            &DataInstanceInfo.un8RowCount,
            0,
            SPORTS_SERVICE_ENTRYNO_BITLEN );

    if( tBitsRead != SPORTS_SERVICE_ENTRYNO_BITLEN )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseDataInstance:"
                "Unable to read entry count\n");

        return FALSE;
    }

    SPORTS_SERVICE_MGR_vPrint( 0/*SSDL_HB_DI_UPDATE*/,
            "bParseDataInstance:"
            "EntryCount: %u\n", (UN32)DataInstanceInfo.un8RowCount);

    *ptBitsConsumed += tBitsRead;

    //==========================================================
    // Get the instance ID we will be writing or updating
    //==========================================================

    // We now have enough information to request an instance ID from the
    // manager object.  He will either:
    //  (1) Create a new instance for us, or
    //  (2) Return an existing instance, after testing for a change
    //      in which case he will clear out the old instance data
    // Once we have an instance, we can go ahead and start writing the
    // data in to the instance table.
    //

    // Begin a table update.
    // Will drop any data we do not have defs for
    // WILL LOCK DB SEM AND CREATE ROLLBACK POINT IF WE ARE TO UPDATE DB
    if(!GsSportsServiceMgrIntf.bBeginTableUpdate(
            (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
            &DataInstanceInfo,
            &sTrans,
            bIsPrimaryTable,
            SS_ROLLBACK_POINT_DI_UPDATE))
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseDataInstance:"
                "No data update InstId: %lld\n", DataInstanceInfo.n64InstanceId);

        return FALSE;
    }

    // Forward label overrides to the manager.
    bOk = bForwardLabelOverrides(
            psHighbandObj,
            &DataInstanceInfo,
            hMyTdef );

    if( !bOk )
    {
        return FALSE;
    }

    //========================================================
    // The manager is ready for receipt of rows.
    // From this point forward, database is locked
    // and has a rollback point set.
    //========================================================

    // are we being asked to return the instance ID?
    if (ptInstanceId)
    {
        *ptInstanceId = DataInstanceInfo.n64InstanceId;
    }
    if (pun8ClassId)
    {
        *pun8ClassId = DataInstanceInfo.un8Tclass;
    }
    //=============================================
    // Start ROW processing:
    //=============================================

    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
            "bParseDataInstance:"
            "CHECK: BitsConsumed = %u\n"
            ,(UN32)(*ptBitsConsumed));    //Column headings

    SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
           "\tLBLID\tDTYPE\tLBTYPE\tSIZE\tVAL\n" );
    n = 0;
    while ( n < DataInstanceInfo.un8RowCount)
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        UN8 un8LblIndex;
        SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *hLabelInfo;
        BOOLEAN bIgnoreLabelValue;

        hEntry = OSAL.hLinkedListFirst ( hMyTdef->hLabelInfoList, (void**)&hLabelInfo);

        // Condition the manager for the start of a row.
        bOk = GsSportsServiceMgrIntf.bBeginRowUpdate(
            (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
            &sTrans );
        if( !bOk )
        {
            break;
        }

        SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
            "r%u", (UN32)n );

        // For each label present, walk through the row, parsing data according to the definition
        for( un8LblIndex = 0; un8LblIndex < hMyTdef->un8LabelCount; un8LblIndex++)
        {
            if ( (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) || (hLabelInfo == NULL) )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                        "bParseDataInstance:"
                        "Lost label sync at Label %u in row %u\n",
                        (UN32)un8LblIndex, (UN32)n);
                bOk = FALSE;
                break;
            }
            tBitsRead = OSAL.tBufferReadHeadBits(
                    psAuInfo->hPayload,
                    &bIsLabelPresent,
                    0,
                    SPORTS_SERVICE_PRESENCE_FLAG_BITLEN);
            if (tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN)
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                        "bParseDataInstance:"
                        "Unable to read label data presence\n");

                // Must always end a table transaction if it was started.
                bOk = FALSE;
                break;
            }

            (*ptBitsConsumed) += tBitsRead;

            if( bIsLabelPresent )
            {
                bIgnoreLabelValue = FALSE;
                un16ActualLabelSize = hLabelInfo->un16LabelSize;

                // Check for overrides and use overridden size
                if( psHighbandObj->aun8LabelOverrides[hLabelInfo->un8LabelId] )
                {
                    if( hLabelInfo->un8Dtype == SS_LABEL_DATATYPE_INTEGER )
                    {
                        un16ActualLabelSize =
                                psHighbandObj->aun8LabelOverrides[hLabelInfo->un8LabelId];

                        if( hLabelInfo->un8LbType <
                            (sizeof(GsLabelVerificationTable) / sizeof(GsLabelVerificationTable[0])) )
                        {
                            if( un16ActualLabelSize >
                                GsLabelVerificationTable[hLabelInfo->un8LbType].un16LbSize )
                            {
                                bIgnoreLabelValue = TRUE;
                            }
                        }
                    }
                    else
                    {
                        // Only integers can be overridden.  For text ignore the override and
                        // bounds check to the table def.  If entry does exceed the
                        // table def baudot size it will not be placed in the db.
                        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                                                "bParseDataInstance:"
                                                "Label override of non int label.\n" );
                    }

                   SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
                            "\t%u\t%u\t%u\t%u(%u)",
                            (UN32)hLabelInfo->un8LabelId,
                            (UN32)hLabelInfo->un8Dtype,
                            (UN32)hLabelInfo->un8LbType,
                            (UN32)un16ActualLabelSize,
                            (UN32)hLabelInfo->un16LabelSize );
                }
                else
                {
                    SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
                            "\t%u\t%u\t%u\t%u",
                            (UN32)hLabelInfo->un8LabelId,
                            (UN32)hLabelInfo->un8Dtype,
                            (UN32)hLabelInfo->un8LbType,
                            (UN32)un16ActualLabelSize );
                }

                switch( hLabelInfo->un8LbType )
                {
                    case SPORTS_LABEL_TYPE_HTEAM:
                    case SPORTS_LABEL_TYPE_VTEAM:
                    case SPORTS_LABEL_TYPE_PARTICIPANT_PARTITION:
                    {
                        bOk = bParseTeamLabel(
                                    psHighbandObj,
                                    psAuInfo->hPayload,
                                    &sTrans,
                                    un16ActualLabelSize,
                                    un8LblIndex,
                                    hLabelInfo->un8LPriority,
                                    hLabelInfo->un8LbType,
                                    ptBitsConsumed );
                        break;
                    }
                    case SPORTS_LABEL_TYPE_TABLEREF:
                    {
                        if( bIsPrimaryTable )
                        {
                            bOk = bParseTref(
                                    psHighbandObj,
                                    psAuInfo->hPayload,
                                    &hTrefList,
                                    &DataInstanceInfo,
                                    n,
                                    un16ActualLabelSize,
                                    un8LblIndex,
                                    hLabelInfo->un8LPriority,
                                    ptBitsConsumed );
                        }
                        else
                        {
                            bOk = FALSE;
                        }
                        break;
                    }
                    default:
                    {
                        bOk = bParseLabel(
                                psHighbandObj,
                                psAuInfo->hPayload,
                                &sTrans,
                                un16ActualLabelSize,
                                un8LblIndex,
                                hLabelInfo->un8LPriority,
                                hLabelInfo->un8LbType,
                                hLabelInfo->un8Dtype,
                                bIgnoreLabelValue,
                                ptBitsConsumed );
                        break;
                    }
                }

                SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
                        "\n");

                if( !bOk )
                {
                    break;
                }
            }
            else
            {
                //      SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
                //          "bParseDataInstance:"
                //          "\t%u  {not present}\n", (UN32)hLabelInfo->un8LabelId);
            }

            hEntry = OSAL.hLinkedListNext ( hEntry, (void**)&hLabelInfo);
        }   // end for labelindex

        if ( bOk )
        {
            // Condition the manager for the end of a row.
            bOk = GsSportsServiceMgrIntf.bEndRowUpdate(
                (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                &sTrans );
        }

        // A row has been written.  If there were any TREFs found, it is
        // now time to update their row ids.  This will allow us
        // to identify which rows need updating later
        if ( bOk && bIsPrimaryTable && (hTrefList != OSAL_INVALID_OBJECT_HDL) )
        {
            // find all the trefs we just collected in this row (match rowindex)
            // and set their primary row id to that last written
            TREF_INFO_STRUCT tmpTrefInfo;
            tmpTrefInfo.un8RowIndex = n;
            bOk = GsSportsServiceMgrIntf.bGetLastRowId(
                    (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                    &tmpTrefInfo.n64PrimaryRowId);

            if( bOk )
            {
                eOsalCode = OSAL.eLinkedListIterate(
                        hTrefList,
                        _bLLIteratorUpdateRowId,
                        &tmpTrefInfo);

                if( eOsalCode == OSAL_NO_OBJECTS )
                {
                }
                else if( eOsalCode != OSAL_SUCCESS )
                {
                    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                            "bParseDataInstance:"
                            "Unable to iterate tref LL eOsalCode %u\n",
                            eOsalCode );
                    bOk = FALSE;
                }
            }
            else
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                        "bParseDataInstance:"
                        "Unable to retrieve last row written\n");
            }
        }

        if( !bOk )
        {
            break;
        }

        n++;
    }   // end while ( n < DataInstanceInfo.un8RowCount)

    // Indicate completion of entire table if we have been successful so far.
    if ( bOk )
    {
        DataInstanceInfo.bTableComplete = TRUE;
    }

    // Must always end a table transaction if it was started.
    // NOTE: THIS WILL NOT RELEASE THE SEMAPHORE OR ROLLBACK POINT
    // (these must stay active for any TREF processing we have to do)
    bOk = GsSportsServiceMgrIntf.bEndTableUpdate(
        (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
        &DataInstanceInfo,
        &sTrans );

    // HANDLE SECONDARY TABLES:
    // Walk through the TREF linked list and find the secondary
    // tables.  Process them in to the database and remove them
    // from the linked list as they are taken care of
    if (  bOk && bIsPrimaryTable && (hTrefList != OSAL_INVALID_OBJECT_HDL) )
    {

        TREF_INFO_STRUCT *tmpTrefInfo;
        int iEntryCnt = 0;

        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                "bParseDataInstance:"
                "Primary Table complete at bit offset %u\n", (UN32)(*ptBitsConsumed));

        // #####################  debug only:
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                "bParseDataInstance:"
                "PRE-PARSE: ");
#if SSDL_HB_TREF
        eOsalCode = OSAL.eLinkedListIterate(
                hTrefList, _bLLIteratorDumpLL, NULL);
#endif // SSDL_HB_TREF
        // find first entry
        hLLEntry = OSAL.hLinkedListFirst(hTrefList, NULL);
        while (hLLEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            size_t tAlignBits;
            size_t tTrefAdjustment;
            BOOLEAN bReturn;
            iEntryCnt++;


            tmpTrefInfo = OSAL.pvLinkedListThis(hLLEntry);

            bReturn = bGetTrefAdjustment(
                    psTrefMap,
                    psAuGroup->un16Count,
                    tmpTrefInfo->un16ByteOffset*8,
                    &tTrefAdjustment );

            if( bReturn == FALSE )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                        "bParseDataInstance:"
                        "Failed to get tref adjustment:"
                        " un16ByteOffset %u ptBitsConsumed %u\n",
                        (UN32)tmpTrefInfo->un16ByteOffset,
                        (UN32)(*ptBitsConsumed) );

                bOk = FALSE;
                break;
            }

            // The adjustment must not make the offset negative.
            if( (size_t)(tmpTrefInfo->un16ByteOffset*8) < tTrefAdjustment )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                        "bParseDataInstance:"
                        "Tref adjustment exceeds offset:"
                        " adj %u off %u\n",
                        (UN32)tTrefAdjustment,
                        (UN32)tmpTrefInfo->un16ByteOffset );

                bOk = FALSE;
                break;
            }

            // The tref offset must >= to our current position in the payload.
            if( (tmpTrefInfo->un16ByteOffset*8 - tTrefAdjustment) < *ptBitsConsumed )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                        "bParseDataInstance:"
                        "Tref offset < current payload position"
                        " off %u ptBitsConsumed %u\n",
                        (UN32)tmpTrefInfo->un16ByteOffset,
                        (UN32)(*ptBitsConsumed) );

                bOk = FALSE;
                break;
            }

            // must now move forward to next TREF
            tAlignBits = (tmpTrefInfo->un16ByteOffset*8 -
                            tTrefAdjustment) - *ptBitsConsumed;

            //move to the start of the table by navigating via BitOffset in the payload
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                    "bParseDataInstance:"
                    "Moving payload bit ptr %u by %u bits"
                    " multiauadj %u originaltrefoffset %u\n",
                    (UN32)(*ptBitsConsumed),
                    (UN32)tAlignBits,
                    (UN32)tTrefAdjustment,
                    (UN32)tmpTrefInfo->un16ByteOffset*8 );

            if ( tAlignBits > 0 )
            {
                // get us back in to byte alignment
                tBitsRead = OSAL.tBufferSeekHeadBits(
                                 psAuInfo->hPayload,
                                 tAlignBits);

                if (tAlignBits != tBitsRead)
                {
                    SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                            "bParseDataInstance:"
                            "unable to regain byte alignment for TREF processing\n"
                                   );
                     bOk = FALSE;
                     break;
                }
                *ptBitsConsumed += tAlignBits;
            }

            //parse table and add it to the database, return the RefInstanceId
            if( bOk )
            {
                bOk = bParseDataInstance(
                         psHighbandObj,
                         psAuGroup,
                         psTrefMap,
                         ptBitsConsumed,
                         pun8ClassId,
                         &tmpTrefInfo->tChildInstanceId,
                         FALSE );
            }

            // debug only:
#if SSDL_HB_TREF
            _bLLIteratorDumpLL((void*)tmpTrefInfo, (void*)NULL);   // manual iteration to Debug DUMP contents
#endif

           // now update the parent instance and child relationships
            if (bOk)
            {
                bOk = GsSportsServiceMgrIntf.bUpdateTrefRelationships(
                        (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                        tmpTrefInfo->un32TdefKey,
                        tmpTrefInfo->un8LabelId,
                        tmpTrefInfo->tChildInstanceId,
                        tmpTrefInfo->n64PrimaryRowId,
                        tmpTrefInfo->tParentInstanceId,
                        tmpTrefInfo->un8LPriority
                        );
            }

            if (!bOk)
            {

                 // Any failure will result in the entire TREF being discarded
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                         "bParseDataInstance:"
                         "Failed to process table instance for InstId=%llu\n",
                                 tmpTrefInfo->tParentInstanceId);
                break;
            }
            else
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                        "bParseDataInstance:"
                         "Secondary Table instance processed for InstId=%llu\n",
                                 tmpTrefInfo->tChildInstanceId);
            }

            // Get next TREF
            hLLEntry = OSAL.hLinkedListNext(hLLEntry, (void*)&tmpTrefInfo);
            if( hLLEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                        "bParseDataInstance:"
                        "TREF # LL entries=%d\n",iEntryCnt);
            }
        }
    }

    // Clean up TREF list if it was created
    if( hTrefList != OSAL_INVALID_OBJECT_HDL )
    {
        // Remove links from the list and destroy
        eOsalCode = OSAL.eLinkedListRemoveAll(
                hTrefList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );
        if (eOsalCode == OSAL_SUCCESS)
        {
            // Destroy the list itself
            OSAL.eLinkedListDelete( hTrefList );
        }
    }

    if( bIsPrimaryTable )
    {
        // Rollback or commit the changes
        // and release the semaphore.
        GsSportsServiceMgrIntf.vEndDataTableInstance(
                psHighbandObj->hSportsDataService,
                SS_ROLLBACK_POINT_DI_UPDATE,
                bOk,
                &DataInstanceInfo );
    }

    return bOk;
}

/*****************************************************************************
*
*   bParseTableHeader
*
*****************************************************************************/
static BOOLEAN bParseTableHeader(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        OSAL_BUFFER_HDL hPayload,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psDataInstanceInfo,
        size_t *ptBitsConsumed
        )
{
    BOOLEAN bOk;
    size_t tBitsRead, tNumSymbols;
    STRING_OBJECT hTitle = STRING_INVALID_OBJECT;

    // Get the table def id
    bOk = OSAL.bBufferReadBitsToUN16(
            hPayload,
            &(psDataInstanceInfo->un16TableId),
            SPORTS_SERVICE_TABLEID_BITLEN );

    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseTableHeader:"
                "Unable to read Data Instance Table ID\n");
        return FALSE;
    }

    *ptBitsConsumed += SPORTS_SERVICE_TABLEID_BITLEN;

    // Get the table def version.
    tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload,
            &(psDataInstanceInfo->un8TableVer),
            0,
            SPORTS_SERVICE_TABVER_BITLEN );

    if( tBitsRead != SPORTS_SERVICE_TABVER_BITLEN )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseTableHeader:"
                "Unable to read table ver\n");
        return FALSE;
    }

    *ptBitsConsumed += tBitsRead;

    // Get the title if present.
    tNumSymbols = 0;
    tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload,
            &bOk,
            0,
            SPORTS_SERVICE_PRESENCE_FLAG_BITLEN );

    if( tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseTableHeader:"
                "Unable to read Title Presence bit\n");
        return FALSE;
    }

    // Title is present.
    if( bOk )
    {
        hTitle = BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    hPayload,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE,
                    SS_MAX_TABLE_TITLE_SYMBOL_COUNT,
                    &tNumSymbols );

        if( hTitle == STRING_INVALID_OBJECT )
        {
            return FALSE;
        }

        if( tNumSymbols > (size_t)SS_MAX_TABLE_TITLE_SYMBOL_COUNT )
        {
            STRING_vDestroy(hTitle);
            return FALSE;
        }

        // copy the string in to our object's char buffer
        // if it fails or the data is truncated, continue anyway
        STRING.tCopyToCStr(
                hTitle,
                psDataInstanceInfo->InstanceTitle,
                sizeof(psDataInstanceInfo->InstanceTitle) );

        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                "bParseTableHeader:"
                "Table Instance title[%u]: %s\n",
                (UN32)tNumSymbols,
                psDataInstanceInfo->InstanceTitle );

        STRING_vDestroy(hTitle);
    }

    *ptBitsConsumed += ( tBitsRead + (tNumSymbols * BAUDOT_CHAR_BITWIDTH));
    return TRUE;
}

/*****************************************************************************
*
*   bParseLabelOverrides
*
*****************************************************************************/
static BOOLEAN bParseLabelOverrides(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        OSAL_BUFFER_HDL hPayload,
        size_t *ptBitsConsumed
        )
{
    BOOLEAN bOk, bOverridePresent = 0;
    UN16 un16LabelId = 0;
    size_t tBitsRead;

    // Table of possible LBID's.  LBSIZE + 1 is the number bits to parse for
    // a label.  Table stores the new (overriden LBSIZE + 1).  Zero is used
    // to indicate no override is present.
    OSAL.bMemSet(  psHighbandObj->aun8LabelOverrides,
                   0,
                   sizeof(psHighbandObj->aun8LabelOverrides) );

    do
    {
        tBitsRead = OSAL.tBufferReadHeadBits(
                hPayload,
                &bOverridePresent,
                0,
                SPORTS_SERVICE_PRESENCE_FLAG_BITLEN );

        if( tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                    "bParseLabelOverrides:"
                    "Unable to read Label Override Presence bit\n");
            return FALSE;
        }

        *ptBitsConsumed += tBitsRead;

        if( bOverridePresent )
        {
            // get the labelid to override
            bOk = OSAL.bBufferReadBitsToUN16(
                    hPayload,
                    &un16LabelId,
                    SPORTS_SERVICE_OVERRIDE_LABELID_BITLEN );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_OVERRIDES,
                        "bParseLabelOverrides:"
                        "Unable to read Label Override Label id\n");
                return FALSE;
            }

            *ptBitsConsumed += SPORTS_SERVICE_OVERRIDE_LABELID_BITLEN;

            if( un16LabelId > SPORTS_SERVICE_HIGHBAND_MAX_LABEL_ID )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_OVERRIDES,
                        "bParseLabelOverrides:"
                        "Invalid label id for Override: LBID %u\n",
                        un16LabelId );
                return FALSE;
            }

            // read the new size in to our table
            tBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload,
                    &psHighbandObj->aun8LabelOverrides[un16LabelId],
                    0,
                    SPORTS_SERVICE_OVERRIDE_LABELSIZE_BITLEN );

            if( tBitsRead != SPORTS_SERVICE_OVERRIDE_LABELSIZE_BITLEN )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_OVERRIDES,
                        "bParseLabelOverrides:"
                        "Error reading new Label Override size\n");
                return FALSE;
            }

            // Make 1+LBSIZE
            psHighbandObj->aun8LabelOverrides[un16LabelId]++;

            *ptBitsConsumed += tBitsRead;

            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_OVERRIDES,
                    "bParseLabelOverrides:"
                    "Overriding label id %d with temp size of %d\n",
                    un16LabelId,
                    psHighbandObj->aun8LabelOverrides[un16LabelId]
                    );
        }
    } while ( bOverridePresent );

    return TRUE;
}

/*****************************************************************************
*
*   bParseTeamLabel
*
*****************************************************************************/
static BOOLEAN bParseTeamLabel(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        OSAL_BUFFER_HDL hPayload,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTrans,
        UN16 un16LabelSize,
        UN8 un8ColumnIndex,
        UN8 un8LPriority,
        UN8 un8LType,
        size_t *ptBitsConsumed
        )
{
    BOOLEAN bOk;
    size_t tBitsRead, tNumSymbols;
    BOOLEAN bDtypeBit = 0;
    UN16 un16Value = 0;
    STRING_OBJECT hName = STRING_INVALID_OBJECT;

    // requires special case handling based on MSB
    // read the MSB first - if 0 then we have an INT else we have a string of Baudot
    tBitsRead = OSAL.tBufferReadHeadBits(
                        hPayload,
                        &bDtypeBit,
                        0,
                        SPORTS_SERVICE_PRESENCE_FLAG_BITLEN );

    if( tBitsRead != SPORTS_SERVICE_PRESENCE_FLAG_BITLEN )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseTeamLabel:"
                "Failed to read bDtypeBit.\n");
        return FALSE;
    }

    *ptBitsConsumed += tBitsRead;

    // Treat as int.  LabelSize is ignored.
    if( bDtypeBit == 0 )
    {
        bOk = OSAL.bBufferReadBitsToUN16(
                hPayload,
                &un16Value,
                SPORTS_LABEL_INTEGER_TEAM_BIT_SIZE );

        if( bOk )
        {
            *ptBitsConsumed += SPORTS_LABEL_INTEGER_TEAM_BIT_SIZE;

            SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
                    "\t%u",
                    (UN32)un16Value);

            // Update the column.
            bOk = GsSportsServiceMgrIntf.bUpdateColumnInt(
                (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                psTrans,
                un8ColumnIndex,
                un8LPriority,
                un8LType,
                un16Value );
        }
        else
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                "bParseTeamLabel:"
                "Failed to read value.\n");
        }
        return bOk;
    }

    // string data
    hName = BAUDOT_hToString(
            SMS_INVALID_OBJECT,
            hPayload,
            BAUDOT_BEHAVIOR_PROCESS_TO_END,
            TRUE, TRUE,
            SPORTS_SERVICE_HIGHBAND_MAX_TNAME_SYMBOL_COUNT,
            &tNumSymbols);

    if( hName == STRING_INVALID_OBJECT )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseTeamLabel:"
                "Failed to create string\n");
        return FALSE;
    }

    *ptBitsConsumed += (tNumSymbols * BAUDOT_CHAR_BITWIDTH);

    SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
            "\t%s",
            STRING.pacCStr(hName));

    // Update the column.
    if( tNumSymbols > (size_t)un16LabelSize )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseTeamLabel:"
                "Too many symbols: lbsize %u symbols %u\n",
                un16LabelSize,
                tNumSymbols );

        // Drop the column silently.
        bOk = TRUE;
    }
    else
    {
        bOk = GsSportsServiceMgrIntf.bUpdateColumnString(
            (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
            psTrans,
            un8ColumnIndex,
            un8LPriority,
            un8LType,
            hName );
    }

    STRING_vDestroy(hName);

    return bOk;
}

/*****************************************************************************
*
*   bParseTref
*
*****************************************************************************/
static BOOLEAN bParseTref(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        OSAL_BUFFER_HDL hPayload,
        OSAL_OBJECT_HDL *phTrefList,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psDataInstanceInfo,
        UN8 un8RowIndex,
        UN16 un16LabelSize,
        UN8 un8ColumnIndex,
        UN8 un8LPriority,
        size_t *ptBitsConsumed
        )
{
    UN16 un16ByteOffset;
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    TREF_INFO_STRUCT *psTrefInfo = NULL;

    // A TREF is a packet-internal pointer to another table.
    // The TREFs are collected in a linked list and once we get
    // to the end of the primary table we process the ref'd tables.
    // Once the ref'd tables are processed, we go back and update
    // the database with the ref'd tables instance IDs.
    // i.e.  DI_{TdefKey}!c{LabelId} = Ref'd Instance ID
    // for a saved RowId.

    if( un16LabelSize > SPORTS_LABEL_MAX_TABLEREF_BIT_SIZE )
    {
        return FALSE;
    }

    // Get tref offset.
    un16ByteOffset = 0;
    bOk = OSAL.bBufferReadBitsToUN16(
            hPayload,
            &un16ByteOffset,
            un16LabelSize );

    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                "bParseDataInstance:"
                "Failed to read TREF.\n");
        return FALSE;
    }

    *ptBitsConsumed += un16LabelSize;

    // Create a linked list if we do not have one.
    if( *phTrefList == OSAL_INVALID_OBJECT_HDL )
    {
        eOsalCode = OSAL.eLinkedListCreate(
            phTrefList,
            SPORTS_SVC_HB_OBJ_NAME":TREFList",
            _n16CompareByteOffsets,   // Trefs must be added to the list in Byte offset order
            OSAL_LL_OPTION_LINEAR );

        if( eOsalCode != OSAL_SUCCESS )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                    "bParseDataInstance:"
                    "Unable to create linked list for TREFS: eOsalCode%u\n",
                    eOsalCode );
            return FALSE;
        }
    }

    // create a new TRefInfo struct
    psTrefInfo = (TREF_INFO_STRUCT *)SMSO_hCreate(
             SPORTS_SVC_HB_OBJ_NAME":TREF",
             sizeof(TREF_INFO_STRUCT),
             SMS_INVALID_OBJECT,
             FALSE );

    if( psTrefInfo == NULL )
    {
        return FALSE;
    }

    // SMSO_hCreate does this for us.
    //OSAL.bMemSet(TRefInfo, 0, sizeof(TREF_INFO_STRUCT));

    // populate the entry references
    psTrefInfo->un8RowIndex = un8RowIndex;
    psTrefInfo->un16ByteOffset = un16ByteOffset;
    psTrefInfo->un8LabelId = un8ColumnIndex;
    psTrefInfo->tParentInstanceId = psDataInstanceInfo->n64InstanceId;
    psTrefInfo->un32TdefKey = psDataInstanceInfo->un32TdefKey;
    psTrefInfo->un8LPriority = un8LPriority;

    SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_TREF,
            "TREF - (%d) DI_%08X InstId:%llu  RI:%u  BOffset:%u  LblId: %u",
            *ptBitsConsumed,
            psDataInstanceInfo->un32TdefKey,
            psDataInstanceInfo->n64InstanceId,
            (UN32)psTrefInfo->un8RowIndex,
            (UN32)psTrefInfo->un16ByteOffset,
            (UN32)psTrefInfo->un8LabelId,
            (UN32)psTrefInfo->un8LPriority);

    // Now add the TREF info to the linked list
    eOsalCode = OSAL.eLinkedListAdd(
            *phTrefList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            (void*)psTrefInfo );

    if( eOsalCode != OSAL_SUCCESS )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                "bParseDataInstance:"
                "Failed to add TREF for InstId=%llu to LL. eOsalCode %u\n",
                psDataInstanceInfo->n64InstanceId,
                eOsalCode );

        SMSO_vDestroy((SMS_OBJECT)psTrefInfo);
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bParseLabel
*
*****************************************************************************/
static BOOLEAN bParseLabel(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        OSAL_BUFFER_HDL hPayload,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTrans,
        UN16 un16LabelSize,
        UN8 un8ColumnIndex,
        UN8 un8LPriority,
        UN8 un8LType,
        UN8 un8Dtype,
        BOOLEAN bIgnoreLabelValue,
        size_t *ptBitsConsumed
        )
{
    size_t tNumSymbols;
    BOOLEAN bOk;
    UN32 un32Result = 0;
    STRING_OBJECT hText = STRING_INVALID_OBJECT;

    // Integer
    if( un8Dtype == 0)
    {
        if( un16LabelSize > SPORTS_LABEL_MAX_INTEGER_BIT_SIZE )
        {
            // An integer could be specified as 4096 bits by the spec.
            // We only save integers up to 32 bits since the lifetime
            // size for all labels in the pvn1 spec is 32 as of 2011_10_12
            // Skip over larger integers.  The db will always have a NULL
            // value for these future large integers.
            tNumSymbols = OSAL.tBufferSeekHeadBits(
                                      hPayload,
                                      un16LabelSize );

            if( tNumSymbols != un16LabelSize )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                        "bParseLabel:"
                        "Skipping large integer: LType %u bits %u\n",
                        un8LType,
                        un16LabelSize );

                return FALSE;
            }

            *ptBitsConsumed += tNumSymbols;
            return TRUE;
        }

        bOk = OSAL.bBufferReadBitsToUN32(
                            hPayload,
                            &un32Result,
                            un16LabelSize);

        if( bOk )
        {
            *ptBitsConsumed += un16LabelSize;

            SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
                    "\t%u",
                    un32Result );

            if( !bIgnoreLabelValue )
            {
                bOk = GsSportsServiceMgrIntf.bUpdateColumnInt(
                    (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                    psTrans,
                    un8ColumnIndex,
                    un8LPriority,
                    un8LType,
                    (int)un32Result );
            }
        }
        else
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                                    "bParseLabel:"
                                    "Buffer read error reading %u bits\n",
                                    (UN32)un16LabelSize);
        }

        return bOk;
    }

    // Text
    hText = BAUDOT_hToString(
                SMS_INVALID_OBJECT,
                hPayload,
                BAUDOT_BEHAVIOR_PROCESS_TO_END,
                TRUE, TRUE,
                un16LabelSize,
                &tNumSymbols);

    if( hText == STRING_INVALID_OBJECT )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseLabel:"
                "Failed to create LBTXT string\n");
        return FALSE;
    }

    *ptBitsConsumed += (tNumSymbols * BAUDOT_CHAR_BITWIDTH);

    SPORTS_SERVICE_MGR_vPrintSilent( SSDL_HB_DI_UPDATE,
            "\t%s",
            STRING.pacCStr(hText));

    // Update the column.
    bOk = TRUE;
    if( tNumSymbols > (size_t)un16LabelSize )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_DI_UPDATE,
                "bParseLabel:"
                "Too many symbols:lbsize %u symbols %u\n",
                un16LabelSize,
                tNumSymbols );

        // Drop the column silently.
    }
    else
    {
        if( !bIgnoreLabelValue )
        {
            bOk = GsSportsServiceMgrIntf.bUpdateColumnString(
                (SPORTS_SERVICE_OBJECT) psHighbandObj->hSportsDataService,
                psTrans,
                un8ColumnIndex,
                un8LPriority,
                un8LType,
                hText );
        }
    }

    STRING_vDestroy(hText);

    return bOk;
}

/*****************************************************************************
*
*   _n16CompareByteOffsets
*   Comparison used by LL to ensure that the TREFs are added in byte offset
*   ascending order.  (i.e. linearly)
*
*****************************************************************************/
static N16 _n16CompareByteOffsets(
        void *pvObj1,
        void *pvObj2
        )
{
    N16 n16Result;
    TREF_INFO_STRUCT *TRefInfo1 = pvObj1;
    TREF_INFO_STRUCT *TRefInfo2 = pvObj2;

    if(TRefInfo1->un16ByteOffset > TRefInfo2->un16ByteOffset)
    {
        n16Result = 1;
    } else if(TRefInfo1->un16ByteOffset < TRefInfo2->un16ByteOffset)
    {
        n16Result = -1;
    } else
    {
        n16Result = 0;
    }

    return n16Result;
}

/*****************************************************************************
*
*   _bLLIteratorUpdateRowId
*   Iterate Linked list and update the primary row id field
*
*****************************************************************************/
static BOOLEAN _bLLIteratorUpdateRowId ( void *pvData, void *pvArg )
{
    TREF_INFO_STRUCT *savedTrefInfo = (TREF_INFO_STRUCT *)pvData;   // member in list
    TREF_INFO_STRUCT *newTrefInfo = (TREF_INFO_STRUCT *)pvArg;  // incoming parameters

    if (savedTrefInfo->un8RowIndex == newTrefInfo->un8RowIndex)
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_HB_TREF,
                "bProcessSportsTableInstance:"
                "Updating TREF @ BitOffset=%u  RowId=%llu\n",
                (UN32)(savedTrefInfo->un16ByteOffset*8),
                newTrefInfo->n64PrimaryRowId
            );
        savedTrefInfo->n64PrimaryRowId = newTrefInfo->n64PrimaryRowId;
    }

    return TRUE;    // keep going
}

/*****************************************************************************
*
*   bVerifyAu
*
*   Sports Service uses a ISO 3309 CRC32 in order to validate packets as
*   they are received over the air.  This function uses tComputeCRC to
*   validate the data available in hPayload.
*
*****************************************************************************/
static BOOLEAN bVerifyAu (
    SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
    AU_INFO_STRUCT *psAuInfo
        )
{
    OSAL_CRC_RESULT tCRC;
    BOOLEAN bOk,
            bPacketValid = FALSE;
    size_t tNumBytesToCRC,
           tBytesRead,
           tBytesProcessed = 0;
    UN8 un8CRCByte0 = 0,
        un8CRCByte1 = 0,
        un8CRCByte2 = 0,
        un8CRCByte3 = 0;

    // Get the bit-size of the entire message
    psAuInfo->tPayloadStartSizeBits = OSAL.tBufferGetSizeInBits(psAuInfo->hPayload);
    if (psAuInfo->tPayloadStartSizeBits <= (SPORTS_SERVICE_CRC_BYTELEN * 8))
    {
        // Have to have some data
    	SMSAPI_DEBUG_vPrintErrorFull(NULL, __LINE__,
    	        			SPORTS_SVC_HB_OBJ_NAME" bVerifyAu - no data\n");
        return FALSE;
    }

    // Calculate the number of bytes to
    // check with a CRC
    tNumBytesToCRC =
        OSAL.tBufferGetSize(psAuInfo->hPayload) - SPORTS_SERVICE_CRC_BYTELEN;

    // Our offset starts at the last byte
    tBytesRead = OSAL.tBufferReadTail(psAuInfo->hPayload, &un8CRCByte0, 1);
    tBytesRead += OSAL.tBufferReadTail(psAuInfo->hPayload, &un8CRCByte1, 1);
    tBytesRead += OSAL.tBufferReadTail(psAuInfo->hPayload, &un8CRCByte2, 1);
    tBytesRead += OSAL.tBufferReadTail(psAuInfo->hPayload, &un8CRCByte3, 1);

    if (tBytesRead != SPORTS_SERVICE_CRC_BYTELEN)
    {
    	SMSAPI_DEBUG_vPrintErrorFull(NULL, __LINE__,
    	        			SPORTS_SVC_HB_OBJ_NAME" bVerifyAu - failed to read CRC bytes\n");
        return FALSE;
    }

    // Initialize the CRC
    bOk = OSAL.bInitializeCRC( psHighbandObj->hCRC, &tCRC );
    if (bOk == FALSE)
    {
    	SMSAPI_DEBUG_vPrintErrorFull(NULL, __LINE__,
    	        			SPORTS_SVC_HB_OBJ_NAME" bVerifyAu - CRC init failed\n");
        return FALSE;
    }

    // Compute the CRC of the entire payload
    tCRC = OSAL.tComputeCRC(
        psHighbandObj->hCRC, tCRC, psAuInfo->hPayload,
        0, tNumBytesToCRC, &tBytesProcessed);

    // Did we process the entire payload?
    if (tBytesProcessed != tNumBytesToCRC)
    {
        // No!
    	SMSAPI_DEBUG_vPrintErrorFull(NULL, __LINE__,
    	        			SPORTS_SVC_HB_OBJ_NAME" bVerifyAu - unable to process entire payload\n");
        return FALSE;
    }

    // Invert the CRC bits (SX-9845-0180 section 3.3.3)
    tCRC ^= SPORTS_SERVICE_CRC_INVERSION_VALUE;

    // The packet is valid if the calculated CRC is
    // equal to the CRC field in the message
    if ((BYTE0(tCRC) == un8CRCByte0) &&
        (BYTE1(tCRC) == un8CRCByte1) &&
        (BYTE2(tCRC) == un8CRCByte2) &&
        (BYTE3(tCRC) == un8CRCByte3))
    {
        bPacketValid = TRUE;

        // Update the current msg crc
        psAuInfo->un32Crc = (UN32)tCRC;
    }

    return bPacketValid;
}
#if(0)
static void DEBUG_DumpBuffer(OSAL_BUFFER_HDL hPayload)
{
	size_t n;
	UN8 dataByte;

	// DEBUG DUMP:
	size_t bufflen = OSAL.tBufferGetSize(hPayload);
	SPORTS_SERVICE_MGR_vPrintSilent( 1,
	    "Dumping buffer %d bytes:\n", bufflen );

	for (n = 0; n < bufflen; n++)
	{
		OSAL.tBufferPeekBits(
				hPayload, &dataByte,
				0, 8, n<<3);

		SPORTS_SERVICE_MGR_vPrintSilent( 1,
		        "[%02X] ", dataByte);

		if ( ((n>0) && (n%8 == 0)) || (n == bufflen-1) )
		{
		    SPORTS_SERVICE_MGR_vPrintSilent( 1, "\n");
		}
	}
}

static void DEBUG_DumpList(SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj)
{
	UN32 n=0;
	SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *TdefInfo;

	for (n=0; n<SPORTS_SERVICE_HIGHBAND_MAX_NUMBER_TABLE_DEFS; n++)
	{
		TdefInfo = (SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT*)&psHighbandObj->hTableDefs[n];
		if (TdefInfo->IsValid)
		{
			printf("TableID[%d]: TableVer: %u  Tclass: %u  CRC: 0x%X  LabelCount: %u List Handle: 0x%X\n",
						n, TdefInfo->un8TableVer, TdefInfo->un8TableClass, TdefInfo->CRC,
						TdefInfo->un8LabelCount, TdefInfo->hLabelInfoList);
		}
	}
}

static void DEBUG_DumpTableDef(SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj, UN16 un16TableId)
{
	// Given a pointer to the SPORTS_SERVICE object and a table ID,
	// dump the formatted contents of the table id to the console
	SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *TdefInfo;
	SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *hLabelInfo = NULL;

	OSAL_LINKED_LIST_ENTRY hEntry;

	if (un16TableId <= SPORTS_SERVICE_HIGHBAND_MAX_NUMBER_TABLE_DEFS)
	{
		TdefInfo = (SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT*)&psHighbandObj->hTableDefs[un16TableId];
		if (TdefInfo->IsValid)
		{
			printf("DEBUG_DumpTableDef:\n"
					"TableID[%d]: TableVer: %u  Tclass: %u  CRC: 0x%X  LabelCount: %u\n",
					un16TableId, TdefInfo->un8TableVer, TdefInfo->un8TableClass, TdefInfo->CRC, TdefInfo->un8LabelCount);

			if(TdefInfo->hLabelInfoList == OSAL_INVALID_OBJECT_HDL)
			{
				printf("\tNo label info list available\n");
			} else {
				// walk the label info list and dump out the label info
				hEntry = OSAL.hLinkedListFirst ( TdefInfo->hLabelInfoList, (void**)&hLabelInfo);
				if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
				{
					printf("\tUnable to retrieve head of list\n");
					return;
				}

				printf("\tID\tDtype\tLBtype\tDMOD\tPri\tLBSIZE\tLBTXT\n");
				while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
				{

					printf("\t{%d}\t%d\t%d", hLabelInfo->un8LabelId, hLabelInfo->un8Dtype, hLabelInfo->un8LbType);
					if (hLabelInfo->bIsDmodPresent)
						printf("\t%d", hLabelInfo->un8Dmod);
					else
						printf("\t--");
					printf("\t%d\t%d", hLabelInfo->un8LPriority, hLabelInfo->un16LabelSize);
					if (hLabelInfo->bIsLabelTxtPresent)
					{
						printf("\t%s", hLabelInfo->LabelText);
					}
					printf("\n");

					hEntry = OSAL.hLinkedListNext ( hEntry, (void**)&hLabelInfo);
				}
			}
		}
	} else {

		printf("DEBUG_DumpTableDef: Table Id %d is out of range\n", un16TableId);

	}

}
#endif

/*****************************************************************************
*
*   bForwardLabelOverrides
*
*****************************************************************************/
static BOOLEAN bForwardLabelOverrides(
        SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT *psHighbandObj,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psDataInfo,
        SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *hMyTdef
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OVERRIDE_FORWARD_HELPER_STRUCT sHelper;

    sHelper.bOk = TRUE;
    sHelper.psHighbandObj = psHighbandObj;
    sHelper.tInstId = psDataInfo->n64InstanceId;

    eOsalCode = OSAL.eLinkedListIterate (
         hMyTdef->hLabelInfoList,
         (OSAL_LL_ITERATOR_HANDLER) bUnsafeForwardOverrideHandler,
         (void*)&sHelper );

    if( eOsalCode == OSAL_NO_OBJECTS )
    {
        bOk = TRUE;
    }
    else if( eOsalCode != OSAL_SUCCESS )
    {
        bOk = FALSE;
    }
    else
    {
        bOk = sHelper.bOk;
    }

    return bOk;
}

/*****************************************************************************
*
*   bUnsafeForwardOverrideHandler
*
*****************************************************************************/
static BOOLEAN bUnsafeForwardOverrideHandler (
    SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *psLabelInfo,
    OVERRIDE_FORWARD_HELPER_STRUCT *psHelper
    )
{
    UN32 un32Limit;

    if( psHelper == NULL )
    {
        return FALSE;
    }

    if( psLabelInfo == NULL )
    {
        psHelper->bOk = FALSE;
        return FALSE;
    }

    un32Limit = sizeof(psHelper->psHighbandObj->aun8LabelOverrides) /
                    sizeof(psHelper->psHighbandObj->aun8LabelOverrides[0]);

    // Range check label id to our override table.
    if( psLabelInfo->un8LabelId >= un32Limit )
    {
        return TRUE;
    }

    // No override for this label.
    if( psHelper->psHighbandObj->aun8LabelOverrides[psLabelInfo->un8LabelId] == 0 )
    {
        return TRUE;
    }

    // Do not forward invalid overrides.
    if( (psLabelInfo->un8Dtype != SS_LABEL_DATATYPE_INTEGER)
        || (psLabelInfo->un8LbType == SPORTS_LABEL_TYPE_HTEAM)
        || (psLabelInfo->un8LbType == SPORTS_LABEL_TYPE_VTEAM)
        || (psLabelInfo->un8LbType == SPORTS_LABEL_TYPE_PARTICIPANT_PARTITION)
        || (psLabelInfo->un8LbType == SPORTS_LABEL_TYPE_TABLEREF) )
    {
        return TRUE;
    }

    // Range check override if within known range.  If failure do not forward.
    // We will not populate the db with an override that exceeds the max limit.
    if( psLabelInfo->un8LbType <
            (sizeof(GsLabelVerificationTable) / sizeof(GsLabelVerificationTable[0])) )
    {
        if( psHelper->psHighbandObj->aun8LabelOverrides[psLabelInfo->un8LabelId] >
            GsLabelVerificationTable[psLabelInfo->un8LbType].un16LbSize )
        {
            return TRUE;
        }
    }
    else
    {
        if( psHelper->psHighbandObj->aun8LabelOverrides[psLabelInfo->un8LabelId] >
            SPORTS_LABEL_MAX_INTEGER_BIT_SIZE )
        {
            return TRUE;
        }
    }

    psHelper->bOk = GsSportsServiceMgrIntf.bAddLabelOverride(
                (SPORTS_SERVICE_OBJECT) psHelper->psHighbandObj->hSportsDataService,
                psHelper->tInstId,
                psLabelInfo->un8LabelId,
                psHelper->psHighbandObj->aun8LabelOverrides[psLabelInfo->un8LabelId] );

    return psHelper->bOk;
}
