/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Safety Camera protocol version 1 implementation
 *  for the Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "string.h"
#include "baudot.h"
#include "dataservice_mgr_obj.h"
#include "location_obj.h"
#include "db_util.h"

#include "safecam_interface.h"
#include "_safecam_pvn1.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

///////////////////////////////////////////////////////////////////////////////
const UN8 aun8NumValidRowsForBaudotColumnNonEscape[BAUDOT_NUM_COLUMNS] = {
    SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LTR_UC,
    SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LTR_LC,
    SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LTR_FIG
};

///////////////////////////////////////////////////////////////////////////////
const UN8 aun8NumValidRowsForBaudotColumnEscape[BAUDOT_NUM_COLUMNS] = {
    SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_UC_ESC,
    SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LC_ESC,
    SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_FIG_ESC
};

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tMinimumOTABufferByteSize
*
*   Calculate the minimum number of bytes needed for this service
*   based upon how it's going to be used
*
*****************************************************************************/
static size_t tMinimumOTABufferByteSize (
    BOOLEAN bDBUpdatesEnabled
        )
{
    // We'll always need this as a starting point
    size_t tByteSize = SAFECAM1_INITIAL_BUFFER_BYTESIZE;

    if (TRUE == bDBUpdatesEnabled)
    {
        tByteSize += SAFECAM1_BUFFER_BYTESIZE;
    }

    return tByteSize;
}

/*****************************************************************************
*
*   hInit
*
*****************************************************************************/
static SAFECAM_INTERFACE_OBJECT hInit(
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService,
    SMS_OBJECT hParent,
    RFD_UPDATE_VERSION tCurrentVersion
        )
{
    BOOLEAN bOwner = FALSE;
    SAFECAM_INTERFACE_OBJECT hResult = SAFECAM_INTERFACE_INVALID_OBJECT;
    SAFECAM1_OBJECT_STRUCT *psObj = (SAFECAM1_OBJECT_STRUCT *)NULL;
    BOOLEAN bTemp = FALSE;

    do
    {
        if (hSafeCamService == SAFETY_CAMERAS_SERVICE_INVALID_OBJECT)
        {
            break;
        }

        // Verify we own service handle
        bOwner = SMSO_bOwner(hParent);
        if (bOwner == FALSE)
        {
            break;
        }

        // Create an instance of this object
        psObj = (SAFECAM1_OBJECT_STRUCT*)
            SMSO_hCreate(
                SAFECAM1_OBJECT_NAME,
                sizeof(*psObj),
                hParent, FALSE);

        if (psObj == NULL)
        {
            break;
        }

        psObj->hSafeCamService = hSafeCamService;

        bTemp = bInitRFD(psObj, tCurrentVersion);
        if (bTemp == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to init RFD"
                    );
            break;
        }

        hResult = (SAFECAM_INTERFACE_OBJECT)psObj;
    } while (FALSE);

    if (hResult == SAFECAM_INTERFACE_INVALID_OBJECT)
    {
        vUninitObject(psObj);
    }

    return hResult;
}

/*****************************************************************************
*
*   vUnInit
*
*****************************************************************************/
static void vUnInit(
    SAFECAM_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner = FALSE;
    SAFECAM1_OBJECT_STRUCT *psObj = (SAFECAM1_OBJECT_STRUCT*)hInterface;

    do {
        if (hInterface == SAFECAM_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        vUninitObject(psObj);
    } while (FALSE);

    return;
}


/*****************************************************************************
*
*   bProcessMessage
*
*****************************************************************************/
static BOOLEAN bProcessMessage(
    SAFECAM_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    SAFECAM1_OBJECT_STRUCT *psObj = (SAFECAM1_OBJECT_STRUCT *)hInterface;
    SAFECAM1_MSG_TYPE_ENUM eMsgType = SAFECAM1_INVALID_MSG;

    printf(SAFECAM1_OBJECT_NAME": [%s] start with parameters "
        "hInterface = %p, phPayload = %p\n",
        __FUNCTION__, hInterface, phPayload
            );

    do
    {
        if (hInterface == SAFECAM_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        if (phPayload == NULL)
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // Process the weather message
        // and update the system
        eMsgType = eGetMessageType(*phPayload);
        printf(SAFECAM1_OBJECT_NAME": Message type is %u\n", eMsgType);

        switch (eMsgType)
        {
            case SAFECAM1_MSG_STU_METADATA:
            case SAFECAM1_MSG_STU:
            {
                RFD_INTERFACE_bProcessPayload(psObj->hRFDSTU, *phPayload);
                *phPayload = OSAL_INVALID_BUFFER_HDL;
            }
            break;

            case SAFECAM1_MSG_LTU_METADATA:
            case SAFECAM1_MSG_LTU:
            {
                RFD_INTERFACE_bProcessPayload(psObj->hRFDLTU, *phPayload);
                *phPayload = OSAL_INVALID_BUFFER_HDL;
            }
            break;

            case SAFECAM1_INVALID_MSG:
            default:
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": invalid message type received"
                        );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    printf(SAFECAM1_OBJECT_NAME": [%s] end with result %d\n", __FUNCTION__,
        bResult
            );

    return bResult;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
#if (DEBUG_OBJECT == 1)
/*******************************************************************************
*
*       vPrintPartitionInfo
*
*******************************************************************************/
static void vPrintPartitionInfo(SAFECAM1_PARTITION_DESC_STRUCT *psPartDesc)
{
    printf(SAFECAM1_OBJECT_NAME"------------------------\n");
    printf(SAFECAM1_OBJECT_NAME": Alert Type Update -"
            " %u\n", psPartDesc->bAlertTypeUpdate);
    printf(SAFECAM1_OBJECT_NAME": Version - "
            " %u\n", psPartDesc->un16Version);
    printf(SAFECAM1_OBJECT_NAME": Offset - "
            " %u\n", psPartDesc->un32ByteSeekOffset);
    printf(SAFECAM1_OBJECT_NAME"------------------------\n");
}
#endif


/*****************************************************************************
*
*   vUninitRFDCtrlFileData
*
*****************************************************************************/
static void vUninitRFDCtrlFileData (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if (psRFDCtrl->hPartsList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psRFDCtrl->hPartsList,
            (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to clean up PartsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        eReturnCode = OSAL.eLinkedListDelete(psRFDCtrl->hPartsList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to delete PartsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        psRFDCtrl->hPartsList = OSAL_INVALID_OBJECT_HDL;
    }

    psRFDCtrl->hCurrentPartEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (psRFDCtrl->hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        // clearing buffer to be sure that old file data will not be used
        // for next update
        eReturnCode = OSAL.eBufferFree(psRFDCtrl->hBuffer);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to free buffer(%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    psRFDCtrl->tStartProgressIndex = 0;
    psRFDCtrl->tCurProgressIndex = 0;
}


/*****************************************************************************
*
*   vUninitRFDCtrl
*
*****************************************************************************/
static void vUninitRFDCtrl (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bOwner = FALSE;

    if (psRFDCtrl == NULL)
    {
        return;
    }

    psRFDCtrl->hRFD = RFD_INTERFACE_INVALID_OBJECT;

    // Disconnect from the database
    if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psRFDCtrl->hDBPath);
        psRFDCtrl->hDBPath = STRING_INVALID_OBJECT;
    }

    vUninitRFDCtrlFileData(psRFDCtrl);

    vDestroyFixedConstants(&(psRFDCtrl->sFixed));

    // Destroy the file buffer
    DATASERVICE_MGR_vDestroyFileBuffer(psRFDCtrl->hBlockPool, psRFDCtrl->hBuffer);
    psRFDCtrl->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    psRFDCtrl->hBuffer = OSAL_INVALID_BUFFER_HDL;

    bOwner = SMSO_bIsOwner((SMS_OBJECT)psRFDCtrl);

    if (bOwner == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)psRFDCtrl);
    }

    return;
}


/*****************************************************************************
*
*   vDestroyRFDCtrl
*
*****************************************************************************/
static void vDestroyRFDCtrl (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl != NULL)
    {
        vUninitRFDCtrl(psRFDCtrl);
        SMSO_bLock((SMS_OBJECT)psRFDCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        SMSO_vDestroy((SMS_OBJECT)psRFDCtrl);
    }

    return;
}


/*******************************************************************************
*
*       vUninitObject
*
*******************************************************************************/
static void vUninitObject(SAFECAM1_OBJECT_STRUCT *psObj)
{
    if (psObj != NULL)
    {
        // Destroy the file buffer
        printf(SAFECAM1_OBJECT_NAME": Disconnecting RFD\n");

        if (psObj->hRFDLTU != RFD_INTERFACE_INVALID_OBJECT)
        {
            RFD_INTERFACE_vDisconnect(psObj->hRFDLTU);
            psObj->hRFDLTU = RFD_INTERFACE_INVALID_OBJECT;
        }

        if (psObj->hRFDSTU != RFD_INTERFACE_INVALID_OBJECT)
        {
            RFD_INTERFACE_vDisconnect(psObj->hRFDSTU);
            psObj->hRFDSTU = RFD_INTERFACE_INVALID_OBJECT;
        }

        if (psObj->psRFDCtrl != NULL)
        {
            vDestroyRFDCtrl(psObj->psRFDCtrl);
            psObj->psRFDCtrl = (SAFECAM1_RFD_CTRL_STRUCT*)NULL;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*******************************************************************************
*
*       bGetNumOfParts
*
*******************************************************************************/
static BOOLEAN bGetNumOfParts(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN16* pun16NumOfParts
        )
{
    UN8 un8Buffer = 0;
    size_t tBitsRead = 0;
    BOOLEAN bResult = TRUE;

    tBitsRead = OSAL.tBufferReadHeadBits(
        psObj->hBuffer, &un8Buffer, 0,
        SAFECAM1_PARTITIONS_BITLEN
            );
    if (tBitsRead != SAFECAM1_PARTITIONS_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read num of partitions from buffer"
                );
        bResult = FALSE;
    }

    *pun16NumOfParts = un8Buffer + 1;

    return bResult;
}

/*******************************************************************************
*
*       bGetVerMin
*
*******************************************************************************/
static BOOLEAN bGetVerMin(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN16* pun16VerMin
        )
{
    BOOLEAN bResult;

    // Read version
    bResult = OSAL.bBufferReadBitsToUN16(
        psObj->hBuffer, pun16VerMin,
        SAFECAM1_VER_MIN_BITLEN
            );
    if (bResult != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read ver min from buffer"
                );
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetAlertUpdateFlag
*
*******************************************************************************/
static BOOLEAN bGetAlertUpdateFlag(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    BOOLEAN *pbAlertTypeUpdate
        )
{
    UN8 un8Buffer = 0;
    size_t tBitsRead = 0;
    BOOLEAN bResult = TRUE;

    tBitsRead = OSAL.tBufferReadHeadBits(
        psObj->hBuffer, &un8Buffer, 0,
        SAFECAM1_FLAGS_BITLEN
            );
    if (tBitsRead != SAFECAM1_FLAGS_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read flags from buffer"
                );
        bResult = FALSE;
    }

    if (un8Buffer & (1 << SAFECAM1_ALERT_UPDATE_FLAG))
    {
        *pbAlertTypeUpdate = TRUE;
    }
    else
    {
        *pbAlertTypeUpdate = FALSE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetPartsList
*
*******************************************************************************/
static BOOLEAN bGetPartsList(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    OSAL_OBJECT_HDL hPartsList
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bAlertTypeUpdate = FALSE;
        BOOLEAN bSuccess;
        UN16 un16PartsNum = 0;
        SAFECAM1_PARTITION_DESC_STRUCT *psPartDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        bSuccess = bGetAlertUpdateFlag(psObj, &bAlertTypeUpdate);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": cannot get alert type update"
                    );
            break;
        }

        printf(SAFECAM1_OBJECT_NAME": Alert Update flag is %u\n", bAlertTypeUpdate);

        bSuccess = bGetNumOfParts(psObj, &un16PartsNum);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": cannot get num of partitions"
                    );
            break;
        }

        printf(SAFECAM1_OBJECT_NAME": Number of parts is %u\n", un16PartsNum);

        while (un16PartsNum-- > 0)
        {
            psPartDesc = (SAFECAM1_PARTITION_DESC_STRUCT*)
                SMSO_hCreate(SAFECAM1_OBJECT_NAME":PartDesc",
                sizeof(*psPartDesc), SMS_INVALID_OBJECT, FALSE
                    );
            if (psPartDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": cannot create part desc struct"
                        );
                bSuccess = FALSE;
                break;
            }

            bSuccess = OSAL.bBufferReadBitsToUN16(psObj->hBuffer,
                &(psPartDesc->un16Version), SAFECAM1_UPDATE_VER_BITLEN
                    );
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": cannot get update version"
                        );
                break;
            }

            bSuccess = OSAL.bBufferReadBitsToUN32(psObj->hBuffer,
                &(psPartDesc->un32ByteSeekOffset), SAFECAM1_PAR_INDEX_BITLEN
                    );
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": cannot get par index"
                        );
                break;
            }

            if (bAlertTypeUpdate == TRUE)
            {
                psPartDesc->bAlertTypeUpdate = TRUE;
                bAlertTypeUpdate = FALSE;
            }

#if (DEBUG_OBJECT == 1)
            vPrintPartitionInfo(psPartDesc);
#endif

            eReturnCode = OSAL.eLinkedListAdd(hPartsList,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psPartDesc);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": cannot add part desc to list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                bSuccess = FALSE;
                break;
            }
            psPartDesc = NULL;
        }

        if (psPartDesc != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psPartDesc);
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*       bGetSubDivision
*
*******************************************************************************/
static BOOLEAN bGetSubDivision(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN16* pun16SubDivision
        )
{
    BOOLEAN bResult = FALSE;

    bResult = OSAL.bBufferReadBitsToUN16(psObj->hBuffer,
        pun16SubDivision,
        SAFECAM1_ALERT_LOCATION_LOC_SUBDIVISION_BITLEN
            );
    if (bResult != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read subdivision from buffer"
                );
    }
    else
    {
        printf(SAFECAM1_OBJECT_NAME":SubDivision is %x\n",
            *pun16SubDivision
                );
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetFlag
*
*******************************************************************************/
static BOOLEAN bGetFlag(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    BOOLEAN *pbFlag
        )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;
    UN8 un8Buffer = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
        &un8Buffer, 0,
        SAFECAM1_FLAG_BITLEN
            );
    if (tBitsRead != SAFECAM1_FLAG_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read flag from buffer"
                );
    }
    else
    {
        bResult = TRUE;
        if (un8Buffer > 0)
        {
            *pbFlag = TRUE;
        }
        else
        {
            *pbFlag = FALSE;
        }
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetUID
*
*******************************************************************************/
static BOOLEAN bGetUID(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo,
    UN8 un8IdSize,
    UN32* pun32UID
        )
{
    BOOLEAN bResult = FALSE;
    UN16 un16Buffer = 0;

    bResult = OSAL.bBufferReadBitsToUN16(psObj->hBuffer, &un16Buffer,
        un8IdSize
            );
    if (bResult != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read ID from buffer"
                );
    }
    else
    {
        *pun32UID = SAFECAM1_UID_FROM_REG_ID(psRegionInfo->un16Region,
            un16Buffer
                );
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetSubRegion
*
*******************************************************************************/
static BOOLEAN bGetSubRegion(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8* pun8SubRegion
        )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, pun8SubRegion, 0,
        SAFECAM1_ALERT_LOCATION_LOC_SUBREGION_BITLEN
            );
    if (tBitsRead != SAFECAM1_ALERT_LOCATION_LOC_SUBREGION_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read subregion from buffer"
                );
    }
    else
    {
        printf(SAFECAM1_OBJECT_NAME":SubRegion is %x\n",
            *pun8SubRegion
                );
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetDivision
*
*******************************************************************************/
static BOOLEAN bGetDivision(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8* pun8Division
        )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, pun8Division, 0,
        SAFECAM1_ALERT_LOCATION_LOC_DIVISION_BITLEN
            );
    if (tBitsRead != SAFECAM1_ALERT_LOCATION_LOC_DIVISION_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read division from buffer"
                );
    }
    else
    {
        printf(SAFECAM1_OBJECT_NAME":Division is %x\n",
            *pun8Division
                );
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetLoc
*
*******************************************************************************/
static BOOLEAN bGetLoc(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo
        )
{
    BOOLEAN bResult = TRUE;
    size_t tBitsRead = 0;
    UN8 un8Buffer = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8Buffer, 0,
        SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_BITLEN
            );
    if (tBitsRead != SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read loc preamble from buffer"
                );
        return FALSE;
    }

    printf(SAFECAM1_OBJECT_NAME": Preamble is %u\n", un8Buffer);

    switch(un8Buffer)
    {
        case SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_SR:
        {
            bResult = bGetSubRegion(psObj, &(psRegionInfo->un8SubRegion));
            if (bResult == FALSE)
            {
                break;
            }
        }
        // Fall through to process next field

        case SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_D:
        {
            bResult = bGetDivision(psObj, &(psRegionInfo->un8Division));
            if (bResult == FALSE)
            {
                break;
            }
        }
        // Fall through to process next field

        case SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_SD:
        {
            bResult = bGetSubDivision(psObj, &(psRegionInfo->un16SubDivision));
        }
        // Fall through to process next field

        case SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_NC:
        {
            //nothing to do
        }
        break;

        default:
        {
            bResult = FALSE;
        }
        break;
    }

    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": loc preamble seems to be wrong"
                );
    }

    return bResult;
}

/*******************************************************************************
*
*       bProcessLocationDelete
*
*******************************************************************************/
static BOOLEAN bProcessLocationDelete(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo,
    UN8 un8IdSize
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        SAFECAM_ALERT_LOCATIONS_ROW_STRUCT sAlertLocation;
        BOOLEAN bSuccess = TRUE;

        OSAL.bMemSet(&sAlertLocation, 0, sizeof(sAlertLocation));

        bSuccess = bGetUID(psObj, psRegionInfo, un8IdSize,
            &(sAlertLocation.un32UID)
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get location UID"
                    );
            break;
        }
        printf(SAFECAM1_OBJECT_NAME": UID field is %u\n", sAlertLocation.un32UID);

        bSuccess = GsSafeCamMgr.bDeleteAlertLocation(
            psObj->hDBConnection,
            psObj->acBuffer,
            SAFECAM_MAX_SQL_STRING_LENGTH,
            &sAlertLocation );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to delete location"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bCreateFixedConstants
*
*   This function creates all fixed values for the calculations that
*   this interface must perform to convert grid points to a lat/lon pair
*
*****************************************************************************/
static BOOLEAN bCreateFixedConstants (
    SAFECAM1_FIXED_CALC_STRUCT *psFixed
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        // Initial latitude offset
        psFixed->hInitialLat = OSAL_FIXED.hCreateFromFixed(
            SAFECAM1_INITIAL_LAT, 0
                );
        if (psFixed->hInitialLat == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Initial longitude offset
        psFixed->hInitialLon = OSAL_FIXED.hCreateFromFixed(
            SAFECAM1_INITIAL_LON, 0
                );
        if (psFixed->hInitialLon == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // The lat / lon computation denominator
        psFixed->hLocDenominator = OSAL_FIXED.hCreateFromFixed(
            SAFECAM1_LOC_DENOMINATOR, 0
                );
        if (psFixed->hLocDenominator == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // The lat / lon offset denominator
        psFixed->hOffsetDenominator = OSAL_FIXED.hCreateFromFixed(
            SAFECAM1_OFFSET_DENOMINATOR, 0
                );
        if (psFixed->hOffsetDenominator == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        bResult = TRUE;
    }
    while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   vDestroyFixedConstants
*
*   Destroys all fixed constants used by this interface.
*
*****************************************************************************/
static void vDestroyFixedConstants (
    SAFECAM1_FIXED_CALC_STRUCT *psFixed
        )
{
    if (psFixed->hInitialLat != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psFixed->hInitialLat);
        psFixed->hInitialLat = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psFixed->hInitialLon != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psFixed->hInitialLon);
        psFixed->hInitialLon = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psFixed->hLocDenominator != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psFixed->hLocDenominator);
        psFixed->hLocDenominator = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psFixed->hOffsetDenominator != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psFixed->hOffsetDenominator);
        psFixed->hOffsetDenominator = OSAL_FIXED_INVALID_OBJECT;
    }

    return;
}

/*******************************************************************************
*
*       bConvertRegionToLoc
*
*******************************************************************************/
static BOOLEAN bConvertRegionToLoc(
    SAFECAM1_FIXED_CALC_STRUCT *psFixed,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        UN32 un32Lat = 0;
        UN32 un32Lon = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        un32Lat = SAFECAM1_GET_LAT_FROM_REGION(psRegionInfo->un16Region,
            psRegionInfo->un8SubRegion, psRegionInfo->un8Division,
            psRegionInfo->un16SubDivision);

        un32Lon = SAFECAM1_GET_LON_FROM_REGION(psRegionInfo->un16Region,
            psRegionInfo->un8SubRegion, psRegionInfo->un8Division,
            psRegionInfo->un16SubDivision);

        psFixed->hLat = OSAL_FIXED.hCreateInMemory(un32Lat, 0,
            &(psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * SAFECAM1_LAT_OBJ_INDEX])
                );

        psFixed->hLon = OSAL_FIXED.hCreateInMemory(un32Lon, 0,
            &(psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * SAFECAM1_LON_OBJ_INDEX])
                );

        /*-- Divide those values by the computation denominator --*/
        eReturnCode = OSAL_FIXED.eDivide(psFixed->hLat,
            psFixed->hLocDenominator, psFixed->hLat );
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eDivide(psFixed->hLon,
            psFixed->hLocDenominator, psFixed->hLon );
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Add that to the initial lat & lon
        eReturnCode = OSAL_FIXED.eAdd(psFixed->hLat,
            psFixed->hInitialLat, psFixed->hLat );
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eAdd(psFixed->hLon,
            psFixed->hInitialLon, psFixed->hLon );
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Latitude first
        psAlertLocation->n32Lat = OSAL_FIXED.n32ScaledValue(psFixed->hLat,
            LOCATION_BINPOINT
                );

        // Then longitude
        psAlertLocation->n32Lon = OSAL_FIXED.n32ScaledValue(psFixed->hLon,
            LOCATION_BINPOINT
                );

#if (DEBUG_OBJECT == 1)
        printf(SAFECAM1_OBJECT_NAME": Latitude:\n");
        OSAL_FIXED.n32FPrintf(psFixed->hLat, stdout, FALSE);
        printf(SAFECAM1_OBJECT_NAME": Longitude:\n");
        OSAL_FIXED.n32FPrintf(psFixed->hLon, stdout, FALSE);
#endif

        bResult = TRUE;
    } while (FALSE);

    return bResult;

}

/*******************************************************************************
*
*       bGetUnits
*
*******************************************************************************/
static BOOLEAN bGetUnits(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    BOOLEAN *pbMilesUnit
        )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;
    UN8 un8Buffer = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8Buffer, 0,
        SAFECAM1_ALERT_LOCATION_UNIT_BITLEN
            );
    if (tBitsRead != SAFECAM1_ALERT_LOCATION_UNIT_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read units from buffer"
                );
    }
    else
    {
        bResult = TRUE;
        if (un8Buffer == 0)
        {
            *pbMilesUnit = TRUE;
        }
        else
        {
            *pbMilesUnit = FALSE;
        }
    }

    return bResult;
}



/*******************************************************************************
*
*       bGetSubNo
*
*******************************************************************************/
static BOOLEAN bGetSubNo(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8* pun8SubNo
        )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, pun8SubNo, 0,
        SAFECAM1_ALERT_LOCATION_SUBNO_BITLEN
            );
    if (tBitsRead != SAFECAM1_ALERT_LOCATION_SUBNO_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read subno from buffer"
                );
    }
    else
    {
        (*pun8SubNo)++; //number of subs is subno + 1
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetExtCnt
*
*******************************************************************************/
static BOOLEAN bGetExtCnt(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN16* pun16ExtCnt
        )
{
    BOOLEAN bResult = FALSE;

    bResult = OSAL.bBufferReadBitsToUN16(psObj->hBuffer, pun16ExtCnt,
        SAFECAM1_ALERT_LOCATION_EXTCNT_BITLEN
            );
    if (bResult != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read ext cnt from buffer"
                );
    }
    else
    {
        (*pun16ExtCnt)++; //number of ext cnt is ext cnt + 1
    }

    return bResult;
}


/*******************************************************************************
*
*       bGetExtData
*
*******************************************************************************/
static BOOLEAN bGetExtData(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN16 un16ExtCnt,
    UN8* pun8ExtData
        )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, pun8ExtData, 0,
        un16ExtCnt
            );
    if (tBitsRead != un16ExtCnt)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read ext data from buffer"
                );
    }
    else
    {
        bResult = TRUE;
    }

    return bResult;
}


/*******************************************************************************
*
*       bGetSubId
*
*******************************************************************************/
static BOOLEAN bGetSubId(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8* pun8SubID
    )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, pun8SubID, 0,
        SAFECAM1_SUB_LOCATION_ID_BITLEN
            );
    if (tBitsRead != SAFECAM1_SUB_LOCATION_ID_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read sub id from buffer"
                );
    }
    else
    {
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetSubType
*
*******************************************************************************/
static BOOLEAN bGetSubType(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8* pun8SubType
    )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, pun8SubType, 0,
        SAFECAM1_SUB_LOCATION_TYPE_BITLEN
            );
    if (tBitsRead != SAFECAM1_SUB_LOCATION_TYPE_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read sub type from buffer"
                );
    }
    else
    {
        bResult = TRUE;
    }

    return bResult;
}


/*******************************************************************************
*
*       bGetSubOffset
*
*******************************************************************************/
static BOOLEAN bGetSubOffset(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8 un8OffsetSize,
    UN32* pun32SubOffset
    )
{
    BOOLEAN bResult = FALSE;

    bResult = OSAL.bBufferReadBitsToUN32(psObj->hBuffer, pun32SubOffset,
        un8OffsetSize
            );
    if (bResult != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read sub offset from buffer"
                );
    }

    return bResult;
}

/*******************************************************************************
*
*       bExtractLatLon
*
*******************************************************************************/
static void vExtractLatLon(
    const UN32 un32Offset,
    const UN8 un8ValSize,
    N16 *pn16Lat, N16 *pn16Lon
        )
{
    const UN16 un16SignBitMask = 1 << (un8ValSize - 1);
    const UN16 un16LatValue = (UN16)(un32Offset >> un8ValSize);

    *pn16Lat = un16LatValue & (un16SignBitMask - 1);
    if (un16LatValue & un16SignBitMask)
    {
        *pn16Lat *= -1;
    }
    *pn16Lon = un32Offset & (un16SignBitMask - 1);
    if (un32Offset & un16SignBitMask)
    {
        *pn16Lon *= -1;
    }

    return;
}

/*******************************************************************************
*
*       bGetNewSubLoc
*
*******************************************************************************/
static BOOLEAN bCalculateSubLoc(
    SAFECAM1_FIXED_CALC_STRUCT *psFixed,
    UN32 un32Offset,
    UN8 un8OffsetSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        N16 n16LatOffset = 0, n16LonOffset = 0;
        OSAL_FIXED_OBJECT hSubLat = OSAL_FIXED_INVALID_OBJECT;
        OSAL_FIXED_OBJECT hSubLon = OSAL_FIXED_INVALID_OBJECT;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        if ((psFixed->hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psFixed->hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            // location is not initialized for some reason
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": alert location not initialized"
                    );
            break;
        }

        vExtractLatLon(un32Offset, un8OffsetSize >> 1, &n16LatOffset, &n16LonOffset);

        hSubLat = OSAL_FIXED.hCreateInMemory(n16LatOffset, 0,
            &(psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * SAFECAM1_SUB_LAT_OBJ_INDEX])
                );
        hSubLon = OSAL_FIXED.hCreateInMemory(n16LonOffset, 0,
            &(psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * SAFECAM1_SUB_LON_OBJ_INDEX])
                );

        /*-- Divide those values by the denominator --*/
        eReturnCode = OSAL_FIXED.eDivide(hSubLat,
            psFixed->hOffsetDenominator, hSubLat
                );
        eReturnCode |= OSAL_FIXED.eDivide(hSubLon,
            psFixed->hOffsetDenominator, hSubLon
                );

        eReturnCode |= OSAL_FIXED.eAdd(hSubLat, psFixed->hLat, hSubLat);
        eReturnCode |= OSAL_FIXED.eAdd(hSubLon, psFixed->hLon, hSubLon);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        psSubLocation->n32Lat = OSAL_FIXED.n32ScaledValue(hSubLat, LOCATION_BINPOINT);
        psSubLocation->n32Lon = OSAL_FIXED.n32ScaledValue(hSubLon, LOCATION_BINPOINT);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*******************************************************************************
*
*       bGetSubDir
*
*******************************************************************************/
static BOOLEAN bGetSubDir(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN16* pun16Dir
        )
{
    BOOLEAN bResult = FALSE;

    bResult = OSAL.bBufferReadBitsToUN16(psObj->hBuffer, pun16Dir,
        SAFECAM1_SUB_LOCATION_DIR_BITLEN
            );
    if (bResult != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read sub dir from buffer"
                );
    }
    else
    {
        *pun16Dir *= SAFECAM_SUB_LOCATION_DIR_MULTIPLIER;
    }

    return bResult;
}


/*******************************************************************************
*
*       bGetSubSpeed
*
*******************************************************************************/
static BOOLEAN bGetSubSpeed(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN16* pun16Speed
        )
{
    BOOLEAN bResult = FALSE;
    UN8 un8Buffer = 0;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8Buffer, 0,
        SAFECAM1_SUB_LOCATION_SPEED_BITLEN
            );
    if (tBitsRead != SAFECAM1_SUB_LOCATION_SPEED_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read sub speed from buffer"
                );
    }
    else
    {
        *pun16Speed = (un8Buffer + 1) * SAFECAM1_SUB_LOCATION_SPEED_MULTIPLIER;
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetSubLocData
*
*******************************************************************************/
static BOOLEAN bGetSubLocData(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8 un8OffsetSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation,
    BOOLEAN *pbDelete
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE, bFlag = FALSE;
        UN32 un32Offset = 0;

        if (pbDelete != NULL)
        {
            *pbDelete = FALSE;
        }

        // getting type flag
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get sub type flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetSubType(psObj, &(psSubLocation->un8Type));
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get sub type"
                        );
                break;
            }
            else
            {
                if ((psSubLocation->un8Type == SAFECAM1_SUB_LOCATION_DELETE_FLAG) &&
                    (pbDelete != NULL))
                {
                    *pbDelete = TRUE;
                }
            }
        }

        // getting offset flag
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get offset flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetSubOffset(psObj, un8OffsetSize, &un32Offset);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get sub offset"
                        );
                break;
            }

            bSuccess = bCalculateSubLoc(&(psObj->sFixed), un32Offset,
                un8OffsetSize, psSubLocation
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to calculate sub loc"
                        );
                break;
            }
        }

        // getting dir flag
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get dir flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetSubDir(psObj, &(psSubLocation->un16Dir));
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get sub dir"
                        );
                break;
            }
        }

        // getting speed1 flag
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get speed1 flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetSubSpeed(psObj, &(psSubLocation->un16Speed1)
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get sub speed 1"
                        );
                break;
            }
        }

        // getting speed2 flag
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get speed2 flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetSubSpeed(psObj, &(psSubLocation->un16Speed2)
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get sub speed 2"
                        );
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);


    return bResult;
}

/*******************************************************************************
*
*       bGetOffsetSize
*
*******************************************************************************/
static BOOLEAN bGetOffsetSize(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8* pun8OffsetSize
        )
{
    BOOLEAN bResult = FALSE;
    UN8 un8Buffer = 0;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8Buffer, 0,
        SAFECAM1_ALERT_LOCATION_OSIZE_BITLEN
            );
    if (tBitsRead != SAFECAM1_ALERT_LOCATION_OSIZE_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read OSIZE from buffer"
                );
    }
    else
    {
        *pun8OffsetSize = SAFECAM1_GET_OFFSET_BITLEN(un8Buffer);
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetAlertLocationData
*
*******************************************************************************/
static BOOLEAN bGetAlertLocData(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation,
    UN8* pun8OffsetSize
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE, bFlag = FALSE;
        size_t tNumSymbolsFound = 0;

        bSuccess = bGetLoc(psObj, psRegionInfo);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get location info"
                    );
            break;
        }

        bSuccess = bConvertRegionToLoc(&psObj->sFixed, psRegionInfo,
            psAlertLocation
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to calculate location"
                    );
            break;
        }

        bSuccess = bGetOffsetSize(psObj, pun8OffsetSize);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed read OSIZE"
                    );
            break;
        }

        // getting flag for description field
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get desc flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            if (psAlertLocation->hInfo != STRING_INVALID_OBJECT)
            {
                SMSO_vDestroy((SMS_OBJECT)psAlertLocation->hInfo);
            }
            psAlertLocation->hInfo =
                BAUDOT_hToString((SMS_OBJECT)psObj, psObj->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END, TRUE, TRUE,
                    SAFECAM1_ALERT_LOCATION_DESC_MAX_SYMBOLS,
                    &tNumSymbolsFound
                        );
            if (psAlertLocation->hInfo == STRING_INVALID_OBJECT)
            {
                //something went wrong
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get desc string"
                        );
                break;
            }
#if (DEBUG_OBJECT == 1)
            printf(SAFECAM1_OBJECT_NAME": Name: ");
            STRING.n32FWrite(psAlertLocation->hInfo, stdout);
            printf("\n");
#endif
        }

        // getting flag for units field
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get units flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetUnits(psObj, &(psAlertLocation->bMilesUnits));
            if (bSuccess == FALSE)
            {
                //something went wrong
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get units"
                        );
                break;
            }
        }

        // getting flag for subno field
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get subno flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetSubNo(psObj, &(psAlertLocation->un8NoSub));
            if (bSuccess == FALSE)
            {
                //something went wrong
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get subno"
                        );
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*       bProcessLocationNew
*
*******************************************************************************/
static BOOLEAN bProcessLocationNew(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo,
    UN8 un8IdSize
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT sAlertLocation;

    OSAL.bMemSet(&sAlertLocation, 0, sizeof(sAlertLocation));

    do
    {
        BOOLEAN bSuccess = TRUE;
        BOOLEAN bFlag = FALSE;
        UN8 un8SubsLeft = 0, un8OffsetSize = 0;
        UN16 un16ExtCnt = 0;

        bSuccess = bGetUID(psObj, psRegionInfo, un8IdSize,
            &(sAlertLocation.un32UID)
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get location UID"
                    );
            break;
        }
        printf(SAFECAM1_OBJECT_NAME": UID field is %u\n", sAlertLocation.un32UID);

        bSuccess = bGetAlertLocData(psObj, psRegionInfo,
            &sAlertLocation, &un8OffsetSize
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get alert location data"
                    );
            break;
        }

        un8SubsLeft = sAlertLocation.un8NoSub;

        while (un8SubsLeft-- > 0)
        {
            SAFECAM_SUB_LOCATIONS_ROW_STRUCT sSubLocation;

            OSAL.bMemSet(&sSubLocation, 0, sizeof(sSubLocation));

            sSubLocation.un32UID = sAlertLocation.un32UID;

            //getting subid flag
            bSuccess = bGetFlag(psObj, &bFlag);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get subid flag"
                        );
                break;
            }

            if (bFlag == TRUE)
            {
                // getting subid
                bSuccess = bGetSubId(psObj, &(sSubLocation.un8SubID));
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM1_OBJECT_NAME": failed to get sub id"
                            );
                    break;
                }

                bSuccess = bGetSubLocData(psObj, un8OffsetSize,
                    &sSubLocation, (BOOLEAN*)NULL
                        );
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM1_OBJECT_NAME": failed to get sub loc data"
                            );
                    break;
                }

                bSuccess = GsSafeCamMgr.bAddSubLocation(psObj->hDBConnection,
                    psObj->acBuffer, SAFECAM_MAX_SQL_STRING_LENGTH,
                    &sSubLocation );

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM1_OBJECT_NAME": failed to add new sub location"
                            );
                    break;
                }
            }
        }

        // getting flag for extension field
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get extension flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetExtCnt(psObj, &un16ExtCnt);
            if (bSuccess == FALSE)
            {
                //something went wrong
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get ext cnt"
                        );
                break;
            }

            bSuccess = bGetExtData(psObj, un16ExtCnt, (UN8*)NULL);
            if (bSuccess == FALSE)
            {
                //something went wrong
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get ext data"
                        );
                break;
            }
        }

        bSuccess = GsSafeCamMgr.bAddAlertLocation(psObj->hDBConnection,
            psObj->acBuffer, SAFECAM_MAX_SQL_STRING_LENGTH,
            &sAlertLocation );
        if (bSuccess == FALSE)
        {
            //something went wrong
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to add alert location"
                    );
            break;
        }

        bSuccess = GsSafeCamMgr.bUpdateAlertLocationRtree(
            psObj->hDBConnection, psObj->acBuffer,
            SAFECAM_MAX_SQL_STRING_LENGTH, &sAlertLocation );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to set alert location rtree"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (sAlertLocation.hInfo != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sAlertLocation.hInfo);
        sAlertLocation.hInfo = STRING_INVALID_OBJECT;
    }

    return bResult;
}

/*******************************************************************************
*
*       bProcessLocationModify
*
*******************************************************************************/
static BOOLEAN bProcessLocationModify(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo,
    UN8 un8IdSize
            )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT sAlertLocation;

    OSAL.bMemSet(&sAlertLocation, 0, sizeof(sAlertLocation));

    do
    {
        BOOLEAN bSuccess = TRUE, bDoNotApply = FALSE;
        BOOLEAN bFlag = FALSE;
        UN8 un8SubsLeft = 0, un8OffsetSize = 0;
        UN16 un16ExtCnt = 0;

        bSuccess = bGetUID(psObj, psRegionInfo, un8IdSize,
            &(sAlertLocation.un32UID)
                );

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get location UID"
                    );
            break;
        }

        bSuccess = GsSafeCamMgr.bGetAlertLocationData(
            psObj->hDBConnection,
            psObj->acBuffer,
            SAFECAM_MAX_SQL_STRING_LENGTH,
            &sAlertLocation
                );
        if (bSuccess == FALSE)
        {
            printf(SAFECAM1_OBJECT_NAME": Alert location UID"
                " %x cannot be read from DB\n", sAlertLocation.un32UID
                    );
            bDoNotApply = TRUE;
        }


        bSuccess = bGetAlertLocData(psObj, psRegionInfo, &sAlertLocation,
            &un8OffsetSize
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get alert location data"
                    );
            break;
        }

        un8SubsLeft = sAlertLocation.un8NoSub;

        while (un8SubsLeft-- > 0)
        {
            SAFECAM_SUB_LOCATIONS_ROW_STRUCT sSubLocation;

            OSAL.bMemSet(&sSubLocation, 0, sizeof(sSubLocation));
            sSubLocation.un32UID = sAlertLocation.un32UID;

            //getting subid flag
            bSuccess = bGetFlag(psObj, &bFlag);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get subid flag"
                        );
                break;
            }

            if (bFlag == TRUE)
            {
                BOOLEAN bDeleteSubLoc = FALSE;

                // getting subid
                bSuccess = bGetSubId(psObj, &(sSubLocation.un8SubID));
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM1_OBJECT_NAME": failed to get sub id"
                            );
                    break;
                }

                if (bDoNotApply == FALSE)
                {
                    bSuccess = GsSafeCamMgr.bGetSubLocationData(
                        psObj->hDBConnection,
                        psObj->acBuffer,
                        SAFECAM_MAX_SQL_STRING_LENGTH,
                        &sSubLocation );
                    if (bSuccess == FALSE)
                    {
                        printf(SAFECAM1_OBJECT_NAME": failed to get sub"
                            " loc data from DB\n"
                                );
                    }
                }

                bSuccess = bGetSubLocData(psObj, un8OffsetSize,
                    &sSubLocation, &bDeleteSubLoc
                        );
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM1_OBJECT_NAME": failed to get sub loc data"
                            );
                    break;
                }

                if (bDoNotApply == FALSE)
                {
                    // this is delete case per 6.2.4.9 sect
                    if (bDeleteSubLoc == TRUE)
                    {
                        bSuccess = GsSafeCamMgr.bDeleteSubLocation(
                            psObj->hDBConnection,
                            psObj->acBuffer,
                            SAFECAM_MAX_SQL_STRING_LENGTH,
                            &sSubLocation
                                );
                    }
                    else
                    {
                        bSuccess = GsSafeCamMgr.bModifySubLocation(
                            psObj->hDBConnection,
                            psObj->acBuffer,
                            SAFECAM_MAX_SQL_STRING_LENGTH,
                            &sSubLocation
                                );
                    }
                    if (bSuccess == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            SAFECAM1_OBJECT_NAME": failed to modify sub location"
                                );
                        break;
                    }
                }
            }
        }

        // getting flag for extension field
        bSuccess = bGetFlag(psObj, &bFlag);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get extension flag"
                    );
            break;
        }

        if (bFlag == TRUE)
        {
            bSuccess = bGetExtCnt(psObj, &un16ExtCnt);
            if (bSuccess == FALSE)
            {
                //something went wrong
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get ext cnt"
                        );
                break;
            }

            bSuccess = bGetExtData(psObj, un16ExtCnt, (UN8*)NULL);
            if (bSuccess == FALSE)
            {
                //something went wrong
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to get ext data"
                        );
                break;
            }
        }

        if (bDoNotApply == FALSE)
        {
            bSuccess = GsSafeCamMgr.bModifyAlertLocation(
                psObj->hDBConnection,
                psObj->acBuffer,
                SAFECAM_MAX_SQL_STRING_LENGTH,
                &sAlertLocation );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to modify alert location" );
                break;
            }

            bSuccess = GsSafeCamMgr.bUpdateAlertLocationRtree(
                psObj->hDBConnection,
                psObj->acBuffer,
                SAFECAM_MAX_SQL_STRING_LENGTH,
                &sAlertLocation );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to set alert location rtree" );
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    if (sAlertLocation.hInfo != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sAlertLocation.hInfo);
        sAlertLocation.hInfo = STRING_INVALID_OBJECT;
    }

    return bResult;
}

/*******************************************************************************
*
*       bGetType
*
*******************************************************************************/
static BOOLEAN bGetType(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    UN8* pun8Type
        )
{
    BOOLEAN bResult = FALSE;
    size_t tBitsRead = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, pun8Type, 0,
        SAFECAM1_ALERT_TYPE_TYPE_BITLEN
            );
    if (tBitsRead != SAFECAM1_ALERT_TYPE_TYPE_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": cannot read alert type from buffer"
                );
    }
    else
    {
        bResult = TRUE;
    }

    return bResult;
}


/*******************************************************************************
*
*       bGetTypeDesc
*
*******************************************************************************/
static BOOLEAN bGetTypeDesc(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    STRING_OBJECT *phDesc
        )
{
    size_t tNumSymbolsFound = 0;
    BOOLEAN bResult = FALSE;

    *phDesc =
        BAUDOT_hToString((SMS_OBJECT)psObj, psObj->hBuffer,
            BAUDOT_BEHAVIOR_PROCESS_TO_END, TRUE, TRUE,
            SAFECAM1_ALERT_TYPE_DESC_MAX_SYMBOLS,
            &tNumSymbolsFound
                );
    if (*phDesc == STRING_INVALID_OBJECT)
    {
        //something went wrong
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM1_OBJECT_NAME": failed to get desc string"
                );
    }
    else
    {
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
*
*       bProcessTypeModify
*
*******************************************************************************/
static BOOLEAN bProcessTypeModify(
    SAFECAM1_RFD_CTRL_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_ALERT_TYPES_ROW_STRUCT sAlertType;

    OSAL.bMemSet(&sAlertType, 0, sizeof(sAlertType));

    do
    {
        BOOLEAN bSuccess = TRUE;

        bSuccess = bGetType(psObj, &(sAlertType.un8Type));
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get alert type"
                    );
            break;
        }

        bSuccess = bGetTypeDesc(psObj, &(sAlertType.hDesc));
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get alert type desc" );
            break;
        }

        bSuccess = GsSafeCamMgr.bAddAlertType(
            psObj->hDBConnection,
            psObj->acBuffer,
            SAFECAM_MAX_SQL_STRING_LENGTH,
            &sAlertType );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to modify alert type" );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (sAlertType.hDesc != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sAlertType.hDesc);
        sAlertType.hDesc = STRING_INVALID_OBJECT;
    }

    return bResult;
}

/*******************************************************************************
*
*       bProcessTypeNew
*
*******************************************************************************/
static BOOLEAN bProcessTypeNew(
    SAFECAM1_RFD_CTRL_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_ALERT_TYPES_ROW_STRUCT sAlertType;

    OSAL.bMemSet(&sAlertType, 0, sizeof(sAlertType));

    do
    {
        BOOLEAN bSuccess = TRUE;

        bSuccess = bGetType(psObj, &(sAlertType.un8Type));
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get alert type"
                    );
            break;
        }

        bSuccess = bGetTypeDesc(psObj, &(sAlertType.hDesc));
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get alert type desc" );
            break;
        }

        bSuccess = GsSafeCamMgr.bAddAlertType(
            psObj->hDBConnection,
            psObj->acBuffer,
            SAFECAM_MAX_SQL_STRING_LENGTH,
            &sAlertType );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to add alert type" );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (sAlertType.hDesc != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sAlertType.hDesc);
        sAlertType.hDesc = STRING_INVALID_OBJECT;
    }

    return bResult;
}

/*******************************************************************************
*
*       bProcessTypeDelete
*
*******************************************************************************/
static BOOLEAN bProcessTypeDelete(SAFECAM1_RFD_CTRL_STRUCT *psObj)
{
    BOOLEAN bResult = FALSE;

    do
    {
        SAFECAM_ALERT_TYPES_ROW_STRUCT sAlertType;
        BOOLEAN bSuccess = TRUE;

        OSAL.bMemSet(&sAlertType, 0, sizeof(sAlertType));

        bSuccess = bGetType(psObj, &(sAlertType.un8Type));
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get alert type" );
            break;
        }
        printf(SAFECAM1_OBJECT_NAME": ID field is %u\n", sAlertType.un8Type);

        bSuccess = GsSafeCamMgr.bDeleteAlertType(
            psObj->hDBConnection,
            psObj->acBuffer,
            SAFECAM_MAX_SQL_STRING_LENGTH,
            &sAlertType );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to delete alert type"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*       bGetTypeUpdate
*
*******************************************************************************/
static BOOLEAN bGetTypeUpdate(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    FILE *psFile
        )
{
    BOOLEAN bResult = TRUE;

    do
    {
        UN8 un8Buffer = 0;
        size_t tBitsRead = 0;

        // Grab more data
        DATASERVICE_MGR_bFillBufferBlock(psFile, psObj->hBuffer);

        tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8Buffer, 0,
            SAFECAM1_ALERT_TYPE_TUTYPE_BITLEN
                );
        if (tBitsRead != SAFECAM1_ALERT_TYPE_TUTYPE_BITLEN)
        {
            bResult = FALSE;
            break;
        }

        printf(SAFECAM1_OBJECT_NAME": Update type field is %u\n", un8Buffer);

        switch (un8Buffer)
        {
            case SAFECAM1_ALERT_TYPE_TUTYPE_DELETE:
            {
                bResult = bProcessTypeDelete(psObj);
            }
            break;

            case SAFECAM1_ALERT_TYPE_TUTYPE_NEW:
            {
                bResult = bProcessTypeNew(psObj);
            }
            break;

            case SAFECAM1_ALERT_TYPE_TUTYPE_MODIFY:
            {
                bResult = bProcessTypeModify(psObj);
            }
            break;

            default:
            {
                bResult = FALSE;
            }
            break;
        }
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*       bGetRegionUpdates
*
*******************************************************************************/
static BOOLEAN bGetRegionUpdates(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    FILE *psFile,
    SAFECAM1_REGION_DATA_STRUCT *psRegionInfo,
    UN8 un8IdSize
        )
{
    BOOLEAN bResult = TRUE;
    BOOLEAN bEOF = FALSE;

    do
    {
        UN8 un8Buffer = 0;
        size_t tBitsRead = 0;

        // Grab more data
        DATASERVICE_MGR_bFillBufferBlock(psFile, psObj->hBuffer);

        tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8Buffer, 0,
            SAFECAM1_ALERT_LOCATION_UTYPE_BITLEN
                );
        if (tBitsRead != SAFECAM1_ALERT_LOCATION_UTYPE_BITLEN)
        {
            bResult = FALSE;
            break;
        }

        printf(SAFECAM1_OBJECT_NAME": Update type field is %u\n", un8Buffer);

        switch (un8Buffer)
        {
            case SAFECAM1_ALERT_LOCATION_UTYPE_DEL:
            {
                bResult = bProcessLocationDelete(psObj, psRegionInfo, un8IdSize);
            }
            break;

            case SAFECAM1_ALERT_LOCATION_UTYPE_NEW:
            {
                bResult = bProcessLocationNew(psObj, psRegionInfo, un8IdSize);
            }
            break;

            case SAFECAM1_ALERT_LOCATION_UTYPE_MOD:
            {
                bResult = bProcessLocationModify(psObj, psRegionInfo, un8IdSize);
            }
            break;

            case SAFECAM1_ALERT_LOCATION_UTYPE_EOF:
            {
                bEOF = TRUE;
            }
            break;

            default:
            {
                bResult = FALSE;
            }
            break;
        }
    } while ((bResult == TRUE) && (bEOF == FALSE));

    return bResult;
}

/*******************************************************************************
*
*       bApplyAlertLocationUpdate
*
*******************************************************************************/
static BOOLEAN bApplyAlertLocationUpdate(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_PARTITION_DESC_STRUCT *psPartDesc,
    FILE *psFile)
{
    BOOLEAN bResult = FALSE;

    do
    {
        int iResult = 0;
        BOOLEAN bSuccess = TRUE;
        UN16 un16RCount = 0;
        size_t tBitsRead = 0;

        iResult = fseek(psFile, psPartDesc->un32ByteSeekOffset, SEEK_SET);
        if (iResult != 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to set file position"
                    );
            break;
        }

        bSuccess = DATASERVICE_MGR_bFillBufferBlock(
            psFile, psObj->hBuffer
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to copy data to block"
                    );
            break;
        }

        bSuccess = OSAL.bBufferReadBitsToUN16(psObj->hBuffer, &un16RCount,
            SAFECAM1_PARTITION_RCOUNT_BITLEN
                );
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": cannot get region count"
                    );
            break;
        }
        un16RCount++;   // number of regions is RCOUNT + 1
        printf(SAFECAM1_OBJECT_NAME": Number of regions is %u\n", un16RCount);

        while(un16RCount-- > 0)
        {
            UN8 un8IdSize = 0;
            SAFECAM1_REGION_DATA_STRUCT sRegionInfo;

            OSAL.bMemSet(&sRegionInfo, 0, sizeof(sRegionInfo));

            // Grab more data
            DATASERVICE_MGR_bFillBufferBlock(
                        psFile, psObj->hBuffer);

            bSuccess = OSAL.bBufferReadBitsToUN16(psObj->hBuffer,
                &(sRegionInfo.un16Region), SAFECAM1_REGION_AREGION_BITLEN
                    );
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": cannot get aregion field"
                        );
                break;
            }

            printf(SAFECAM1_OBJECT_NAME": ARegion field is %u\n",
                sRegionInfo.un16Region
                    );

            tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8IdSize,
                0, SAFECAM1_REGION_IDSIZE_BITLEN
                    );
            if (tBitsRead != SAFECAM1_REGION_IDSIZE_BITLEN)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": cannot get idsize field"
                        );
                bSuccess = FALSE;
                break;
            }
            un8IdSize++; // ID size in bits = IDSIZE + 1
            printf(SAFECAM1_OBJECT_NAME": ID size field is %u\n", un8IdSize);

            bSuccess = bGetRegionUpdates(psObj, psFile, &sRegionInfo, un8IdSize);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to read region updates"
                        );
                break;
            }

        }

        if (bSuccess == FALSE)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*       bApplyAlertTypeUpdate
*
*******************************************************************************/
static BOOLEAN bApplyAlertTypeUpdate(
    SAFECAM1_RFD_CTRL_STRUCT *psObj,
    SAFECAM1_PARTITION_DESC_STRUCT *psPartDesc,
    FILE *psFile
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        int iResult = 0;
        BOOLEAN bSuccess = FALSE;
        UN8 un8TypeNum = 0;
        size_t tBitsRead = 0;

        iResult = fseek(psFile, psPartDesc->un32ByteSeekOffset, SEEK_SET);
        if (iResult != 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to set file position"
                    );
            break;
        }

        bSuccess = DATASERVICE_MGR_bFillBufferBlock(
            psFile, psObj->hBuffer
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to copy data to block"
                    );
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer, &un8TypeNum, 0,
            SAFECAM1_ALERT_TYPE_TYPE_NUM_BITLEN
                );
        if (tBitsRead != SAFECAM1_ALERT_TYPE_TYPE_NUM_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": cannot get num of alert type updates"
                    );
            break;
        }
        un8TypeNum++;   // number of types is TYPE_NUM + 1
        printf(SAFECAM1_OBJECT_NAME": Number of types is %u\n", un8TypeNum);

        while(un8TypeNum-- > 0)
        {
            bSuccess = bGetTypeUpdate(psObj, psFile);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to read type update"
                        );
                break;
            }
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   eGetMessageType
*
*****************************************************************************/
static SAFECAM1_MSG_TYPE_ENUM eGetMessageType(
    OSAL_BUFFER_HDL hPayload
        )
{
    UN8 un8CarouselID = 0;
    PVN tPVN = (PVN)0;
    size_t tBitsRead;
    SAFECAM1_MSG_TYPE_ENUM eMsgType = SAFECAM1_INVALID_MSG;

    do
    {
        // Peek at the PVN
        tBitsRead = OSAL.tBufferPeekBits(
            hPayload, &tPVN, 0,
            SAFECAM1_PVN_BITLEN, 0
                );

        if (tBitsRead != SAFECAM1_PVN_BITLEN)
        {
            // Peek failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            printf(SAFECAM1_OBJECT_NAME":  Unable to read PVN\n");
            break;
        }

        // Verify the PVN
        if (tPVN != SAFECAM1_PVN)
        {
            printf(SAFECAM1_OBJECT_NAME": Incorrect PVN - got %u, expected %u\n",
                tPVN, SAFECAM1_PVN
                    );
            break;
        }

        // Peek at the Carousel Id
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8CarouselID, 0,
            SAFECAM1_CAROUSEL_BITLEN, SAFECAM1_PVN_BITLEN
                );
        if (tBitsRead != SAFECAM1_CAROUSEL_BITLEN)
        {
            // Peek failed -- the message is probably
            // garbled
            printf(SAFECAM1_OBJECT_NAME":  Unable to read Carousel Id\n");
            break;
        }

        // determine message type
        switch (un8CarouselID)
        {
            case SAFECAM1_STU_CAROUSEL_ID:
            {
                eMsgType = SAFECAM1_MSG_STU;
            }
            break;
            case SAFECAM1_LTU_CAROUSEL_ID:
            {
                eMsgType = SAFECAM1_MSG_LTU;
            }
            break;
            case SAFECAM1_METADATA_CAROUSEL_ID:
            {
                eMsgType = eGetMetadataType(hPayload);
            }
            break;
            default:
            break;
        }
    } while (FALSE);

    return eMsgType;
}


/*****************************************************************************
*
*   bRFDFileVersionReader
*
*****************************************************************************/
static BOOLEAN bRFDFileVersionReader(
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptVersionMin,
    RFD_UPDATE_VERSION *ptFileVersion
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        char acBuffer[SAFECAM1_UPDATE_VER_STRING_LEN + 1];
        UN8 un8Idx = 0;
        size_t tFileNameStrLen = 0;

        // Verify that the length of pcFileName is correct
        tFileNameStrLen = strlen(pcFileName);

        if (tFileNameStrLen != SAFECAM1_UPDATE_FILE_NAME_LEN)
        {
            break;
        }

        // Verify that the first character is 'U'
        if (*pcFileName != SAFECAM1_UPDATE_FILE_START_CHAR)
        {
            break;
        }

        pcFileName++;

        // Verify that the 2nd character is 'S' or 'L'
        if ((*pcFileName != SAFECAM1_UPDATE_FILE_LTU_CHAR) &&
            (*pcFileName != SAFECAM1_UPDATE_FILE_STU_CHAR))
        {
            break;
        }

        pcFileName++;

        // Parse the first 5 characters into a number
        for (un8Idx = 0; un8Idx < SAFECAM1_UPDATE_VER_STRING_LEN;
            un8Idx++, pcFileName++)
        {
            // Offsetting un8Idx by one to account for the first char
            acBuffer[un8Idx] = *pcFileName;
        }

        acBuffer[SAFECAM1_UPDATE_VER_STRING_LEN] = '\0';

        *ptVersionMin = (RFD_UPDATE_VERSION)
            atoi((const char *)acBuffer);

        // skipping separator
        pcFileName++;

        // Parse the last 5 characters into a number
        for (un8Idx = 0; un8Idx < SAFECAM1_UPDATE_VER_STRING_LEN; un8Idx++,
            pcFileName++)
        {
            // Offsetting un8Idx by one to account for the first char
            // and the first version number
            acBuffer[un8Idx] = *pcFileName;
        }

        *ptFileVersion = (RFD_UPDATE_VERSION)
            atoi((const char *)acBuffer);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   vNotifyDBExpired
*
*****************************************************************************/
static void vNotifyDBExpired(
    RFD_INTERFACE_OBJECT hConnection,
    void *pvCallbackArg
        )
{
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl =
        (SAFECAM1_RFD_CTRL_STRUCT*)pvCallbackArg;

    if (psRFDCtrl != NULL)
    {
        GsSafeCamMgr.vTooOldUpdate(psRFDCtrl->hSafeCamService);
    }

    return;
}


/*****************************************************************************
*
*   bConnectToDB
*
*****************************************************************************/
static BOOLEAN bConnectToDB (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        )
{
    SQL_INTERFACE_OPTIONS_MASK tOptions = SQL_INTERFACE_OPTIONS_NONE;
    BOOLEAN bConnected = FALSE;

    if (psRFDCtrl->hDBPath != NULL)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);

        if (bPerformIntegrityCheck == FALSE)
        {
            tOptions |= SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK;
        }

        // Connect to the database
        psRFDCtrl->hDBConnection = SQL_INTERFACE.hConnect(
            pcDBPath,
            tOptions,
            (DATASERVICE_ERROR_CODE_ENUM *)NULL );

        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            bConnected = TRUE;
        }
    }

    return bConnected;
}

/*****************************************************************************
*
*   bFinalizeDB
*
*****************************************************************************/
static BOOLEAN bFinalizeDB (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bDBReady;

    bDBReady = bConnectToDB(psRFDCtrl, TRUE);
    if (bDBReady == TRUE)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    return bDBReady;
}

/*****************************************************************************
*
*   vDeleteDB
*
*****************************************************************************/
static void vDeleteDB (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);
        remove(pcDBPath);
    }
    return;
}

/*****************************************************************************
*
*   bInitRFDCtrl
*
*****************************************************************************/
static BOOLEAN bInitRFDCtrl (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex,
    SAFECAM1_RFD_TYPE_ENUM eUpdateType
        )
{
    BOOLEAN bResult = FALSE, bLocked = FALSE;
    STRING_OBJECT hCurDB = STRING_INVALID_OBJECT;

    do
    {
        BOOLEAN bSuccess = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        bLocked = SMSO_bLock((SMS_OBJECT)psRFDCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to lock RFD Ctrl object"
                    );
            break;
        }

        psRFDCtrl->eRFDType = eUpdateType;

        // Store the RFD connection handle
        psRFDCtrl->hRFD = hRFD;

        // Initialize the structure
        psRFDCtrl->tStartProgressIndex = 0;
        psRFDCtrl->tCurProgressIndex = 0;

        // Create FIXED constants that are used to work
        // with location updates
        bSuccess = bCreateFixedConstants(&psRFDCtrl->sFixed);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to create fixed constants"
                    );
            break;
        }

        // Create our file reader buffer
        bSuccess = DATASERVICE_MGR_bCreateFileBuffer(
            &psRFDCtrl->hBlockPool,
            &psRFDCtrl->hBuffer,
            SAFECAM1_RFD_READ_BLOCK_SIZE
                );

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to create file reader buffer"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&psRFDCtrl->hPartsList,
            SAFECAM1_OBJECT_NAME":PartsList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR|OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to create PartsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Find out where to place our working copy
        bSuccess = GsSafeCamMgr.bRefDBBank(
            psRFDCtrl->hSafeCamService, &hCurDB, &psRFDCtrl->hDBPath
                );
        if (bSuccess == FALSE)
        {
            break;
        }

        // Are we starting update from the beginning?
        if (tStartingIndex == 0)
        {
            // Yes!  Get a copy of the db
            bSuccess = DB_UTIL_bCopyDB(
                STRING.pacCStr(psRFDCtrl->hDBPath),
                STRING.pacCStr(hCurDB)
                    );
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Connect to our working database file now
        bSuccess = bConnectToDB(psRFDCtrl, FALSE);
        if (bSuccess == FALSE)
        {
            break;
        }

        // if update starts from the beginning, need to prepare DB
        if (tStartingIndex == 0)
        {
            bSuccess = GsSafeCamMgr.bDBUpdateBegin(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->acBuffer,
                SAFECAM_MAX_SQL_STRING_LENGTH
                    );
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    if (hCurDB != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hCurDB);
    }

    if ((bResult == FALSE) && (bLocked == TRUE))
    {
        SMSO_vUnlock((SMS_OBJECT)psRFDCtrl);
    }

    return bResult;
}

/*****************************************************************************
*
*   bProcessRFDHeader
*
*****************************************************************************/
static BOOLEAN bProcessRFDHeader (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psRFDFile
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;
        UN16 un16VerMin = 0;

        // Fill the buffer with file data
        bSuccess = DATASERVICE_MGR_bFillBufferBlock(
            psRFDFile, psRFDCtrl->hBuffer
                );
        if (bSuccess == FALSE)
        {
            // Couldn't access the file
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": Unable to read file data to buffer"
                );
            break;
        }

        // reading ver min
        bSuccess = bGetVerMin(psRFDCtrl, &un16VerMin);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get ver min"
                    );
            break;
        }

        psRFDCtrl->tFileBaseVersion = (RFD_UPDATE_VERSION)un16VerMin;

        bSuccess = bGetPartsList(psRFDCtrl, psRFDCtrl->hPartsList);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to get partitions list"
                    );
            break;
        }

        // Time to clear out our buffer -- just seek past everything
        OSAL.tBufferSeekHeadBits(psRFDCtrl->hBuffer, N32_MAX);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   eProcessUpdateGroup
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessUpdateGroup (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;

    do
    {
        SAFECAM1_PARTITION_DESC_STRUCT *psPartDesc =
            (SAFECAM1_PARTITION_DESC_STRUCT*)NULL;
        BOOLEAN bSuccess = FALSE;

        // need to init current partition if empty
        if (psRFDCtrl->hCurrentPartEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            psRFDCtrl->hCurrentPartEntry =
                OSAL.hLinkedListFirst(psRFDCtrl->hPartsList,
                (void**)&psPartDesc
                    );
        }
        else
        {
            psRFDCtrl->hCurrentPartEntry =
                OSAL.hLinkedListNext(psRFDCtrl->hCurrentPartEntry,
                (void**)&psPartDesc
                    );
        }

        // if current entry is NULL, we've completed all parts
        if (psRFDCtrl->hCurrentPartEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eResult = RFD_PROCESS_RESULT_COMPLETE;
            break;
        }

        //reinit buffer for every partition to avoid block issues
        if (psRFDCtrl->hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            eReturnCode = OSAL.eBufferFree(psRFDCtrl->hBuffer);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to free buffer (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }

            psRFDCtrl->hBuffer = OSAL.hBufferAllocate(
                psRFDCtrl->hBlockPool, FALSE, FALSE,
                OSAL_BUFFER_ALLOCATE_OPTION_NONE
                    );
            if (psRFDCtrl->hBuffer == OSAL_INVALID_BUFFER_HDL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to allocate buffer",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        if (psPartDesc->bAlertTypeUpdate == TRUE)
        {
            bSuccess =
                bApplyAlertTypeUpdate(psRFDCtrl, psPartDesc, psFile);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to apply alert"
                    " type update"
                        );
                break;
            }
        }
        else
        {
            bSuccess =
                bApplyAlertLocationUpdate(psRFDCtrl, psPartDesc, psFile);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM1_OBJECT_NAME": failed to apply alert"
                    " location update"
                        );
                break;
            }
        }

        eResult = RFD_PROCESS_RESULT_INCOMPLETE;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eProcessNextRFDEntry
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    RFD_PROCESS_RESULT_ENUM eResult;
    BOOLEAN bSuccess = TRUE;

    // Process the next group
    eResult = eProcessUpdateGroup(psRFDCtrl, psFile);

    if (eResult != RFD_PROCESS_RESULT_ERROR)
    {
        // Increment our progress index
        psRFDCtrl->tCurProgressIndex++;

        if (psRFDCtrl->tCurProgressIndex > psRFDCtrl->tStartProgressIndex)
        {
            // Report our progress
            bSuccess = RFD_INTERFACE_bReportProgress(
                psRFDCtrl->hRFD, psRFDCtrl->tCurProgressIndex
                    );
            if (bSuccess == FALSE)
            {
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }
    }

    // Are we done?
    if (eResult == RFD_PROCESS_RESULT_COMPLETE)
    {
        // determining DB version to set
        // if current RFD interface is STU
        if (psRFDCtrl->eRFDType == SAFECAM1_RFD_TYPE_STU)
        {
            if (psRFDCtrl->tFileBaseVersion > psRFDCtrl->tCurrentVersion)
            {
                // if file base version is more than current,
                // DB version stays the same
                tUpdateVersion = psRFDCtrl->tCurrentVersion;
            }
        }

        bSuccess = GsSafeCamMgr.bDBUpdateEnd(
            psRFDCtrl->hDBConnection,
            psRFDCtrl->acBuffer,
            SAFECAM_MAX_SQL_STRING_LENGTH,
            tUpdateVersion
                );
        if (bSuccess == FALSE)
        {
            eResult = RFD_PROCESS_RESULT_ERROR;
        }
        else
        {
            psRFDCtrl->tCurrentVersion = tUpdateVersion;
        }
    }

    return eResult;
}


/*****************************************************************************
*
*   eRFDSTUFileProcessor
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDSTUFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        )
{
    return eRFDFileProcessor(hConnection, eProcessStatus, psRFDFile,
        tFileVersion, tProgressIndex, pvCallbackArg, SAFECAM1_RFD_TYPE_STU
            );
}

/*****************************************************************************
*
*   eRFDSTUNeeded
*
*****************************************************************************/
static RFD_FILE_STATUS_ENUM eRFDSTUNeeded (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        )
{
    RFD_FILE_STATUS_ENUM eStatus = RFD_FILE_STATUS_INVALID;

    if ( pvCallbackArg != NULL )
    {
        SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl =
            (SAFECAM1_RFD_CTRL_STRUCT *)pvCallbackArg;
        RFD_UPDATE_VERSION tVersionMin = 0;
        BOOLEAN bSuccess = FALSE;

        // Get the version info from the file name
        bSuccess = bRFDFileVersionReader(
            pcFileName, &tVersionMin, ptFileVersion );

        // Did we extract that information without error?
        if (bSuccess == TRUE)
        {
            // Determine if this update is needed
            eStatus = RFD_INTERFACE_eCompareFileVersions(
                tVersionMin, psRFDCtrl->tCurrentVersion,
                *ptFileVersion, tVersionBitWidth,
                RFD_INTERFACE_BASELINE_CHECK_SKIP);
        }
    }

    return eStatus;
}

/*****************************************************************************
*
*   eRFDLTUNeeded
*
*****************************************************************************/
static RFD_FILE_STATUS_ENUM eRFDLTUNeeded (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        )
{
    RFD_FILE_STATUS_ENUM eStatus = RFD_FILE_STATUS_INVALID;

    if ( pvCallbackArg != NULL )
    {
        SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl =
            (SAFECAM1_RFD_CTRL_STRUCT *)pvCallbackArg;
        RFD_UPDATE_VERSION tVersionMin = 0;
        BOOLEAN bSuccess = FALSE;

        // Get the version info from the file name
        bSuccess = bRFDFileVersionReader(
            pcFileName, &tVersionMin, ptFileVersion );

        // Did we extract that information without error?
        if (bSuccess == TRUE)
        {
            // Determine if this update is needed
            eStatus = RFD_INTERFACE_eCompareFileVersions(
                tVersionMin, psRFDCtrl->tCurrentVersion,
                *ptFileVersion, tVersionBitWidth,
                RFD_INTERFACE_BASELINE_CHECK_PERFORM );
        }
    }

    return eStatus;
}


/*****************************************************************************
*
*   eRFDLTUFileProcessor
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDLTUFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        )
{
    return eRFDFileProcessor(hConnection, eProcessStatus, psRFDFile,
        tFileVersion, tProgressIndex, pvCallbackArg, SAFECAM1_RFD_TYPE_LTU
            );
}

/*****************************************************************************
*
*   eRFDFileProcessor
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg,
    SAFECAM1_RFD_TYPE_ENUM eUpdateType
        )
{
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl = (SAFECAM1_RFD_CTRL_STRUCT*)pvCallbackArg;
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;

    if (psRFDCtrl == NULL)
    {
        return RFD_PROCESS_RESULT_ERROR;
    }

    if (eProcessStatus == RFD_PROCESS_STATUS_BEGIN)
    {
        BOOLEAN bOk;

        // Initialize the RFD Ctrl object and lock it
        bOk = bInitRFDCtrl(psRFDCtrl, hConnection, tProgressIndex, eUpdateType);

        if (bOk == TRUE)
        {
            // Process the update header now
            bOk = bProcessRFDHeader(psRFDCtrl, psRFDFile);
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": Unable to process RFD file header"
                    );

            return RFD_PROCESS_RESULT_ERROR;
        }
    }
    else if (eProcessStatus == RFD_PROCESS_STATUS_STOP)
    {
        // Free associated memory if we're stopping
        vUninitRFDCtrl(psRFDCtrl);

        return RFD_PROCESS_RESULT_INCOMPLETE;
    }

    // Process the file contents and update the database
    eResult = eProcessNextRFDEntry(
        psRFDCtrl, psRFDFile, tFileVersion
            );
    if (eResult != RFD_PROCESS_RESULT_INCOMPLETE)
    {
        // Close our connection to the DB
        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
            psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Are we done?
        if (eResult == RFD_PROCESS_RESULT_COMPLETE)
        {
            BOOLEAN bFinalized;

            // Finalize the new database
            bFinalized = bFinalizeDB(psRFDCtrl);
            if (bFinalized == FALSE)
            {
                // We failed to get the DB ready for use
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }

        // Did we experience an error?
        if (eResult  >= RFD_PROCESS_RESULT_ERROR)
        {
            // Delete database
            vDeleteDB(psRFDCtrl);
        }
    }

    return eResult;
}


/*****************************************************************************
*
*   psCreateRFDCtrl
*
*****************************************************************************/
static SAFECAM1_RFD_CTRL_STRUCT *psCreateRFDCtrl(
    SAFECAM1_OBJECT_STRUCT *psObj
        )
{
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl;

    // Create the RFD processor object
    psRFDCtrl = (SAFECAM1_RFD_CTRL_STRUCT*)
        SMSO_hCreate(
        SAFECAM1_OBJECT_NAME":RFDCtrl",
        sizeof(*psRFDCtrl),
        SMS_INVALID_OBJECT, TRUE
            );
    if (psRFDCtrl == NULL)
    {
        return (SAFECAM1_RFD_CTRL_STRUCT*)NULL;
    }

    // Populate the structure with the minimum required
    psRFDCtrl->hSafeCamService = psObj->hSafeCamService;

    return psRFDCtrl;
}


/*****************************************************************************
*
*   bInitRFD
*
*****************************************************************************/
static BOOLEAN bInitRFD (
    SAFECAM1_OBJECT_STRUCT *psObj,
    RFD_UPDATE_VERSION tCurrentVersion
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        RFD_OPTIONAL_CALLBACKS_STRUCT sCallbacks;

        OSAL.bMemSet(&sCallbacks, 0, sizeof(sCallbacks));

        // Update determination
        sCallbacks.eUpdateNeeded = eRFDSTUNeeded;

        // Create the RFD processor object
        psObj->psRFDCtrl = psCreateRFDCtrl(psObj);
        if (psObj->psRFDCtrl == NULL)
        {
            break;
        }

        psObj->psRFDCtrl->tCurrentVersion = tCurrentVersion;

        // Connect to RFD STU
        psObj->hRFDSTU = RFD_INTERFACE_hConnect (
            SAFECAM1_STU_RFD_CLIENT_ID,
            tCurrentVersion,
            SAFECAM1_MAX_VERSION_BITLEN,
            eRFDSTUFileProcessor,
            &sCallbacks,
            psObj->psRFDCtrl
                );

        if (psObj->hRFDSTU == RFD_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Update determination
        sCallbacks.eUpdateNeeded = eRFDLTUNeeded;

        // DB Expiration
        sCallbacks.vReportExpiredDB = vNotifyDBExpired;

        // Connect to RFD LTU
        psObj->hRFDLTU = RFD_INTERFACE_hConnect (
            SAFECAM1_LTU_RFD_CLIENT_ID,
            tCurrentVersion,
            SAFECAM1_MAX_VERSION_BITLEN,
            eRFDLTUFileProcessor,
            &sCallbacks,
            psObj->psRFDCtrl
                );

        if (psObj->hRFDLTU == RFD_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Now RFD Ctrl is ready and can be used
        SMSO_vUnlock((SMS_OBJECT)psObj->psRFDCtrl);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   bGetBaudotFieldBitsSize
*
*****************************************************************************/
static BOOLEAN bGetBaudotFieldBitsSize(
    OSAL_BUFFER_HDL hBuffer,
    size_t tBitsOffset,
    size_t *ptFieldBitsSize
        )
{
    SAFECAM1_BAUDOT_COL_NUM_ENUM eCurrColumnNo =
        SAFECAM1_BAUDOT_COL_NUM_LTR_UC;
    BOOLEAN bIsEscapeMode = FALSE, bIsLineFeedInsertRequested = FALSE,
        bFoundBaudotEndCode = FALSE;
    UN8 un8Buffer = 0;
    size_t tBitsRead = 0;
    BOOLEAN bSuccess = TRUE;

    while(bFoundBaudotEndCode == FALSE)
    {
        // Check for Line Feed request.
        if(bIsLineFeedInsertRequested == TRUE)
        {
            // signal to add line feed following carriage return.
            bIsLineFeedInsertRequested = FALSE;
            continue;
        }

        un8Buffer = 0;
        tBitsRead = OSAL.tBufferPeekBits(hBuffer, &un8Buffer, 0,
                BAUDOT_CHAR_BITWIDTH, tBitsOffset
                    );
        if (tBitsRead != BAUDOT_CHAR_BITWIDTH)
        {
            //Looks like buffer is corrupted, skip
            bSuccess = FALSE;
            break;
        }
        else
        {
            (*ptFieldBitsSize) += tBitsRead;
            tBitsOffset += tBitsRead;
        }

        if(bIsEscapeMode == TRUE)
        {
            bIsEscapeMode = FALSE;
            if(un8Buffer >= aun8NumValidRowsForBaudotColumnEscape[eCurrColumnNo])
            {
                //Looks like buffer is corrupted, skip
                bSuccess = FALSE;
                break;
            }
        }
        else
        {
            // regular mode, not escape
            switch(un8Buffer)
            {
                case SAFECAM1_BAUDOT_CODE_END:
                {
                    bFoundBaudotEndCode = TRUE;
                }
                break;

                case SAFECAM1_BAUDOT_CODE_ESC:
                {
                    bIsEscapeMode = TRUE;
                }
                break;

                case SAFECAM1_BAUDOT_CODE_UC_LC:
                {
                    if(eCurrColumnNo == SAFECAM1_BAUDOT_COL_NUM_LTR_UC)
                    {
                        eCurrColumnNo = SAFECAM1_BAUDOT_COL_NUM_LTR_LC;
                    }
                    else if(eCurrColumnNo == SAFECAM1_BAUDOT_COL_NUM_LTR_LC)
                    {
                        eCurrColumnNo = SAFECAM1_BAUDOT_COL_NUM_LTR_UC;
                    }
                    else if(un8Buffer >=
                        aun8NumValidRowsForBaudotColumnNonEscape[eCurrColumnNo])
                    {
                        //Looks like buffer is corrupted, skip
                        bSuccess = FALSE;
                    }
                }
                break;

                case SAFECAM1_BAUDOT_CODE_FIG_LTR:
                {
                    if(eCurrColumnNo == SAFECAM1_BAUDOT_COL_NUM_LTR_UC ||
                        eCurrColumnNo == SAFECAM1_BAUDOT_COL_NUM_LTR_LC )
                    {
                        eCurrColumnNo = SAFECAM1_BAUDOT_COL_NUM_FIG;
                    }
                    else
                    {
                        eCurrColumnNo = SAFECAM1_BAUDOT_COL_NUM_LTR_UC;
                    }
                }
                break;

                case SAFECAM1_BAUDOT_CODE_CR:
                {
                    if(un8Buffer <
                        aun8NumValidRowsForBaudotColumnNonEscape[eCurrColumnNo])
                    {
                        bIsLineFeedInsertRequested = TRUE;
                    }
                    else
                    {
                        //Looks like buffer is corrupted, skip
                        bSuccess = FALSE;
                    }
                }
                break;

                default:
                {
                    // Not a control or special code.
                    if(un8Buffer >=
                        aun8NumValidRowsForBaudotColumnNonEscape[eCurrColumnNo])
                    {
                        //Looks like buffer is corrupted, skip
                        bSuccess = FALSE;
                    }
                }
                break;
            }

            if (bSuccess == FALSE)
            {
                break;
            }
        }
    }

    return bSuccess;
}


/*****************************************************************************
*
*   eGetMetadataType
*
*****************************************************************************/
static SAFECAM1_MSG_TYPE_ENUM eGetMetadataType(
    OSAL_BUFFER_HDL hBuffer
        )
{
    SAFECAM1_MSG_TYPE_ENUM eResult = SAFECAM1_INVALID_MSG;

    do
    {
        BOOLEAN bSuccess = FALSE;
        size_t tNameBitsSize = 0, tBitsRead = 0;
        UN8 un8Buffer = 0;

        bSuccess = bGetBaudotFieldBitsSize(hBuffer,
            SAFECAM1_PVN_BITLEN + SAFECAM1_CAROUSEL_BITLEN, &tNameBitsSize
                );
        if (bSuccess == FALSE)
        {
            // cannot get name field, buffer corrupted
            break;
        }

        tBitsRead = OSAL.tBufferPeekBits(hBuffer, &un8Buffer, 0,
            SAFECAM1_CAR_BITLEN, SAFECAM1_PVN_BITLEN +
            SAFECAM1_CAROUSEL_BITLEN + tNameBitsSize + SAFECAM1_SIZE_BITLEN
                );
        if (tBitsRead != SAFECAM1_CAR_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM1_OBJECT_NAME": failed to peek CAR field"
                    );
            break;
        }

        switch (un8Buffer)
        {
            case SAFECAM1_STU_CAROUSEL_ID:
            {
                eResult = SAFECAM1_MSG_STU_METADATA;
            }
            break;

            case SAFECAM1_LTU_CAROUSEL_ID:
            {
                eResult = SAFECAM1_MSG_LTU_METADATA;
            }
            break;

            default:
            break;
        }

    } while (FALSE);


    return eResult;
}

