/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Non-Nav Maps protocol version 1 implementation
 *  for the Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "string.h"
#include "baudot.h"
#include "location_obj.h"
#include "db_util.h"
#include "dataservice_base.h"
#include "dataservice_mgr_obj.h"

#include "rfd_interface_obj.h"
#include "maps_interface.h"
#include "_maps_pvn1.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   hInit
 *
 *****************************************************************************/
static MAPS_INTERFACE_OBJECT hInit(
    MAPS_SERVICE_OBJECT hService,
    SMS_OBJECT hParent
        )
{
    MAPS_INTERFACE_OBJECT hResult = MAPS_INTERFACE_INVALID_OBJECT;
    MAPS1_OBJECT_STRUCT *psObj = (MAPS1_OBJECT_STRUCT*)NULL;

    do
    {
        BOOLEAN bOk;

        if (hService == MAPS_SERVICE_INVALID_OBJECT ||
            hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Verify we own service handle
        bOk = SMSO_bOwner((SMS_OBJECT)hParent);
        if (bOk == FALSE)
        {
            break;
        }

        // Create an instance of this object
        psObj = (MAPS1_OBJECT_STRUCT*)
            SMSO_hCreate(MAPS1_OBJECT_NAME,
                sizeof(*psObj), (SMS_OBJECT)hService,
                FALSE
                    );
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS1_OBJECT_NAME": failed to create object" );
            break;
        }

        // Initialize/connect to MFM now
        bOk = bMFMInit(psObj, hService);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS1_OBJECT_NAME": failed to init MFM" );
            break;
        }

        hResult = (MAPS_INTERFACE_OBJECT)psObj;
    } while (FALSE);

    if (hResult == MAPS_INTERFACE_INVALID_OBJECT)
    {
        vUninitObject(psObj);
    }

    return hResult;
}

/*****************************************************************************
 *
 *   vUnInit
 *
 *****************************************************************************/
static void vUnInit(
    MAPS_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (TRUE == bOwner)
    {
        MAPS1_OBJECT_STRUCT *psObj =
            (MAPS1_OBJECT_STRUCT *)hInterface;

        vUninitObject(psObj);
    }

    return;
}

/*****************************************************************************
 *
 *   bProcessMessage
 *
 *****************************************************************************/
static BOOLEAN bProcessMessage(
    MAPS_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bSuccess = FALSE;

    if ((phPayload == NULL) || (*phPayload == OSAL_INVALID_BUFFER_HDL))
    {
        return FALSE;
    }

    do
    {
        BOOLEAN bOwner;
        size_t tPayloadSize, tBytesPeeked;
        RFD_STATUS eStatusCode;
        MAPS1_OBJECT_STRUCT *psObj = (MAPS1_OBJECT_STRUCT *)hInterface;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // Copy payload to our shared buffer
        tPayloadSize = OSAL.tBufferGetSize(*phPayload);
        tBytesPeeked = OSAL.tBufferPeek(*phPayload,
            &psObj->psMFMCtrl->aun8PayloadBuffer[0],
            MAPS1_MAX_AU_BYTESIZE, 0);

        // Did we copy the right amount?
        if (tBytesPeeked != tPayloadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS1_OBJECT_NAME": bProcessMessage(): Cannot copy full payload!");
            break;
        }

        eStatusCode = MFM_PutRfdMessage(psObj->psMFMCtrl->hMFM,
            &psObj->psMFMCtrl->aun8PayloadBuffer[0],
            tPayloadSize,
            RFD_DMI_UNUSED_VALUE);

        if (eStatusCode != RFD_STATUS_OK)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS1_OBJECT_NAME": Error adding payload to MFM: %d\n",
                eStatusCode);
            break;
        }

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/******************************************************************************
                             PRIVATE FUNCTIONS
 *****************************************************************************/
/******************************************************************************
 *
 *   vUninitObject
 *
 *****************************************************************************/
static void vUninitObject(
    MAPS1_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        // Clean up MFM now
        vDestroyMFMCtrl(psObj->psMFMCtrl);
        psObj->psMFMCtrl = (MAPS1_MFM_CTRL_STRUCT *)NULL;

        // Destroy this object
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
 *
 *   bMFMInit
 *
 *****************************************************************************/
static BOOLEAN bMFMInit(
    MAPS1_OBJECT_STRUCT *psObj,
    MAPS_SERVICE_OBJECT hMapsService
        )
{
    MAPS1_MFM_CTRL_STRUCT *psMFMCtrl;

    // Create the control object for MFM interaction
    psMFMCtrl = psCreateMFMCtrl(psObj);

    if ((MAPS1_MFM_CTRL_STRUCT *)NULL == psMFMCtrl)
    {
        return FALSE;
    }

    do
    {
        RFD_STATUS eMFMReturnCode;
        MFM_RFD_CLIENT_STRUCT asConnections[MAPS1_NUM_RFD_CLIENTS];
        MFM_APP_CALLBACK_INFO_STRUCT sCallbacks;
        MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT sPolicy;
        BOOLEAN bSuccess;

        // Attach the MFM to the service Maps PVN
        psObj->psMFMCtrl = psMFMCtrl;

        // Store the maps service handle now
        psMFMCtrl->hMapsService = hMapsService;

        // Get the connections to RFD from our interface object
        psMFMCtrl->bRFDConnected = RFD_INTERFACE_bAcquireRFDConnectionsForMFM(
            MAPS1_RFD_CLIENT_ID,
            &asConnections[0], MAPS1_NUM_RFD_CLIENTS);
        if (FALSE == psMFMCtrl->bRFDConnected)
        {
            break;
        }

        // RFD Interface provides this function for us
        sCallbacks.appGetTimeCallbackFcn = RFD_INTERFACE_bMFMReportTime;
        sCallbacks.appGetTimeCallbackArg = NULL;

        // We don't need to be given DMI status updates
        sCallbacks.appBlkMsgDmiStatusCallbackFcn = NULL;
        sCallbacks.appBlkMsgDmiStatusCallbackArg = NULL;

        // Callback informs MFM if the subname is valid
        // for this service
        sCallbacks.appIsSubnameInServiceCallbackFcn = bIsSubnameInService;
        sCallbacks.appIsSubnameInServiceCallbackArg = (void *)psObj;

        // Callback which gives us all new files
        sCallbacks.appRfdFileTransferCallbackFcn = bMFMFileTransfer;
        sCallbacks.appRfdFileTransferCallbackArg = (void *)psObj;

        // Callback which parses an RFD Metadata Name field
        // to populate a MFM_FILE_IDENTITY_STRUCT
        sCallbacks.mfmFileIdentityParseCallbackFcn = bMFMParseIdentity;
        sCallbacks.mfmFileIdentityParseCallbackArg = (void *)psObj;

        // Callbacks which compare updates for prioritization purposes,
        // and indicate a start/end of sorting
        sCallbacks.mfmFilePriorityCallbackFcns.filePrioritySortBeginFcn =
            bMFMSortBegin;
        sCallbacks.mfmFilePriorityCallbackFcns.filePrioritySortEndFcn =
            vMFMSortEnd;
        sCallbacks.mfmFilePriorityCallbackFcns.filePriorityCompareFcn =
            iMFMSortUpdates;
        sCallbacks.mfmFilePriorityCallbackArg = NULL;

        // Maps will prioritize updates
        sPolicy.isFilePrioritySortingEnabled = TRUE;

        // Maps uses the "absolute" model
        sPolicy.updateType = MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES;

        // Create our file reader buffer
        bSuccess = DATASERVICE_MGR_bCreateFileBuffer(
            &psMFMCtrl->hBlockPool,
            &psMFMCtrl->hBuffer,
            MAPS1_MAX_RFD_UPDATE_HEADER_BYTESIZE
                );
        if (FALSE == bSuccess)
        {
            break;
        }

        // Connect to MFM now
        eMFMReturnCode = MFM_CreateExt(
            &psMFMCtrl->hMFM,                   // MFM Connection Handle
            MAPS1_MFM_CLIENT_ID,                // MFM Client ID
            MAPS1_OBJECT_NAME":MFMConnection",  // MFM Client Name
            &asConnections[0],                  // RFD Connection Array
            MAPS1_NUM_RFD_CLIENTS,              // Size of Connection Array
            &sCallbacks,                        // Callbacks
            &sPolicy,                           // File Management Policy
            RFD_INTERFACE_MFM_POLLRATE,         // Poll Interval
            FALSE                               // Multi-threaded Mode
                );
        if (RFD_STATUS_OK != eMFMReturnCode)
        {
            break;
        }

        return TRUE;
    } while (FALSE);

    // Detach control object
    psObj->psMFMCtrl = (MAPS1_MFM_CTRL_STRUCT *)NULL;
    // Clean up the control object
    vDestroyMFMCtrl(psMFMCtrl);

    return FALSE;
}

/*****************************************************************************
 *
 *   psCreateMFMCtrl
 *
 *****************************************************************************/
static MAPS1_MFM_CTRL_STRUCT *psCreateMFMCtrl (
    MAPS1_OBJECT_STRUCT *psObj
        )
{
    MAPS1_MFM_CTRL_STRUCT *psMFMCtrl;

    // Create the RFD processor object
    psMFMCtrl = (MAPS1_MFM_CTRL_STRUCT *)
        SMSO_hCreate(
            MAPS1_OBJECT_NAME":MFMCtrl",
            sizeof(MAPS1_MFM_CTRL_STRUCT),
            SMS_INVALID_OBJECT, FALSE);
    if (NULL == psMFMCtrl)
    {
        return (MAPS1_MFM_CTRL_STRUCT *)NULL;
    }

    // RFD is not yet connected
    psMFMCtrl->bRFDConnected = FALSE;

    return psMFMCtrl;
}

/*****************************************************************************
 *
 *   vDestroyMFMCtrl
 *
 *****************************************************************************/
static void vDestroyMFMCtrl (
    MAPS1_MFM_CTRL_STRUCT *psMFMCtrl
        )
{
    if (psMFMCtrl == NULL)
    {
        return;
    }

    // Clear the service handle and the MFM handle
    psMFMCtrl->hMapsService = MAPS_SERVICE_INVALID_OBJECT;

    // Is MFM connected?
    if ((MULTIFILE_MANAGER_HANDLE)NULL != psMFMCtrl->hMFM)
    {
        // Yes clean it up now
        puts(MAPS1_OBJECT_NAME": Disconnecting MFM");
        MFM_Delete(psMFMCtrl->hMFM);
        psMFMCtrl->hMFM = (MULTIFILE_MANAGER_HANDLE)NULL;
    }

    if (TRUE == psMFMCtrl->bRFDConnected)
    {
        // Release our RFD connections now
        RFD_INTERFACE_vReleaseRFDConnectionsForMFM(MAPS1_RFD_CLIENT_ID);
    }

    // Destroy the file buffer
    DATASERVICE_MGR_vDestroyFileBuffer(
        psMFMCtrl->hBlockPool, psMFMCtrl->hBuffer);

    psMFMCtrl->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    psMFMCtrl->hBuffer = OSAL_INVALID_BUFFER_HDL;

    // Destroy object now
    SMSO_vDestroy((SMS_OBJECT)psMFMCtrl);
    return;
}

/*****************************************************************************
 *
 *   bIsSubnameInService
 *
 *****************************************************************************/
static BOOL bIsSubnameInService (
    TCHAR *pacSubname,
    UINT32 *pun32AppStoredFileIdentifyVersionToPtr,
    void *pvCallbackArg
        )
{
    MAPS1_OBJECT_STRUCT *psObj = (MAPS1_OBJECT_STRUCT *)pvCallbackArg;
    BOOL bIsInService = FALSE;

    if ((pacSubname == NULL) ||
        (pun32AppStoredFileIdentifyVersionToPtr == NULL) ||
        (pvCallbackArg == NULL))
    {
        return FALSE;
    }

    // Try to get current version for provided subname.
    *pun32AppStoredFileIdentifyVersionToPtr =
        GsMapsMgrIntf.un32GetVersionForSubname(
            psObj->psMFMCtrl->hMapsService,
            (const char *)pacSubname);

    if (*pun32AppStoredFileIdentifyVersionToPtr
        != MFM_FILE_IDENTITY_VERSION_NUM_UNUSED)
    {
        // We are in service
        bIsInService = TRUE;
    }

    return bIsInService;
}

/*****************************************************************************
 *
 *   bMFMFileTransfer
 *
 *****************************************************************************/
static BOOL bMFMFileTransfer(
    const TCHAR acFilePath[],
    RFD_CONSUMER_FILE_INFO_STRUCT *psRFDMetadataFileInfo,
    void *pvCallbackArg
        )
{
    FILE *psRFDFile;
    BOOL bOk = FALSE;
    MFM_FILE_IDENTITY_STRUCT sIdentity;
    MAPS1_OBJECT_STRUCT *psObj = (MAPS1_OBJECT_STRUCT *)pvCallbackArg;
    MAPS1_RFD_IMAGE_DATA_STRUCT asImageData[MAPS1_IMAGE_FILES_IN_UPDATE_MAX];

    if ((acFilePath == NULL) ||
        (psRFDMetadataFileInfo == NULL) ||
        (pvCallbackArg == NULL))
    {
        return FALSE;
    }

    psRFDFile = fopen(acFilePath, "rb");

    if (psRFDFile == NULL)
    {
        return FALSE;
    }

    do
    {
        bOk = bMFMParseIdentity(
            psRFDMetadataFileInfo->name, &sIdentity, (void *)psObj);
        if (bOk != TRUE)
        {
            break;
        }

        OSAL.bMemSet(&asImageData[0], 0, sizeof(asImageData));
        bOk = bParseRFDFileHeader(
            psRFDFile, &asImageData[0], psObj->psMFMCtrl);
        if (bOk != TRUE)
        {
            break;
        }

        bOk = bExtractImageFiles(psObj->psMFMCtrl, psRFDFile, &asImageData[0],
            &sIdentity.subname[0], sIdentity.versionTo);
    } while (FALSE);

    fclose(psRFDFile);

    return bOk;
}

/*****************************************************************************
 *
 *   bParseRFDMetadataName
 *
 *****************************************************************************/
static BOOLEAN bParseRFDMetadataName(
    const TCHAR pacRFDMetadataName[],
    char *pacName,
    size_t tNameSize,
    UINT32 *pun32Version
        )
{
    BOOLEAN bOk = FALSE;

    if ((pacName != NULL) &&
        (tNameSize > MAPS1_RFD_METADATA_FILE_NAME_BYTELEN) &&
        (pun32Version != NULL))
    {
        bOk = OSAL.bMemCpy(pacName, pacRFDMetadataName, MAPS1_RFD_METADATA_FILE_NAME_BYTELEN);
        pacName[MAPS1_RFD_METADATA_FILE_NAME_BYTELEN] = '\0';

        // Read out version
        *pun32Version =
            (UINT32)strtol(pacRFDMetadataName + MAPS1_RFD_METADATA_FILE_NAME_BYTELEN,
                           (char**)NULL, 10 /* DEC */);
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bMFMParseIdentity
 *
 *****************************************************************************/
static BOOL bMFMParseIdentity (
    const TCHAR pacRFDMetadataName[],
    MFM_FILE_IDENTITY_STRUCT *psFileIdentityInfo,
    void *pvCallbackArg
        )
{
    // Parse the name to populate psFileIdentityInfo
    BOOLEAN bOk;
    MAPS1_OBJECT_STRUCT *psObj = (MAPS1_OBJECT_STRUCT *)pvCallbackArg;

    if ((pacRFDMetadataName == NULL) ||
        (psFileIdentityInfo == NULL) ||
        (pvCallbackArg == NULL))
    {
        // Bad param
        return FALSE;
    }

    // Extract name and version
    bOk = bParseRFDMetadataName(pacRFDMetadataName,
        psFileIdentityInfo->subname,
        sizeof(psFileIdentityInfo->subname),
        &psFileIdentityInfo->versionTo);

    if (bOk == TRUE)
    {
        psFileIdentityInfo->versionFrom =
            GsMapsMgrIntf.un32GetVersionForSubname(
                psObj->psMFMCtrl->hMapsService,
                (const char *)&psFileIdentityInfo->subname[0]);

        if (psFileIdentityInfo->versionFrom
            == MFM_FILE_IDENTITY_VERSION_NUM_UNUSED)
        {
            bOk = FALSE;
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bMFMSortBegin
 *
 *****************************************************************************/
static BOOL bMFMSortBegin (
    void *pvCallbackArg
        )
{
    return FALSE;
}

/*****************************************************************************
 *
 *   vMFMSortEnd
 *
 *****************************************************************************/
static void vMFMSortEnd (
    void *pvCallbackArg
        )
{
    return;
}

/*****************************************************************************
 *
 *   iMFMSortUpdates
 *
 *****************************************************************************/
static int iMFMSortUpdates (
    const TCHAR *pacSubNameA,
    const TCHAR *pacSubNameB,
    void *pvCallbackArg
        )
{
    MAPS1_OBJECT_STRUCT *psObj = (MAPS1_OBJECT_STRUCT *)pvCallbackArg;

    if (psObj == NULL)
    {
        return 0;
    }

    return GsMapsMgrIntf.n8ComparePriority(psObj->psMFMCtrl->hMapsService,
        pacSubNameA, pacSubNameB);
}


/*****************************************************************************
 *
 *   bParseRFDFileHeader
 *
 *   Parse the Maps Image Update File header.
 *   Header have the following format:
 *
 *   Reserved - 4bits (For future use)
 *   UCOUNT - 4bits (MAPS1_MAX_AU_BYTESIZE)
 *   HOR - 16bits (Horizontal Resolution of Image)
 *   OFFSET - 24bits (File Offset to Image Update)
 *
 *****************************************************************************/
static BOOLEAN bParseRFDFileHeader (
    FILE *psRFDFile,
    MAPS1_RFD_IMAGE_DATA_STRUCT *pasRFDImageDataStruct,
    void *pvArg
        )
{
    MAPS1_MFM_CTRL_STRUCT *psMFMCtrl = (MAPS1_MFM_CTRL_STRUCT *)pvArg;
    BOOLEAN bSuccess = FALSE;
    UN16 un16Count = 0;
    UN8 un8Index;
    size_t tReadBits;

    do
    {
        if ((psRFDFile == NULL) ||
            (pasRFDImageDataStruct == NULL) ||
            (pvArg == NULL))
        {
            break;
        }

        bSuccess = DATASERVICE_MGR_bFillBufferBlock(
                psRFDFile, psMFMCtrl->hBuffer);
        if (bSuccess != TRUE)
        {
            break;
        }

        tReadBits = OSAL.tBufferSeekHeadBits(psMFMCtrl->hBuffer,
            MAPS1_IMAGE_RESERVED_BITLEN);
        if (tReadBits != MAPS1_IMAGE_RESERVED_BITLEN)
        {
            bSuccess = FALSE;
            break;
        }

        bSuccess = OSAL.bBufferReadBitsToUN16(
                psMFMCtrl->hBuffer, &un16Count,
                MAPS1_IMAGE_UCOUNT_BITLEN);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Increment un16Count since according to
        // protocol specification: "Items 3 - 4 repeat UCOUNT+1 times."
        ++un16Count;

        if (un16Count > MAPS1_IMAGE_FILES_IN_UPDATE_MAX)
        {
            // Invalid images count.
            bSuccess = FALSE;
            break;
        }

        for (un8Index = 0; un8Index < un16Count; un8Index++)
        {
            bSuccess  = OSAL.bBufferReadBitsToUN16(
                psMFMCtrl->hBuffer,
                &pasRFDImageDataStruct[un8Index].un16HorResolution,
                MAPS1_IMAGE_HOR_BITLEN);
            if (bSuccess != TRUE)
            {
                break;
            }

            bSuccess  = OSAL.bBufferReadBitsToUN32(
                    psMFMCtrl->hBuffer,
                    &pasRFDImageDataStruct[un8Index].un32Offset,
                    MAPS1_IMAGE_OFFSET_BITLEN);
            if (bSuccess != TRUE)
            {
                break;
            }
        }

        // Seek past the rest of the bits
        OSAL.tBufferSeekHeadBits(psMFMCtrl->hBuffer, N32_MAX);
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
 *
 *   bExtractImageFiles
 *
 *   Extract *.png files from Maps Image Update File based on their offset.
 *
 *****************************************************************************/
static BOOLEAN bExtractImageFiles (
    MAPS1_MFM_CTRL_STRUCT *psMFMCtrl,
    FILE *psRFDFile,
    const MAPS1_RFD_IMAGE_DATA_STRUCT *pasRFDImageDataStruct,
    const char *pacSubname,
    UN32 un32UpdateVersion
        )
{
    N8 n8Index;
    BOOLEAN bSuccess = FALSE;
    size_t tRFDFileSize, tNewFileSize;

    if ((psRFDFile == NULL) ||
        (pasRFDImageDataStruct == NULL) ||
        (pacSubname == NULL) ||
        (psMFMCtrl == NULL))
    {
        return FALSE;
    }

    bSuccess = OSAL.bFileSystemGetFileSize(psRFDFile, &tRFDFileSize);
    if (bSuccess != TRUE)
    {
        return FALSE;
    }

    for (n8Index = 0; (n8Index < MAPS1_IMAGE_FILES_IN_UPDATE_MAX) &&
            (pasRFDImageDataStruct[n8Index].un16HorResolution != 0); n8Index++)
    {
        // Calculate new file length
        if ((n8Index < MAPS1_IMAGE_FILES_IN_UPDATE_MAX - 1) &&
            (pasRFDImageDataStruct[n8Index + 1].un32Offset != 0))
        {
            tNewFileSize =
                pasRFDImageDataStruct[n8Index + 1].un32Offset - pasRFDImageDataStruct[n8Index].un32Offset;
        }
        else
        {
            tNewFileSize =
                tRFDFileSize - pasRFDImageDataStruct[n8Index].un32Offset;
        }

        // Seek to the next Image File start position
        fseek(psRFDFile, pasRFDImageDataStruct[n8Index].un32Offset, SEEK_SET);

        // Update Persistent DB
        bSuccess = GsMapsMgrIntf.bAddUpdate(
            psMFMCtrl->hMapsService,
            pacSubname,
            un32UpdateVersion,
            pasRFDImageDataStruct[n8Index].un16HorResolution,
            psRFDFile,
            tNewFileSize);

        if (bSuccess == FALSE)
        {
            break;
        }
    }

    return bSuccess;
}
