/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the PVN 1 specific fuel price data service
 *  for the Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include <string.h>
#include <stdio.h>

#include "sms_api.h"
#include "sms_obj.h"
#include "baudot.h"
#include "string_obj.h"
#include "location_obj.h"
#include "rfd_interface_obj.h"
#include "dataservice_mgr_obj.h"
#include "fuel_db_constants.h"
#include "sql_interface_obj.h"
#include "db_util.h"
#include "ds_util.h"
#include "fuel_interface.h"
#include "fuel_mgr_obj.h"
#include "_fuel_pvn1.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hInit
*
*   Creates and initializes the PVN0 fuel interface object.
*
*   Inputs:
*       hFuelService - A valid handle to a FUEL_SERVICE_OBJECT representing
*           the central fuel service manager.
*
*   Output:
*       An object handle representing this instantiation of the
*       fuel1 interface (representing high-band support)
*
*****************************************************************************/
static FUEL_OTA_INTERFACE_OBJECT hInit(
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface
        )
{
    BOOLEAN bOwner;

    // Verify we own service handle
    bOwner = SMSO_bOwner((SMS_OBJECT)hParent);
    if (bOwner == TRUE)
    {
        FUEL1_OBJECT_STRUCT *psObj;

        // Create an instance of this object
        psObj = (FUEL1_OBJECT_STRUCT *)
            SMSO_hCreate(
                FUEL1_OBJECT_NAME,
                sizeof(FUEL1_OBJECT_STRUCT),
                hParent, FALSE );

        if (psObj == NULL)
        {
            return FUEL_OTA_INTERFACE_INVALID_OBJECT;
        }

        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Save the service handles
            psObj->hFuelService = hFuelService;
            psObj->hDBInterface = hDBInterface;
            psObj->psDBInterface = psDBInterface;
            psObj->psOTAInterface = psOTAInterface;

            // Request access to ISO 3309 CRC32
            eReturnCode = OSAL.eGetCRC(
                &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32 );

            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error! destroy object
                break;
            }

            // Ensure this is cleared
            psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;

            // Price updates aren't enabled yet
            psObj->sPriceMem.bEnabled = FALSE;

            return (FUEL_OTA_INTERFACE_OBJECT)psObj;

        } while (FALSE);

        vUnInit((FUEL_OTA_INTERFACE_OBJECT)psObj);
    }

    return FUEL_OTA_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vUnInit
*
*   Uninitializes and destroys the PVN 1 fuel price interface object.
*   This API must only be called when already in the context of the
*   fuel price manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*
*****************************************************************************/
static void vUnInit (
    FUEL_OTA_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        FUEL1_OBJECT_STRUCT *psObj =
            (FUEL1_OBJECT_STRUCT *)hInterface;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy the CRC object if it exists
        if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
        {
            // Release our hold of the CRC computation handle
            eReturnCode = OSAL.eReleaseCRC( psObj->hCRC );

            // Only finish clean up if that succeeded.  This allows
            // us to more easily track this kind of issue
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bProductStateChange
*
*****************************************************************************/
static BOOLEAN bProductStateChange (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        )
{
    BOOLEAN bOwner, bSuccess = TRUE;
    FUEL1_OBJECT_STRUCT *psObj = 
        (FUEL1_OBJECT_STRUCT *)hInterface;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if ((FALSE == bOwner) ||
        (DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *)NULL == psState)
    {
        return FALSE;
    }

    switch (psState->eType)
    {
        case DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES:
        {
            // Let the prices FSM deal with it
            bSuccess = bPriceProductStateChange(psObj, psState);
        }
        break;

        case DATA_PRODUCT_TYPE_DB_UPDATES:
        {
            // Let the RFD FSM deal with it
            bSuccess = bRFDProductStateChange(psObj, psState);
        }
        break;

        default:
        {
            // Don't know how to handle this for any other products
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   eNextProductState
*
*****************************************************************************/
static DATA_PRODUCT_STATE_ENUM eNextProductState (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eType,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eDSIState,
    SXM_DMI *pDMIs,
    UN8 un8DMICount
        )
{
    DATA_PRODUCT_STATE_ENUM eNextState = 
        DATA_PRODUCT_STATE_ERROR;

    switch (eType)
    {
        // Fuel Price product
        case DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES:
        {
            switch (eDSIState)
            {
                case DATASERVICE_STATE_INITIAL:
                {
                    eNextState = DATA_PRODUCT_STATE_INITIAL;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // Both of these mean no price updates!
                    eNextState = DATA_PRODUCT_STATE_UNSUBSCRIBED;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    eNextState = DATA_PRODUCT_STATE_READY;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    eNextState = DATA_PRODUCT_STATE_DISABLED;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                case DATASERVICE_STATE_ERROR:
                case DATASERVICE_STATE_INVALID:
                {
                    eNextState = DATA_PRODUCT_STATE_ERROR;
                }
                break;
            }
        }
        break;

        // RFD Product
        case DATA_PRODUCT_TYPE_DB_UPDATES:
        {
            switch (eDSIState)
            {
                case DATASERVICE_STATE_INITIAL:
                {
                    eNextState = DATA_PRODUCT_STATE_INITIAL;
                }
                break;

                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    eNextState = DATA_PRODUCT_STATE_UNSUBSCRIBED;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                case DATASERVICE_STATE_READY:
                {
                    // Both of these mean RFD updates are active
                    eNextState = DATA_PRODUCT_STATE_READY;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    eNextState = DATA_PRODUCT_STATE_DISABLED;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                case DATASERVICE_STATE_ERROR:
                case DATASERVICE_STATE_INVALID:
                {
                    eNextState = DATA_PRODUCT_STATE_ERROR;
                }
                break;
            }
        }
        break;

        default:
        {
            // Nothing to do
        }
        break;
    }

    return eNextState;
}

/*****************************************************************************
*
*   bGetDSIForProduct
*
*   Provides the caller with the mapping of products/DSIs used by this 
*   interface
*
*****************************************************************************/
static BOOLEAN bGetDSIForProductGeneric (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo,
    DSI tDSI
        )
{
    BOOLEAN bSuccess = FALSE;

    if ((eType == DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES) ||
        (eType == DATA_PRODUCT_TYPE_DB_UPDATES))
    {
        psDSIInfo->bEnableAllDMIs = TRUE;
        psDSIInfo->tDSI = tDSI;
        psDSIInfo->tSuggestedOTABufferByteSize =
            FUEL1_OTA_BUFFER_SIZE;
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bGetDSIForProduct
*
*   Provides the caller with the mapping of products/DSIs used by this
*   interface
*
*****************************************************************************/
static BOOLEAN bGetDSIForProduct (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        )
{
    return bGetDSIForProductGeneric(eType, psDSIInfo, FUEL1_DSI);
}

/*****************************************************************************
*
*   bGetDSIForCanadianProduct
*
*   Provides the caller with the mapping of products/DSIs used by this
*   interface
*
*****************************************************************************/
static BOOLEAN bGetDSIForCanadianProduct (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        )
{
    return bGetDSIForProductGeneric(eType, psDSIInfo, FUEL1_CAN_DSI);
}

/*****************************************************************************
*
*   bVerifyMessageNeeded
*
*****************************************************************************/
static BOOLEAN bVerifyMessageNeeded (
    FUEL1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    UN8 *pun8CarouselID
        )
{
    BOOLEAN bOk, bPacketNeeded = FALSE;

    do 
    {
        PVN tPVN = (PVN)0;
        DSI tDSI = (DSI)0;
        size_t tBitsRead, tBytesRead;

        // The DSI is written to the head of the buffer in the DSM for 
        // multi-product services.  So read & validate it.
        tBytesRead = OSAL.tBufferReadHead (
                hPayload, &tDSI, sizeof(tDSI));

        if (tBytesRead != sizeof(tDSI))
        {
            // Unable to read DSI from payload
            return FALSE;
        }

        if (tDSI != psObj->psOTAInterface->tDSI)
        {
            // I don't know who this is for -
            // why did it get routed to fuel?
            return FALSE;
        }

        // Peek at the PVN
        tBitsRead = OSAL.tBufferPeekBits(
            hPayload, &tPVN, 0,
            FUEL1_PVN_BITLEN, 0);

        if (tBitsRead != FUEL1_PVN_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled. Don't do anything rash here.
            puts(FUEL1_OBJECT_NAME" Unable to read PVN\n");

            break;
        }

        // Verify the PVN
        if (tPVN != FUEL1_PVN)
        {
            printf(FUEL1_OBJECT_NAME" Incorrect PVN - got %u, expected %u\n",
                tPVN, FUEL1_PVN);

            break;
        }

        // Peek at the Carousel Id
        tBitsRead = OSAL.tBufferPeekBits(hPayload, pun8CarouselID, 0,
            FUEL1_CAROUSEL_BITLEN, FUEL1_PVN_BITLEN);

        if (tBitsRead != FUEL1_CAROUSEL_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(FUEL1_OBJECT_NAME" Unable to read Carousel Id\n");

            break;
        }

        // Checking some specific message fields
        switch (*pun8CarouselID)
        {
            case FUEL1_PRICE_CAROUSEL_ID:
            {
                FUEL_REGION tRegion = 0;
                UN8 un8TextVer = 0;

                // Read the text version field
                tBitsRead = OSAL.tBufferPeekBits(
                    hPayload, &un8TextVer, 0,
                    FUEL1_TEXT_VER_BITLEN,
                    FUEL1_PVN_BITLEN + FUEL1_CAROUSEL_BITLEN);
                if (tBitsRead != FUEL1_TEXT_VER_BITLEN)
                {
                    break;
                }

                // Peek the region id field
                bOk = OSAL.bBufferPeekBitsToUN16(
                    hPayload, (UN16 *)&tRegion,
                    FUEL1_REGION_BITLEN, 
                    FUEL1_PVN_BITLEN + FUEL1_CAROUSEL_BITLEN + FUEL1_TEXT_VER_BITLEN);

                if (bOk == FALSE)
                {
                    break;
                }

                // Does the manager want to process prices for
                // this region?
                bPacketNeeded = GsFuelMgrIntf.bIsRegionNeeded(
                    psObj->hFuelService,
                    tRegion, un8TextVer);
            }
            break;

            default:
            {
                // All other messages are needed
                bPacketNeeded = TRUE;
            }
        }

    } while (FALSE);

    return bPacketNeeded;
}

/*****************************************************************************
*
*   bProcessMessage
*
*   The entry point for the interface's OTA message processing.  All messages
*   received by the Fuel manager come through this API.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*       hPayload - A valid OSAL_BUFFER_HDL containing the OTA fuel data
*
*   Output:
*       TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bProcessMessage (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify inputs
    if ((hInterface == FUEL_OTA_INTERFACE_INVALID_OBJECT) ||
        (phPayload == NULL)
       )
    {
        return FALSE;
    }

    // Verify buffer
    if (*phPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return FALSE;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        FUEL1_OBJECT_STRUCT *psObj =
            (FUEL1_OBJECT_STRUCT *)hInterface;
        BOOLEAN bMessageValid;
        UN8 un8CarouselID = 0;
        size_t tBitsRead;

        // Check some message fields to detect if this message 
        // is not to be processed further
        bMessageValid = bVerifyMessageNeeded(psObj, 
            *phPayload, &un8CarouselID);
        if (bMessageValid == FALSE)
        {
            puts(FUEL1_OBJECT_NAME" Packet is rejected by manager\n");

            return TRUE;
        }

        // Validate the message (don't consume the CRC bytes though)
        bMessageValid = DS_UTIL_bIsCRCValid(psObj->hCRC, *phPayload,
                            (OSAL_CRC_RESULT*)&psObj->un32CurrentMsgCRC);
        if (bMessageValid == FALSE)
        {
            puts(FUEL1_OBJECT_NAME" Packet Invalid\n");

            return TRUE;
        }

        // Now send the message on it's way
        switch (un8CarouselID)
        {
            case FUEL1_DB_UPDATE_CAROUSEL_ID:
            case FUEL1_METADATA_CAROUSEL_ID:
            {
                printf(FUEL1_OBJECT_NAME": RFD - ");
                if (un8CarouselID == FUEL1_DB_UPDATE_CAROUSEL_ID)
                {
                    puts("Database Update");
                }
                else
                {
                    puts("Meta-data Update");
                }

                // Pass the RFD AU to the RFD Interface
                bSuccess = RFD_INTERFACE_bProcessPayload(
                    psObj->hRFD, *phPayload);

                // Regardless of the error code, we don't
                // own this payload anymore
                *phPayload = OSAL_INVALID_BUFFER_HDL;

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        FUEL1_OBJECT_NAME
                        ": RFD_INTERFACE_bProcessPayload() failed");
                }
            }
            break;

            case FUEL1_TEXT_CAROUSEL_ID:
            case FUEL1_PRICE_CAROUSEL_ID:
            {
                // We can now get rid of the CRC
                tBitsRead = OSAL.tBufferSeekTail(*phPayload, FUEL1_CRC_BYTELEN);
                tBitsRead *= 8; // make this a bit count

                // We can also now seek past the PVN & Car ID
                tBitsRead += OSAL.tBufferSeekHeadBits(
                    *phPayload, FUEL1_PVN_BITLEN + FUEL1_CAROUSEL_BITLEN);

                if (tBitsRead == FUEL1_TRIM_BITLEN)
                {
                    if (un8CarouselID == FUEL1_PRICE_CAROUSEL_ID)
                    {
                        // Process the price update message
                        bSuccess = bProcessPriceMessage( psObj, *phPayload );
                    }
                    else
                    {
                        // Process the text message
                        bSuccess = bProcessTextMessage(psObj, *phPayload);
                    }
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        FUEL1_OBJECT_NAME": Unable to seek past Message Header / Footer");
                    bSuccess = FALSE;
                }
            }
            break;

            default:
            {
                // Don't make this a big deal
                bSuccess = TRUE;

                printf(FUEL1_OBJECT_NAME": Unknown Car ID: %u\n",
                    un8CarouselID);
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bAddRegionsForLocation
*
*   This function provides the fuel manager with the ability to map N fuel
*   regions to a central lat/lon with a given radius.
*
*****************************************************************************/
static BOOLEAN bAddRegionsForLocation (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    LOCATION_OBJECT hCenterLocation,
    void *pvArg
        )
{
    FUEL_REGION tRegion,
                tCenterRegion,
                tStartRegion,
                tSWRegion,
                tNERegion,
                tDiff;
    UN8 un8Lat;
    N8  n8Lon;
    UN8 un8CurRow, un8CurCol;
    UN8 un8NumColsBefore,
        un8NumColsAfter,
        un8NumRowsBefore,
        un8NumRowsAfter;
    UN8 un8NumColsToProcess, un8NumRowsToProcess;
    UN32 un32Offset;
    BOOLEAN bIsPoint = FALSE;
    OSAL_FIXED_OBJECT hCenterLat,
                      hNELat,
                      hNELon,
                      hLat,
                      hLatDiff,
                      hCenterLon,
                      hLon,
                      hLonDiff;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE*2];
    OSAL_FIXED_OBJECT_DATA atNEFixedData[OSAL_FIXED_OBJECT_SIZE*2];
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    BOOLEAN bSuccess;
    FUEL1_OBJECT_STRUCT *psObj =
        (FUEL1_OBJECT_STRUCT *)hInterface;

    // Verify inputs
    if ((hCenterLocation == LOCATION_INVALID_OBJECT) ||
        (hInterface == FUEL_OTA_INTERFACE_INVALID_OBJECT))
    {
        return FALSE;
    }

    // Init memory
    OSAL.bMemSet(&atNEFixedData[0], 0, sizeof(atNEFixedData));
    OSAL.bMemSet(&atFixedData[0], 0, sizeof(atNEFixedData));

    // Get the center location's attributes
    hCenterLat = LOCATION.hLat(hCenterLocation);
    hCenterLon = LOCATION.hLon(hCenterLocation);
    eReturn = LOCATION.eIsPoint(hCenterLocation, &bIsPoint);

    if ( ((eReturn == SMSAPI_RETURN_CODE_SUCCESS) &&
          (bIsPoint == TRUE)) || // The location is an empty area
         (hCenterLat == OSAL_FIXED_INVALID_OBJECT) ||
         (hCenterLon == OSAL_FIXED_INVALID_OBJECT)) // .. or does not have a base point
    {
        // Just move on
        return TRUE;
    }

    /*****************************************************
     * Calculate the region for the center of the location
     *****************************************************/

    // Just grab the lat / lon values as whole numbers
    un8Lat = (UN8)OSAL_FIXED.n32Floor(hCenterLat);
    n8Lon = (N8)OSAL_FIXED.n32Floor(hCenterLon);

    // Using the latitude, calculate the offset from
    // the service's initial latitude to start computing
    // the region
    un32Offset = (un8Lat - FUEL1_INITIAL_LAT) / FUEL1_REG_SIZE;
    tCenterRegion = FUEL1_REG_FROM_LAT_OFFSET(un32Offset);

    // Using the longitude, calculate the offset from
    // the service's initial longitude to complete computing
    // the region
    un32Offset = (n8Lon - FUEL1_INITIAL_LON) / FUEL1_REG_SIZE;
    tCenterRegion |= FUEL1_REG_FROM_LON_OFFSET(un32Offset);

    /*****************************************
     * Calculate the region for the NE corner
     *****************************************/
    // Get NE corner coordinates via LOCATION_bTopRight()
    // since radius distance is not applicable for rectangle
    hLat = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);
    hLon = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);

    LOCATION_bTopRight(hCenterLocation, hLat, hLon);

    // Keep a copy of the Lat
    OSAL_FIXED.bCopyToMemory(hLat, &atNEFixedData[0]);
    hNELat = (OSAL_FIXED_OBJECT)&atNEFixedData[0];

    // Keep a copy of the Lon
    OSAL_FIXED.bCopyToMemory(hLon, &atNEFixedData[OSAL_FIXED_OBJECT_SIZE]);
    hNELon = (OSAL_FIXED_OBJECT)&atNEFixedData[OSAL_FIXED_OBJECT_SIZE];

    // Just grab the lat / lon values as whole numbers
    un8Lat = (UN8)OSAL_FIXED.n32Floor(hNELat);
    n8Lon = (N8)OSAL_FIXED.n32Floor(hNELon);

    // Using the latitude, calculate the offset from
    // the service's initial latitude to start computing
    // the region
    un32Offset = (un8Lat - FUEL1_INITIAL_LAT) / FUEL1_REG_SIZE;
    tNERegion = FUEL1_REG_FROM_LAT_OFFSET(un32Offset);

    // Using the longitude, calculate the offset from
    // the service's initial longitude to complete computing
    // the region
    un32Offset = (n8Lon - FUEL1_INITIAL_LON) / FUEL1_REG_SIZE;
    tNERegion |= FUEL1_REG_FROM_LON_OFFSET(un32Offset);

    /*****************************************
     * Calculate the region for the SW corner
     *****************************************/

    // Calculate the latitude difference from the center to the NE
    hLatDiff = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);
    eReturnCode = OSAL_FIXED.eSubtract(hNELat, hCenterLat, hLatDiff);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // So, the latitude is centerlat - latdiff
    hLat = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE]);
    eReturnCode = OSAL_FIXED.eSubtract(hCenterLat, hLatDiff, hLat);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Extract the latitude as a whole number
    un8Lat = (UN8)OSAL_FIXED.n32Floor(hLat);

    // Calculate the longitude difference from the center to the NE
    hLonDiff = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);
    eReturnCode = OSAL_FIXED.eSubtract(hNELon, hCenterLon, hLonDiff);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // So, the longitude is centerlon - londiff
    hLon = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE]);
    eReturnCode = OSAL_FIXED.eSubtract(hCenterLon, hLonDiff, hLon);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Extract the longitude as a whole number
    n8Lon = (N8)OSAL_FIXED.n32Floor(hLon);

    // Using the latitude, calculate the offset from
    // the service's initial latitude to start computing
    // the region
    un32Offset = (un8Lat - FUEL1_INITIAL_LAT) / FUEL1_REG_SIZE;
    tSWRegion = FUEL1_REG_FROM_LAT_OFFSET(un32Offset);

    // Using the longitude, calculate the offset from
    // the service's initial longitude to complete computing
    // the region
    un32Offset = (n8Lon - FUEL1_INITIAL_LON) / FUEL1_REG_SIZE;
    tSWRegion |= FUEL1_REG_FROM_LON_OFFSET(un32Offset);

    /***************************************
     * All Regions have now been calculated
     ***************************************/

    // We now have the center region and the SW region and the NE region

    // Determine the number of rows & columns
    // before (south and west of) the center region that
    // must be processed
    tDiff = tCenterRegion - tSWRegion;
    un8NumColsBefore = (UN8)(tDiff / FUEL1_REG_GRID_COL_HEIGHT);
    un8NumRowsBefore = (UN8)(tDiff % FUEL1_REG_GRID_COL_HEIGHT);

    // Determine the number of rows & columns
    // after (north and east of) the center region that
    // must be processed
    tDiff = tNERegion - tCenterRegion;
    un8NumColsAfter = (UN8)(tDiff / FUEL1_REG_GRID_COL_HEIGHT);
    un8NumRowsAfter = (UN8)(tDiff % FUEL1_REG_GRID_COL_HEIGHT);

    // Start at the region in the SW
    tStartRegion = tCenterRegion - (FUEL1_REG_GRID_COL_HEIGHT * un8NumColsBefore);
    if (tStartRegion > 0)
    {
        tStartRegion -= un8NumRowsBefore;
    }

    if (tStartRegion < 0)
    {
        tStartRegion = 0;
    }

    // Calculate the total number of columns to process
    un8NumColsToProcess = un8NumColsBefore + un8NumColsAfter + 1; // + 1 for the center

    // Calculate the total number of rows to process
    un8NumRowsToProcess = un8NumRowsBefore + un8NumRowsAfter + 1; // + 1 for the center

    // Iterate through all the columns & rows
    for (un8CurCol = 0; un8CurCol < un8NumColsToProcess; un8CurCol++)
    {
        // Calculate the "base" region for this column
        tRegion = tStartRegion + (FUEL1_REG_GRID_COL_HEIGHT * un8CurCol);

        for (un8CurRow = 0; un8CurRow < un8NumRowsToProcess; un8CurRow++)
        {
            // Tell the manager to add this region
            bSuccess = GsFuelMgrIntf.bAddRegionOfInterest(
                psObj->hFuelService, (FUEL_REGION)(tRegion + un8CurRow), pvArg);

            // Ensure it succeeded
            if (bSuccess == FALSE)
            {
                return FALSE;
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   eProcessAmenities
*
*   Process protocol-specific amenities for the caller
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProcessAmenities (
    size_t tNumAmenities,
    AMENITY_STRUCT *pasFuelAmenities,
    UN32 un32RawAmenitiesData
        )
{
    // Validate inputs
    if ((tNumAmenities < FUEL_STATION_NUM_AMENITIES) ||
        (pasFuelAmenities == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else
    {
        UN8 un8AmenityOffset = 0, un8RawAmen;
        FUEL_STATION_AMENITY_ENUM eCurAmenity = FUEL1_STATION_BASE_AMENITY;
        SMSAPI_RETURN_CODE_ENUM eReturnCode =
            SMSAPI_RETURN_CODE_SUCCESS;

        do
        {
            switch (eCurAmenity)
            {
                case FUEL_STATION_AMENITY_OPEN_24HR: // index 0
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_24HR_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_OPEN_24HR;

                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_EMERGENCY_ROAD_SERVICE: // index 1
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_EMERGENCY_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_EMERGENCY_ROAD_SERVICE;

                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_FULL_SERVICE: // index 2
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_SERVICE_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_FULL_SERVICE;
                        if (un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_OIL_CHANGE: // index 2
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_OILCHANGE_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_OIL_CHANGE;
                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_INTERSTATE_ACCESS_UNDER_HALF_MILE: // index 3
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_UNDERHALF_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_INTERSTATE_ACCESS_UNDER_HALF_MILE;
                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_INTERSTATE_ACCESS_OVER_HALF_MILE: // index 3
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_OVERHALF_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_INTERSTATE_ACCESS_OVER_HALF_MILE;
                        if (un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_CASH_DISCOUNT: // index 4
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_DISCOUNT_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_CASH_DISCOUNT;
                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_COVENIENCE_STORE: // index 5
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_STORE_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_COVENIENCE_STORE;
                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_SUPERMARKET: // index 5
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_SUPERMKT_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_SUPERMARKET;
                        if (un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_SNACKS_FAST_FOOD: // index 6
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_SNACKS_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_SNACKS_FAST_FOOD;
                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_RESTAURANT: // index 6
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_RESTAURANT_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_RESTAURANT;
                        if (un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_TRUCK_STOP: // index 7
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_TRUCKSTOP_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_TRUCK_STOP;
                        if (un8RawAmen == FUEL1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL: // index 7
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> FUEL1_HOTEL_AMENITY_OFFSET)
                            & FUEL1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == FUEL1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL;
                        if (un8RawAmen == FUEL1_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                default:
                {
                    // We can't identify these amenities
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                    break;
                }
            }
        } while (++eCurAmenity <= FUEL1_STATION_MAX_AMENITY);

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (un8AmenityOffset == 0))
        {
            // We didn't find any amenities, let the caller know
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        return eReturnCode;
    }
}

/*****************************************************************************
*
*   psCreateRFDCtrl
*
*****************************************************************************/
static FUEL1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface
        )
{
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl;

    // Create the RFD processor object
    psRFDCtrl = (FUEL1_RFD_CTRL_STRUCT *)
        SMSO_hCreate(
            FUEL1_OBJECT_NAME":RFDCtrl",
            sizeof(FUEL1_RFD_CTRL_STRUCT),
            SMS_INVALID_OBJECT, FALSE);
    if (psRFDCtrl == NULL)
    {
        return (FUEL1_RFD_CTRL_STRUCT *)NULL;
    }

    // Populate the structure with the minimum required
    psRFDCtrl->hDBInterface = hDBInterface;
    psRFDCtrl->psDBInterface = psDBInterface;
    psRFDCtrl->psOTAInterface = psOTAInterface;

    // Get the database content version
    psRFDCtrl->tCurrentVersion = (RFD_UPDATE_VERSION)
        psRFDCtrl->psDBInterface->un16DBContentVer(hDBInterface);

    return psRFDCtrl;
}

/*****************************************************************************
*
*   bInitRFDCtrl
*
*****************************************************************************/
static BOOLEAN bInitRFDCtrl (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        )
{
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hNoBrand, hCurDB = STRING_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Store the RFD connection handle
        psRFDCtrl->hRFD = hRFD;

        // Intialize the sql command buffer
        psRFDCtrl->pcSQLCommandBuffer = (char *)
            SMSO_hCreate(
            FUEL1_OBJECT_NAME": SQLBuffer",
                FUEL_MAX_SQL_STRING_LENGTH,
                SMS_INVALID_OBJECT, FALSE);
        if (psRFDCtrl->pcSQLCommandBuffer == NULL)
        {
            break;
        }

        psRFDCtrl->tBufferSize = FUEL_MAX_SQL_STRING_LENGTH;

        // Initialize the structure
        psRFDCtrl->tStartProgressIndex = tStartingIndex;
        psRFDCtrl->tCurProgressIndex = 0;

        // Create the brand text list
        eReturnCode = OSAL.eLinkedListCreate(
            &psRFDCtrl->hBrandTexts,
            FUEL1_OBJECT_NAME":BrandTexts",
            (OSAL_LL_COMPARE_HANDLER)n16CompareTextEntries,
            OSAL_LL_OPTION_UNIQUE
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create the brand string for stations without a brand
        hNoBrand = STRING.hCreate( FUEL1_UNBRANDED_TEXT,
                                   strlen(FUEL1_UNBRANDED_TEXT) );

        if (hNoBrand == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Add it to our brand list
        bSuccess = bAddBrandText(
            psRFDCtrl, FUEL1_UNBRANDED_ID, hNoBrand);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Create our file reader buffer
        bSuccess = DATASERVICE_MGR_bCreateFileBuffer(
            &psRFDCtrl->hBlockPool,
            &psRFDCtrl->hBuffer,
            FUEL1_RFD_READ_BLOCK_SIZE
                );
        if (bSuccess == FALSE)
        {
            // Error! destroy object
            break;
        }

        // Create all of our fixed objects used
        // for constant values
        bSuccess = bCreateFixedConstants(&psRFDCtrl->sFixed);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Find out where to place our working copy
        bSuccess = psRFDCtrl->psDBInterface->bRefDBBank(
            psRFDCtrl->hDBInterface, &hCurDB, &psRFDCtrl->hDBPath);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Are we starting a new update?
        if (tStartingIndex == 0)
        {
            // Yes!  Get a copy of the db
            bSuccess = DB_UTIL_bCopyDB(
                STRING.pacCStr(psRFDCtrl->hDBPath),
                STRING.pacCStr(hCurDB));
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Connect to our working database file now
        bSuccess = bConnectToDB(psRFDCtrl, FALSE);
        if (bSuccess == FALSE)
        {
            break;
        }

        if (tStartingIndex == 0)
        {
            bSuccess = psRFDCtrl->psDBInterface->bDBUpdateBegin(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->pcSQLCommandBuffer,
                psRFDCtrl->tBufferSize);

            if (bSuccess == FALSE)
            {
                break;
            }
        }

    } while (FALSE);

    if (hCurDB != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hCurDB);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vDestroyRFDCtrl
*
*****************************************************************************/
static void vDestroyRFDCtrl (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psRFDCtrl == NULL)
    {
        return;
    }

    psRFDCtrl->hRFD = RFD_INTERFACE_INVALID_OBJECT;

    // Disconnect from the database
    if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psRFDCtrl->hDBPath);
        psRFDCtrl->hDBPath = STRING_INVALID_OBJECT;
    }

    psRFDCtrl->tStartProgressIndex = 0;
    psRFDCtrl->tCurProgressIndex = 0;

    if (psRFDCtrl->pcSQLCommandBuffer != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psRFDCtrl->pcSQLCommandBuffer);
        psRFDCtrl->pcSQLCommandBuffer = NULL;
    }

    // Empty out the text list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psRFDCtrl->hBrandTexts,
        (OSAL_LL_RELEASE_HANDLER)vRemoveTextEntry );
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psRFDCtrl->hBrandTexts);
        psRFDCtrl->hBrandTexts = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the file buffer
    DATASERVICE_MGR_vDestroyFileBuffer(
        psRFDCtrl->hBlockPool, psRFDCtrl->hBuffer);
    psRFDCtrl->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    psRFDCtrl->hBuffer = OSAL_INVALID_BUFFER_HDL;

    // Destroy all of our fixed objects
    vDestroyFixedConstants(&psRFDCtrl->sFixed);

    // Destroy this object
    SMSO_vDestroy((SMS_OBJECT)psRFDCtrl);

    return;
}

/*****************************************************************************
*
*   vDeleteDB
*
*****************************************************************************/
static void vDeleteDB (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);
        remove(pcDBPath);
    }
    return;
}

/*****************************************************************************
*
*   bConnectToDB
*
*****************************************************************************/
static BOOLEAN bConnectToDB (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        )
{
    SQL_INTERFACE_OPTIONS_MASK tOptions = SQL_INTERFACE_OPTIONS_NONE;
    BOOLEAN bConnected = FALSE;

    if (psRFDCtrl->hDBPath != NULL)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);

        if (bPerformIntegrityCheck == FALSE)
        {
            tOptions |= SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK;
        }

        // Connect to the database
        psRFDCtrl->hDBConnection =
            SQL_INTERFACE.hConnect(
                pcDBPath, tOptions,
                (DATASERVICE_ERROR_CODE_ENUM *)NULL);

        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            bConnected = TRUE;
        }
    }

    return bConnected;
}

/*****************************************************************************
*
*   bFinalizeDB
*
*****************************************************************************/
static BOOLEAN bFinalizeDB (
   FUEL1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bDBReady;

    bDBReady = bConnectToDB(psRFDCtrl, TRUE);
    if (bDBReady == TRUE)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    return bDBReady;
}

/*****************************************************************************
*
*   bRFDProductStateChange
*
*   Implements state FSM for the RFD product
*
*****************************************************************************/
static BOOLEAN bRFDProductStateChange (
    FUEL1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        )
{
    BOOLEAN bSuccess = TRUE, bActionNeeded = FALSE;

    switch (psState->ePreviousState)
    {
        case DATA_PRODUCT_STATE_INITIAL:
        {
            // Only perform an action when moving to READY
            if (DATA_PRODUCT_STATE_READY == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_ERROR:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_READY:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_UNSUBSCRIBED:
        {
            if (DATA_PRODUCT_STATE_READY == psState->eState ||
                DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_DISABLED:
        default:
        {
            // Nothing to do
        }
        break;
    }

    if (TRUE == bActionNeeded)
    {
        if (DATA_PRODUCT_STATE_READY == psState->eState)
        {
            // Enable RFD updates when we enter the READY
            // state under the right circumstances
            bSuccess = bEnableRFDUpdates(psObj);
        }
        else if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
        {
            // Disable RFD updates under the right circumstances
            vDisableRFDUpdates(psObj);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bEnableRFDUpdates
*
*****************************************************************************/
static BOOLEAN bEnableRFDUpdates (
    FUEL1_OBJECT_STRUCT *psObj
        )
{
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl = (FUEL1_RFD_CTRL_STRUCT *)NULL;

    if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
    {
        // Already initialized
        return TRUE;
    }

    do
    {
        // Create the RFD processor object
        psRFDCtrl = psCreateRFDCtrl(psObj, psObj->hDBInterface,
                                           psObj->psDBInterface,
                                           psObj->psOTAInterface);
        if (psRFDCtrl == NULL)
        {
            break;
        }

        // Connect to RFD
        psObj->hRFD = RFD_INTERFACE_hConnect(
            psRFDCtrl->psOTAInterface->un8RFDClientId,
            psRFDCtrl->tCurrentVersion,
            psRFDCtrl->psOTAInterface->tMaxVersionBitlen,
            eRFDFileProcessor,
            (RFD_OPTIONAL_CALLBACKS_STRUCT *)NULL,
            (void *)psRFDCtrl);

        if (psObj->hRFD == RFD_INTERFACE_INVALID_OBJECT)
        {
            // Error! destroy object
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL1_OBJECT_NAME": Can't create RFD connection");
            break;
        }

        return TRUE;

    } while (FALSE);

    vDestroyRFDCtrl(psRFDCtrl);

    return FALSE;
}

/*****************************************************************************
*
*   vDisableRFDUpdates
*
*****************************************************************************/
static void vDisableRFDUpdates (
    FUEL1_OBJECT_STRUCT *psObj
        )
{
    // Disconnect RFD interface if it is connected
    if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
    {
        RFD_INTERFACE_vDisconnect(psObj->hRFD);
        psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bPriceProductStateChange
*
*****************************************************************************/
static BOOLEAN bPriceProductStateChange (
    FUEL1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        )
{
    BOOLEAN bActionNeeded = FALSE, bEnabled = FALSE;

    switch (psState->ePreviousState)
    {
        case DATA_PRODUCT_STATE_INITIAL:
        {
            // Only perform an action when moving to READY
            if (DATA_PRODUCT_STATE_READY == psState->eState)
            {
                bActionNeeded = TRUE;
                bEnabled =  TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_ERROR:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_READY:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_UNSUBSCRIBED:
        {
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
            else if (DATA_PRODUCT_STATE_READY == psState->eState)
            {
                bActionNeeded = TRUE;
                bEnabled =  TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_DISABLED:
        default:
        {
            // Nothing to do
        }
        break;
    }

    if (TRUE == bActionNeeded)
    {
        if (TRUE == bEnabled)
        {
            // Initialize and prepare the price processing memory
            // under the right circumstances
            vInitializePriceMemory(&psObj->sPriceMem);
        }
        else
        {
            // Uninitialize the price memory under the 
            // right circumstances
            vUninitPriceMemory(&psObj->sPriceMem);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bProcessTextMessage
*
*****************************************************************************/
static BOOLEAN bProcessTextMessage (
    FUEL1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bProcessMessage, bMore;
    FUEL1_TEXT_WORK_STRUCT sWork;

    // Initialize the work unit
    sWork.un8NumEntries = 0;
    sWork.un8Version = 0;

    // We don't support the long text version
    sWork.sText.hLongText = STRING_INVALID_OBJECT;

    // First, process the text message header and
    // populate the work
    bProcessMessage = bProcessTextMessageHeader(
        psObj, &sWork, hPayload );
    if (bProcessMessage == FALSE)
    {
        return TRUE;
    }

    // Now, can we begin updating the text table now?
    bProcessMessage = GsFuelMgrIntf.bTextUpdateBegin(
        psObj->hFuelService, sWork.un8Version);
    if (FALSE == bProcessMessage)
    {
        return TRUE;
    }

    do
    {
        // Now, continue to process the message until we're done
        bMore = bProcessTextMessageData(psObj, &sWork, hPayload);

    } while (bMore == TRUE);

    // Tell the manager we're done now
    GsFuelMgrIntf.vTextUpdateEnd(psObj->hFuelService, sWork.un8Version);

    return TRUE;
}

/*****************************************************************************
*
*   bProcessTextMessageHeader
*
*****************************************************************************/
static BOOLEAN bProcessTextMessageHeader (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_TEXT_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    do
    {
        size_t tBitsRead = 0;
        BOOLEAN bProcessThisMessage;
        UN8 un8Language = 0;

        // Read the text version field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8Version, 0,
            FUEL1_TEXT_VER_BITLEN);

        if (tBitsRead != FUEL1_TEXT_VER_BITLEN)
        {
            break;
        }

        // Ask the DB if we need to continue
        bProcessThisMessage = psObj->psDBInterface->bTextQuery(
            psObj->hDBInterface, psWork->un8Version );

        if (bProcessThisMessage == FALSE)
        {
            // We don't need to process this message
            break;
        }

        // Skip the reserved bits
        tBitsRead = OSAL.tBufferSeekHeadBits(
            hPayload, FUEL1_TEXT_RSVD_BITLEN);

        if (tBitsRead != FUEL1_TEXT_RSVD_BITLEN)
        {
            break;
        }

        // Read the text language field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8Language, 0,
            FUEL1_TEXT_LANG_BITLEN);

        if (tBitsRead != FUEL1_TEXT_LANG_BITLEN)
        {
            break;
        }

        if (un8Language != FUEL1_ENGLISH)
        {
            // Only support English for now
            // TODO: locales
            break;
        }

        // Get the number of fuel entries listed
        psWork->un8NumEntries = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8NumEntries, 0,
            FUEL1_TEXT_NUM_ENTRIES_BITLEN);
        if (tBitsRead != FUEL1_TEXT_NUM_ENTRIES_BITLEN)
        {
            break;
        }

        /*
         * SX-9845-0023 Section 5.2.2:
         * COUNT + 1 is the number of Fuel Types
         * listed in this Access Unit
         * */
        psWork->un8NumEntries++;

        // Initialize text work unit data
        psWork->sText.un8TextId = 0;
        psWork->sText.bBrandText = FALSE;

        return TRUE;
    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bProcessTextMessageData
*
*   This function is utilized when processing a work unit based on text
*   message data.  Each text entry is provided to the manager via the
*   manager interface.
*
*****************************************************************************/
static BOOLEAN bProcessTextMessageData (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_TEXT_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tBitsRead;
    BOOLEAN bMoreToProcess = TRUE,
            bSuccess = TRUE;

    // Get the text type as the id
    psWork->sText.un8TextId = 0;
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &psWork->sText.un8TextId, 0,
        FUEL1_TEXT_TYPE_BITLEN);

    if (tBitsRead != FUEL1_TEXT_TYPE_BITLEN)
    {
        bMoreToProcess = FALSE;
    }

    // Extract the next fuel type

    // Read the text (baudot) as a STRING_OBJECT
    psWork->sText.hText = BAUDOT_hToString(
        SMS_INVALID_OBJECT, hPayload,
        BAUDOT_BEHAVIOR_PROCESS_TO_END,
        TRUE, TRUE, 0, NULL);

    if (psWork->sText.hText == STRING_INVALID_OBJECT)
    {
        return FALSE;
    }

    // Tell the manager about the new text
    bSuccess = GsFuelMgrIntf.bTextUpdate(
        psObj->hFuelService, &psWork->sText);
    if (bSuccess == FALSE)
    {
        bMoreToProcess = FALSE;
    }

    // We have one less entry to process
    psWork->un8NumEntries--;

    // Did we reach the end of this list of entries?
    if (psWork->un8NumEntries == 0)
    {
        // We've finished processing the fuel type list
        bMoreToProcess = FALSE;
    }

    return bMoreToProcess;
}

/*****************************************************************************
*
*   bProcessPriceMessage
*
*****************************************************************************/
static BOOLEAN bProcessPriceMessage (
    FUEL1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bProcessMessage, bMore;
    FUEL1_PRICE_WORK_STRUCT sWork;

    // Do we want to process price messages?
    if (psObj->sPriceMem.bEnabled == FALSE)
    {
        // Nope!
        return TRUE;
    }

    // Initialize the work unit
    OSAL.bMemSet((void *)&sWork, 0, sizeof(FUEL1_PRICE_WORK_STRUCT));

    // First, process the price message header and
    // populate the work unit
    bProcessMessage = bProcessPriceMessageHeader(
        psObj, &sWork, hPayload );
    if (bProcessMessage == FALSE)
    {
        return TRUE;
    }

    do
    {
        // Now, continue to process the message until we're done
        bMore = bProcessPriceData(psObj, &sWork, hPayload);

    } while (bMore == TRUE);

    // All done!
    vReleasePriceWorkData(&sWork);

    return TRUE;
}

/*****************************************************************************
*
*   bProcessPriceMessageHeader
*
*****************************************************************************/
static BOOLEAN bProcessPriceMessageHeader (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_PRICE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOk;
    size_t tBitsRead;
    FUEL1_PRICE_BASELINE_STRUCT *psCurBaseline;
    FUEL1_PRICE_BASELINE_STRUCT *psPrevBaseline;
    FUEL1_PRICE_SCALING_FACTOR_ENUM  eScalingFactor;
    BOOLEAN bRegionUpdateNeeded;
    FUEL_REGION tRegion = 0;
    UN8 un8TextVer = 0;

    // Read the text version field
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &un8TextVer, 0,
        FUEL1_TEXT_VER_BITLEN);
    if (tBitsRead != FUEL1_TEXT_VER_BITLEN)
    {
        return FALSE;
    }

    // Region the region id field
    bOk = OSAL.bBufferReadBitsToUN16(
        hPayload, (UN16 *)&tRegion,
        FUEL1_REGION_BITLEN);

    if (bOk == FALSE)
    {
        return FALSE;
    }

    // Start region update by checking filter on manager side
    bRegionUpdateNeeded = GsFuelMgrIntf.bRegionUpdateBegin(
        psObj->hFuelService,
        psObj->un32CurrentMsgCRC,
        tRegion, un8TextVer);
    if (bRegionUpdateNeeded == FALSE)
    {
        // Manager wants this to be filtered out
        return FALSE;
    }

    // Is AU Count size present?
    if (bFieldPresent(hPayload) == TRUE)
    {
        UN8 un8CtSize = 0;

        // Seek past the AU stats -- who needs this?

        // Read the CTSize field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8CtSize, 0,
            FUEL1_PRICE_CTSIZE_BITLEN);
        if (tBitsRead != FUEL1_PRICE_CTSIZE_BITLEN)
        {
            return FALSE;
        }

        /*
         * RX 185, section 6.2.5:
         *
         * CTSIZE is a four bit field whose value determines
         * the size of the two subsequent fields. [CTSIZE] + 1
         * is the size, in bits, of AUTOT and AUCT.
         * */
        un8CtSize++;

        // Seek past the AU-related fields
        tBitsRead = OSAL.tBufferSeekHeadBits(
            hPayload, (un8CtSize*2));
        if (tBitsRead != (unsigned)(un8CtSize * 2 ))
        {
            return FALSE;
        }
    }

    // Read the FSUID size field
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &psWork->un8FSSize, 0,
        FUEL1_FSSIZE_BITLEN);

    if (tBitsRead != FUEL1_FSSIZE_BITLEN)
    {
        return FALSE;
    }

    // Initialize the static price baselines
    vInitializeStaticPriceBaselines(psWork);

    // Grab the first price baseline and use it
    // as the current / previous
    psPrevBaseline = psCurBaseline =
        &psWork->asPriceBaselines[0];

    // Process all the baseline fuel price info now
    do
    {
        // Do we need to allocate a new price baseline?
        if (psCurBaseline == NULL)
        {
            // Yes -- create one now
            psCurBaseline = psCreateNewPriceBaseline(psObj);
            if (psCurBaseline == NULL)
            {
                puts(FUEL1_OBJECT_NAME": Unable to create new price baseline");

                vReleasePriceWorkData(psWork);
                return FALSE;
            }

            // Add this baseline to the list
            psPrevBaseline->psNext = psCurBaseline;
        }

        // Read the fuel type
        psCurBaseline->un8FuelType = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psCurBaseline->un8FuelType, 0,
            FUEL1_PRICE_FUEL_TYPE_BITLEN);

        // Read the fuel scaling factor
        eScalingFactor = (FUEL1_PRICE_SCALING_FACTOR_ENUM)0;
        tBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &eScalingFactor, 0,
            FUEL1_PRICE_FPSCALE_BITLEN);

        switch (eScalingFactor)
        {
            case FUEL1_PRICE_SCALING_FACTOR_NONE:
            {
                // The reported prices are in the
                // lowest denomination.  We report
                // prices to the application in
                // tenths of the lowest denomination,
                // so we need to multiply by 10
                psCurBaseline->un8ScalingFactor = 10;
            }
            break;

            case FUEL1_PRICE_SCALING_FACTOR_10X:
            {
                // The reported prices are in the
                // lowest denomination, and need to
                // be multiplied by 10 before they
                // are reported to the application.
                // We report prices to the application in
                // tenths of the lowest denomination,
                // so we need to multiply by 100
                psCurBaseline->un8ScalingFactor = 100;
            }
            break;

            case FUEL1_PRICE_SCALING_FACTOR_ONE_TENTH:
            {
                // The reported prices are in tenths
                // of a cent, which is how we report
                // them to the application.  We
                // don't have to do any scaling now
                psCurBaseline->un8ScalingFactor = 1;
            }
            break;

            case FUEL1_PRICE_SCALING_FACTOR_UNDEF:
            case FUEL1_PRICE_SCALING_MAX_FACTORS:
            default:
            {
                puts(FUEL1_OBJECT_NAME": Bad scaling factor in message");

                vReleasePriceWorkData(psWork);
                return FALSE;
            }
        }

        // Read fp min
        psCurBaseline->un16BasePrice = 0;
        bOk = OSAL.bBufferReadBitsToUN16(
            hPayload, &psCurBaseline->un16BasePrice,
            FUEL1_PRICE_FPMIN_BITLEN);
        if (bOk == TRUE)
        {
            tBitsRead += FUEL1_PRICE_FPMIN_BITLEN;
        }

        // Read price size
        psCurBaseline->un8PriceSize = 0;
        tBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &psCurBaseline->un8PriceSize, 0,
            FUEL1_PRICE_FPRNG_BITLEN);

        if (tBitsRead != FUEL1_PRICE_HDR_ENTRY_BITLEN)
        {
            // That failed, so release the work unit
            // and stop here
            vReleasePriceWorkData(psWork);

            return FALSE;
        }

        // RX 185 Section 6.2.12
        psCurBaseline->un8PriceSize += FUEL_PRICE_FPRNG_ADJUSTMENT;

        // Calculate the "out of fuel" flag for this message
        psCurBaseline->un32OutOfFuelCheckValue = FUEL1_PRICE_UNAVAIL_BASE_FLAG;
        psCurBaseline->un32OutOfFuelCheckValue >>= (FUEL1_PRICE_UNAVAIL_BASE_FLAG_BITLEN -
                        psCurBaseline->un8PriceSize);

        // We have another valid baseline
        psWork->tNumValidBaselines++;

        // Move to the next baseline structure
        psPrevBaseline = psCurBaseline;
        psCurBaseline = psCurBaseline->psNext;

    } while (bFieldPresent(hPayload) == TRUE);

    // Prepare the price memory now
    bOk = bPreparePriceMemory(&psObj->sPriceMem, psWork);

    return bOk;
}

/*****************************************************************************
*
*   bProcessPriceData
*
*   This function is utilized when processing a work unit based on price
*   update data.  Each entry is provided to the manager via the
*   manager interface.
*
*****************************************************************************/
static BOOLEAN bProcessPriceData (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_PRICE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOk, bPriceAvailable,
            bMoreToProcess = TRUE;
    size_t tBitsRead,
           tBitsRemaining;
    UN8 un8VLCBit = 0, un8PriceAgeInDays, un8Index;
    FUEL1_PRICE_BASELINE_STRUCT *psCurBaseline;
    FUEL_STATION_ID tStation;
    FUEL_PRICE_ENTRY_STRUCT *psCurPriceEntry;

    // We don't have any prices just yet
    psWork->sPriceRow.tNumPrices = 0;

    do
    {
        for (un8Index = 0; un8Index < FUEL1_PRICE_MAX_VLC_CODE_LEN; un8Index++)
        {
            un8VLCBit = 0;
            tBitsRead =  OSAL.tBufferReadHeadBits(
                    hPayload, &un8VLCBit, 0, FUEL1_PRICE_VLC_BITLEN);

            if (tBitsRead != FUEL1_PRICE_VLC_BITLEN)
            {
                bMoreToProcess = FALSE;
                break;
            }

            if (un8VLCBit != 0)
            {
                break;
            }
        }

        if (un8VLCBit != 0)
        {
            tStation = psWork->tPrevStation + un8Index + 1;
        }
        else
        {
            // No, just read the station's ID
            tStation = 0;
            bOk = OSAL.bBufferReadBitsToUN32(
                hPayload, (UN32 *)&tStation,
                psWork->un8FSSize);
            if (bOk == FALSE)
            {
                bMoreToProcess = FALSE;
                break;
            }
        }

        // Save this station ID as our previous
        psWork->tPrevStation = tStation;

        // Grab the first baseline
        psCurBaseline = &psWork->asPriceBaselines[0];

        // Start processing the price list for this station
        for (un8Index = 0; un8Index < psWork->tNumValidBaselines; un8Index++)
        {
            // Grab the current price entry
            psCurPriceEntry = &psWork->sPriceRow.pasPrices[psWork->sPriceRow.tNumPrices];

            // NULL check these pointers
            if ((psCurPriceEntry == NULL) ||
                (psCurBaseline == NULL))
            {
                bMoreToProcess = FALSE;
                break;
            }

            // Read the marker to determine
            // if this price is available
            bPriceAvailable = FALSE;
            tBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &bPriceAvailable, 0,
                FUEL1_PRICE_MARKER_BITLEN );
            if (tBitsRead != FUEL1_PRICE_MARKER_BITLEN)
            {
                bMoreToProcess = FALSE;
                break;
            }

            if (bPriceAvailable == FALSE)
            {
                // Move to the next baseline
                psCurBaseline = psCurBaseline->psNext;

                continue;
            }

            // Read the price age field
            un8PriceAgeInDays = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &un8PriceAgeInDays, 0,
                FUEL1_PRICE_AGE_BITLEN );

            if (tBitsRead != FUEL1_PRICE_AGE_BITLEN)
            {
                bMoreToProcess = FALSE;
                break;
            }

            // Initialize this to the age in days (it'll be converted later)
            psCurPriceEntry->un32PriceAgeUTCSeconds = (UN32)un8PriceAgeInDays;

            // Read the price field
            psCurPriceEntry->un32FuelPrice = 0;
            bOk = OSAL.bBufferReadBitsToUN32(
                hPayload, &psCurPriceEntry->un32FuelPrice,
                psCurBaseline->un8PriceSize);
            if (bOk == FALSE)
            {
                bMoreToProcess = FALSE;
                break;
            }

            // Is this the "price unavailable" flag?  If a price is
            // not available, the un32FuelPrice value will be all '1's, so
            // we'll do an xor on the appropriate number of '1' bits and see
            // if that results in a match (0)
            if ((psCurPriceEntry->un32FuelPrice ^ psCurBaseline->un32OutOfFuelCheckValue) != 0)
            {
                // This fuel type is available at this station
                psCurPriceEntry->eAvailable = FUEL_AVAILABLE;

                // Finish calculating the price now
                psCurPriceEntry->un32FuelPrice += (UN32)
                    psCurBaseline->un16BasePrice;

                // Now apply the scaling factor
                psCurPriceEntry->un32FuelPrice *=
                    psCurBaseline->un8ScalingFactor;
            }
            else
            {
                // This fuel type is sold at this station, but
                // they are out (blame Moammar Gadhafi)
                psCurPriceEntry->eAvailable = FUEL_NOT_AVAILABLE;
            }

            // We are now ready to inform the manager
            // of new price information
            // Populate the fuel type field
            // using the fuel type from the current baseline
            psCurPriceEntry->un8FuelType = psCurBaseline->un8FuelType;

            // We now have one more price entry
            psWork->sPriceRow.tNumPrices++;

            // Move to the next baseline
            psCurBaseline = psCurBaseline->psNext;
        }

        // If we didn't experience an issue processing these prices,
        // provide them to the manager
        if (bMoreToProcess == TRUE)
        {
            // Provide this info to the manager --
            // if we experience an error, then stop here
            bMoreToProcess = GsFuelMgrIntf.bPriceUpdate(
                psObj->hFuelService,
                tStation, &psWork->sPriceRow);
        }

    } while (FALSE);

    // RX 124-G, section 6.2.1.1: Bit Stuffing
    /*
        The packing of the groups of data carries
        no byte alignment; the last Fuel Price group of
        data may not end on a byte boundary requiring
        stuffing bits to be added to provide byte
        alignment for the CRC field that occupies the
        last 2 bytes of the Access Unit. There may
        be zero to seven stuffing bits and all must be 0.
    */

    // How many bits are left?
    tBitsRemaining = OSAL.tBufferGetSizeInBits(hPayload);

    // If there are less than 8 bits left, then we
    // only have bit stuffing left in this message
    if (tBitsRemaining <= 7)
    {
        bMoreToProcess = FALSE;
    }

    // If the work unit is complete then tell
    // the manager when we're all done
    if (bMoreToProcess == FALSE)
    {
        // Tell the manager this price update is now
        // complete
        GsFuelMgrIntf.vRegionUpdateComplete(psObj->hFuelService);
    }

    return bMoreToProcess;
}

/*****************************************************************************
*
*   bProcessRFDHeader
*
*****************************************************************************/
static BOOLEAN bProcessRFDHeader (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    BOOLEAN bDataTransferred, bEntryAdded;
    size_t tBitsRead;
    UN8 un8BRNum = 0,
        un8BRCode;
    STRING_OBJECT hBrand;
    BOOLEAN bRead;

    // Fill the buffer with file data
    bDataTransferred = DATASERVICE_MGR_bFillBufferBlock(
        psFile, psRFDCtrl->hBuffer);
    if (bDataTransferred == FALSE)
    {
        // Couldn't access the file
        return FALSE;
    }

    // Read the BRNUM field
    tBitsRead = OSAL.tBufferReadHeadBits(
        psRFDCtrl->hBuffer, &un8BRNum, 0,
        FUEL1_BRNUM_BITLEN);
    if (tBitsRead != FUEL1_BRNUM_BITLEN)
    {
        return FALSE;
    }

    // RX 185, Section 7.2.1
    un8BRNum++;

    do
    {
        // Read the BRCODE field
        un8BRCode = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8BRCode, 0,
            FUEL1_BRCODE_BITLEN);
        if (tBitsRead != FUEL1_BRCODE_BITLEN)
        {
            return FALSE;
        }

        // Read the BAUDOT string for this brand code
        hBrand = BAUDOT_hToString(
            SMS_INVALID_OBJECT,
            psRFDCtrl->hBuffer,
            BAUDOT_BEHAVIOR_PROCESS_TO_END,
            TRUE, TRUE, 0, NULL);
        if (hBrand == STRING_INVALID_OBJECT)
        {
            return FALSE;
        }

        // Add a new brand text entry now
        bEntryAdded = bAddBrandText(
            psRFDCtrl, un8BRCode, hBrand);
        if (bEntryAdded == FALSE)
        {
            return FALSE;
        }

    } while (--un8BRNum > 0);

    // Read the region count field
    psRFDCtrl->un16NumRegionsInUpdate = 0;
    bRead = OSAL.bBufferReadBitsToUN16(
        psRFDCtrl->hBuffer,
        &psRFDCtrl->un16NumRegionsInUpdate,
        FUEL1_DESC_REG_COUNT_BITLEN);
    if (bRead == FALSE)
    {
        return FALSE;
    }

    if ((psRFDCtrl->un16NumRegionsInUpdate == 0) ||
        (psRFDCtrl->un16NumRegionsInUpdate > 0x7FF))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL1_OBJECT_NAME": Invalid number of regions in file header");

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   eProcessNextRFDEntry
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_INCOMPLETE;
    BOOLEAN bSuccess = TRUE;

    // Do we have any work left?
    if (psRFDCtrl->un16NumRegionsInUpdate != psRFDCtrl->tCurProgressIndex)
    {
        // Process the next region
        bSuccess = bProcessRegion(psRFDCtrl, psFile);

        if (bSuccess == TRUE)
        {
            // Increment our progress index
            psRFDCtrl->tCurProgressIndex++;

            if (psRFDCtrl->tCurProgressIndex > psRFDCtrl->tStartProgressIndex)
            {
                // Report our progress
                bSuccess = RFD_INTERFACE_bReportProgress(
                    psRFDCtrl->hRFD, psRFDCtrl->tCurProgressIndex);
            }
        }
    }

    if (bSuccess == TRUE)
    {
        // Are we done?
        if (psRFDCtrl->un16NumRegionsInUpdate == psRFDCtrl->tCurProgressIndex)
        {
            // Finalize the DB
            bSuccess = psRFDCtrl->psDBInterface->bDBUpdateEnd(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->pcSQLCommandBuffer,
                psRFDCtrl->tBufferSize, (UN8)tUpdateVersion);

            if (bSuccess == TRUE)
            {
                // Yes -- tell the caller
                eResult = RFD_PROCESS_RESULT_COMPLETE;
            }
        }
    }

    if (bSuccess == FALSE)
    {
        eResult = RFD_PROCESS_RESULT_ERROR;
    }

    return eResult;
}

/*****************************************************************************
*
*   bProcessRegion
*
*   Process a region entry in the DB update file
*
*****************************************************************************/
static BOOLEAN bProcessRegion(
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    UN8 un8UType,
        un8FSSize = 0,
        un8Div = 0,
        un8SubReg = 0;
    size_t tBitsRead;
    BOOLEAN bOk;
    UN16 un16SubDiv = 0;
    FUEL_STATION_ROW_STRUCT sStationRow;
    BOOLEAN bInformManager = TRUE;
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType =
        FUEL_STATION_UPDATE_MAX_TYPES;
    N32 n32PrevLat = -1, n32PrevLon = -1;

    // Did we already process this region?
    if (psRFDCtrl->tCurProgressIndex < psRFDCtrl->tStartProgressIndex)
    {
        bInformManager = FALSE;
    }

    // Read the region id field
    psRFDCtrl->tRegion = 0;
    bOk = OSAL.bBufferReadBitsToUN16(
        psRFDCtrl->hBuffer,
        (UN16 *)&psRFDCtrl->tRegion,
        FUEL1_REGION_BITLEN);

    if (bOk == FALSE)
    {
        return FALSE;
    }

    if (bInformManager == TRUE)
    {
        // We now have the region ID, and version.
        // See if the manager wants the update
        bOk = psRFDCtrl->psDBInterface->bRegionUpdateBegin(
            psRFDCtrl->hDBConnection,
            psRFDCtrl->pcSQLCommandBuffer,
            psRFDCtrl->tBufferSize,
            psRFDCtrl->tRegion);
        if (bOk == FALSE)
        {
            return FALSE;
        }
    }

    do
    {
        BOOLEAN bBreak = FALSE;

        // Read the FSSIZE field
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8FSSize, 0,
            FUEL1_FSSIZE_BITLEN);
        if (tBitsRead != FUEL1_FSSIZE_BITLEN)
        {
            break;
        }

        // RX 185, Section 7.2.2.2
        un8FSSize++;

        do
        {
            // Fill the buffer with file data
            DATASERVICE_MGR_bFillBufferBlock(
                psFile, psRFDCtrl->hBuffer);

            // Read the UTYPE field
            un8UType = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8UType, 0,
                FUEL1_DESC_UPDATE_TYPE_BITLEN);
            if (tBitsRead != FUEL1_DESC_UPDATE_TYPE_BITLEN)
            {
                break;
            }

            switch (un8UType)
            {
                case FUEL1_UPDATE_TYPE_DELETE:
                {
                    eUpdateType = FUEL_STATION_UPDATE_TYPE_DELETE;
                }
                break;

                case FUEL1_UPDATE_TYPE_NEW:
                {
                    eUpdateType = FUEL_STATION_UPDATE_TYPE_NEW;
                }
                break;

                case FUEL1_UPDATE_TYPE_MODIFY:
                {
                    eUpdateType = FUEL_STATION_UPDATE_TYPE_MODIFY_GIVEN_ATTRIBS;
                }
                break;

                case FUEL1_UPDATE_TYPE_END:
                {
                    // All done with this region
                    bBreak = TRUE;
                }
                break;

                default:
                {
                    // Don't know what we've read,
                    // and we don't know how to continue either
                    bBreak = TRUE;
                }
            }

            // Break if we experienced a problem or if
            // we don't have a valid update type
            if ((bBreak == TRUE) ||
                (eUpdateType >= FUEL_STATION_UPDATE_MAX_TYPES))
            {
                break;
            }

            // Clear the row
            OSAL.bMemSet(&sStationRow, 0, sizeof(FUEL_STATION_ROW_STRUCT));

            // Clear State ID
            sStationRow.tStateID = STATE_INVALID_ID;

            // Clear Logo ID
            sStationRow.n16LogoId = -1;

            // All station data is for this region
            // and this version
            sStationRow.tRegion = psRFDCtrl->tRegion;

            // Put back coordinates of previous station
            sStationRow.n32PrevLat = n32PrevLat;
            sStationRow.n32PrevLon = n32PrevLon;

            // Process the station entry and provide
            // it to the manager if necessary
            bOk = bProcessStation(
                psRFDCtrl, bInformManager, &sStationRow, eUpdateType,
                un8FSSize, &un8SubReg, &un8Div, &un16SubDiv);

            // Saving coordinates for next time
            n32PrevLat = sStationRow.n32PrevLat;
            n32PrevLon = sStationRow.n32PrevLon;

        } while (bOk == TRUE);

        if (bOk == FALSE)
        {
            break;
        }

        if (bInformManager == TRUE)
        {
            // Inform the manager that the update
            // for this region is done
            bOk = psRFDCtrl->psDBInterface->bRegionUpdateEnd(
                            psRFDCtrl->hDBConnection);
        }

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bProcessStation
*
*   Process a station entry in the DB update file
*
*****************************************************************************/
static BOOLEAN bProcessStation(
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    UN8 un8FSSize,
    UN8 *pun8LastSubRegion,
    UN8 *pun8LastDivision,
    UN16 *pun16LastSubDivision
        )
{
    BOOLEAN bOk,
            bLocationProvided = FALSE,
            bAmenitiesUpdated = FALSE;
    size_t tBitsRead;
    UN8 un8LocType = 0,
        un8BRCode = FUEL_INVALID_BRAND_ID;

    // Get the station ID
    psStationRow->tStationId = 0;
    bOk = OSAL.bBufferReadBitsToUN32(
        psRFDCtrl->hBuffer,
        (UN32 *)&psStationRow->tStationId,
        un8FSSize);
    if (bOk == FALSE)
    {
        return FALSE;
    }

    // If we're deleting an entry, we don't
    // need to read anything else.
    if (eUpdateType != FUEL_STATION_UPDATE_TYPE_DELETE)
    {
        // Extract this station's LOC information
        // Read the LOC type
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8LocType, 0,
            FUEL1_LOC_BITLEN);

        // Ensure that was read correctly
        if (tBitsRead != FUEL1_LOC_BITLEN)
        {
            return FALSE;
        }

        switch (un8LocType)
        {
            case FUEL1_LOC_SUBREG:
            {
                *pun8LastSubRegion = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    psRFDCtrl->hBuffer, pun8LastSubRegion, 0,
                    FUEL1_LOC_SUBREG_BITLEN);
                if (tBitsRead != FUEL1_LOC_SUBREG_BITLEN)
                {
                    return FALSE;
                }
            }
            /* no break */

            case FUEL1_LOC_DIV:
            {
                *pun8LastDivision = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    psRFDCtrl->hBuffer, pun8LastDivision, 0,
                    FUEL1_LOC_DIV_BITLEN);
                if (tBitsRead != FUEL1_LOC_SUBREG_BITLEN)
                {
                    return FALSE;
                }
            }
            /* no break */

            case FUEL1_LOC_SUBDIV:
            {
                // Broadcast gave us a new location
                bLocationProvided = TRUE;
                *pun16LastSubDivision = 0;
                bOk = OSAL.bBufferReadBitsToUN16(
                    psRFDCtrl->hBuffer, pun16LastSubDivision,
                    FUEL1_LOC_SUBDIV_BITLEN);
                if (bOk == FALSE)
                {
                    return FALSE;
                }
            }
            /* no break */
        }

        if (bLocationProvided == TRUE)
        {
            // Compute the lat / lon for this station
            bOk = bComputeLatLonFromGrid(
                psRFDCtrl, psStationRow,
                *pun8LastSubRegion,
                *pun8LastDivision,
                *pun16LastSubDivision);
            if (bOk == FALSE)
            {
                return FALSE;
            }
            
            // Saving calculated values for future use
            psStationRow->n32PrevLat = psStationRow->n32Lat;
            psStationRow->n32PrevLon = psStationRow->n32Lon;
        }
        else // Location info not provided
        {
            if (un8LocType == FUEL1_LOC_PREV)
            {
                // Use previous values
                psStationRow->n32Lat = psStationRow->n32PrevLat;
                psStationRow->n32Lon = psStationRow->n32PrevLon;
            }
            else
            {
                // Invalidate these fields
                psStationRow->n32Lat = -1;
                psStationRow->n32Lon = -1;
            }
        }

        // "Unknown Brand" by default
        un8BRCode = 0;

        // Read the brand code
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            // Read the BRCode field
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8BRCode, 0,
                FUEL1_BRCODE_BITLEN);

            if (tBitsRead != FUEL1_BRCODE_BITLEN)
            {
                return FALSE;
            }
        }

        // Get the brand string from the brand code
        psStationRow->hBrand = hGetBrandString(psRFDCtrl, un8BRCode );

        // Read the Fuel Station name (Baudot)
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            psStationRow->hName =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 0, NULL);
            if (psStationRow->hName == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }
        }
        else
        {
            psStationRow->hName = STRING_INVALID_OBJECT;
        }

        // Read the address (Baudot)
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            psStationRow->hAddr =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    FALSE, TRUE, 0, NULL);
            if (psStationRow->hAddr == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }
        }
        else
        {
            psStationRow->hAddr = STRING_INVALID_OBJECT;
        }

        // Read the city (Baudot)
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            psStationRow->hCity =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 0, NULL);
            if (psStationRow->hCity == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }
        }
        else
        {
            psStationRow->hCity = STRING_INVALID_OBJECT;
        }

        // Read state code
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN8 un8State = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8State, 0,
                FUEL1_DESC_STATE_BITLEN);

            if (tBitsRead != FUEL1_DESC_STATE_BITLEN)
            {
                return FALSE;
            }

            // Save the state id
            psStationRow->tStateID = (STATE_ID)un8State;
        }
        else
        {
            psStationRow->tStateID = STATE_INVALID_ID;
        }

        // Read ZIP
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN8 un8ZipType = 0;

            // First read the zip type
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8ZipType, 0,
                FUEL1_ZIP_TYPE_BITLEN);

            if (tBitsRead != FUEL1_ZIP_TYPE_BITLEN)
            {
                return FALSE;
            }

            if (un8ZipType == FUEL1_ZIP_TYPE_INTEGER)
            {
                UN32 un32Zip = 0;
                char acZip[FUEL1_MAX_ZIP_INT_CHARS + 1];

                // Read the zip code as an integer
                bOk = OSAL.bBufferReadBitsToUN32(
                    psRFDCtrl->hBuffer, &un32Zip,
                    FUEL1_ZIP_BITLEN);
                if (bOk == FALSE)
                {
                    return FALSE;
                }

                // Create a string for the zip code
                snprintf(&acZip[0], sizeof(acZip), "%05u", un32Zip);
                psStationRow->hZIP = STRING.hCreate(&acZip[0], 0);
            }
            else // Read ZIP as a string (baudot)
            {
                psStationRow->hZIP =
                    BAUDOT_hToString(
                        SMS_INVALID_OBJECT,
                        psRFDCtrl->hBuffer,
                        BAUDOT_BEHAVIOR_PROCESS_TO_END,
                        TRUE, TRUE, 0, NULL);
            }
        }
        else
        {
            psStationRow->hZIP = STRING_INVALID_OBJECT;
        }

        // Read Phone
        psStationRow->n64Phone = 0;
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN16 un16AreaCode = 0,
                 un16Exchange = 0,
                 un16Number = 0;
            OSAL.bBufferReadBitsToUN16(
                psRFDCtrl->hBuffer, &un16AreaCode, 10);
            OSAL.bBufferReadBitsToUN16(
                psRFDCtrl->hBuffer, &un16Exchange, 10);
            OSAL.bBufferReadBitsToUN16(
                psRFDCtrl->hBuffer, &un16Number, 14);

            // Convert to single int
            psStationRow->n64Phone =
                FUEL_MGR_n64GeneratePhoneFromFields(
                    un16AreaCode, un16Exchange, un16Number);
        }

        // Read Amenities 1
        psStationRow->un32Amenities = 0;
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            bOk = bProcessAmenities(
                psRFDCtrl, TRUE, &psStationRow->un32Amenities);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            bAmenitiesUpdated = TRUE;
        }

        // Read Amenities 2
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            bOk = bProcessAmenities(
                psRFDCtrl, FALSE, &psStationRow->un32Amenities);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            bAmenitiesUpdated = TRUE;
        }

        // Read Extension
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN8 un8NumExtensionBytes = 0;
            UN16 un16ExtensionBitSize = 0;

            // Read the extension size
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8NumExtensionBytes, 0,
                FUEL1_EXTENSION_SIZE_BITLEN);
            if (tBitsRead != FUEL1_EXTENSION_SIZE_BITLEN)
            {
                return FALSE;
            }

            // The size of the extension field is the number
            // of bytes times byte size
            un16ExtensionBitSize = un8NumExtensionBytes * 8;

            if (un16ExtensionBitSize >= FUEL1_UPDATE_ENTRY_VER_BITLEN)
            {
                // We have the update entry version here
                UN16 un16FieldVersion = 0;

                 bOk = OSAL.bBufferReadBitsToUN16(
                    psRFDCtrl->hBuffer, &un16FieldVersion,
                    FUEL1_UPDATE_ENTRY_VER_BITLEN);

                if (bOk == FALSE)
                {
                    return FALSE;
                }

                if (un16FieldVersion <= (UN16)psRFDCtrl->tCurrentVersion)
                {
                    // Skip this entry - we already know about it
                    bInformManager = FALSE;
                }

                un16ExtensionBitSize -= FUEL1_UPDATE_ENTRY_VER_BITLEN;
            }

            // Seek past the rest of the bits
            // since we dont' understand them
            OSAL.tBufferSeekHeadBits(
                psRFDCtrl->hBuffer, (size_t)un16ExtensionBitSize);
        }
    }

    if (bInformManager == TRUE)
    {
        // Provide the DB with the station update now
        bOk = psRFDCtrl->psDBInterface->bStationUpdate(
            psRFDCtrl->hDBConnection,
            &psRFDCtrl->pcSQLCommandBuffer,
            &psRFDCtrl->tBufferSize,
            psStationRow, eUpdateType,
            bAmenitiesUpdated);
    }
    else
    {
        bOk = TRUE;
    }

    return bOk;
}

/*****************************************************************************
*
*   bProcessAmenities
*
*   Process the amenities entry for a station
*
*****************************************************************************/
static BOOLEAN bProcessAmenities (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bAmenities1,
    UN32 *pun32Amenities
        )
{
    size_t tBitsRead = 0,
           tBitOffset = 0;
    UN8 un8CurrentAmen,
        un8RawAmenValue;
    UN32 un32Temp;

    // Are we working on the first amenities field?
    if (bAmenities1 == FALSE)
    {
        // No, make sure we don't overwrite amen1 with
        // these values from amen2
        tBitOffset = FUEL_AMENITIES_ARRAY_SIZE * FUEL1_AMEN_FIELD_BITLEN;
    }

    for (un8CurrentAmen = 0;
         un8CurrentAmen < FUEL_AMENITIES_ARRAY_SIZE;
         un8CurrentAmen++)
    {
        // Read two bits per field
        un8RawAmenValue = 0;
        tBitsRead += OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8RawAmenValue, 0,
            FUEL1_AMEN_FIELD_BITLEN);

        // Take that value
        un32Temp = (UN32)un8RawAmenValue;

        // Place it at the correct bit position
        un32Temp <<= tBitOffset;

        // Save the result
        *pun32Amenities |= un32Temp;

        // Update the bit position
        tBitOffset += FUEL1_AMEN_FIELD_BITLEN;
    }

    // Make sure we read it all
    if (tBitsRead != (FUEL_AMENITIES_ARRAY_SIZE * FUEL1_AMEN_FIELD_BITLEN))
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bFieldPresent
*
*   A simple function which reads 1 bits and reports TRUE or FALSE
*   based on that bit's value.
*
*****************************************************************************/
static BOOLEAN bFieldPresent(
    OSAL_BUFFER_HDL hBuffer
        )
{
    BOOLEAN bPresent = FALSE;
    size_t tBitsRead;

    // Read the presence flag
    tBitsRead = OSAL.tBufferReadHeadBits(
        hBuffer, &bPresent, 0, FUEL1_PRESENCE_FLAG_BITLEN);

    if (tBitsRead != FUEL1_PRESENCE_FLAG_BITLEN)
    {
        SMSAPI_DEBUG_vPrint(FUEL1_OBJECT_NAME, 1, 
            "Cannot read buffer, returning \"fake\" presence flag");

        // Indicate presence if we can't read the buffer
        // to force the caller to fail as well
        return TRUE;
    }

    return bPresent;
}

/*****************************************************************************
*
*   eRFDFileProcessor
*
*   The callback invoked by RFD when an update file has been fully recieved
*   and is ready for processing
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        )
{
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl = (FUEL1_RFD_CTRL_STRUCT *)pvCallbackArg;
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;

    if (psRFDCtrl == NULL)
    {
        return RFD_PROCESS_RESULT_ERROR;
    }

    if (eProcessStatus == RFD_PROCESS_STATUS_BEGIN)
    {
        BOOLEAN bOk;

        // Initialize the RFD Ctrl object now
        bOk = bInitRFDCtrl(psRFDCtrl, hConnection, tProgressIndex);

        if (bOk == TRUE)
        {
            // Process the update header now
            bOk = bProcessRFDHeader(psRFDCtrl, psRFDFile);
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL1_OBJECT_NAME": Unable to process RFD file header");

            return RFD_PROCESS_RESULT_ERROR;
        }
    }
    else if (eProcessStatus == RFD_PROCESS_STATUS_STOP)
    {
        // Free associated memory if we're stopping
        vDestroyRFDCtrl(psRFDCtrl);

        return RFD_PROCESS_RESULT_INCOMPLETE;
    }

    // Process the file contents and update the database
    eResult = eProcessNextRFDEntry(
        psRFDCtrl, psRFDFile, tFileVersion);
    if (eResult != RFD_PROCESS_RESULT_INCOMPLETE)
    {
        // Close our connection to the DB
        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
            psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Are we done?
        if (eResult == RFD_PROCESS_RESULT_COMPLETE)
        {
            BOOLEAN bFinalized;

            // Finalize the new database
            bFinalized = bFinalizeDB(psRFDCtrl);
            if (bFinalized == FALSE)
            {
                // We failed to get the DB ready for use
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }

        // Did we experience an error?
        if (eResult  >= RFD_PROCESS_RESULT_ERROR)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL1_OBJECT_NAME": RFD processor returned error (%d)",
                eResult);

            // Delete database
            vDeleteDB(psRFDCtrl);
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   bComputeLatLonFromGrid
*
*   This function is utilized a station update is received in order to
*   generate a lat/lon pair for that station based upon its location in the
*   region/sub-region/division/sub-division grid.
*
*****************************************************************************/
static BOOLEAN bComputeLatLonFromGrid (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    UN8 un8SubRegion,
    UN8 un8Division,
    UN16 un16SubDivision
        )
{
    FUEL1_FIXED_CALC_STRUCT *psFixed =
        &psRFDCtrl->sFixed;
    OSAL_FIXED_OBJECT hLat;
    OSAL_FIXED_OBJECT hLon;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8RegionLatOffset = FUEL1_REG_LAT_OFFSET(psStationRow->tRegion);
    UN8 un8RegionLonOffset = FUEL1_REG_LON_OFFSET(psStationRow->tRegion);
    UN32 un32Lon, un32Lat;
    UN8 un8SubRegionLon = FUEL1_GET_LON(un8SubRegion);
    UN8 un8SubRegionLat = FUEL1_GET_LAT(un8SubRegion);
    UN8 un8DivLon = FUEL1_GET_LON(un8Division);
    UN8 un8DivLat = FUEL1_GET_LAT(un8Division);
    UN8 un8SubDivLon = FUEL1_GET_SUBDIV_LON(un16SubDivision);
    UN8 un8SubDivLat = FUEL1_GET_SUBDIV_LAT(un16SubDivision);

    /*
        RX 185, Section 7.5

        lon = -168 +{((r_lon << 14) | (sr_lon << 10) | (d_lon << 6) | (sd_lon)) / 8192.0};
        lat = 12 + {((r_lat << 14) | (sr_lat << 10) | (d_lat << 6) | (sd_lat )) / 8192.0};
    */

    do
    {
        // Compute the integer math section of the lat & lon computations
        un32Lon = (
           ((un8RegionLonOffset << 14) | (un8SubRegionLon << 10) | (un8DivLon << 6) | (un8SubDivLon))
                );

        un32Lat = (
            (un8RegionLatOffset << 14) | (un8SubRegionLat << 10) | (un8DivLat << 6) | (un8SubDivLat)
                 );

        // Create a fixed for the lat & lon computations, start them out
        // with our computed values
        hLat = OSAL_FIXED.hCreateInMemory(un32Lat, 0,
            &psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * FUEL1_LAT_OBJ_INDEX]);
        hLon = OSAL_FIXED.hCreateInMemory(un32Lon, 0,
            &psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * FUEL1_LON_OBJ_INDEX]);

        if ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            // Error!
            break;
        }

        /*-- Divide those values by the computation denominator --*/
        eReturnCode = OSAL_FIXED.eDivide(hLat, psFixed->hComputeDenominator, hLat);
        eReturnCode |= OSAL_FIXED.eDivide(hLon, psFixed->hComputeDenominator, hLon);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Add that to the initial lat & lon
        eReturnCode |= OSAL_FIXED.eAdd( hLat, psFixed->hInitialLat, hLat );
        eReturnCode |= OSAL_FIXED.eAdd( hLon, psFixed->hInitialLon, hLon );

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // We now have fixed-point values for the lat & lon,
        // but we don't know their bin point.  We do know that
        // we must conform to the fuel manager's bin point however.
        // So, get these correctly scaled value now

        // Latitude first
        psStationRow->n32Lat = OSAL_FIXED.n32ScaledValue(hLat, LOCATION_BINPOINT);

        // Then longitude
        psStationRow->n32Lon = OSAL_FIXED.n32ScaledValue(hLon, LOCATION_BINPOINT);

        return TRUE;
    }
    while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vReleasePriceWorkData
*
*   This function handles all required steps needed to free any extra
*   memory associated with a price work unit
*
*****************************************************************************/
static void vReleasePriceWorkData (
    FUEL1_PRICE_WORK_STRUCT *psWork
        )
{
    // Ensure we destroy any price baselines
    // we allocated
    vDestroyAllocatedPriceBaselines( psWork );

    return;
}

/*****************************************************************************
*
*   bCreateFixedConstants
*
*   This function creates all fixed values for the calculations that
*   this interface must perform to convert grid points to a lat/lon pair
*
*****************************************************************************/
static BOOLEAN bCreateFixedConstants (
    FUEL1_FIXED_CALC_STRUCT *psFixed
        )
{
    do
    {
        // Initial latitude offset
        psFixed->hInitialLat = OSAL_FIXED.hCreateFromFixed(
            FUEL1_INITIAL_LAT, 0);

        if (psFixed->hInitialLat == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Initial longitude offset
        psFixed->hInitialLon = OSAL_FIXED.hCreateFromFixed(
            FUEL1_INITIAL_LON, 0);
        if (psFixed->hInitialLon == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // The lat / lon computation denominator
        psFixed->hComputeDenominator = OSAL_FIXED.hCreateFromFixed(
            FUEL1_COMPUTE_DENOMINATOR, 0);
        if (psFixed->hComputeDenominator == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        return TRUE;
    }
    while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vDestroyFixedConstants
*
*   Destroys all fixed constants used by this interface.
*
*****************************************************************************/
static void vDestroyFixedConstants (
    FUEL1_FIXED_CALC_STRUCT *psFixed
        )
{
    if (psFixed->hInitialLat != OSAL_FIXED_INVALID_OBJECT)
    {
    	OSAL_FIXED.vDestroy(psFixed->hInitialLat);
        psFixed->hInitialLat = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psFixed->hInitialLon != OSAL_FIXED_INVALID_OBJECT)
    {
    	OSAL_FIXED.vDestroy(psFixed->hInitialLon);
        psFixed->hInitialLon = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psFixed->hComputeDenominator != OSAL_FIXED_INVALID_OBJECT)
    {
    	OSAL_FIXED.vDestroy(psFixed->hComputeDenominator);
        psFixed->hComputeDenominator = OSAL_FIXED_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bAddBrandText
*
*   This function adds a brand text entry using the provided parameters.
*
*****************************************************************************/
static BOOLEAN bAddBrandText (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode,
    STRING_OBJECT hString
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL1_BRAND_TEXT_STRUCT *psText;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Create the name for this text entry
    snprintf( &acName[0],
        OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
        FUEL1_OBJECT_NAME":Text(Code:%u)",
        un8BrandCode );

    // Allocate the text row and add it
    psText = (FUEL1_BRAND_TEXT_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(FUEL1_BRAND_TEXT_STRUCT),
            (SMS_OBJECT)psRFDCtrl, FALSE);

    if (psText == NULL)
    {
        return FALSE;
    }

    // Populate the text entry
    psText->un8Code = un8BrandCode;
    psText->hString = hString;

    // Add the new entry to the list
    eReturnCode = OSAL.eLinkedListAdd(
        psRFDCtrl->hBrandTexts, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void *)psText );

    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSO_vDestroy((SMS_OBJECT)psText);

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   hGetBrandString
*
*****************************************************************************/
static STRING_OBJECT hGetBrandString(
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    FUEL1_BRAND_TEXT_STRUCT sSearchParameters;

    // Populate the search parameters
    sSearchParameters.hString = STRING_INVALID_OBJECT;
    sSearchParameters.un8Code = un8BrandCode;

    // Search for the code provided
    eReturnCode = OSAL.eLinkedListSearch(
        psRFDCtrl->hBrandTexts,
        &hEntry,
        &sSearchParameters);
    if (eReturnCode == OSAL_SUCCESS)
    {
        FUEL1_BRAND_TEXT_STRUCT *psText;

        // Extract the list contents
        psText = (FUEL1_BRAND_TEXT_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        if (psText != NULL)
        {
            // Pull out the string handle
            hString = psText->hString;
        }
    }

    return hString;
}

/*****************************************************************************
*
*   n16CompareTextEntries
*
*****************************************************************************/
static N16 n16CompareTextEntries (
    FUEL1_BRAND_TEXT_STRUCT *psText1,
    FUEL1_BRAND_TEXT_STRUCT *psText2
        )
{
    if ((psText1 != NULL) && (psText2 != NULL))
    {
        // Sort by code
        if (psText1->un8Code < psText2->un8Code)
        {
            return -1;
        }

        if (psText1->un8Code > psText2->un8Code)
        {
            return 1;
        }

        return 0;
    }

    return N16_MIN;
}

/*****************************************************************************
*
*   vRemoveTextEntry
*
*****************************************************************************/
static void vRemoveTextEntry (
    FUEL1_BRAND_TEXT_STRUCT *psText
        )
{
    if (psText != NULL)
    {
        // Set the code to the "unbranded" code value
        psText->un8Code = 0;

        // Destroy / Clear the string handle
        STRING_vDestroy(psText->hString);
        psText->hString = STRING_INVALID_OBJECT;

        // Destroy the object
        SMSO_vDestroy((SMS_OBJECT)psText);
    }

    return;
}

/*****************************************************************************
*
*   vInitializeStaticPriceBaselines
*
*   This function initializes the static price baselines used when a price
*   update is first received.  The number of static price baselines is equal
*   to FUEL1_NUM_WELL_KNOWN_FUEL_TYPES.
*
*****************************************************************************/
static void vInitializeStaticPriceBaselines (
    FUEL1_PRICE_WORK_STRUCT *psPriceWork
        )
{
    size_t tIndex = 0;

    do
    {
        // Populate the "next" pointer
        psPriceWork->asPriceBaselines[tIndex].psNext =
            &psPriceWork->asPriceBaselines[tIndex+1];

        // Mark as "static"
        psPriceWork->asPriceBaselines[tIndex].bStatic = TRUE;
    } while (++tIndex < (FUEL1_NUM_WELL_KNOWN_FUEL_TYPES - 1));

    // The last work unit has no "next"
    psPriceWork->asPriceBaselines[tIndex].psNext = NULL;

    // Mark as "static"
    psPriceWork->asPriceBaselines[tIndex].bStatic = TRUE;

    return;
}

/*****************************************************************************
*
*   vInitializePriceMemory
*
*   Initializes the memory used by this object to store price entry info
*   as a price message is being processed
*
*****************************************************************************/
static void vInitializePriceMemory (
    FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT *psPriceMem
        )
{
    if (psPriceMem->bEnabled == FALSE)
    {
        // Initialize the price entries

        // Prices are enabled
        psPriceMem->bEnabled = TRUE;

        // Nothing allocated yet
        psPriceMem->tNumEntriesAllocated = 0;
        psPriceMem->pasEntries = NULL;
    }

    return;
}

/*****************************************************************************
*
*   vUninitPriceMemory
*
*   Cleans up memory associated with price entries for interface shutdown
*
*****************************************************************************/
static void vUninitPriceMemory (
    FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT *psPriceMem
        )
{
    if (psPriceMem->bEnabled == TRUE)
    {
        // Price processing is now disabled
        psPriceMem->bEnabled = FALSE;

        // Clear the counter
        psPriceMem->tNumEntriesAllocated = 0;

        // Free any memory we're using
        if (psPriceMem->pasEntries != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psPriceMem->pasEntries);
            psPriceMem->pasEntries = NULL;
        }
    }

    return;
}

/*****************************************************************************
*
*   bPreparePriceMemory
*
*   Ensures there is enough memory allocated to handle a particular
*   price message
*
*****************************************************************************/
static BOOLEAN bPreparePriceMemory (
    FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT *psPriceMem,
    FUEL1_PRICE_WORK_STRUCT *psWork
    )
{
    if (psPriceMem->tNumEntriesAllocated < psWork->tNumValidBaselines)
    {
        if (psPriceMem->pasEntries != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psPriceMem->pasEntries);
        }

        // Reset the allocated count
        psPriceMem->tNumEntriesAllocated = 0;

        psPriceMem->pasEntries = (FUEL_PRICE_ENTRY_STRUCT *)
            SMSO_hCreate(FUEL1_OBJECT_NAME": PriceMemory",
                psWork->tNumValidBaselines * sizeof(FUEL_PRICE_ENTRY_STRUCT),
                SMS_INVALID_OBJECT, FALSE);
    }

    if (psPriceMem->pasEntries != NULL)
    {
        // We have now allocated memory to meet
        // the required number of entries
        psPriceMem->tNumEntriesAllocated = psWork->tNumValidBaselines;

        // Provide the price row with price entry memory
        psWork->sPriceRow.pasPrices = psPriceMem->pasEntries;

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   psCreateNewPriceBaseline
*
*   When a price update contains pricing information for more than
*   FUEL1_NUM_WELL_KNOWN_FUEL_TYPES, we must allocate new ones.  This function
*   allocates a new price baseline and initializes it.
*
*****************************************************************************/
static FUEL1_PRICE_BASELINE_STRUCT *psCreateNewPriceBaseline (
    FUEL1_OBJECT_STRUCT *psObj
        )
{
    FUEL1_PRICE_BASELINE_STRUCT *psNewBaseline;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    static UN32 un32Instance = 0;

    // Generate a name for this baseline entry
    snprintf(&acName[0], sizeof(acName),
        FUEL1_OBJECT_NAME": Price baseline %u",
        un32Instance++);

    // Create an instance of this object
    psNewBaseline = (FUEL1_PRICE_BASELINE_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(FUEL1_PRICE_BASELINE_STRUCT),
            (SMS_OBJECT)psObj, FALSE);

    if (psNewBaseline != NULL)
    {
        // This is a dynamically allocated entry
        psNewBaseline->bStatic = FALSE;
        psNewBaseline->psNext = NULL;
    }

    return psNewBaseline;
}

/*****************************************************************************
*
*   vDestroyAllocatedPriceBaselines
*
*   When a price update contains pricing information for more than.
*   Destroys all price baselines created via psCreateNewPriceBaseline. This
*   is generally done when we're done processing a price update.
*
*****************************************************************************/
static void vDestroyAllocatedPriceBaselines (
    FUEL1_PRICE_WORK_STRUCT *psPriceWork
        )
{
    FUEL1_PRICE_BASELINE_STRUCT *psBaseline =
        &psPriceWork->asPriceBaselines[0];
    FUEL1_PRICE_BASELINE_STRUCT *psNext = NULL;

    while (psBaseline != NULL)
    {
        // Get the next entry
        psNext = psBaseline->psNext;
        if (psBaseline->bStatic == FALSE)
        {
            SMSO_vDestroy((SMS_OBJECT)psBaseline);
        }
        psBaseline = psNext;
    }

    return;
}
