/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the High-Band specific channel art functions
 *  for the Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <stdlib.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_api_debug.h"
#include "sms_obj.h"
#include "sms.h"

#include "ds_util.h"
#include "dataservice_mgr_impl.h"

#include "_channel_art_highband.h"
#include "channel_art_db_constants.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hInit
*
*   Creates and initializes the "highband" channel art interface object.
*
*   Inputs:
*       hChannelArtService - A handle to the central channel art
*           manager object handle
*       vUpdateDefault - The function pointer this interface must
*           call when a default association has been updated
*
*   Output:
*       An object handle representing this instantiation of the
*       channel art interface (representing high-band support)
*
*****************************************************************************/
static CHANNEL_ART_INTERFACE_OBJECT hInit (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    SMS_OBJECT hParent
        )
{
    CHANNEL_ART_INTERFACE_OBJECT hHighBand =
        CHANNEL_ART_INTERFACE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify we own the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hParent);
    if (bOwner == TRUE)
    {
        do
        {
            CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj;
            OSAL_RETURN_CODE_ENUM eReturnCode;
            BOOLEAN bHaveDeviceGroup;
            DEVICE_GROUP tDeviceGroup;
            UN8 un8Index;

            bHaveDeviceGroup = SMS_bDeviceGroup(&tDeviceGroup);
            if (bHaveDeviceGroup == FALSE)
            {
                // Error!
                break;
            }

            // Create an instance of this object
            psObj = (CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *)
                SMSO_hCreate(
                    CHANNEL_ART_HIGHBAND_OBJECT_NAME,
                    sizeof(CHANNEL_ART_HIGHBAND_OBJECT_STRUCT),
                    hParent, FALSE );

            if (psObj == NULL)
            {
                // Error!
                break;
            }

            // Initialize the CRC
            eReturnCode = OSAL.eGetCRC(&psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                break;
            }

            // Save the service handle
            psObj->hChannelArtService = hChannelArtService;

            // Was this device group provisioned?
            if (tDeviceGroup == DEVICE_GROUP_NOT_PROVISIONED)
            {
                // Nope -- just clear the mask
                psObj->tDeviceMask = DEVICE_MASK_EMPTY;
            }
            else
            {
                // Set the device group
                psObj->tDeviceMask = DEVICE_GROUP_TO_MASK(tDeviceGroup);
            }

            // We are not currently processing
            // an association message or an
            // image message of any type
            psObj->sAssocCtrl.un8CurrentAssocMessageType =
                CHANNEL_ART_HIGHBANG_INVALID_MSG;

            // Intialize flag to indicate we're sending
            // association info to the manager
            psObj->sAssocCtrl.bSendAssocInfoToMgr = TRUE;

            // Initalize assoc tracking
            for (un8Index = 0; un8Index < CHANNEL_ART_HIGHBAND_MAX_INDEX; un8Index++)
            {
                // All assoc types initialize these the same way
                psObj->sAssocCtrl.asAssocTracking[un8Index].bCompleted = FALSE;
                psObj->sAssocCtrl.asAssocTracking[un8Index].n16LastAssocSeqNo =
                    CHANNEL_ART_HIGHBAND_REF_SEQ_NO_INVALID;

                // Start off with this enabled
                psObj->sAssocCtrl.asAssocTracking[un8Index].bReportFirstInstance = TRUE;

                // Is this a "dynamic" reference message of any type?
                if (un8Index >= CHANNEL_ART_HIGHBAND_DYN_CHAN_INDEX)
                {
                    // Yes, disable this flag since the manager doesn't want to know this
                    // for messages of this type
                    psObj->sAssocCtrl.asAssocTracking[un8Index].bReportFirstInstance = FALSE;
                }
            }

            hHighBand = (CHANNEL_ART_INTERFACE_OBJECT)psObj;

        } while (0);
    }

    return hHighBand;
}

/*****************************************************************************
*
*   vUnInit
*
*   Uninitializes and destroys the "highband" channel art interface object.
*   This API must only be called when already in the context of the
*   channel art manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*
*****************************************************************************/
static void vUnInit (
    CHANNEL_ART_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *)hInterface;

        // Release the CRC
        OSAL.eReleaseCRC(psObj->hCRC);

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/***************************************************************************
*
*   bProcessMessage
*
*   This object interface function is used by the art manager in order
*   to indicate that a new message has been received.
*
*   This API must only be called when already in the
*   context of the channel art manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*       *phPayload - A pointer to valid STI payload
*
*   Output:
*       TRUE on success, FALSE on error.
*
***************************************************************************/
static BOOLEAN bProcessMessage (
    CHANNEL_ART_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj =
        (CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *)hInterface;
    CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ENUM eMessageType;
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify inputs
    if (phPayload == NULL)
    {
        return FALSE;
    }

    if (*phPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return FALSE;
    }

    // Verify we own the interface
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == FALSE)
    {
        return FALSE;
    }

    // First, we need to parse the message header
    eMessageType = eParseMessageHeader ( hInterface, phPayload);

    switch(eMessageType)
    {
        case CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_IMAGE:
        {
            // Process an image message
            bSuccess = bProcessImageMessage(psObj, *phPayload );
        }
        break;

        case CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ASSOC:
        {
            // Process the association message
            bSuccess = bProcessAssociationMessage(psObj, *phPayload);
        }
        break;

        case CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_UNKNOWN:
        default:
        {
            // Either this is a new message type or
            // one of the deprecated messages --
            // don't worry about it
            bSuccess = TRUE;
        }
        break;
    }

    return bSuccess;
}

/***************************************************************************
*
*   eParseMessageHeader
*
***************************************************************************/
static CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ENUM eParseMessageHeader (
    CHANNEL_ART_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    OSAL_BUFFER_HDL hPayload = OSAL_INVALID_BUFFER_HDL;
    CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ENUM eMessageType =
        CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_UNKNOWN;
    BOOLEAN bUsable, bOwner;
    UN8 un8MTI = 0;

    // Verify input
    if (*phPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_UNKNOWN;
    }

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == FALSE)
    {
        return CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_UNKNOWN;
    }

    bUsable = bIsPayloadUsable (
        (CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *)hInterface,
        phPayload);

    if (bUsable == FALSE)
    {
        return CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_UNKNOWN;
    }

    // We have a good payload, so dequeue the pointer for our use
    puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Packet Valid");
    hPayload = *phPayload;

    // Extract the Message Type Identifier, but
    // don't consume it. (we'll need it later to
    // differentiate between the message sub-types)
    un8MTI = un8ExtractMTI(TRUE, hPayload);

    if (un8MTI == CHANNEL_ART_HIGHBANG_INVALID_MSG)
    {
        // Read failed -- the message is probably
        // garbled, although that is unlikely since
        // the CRC checked out. Don't do anything rash here.
        puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Unable to read MTI");

        return CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_UNKNOWN;
    }

    printf(CHANNEL_ART_HIGHBAND_OBJECT_NAME": MTI: %u\n", un8MTI);

    // Populate the message type enum based on
    // which message type identifier we've just received
    switch(un8MTI)
    {
        // These two messages provide an image
        // and its meta data
        case CHANNEL_ART_HIGHBAND_MSG_LOGO:
        case CHANNEL_ART_HIGHBAND_MSG_BKGND:
        case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_LOGO:
        {
            eMessageType = CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_IMAGE;
        }
        break;

        // These three messages associate images with
        // channels/categories
        case CHANNEL_ART_HIGHBAND_MSG_STATIC_SVC:
        case CHANNEL_ART_HIGHBAND_MSG_DYNAMIC_SVC:
        case CHANNEL_ART_HIGHBAND_MSG_CAT:
        case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_STATIC_SVC:
        case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_DYNAMIC_SVC:
        {
            eMessageType = CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ASSOC;
        }
        break;

        default:
        {
            // NOTE: We don't support the deprecated MTIs
            // of 3 and 5.  These can be ignored.
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Invalid Carousel Id");
        }
    }

    return eMessageType;
}

/*****************************************************************************
*
*   bProcessImageMessage
*
*   The channel graphics data service protocol (XM-TEC-0-0801-DD, Rev 2.10)
*   specifies that each image message shall be completely encapsulated in one
*   packet. It also specifies that some messages may contain two images
*   (Section 7.4).
*
*   Output: TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bProcessImageMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    CHANNEL_ART_ATTRIB_ROW_STRUCT sAttribRow;
    BOOLEAN bResultAvailable = FALSE,
            bNewMessage = TRUE,
            bSuccess = TRUE;
    UN8 un8MTI;
    size_t tImageDataByteLen = 0;

    // Initialize the structure
    OSAL.bMemSet(&sAttribRow, 0, sizeof(CHANNEL_ART_ATTRIB_ROW_STRUCT));

    // If this is a new message, we need to
    // pull the MTI from the payload
    // Read the MTI to see what kind of image message this is
    // and consume it from the payload
    un8MTI = un8ExtractMTI( FALSE, hPayload );

    do
    {
        switch(un8MTI)
        {
            case CHANNEL_ART_HIGHBAND_MSG_LOGO:
            {
                // Process a logo message
                bResultAvailable =
                    bProcessLogoMessage(
                        psObj, FALSE, bNewMessage, &sAttribRow,
                        &tImageDataByteLen,
                        hPayload);
            }
            break;

            case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_LOGO:
            {
                // Process an overlay logo message
                bResultAvailable =
                    bProcessLogoMessage(
                        psObj, TRUE, bNewMessage, &sAttribRow,
                        &tImageDataByteLen,
                        hPayload);
            }
            break;

            case CHANNEL_ART_HIGHBAND_MSG_BKGND:
            {
                // Process a background message
                bResultAvailable =
                    bProcessBackgroundMessage (
                        psObj, bNewMessage, &sAttribRow,
                        &tImageDataByteLen,
                        hPayload );
            }
            break;

            default:
            {
                puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Invalid Message");
            }
        }

        if (bResultAvailable == TRUE)
        {
            // Tell the manager about what we found
            bSuccess = GsArtMgrIntf.bChanArtImageUpdate(
                psObj->hChannelArtService, &sAttribRow,
                tImageDataByteLen, hPayload);
            if (bSuccess == FALSE)
            {
                // Make sure we stop iterating
                bResultAvailable = FALSE;
            }
        }

        // We are no longer processing a new message
        bNewMessage = FALSE;

    } while (bResultAvailable == TRUE);

    // Tell the caller how that went
    return bSuccess;
}

/*****************************************************************************
*
*   bProcessAssociationMessage
*
*   The channel graphics protocol (XM-TEC-0-0801-DD, Rev 2.10) specifies that
*   any number of associations may be provided in this message.
*
*   This message contains a list of entries which must be processed
*   separately, so this function is used to process the next "segment" of
*   data within the association message.
*
*   Output: TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bProcessAssociationMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eResult;
    CHANNEL_ART_ASSOC_ROW_STRUCT sAssocRow;
    BOOLEAN bSuccess = FALSE;

    // We're sending the first association to the manager
    psObj->sAssocCtrl.bSendAssocInfoToMgr = TRUE;

    // We need to parse the association header now
    eResult = eProcessAssocHeader ( psObj, hPayload );

    if (eResult == CHANNEL_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE)
    {
        do
        {
            // Now process a portion of the association message
            // and use that return value for our result
            eResult = eProcessAssocBody (
                psObj, &sAssocRow, hPayload );
        } while (eResult == CHANNEL_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE);
    }

    // Tell the manager we're done now (even if an error was experienced earlier)
    bSuccess = GsArtMgrIntf.bChanArtAssocEnd(psObj->hChannelArtService);

    // If an error was experienced while processing the message,
    // ensure the caller knows about it
    if (eResult != CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE)
    {
        bSuccess = FALSE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vAllAssociationsCompleted
*
*   The channel graphics protocol (XM-TEC-0-0801-DD, Rev 2.10) specifies that
*   any number of associations may be provided in an association message.
*
*   When the central channel art manager has determined a particular
*   association message has been completed, it calls this function
*   to inform the interface of that fact
*
*   This API must only be called when already in the
*   context of the channel art manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*
*   Output: None
*
*****************************************************************************/
static void vAllAssociationsCompleted (
    CHANNEL_ART_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *)hInterface;
        CHANNEL_ART_HIGHBAND_ASSOC_TRACKING_STRUCT *psTracking =
            (CHANNEL_ART_HIGHBAND_ASSOC_TRACKING_STRUCT *)NULL;

        // Update the appropriate attribute
        switch(psObj->sAssocCtrl.un8CurrentAssocMessageType)
        {
            case CHANNEL_ART_HIGHBAND_MSG_CAT:
            {
                psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_CAT_INDEX];
            }
            break;

            case CHANNEL_ART_HIGHBAND_MSG_STATIC_SVC:
            {
                psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_CHAN_INDEX];
            }
            break;

            case CHANNEL_ART_HIGHBAND_MSG_DYNAMIC_SVC:
            {
                psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_DYN_CHAN_INDEX];
            }
            break;

            case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_DYNAMIC_SVC:
            {
                psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_OVERLAY_DYN_CHAN_INDEX];
            }
            break;

            case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_STATIC_SVC:
            {
                psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_OVERLAY_CHAN_INDEX];
            }
            break;

            default:
            {
                // Nuttin'
                return;
            }
        }

        // Just mark this entry as completed
        psTracking->bCompleted = TRUE;
    }

    return;
}

/*****************************************************************************
*
*   vResetAssociationTracking
*
*   When the central channel art manager want the highband interface to
*   reset message tracking in order to re-open any filter this is called.
*
*   This API must only be called when already in the
*   context of the channel art manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*
*   Output: None
*
*****************************************************************************/
static void vResetAssociationTracking (
    CHANNEL_ART_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *)hInterface;
        UN8 un8Index;

        // Iterate all tracking attributes
        for (un8Index = 0; un8Index < CHANNEL_ART_HIGHBAND_MAX_INDEX; un8Index++)
        {
            // Clear the completed flag on all
            psObj->sAssocCtrl.asAssocTracking[un8Index].bCompleted = FALSE;
        }
    }

    return;
}

/*******************************************************************************
*
*   n8CompareAssociationVersions
*
*   This function compares the versions of two associations (given by a pointer
*   to an CHANNEL_ART_ASSOC_ROW_STRUCT and orders them by the data within.
*   Providing this function as a part of the interface allows the central
*   channel art service to utilize protocol-specific rules to compare
*   association versions.
*
*   This API must only be called when already in the
*   context of the channel art manager.
*
*   Inputs:
*       *psOldAssocRow - A pointer to an association row considered to
*           be "old", as in a pre-existing association
*       *psNewAssocRow - A pointer to an association row considered to
*           be "new", as in a newly received association (likely from
*           a message that recently came in)
*
*   Outputs:
*       TRUE - Versions are equal
*       FALSE - Versions are not equal
*
*******************************************************************************/
static BOOLEAN bCompareAssociationVersions (
    CHANNEL_ART_ASSOC_ROW_STRUCT *psOldAssocRow,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psNewAssocRow,
    BOOLEAN *pbOldNeedsConfirmation
        )
{
    if ((NULL == psOldAssocRow) ||
        (NULL == psNewAssocRow) ||
        (NULL == pbOldNeedsConfirmation))
    {
        // Just say the versions are different.  That should
        // make the caller pay!
        return FALSE;
    }

    // Initialize this to the common case
    *pbOldNeedsConfirmation = FALSE;

    // Has this association been confirmed yet?
    if (psOldAssocRow->bAssocVerConfirmed == FALSE)
    {
        // No, tell the caller they gotta check it out
        *pbOldNeedsConfirmation = TRUE;
    }

    // Do both associations have the same version?
    if (psOldAssocRow->un8AssocVer == psNewAssocRow->un8AssocVer)
    {
        // We need this if the assoc version is the same,
        // but we're replacing a low-priority assoc with
        // a high priority assoc
        if ((psOldAssocRow->bHighPriority == FALSE) &&
            (psNewAssocRow->bHighPriority == TRUE))
        {
            return FALSE;
        }

        // This is a match
        return TRUE;
    }

    // Now we're at the point at which we know that
    // the association version numbers are different.
    // We need to just say the info from broadcast is newer
    return FALSE;
}

/*******************************************************************************
*
*   bCompareImageVersions
*
*   This function compares the versions of two images (given by a pointer
*   to an CHANNEL_ART_ATTRIB_ROW_STRUCT and orders them by the data within.
*   Providing this function as a part of the interface allows the central
*   channel art service to utilize protocol-specific rules to compare
*   association versions.  Highband images actually have two version
*   fields: sequence number and revision number.  Both of these numbers
*   have been packed within the version field in the attribute row.
*
*   This API must only be called when already in the
*   context of the channel art manager.
*
*   Inputs:
*       *psAttribRow1 - A pointer to an attribute row considered to
*           be "old", as in pre-existing image data
*       *psAttribRow2 - A pointer to an attribute row considered to
*           be "new", as in newly received attributes (likely from
*           a message that recently came in)
*       *pbImageCompatible - A pointer to a BOOLEAN which indicates if
*           the two versions indicated in the previous two arguments
*           represent "compatible" images. Here in the highband world
*           that means two images have the same sequence number and
*           we don't care about the revision numbers. This helps when
*           we are comparing the image sequence number reported in association
*           messages with the seq/rev pair we have in image metadata.
*
*   Outputs:
*       TRUE - Versions are equal
*       FALSE - Versions are not equal
*
*******************************************************************************/
static BOOLEAN bCompareImageVersions (
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow1,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow2,
    BOOLEAN *pbImageCompatible
        )
{
    BOOLEAN bRevPresent, bImageCompatible;
    UN8 un8SeqNo1, un8SeqNo2;

    if ((psAttribRow1 == NULL) ||
        (psAttribRow2 == NULL))
    {
        // Just say the versions are different.  That should
        // make the caller pay!
        return FALSE;
    }

    // Use local variable if necessary
    if (pbImageCompatible == NULL)
    {
        pbImageCompatible = &bImageCompatible;
    }

    // We haven't detected that the image has been updated yet
    *pbImageCompatible = FALSE;

    // Grab the sequence numbers
    un8SeqNo1 = CHANNEL_ART_HIGHBAND_VER_TO_SEQ(psAttribRow1->un16ImageVer);
    un8SeqNo2 = CHANNEL_ART_HIGHBAND_VER_TO_SEQ(psAttribRow2->un16ImageVer);

    // Compare sequence numbers first -- just
    // check if they're different
    if (un8SeqNo1 != un8SeqNo2)
    {
        // Sequence number changes mean that these two are different
        return FALSE;
    }

    // All image updates from this point on are "compatible"
    *pbImageCompatible = TRUE;

    // Do these attribute rows both contain
    // revision information?
    bRevPresent = CHANNEL_ART_HIGHBAND_VER_HAS_REV(psAttribRow1->un16ImageVer);
    bRevPresent &= CHANNEL_ART_HIGHBAND_VER_HAS_REV(psAttribRow2->un16ImageVer);

    if (bRevPresent == TRUE)
    {
        UN8 un8RevNo1, un8RevNo2;

        // Grab the revision numbers
        un8RevNo1 = CHANNEL_ART_HIGHBAND_VER_TO_REV(psAttribRow1->un16ImageVer);
        un8RevNo2 = CHANNEL_ART_HIGHBAND_VER_TO_REV(psAttribRow2->un16ImageVer);

        // Both attribute rows contain revision
        // numbers -- compare them

        // Just detect a difference
        if (un8RevNo1 != un8RevNo2)
        {
            // The image has been updated
            return FALSE;
        }
    }

    // These images are the same
    return TRUE;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   un8ExtractMTI
*
*   This function pulls the message type identifier out of a message by
*   either consuming the appropriate byte or just peeking at it.
*
*******************************************************************************/
static UN8 un8ExtractMTI (
    BOOLEAN bPeek,
    OSAL_BUFFER_HDL hPayload
        )
{
    UN8 un8MTI = 0;
    size_t tBitsRead;


    if (hPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return CHANNEL_ART_HIGHBANG_INVALID_MSG;
    }

    if (bPeek == TRUE)
    {
        // Peek at the Message Type Identifier
        // (don't remove it from the buffer)
        tBitsRead = OSAL.tBufferPeekBits(
            hPayload, &un8MTI, 0,
            CHANNEL_ART_HIGHBAND_MTI_BITLEN, 0);
    }
    else
    {
        // Read the Message Type Identifier
        // (consume it from the buffer)
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &un8MTI, 0,
                CHANNEL_ART_HIGHBAND_MTI_BITLEN );
    }

    // We were unable to read the MTI
    if (tBitsRead != CHANNEL_ART_HIGHBAND_MTI_BITLEN)
    {
        return CHANNEL_ART_HIGHBANG_INVALID_MSG;
    }

    // If this message is using an alternate id, convert
    // it to the "regular" id for use in the switch below
    if ( (un8MTI >= CHANNEL_ART_HIGHBAND_MSG_ALT_MODIFIER) &&
         (un8MTI < CHANNEL_ART_HIGHBAND_MSG_OVERLAY_STATIC_SVC) )
    {
        un8MTI -= CHANNEL_ART_HIGHBAND_MSG_ALT_MODIFIER;
    }

    return un8MTI;
}

/*******************************************************************************
*
*   eProcessServiceAssoc
*
*   This function handles the processing of service (channel) association
*   messages, whether they be the static or dynamic type.  The return value
*   of this message indicates to the caller if the current entry has been fully
*   processed.  If all portions of an association entry have been processed,
*   then this will indicate CHANNEL_ART_PROCESS_RESULT_COMPLETE.  If there
*   are pending portions of the current entyr, then this function will indicate
*   CHANNEL_ART_PROCESS_RESULT_INCOMPLETE.
*
*   CHANNEL_ART_PROCESS_RESULT_ERROR is not used here, because message
*   processing errors aren't considered to be very serious.  If an error
*   occurs, we'll say just we're done.
*
*******************************************************************************/
static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessServiceAssoc (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN bIsOverlay,
    BOOLEAN *pbRowPopulated,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tTotalBitsRead;
    UN8 un8Source = 0, un8ImageId = 0;

    // Are we in the middle of processing an entry's image
    // list or are we starting to process a new entry?
    if (psObj->sAssocCtrl.bProcessingImageList == FALSE)
    {
        // Are we in the middle of processing an entry's priority list?
        if (psObj->sAssocCtrl.bProcessingAssocPriorityList == FALSE)
        {
            // No, Read the logo attributes and source id

            // Since this is a new entry we're going to be
            // sending this assoc to the manager whether they like it or not
            psObj->sAssocCtrl.bSendAssocInfoToMgr = TRUE;

            // This is the high-priority assoc for this id
            psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority = TRUE;

            // Read the source Id
            tTotalBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &un8Source, 0,
                CHANNEL_ART_HIGHBAND_SVC_ID_BITLEN);
            psObj->sAssocCtrl.sCurrentAssocRow.un16Source = (UN16)un8Source;

            if (tTotalBitsRead != CHANNEL_ART_HIGHBAND_SVC_ID_BITLEN)
            {
                // Read failed -- the message is probably
                // garbled, although that is unlikely since
                // the CRC checked out. Don't do anything rash here.
                return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
            }

            if (bIsOverlay == TRUE)
            {
                // Overlay assoc source IDs are offset by 256
                psObj->sAssocCtrl.sCurrentAssocRow.un16Source +=
                    CHANNEL_ART_HIGHBAND_OVERLAY_SOURCE_ID_OFFSET;
            }

            // Read the Channel Logo Reference Id
            tTotalBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &un8ImageId, 0,
                CHANNEL_ART_HIGHBAND_LOGO_ID_BITLEN);
            psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId = (UN16)un8ImageId;

            if (bIsOverlay == TRUE)
            {
                if (  (psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId
                             >= CHANNEL_ART_HIGHBAND_OVERLAY_REF_ID_OFFSET_RANGE_START) &&
                      (psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId
                            <= CHANNEL_ART_HIGHBAND_OVERLAY_REF_ID_OFFSET_RANGE_STOP)  )
                {
                    // Overlay assoc reference IDs are offset by 256 if they
                    // are between 0 and 127.
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId +=
                        CHANNEL_ART_HIGHBAND_OVERLAY_REF_ID_OFFSET;
                }

                printf("Overlay Assoc found for %u, %u\n",
                    psObj->sAssocCtrl.sCurrentAssocRow.un16Source,
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId);
            }

            // Read the association ordering flag
            psObj->sAssocCtrl.bTryOTAImageFirst = FALSE;
            tTotalBitsRead += OSAL.tBufferReadHeadBits(
                hPayload, &psObj->sAssocCtrl.bTryOTAImageFirst, 0,
                CHANNEL_ART_HIGHBAND_REF_DB_SEARCH_ORDER_BITLEN);

            // If we are to try the OTA image first,
            // then we need to generate an OTA Image Id
            if (psObj->sAssocCtrl.bTryOTAImageFirst == TRUE)
            {
                psAssocRow->un16ImageId =
                    CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(
                        psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId);
            }
            else // Just use the static image Id
            {
                psAssocRow->un16ImageId =
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId;
            }

            // Read the Logo Image sequence number
            psObj->sAssocCtrl.un8CurrentImageSeqNo = 0;
            tTotalBitsRead += OSAL.tBufferReadHeadBits(
                    hPayload, &psObj->sAssocCtrl.un8CurrentImageSeqNo, 0,
                    CHANNEL_ART_HIGHBAND_LOGO_SEQ_BITLEN);

            if (tTotalBitsRead != CHANNEL_ART_HIGHBBAND_SVC_LIST_ENTRY_BITLEN)
            {
                // Read failed -- the message is probably
                // garbled, although that is unlikely since
                // the CRC checked out. Don't do anything rash here.
                return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
            }

            // This is the primary large logo
            psObj->sAssocCtrl.sCurrentAssocRow.un8ImageType =
                CHANNEL_ART_IMAGETYPE_LOGO;

            if (psObj->sAssocCtrl.un8CurrentImageSeqNo == CHANNEL_ART_HIGHBAND_UNREFERENCED_FLAG)
            {
                // We have been told to remove this
                // association if possible
                psObj->sAssocCtrl.sCurrentAssocRow.bRemove = TRUE;
            }
            else
            {
                psObj->sAssocCtrl.sCurrentAssocRow.bRemove = FALSE;
            }

            // Were we told to remove this association?
            if (psObj->sAssocCtrl.sCurrentAssocRow.bRemove == FALSE)
            {
                // No, we weren't -- this means that we'll enter
                // the "processing assoc priority list" mode because
                // we need to try both associations present in this
                // message for this channel entry.  If this is a
                // remove, it doesn't matter which order the
                // associations are in so we don't need
                // to process the entry's list
                psObj->sAssocCtrl.bProcessingAssocPriorityList = TRUE;
            }
            else
            {
                // Move on to the processing image list state
                psObj->sAssocCtrl.bProcessingImageList = TRUE;
            }
        }
        else // We are in the middle of processing the
             // assoc priority list
        {

            // This is the low-priority assoc for this id
            psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority = FALSE;

            // Were we told to try the OTA image
            // second?
            if (psObj->sAssocCtrl.bTryOTAImageFirst == FALSE)
            {
                // We tried the static image id first,
                // so now we'll try the OTA image Id
                psAssocRow->un16ImageId =
                    CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(
                        psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId);
            }
            else // Just use the static image Id
            {
                psAssocRow->un16ImageId =
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId;
            }

            // We are no longer processing an assoc priority
            // list for an assoc entry
            psObj->sAssocCtrl.bProcessingAssocPriorityList = FALSE;

            // We are now processing the image list
            psObj->sAssocCtrl.bProcessingImageList = TRUE;
        }
    }
    else // Processing an assoc's image list
    {
        // Are we in the middle of processing an entry's priority list?
        if (psObj->sAssocCtrl.bProcessingAssocPriorityList == FALSE)
        {
            // No, Read the background attributes

            // This is the high-priority assoc for this id
            psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority = TRUE;

            // Since this is a new entry we're going to be
            // sending this assoc to the manager whether they like it or not
            psObj->sAssocCtrl.bSendAssocInfoToMgr = TRUE;

            // Read the Channel Background Reference Id
            tTotalBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &un8ImageId, 0,
                CHANNEL_ART_HIGHBAND_BKGRND_ID_BITLEN);
            psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId = (UN16)un8ImageId;

            // Read the association ordering flag
            psObj->sAssocCtrl.bTryOTAImageFirst = FALSE;
            tTotalBitsRead += OSAL.tBufferReadHeadBits(
                hPayload, &psObj->sAssocCtrl.bTryOTAImageFirst, 0,
                CHANNEL_ART_HIGHBAND_REF_DB_SEARCH_ORDER_BITLEN);

            // If we are to try the OTA image first,
            // then we need to generate an OTA Image Id
            if (psObj->sAssocCtrl.bTryOTAImageFirst == TRUE)
            {
                psAssocRow->un16ImageId =
                    CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(
                        psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId);
            }
            else // Just use the static image Id
            {
                psAssocRow->un16ImageId =
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId;
            }

            // Read the Background Image sequence number
            psObj->sAssocCtrl.un8CurrentImageSeqNo = 0;
            tTotalBitsRead += OSAL.tBufferReadHeadBits(
                    hPayload, &psObj->sAssocCtrl.un8CurrentImageSeqNo, 0,
                    CHANNEL_ART_HIGHBAND_BKGRND_SEQ_BITLEN);

            if (tTotalBitsRead != CHANNEL_ART_HIGHBBAND_SVC_LIST_ENTRY_BITLEN)
            {
                // Read failed -- the message is probably
                // garbled, although that is unlikely since
                // the CRC checked out. Don't do anything rash here.
                return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
            }

            // Now the reference entry variable data contains nothing
            // that we understand -- so seek past it
            tTotalBitsRead = OSAL.tBufferSeekHead(
                hPayload, psObj->sAssocCtrl.un8RefEntryVarDataByteSize);

            if (tTotalBitsRead != psObj->sAssocCtrl.un8RefEntryVarDataByteSize)
            {
                // Read failed -- the message is probably
                // garbled, although that is unlikely since
                // the CRC checked out. Don't do anything rash here.
                return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
            }

            // This is the background image
            psObj->sAssocCtrl.sCurrentAssocRow.un8ImageType =
                (UN8)CHANNEL_ART_IMAGETYPE_BKGRND;

            if (psObj->sAssocCtrl.un8CurrentImageSeqNo == CHANNEL_ART_HIGHBAND_UNREFERENCED_FLAG)
            {
                // We have been told to remove this
                // association if possible
                psObj->sAssocCtrl.sCurrentAssocRow.bRemove = TRUE;
            }
            else
            {
                psObj->sAssocCtrl.sCurrentAssocRow.bRemove = FALSE;
            }

            // Were we told to remove this association?
            if (psObj->sAssocCtrl.sCurrentAssocRow.bRemove == FALSE)
            {
                // No, we weren't -- this means that we'll enter
                // the "processing assoc priority list" mode because
                // we need to try both associations present in this
                // message for this category entry.  If this is a
                // remove, it doesn't matter which order the
                // associations are in so we don't need
                // to process the entry's list
                psObj->sAssocCtrl.bProcessingAssocPriorityList = TRUE;
            }
            else
            {
                // Move out of the processing image list state
                psObj->sAssocCtrl.bProcessingImageList = FALSE;
            }
        }
        else // We are in the middle of processing the
             // assoc priority list
        {
            // This is the low-priority assoc for this id
            psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority = FALSE;

            // Were we told to try the OTA image
            // second?
            if (psObj->sAssocCtrl.bTryOTAImageFirst == FALSE)
            {
                // We tried the static image id first,
                // so now we'll try the OTA image Id
                psAssocRow->un16ImageId =
                    CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(
                        psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId);
            }
            else // Just use the static image Id
            {
                psAssocRow->un16ImageId =
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId;
            }

            // We are no longer processing an assoc priority
            // list for an assoc entry
            psObj->sAssocCtrl.bProcessingAssocPriorityList = FALSE;

            // We are no longer processing the image list
            psObj->sAssocCtrl.bProcessingImageList = FALSE;
        }
    }

    // Copy the results to the caller
    psAssocRow->bCategory = FALSE;
    psAssocRow->bDefault = FALSE;
    psAssocRow->bRemove = psObj->sAssocCtrl.sCurrentAssocRow.bRemove;
    psAssocRow->bContent = psObj->sAssocCtrl.sCurrentAssocRow.bContent;
    psAssocRow->un16Source = psObj->sAssocCtrl.sCurrentAssocRow.un16Source;
    psAssocRow->un8AssocVer = psObj->sAssocCtrl.sCurrentAssocRow.un8AssocVer;
    psAssocRow->bAssocVerConfirmed = TRUE;
    psAssocRow->un8ImageType = psObj->sAssocCtrl.sCurrentAssocRow.un8ImageType;
    psAssocRow->bHighPriority = psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority;

    // Don't fill in the content flag -- that is done
    // in eProcessAssocBody (one step up in the call stack)

    // Generate the image version number based upon the sequence number
    psAssocRow->un16ImageVer = CHANNEL_ART_HIGHBAND_SEQ_TO_VER(psObj->sAssocCtrl.un8CurrentImageSeqNo);

    // The association row now
    // contains useful data
    *pbRowPopulated = TRUE;

    // Now tell the caller if we're done with this entry --
    // We know we're done with this entry when we are no
    // longer processing this entry's image list or
    // assoc priority list
    if (   (psObj->sAssocCtrl.bProcessingAssocPriorityList == FALSE)
        && (psObj->sAssocCtrl.bProcessingImageList == FALSE))
    {
        // We are all done with this entry
        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    // We have more for this entry
    return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE;
}

/*******************************************************************************
*
*   eProcessCategoryAssoc
*
*   This function handles the processing of category association messages
*   The return value of this message indicates to the caller if the current
*   entry has been fully processed.  If all portions of an association entry
*   have been processed, then this will indicate
*   CHANNEL_ART_PROCESS_RESULT_COMPLETE.  If there are pending portions of
*   the current entyr, then this function will indicate
*   CHANNEL_ART_PROCESS_RESULT_INCOMPLETE.
*
*   CHANNEL_ART_PROCESS_RESULT_ERROR is not used here, because message
*   processing errors aren't considered to be very serious.  If an error
*   occurs, we'll say just we're done.
*
*******************************************************************************/
static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessCategoryAssoc(
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN *pbRowPopulated,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload
        )
{
    // Are we processing a new association, or are
    // we in the middle of processing an association
    // entry's priority list?
    if (psObj->sAssocCtrl.bProcessingAssocPriorityList == FALSE)
    {
        size_t tTotalBitsRead;
        UN8 un8ImageSeqNo = 0,
            un8Source = 0,
            un8ImageId = 0;

        // Starting a new entry -- parse
        // the message

        // This is the high-priority assoc for this category
        psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority = TRUE;

        // Since this is a new entry we're going to be
        // sending this assoc to the manager whether they like it or not
        psObj->sAssocCtrl.bSendAssocInfoToMgr = TRUE;

        // Read the source id field
        tTotalBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8Source, 0, CHANNEL_ART_HIGHBAND_CAT_ID_BITLEN);
        psObj->sAssocCtrl.sCurrentAssocRow.un16Source = (UN16)un8Source;

        // Read the image id field
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &un8ImageId, 0, CHANNEL_ART_HIGHBAND_BKGRND_ID_BITLEN);
        psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId = (UN16)un8ImageId;

        // Read the association ordering flag
        psObj->sAssocCtrl.bTryOTAImageFirst = FALSE;
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &psObj->sAssocCtrl.bTryOTAImageFirst, 0,
            CHANNEL_ART_HIGHBAND_REF_DB_SEARCH_ORDER_BITLEN);

        // If we are to try the OTA image first,
        // then we need to generate an OTA Image Id
        if (psObj->sAssocCtrl.bTryOTAImageFirst == TRUE)
        {
            psAssocRow->un16ImageId =
                CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId);
        }
        else // Just use the static image Id
        {
            psAssocRow->un16ImageId =
                psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId;
        }

        // Read the Background Image sequence number
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &un8ImageSeqNo, 0,
            CHANNEL_ART_HIGHBAND_BKGRND_SEQ_BITLEN);

        // Generate the image version number based upon
        // the sequence number
        psObj->sAssocCtrl.sCurrentAssocRow.un16ImageVer =
            CHANNEL_ART_HIGHBAND_SEQ_TO_VER(un8ImageSeqNo);

        if (un8ImageSeqNo == CHANNEL_ART_HIGHBAND_UNREFERENCED_FLAG)
        {
            // We have been told to remove this
            // association if possible
            psObj->sAssocCtrl.sCurrentAssocRow.bRemove = TRUE;
        }
        else
        {
            psObj->sAssocCtrl.sCurrentAssocRow.bRemove = FALSE;
        }

        // Were we told to remove this association?
        if (psObj->sAssocCtrl.sCurrentAssocRow.bRemove == FALSE)
        {
            // No, we weren't -- this means that we'll enter
            // the "processing assoc priority list" mode because
            // we need to try both associations present in this
            // message for this category entry.  If this is a
            // remove, it doesn't matter which order the
            // associations are in so we don't need
            // to process the entry's list
            psObj->sAssocCtrl.bProcessingAssocPriorityList = TRUE;
        }

        // Make sure all the reads add up
        if (tTotalBitsRead != CHANNEL_ART_HIGHBAND_CAT_LIST_ENTRY_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
        }

        // the reference entry variable data contains nothing
        // that we understand -- so seek past it
        tTotalBitsRead = OSAL.tBufferSeekHead(
            hPayload, psObj->sAssocCtrl.un8RefEntryVarDataByteSize);

        if (tTotalBitsRead != psObj->sAssocCtrl.un8RefEntryVarDataByteSize)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
        }
    }
    else // We are in the middle of processing the
         // assoc priority list
    {
        // This is the low-priority assoc for this category
        psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority = FALSE;

        // Were we told to try the OTA image
        // second?
        if (psObj->sAssocCtrl.bTryOTAImageFirst == FALSE)
        {
            // We tried the static image id first,
            // so now we'll try the OTA image Id
            psAssocRow->un16ImageId =
                CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(
                    psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId);
        }
        else // Just use the static image Id
        {
            psAssocRow->un16ImageId =
                psObj->sAssocCtrl.sCurrentAssocRow.un16ImageId;
        }

        psObj->sAssocCtrl.bProcessingAssocPriorityList = FALSE;
    }

    // This is always a static, non-default category association
    psAssocRow->bDefault = FALSE;
    psAssocRow->bCategory = TRUE;
    psAssocRow->bContent = FALSE;

    // The image type is always background
    psAssocRow->un8ImageType = (UN8)CHANNEL_ART_IMAGETYPE_BKGRND;

    // The association version was determined
    // previously and stored
    psAssocRow->un8AssocVer = psObj->sAssocCtrl.sCurrentAssocRow.un8AssocVer;
    psAssocRow->bAssocVerConfirmed = TRUE;
    psAssocRow->bHighPriority = psObj->sAssocCtrl.sCurrentAssocRow.bHighPriority;

    // Copy the result data to the caller's pointer
    psAssocRow->un16Source = psObj->sAssocCtrl.sCurrentAssocRow.un16Source;
    psAssocRow->un16ImageVer = psObj->sAssocCtrl.sCurrentAssocRow.un16ImageVer;
    psAssocRow->bRemove = psObj->sAssocCtrl.sCurrentAssocRow.bRemove;

    // The caller may utilize the data in the
    // association row now
    *pbRowPopulated = TRUE;

    // Now tell the caller if we're done with this entry --
    // We know we're done with this entry when we are no
    // longer processing this entry's assoc priority list
    if (psObj->sAssocCtrl.bProcessingAssocPriorityList == FALSE)
    {
        // We are all done with this entry
        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    // We have more for this entry
    return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE;
}

/*******************************************************************************
*
*   eProcessAssocHeader
*
*   The function processes the header of an association message.  The header
*   may specify a channel and/or a category default image.  In addition, an
*   association version is specified in the header which needs to be applied
*   to all associations present in the body as well.
*
*******************************************************************************/
static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessAssocHeader (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tTotalBitsRead,
           tMessageLen,
           tComputedMessageLen;
    UN8 un8HdrVarDataByteSize = 0,
        un8NumListEntries = 0,
        un8ListEntryByteLen;
    UN8 un8MTI, un8CurrentSeqNo = 0;
    BOOLEAN bCategory = FALSE,
            bDynamic = FALSE,
            bOverlay = FALSE,
            bFirstInstance = FALSE;
    CHANNEL_ART_MGR_PROCESS_ASSOC_ENUM eProcess;
    CHANNEL_ART_HIGHBAND_ASSOC_TRACKING_STRUCT *psTracking;

    // We are not currently processing an
    // image list for an association now
    // that we are starting a new message
    psObj->sAssocCtrl.bProcessingImageList = FALSE;
    psObj->sAssocCtrl.bProcessingAssocPriorityList = FALSE;

    /***************************************
     * Determine if this is a new sequence *
     * number for the message type         *
     ***************************************/

    // Read the MTI to see what kind of message this is
    // and consume it from the payload
    un8MTI = un8ExtractMTI( FALSE, hPayload );

    switch(un8MTI)
    {
        case CHANNEL_ART_HIGHBAND_MSG_CAT:
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Category Association Message:");

            psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_CAT_INDEX];

            un8ListEntryByteLen =
                CHANNEL_ART_HIGHBAND_LIST_ENTRY_BYTE_SIZE_CAT;

            // Set the current association as not
            // for content
            psObj->sAssocCtrl.sCurrentAssocRow.bContent = FALSE;

            // This is a category association message
            bCategory = TRUE;
        }
        break;

        case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_STATIC_SVC:
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Channel Association Message (static, overlay):");

            psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_OVERLAY_CHAN_INDEX];

            un8ListEntryByteLen =
                CHANNEL_ART_HIGHBAND_LIST_ENTRY_BYTE_SIZE_SVC;

            // Set the current association as not
            // for content
            psObj->sAssocCtrl.sCurrentAssocRow.bContent = FALSE;

            // This is an overlay association message
            bOverlay = TRUE;
        }
        break;

        case CHANNEL_ART_HIGHBAND_MSG_STATIC_SVC:
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Channel Association Message (static):");

            psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_CHAN_INDEX];

            un8ListEntryByteLen =
                CHANNEL_ART_HIGHBAND_LIST_ENTRY_BYTE_SIZE_SVC;

            // Set the current association as not
            // for content
            psObj->sAssocCtrl.sCurrentAssocRow.bContent = FALSE;
        }
        break;

        case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_DYNAMIC_SVC:
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Channel Association Message (dynamic, overlay):");

            psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_OVERLAY_DYN_CHAN_INDEX];

            un8ListEntryByteLen =
                CHANNEL_ART_HIGHBAND_LIST_ENTRY_BYTE_SIZE_SVC;

            // Indicate the current association is
            // content related
            psObj->sAssocCtrl.sCurrentAssocRow.bContent = TRUE;

            // This is a dynamic channel association message
            bDynamic = TRUE;

            // This is an overlay association message
            bOverlay = TRUE;
        }
        break;

        case CHANNEL_ART_HIGHBAND_MSG_DYNAMIC_SVC:
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Channel Association Message (dynamic):");

            psTracking = &psObj->sAssocCtrl.asAssocTracking[CHANNEL_ART_HIGHBAND_DYN_CHAN_INDEX];

            un8ListEntryByteLen =
                CHANNEL_ART_HIGHBAND_LIST_ENTRY_BYTE_SIZE_SVC;

            // Indicate the current association is
            // content related
            psObj->sAssocCtrl.sCurrentAssocRow.bContent = TRUE;

            // This is a dynamic channel association message
            bDynamic = TRUE;
        }
        break;

        default:
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Invalid Message");

            // The message is probably garbled,
            // although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
        }
    }

    // Keep track of what our current association
    // message type is
    psObj->sAssocCtrl.un8CurrentAssocMessageType = un8MTI;

    // Read the Association version field
    tTotalBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &un8CurrentSeqNo, 0,
        CHANNEL_ART_HIGHBAND_REF_SEQ_BITLEN);

    if (tTotalBitsRead != CHANNEL_ART_HIGHBAND_REF_SEQ_BITLEN)
    {
        // Read failed -- the message is probably
        // garbled, although that is unlikely since
        // the CRC checked out. Don't do anything rash here.
        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    printf(CHANNEL_ART_HIGHBAND_OBJECT_NAME
            ":\tAssoc SeqNo: %u (Last SeqNo: %d)\n",
            un8CurrentSeqNo, psTracking->n16LastAssocSeqNo);

    // Has this association message been completely
    // handled already?
    if ((un8CurrentSeqNo == psTracking->n16LastAssocSeqNo) &&
        (psTracking->bCompleted == TRUE))
    {
        puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Skipping association message");

        // This message has already been completed
        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    /**********************************
     * Process the rest of the header *
     **********************************/

    // Read the number of entries in the association
    // list -- we'll validate message length with it
    tTotalBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &un8NumListEntries, 0,
        CHANNEL_ART_HIGHBAND_REF_LIST_SIZE_BITLEN);

    // Read the byte size of the entries' variable
    // data field
    psObj->sAssocCtrl.un8RefEntryVarDataByteSize = 0;
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
        hPayload, &psObj->sAssocCtrl.un8RefEntryVarDataByteSize, 0,
        CHANNEL_ART_HIGHBAND_REF_ENTRY_VAR_DATA_SIZE_BITLEN);

    // Read the byte size of the message header's
    // variable data field
    un8HdrVarDataByteSize = 0;
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
        hPayload, &un8HdrVarDataByteSize, 0,
        CHANNEL_ART_HIGHBAND_REF_HDR_VAR_DATA_SIZE_BITLEN);

    if (tTotalBitsRead != CHANNEL_ART_HIGHBAND_REF_STATIC_HEADER_BITLEN)
    {
        // Read failed -- the message is probably
        // garbled, although that is unlikely since
        // the CRC checked out. Don't do anything rash here.
        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    // Do we have the device mask present?
    if (un8HdrVarDataByteSize >=
        CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_DEVICE_MASK_BYTELEN)
    {
        BOOLEAN bProcessMessage;
        size_t tBytesConsumed = 0;

        // Check to see if this device mask matches
        // our current device type(s)
        bProcessMessage = bDeviceMaskMatches(
            psObj, hPayload, &tBytesConsumed );

        if (bProcessMessage == FALSE)
        {
            // We don't need to process this
            // message -- stop here
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME
                 ": Skipping processing: device mask does not match");

            return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
        }

        // The variable data fields have been at
        // least partially consumed.  Calculate how
        // much is left
        un8HdrVarDataByteSize -= (UN8)tBytesConsumed;
    }

    // Seek past whatever is left in the variable
    // data field since we don't understand it
    tTotalBitsRead = OSAL.tBufferSeekHeadBits(
        hPayload,
        (size_t)un8HdrVarDataByteSize * 8 // Number of remaining extension bits
            );

    if (tTotalBitsRead != (size_t)(un8HdrVarDataByteSize * 8))
    {
        // Read failed -- the message is probably
        // garbled, although that is unlikely since
        // the CRC checked out. Don't do anything rash here.
        puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME
             ": Skipping processing: message corrupt?");

        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    // Verify message length is equal to the number
    // of entries in the list * (list entry size +
    // variable data byte size)
    tMessageLen = OSAL.tBufferGetSize( hPayload );

    // Compute the message length
    tComputedMessageLen =
        un8NumListEntries *
            (un8ListEntryByteLen
                + psObj->sAssocCtrl.un8RefEntryVarDataByteSize);

    if (tMessageLen != tComputedMessageLen)
    {
        // Something is wrong with the message length -
        // Don't do anything rash here.
        puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME
             ": Skipping processing: message length calculation failed");

        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    // Keep the current version in our
    // current assoc row structure as well
    psObj->sAssocCtrl.sCurrentAssocRow.un8AssocVer = un8CurrentSeqNo;

    // Indicate this is the first instance of a message if:
    // 1) We're supposed to do that for this message type
    // 2) It is the first instance
    if ((psTracking->bReportFirstInstance == TRUE) &&
        (psTracking->n16LastAssocSeqNo != (N16)un8CurrentSeqNo))
    {
        bFirstInstance = TRUE;
    }

    // Update the last seq no and
    // ensure the completed flag is unset
    psTracking->n16LastAssocSeqNo = (N16)un8CurrentSeqNo;
    psTracking->bCompleted = FALSE;

    // Tell the manager we're about to start a new association list
    // The last argument here is actually called bVerifyLineup by
    // the interface header.  We do want the manager to verify all
    // channels / categories in order to ensure they're all
    // using good images, but only if this is the first instance
    // of this association message.
    eProcess = GsArtMgrIntf.eChanArtAssocBegin(
        psObj->hChannelArtService,
        (bCategory == TRUE)?CHANNEL_ART_IMAGETYPE_BKGRND:CHANNEL_ART_IMAGETYPE_LOGO,
        bFirstInstance, bCategory, bOverlay);
    if (eProcess == CHANNEL_ART_MGR_PROCESS_ASSOC_IGNORE)
    {
        // We don't need to process this
        // message -- stop here
        puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME
             ": Skipping processing: manager doesn't want this type");

        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }
    // CHANNEL_ART_MGR_PROCESS_ASSOC_ERROR or other
    else if (eProcess != CHANNEL_ART_MGR_PROCESS_ASSOC_BEGIN)
    {
        // The manager had a problem
        puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME
             ": Failed to process association");

        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ERROR;
    }

    // Now, if this association message contains no entries, it means
    // we are supposed to reset all associations of the same type
    if (un8NumListEntries == CHANNEL_ART_HIGHBAND_RESET_ASSOCS_FLAG)
    {
        // Tell the mgr to reset its associations which
        // match this type
        GsArtMgrIntf.bChanArtResetAssociations(
            psObj->hChannelArtService,
            bDynamic, bOverlay, bCategory );

        // Nothing else to do here
        puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME
             ": Association reset");

        return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    }

    // We're not done with this message yet -- we still
    // need to process the message body
    puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME
         ": Processing incomplete. Continuing...");

    return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE;
}

/*******************************************************************************
*
*   eProcessAssocBody
*
*   The channel graphics protocol (XM-TEC-0-0801-DD, Rev 2.10) specifies that
*   any number of associations may be provided in this message.
*
*   This function delegates association message processing to the
*   appropriate processors.  Those processor functions will indicate if
*   the association entry they are currently working on have been completed.
*
*******************************************************************************/
static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessAssocBody (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload
        )
{
    CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eResult =
        CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
    size_t tBitsExpected;
    BOOLEAN bRowPopulated = FALSE,
            bSuccess = FALSE;

    switch (psObj->sAssocCtrl.un8CurrentAssocMessageType)
    {
        case CHANNEL_ART_HIGHBAND_MSG_CAT:
        {
            // Process the category association message
            eResult = eProcessCategoryAssoc(
                psObj, &bRowPopulated, psAssocRow, hPayload);

            // We need at least this many bytes for another
            // association entry
            tBitsExpected = CHANNEL_ART_HIGHBAND_CAT_LIST_ENTRY_BITLEN;
        }
        break;

        case CHANNEL_ART_HIGHBAND_MSG_STATIC_SVC:
        case CHANNEL_ART_HIGHBAND_MSG_DYNAMIC_SVC:
        {
            // Process the channel association message
            eResult = eProcessServiceAssoc(
                psObj, FALSE, &bRowPopulated, psAssocRow, hPayload);

            // We need at least this many bytes for another
            // association entry
            tBitsExpected = CHANNEL_ART_HIGHBBAND_SVC_LIST_ENTRY_BITLEN;
        }
        break;

        case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_STATIC_SVC:
        case CHANNEL_ART_HIGHBAND_MSG_OVERLAY_DYNAMIC_SVC:
        {
            // Process the channel association message
            eResult = eProcessServiceAssoc(
                psObj, TRUE, &bRowPopulated, psAssocRow, hPayload);

            // We need at least this many bytes for another
            // association entry
            tBitsExpected = CHANNEL_ART_HIGHBBAND_SVC_LIST_ENTRY_BITLEN;
        }
        break;

        default:
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME": Invalid Message");

            // The message is probably garbled,
            // although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE;
        }
    }

    // Provide the new association data to the mgr if we didn't
    // experience an error
    if ((eResult != CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ERROR) &&
        (bRowPopulated == TRUE))
    {
        // Does the manager want to know about this?
        if (psObj->sAssocCtrl.bSendAssocInfoToMgr == TRUE)
        {
            // Yes, tell the manager about the new association
            bSuccess = GsArtMgrIntf.bChanArtAssociationUpdate(
                psObj->hChannelArtService, psAssocRow,
                &psObj->sAssocCtrl.bSendAssocInfoToMgr);
        }
        else
        {
            // No, but we're gonna tell them about the next
            // one for sure
            psObj->sAssocCtrl.bSendAssocInfoToMgr = TRUE;
            bSuccess = TRUE;
        }

        if (bSuccess == FALSE)
        {
            // This shouldn't happen
            return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ERROR;
        }
    }

    // The message processors tell us if they are done
    // with a particular message entry via their return
    // value.  If they are done with a message entry, see
    // if there is room for another one.  If not, we're done
    // with the message.  If the message processor indicates
    // they're not done with the message, then just pass that
    // indication along. Ditto with error
    if (eResult == CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE)
    {
        size_t tBitsRemaining;

        // Determine if we are done with the entire message
        // or if there is more in the association list to process
        tBitsRemaining = OSAL.tBufferGetSizeInBits(hPayload);
        if (tBitsRemaining >= tBitsExpected)
        {
            // There are more entries to process
            return CHANNEL_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE;
        }
    }

    return eResult;
}

/*******************************************************************************
*
*   bProcessLogoMessage
*
*   The channel graphics protocol (XM-TEC-0-0801-DD, Rev 2.10) specifies that
*   a logo message may contain up to two images (a primary logo and a
*   secondary logo).
*
*   This function processes a logo message in order to provide the caller
*   with the logo's attributes & image data.
*
*******************************************************************************/
static BOOLEAN bProcessLogoMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN bIsOverlay,
    BOOLEAN bNewMessage,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t *ptImageDataByteLen,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bRead;
    UN16 un16LogoLen = 0;
    IMAGE_BACKGROUND_GRAPHICS_STRUCT *psBackground =
        &psAttribRow->sBackgroundGfx;

    if (bNewMessage == TRUE)
    {
        size_t tTotalBitsRead;
        UN8 un8ImageId = 0,
            un8SeqNo = 0,
            un8RevNo = 0,
            un8LineBitmapIndex = 0,
            un8Red = 0,
            un8Green = 0,
            un8Blue = 0,
            un8VariableDataLen = 0;
        BOOLEAN bMessageMatch = TRUE;

        // If we are processing a new message
        // then pull the info from the message

        // Read the Image Message Header

        // Read the Image Id
        tTotalBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8ImageId, 0, CHANNEL_ART_HIGHBAND_LOGO_ID_BITLEN );
        psAttribRow->un16ImageId = (UN16)un8ImageId;

        if (bIsOverlay == TRUE)
        {
            // Overlay Logo Image IDs are offset by 256
            psAttribRow->un16ImageId += CHANNEL_ART_HIGHBAND_OVERLAY_IMAGE_ID_OFFSET;
        }

        if (psAttribRow->un16ImageId > CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
			    CHANNEL_ART_HIGHBAND_OBJECT_NAME
			    ": Out-of-range Image Reference ID found: %u", psAttribRow->un16ImageId);
            return FALSE;
        }

        // Generate a new Id based on what we received (this service
        // never updates the image ids which are present in the
        // initial database.  So, we'll generate an Id now which
        // never conflicts with anything in the initial database)
        psAttribRow->un16ImageId =
            CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(
                psAttribRow->un16ImageId);

        // Seek past the reserved field
        tTotalBitsRead += OSAL.tBufferSeekHeadBits(
            hPayload, CHANNEL_ART_HIGHBAND_LOGO_RSVD1_BITLEN);

        // Read the Image Sequence number
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &un8SeqNo, 0,
            CHANNEL_ART_HIGHBAND_LOGO_SEQ_BITLEN);

        // Read the Image Revision field
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &un8RevNo, 0,
            CHANNEL_ART_HIGHBAND_LOGO_REV_BITLEN);

        // The image version is an amalgamation of the
        // Image Sequence and the Image Revision
        psAttribRow->un16ImageVer =
            CHANNEL_ART_HIGHBAND_SEQ_REV_TO_VER(un8SeqNo, un8RevNo);

        // Read the Background Color Index field
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &un8LineBitmapIndex, 0,
            CHANNEL_ART_HIGHBAND_LOGO_BKGRND_COLOR_INDEX_BITLEN);

        // Red
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload,
            &un8Red, 0,
            CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN);

        // Green
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload,
            &un8Green, 0,
            CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN);

        // Blue
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload,
            &un8Blue, 0,
            CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN);

        // Construct the background color
        psBackground->un32BackgroundColor =
            CHANNEL_ART_GET_BACKGROUND_COLOR(un8Red, un8Green, un8Blue );

        // Seek past the reserved field
        tTotalBitsRead += OSAL.tBufferSeekHeadBits(
            hPayload, CHANNEL_ART_HIGHBAND_LOGO_RSVD2_BITLEN);

        // Read the Secondary Image available flag
        psObj->sLogoCtrl.bMoreLogosAvailable = FALSE;
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &psObj->sLogoCtrl.bMoreLogosAvailable, 0,
            CHANNEL_ART_HIGHBAND_LOGO_SECOND_IMG_FLAG_BITLEN);

        // This value is also used here in the image attribute row
        psAttribRow->bSecondaryAvailable = psObj->sLogoCtrl.bMoreLogosAvailable;

        // Seek past the reserved field
        tTotalBitsRead += OSAL.tBufferSeekHeadBits(
            hPayload, CHANNEL_ART_HIGHBAND_LOGO_RSVD3_BITLEN);

        // Read the variable data length field
        tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &un8VariableDataLen, 0,
            CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_BITLEN);

        // Verify we have been able to read static
        // portion of the logo message header
        if (tTotalBitsRead != CHANNEL_ART_HIGHBAND_LOGO_STATIC_HEADER_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out.
            return FALSE;
        }

        // Now, process the variable data field(s)

        // Do we have the device mask present?
        if (un8VariableDataLen >=
            CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_DEVICE_MASK_BYTELEN)
        {
            size_t tBytesConsumed = 0;

            // Check to see if this device mask matches
            // our current device type(s)
            bMessageMatch = bDeviceMaskMatches(
                psObj, hPayload, &tBytesConsumed );

            // The variable data fields have been at
            // least partially consumed.  Calculate how
            // much is left
            un8VariableDataLen -= (UN8)tBytesConsumed;
        }

        // Is this message directed at us?
        if (bMessageMatch == FALSE)
        {
            // No -- this message is not
            // meant for us
            return FALSE;
        }

        // Seek past whatever is left in the variable
        // data field since we don't understand it
        tTotalBitsRead = OSAL.tBufferSeekHeadBits(
                hPayload,
                (size_t)un8VariableDataLen * 8 // Number of remaining extension bits
                    );

        if (tTotalBitsRead != (size_t)(un8VariableDataLen * 8))
        {
            // Seek failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out.
            return FALSE;
        }

        // Read the Primary Logo length field
        bRead = OSAL.bBufferReadBitsToUN16(
            hPayload, &un16LogoLen,
            CHANNEL_ART_HIGHBAND_LOGO_DATA_LEN_BITLEN);

        if (bRead == FALSE)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out.
            return FALSE;
        }

        psObj->sLogoCtrl.tPrimaryLogoLen = (size_t)un16LogoLen;

        // Process the background color index
        vProcessBackgroundOptions (
            psObj, un8LineBitmapIndex, psBackground);

        // The Code Type field is always PNG
        // for any image in this message
        psAttribRow->un8CodeType = (UN8)IMAGE_FORMAT_PNG;

        // The Image Type field is always the primary
        // large logo for the first image
        psAttribRow->un8ImageType =
            (UN8)CHANNEL_ART_IMAGETYPE_LOGO;
    }
    else if (psObj->sLogoCtrl.bMoreLogosAvailable == TRUE)
    {
        size_t tSeekBytes;

        // At most only one more logo is available, so
        // there is nothing after this
        psObj->sLogoCtrl.bMoreLogosAvailable = FALSE;

        // No Background color junk for this image type
        psBackground->tBackgroundOptions = CHANNEL_ART_BACKGROUND_OPTION_NONE;
        psBackground->sLineBitmap.pun8LineBitmap = NULL;
        psBackground->sLineBitmap.tLineBitmapByteSize = 0;
        psBackground->sLineBitmap.tLineBitmapIndex = 0;

        // Seek past the primary logo
        tSeekBytes = OSAL.tBufferSeekHead(hPayload, psObj->sLogoCtrl.tPrimaryLogoLen);
        if (tSeekBytes != psObj->sLogoCtrl.tPrimaryLogoLen)
        {
            // That was weird
            return FALSE;
        }

        // Read the Primary Logo length field
        bRead = OSAL.bBufferReadBitsToUN16(
            hPayload, &un16LogoLen,
            CHANNEL_ART_HIGHBAND_LOGO_DATA_LEN_BITLEN);
        if (bRead == FALSE)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out.
            return FALSE;
        }

        // The Image Type field is always the secondary
        // large logo for the second image
        psAttribRow->un8ImageType =
            (UN8)CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO;

        // The secondary logo does not have a secondary image
        psAttribRow->bSecondaryAvailable = FALSE;
    }
    else
    {
        // We have nothing more to provide
        // to the caller
        return FALSE;
    }

    // Now, provide the image attributes to the caller
    *ptImageDataByteLen = un16LogoLen;

    return TRUE;
}

/*******************************************************************************
*
*   bProcessBackgroundMessage
*
*   The channel graphics protocol (XM-TEC-0-0801-DD, Rev 2.10) specifies that
*   a background image message only contains one image.
*
*   This function processes a background message in order to provide the caller
*   with the background's attributes & image data.
*
*******************************************************************************/
static BOOLEAN bProcessBackgroundMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN bNewMessage,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t *ptImageDataByteLen,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bRead, bResultAvailable = TRUE;
    size_t tTotalBitsRead;
    UN8 un8ImageId = 0,
        un8SeqNo = 0,
        un8RevNo = 0,
        un8Red = 0,
        un8Green = 0,
        un8Blue = 0,
        un8VariableDataLen = 0;
    UN16 un16ImageLen = 0;
    IMAGE_BACKGROUND_GRAPHICS_STRUCT *psBackground =
        &psAttribRow->sBackgroundGfx;

    // We only have one image in this message, so
    // if this is a repeated call on the same
    // message, we have nothing for the caller
    if (bNewMessage == FALSE)
    {
        return FALSE;
    }

    // Backgrounds don't have line bitmaps,
    // so clear that out
    psBackground->sLineBitmap.pun8LineBitmap = NULL;
    psBackground->sLineBitmap.tLineBitmapByteSize = 0;
    psBackground->sLineBitmap.tLineBitmapIndex = 0;

    // Read the Image Message Header

    // Read the Image Id
    tTotalBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &un8ImageId, 0, CHANNEL_ART_HIGHBAND_BKGRND_ID_BITLEN);
    psAttribRow->un16ImageId = (UN16)un8ImageId;

    // Generate a new Id based on what we received (this service
    // never updates the image ids which are present in the
    // initial database.  So, we'll generate an Id now which
    // never conflicts with anything in the initial database)
    psAttribRow->un16ImageId =
        CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(psAttribRow->un16ImageId);

    // Seek past the reserved field
    tTotalBitsRead += OSAL.tBufferSeekHeadBits(
        hPayload, CHANNEL_ART_HIGHBAND_BKGRND_RSVD1_BITLEN);

    // Read the Image message sequence number
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
            hPayload, &un8SeqNo, 0,
            CHANNEL_ART_HIGHBAND_BKGRND_SEQ_BITLEN);

    // Read the Image Version field
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
        hPayload, &un8RevNo, 0,
        CHANNEL_ART_HIGHBAND_BKGRND_REV_BITLEN);

    // The image version is an amalgamation of the
    // Image Sequence and the Image Revision
    psAttribRow->un16ImageVer =
        CHANNEL_ART_HIGHBAND_SEQ_REV_TO_VER(un8SeqNo, un8RevNo);

    // Read the Background Color fields

    // Red
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
        hPayload,
        &un8Red, 0,
        CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN);

    // Green
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
        hPayload,
        &un8Green, 0,
        CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN);

    // Blue
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
        hPayload,
        &un8Blue, 0,
        CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN);

    // Construct the background color
    psBackground->un32BackgroundColor =
        CHANNEL_ART_GET_BACKGROUND_COLOR(un8Red, un8Green, un8Blue );

    // Read the variable data length field
    tTotalBitsRead += OSAL.tBufferReadHeadBits(
        hPayload, &un8VariableDataLen, 0,
        CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_BITLEN);

    // Verify we have been able to read the
    // background image message header
    if (tTotalBitsRead != CHANNEL_ART_HIGHBAND_BKGRND_STATIC_HEADER_BITLEN)
    {
        return FALSE;
    }

    // Do we have the device mask present?
    if (un8VariableDataLen >=
        CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_DEVICE_MASK_BYTELEN)
    {
        size_t tBytesConsumed = 0;

        // Check to see if this device mask matches
        // our current device type(s)
        bResultAvailable = bDeviceMaskMatches(
            psObj, hPayload, &tBytesConsumed );

        // The variable data fields have been at
        // least partially consumed.  Calculate how
        // much is left
        un8VariableDataLen -= (UN8)tBytesConsumed;
    }

    // Is this message directed at us?
    if (bResultAvailable == FALSE)
    {
        // No -- this message is not
        // meant for us
        return FALSE;
    }

    // Seek past whatever is left in the variable
    // data field since we don't understand it
    tTotalBitsRead = OSAL.tBufferSeekHeadBits(
            hPayload,
            (size_t)un8VariableDataLen * 8 // Number of remaining extension bits
                );

    if (tTotalBitsRead != ((size_t)(un8VariableDataLen * 8)))
    {
        // Read failed -- the message is probably
        // garbled, although that is unlikely since
        // the CRC checked out.
        return FALSE;
    }

    // Read the Background Image length field
    bRead = OSAL.bBufferReadBitsToUN16(
        hPayload, &un16ImageLen,
        CHANNEL_ART_HIGHBAND_BKGRND_DATA_LEN_BITLEN);

    if (bRead == FALSE)
    {
        // Read failed -- the message is probably
        // garbled, although that is unlikely since
        // the CRC checked out.
        return FALSE;
    }

    // The Code Type field is always JPEG
    psAttribRow->un8CodeType = (UN8)IMAGE_FORMAT_JPEG;

    // The Image Type field is always the background image
    psAttribRow->un8ImageType =
        (UN8)CHANNEL_ART_IMAGETYPE_BKGRND;

    // We always have background color with these messages
    psAttribRow->sBackgroundGfx.tBackgroundOptions =
        CHANNEL_ART_BACKGROUND_OPTION_BACKGROUND_COLOR;

    // We never have secondary images for backgrounds
    psAttribRow->bSecondaryAvailable = FALSE;

    *ptImageDataByteLen = un16ImageLen;

    return bResultAvailable;
}

/*****************************************************************************
*
*   vProcessBackgroundOptions
*
*   The channel graphics protocol (XM-TEC-0-0801-DD, Rev 2.10, section
*   3.2.5.2.1) specifies the background for a logo image may be a solid color,
*   a "line bitmap" (resized to the App's needs), or a color merged with the
*   "line bitmap" which produces a snazzy effect.
*
*   If the line bitmap index provided in the message is invalid (using the
*   specified, constant, invalid index value), it means there is no "line
*   bitmap" and the background color is to be displayed as is (solid). This
*   is the "solid" flag.
*
*   In addition, there is a "tint adjust" flag.  If this flag is set, it
*   means the line bitmap (given via line bitmap, which better be valid now)
*   should be used in the display.
*
*   If the "solid" flag is set, it means we can ignore the value in the "tint
*   adjust" flag.
*
*   If the "solid" flag is not set, then:
*      If the "tint adjust" flag is set, then the background color is to be
*      used along with the line bitmap in order to generate the previously
*      mentioned snazzy effect.
*      If the "tint adjust" flag is not set, then the background color is
*      not to be used at all.  The display should just use the line bitmap.
*
*   Whew.
*
*****************************************************************************/
static void vProcessBackgroundOptions (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    UN8 un8BackgroundColorIndex,
    IMAGE_BACKGROUND_GRAPHICS_STRUCT *psBackground
        )
{
    UN8 un8LineBitmapIndex =
        un8BackgroundColorIndex & CHANNEL_ART_HIGHBAND_LINE_BITMAP_INDEX_MASK;

    // If the line bitmap index is invalid, it means the "solid"
    // flag has been set and we can ignore the "tint adjust" flag
    if (un8LineBitmapIndex == CHANNEL_ART_HIGHBAND_LINE_BITMAP_INVALID)
    {
        // We should utilize the background color only
        psBackground->tBackgroundOptions =
            CHANNEL_ART_BACKGROUND_OPTION_BACKGROUND_COLOR;
    }
    else
    {
        // We know we'll be using the line bitmap now
        psBackground->tBackgroundOptions =
            CHANNEL_ART_BACKGROUND_OPTION_LINE_BITMAP;

        // Save the line bitmap index
        psBackground->sLineBitmap.tLineBitmapIndex = un8LineBitmapIndex;

        // Check the "tint adjust" flag
        if ((un8BackgroundColorIndex
               & CHANNEL_ART_HIGHBAND_TINT_ADJUST_FLAG_MASK) != 0
           )
        {
            // The flag is set -- use both the color as well
            psBackground->tBackgroundOptions |=
                CHANNEL_ART_BACKGROUND_OPTION_BACKGROUND_COLOR;
        }
    }

    return;
}

/*****************************************************************************
*
*   bDeviceMaskMatches
*
*   The channel graphics protocol (XM-TEC-0-0801-DD, Rev 2.10, section 6.6)
*   specifies a device mask which is utilized to direct messages to certain
*   types of hardware.
*
*****************************************************************************/
static BOOLEAN bDeviceMaskMatches (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    size_t *ptBytesConsumed
        )
{
    size_t tBitsRead;
    UN8 un8DeviceMaskId = 0;
    UN32 un32DeviceMask;
    BOOLEAN bRead;

    if (ptBytesConsumed == NULL)
    {
        return FALSE;
    }

    // Initialize this now
    *ptBytesConsumed = 0;

    // Peek at the Device Mask ID field -
    // we don't want to consume this data yet
    tBitsRead = OSAL.tBufferPeekBits(
        hPayload, &un8DeviceMaskId, 0,
        CHANNEL_ART_HIGHBAND_DEVICE_MASK_ID_BITLEN, 0);

    if (tBitsRead != CHANNEL_ART_HIGHBAND_DEVICE_MASK_ID_BITLEN)
    {
        // There was a problem reading this portion of the
        // message -- that's not good, so just tell the
        // caller that this message is not for them
        return FALSE;
    }

    // If this field is actually the device mask Id, it will
    // have the expected value.
    if (un8DeviceMaskId != CHANNEL_ART_HIGHBAND_DEVICE_MASK_ID)
    {
        // This isn't the device mask, so we don't
        // filter on this field
        return TRUE;
    }

    // We've verified that this is a device mask field,
    // so we can consume these bits now
    tBitsRead = OSAL.tBufferSeekHeadBits(
        hPayload, CHANNEL_ART_HIGHBAND_DEVICE_MASK_ID_BITLEN);

    if (tBitsRead != CHANNEL_ART_HIGHBAND_DEVICE_MASK_ID_BITLEN)
    {
        // There was a problem reading this portion of the
        // message -- that's not good, so just tell the
        // caller that this message is not for them
        return FALSE;
    }

    // Track how many bytes we consumed
    *ptBytesConsumed = (tBitsRead / 8);

    // Read the device mask
    bRead = OSAL.bBufferReadBitsToUN32(
        hPayload, &un32DeviceMask,
        CHANNEL_ART_HIGHBAND_DEVICE_MASK_VALUE_BITLEN);
    if (bRead == FALSE)
    {
        // This is a device mask, but we can't read it
        // that means we can't let anybody know about this
        return FALSE;
    }

    // Track how many bits we consumed
    *ptBytesConsumed +=
        (CHANNEL_ART_HIGHBAND_DEVICE_MASK_VALUE_BITLEN / 8);

    // Now, is this message appropriate for
    // our provisioned device group?
    if ((un32DeviceMask & psObj->tDeviceMask) != 0)
    {
        // This is for us
        return TRUE;
    }

    // We don't want this
    return FALSE;
}

/*****************************************************************************
*
*   bIsPayloadUsable
*
******************************************************************************/
static BOOLEAN bIsPayloadUsable (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bSuccess;
    size_t tBitsRead;
    PVN tPVN = (PVN)0;
    UN8 un8CarouselID = 0, un8RFU = 0;

    do
    {
        // Validate the payload
        bSuccess = DS_UTIL_bIsCRCValid(psObj->hCRC, *phPayload, NULL);
        if (bSuccess == FALSE)
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME" SDTP AU Payload Invalid");
            break;
        }

        // Cut off CRC
        bSuccess = DS_UTIL_bCutCRC(*phPayload);
        if (bSuccess == FALSE)
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME" failed to cut off CRC");
            break;
        }

        // Read the PVN
        tBitsRead = OSAL.tBufferReadHeadBits(
            *phPayload, &tPVN, 0,
            CHANNNEL_ART_HIGHBAND_PVN_BITLEN);

        if (tBitsRead != CHANNNEL_ART_HIGHBAND_PVN_BITLEN)
        {
            // Read failed -- the message is probably garbled
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME" Unable to read PVN\n");

            break;
        }

        // Verify the PVN
        if (tPVN != CHANNNEL_ART_HIGHBAND_PVN)
        {
            printf(CHANNEL_ART_HIGHBAND_OBJECT_NAME" Incorrect PVN - got %u, expected %u\n",
                tPVN, CHANNNEL_ART_HIGHBAND_PVN);

            break;
        }

        // Read the Carousel Id, do nothing with it
        tBitsRead = OSAL.tBufferReadHeadBits(
            *phPayload, &un8CarouselID, 0,
            CHANNNEL_ART_HIGHBAND_CARID_BITLEN);

        if (tBitsRead != CHANNNEL_ART_HIGHBAND_CARID_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME" Unable to read Carousel Id\n");

            break;
        }

        // Make sure that the carousel ID is what we are looking for.
        if (un8CarouselID != CHANNNEL_ART_HIGHBAND_CARID)
        {
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME" Unknown carousel ID found, ignoring\n");

            break;
        }

        // Read the RFU, do nothing with it
        tBitsRead = OSAL.tBufferReadHeadBits(
            *phPayload, &un8RFU, 0,
            CHANNNEL_ART_HIGHBAND_RFU_BITLEN);

        if (tBitsRead != CHANNNEL_ART_HIGHBAND_RFU_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled
            puts(CHANNEL_ART_HIGHBAND_OBJECT_NAME" Unable to read RFU\n");

            break;
        }

        return TRUE;

    } while (FALSE);

    // Error processing payload, so free it
    // Free the new payload
    DATASERVICE_IMPL_bFreeDataPayload(*phPayload);

    //invalidate the callers handle
    *phPayload = OSAL_INVALID_BUFFER_HDL;

    return FALSE;
}
