/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains an implementation of an album art decoder
 *  that matches CAI protocol version (PVN) 1.0, herein referred to as
 *  "album art." See document SX-9845-0231 for the protocol specification.
 *
 ******************************************************************************/
#include <stdlib.h>

#include "standard.h"
#include "osal.h"
#include "baudot.h"
#include "dataservice_mgr_impl.h"
#include "sms_api.h"
#include "sms_api_debug.h"
#include "sms_obj.h"
#include "sms.h"
#include "ds_util.h"

#include "_album_art_pvn1.h"

static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   hInit
 *
 *   Creates and initializes the "highband" album art interface object.
 *
 *   Inputs:
 *       hChannelArtService - A handle to the central channel art
 *           manager object (album art is a product within the
 *           channel art service, so we use their manager.)
 *       hParent - The SMS object that owns this instance of the
 *          album art interface
 *
 *   Output:
 *       An object handle representing this instantiation of the
 *       protocol version 1 album art interface
 *
 *****************************************************************************/
static ALBUM_ART_INTERFACE_OBJECT hInit (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    SMS_OBJECT hParent
        )
{
    // Set the album art interface object to NULL for now. If
    // we have any errors, we'll just return this invalid object
    // in place of a real album art interface to signify something
    // went wrong.
    ALBUM_ART_INTERFACE_OBJECT hInterface = ALBUM_ART_INTERFACE_INVALID_OBJECT;

    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj;
    BOOLEAN bOwner, bHaveDeviceGroup;
    DEVICE_GROUP tDeviceGroup;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify we own the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT) hParent);
    if( FALSE == bOwner )
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                gpacThisFile,
                __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": Failed ownership check!"
               );

        return hInterface;
    }

    // Make sure we have a device group as well ...
    bHaveDeviceGroup = SMS_bDeviceGroup(&tDeviceGroup);
    if (bHaveDeviceGroup == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                gpacThisFile,
                __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": We don't have a device group!"
               );

        return hInterface;
    }

    // Now that we've got our pre-reqs out of the way, try and create the
    // album art object.
    psObj = (ALBUM_ART_HIGHBAND_OBJECT_STRUCT *) SMSO_hCreate(
                ALBUM_ART_HIGHBAND_OBJECT_NAME,
                sizeof(ALBUM_ART_HIGHBAND_OBJECT_STRUCT), hParent, FALSE
            );
    if (psObj == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                gpacThisFile,
                __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": SMS Album Art Object Creation Failed!"
               );

        return hInterface;
    }

    // Initialize the CRC
    eReturnCode = OSAL.eGetCRC( &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32 );
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj);

        SMSAPI_DEBUG_vPrintErrorFull(
                gpacThisFile,
                __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": Failed calling OSAL.eGetCRC!"
               );

       return hInterface;
    }

    // Save the service handle
    psObj->hChannelArtService = hChannelArtService;

    hInterface = (ALBUM_ART_INTERFACE_OBJECT)psObj;

    // Initialize image tracking for both high and low carousels
    vResetImageTracking( &psObj->sLowImageCtrl );
    vResetImageTracking( &psObj->sHighImageCtrl );

    return hInterface;
}

/*****************************************************************************
 *
 *   vUnInit
 *
 *   Un-initializes and destroys the album art interface object.
 *   This API must only be called when already in the context of the
 *   channel art manager.
 *
 *   Inputs:
 *       hInterface - The interface object handle created with a
 *           previous call to hInit.
 *
 *****************************************************************************/
static void vUnInit (
    ALBUM_ART_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner before we destroy
    // anything.
    bOwner = SMSO_bOwner((SMS_OBJECT) hInterface);
    if ( TRUE == bOwner )
    {

        ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj =
                (ALBUM_ART_HIGHBAND_OBJECT_STRUCT *) hInterface;

        // Release the CRC
        OSAL.eReleaseCRC(psObj->hCRC);

        // Destroy any in-progress logo buffer data (high-prio)
        if ( psObj->sHighImageCtrl.hImageData != OSAL_INVALID_BUFFER_HDL )
        {
            OSAL.eBufferFree(psObj->sHighImageCtrl.hImageData);
            psObj->sHighImageCtrl.hImageData = OSAL_INVALID_BUFFER_HDL;
        }

        // Destroy any in-progress logo buffer data (low-prio)
        if ( psObj->sLowImageCtrl.hImageData != OSAL_INVALID_BUFFER_HDL )
        {
            OSAL.eBufferFree(psObj->sLowImageCtrl.hImageData);
            psObj->sLowImageCtrl.hImageData = OSAL_INVALID_BUFFER_HDL;
        }

        SMSO_vDestroy((SMS_OBJECT) psObj);
    }

    return;
}

/***************************************************************************
 *
 *   bProcessMessage
 *
 *   This object interface function is used by the art manager in order
 *   to indicate that a new message has been received.
 *
 *   This API must only be called when already in the
 *   context of the channel art manager.
 *
 *   Inputs:
 *       hInterface - The album art interface object handle created
 *           with a previous call to hInit.
 *       *phPayload - A pointer to valid STI payload
 *
 *   Output:
 *       TRUE on success, FALSE on error.
 *
 ***************************************************************************/
static BOOLEAN bProcessMessage (
    ALBUM_ART_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj =
            (ALBUM_ART_HIGHBAND_OBJECT_STRUCT *) hInterface;
    BOOLEAN bOwner = FALSE, bSuccess = FALSE;
    size_t tBitsRead = (size_t)0;

    // We don't know what the carosuel is yet
    ALBUM_ART_HIGHBAND_CARID_ENUM eCarosuelId
        = ALBUM_ART_HIGHBAND_CARID_UNKNOWN;

    // Verify that our pointer is valid ...
    if (phPayload == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                    gpacThisFile,
                    __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": Invalid payload!"
               );

        return FALSE;
    }

    // Verify that the handle pointed to is valid as well.
    if (*phPayload == OSAL_INVALID_BUFFER_HDL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                    gpacThisFile,
                    __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": Invalid payload handle!"
               );

        return FALSE;
    }

    // Verify we own the interface
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                    gpacThisFile,
                    __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": Failed ownership check!"
               );

        return FALSE;
    }

    // Album Art (unlike Channel Art) doesn't use Message IDs. Image files
    // come in on carousel IDs 0 & 1, and the default image associations
    // come in on carousel ID 2. Grab our carousel ID (which will validate
    // the message as well; any invalid payloads will return with
    // carousel ID "unknown".)
    eCarosuelId = eParseMessageHeader ( hInterface, phPayload, &tBitsRead );

    switch(eCarosuelId)
    {
        // The "priority" of these images apply to the carousel timing
        // only; thus, we process both high and low priority
        // images the  same way, although they each track their image
        // updates using a separate image control struct. We also have
        // to tell the image message the number of bits read during
        // header processing, so it knows how many stuff bits are
        // in the message
        case ALBUM_ART_HIGHBAND_CARID_IMAGE_HIGH_PRIO:
        {
            bSuccess = bProcessImageMessage( psObj, phPayload,
                    tBitsRead, &(psObj->sHighImageCtrl)
                );
        }
        break;
        case ALBUM_ART_HIGHBAND_CARID_IMAGE_LOW_PRIO:
        {
            bSuccess = bProcessImageMessage( psObj, phPayload,
                    tBitsRead, &(psObj->sLowImageCtrl)
                );
        }
        break;

        case ALBUM_ART_HIGHBAND_CARID_DEFAULT_ASSOC:
        {
            // Process the association message
            bSuccess = bProcessAssociationMessage( psObj, phPayload );
        }
        break;

        default:
        {
            // Could be a new message type (or carousel) we don't
            // support yet. Just return "success."
            bSuccess = TRUE;
        }
        break;
    }

    // If something didn't work, let the app know
    if ( FALSE == bSuccess )
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                gpacThisFile,
                __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": Album art decoder failed to parse an incoming message!"
            );
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bProcessImageMessage
 *
 *   The album art / CAI service protocol (SX 9845-0231) specifies the format
 *   and layout of incoming image messages in section 5. Both high- and low-
 *   priority image files are decoded the same way, and are handled by
 *   this function.
 *
 *   Completed associations and image data are sent to the channel art manager.
 *   Images typically span multiple AUs. Image data from successive AUs is
 *   appended to the end of the first AU's buffer.
 *
 *   This API must only be called when already in the
 *   context of the channel art manager.
 *
 *   Inputs:
 *       *psObj - A pointer to the album art highband object
 *       *phPayload - A pointer to a buffer handle containing image
 *           data that needs processing.
 *
 *   Output: Output: TRUE on success, FALSE on error
 *
 *******************************************************************************/
static BOOLEAN bProcessImageMessage (
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    size_t tTotalBitsRead,
    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT *psImageCtrl
        )
{
    ALBUM_ART_ASSOC_ROW_STRUCT sAssocRow;
    BOOLEAN bSuccess,
            bPresent = FALSE;

    STRING_OBJECT hCaption = STRING_INVALID_OBJECT;

    size_t tLeftover = 0;

    UN8 un8Action = 0,
        un8Compression = 0,
        un8AUTotal = 1,
        un8AUCount = 0;

    // Clear out the association row
    OSAL.bMemSet(&sAssocRow, (UN8)0, sizeof(ALBUM_ART_ASSOC_ROW_STRUCT));

    // Read the action (and verify that it was read properly); the action
    // itself doesn't need to be stored in the association, as we can infer
    // later what type of action this from the presence (or absence) of data
    // items such as PIDs, association timeouts, and blank/default flags.
    bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_ACTION_BITLEN,
                          &tTotalBitsRead, &un8Action, "action" );
    if ( bSuccess == FALSE )
    {
        return FALSE;
    }

    // Make the sure the action is valid
    if ( ALBUM_ART_ACTION_UNKNOWN <= un8Action )
    {
      SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
              ALBUM_ART_HIGHBAND_OBJECT_NAME
              ": Out-of-range album art action found: %u",
              un8Action );
      return FALSE;
    }

    // Per draft H of the Pix UI Spec (SX-845-0233), only actions 0x0
    // (assoc to PID) and 0x4 (service default update) are supported right
    // now; we should never see the others over the air, but if we do
    // they're to be ignored for the moment.

    if ( ( ALBUM_ART_ACTION_TIMED_SID_ASSOCIATE == un8Action ) ||
         ( ALBUM_ART_ACTION_SID_BLANK == un8Action ) ||
         ( ALBUM_ART_ACTION_SID_DEFAULT == un8Action ) )
    {
        // Ignore and return success
        return TRUE;
    }

    // Read the compression type (and verify that it was read properly.)
    bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_COMPRESSION_BITLEN,
                          &tTotalBitsRead, &un8Compression, "compression" );
    if ( bSuccess == FALSE )
    {
        return FALSE;
    }

    // Only JPEG (0x0) compression is available right now. If we get something
    // else in the compression field, just bail out but return "success", as this
    // isn't technically an error.
    if ( (UN8)ALBUM_ART_COMPRESSION_UNKNOWN <= un8Compression )
    {
        // Just return "success". We'll free the payload
        // further up the call chain.
        return TRUE;
    }

#if 0
    // Removed pending final UIRR
    // Note that if the action is default or blank, we need to set that on our assoc
    // row.
    if ( un8Action == ALBUM_ART_ACTION_SID_BLANK )
    {
        sAssocRow.bBlank = TRUE;
    }
    else if ( un8Action == ALBUM_ART_ACTION_SID_DEFAULT )
    {
        sAssocRow.bDefault = TRUE;
    }
#endif

    // From here on out, the message will vary slightly dependent on the action we just
    // received. The SID comes next in four of the five action "message subtypes".
    switch ( un8Action )
    {
        // Note: intentional fall-through ahead
        case ALBUM_ART_ACTION_PID_ASSOCIATE:
#if 0
        // Removed pending final UIRR
        case ALBUM_ART_ACTION_TIMED_SID_ASSOCIATE:
        case ALBUM_ART_ACTION_SID_DEFAULT:
        case ALBUM_ART_ACTION_SID_BLANK:
#endif
        {
            // Read the SID (and verify that it was read properly.)
            bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_SID_BITLEN,
                    &tTotalBitsRead, &sAssocRow.un16ServiceId, "SID" );
            break;
        }

        // Only service default images have image IDs
        case ALBUM_ART_ACTION_DEFAULT_UPDATE:
        {
            // Read the Image ID(and verify that it was read properly.)
            bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_IMAGEID_BITLEN,
                    &tTotalBitsRead, &sAssocRow.un16ImageId, "Image ID" );
            break;
        }

        default:
        {
            // Will never happen, as we check for action validity above; Still, adding
            // a default case means one less thing for CLANG static analysis to complain
            // about.
            bSuccess = TRUE;
            break;
        }
    }

    // Let's see if we completed our last read successfully ...
    if ( bSuccess == FALSE )
    {
        return FALSE;
    }

    // Some actions also have supplemental info; parse those now.
    switch ( un8Action )
    {
        case ALBUM_ART_ACTION_PID_ASSOCIATE:
        {
            // Read the PID ID(and verify that it was read properly.)
            bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_PID_BITLEN,
                    &tTotalBitsRead, &sAssocRow.un32ProgramId, "PID ID" );
            break;
        }

#if 0
        // Removed pending final UIRR
        case ALBUM_ART_ACTION_TIMED_SID_ASSOCIATE:
        {
            UN8 un8Duration = 0;

            // Read the association duration (and verify that it was read properly.)
            bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_DURATION_BITLEN,
                    &tTotalBitsRead, &un8Duration, "duration" );

            // We can't just assign the duration straight to the association row, as the
            // association expires un8Duation seconds after the message is received. We have
            // to construct that value by grabbing the current time.
            if ( bSuccess == TRUE )
            {
                UN32 un32CurSeconds = 0;
                OSAL_RETURN_CODE_ENUM eSuccess;

                eSuccess = OSAL.eTimeGet ( &un32CurSeconds );

                if ( OSAL_SUCCESS != eSuccess )
                {
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                            ALBUM_ART_HIGHBAND_OBJECT_NAME
                            ": Problem calling OSAL.eTimeGet()" );
                    return FALSE;
                }

                // The unit of measure for the duration is 15 second
                // increments, so we adjust accordingly before creating
                // our expiration delta.
                sAssocRow.un32Expiration = ( un32CurSeconds +
                        ( ALBUM_ART_DURATION_SECS * (UN32)un8Duration ) );
            }
            break;
        }
#endif

        case ALBUM_ART_ACTION_DEFAULT_UPDATE:
        {
            // Read the PID ID(and verify that it was read properly.)
            bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_IMAGEVER_BITLEN,
                    &tTotalBitsRead, &sAssocRow.un16ImageVer, "image version" );
            break;
        }

        default:
        {
            // We do nothing in this case, as for the other two actions there's nothing
            // to read here; Still, adding a default case means one less thing for
            // CLANG static analysis to complain about.
            bSuccess = TRUE;
            break;
        }
    }

    // Check again to see if our last round of decoding succeeded ...
    if ( bSuccess == FALSE )
    {
        return FALSE;
    }

    // Next up:  there may (or may not) be a caption depending on the presence field
    // in the bitstream.
    bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_PRESENCE_BITLEN,
            &tTotalBitsRead, &bPresent, "caption presence flag" );
    if ( bSuccess == FALSE )
    {
        return FALSE;
    }

    // If it's there, grab the caption. If it's not, the caption will be a
    // defacto NULL object as we zeroed the association row out above.
    if ( bPresent == TRUE )
    {
        size_t tNumSymbolsFound = (size_t)0;

        hCaption = BAUDOT_hToString( SMS_INVALID_OBJECT, *phPayload,
                BAUDOT_BEHAVIOR_PROCESS_TO_END, TRUE, TRUE,
                ALBUM_ART_HIGHBAND_MAX_CAPTION_SYMBOL_COUNT,
                &tNumSymbolsFound );

        if( STRING_INVALID_OBJECT == hCaption )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": Could not read the caption; invalid object returned." );
            return FALSE;
        }

        tTotalBitsRead += ( BAUDOT_CHAR_BITWIDTH * tNumSymbolsFound );
    }

    // Read the Extdata Presence flag (and verify that it was read properly.)
    bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_PRESENCE_BITLEN,
                          &tTotalBitsRead, &bPresent, "extdata presence flag" );
    if ( bSuccess == FALSE )
    {
        return FALSE;
    }

    // If it's there (which would be odd ... see below), grab the extdata size.
    if ( bPresent == TRUE )
    {
        UN8 un8ExtDataSize;
        size_t tBitsRead = 0;

        // Read the PID ID(and verify that it was read properly.)
        bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_EXTDATA_COUNT_BITLEN,
                &tTotalBitsRead, &un8ExtDataSize, "extension data size" );
        if ( bSuccess == FALSE )
        {
            return FALSE;
        }
        
        // Ignore the extension data for now, as it's undefined per
        // SX-9845-0231 section 5.1.8; note that the ext data length
        // (in bytes) is actually un8ExtDataSize + 1 (per section 6.2.4)
        tBitsRead = OSAL.tBufferSeekHead( *phPayload, un8ExtDataSize+1 );

        if ( tBitsRead != (size_t)( ALBUM_ART_HIGHBAND_EXTDATA_BLOCK_BITLEN *
                                un8ExtDataSize+1 ) )
        {
            return FALSE;
        }

        tTotalBitsRead += tBitsRead;
    }

    // Read the AU count / totals presence flag (and verify that it was read properly.)
    bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_PRESENCE_BITLEN,
                &tTotalBitsRead, &bPresent, "AU count presence flag" );
    if ( bSuccess == FALSE )
    {
        return FALSE;
    }

    // If the AU count is there, grab its size
    if ( TRUE == bPresent )
    {
        UN8 un8AUCountSize = 0;

        // Read the AU count size (and verify that it was read properly.)
        bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_AU_COUNT_BITLEN,
                              &tTotalBitsRead, &un8AUCountSize, "AU count size" );
        if ( bSuccess == FALSE )
        {
            return FALSE;
        }

        // AU Count size is defined as one greater than
        // what's read off the wire. (See the CAI spec 5.1.9)
        un8AUCountSize += 1;        

        // Read the AU total (and verify that it was read properly.)
        bSuccess = bReadNext( *phPayload, un8AUCountSize, &tTotalBitsRead,
                    &un8AUTotal, "AU total" );
        if ( bSuccess == FALSE )
        {
            return FALSE;
        }

        // AU Total is also defined as one greater than what's read
        // off the wire  (See the CAI spec 5.1.10)
        un8AUTotal += 1;         

        // Read the AU total (and verify that it was read properly.)
        bSuccess = bReadNext( *phPayload, un8AUCountSize,
                    &tTotalBitsRead, &un8AUCount, "AU count" );
        if ( bSuccess == FALSE )
        {
            return FALSE;
        }
    }

    // If our last sequence number is the intial val, or if we get something unexpected
    // for the association, then we have to re-init tracking.
    if ( ( psImageCtrl->n8LastSeqNum == ALBUM_ART_LAST_SEQ_NO_INITIAL_VAL ) ||
         ( (sAssocRow.un16ServiceId  != psImageCtrl->sImageInProcess.un16ServiceId)  ||
           (sAssocRow.un32ProgramId  != psImageCtrl->sImageInProcess.un32ProgramId)  ||
           (sAssocRow.un16ImageId    != psImageCtrl->sImageInProcess.un16ImageId)    ||
           (sAssocRow.un16ImageVer   != psImageCtrl->sImageInProcess.un16ImageVer)   ||
           (sAssocRow.un32Expiration != psImageCtrl->sImageInProcess.un32Expiration) ||
           (un8AUCount != (UN8)(psImageCtrl->n8LastSeqNum + 1) ) ) )
    {
        // Either we've lost sync, or we started in the middle of a sequence of AUs.
        // We'll try to resync on the current AU.
        vResetImageTracking(psImageCtrl);
        psImageCtrl->sImageInProcess = sAssocRow;
    }

    // The image data is byte-aligned, so we need to skip ahead to the next
    // byte boundary. See how many bits we have left over.
    tLeftover = tTotalBitsRead % ALBUM_BITS_PER_BYTE;

    // If we have leftovers, use that to figure out how many bits
    // we need to skip ahead
    if ( (size_t)0 != tLeftover )
    {
        size_t tStuffBits;
        UN8 un8Stuffing = 0;

        tStuffBits = ALBUM_BITS_PER_BYTE - tLeftover;

        // Read any stuff bits, if necessary
        bSuccess = bReadNext( *phPayload, tStuffBits,
                              &tTotalBitsRead, &un8Stuffing, "stuffing"
                            );

        // Make sure the stuffing is zero; if it's not, then we mis-calculated
        // our number of stuff bits.
        if ( ( (UN8)0 != un8Stuffing ) ||
             ( FALSE == bSuccess ) )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": Problem handling stuff bits; Stuffing: %u, Success: %s",
                    un8Stuffing, ( TRUE == bSuccess ) ? "True" : "False" );
            return FALSE;
        }
    }

    // Everything that could go wrong in decoding would have done so by now; let's mark
    // our sequence number for next time.
    psImageCtrl->n8LastSeqNum = un8AUCount;

#if 0
    // Removed pending final UIRR
    // Actions to blank or default a given SID have no image data associated
    // with them. We just send the association over right away; the other
    // action types will send the association over with the last image AU.
    if ( ( ALBUM_ART_ACTION_SID_BLANK == un8Action ) ||
         ( ALBUM_ART_ACTION_SID_DEFAULT == un8Action ) )
    {
        bSuccess = GsArtMgrIntf.bAlbumArtAssociationUpdate(
                                psObj->hChannelArtService,
                                &sAssocRow, NULL
                             );
    }
#endif

    // If our action is 0x0, 0x1, or 0x4 then we have to
    // hand the image over to the reassembler, which will
    // take care of sending the image (and any association,
    // if necessary) to the manager upon completion.
    // Note: As of SX-9845-0233 Draft H, we're ignoring any
    // captions for the moment (although we should never see one OTA.)
    if ( ( ALBUM_ART_ACTION_PID_ASSOCIATE == un8Action ) ||
#if 0
        // Removed pending final UIRR
         ( ALBUM_ART_ACTION_TIMED_SID_ASSOCIATE == un8Action ) ||
#endif
         ( ALBUM_ART_ACTION_DEFAULT_UPDATE == un8Action ))
    {
        bSuccess = bImageReassembler( psObj, un8AUTotal, un8AUCount,
                &sAssocRow, NULL, phPayload, psImageCtrl, un8Action );
    }

    // Tell the caller how it all went
    return bSuccess;
};

/*****************************************************************************
*
*   vResetImageTracking
*
*   This function resets our sequence number and free any working image
*   data in the buffer. This typically happens as a result of lost sync or if an
*   image has been completed.
*
*   Inputs:
*       *psImageCtrl - The album art object's image control struct; needed
*           for access to the buffers and sequence number.
*
*******************************************************************************/
static void vResetImageTracking (
    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT *psImageCtrl
        )
{
    // Destroy any in-progress logo buffer data
    if ( psImageCtrl->hImageData != OSAL_INVALID_BUFFER_HDL )
    {
        OSAL.eBufferFree(psImageCtrl->hImageData);
        psImageCtrl->hImageData = OSAL_INVALID_BUFFER_HDL;
    }

    // Clear the last sequence number attribute
    // (this is the key value in this structure)
    psImageCtrl->n8LastSeqNum = ALBUM_ART_LAST_SEQ_NO_INITIAL_VAL;

    return;
}

/*****************************************************************************
 *
 *   bImageReassembler
 *
 *   This function is utilized when processing a work unit based on image
 *   message data.  Each image is reassembled from component AUs and
 *   is is provided to the manager via the manager interface upon
 *   completion (along with the corresponding association for
 *   ephemereal images.)
 *
 *   Inputs:
 *       *psObj - A pointer to the album art object struct
 *       un8AUTotal - The total number of AUs expected for this image
 *       un8AUCount - The sequence number of this AU
 *       *psAssocRoc - A pointer to the association row that describes the
 *           SID/PID association and image ID info for this image.
 *       *phCpation - A pointer to handle for a string object representing
 *           this image's caption
 *       *phPayload - A pointer to a handle to the message buffer containing
 *           the image data.
 *       eAction - The type of action being performed; used to determine which
 *           interface methods get called on the data service end
 *
 *
 *   Output: Output: TRUE on success, FALSE on error
 *
 *******************************************************************************/
static BOOLEAN bImageReassembler (
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    UN8 un8AUTotal,
    UN8 un8AUCount,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    STRING_OBJECT *phCaption,
    OSAL_BUFFER_HDL *phPayload,
    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT *psImageCtrl,
    UN8 un8Action
        )
{
    BOOLEAN bOk = TRUE,
            bResetProgress = FALSE;
        
    // Null out the art that we'll be updating
    CHANNEL_ART_OBJECT hArtToUpdate = CHANNEL_ART_INVALID_OBJECT;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturn = OSAL_SUCCESS;

        // If we're starting a new image, initialize our tracking info;
        // we'll just use the starting AU's buffer to append image data into...
        if ( un8AUCount == ALBUM_ART_FIRST_MULTI_AU_SEQ_NO )
        {
            // Make sure we don't have any work data in the buffer
            if ( psImageCtrl->hImageData != OSAL_INVALID_BUFFER_HDL )
            {
                // Odd .. make sure we clear this out.
                vResetImageTracking(psImageCtrl);

                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": At ALBUM_ART_FIRST_MULTI_AU_SEQ_NO with non-"
                    "empty buffer" );
            }

            // Initialize the sequence number, # of messages processed,
            // and our working image data buffer
            psImageCtrl->n8LastSeqNum = (N8)un8AUCount;
            psImageCtrl->un8NumMessagesProcessed = (UN8)0;
            psImageCtrl->hImageData = *phPayload;

            // We notify the upper layers that we're taking ownership
            // of this payload by nulling out the payload pointer.
            *phPayload = OSAL_INVALID_BUFFER_HDL;
        }
        // ... on the other hand, if this isn't the first AU in a sequence,
        // save the new data into the buffer that's already been set up
        // for us.
        else 
        {
            // If we're not at the start of an image sequence, and we have a valid
            // buffer, glom our data onto the end. If we're mid-sequence *without*
            // a buffer to work on, this means we missed the start of the sequence;
            // there's no way to recover from this, so just wait and hope to
            // re-sync at the next opportunity.
            if ( psImageCtrl->hImageData != OSAL_INVALID_BUFFER_HDL )
            {
                eReturn = OSAL.eBufferAppend( psImageCtrl->hImageData, *phPayload );
            }
            else
            {
                break;
            }
        }

        // Check to make sure that data copy worked ...
        if ( OSAL_SUCCESS != eReturn )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": Problem encountered during buffer append: %s",
                    OSAL.pacGetReturnCodeName(eReturn) );

            bResetProgress = TRUE;
            break;
        }

        // That worked ... now increment the number of messages we've
        // processed for this image
        psImageCtrl->un8NumMessagesProcessed++;

        // We know this image is done when we've hit the expected
        // number of messages (as specified by un8AUTotal; if we're
        // not in the right spot just bail out
        if ( ( un8AUCount != (un8AUTotal - 1) ) ||
             ( psImageCtrl->un8NumMessagesProcessed != un8AUTotal ) )
        {
            break;
        }

        // If we've made it here, we have the right number of messages;
        // we have to make sure and reset our image tracking, even if we
        // have trouble passing this through to the manager.
        bResetProgress = TRUE;

        // If we're updating a default image, call default update handler
        // to write out the image data and bump the corresponding version in
        // the assoc table
        if ( ALBUM_ART_ACTION_DEFAULT_UPDATE == un8Action )
        {
            // For default image updates, we're not adding an association,
            // but we still need to pass the image data to the manager for
            // to be written to disk.
            bOk = GsArtMgrIntf.bAlbumArtImageUpdate( psObj->hChannelArtService,
                    psAssocRow, psImageCtrl->hImageData, phCaption, NULL );

        }
        // Otherwise, we're associating to a SID, so we need to set up an
        // association as well as pass the image data over.
        else
        {
            // Pass the association to the channel art manager. The channel art manager
            // will pass us back a a pointer to a channel art object that we can
            // place our image into.
            bOk = GsArtMgrIntf.bAlbumArtAssociationUpdate( psObj->hChannelArtService,
                                psAssocRow, &hArtToUpdate );


            // If everything's okay, but we don't have any art to update, then this
            // is just a duplicate. Don't bother saving anything off.
            if ( ( TRUE == bOk ) &&
                 ( CHANNEL_ART_INVALID_OBJECT == hArtToUpdate ) )
            {
                break;
            }

            // Check to make sure that everything went all right
            if ( FALSE == bOk )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        ALBUM_ART_HIGHBAND_OBJECT_NAME
                        ": Problem setting album association!" );
                break;
            }

            // Using the album art handle we received above, pass this back through
            // to the manager to update its album art image.
            bOk = GsArtMgrIntf.bAlbumArtImageUpdate( psObj->hChannelArtService,
                                psAssocRow, psImageCtrl->hImageData,
                                phCaption, &hArtToUpdate );

            if ( FALSE == bOk )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        ALBUM_ART_HIGHBAND_OBJECT_NAME
                        ": Problem setting image data!!" );
                break;
            }
        }
    } while(FALSE);

    // We need to reset if we've either failed somehow, or have completed
    // an image sequence.
    if ( bResetProgress == TRUE )
    {
        vResetImageTracking( psImageCtrl );
    }

    return bOk;
}

/*****************************************************************************
 *
 * bProcessAssociationMessage
 *
 * An album art association message (see SX-9845-0231, section 6) has a
 * series of service ID to image ID mappings.
 *
 * The associations are provided as a monotonically increasing array
 * of SIDs. The size of this array is given by the SIDCNT paramter.
 *
 * Note that we don't remove any associations *not* referenced by this
 * message, as update messages have been seen (as of June 2013) to only
 * go up to SID ~230, while the baseline image package contained images
 * for SIDs up to 904. "Removal" is instead handled by the DBehavior
 * parameter, which merely sets the SID's default art to the service
 * default instead.
 *
 *   Inputs:
 *       *psObj - A pointer to the album art highband object
 *       *phPayload - A pointer to the association payload that
 *           needs processing.
 *
 *    Output: TRUE on success, FALSE on error
 *
 *******************************************************************************/
static BOOLEAN bProcessAssociationMessage (
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    ALBUM_ART_ASSOC_ROW_STRUCT sAssocRow;
    BOOLEAN bSuccess;

    UN16 un16CurrentSID = 1,
         un16SIDCount = 0;
    UN8 un8ImageAssVersion = 0;

    // Clean out the assoc row before we do anything with it
    OSAL.bMemSet(&sAssocRow, (UN8)0, sizeof(ALBUM_ART_ASSOC_ROW_STRUCT));

    // Read the AU count / totals presence flag
    // (and verify that it was read properly.)
    bSuccess = bReadNext( *phPayload,
            ALBUM_ART_HIGHBAND_ASSIGNMENT_VERSION_BITLEN,
            TOTAL_BITS_DONTCARE, &un8ImageAssVersion,
            "default assignment version" );

    if ( bSuccess == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": Problem reading default assignment!!" );
        return FALSE;
    }

    // Read the SID count  (and verify that it was read properly.)
    bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_SID_COUNT_BITLEN,
            TOTAL_BITS_DONTCARE, &un16SIDCount, "SID Count" );
    if ( bSuccess == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": Problem reading SID count!!" );
        return FALSE;
    }

    // Tell the manager we're about to start a batch of updates; this is mostly
    // so the channel art manager can open a transaction
    bSuccess = GsArtMgrIntf.bAlbumArtAssocBegin( psObj->hChannelArtService );
    if ( bSuccess == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
            ALBUM_ART_HIGHBAND_OBJECT_NAME
            ": Cannot start DB transaction!!" );
        return FALSE;
    }

    // Per SX-0845-0321, 6.2, we are actually to repeat this SID count + 1 times.
    for ( un16CurrentSID = 1; un16CurrentSID <= (un16SIDCount+1); ++un16CurrentSID )
    {
        // Now process a portion of the association message
        // and use that return value for our result

        UN8 un8DefBehavior = 0;
        UN16 un16DefID = 0;

        // Read the default behavior param for this SID. If equal to 0,
        // we merely set the SID's default art to the service default image.
        // It can also be used to set up more complicated relationships based
        // on CDO etype (see SX-9845-0231, Sec 6.2.2, table 8)
        bSuccess = bReadNext( *phPayload,
                              ALBUM_ART_HIGHBAND_DEFUALT_BEHAVIOR_BITLEN,
                              TOTAL_BITS_DONTCARE,
                              &un8DefBehavior,
                              "default behavior" );
        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": Problem reading default behavior!!" );
            return FALSE;
        }

        // The upper two bits are don't-cares (for now),
        // so mask them out to remove any surprises.
        un8DefBehavior &= ALBUM_ART_HIGHBAND_DEFAUT_BEHAVIOR_BITMASK;

        // Per SX-9845-0233 Draft H, we're treating values of 10 and 11
        // for DBEHAVIOR as 01; although (like before) we whouldn't be
        // seeing these OTA, and the full support is included below this
        // line in case future drafts restore this functionality.
        if  ( ( ALBUM_ART_USE_PROVIDED_SPORTS_ONLY == un8DefBehavior ) ||
              ( ALBUM_ART_USE_PROVIDED_MUSIC_ONLY == un8DefBehavior ) )
        {
            un8DefBehavior = ALBUM_ART_USE_PROVIDED;
        }


        // We only have a default ID for behaviors > 0 (ALBUM_ART_USE_DEFAULT)
        if ( un8DefBehavior > ALBUM_ART_USE_DEFAULT )
        {
            //
            // Read the SID count  (and verify that it was read properly.)
            //
            bSuccess = bReadNext( *phPayload,
                                  ALBUM_ART_HIGHBAND_DEFUALT_ID_BITLEN,
                                  TOTAL_BITS_DONTCARE,
                                  &un16DefID,
                                  "default ID" );
            if ( bSuccess == FALSE )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        ALBUM_ART_HIGHBAND_OBJECT_NAME
                        ": Problem reading default ID!!" );
                return FALSE;
            }
        }

        // We can just do a direct comparison here as we've already
        // masked off the upper two bits. NULL CDO etypes mean
        // this association can apply to *ANY* type of content.
        switch ( un8DefBehavior )
        {
            case ALBUM_ART_USE_DEFAULT:
            {
                sAssocRow.un16ImageId = ALBUM_ART_SERVICE_DEFAULT_ID;
                sAssocRow.eType = (CDO_TYPE_ENUM)0;
            }
            break;
            case ALBUM_ART_USE_PROVIDED:
            {
                sAssocRow.un16ImageId = un16DefID;
                sAssocRow.eType = (CDO_TYPE_ENUM)0;
            }
            break;
            case ALBUM_ART_USE_PROVIDED_MUSIC_ONLY:
            {
                sAssocRow.un16ImageId = un16DefID;
                sAssocRow.eType = CDO_MUSIC;
            }
            break;
            case ALBUM_ART_USE_PROVIDED_SPORTS_ONLY:
            {
                sAssocRow.un16ImageId = un16DefID;
                sAssocRow.eType = CDO_SPORTS;
            }
            break;
            default:
            {
                // Should never get this, but ...
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        ALBUM_ART_HIGHBAND_OBJECT_NAME
                        ": Received invalid un8DefBehavior: %u!!",
                        un8DefBehavior );
                return FALSE;
            }
        }

        sAssocRow.un16ServiceId = un16CurrentSID;

        // Tell the manager about the new association
        bSuccess = GsArtMgrIntf.bAlbumArtAssociationUpdate (
                    psObj->hChannelArtService, &sAssocRow,
                    NULL );
        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                ALBUM_ART_HIGHBAND_OBJECT_NAME
                ": Cannot update DB!!" );
            return FALSE;
        }
    }

    // Tell the manager we're done now (even if an error was experienced earlier)
    bSuccess = GsArtMgrIntf.bAlbumArtAssocEnd( psObj->hChannelArtService );
    if ( bSuccess == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
            ALBUM_ART_HIGHBAND_OBJECT_NAME
            ": Cannot complete DB transaction!!" );
    }

    // Technically there's also ExtData back here, as well as stuffing, but we don't
    // particularly care about those (especially as ExtData is undefined at present.)
    // See SX-9845-0231, section 6.2.5

    return bSuccess;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/***************************************************************************
*
*   eParseMessageHeader
*
*   This function parses the data common to all types of Album Art images:
*   namely, the CRC, the PVN, and the carousel identifier. It returns
*   the carousel ID so that the main message handling routine can determine
*   which message handler to call (as Album Art images are distinguished by
*   their carousel.)
*
*   Inputs:
*       hInterface - A handle to the album art interface object
*       *phPayload- A pointer to a handle to an SMS buffer object
*           whose header we'll be reading.
*       *ptBitsRead - A pointer to a size_t var that will be updated
*           with the number of bits read during header processing.
*
*   Outputs:
*       An ALBUM_ART_HIGHBAND_CARID_ENUM which represents the carousel
*       ID of this message. Will be set to ALBUM_ART_HIGHBAND_CARID_UNKNOWN
*       upon any error in processing.
*
*******************************************************************************/
static ALBUM_ART_HIGHBAND_CARID_ENUM eParseMessageHeader (
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    size_t *ptBitsRead
        )
{
    // We don't know what the carousel is yet
    ALBUM_ART_HIGHBAND_CARID_ENUM eCarouselId
        = ALBUM_ART_HIGHBAND_CARID_UNKNOWN;

    do
    {
        BOOLEAN bSuccess;
        PVN tPVN = (PVN)0;

        // Double check that we have a valid payload before continuing ...
        if ( OSAL_INVALID_BUFFER_HDL == *phPayload )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                  ALBUM_ART_HIGHBAND_OBJECT_NAME
                  ": Invalid buffer handle passed to eParseMessageHeader!" );
            break;
        }

        // We're here? Great, now validate the data's CRC.
        bSuccess = DS_UTIL_bIsCRCValid(psObj->hCRC, *phPayload, NULL);
        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": SDTP AU Payload Invalid!!" );
            break;
        }

        // Cut the CRC from the paylaod
        bSuccess = DS_UTIL_bCutCRC(*phPayload);
        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    ": failed to cut off CRC" );
            break;
        }

        // Read the PVN
        bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_PVN_BITLEN,
                ptBitsRead, &tPVN, "PVN" );
        if ( bSuccess == FALSE )
        {
            // Error printing handled by bReadNext
            break;
        }

        // Verify the PVN
        if ( ALBUM_ART_HIGHBAND_PVN != tPVN )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME
                    " Incorrect PVN - got %u, expected %u",
                    tPVN, ALBUM_ART_HIGHBAND_PVN );
            break;
        }

        // Read the Carousel Id
        bSuccess = bReadNext( *phPayload, ALBUM_ART_HIGHBAND_CARID_BITLEN,
                               ptBitsRead, &eCarouselId, "Carousel ID" );
        if ( bSuccess == FALSE )
        {
            // Error printing handled by bReadNext
            break;
        }

        // Make sure that the carousel ID is what we are looking for.
        if ( eCarouselId >= ALBUM_ART_HIGHBAND_CARID_UNKNOWN )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    ALBUM_ART_HIGHBAND_OBJECT_NAME" Invalid Carousel ID: %u",
                    eCarouselId );
            eCarouselId = ALBUM_ART_HIGHBAND_CARID_UNKNOWN;
            break;
        }

        return eCarouselId;

    } while (FALSE);


    // If we wound up here, there was an error processing our payload,
    // so we make sure to free it and clean up the buffer handle
    DATASERVICE_IMPL_bFreeDataPayload(*phPayload);
    *phPayload = OSAL_INVALID_BUFFER_HDL;

    return eCarouselId;
}

/*******************************************************************************
*
*   bReadNext
*
*   This is a utility function designed to handle reading data (using the
*   appropriate endian-preserving buffer function), as well as printing of debug
*   messages and updating the number of bits read.
*
*   Inputs:
*       hPayload - A handle to the payload buffer
*       tBitCount - A size_t variable that specifies how many bits
*           are to be read from the buffer provided by hPayload.
*       *ptTotalBitsRead - A pointer a size_t variable that keeps a
*           tally of the number of bits read (thus far) during the
*           processing of the buffer provided by hPayload. Can be NULL,
*           in which case the number of bits read is discarded.
*       *pvVariable - A void pointer to the variable where the "incoming" data
*           will be stored.
*       *pacName - A c-style string that will be used to provide the name of the
*           variable should this function need to print an error message.
*
*   Outputs:
*       Return TRUE on a successful read, FALSE otherwise.
*
*******************************************************************************/
static BOOLEAN bReadNext(
    OSAL_BUFFER_HDL hPayload,
    size_t tBitCount,
    size_t *ptTotalBitsRead,
    void *pvVariable,
    char *pacName
        )
{
    BOOLEAN bSuccess = FALSE;
    size_t tBitsRead = 0;

    // Use the number of bits to be read to determine which
    // OSAL buffer read function that we'll need.
    if ( tBitCount <= 8 )
    {
        tBitsRead = OSAL.tBufferReadHeadBits( hPayload, pvVariable,
                        0, tBitCount );

        if ( tBitsRead == tBitCount )
        {
            bSuccess = TRUE;
        }
    }
    else
    {
        if ( tBitCount <= 16 )
        {
            bSuccess
                = OSAL.bBufferReadBitsToUN16( hPayload, pvVariable, tBitCount );
        }
        else
        {
            bSuccess
                = OSAL.bBufferReadBitsToUN32( hPayload, pvVariable, tBitCount );
        }
        tBitsRead = tBitCount;
    }

    //
    // Were we able to read the field?
    //
    if ( bSuccess == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        ALBUM_ART_HIGHBAND_OBJECT_NAME
                        ": Incorrect Number of bits read for the %s: %u",
                        pacName, (UN32)tBitsRead );

        return FALSE;
    }

    // If the user cares about the total # of bits, then
    // update the total.
    if ( NULL != ptTotalBitsRead )
    {
        *ptTotalBitsRead += tBitsRead;
    }

    return TRUE;
}
