/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/

#ifndef _TRAFCAM_PVN1_H_
#define _TRAFCAM_PVN1_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include <stdio.h>

#include "sms_api.h"
#include "trafcam_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

// PVN 1 - Definitions
#define TRAFCAM1_OBJECT_NAME "TRAFCAM1"

#define TRAFCAM1_DSI                 (492)

// The protocol version supported
// by SMS for this interface
#define TRAFCAM1_PVN (PVN)(1)

#define TRAFCAM1_PACKET_MAX_SIZE (5120)

// The pool size is calculated as max image size without single packet (because
// we store the image to file right when last packet received, so we could use
// current payload buffer for this purpose) multiplied to number of active DMIs
// Every DMI receives a sequence of packets for a single image.
// Additionally, space is reserved for just received payload and
// currently collecting payload - 5 Kb for each of them.
#define TRAFCAM1_BUFFER_BYTESIZE ((TRAFCAM1_PACKET_MAX_SIZE * \
    (TRAFCAM1_AUTOT_MAX - 1) * TRAFCAM1_NUM_OF_DMIS ) + \
    TRAFCAM1_PACKET_MAX_SIZE * 2)

// number of TMC markets
#define TRAFFIC_MARKETS_NUMBER (36)

// this buffer is used to calculate CRC for image info
#define TRAFCAM1_CRC_CALC_BUFFER_SIZE (100)

// This is size of block in block pool used for CRC calculation
#define TRAFCAM1_CRC_CALC_BLOCK_SIZE (100)

// Protocol related definitions

// Access unit header

// General access unit definitions
#define TRAFCAM1_PVN_BITLEN (4)
#define TRAFCAM1_CAROUSEL_BITLEN (4)
#define TRAFCAM1_CRC_BYTELEN (4)

// Carousel ID
#define TRAFCAM1_CAROUSEL_ID (0)

#define TRAFCAM1_TABLE_BITLEN (8)
#define TRAFCAM1_BSA_BITLEN (8)
#define TRAFCAM1_SIG_BITLEN (16)
#define TRAFCAM1_AUTOT_BITLEN (4)
#define TRAFCAM1_AUCT_BITLEN (4)

// Bitfield offsets
#define TRAFCAM1_TABLE_OFFSET (TRAFCAM1_PVN_BITLEN + \
                               TRAFCAM1_CAROUSEL_BITLEN)
#define TRAFCAM1_BSA_OFFSET (TRAFCAM1_TABLE_OFFSET + \
                             TRAFCAM1_TABLE_BITLEN)

// Image metadata
#define TRAFCAM1_SOI_BITLEN (8)
#define TRAFCAM1_FLAG_BITLEN (1)
#define TRAFCAM1_DIR_BITLEN (3)
#define TRAFCAM1_TIME_BITLEN (11)
#define TRAFCAM1_TTL_BITLEN (6)

// Location type TMC
#define TRAFCAM1_TMC_BITLEN (16)
#define TRAFCAM1_OFFSET_BITLEN (3)
#define TRAFCAM1_DIRECTION_BITLEN (1)

// Location type geographic-referenced
#define TRAFCAM1_LON_BITLEN (25)
#define TRAFCAM1_LON_FRAC_BITS (17)
#define TRAFCAM1_LAT_BITLEN (24)
#define TRAFCAM1_LAT_FRAC_BITS (17)

// Direction type definitions
#define TRAFCAM1_DIRECTION_POSITIVE (0)
#define TRAFCAM1_DIRECTION_NEGATIVE (1)

// Camera description max length
#define TRAFCAM1_DESC_MAX_LENGTH (40)

// Camera description symbol bitlen
#define TRAFCAM1_DESC_SYMBOL_BITLEN (8)

// Max TTL value
#define TRAFCAM1_TTL_MAX (60)

// Maximum number of access units in a single group
#define TRAFCAM1_AUTOT_MAX (8)

// Starting buffer size
#define TRAFCAM1_BUFFER_SIZE (256)

// number of 1st active DMI
#define TRAFCAM1_FIRST_DMI (740)

// number of active DMIs
#define TRAFCAM1_NUM_OF_DMIS (6)

// Invalid slot number
#define TRAFCAM1_INVALID_SLOT (N8_MIN)


  /**************/
 /** TYPEDEFS **/
/**************/

typedef struct trafcam1_process_slot_struct
{
    OSAL_BUFFER_HDL hBuffer;
    TRAFFIC_MARKET tMarket;
    UN16 un16SIG;
    UN8 un8BSA;
    UN8 un8AUCT;
    UN8 un8RefCount;
    BOOLEAN bChanged;
} TRAFCAM1_PROCESS_SLOT_STRUCT;

typedef struct trafcam1_object_struct
{
    // Handle to the top-level TRAFCAM service
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService;

    // CRC object handle
    OSAL_OBJECT_HDL hCRC;

    // Filter and messages data
    OSAL_OBJECT_HDL hMessagesData;

    char acBuffer[TRAFCAM1_BUFFER_SIZE];

    TRAFCAM1_PROCESS_SLOT_STRUCT asSlot[TRAFCAM1_NUM_OF_DMIS];

    // Apogee Traffic provides the images for the same traffic cameras
    // repeatedly. Sometimes images are the same, and we just update the timeout
    // value. Such messages could be detected using the market, BSA and SIG
    // values from the packet header.
    // However, we had to find the way how to deal with new images for the same
    // traffic cameras. Such packets have different SIG value but the image is
    // still related to the same traffic camera. And, previous image could still
    // be valid per the timeout value.
    // We decided to take the packet header part: lat, lon, tmc code,
    // description and direction, compute CRC for these fields and store it.
    // Thus, if receive the image with the same CRC, but different SIG, we know
    // that image has been updated and application is notified using the DSRL
    // entry change.
    // This block pool is used for buffer allocation, and this buffer is used as
    // temporary storage of data for CRC calculation.
    OSAL_OBJECT_HDL hCRCCalcBlockPool;

} TRAFCAM1_OBJECT_STRUCT;

typedef struct trafcam1_bsa_data_struct
{
    TRAFFIC_BSA tBSA;
    OSAL_OBJECT_HDL hGroups;
    OSAL_LINKED_LIST_ENTRY hEntry;
} TRAFCAM1_BSA_DATA_STRUCT;

typedef struct trafcam1_group_struct
{
    UN16 un16SIG;
    OSAL_LINKED_LIST_ENTRY hEntry;
    OSAL_CRC_RESULT tCRC;
    TRAFCAM1_OBJECT_STRUCT *psObj;
} TRAFCAM1_GROUP_STRUCT;

typedef struct trafcam1_market_data_struct
{
    TRAFFIC_MARKET tMarket;
    OSAL_OBJECT_HDL hBSAData;
    TRAFCAM1_PROCESS_SLOT_STRUCT *psSlot;
    OSAL_LINKED_LIST_ENTRY hEntry;
} TRAFCAM1_MARKET_DATA_STRUCT;

typedef struct trafcam1_filter_iterator_struct
{
    BOOLEAN bResult;
    OSAL_OBJECT_HDL hEntriesList;
    TRAFCAM1_OBJECT_STRUCT *psObj;
} TRAFCAM1_FILTER_ITERATOR_STRUCT;

typedef struct trafcam1_filter_bsa_iterator_struct
{
    BOOLEAN bResult;
    OSAL_LINKED_LIST_ENTRY hEntry;
    TRAFFIC_MARKET tMarket;
    OSAL_OBJECT_HDL hEntriesList;
} TRAFCAM1_FILTER_BSA_ITERATOR_STRUCT;

typedef struct trafcam1_packet_info_struct
{
    UN16 un16SIG;
    UN8 un8AUTOT;
    UN8 un8AUCT;
} TRAFCAM1_PACKET_INFO_STRUCT;

typedef struct trafcam1_bsa_search_struct
{
    TRAFFIC_BSA tBSA;
    TRAFFIC_MARKET tMarket;
} TRAFCAM1_BSA_SEARCH_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

// Public interface APIs

static TRAFCAM_INTERFACE_OBJECT hInit (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameraService,
    SMS_OBJECT hParent
        );

static void vUnInit (
    TRAFCAM_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProcessMessage (
    TRAFCAM_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bSetFilter (
    TRAFCAM_INTERFACE_OBJECT hInterface,
    OSAL_OBJECT_HDL hFilter
        );

static BOOLEAN bReportExpiredHash (
    TRAFCAM_INTERFACE_OBJECT hInterface,
    TRAFCAM_IMAGE_HASH tHash
        );

/* Object Private Prototypes */

static void vUninitObject (
    TRAFCAM1_OBJECT_STRUCT *psObj
        );

static N16 n16CompareMarkets (
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData1,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData2
        );

static BOOLEAN bCheckPacketHeader(
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bParsePacketData(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData
        );

static BOOLEAN bProcessCleanSlot (
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        );

static BOOLEAN bProcessNonCleanSlot(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        );

static BOOLEAN bProcessNewGroup (
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        );

static BOOLEAN bProcessContinuingGroup(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        );

static BOOLEAN bCompleteGroup(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    UN16 un16SIG
        );

static BOOLEAN bIterateToRemove(
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData,
    TRAFCAM1_FILTER_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bIterateToAdd (
    TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter,
    TRAFCAM1_FILTER_ITERATOR_STRUCT *psIteratorArg
        );

static void vReleaseBSAData(
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData
        );

static void vReleaseMarketData(
    TRAFCAM1_MARKET_DATA_STRUCT *psMarketData
        );

static N16 n16CompareSIGs(
    TRAFCAM1_GROUP_STRUCT *psGroup1,
    TRAFCAM1_GROUP_STRUCT *psGroup2
        );

static N16 n16CompareBSAData(
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData1,
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData2
        );

static BOOLEAN bCleanSlot(
    TRAFCAM1_PROCESS_SLOT_STRUCT *psSlot,
    TRAFFIC_MARKET tMarket
        );

static BOOLEAN bUpdateDMIs(
    TRAFCAM1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCalculateCRC(
    OSAL_OBJECT_HDL hCRC,
    OSAL_BUFFER_HDL hBuffer,
    OSAL_CRC_RESULT *ptCRC
        );

static BOOLEAN bGetMarketData(
    OSAL_OBJECT_HDL hMarketsList,
    OSAL_BUFFER_HDL hPayload,
    TRAFCAM1_MARKET_DATA_STRUCT **ppsMarketData,
    TRAFCAM1_BSA_DATA_STRUCT **ppsBSAData
        );

static BOOLEAN bIterateBSAToRemove(
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_FILTER_BSA_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bReadImageInfo(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TRAFCAM_IMAGE_INFO_STRUCT *psImageInfo,
    TRAFCAM1_GROUP_STRUCT *psGroup
        );

static BOOLEAN bReadPacketData(
    OSAL_BUFFER_HDL hPayload,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        );

static N16 n16FindGroupByCRC(
    TRAFCAM1_GROUP_STRUCT *psGroup,
    OSAL_CRC_RESULT *ptCRC
        );

static N16 n16FindMarket(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFFIC_MARKET tMarket
        );

static TRAFCAM1_BSA_DATA_STRUCT *psCreateBSAData(
    OSAL_OBJECT_HDL hBSAList,
    TRAFFIC_BSA tBSA
        );

static TRAFCAM1_GROUP_STRUCT *psCreateGroup (
    OSAL_OBJECT_HDL hGroupsList,
    TRAFCAM1_GROUP_STRUCT *psGroup
        );

static TRAFCAM1_MARKET_DATA_STRUCT *psCreateMarketData(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hFilter,
    TRAFFIC_MARKET tMarket
        );

static TRAFCAM1_GROUP_STRUCT *psExistingGroup (
    TRAFCAM1_BSA_DATA_STRUCT *psBSAData,
    TRAFCAM1_PACKET_INFO_STRUCT *psPacketInfo
        );

static TRAFCAM1_PROCESS_SLOT_STRUCT *psRegisterMarket(
    TRAFCAM1_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket
        );

static void vReleaseGroupWithNotify(
    TRAFCAM1_GROUP_STRUCT *psGroup
        );

static void vUnRegisterSlot(
    TRAFCAM1_PROCESS_SLOT_STRUCT *psSlot,
    TRAFFIC_MARKET tMarket
        );

  /***************/
 /** VARIABLES **/
/***************/

const TRAFCAM_PLUGIN_INTERFACE_STRUCT GsTrafcamIntf =
{
    /*.tOTABufferByteSize = */TRAFCAM1_BUFFER_BYTESIZE,
    /*.tDSI = */TRAFCAM1_DSI,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProcessMessage = */bProcessMessage,
    /*.bSetFilter = */bSetFilter,
    /*.bReportExpiredHash = */bReportExpiredHash
};

// this array contains mapping of market to slot; array index is market #
const N8 an8MarketToSlot[] =
{
    TRAFCAM1_INVALID_SLOT, // market 0 doesn't exist
    2,
    2,
    4,
    5,
    0,
    1,
    3,
    3,
    TRAFCAM1_INVALID_SLOT,
    4,
    2,
    2,
    2,
    0,
    1,
    1,
    0,
    3,
    3,
    5,
    4,
    4,
    TRAFCAM1_INVALID_SLOT,
    TRAFCAM1_INVALID_SLOT,
    4,
    TRAFCAM1_INVALID_SLOT,
    TRAFCAM1_INVALID_SLOT,
    TRAFCAM1_INVALID_SLOT,
    5
};

#endif /* _TRAFCAM_PVN1_H_ */
