/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _TDS_UTIL_H_
#define _TDS_UTIL_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "tds_util.h"

  /***************/
 /** CONSTANTS **/
/***************/

#define TDS_OBJECT_NAME "TDS"

#define TDS_PVN1        ((PVN)1)
#define TDS_CRC_BYTELEN (4)

// General
#define TDS_PVN_BITLEN (4)
#define TDS_CAR_BITLEN (3)
#define TDS_TABID_BITLEN (10)
#define TDS_TABVER_BITLEN (8)

// Table Def
#define TDS_TTIME_BITLEN (12)
#define TDS_RFU_BITLEN (4)
#define TDS_LBNO_BITLEN (6)
#define TDS_LBID_BITLEN (8)
#define TDS_LBTYPE_BITLEN (1)
#define TDS_DTYPE_BITLEN (3)
#define TDS_LBSIZE_BITLEN (12)
#define TDS_LENSIZE_BITLEN (4)
#define TDS_EXTCNT_BITLEN (8)
#define TDS_PRESENCE_FLAG_BITLEN (1)

#define TDS_DEF_HDR_BITLEN ( \
    TDS_RFU_BITLEN + \
    TDS_LBNO_BITLEN \
        )

#define TDS_LABEL_ENTRY_HDR_BITLEN ( \
    TDS_LBID_BITLEN + \
    TDS_LBTYPE_BITLEN + \
    TDS_DTYPE_BITLEN + \
    TDS_LBSIZE_BITLEN \
        )

// Table Entry
#define TDS_TDVER_BITLEN (16)
#define TDS_ENTCT_BITLEN (16)
#define TDS_CTSIZE_BITLEN (4)

// Raw Data Types
#define TDS_DTYPE_INT (0)
#define TDS_DTYPE_STRING (1)
#define TDS_DTYPE_BAUDOT (2)
#define TDS_DTYPE_INT_ARRAY (3)
#define TDS_DTYPE_SCODE (4)

// Data Symbol sizes
#define TDS_INT_SYM_COUNT (1)
#define TDS_CSTRING_SYM_BITLEN (8)
#define TDS_BAUDOT_SYM_BITLEN (5)
#define TDS_SCODE_SYM_BITLEN (6)

  /**************/
 /** TYPEDEFS **/
/**************/

typedef enum tds_raw_string_type_enum
{
    TDS_STRING_RAW_TYPE_ASCII,
    TDS_STRING_RAW_TYPE_BAUDOT,
    TDS_STRING_RAW_TYPE_SCODE

} TDS_RAW_STRING_TYPE_ENUM;

typedef struct tds_label_struct
{
    // Public attributes
    TDS_COLUMN_ENTRY_STRUCT sEntry;

    // Private attributes
    OSAL_LINKED_LIST_ENTRY hEntry;

    BOOLEAN bVariableData;
    TDS_RAW_STRING_TYPE_ENUM eRawStringType;
    UN32 un32SymbolCount;
    UN8 un8SymbolSize;
    UN16 un16ArrayLenSize;

    UN8 un8RowOrderId;
    /*
        But wait!  You said this isn't a sorted
        list -- doesn't this serve as a sorting attribute?

        Yeah, kinda.  Look, here's the deal.

        Let's reference the (annotated) TDS spec (SX-9845-0022)
        Section 5.2 (Table Entry Payload):

        A) In table entry payloads, labels are transmitted in
        the order in which they are listed in the table definition

        B) Labels are not transmitted in a numerical order
        based on their LBID in either the table entry message or
        the table defintion message

        Okay, so A says that whatever order labels are presented
        in (in the table definition message) is how they will
        be referenced in the table entry payload.

        B says you can't assume that labels are presented in a sorted
        order based on Label ID in the table defintion payload.

        However, the goal of this object is to centralize as much
        processing as possible, and it would be nice if it
        could tell anybody using it when table entry additions are
        a part of the current row being processed, or if they are
        the start of the next row.

        That's all this allows us to do.
    */

} TDS_LABEL_STRUCT;

typedef struct tds_in_progress_struct
{
    TDS_TABLE_DEF_VERSION tDefVer;
    TDS_TABLE_CONTENT_VERSION tContentVer;
    TDS_UTIL_TABLE_INSTANCE hTable;
    size_t tNumEntryMessagesLeft;

    // LL tracks the label definitions
    // for this table definition, if known
    OSAL_OBJECT_HDL hLabels;

    // LL tracks the content messages
    // we have already processed
    // for this update
    OSAL_OBJECT_HDL hContentMsgs;

    // The smallest symbol size this table
    // utilizes
    size_t tSmallestSymbolSizeInBits;
    BOOLEAN bSmallestSizeKnown;

} TDS_IN_PROGRESS_STRUCT;

typedef struct tds_table_tracking_struct
{
    TDS_TABLE_DESC_STRUCT sDesc;
    TDS_IN_PROGRESS_STRUCT sInProgress;

    // LL Entry handle for this entry
    OSAL_LINKED_LIST_ENTRY hEntry;

} TDS_TABLE_TRACKING_STRUCT;

typedef struct tds_scode_struct
{
    // Lookup table
    const char **ppacSCodeLUT;

    // Size of that table
    size_t tNumEntriesInLUT;

} TDS_SCODE_STRUCT;

typedef struct tds_object_struct
{
    SMS_OBJECT hRowDataParent;

    // CRC handle (used to compute ISO 3309 CRC32 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    OSAL_OBJECT_HDL hTables;

    BOOLEAN bRestrictTables;

    // Callback definitions from client
    TDS_CALLBACKS_STRUCT sCallbacks;

    // SCode-related info
    TDS_SCODE_STRUCT sScodeLUT;

} TDS_OBJECT_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Private Prototypes */

static BOOLEAN bAddTableToTrackingList (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_DESC_STRUCT *psTableDesc
        );

static BOOLEAN bCreateTable (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    TDS_TABLE_DEF_VERSION tTabDefVer,
    TDS_TABLE_TIME tTabTime
        );

static void vResetInProgressTable (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable
        );

static BOOLEAN bTableChangeover (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable
        );

static TDS_TABLE_TRACKING_STRUCT *psGetTableById (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_ID tId
        );

static BOOLEAN bProcessDefinitionMsg (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_DEF_VERSION tNewDefVer,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessContentMsg (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_DEF_VERSION tNewDefVer,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessLabels (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessLabelData (
    TDS_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    TDS_LABEL_STRUCT *psLabel
        );

static BOOLEAN bProcessStringData (
    TDS_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TDS_LABEL_STRUCT *psLabel
        );

static BOOLEAN bFieldPresent(
    TDS_IN_PROGRESS_STRUCT *psInProgress,
    OSAL_BUFFER_HDL hBuffer,
    BOOLEAN *pbBufferEmpty
        );

static BOOLEAN bIsProcessedMsg (
    TDS_TABLE_TRACKING_STRUCT *psTable,
    size_t tMsgId
        );

static BOOLEAN bMarkMsgProcessed (
    TDS_TABLE_TRACKING_STRUCT *psTable,
    size_t tMsgId
        );

// Linked list

static N16 n16CompareTables(
    TDS_TABLE_TRACKING_STRUCT *psTable1,
    TDS_TABLE_TRACKING_STRUCT *psTable2
        );

static BOOLEAN bRemoveTableEntry(
    TDS_TABLE_TRACKING_STRUCT *psTable,
    TDS_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRemoveLabel (
    TDS_LABEL_STRUCT *psLabel,
    void *pvUnused
        );

  /***************/
 /** VARIABLES **/
/***************/

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _TDS_UTIL_H_
