/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _MOVIES_PVN1_H_
#define _MOVIES_PVN1_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "rfd_interface_obj.h"
#include "baudot.h"
#include "movies_db_constants.h"
#include "movies_interface.h"
#include "theater_times_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

// As defined in rfd_receiver_config.h
#define MOVIES1_RFD_CLIENT_ID    (1)

#define MOVIES1_RFD_UPDATE_CAR_ID   (2)
#define MOVIES1_RFD_METADATA_CAR_ID (3)

/* Data Services */
#define MOVIES1_DSI                 (610)
#define MOVIES1_PVN                 (1)
#define MOVIES1_OBJECT_NAME         "MOVIES1"

/* Movies Data Service Protocol related constants */

#define MOVIES1_CRC_BYTELEN             (4)
#define MOVIES1_PVN_BITLEN              (4)
#define MOVIES1_CAROUSEL_BITLEN         (3)
#define MOVIES1_VER_BITLEN              (4)
#define MOVIES1_STATEID_BITLEN          (7)
#define MOVIES_STATEID_PEEK_INDEX \
    (MOVIES1_PVN_BITLEN + MOVIES1_CAROUSEL_BITLEN + MOVIES1_VER_BITLEN)

#define MOVIES1_PRESENT_FLAG_BITLEN     (1)

#define MOVIES1_RSYSVER_BITLEN          (4)
#define MOVIES1_LANG_BITLEN             (6)
#define MOVIES1_MAPNO_BITLEN            (4)
#define MOVIES1_RMAP_BITLEN             (4)
#define MOVIES1_SCODE_BITLEN            (3)
#define MOVIES1_MAX_RATING_MAPPINGS     (16)
#define MOVIES1_MAX_RATING_EXCEPTIONS   (256)

#define MOVIES1_MVID_BITLEN             (8)
#define MOVIES1_RCODE_BITLEN            (4)
#define MOVIES1_MVRUN_BITLEN            (5)
#define MOVIES1_MVRUN_BASE_MINUTES      (60)
#define MOVIES1_MVRUN_INCR_MINUTES      (5)
#define MOVIES1_MORESIZE_BITLEN         (8)

#define MOVIES1_CTSIZE_BITLEN           (4)
#define MOVIES1_REXCEPT_BITLEN          (8)
#define MOVIES1_EPOCH_BITLEN            (16)
#define MOVIES1_ETIME_BITLEN            (5)
#define MOVIES1_THIDSIZE_BITLEN         (4)
#define MOVIES1_MVNO_BITLEN             (5)
#define MOVIES1_SHOWNO_BITLEN           (4)
#define MOVIES1_ESHOWNO_BITLEN          (4)
#define MOVIES1_ESHOWNO_PRESENT_VALUE   (0xF)
#define MOVIES1_MAX_NUM_TIMES_FOR_SHOWNO (16)
#define MOVIES1_NO_SHOWTIMES_VALUE      (0xF)
#define MOVIES1_TIME_BITLEN             (6)
#define MOVIES1_EXT_TIME_VALUE          (0x3F)

#define MOVIES1_UFILE_BITLEN            (2)
#define MOVIES1_THVER_BITLEN            (8)
#define MOVIES1_NSTATE_BITLEN           (7)
#define MOVIES1_UTYPE_BITLEN            (2)
#define MOVIES1_STATE_BITLEN            (7)
#define MOVIES1_THSIZE_BITLEN           (4)
#define MOVIES1_THLAT_BITLEN            (20)
#define MOVIES1_THLON_BITLEN            (20)
#define MOVIES1_ZTYPE_BITLEN            (1)
#define MOVIES1_ZTYPE_INTEGER           (0)
#define MOVIES1_ZTYPE_BAUDOT            (1)
#define MOVIES1_MAX_ZIP_INT_CHARS       (6)
#define MOVIES1_ZIP_INTEGER_BITLEN      (17)
#define MOVIES1_THZIP_NUM_SYMBOLS       (15)
#define MOVIES1_MAX_PHONE_CHARS         (13)
#define MOVIES1_PHONE_AREA_BITLEN       (10)
#define MOVIES1_PHONE_EXCH_BITLEN       (10)
#define MOVIES1_PHONE_NUM_BITLEN        (14)
#define MOVIES1_AMEN_FIELD_BITLEN       (2)
#define MOVIES1_EXTENSION_SIZE_BITLEN   (8)

#define MOVIES1_LANG_ENGLISH_MASK       (0x20)
#define MOVIES1_LANG_SPANISH_MASK       (0x10)
#define MOVIES1_LANG_FRENCH_MASK        (0x08)
#define MOVIES1_LANG_RESERVED1_MASK     (0x04)
#define MOVIES1_LANG_RESERVED2_MASK     (0x02)
#define MOVIES1_LANG_RESERVED3_MASK     (0x01)

#define MOVIES1_NUM_LANGUAGES           (6)

#define MOVIES1_MAX_STEXT_SYMBOLS      (10)
#define MOVIES1_MAX_RTEXT_SYMBOLS      (10)

// Calculated by the max size of an update
// entry, rounded up to the nearest power of 2
#define MOVIES1_RFD_READ_BLOCK_SIZE     (2048)

#define MOVIES1_LAT_LON_DIVSOR_VALUE    (8192)
#define MOVIES1_LON_BASE_VALUE          (-50)

// Constant which defines the bit width of the maximum file update
// version, which is then passed to RFD_INTERFACE_hConnect.
// This is used in determining the applicability of incoming
// RFD updates.
// Note: 99 => 0110 0011 =  7 bits
#define MOVIES1_MAX_VERSION_BITLEN ( 7 )

/* Max message sizes (from SX-9845-0027)
*   Section 4.3 says:
*       Access Units for the Movie Description Carousel and the Movie Times
*       Carousel have a maximum size of 4096 bytes.  Access Units for the
*       Movie Theater Database Update RFD block messages have a maximum size
*       of 5120 bytes.
*
*   There is also the RFD metadata message carousel, but no info
*   on the max size there, so we'll play it safe and say it's the same
*   as the block carousel.
*
*   In addition, let's make sure we have enough memory for
*   a few queued times/desc messages -- let's say 4.
*/

// Bosch ID#5: Increase the payload size as discussed with SXM
#define MOVIES1_TIMES_PAYLOAD_MAX_SIZE (4096 * 4 * 10)
#define MOVIES1_DESC_PAYLOAD_MAX_SIZE (4096 * 4)
#define MOVIES1_RFD_META_PAYLOAD_MAX_SIZE (5120)
#define MOVIES1_RFD_BLOCK_PAYLOAD_MAX_SIZE (5120)

  /**************/
 /** TYPEDEFS **/
/**************/

// Structure which groups ratings mapping
// data for parsing movie descriptions
typedef struct movie_ratings_mapping_struct
{
    // These two values are populated by the
    // description header
    UN8 un8RMap;
    MOVIE_RATING_SYS_CODE tSCode;
    
} MOVIE_RATINGS_MAPPING_STRUCT;

typedef struct movie_description_ctrl_struct
{
    MOVIE_VERSION tCurAccessUnitVersion;
    UN16 un16CurAccessUnitAUTOT;
    UN16 un16CurAccessUnitAUNUM;

    // The languages that the current
    // description message contains
    UN8 un8LangMap;
    
    // Array of rating mappings
    MOVIE_RATINGS_MAPPING_STRUCT asMappings[MOVIES1_MAX_RATING_MAPPINGS];
    UN8 un8NumRatingsMappings;

    // Array of movie ratings used for each movie listed
    // in the description message
    MOVIE_RATING_STRUCT asRatings[MOVIES1_MAX_RATING_MAPPINGS];

    MOVIE_VERSION tCurVersion;
    size_t tAUVectorSize;
    UN8 *pun8AUVector;

} MOVIE_DESCRIPTION_CTRL_STRUCT;

typedef struct movie_times_state_carousel_struct
{
    MOVIE_STATE_ID tID;
    BOOLEAN bActive;
    UN16 un16NumAUs;
    size_t tAUVectorSize;
    UN8 *pun8AUVector;

} MOVIE_TIMES_STATE_CAROUSEL_STRUCT;

typedef struct movie_times_ratings_exception_struct
{
    MOVIE_ID tMovieID;
    MOVIE_RATING_CODE tRating;
} MOVIE_TIMES_RATINGS_EXCEPTION_STRUCT;

typedef struct movie_times_ctrl_struct
{
    MOVIE_VERSION tCurAccessUnitVersion;
    MOVIE_STATE_ID tCurAccessUnitStateID;
    UN16 un16CurAccessUnitAUTOT;
    UN16 un16CurAccessUnitAUNUM;
    UN16 un16CurAccessUnitEPOCH;
    UN8 un8CurAccessUnitETIME;
    UN8 un8CurAccessUnitTHIDSIZE;
    MOVIE_RATINGS_MAPPING_STRUCT *psRatings;

    MOVIE_VERSION tCurVersion;
    OSAL_OBJECT_HDL hStateCarouselList;

    UN16 un16NumExceptions;
    MOVIE_TIMES_RATINGS_EXCEPTION_STRUCT asExceptions[MOVIES1_MAX_RATING_EXCEPTIONS];

} MOVIE_TIMES_CTRL_STRUCT;

typedef struct movies1_object_struct
{
    // Handle to the top-level movies service
    MOVIES_SERVICE_OBJECT hMoviesService;

    // CRC handle (used to compute CCITT-CRC16 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    MOVIE_DESCRIPTION_CTRL_STRUCT sDescCtrl;
    MOVIE_TIMES_CTRL_STRUCT sTimesCtrl;

    RFD_INTERFACE_OBJECT hRFD;

} MOVIES1_OBJECT_STRUCT;

typedef enum movie1_ufile_type_enum
{
    MOVIE1_UFILE_TYPE_EMPTY = 0,
    MOVIE1_UFILE_TYPE_ONLY_THEATERS,
    MOVIE1_UFILE_TYPE_ONLY_RATINGS,
    MOVIE1_UFILE_TYPE_THEATERS_AND_RATINGS,

} MOVIE1_UFILE_TYPE_ENUM;

typedef struct movies_rfd_ctrl_struct
{
    // Handle to the top-level movies service
    MOVIES_SERVICE_OBJECT hMoviesService;

    RFD_INTERFACE_OBJECT hRFD;

    // Our path / connection to the DB under construction
    SQL_INTERFACE_OBJECT hDBConnection;
    STRING_OBJECT hDBPath;

    // Progress tracking
    RFD_UPDATE_VERSION tNewVersion;
    RFD_PROGRESS_INDEX tCurProgressIndex;
    RFD_PROGRESS_INDEX tStartProgressIndex;
    UN8 un8NumEntriesInUpdate;

    // The block pool used to process RFD files
    OSAL_OBJECT_HDL hBlockPool;

    // The buffer used to access the block pool
    OSAL_BUFFER_HDL hBuffer;

    // Indicates what type of update we're processing
    MOVIE1_UFILE_TYPE_ENUM eFileType;

    // Fixed objects to aid math operations
    OSAL_FIXED_OBJECT hDegDivsor;
    OSAL_FIXED_OBJECT hLonBase;

    // Command buffer used to
    // generate comments
    char acSQLCommandBuffer[MOVIES_MAX_SQL_STRING_LENGTH];

} MOVIES1_RFD_CTRL_STRUCT;

typedef enum movie1_utype_enum
{
    MOVIE1_UTYPE_DELETE = 0,
    MOVIE1_UTYPE_NEW,
    MOVIE1_UTYPE_MODIFY,
    MOVIE1_UTYPE_END_OF_LIST,

} MOVIE1_UTYPE_ENUM;

  /************/
 /** MACROS **/
/************/

// Convert the two parts of the movie times header
// into a unixtime (seconds since 1-1-1970).  This converted
// times should be treated as the local time for the location
// of the theater that the movie is playing in.
// This macro uses the SECONDS_IN_DAY and SECONDS_IN_HOUR
// constants to get work done
#define EPOCH_TO_UNIXTIME(epoch, etime) \
    ((UN32)((epoch*SECONDS_IN_DAY) + (etime * SECONDS_IN_HOUR)))

#define MAKE_UNIQUE_THID(stateId, theaterID)    ((stateId << 16) | theaterID)

  /****************/
 /** PROTOTYPES **/
/****************/

// Interface APIs

static size_t tMinimumOTABufferByteSize (
    BOOLEAN bDBUpdatesEnabled
        );

static MOVIES_INTERFACE_OBJECT hInit (
    MOVIES_SERVICE_OBJECT hMoviesService,
    SMS_OBJECT hParent,
    BOOLEAN bDBUpdatesEnabled,
    UN8 un8CurDBVersion
        );

static MOVIE_MESSAGE_TYPE_ENUM eParseMessageHeader (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    OSAL_BUFFER_HDL hPayload,
    MOVIE_VERSION *ptVersion
        );

static MOVIE_OBJECT hParseDescription (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    SMS_OBJECT hMovieParent,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN *pbMoreData
        );

static BOOLEAN bMarkDescriptionPayloadAsParsed (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    BOOLEAN *pbCarouselComplete
        );

static BOOLEAN bResetDescriptionPayloadTracking (
    MOVIES_INTERFACE_OBJECT hInterfaceObj
        );

static BOOLEAN bParseTimes (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    SMS_OBJECT hTimesParent,
    OSAL_BUFFER_HDL hPayload,
    THEATER_TIMES_OBJECT *phTheaterTimes,
    THEATER_ID *ptTHID,
    BOOLEAN *pbMoreData
        );

static BOOLEAN bMarkTimesPayloadAsParsed (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    BOOLEAN *pbCarouselComplete
        );

static BOOLEAN bResetTimesPayloadTracking (
    MOVIES_INTERFACE_OBJECT hInterfaceObj
        );

static void vUninit (
    MOVIES_INTERFACE_OBJECT hInterfaceObj
        );

// Private APIs

static MOVIES1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    MOVIES1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitRFDCtrl (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        );

static void vDestroyRFDCtrl (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bParseDescriptionHeader (
    MOVIES1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN *pbShouldProcess
        );

static BOOLEAN bParseTimesHeader (
    MOVIES1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN *pbShouldProcess
        );

static BOOLEAN bInitDescAUVector (
    MOVIES1_OBJECT_STRUCT *psObj,
    UN16 un16NumAUs
        );

static BOOLEAN bFindStateAUEntry (
    MOVIES1_OBJECT_STRUCT *psObj,
    MOVIE_STATE_ID tStateID,
    UN16 un16AUSeqNum,
    BOOLEAN bMatchAUSeqNum,
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT **ppsEntry
        );

static BOOLEAN bMarkTimesAUNumAsParsed(
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry,
    UN16 un16NumAUs,
    UN16 un16AUSeqNum,
    BOOLEAN *pbCarouselComplete
        );

static BOOLEAN bGetAUNumParsed(
    UN8 *pun8AUList,
    UN16 un16NumAUs,
    UN16 un16AUSeqNum
        );

static BOOLEAN bSetAUNumAsParsed(
    UN8 *pun8AUList,
    UN16 un16NumAUs,
    UN16 un16AUSeqNum,
    BOOLEAN *pbCarouselComplete
        );

static BOOLEAN bResetStateAUEntryIterator (
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry,
    void *pvArg
        );

static void vReleaseStateCarEntry (
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry
        );

static N16 n16CompareStateCarEntry (
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry1,
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry2
        );

static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        );

static BOOLEAN bProcessRFDHeader (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        );

static BOOLEAN bProcessRatings (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static BOOLEAN bProcessRatingsSystem (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    MOVIE_RATINGS_ROW_STRUCT *psRatingsRow,
    FILE *psFile
        );

static BOOLEAN bProcessState (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static BOOLEAN bProcessTheatersForState (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager,
    FILE *psFile,
    UN8 un8State,
    UN8 un8THSIZE,
    RFD_UPDATE_VERSION tVersion
        );

static BOOLEAN bProcessAmenities (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bAmenities1,
    UN16 *pun16Amenities
        );

static BOOLEAN bFieldPresent(
    OSAL_BUFFER_HDL hBuffer
        );

static void vDeleteDB (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bConnectToDB (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        );

static BOOLEAN bFinalizeDB (
   MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl
        );

  /***************/
 /** VARIABLES **/
/***************/

const MOVIES_PLUGIN_INTERFACE_STRUCT GsMoviesIntf =
{
    /*.tDSI = */MOVIES1_DSI,
    /*.tMaxVersionBitlen = */MOVIES1_MAX_VERSION_BITLEN,
    /*.tMinimumOTABufferByteSize = */tMinimumOTABufferByteSize,
    /*.hInit = */hInit,
    /*.eParseMessageHeader = */eParseMessageHeader,
    /*.hParseDescription = */hParseDescription,
    /*.bMarkDescriptionPayloadAsParsed = */bMarkDescriptionPayloadAsParsed,
    /*.bResetDescriptionPayloadTracking = */bResetDescriptionPayloadTracking,
    /*.bParseTimes = */bParseTimes,
    /*.bMarkTimesPayloadAsParsed = */bMarkTimesPayloadAsParsed,
    /*.bResetTimesPayloadTracking = */bResetTimesPayloadTracking,
    /*.vUninit = */vUninit
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _MOVIES_PVN1_H_
