/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _FUEL_PVN1_H_
#define _FUEL_PVN1_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include <stdio.h>

#include "sms_api.h"
#include "rfd_interface_obj.h"
#include "fuel_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

// PVN 1 - Definitions
#define FUEL1_OBJECT_NAME "Fuel1"

// The protocol version supported
// by SMS for this interface
#define FUEL1_PVN (PVN)(1)
#define FUEL1_DSI (DSI)(600)
#define FUEL1_CAN_DSI (DSI)(601)

#define FUEL1_UNBRANDED_TEXT "Brand Unknown"
#define FUEL1_UNBRANDED_ID (0)

// General Carousel Definitions

// Carousel IDs
#define FUEL1_PRICE_CAROUSEL_ID (0)
#define FUEL1_DB_UPDATE_CAROUSEL_ID (1)
#define FUEL1_TEXT_CAROUSEL_ID (2)
#define FUEL1_METADATA_CAROUSEL_ID (3)

// As defined in rfd_receiver_config.h
#define FUEL1_RFD_CLIENT_ID    (0)
#define FUEL1_RFD_CAN_CLIENT_ID    (9)

// General Access Unit Definitions
#define FUEL1_PVN_BITLEN (4)
#define FUEL1_CAROUSEL_BITLEN (3)
#define FUEL1_REGION_BITLEN (11)
#define FUEL1_FSSIZE_BITLEN (4)
#define FUEL1_CRC_BYTELEN (4)
#define FUEL1_PRESENCE_FLAG_BITLEN (1)
#define FUEL1_TRIM_BITLEN (39)

// Text Message Definitions
#define FUEL1_TEXT_VER_BITLEN (8)
#define FUEL1_TEXT_NUM_TYPES_BITLEN (3)
#define FUEL1_TEXT_RSVD_BITLEN (6)
#define FUEL1_TEXT_LANG_BITLEN (3)
#define FUEL1_TEXT_NUM_ENTRIES_BITLEN (6)
#define FUEL1_TEXT_TYPE_BITLEN (6)

// Station Description Definitions
#define FUEL1_DESC_REG_COUNT_BITLEN (11)
#define FUEL1_DESC_UPDATE_TYPE_BITLEN (2)
#define FUEL1_DESC_AUNUM_BITLEN (8)
#define FUEL1_DESC_AUTOTAL_BITLEN (8)
#define FUEL1_LOC_BITLEN (2)
#define FUEL1_DESC_STATE_BITLEN (7)
#define FUEL1_ZIP_BITLEN (17)
#define FUEL1_ZIP_TYPE_BITLEN (1)
#define FUEL1_AMEN_FIELD_BITLEN (2)
#define FUEL1_EXTENSION_SIZE_BITLEN (8)
#define FUEL1_DCODE_ENTRY_BITLEN ( \
    FUEL1_DCODE_BITLEN + \
    FUEL1_DCODE_ARG_BITLEN \
    )
#define FUEL1_UPDATE_ENTRY_VER_BITLEN (7)

// ZIP / Postal Code Types
#define FUEL1_ZIP_TYPE_INTEGER (0)
#define FUEL1_ZIP_TYPE_STRING (1)
#define FUEL1_MAX_ZIP_INT_CHARS (6)

// Location code types / info
#define FUEL1_LOC_PREV (0)
#define FUEL1_LOC_SUBDIV (1)
#define FUEL1_LOC_DIV (2)
#define FUEL1_LOC_SUBREG (3)
#define FUEL1_LOC_SUBDIV_BITLEN (12)
#define FUEL1_LOC_DIV_BITLEN (8)
#define FUEL1_LOC_SUBREG_BITLEN (8)

// Price Update Definitions
#define FUEL1_PRICE_CTSIZE_BITLEN (4)
#define FUEL1_PRICE_FUEL_TYPE_BITLEN (6)
#define FUEL1_PRICE_FPSCALE_BITLEN (2)
#define FUEL1_PRICE_FPMIN_BITLEN (12)
#define FUEL1_PRICE_FPRNG_BITLEN (4)
#define FUEL1_PRICE_HDR_ENTRY_BITLEN ( \
    FUEL1_PRICE_FUEL_TYPE_BITLEN + \
    FUEL1_PRICE_FPSCALE_BITLEN   + \
    FUEL1_PRICE_FPMIN_BITLEN     + \
    FUEL1_PRICE_FPRNG_BITLEN       \
    )
#define FUEL1_PRICE_VLC_BITLEN (1)
#define FUEL1_PRICE_MARKER_BITLEN (1)
#define FUEL1_PRICE_AGE_BITLEN (2)
#define FUEL1_PRICE_UNAVAIL_BASE_FLAG (0x1FFFF)
#define FUEL1_PRICE_UNAVAIL_BASE_FLAG_BITLEN (17)

// Brand Table Definitions
#define FUEL1_BRNUM_BITLEN (7)
#define FUEL1_BRCODE_BITLEN (7)
#define FUEL1_BRAND_TABLE_BYTE_SIZE(num_entries) \
    ( ((num_entries) * \
       (((FUEL1_BRCODE_BITLEN + FUEL1_BRTEXT_MAX_BITLEN) / 8) + 1)))

// RX 185 Section 6.2.12
#define FUEL_PRICE_FPRNG_ADJUSTMENT (2)

// The maximum number of bits used
// in the VL code
#define FUEL1_PRICE_MAX_VLC_CODE_LEN (6)

// Lat / Lon calculation values
#define FUEL1_INITIAL_LAT (12)
#define FUEL1_INITIAL_LON (-168)
#define FUEL1_COMPUTE_DENOMINATOR (8192)
#define FUEL1_REG_SIZE (2)
#define FUEL1_INPUT_FIXED_BITS (31)

// Region grid info
#define FUEL1_REG_GRID_COL_HEIGHT (32)
#define FUEL1_NE_BEARING (45)

// Macros used to pull offsets / grid info from field values
#define FUEL1_REG_LAT_OFFSET(x) ((x) & 0x1F)
#define FUEL1_REG_LON_OFFSET(x) (((x) >> 5) & 0x3F)

// Macro used to build a region from the offsets
#define FUEL1_REG_FROM_LAT_OFFSET(x) FUEL1_REG_LAT_OFFSET(x)
#define FUEL1_REG_FROM_LON_OFFSET(x) (((x) & 0x3F) << 5)

// Col is used for Lat
#define FUEL1_GET_LAT(x) ((x) & 0x0F)
#define FUEL1_GET_SUBDIV_LAT(x) ((x) & 0x3F)

// Row is used for Lon
#define FUEL1_GET_LON(x) (((x) >> 4) & 0x0F)
#define FUEL1_GET_SUBDIV_LON(x) ((x >> 6) & 0x3F)

// Constants used in the utilization of fixed objects
#define FUEL1_LAT_OBJ_INDEX (0)
#define FUEL1_LON_OBJ_INDEX (1)
#define FUEL1_MAX_OBJ_INDEX (FUEL1_LON_OBJ_INDEX)
#define FUEL1_NUM_MEMORY_FIXED (FUEL1_MAX_OBJ_INDEX + 1)

// The number of well known (and persistent) fuel types
// From RX185, Section 5.2.3.1
#define FUEL1_NUM_WELL_KNOWN_FUEL_TYPES (4)

// Localization definitions
#define FUEL1_ENGLISH (0)
#define FUEL1_SPANISH (1)
#define FUEL1_FRENCH (2)
#define FUEL1_END_OF_LANG_LIST (7)

// Constant defined for amenities fields
#define FUEL1_INITIAL_AMENITIES 'U'

// Calculated by the max size of an update
// entry, rounded up to the nearest power of 2
#define FUEL1_RFD_READ_BLOCK_SIZE (2048)

// Amenities processing
#define FUEL1_STATION_BASE_AMENITY (FUEL_STATION_AMENITY_OPEN_24HR)
#define FUEL1_STATION_MAX_AMENITY (FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL)
#define FUEL1_STATION_AMENITY_BITMASK (0x3)
#define FUEL1_AMENITY_AVAILABLE (1)
#define FUEL1_AMENITY_SECONDARY_AVAILABLE (2)
#define FUEL1_AMENITY_NOT_AVAILABLE (3)
#define FUEL1_24HR_AMENITY_OFFSET (0)
#define FUEL1_EMERGENCY_AMENITY_OFFSET (2)
#define FUEL1_SERVICE_AMENITY_OFFSET (4)
#define FUEL1_OILCHANGE_AMENITY_OFFSET (4)
#define FUEL1_UNDERHALF_AMENITY_OFFSET (6)
#define FUEL1_OVERHALF_AMENITY_OFFSET (6)
#define FUEL1_DISCOUNT_AMENITY_OFFSET (8)
#define FUEL1_STORE_AMENITY_OFFSET (10)
#define FUEL1_SUPERMKT_AMENITY_OFFSET (10)
#define FUEL1_SNACKS_AMENITY_OFFSET (12)
#define FUEL1_RESTAURANT_AMENITY_OFFSET (12)
#define FUEL1_TRUCKSTOP_AMENITY_OFFSET (14)
#define FUEL1_HOTEL_AMENITY_OFFSET (14)

// Constant which defines the bit width of the maximum file update
// version, which is then passed to RFD_INTERFACE_hConnect.
// This is used in determining the applicability of incoming
// RFD updates.
// Note: 99 => 0110 0011 =  7 bits
#define FUEL1_MAX_VERSION_BITLEN ( 7 )

/* Max message sizes (from SX-9845-0023)
*   Section 4.3 says:
*       Access Units for the Service Text Carousel and Fuel Price Carousel
*       have a maximum size of 4096 bytes. Access Units for the Fuel Station
*       Database Update RFD block messages have a maximum size of 5120 bytes.
*
*   There is also the RFD metadata message carousel, but no info
*   on the max size there, so we'll play it safe and say it's the same
*   as the block carousel.
*
*   In addition, let's make sure we have enough memory for
*   a few queued price messages -- let's say 8 (they come in fast)
*/
#define FUEL1_TEXT_PAYLOAD_MAX_SIZE (4096)
#define FUEL1_PRICE_PAYLOAD_MAX_SIZE (4096 * 8)
#define FUEL1_RFD_META_PAYLOAD_MAX_SIZE (5120)
#define FUEL1_RFD_BLOCK_PAYLOAD_MAX_SIZE (5120)

// Buffer size needed for this DSI
#define FUEL1_OTA_BUFFER_SIZE ( \
    FUEL1_TEXT_PAYLOAD_MAX_SIZE + \
    FUEL1_PRICE_PAYLOAD_MAX_SIZE + \
    FUEL1_RFD_META_PAYLOAD_MAX_SIZE + \
    FUEL1_RFD_BLOCK_PAYLOAD_MAX_SIZE \
    )

  /**************/
 /** TYPEDEFS **/
/**************/

// Update types
typedef enum fuel1_update_type_enum
{
    FUEL1_UPDATE_TYPE_DELETE = 0,
    FUEL1_UPDATE_TYPE_NEW,
    FUEL1_UPDATE_TYPE_MODIFY,
    FUEL1_UPDATE_TYPE_END,
    FUEL1_INVALID_UPDATE_TYPE

} FUEL1_UPDATE_TYPE_ENUM;

// Structure utilized for text update work
typedef struct fuel1_text_work_struct
{
    FUEL_TEXT_ROW_STRUCT sText;
    UN8 un8Version;
    UN8 un8NumEntries;

} FUEL1_TEXT_WORK_STRUCT;

// Enumeration for the ways in which
// price values are scaled in price messages
typedef enum fuel1_price_scaling_factor_enum
{
    FUEL1_PRICE_SCALING_FACTOR_NONE = 0,
    FUEL1_PRICE_SCALING_FACTOR_UNDEF = 1,
    FUEL1_PRICE_SCALING_FACTOR_10X = 2,
    FUEL1_PRICE_SCALING_FACTOR_ONE_TENTH = 3,
    FUEL1_PRICE_SCALING_MAX_FACTORS

} FUEL1_PRICE_SCALING_FACTOR_ENUM;

// Structure utilized to aid in the processing
// of price updates
typedef struct fuel1_price_baseline_struct
{
    // General price baseline data
    UN8 un8FuelType;
    UN8 un8ScalingFactor;
    UN16 un16BasePrice;
    UN8 un8PriceSize;

    // The "out of fuel" value to
    // check against.  This varies (real smart!)
    // which is why it is here
    UN32 un32OutOfFuelCheckValue;

    // Flag indicating if this entry is static
    // or dynamically allocated
    BOOLEAN bStatic;

    // Next pointer for a very simple LL
    struct fuel1_price_baseline_struct *psNext;

} FUEL1_PRICE_BASELINE_STRUCT;

// Structure utilized for price work
typedef struct fuel1_price_work_struct
{
    // The price info which is populated
    // by this work
    FUEL_PRICE_ROW_STRUCT sPriceRow;

    // The size of the FSUID field in this work
    UN8 un8FSSize;

    // The previously processed station ID
    FUEL_STATION_ID tPrevStation;

    // The array of price baselines which is
    // used to provide prices to the manager
    FUEL1_PRICE_BASELINE_STRUCT asPriceBaselines[FUEL1_NUM_WELL_KNOWN_FUEL_TYPES];
    size_t tNumValidBaselines;

} FUEL1_PRICE_WORK_STRUCT;

typedef struct fuel1_price_dynamic_memory_struct
{
    BOOLEAN bEnabled;
    FUEL_PRICE_ENTRY_STRUCT *pasEntries;
    size_t tNumEntriesAllocated;

} FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT;

// Interface definition
typedef struct fuel1_object_struct
{
    // Handle to the fuel manager
    FUEL_SERVICE_OBJECT hFuelService;

    // Handle to the DB interface for the fuel service
    FUEL_DB_INTERFACE_OBJECT hDBInterface;

    // DB interface implementation
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface;

    // OTA Interface implementation
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface;

    // CRC handle (used to compute ISO 3309 CRC32 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    // Handle to RFD client
    RFD_INTERFACE_OBJECT hRFD;

    // Structure which groups all
    // memory which is dynamically allocated
    // to handle price processing
    FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT sPriceMem;

    // The CRC value for the current message
    UN32 un32CurrentMsgCRC;

} FUEL1_OBJECT_STRUCT;

// Structure used to group all
// fixed object handles and memory
typedef struct fuel1_fixed_calc_struct
{
    // Fixed objects created to
    // represent constant float values
    OSAL_FIXED_OBJECT hInitialLat;
    OSAL_FIXED_OBJECT hInitialLon;
    OSAL_FIXED_OBJECT hComputeDenominator;

    // A memory segment used for temporary
    // values used to compute lat/lon
    OSAL_FIXED_OBJECT_DATA atFixedData[
        OSAL_FIXED_OBJECT_SIZE * FUEL1_NUM_MEMORY_FIXED];

} FUEL1_FIXED_CALC_STRUCT;

typedef struct fuel1_rfd_ctrl_struct
{
    FUEL_DB_INTERFACE_OBJECT hDBInterface;
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface;
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface;
    RFD_INTERFACE_OBJECT hRFD;

    // Our path / connection to the DB under construction
    SQL_INTERFACE_OBJECT hDBConnection;
    STRING_OBJECT hDBPath;

    // Progress tracking
    RFD_UPDATE_VERSION tCurrentVersion;
    RFD_PROGRESS_INDEX tCurProgressIndex;
    RFD_PROGRESS_INDEX tStartProgressIndex;
    UN16 un16NumRegionsInUpdate;

    // Current region under process
    FUEL_REGION tRegion;

    // The block pool used to process RFD files
    OSAL_OBJECT_HDL hBlockPool;

    // The buffer used to access the block pool
    OSAL_BUFFER_HDL hBuffer;

    // Command buffer used to
    // generate comments
    char *pcSQLCommandBuffer;
    size_t tBufferSize;

    // The list of brand text strings
    // provided in an RFD update
    OSAL_OBJECT_HDL hBrandTexts;

    // Structure which groups all
    // fixed-object attributes used in
    // computing lat/lon
    FUEL1_FIXED_CALC_STRUCT sFixed;

} FUEL1_RFD_CTRL_STRUCT;

// Structure which defines a brand text entry
typedef struct fuel1_brand_text_struct
{
    UN8 un8Code;
    STRING_OBJECT hString;

} FUEL1_BRAND_TEXT_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

// Public interface APIs

static FUEL_OTA_INTERFACE_OBJECT hInit(
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface
        );

static void vUnInit(
    FUEL_OTA_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProductStateChange (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        );

static BOOLEAN bGetDSIForProductGeneric (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo,
    DSI tDSI
        );

static BOOLEAN bGetDSIForProduct (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        );

static BOOLEAN bGetDSIForCanadianProduct (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        );

static DATA_PRODUCT_STATE_ENUM eNextProductState (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eType,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eDSIState,
    SXM_DMI *pDMIs,
    UN8 un8DMICount
        );

static BOOLEAN bProcessMessage (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bAddRegionsForLocation (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    LOCATION_OBJECT hCenterLocation,
    void *pvArg
        );

static SMSAPI_RETURN_CODE_ENUM eProcessAmenities (
    size_t tNumAmenities,
    AMENITY_STRUCT *pasFuelAmenities,
    UN32 un32RawAmenitiesData
        );

/* Object Private Prototypes */

/* Message Processors */

static FUEL1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface
        );

static BOOLEAN bInitRFDCtrl (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        );

static void vDestroyRFDCtrl (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static void vDeleteDB (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bConnectToDB (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        );

static BOOLEAN bFinalizeDB (
   FUEL1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bRFDProductStateChange (
    FUEL1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        );

static BOOLEAN bEnableRFDUpdates (
    FUEL1_OBJECT_STRUCT *psObj
        );

static void vDisableRFDUpdates (
    FUEL1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPriceProductStateChange (
    FUEL1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        );

static BOOLEAN bProcessTextMessage (
    FUEL1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessTextMessageHeader (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_TEXT_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessTextMessageData (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_TEXT_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessPriceMessage (
    FUEL1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessPriceMessageHeader (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_PRICE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessPriceData (
    FUEL1_OBJECT_STRUCT *psObj,
    FUEL1_PRICE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessRFDHeader (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        );

static BOOLEAN bProcessRegion (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static BOOLEAN bProcessStation(
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    UN8 un8FSSize,
    UN8 *pun8LastSubRegion,
    UN8 *pun8LastDivision,
    UN16 *pun16LastSubDivision
        );

static BOOLEAN bProcessAmenities (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bAmenities1,
    UN32 *pun32Amenities
        );

static BOOLEAN bFieldPresent(
    OSAL_BUFFER_HDL hBuffer
        );

static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        );

static BOOLEAN bComputeLatLonFromGrid (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    UN8 un8SubRegion,
    UN8 un8Division,
    UN16 un16SubDivision
        );

static BOOLEAN bVerifyMessageNeeded (
    FUEL1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    UN8 *pun8CarouselID
        );

static void vReleasePriceWorkData (
    FUEL1_PRICE_WORK_STRUCT *psWork
        );

static BOOLEAN bCreateFixedConstants (
    FUEL1_FIXED_CALC_STRUCT *psFixed
        );

static void vDestroyFixedConstants (
    FUEL1_FIXED_CALC_STRUCT *psFixed
        );

static BOOLEAN bAddBrandText (
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode,
    STRING_OBJECT hString
        );

static STRING_OBJECT hGetBrandString(
    FUEL1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode
        );

static N16 n16CompareTextEntries (
    FUEL1_BRAND_TEXT_STRUCT *psText1,
    FUEL1_BRAND_TEXT_STRUCT *psText2
        );

static void vRemoveTextEntry (
    FUEL1_BRAND_TEXT_STRUCT *psText
        );

static void vInitializeStaticPriceBaselines (
    FUEL1_PRICE_WORK_STRUCT *psPriceWork
        );

static void vInitializePriceMemory (
    FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT *psPriceMem
        );

static void vUninitPriceMemory (
    FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT *psPriceMem
        );

static BOOLEAN bPreparePriceMemory (
    FUEL1_PRICE_DYNAMIC_MEMORY_STRUCT *psPriceMem,
    FUEL1_PRICE_WORK_STRUCT *psWork
        );

static FUEL1_PRICE_BASELINE_STRUCT *psCreateNewPriceBaseline (
    FUEL1_OBJECT_STRUCT *psObj
        );

static void vDestroyAllocatedPriceBaselines (
    FUEL1_PRICE_WORK_STRUCT *psPriceWork
        );

  /***************/
 /** VARIABLES **/
/***************/

const DATASERVICE_PRODUCT_DESCRIPTOR_STRUCT GasFuelProducts[] = 
{
    // Fuel Prices
    {
        DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES,
        DATA_PRODUCT_MASK_NONE
    },

    // Database updates
    {
        DATA_PRODUCT_TYPE_DB_UPDATES,
        DATA_PRODUCT_MASK_NONE
    }
};

const FUEL_OTA_INTERFACE_STRUCT GsPriceOTAIntf =
{
    /*.tDSI = */FUEL1_DSI,
    /*.tDataID = */DATASERVICE_ID_FUEL_PRICES,
    /*.tMaxVersionBitlen = */FUEL1_MAX_VERSION_BITLEN,
    /*.psProducts = */GasFuelProducts, 
    /*.un8NumProducts = */sizeof(GasFuelProducts) / sizeof(*GasFuelProducts),
    /*.un8RFDClientId = */FUEL1_RFD_CLIENT_ID,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProductStateChange = */bProductStateChange,
    /*.bGetDSIForProduct = */bGetDSIForProduct,
    /*.eNextProductState = */eNextProductState,
    /*.bProcessMessage = */bProcessMessage,
    /*.hAddRegionsForLocation = */bAddRegionsForLocation,
    /*.eProcessAmenities = */eProcessAmenities
};

const FUEL_OTA_INTERFACE_STRUCT GsCanadianPriceOTAIntf =
{
    /*.tDSI = */FUEL1_CAN_DSI,
    /*.tDataID = */DATASERVICE_ID_CANFUEL_PRICES,
    /*.tMaxVersionBitlen = */FUEL1_MAX_VERSION_BITLEN,
    /*.psProducts = */GasFuelProducts,
    /*.un8NumProducts = */sizeof(GasFuelProducts) / sizeof(*GasFuelProducts),
    /*.un8RFDClientId = */FUEL1_RFD_CAN_CLIENT_ID,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProductStateChange = */bProductStateChange,
    /*.bGetDSIForProduct = */bGetDSIForCanadianProduct,
    /*.eNextProductState = */eNextProductState,
    /*.bProcessMessage = */bProcessMessage,
    /*.hAddRegionsForLocation = */bAddRegionsForLocation,
    /*.eProcessAmenities = */eProcessAmenities
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _FUEL_PVN1_H_
