/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _EVCHARGING_PVN1_H_
#define _EVCHARGING_PVN1_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include <stdio.h>

#include "sms_api.h"
#include "rfd_interface_obj.h"
#include "fuel_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

// PVN 1 - Definitions
#define EV1_OBJECT_NAME "EV1"

// The protocol version supported
// by SMS for this interface
#define EV1_PVN (PVN)(1)
#define EV1_BASE_DSI (DSI)(630)
#define EV1_UPDATE_DSI (DSI)(631)
#define EV1_NUM_PRODUCTS (3)

#define EV1_UNBRANDED_TEXT "Brand Unknown"
#define EV1_UNBRANDED_ID (0)

// General Carousel Definitions

// Carousel IDs
#define EV1_LOGO_UPDATE_CAROUSEL_ID (1)
#define EV1_STATION_CAROUSEL_ID (2)
#define EV1_METADATA_CAROUSEL_ID (3)
#define EV1_CHARGER_AVAILABILITY_1_CAROUSEL_ID (4)
#define EV1_CHARGER_AVAILABILITY_2_CAROUSEL_ID (5)
#define EV1_CHARGER_TYPE_CAROUSEL_ID (6)

// As defined in rfd_receiver_config.h
#define EV1_RFD_CLIENT_ID    (6)

// General Access Unit Definitions
#define EV1_PVN_BITLEN (4)
#define EV1_CAROUSEL_BITLEN (3)
#define EV1_REGION_BITLEN (11)
#define EV1_CSSIZE_BITLEN (4)
#define EV1_CRC_BYTELEN (4)
#define EV1_PRESENCE_FLAG_BITLEN (1)
#define EV1_TRIM_BITLEN (39)

// Logo Message Definitions
#define EV1_TEXT_VER_BITLEN (8)
#define EV1_TEXT_NUM_TYPES_BITLEN (3)
#define EV1_TEXT_RSVD_BITLEN (6)
#define EV1_TEXT_LANG_BITLEN (3)
#define EV1_TEXT_NUM_ENTRIES_BITLEN (6)
#define EV1_TEXT_TYPE_BITLEN (6)

// Charger Availability Definitions
#define EV1_AVAIL_TIME_BITLEN (11)
#define EV1_AVAIL_CT_AUVER_BITLEN (8)
#define EV1_AVAIL_CSSIZE_BITLEN (4)
#define EV1_AVAIL_ATYPESIZE_BITLEN (3)
#define EV1_AVAIL_CHG_TYPESIZE_BITLEN (3)
#define EV1_AVAIL_NUMSIZE_BITLEN (4)
#define EV1_AVAIL_CTSIZE_BITLEN (4)
#define EV1_AVAIL_MAX_VLC_CODE_LEN (6)
#define EV1_AVAIL_VLC_BITLEN (1)
#define EV1_AVAIL_ENO_BITLEN (3)
#define EV1_AVAIL_EXT_SIZE_BITLEN (7)
#define EV1_AVAIL_MAX_TYPES (1 << EV1_AVAIL_ENO_BITLEN)

// Charger Type Definitions
#define EV1_CHGTYPE_CT_AUVER_BITLEN (8)
#define EV1_CHGTYPE_ATYPESIZE_BITLEN (3)
#define EV1_CHGTYPE_NUM_BITLEN (6)
#define EV1_CHGTYPE_CHTYPE_BITLEN (6)
#define EV1_CHGTYPE_MAX_ATYPESIZE ((1 << EV1_CHGTYPE_ATYPESIZE_BITLEN ) - 1)
#define EV1_CHGTYPE_MAX_MAPPINGS (1 << EV1_CHGTYPE_MAX_ATYPESIZE)

// Station Description Definitions
#define EV1_DESC_REG_COUNT_BITLEN (11)
#define EV1_DESC_UPDATE_TYPE_BITLEN (2)
#define EV1_DESC_AUNUM_BITLEN (8)
#define EV1_DESC_AUTOTAL_BITLEN (8)
#define EV1_LOC_BITLEN (2)
#define EV1_DESC_STATE_BITLEN (7)
#define EV1_ZIP_BITLEN (17)
#define EV1_ZIP_TYPE_BITLEN (1)
#define EV1_AMEN_FIELD_BITLEN (2)
#define EV1_EXTENSION_SIZE_BITLEN (8)
#define EV1_DCODE_ENTRY_BITLEN ( \
    EV1_DCODE_BITLEN + \
    EV1_DCODE_ARG_BITLEN \
    )
#define EV1_UPDATE_ENTRY_VER_BITLEN (10)
#define EV1_ENO_BITLEN (3)

// ZIP / Postal Code Types
#define EV1_ZIP_TYPE_INTEGER (0)
#define EV1_ZIP_TYPE_STRING (1)
#define EV1_MAX_ZIP_INT_CHARS (6)

// Location code types / info
#define EV1_LOC_PREV (0)
#define EV1_LOC_SUBDIV (1)
#define EV1_LOC_DIV (2)
#define EV1_LOC_SUBREG (3)
#define EV1_LOC_SUBDIV_BITLEN (12)
#define EV1_LOC_DIV_BITLEN (8)
#define EV1_LOC_SUBREG_BITLEN (8)

// Charge Table Definitions
#define EV1_CHTYPE_NUM_BITLEN (6)
#define EV1_CHTYPE_BITLEN (6)
#define EV1_CHNUM_BITLEN (7)
#define EV1_CVER_BITLEN (8)

// Logo Table Definitions
#define EV1_LOGOID_BITLEN (9)
#define EV1_LOGO_UVER_BITLEN (8)
#define EV1_LOGO_TYPE_BITLEN (2)
#define EV1_LOGO_OPER_BITLEN (1)
#define EV1_LOGO_FVER_BITLEN (8)
#define EV1_LOGO_CTSIZE_BITLEN (4)

// Brand Table Definitions
#define EV1_BRNUM_BITLEN (8)
#define EV1_BRCODE_BITLEN (8)
#define EV1_BRAND_TABLE_BYTE_SIZE(num_entries) \
    ( ((num_entries) * \
       (((EV1_BRCODE_BITLEN + EV1_BRTEXT_MAX_BITLEN) / 8) + 1)))

// Lat / Lon calculation values
#define EV1_INITIAL_LAT (12)
#define EV1_INITIAL_LON (-168)
#define EV1_COMPUTE_DENOMINATOR (8192)
#define EV1_REG_SIZE (2)
#define EV1_INPUT_FIXED_BITS (31)

// Region grid info
#define EV1_REG_GRID_COL_HEIGHT (32)
#define EV1_NE_BEARING (45)

// Macros used to pull offsets / grid info from field values
#define EV1_REG_LAT_OFFSET(x) ((x) & 0x1F)
#define EV1_REG_LON_OFFSET(x) (((x) >> 5) & 0x3F)

// Macro used to build a region from the offsets
#define EV1_REG_FROM_LAT_OFFSET(x) EV1_REG_LAT_OFFSET(x)
#define EV1_REG_FROM_LON_OFFSET(x) (((x) & 0x3F) << 5)

// Col is used for Lat
#define EV1_GET_LAT(x) ((x) & 0x0F)
#define EV1_GET_SUBDIV_LAT(x) ((x) & 0x3F)

// Row is used for Lon
#define EV1_GET_LON(x) (((x) >> 4) & 0x0F)
#define EV1_GET_SUBDIV_LON(x) ((x >> 6) & 0x3F)

// Constants used in the utilization of fixed objects
#define EV1_LAT_OBJ_INDEX (0)
#define EV1_LON_OBJ_INDEX (1)
#define EV1_MAX_OBJ_INDEX (EV1_LON_OBJ_INDEX)
#define EV1_NUM_MEMORY_FIXED (EV1_MAX_OBJ_INDEX + 1)

// Constant defined for amenities fields
#define EV1_INITIAL_AMENITIES 'U'

// Calculated by the max size of an update
// entry, rounded up to the nearest power of 2
#define EV1_RFD_READ_BLOCK_SIZE (2048)

#define EV1_LAST_SEQ_NO_INITIAL_VAL (-1)
#define EV1_FIRST_MULTI_AU_SEQ_NO (0)

// Amenities processing
#define EV1_STATION_BASE_AMENITY (FUEL_STATION_AMENITY_OPEN_24HR)
#define EV1_STATION_MAX_AMENITY (FUEL_STATION_AMENITY_RESERVATIONS)

#define EV1_STATION_AMENITY_BITMASK (0x3)
#define EV1_AMENITY_AVAILABLE (1)
#define EV1_AMENITY_SECONDARY_AVAILABLE (2)
#define EV1_AMENITY_NOT_AVAILABLE (3)

#define EV1_24HR_AMENITY_NOT_AVAILABLE (2)
#define EV1_CREDITDEBUG_AMENITY_NOT_AVAILABLE (2)
#define EV1_CASH_AMENITY_NOT_AVAILABLE (2)
#define EV1_RESERVATIONS_AMENITY_NOT_AVAILABLE (2)

#define EV1_24HR_AMENITY_OFFSET (0)
#define EV1_AFFIL_CARDS_AMENITY_OFFSET (2)
#define EV1_ALL_CARDS_AMENITY_OFFSET (2)
#define EV1_CREDITDEBIT_AMENITY_OFFSET (4)
#define EV1_PUBLIC_ACCESS_AMENITY_OFFSET (6)
#define EV1_PATRON_ACCESS_AMENITY_OFFSET (6)
#define EV1_CASH_AMENITY_OFFSET (8)
#define EV1_RESERVATIONS_AMENITY_OFFSET (10)

// RFD related
#define EV1_RFD_UPDATE_VER_STRING_LEN (3)
#define EV1_RFD_UPDATE_FILE_NAME_LEN (7)
#define EV1_RFD_UPDATE_FILE_START_CHAR 'U'
#define EV1_RFD_UPDATE_FILE_MAX_VERSION (999)

// Constant which defines the bit width of the maximum file update
// version, which is then passed to RFD_INTERFACE_hConnect.
// This is used in determining the applicability of incoming
// RFD updates.
// Note: 255 => 1111 1111 => 8 bits
#define EV1_MAX_VERSION_BITLEN ( 8 )

/* Max message sizes (from SX-9845-0219)
*   Section 4.3 says:
*       The maximum size of any Access Unit is 5120 bytes.
*   Section 5 says:
*       The broadcast update of a single Logo Image file for the EV Charge
*       Station Service shall require from one to a maximum of 4 Access Units.
*
*   This means that the logo carousel can take up to 5120*4 bytes.  Let's
*   have enough memory on hand to deal with 2 images in the queue.
*
*   The availability messages have a maximum AU size of 5120 (as mentioned
*   in Section 4.3, noted above).  But, the availability messages may
*   require up to 4 AUs to complete a message.  These messages are
*   carried on 2 carousels, and we want to be able to queue 2 of
*   these maximally sized messages on each carousel.
*
*   The charger type messages have a maximum size of 5120 (as mentioned
*   in Section 4.3, noted above).  These are on their own carousel as well.
*/
#define EV1_LOGO_PAYLOAD_MAX_SIZE (5120 * 4 * 2)
#define EV1_RFD_META_PAYLOAD_MAX_SIZE (5120)
#define EV1_RFD_BLOCK_PAYLOAD_MAX_SIZE (5120)
#define EV1_CHGTYPE_PAYLOAD_MAX_SIZE (5120)
#define EV1_AVAIL_PAYLOAD_MAX_SIZE (5120 * 4 * 2 * 2)

// Buffer size needed for base DSI
#define EV1_OTA_BUFFER_SIZE_BASE_DSI ( \
    EV1_LOGO_PAYLOAD_MAX_SIZE + \
    EV1_RFD_META_PAYLOAD_MAX_SIZE + \
    EV1_RFD_BLOCK_PAYLOAD_MAX_SIZE \
    )

// Buffer size needed for update DSI
#define EV1_OTA_BUFFER_SIZE_UPDATE_DSI ( \
    EV1_CHGTYPE_PAYLOAD_MAX_SIZE + \
    EV1_AVAIL_PAYLOAD_MAX_SIZE \
    )

  /**************/
 /** TYPEDEFS **/
/**************/

// Update types
typedef enum evc1_update_type_enum
{
    EV1_UPDATE_TYPE_DELETE = 0,
    EV1_UPDATE_TYPE_NEW,
    EV1_UPDATE_TYPE_MODIFY,
    EV1_UPDATE_TYPE_END,
    EV1_INVALID_UPDATE_TYPE

} EVC1_UPDATE_TYPE_ENUM;

typedef struct ev1_logo_ctrl_struct
{
    // Flag indicates if logo updates are enabled
    BOOLEAN bEnabled;

    // The attributes of the logo itself
    FUEL_LOGO_ROW_STRUCT sLogoInProgress;
    OSAL_BUFFER_HDL hLogoData;

    // Messaging attributes
    N32 n32LastSeqNo;
    UN16 un16MessagesProcessed;

} EV1_LOGO_CTRL_STRUCT;

typedef struct ev1_availability_ctrl_struct
{
    // Region we're currently operating on
    FUEL_REGION tRegion;

    // The positions reported for each fuel type
    // come in a table that is bounded by the size
    // of the ENO field, so we'll use that as the way
    // to determine how large this array needs to be
    FUEL_POSITION_UPDATE_STRUCT asPositions[EV1_AVAIL_MAX_TYPES];
    
} EV1_AVAILABILITY_CTRL_STRUCT;

typedef struct ev1_charger_mapping_entry_struct
{
    // The charger type this entry maps to
    UN8 un8ChargerType;

    // The text representations for it (if known)
    STRING_OBJECT hShortText;
    STRING_OBJECT hLongText;

} EV1_CHARGER_MAPPING_ENTRY_STRUCT;

typedef struct ev1_charger_type_table_struct
{
    // The version of the table
    UN8 un8Version;

    // The table entries.  We allocate an array of this size
    // because we use the ATYPE as the index number into this array
    // and in order to do this, we must use an array big enough to 
    // index the largest mapping value possible
    EV1_CHARGER_MAPPING_ENTRY_STRUCT asMappings[EV1_CHGTYPE_MAX_MAPPINGS];

} EV1_CHARGER_TYPE_TABLE_STRUCT;

typedef struct ev1_charger_type_ctrl_struct
{
    // A list of the tables we're currently using
    OSAL_OBJECT_HDL hChargerTypeTables;

    // Structure allocated on the heap used to 
    // aid in searches
    EV1_CHARGER_TYPE_TABLE_STRUCT sSearchStruct;

} EV1_CHARGER_TYPE_CTRL_STRUCT;

// Interface definition
typedef struct ev1_object_struct
{
    // Handle to the fuel manager
    FUEL_SERVICE_OBJECT hFuelService;

    // Handle to the DB interface for the fuel service
    FUEL_DB_INTERFACE_OBJECT hDBInterface;

    // CRC handle (used to compute ISO 3309 CRC32 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    // Handle to RFD client
    RFD_INTERFACE_OBJECT hRFD;

    // Structure used to track logo progress
    EV1_LOGO_CTRL_STRUCT sLogoCtrl;

    // Structured used to aid in the processing
    // of availability messages
    EV1_AVAILABILITY_CTRL_STRUCT sAvailCtrl;

    // Structure used to aid in the processing
    // of charger type messages
    EV1_CHARGER_TYPE_CTRL_STRUCT sChgTypeCtrl;

} EV1_OBJECT_STRUCT;

// Structure used to group all
// fixed object handles and memory
typedef struct ev1_fixed_calc_struct
{
    // Fixed objects created to
    // represent constant float values
    OSAL_FIXED_OBJECT hInitialLat;
    OSAL_FIXED_OBJECT hInitialLon;
    OSAL_FIXED_OBJECT hComputeDenominator;

    // A memory segment used for temporary
    // values used to compute lat/lon
    OSAL_FIXED_OBJECT_DATA atFixedData[
        OSAL_FIXED_OBJECT_SIZE * EV1_NUM_MEMORY_FIXED];

} EV1_FIXED_CALC_STRUCT;

typedef struct ev1_rfd_ctrl_struct
{
    FUEL_DB_INTERFACE_OBJECT hDBInterface;
    RFD_INTERFACE_OBJECT hRFD;

    // Our path / connection to the DB under construction
    SQL_INTERFACE_OBJECT hDBConnection;
    STRING_OBJECT hDBPath;

    // Progress tracking
    RFD_UPDATE_VERSION tCurrentVersion;
    RFD_PROGRESS_INDEX tCurProgressIndex;
    RFD_PROGRESS_INDEX tStartProgressIndex;
    UN16 un16NumRegionsInUpdate;

    // Current region under process
    FUEL_REGION tRegion;

    // The block pool used to process RFD files
    OSAL_OBJECT_HDL hBlockPool;

    // The buffer used to access the block pool
    OSAL_BUFFER_HDL hBuffer;

    // Command buffer used to
    // generate comments
    char *pcSQLCommandBuffer;
    size_t tBufferSize;

    // The list of brand text strings
    // provided in an RFD update
    OSAL_OBJECT_HDL hBrandTexts;

    // Structure which groups all
    // fixed-object attributes used in
    // computing lat/lon
    EV1_FIXED_CALC_STRUCT sFixed;

    // Fuel types array, which is used as temporary
    // storage while processing updates
    FUEL_REFUELING_TYPE_STRUCT asTypes[FUEL_REFUELING_TYPES_ARRAY_SIZE];

} EV1_RFD_CTRL_STRUCT;

// Structure which defines a brand text entry
typedef struct ev1_brand_text_struct
{
    UN8 un8Code;
    STRING_OBJECT hString;

} EV1_BRAND_TEXT_STRUCT;

// Structure utilized for logo update work
typedef struct ev1_logo_work_struct
{
    // Logo attributes
    FUEL_LOGO_ROW_STRUCT sLogo;

    // Are we updating this logo
    // or deleting it?
    BOOLEAN bUpdateLogo;

    // The sequence number of the
    // logo message being processed
    UN16 un16SeqNo;

    // The number of access units
    // which make up this logo
    UN16 un16MessagesNeeded;

} EV1_LOGO_WORK_STRUCT;

typedef struct ev1_availability_work_struct
{
    // The version of the data being processed
    UN8 un8Version;

    // Message instance specific values
    UN8 un8CSSize;
    UN8 un8ATypeSize;
    UN8 un8NumSize;
    UN16 un16Time;

    // The previously processed station ID
    FUEL_STATION_ID tPrevStation;

    // The charger table we're referencing
    EV1_CHARGER_TYPE_TABLE_STRUCT *psChargerTable;

} EV1_AVAILABILITY_WORK_STRUCT;

typedef struct ev1_charger_type_work_struct
{
    // The number of charger types in this message
    UN8 un8ChargeTypeCount;

    // Size number of bits used in the "AType" Field
    UN8 un8ATypeSize;

    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable;
} EV1_CHARGER_TYPE_WORK_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

// Public interface APIs

static FUEL_OTA_INTERFACE_OBJECT hInit(
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface
        );

static void vUnInit(
    FUEL_OTA_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProductStateChange (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        );

static BOOLEAN bGetDSIForProduct (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        );

static DATA_PRODUCT_STATE_ENUM eNextProductState (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eType,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eDSIState,
    SXM_DMI *pDMIs,
    UN8 un8DMICount
        );

static BOOLEAN bProcessMessage (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bAddRegionsForLocation (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    LOCATION_OBJECT hCenterLocation,
    void *pvArg
        );

static SMSAPI_RETURN_CODE_ENUM eProcessAmenities (
    size_t tNumAmenities,
    AMENITY_STRUCT *pasFuelAmenities,
    UN32 un32RawAmenitiesData
        );

/* Object Private Prototypes */

/* Message Processors */

static EV1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    EV1_OBJECT_STRUCT *psObj,
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        );

static BOOLEAN bInitRFDCtrl (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        );

static void vDestroyRFDCtrl (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static void vDeleteDB (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bConnectToDB (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        );

static BOOLEAN bFinalizeDB (
   EV1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bLogoProductStateChange (
    EV1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        );

static BOOLEAN bRFDProductStateChange (
    EV1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        );

static BOOLEAN bAvailabilityProductStateChange (
    EV1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        );

static BOOLEAN bEnableAvailabilityUpdates (
    EV1_OBJECT_STRUCT *psObj
        );

static void vDisableAvailabilityUpdates (
    EV1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProcessAvailabilityMessage (
    EV1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessAvailabilityMessageHeader (
    EV1_OBJECT_STRUCT *psObj,
    EV1_AVAILABILITY_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessAvailabilityMessageData (
    EV1_OBJECT_STRUCT *psObj,
    EV1_AVAILABILITY_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessChargerTypeMessage (
    EV1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessChargerTypeMessageHeader (
    EV1_OBJECT_STRUCT *psObj,
    EV1_CHARGER_TYPE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessChargerTypeMessageData (
    EV1_OBJECT_STRUCT *psObj,
    EV1_CHARGER_TYPE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static N16 n16CompareChargerMappings (
    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable1,
    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable2
        );

static EV1_CHARGER_TYPE_TABLE_STRUCT *psChargerTable (
    EV1_OBJECT_STRUCT *psObj,
    UN8 un8Version
        );

static BOOLEAN bProcessLogoMessage (
    EV1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bProcessLogoMessageHeader (
    EV1_OBJECT_STRUCT *psObj,
    EV1_LOGO_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessLogoMessageData (
    EV1_OBJECT_STRUCT *psObj,
    EV1_LOGO_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL *phPayload
        );

static void vInitLogoTracking (
    EV1_LOGO_CTRL_STRUCT *psLogoCtrl,
    BOOLEAN bEnabled
        );

static BOOLEAN bCanProcessLogoMessage (
    EV1_LOGO_CTRL_STRUCT *psLogoCtrl,
    EV1_LOGO_WORK_STRUCT *psWork
        );

static BOOLEAN bEnableRFDUpdates (
    EV1_OBJECT_STRUCT *psObj
        );

static void vDisableRFDUpdates (
    EV1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProcessRFDHeader (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static BOOLEAN bProcessChargeTypes (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        );

static BOOLEAN bProcessRegion (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static BOOLEAN bProcessStation(
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    UN8 un8CSSize,
    UN8 *pun8LastSubRegion,
    UN8 *pun8LastDivision,
    UN16 *pun16LastSubDivision
        );

static BOOLEAN bProcessAmenities (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bAmenities1,
    UN32 *pun32Amenities
        );

static BOOLEAN bFieldPresent(
    OSAL_BUFFER_HDL hBuffer
        );

static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        );

static BOOLEAN bComputeLatLonFromGrid (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    UN8 un8SubRegion,
    UN8 un8Division,
    UN16 un16SubDivision
        );

static BOOLEAN bCreateFixedConstants (
    EV1_FIXED_CALC_STRUCT *psFixed
        );

static void vDestroyFixedConstants (
    EV1_FIXED_CALC_STRUCT *psFixed
        );

static BOOLEAN bAddBrandText (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode,
    STRING_OBJECT hString
        );

static STRING_OBJECT hGetBrandString(
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode
        );

static N16 n16CompareTextEntries (
    EV1_BRAND_TEXT_STRUCT *psText1,
    EV1_BRAND_TEXT_STRUCT *psText2
        );

static void vRemoveTextEntry (
    EV1_BRAND_TEXT_STRUCT *psText
        );

/* RFD Interface callbacks */
static BOOLEAN bExtractVersionFromName (
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptBaseVersion,
    RFD_UPDATE_VERSION *ptNewVersion
        );

static RFD_FILE_STATUS_ENUM eRFDUpdateNeeded (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        );

static void vReleaseChargerTypeTable(
    void *pvData
        );

  /***************/
 /** VARIABLES **/
/***************/

const DATASERVICE_PRODUCT_DESCRIPTOR_STRUCT GasEVProducts[] = 
{
    // Base EV Product
    {
        DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES,
        DATA_PRODUCT_MASK_NONE
    },

    // Base EV Product
    {
        DATA_PRODUCT_TYPE_DB_UPDATES,
        DATA_PRODUCT_MASK_NONE
    },

    // EV Availability Product
    {
        DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES,
        DATA_PRODUCT_MASK_NONE
    }
};

const FUEL_OTA_INTERFACE_STRUCT GsEVChargingOTAIntf =
{
    /*.tDSI = */EV1_BASE_DSI,
    /*.tDataID = */DATASERVICE_ID_EV,
    /*.tMaxVersionBitlen = */EV1_MAX_VERSION_BITLEN,
    /*.psProducts = */GasEVProducts,
    /*.un8NumProducts = */EV1_NUM_PRODUCTS,
    /*.un8RFDClientId = */EV1_RFD_CLIENT_ID,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProductStateChange = */bProductStateChange,
    /*.bGetDSIForProduct = */bGetDSIForProduct,
    /*.eNextProductState = */eNextProductState,
    /*.bProcessMessage = */bProcessMessage,
    /*.hAddRegionsForLocation = */bAddRegionsForLocation,
    /*.eProcessAmenities = */eProcessAmenities
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _EVCHARGING_PVN1_H_
