/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _EPG_PVN1_H_
#define _EPG_PVN1_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "eprogram_obj.h"
#include "epg_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define EPG_PARSER_DBG_PREFIX   "EPG PARSER"

/* Data Services */
#define EPG_DSI                 (300)
#define EPG_INTF_OBJECT_NAME    "EPG1"

/*##################################
  # EPG Protocol related constants #
  ##################################*/
// VERSION OF PROTOCOL SUPPORTED BY THIS IMPLEMENTATION
#define EPG_PVN_VAL                 (1)

/*-------------------------------------------------
  - Common Access Unit Header related definitions -
  - (common for all EPG Access Units)             -
  -------------------------------------------------*/
// Common AU Header fields lengths
#define EPG_AU_HEADER_PVN_BITLEN                (4)
#define EPG_AU_HEADER_CARID_BITLEN              (3)
// CARID field values
#define EPG_AU_HEADER_CARID_SCHEDULE_TODAY      (0)
#define EPG_AU_HEADER_CARID_SCHEDULE_UPCOMING   (1)
#define EPG_AU_HEADER_CARID_PAM                 (2)
#define EPG_AU_HEADER_CARID_TAD                 (3)
#define EPG_AU_HEADER_CARID_PROFILE_CONFIG      (4)
#define EPG_AU_HEADER_CARID_SCHEDULE_VERSIONING (5)

/*---------------------------------------------------------------
  - Common Schedule Access Unit Header related definitions      -
  - (in addition to Common AU Header fields)                    -
  - (CARID = CARID_SCHEDULE_TODAY or CARID_SCHEDULE_UPCOMING)   -
  ---------------------------------------------------------------*/
// Common Schedule AU Header fields lengths
#define EPG_AU_HEADER_TFLAG_BITLEN          (1)
#define EPG_AU_HEADER_SHVER_BITLEN          (5)
#define EPG_AU_HEADER_EPOCH_BITLEN          (16)
#define EPG_AU_HEADER_NOSEG_BITLEN          (3)
#define EPG_AU_HEADER_SHSEG_BITLEN          (3)
#define EPG_AU_HEADER_CTSIZE_BITLEN         (4)
// TFLAG field values
#define EPG_AU_HEADER_TFLAG_GRID            (0)
#define EPG_AU_HEADER_TFLAG_TEXT            (1)

/*---------------------------------------------------------------
  - Grid Access Unit Header related definitions                 -
  - (in addition to Common Schedule AU Header fields)           -
  - (CARID = CARID_SCHEDULE_TODAY or CARID_SCHEDULE_UPCOMING)   -
  - (TFLAG = TFLAG_GRID)                                        -
  ---------------------------------------------------------------*/
// Grid AU Header fields lengths
#define EPG_AU_HEADER_SERSIZE_BITLEN        (5)
#define EPG_AU_HEADER_PROGSIZE_BITLEN       (5)
#define EPG_AU_HEADER_ISIZE_BITLEN          (4)
#define EPG_AU_HEADER_TSIZE_BITLEN          (4)

/*--------------------------------------------------
  - Program Event Block Header related definitions -
  --------------------------------------------------*/
// PEB header fields lengths
#define EPG_PEB_HEADER_SINC_BITLEN          (2)
#define EPG_PEB_HEADER_SID_BITLEN           (10)
#define EPG_PEB_HEADER_SHOWNO_BITLEN        (6)
// SINC field values
#define EPG_PEB_HEADER_SINC_CODE_END_OF_LIST (0)
#define EPG_PEB_HEADER_SINC_CODE_INC_OF_ONE (1)
#define EPG_PEB_HEADER_SINC_CODE_INC_OF_TWO (2)
#define EPG_PEB_HEADER_SINC_CODE_ABSOLUTE   (3)

/*---------------------------------------------
  - Program Event Message related definitions -
  ---------------------------------------------*/
// PEM fields lengths
#define EPG_PEM_PTYPE_BITLEN                (2)
#define EPG_PEM_FLAGS1_BITLEN               (6)
#define EPG_PEM_FLAGS2_BITLEN               (6)
#define EPG_PEM_DUR_BITLEN                  (3)
#define EPG_PEM_ODUR_BITLEN                 (9)
#define EPG_PEM_ORIGIN_BITLEN               (16)
#define EPG_PEM_STNUM_BITLEN                (5)
#define EPG_PEM_START_BITLEN                (9)
#define EPG_PEM_DAY_BITLEN                  (7)
#define EPG_PEM_TCNT_BITLEN                 (3)
#define EPG_PEM_EXTCNT_BITLEN               (8)
#define EPG_PEM_EXTCNT_MULTIPLIER           (8)
// PTYPE field values
#define EPG_PEM_PTYPE_FULL_EVENT            (0)
#define EPG_PEM_PTYPE_MODIFY_EVENT          (1)
#define EPG_PEM_PTYPE_ADD_EVENT             (2)
#define EPG_PEM_PTYPE_DURATION_ONLY_EVENT   (3)
// FLAGS field values
#define EPG_PEM_FLAGS_UNDEFINED             (0x0000)
#define EPG_PEM_FLAGS_FEATURED              (0x0001) // 1 bit
#define EPG_PEM_FLAGS_HIGHLIGHTED           (0x0002) // 1 bit
#define EPG_PEM_FLAGS_LIVE                  (0x0004) // 1 bit
#define EPG_PEM_FLAGS_NEW_THIS              (0x0008) // 1 bit
#define EPG_PEM_FLAGS_RECORD_BIT_MASK       (0x0030) // 2 bits
#define EPG_PEM_FLAGS_RECORD_ALWAYS         (0x0000)
#define EPG_PEM_FLAGS_RECORD_RESTRICTED     (0x0010)
#define EPG_PEM_FLAGS_RECORD_APPROVED       (0x0020)
#define EPG_PEM_FLAGS_RECORD_NONE           (0x0030)
#define EPG_PEM_FLAGS_ALL           \
       (EPG_PEM_FLAGS_FEATURED    | \
        EPG_PEM_FLAGS_HIGHLIGHTED | \
        EPG_PEM_FLAGS_LIVE        | \
        EPG_PEM_FLAGS_NEW_THIS    | \
        EPG_PEM_FLAGS_RECORD_BIT_MASK)
// DUR field values
#define EPG_PEM_DUR_30MIN                   (0)
#define EPG_PEM_DUR_1HOUR                   (1)
#define EPG_PEM_DUR_2HOURS                  (2)
#define EPG_PEM_DUR_3HOURS                  (3)
#define EPG_PEM_DUR_4HOURS                  (4)
#define EPG_PEM_DUR_5HOURS                  (5)
#define EPG_PEM_DUR_END_OF_SEGMENT          (6)
#define EPG_PEM_DUR_ESCAPE_TO_ODUR          (7)
// ODUR field LSB value (in minutes)
#define EPG_PEM_ODUR_LSB_VALUE              (5)
// START field LSB value ( in minutes)
#define EPG_PEM_START_LSB_VALUE             (5)

/*-------------------------------------------------------------
  - Compressed Text Access Unit Header related definitions    -
  - (in addition to Common Schedule AU Header fields)         -
  - (CARID = CARID_SCHEDULE_TODAY or CARID_SCHEDULE_UPCOMING) -
  - (TFLAG = TFLAG_TEXT)                                      -
  -------------------------------------------------------------*/
// Compressed Text AU Header fields lengths
#define EPG_AU_HEADER_NOSTRING_BITLEN       (16)
#define EPG_AU_HEADER_FILESIZE_BITLEN       (24)
#define EPG_CT_HEADER_BASE_BITLEN (EPG_AU_HEADER_PVN_BITLEN \
                                 + EPG_AU_HEADER_CARID_BITLEN \
                                 + EPG_AU_HEADER_TFLAG_BITLEN \
                                 + EPG_AU_HEADER_SHVER_BITLEN \
                                 + EPG_AU_HEADER_EPOCH_BITLEN \
                                 + EPG_AU_HEADER_NOSEG_BITLEN \
                                 + EPG_AU_HEADER_SHSEG_BITLEN \
                                 + EPG_AU_HEADER_NOSTRING_BITLEN \
                                 + EPG_AU_HEADER_FILESIZE_BITLEN \
                                 + 1) // CTSIZE field presence flag (1 bit)

/*----------------------------------------------------------------------
  - Schedule Segment Versioning Access Unit Header related definitions -
  - (in addition to Common AU Header fields)                           -
  - (CARID = SCHEDULE_VERSIONING)                                      -
  ----------------------------------------------------------------------*/
// Schedule Segment Versioning AU Header fields lengths
#define EPG_AU_HEADER_SSV_RESERVED_BITLEN   (1)

/*-------------------------------------------------------
  - Schedule Segment Version Payoad related definitions -
  -------------------------------------------------------*/
// Schedule Segment Version Payload fields lengths
#define EPG_SSV_SCHVER_BITLEN               (5)
#define EPG_SSV_EPOCH_0_BITLEN              (16)
#define EPG_SSV_NOSEG_BITLEN                (3)
#define EPG_SSV_SEGVER_BITLEN               (5)
#define EPG_SSV_EXTCNT_BITLEN               (8)


/*#############################################
  # Internal implementation related constants #
  #############################################*/
#define EPG_BLOCKPOOLBUF_SIZE   (1024)
#define EPG_BLOCKPOOLBUF_NUM    (5)

// Allocate this to be 300k (just based on what it was set to in
// earlier revisions)
#define EPG_OTA_BUFFER_BYTESIZE (300 * 1024)

// Maximum allowed number of segments supported by this protocol
#define EPG_MAX_TOTAL_NUMBER_OF_SEGMENTS (1 << EPG_AU_HEADER_NOSEG_BITLEN)

// Maximum allowed schedule version value
#define EPG_SCHVER_MAX      ((1 << EPG_SSV_SCHVER_BITLEN) - 1)

// Maximum allowed amount of topics associated with program
#define EPG_TOPICS_CNT_MAX  (1 << EPG_PEM_TCNT_BITLEN)

// Maximum allowed amount of additional start times of the program
#define EPG_ADDITIONAL_START_TIMES_CNT_MAX (1 << EPG_PEM_STNUM_BITLEN)

  /**************/
 /** TYPEDEFS **/
/**************/

// Access Unit (AU) payload data
typedef struct epg_au_data_struct
{
    void *pvAuPayload;
    size_t tPayloadSize;

} EPG_AU_DATA_STRUCT;

// Grid AU header specific data
typedef struct epg_au_grid_header
{
    UN8 un8SeriesIdSize;
    UN8 un8ProgramIdSize;
    UN8 un8TextIndexSize;
    UN8 un8TopicIdSize;
    UN32 un32AuSequenceNumber;
    UN32 un32NumAuTotal;

} EPG_AU_GRID_HEADER;

// Grid data for one segment (Grid AUs collection)
typedef struct epg_seg_grid_data_struct
{
    UN32 un32NumAuTotal;
    UN32 un32NumAuCollected; // Number of AUs already parsed
    EPG_AU_DATA_STRUCT *asAuGridData; // Dynamic array (size = un32NumAuTotal)

} EPG_SEG_GRID_DATA_STRUCT;

// Text AU header specific data
typedef struct epg_au_text_header
{
    UN16 un16StringsCount;
    UN32 un32DecompressedFileSize;
    UN32 un32AuSequenceNumber;
    UN32 un32NumAuTotal;

} EPG_AU_TEXT_HEADER;

// Compressed text data of segment (Text AUs collection)
typedef struct epg_seg_text_data_struct
{
    // Text AU header data
    EPG_AU_TEXT_HEADER sTextAuHeader;
    UN32 un32NumAuCollected;
    EPG_AU_DATA_STRUCT *asAuTextData; // Dynamic array (size = sTextAuHeader.un32NumAuTotal)

} EPG_SEG_TEXT_DATA_STRUCT;

// Segment data (Grid and Text AUs collection)
typedef struct epg_segment_struct
{
    // Common segment information
    UN16 un16SegmentEpoch;
    UN8 un8SegmentVersion;
    UN8 un8SegmentNumber;

    // Grid data
    EPG_SEG_GRID_DATA_STRUCT sSegGridData;

    // Text data
    EPG_SEG_TEXT_DATA_STRUCT sSegTextData;

} EPG_SEGMENT_STRUCT;

// Main Parser object
typedef struct epg_parser_struct
{
    // Collected schedule information
    UN8 un8NumSegmentsToRecieve;
    UN8 un8NumSegmentsGridCollected;
    UN8 un8NumSegmentsTextCollected;
    EPG_SEGMENT_STRUCT *asSegments; // Dynamic array (size = un8NumSegmentToRecieve)

    // Parser options
    UN32 un32Options;
    BOOLEAN bAcceptNewVersionOnly;

    // Callback to check if ongoing processing is cancelled
    CANCELLATION_CHECK_CALLBACK bCancellationCallback;
    void *pvCancellationCallbackArg;

    // EPG Parser's client interface.
    // Used to bring parsed data to a parser's client.
    EPG_CLIENT_INTERFACE_STRUCT *psEpgClient;

    // Schedule parser internal utility data
    OSAL_OBJECT_HDL hCRC;
    OSAL_OBJECT_HDL hTempBlockPool;
    OSAL_BUFFER_HDL hTempBuffer;

} EPG_PARSER_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Interface APIs */

static EPG_PARSER_INTERFACE_OBJECT hEpgIntfInit (
        EPG_CLIENT_INTERFACE_STRUCT *psClientIntf,
        UN8 un8NumSegments,
        UN32 un32Options
            );

static void vEpgIntfUninit (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj
            );

static EPG_RETURN_CODES_ENUM eEpgIntfProcessMessage (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        OSAL_BUFFER_HDL hPayload,
        CANCELLATION_CHECK_CALLBACK bCancellationCallback,
        void *pvCancellationCallbackArg
            );

static EPG_RETURN_CODES_ENUM eEpgIntfLoadAuData (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        EPG_AU_INFO_STRUCT *psAuData,
        CANCELLATION_CHECK_CALLBACK bCancellationCallback,
        void *pvCancellationCallbackArg
            );

static EPG_RETURN_CODES_ENUM eEpgIntfGetOptions (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        UN32 *pun32Options
            );

static EPG_RETURN_CODES_ENUM eEpgIntfSetOptions (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        UN32 un32Options
            );

/* Private functions */

/* Utility functions */

static void vEpgParserCleanParserEvent (
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static void vEpgParserCleanAu (
       EPG_AU_DATA_STRUCT *psAuGridData
            );

static EPG_PEM_DURATION_CODE_ENUM eDurationCodeFromValue (
        UN32 un32DurCodeValue
            );

/* General */

static EPG_PARSER_STRUCT *psEpgParserCreate (
        UN8 un8NumSegments
            );

static void vEpgParserDestroy (
        EPG_PARSER_STRUCT *psEpgParser
            );

static void vEpgParserDestroyAllSegments (
        EPG_PARSER_STRUCT *psEpgParser
            );

static void vEpgParserCleanAllSegments (
        EPG_PARSER_STRUCT *psEpgParser
            );

static void vEpgParserCleanAllSegmentsGrid (
        EPG_PARSER_STRUCT *psEpgParser
            );

static void vEpgParserCleanSegmentGrid (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber
            );

static void vEpgParserCleanSegmentText (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber
            );

static BOOLEAN bEpgParserCreateTempBuffer (
        EPG_PARSER_STRUCT *psEpgParser
            );

static void vEpgParserDestroyTempBuffer (
        EPG_PARSER_STRUCT *psEpgParser
            );

static BOOLEAN bEpgParserCopyDataToTempBuffer (
        EPG_PARSER_STRUCT *psEpgParser,
        void *pvData,
        size_t tDataSize
            );

static BOOLEAN bEpgParserIsCancelled (
        EPG_PARSER_STRUCT *psEpgParser
            );

/* Message Processors */

static BOOLEAN bEpgParserExtractGridHeader (//TODO: Add function to peek data
        OSAL_BUFFER_HDL hPayload,
        EPG_AU_GRID_HEADER *psGridHeader
            );

static BOOLEAN bEpgParserExtractTextHeader (//TODO: Add function to peek data
        OSAL_BUFFER_HDL hPayload,
        EPG_AU_TEXT_HEADER *psTextHeader
            );

static void vEpgParserParseProgramFlags (
        UN16 un16Flags,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractProgramFlags (
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractDuration (
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        BOOLEAN bMandatory
            );

static BOOLEAN bEpgParserExtractAdditionalTimes (
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractTopics (
        EPG_AU_GRID_HEADER *psGridHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractExtraData (
        OSAL_BUFFER_HDL hPayload,
        void **ppvDst,
        UN16 *pun16Size
            );

static BOOLEAN bEpgParserProcessScheduleVersion (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessScheduleMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessGridMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessNextNearestGridAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessGridAu (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessTextMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessNextNearestTextAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessProfileConfigMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessPamMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessTadMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessScheduleVersioningMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessGridPayload (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber,
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        BOOLEAN bIgnoreEventProcessingErrors
            );

static EPG_RETURN_CODES_ENUM eEpgParserProcessTextPayload (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber,
        EPG_AU_TEXT_HEADER *psTextAuHeader,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserSaveGridAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            );

static EPG_RETURN_CODES_ENUM eEpgParserSaveTextAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            );

static BOOLEAN bEpgParserExtractProgramEvent (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractPemFull (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractPemMod (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractPemAdd (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgParserExtractPemDur (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static EPG_RETURN_CODES_ENUM eEpgParserParseSavedGridAus (
        EPG_PARSER_STRUCT *psEpgParser,
        BOOLEAN bReprocessAll
            );

static EPG_RETURN_CODES_ENUM eEpgParserParseSavedTextAus (
        EPG_PARSER_STRUCT *psEpgParser,
        BOOLEAN bReprocessAll
            );

  /***************/
 /** VARIABLES **/
/***************/

const EPG_PARSER_INTERFACE_STRUCT GsEpgIntf =
{
    /*.tPVN = */EPG_PVN_VAL,
    /*.tDSI = */EPG_DSI,
    /*.tOTABufferByteSize = */EPG_OTA_BUFFER_BYTESIZE,
    /*.un8MaxTotalSegments = */EPG_MAX_TOTAL_NUMBER_OF_SEGMENTS,
    /*.hInit = */hEpgIntfInit,
    /*.vUninit = */vEpgIntfUninit,
    /*.eProcessMessage = */eEpgIntfProcessMessage,
    /*.eLoadAuData = */eEpgIntfLoadAuData,
    /*.eGetOptions = */eEpgIntfGetOptions,
    /*.eSetOptions = */eEpgIntfSetOptions,
};

#endif  // _EPG_PVN1_H_
