/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CHANNEL_ART_HIGHBAND_H_
#define _CHANNEL_ART_HIGHBAND_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "channel_art_interface.h"

  /************************/
 /** CONSTANTS / MACROS **/
/************************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define CHANNEL_ART_HIGHBAND_OBJECT_NAME "ChanArt:HB"

// We use the "SXM Channel Graphics" DSI.  Other alisases are "G4 Channel Graphics" or "Overlay Channel Graphics".
#define CHANNEL_GRAPHICS_DSI            (311)

// SX-9845-0166:
// Wide-background images max out at 60k (section 3.1.1)
// Logo images max out at 30k (section 3.2.3)
//
/* Max message sizes (from SX-9845-0166)
*   Section 3.1.1 says (Wide background images):
*       Image Data Size: Max 60,000 bytes.
*   Section 3.2.3 says (Large PNG logo images):
*       Image Data Size: Max 15,000 bytes (15,000 bytes for the
*       Primary Logo, and 15,000 bytes for the optional Secondary Logo).
*
*   No specification on reference messages, but
*   a fair guess is 5k max.  There is only 1 carousel
*   used in this service, which means that 60k is fine
*   to use as the number of bytes needed for this service.
*/

// Bosch ID#4: Increase the buffer size as discussed with SXM
#define CHANNEL_ART_HIGHBAND_BUFFER_BYTESIZE (60 * 1024 * 2)

// CGS definitions
#define CHANNNEL_ART_HIGHBAND_PVN_BITLEN                4
#define CHANNNEL_ART_HIGHBAND_CARID_BITLEN              3
#define CHANNNEL_ART_HIGHBAND_RFU_BITLEN                1

#define CHANNNEL_ART_HIGHBAND_PVN                       (PVN)1
#define CHANNNEL_ART_HIGHBAND_CARID                     (PVN)0

#define CHANNEL_ART_HIGHBAND_OVERLAY_IMAGE_ID_OFFSET            256
#define CHANNEL_ART_HIGHBAND_OVERLAY_SOURCE_ID_OFFSET           256
#define CHANNEL_ART_HIGHBAND_OVERLAY_REF_ID_OFFSET              256

#define CHANNEL_ART_HIGHBAND_OVERLAY_REF_ID_OFFSET_RANGE_START  0
#define CHANNEL_ART_HIGHBAND_OVERLAY_REF_ID_OFFSET_RANGE_STOP   127

#define CHANNNEL_ART_HIGHBAND_INITIAL_SERVICE_ID    1
#define CHANNNEL_ART_HIGHBAND_FINAL_SERVICE_ID      255

#define CHANNNEL_ART_HIGHBAND_OVERLAY_INITIAL_SERVICE_ID    256
#define CHANNNEL_ART_HIGHBAND_OVERLAY_FINAL_SERVICE_ID      383

//CSH commented out old XMAppID constants
#if 0
#define CHANNNEL_ART_HIGHBAND_SYNC_BYTELEN          6
#define CHANNNEL_ART_HIGHBAND_SYNC_BYTE0            0x24
#define CHANNNEL_ART_HIGHBAND_SYNC_BYTE1            0x58
#define CHANNNEL_ART_HIGHBAND_SYNC_BYTE2            0x4D
#define CHANNNEL_ART_HIGHBAND_SYNC_BYTE3            0x4D
#define CHANNNEL_ART_HIGHBAND_SYNC_BYTE4            0x58
#define CHANNNEL_ART_HIGHBAND_SYNC_BYTE5            0x24
#define CHANNNEL_ART_HIGHBAND_PAYLOAD_LEN_BITLEN    16
#define CHANNNEL_ART_HIGHBAND_CRC_BYTELEN           2
#endif

// Message Identifier Definitions
#define CHANNEL_ART_HIGHBAND_MSG_CAT (1)
#define CHANNEL_ART_HIGHBAND_MSG_STATIC_SVC (2)
#define CHANNEL_ART_HIGHBAND_MSG_DYNAMIC_SVC (8)
#define CHANNEL_ART_HIGHBAND_MSG_LOGO (9)
#define CHANNEL_ART_HIGHBAND_MSG_BKGND (10)
#define CHANNEL_ART_HIGHBAND_MSG_OVERLAY_STATIC_SVC (112)
#define CHANNEL_ART_HIGHBAND_MSG_OVERLAY_DYNAMIC_SVC (118)
#define CHANNEL_ART_HIGHBAND_MSG_OVERLAY_LOGO (119)
#define CHANNEL_ART_HIGHBANG_INVALID_MSG  (255)

#define CHANNEL_ART_HIGHBAND_MSG_ALT_MODIFIER (100)
#define CHANNEL_ART_HIGHBAND_MSG_ALT_ID(x) \
    ( (x) + CHANNEL_ART_HIGHBAND_MSG_ALT_MODIFIER )

// General Message Field Length Definitions
#define CHANNEL_ART_HIGHBAND_MTI_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_DEVICE_MASK_ID_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_DEVICE_MASK_VALUE_BITLEN (32)

// General Message Field Value Definitions
#define CHANNEL_ART_HIGHBAND_MAX_ASSOC_SEQ (255)
#define CHANNEL_ART_HIGHBAND_UNREFERENCED_FLAG (127)
#define CHANNEL_ART_HIGHBAND_RESET_ASSOCS_FLAG (0)

// Device Mask Definitions
#define CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_DEVICE_MASK_BYTELEN (5)
#define CHANNEL_ART_HIGHBAND_DEVICE_MASK_ID (0x43)

// Logo Message Field Length Definitions
#define CHANNEL_ART_HIGHBAND_LOGO_ID_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_LOGO_RSVD1_BITLEN (1)
#define CHANNEL_ART_HIGHBAND_LOGO_SEQ_BITLEN (7)
#define CHANNEL_ART_HIGHBAND_LOGO_REV_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_LOGO_BKGRND_COLOR_INDEX_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_LOGO_RSVD2_BITLEN (1)
#define CHANNEL_ART_HIGHBAND_LOGO_SECOND_IMG_FLAG_BITLEN (1)
#define CHANNEL_ART_HIGHBAND_LOGO_RSVD3_BITLEN (6)
#define CHANNEL_ART_HIGHBAND_LOGO_DATA_LEN_BITLEN (16)

#define CHANNEL_ART_HIGHBAND_IMG_ID_OFFSET (8)
#define CHANNEL_ART_HIGHBAND_IMG_SEQ_OFFSET (17)
#define CHANNEL_ART_HIGHBAND_IMG_REV_OFFSET (24)
#define CHANNEL_ART_HIGHBAND_LOGO_SECOND_IMG_FLAG_OFFSET (65)
#define CHANNEL_ART_HIGHBAND_LOGO_LOOKAHEAD_LEN ( \
            CHANNEL_ART_HIGHBAND_LOGO_SECOND_IMG_FLAG_OFFSET + 1 )

#define CHANNEL_ART_HIGHBAND_LOGO_LOOKAHEAD_READ_LEN ( \
    CHANNEL_ART_HIGHBAND_LOGO_ID_BITLEN +              \
    CHANNEL_ART_HIGHBAND_LOGO_SEQ_BITLEN +             \
    CHANNEL_ART_HIGHBAND_LOGO_REV_BITLEN +             \
    CHANNEL_ART_HIGHBAND_LOGO_SECOND_IMG_FLAG_BITLEN   \
    )

// These fields exist in every logo message
#define CHANNEL_ART_HIGHBAND_LOGO_STATIC_HEADER_BITLEN ( \
    CHANNEL_ART_HIGHBAND_LOGO_ID_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_RSVD1_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_SEQ_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_REV_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_BKGRND_COLOR_INDEX_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_RSVD2_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_SECOND_IMG_FLAG_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_RSVD3_BITLEN + \
    CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_BITLEN \
    )

// Background Message Field Length Definitions
#define CHANNEL_ART_HIGHBAND_BKGRND_ID_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_BKGRND_RSVD1_BITLEN (1)
#define CHANNEL_ART_HIGHBAND_BKGRND_SEQ_BITLEN (7)
#define CHANNEL_ART_HIGHBAND_BKGRND_REV_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_BKGRND_DATA_LEN_BITLEN (16)

#define CHANNEL_ART_HIGHBAND_BKGRND_LOOKAHEAD_LEN ( \
    CHANNEL_ART_HIGHBAND_IMG_REV_OFFSET +           \
    CHANNEL_ART_HIGHBAND_BKGRND_REV_BITLEN      \
    )

#define CHANNEL_ART_HIGHBAND_BKGRND_LOOKAHEAD_READ_LEN ( \
    CHANNEL_ART_HIGHBAND_BKGRND_ID_BITLEN +              \
    CHANNEL_ART_HIGHBAND_BKGRND_SEQ_BITLEN +       \
    CHANNEL_ART_HIGHBAND_BKGRND_REV_BITLEN               \
    )

// These fields exist in every background image message
#define CHANNEL_ART_HIGHBAND_BKGRND_STATIC_HEADER_BITLEN ( \
    CHANNEL_ART_HIGHBAND_BKGRND_ID_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_RSVD1_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_SEQ_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_REV_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_COLOR_BITLEN + \
    CHANNEL_ART_HIGHBAND_VAR_DATA_LEN_BITLEN \
    )

// General Image Association MessageField Length Definitions
#define CHANNEL_ART_HIGHBAND_REF_SEQ_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_REF_LIST_SIZE_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_REF_ENTRY_VAR_DATA_SIZE_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_REF_HDR_VAR_DATA_SIZE_BITLEN (8)
#define CHANNEL_ART_HIGHBAND_REF_DB_SEARCH_ORDER_BITLEN (1)

// These fields exist in every category association message
#define CHANNEL_ART_HIGHBAND_REF_STATIC_HEADER_BITLEN ( \
    CHANNEL_ART_HIGHBAND_REF_LIST_SIZE_BITLEN + \
    CHANNEL_ART_HIGHBAND_REF_ENTRY_VAR_DATA_SIZE_BITLEN + \
    CHANNEL_ART_HIGHBAND_REF_HDR_VAR_DATA_SIZE_BITLEN \
    )

// Service Association Message Field Definitions
#define CHANNEL_ART_HIGHBAND_SVC_ID_BITLEN (8)

// Category Association Message Field Definitions
#define CHANNEL_ART_HIGHBAND_CAT_ID_BITLEN (8)

// Each entry in a channel association list
// contains these fields
#define CHANNEL_ART_HIGHBBAND_SVC_LIST_ENTRY_BITLEN ( \
    CHANNEL_ART_HIGHBAND_SVC_ID_BITLEN + \
    CHANNEL_ART_HIGHBAND_REF_DB_SEARCH_ORDER_BITLEN + \
    CHANNEL_ART_HIGHBAND_LOGO_SEQ_BITLEN \
    )

// Each entry in an category association list
// contains these fields
#define CHANNEL_ART_HIGHBAND_CAT_LIST_ENTRY_BITLEN ( \
    CHANNEL_ART_HIGHBAND_CAT_ID_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_ID_BITLEN + \
    CHANNEL_ART_HIGHBAND_REF_DB_SEARCH_ORDER_BITLEN + \
    CHANNEL_ART_HIGHBAND_BKGRND_SEQ_BITLEN \
    )

// General reference list entry definitions
#define CHANNEL_ART_HIGHBAND_LIST_ENTRY_BYTE_SIZE_SVC (5)
#define CHANNEL_ART_HIGHBAND_LIST_ENTRY_BYTE_SIZE_CAT (3)

// General Protocol Definitions
#define CHANNEL_ART_HIGHBAND_LINE_BITMAP_INVALID    (15)
#define CHANNEL_ART_HIGHBAND_TINT_ADJUST_FLAG_MASK  (0x80)
#define CHANNEL_ART_HIGHBAND_LINE_BITMAP_INDEX_MASK (0x0F)
#define CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID       (384)

// Image Sequence / Revision Definitions
#define CHANNEL_ART_HIGHBAND_ASSOC_MAX_SEQ_NO (255)
#define CHANNEL_ART_HIGHBAND_IMAGE_MAX_SEQ_NO (126)
#define CHANNEL_ART_HIGHBAND_IMAGE_MAX_REV_NO (255)
#define CHANNEL_ART_HIGHBAND_UNKNOWN_REV (0x8000)

#define CHANNEL_ART_HIGHBAND_REF_SEQ_NO_INVALID (-1)

#define CHANNEL_ART_HIGHBAND_CAT_INDEX (0)
#define CHANNEL_ART_HIGHBAND_CHAN_INDEX (1)
#define CHANNEL_ART_HIGHBAND_OVERLAY_CHAN_INDEX (2)
#define CHANNEL_ART_HIGHBAND_DYN_CHAN_INDEX (3)
#define CHANNEL_ART_HIGHBAND_OVERLAY_DYN_CHAN_INDEX (4)
#define CHANNEL_ART_HIGHBAND_MAX_INDEX (5)

// These macros either generate a 16 bit image
// version number from a 7 bit sequence number
// and a 8 bit revision number or they extract
// the sequence & revision from the image
// version number
#define CHANNEL_ART_HIGHBAND_VER_TO_SEQ(ver) \
    ( ( (ver) & 0x7f00) >> 8)
#define CHANNEL_ART_HIGHBAND_VER_TO_REV(ver) \
    ( (ver) & 0xFF)
#define CHANNEL_ART_HIGHBAND_SEQ_REV_TO_VER(seq, rev) \
    ( (((seq) & 0x7f) << 8) | (rev) )
#define CHANNEL_ART_HIGHBAND_SEQ_TO_VER(seq) \
    ( (((seq) & 0x7f) << 8) | CHANNEL_ART_HIGHBAND_UNKNOWN_REV )
#define CHANNEL_ART_HIGHBAND_VER_HAS_REV(ver) \
    ((((ver) & CHANNEL_ART_HIGHBAND_UNKNOWN_REV) \
    != CHANNEL_ART_HIGHBAND_UNKNOWN_REV) ? TRUE : FALSE)

// Image Id Definitions

// Number of bits the largest channel ID will be
#define CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID_BITLEN (9)
COMPILE_TIME_ASSERT( CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID < ((1 << CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID_BITLEN)-1),
    CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID_has_to_fit_CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID_BITLEN);

#define CHANNEL_ART_HIGHBAND_OTA_ID_FLAG (1 << (CHANNEL_ART_HIGHBAND_IMAGE_MAX_REF_ID_BITLEN))
#define CHANNEL_ART_HIGHBAND_GENERATE_OTA_ID(id) \
    ( (UN16)((id) | CHANNEL_ART_HIGHBAND_OTA_ID_FLAG) )
#define CHANNEL_ART_HIGHBAND_EXTRACT_ID_FROM_OTA(ota) \
    ( (ota) & (CHANNEL_ART_HIGHBAND_OTA_ID_FLAG-1) )

  /**************/
 /** TYPEDEFS **/
/**************/

/* Channel Graphics Data Service Protocol related typedefs */

// Enumeration of the message types supported
// by the high-band channel art service
typedef enum channel_art_highband_message_type_enum
{
    CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_IMAGE,
    CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ASSOC,
    CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_UNKNOWN

} CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ENUM;

// Enumeration which specifies the result of
// processing for an association message
typedef enum channel_art_highband_process_result_enum
{
    CHANNEL_ART_HIGHBAND_PROCESS_RESULT_COMPLETE,
    CHANNEL_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE,
    CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ERROR

} CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM;

// This structure groups all the attributes used
// to process Logo messages
typedef struct channel_art_highband_logo_processing_struct
{
    size_t tPrimaryLogoLen;

    // Flag indicating if more logos are available
    // in the current message
    BOOLEAN bMoreLogosAvailable;

} CHANNEL_ART_HIGHBAND_LOGO_PROCESSING_STRUCT;

typedef struct channel_art_highband_assoc_tracking_struct
{
    BOOLEAN bCompleted;
    BOOLEAN bReportFirstInstance;
    N16 n16LastAssocSeqNo;

} CHANNEL_ART_HIGHBAND_ASSOC_TRACKING_STRUCT;

typedef struct channel_art_highband_assoc_ctrl_processing_struct
{
    // The current type of association
    // message being processed
    UN8 un8CurrentAssocMessageType;

    // Current Association message attribute row
    CHANNEL_ART_ASSOC_ROW_STRUCT sCurrentAssocRow;
    UN16 un8CurrentImageSeqNo;

    // Flag indicating if the OTA image should be
    // tried first in an association priority list
    BOOLEAN bTryOTAImageFirst;

    // Flag indicating if we are currently
    // processing an image list within an assoc message
    BOOLEAN bProcessingImageList;

    // Flag indicating if we are currently
    // processing the association priority
    // list in the current image list entry
    BOOLEAN bProcessingAssocPriorityList;

    // Flag indicating if we actually need to send this update
    // to the manager
    BOOLEAN bSendAssocInfoToMgr;

    // The current size of the data within
    // a particular assoc message's variable data field
    UN8 un8RefEntryVarDataByteSize;

    // Association message tracking attributes
    CHANNEL_ART_HIGHBAND_ASSOC_TRACKING_STRUCT asAssocTracking[CHANNEL_ART_HIGHBAND_MAX_INDEX];

} CHANNEL_ART_HIGHBAND_ASSOC_CTRL_PROCESSING_STRUCT;

typedef struct channel_art_highband_object_struct
{
    // Central Channel art service handle
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService;

    // The device mask
    DEVICE_MASK tDeviceMask;

    // CRC handle (used to compute SDTP AU ISO 3309 CRC32 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    // The logo processing control structure
    CHANNEL_ART_HIGHBAND_LOGO_PROCESSING_STRUCT sLogoCtrl;

    // The association message control structure
    CHANNEL_ART_HIGHBAND_ASSOC_CTRL_PROCESSING_STRUCT sAssocCtrl;

} CHANNEL_ART_HIGHBAND_OBJECT_STRUCT;

  /****************/
 /** PROTOTYPES **/
/****************/

static CHANNEL_ART_INTERFACE_OBJECT hInit (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    SMS_OBJECT hParent
        );

static void vUnInit (
    CHANNEL_ART_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProcessMessage (
    CHANNEL_ART_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

static void vAllAssociationsCompleted (
    CHANNEL_ART_INTERFACE_OBJECT hInterface
        );

static void vResetAssociationTracking (
    CHANNEL_ART_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bCompareAssociationVersions (
    CHANNEL_ART_ASSOC_ROW_STRUCT *psOldAssocRow,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psNewAssocRow,
    BOOLEAN *pbOldNeedsConfirmation
        );

static BOOLEAN bCompareImageVersions (
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psOldAttribRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psNewAttribRow,
    BOOLEAN *pbImageCompatible
        );

/* Object Private Prototypes */

/* Message Processors */

static UN8 un8ExtractMTI (
    BOOLEAN bPeek,
    OSAL_BUFFER_HDL hPayload
        );

static CHANNEL_ART_HIGHBAND_MESSAGE_TYPE_ENUM eParseMessageHeader (
    CHANNEL_ART_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bProcessImageMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessAssociationMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessServiceAssoc (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN bIsOverlay,
    BOOLEAN *pbRowPopulated,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload
        );

static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessCategoryAssoc(
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN *pbRowPopulated,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload
        );

static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessAssocHeader (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static CHANNEL_ART_HIGHBAND_PROCESS_RESULT_ENUM eProcessAssocBody (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessLogoMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN bIsOverlay,
    BOOLEAN bNewMessage,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t *ptImageDataByteLen,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessBackgroundMessage (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    BOOLEAN bNewMessage,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t *ptImageDataByteLen,
    OSAL_BUFFER_HDL hPayload
        );

static void vProcessBackgroundOptions (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    UN8 un8BackgroundColorIndex,
    IMAGE_BACKGROUND_GRAPHICS_STRUCT *psBackground
        );

static BOOLEAN bDeviceMaskMatches (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    size_t *ptBytesConsumed
        );

static BOOLEAN bIsPayloadUsable (
    CHANNEL_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        );

  /***************/
 /** VARIABLES **/
/***************/

const CHANNEL_ART_PLUGIN_INTERFACE_STRUCT GsChannelArtIntf =
{
    /*.tDSI = */CHANNEL_GRAPHICS_DSI,
    /*.tOTABufferByteSize = */CHANNEL_ART_HIGHBAND_BUFFER_BYTESIZE,
    /*.tStartServiceID = */CHANNNEL_ART_HIGHBAND_INITIAL_SERVICE_ID,
    /*.tEndServiceID = */CHANNNEL_ART_HIGHBAND_FINAL_SERVICE_ID,
    /*.tOverlayStartServiceID = */CHANNNEL_ART_HIGHBAND_OVERLAY_INITIAL_SERVICE_ID,
    /*.tOverlayEndServiceID = */CHANNNEL_ART_HIGHBAND_OVERLAY_FINAL_SERVICE_ID,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProcessMessage = */bProcessMessage,
    /*.vAllAssociationsCompleted = */vAllAssociationsCompleted,
    /*.vResetAssociationTracking = */vResetAssociationTracking,
    /*.bCompareAssociationVersions = */bCompareAssociationVersions,
    /*.bCompareImageVersions = */bCompareImageVersions
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _CHANNEL_ART_HIGHBAND_H_
