/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the common Radio interface implementation for the
 *  Satellite Module Services (SMS) that are used by all radio modules
 *
 ******************************************************************************/
#include <string.h>
#include <time.h>

#include "standard.h"
#include "osal.h"
#include "sti_api.h"

#include "srh.h"

#include "srm_obj.h"
#include "module_obj.h"

// Radio includes
#include "radio.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   RADIO_bFindCapableDevices
 *
 *****************************************************************************/
BOOLEAN RADIO_bFindCapableDevices(
    SRM_OBJECT hSRM, MODULE_OBJECT *phModule, DECODER_OBJECT *phDecoder,
    const char **ppacDecoderName,
    SRH_DEVICE_CAPABILITIES_MASK tRequestedCapabilities)
{
    BOOLEAN bLocked, bFound = FALSE;

    bLocked = SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        
        const char *pacDecoderName = NULL;
        MODULE_OBJECT hModule = MODULE_INVALID_OBJECT;
        DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
        SRH_DEVICE_CAPABILITIES_MASK tCapabilities = SRH_DEVICE_CAPABILITY_NONE;
        RADIO_SRM_OBJECT_STRUCT *psSRM;

        // Obtain radio data
        psSRM = (RADIO_SRM_OBJECT_STRUCT *)SRM_hGetRadioSpecificData(hSRM);
        if(psSRM != NULL)
        {
            SRH_DEVICE_MODULE_TUPLE_STRUCT *psModuleTuple;
            UN32 un32NumModules;
            N32 n32Err;

            // Extract SRM name
            if(psSRM->pacName == NULL)
            {
                // Nothing found.
                SMSO_vUnlock((SMS_OBJECT)hSRM);
                return FALSE;
            }

            // Condition inputs and use provided or local vars as necessary
            if(phModule == NULL)
            {
                // Use local
                phModule = &hModule;
            }

            if(phDecoder == NULL)
            {
                // Use local
                phDecoder = &hDecoder;
            }

            if(ppacDecoderName == NULL)
            {
                // Use local
                ppacDecoderName = &pacDecoderName;
            }

            // Ask the SRH driver about the modules associated with this SRM
            // This will allocate memory for psTuple which we will have to
            // clean up ourselves when we're done with it
            n32Err = ioctl(psSRM->psDevice, SRH_IOCTL_DESCRIBE_MODULES, psSRM->pacName,
                &psModuleTuple, &un32NumModules);
            if(n32Err == DEV_OK)
            {
                UN32 un32ModuleIndex;

                // Iterate over the modules we discovered and
                // inspect our module descriptor list with
                // the relevant data
                for(un32ModuleIndex = 0; un32ModuleIndex < un32NumModules;
                    un32ModuleIndex++)
                {
                    // Process the relevant information

                    // This SRM's capabilities are equal to the capabilities
                    // of all it's constituent modules
                    tCapabilities
                            |= psModuleTuple[un32ModuleIndex].tModuleCapabilities;

                    // Does this MODULE have the capabilities requested?
                    if((psModuleTuple[un32ModuleIndex].tModuleCapabilities
                                    & tRequestedCapabilities)
                                    == tRequestedCapabilities)
                    {
                        UN32 un32NumDecoders;
                        SRH_DEVICE_DECODER_TUPLE_STRUCT *psDecoderTuple;

                        // Module has requested capabilities

                        // Find DECODER with these capabilities
                        // Ask the SRH driver about the decoders associated with
                        // this module This will allocate memory for psTuple
                        // which we will have to clean up ourselves when we're
                        // done with it
                        n32Err = ioctl(psSRM->psDevice,
                            SRH_IOCTL_DESCRIBE_DECODERS,
                            psModuleTuple[un32ModuleIndex].tModuleId,
                            &psDecoderTuple, &un32NumDecoders);
                        if(n32Err == DEV_OK)
                        {
                            UN32 un32DecoderIndex;

                            // Iterate over the decoders we discovered and
                            // inspect our decoder descriptor list with
                            // the relevant data
                            for(un32DecoderIndex = 0;
                                    (un32DecoderIndex < un32NumDecoders) &&
                                    (bFound == FALSE);
                                un32DecoderIndex++
                                    )
                            {
                                // Process the relevant information

                                // This SRM's capabilities are equal to the
                                // capabilities of all it's constituent decoders
                                tCapabilities |= psDecoderTuple[un32DecoderIndex].
                                    tDecoderCapabilities;

                                // Does this DECODER have the capabilities
                                // requested?
                                if((psDecoderTuple[un32DecoderIndex].
                                                tDecoderCapabilities
                                                & tRequestedCapabilities)
                                                == tRequestedCapabilities)
                                {
                                    MODULE_OBJECT hThisModule;
                                    DECODER_OBJECT hThisDecoder =
                                        DECODER_INVALID_OBJECT;

                                    // DECODER has requested capabilities

                                    // Find this DECODER's MODULE (if it exists)
                                    hThisModule
                                        = SRM_hModule(
                                            hSRM,
                                            psModuleTuple[un32ModuleIndex].
                                                tModuleId);
                                    if(hThisModule != MODULE_INVALID_OBJECT)
                                    {
                                        // Find this DECODER's Handle (if it exists)
                                        hThisDecoder
                                            = MODULE_hDecoder(
                                                hThisModule,
                                                psDecoderTuple[un32DecoderIndex].
                                                pacDecoderName);

                                        // Handle is always returned locked
                                        SMSO_vUnlock((SMS_OBJECT)hThisModule);
                                    }

                                    // Fill in all we can...

                                    // If we have a MODULE handle which is this one
                                    // Then we know we need to find another
                                    if((*phModule == hThisModule) &&
                                        (hThisModule != MODULE_INVALID_OBJECT))
                                    {
                                        // Skip this one, next time any matching
                                        // MODULE will be used.
                                        *phModule = MODULE_INVALID_OBJECT;
                                    }
                                    else if(*phModule == MODULE_INVALID_OBJECT)
                                    {
                                        // Use this one, we're done
                                        *phModule = hThisModule;
                                        bFound = TRUE;
                                    }

                                    // If we have a MODULE handle which is this one
                                    // Then we know we need to find another
                                    if((*phDecoder == hThisDecoder) &&
                                        (hThisDecoder != DECODER_INVALID_OBJECT))
                                    {
                                        // Skip this one, next time any matching
                                        // DECODER will be used.
                                        *phDecoder = DECODER_INVALID_OBJECT;
                                    }
                                    else if(*phDecoder == DECODER_INVALID_OBJECT)
                                    {
                                        // Use this one, we're done
                                        *phDecoder = hThisDecoder;
                                        bFound = TRUE;
                                    }

                                    // Record DECODER name
                                    *ppacDecoderName
                                        = psDecoderTuple[un32DecoderIndex].
                                        pacDecoderName;
                                }
                            }

                            // Free the tuple's memory allocated by the IOCTL
                            OSAL.vMemoryFree(psDecoderTuple);
                        }
                    }
                }

                // Free the tuple's memory allocated by the IOCTL
                OSAL.vMemoryFree(psModuleTuple);
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hSRM);
    }

    return bFound;
}


/*****************************************************************************
 *
 *   RADIO_bFreeDataPayload
 *
 *****************************************************************************/
BOOLEAN RADIO_bFreeDataPayload (
    OSAL_BUFFER_HDL hPayload
        )
{
    return STI_bFreePayload(hPayload);
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/
