/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*    LOCAL HEADER
*******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SDTPRX_H_
#define _SDTPRX_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <standard.h>
#include <stdio.h>
#include <time.h>

#include "osal.h"
#include "sdtp.h"

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/**************/
 /** TYPEDEFS **/
/**************/

/*

    SDTPRx Field Types

    Good idea to never use native types in protocols as they may change!
    So instead we define our own types for each of the fields that is a
    standard size.

    RX000107-B, Section 3 currently
    defines each of these fields.
*/

typedef UN8     SDTPRx_SYNC;      // 8-bits
typedef BOOLEAN SDTPRx_START_AU;  // 1-bit
typedef BOOLEAN SDTPRx_END_AU;    // 1-bit
typedef UN8     SDTPRx_RFU;       // 2-bit
typedef UN16    SDTPRx_LEN_COUNT; // 10-bits
// PSI                            // 10-bits
// Payload                        // 1-1024 bytes
typedef UN8     SDTPRx_SUM;       // 8-bits


////////////////////////////////////////////////////////////////////////////////
// SDTP Frame Header Structure
//
// This structure must be "packed" as direct memory access is performed
//
// ********************************
// *       *              *       *
// * SYNC  *  SDTP FRAME  *  SUM  *
// *       *              *       *
// ********************************
//
////////////////////////////////////////////////////////////////////////////////

typedef struct sdtp_hdr_struct
{
    SDTPRx_START_AU  tStartAu:1;
    SDTPRx_END_AU    tEndAu:1;
    SDTPRx_RFU       tRFU:2;
    PSI              tPSI:10;
    SDTPRx_LEN_COUNT tLenOrCount:10;

} SDTP_FRAME_HDR_STRUCT;


// SDTP Statistics
typedef struct sdtp_statistics_struct
{
    UN32 un32Frames; // Number of received frames
    UN32 un32AUs;    // Number of received Access Units

    UN32 un32NoMoreBuffers;     // Unable to get a buffer handle for a frame
    UN32 un32BufferOverflow;    // Unable to add to a frame buffer
    UN32 un32OutOfSync;         // Reading data while out of sync
    UN32 un32SyncEmulation;     // Number of types a sync has failed verfication
    UN32 un32UnknownStream;     // Number of frames received that are not part of any connection
    UN32 un32UnexpectedSOA;     // Frame with SOA recieved unexpectedly
    UN32 un32UnexpectedPLPC;    // Frame PLPC doesn't match expected PLPC (doesn't include unex SOA)
    UN32 un32DroppedFrames;     // Frames dropped
    UN32 un32Errors;            // General errors

} SDTP_STATISTICS_STRUCT;

/*
    SDTP Connection Structure

    This structure contains the list of Access Units for the
    PSIs associated with this connection, and the current
    status of each AU.

*/
typedef struct sdtp_connection_struct
{
    // SDTPRx access unit list which defines this unique connection and
    // stores state data for each access unit managed
    OSAL_OBJECT_HDL hAccessUnits;

    // Stats collected for this connection
    SDTP_STATISTICS_STRUCT sStatistics;

} SDTP_CONNECTION_STRUCT;

/*
    SDTP Access Unit structure.

    This structure is used to manage the re-assembly process for
    SDTP packets associated with a particular access unit
*/
typedef struct sdtp_access_unit_struct
{
    // PSI for this access unit
    PSI tPSI;

    // Number of SDTP packets we believe are left in this access unit
    SDTPRx_LEN_COUNT tNumPacketsLeft;

    // The length of this access unit
    SDTPRx_LEN_COUNT tPayloadLen;

    // Buffer handle for this access unit
    OSAL_BUFFER_HDL hPayload;

    // The LL entry this AU belongs to
    OSAL_LINKED_LIST_ENTRY hEntry;

} SDTP_ACCESS_UNIT_STRUCT;

// Receiver defines
#define SDTPRx_INPUT_BUFFER_SIZE ( 256 )
#define SDTPRx_HDR_LEN         ( 3 )

  /***************/
 /** CONSTANTS **/
/***************/

#define SDTP_OUTPUT_PREFIX "SDTP"

// Framing Codes & byte-stuffing sequences
//
// From RX000107-B, pg.6 Section 3
#define SDTPRx_SOF     (0x7E)

// SDTPRx Macros

// SDTPRx Frame Elements and offsets from beginning of the SDTP frame
// All SDTP header fields have sizes that are not byte-aligned
#define SDTPRx_BITFIELDS_OFFSET   ( 0 )
 // These fields are 4 bits, thus they are less than 1 byte
#define SDTPRx_BITFIELDS_BYTE_LEN ( 0 )

#define PSI_OFFSET       ( SDTPRx_BITFIELDS_OFFSET + SDTPRx_BITFIELDS_BYTE_LEN )
// This field is 10 bits, thus it is less than 2 bytes
#define PSI_BYTE_LEN     ( 1 )

#define SDTPRx_COUNT_OFFSET     ( PSI_OFFSET + PSI_BYTE_LEN )
// This field is 10 bits, thus it is less than 2 bytes
#define SDTPRx_COUNT_OFFSET_LEN     ( 1 )

/*
    Bitfield get macros
*/

#define SDTPRx_GET_SOA(x)    (( (x) >> 7) & 0x1)
#define SDTPRx_GET_EOA(x)    (( (x) >> 6) & 0x1)
#define SDTPRx_GET_RFU(x)    (( (x) >> 4) & 0x3)

/*
    OSAL Names for SDTPRx allocations
*/
#define SDTP_CONNECTION_NAME "SDTP"
#define SDTP_AULIST_NAME "AUs"

// defines for outputting SDTP stats
#define SDTP_LOG_STATS                            \
    "\n\tSDTP Connection Statistics:\n"           \
    "\t  Frames: %u  AUs: %u\n"                   \
    "\t  NoMoreBuffers: %u  BufOverflow: %u\n"    \
    "\t  OutOfSync: %u  SyncEmulation: %u\n"      \
    "\t  UnknownStream: %u  UnexpectedSOA: %u\n"  \
    "\t  UnexpectedPLPC: %u  DroppedFrames: %u\n" \
    "\t  Errors:%u\n"

#define SDTP_STATS_CSV "\n\t  CSV: %u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u\n"

  /****************/
 /** PROTOTYPES **/
/****************/

// Local (helper) functions

static BOOLEAN bSyncStream (
    SDTP_STATISTICS_STRUCT *psStats,
    SDTP_CONNECTION_STRUCT *psConnection,
    SDTP_ACCESS_UNIT_STRUCT *psAU,
    SDTP_FRAME_HDR_STRUCT *psFrameHdr,
    OSAL_BUFFER_HDL *phFrame
        );

static void vExtractHeader (
    UN8 *pun8RawHdr,
    SDTP_FRAME_HDR_STRUCT *psHdr
        );

static SDTP_ACCESS_UNIT_STRUCT *psAddAU (
    SDTP_CONNECTION_STRUCT *psConnection,
    PSI tPSI
        );

static SDTP_ACCESS_UNIT_STRUCT *psFindAU (
    SDTP_CONNECTION_STRUCT *psConnection,
    PSI tPSI
        );

static N16 n16CompareAU (
    SDTP_ACCESS_UNIT_STRUCT *psAU1,
    SDTP_ACCESS_UNIT_STRUCT *psAU2
        );

static void vReleasePSINode(
    void *pvData
        );

  /***************/
 /** VARIABLES **/
/***************/

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif    // _SDTPRX_H_
