/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
#include <stdlib.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"
#include "sti_api.h"
#include "sxi.h"
#include "sxi_rw.h"
#include "sxiapi.h"

/*****************************************************************************
*
*       SXIAPI_eStatusMonCmd
*
*       Inputs:
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eStatusMonCmd (
    STI_HDL hSTI,
    SXIAPI_MONITOR_OP_ENUM eOperation,
    const UN8 un8ListCnt,
    const SXIAPI_STATUS_ITEM_ENUM *peMonitorList
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    if (un8ListCnt > SXIAPI_MAX_STATUS_MONITOR_COUNT ||
        peMonitorList == NULL)
    {
        // Error!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_STATUS);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 7.5
        SXIAPI_bWriteUint8(hPayload, (UN8)eOperation);

        if (eOperation == SXIAPI_MONITOR_OP_DISABLE_ALL)
        {
            // Write a list count value of zero, per the spec
            SXIAPI_bWriteUint8(hPayload, 0);
        }
        else
        {
            UN8 un8Idx;

            // Write out the count and the list of monitor IDs
            SXIAPI_bWriteUint8(hPayload, un8ListCnt);

            for (un8Idx = 0; un8Idx < un8ListCnt; un8Idx++)
            {
                SXIAPI_bWriteUint8(hPayload, (UN8)peMonitorList[un8Idx]);
            }
        }

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eFeatureMonCmd
*
*       Inputs:
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eFeatureMonCmd (
    STI_HDL hSTI,
    SXIAPI_MONITOR_OP_ENUM eOperation,
    UN8 un8ListCnt,
    const SXIAPI_FEATURE_MONITOR_ENUM *peMonitorList
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    if (un8ListCnt > SXIAPI_MAX_FEATURE_MONITOR_COUNT ||
         ( (peMonitorList == NULL) && (eOperation != SXIAPI_MONITOR_OP_DISABLE_ALL) ) )
    {
        // Error!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_FEATUREMON);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 7.9
        SXIAPI_bWriteUint8(hPayload,  (UN8)eOperation);

        if (eOperation == SXIAPI_MONITOR_OP_DISABLE_ALL)
        {
            // Write a list count value of zero, per the spec
            SXIAPI_bWriteUint8(hPayload, 0);
        }
        else
        {
            UN8 un8Idx;

            // Write out the count and the list of monitor IDs
            SXIAPI_bWriteUint8(hPayload, un8ListCnt);

            for (un8Idx = 0; un8Idx < un8ListCnt; un8Idx++)
            {
                SXIAPI_bWriteUint8(hPayload, (UN8)peMonitorList[un8Idx]);
            }
        }

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_ePingCmd
*
*       Inputs:
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_ePingCmd (
    STI_HDL hSTI
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_PINGCMD);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

