/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
#include <stdlib.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"
#include "sti_api.h"
#include "sxi.h"
#include "sxi_rw.h"
#include "sxiapi.h"


/*****************************************************************************
*
*       SXIAPI_eDataServiceMonCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           SXIAPI_DATASERVICE_MONITOR_OP_ENUM eMonOp,
*           SXIAPI_DSI tDSI
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eDataServiceMonCmd (
    STI_HDL hSTI,
    SXIAPI_DATASERVICE_MONITOR_OP_ENUM eMonOp,
    SXIAPI_DSI tDSI
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_DATASERVICE);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 14.1
        SXIAPI_bWriteUint8(hPayload, (UN8)eMonOp);
        SXIAPI_bWriteUint16(hPayload, (UN16)tDSI);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eDataServiceFilterCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           SXIAPI_DATASERVICE_FILTER_UPDATE_ENUM eUpdateType,
*           SXIAPI_DSI tDSI,
*           UN8 un8DMICnt,
*           SXIAPI_DMI *ptDMIList,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eDataServiceFilterCmd (
    STI_HDL hSTI,
    SXIAPI_DATASERVICE_FILTER_UPDATE_TYPE_ENUM eUpdateType,
    SXIAPI_DSI tDSI,
    UN8 un8DMICnt,
    SXIAPI_DMI *ptDMIList
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;
    UN8 un8DMIIdx;

    if (un8DMICnt <= SXIAPI_DMI_CNT_MAX)
    {
        // Allocate a payload buffer for this command
        hPayload = STI_hAllocatePayload( hSTI );
        if(hPayload != OSAL_INVALID_BUFFER_HDL)
        {
            STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
            SXIAPI_TRANID tTransactionId;

            // Get transaction id
            tTransactionId =
                SXI_tGetNextTransIDForConnection(hSTI);

            // Populate command header
            SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_DATASERVICE_FILTER);
            SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

            // Populate message
            // See SX-9845-0097, sec 14.6
            SXIAPI_bWriteUint8(hPayload, (UN8)eUpdateType);
            SXIAPI_bWriteUint16(hPayload, (UN16)tDSI);
            SXIAPI_bWriteUint8(hPayload, (UN8)un8DMICnt);

            for (un8DMIIdx = 0; un8DMIIdx < un8DMICnt; un8DMIIdx++)
            {
                SXIAPI_bWriteUint16(hPayload, (UN16)ptDMIList[un8DMIIdx]);
            }

            // Send payload
            eResultCode =
                SXI_eSendPayload(
                    hSTI, hPayload,
                    SXI_DEFAULT_RETRIES,
                    SXI_DEFAULT_TIMEOUT_MSEC,
                    &eStatusCode
                        );
            if((STI_RESULT_CODE_OK != eResultCode) && // ok
                (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                    )
            {
                // Error! Unable to send this payload.
                STI_bFreePayload(hPayload);
            }
        }
    }

    return eStatusCode;
}


