/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
#include <stdlib.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"
#include "sti_api.h"
#include "sxi.h"
#include "sxi_rw.h"
#include "sxiapi.h"

/*****************************************************************************
*
*       SXIAPI_eModuleCfgCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           SXIAPI_MODULECFGCMD_STRUCT *psCfg
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eModuleCfgCmd (
    STI_HDL hSTI,
    SXIAPI_MODULECFGCMD_STRUCT const *psCfg
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;


    // Check input parameters
    if( psCfg == NULL)
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_MODULECFG);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 6.1
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eFading);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eMaxCatLabelLen);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eMaxChanLabelLen);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eMaxMetadataLabelLen);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eMaxPendingInds);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->tConfirmationWaitTime);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eIRControl);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eIRDeleteOnTune);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eIRMarkNewTrack);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->tRecordControl);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->tExtendedControl);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->tPrioritySmartFavCnt);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}


/*****************************************************************************
*
*       SXIAPI_ePwrModeCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           SXIAPI_PWRMODE_ENUM ePwrMode,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_ePwrModeCmd (
    STI_HDL hSTI,
    SXIAPI_PWRMODE_ENUM ePwrMode
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_PWRMODE);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 6.5
        SXIAPI_bWriteUint8(hPayload, (UN8)ePwrMode);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eTimeCfgCmd
*
*       Inputs:
*          SXIAPI_TIMEZONE_ENUM eTimeZone,
*          BOOLEAN bEnableDST,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eTimeCfgCmd (
    STI_HDL hSTI,
    SXIAPI_TIMEZONE_ENUM eTimeZone,
    BOOLEAN bEnableDST
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_TIME);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 6.11
        SXIAPI_bWriteUint8(hPayload, (UN8)eTimeZone);
        SXIAPI_bWriteUint8(hPayload, bEnableDST);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
   }

    return eStatusCode;
}


