/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sti_api.h"
#include "sxi.h"
#include "sxi_rw.h"
#include "sxiapi.h"

#include "_channel.h"

/*****************************************************************************
*
*       SXIAPI_eChanBrowseCmd
*
*       Inputs:
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eChanBrowseCmd (
    STI_HDL hSTI,
    SXIAPI_CHAN_BROWSE_TYPE_ENUM eBrowseType,
    SXIAPI_SID tChanOrSID,
    SXIAPI_CATEGORY_ID tCatID,
    SXIAPI_CH_ATTRIB tOverrides
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_CHANBROWSE);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 8.1
        SXIAPI_bWriteUint8(hPayload,  (UN8)eBrowseType);
        SXIAPI_bWriteUint16(hPayload, (UN16)tChanOrSID);
        SXIAPI_bWriteUint8(hPayload,  (UN8)tCatID);
        SXIAPI_bWriteUint8(hPayload,  (UN8)tOverrides);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eChanSelectCmd
*
*       Inputs:
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eChanSelectCmd (
    STI_HDL hSTI,
    SXIAPI_CHAN_SELECT_TYPE_ENUM eSelType,
    SXIAPI_SID tChanOrSID,
    SXIAPI_CATEGORY_ID tCatID,
    SXIAPI_CH_ATTRIB tOverrides
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_CHANSELECT);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 9.5
        SXIAPI_bWriteUint8(hPayload,  (UN8)eSelType);
        SXIAPI_bWriteUint16(hPayload, (UN16)tChanOrSID);
        SXIAPI_bWriteUint8(hPayload,  (UN8)tCatID);
        SXIAPI_bWriteUint8(hPayload,  (UN8)tOverrides);
        SXIAPI_bWriteUint8(hPayload,  (UN8)SXIAPI_ROUTING_I2S_AUDIO_PATH);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eExtMetadataMonCmd
*
*       Inputs:
*        hSTI - STI connection handle
*        eMonitorSelection - the monitor to enable/disable
*        eOperation - the operation to perform
*        un8ListCnt - then number of EMIs to operate on
*        ptList - the list of EMIs
*        ppsResp - the response structure
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eExtMetadataMonCmd (
    STI_HDL hSTI,
    SXIAPI_EXT_METADATA_MONITOR_ENUM eMonitorSelection,
    SXIAPI_MONITOR_OP_ENUM eOperation,
    UN8 un8ListCnt,
    const SXIAPI_METADATA_IDENTIFIER *ptList
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    if ( un8ListCnt > SXIAPI_MAX_EXT_METADATA_EMI_COUNT ||
         ( (ptList == NULL) && (eOperation != SXIAPI_MONITOR_OP_DISABLE_ALL) ) )
    {
        // Error!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_EXT_METADATAMON);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845, sec 9.9
        SXIAPI_bWriteUint8(hPayload, (UN8)eMonitorSelection);
        SXIAPI_bWriteUint8(hPayload, (UN8)eOperation);

        if (eOperation == SXIAPI_MONITOR_OP_DISABLE_ALL)
        {
            // Write a list count value of zero, per the spec
            SXIAPI_bWriteUint8(hPayload, 0);
        }
        else
        {
            UN8 un8Idx;

            // Write out the number of EMIs we want to monitor
            SXIAPI_bWriteUint8(hPayload, un8ListCnt);

            // write out list of EMIs to monitor
            for (un8Idx = 0; un8Idx < un8ListCnt; un8Idx++)
            {
                SXIAPI_bWriteUint16(hPayload, (UN16)ptList[un8Idx]);
            }
        }

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eSeekMonCmd
*
*       Inputs:
*        hSTI - STI connection handle
*        un8SeekMonId - the ID of the monitor to enable/disable
*        eOperation - the operation to perform
*        eTMI - the TMI to monitor
*        un8ValueCnt - then number of TM values to operate on
*        un8ValueLen - the size of each TM value
*        pvValues - the list of TM Values
*        un8ReportCnt - the number of TMIs to report
*        ptReport - the list of TMIs to report
*        un8SeekControlMask - the Seek Control mask
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eSeekMonCmd (
    STI_HDL hSTI,
    UN8 un8SeekMonId,
    SXIAPI_MONITOR_OP_ENUM eOperation,
    SXIAPI_TMI_ENUM eTMI,
    UN8 un8ValueCnt,
    UN8 un8ValueLen,
    const void *pvValues,
    UN8 un8ReportCnt,
    const SXIAPI_METADATA_IDENTIFIER *ptReport,
    UN8 un8SeekControlMask
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    if ((pvValues == NULL) && (eOperation != SXIAPI_MONITOR_OP_DISABLE) &&
        (eOperation != SXIAPI_MONITOR_OP_DISABLE_ALL))
    {
        // Error!
        return SXIAPI_STATUSCODE_ERROR;
    }

    if ((pvValues == NULL) && (un8ValueCnt != 0 ))
    {
        // Error!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;
        UN8 un8Idx;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_SEEK_MON);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        SXIAPI_bWriteUint8(hPayload, un8SeekMonId);
        SXIAPI_bWriteUint8(hPayload, (UN8)eOperation);

        // Write out the TMI value to monitor
        SXIAPI_bWriteUint16(hPayload, (UN16)eTMI);

        // Write out the number of TM values we want to monitor
        SXIAPI_bWriteUint8(hPayload, un8ValueCnt);

        // Write out the size of each TM value we want to monitor
        SXIAPI_bWriteUint8(hPayload, un8ValueLen);

        // Write out list of TM values to monitor
        for (un8Idx = 0; un8Idx < un8ValueCnt; un8Idx++)
        {
            switch (un8ValueLen)
            {
                case (sizeof(UN8)):
                    SXIAPI_bWriteUint8(hPayload, ((UN8*)pvValues)[un8Idx]);
                    break;
                case (sizeof(UN16)):
                    SXIAPI_bWriteUint16(hPayload, ((UN16*)pvValues)[un8Idx]);
                    break;
                case (sizeof(UN32)):
                    SXIAPI_bWriteUint32(hPayload, ((UN32*)pvValues)[un8Idx]);
                    break;
                default:
                    SXIAPI_bWriteBlob(hPayload, pvValues,
                            (size_t)un8ValueLen * (size_t)un8ValueCnt);
            }
        }
        // Write out the number of Report TMIs we want to monitor
        SXIAPI_bWriteUint8(hPayload, un8ReportCnt);

        // Write out list of Report TMIs to monitor
        for (un8Idx = 0; un8Idx < un8ReportCnt; un8Idx++)
        {
            SXIAPI_bWriteUint16(hPayload, ptReport[un8Idx]);
        }

        // Write out the SeekControl mask
        SXIAPI_bWriteUint8(hPayload, un8SeekControlMask);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eBulletinMonCmd
*
*       Inputs:
*        hSTI - STI connection handle
*        un8BulletinType - the type of the Bulletin monitor
*        un8ParamCnt - then number of parameters in pun16ParamList
*        pun16ParamList - the list of parameters
*
*       Outputs:
*           SXIAPI_STATUSCODE_ENUM
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eBulletinMonCmd (
    STI_HDL hSTI,
    UN8 un8BulletinType,
    UN8 un8ParamCnt,
    const UN16 *pun16ParamList
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    if ((pun16ParamList == NULL) && (un8ParamCnt > 0))
    {
        // Error!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;
        UN8 un8Idx;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_BULLETIN_MON);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        SXIAPI_bWriteUint8(hPayload, un8BulletinType);
        SXIAPI_bWriteUint8(hPayload, un8ParamCnt);

        for (un8Idx = 0; un8Idx < un8ParamCnt; un8Idx++)
        {
            SXIAPI_bWriteUint16(hPayload, pun16ParamList[un8Idx]);
        }

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*   bPopulateSIDV
*
*****************************************************************************/
static BOOLEAN bPopulateSIDV(
    OSAL_BUFFER_HDL *phPayload,
    SXIAPI_SID tServiceId[],
    UN16 un16NumServiceIds
        )
{
    UN16 un16ServiceIdIndex;
    BOOLEAN bSuccess = TRUE;
    UN8 aun8SIDV[SXIAPI_SERVICE_ID_VECTOR_SIZE];

    if (un16NumServiceIds > SXIAPI_SERVICE_ID_BROADCAST_MAX)
    {
        return FALSE;
    }

    // Initialize vector
    OSAL.bMemSet(&aun8SIDV, 0, sizeof(aun8SIDV));

    // Translate ServiceIDs to bitvector
    for(un16ServiceIdIndex = 0;
        un16ServiceIdIndex < un16NumServiceIds;
        un16ServiceIdIndex++)
    {
        // Calculate bit position in the target byte
        UN8 un8BitId = tServiceId[un16ServiceIdIndex] % 8;

        // Calculate target byte
        UN8 un8ByteId =
            (SXIAPI_SERVICE_ID_VECTOR_SIZE - 1 -
                (tServiceId[un16ServiceIdIndex] / 8));

        // Write result into vector
        aun8SIDV[un8ByteId] |= (1 << un8BitId);
    }

    // Populate payload
    for(un16ServiceIdIndex = 0;
        ((un16ServiceIdIndex < SXIAPI_SERVICE_ID_VECTOR_SIZE) && (bSuccess == TRUE));
        un16ServiceIdIndex++)
    {
        bSuccess = SXIAPI_bWriteUint8(*phPayload, aun8SIDV[un16ServiceIdIndex]);
    }

    return bSuccess;
}

/*****************************************************************************
*
*       SXIAPI_eChanAttrListCmd
*
*       Inputs:
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eChanAttrListCmd (
    STI_HDL hSTI,
    SXIAPI_CH_ATTRIB tChangeAttrib,
    SXIAPI_SID tServiceId[],
    UN16 un16NumServiceIds
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;
    UN16 un16ServiceIdIndex;

    if(un16NumServiceIds > SXIAPI_SERVICE_ID_BROADCAST_MAX)
    {
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_CHAN_ATTR_LIST);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX TBD, sec TBD

        SXIAPI_bWriteUint8(hPayload,  (UN8)tChangeAttrib);
        SXIAPI_bWriteUint16(hPayload,  (UN16)un16NumServiceIds);
        for(un16ServiceIdIndex = 0; un16ServiceIdIndex < un16NumServiceIds; un16ServiceIdIndex++)
        {
            SXIAPI_bWriteUint16(hPayload,  (UN16)tServiceId[un16ServiceIdIndex]);
        }

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
   }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eChanAttrCfgCmd
*
*       Inputs:
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eChanAttrCfgCmd (
    STI_HDL hSTI,
    SXIAPI_CH_ATTRIB tChangeAttrib,
    SXIAPI_SID tServiceId[],
    UN16 un16NumServiceIds
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    if (un16NumServiceIds > SXIAPI_SERVICE_ID_BROADCAST_MAX)
    {
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_CHAN_ATTR_CFG);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845-0097, Section 4.2.3

        SXIAPI_bWriteUint8(hPayload, (UN8)tChangeAttrib);
        bPopulateSIDV(&hPayload, tServiceId, un16NumServiceIds);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
   }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eChanSelectCfgCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           SXIAPI_ADV_IR_CFG_CMD_STRUCT *psCfg
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if successful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eChanSelectCfgCmd (
    STI_HDL hSTI,
    SXIAPI_CHAN_SELECT_CFG_CMD_STRUCT *psCfg
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( psCfg == NULL)
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_CHANSELECT_CFG);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX TBD, sec TBD

        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->eFeatureControl);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->ePlayPoint);
        SXIAPI_bWriteUint8(hPayload, (UN8)psCfg->tPlaySeconds);
        SXIAPI_bWriteUint16(hPayload, (UN16)psCfg->tChanScanIncludeMask);
        SXIAPI_bWriteUint16(hPayload, (UN16)psCfg->tChanScanExcludeMask);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}
