/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
#include <stdlib.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"
#include "sti_api.h"
#include "sxi.h"
#include "sxi_rw.h"
#include "sxiapi.h"

/*****************************************************************************
*
*       SXIAPI_eAudioMuteCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           SXIAPI_MUTE_ENUM eMute
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eAudioMuteCmd (
    STI_HDL hSTI,
    SXIAPI_MUTE_ENUM eMute
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( eMute >= SXIAPI_MUTE_UNKNOWN)
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_AUDIOMUTE);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 10.3
        SXIAPI_bWriteUint8(hPayload, (UN8)eMute);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eAudioExciterCmd
*
*       Inputs:
*           STI_HDL hSTI,
*            SXIAPI_EXCITER_ENUM eExciter
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eAudioExciterCmd (
    STI_HDL hSTI,
    SXIAPI_EXCITER_ENUM eExciter
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( eExciter >= SXIAPI_EXCITER_UNKNOWN)
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_EXCITER);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See RX223, sec 10.3
        SXIAPI_bWriteUint8(hPayload, (UN8)eExciter);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eAudioVolCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           N8 n8VolumeAdj
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eAudioVolCmd (
    STI_HDL hSTI,
    N8 n8VolumeAdj
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( (n8VolumeAdj < SXIAPI_VOL_MIN) ||
        (n8VolumeAdj > SXIAPI_VOL_MAX) )
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_AUDIOVOL);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845-0097, sec 10.1
        SXIAPI_bWriteSint8(hPayload, n8VolumeAdj);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eAudioTonalCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           N8 n8BassAdj,
*           N8 n8TrebleAdj
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eAudioTonalCmd (
    STI_HDL hSTI,
    N8 n8BassAdj,
    N8 n8TrebleAdj
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( (n8BassAdj < SXIAPI_TONAL_ADJ_MIN) ||
        (n8BassAdj > SXIAPI_TONAL_ADJ_MAX) )
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    if( (n8TrebleAdj < SXIAPI_TONAL_ADJ_MIN) ||
        (n8TrebleAdj > SXIAPI_TONAL_ADJ_MAX) )
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_AUDIOTONAL);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845-0097, sec 10.5
        SXIAPI_bWriteSint8(hPayload, n8BassAdj);

        SXIAPI_bWriteSint8(hPayload, n8TrebleAdj);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eAudioEqCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           N8 *pn8BandGain
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eAudioEqCmd (
    STI_HDL hSTI,
    N8  *pn8BandGain
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( pn8BandGain == NULL)
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        UN8 un8Band;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_AUDIOEQ);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845-0097, sec 10.7

        for (un8Band=0; un8Band < SXIAPI_MAX_EQ_BANDS; un8Band++)
        {
            SXIAPI_bWriteSint8(hPayload, pn8BandGain[un8Band]);
        }

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
   }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eToneGenerateCmd
*
*       Inputs:
*           STI_HDL hSTI,
*           UN8 un8ToneFreq,
*           UN8 un8AudioToneOptions
*           N8  n8Volume
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eToneGenerateCmd (
    STI_HDL hSTI,
	UN8 un8ToneFreq,
    UN8 un8AudioToneOptions,
    N8  n8ToneVolume
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( un8AudioToneOptions >= SXIAPI_TONE_GEN_OPTIONS_INVALID)
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_TONEGEN);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845-0097, sec 10.3
        SXIAPI_bWriteUint8(hPayload, un8ToneFreq);

        SXIAPI_bWriteUint8(hPayload, un8AudioToneOptions);

        SXIAPI_bWriteSint8(hPayload, n8ToneVolume);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}
