/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the SPORTS ID implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <stdarg.h>

#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sports_obj.h"
#include "sports.h"

#include "cme.h"
#include "cdo_obj.h"
#include "league_obj.h"

// Include SXI SPORTS ID module headers
#include "sxi_sports_id.h"
#include "_sxi_sports_id.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bInitialize
*
* There are a number of static SXi IDs we need to be created to facilitate the
* creation of CIDs and to represent them as unique content ids. This routine
* is responsible for initializing all the static SXi ID content and registering
* it with the content object abstractions.
*
* This needs to be called once when SMS is started.
* It initializes all the CDO static content such as; Report Markets,
* Sports Types, Leagues and Teams.
*
*****************************************************************************/
static BOOLEAN bInitialize ( void )
{
    BOOLEAN bSuccess;

    bSuccess = bInitializeSports();

    // Check if any error occurred
    if(bSuccess == FALSE)
    {
        // Error!
        // We failed initialization, clean up anything which did work
        vUnInitialize();
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vUnInitialize
*
* Un-initialize all SXi ID static content. Needs to be called when SMS is
* stopped.
*
*****************************************************************************/
static void vUnInitialize ( void )
{
    // Note: Leagues and Sports have nothing to uninitialize.

    return;
}

/*****************************************************************************
*
*   bInitializeSports
*
*****************************************************************************/
static BOOLEAN bInitializeSports( void )
{
    size_t tSportIndex;
    BOOLEAN bReturn;

    // Create a CID for each SPORT we support
    for(tSportIndex = 0; tSportIndex < NUM_SPORTS; tSportIndex++)
    {
        CID_OBJECT hId;
        UN32 un32Integer = (UN32)gasSport[tSportIndex].eSubType;

        // Create a CID from the provided PID
        hId = CID_hCreateConst(CID_SXI_SPORTS_ID, (void *)&un32Integer);

        if(hId != CID_INVALID_OBJECT)
        {
            BOOLEAN bAdded;
            // Add this sport
            // We do not need to make a copy of the sport since
            // it is already in ROM.
            bAdded = SPORTS_bAdd(hId, &gasSport[tSportIndex], FALSE);
            if(bAdded == FALSE)
            {
                // Error! Cannot add
                CID_vDestroy(hId);
                break;
            }
        }
        else
        {
            // Error! Cannot create CID
            break;
        }
    }

    // Did we finish all the sports?
    if(tSportIndex != NUM_SPORTS)
    {
        // Error!
        vUnInitialize();
        bReturn = FALSE;
    }
    else
    {
        BOOLEAN bLoadedLeagues;

        bLoadedLeagues = LEAGUE_bLoad(CID_SXI_SPORTS_ID);
        if(bLoadedLeagues == TRUE)
        {
            // Load in teams
            TEAM_bLoad(CID_SXI_SPORTS_ID);
        }

        // always return TRUE because we don't want to prevent sms from running
        // if the leagues/teams weren't loaded. if that happens it still ok we will
        // just have all leagues and teams "unknown"
        bReturn = TRUE;
    }

    return bReturn;
}

/*****************************************************************************
*
*   bParse
*
* Parse provided data to populate the SPORT object. This function uses
* the data to populate the SPORT CDO
* structure.
*
* Inputs:
*
*   hCDO - A handle to a valid CDO to populate.
*   pvData - The pointer to the source data
*
* Outputs:
*
*   BOOLEAN     FALSE if the provided SXI ID was invalid, or parsed but no
*               update was performed because it had not changed from what this
*                CDO's previous SXI ID was. -or-
*               TRUE if the parser successfully parsed the SXI ID and it
*               caused and update (or change) in the CDO's content.
*
*****************************************************************************/
static BOOLEAN bParse (
    CD_OBJECT hCDO,
    const void *pvData
        )
{
    BOOLEAN bUpdated = FALSE;

    // Verify this CDO's type is SPORTS. Anything other than SPORTS
    // makes no sense.
    if(CDO.eType(hCDO) == CDO_SPORTS)
    {
        SPORTS_PARSE_STRUCT *psSports = (SPORTS_PARSE_STRUCT *)pvData;


        SXI_ID_LEAGUE_ITERATOR_STRUCT sStruct;
        sStruct.un8Id = psSports->un8LeagueId;
        sStruct.eSport = SPORTS_UNKNOWN;
        sStruct.bFound = FALSE;

        LEAGUE_bIterateContent(bLeagueIterator, &sStruct);

        if (sStruct.bFound == TRUE)
        {
            CID_OBJECT hId;
            UN32 un32Integer;

            psSports->bFound = TRUE;

            un32Integer = (UN32)sStruct.eSport;

            hId = CDO_hCidCreate(hCDO, CID_SXI_SPORTS_ID, (void *)&un32Integer);

            // sport
            bUpdated = SPORTS_bAssignId(hCDO, hId);

            // We no longer need this CID, put it back in the recycling bin
            CID_vDestroy(hId);

            // broadcast type
            bUpdated |= SPORTS_bAssignBCastType(hCDO, psSports->eBCastType);
            if (psSports->eBCastType == SPORTS_BROADCAST_TEAM)
            {
                bUpdated |= SPORTS_bAssignBCastTeamIndex(hCDO, psSports->un8BCastTeamIndex);
            }

            // league
            bUpdated |= bParseLeague(hCDO, psSports);

            // teams
            bUpdated |= bParseTeams(hCDO, psSports);
        }
        else
        {
            psSports->bFound = FALSE;
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*   bParseLeague
*
*****************************************************************************/
static BOOLEAN bParseLeague(
    CD_OBJECT hCDO,
    const SPORTS_PARSE_STRUCT *psSports
        )
{
    UN32 un32Integer;
    BOOLEAN bUpdated;
    CID_OBJECT hId;
    LEAGUE_ENUM eLeague;

    un32Integer = (UN32)psSports->un8LeagueId;

    eLeague = RADIO_eMapLeague(un32Integer);

    hId = CDO_hCidCreate(hCDO, CID_SXI_SPORTS_ID, (void *)&un32Integer);

    // league
    bUpdated = SPORTS_bAssignLeagueId(hCDO, hId, eLeague);

    // We no longer need this CID, put it back in the recycling bin
    CID_vDestroy(hId);

    return bUpdated;
}

/*****************************************************************************
*
*   bParseTeams
*
*****************************************************************************/
static BOOLEAN bParseTeams(
    CD_OBJECT hCDO,
    const SPORTS_PARSE_STRUCT *psSports
        )
{
    UN32 un32Integer;
    CID_OBJECT hIdTeam1= CID_INVALID_OBJECT, hIdTeam2= CID_INVALID_OBJECT;
    BOOLEAN bUpdated = FALSE, bFlush = FALSE;

    if ( (psSports->un16Team1 == 0) && (psSports->un16Team2 == 0) )
    {
        // no teams
        bUpdated = SPORTS_bFlushTeams(hCDO);
    }
    else
    {
        TEAM_OBJECT hTeam;
        CID_OBJECT hCurrentTeamId = CID_INVALID_OBJECT;

        // did we receive team 1?
        if (psSports->un16Team1 != 0)
        {
            // yes create a cid for this team id
            un32Integer = (UN32)psSports->un16Team1;
            hIdTeam1 = CDO_hCidCreate(
            		hCDO, CID_SXI_SPORTS_ID, (void *)&un32Integer);
        }

        // get the team that is currently in team 1's spot in the cdo
        hTeam = SPORTS.hTeam(hCDO, 0);
        if (hTeam != TEAM_INVALID_OBJECT)
        {
            hCurrentTeamId = TEAM.hId(hTeam);
            if (CID.n16Compare(hCurrentTeamId, hIdTeam1) != 0)
            {
                bFlush |= TRUE;
            }
        }

        // team 2
        if (psSports->un16Team2 != 0)
        {
            un32Integer = (UN32)psSports->un16Team2;
            hIdTeam2 = CDO_hCidCreate(
            		hCDO, CID_SXI_SPORTS_ID, (void *)&un32Integer);
        }

        hTeam = SPORTS.hTeam(hCDO, 1);
        if (hTeam != TEAM_INVALID_OBJECT)
        {
            hCurrentTeamId = TEAM.hId(hTeam);
            if (CID.n16Compare(hCurrentTeamId, hIdTeam2) != 0)
            {
                bFlush |= TRUE;
            }
        }

        if (bFlush == TRUE)
        {
            bUpdated |= SPORTS_bFlushTeams(hCDO);
        }

        if (hIdTeam1 != CID_INVALID_OBJECT)
        {
            bUpdated |= SPORTS_bAssignTeamId(hCDO, hIdTeam1);

            // We no longer need this CID, put it back in the recycling bin
            CID_vDestroy(hIdTeam1);
        }

        if (hIdTeam2 != CID_INVALID_OBJECT)
        {
            bUpdated |= SPORTS_bAssignTeamId(hCDO, hIdTeam2);

            // We no longer need this CID, put it back in the recycling bin
            CID_vDestroy(hIdTeam2);
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*   bLeagueIterator
*
*****************************************************************************/
static BOOLEAN bLeagueIterator(
    LEAGUE_OBJECT hLeague,
    void *pvContentIteratorCallbackArg
        )
{
    CID_OBJECT hThisLeagueId;
    SXI_ID_LEAGUE_ITERATOR_STRUCT *psStruct =
        (SXI_ID_LEAGUE_ITERATOR_STRUCT *)pvContentIteratorCallbackArg;
    UN32 un32Id;
    void *pvId = &un32Id;

    hThisLeagueId = LEAGUE.hId(hLeague);

    CID_n32GetValue(hThisLeagueId, &pvId);

    if ((UN8)(un32Id & 0xFF) == psStruct->un8Id)
    {
        psStruct->eSport = LEAGUE.eSport(hLeague);
        psStruct->bFound = TRUE;
        return FALSE;
    }

    return TRUE;
}
