/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the MARKET ID implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_api.h"
#include "sms_obj.h"

#include "cme.h"
#include "cdo_obj.h"

// Include SXI AMRKET ID module headers
#include "sxi_market_id.h"
#include "_sxi_market_id.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bInitialize
*
* There are a number of static SXi IDs we need to be created to facilitate the
* creation of CIDs and to represent them as unique content ids. This routine
* is responsible for initializing all the static SXi ID content and registering
* it with the content object abstractions.
*
* This needs to be called once when SMS is started.
* It initializes all the CDO static content such as; Report Markets,
* Sports Types, Leagues and Teams.
*
*****************************************************************************/
static BOOLEAN bInitialize ( void )
{
    BOOLEAN bSuccess;

    do
    {
        bSuccess = bInitializeReports();
        if(bSuccess == FALSE)
        {
            // Error!
            break;
        }

    } while (FALSE);

    // Check if any error occurred
    if(bSuccess == FALSE)
    {
        // Error!
        // We failed initialization, clean up anything which did work
        vUnInitialize();
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vUnInitialize
*
* Un-initialize all SXi ID static content. Needs to be called when SMS is
* stopped.
*
*****************************************************************************/
static void vUnInitialize ( void )
{
    vUnInitializeReports();

    return;
}

/*****************************************************************************
*
*   bInitializeReports
*
*****************************************************************************/
static BOOLEAN bInitializeReports ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOk;
    size_t tMarketIndex;

    // Initialize MARKET(REPORT) SXI IDs
    // Used for mapping incoming SXI IDs to textual content information.

    // Create a CID for each MARKET we support
    for(tMarketIndex = 0; tMarketIndex < NUM_MARKETS; tMarketIndex++)
    {
        CID_OBJECT hId;

        // Create a CID from the provided PID
        hId = CID_hCreateConst(
            CID_SXI_MARKET_ID, (void *)&gasMarkets[tMarketIndex].un8MarketNum);

        if(hId != CID_INVALID_OBJECT)
        {
            BOOLEAN bAdded;

            // Add this Market
            bAdded = REPORT_bAdd(hId, &gasMarkets[tMarketIndex].sMarketInfo, FALSE);
            if(bAdded == FALSE)
            {
                // Error! Cannot add
                CID_vDestroy(hId);
                break;
            }
        }
    }

    // Did we finish?
    if(tMarketIndex != NUM_MARKETS)
    {
        // Error!
        vUnInitialize();
        return FALSE;
    }

    // load the cid create function
    bOk = REPORT_bLoadMarketCidCreator(SXI_MARKET_ID_hCreate);
    if (bOk == FALSE)
    {
        // Error!
        vUnInitialize();
        return FALSE;
    }

    // Create a dynamic market list to hold any new markets
    // we might discover.
    eReturnCode = OSAL.eLinkedListCreate(
        &gsSxiMarketIdCtrl.hDiscoveredMarkets,
        SXI_ID_OBJECT_NAME":DiscoveredMarketsList",
        (OSAL_LL_COMPARE_HANDLER)NULL,
        OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE | OSAL_LL_OPTION_PROTECT
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vUnInitialize();
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   vUnInitializeReports
*
*****************************************************************************/
static void vUnInitializeReports ( void )
{
    // Check if discovered market list exists
    if(gsSxiMarketIdCtrl.hDiscoveredMarkets != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy all discovered market entries
        eReturnCode =
            OSAL.eLinkedListRemoveAll(
                gsSxiMarketIdCtrl.hDiscoveredMarkets,
                (OSAL_LL_RELEASE_HANDLER)vDestroyDiscoveredMarket
                    );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy the list itself
            OSAL.eLinkedListDelete(gsSxiMarketIdCtrl.hDiscoveredMarkets);
            gsSxiMarketIdCtrl.hDiscoveredMarkets = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return;
}

/*****************************************************************************
*
*   bParse
*
* Parse provided data to populate the REPORT object. This function uses
* the data to populate the REPORT CDO
* structure.
*
* Inputs:
*
*   hCDO - A handle to a valid CDO to populate.
*   pvData - The pointer to the source data
*
* Outputs:
*
*   BOOLEAN     FALSE if the provided SXI ID was invalid, or parsed but no
*               update was performed because it had not changed from what this
*                CDO's previous SXI ID was. -or-
*               TRUE if the parser successfully parsed the SXI ID and it
*               caused and update (or change) in the CDO's content.
*
*****************************************************************************/
static BOOLEAN bParse (
    CD_OBJECT hCDO,
    const void *pvData
        )
{
    BOOLEAN bMarketIdUpdated = FALSE;

    // Verify this CDO's type is REPORT. Anything other than REPORT
    // makes no sense.
    if(CDO.eType(hCDO) == CDO_REPORT)
    {
        REPORT_OBJECT_STRUCT *psObj;

        // Go get REPORT CDO structure from provided CDO
        psObj = (REPORT_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            if(pvData != NULL)
            {
                CID_OBJECT hId;
                MARKET_PARSE_STRUCT *psStruct;

                psStruct = (MARKET_PARSE_STRUCT *)pvData;

                // Create the CID for this market
                hId = hCreateCID(hCDO, CID_POOL_INVALID_OBJECT, psStruct->un32MarketId);

                bMarketIdUpdated = REPORT_bAssignId(hCDO, hId, &(psStruct->bMktFound));

                // We no longer need this CID, put it back in the recycling bin
                CID_vDestroy(hId);
            }
        }
    }

    return bMarketIdUpdated;
}

/*****************************************************************************
*
*   vDestroyDiscoveredMarket
*
* Destroy a discovered market entry (an entry in the list)
*
*****************************************************************************/
static void vDestroyDiscoveredMarket (
    const MARKET_STRUCT *psMarket
        )
{
    // Destroy object
    SMSO_vDestroy((SMS_OBJECT)psMarket);
    return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
 *
 *   SXI_MARKET_ID_hCreate
 *
 *****************************************************************************/
CID_OBJECT SXI_MARKET_ID_hCreate (
    CID_POOL hCidPool,
    UN32 un32MarketId
        )
{
    CID_OBJECT hMarketId;

    // Create a market CID using the provided cid pool
    //(we don't have a related CDO)
    hMarketId = hCreateCID(CD_INVALID_OBJECT, hCidPool, un32MarketId);

    return hMarketId;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
 *
 *   hCreateCID
 *
 *****************************************************************************/
static CID_OBJECT hCreateCID (
    CD_OBJECT hCDO,
    CID_POOL hCidPool,
    UN32 un32MarketId
        )
{
    CID_OBJECT hMarketId = CID_INVALID_OBJECT;

    if (hCDO == CD_INVALID_OBJECT)
    {
        // No associated CDO provided -- create a constant CID
        hMarketId = CID_hCreate(hCidPool, CID_SXI_MARKET_ID, (void *)&un32MarketId);
    }
    else
    {
        hMarketId = CDO_hCidCreate(hCDO, CID_SXI_MARKET_ID, (void *)&un32MarketId);
    }

    return hMarketId;
}
