/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the RDS implementation.
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_version.h"
#include "sms_api.h"
#include "radio.h"

#include "sxi.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "radio_event_types.h"
#include "dataservice_mgr_obj.h"

// Include radio data service headers
#include "radio_data_service.h"
#include "_radio_data_service.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   RADIO_eDataServiceInit
*
*   This function is used to initialize radio layer data of a service.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceInit (
    RADIO_PRIVATE_DATA_OBJECT hRadioData,
    DATASERVICE_MGR_OBJECT hManager,
    DATASERVICE_ID tDataId
        )
{
    RADIO_DATA_SERVICE_STRUCT *psService =
        (RADIO_DATA_SERVICE_STRUCT *)NULL;
    BOOLEAN bOwner;

    // Verify ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hRadioData);
    if (bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    }

    if (bOwner == FALSE)
    {
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Initialize service
    psService = psCreateDataServiceRadioData(hManager, tDataId);
    if (NULL == psService)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME
            ": Failed to initialize radio data for service %u",
            tDataId);

        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Set radio data
    DATASERVICE_MGR_vSetRadioSpecificData(
        hManager, (RADIO_PRIVATE_DATA_OBJECT)psService);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   RADIO_eDataServiceDestroy
*
*   This function is used to destroy the service resources based on the
*   manager handle.
*
*   NOTE: It is assumed that this function is only invoked after a call
*   to RADIO_eDataServiceStop and it is the ONLY data-related function
*   called after RADIO_eDataServiceStop
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceDestroy (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    RADIO_DATA_SERVICE_STRUCT *psService;
    BOOLEAN bOwner;

    // Ensure we own the manager object
    bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    if (bOwner == FALSE)
    {
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Retrieve this Data Service's radio specific data.
    psService = (RADIO_DATA_SERVICE_STRUCT *)
        DATASERVICE_MGR_hGetRadioSpecificData(hManager);
    if (psService == NULL)
    {
        // Nothin to shut down, no problem!
        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    // Invalidate the data service's handle on this memory
    DATASERVICE_MGR_vSetRadioSpecificData(
        hManager, RADIO_PRIVATE_DATA_INVALID_OBJECT);

    vDestroyDataServiceRadioData(psService);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   RADIO_eDataServiceManageStream
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceManageStream (
    RADIO_PRIVATE_DATA_OBJECT hDataCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DATASERVICE_DMI_CONFIG_STRUCT const *psDMIsToConfigure,
    size_t tNumDMIsToConfigure
        )
{
    BOOLEAN bOk = TRUE;
    RADIO_DATA_SERVICE_STRUCT *psService;
    UN8 un8ModifyIndex;
    RADIO_DATA_UPDATE_DMI_FILTER_ITERATOR_ARG sArg;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_SUCCESS;

    // Verify inputs
    if (((DATASERVICE_DMI_CONFIG_STRUCT *)NULL == psDMIsToConfigure) ||
        (0 == tNumDMIsToConfigure))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Validate ownership
    bOk = SMSO_bOwner((SMS_OBJECT)hDataCtrl);
    if (TRUE == bOk)
    {
        bOk = SMSO_bOwner((SMS_OBJECT)hManager);
    }

    if (FALSE == bOk)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Not owner of manager %p",
            hManager);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Retrieve this Data Service's radio specific data.
    psService = (RADIO_DATA_SERVICE_STRUCT *)
        DATASERVICE_MGR_hGetRadioSpecificData(hManager);

    if (NULL == psService)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Since demux list is only modified from the same context
    // locking is not needed for searching and reading data
    for (un8ModifyIndex = 0; un8ModifyIndex < tNumDMIsToConfigure;
        un8ModifyIndex++)
    {
        RADIO_DMI_STRUCT *psDMI;

        psDMI = psRadioDataFindDMI(
            psService->hDemuxList,
            psDMIsToConfigure[un8ModifyIndex].tDMI);
        if (NULL != psDMI)
        {
            SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
                PC_GREEN"DMI %u is found in the Demux List. "
                "Mark as %s.\n"PC_RESET,
                psDMIsToConfigure[un8ModifyIndex].tDMI,
                psDMIsToConfigure[un8ModifyIndex].bEnable ? "ENABLED" : "DISABLED");

            // It is safe to modify this field without locking demux list
            // because it is not used in other contexts
            vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_ENABLED,
                psDMIsToConfigure[un8ModifyIndex].bEnable);
        }
        else
        {
            SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 1,
                PC_BLUE"DMI %u (%s) is not found in the Demux List.\n"PC_RESET,
                psDMIsToConfigure[un8ModifyIndex].tDMI,
                psDMIsToConfigure[un8ModifyIndex].bEnable ? "ENABLED" : "DISABLED");

            // This means that requested DMI is currently unavailable (unsubscribed).

            // Enabled DMI should be added to the Demux List
            // so it can be added to the DMI Filter when DMI becomes available.
            if (TRUE == psDMIsToConfigure[un8ModifyIndex].bEnable)
            {
                SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
                    PC_BLUE"Try to add empty DMI %u (ENABLED).\n"PC_RESET,
                    psDMIsToConfigure[un8ModifyIndex].tDMI);

                // Create empty DMI and add it to the Service's Demux List.
                psDMI = psRadioDataCreateNewDMI(psDMIsToConfigure[un8ModifyIndex].tDMI);
                if (NULL == psDMI)
                {
                    eResult = SMSAPI_RETURN_CODE_ERROR;
                }
                else
                {
                    bOk = bRadioDataRegisterDMIinDemuxList(psService, psDMI);
                    if (TRUE == bOk)
                    {
                        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
                            PC_GREEN"DMI %u is successfully added to the Demux List. "
                            "Mark as %s.\n"PC_RESET,
                            psDMIsToConfigure[un8ModifyIndex].tDMI,
                            psDMIsToConfigure[un8ModifyIndex].bEnable ? "ENABLED" : "DISABLED");

                        vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_ENABLED,
                            psDMIsToConfigure[un8ModifyIndex].bEnable);
                    }
                    else
                    {
                        // Rollback
                        vRadioDataDestroyDMI(psDMI);
                        eResult = SMSAPI_RETURN_CODE_ERROR;
                    }
                }
            }
        }
    }

    // Update DMI filter for all DSIs used by this service
    // because the affected DSIs were not tracked
    sArg.tDSI = DSI_INVALID_ID;
    sArg.bTurnFilterOn = TRUE;
    sArg.bSuccess = TRUE;
    sArg.psDataCtrl = (RADIO_DATA_CTRL_STRUCT *)hDataCtrl;

    eOsalReturnCode = OSAL.eLinkedListIterate(
        psService->hDSIList,
        (OSAL_LL_ITERATOR_HANDLER)bRadioDataUpdateDMIFilterIterator,
        &sArg);

    if ((FALSE == sArg.bSuccess) ||
        ((OSAL_NO_OBJECTS != eOsalReturnCode) &&
         (OSAL_SUCCESS != eOsalReturnCode)))
    {
        eResult = SMSAPI_RETURN_CODE_ERROR;
    }

    // eResult == SMSAPI_RETURN_CODE_ERROR
    // if not all DMIs are successfully added to the Demux List or to the DMI Filter
    return eResult;
}

/*****************************************************************************
*
*   RADIO_eDataServiceControlFlow
*
*   This function is used to enable/disable the flow of data for a data service
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceControlFlow (
    RADIO_PRIVATE_DATA_OBJECT hDataCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DSI tDSI,
    BOOLEAN bEnableData
        )
{
    BOOLEAN bOwner;
    RADIO_DATA_SERVICE_STRUCT *psService;
    RADIO_DATA_UPDATE_DMI_FILTER_ITERATOR_ARG sArg;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    // Validate ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDataCtrl);
    if (TRUE == bOwner)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    }

    if (FALSE == bOwner)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Not owner of manager %p",
            hManager);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Retrieve this Data Service's radio specific data.
    psService = (RADIO_DATA_SERVICE_STRUCT *)
        DATASERVICE_MGR_hGetRadioSpecificData(hManager);

    if (psService == NULL)
    {
        // Nothing to do!
        return SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    // Turn on/off DMI filters for the given DSI
    sArg.tDSI = tDSI;
    sArg.bTurnFilterOn = bEnableData;
    sArg.bSuccess = TRUE;
    sArg.psDataCtrl = (RADIO_DATA_CTRL_STRUCT *)hDataCtrl;

    eOsalReturnCode = OSAL.eLinkedListIterate(
        psService->hDSIList,
        (OSAL_LL_ITERATOR_HANDLER)bRadioDataUpdateDMIFilterIterator,
        (void *)&sArg);

    if (OSAL_NO_OBJECTS != eOsalReturnCode)
    {
        if (OSAL_SUCCESS != eOsalReturnCode)
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        if (FALSE == sArg.bSuccess)
        {
            return SMSAPI_RETURN_CODE_INVALID_OPTIONS;
        }
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   RADIO_eDataServiceStop
*
*   This function is used to stop a service based on the manager handle.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceStop (
    RADIO_PRIVATE_DATA_OBJECT hDataCtrl,
    DATASERVICE_MGR_OBJECT hManager
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        BOOLEAN bOwner;
        RADIO_DATA_SERVICE_STRUCT *psService;
        RADIO_DATA_CTRL_STRUCT *psDataCtrl = 
            (RADIO_DATA_CTRL_STRUCT *)hDataCtrl;

        // Validate ownership
        bOwner = SMSO_bOwner((SMS_OBJECT)hDataCtrl);
        if (TRUE == bOwner)
        {
            bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
        }

        if (FALSE == bOwner)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": Not owner of manager %p",
                hManager);
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Retrieve this Data Service's radio specific data.
        psService = (RADIO_DATA_SERVICE_STRUCT *)
            DATASERVICE_MGR_hGetRadioSpecificData(hManager);

        // If we have something here then we have some work to do
        if (psService != NULL)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            // Stop monitoring all associated DSIs
            eOsalReturnCode = OSAL.eLinkedListIterate(
                psService->hDSIList,
                (OSAL_LL_ITERATOR_HANDLER)bRadioDataStopDSIIterator,
                (void *)psDataCtrl);

            if ((OSAL_SUCCESS != eOsalReturnCode) &&
                (OSAL_NO_OBJECTS != eOsalReturnCode))
            {
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }
        }

        return SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   RADIO_eDataServiceAddDSI
*
*   This function is used to add a DSI to an existing data service
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceAddDSI (
    RADIO_PRIVATE_DATA_OBJECT hDataCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DSI tDSI,
    size_t tSuggestedOTABufferByteSize,
    BOOLEAN bEnableAllDMIs
        )
{
    BOOLEAN bOwner;
    RADIO_DATA_SERVICE_STRUCT *psService;
    RADIO_DATA_CTRL_STRUCT *psDataCtrl = 
        (RADIO_DATA_CTRL_STRUCT *)hDataCtrl;
    BOOLEAN bOk;

    // Validate ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDataCtrl);
    if (TRUE == bOwner)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    }

    if (FALSE == bOwner)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Not owner of manager %p",
            hManager);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Retrieve this Data Service's radio specific data.
    psService = (RADIO_DATA_SERVICE_STRUCT *)
        DATASERVICE_MGR_hGetRadioSpecificData(hManager);

    if (psService == NULL)
    {
        // Nothing to do!
        return SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    // Add DSI
    bOk = bRadioDataCreateDSI(
        psDataCtrl, psService, tDSI, 
        tSuggestedOTABufferByteSize, bEnableAllDMIs);

    if (FALSE == bOk)
    {
        SMSAPI_DEBUG_vPrintErrorFull(
                    gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": cannot create DSI %u object",
            tDSI);
        return SMSAPI_RETURN_CODE_ERROR;
    }

    SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 4,
        "Start DSI %u\n", tDSI);

    // Request that we monitor this DSI (don't worry about
    // failures here. If the link is dead SMS will
    // become aware of it. If the link is busy and we just
    // didn't get a CFM in time it'll make this call return a
    // failure, but that failure we be a false negative
    (void) SXIAPI_eDataServiceMonCmd(
        psDataCtrl->hControlCxn,
        SXIAPI_DATASERVICE_MONITOR_OP_START,
        (SXIAPI_DSI)tDSI);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   RADIO_eDataServiceRemoveDSI
*
*   This function is used to remove a DSI from an existing data service
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceRemoveDSI (
    RADIO_PRIVATE_DATA_OBJECT hDataCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DSI tDSI
        )
{
    BOOLEAN bOwner;
    RADIO_DATA_SERVICE_STRUCT *psService;
    RADIO_DATA_CTRL_STRUCT *psDataCtrl = 
        (RADIO_DATA_CTRL_STRUCT *)hDataCtrl;
    RADIO_DSI_STRUCT *psDSI = (RADIO_DSI_STRUCT *)NULL;

    // Validate ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDataCtrl);
    if (TRUE == bOwner)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    }

    if (FALSE == bOwner)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Not owner of manager %p",
            hManager);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Retrieve this Data Service's radio specific data.

    psService = (RADIO_DATA_SERVICE_STRUCT *)
        DATASERVICE_MGR_hGetRadioSpecificData(hManager);

    if (NULL != psService)
    {
        psDSI = psRadioDataFindDSI(psService, tDSI);
    }

    if (NULL == psDSI)
    {
        return SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    // Stop monitoring it
    vRadioDataStopDSI(psDataCtrl, psDSI);

    // Disconnect from SDTP now
    if ( SDTP_INVALID_HDL != psDSI->hSDTP )
    {
       SDTP_vDisconnect(psDSI->hSDTP);
       psDSI->hSDTP = SDTP_INVALID_HDL;
    }

    // Destroy DSI
    vRadioDataDestroyDSI(psDSI);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   RADIO_eDataServiceGetDMIsCount
*
*   This function is used to get the number of available DMIs
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceGetDMIsCount (
    DATASERVICE_MGR_OBJECT hManager,
    DSI tDSI,
    UN8 *pun8DMIsCount
        )
{
    BOOLEAN bOwner;
    RADIO_DATA_SERVICE_STRUCT *psService;
    RADIO_DSI_STRUCT *psDSI = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    RADIO_DATA_DMI_INFO_ITERATOR_ARG sArg;

    if (NULL == pun8DMIsCount)
    {
        return SMSAPI_RETURN_CODE_BAD_ARGUMENT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    if (FALSE == bOwner)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Not owner of manager %p",
            hManager);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    psService = (RADIO_DATA_SERVICE_STRUCT *)
        DATASERVICE_MGR_hGetRadioSpecificData(hManager);
    if (NULL != psService)
    {
        psDSI = psRadioDataFindDSI(psService, tDSI);
    }

    if (NULL == psDSI)
    {
        return SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    // Retrieve DMIs count for specified DSI.
    sArg.tDSI = tDSI;
    sArg.un8DMIsCountMax = 0;
    sArg.ptDMIs = NULL;
    sArg.apsDMIs = NULL;
    sArg.un8DMIsCount = 0;

    eOsalReturnCode = OSAL.eLinkedListIterate(
        psService->hDemuxList,
        (OSAL_LL_ITERATOR_HANDLER)bRadioDataDMIInfoIterator,
        &sArg);
    if (OSAL_SUCCESS != eOsalReturnCode)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    *pun8DMIsCount = sArg.un8DMIsCount;

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   RADIO_eDataServiceGetDMIs
*
*   This function is used to get the list of available DMIs
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eDataServiceGetDMIs (
    DATASERVICE_MGR_OBJECT hManager,
    DSI tDSI,
    SXM_DMI *ptDMIs,
    UN8 un8DMIsSize
        )
{
    BOOLEAN bOwner;
    RADIO_DATA_SERVICE_STRUCT *psService;
    RADIO_DSI_STRUCT *psDSI = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    RADIO_DATA_DMI_INFO_ITERATOR_ARG sArg;

    if ((NULL == ptDMIs) || (0 == un8DMIsSize))
    {
        return SMSAPI_RETURN_CODE_BAD_ARGUMENT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    if (FALSE == bOwner)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Not owner of manager %p",
            hManager);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Retrieve this Data Service's radio specific data.
    psService = (RADIO_DATA_SERVICE_STRUCT *)
        DATASERVICE_MGR_hGetRadioSpecificData(hManager);
    if (NULL != psService)
    {
        psDSI = psRadioDataFindDSI(psService, tDSI);
    }

    if (NULL == psDSI)
    {
        return SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    // Retrieve DMIs for specified DSI.
    sArg.tDSI = tDSI;
    sArg.un8DMIsCountMax = un8DMIsSize;
    sArg.ptDMIs = ptDMIs;
    sArg.apsDMIs = NULL;
    sArg.un8DMIsCount = 0;

    eOsalReturnCode = OSAL.eLinkedListIterate(
        psService->hDemuxList,
        (OSAL_LL_ITERATOR_HANDLER)bRadioDataDMIInfoIterator,
        &sArg);
    if (OSAL_SUCCESS != eOsalReturnCode)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Empty array's tail
    while (sArg.un8DMIsCount != un8DMIsSize)
    {
        sArg.ptDMIs[sArg.un8DMIsCount++] =
            SXM_DMI_INVALID_ID;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
 *
 *   RADIO_hDataServiceInitialize
 *
 *   This function is used to initialize data services on the radio.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DSM.
 *
 *****************************************************************************/
RADIO_PRIVATE_DATA_OBJECT RADIO_hDataServiceInitialize(
    SMS_OBJECT hDataCtrl,
    STI_HDL hSTI
        )
{
    RADIO_DATA_CTRL_STRUCT *psDataCtrl =
        (RADIO_DATA_CTRL_STRUCT *)NULL;
    BOOLEAN bOwner;

    // Verify inputs, caller owns the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hDataCtrl);
    if(bOwner == FALSE)
    {
        // Error!
        return RADIO_PRIVATE_DATA_INVALID_OBJECT;
    }

    // Allocate radio specific data for data services now
    psDataCtrl = (RADIO_DATA_CTRL_STRUCT *)SMSO_hCreate(
        RADIO_DATA_SERVICE_NAME":Data", sizeof(RADIO_DATA_CTRL_STRUCT),
        (SMS_OBJECT)hDataCtrl, // Object is parent,
        FALSE                  // inherit lock-feature
        );
    if(psDataCtrl == NULL)
    {
        // Error!
        return RADIO_PRIVATE_DATA_INVALID_OBJECT;
    }

    // We need to get the connection handle to use from the MODULE
    // object this DSM is associated with and assigned to.
    psDataCtrl->hControlCxn = hSTI;

    // Associate the data control object with this one
    psDataCtrl->hSMSDataCtrl = hDataCtrl;

#if SMS_LOGGING == 1

    // Store the current uptime in our last log attribute
    psDataCtrl->un32LastLogSeconds = 0;
    OSAL.vTimeUp(&psDataCtrl->un32LastLogSeconds, (UN16 *)NULL);

#endif

    return (RADIO_PRIVATE_DATA_OBJECT)psDataCtrl;
}

/*****************************************************************************
*
 *   RADIO_vDataServiceUninitialize
 *
 *   This function is used to uninitialize the radio-specific data object
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DSM.
 *
 *****************************************************************************/
void RADIO_vDataServiceUninitialize (
    RADIO_PRIVATE_DATA_OBJECT hRadioDataCtrl
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hRadioDataCtrl);
    if (bOwner == TRUE)
    {
        RADIO_DATA_CTRL_STRUCT *psDataCtrl =
            (RADIO_DATA_CTRL_STRUCT *)hRadioDataCtrl;

        // Clear handles to things we don't own
        psDataCtrl->hSMSDataCtrl = SMS_INVALID_OBJECT;

        // Clear STI handle
        psDataCtrl->hControlCxn = STI_INVALID_HDL;

        // Destroy control object
        SMSO_vDestroy((SMS_OBJECT)psDataCtrl);
    }

    return;
}

/*****************************************************************************
 *
 *   RADIO_bDataServiceEventHandler
 *
 *   This event handler is called by SMS whenever the DSM needs
 *   to process an event and that event has not been handled by SMS itself
 *   (meaning the event should be handled by specific RADIO hardware).
 *
 *   NOTE: This handler always runs in the context of the DSM
 *   thus it should be assumed the implementation here has exclusive access
 *   to the object already.
 *
 *   Inputs:
 *       hRadioDataCtrl - A valid handle to Data Control object mapped to the
 *           physical hardware to process this event.
 *       hEvent - A handle to the event being processed.
 *
 *   Returns:
 *       BOOLEAN - TRUE if this handler handled the event. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bDataServiceEventHandler(
    RADIO_PRIVATE_DATA_OBJECT hRadioDataCtrl,
    SMS_EVENT_TYPE_ENUM eEventType,
    const void *pvEventData
        )
{
    SMS_EVENT_DATA_UNION const *puEventData =
        (SMS_EVENT_DATA_UNION const *)pvEventData;
    BOOLEAN bOwner, bHandledEvent = TRUE;
    RADIO_DATA_CTRL_STRUCT *psDataCtrl =
        (RADIO_DATA_CTRL_STRUCT *)hRadioDataCtrl;

    bOwner = SMSO_bOwner((SMS_OBJECT)hRadioDataCtrl);
    if (bOwner == FALSE)
    {
        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 4,
            "Ignored DSM RADIO event: %d, hRadioDataCtrl: %p\n",
            eEventType, hRadioDataCtrl);
        return FALSE;
    }

    // Process event
    switch((RADIO_EVENT_TYPE_ENUM)eEventType)
    {
        case RADIO_EVENT_SXI_RX_DATA:
        {
            SXIAPI_OPTYPE tOpType;
            SXIAPI_INDCODE_ENUM eIndCode;
            SXIAPI_DATASERVICESTATUSIND_STRUCT sDataStatus;
            SXIAPI_RX_STRUCT *psRxData =
                (SXIAPI_RX_STRUCT *)*puEventData->sRadio.apvData;

            // Extract data from the Rx data
            tOpType = psRxData->sHdr.tOpType;
            eIndCode = psRxData->uData.sInd.eIndCode;
            sDataStatus = psRxData->uData.sInd.uData.sDataStatus;

            // Free data
            SXI_vFreeResponse(psRxData);

            // Process SXI Data Ind
            vProcessDataInd(psDataCtrl, tOpType, eIndCode, &sDataStatus);
        }
        break;

        case RADIO_EVENT_SXI_DATA_PAYLOAD:
        {
            SXIAPI_DATA_PAYLOAD_EVENT_STRUCT *psDataPayload =
                (SXIAPI_DATA_PAYLOAD_EVENT_STRUCT *)
                    puEventData->sRadio.apvData;
            RADIO_DSI_STRUCT *psDSI;
            OSAL_BUFFER_HDL hInputPayload = psDataPayload->hPayload;

            // Find the destination service for this payload
            psDSI = psRadioDataFindDSIAnywhere(psDataCtrl, psDataPayload->tDSI);

            if (NULL != psDSI)
            {
                OSAL_BUFFER_HDL hOutputPayload;

                // If this is SDTP then we need to re-assemble it now
                if (psDataPayload->eType == SXIAPI_DATA_PACKET_TYPE_SDTP)
                {
                    // Send the payload to the SDTP connection,
                    // use whatever it returns to us
                    hOutputPayload = hHandleSDTPPacket(psDSI, &hInputPayload);
                }
                else
                {
                    // Just a transfer
                    hOutputPayload = hInputPayload;
                    hInputPayload = OSAL_INVALID_BUFFER_HDL;
                }

                // Do we still have something to send?
                if (hOutputPayload != OSAL_INVALID_BUFFER_HDL)
                {
                    // Inform the manager that we have received
                    // a new payload
                    DATASERVICE_MGR_vReceivePayload(
                        psDSI->psService->hManager,
                        psDSI->tDSI,
                        hOutputPayload);
                }

#if SMS_LOGGING == 1
                {
                    UN32 un32Time;
                    BOOLEAN bUpdateLog = FALSE;

                    // How long have we been up?
                    OSAL.vTimeUp(&un32Time, (UN16 *)NULL);

                    // Is it time to log data service stats?
                    if (un32Time < psDataCtrl->un32LastLogSeconds)
                    {
                        // I don't know how long it's been,
                        // so let's just print the log
                        bUpdateLog = TRUE;
                    }
                    else if ((un32Time - psDataCtrl->un32LastLogSeconds)
                               >= DS_LOGGING_RATE_SECONDS)
                    {
                        // It's time to update
                        bUpdateLog = TRUE;
                    }

                    if (bUpdateLog == TRUE)
                    {
                        // Yes!
                        psDataCtrl->un32LastLogSeconds = un32Time;

                        DATASERVICE_MGR_vLog(DS_LOGGING_BANNER);

                        DATASERVICE_MGR_bIterateRadioData(
                            psDataCtrl->hSMSDataCtrl,
                            (DATASERVICE_RADIO_DATA_ITERATOR)bIterateServicesForLog,
                            (void *)psDataPayload->hBlockPool);

                        DATASERVICE_MGR_vLog(DS_LOGGING_END);
                    }
                }
#endif
            }

            // Free payloads if they still exist
            RADIO_bFreeDataPayload(hInputPayload);
        }
        break;

        default:
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": Unhandled RADIO event.\n");
            bHandledEvent = FALSE;
        break;
    };

    return bHandledEvent;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
 *
 *   psRadioDataFindDMI
 *
 *****************************************************************************/
static RADIO_DMI_STRUCT *psRadioDataFindDMI (
    OSAL_OBJECT_HDL hDemuxList,
    SXM_DMI tDMI
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    RADIO_DMI_STRUCT sSearchCriteria;
    RADIO_DMI_STRUCT *psDMI = (RADIO_DMI_STRUCT *) NULL;

    sSearchCriteria.tDMI = tDMI;

    eOsalReturnCode = OSAL.eLinkedListSearch(
        hDemuxList,
        &hEntry,
        &sSearchCriteria);

    if (OSAL_SUCCESS == eOsalReturnCode)
    {
        psDMI = (RADIO_DMI_STRUCT *)OSAL.pvLinkedListThis(hEntry);
    }

    return psDMI;
}

/*****************************************************************************
 *
 *   bRadioDataFindDSIIterator
 *
 *****************************************************************************/
static BOOLEAN bRadioDataFindDSIIterator (
    RADIO_PRIVATE_DATA_OBJECT hRadioData,
    RADIO_DATA_FIND_DSI_ITERATOR_ARG *psArg
        )
{
    RADIO_DATA_SERVICE_STRUCT *psService =
        (RADIO_DATA_SERVICE_STRUCT *)hRadioData;

    if (psService == NULL)
    {
        // Skip this entry
        return TRUE;
    }

    psArg->psDSI = psRadioDataFindDSI(psService, psArg->tDSI);

    // continue searching if not found
    return (NULL == psArg->psDSI);
}

/*****************************************************************************
 *
 *   bRadioDataUpdateDMIFilterIterator
 *
 *****************************************************************************/
static BOOLEAN bRadioDataUpdateDMIFilterIterator (
    RADIO_DSI_STRUCT *psDSI,
    RADIO_DATA_UPDATE_DMI_FILTER_ITERATOR_ARG *psArg
        )
{
    // An invalid DSI means "all DSIs".  Otherwise
    // match the given DSI
    if ((DSI_INVALID_ID == psArg->tDSI) ||
        (psDSI->tDSI == psArg->tDSI))
    {
        // Update DMI filter
        psArg->bSuccess = bRadioDataUpdateDMIFilter(
            psArg->psDataCtrl, psDSI, psArg->bTurnFilterOn);
    }

    // stop iterating in error case
    return psArg->bSuccess;
}

/*****************************************************************************
 *
 *   bRadioDataUpdateDMIFilter
 *
 *****************************************************************************/
static BOOLEAN bRadioDataUpdateDMIFilter (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    RADIO_DSI_STRUCT *psDSI,
    BOOLEAN bTurnFilterOn
        )
{
    SXIAPI_DMI atDMIs[SXIAPI_DMI_CNT_MAX];
    RADIO_DATA_DMI_INFO_ITERATOR_ARG sArg;
    SXIAPI_STATUSCODE_ENUM eStatusCode;

    sArg.tDSI = psDSI->tDSI;
    sArg.un8DMIsCountMax = SXIAPI_DMI_CNT_MAX;
    sArg.ptDMIs = atDMIs;
    sArg.apsDMIs = NULL;
    sArg.un8DMIsCount = 0;

    if (TRUE == bTurnFilterOn)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Compose DMI filter
        // (add available & enabled DMIs for specified DSI to sArg)
        eOsalReturnCode = OSAL.eLinkedListIterate(
            psDSI->psService->hDemuxList,
            (OSAL_LL_ITERATOR_HANDLER)bRadioDataFilterComposeIterator,
            &sArg);

        if (OSAL_SUCCESS != eOsalReturnCode)
        {
            if (OSAL_NO_OBJECTS == eOsalReturnCode)
            {
                // no associated DMIs -- nothing to do
                return TRUE;
            }

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": Failed to compose "
                "dataservice filter for DSI %u.",
                psDSI->tDSI);

            return FALSE;
        }
    }

    SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
        PC_GREEN"Send request to update the filter "
        "(DSI %u; %u DMIs).\n"PC_RESET,
        psDSI->tDSI, sArg.un8DMIsCount);

    // Request to update the filter
    eStatusCode = SXIAPI_eDataServiceFilterCmd(
        psDataCtrl->hControlCxn,
        SXIAPI_DATASERVICE_FILTER_UPDATE_TYPE_START,
        psDSI->tDSI, sArg.un8DMIsCount, sArg.ptDMIs);

    if (SXIAPI_STATUSCODE_MSG_RECEIVED != eStatusCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME
            ": SXIAPI_eDataServiceFilterCmd() returned %u.",
            eStatusCode);

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   bRadioDataStopDSIIterator
 *
 *****************************************************************************/
static BOOLEAN bRadioDataStopDSIIterator (
    RADIO_DSI_STRUCT *psDSI,
    RADIO_DATA_CTRL_STRUCT *psDataCtrl
        )
{
    // stop monitoring this DSI
    vRadioDataStopDSI(psDataCtrl, psDSI);

    if (SDTP_INVALID_HDL != psDSI->hSDTP)
    {
        // Disconnect from SDTP now
        SDTP_vDisconnect(psDSI->hSDTP);
        psDSI->hSDTP = SDTP_INVALID_HDL;

        // This feature is inactive now, this flag
        // is necessary because the SDTP connection
        // is made automatically if we recieve an SDTP
        // frame and we don't have an SDTP connection
        // at the time of reception.  Well, we don't
        // want the SDTP re-built for this DSI anymore
        psDSI->bSDTPActive = FALSE;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   vRadioDataStopDSI
 *
 *****************************************************************************/
static void vRadioDataStopDSI (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    RADIO_DSI_STRUCT *psDSI
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode;

    SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 4,
        "Stop DSI %u\n", psDSI->tDSI);

    // Request that we stop monitor for this DSI
    eStatusCode = SXIAPI_eDataServiceMonCmd(
        psDataCtrl->hControlCxn,
        SXIAPI_DATASERVICE_MONITOR_OP_STOP,
        (SXIAPI_DSI)psDSI->tDSI);

    if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        // Just keep going if we have a failure here.
        // We can still shut our service down and just trash
        // the payloads/indications for this service if
        // and when they arrive
        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 1,
                PC_BLUE"SXIAPI_eDataServiceMonCmd() failed. "
                "Status Code %d.\n"PC_RESET,
                eStatusCode);
    }

    // Destroy associated DMIs, ignore result
    bRadioDataDestroyAllDMIs(psDSI);

    return;
}

/*****************************************************************************
 *
 *   bRadioDataCreateDSI
 *
 *****************************************************************************/
static BOOLEAN bRadioDataCreateDSI (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    RADIO_DATA_SERVICE_STRUCT *psService,
    DSI tDSI,
    size_t tSuggestedOTABufferByteSize,
    BOOLEAN bEnableAllDMIs
        )
{
    RADIO_DSI_STRUCT * psDSI;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    // ensure this DSI is not used by any other service
    psDSI = psRadioDataFindDSIAnywhere(psDataCtrl, tDSI);
    if (NULL != psDSI)
    {
        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 1,
                PC_BLUE"DSI %u is already used by service %u.\n"PC_RESET,
                tDSI, DATASERVICE_MGR_tGetDataId(psDSI->psService->hManager));
        return FALSE;
    }

    psDSI = (RADIO_DSI_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
        RADIO_DATA_SERVICE_NAME" DataService ID",
        sizeof(RADIO_DSI_STRUCT),
        FALSE);

    if (NULL == psDSI)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": cannot allocate memory for DSI %u",
            tDSI);
        return FALSE;
    }

    psDSI->tDSI = tDSI;
    psDSI->bSDTPActive = TRUE;
    psDSI->tSuggestedOTABufferByteSize = tSuggestedOTABufferByteSize;
    psDSI->bEnableAllDMIs = bEnableAllDMIs;
    psDSI->eStatus = SXIAPI_DATASERVICE_STATUS_UNKNOWN;
    psDSI->psService = psService;
    psDSI->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psDSI->hSDTP = SDTP_INVALID_HDL;

    do
    {
        // Add DSI entry to the DSI list
        eOsalReturnCode = OSAL.eLinkedListAdd(
            psService->hDSIList,
            &psDSI->hEntry,
            psDSI);
        if (OSAL_SUCCESS != eOsalReturnCode)
        {
            if (OSAL_ERROR_LIST_ITEM_NOT_UNIQUE == eOsalReturnCode)
            {
                SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 1,
                    PC_BLUE"DSI %u is already used.\n"PC_RESET,
                    tDSI);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_DATA_SERVICE_NAME": cannot add DSI %u",
                    tDSI);
            }
            break;
        }

        // Successfully created
        return TRUE;

    } while (FALSE);

    // Rollback
    vRadioDataDestroyDSI(psDSI);
    return FALSE;
}

/*****************************************************************************
 *
 *   vRadioDataDestroyDSI
 *
 *****************************************************************************/
static void vRadioDataDestroyDSI (
    RADIO_DSI_STRUCT *psDSI
        )
{
    BOOLEAN bOk;

    bOk = bRadioDataDestroyAllDMIs(psDSI);
    if (TRUE != bOk)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": cannot destroy DMIs for DSI %u.",
            psDSI->tDSI);
    }

    if (OSAL_INVALID_LINKED_LIST_ENTRY != psDSI->hEntry)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        eOsalReturnCode = OSAL.eLinkedListRemove(psDSI->hEntry);
        if (OSAL_SUCCESS == eOsalReturnCode)
        {
            psDSI->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": cannot remove DSI %u from list",
                psDSI->tDSI);
        }
    }

    OSAL.vLinkedListMemoryFree(psDSI);
    return;
}

/*****************************************************************************
 *
 *   bRadioDataDestroyAllDMIs
 *
 *****************************************************************************/
static BOOLEAN bRadioDataDestroyAllDMIs (
    RADIO_DSI_STRUCT *psDSI
        )
{
    RADIO_DMI_STRUCT *apsDMIs[SXIAPI_DMI_CNT_MAX];
    RADIO_DATA_DMI_INFO_ITERATOR_ARG sArg;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    // Find DMIs to be removed
    sArg.tDSI = psDSI->tDSI;
    sArg.un8DMIsCountMax = SXIAPI_DMI_CNT_MAX;
    sArg.ptDMIs = NULL;
    sArg.apsDMIs = apsDMIs;
    sArg.un8DMIsCount = 0;

    eOsalReturnCode = OSAL.eLinkedListIterate(
        psDSI->psService->hDemuxList,
        (OSAL_LL_ITERATOR_HANDLER)bRadioDataDMIInfoIterator,
        &sArg);
    if (OSAL_SUCCESS != eOsalReturnCode &&
        OSAL_NO_OBJECTS != eOsalReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": cannot find DMIs for DSI %u.",
            psDSI->tDSI);
        return FALSE;
    }

    while (sArg.un8DMIsCount > 0)
    {
        sArg.un8DMIsCount--;

        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
            PC_GREEN"Remove DMI %u (DSI %u).\n"PC_RESET,
            (sArg.apsDMIs[sArg.un8DMIsCount])->tDMI, psDSI->tDSI);

        vRadioDataDestroyDMI(sArg.apsDMIs[sArg.un8DMIsCount]);
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   psRadioDataFindDSI
 *
 *****************************************************************************/
static RADIO_DSI_STRUCT *psRadioDataFindDSI (
    RADIO_DATA_SERVICE_STRUCT *psService,
    DSI tDSI
        )
{
    RADIO_DSI_STRUCT *psDSI = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    RADIO_DSI_STRUCT sSearchCriteria;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    sSearchCriteria.tDSI = tDSI;

    eOsalReturnCode = OSAL.eLinkedListSearch(
        psService->hDSIList,
        &hEntry,
        &sSearchCriteria);

    if (OSAL_SUCCESS == eOsalReturnCode)
    {
        psDSI = (RADIO_DSI_STRUCT *)OSAL.pvLinkedListThis(hEntry);
    }

    return psDSI;
}

/*****************************************************************************
 *
 *   psRadioDataFindDSIAnywhere
 *
 *****************************************************************************/
static RADIO_DSI_STRUCT *psRadioDataFindDSIAnywhere (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    DSI tDSI
        )
{
    RADIO_DATA_FIND_DSI_ITERATOR_ARG sFindDSIArg;

    sFindDSIArg.tDSI = tDSI;
    sFindDSIArg.psDSI = NULL;

    DATASERVICE_MGR_bIterateRadioData(
        psDataCtrl->hSMSDataCtrl,
        (DATASERVICE_RADIO_DATA_ITERATOR)bRadioDataFindDSIIterator,
        &sFindDSIArg);

    return sFindDSIArg.psDSI;
}

/*****************************************************************************
 *
 *   n16RadioDataDSICompare
 *
 *****************************************************************************/
static N16 n16RadioDataDSICompare (
    RADIO_DSI_STRUCT *psDSI1,
    RADIO_DSI_STRUCT *psDSI2
        )
{
    return COMPARE(psDSI1->tDSI, psDSI2->tDSI);
}

/*****************************************************************************
 *
 *   vRadioDataSetDSIStatus
 *
 *****************************************************************************/
static void vRadioDataSetDSIStatus (
    RADIO_DSI_STRUCT *psDSI,
    SXIAPI_DATASERVICE_STATUS_ENUM eStatus
        )
{
    DATASERVICE_STATE_ENUM eState;

    // Set actual DSI status
    psDSI->eStatus = eStatus;

    // transform it to appropriate state value
    switch (eStatus)
    {
        case SXIAPI_DATASERVICE_STATUS_FULLY_SUBSCRIBED:
        {
            eState = DATASERVICE_STATE_READY;
        }
        break;

        case SXIAPI_DATASERVICE_STATUS_PARTIALLY_SUBSCRIBED:
        {
            eState = DATASERVICE_STATE_POI_UPDATES_ONLY;
        }
        break;

        case SXIAPI_DATASERVICE_STATUS_UNSUBSCRIBED:
        {
            eState = DATASERVICE_STATE_UNSUBSCRIBED;
        }
        break;

        case SXIAPI_DATASERVICE_STATUS_INVALID_DSI:
        {
            // Ignore this state because the chipset will learn
            // about this DSI sometime later
            // see also this case in bRadioDataUpdateDSIStatus()
            return;
        }

        // Both of these indications mean that the 
        // service is unavailable
        case SXIAPI_DATASERVICE_STATUS_INSUF_SYS_RESOURCES:
        case SXIAPI_DATASERVICE_STATUS_UNKNOWN:
        {
            eState = DATASERVICE_STATE_UNAVAILABLE;
        }
        break;

        default:
        {
            // do not inform about these states
            return;
        }
    }

    // Inform service about DSI status change
    DATASERVICE_MGR_vDSIStateChange(psDSI->psService->hManager, psDSI->tDSI, eState);

    return;
}

/*****************************************************************************
 *
 *   vRadioDataDSIReleaseHandler
 *
 *****************************************************************************/
static void vRadioDataDSIReleaseHandler (
    RADIO_DSI_STRUCT *psDSI
        )
{
    // Invalidate master entry in order to
    // avoid removing it from master list
    // (will be removed by OSAL.eLinkedListRemoveAll)
    psDSI->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Destroy DSI
    vRadioDataDestroyDSI(psDSI);

    return;
}

/*****************************************************************************
 *
 *   bRadioDataCreateNewDMI
 *
 *****************************************************************************/
static RADIO_DMI_STRUCT * psRadioDataCreateNewDMI (
    SXM_DMI tDMI
        )
{
    RADIO_DMI_STRUCT *psDMI =
        (RADIO_DMI_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
            RADIO_DATA_SERVICE_NAME":DMI",
            sizeof(*psDMI),
            TRUE);
    if (NULL != psDMI)
    {
        psDMI->tDMI = tDMI;
        vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_AVAILABLE, FALSE);
        vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_ENABLED, FALSE);
        psDMI->hDemuxEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psDMI->psDSI = NULL;
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Failed to allocate memory for DMI %u",
            tDMI);
    }

    return psDMI;
}

/*****************************************************************************
 *
 *   bRadioDataRegisterNewDMI
 *
 *****************************************************************************/
static BOOLEAN bRadioDataRegisterNewDMI (
    SXM_DMI tDMI,
    BOOLEAN bEnabled,
    RADIO_DSI_STRUCT *psDSI
        )
{
    BOOLEAN bResult = FALSE;

    // Create the empty object
    RADIO_DMI_STRUCT *psDMI = psRadioDataCreateNewDMI(tDMI);
    if (NULL == psDMI)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Failed to allocate memory for DMI %u",
            tDMI);
        return FALSE;
    }

    // Init the object appropriately
    psDMI->tDMI = tDMI;
    vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_AVAILABLE, TRUE);
    vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_ENABLED, bEnabled);
    psDMI->hDemuxEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psDMI->psDSI = psDSI;

    // Register DMI in the demux list
    bResult = bRadioDataRegisterDMIinDemuxList(psDSI->psService, psDMI);
    if (FALSE == bResult)
    {
        // Rollback
        vRadioDataDestroyDMI(psDMI);
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bRadioDataRegisterDMIinDemuxList
 *
 *****************************************************************************/
static BOOLEAN bRadioDataRegisterDMIinDemuxList (
    RADIO_DATA_SERVICE_STRUCT *psService,
    RADIO_DMI_STRUCT *psDMI
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    BOOLEAN bResult = TRUE;

    if (OSAL_INVALID_LINKED_LIST_ENTRY != psDMI->hDemuxEntry)
    {
        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 1,
            PC_BLUE"DMI %u is already registered in the Demux List.\n"PC_RESET,
            psDMI->tDMI);

        return TRUE;
    }

    // Register DMI in the demux list
    eOsalReturnCode = OSAL.eLinkedListAdd(
        psService->hDemuxList, &psDMI->hDemuxEntry, psDMI);
    if (OSAL_SUCCESS != eOsalReturnCode)
    {
        if (OSAL_ERROR_LIST_ITEM_NOT_UNIQUE == eOsalReturnCode)
        {
            SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 1,
                PC_BLUE"DMI %u is already in the Demux List.\n"PC_RESET,
                psDMI->tDMI);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": Failed to register "
                "DMI %u in the Demux List",
                psDMI->tDMI);

            bResult = FALSE;
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   vRadioDataSetDMIAttribute
 *
 *****************************************************************************/
static void vRadioDataSetDMIAttribute (
    RADIO_DMI_STRUCT * psDMI,
    UN8 un8DMIAttribute,
    BOOLEAN bAttrValue
        )
{
    if (TRUE == bAttrValue)
    {
        psDMI->un8Attributes |= un8DMIAttribute;
    }
    else
    {
        psDMI->un8Attributes &= ~un8DMIAttribute;
    }

    return;
}

/*****************************************************************************
 *
 *   bRadioDataGetDMIAttribute
 *
 *****************************************************************************/
static BOOLEAN bRadioDataGetDMIAttribute (
    RADIO_DMI_STRUCT * psDMI,
    UN8 un8DMIAttribute
        )
{
    return (0!= (psDMI->un8Attributes & un8DMIAttribute));
}

/*****************************************************************************
 *
 *   vRadioDataDestroyDMI
 *
 *****************************************************************************/
static void vRadioDataDestroyDMI (
    RADIO_DMI_STRUCT *psDMI
        )
{
    if (OSAL_INVALID_LINKED_LIST_ENTRY != psDMI->hDemuxEntry)
    {
        // Remove from the demux list
        OSAL_RETURN_CODE_ENUM eOsalReturnCode =
            OSAL.eLinkedListRemove(psDMI->hDemuxEntry);
        if (OSAL_SUCCESS != eOsalReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": Failed to unregister DMI %u from demux list",
                psDMI->tDMI);
        }
    }

    OSAL.vLinkedListMemoryFree(psDMI);

    return;
}

/*****************************************************************************
 *
 *   n16RadioDataDMICompare
 *
 *****************************************************************************/
static N16 n16RadioDataDMICompare (
    RADIO_DMI_STRUCT *psDMI1,
    RADIO_DMI_STRUCT *psDMI2
        )
{
    return COMPARE(psDMI1->tDMI, psDMI2->tDMI);
}

/*****************************************************************************
 *
 *   bRadioDataDMIListUpdateIterator
 *
 * This iterator performs analyzis of 2 sets of DMIs extracting
 * new and to-be-removed DMIs. The incoming list of DMIs which shall replace
 * the old one is given in iteratator argument. The old list of DMIs is the
 * itarated list.
 *
 * INPUTS:
 *  the iterated list - the "old" list of DMIs
 *  psArg->psDSI - DSI which DMIs belong to
 *  psArg->atNewDMIs - contains the incoming DMIs
 *  psArg->un8NewDMIsCount - the number of items in the above array
 *
 * OUTPUTS:
 *  psArg->atNewDMIs - contains the DMIs detected as new
 *  psArg->un8NewDMIsCount - the number of items in the above array
 *  psArg->ahDMIsToRemove - the DMIs detected as to-be-removed
 *  psArg->un8DMIsToRemoveCount - the number of items in above array
 *  psArg->bExistingDMIsChanged - flag indicating change of attributes
 *                                of any existing DMI
 *
 *****************************************************************************/
static BOOLEAN bRadioDataDMIListUpdateIterator (
    RADIO_DMI_STRUCT * psDMI,
    RADIO_DATA_DMI_LIST_UPDATE_ITERATOR_ARG *psArg
        )
{
    UN8 un8Idx = 0;
    BOOLEAN bResult = FALSE;

    for (un8Idx = 0; un8Idx < psArg->un8NewDMIsCount; un8Idx++)
    {
        if (psArg->atNewDMIs[un8Idx] == psDMI->tDMI)
        {
            SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
                PC_GREEN"bRadioDataDMIListUpdateIterator(): "
                "DMI %u (attrs: 0x%X) is already in the Demux List.\n"PC_RESET,
                psDMI->tDMI, psDMI->un8Attributes);

            // Check if this DMI is currently marked as available
            bResult = bRadioDataGetDMIAttribute(psDMI, DMI_ATTR_AVAILABLE);
            if (FALSE == bResult)
            {
                SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
                    PC_GREEN"bRadioDataDMIListUpdateIterator(): "
                    "Mark DMI %u as available.\n"PC_RESET,
                    psDMI->tDMI);

                // Set DSI
                psDMI->psDSI = psArg->psDSI;

                // Mark DMI as available
                vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_AVAILABLE, TRUE);

                psArg->bExistingDMIsChanged = TRUE;
            }

            // Remove this DMI from the list of new DMIs
            psArg->atNewDMIs[un8Idx] =
                psArg->atNewDMIs[--psArg->un8NewDMIsCount];

            return TRUE;
        }
    }

    // Mark DMI as unavailable
    vRadioDataSetDMIAttribute(psDMI, DMI_ATTR_AVAILABLE, FALSE);

    return TRUE;
}

/*****************************************************************************
 *
 *   bRadioDataFilterComposeIterator
 *
 *****************************************************************************/
static BOOLEAN bRadioDataFilterComposeIterator (
    RADIO_DMI_STRUCT *psDMI,
    RADIO_DATA_DMI_INFO_ITERATOR_ARG *psArg
        )
{
    BOOLEAN bFiltered = FALSE;

    // Check if DMI is registered with DSI
    if (NULL == psDMI->psDSI)
    {
        // Skip this DMI
        return TRUE;
    }

    bFiltered = (psArg->tDSI == psDMI->psDSI->tDSI) &&
        bRadioDataGetDMIAttribute(psDMI, DMI_ATTR_AVAILABLE) &&
        bRadioDataGetDMIAttribute(psDMI, DMI_ATTR_ENABLED);
    if (TRUE == bFiltered)
    {
        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
            PC_GREEN"Add DMI %u (DSI %u) to DMI Filter\n"PC_RESET,
            psDMI->tDMI, psDMI->psDSI->tDSI);

        psArg->ptDMIs[psArg->un8DMIsCount++] = psDMI->tDMI;
    }

    return (psArg->un8DMIsCount != psArg->un8DMIsCountMax);
}

/*****************************************************************************
 *
 *   bRadioDataDMIInfoIterator
 *
 *****************************************************************************/
static BOOLEAN bRadioDataDMIInfoIterator (
    RADIO_DMI_STRUCT *psDMI,
    RADIO_DATA_DMI_INFO_ITERATOR_ARG *psArg
        )
{
    BOOLEAN bResult = TRUE;

    do
    {
        // Check if DMI is registered with DSI
        if (NULL == psDMI->psDSI)
        {
            // Skip this DMI
            break;
        }

        // Match DSI
        if (psDMI->psDSI->tDSI != psArg->tDSI)
        {
            // Skip this DMI
            break;
        }

        // Copy DMIs if required
        if (0 != psArg->un8DMIsCountMax)
        {
            if (NULL != psArg->ptDMIs)
            {
                psArg->ptDMIs[psArg->un8DMIsCount] = psDMI->tDMI;
            }

            if (NULL != psArg->apsDMIs)
            {
                psArg->apsDMIs[psArg->un8DMIsCount] = psDMI;
            }

            psArg->un8DMIsCount++;

            bResult = (psArg->un8DMIsCount != psArg->un8DMIsCountMax);
        }
        else
        {
            psArg->un8DMIsCount++;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bRadioDataUpdateDSIStatus
*
*****************************************************************************/
static BOOLEAN bRadioDataUpdateDSIStatus (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    SXIAPI_DATASERVICESTATUSIND_STRUCT *psDataStatus
        )
{
    BOOLEAN bSuccess = TRUE;
    RADIO_DSI_STRUCT *psDSI = NULL;
    BOOLEAN bDMIListChanged = FALSE;
    RADIO_DATA_DMI_LIST_UPDATE_ITERATOR_ARG *psArg = &psDataCtrl->sDMIUpdateArg;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    // Check input
    if (psDataStatus == NULL)
    {
        return FALSE;
    }

    // Find DSI
    psDSI = psRadioDataFindDSIAnywhere(psDataCtrl, (DSI)psDataStatus->tDSI);
    if (NULL == psDSI)
    {
        // Thiss is queued indication for already stopped service
        SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 4,
            "Status update for unregistered DSI %u.\n",
            psDataStatus->tDSI);

        return TRUE;
    }

    // Initialize iterator argument with common values
    psArg->psDSI = psDSI;
    psArg->un8NewDMIsCount = 0;
    psArg->bExistingDMIsChanged = FALSE;

    // New DMIs will be determined in case of  FULLY_SUBSCRIBED
    // or PARTIALLY_SUBSCRIBED statuses.
    // In other cases number of new DMIs is kept as 0.
    // All unavailable DMIs should be removed from the Demux List
    // if they are disabled only, all currently enabled but unavailable DMIs
    // are marked as unavailable but kept in the Demux List
    // so they can be added to the DMI Filter when they become available. 

    switch (psDataStatus->eStatus)
    {
        case SXIAPI_DATASERVICE_STATUS_FULLY_SUBSCRIBED:
        case SXIAPI_DATASERVICE_STATUS_PARTIALLY_SUBSCRIBED:
        {
            // Prepare for detection of new and to-be-removed DMIs
            OSAL.bMemCpy(
                psArg->atNewDMIs,
                psDataStatus->atDMIs,
                psDataStatus->un8DMICount * sizeof(psArg->atNewDMIs[0]));

            psArg->un8NewDMIsCount = psDataStatus->un8DMICount;
        }
        break;

        case SXIAPI_DATASERVICE_STATUS_INVALID_DSI:
        {
            // Print out that an invalid DSI indication was sent.
            // This doesn't mean anything really...the chipset
            // will learn of this DSI eventually, so don't sweat it!
            SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 1,
                PC_BLUE"Radio reports invalid DSI (%u).\n"PC_RESET,
                psDataStatus->tDSI);
        }
        break;

        case SXIAPI_DATASERVICE_STATUS_INSUF_SYS_RESOURCES:
            // Skip
        break;

        case SXIAPI_DATASERVICE_STATUS_UNSUBSCRIBED:
            // Skip
        break;

        default:
        {
            bSuccess = FALSE;
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": Unknown data status: %u",
                psDataStatus->eStatus);
        }
        break;
    }

    // Check if we have to proceed
    if (FALSE == bSuccess)
    {
        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
        PC_GREEN"Status %u received for DSI %u. "
        "DMIs count: %u.\n"PC_RESET,
        psDataStatus->eStatus,
        psDataStatus->tDSI,
        psDataStatus->un8DMICount);

    // Detect new and to-be-removed DMIs as well as update existing DMIs.
    eOsalReturnCode = OSAL.eLinkedListIterate(
        psDSI->psService->hDemuxList,
        (OSAL_LL_ITERATOR_HANDLER)bRadioDataDMIListUpdateIterator,
        psArg);

    if ((OSAL_SUCCESS != eOsalReturnCode) &&
        (OSAL_NO_OBJECTS != eOsalReturnCode))
    {
        // THIS IS EXCEPTIONAL ERROR WHICH NEVER SHOULD OCCUR!

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_DATA_SERVICE_NAME": Failed to update DMIs list.");

        return FALSE;
    }

    // Do we have any new DMIs detected?
    if (0 != psArg->un8NewDMIsCount)
    {
        // It is time to update the payload demux
        SXI_ADD_DATA_PAYLOAD_DEST_STRUCT *psAddDest;

        // Create an object to store the new DMIs...we're gonna send this to SXI
        psAddDest = (SXI_ADD_DATA_PAYLOAD_DEST_STRUCT *)
            SMSO_hCreate(RADIO_DATA_SERVICE_NAME":NewPayloadDest",
            sizeof(SXI_ADD_DATA_PAYLOAD_DEST_STRUCT), SMS_INVALID_OBJECT, FALSE);

        if (psAddDest != (SXI_ADD_DATA_PAYLOAD_DEST_STRUCT *)NULL)
        {
            BOOLEAN bSent;

            // All of these new destinations are for this DSI
            psAddDest->tDSI = psDataStatus->tDSI;

            // How big the block pool should be
            psAddDest->tSuggestedOTABufferByteSize =
                psDSI->tSuggestedOTABufferByteSize;

            // How many we have
            psAddDest->un8NumDMIs = psArg->un8NewDMIsCount;

            // Copy the DMIs into this array now
            OSAL.bMemCpy(
                &psAddDest->atNewDMIs[0],
                &psArg->atNewDMIs[0],
                psArg->un8NewDMIsCount * sizeof(SXIAPI_DMI));

            // Send this new data to SXI now
            bSent = STI_bSendOOB(
                psDataCtrl->hControlCxn,
                STI_OOB_TYPE_PROTOCOL_SPECIFIC, (void *)(size_t)psAddDest);
            if (FALSE == bSent)
            {
                SMSO_vDestroy((SMS_OBJECT)psAddDest);
            }
        }

        // Add new DMIs
        while (psArg->un8NewDMIsCount > 0)
        {
            psArg->un8NewDMIsCount--;

            SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
                PC_GREEN"Add DMI %u for DSI %u.\n"PC_RESET,
                psArg->atNewDMIs[psArg->un8NewDMIsCount], psDSI->tDSI);

            bSuccess = bSuccess && bRadioDataRegisterNewDMI(
                psArg->atNewDMIs[psArg->un8NewDMIsCount],
                psDSI->bEnableAllDMIs, psDSI);
        }

        bDMIListChanged = TRUE;
    }

    if (TRUE == bDMIListChanged ||
        TRUE == psArg->bExistingDMIsChanged)
    {
        if (SDTP_INVALID_HDL != psDSI->hSDTP)
        {
            SMSAPI_DEBUG_vPrint(RADIO_DATA_SERVICE_NAME, 5,
                PC_GREEN"DMI list is changed for DSI %u. "
                "Reset SDTP connection.\n"PC_RESET,
                psDSI->tDSI);

            // Forces SDTP to re-learn which DMIs need to
            // be processed by this connection
            SDTP_vResetConnection(psDSI->hSDTP);
        }
    }

    // Set DSI status
    vRadioDataSetDSIStatus(psDSI, psDataStatus->eStatus);

    return bSuccess;
}

/*****************************************************************************
 *
 *   psCreateDataServiceRadioData
 *
 *****************************************************************************/
static RADIO_DATA_SERVICE_STRUCT * psCreateDataServiceRadioData (
    DATASERVICE_MGR_OBJECT hManager,
    DATASERVICE_ID tDataId
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];
    RADIO_DATA_SERVICE_STRUCT *psService =
        (RADIO_DATA_SERVICE_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    do
    {
        // Create object name
        snprintf(&acName[0], sizeof(acName),
            RADIO_DATA_SERVICE_NAME":DataService:%u", tDataId);

        // Create the object
        psService = (RADIO_DATA_SERVICE_STRUCT *)SMSO_hCreate(
            &acName[0],
            sizeof(RADIO_DATA_SERVICE_STRUCT),
            (SMS_OBJECT)hManager,
            FALSE);

        if (NULL == psService)
        {
            break;
        }

        // Set Data Service Manager handle and attribs
        psService->hManager = hManager;

        // Create DSI list
        eOsalReturnCode = OSAL.eLinkedListCreate(
            &psService->hDSIList,
            RADIO_DATA_SERVICE_NAME":DSIList",
            (OSAL_LL_COMPARE_HANDLER)n16RadioDataDSICompare,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);

        if (OSAL_SUCCESS != eOsalReturnCode)
        {
            break;
        }

        // Create Demux List
        eOsalReturnCode = OSAL.eLinkedListCreate(
            &psService->hDemuxList,
            RADIO_DATA_SERVICE_NAME":PayloadDemux",
            (OSAL_LL_COMPARE_HANDLER)n16RadioDataDMICompare,
            OSAL_LL_OPTION_RBTREE);

        if ((OSAL_SUCCESS != eOsalReturnCode) ||
            (OSAL_INVALID_OBJECT_HDL == psService->hDemuxList))
        {
            break;
        }

        // Everything's OK
        return psService;

    } while (FALSE);

    // Rollback
    vDestroyDataServiceRadioData(psService);

    return NULL;
}

/*****************************************************************************
 *
 *   vDestroyDataServiceRadioData
 *
 *****************************************************************************/
static void vDestroyDataServiceRadioData (
    RADIO_DATA_SERVICE_STRUCT *psService
        )
{
    if (psService == NULL)
    {
        return;
    }

    if (OSAL_INVALID_OBJECT_HDL != psService->hDSIList)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Destroy associated DSIs
        eOsalReturnCode = OSAL.eLinkedListRemoveAll(
            psService->hDSIList,
            (OSAL_LL_RELEASE_HANDLER)vRadioDataDSIReleaseHandler);

        if (OSAL_SUCCESS == eOsalReturnCode)
        {
            eOsalReturnCode = OSAL.eLinkedListDelete(psService->hDSIList);

            if (OSAL_SUCCESS != eOsalReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_DATA_SERVICE_NAME": failed to destroy DSI list of service %u",
                    DATASERVICE_MGR_tGetDataId(psService->hManager));
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME": failed to destroy DSIs of service %u",
                DATASERVICE_MGR_tGetDataId(psService->hManager));
        }
    }

    // Destroy Demux List
    if (OSAL_INVALID_OBJECT_HDL != psService->hDemuxList)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Delete demux list - it's empty
        eOsalReturnCode = OSAL.eLinkedListDelete(
            psService->hDemuxList);

        if (OSAL_SUCCESS != eOsalReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME
                ": Failed to delete Demux List, error %d",
                eOsalReturnCode);
        }
    }

    // Remove Data Service Radio Specific Data
    SMSO_vDestroy((SMS_OBJECT)psService);

    return;
}

/*******************************************************************************
 *
 *   vProcessDataInd
 *
 *******************************************************************************/
static void vProcessDataInd (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    SXIAPI_OPTYPE tOpType,
    SXIAPI_INDCODE_ENUM eIndCode,
    SXIAPI_DATASERVICESTATUSIND_STRUCT *psDataStatus
        )
{
    // Parse payload based on optype
    switch(tOpType)
    {
        case SXIAPI_MESSOP_DATASERVICE:
        {
            if (eIndCode == SXIAPI_INDCODE_NOMINAL)
            {
                BOOLEAN bOk;

                bOk = bRadioDataUpdateDSIStatus(psDataCtrl, psDataStatus);
                if (FALSE == bOk)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_DATA_SERVICE_NAME
						": Failed to process data ind for DSI %u",
                        psDataStatus->tDSI);
                }
            }
        }
        break;

        default:
            // Do nothing
        break;
    }
    return;
}

/*****************************************************************************
 *
 *   hHandleSDTPPacket
 *
 *****************************************************************************/
static OSAL_BUFFER_HDL hHandleSDTPPacket (
    RADIO_DSI_STRUCT *psDSI,
    OSAL_BUFFER_HDL *phBuffer
        )
{
    OSAL_BUFFER_HDL hAssembledMessage = OSAL_INVALID_BUFFER_HDL;

    // Connect to SDTP if necessary
    if ((psDSI->hSDTP == SDTP_INVALID_HDL) &&
        (psDSI->bSDTPActive == TRUE))
    {
        // Connect to SDTP
        psDSI->hSDTP = SDTP_hConnect(
            (SMS_OBJECT)psDSI->psService);

        if (psDSI->hSDTP == SDTP_INVALID_HDL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_DATA_SERVICE_NAME
                ": unable to connect to SDTP.");
        }
    }

    if (psDSI->hSDTP != SDTP_INVALID_HDL)
    {
        // Send the payload to the SDTP re-assember
        hAssembledMessage = SDTP_hReAssemble(psDSI->hSDTP, phBuffer);
    }
    else
    {
        // Just dump this buffer now
        RADIO_bFreeDataPayload(*phBuffer);

        // Clear this handle
        *phBuffer = OSAL_INVALID_BUFFER_HDL;
    }

    return hAssembledMessage;
}

#if SMS_LOGGING == 1

/*****************************************************************************
*
*   bDSILogIterator
*
*****************************************************************************/
static BOOLEAN bDSILogIterator (
    RADIO_DSI_STRUCT *psDSI,
    OSAL_OBJECT_HDL hBlockPool
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_BLOCK_POOL_INFO_STRUCT sInfo;

    // Retrieve the block pool information for this DSI
    OSAL.bMemSet(&sInfo, 0, sizeof(OSAL_BLOCK_POOL_INFO_STRUCT));
    eOsalReturnCode = OSAL.eBlockPoolGetInfo(hBlockPool, &sInfo);

    // print header information
    DATASERVICE_MGR_vLog("\n%s\n\tDSI: %u\n",
        DATASERVICE_MGR_pacName(psDSI->psService->hManager),
        psDSI->tDSI);

    // We have something to report if the block pool is being utilized
    // (and we can successfully use the OSAL API)
    if ((eOsalReturnCode == OSAL_SUCCESS) &&
        (sInfo.un16MaxAllocated > 0))
    {
        UN32 un32BufferByteSize = sInfo.un16NumBlocks    * sInfo.un32ActualBlockSize,
             un32NumBytesUsed   = sInfo.un16Allocated    * sInfo.un32ActualBlockSize,
             un32MaxBytesUsed   = sInfo.un16MaxAllocated * sInfo.un32ActualBlockSize;

        if (psDSI->hSDTP == SDTP_INVALID_HDL)
        {
            // Label this connection as XMAPP
            DATASERVICE_MGR_vLog("\tXM APP\n");
        }
        else
        {
            // Print SDTP stats to the DSM log
            SDTP_vPrintToLog(psDSI->hSDTP,
                (SDTP_LOGGING_CALLBACK)DATASERVICE_MGR_vLog);
        }

        // Human-readable output & CSV output
        // for post-processing
        DATASERVICE_MGR_vLog(
            DS_BUFFER_STATS_OUTPUT_COMBINED,

            // Human-readable output
            un32BufferByteSize,
            un32NumBytesUsed,
            un32MaxBytesUsed,
            sInfo.un32OverrunCnt,

            // CSV output
            psDSI->tDSI,
            un32BufferByteSize,
            un32NumBytesUsed,
            un32MaxBytesUsed,
            sInfo.un32OverrunCnt);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateServicesForLog
*
*****************************************************************************/
static BOOLEAN bIterateServicesForLog (
    RADIO_PRIVATE_DATA_OBJECT hDataServiceRadioData,
    OSAL_OBJECT_HDL hBlockPool
        )
{
    RADIO_DATA_SERVICE_STRUCT *psService =
        (RADIO_DATA_SERVICE_STRUCT *)hDataServiceRadioData;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    if (psService == NULL)
    {
        // Skip this entry
        return TRUE;
    }

    eOsalReturnCode = OSAL.eLinkedListIterate(
        psService->hDSIList,
        (OSAL_LL_ITERATOR_HANDLER)bDSILogIterator,
        (void *)hBlockPool);

    if (OSAL_NO_OBJECTS == eOsalReturnCode)
    {
        // print header information
        DATASERVICE_MGR_vLog("\n%s\n\tDSI: none\n",
            DATASERVICE_MGR_pacName(psService->hManager));
    }

    return TRUE;
}
#endif
