/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _RADIO_H_
#define _RADIO_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"
#include "sti_api.h"
#include "sms_api.h"

// SMS Includes
#include "sms_event.h"
#include "channel_obj.h"
#include "radio_data_service.h"

// Protocol Specific Includes (SXi)
#include "sxi.h"
#include "sxiapi.h"
#include "sxi_fsm.h"

// Protocol Specific Includes (SDTP)
#include "sdtp.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// If radio logging hasn't been configured already
// then only enable when we're in debug
#ifndef SMS_LOGGING
#if SMS_DEBUG == 1
#define SMS_LOGGING 1
#else
#define SMS_LOGGING 0
#endif
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define RADIO_OBJECT_NAME "RADIO"

// SXI does not support dst
#define SXIAPI_RADIO_DEFAULT_DST (FALSE)

/* Category ID Ranges */

// Category Name Limits (from SXi)
#define CATEGORY_SHORT_NAME_MAX_LEN  (8)
#define CATEGORY_MEDIUM_NAME_MAX_LEN (10)
#define CATEGORY_LONG_NAME_MAX_LEN   (16)

// Broadcast category range (from SXi)
#define CATEGORY_ID_BROADCAST_MAX (63)
#define CATEGORY_ID_BROADCAST_MIN (1)
#define CATEGORY_ID_BROADCAST_NOT_ASSIGNED (0x00)

// Derived category range
#define CATEGORY_ID_DERIVED_MIN (256)
#define CATEGORY_ID_DERIVED_GROUP_SIZE (1024)
#define CATEGORY_ID_DERIVED_MAX ( \
    CATEGORY_ID_DERIVED_MIN + \
    CATEGORY_ID_DERIVED_GROUP_SIZE  - 1)

/* Channel ID Limites (from SXi) */
#define CHANNEL_ID_MAX              (999)


/* Channel Name Limits (from SXi) */
#define CHANNEL_SHORT_NAME_MAX_LEN  (8)
#define CHANNEL_MEDIUM_NAME_MAX_LEN (10)
#define CHANNEL_LONG_NAME_MAX_LEN   (20)

/* Channel Description Limits (from SXi) */
#define CHANNEL_SHORT_DESCRIPTION_MAX_LEN (SXIAPI_MAX_CHAN_SHORT_DESC)
#define CHANNEL_LONG_DESCRIPTION_MAX_LEN (SXIAPI_MAX_CHAN_LONG_DESC)

/* Artist, Title, Composer Text Limits (from SXi) */
#define ARTIST_TEXT_MAX_LEN (36)
#define TITLE_TEXT_MAX_LEN (36)
#define COMPOSER_TEXT_MAX_LEN (36)

// The initial service ID to tune the radio to
#define DEFAULT_SERVICE_ID      (1)

// The "safe" service ID to turn to.  Typically this isn't a
// real service id in the broadcast and therefore is always available
// no matter the signal conditions
#define SAFE_SERVICE_ID         (0)

// the SXI specified min and max DSI values for range checking.
#define DSI_MIN_ID     (0)
#define DSI_MAX_ID     (1023)

// SDTP Defines for Virtual Mem Buffer Device
// Choose a block size of one fourth of a max frame
#define SDTPRx_FRAME_LEN_MAX ( 1029 )
#define SDTPRx_RX_FRAME_BLOCK_RATIO (4)
#define SDTPRx_RX_FRAME_BLOCK_SIZE ((SDTPRx_FRAME_LEN_MAX / SDTPRx_RX_FRAME_BLOCK_RATIO + 1))

/* Playback Definitions */
#define PLAYBACK_DEFAULT_WARNING_OFFSET ((UN32)10) // (seconds)

#define RADIO_EPOCH                       (70) // 1970 - 1900
#define RADIO_SXI_SUB_NOTIFICATION_OFFSET (10)

// Smart Favorites Default Number
#define RADIO_SMART_FAVORITES_DEFAULT     (6)

// TuneMix Channel Attributes offset
#define TUNEMIX_CHANNELATTRIB_OFFSET      (0x40)

  /**************/
 /** TYPEDEFS **/
/**************/

typedef enum radio_fwupdate_state_enum
{
    RADIO_FWUPDATE_STATE_INITIAL,
    RADIO_FWUPDATE_STATE_ERASING,
    RADIO_FWUPDATE_STATE_SENDING,
    RADIO_FWUPDATE_STATE_ABORTING

} RADIO_FWUPDATE_STATE_ENUM;

typedef enum radio_fwupdate_event_enum
{
    RADIO_FWUPDATE_EVENT_READ_COMPLETE,
    RADIO_FWUPDATE_EVENT_ERASE_COMPLETE,
    RADIO_FWUPDATE_EVENT_SENT_PACKET,
    RADIO_FWUPDATE_EVENT_LOAD_COMPLETE,
    RADIO_FWUPDATE_EVENT_ABORT,
    RADIO_FWUPDATE_EVENT_ERROR

} RADIO_FWUPDATE_EVENT_ENUM;

typedef enum radio_fwupdate_error_enum
{
    RADIO_FWUPDATE_ERROR_ERASE_NONE,
    RADIO_FWUPDATE_ERROR_ERASE_FAILED,
    RADIO_FWUPDATE_ERROR_ERASE_TIMEOUT,
    RADIO_FWUPDATE_ERROR_GENERAL

} RADIO_FWUPDATE_ERROR_ENUM;

typedef struct radio_fwupdate_struct
{
    RADIO_FWUPDATE_EVENT_ENUM eFWUpdateEvent;
    RADIO_FWUPDATE_ERROR_ENUM eFWUpdateError;
    FILE *psFWUpdateFile;
    UN8 *pun8FWUpdateBuffer;
    UN32 un32FWUpdateNumberPackets;

} RADIO_FWUPDATE_STRUCT;

// *** MODULE ***

typedef struct radio_module_sxi_struct
{
    // SXi Link FSM
    SXI_FSM_HDL hFSM;

    // SXi Control Connection
    STI_HDL hControlCxn;

    // Most recent DispAdvisoryInd IndCode
    SXIAPI_INDCODE_ENUM eCurDispAdvisoryIndCode;

    // Last received package information
    SXIAPI_PKG_IND_STRUCT sPkgInd;

} RADIO_MODULE_SXI_STRUCT;

typedef struct radio_module_fwu_struct
{
    FILE *psFWUpdateFile;
    UN8 *pun8FWUpdateBuffer;
    UN32 un32FWUpdateBufferSize;
    UN32 un32FWUpdateCurrentPacket;
    UN32 un32FWUpdateNumberPackets;
    OSAL_OBJECT_HDL hFWUpdateEraseTimer;
    RADIO_FWUPDATE_STATE_ENUM eFWUpdateState;
    SMS_EVENT_HDL hEraseTimeoutEvent;

} RADIO_MODULE_FWU_STRUCT;

typedef struct radio_module_sub_status_struct
{
    SXIAPI_SUB_STATUS_ENUM eStatus;
    MODULE_SUBSTATUS_REASON_CODE tReasonCode;
    UN32 un32UTCTime;
    STRING_OBJECT hReasonText;
    STRING_OBJECT hPhoneNumber;
    BOOLEAN bIsAudioSubscribed;

} RADIO_MODULE_SUB_STATUS_STRUCT;

typedef struct radio_module_object_struct
{
    // Associated SRM
    SRM_OBJECT hSRM;

    // The SRM's name
    const char *pacSRMName;

    // The SRM's I/O device
    FILE *psDevice;

    // This MODULE object
    MODULE_OBJECT hModule;
    MODULE_ID tId;

    // ESN
    STRING_OBJECT hESN;
    const char *pacESN;

    // MODULE Capabilities
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities;

    // MODULE SXi info
    RADIO_MODULE_SXI_STRUCT sSXI;

    UN16 un16DurationOfBuffer;

    // Substatus
    RADIO_MODULE_SUB_STATUS_STRUCT sSubstatus;

    RADIO_MODULE_FWU_STRUCT sFWU;

    // Event handler for parent object
    SMS_EVENT_HANDLER hEventHdlr;

    // flag to indicate if RESET should happen
    BOOLEAN bReset;

    // Decoder tuple info
    SRH_DEVICE_DECODER_TUPLE_STRUCT *psDecoderTuple;
    UN32 un32NumDecoders;

} RADIO_MODULE_OBJECT_STRUCT;

// *** DECODER ***


//TODO: I believe this entire structure (and thus
//the functions which use them can be eliminated
//for CCACHE functions which are common and do the
//same thing.
typedef struct radio_decoder_category_struct
{
    // *** Categories ***

    // Number of Broadcast Categories
    N16 n16NumCategories;

    // Category Vector
    UN8 aun8CatIDVector[SXIAPI_CAT_ID_VECTOR_SIZE];

    // Linked list to store currently known categories
    OSAL_OBJECT_HDL hCatList;

} RADIO_DECODER_CATEGORY_STRUCT;

// If we tried to tune when we were in a non-NOMINAL state
// keep track of the ID until we are NOMINAL
typedef struct radio_channel_selection_struct
{
    SXIAPI_CHAN_SELECT_TYPE_ENUM eSelType;

    union
    {
        SXIAPI_SID tService;
        SXIAPI_CHANNEL_ID tChannel;

    } uId;

    SXIAPI_CATEGORY_ID tCatId;
    SXIAPI_CH_ATTRIB tOverrides;
    SXIAPI_SID tNominalId;

} RADIO_CHANNEL_SELECTION_STRUCT;

typedef struct radio_decoder_playback_struct
{
    SXIAPI_PLAYBACK_STATE_ENUM eState;
    UN8 un8BufferUsage;
    UN8 un8PlayPt;
    UN16 un16TimeOffset;
    UN16 un16DurationOfBuffer;
    UN16 un16WarningLimit;
    SXIAPI_PLAYBACK_ID tOldestPlaybackId;
    SXIAPI_PLAYBACK_ID tCurrentPlaybackId;
    SXIAPI_PLAYBACK_ID tNewestPlaybackId;
    UN16 un16DurationOfOldestTrack;
    UN16 un16DurationOfNewestTrack;
    UN32 un32TimeFromTrackStart;
    UN32 un32DurationOfTrack;
    UN16 un16TracksRemaining;
    UN16 un16TracksBefore;
    UN16 un16TimeBefore;
    BOOLEAN bIRRecordInfoValid;
    BOOLEAN bIRPlaybackInfoValid;
    BOOLEAN bFlushed;
    BOOLEAN bFlush;
    BOOLEAN bReady;
    BOOLEAN bReadySent;
    BOOLEAN bOffsetReady;
    UN16 un16Duration; // duration of content

    BOOLEAN bWarningSent;
    BOOLEAN bLimitSent;
    BOOLEAN bPaused;

} RADIO_DECODER_PLAYBACK_STRUCT;

typedef struct radio_decoder_substatus_struct
{
    MODULE_SUBSTATUS_ENUM eStatus;
    MODULE_SUBSTATUS_REASON_CODE tReasonCode;
    UN32 un32SubSuspendDate;

} RADIO_DECODER_SUBSTATUS_STRUCT;

typedef struct radio_decoder_object_struct
{
    // This DECODER object
    DECODER_OBJECT hDecoder;

    // Decoder Id
    DECODER_ID tDecoderId;

    // Associated control connection, provided by
    // the parent MODULE object.
    STI_HDL hControlCxn;

    // Decoder's ccache
    CCACHE_OBJECT hCCache;

    // Decoder capabilities
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities;

    ANTENNA_STATE_ENUM eAntennaState;
    SIGNAL_QUALITY_STRUCT sSignalQuality;

    // Initialization FSM - Current state
    DECODER_STATE_ENUM eState;

    // Most recent DispAdvisoryInd IndCode
    SXIAPI_INDCODE_ENUM eCurDispAdvisoryIndCode;

    // Booleans to track if we have enough changes to
    // enter the RADIO_READY state
    BOOLEAN bChannel0IndRxd;

    // *** Categories ***
    RADIO_DECODER_CATEGORY_STRUCT sCategory;

    // *** Selection ***
    RADIO_CHANNEL_SELECTION_STRUCT sRadioSelection;

    RADIO_DECODER_PLAYBACK_STRUCT sPlayback;

    RADIO_DECODER_SUBSTATUS_STRUCT sSubStatus;

    // Event handler for parent object
    SMS_EVENT_HANDLER hEventHdlr;

    SXIAPI_CHAN_SELECT_CFG_CMD_STRUCT sChanSelConfig;

    // Indicates Self Tune procedure completion
    BOOLEAN bSelfTuneComplete;

} RADIO_DECODER_OBJECT_STRUCT;

/** Helper Structs **/

typedef union radio_event_union
{
    RADIO_FWUPDATE_STRUCT sFirmwareUpdate;
    SXIAPI_DATA_PAYLOAD_EVENT_STRUCT sDataPayload;
    SXIAPI_RX_PAYLOAD_STRUCT sRxPayload;
    SXIAPI_RX_STRUCT *psRxData;
    SXI_FSM_EVENT_STRUCT sFsmEvent;
    STI_PROTOCOL_OOB_STRUCT sOOB;

} RADIO_EVENT_UNION;

typedef struct radio_event_dispatch_struct
{
    RADIO_EVENT_UNION uRadio;

} RADIO_EVENT_DISPATCH_STRUCT;

  /************/
 /** MACROS **/
/************/

#define RADIO_EXTRACT_LEAGUE_ID(x) (UN8)((x) >> 16)
#define RADIO_EXTRACT_TEAM_ID(x) (UN16)((x) & 0x0000FFFF)

// Sanity check; please put this in each RADIO implementation
// Verify that the above RADIO_EVENT_UNION is less than
// SMS_EVENT_DATA_UNION. If it is not, the you will probably get a compiler
// error which says..."size of array 'msg' is negative", which means this
// structure is too large. YOU MUST FIX THIS IF IT IS NOT!
COMPILE_TIME_ASSERT(sizeof(RADIO_EVENT_UNION) <= \
    sizeof(SMS_EVENT_DATA_UNION), \
    RADIO_EVENT_UNION_is_too_big_you_need_to_reduce_the_size);

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

/* Object Private Prototypes */

static void vUninitializeSrm (
    RADIO_SRM_OBJECT_STRUCT *psSRM
        );

static void vUninitializeModule(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        );

static void vUninitializeDecoder(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        );

static void vProcessModuleInd(
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    SXIAPI_RX_STRUCT *psRxData,
    BOOLEAN *pbDispatchToDecoders,
    BOOLEAN *pbEngineeringData
        );

static void vProcessDecoderInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_RX_STRUCT *psRxData
        );

static void vUpdateSCacheState(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SCACHE_OBJECT hSCache
        );

static void vProcessSeekInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    BOOLEAN bEnd,
    SXIAPI_SEEKMON_IND_STRUCT *psInd
        );

static void vProcessFlashInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_FLASH_IND_STRUCT *psInd
        );

static void vUpdateCategory(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_CATEGORY_ID tCatID,
    const char *psCatNameShort,
    const char *psCatNameMedium,
    const char *psCatNameLong,
    BOOLEAN bRemove
        );

static void vUpdateChannel(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    CHANNEL_ID tChannelId,
    SERVICE_ID tServiceId,
    const SXIAPI_CHANNEL_STRUCT *psChannel,
    const SXIAPI_TRACK_INFO_STRUCT *psTrack
        );

static void vUpdateCategoryVector(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_CATEGORY_ID tCatID,
    BOOLEAN bPresent
        );

static void vUpdateChannelInfo(
    CHANNEL_OBJECT hChannel,
    const SXIAPI_CHANNEL_STRUCT *psChannel
        );

static SPORTS_BROADCAST_ENUM eProcessSportsBroadcastType(
    UN32 un32ValidFieldMask,
    UN8 un8SportBCastType,
    UN8 *pun8Index
        );

static void vUpdateCDO(
    CHANNEL_OBJECT hChannel,
    const SXIAPI_TRACK_INFO_STRUCT *psTrack
        );

static void vUpdateStatus(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    const SXIAPI_STATUSIND_STRUCT *psStatus);

static void vUpdateDisplayAdvisories (
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_INDCODE_ENUM eIndCode,
    BOOLEAN bInitialUpdate);

static void vReleaseESN(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        );

static BOOLEAN bDispatchToDecoder(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TYPE_ENUM eType,
    const void *pvArg
        );

static BOOLEAN bPowerOnSRM (
    FILE *psDevice,
    const char *pacSRMName
        );

static BOOLEAN bPowerOffSRM (
    FILE *psDevice,
    const char *pacSRMName
        );

static BOOLEAN bInitializeAudioDecoder(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        );

static void vUnInitializeAudioDecoder(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        );

static void vSetDecoderAudioSignalState (
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SIGNAL_STATE_ENUM eSignalState
        );

static N16 n16CompareCatID (
    void *pvObj1,
    void *pvObj2
        );

static SIGNAL_QUALITY_ENUM eConvertRSLToQuality (
    UN8 un8SignalRSLValue
        );

static void vProcessFWUpdateEvent(
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    RADIO_FWUPDATE_STRUCT const *psFWUpdateInfo
        );

static void vFWUpdateHandleError(
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    MODULE_ERROR_CODE_ENUM eErrorCode
        );

static void vFWUpdateCleanup(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        );

static SMSAPI_RETURN_CODE_ENUM eLoadFWUpdate(
    MODULE_OBJECT hModule,
    const char *pcFirmwareFileName,
    RADIO_FWUPDATE_STRUCT *psFWUpdateInfo
        );

static BOOLEAN bVerifyFWUpdateFileValid(
    MODULE_OBJECT hModule,
    RADIO_FWUPDATE_STRUCT *psModule
        );

static void vFWUpdateEraseTimerCallback(
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static MODULE_ERROR_CODE_ENUM eFWUpdateTransitionToErasing(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        );

static MODULE_ERROR_CODE_ENUM eFWUpdateTransitionToSending(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        );

static MODULE_ERROR_CODE_ENUM eFWUpdateHandleSent(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        );

static void vFWUpdateHandleAbort(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        );

static BOOLEAN bScan (
    DECODER_OBJECT hDecoder,
    RADIO_CHANNEL_SELECTION_STRUCT const *psRadioSelection
        );

static BOOLEAN bPostModuleEvent (
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    const void *pvData
        );

static BOOLEAN bPostDecoderEvent (
    DECODER_OBJECT hDecoder,
    RADIO_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    const void *pvData
        );

static BOOLEAN bPostDataEvent (
    SMS_EVENT_TYPE_ENUM eEvent,
    const void *pvData
        );

static MODULE_SUBSTATUS_ENUM eSXISubStatusToSMSSubStatus(
    SXIAPI_SUB_STATUS_ENUM eSXISubStatus
        );

static CHANNEL_PLAY_ON_SELECT_METHOD_ENUM eMapPlayOnSelectMethod (
    UN8 un8PlayOnSelectMethod
        );

static CONTENT_TYPE_ENUM eMapContentType (
    UN8 un8ContentType
        );

static IR_NAVIGATION_CLASS_ENUM eMapIRNavigationClass (
    UN8 un8IRNavigationClass
        );

static CHANNEL_ACO tMapACO (
    UN16 un16ACO
        );

static BOOLEAN bHandleModuleInd (
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    SXIAPI_RX_STRUCT *psRxData
        );

static void vUpdateLastTunedChannelSelection(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_SID tSID
        );

static void vUpdateChannelSelection(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_SID tSID,
    BOOLEAN bNominal
        );

static const char *pacPlaybackStateToString(
    SXIAPI_PLAYBACK_STATE_ENUM eState
        );

static void vFlushSCache(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        );

static BOOLEAN bSetModuleError (
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    MODULE_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bStatusMonitorSwitch (
    DECODER_OBJECT hDecoder,
    SXIAPI_STATUS_ITEM_ENUM const *peStatusItems,
    size_t tStatusItemsNum,
    BOOLEAN bEnable
        );

static void vProcessBulletinStatusInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_BULLETIN_STATUS_IND_STRUCT *psInd
        );

  /***************/
 /** VARIABLES **/
/***************/

// Most recent DispAdvisoryInd IndCode (note this is global!)
// TODO: Note this doesn't work if we have multiple MODULE
// objects. So we must revisit if that can happen.
static SXIAPI_INDCODE_ENUM geDispAdvisoryIndCode = SXIAPI_INDCODE_NOMINAL;
static UN16 gun16DurationOfBuffer = 0;

// Default text used by un-subscribed channels
const char GacUnSubscribedTextDefault[] =
    "Call 800-643-2112 to Subscribe";

// Radio constants (for SXi)
const RADIO_CONSTANTS_STRUCT GsRadio =
{
    // Category
    {
        CATEGORY_SHORT_NAME_MAX_LEN,
        CATEGORY_MEDIUM_NAME_MAX_LEN,
        CATEGORY_LONG_NAME_MAX_LEN,
        CATEGORY_ID_BROADCAST_MAX,
        CATEGORY_ID_BROADCAST_MIN,
        CATEGORY_ID_BROADCAST_NOT_ASSIGNED,
        CATEGORY_ID_DERIVED_MAX,
        CATEGORY_ID_DERIVED_MIN,
        CATEGORY_ID_DERIVED_GROUP_SIZE,

    },
    // Channel
    {
        CHANNEL_SHORT_NAME_MAX_LEN,
        CHANNEL_MEDIUM_NAME_MAX_LEN,
        CHANNEL_LONG_NAME_MAX_LEN,
        ARTIST_TEXT_MAX_LEN,
        TITLE_TEXT_MAX_LEN,
        COMPOSER_TEXT_MAX_LEN,
        CHANNEL_SHORT_DESCRIPTION_MAX_LEN,
        CHANNEL_LONG_DESCRIPTION_MAX_LEN,
        DEFAULT_SERVICE_ID,
        SAFE_SERVICE_ID
    },
    // Playback
    {
        PLAYBACK_DEFAULT_WARNING_OFFSET
    },
    // Misc
    {
        //Daylight Savings Time observed
        TRUE,
        DSI_MIN_ID,
        DSI_MAX_ID

    }
};

typedef struct mod_version_name_struct
{
    UN32 un32Type;
    const char *pacNameString;

} RADIO_SXI_MOD_VERSION_NAME_STRUCT;

static const RADIO_SXI_MOD_VERSION_NAME_STRUCT sModVersionStruct[] =
{
    {
        0x10100,
        "X65"
    },
    {
        0x10101,
        "X65H"
    },
    {
        0x10102,
        "X65H2"
    },
    {
        0x10300,
        "SXV100"
    },
    {
        0x10302,
        "SXV200"
    },
    {
        0x20101,
        "X65HA"
    },
    {
        0x20300,
        "SXV300"
    },
    {
        0x30100,
        "X40HA"
    },
    {
        0x30700,
        "X40HAf"
    }
};

// Signal Status monitor
static const SXIAPI_STATUS_ITEM_ENUM gaeSignalStatusItems[] =
{
    SXIAPI_STATUS_ITEM_SIGNAL
};

// Antenna Aiming data monitor
static const SXIAPI_STATUS_ITEM_ENUM gaeAntennaStatusItems[] =
{
    SXIAPI_STATUS_ITEM_ANTENNA_AIMING
};

// Audio Presence data monitor
static const SXIAPI_STATUS_ITEM_ENUM gaeAudioPresenceItems[] =
{
    SXIAPI_STATUS_ITEM_AUDIO_PRESENCE
};

// Seek Monitor TMI List
static const SXIAPI_TMI_ENUM gaeSeekMonTMI[] =
{
    // TMI for Sports Flash seek monitor
    SXIAPI_TMI_TEAM_BCAST_ID
};

// Seek Monitor Sports Flash Report TMI list
static const SXIAPI_METADATA_IDENTIFIER gatSeekMonSportsFlashReportTMI[] =
{
    SXIAPI_TMI_TEAM_BCAST_ID,
    SXIAPI_TMI_SPORT_BCAST_TYPE
};

// Seek Monitor Control Mask list
static const UN8 gaun8SeekMonControlMask[] =
{
    // Seek Control Mask for Sports Flash seek monitor
    SXIAPI_SEEK_CONTROL_ALL
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _RADIO_H_
