/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the SXMTraffic specific functions for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "dataservice_base.h"
#include "traffic_mgr_obj.h"
#include "traffic_interface.h"

#include "dataservice_mgr_impl.h"

#include "_SXMTraffic.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/
/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bConfigureDataFilter
*
* This object interface method is a function that is used to configure
* the data filter in legacy low-band hardware that implements SSPv1 or SSPv2.
*
*****************************************************************************/

static BOOLEAN bConfigureDataFilter(
    TRAFFIC_INTERFACE_OBJECT hTrafficData,
    TRAFFIC_SERVICE_OBJECT hTrafficService,
    BOOLEAN bAdd,
    TRAFFIC_MARKET tMarket
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_DMI_CONFIG_STRUCT asDMIs[TRAFFIC_NUM_DMIS_CONFIGURED];

    // Check inputs
    if (hTrafficService == TRAFFIC_SERVICE_INVALID_OBJECT ||
        tMarket == TRAFFIC_INVALID_MARKET)
    {
        return FALSE;
    }

    // Calculate which bits to use for our markets

    // Incident DMI
    asDMIs[TRAFFIC_INDICIDENT_DMI_INDEX].bEnable = bAdd;
    asDMIs[TRAFFIC_INDICIDENT_DMI_INDEX].tDMI =
        TRAFFIC_BASE_INCIDENT_PSI + tMarket;

    // Speed and flow DMI
    asDMIs[TRAFFIC_SPEED_AND_FLOW_DMI_INDEX].bEnable = bAdd;
    asDMIs[TRAFFIC_SPEED_AND_FLOW_DMI_INDEX].tDMI =
        TRAFFIC_BASE_SPEED_AND_FLOW_PSI + tMarket;

    // Update the data stream configuration now
    eReturnCode = DATASERVICE_IMPL_eManageDataStream(
        (DATASERVICE_IMPL_HDL)hTrafficService, &asDMIs[0], TRAFFIC_NUM_DMIS_CONFIGURED);

    return (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS);
}

/*****************************************************************************
*
*   bParseHeader
*
* This object interface method is a function that is used to parse
* the traffic header from a SDTP payload that was received from low-band
* hardware.
*
* Inputs:
*
*   hPayload - A valid STI payload that starts with the traffic header
*   phTrafficID - A pointer to a handle where the TRAFFIC_LOCID parsed from
*                 the header will be saved.
*   peMsgType - A pointer to the enum that represents what type of Alert-C
*               messages are to be parsed from the payload
*   peMarketProcState - A pointer to a enum that indicates if this is the
*                       the final messages for this market of this message
*                       type
*   ptNumAlertCBytes - A pointer to a size_t that indicates the number of
*                      bytes in the payload that contain Alert-C messages
*
* Outputs:
*
*   TRUE if the header was successfully parsed.  The caller can now parse
*   Alert-C messages from the payload.
*
*   FALSE if there was some error.  The payload should be freed and marked
*   as an error in parsing.
*
*****************************************************************************/
static BOOLEAN bParseHeader(
    TRAFFIC_INTERFACE_OBJECT hTrafficData,
    OSAL_BUFFER_HDL *phPayload,
    BOOLEAN bNewPayload,
    TRAFFIC_LOCID_OBJECT *phTrafficID,
    TRAFFIC_MSG_TYPE_ENUM *peMsgType,
    TRAFFIC_MARKET_PROC_STATE_ENUM *peMarketProcState,
    size_t *ptNumAlertCBytes
        )
{
    BOOLEAN bSuccess = FALSE;

    // Local variables from the payload
    BOOLEAN bAllMessagesTXed = FALSE,
            bContainsIncident = FALSE,
            bContainsSF = FALSE;
    UN8 un8TrafficSyncByte = 0,
        un8AlertCMsgCount = 0;
    size_t tNum;
    TRAFFIC_MARKET tMarket = 0;
    OSAL_BUFFER_HDL hPayload = *phPayload;

    do
    {
        // Read the traffic sync byte.  Should always be correct, but lets just
        // do the sanity check since we can
        tNum = OSAL.tBufferReadHead(hPayload,
            &un8TrafficSyncByte, TRAFFIC_SDTP_SYNC_BYTELEN);

        if ((tNum != TRAFFIC_SDTP_SYNC_BYTELEN) ||
            (un8TrafficSyncByte != TRAFFIC_SDTP_SYNC_BYTE))
        {
            break;
        }

        // Grab the "All Messages Transmitted" flag to see if we are done
        // with this market
        tNum = OSAL.tBufferReadHeadBits(hPayload, &bAllMessagesTXed,
            0, TRAFFIC_SDTP_ALL_MESSAGES_TXED_BITLEN);

        if (tNum != TRAFFIC_SDTP_ALL_MESSAGES_TXED_BITLEN)
        {
            break;
        }

        // Ignore the next chunk of data.  This includes the RESERVED
        // field and the M and W flag fields
        tNum = OSAL.tBufferReadHeadBits(hPayload, NULL, 0,
            TRAFFIC_SDTP_RESERVED_PLUS_M_W_FLAGS_BITLEN);

        if (tNum != TRAFFIC_SDTP_RESERVED_PLUS_M_W_FLAGS_BITLEN)
        {
            break;
        }

        // Read the S flag to see if this payload contains S&F messages
        tNum = OSAL.tBufferReadHeadBits(hPayload, &bContainsSF, 0,
            TRAFFIC_SDTP_S_FLAG_BITLEN);

        if (tNum != TRAFFIC_SDTP_S_FLAG_BITLEN)
        {
            break;
        }

        // Read the I flag to see if this payload contains Incident messages
        tNum = OSAL.tBufferReadHeadBits(hPayload, &bContainsIncident, 0,
            TRAFFIC_SDTP_I_FLAG_BITLEN);

        if (tNum != TRAFFIC_SDTP_I_FLAG_BITLEN)
        {
            break;
        }

        // Read the count of Alert-C messages
        tNum = OSAL.tBufferReadHeadBits(hPayload, &un8AlertCMsgCount, 0,
            TRAFFIC_SDTP_MSG_COUNT_BITLEN);

        if (tNum != TRAFFIC_SDTP_MSG_COUNT_BITLEN)
        {
            break;
        }

        // Read past the message type field
        tNum = OSAL.tBufferReadHeadBits(hPayload, NULL, 0,
            TRAFFIC_SDTP_MSG_TYPE_BITLEN);

        if (tNum != TRAFFIC_SDTP_MSG_TYPE_BITLEN)
        {
            break;
        }

        // Read the Market ID
        tNum = OSAL.tBufferReadHead(hPayload, &tMarket,
            TRAFFIC_SDTP_MARKET_ID_BYTELEN);

        if ((tNum != TRAFFIC_SDTP_MARKET_ID_BYTELEN)
             ||
            (tMarket >= NUM_TRAFFIC_MARKETS ))
        {
            break;
        }

        // Read past the rest of the header
        tNum = OSAL.tBufferReadHeadBits(hPayload, NULL, 0,
            TRAFFIC_SDTP_REST_OF_HEADER_BITLEN);

        if (tNum != TRAFFIC_SDTP_REST_OF_HEADER_BITLEN)
        {
            break;
        }

        printf("SXMTRAFFIC: Payload Found For Market %d\n", tMarket);

        // Create the TRAFFIC_LOCID for this SDTP payload
        // For Sirius Traffic, all we know is the market
        *phTrafficID = TRAFFIC_LOCID.hCreate(
            TRAFFIC_INVALID_POS_CODE, tMarket, TRAFFIC_INVALID_BSA);

        if (*phTrafficID == TRAFFIC_LOCID_INVALID_OBJECT)
        {
            break;
        }

        if (bAllMessagesTXed)
        {
            *peMarketProcState =
                TRAFFIC_MARKET_PROC_STATE_COMPLETE;
        }
        else
        {
            *peMarketProcState =
                TRAFFIC_MARKET_PROC_STATE_INCOMPLETE;
        }

        if (bContainsSF == TRUE)
        {
            *peMsgType = TRAFFIC_MSG_TYPE_SPEED_AND_FLOW;
        }
        else
        {
            *peMsgType = TRAFFIC_MSG_TYPE_INCIDENT;
        }

        // For low-band, the rest of the payload will be alert-c
        // messages.  Get the buffer length.
        *ptNumAlertCBytes = OSAL.tBufferGetSize(hPayload);

        // We got here so we are a success
        bSuccess = TRUE;
    }
    while (FALSE);

    return bSuccess;
}

