/******************************************************************************/
/*                   Copyright (c) Sirius-XM Satellite Radio, Inc.            */
/*                            All Rights Reserved                             */
/*        Licensed Materials - Property of Sirius-XM Satellite Radio, Inc.    */
/******************************************************************************/

#include <stdlib.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#if OSAL_DEBUG == 1
#include "com_test.h"
#endif

#include "sms_api.h"
#include "sms_cli.h"

#ifdef SUPPORT_CUNIT
#include "sms_cunit.h"
#endif

#include "_sms_app.h"

// Puts & Printf are normally disabled for release builds; in our case, we actually
// want output to the console, so we #undef the OSAL definitions.
#undef puts
#undef printf

// Common local functions

/*******************************************************************************
*
*   vMonitorViolationHandler
*
*******************************************************************************/
static void vMonitorViolationHandler (
    OSAL_RETURN_CODE_ENUM eErrorCode,
    const OSAL_TASK_INFO_STRUCT *psTaskInfo,
    void *pvArg
        )
{
    fprintf(stderr,"\n============ OSAL Monitor Notification ===========\n");
    fprintf(stderr,"Task: %s\n",
#if OSAL_DEBUG == 1
        psTaskInfo->sDebug.pacName
#else
        "Unknown"
#endif
        );
    fprintf(stderr,"Status: %s\n", OSAL.pacGetReturnCodeName(eErrorCode));
    fprintf(stderr,"\n==================================================\n");

    return;
}

/*******************************************************************************
*
*   SMS_APP_bStartHandler
*
*******************************************************************************/
BOOLEAN SMS_APP_bStartHandler (
    const void *pvArg
        )
{
    BOOLEAN bTempDirSet;
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_OBJECT_HDL hTask;

    printf( "OSAL Bootstrap Function ENTER\n" );

    /////////////////////////////////////////
    // Set the Temp. Directory Path
    //////////////////////////////////////////
    bTempDirSet = OSAL.bFileSystemSetTempPath(gpacTempDirPath);

    if ( bTempDirSet == FALSE )
    {
        printf( "Error! Cannot set the temp path as %s\n",
            gpacTempDirPath );
        return FALSE;
    }
    else
    {
        printf( "The TEMP path has been set successfully as %s\n",
            gpacTempDirPath );
    }

    // Call any platform-specific initialization routines. Win32, in
    // particular, needs to open a serial I/O shim for GPIO.

    bResult = bPlatformSpecificInit();
    if ( bResult == FALSE )
    {
        puts( "Platform-specific initialization failed!" );
        return FALSE;
    }

    printf( "Setting up OSAL queue / task.\n" );

    // Initialize the queue

    eOsalReturnCode = OSAL.eQueueCreate(&gsAppData.hMsgQueueObj,
                                    "SMS App Cycler Q",
                                    (NUM_ASSETS + 1) * ASSET_NUM_STATES,
                                    sizeof( SMS_APP_MSG_STRUCT ),
                                    OSAL_QUEUE_OPTION_FIXED_SIZE);

    if (eOsalReturnCode != OSAL_SUCCESS)
    {
        puts( "Failed to create the SMS App Cycler Queue!" );
        return FALSE;
    }

    // Create the task

    eOsalReturnCode = OSAL.eTaskCreate(
                        &hTask,
                        "SMS Task Cycler",
                        SMS_APP_n32Task,
                        NULL,
                        SMSD_TASK_STACK_SIZE,
                        OSAL_TASK_PRIORITY_LOW,
                        SMSD_TASK_OPTIONS);

    if ( eOsalReturnCode != OSAL_SUCCESS )
    {
        puts("Failed to create the SMS Cycler Task!");
        return FALSE;
    }

    // Install the debug handler task. Note that this installs
    // the SXM command line interface, which is quite
    // handy for development/testing, but not something you
    // would  ever do for a normal, production application.

    // Protip: Don't run SMSD.eTaskInstall unless you're doing
    // testing or debugging!

    if (0 == (gsAppData.tProps & SMS_APP_CYCLER_PROP_CYCLE_FOREVER))
    {
#if SMS_DEBUG != 0
        eOsalReturnCode = SMSD.eTaskInstall(
                                gsAppData.eAppTaskPriority,
                                gpacDefaultSmsCfgPath,
                                gpacDefaultCliResourcesDir,
                                "srh:" );

        if ( eOsalReturnCode != OSAL_SUCCESS )
        {
            puts( "Failed to install the SMS Debug Task!" );
            return FALSE;
        }
#endif
    }

#if OSAL_DEBUG == 1
    // Start the com test. Note that this installs the SXM command
    // communication test task, which is quite
    // handy for development/testing, but not something you would
    // ever do for a normal, production application.

    // Protip: Don't run COM_bStartTest unless you're doing
    // testing or debugging!

    if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_COM_TEST))
    {
        COM_bStartTest( "COM Test", OSAL_TASK_PRIORITY_LOW );
    }
#endif

    // Kick off CUnit (if necessary)

    if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_RUN_CUNIT))
    {
        SMS_APP_MSG_STRUCT sMsg;
        sMsg.eMsgType = SMS_APP_MSG_START_CUNIT;
        SMS_APP_bSendMessage( &sMsg );
    }

    // Kick off the cycling (if necessary)

    if (gsAppData.un32MaxCycles > 0 ||
        0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_CYCLE_FOREVER))
    {
        SMS_APP_MSG_STRUCT sMsg;

        // Post the message to start the cycler
        sMsg.eMsgType = SMS_APP_MSG_START_CYCLE;
        bResult = SMS_APP_bSendMessage(&sMsg);
    }
#if (SMS_DEBUG == 1)
    else if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_START_ALL))
    {
        SMSD.vFullStart();
    }
#endif

    return bResult;
}

/*****************************************************************************
*
*   SMS_APP_vGenericStateCallback
*
*  A generic callback that sends a message to the application when
*  an asset changes state.
*
*****************************************************************************/
static void SMS_APP_vGenericStateCallback (
    SMS_APP_ASSET_ENUM eAsset,
    UN32 un32StateValue
        )
{
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo = &(gasAssetInfo[eAsset]);
    SMS_APP_MSG_STRUCT sMsg;

    // Filter state (we need READY, STOPPED and ERROR only).
    if (un32StateValue == psAssetInfo->sStatesMap.un32ReadyState)
    {
        sMsg.uMsg.sStateUpdate.eState = ASSET_STATE_STARTED;
    }
    else if (un32StateValue == psAssetInfo->sStatesMap.un32StoppedState)
    {
        sMsg.uMsg.sStateUpdate.eState = ASSET_STATE_STOPPED;
    }
    else if (un32StateValue == psAssetInfo->sStatesMap.un32ErrorState)
    {
        sMsg.uMsg.sStateUpdate.eState = ASSET_STATE_ERROR;
    }
    else
    {
        // Ignore state we are not interested in.
        return;
    }

    sMsg.eMsgType = SMS_APP_MSG_ASSET_STATE;
    sMsg.uMsg.sStateUpdate.eAsset = eAsset;

    SMS_APP_bSendMessage(&sMsg);

    return;
}

/*****************************************************************************
*
*   SMS_APP_bUpdateMemoryStatistics
*
*   Checks and prints memory usage as reported by OSAL.
*
*****************************************************************************/
static BOOLEAN SMS_APP_bUpdateMemoryStatistics (
    BOOLEAN bCheckMemoryUsage
        )
{
    UN32 un32MaxAllocatedBlocks;
    UN32 un32MaxActualAllocatedBlocks;
    UN32 un32MaxUserBytes;
    UN32 un32MaxActualBytes;
    UN32 un32TotalSystemBytes;
    BOOLEAN bOk;
    TIME_T tTod;
    UN32 un32Secs = 0;
    char cBuf[OSAL_ASCBUFSIZE];

    // Get the memory usage from OSAL ...

    bOk = OSAL.bMemoryUsage( NULL,
                             NULL,
                             NULL,
                             NULL,
                             &un32MaxAllocatedBlocks,
                             &un32MaxActualAllocatedBlocks,
                             &un32MaxUserBytes,
                             &un32MaxActualBytes,
                             &un32TotalSystemBytes
                           );

    // ... and if it worked, present the information to the user.

    if (FALSE == bOk)
    {
        printf("ERROR: Cannot retrieve memory usage from OSAL.\n");
        return FALSE;
    }

    // Get current time from the system
    OSAL.eTimeGetLocal(&un32Secs);
    tTod = un32Secs;

    puts(MID_CYCLE_SEPARATOR);
    printf("Memory Usage at %s\n", OSAL.ctime_r(&tTod, cBuf));
    puts(MID_CYCLE_SEPARATOR);

    vIndentedPrint(1, ' ', LOG_HEADER_COLUMN_LEN_MAX, "Blocks:");
    printf("%u\n", un32MaxAllocatedBlocks);
    vIndentedPrint(1, ' ', LOG_HEADER_COLUMN_LEN_MAX, "Actual blocks:");
    printf("%u\n", un32MaxActualAllocatedBlocks);
    vIndentedPrint(1, ' ', LOG_HEADER_COLUMN_LEN_MAX, "User bytes:");
    printf("%u \n", un32MaxUserBytes);
    vIndentedPrint(1, ' ', LOG_HEADER_COLUMN_LEN_MAX, "Actual bytes:");
    printf("%u \n", un32MaxActualBytes);
    vIndentedPrint(1, ' ', LOG_HEADER_COLUMN_LEN_MAX, "OS-Overhead:");
    printf("%.1f%%\n",
        un32MaxUserBytes != 0 ?
                (((double)(un32MaxActualBytes - un32MaxUserBytes))
                    / un32MaxUserBytes) * 100 : 100);

    if (TRUE == bCheckMemoryUsage)
    {
        if (un32MaxUserBytes > gsAppData.un32MaxUserBytes)
        {
            vIndentedPrint(1, ' ', LOG_HEADER_COLUMN_LEN_MAX,
                "Memory leak detected:");
            printf("%u bytes\n",
                un32MaxUserBytes - gsAppData.un32MaxUserBytes);

            gsAppData.un32NumMemoryLeaks++;
        }

        vIndentedPrint(1, ' ', LOG_HEADER_COLUMN_LEN_MAX,
                "Detected memory leaks:");
        printf("%u\n",
            gsAppData.un32NumMemoryLeaks);
    }

    // Save memory usage
    gsAppData.un32MaxUserBytes = un32MaxUserBytes;

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_vPrintStatDump
*
*******************************************************************************/
static void SMS_APP_vPrintStatDump ( void )
{
    UN32 un32UpTimeInSec;

    OSAL.vTimeUp(&un32UpTimeInSec, NULL);

    vIndentedPrint(0, ' ', LOG_HEADER_COLUMN_LEN_MAX, "Elapsed Time for Cycle:");
    printf("%u seconds\n", un32UpTimeInSec - gsAppData.sCycleInfo.un32StartTime);

    vIndentedPrint(0, ' ', LOG_HEADER_COLUMN_LEN_MAX, "Successful runs:");
    printf("%u\n", gsAppData.un32SuccessfulCycles);

    vIndentedPrint(0, ' ', LOG_HEADER_COLUMN_LEN_MAX, "Unsuccessful runs:");
    printf("%u\n", gsAppData.un32FailedCycles);

    return;
}

/*****************************************************************************
*
*   SMS_APP_bSendMessage
*
*   This function will allocate memory for a message and put it on
*   the Sample App Task's Queue.  Since the allocation is blocking, this fxn
*   should not be called from the context of the SMSD task.
*
*   Inputs:
*           psMsg - Ptr to a populated message to be placed in the Task's queue
*
*   Outputs:
*           BOOLEAN - the success (or failure) of this function
*
*****************************************************************************/
static BOOLEAN SMS_APP_bSendMessage (
    SMS_APP_MSG_STRUCT *psMsg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SMS_APP_MSG_STRUCT *psAllocatedMsg;

    do
    {
        // Allocate an available message from the queue.
        // We ask for non-blocking because it's just someone at the
        // CLI typing in commands. If our queue is full, well let them
        // know.
        eReturnCode = OSAL.eMessageAllocate(gsAppData.hMsgQueueObj,
                                            (void**)&psAllocatedMsg,
                                            OSAL_QUEUE_FLAG_NONBLOCK);
        if (OSAL_SUCCESS != eReturnCode)
        {
            printf("ERROR: Unable to allocate message: %d (%s).\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // We were able to allocate an available msg from the queue
        psAllocatedMsg->eMsgType = psMsg->eMsgType;
        psAllocatedMsg->uMsg = psMsg->uMsg;

        // Put the message on the queue
        eReturnCode = OSAL.eQueuePut(psAllocatedMsg,
                                     sizeof(SMS_APP_MSG_STRUCT));
        if (OSAL_SUCCESS != eReturnCode)
        {
            // The queue put was unsuccessful, this should only happen if
            // we passed in a bad size for the message (>block size)
            printf("ERROR: Unable to put message to the queue: %d (%s).\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            // Give back the allocated memory to the queue
            eReturnCode = OSAL.eMessageFree(psAllocatedMsg);
            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("ERROR: Unable to free message: %d (%s).\n",
                    eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            }

            break;
        }

        // Everything's OK
        return TRUE;

    } while (FALSE);

    // Error!
    return FALSE;
}

/*****************************************************************************
*
*   SMS_APP_bStartIntervalTimer
*
*****************************************************************************/
static BOOLEAN SMS_APP_bStartIntervalTimer (
    UN32 un32Timeout
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode =
        OSAL.eTimerStartRelative(gsAppData.hTimer, un32Timeout, 0);
    if (OSAL_SUCCESS != eReturnCode)
    {
        printf("ERROR: Unable to start cycler timer: %d (%s).\n",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*       SMS_APP_n32Task
*
*       This is the Smsd Task Handler.
*
*       Inputs:
*               pvArg - Required by the OS
*
*       Outputs:
*               N32 - Required by OS
*
*****************************************************************************/
static N32 SMS_APP_n32Task (
    void *pvArg
        )
{
    SMS_APP_MSG_STRUCT *psMsg;

    UN32 un32TaskMsgSize = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    BOOLEAN bContinue = TRUE;

    while (TRUE == bContinue)
    {
        // Pend on a message from the queue
        eReturnCode = OSAL.eQueueGet(gsAppData.hMsgQueueObj,
                                     (void **)&psMsg,
                                     &un32TaskMsgSize,
                                     OSAL_OBJ_TIMEOUT_INFINITE);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Some unexpected error
            printf("ERROR: OSAL.eQueueGet() returned %d (%s).\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        if (psMsg->eMsgType >= SMS_APP_MSG_MAX)
        {
            printf("ERROR: Invalid message received: %d.\n",
                psMsg->eMsgType);
            break;
        }

        // Call an appropriate handler
        bContinue = gsCyclerMsgHandlers[psMsg->eMsgType](psMsg);

        // Now we're done with it
        eReturnCode = OSAL.eMessageFree(psMsg);
        if (OSAL_SUCCESS != eReturnCode)
        {
            printf("ERROR: OSAL.eMessageFree() returned %d (%s).\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }
    } // while (TRUE == bContinue)

    // if the Queue hasn't been destroyed yet, destroy it now
    if (OSAL_INVALID_OBJECT_HDL != gsAppData.hMsgQueueObj)
    {
        eReturnCode = OSAL.eQueueDelete(gsAppData.hMsgQueueObj);
        if (OSAL_SUCCESS == eReturnCode)
        {
            gsAppData.hMsgQueueObj = OSAL_INVALID_OBJECT_HDL;
        }
        else
        {
            printf("ERROR: Unable to delete the queue: %d (%s).\n",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    puts(OUTER_CYCLER_TASK_STOP);

    // This task is now done
    return OSAL_TASK_REPORT_NO_ERROR;
}

/*****************************************************************************
*
* SMS_APP_vHeartbeat
*
* Calls SMS_APP_bUpdateMemoryStatistics roughly once per minute.
*
*****************************************************************************/
static void SMS_APP_vHeartbeat (
    void *pvArg
        )
{
#if SMS_DEBUG == 0
    static UN16 count = 0;

    if ( ++count > 128 )
    {
       SMS_APP_bUpdateMemoryStatistics(FALSE);
       count = 0;
    }
#endif
}

/*******************************************************************************
*
*   vIndentedPrint
*
*   It's supposed to be a special replacement for printf().
*
*******************************************************************************/
static void vIndentedPrint (
    UN8 un8Indent,
    char cPaddingCharacter,
    UN8 un8DstLen,
    const char *format,
    ...
        )
{
    // A "special" version of printf that uses the indent setting for the current
    // cycler state to offset what's being printed.

    UN32 un32NumPrinted;
    UN32 un32NumPaddingChars = 0;
    
    va_list args;
    va_start(args, format);

    // Print indent
    printf("%*.*s", INDENT_WIDTH * un8Indent, INDENT_WIDTH * un8Indent, "");

    // Print arguments
    un32NumPrinted = vprintf(format, args);

    // Calculate number of padding characters
    un32NumPrinted += (un8Indent * INDENT_WIDTH);
    if (un8DstLen > un32NumPrinted)
    {
        un32NumPaddingChars = un8DstLen - un32NumPrinted;
    }

    // Print padding characters
    while (0 != un32NumPaddingChars--)
    {
        putchar(cPaddingCharacter);
    }

    va_end(args);

    return;
}

/*******************************************************************************
*
*   vCharPaddingPrint
*
*******************************************************************************/
static void vCharPaddingPrint (
    char cPaddingCharacter,
    UN8 un8DstLen,
    const char *format,
    ...
        )
{
    UN32 un32NumPrinted;
    UN32 un32NumPaddingChars = 0;

    va_list args;
    va_start(args, format);

    // Print arguments
    un32NumPrinted = vprintf(format, args);

    // Calculate number of padding characters
    if (un8DstLen > un32NumPrinted)
    {
        un32NumPaddingChars = un8DstLen - un32NumPrinted;
    }

    // Print padding characters
    while (0 != un32NumPaddingChars--)
    {
        putchar(cPaddingCharacter);
    }

    va_end(args);

    return;
}

/*******************************************************************************
*
*   SMS_APP_vTimeoutCallback
*
*******************************************************************************/
static void SMS_APP_vTimeoutCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    SMS_APP_MSG_STRUCT sMsg;

    // Post event to the test task

    sMsg.eMsgType = SMS_APP_MSG_TIME_INTERVAL;

    SMS_APP_bSendMessage(&sMsg);

    return;
}

/*******************************************************************************
*
*   SMS_APP_pacAssetStateToString
*
*******************************************************************************/
static const char *SMS_APP_pacAssetStateToString (
    SMS_APP_ASSET_STATE_ENUM eState
        )
{
    const char *pacState = "UNKNOWN";

    // Log event
    switch (eState)
    {
        case ASSET_STATE_STOPPED:
        {
            pacState = "STOPPED";
        }
        break;

        case ASSET_STATE_STARTING:
        {
            pacState = "STARTING";
        }
        break;

        case ASSET_STATE_STARTED:
        {
            pacState = "STARTED";
        }
        break;

        case ASSET_STATE_START_FAILED:
        {
            pacState = "START_FAILED";
        }
        break;

        case ASSET_STATE_STOPPING:
        {
            pacState = "STOPPING";
        }
        break;

        case ASSET_STATE_ERROR:
        {
            pacState = "ERROR";
        }
        break;

        default:
        {
            printf("ERROR: Unexpected state: %u.\n", eState);
        }
        break;
    }

    return pacState;
}

// Functions to get test runtime related values
// depending on specified parameters

/*******************************************************************************
*
*   SMS_APP_un32GetTestValue
*
*   Depending on the SMS_APP_CYCLER_PROP_RANDOM_TEST flag specified
*   the function returns:
*       value between 0 and un32MaxValue - if RANDOM flag is set
*       un32DefaultValue                 - if RANDOM flag is not set
*
*******************************************************************************/
static UN32 SMS_APP_un32GetTestValue (
    UN32 un32MaxValue,
    UN32 un32DefaultValue
        )
{
    if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_RANDOM_TEST))
    {
        if (0 != un32MaxValue)
        {
            un32MaxValue = rand() % (un32MaxValue + 1);
        }
    }
    else
    {
        un32MaxValue = un32DefaultValue;
    }

    return un32MaxValue;
}

/*******************************************************************************
*
*   SMS_APP_un32GetAssetStartWaitTime
*
*******************************************************************************/
static UN32 SMS_APP_un32GetAssetStartWaitTime (
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo
        )
{
    BOOLEAN bResult;
    UN32 un32AssetStartTime = 0;

    // Get absolute or random condition
    bResult = (SMS_APP_un32GetTestValue(
        TRUE, DEFAULT_ASSET_START_INTERVAL) != 0);
    if (TRUE == bResult)
    {
        // Get absolute or random value.
        un32AssetStartTime = SMS_APP_un32GetTestValue(
            psAssetInfo->un32StartInterval,
            DEFAULT_ASSET_START_INTERVAL);
    }

    return un32AssetStartTime;
}

/*******************************************************************************
*
*   SMS_APP_un32GetAssetStopWaitTime
*
*******************************************************************************/
static UN32 SMS_APP_un32GetAssetStopWaitTime (
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo
        )
{
    BOOLEAN bResult;
    UN32 un32AssetStopTime = 0;

    // Get absolut or random condition
    bResult = (SMS_APP_un32GetTestValue(
        TRUE, DEFAULT_ASSET_STOP_INTERVAL) != 0);
    if (TRUE == bResult)
    {
        // Get absolute or random value.
        un32AssetStopTime = SMS_APP_un32GetTestValue(
            psAssetInfo->un32StopInterval,
            DEFAULT_ASSET_STOP_INTERVAL);
    }

    return un32AssetStopTime;
}

/*******************************************************************************
*
*   SMS_APP_un32GetOperationalTime
*
*******************************************************************************/
static UN32 SMS_APP_un32GetOperationalTime ( void )
{
    UN32 un32OperationalTime;

    // If operational time is specified, use specified value
    if (UN32_MAX != gsAppData.sCycleInfo.un32OperationalTime)
    {
        // Get absolute or random operational time value.
        un32OperationalTime = SMS_APP_un32GetTestValue(
            gsAppData.sCycleInfo.un32OperationalTime,
            gsAppData.sCycleInfo.un32OperationalTime);
    }
    // Otherwise use default values
    else
    {
        // Get absolute or random operational time value.
        un32OperationalTime = SMS_APP_un32GetTestValue(
            RANDOM_OPERATIONAL_TIME_MAX, DEFAULT_OPERATIONAL_TIME);
    }

    return un32OperationalTime;
}

// Message handlers, called in response to specific
// messages sent to the test task.
// These functions cache the state data of various SMS objects into our
// appData struct; we need to do this, as SMS forbids retrieving
// state info from this objects outside of a callback.

/*******************************************************************************
*
*   SMS_APP_bHandleStartCycleMsg
*
*******************************************************************************/
static BOOLEAN SMS_APP_bHandleStartCycleMsg (
    SMS_APP_MSG_STRUCT *psMsg
        )
{
    UN32 un32TotalIterations =
        gsAppData.un32SuccessfulCycles + gsAppData.un32FailedCycles;
    BOOLEAN bResult;

    // Initializa the cycler if it was not initialized
    bResult = SMS_APP_bInitCycler();
    if (FALSE == bResult)
    {
        return FALSE;
    }

    // Set the console color for this cycles (to help make each cycles stand
    // out a little more.) We'll update the color for the next iteration after
    // this.

    vToggleSmsConsoleColor();

    // Now print out the start-of-cycle header

    puts("\n\n");
    puts(OUTER_CYCLE_SEPARATOR);
    vIndentedPrint(0, ' ', LOG_HEADER_COLUMN_LEN_MAX, "Starting next cycle:");

    if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_CYCLE_FOREVER))
    {
        printf("%d (Continuous Loop Mode)",
            un32TotalIterations + 1);
    }
    else
    {
        printf("%d of %d", un32TotalIterations + 1,
            gsAppData.un32MaxCycles);
    }

    if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_RANDOM_TEST))
    {
        printf(" (Random)");
    }
    printf("\n");

    // Set number of assets to start in this cycle.
    // Get absolute or random value.
    gsAppData.sCycleInfo.un8NumAssetsToStart =
        (UN8)SMS_APP_un32GetTestValue(NUM_ASSETS - 1, NUM_ASSETS - 1) + 1;

    vIndentedPrint(0, ' ', LOG_HEADER_COLUMN_LEN_MAX,
        "Number of assets to start:");
    printf("%u of %u\n",
        gsAppData.sCycleInfo.un8NumAssetsToStart, NUM_ASSETS);

    // Memory usage verification is beginning from the second iteration.
    if (un32TotalIterations > 0)
    {
        bResult = SMS_APP_bUpdateMemoryStatistics(FALSE);
        if (FALSE == bResult)
        {
            return FALSE;
        }
    }

    puts(INNER_CYCLE_SEPARATOR);
    puts("S T A R T I N G . . .");

    // Reset cycle statistics
    gsAppData.sCycleInfo.un16NumErrors = 0;
    gsAppData.sCycleInfo.un8NumStartedAssets = 0;
    gsAppData.sCycleInfo.ePendingAsset = NUM_ASSETS;
    gsAppData.sCycleInfo.eState = SMS_APP_CYCLE_STATE_STARTING;

    // Get the up-time in seconds
    OSAL.vTimeUp(&(gsAppData.sCycleInfo.un32StartTime), NULL);

    // Start asset(s)
    bResult = SMS_APP_bStartNext();

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bHandleStateChangeMsg
*
*******************************************************************************/
static BOOLEAN SMS_APP_bHandleStateChangeMsg (
    SMS_APP_MSG_STRUCT *psMsg
        )
{
    BOOLEAN bResult = TRUE;
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo =
        &(gasAssetInfo[psMsg->uMsg.sStateUpdate.eAsset]);

    // Handle state change for asset which is out of initial state
    if (ASSET_STATE_STOPPED != psAssetInfo->eState &&
        psMsg->uMsg.sStateUpdate.eState != psAssetInfo->eState)
    {
        // Handle event depending on the current cycle's state
        switch (gsAppData.sCycleInfo.eState)
        {
            case SMS_APP_CYCLE_STATE_STARTING:
            {
                bResult = SMS_APP_bOnStartingEventHandler(
                    psMsg->uMsg.sStateUpdate.eAsset,
                    psAssetInfo,
                    psMsg->uMsg.sStateUpdate.eState);
            }
            break;

            case SMS_APP_CYCLE_STATE_STOPPING:
            {
                bResult = SMS_APP_bOnStoppingEventHandler(
                    psMsg->uMsg.sStateUpdate.eAsset,
                    psAssetInfo,
                    psMsg->uMsg.sStateUpdate.eState);
            }
            break;

            default:
            {
                // Ignore event
            }
            break;
        }
    }

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bHandleStartCUnitMsg
*
*******************************************************************************/
static BOOLEAN SMS_APP_bHandleStartCUnitMsg( SMS_APP_MSG_STRUCT *psMsg )
{
#ifdef SUPPORT_CUNIT
    vIndentedPrint(0, ' ', 0,
        "Run CUnit Testing...\n\n");
    SMS_CUNIT_bRun(SMS_CUNIT_OPTION_NORMAL, gsAppData.tCUnitSuite);
#else
    puts("CUnit support is turned off");
#endif

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bHandleTimerMsg
*
*******************************************************************************/
static BOOLEAN SMS_APP_bHandleTimerMsg (
    SMS_APP_MSG_STRUCT *psMsg
        )
{
    BOOLEAN bResult = TRUE;

    // Log event
    puts("DONE");

    // Handle event depending on the current cycle's state
    switch (gsAppData.sCycleInfo.eState)
    {
        case SMS_APP_CYCLE_STATE_STARTING:
        {
            // Check if all assets are now started
            if (gsAppData.sCycleInfo.un8NumStartedAssets ==
                gsAppData.sCycleInfo.un8NumAssetsToStart)
            {
                bResult = SMS_APP_bBeginCycleStop();
            }
            else
            {
                // Continue. Try to start next asset.
                gsAppData.sCycleInfo.un8NumStartedAssets++;
                bResult = SMS_APP_bStartNext();
            }
        }
        break;

        case SMS_APP_CYCLE_STATE_STOPPING:
        {
            // Continue. Try to stop next asset.
            gsAppData.sCycleInfo.un8NumStartedAssets--;
            bResult = SMS_APP_bStopNext();
        }
        break;

        default:
        {
            puts("ERROR: Invalid Cycle state.");
            bResult = FALSE;
        }
        break;
    }

    return bResult;
}

// Events "action" functions for starting and stopping a cycle.
// (These also trigger printing the cycle header / footer).

/*******************************************************************************
*
*   SMS_APP_bInitCycler
*
*******************************************************************************/
static BOOLEAN SMS_APP_bInitCycler ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32Seconds;
    UN16 un16Milliseconds;

    if (OSAL_INVALID_OBJECT_HDL != gsAppData.hTimer)
    {
        // The cycler is already initialized
        return TRUE;
    }

    // Create the timer
    eReturnCode = OSAL.eTimerCreate(&(gsAppData.hTimer),
        "SMS_APP:CyclerTimer", SMS_APP_vTimeoutCallback, &gsAppData);
    if (OSAL_SUCCESS != eReturnCode)
    {
        printf("ERROR: Unable to create cycler timer: %d (%s)\n",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
        return FALSE;
    }

    // Init random seed
    OSAL.vTimeUp(&un32Seconds, &un16Milliseconds);
    srand(un32Seconds * MILLISECONDS_IN_SECOND + un16Milliseconds);

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bBeginCycleStop
*
*******************************************************************************/
static BOOLEAN SMS_APP_bBeginCycleStop ( void )
{
    BOOLEAN bResult;
    SMS_APP_ASSET_ENUM eAsset;
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo;

    // Get last started asset's info (if any)
    if (gsAppData.sCycleInfo.un8NumStartedAssets > 0)
    {
        eAsset = (SMS_APP_ASSET_ENUM)
            (gsAppData.sCycleInfo.un8NumStartedAssets - 1);

        psAssetInfo = &(gasAssetInfo[eAsset]);

        vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
            "S T O P P I N G . . .\n");
    }

    // Reset pending asset
    gsAppData.sCycleInfo.ePendingAsset = NUM_ASSETS;
    // Set stopping state
    gsAppData.sCycleInfo.eState = SMS_APP_CYCLE_STATE_STOPPING;

    // Stop asset(s)
    bResult = SMS_APP_bStopNext();

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bCompleteCycle
*
*******************************************************************************/
static BOOLEAN SMS_APP_bCompleteCycle ( void )
{
    BOOLEAN bResult;
    UN32 un32TotalIterations;
    SMS_APP_MSG_STRUCT sMsg;

    gsAppData.sCycleInfo.eState = SMS_APP_CYCLE_STATE_STOPPED;

    if (0!= gsAppData.sCycleInfo.un16NumErrors)
    {
        gsAppData.un32FailedCycles++;
    }
    else
    {
        gsAppData.un32SuccessfulCycles++;
    }

    un32TotalIterations =
        gsAppData.un32SuccessfulCycles + gsAppData.un32FailedCycles;

    puts(INNER_CYCLE_SEPARATOR);
    vIndentedPrint(0, ' ', LOG_HEADER_COLUMN_LEN_MAX,
        "Current Cycle Finished:");

    if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_CYCLE_FOREVER))
    {
        printf("%d (Continuous Loop Mode)",
            un32TotalIterations);
    }
    else
    {
        printf("%d of %d",
            un32TotalIterations, gsAppData.un32MaxCycles);
    }

    if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_RANDOM_TEST))
    {
        printf(" (Random)");
    }
    printf("\n");

    SMS_APP_vPrintStatDump();

    // Memory verification is beginning from the second iteration.
    // Just print memory statistics at the end of the first iteration.
    bResult = (un32TotalIterations > 1) ? TRUE : FALSE;
    bResult = SMS_APP_bUpdateMemoryStatistics(bResult);
    if (FALSE == bResult)
    {
        return FALSE;
    }

    puts(OUTER_CYCLE_SEPARATOR);

    // If we're not cycling continuously, go ahead and check
    // the number of iterations to see if we need to stop.

    if (0 == (gsAppData.tProps & SMS_APP_CYCLER_PROP_CYCLE_FOREVER) &&
        un32TotalIterations >= gsAppData.un32MaxCycles)
    {
        SMS_APP_bStopCycler();
        return FALSE;
    }

    // Next cycle shall be started.
    // Post a message instead of calling the start handler directly.

    sMsg.eMsgType = SMS_APP_MSG_START_CYCLE;
    bResult = SMS_APP_bSendMessage(&sMsg);

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bStartNext
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartNext ( void )
{
    BOOLEAN bResult = TRUE;
    SMS_APP_ASSET_ENUM eCurrentAsset;
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo;
    BOOLEAN bMandatoryStart = FALSE;
    BOOLEAN bStartRequired = TRUE;
    UN32 un32Time;

    while (gsAppData.sCycleInfo.un8NumStartedAssets <
           gsAppData.sCycleInfo.un8NumAssetsToStart)
    {
        // Select next asset to start
        eCurrentAsset =
            (SMS_APP_ASSET_ENUM)gsAppData.sCycleInfo.un8NumStartedAssets;
        psAssetInfo = &(gasAssetInfo[eCurrentAsset]);

        // Set initial state
        psAssetInfo->eState = ASSET_STATE_STOPPED;

        // Get MANDATORY_START flag value
        bMandatoryStart = (0 != (psAssetInfo->tProps &
                                 SMS_APP_ASSET_PROP_MANDATORY_START));

        // Check is asset start is mandatory required
        if (FALSE == bMandatoryStart)
        {
            // Get absolute or random condition
            BOOLEAN bAssetStartRequired =
                (SMS_APP_un32GetTestValue(TRUE, TRUE) != 0);

            if (FALSE == bAssetStartRequired)
            {
                // Start is not required.
                // Continue. Try to start next asset.
                gsAppData.sCycleInfo.un8NumStartedAssets++;
                continue;
            }
        }

        // Check if this asset has precedense
        if (NUM_ASSETS != psAssetInfo->ePrecedentAsset)
        {
            // Check if precedent asset start failed
            if (ASSET_STATE_START_FAILED ==
                gasAssetInfo[psAssetInfo->ePrecedentAsset].eState ||
                ASSET_STATE_ERROR ==
                gasAssetInfo[psAssetInfo->ePrecedentAsset].eState)
            {
                // This asset cannot be started without precedent asset start
                // Log error
                vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                    "Cannot start %s (%s) because of precedent asset (%s) error.\n",
                    psAssetInfo->pacName,
                    (TRUE == bMandatoryStart) ? "mandatory" : "optional",
                    gasAssetInfo[psAssetInfo->ePrecedentAsset].pacName);
                // Precedent asset won't be started
                bResult = FALSE;
            }
            // Check if precedent asset is already started
            else if (ASSET_STATE_STARTED ==
                gasAssetInfo[psAssetInfo->ePrecedentAsset].eState)
            {
                bResult = TRUE;
            }
            // Check if precedent asset is currently being started
            else if (ASSET_STATE_STARTING ==
                gasAssetInfo[psAssetInfo->ePrecedentAsset].eState)
            {
                // Set pending asset. Wait for precedent asset start.
                gsAppData.sCycleInfo.ePendingAsset = eCurrentAsset;

                return TRUE;
            }
            else
            {
                // Precedent asset was skipped
                bResult = FALSE;
                bStartRequired = FALSE;
            }
        }
        else
        {
            // No precedent asset required
            bResult = TRUE;
        }

        // If precedent asset is started or not required,
        // initiate this asset start.
        if (TRUE == bResult)
        {
            vIndentedPrint(psAssetInfo->un8Indent,
                '.', LOG_ACTION_COLUMN_LEN_MAX,
                "Start %s", psAssetInfo->pacName);

            bResult =
                psAssetInfo->tFnStart(&(psAssetInfo->pvObjectData));
            if (FALSE == bResult)
            {
                // Log start error
                printf("FAILED (%s)\n",
                    (TRUE == bMandatoryStart) ? "mandatory" : "optional");
            }
        }

        if (FALSE == bResult)
        {
            if (TRUE == bStartRequired)
            {
                // Set appropriate state
                psAssetInfo->eState = ASSET_STATE_START_FAILED;

                // Count error
                gsAppData.sCycleInfo.un16NumErrors++;

                // If the following flag is specified
                // the cycler shall be stopped immediately
                // without completion of the current cycle.
                if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_STOP_ON_ERROR))
                {
                    vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                        "STOP_ON_ERROR is set. Stop cycler immediately.\n");
                    return FALSE;
                }

                // Check MANDATORY_START flag value
                if (TRUE == bMandatoryStart)
                {
                    // Initiate current cycle stop
                    bResult = SMS_APP_bBeginCycleStop();

                    return bResult;
                }
            }
        }
        else
        {
            // Get props values
            BOOLEAN bAssetStartWaitSupported =
                (0 != (psAssetInfo->tProps &
                       SMS_APP_ASSET_PROP_READY_EVENT_SUPPORTED));
            BOOLEAN bAssetStartWaitRequired =
                (0 != (psAssetInfo->tProps &
                       SMS_APP_ASSET_PROP_READY_EVENT_REQUIRED));
            BOOLEAN bCommonStartWaitDisabled =
                (0 != (gsAppData.tProps &
                       SMS_APP_CYCLER_PROP_NO_START_WAIT));

            // Check if waiting for READY state is supported for this asset
            if (FALSE == bAssetStartWaitSupported)
            {
                // Start (READY) event is not supported.
                // Update asset's state.
                psAssetInfo->eState = ASSET_STATE_STARTED;
            }
            else
            {
                // Update asset's state
                psAssetInfo->eState = ASSET_STATE_STARTING;

                // Check if waiting for READY state is REQUIRED for this asset
                if (FALSE == bAssetStartWaitRequired)
                {
                    // Check for common "no wait" condition
                    if (TRUE == bCommonStartWaitDisabled)
                    {
                        // Log starting (no wait)
                        puts("STARTING");

                        // Continue. Try to start next asset.
                        gsAppData.sCycleInfo.un8NumStartedAssets++;
                        continue;
                    }

                    // Get absolute or random condition
                    bAssetStartWaitRequired =
                        (SMS_APP_un32GetTestValue(
                            TRUE, DEFAULT_START_WAIT_CONDITION) != 0);
                }

                if (TRUE == bAssetStartWaitRequired)
                {
                    // Set pending asset
                    gsAppData.sCycleInfo.ePendingAsset = eCurrentAsset;

                    // Log starting (wait)
                    vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX, "STARTING");

                    // Nothing else to do here. Waiting for state event.
                    return TRUE;
                }
            }

            // Check for common "no wait" condition
            if (FALSE == bCommonStartWaitDisabled)
            {
                // Check if we should wait some time
                // before next asset start.

                // Get absolute or random value.
                un32Time = SMS_APP_un32GetAssetStartWaitTime(psAssetInfo);
                if (0 != un32Time)
                {
                    // Reset pending asset
                    gsAppData.sCycleInfo.ePendingAsset = NUM_ASSETS;

                    // Log starting (wait)
                    vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX,
                        (TRUE == bAssetStartWaitSupported) ? "STARTING" : "STARTED");
                    // Log timeout (wait)
                    vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX, "wait %u ms", un32Time);

                    // Start the timer
                    bResult = SMS_APP_bStartIntervalTimer(un32Time);

                    return bResult;
                }
            }

            // Log starting (no wait)
            puts((TRUE == bAssetStartWaitSupported) ? "STARTING" : "STARTED");
        }

        // Continue. Try to start next asset.
        gsAppData.sCycleInfo.un8NumStartedAssets++;
    }

    // At this point all assets are started.

    // Get last started asset's info
    eCurrentAsset =
        (SMS_APP_ASSET_ENUM)(gsAppData.sCycleInfo.un8NumStartedAssets - 1);
    psAssetInfo = &(gasAssetInfo[eCurrentAsset]);

    // Get absolute or random operational time value
    un32Time = SMS_APP_un32GetOperationalTime();
    if (0 == un32Time)
    {
        // Initiate current cycle stop now
        bResult = SMS_APP_bBeginCycleStop();
    }
    else
    {
        vIndentedPrint(psAssetInfo->un8Indent, '.',
            LOG_ACTION_COLUMN_LEN_MAX, "STAY WORKING: %u ms (%u s)",
            un32Time, un32Time / MILLISECONDS_IN_SECOND);

        // Start the timer. Cycle stop will be initiated on timer event.
        bResult = SMS_APP_bStartIntervalTimer(un32Time);
    }

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bStopNext
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopNext ( void )
{
    BOOLEAN bResult = TRUE;
    SMS_APP_ASSET_ENUM eCurrentAsset;
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo;
    BOOLEAN bMandatoryStop = FALSE;
    UN32 un32Time;

    while (0 != gsAppData.sCycleInfo.un8NumStartedAssets)
    {
        // Select next asset to stop
        eCurrentAsset =
            (SMS_APP_ASSET_ENUM)(gsAppData.sCycleInfo.un8NumStartedAssets - 1);
        psAssetInfo = &(gasAssetInfo[eCurrentAsset]);

        // Check if asset stop is applicable
        if (NULL == psAssetInfo->tFnStop ||
            ASSET_STATE_STOPPED == psAssetInfo->eState ||
            ASSET_STATE_START_FAILED == psAssetInfo->eState ||
            ASSET_STATE_ERROR == psAssetInfo->eState)
        {
            // Stop is not applicable for this asset.
            // Continue. Try to stop next asset.
            gsAppData.sCycleInfo.un8NumStartedAssets--;
            continue;
        }

        // Get MANDATORY_STOP flag value
        bMandatoryStop = (0 != (psAssetInfo->tProps &
                                SMS_APP_ASSET_PROP_MANDATORY_STOP));

        // Check is asset stop is mandatory required
        if (FALSE == bMandatoryStop)
        {
            // Get absolute or random condition
            BOOLEAN bAssetStopRequired =
                (SMS_APP_un32GetTestValue(TRUE, TRUE) != 0);

            if (FALSE == bAssetStopRequired)
            {
                // Stop is not required.
                // Continue. Try to stop next asset.
                gsAppData.sCycleInfo.un8NumStartedAssets--;
                continue;
            }
        }

        // Initiate asset stop

        vIndentedPrint(psAssetInfo->un8Indent, '.',
            LOG_ACTION_COLUMN_LEN_MAX, "Stop %s", psAssetInfo->pacName);

        bResult =
            psAssetInfo->tFnStop(&(psAssetInfo->pvObjectData));
        if (FALSE == bResult)
        {
            // Count error
            gsAppData.sCycleInfo.un16NumErrors++;

            // Log error
            puts("FAILED");

            if (TRUE == bMandatoryStop)
            {
                // This is critical error.
                // Current cycle cannot be completed,
                // so the cycler shall be stopped immediately.
                vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                    "Critical error. Stop cycler immediately.\n");

                return FALSE;
            }
        }
        else
        {
            // Now current asset stop is initiated

            // Get props values
            BOOLEAN bAssetStopWaitSupported =
                (0 != (psAssetInfo->tProps &
                       SMS_APP_ASSET_PROP_STOPPED_EVENT_SUPPORTED));
            BOOLEAN bAssetStopWaitRequired =
                (0 != (psAssetInfo->tProps &
                       SMS_APP_ASSET_PROP_STOPPED_EVENT_REQUIRED));
            BOOLEAN bCommonStopWaitDisabled =
                (0 != (gsAppData.tProps &
                       SMS_APP_CYCLER_PROP_NO_STOP_WAIT));

            // Check if waiting for STOPPED state is supported for this asset
            if (FALSE == bAssetStopWaitSupported)
            {
                // Stop (RELEASED) event is not supported.
                // Update asset's state.
                psAssetInfo->eState = ASSET_STATE_STOPPED;
            }
            else
            {
                // Update asset's state
                psAssetInfo->eState = ASSET_STATE_STOPPING;

                // Check if waiting for STOPPED state is REQUIRED for this asset
                if (FALSE == bAssetStopWaitRequired)
                {
                    // Check for common "no wait" condition
                    if (TRUE == bCommonStopWaitDisabled)
                    {
                        // Log stopping (no wait)
                        puts("STOPPING");

                        // Continue. Try to stop next asset.
                        gsAppData.sCycleInfo.un8NumStartedAssets--;
                        continue;
                    }

                    // Get absolute or random condition
                    bAssetStopWaitRequired =
                        (SMS_APP_un32GetTestValue(
                            TRUE, DEFAULT_STOP_WAIT_CONDITION) != 0);
                }

                if (TRUE == bAssetStopWaitRequired)
                {
                    // Set pending asset
                    gsAppData.sCycleInfo.ePendingAsset = eCurrentAsset;

                    // Log stopping (wait)
                    vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX, "STOPPING");

                    // Nothing else to do here. Waiting for state event.
                    return TRUE;
                }
            }

            // Check if we should wait some time
            // before next asset stop.

            // Check for common "no wait" condition
            if (FALSE == bCommonStopWaitDisabled &&
                gsAppData.sCycleInfo.un8NumStartedAssets > 1)
            {
                // Get absolute or random value.
                un32Time = SMS_APP_un32GetAssetStopWaitTime(psAssetInfo);
                if (0 != un32Time)
                {
                    // Reset pending asset
                    gsAppData.sCycleInfo.ePendingAsset = NUM_ASSETS;

                    // Log stopping (wait)
                    vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX,
                        (TRUE == bAssetStopWaitSupported) ? "STOPPING" : "STOPPED");
                    // Log timeout (wait)
                    vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX, "wait %u ms", un32Time);

                    // Start the timer
                    bResult = SMS_APP_bStartIntervalTimer(un32Time);

                    // Nothing else to do here. Waiting for timer event.
                    return bResult;
                }
            }

            // Log stopping (no wait)
            puts((TRUE == bAssetStopWaitSupported) ? "STOPPING" : "STOPPED");
        }

        // Continue. Try to stop next asset.
        gsAppData.sCycleInfo.un8NumStartedAssets--;
    }

    // At this point all assets are stopped
    bResult = SMS_APP_bCompleteCycle();

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bStopCycler
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopCycler ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bResult = TRUE;

    vResetSmsConsoleColor();

    // Delete the timer
    eReturnCode = OSAL.eTimerDelete(gsAppData.hTimer);
    if (OSAL_SUCCESS != eReturnCode)
    {
        printf("ERROR: Unable to delete cycler timer: %d (%s)\n",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
        bResult = FALSE;
    }

    gsAppData.hTimer = OSAL_INVALID_OBJECT_HDL;

    puts(INNER_CYCLE_SEPARATOR);
    puts(OUTER_CYCLER_TEST_STOP);
    puts(INNER_CYCLE_SEPARATOR);

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bOnStartingEventHandler
*
*******************************************************************************/
static BOOLEAN SMS_APP_bOnStartingEventHandler (
    SMS_APP_ASSET_ENUM eAsset,
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo,
    SMS_APP_ASSET_STATE_ENUM eState
        )
{
    BOOLEAN bResult = TRUE;
    BOOLEAN bMandatoryStart =
        (0 != (psAssetInfo->tProps & SMS_APP_ASSET_PROP_MANDATORY_START));
    SMS_APP_ASSET_INFO_STRUCT *psPendingAssetInfo;
    UN32 un32Time;

    // STARTED and ERROR states should be handled only
    if (ASSET_STATE_STARTED != eState
        && ASSET_STATE_ERROR != eState)
    {
        return TRUE;
    }

    // Update asset's state
    psAssetInfo->eState = eState;

    // Check for ERROR state
    if (ASSET_STATE_ERROR == eState)
    {
        // Count error
        gsAppData.sCycleInfo.un16NumErrors++;

        // Log error
        if (eAsset == gsAppData.sCycleInfo.ePendingAsset)
        {
            printf("%s (%s)\n",
                SMS_APP_pacAssetStateToString(psAssetInfo->eState),
                TRUE == bMandatoryStart ? "mandatory" : "optional");
        }
        else
        {
            printf("\n");
            vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                "%s state received for %s (%s).\n",
                SMS_APP_pacAssetStateToString(psAssetInfo->eState),
                psAssetInfo->pacName,
                TRUE == bMandatoryStart ? "mandatory" : "optional");
        }

        // If following flag is specified
        // the cycler shall be stopped immediately.
        if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_STOP_ON_ERROR))
        {
            vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                "STOP_ON_ERROR is set. Stop cycler immediately.\n");
            return FALSE;
        }

        // Check if this is mandatory asset which must be obligatory started.
        if (TRUE == bMandatoryStart)
        {
            // This is mandatory asset. We cannot go on if it is not started.
            // So we have to stop all previously started assets
            // and complete current cycle.

            // Count this asset as started since it shall be stopped
            gsAppData.sCycleInfo.un8NumStartedAssets++;
            // Initiate current cycle stop
            bResult = SMS_APP_bBeginCycleStop();

            return bResult;
        }
    }

    // Check if pending asset is set
    if (NUM_ASSETS == gsAppData.sCycleInfo.ePendingAsset)
    {
        // Skip further processing
        return TRUE;
    }

    // Get pending asset info
    psPendingAssetInfo =
        &(gasAssetInfo[gsAppData.sCycleInfo.ePendingAsset]);

    // Check pending asset state
    if (ASSET_STATE_STOPPED == psPendingAssetInfo->eState)
    {
        // Here we should be waiting for precedent asset start
        if (NUM_ASSETS == psPendingAssetInfo->ePrecedentAsset)
        {
            // Log error
            printf("ERROR: Pending asset %u (%s) in STOPPED state"
                   " has no precedent asset.\n",
                   gsAppData.sCycleInfo.ePendingAsset,
                   psPendingAssetInfo->pacName);
            return FALSE;
        }
        else if (eAsset == psPendingAssetInfo->ePrecedentAsset)
        {
            // Reset pending asset
            gsAppData.sCycleInfo.ePendingAsset = NUM_ASSETS;

            // Keep trying to start current asset.
            // Precedent asset's state will be handled
            // in the SMS_APP_bStartNext().
            bResult = SMS_APP_bStartNext();

            return bResult;
        }
    }

    // Check if this is pending asset's event
    if (eAsset != gsAppData.sCycleInfo.ePendingAsset)
    {
        // Skip further processing
        return TRUE;
    }

    // This is pending asset's event. Reset pending asset.
    gsAppData.sCycleInfo.ePendingAsset = NUM_ASSETS;

    if (ASSET_STATE_STARTED == psAssetInfo->eState)
    {
        // Check if we should wait some time
        // before next asset start.

        // Check if start wait is disabled (common app parameter).
        bResult = (0 != (gsAppData.tProps &
                         SMS_APP_CYCLER_PROP_NO_START_WAIT));
        if (FALSE == bResult)
        {
            // Get absolute or random value.
            un32Time = SMS_APP_un32GetAssetStartWaitTime(psAssetInfo);
            if (0 != un32Time)
            {
                vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX, "%s",
                    SMS_APP_pacAssetStateToString(psAssetInfo->eState));
                vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX,
                    "wait %u ms", un32Time);

                // Start the timer
                bResult = SMS_APP_bStartIntervalTimer(un32Time);

                return bResult;
            }
        }

        // Just log state (no wait)
        printf("%s\n", SMS_APP_pacAssetStateToString(psAssetInfo->eState));
    }

    // Move to the next asset
    gsAppData.sCycleInfo.un8NumStartedAssets++;
    bResult = SMS_APP_bStartNext();

    return bResult;
}

/*******************************************************************************
*
*   SMS_APP_bOnStoppingEventHandler
*
*******************************************************************************/
static BOOLEAN SMS_APP_bOnStoppingEventHandler (
    SMS_APP_ASSET_ENUM eAsset,
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo,
    SMS_APP_ASSET_STATE_ENUM eState
        )
{
    BOOLEAN bMandatoryStop =
        (0 != (psAssetInfo->tProps & SMS_APP_ASSET_PROP_MANDATORY_STOP));
    BOOLEAN bResult = FALSE;
    UN32 un32Time;

    // STOPPED and ERROR states should be handled only
    if (ASSET_STATE_STOPPED != eState
        && ASSET_STATE_ERROR != eState)
    {
        return TRUE;
    }

    // Update asset's state
    psAssetInfo->eState = eState;

    if (ASSET_STATE_ERROR == eState)
    {
        // Count error
        gsAppData.sCycleInfo.un16NumErrors++;

        // Log error
        if (eAsset == gsAppData.sCycleInfo.ePendingAsset)
        {
            printf("%s (%s)\n",
                SMS_APP_pacAssetStateToString(psAssetInfo->eState),
                TRUE == bMandatoryStop ? "mandatory" : "optional");
        }
        else
        {
            printf("\n");
            vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                "%s state received for %s (%s).\n",
                SMS_APP_pacAssetStateToString(psAssetInfo->eState),
                psAssetInfo->pacName,
                TRUE == bMandatoryStop ? "mandatory" : "optional");
        }

        // If following flag is specified
        // the cycler shall be stopped immediately.
        if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_STOP_ON_ERROR))
        {
            vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                "STOP_ON_ERROR is set. Stop cycler immediately.\n");
            return FALSE;
        }

        if (TRUE == bMandatoryStop)
        {
            // This is critical error.
            // Current cycle cannot be completed,
            // so the cycler shall be stopped immediately.
            vIndentedPrint(psAssetInfo->un8Indent, ' ', 0,
                "Critical error. Stop cycler immediately.\n");

            return FALSE;
        }
    }

    // Check if this is event for asset we are pending
    if (eAsset != gsAppData.sCycleInfo.ePendingAsset)
    {
        // Skip further processing
        return TRUE;
    }

    // This is pending asset's event. Reset pending asset.
    gsAppData.sCycleInfo.ePendingAsset = NUM_ASSETS;

    if (ASSET_STATE_STOPPED == psAssetInfo->eState)
    {
        // Check if we should wait some time
        // before next asset stop.

        // Check if stop wait is disabled (common app property).
        bResult = (0 != (gsAppData.tProps &
                         SMS_APP_CYCLER_PROP_NO_STOP_WAIT));
        if (FALSE == bResult)
        {
            // Get absolute or random value.
            un32Time = SMS_APP_un32GetAssetStopWaitTime(psAssetInfo);
            if (0 != un32Time)
            {
                vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX, "%s",
                    SMS_APP_pacAssetStateToString(psAssetInfo->eState));
                vCharPaddingPrint('.', LOG_STATE_COLUMN_LEN_MAX,
                    "wait %u ms", un32Time);

                // Start the timer
                bResult = SMS_APP_bStartIntervalTimer(un32Time);

                return bResult;
            }
        }
    }

    // Just log state (no wait)
    printf("%s\n", SMS_APP_pacAssetStateToString(psAssetInfo->eState));

    // Move to the next asset
    gsAppData.sCycleInfo.un8NumStartedAssets--;
    bResult = SMS_APP_bStopNext();

    return bResult;
}

// SMS asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartSMS
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartSMS (
    void *pvAssetObjectData
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bConfigFileReset = FALSE;

    eReturnCode = SMS.eInitialize(gpacDefaultSmsCfgPath,
                                  gpacDefaultSmsCfgPath,
                                  SXM_INITIALIZER_FILE,
                                  &bConfigFileReset);
    if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopSMS
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopSMS (
    void *pvAssetObjectData
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = SMS.eUninitialize();
    if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
    {
        return FALSE;
    }

    return TRUE;
}

// SRM asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartSRM
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartSRM (
    void *pvAssetObjectData
        )
{
    SRM_OBJECT *phSrm = (SRM_OBJECT *)pvAssetObjectData;

    *phSrm = SRM.hGet( "srh:", "srm:", SRM_OBJECT_EVENT_ALL,
        SRM_vEventCallback, NULL );
    if (SRM_INVALID_OBJECT == *phSrm)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopSRM
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopSRM (
    void *pvAssetObjectData
        )
{
    SRM_OBJECT *phSrm = (SRM_OBJECT *)pvAssetObjectData;

    SRM.vRelease(*phSrm);
    *phSrm = SRM_INVALID_OBJECT;

    return TRUE;
}

/*****************************************************************************
*
*   SRM_vEventCallback
*
*  We use this event callback to monitor the state of the SRM.
*
*****************************************************************************/
static void SRM_vEventCallback (
    SRM_OBJECT hSRM,
    SRM_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & SRM_OBJECT_EVENT_STATE))
    {
        SRM_STATE_ENUM eSrmState;

        eSrmState = SRM.eState(hSRM);

        SMS_APP_vGenericStateCallback(SRM_ASSET, (UN32)eSrmState);
    }

    return;
}

/*******************************************************************************
*
*   SMS_APP_hSrm
*
*   Returns SRM object handle from the SRM asset info.
*
*******************************************************************************/
static SRM_OBJECT SMS_APP_hSrm ( void )
{
    return (SRM_OBJECT)gasAssetInfo[SRM_ASSET].pvObjectData;
}

// MODULE asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartModule
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartModule (
    void *pvAssetObjectData
        )
{
    MODULE_OBJECT *phModule = (MODULE_OBJECT *)pvAssetObjectData;
    SRM_OBJECT hSRM = SMS_APP_hSrm();

    *phModule = MODULE.hGet(hSRM,
                            DECODER_NAME,
                            MODULE_OBJECT_EVENT_STATE,
                            MODULE_vEventCallback,
                            NULL);
    if (MODULE_INVALID_OBJECT == *phModule)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopModule
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopModule (
    void *pvAssetObjectData
        )
{
    MODULE_OBJECT *phModule = (MODULE_OBJECT *)pvAssetObjectData;

    MODULE.vRelease(*phModule);
    *phModule = MODULE_INVALID_OBJECT;

    return TRUE;
}

/*****************************************************************************
*
*   MODULE_vEventCallback
*
*****************************************************************************/
static void MODULE_vEventCallback (
    MODULE_OBJECT hModule,
    MODULE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & MODULE_OBJECT_EVENT_STATE))
    {
        MODULE_STATE_ENUM eModuleState;

        eModuleState = MODULE.eState(hModule);

        SMS_APP_vGenericStateCallback(MODULE_ASSET, (UN32)eModuleState);
    }

    return;
}

/*******************************************************************************
*
*   SMS_APP_hModule
*
*   Returns MODULE object handle from the MODULE asset info.
*
*******************************************************************************/
static MODULE_OBJECT SMS_APP_hModule ( void )
{
    return (MODULE_OBJECT)gasAssetInfo[MODULE_ASSET].pvObjectData;
}

// DECODER and DECODER_TUNE assets implementation

/*******************************************************************************
*
*   SMS_APP_bStartDecoder
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartDecoder (
    void *pvAssetObjectData
        )
{
    DECODER_OBJECT *phDecoder = (DECODER_OBJECT *)pvAssetObjectData;
    SRM_OBJECT hSrm = SMS_APP_hSrm();

    *phDecoder = DECODER.hGet(hSrm,
                              DECODER_NAME,
                              DECODER_OBJECT_EVENT_STATE |
                                DECODER_OBJECT_EVENT_TUNE,
                              DECODER_vEventCallback,
                              NULL);
    if (DECODER_INVALID_OBJECT == *phDecoder)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopDecoder
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopDecoder (
    void *pvAssetObjectData
        )
{
    DECODER_OBJECT *phDecoder = (DECODER_OBJECT *)pvAssetObjectData;

    DECODER.vRelease(*phDecoder);
    *phDecoder = DECODER_INVALID_OBJECT;

    return TRUE;
}

/*****************************************************************************
*
*   DECODER_vEventCallback
*
*****************************************************************************/
static void DECODER_vEventCallback (
    DECODER_OBJECT hDecoder,
    DECODER_EVENT_MASK tEventMask,
    CHANNEL_OBJECT hChannel,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DECODER_OBJECT_EVENT_STATE))
    {
        DECODER_STATE_ENUM eDecoderState;

        eDecoderState = DECODER.eState(hDecoder);

        SMS_APP_vGenericStateCallback(DECODER_ASSET, (UN32)eDecoderState);
    }
    else if (0 != (tEventMask & DECODER_OBJECT_EVENT_TUNE))
    {
        TUNE_STATE_ENUM eTuneState;

        eTuneState = DECODER.eTuneState(hDecoder);
        if (TUNE_STATE_TUNED == eTuneState)
        {
            SMS_APP_vGenericStateCallback(DECODER_TUNE_ASSET,
                                          (UN32)ASSET_STATE_STARTED);
        }
    }

    return;
}

/*******************************************************************************
*
*   SMS_APP_bStartDecoderTune
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartDecoderTune (
    void *pvAssetObjectData
        )
{
    DECODER_OBJECT hDecoder = SMS_APP_hDecoder();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DECODER.eTuneDirect(hDecoder, DEFAULT_TUNE_CHANNEL_ID, TRUE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   SMS_APP_hDecoder
*
*****************************************************************************/
static DECODER_OBJECT SMS_APP_hDecoder ( void )
{
    return (DECODER_OBJECT)gasAssetInfo[DECODER_ASSET].pvObjectData;
}

// Art Service and Art Products assets implementation

/*******************************************************************************
*
*   SMS_APP_bStartArtService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartArtService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = CHANNEL_ART.hStart(SRH_DRIVER_NAME,
                                    DATASERVICE_EVENT_ALL,
                                    CHANNEL_ART_vEventCallback,
                                    (void*)&gasAssetInfo[DECODER_ASSET],
                                    CHANNEL_ART_AVAILABLE_IMAGE_TYPE_ALL,
                                    CHANNEL_ART_OPTION_NONE);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }
    
    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopArtService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopArtService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    CHANNEL_ART.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   CHANNEL_ART_vEventCallback
*
*******************************************************************************/
static void CHANNEL_ART_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & CHANNEL_ART_SERVICE_EVENT_SERVICE_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = CHANNEL_ART.eState(hDataService);

        SMS_APP_vGenericStateCallback(ART_SERVICE_ASSET, (UN32)eState);
    }
    else if (0 != (tEventMask & CHANNEL_ART_SERVICE_EVENT_ALBUM_PROD_STATE))
    {
        DATA_PRODUCT_MASK eMask;
        DATA_PRODUCT_STATE_ENUM eState;
        SMSAPI_RETURN_CODE_ENUM eReturn;

        // Get the state of the product
        eReturn = DATA.eProductState(hDataService,
                                     DATA_PRODUCT_TYPE_ALBUM_ART,
                                     &eMask, &eState);
        if (SMSAPI_RETURN_CODE_SUCCESS != eReturn)
        {
            eState = DATA_PRODUCT_STATE_ERROR;
        }

        SMS_APP_vGenericStateCallback(ALBUM_ART_PRODUCT_ASSET, (UN32)eState);
    }
    else if (0 != (tEventMask & CHANNEL_ART_SERVICE_EVENT_CHAN_PROD_STATE))
    {
        DATA_PRODUCT_MASK eMask;
        DATA_PRODUCT_STATE_ENUM eState;
        SMSAPI_RETURN_CODE_ENUM eReturn;

        // Get the state of the product
        eReturn = DATA.eProductState(hDataService,
                                     DATA_PRODUCT_TYPE_CHANNEL_ART,
                                     &eMask, &eState);
        if (SMSAPI_RETURN_CODE_SUCCESS != eReturn)
        {
            eState = DATA_PRODUCT_STATE_ERROR;
        }

        SMS_APP_vGenericStateCallback(CHANNEL_ART_PRODUCT_ASSET, (UN32)eState);
    }

    return;
}

static DATASERVICE_MGR_OBJECT SMS_APP_hArtService ( void )
{
    return (DATASERVICE_MGR_OBJECT)gasAssetInfo[ART_SERVICE_ASSET].pvObjectData;
}

static BOOLEAN SMS_APP_bStartAlbumArtProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hArtService = SMS_APP_hArtService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eEnableProduct(hArtService,
                                  DATA_PRODUCT_TYPE_ALBUM_ART,
                                  DATA_PRODUCT_MASK_NONE);

    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

static BOOLEAN SMS_APP_bStopAlbumArtProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hArtService = SMS_APP_hArtService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eDisableProduct(hArtService,
                                   DATA_PRODUCT_TYPE_ALBUM_ART);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

static BOOLEAN SMS_APP_bStartChannelArtProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hArtService = SMS_APP_hArtService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eEnableProduct(hArtService,
                                  DATA_PRODUCT_TYPE_CHANNEL_ART,
                                  DATA_PRODUCT_MASK_NONE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

static BOOLEAN SMS_APP_bStopChannelArtProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hArtService = SMS_APP_hArtService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eDisableProduct(hArtService,
                                   DATA_PRODUCT_TYPE_CHANNEL_ART);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

// EPG Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartEPG
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartEPG (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = EPG.hStart(SRH_DRIVER_NAME,
                            EPG_NUM_SEGMENTS_MAX,
                            DATASERVICE_EVENT_ALL,
                            EPG_vEventCallback,
                            (void*)&gasAssetInfo[EPG_ASSET],
                            NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }
    
    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopEPG
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopEPG (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    EPG.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   EPG_vEventCallback
*
*******************************************************************************/
static void EPG_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = EPG.eState(hDataService);

        SMS_APP_vGenericStateCallback(EPG_ASSET, (UN32)eState);
    }

    return;
}

#ifdef USE_GENERIC_TRAFFIC
// Generic Traffic Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartGenericTraffic
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartGenericTraffic (
    void *pvAssetObjectData
        )
{
    // Generic traffic is slightly different than the other services,
    // as it's a generic data service.

    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = DATA.hStart(SRH_DRIVER_NAME,
                             TRAFFIC_DSI,
                             FALSE,
                             DATASERVICE_EVENT_ALL,
                             GENERIC_TRAFFIC_vEventCallback,
                             (void*)&gasAssetInfo[TRAFFIC_ASSET]);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopGenericTraffic
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopGenericTraffic (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    DATA.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*****************************************************************************
*
*  GEN_TRAFFIC_vEventCallback
*
*  We use this event callback to monitor the state of our data services, and
*  jump into the next appropriate state if our services have completed starting
*  or stopping.
*
*****************************************************************************/
static void GENERIC_TRAFFIC_vEventCallback(
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = DATA.eState(hDataService);

        SMS_APP_vGenericStateCallback(TRAFFIC_ASSET, (UN32)eState);

        if (eState == DATASERVICE_STATE_READY)
        {
            size_t tNumDMIs;
            DATASERVICE_DMI_CONFIG_STRUCT sDMIs[] = {
                {513, TRUE}, {514, TRUE}, {515, TRUE}, {516, TRUE}, {517, TRUE},
                {518, TRUE}, {519, TRUE}, {520, TRUE}, {521, TRUE}, {522, TRUE},
                {523, TRUE}, {524, TRUE}, {525, TRUE}, {526, TRUE}, {527, TRUE},
                {528, TRUE}, {529, TRUE}, {530, TRUE}, {531, TRUE}, {532, TRUE},
                {533, TRUE}, {534, TRUE}, {535, TRUE}, {536, TRUE}, {537, TRUE},
                {538, TRUE}, {539, TRUE}, {540, TRUE}, {541, TRUE}, {542, TRUE},
                {543, TRUE}, {544, TRUE}, {545, TRUE}, {546, TRUE}, {547, TRUE},
                {548, TRUE}, {549, TRUE}, {550, TRUE}, {551, TRUE}, {552, TRUE},
                {553, TRUE}, {554, TRUE}, {555, TRUE}, {556, TRUE}, {557, TRUE},
                {558, TRUE}, {559, TRUE}, {560, TRUE}, {561, TRUE}, {562, TRUE},
                {563, TRUE}, {564, TRUE}, {565, TRUE}, {566, TRUE}, {567, TRUE},
                {568, TRUE}, {569, TRUE}, {570, TRUE}, {571, TRUE}, {572, TRUE},
                {573, TRUE}, {574, TRUE}, {575, TRUE}
            };

            tNumDMIs = sizeof(sDMIs)/sizeof(DATASERVICE_DMI_CONFIG_STRUCT);

            // After transition to READY we enable all DMIs for the service
            // to simulate maximum workload for testing purposes
            DATA.eManageDataStream(hDataService, sDMIs, tNumDMIs);
        }
        
    }
    else if (0 != (tEventMask & DATASERVICE_EVENT_NEW_DATA))
    {
        // Generic service payloads will come in with a tEventMask
        // set to DATASERVICE_EVENT_NEW_DATA; normally this
        // event would be disabled, but we're leaving it on for
        // example purposes. As we actually don't need this
        // payload, we just go ahead and free it.

        OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
        OSAL.eBufferFree(hPayload);
    }

    return;
}
#else // #ifdef USE_GENERIC_TRAFFIC

// Traffic Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartTraffic
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartTraffic (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = TRAFFIC.hStart(SRH_DRIVER_NAME,
                                NULL,
                                NULL,
                                DATASERVICE_EVENT_ALL,
                                TRAFFIC_vEventCallback,
                                (void*)&gasAssetInfo[TRAFFIC_ASSET],
                                NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopTraffic
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopTraffic (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    TRAFFIC.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*****************************************************************************
*
*  TRAFFIC_vEventCallback
*
*****************************************************************************/
static void TRAFFIC_vEventCallback(
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = TRAFFIC.eState(hDataService);

        SMS_APP_vGenericStateCallback(TRAFFIC_ASSET, (UN32)eState);
    }

    return;
}
#endif

// Movies Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartMovies
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartMovies (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = MOVIES.hStart(SRH_DRIVER_NAME,
                               DATASERVICE_EVENT_ALL,
                               MOVIES_vEventCallback,
                               (void*)&gasAssetInfo[MOVIE_LISTINGS_ASSET],
                               NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopMovies
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopMovies (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    MOVIES.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   MOVIES_vEventCallback
*
*******************************************************************************/
static void MOVIES_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = MOVIES.eState(hDataService);

        SMS_APP_vGenericStateCallback(MOVIE_LISTINGS_ASSET, (UN32)eState);
    }

    return;
}

// Fuel Price Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartFuelPriceService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartFuelPriceService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = FUEL.hStart(SRH_DRIVER_NAME,
                             FUEL_PRICE_SORT_METHOD_FUELTYPE,
                             DATASERVICE_EVENT_ALL,
                             FUEL_PRICE_vEventCallback,
                             (void*)&gasAssetInfo[FUEL_PRICE_SERVICE_ASSET],
                             NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopFuelPriceService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopFuelPriceService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    FUEL.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   FUEL_PRICE_vEventCallback
*
*******************************************************************************/
static void FUEL_PRICE_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    static DATASERVICE_STATE_ENUM eLastServiceState =
        DATASERVICE_STATE_STOPPED;
    static DATA_PRODUCT_STATE_ENUM eLastDbUpdateState =
        DATA_PRODUCT_STATE_DISABLED;
    static DATA_PRODUCT_STATE_ENUM eLastFuelPriceState =
        DATA_PRODUCT_STATE_DISABLED;

    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eServiceState;
        DATA_PRODUCT_MASK tMask;
        DATA_PRODUCT_STATE_ENUM eProductState;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Get Service state
        eServiceState = FUEL.eState(hDataService);
        if (eLastServiceState != eServiceState)
        {
            eLastServiceState = eServiceState;
            SMS_APP_vGenericStateCallback(FUEL_PRICE_SERVICE_ASSET,
                                          (UN32)eServiceState);
        }

        // Get the product state for DB updates
        eReturnCode = DATA.eProductState(hDataService,
                                         DATA_PRODUCT_TYPE_DB_UPDATES,
                                         &tMask, &eProductState);
        if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
        {
            eProductState = DATA_PRODUCT_STATE_ERROR;
        }
        if (eLastDbUpdateState != eProductState)
        {
            eLastDbUpdateState = eProductState;
            SMS_APP_vGenericStateCallback(FUEL_DB_UPDATE_PRODUCT_ASSET,
                                          (UN32)eProductState);
        }

        // Get the product state for price updates
        eReturnCode = DATA.eProductState(hDataService,
                                         DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES,
                                         &tMask, &eProductState);
        if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
        {
            eProductState = DATA_PRODUCT_STATE_ERROR;
        }
        if (eLastFuelPriceState != eProductState)
        {
            eLastFuelPriceState = eProductState;
            SMS_APP_vGenericStateCallback(FUEL_PRICE_PRODUCT_ASSET,
                                          (UN32)eProductState);
        }
    }

    return;
}

/*******************************************************************************
*
*   SMS_APP_hFuelService
*
*******************************************************************************/
static DATASERVICE_MGR_OBJECT SMS_APP_hFuelService ( void )
{
    return (DATASERVICE_MGR_OBJECT)gasAssetInfo[FUEL_PRICE_SERVICE_ASSET].pvObjectData;
}

/*******************************************************************************
*
*   SMS_APP_bStartFuelDbUpdateProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartFuelDbUpdateProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hFuelService = SMS_APP_hFuelService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eEnableProduct(hFuelService,
                                  DATA_PRODUCT_TYPE_DB_UPDATES,
                                  DATA_PRODUCT_MASK_NONE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopFuelDbUpdateProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopFuelDbUpdateProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hFuelService = SMS_APP_hFuelService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eDisableProduct(hFuelService,
                                   DATA_PRODUCT_TYPE_DB_UPDATES);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStartFuelPriceProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartFuelPriceProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hFuelService = SMS_APP_hFuelService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eEnableProduct(hFuelService,
                                  DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES,
                                  DATA_PRODUCT_MASK_NONE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopFuelPriceProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopFuelPriceProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hFuelService = SMS_APP_hFuelService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eDisableProduct(hFuelService,
                                   DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

// EV Stations Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartEvService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartEvService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = FUEL.hEVStart(SRH_DRIVER_NAME,
                               FUEL_PRICE_SORT_METHOD_FUELTYPE,
                               DATASERVICE_EVENT_ALL,
                               EV_vEventCallback,
                               (void*)&gasAssetInfo[EV_SERVICE_ASSET],
                               NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopEvService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopEvService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    FUEL.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   EV_vEventCallback
*
*******************************************************************************/
static void EV_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    static DATASERVICE_STATE_ENUM eLastServiceState =
        DATASERVICE_STATE_STOPPED;
    static DATA_PRODUCT_STATE_ENUM eLastDbUpdateState =
        DATA_PRODUCT_STATE_DISABLED;
    static DATA_PRODUCT_STATE_ENUM eLastLogoUpdateState =
        DATA_PRODUCT_STATE_DISABLED;
    static DATA_PRODUCT_STATE_ENUM eLastAvailabilityState =
        DATA_PRODUCT_STATE_DISABLED;

    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eServiceState;
        DATA_PRODUCT_MASK tMask;
        DATA_PRODUCT_STATE_ENUM eProductState;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Get Service state
        eServiceState = FUEL.eState(hDataService);
        if (eLastServiceState != eServiceState)
        {
            eLastServiceState = eServiceState;
            SMS_APP_vGenericStateCallback(EV_SERVICE_ASSET,
                                          (UN32)eServiceState);
        }

        // Get the product state for DB updates
        eReturnCode = DATA.eProductState(hDataService,
                                         DATA_PRODUCT_TYPE_DB_UPDATES,
                                         &tMask, &eProductState);
        if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
        {
            eProductState = DATA_PRODUCT_STATE_ERROR;
        }
        if (eLastDbUpdateState != eProductState)
        {
            eLastDbUpdateState = eProductState;
            SMS_APP_vGenericStateCallback(EV_DB_UPDATE_PRODUCT_ASSET,
                                          (UN32)eProductState);
        }

        // Get the product state for logo updates
        eReturnCode = DATA.eProductState(hDataService,
                                         DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES,
                                         &tMask, &eProductState);
        if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
        {
            eProductState = DATA_PRODUCT_STATE_ERROR;
        }
        if (eLastLogoUpdateState != eProductState)
        {
            eLastLogoUpdateState = eProductState;
            SMS_APP_vGenericStateCallback(EV_STATION_LOGO_PRODUCT_ASSET,
                                          (UN32)eProductState);
        }

        // Get the product state for availability updates
        eReturnCode = DATA.eProductState(hDataService,
                                         DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES,
                                         &tMask, &eProductState);
        if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
        {
            eProductState = DATA_PRODUCT_STATE_ERROR;
        }
        if (eLastAvailabilityState != eProductState)
        {
            eLastAvailabilityState = eProductState;
            SMS_APP_vGenericStateCallback(EV_AVAILABILITY_PRODUCT_ASSET,
                                          (UN32)eProductState);
        }
    }

    return;
}

/*******************************************************************************
*
*   SMS_APP_hEvService
*
*******************************************************************************/
static DATASERVICE_MGR_OBJECT SMS_APP_hEvService ( void )
{
    return (DATASERVICE_MGR_OBJECT)gasAssetInfo[EV_SERVICE_ASSET].pvObjectData;
}

/*******************************************************************************
*
*   SMS_APP_bStartEvDbUpdateProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartEvDbUpdateProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hEvService = SMS_APP_hEvService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eEnableProduct(hEvService,
                                  DATA_PRODUCT_TYPE_DB_UPDATES,
                                  DATA_PRODUCT_MASK_NONE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopEvDbUpdateProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopEvDbUpdateProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hEvService = SMS_APP_hEvService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eDisableProduct(hEvService,
                                   DATA_PRODUCT_TYPE_DB_UPDATES);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStartEvStationLogoProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartEvStationLogoProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hEvService = SMS_APP_hEvService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eEnableProduct(hEvService,
                                  DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES,
                                  DATA_PRODUCT_MASK_NONE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopEvStationLogoProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopEvStationLogoProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hEvService = SMS_APP_hEvService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eDisableProduct(hEvService,
                                   DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStartEvAvailabilityProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartEvAvailabilityProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hEvService = SMS_APP_hEvService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eEnableProduct(hEvService,
                                  DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES,
                                  DATA_PRODUCT_MASK_NONE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopEvAvailabilityProduct
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopEvAvailabilityProduct (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT hEvService = SMS_APP_hEvService();
    SMSAPI_RETURN_CODE_ENUM eResult;

    eResult = DATA.eDisableProduct(hEvService,
                                   DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES);
    if (SMSAPI_RETURN_CODE_SUCCESS != eResult)
    {
        return FALSE;
    }

    return TRUE;
}

// Sports Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartSportsService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartSportsService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = SPORTS_SERVICE.hStart(SRH_DRIVER_NAME,
                               DATASERVICE_EVENT_ALL,
                               SPORTS_SERVICE_vEventCallback,
                               (void*)&gasAssetInfo[SPORTS_SERVICE_ASSET],
                               NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopSportsService
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopSportsService (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    SPORTS_SERVICE.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   SPORTS_SERVICE_vEventCallback
*
*******************************************************************************/
static void SPORTS_SERVICE_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = SPORTS_SERVICE.eState(hDataService);

        SMS_APP_vGenericStateCallback(SPORTS_SERVICE_ASSET, (UN32)eState);
    }

    return;
}

// Graphical Weather  Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartGraphicalWeather
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartGraphicalWeather (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = AGW.hStart(SRH_DRIVER_NAME,
                            gpacDefaultCliResourcesDir,
                            DATASERVICE_EVENT_ALL,
                            GRAPHICAL_WEATHER_vEventCallback,
                            (void*)&gasAssetInfo[GRAPHICAL_WEATHER_ASSET],
                            NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopGraphicalWeather
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopGraphicalWeather (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    AGW.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   GRAPHICAL_WEATHER_vEventCallback
*
*******************************************************************************/
static void GRAPHICAL_WEATHER_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = AGW.eState(hDataService);

        SMS_APP_vGenericStateCallback(GRAPHICAL_WEATHER_ASSET, (UN32)eState);
    }

    return;
}

// Tabular Weather Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartTabularWeather
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartTabularWeather (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = WEATHER.hStart(SRH_DRIVER_NAME,
                                DATASERVICE_EVENT_ALL,
                                TABULAR_WEATHER_vEventCallback,
                                (void*)&gasAssetInfo[TABULAR_WEATHER_ASSET],
                                NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopTabularWeather
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopTabularWeather (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    WEATHER.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   TABULAR_WEATHER_vEventCallback
*
*******************************************************************************/
static void TABULAR_WEATHER_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = WEATHER.eState(hDataService);

        SMS_APP_vGenericStateCallback(TABULAR_WEATHER_ASSET, (UN32)eState);
    }

    return;
}

// Stocks Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartStockTicker
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartStockTicker (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = STOCK_TICKER.hStart(SRH_DRIVER_NAME,
                                DATASERVICE_EVENT_ALL,
                                STOCK_TICKER_vEventCallback,
                                (void*)&gasAssetInfo[STOCK_TICKER_ASSET],
                                NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopStockTicker
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopStockTicker (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    STOCK_TICKER.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   STOCK_TICKER_vEventCallback
*
*******************************************************************************/
static void STOCK_TICKER_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = STOCK_TICKER.eState(hDataService);

        SMS_APP_vGenericStateCallback(STOCK_TICKER_ASSET, (UN32)eState);
    }

    return;
}

// Saftety Cameras Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartSafetyCameras
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartSafetyCameras (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;
    DATASERVICE_OPTIONS_STRUCT sOption;

    sOption.tMask = DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES;

    *phService = SAFETY_CAMERAS.hStart(SRH_DRIVER_NAME,
                                       DATASERVICE_EVENT_ALL,
                                       SAFETY_CAMERAS_vEventCallback,
                                       (void*)&gasAssetInfo[SAFETY_CAMERAS_ASSET],
                                       &sOption);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopSafetyCameras
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopSafetyCameras (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    SAFETY_CAMERAS.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   SAFETY_CAMERAS_vEventCallback
*
*******************************************************************************/
static void SAFETY_CAMERAS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = SAFETY_CAMERAS.eState(hDataService);

        SMS_APP_vGenericStateCallback(SAFETY_CAMERAS_ASSET, (UN32)eState);
    }

    return;
}

// Traffic Cameras Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartTrafficCameras
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartTrafficCameras (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = TRAFFIC_CAMERAS.hStart(SRH_DRIVER_NAME,
                                        DATASERVICE_EVENT_ALL,
                                        TRAFFIC_CAMERAS_vEventCallback,
                                        (void*)&gasAssetInfo[TRAFFIC_CAMERAS_ASSET],
                                        NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopTrafficCameras
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopTrafficCameras (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    TRAFFIC_CAMERAS.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   TRAFFIC_CAMERAS_vEventCallback
*
*******************************************************************************/
static void TRAFFIC_CAMERAS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = TRAFFIC_CAMERAS.eState(hDataService);

        SMS_APP_vGenericStateCallback(TRAFFIC_CAMERAS_ASSET, (UN32)eState);
    }

    return;
}

// WS Alerts Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartWsAlerts
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartWsAlerts (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = WS_ALERTS.hStart(SRH_DRIVER_NAME,
                                  DATASERVICE_EVENT_ALL,
                                  WS_ALERTS_vEventCallback,
                                  (void*)&gasAssetInfo[WS_ALERTS_ASSET],
                                  NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopWsAlerts
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopWsAlerts (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    WS_ALERTS.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   WS_ALERTS_vEventCallback
*
*******************************************************************************/
static void WS_ALERTS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = WS_ALERTS.eState(hDataService);

        SMS_APP_vGenericStateCallback(WS_ALERTS_ASSET, (UN32)eState);
    }

    return;
}

// Maps Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartMaps
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartMaps (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = MAPS.hStart(SRH_DRIVER_NAME,
                             NULL,
                             DATASERVICE_EVENT_ALL,
                             MAPS_vEventCallback,
                             (void*)&gasAssetInfo[MAPS_ASSET],
                             NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopMaps
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopMaps (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    MAPS.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   MAPS_vEventCallback
*
*******************************************************************************/
static void MAPS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = MAPS.eState(hDataService);

        SMS_APP_vGenericStateCallback(MAPS_ASSET, (UN32)eState);
    }

    return;
}

// Phonetics Service asset implementation

/*******************************************************************************
*
*   SMS_APP_bStartPhonetics
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStartPhonetics (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    *phService = PHONETICS.hStart(SRH_DRIVER_NAME,
                                  gpacDefaultCliResourcesDir,
                                  DATASERVICE_EVENT_ALL,
                                  PHONETICS_vEventCallback,
                                  (void*)&gasAssetInfo[PHONETICS_ASSET],
                                  NULL, NULL, NULL);
    if (DATASERVICE_MGR_INVALID_OBJECT == *phService)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   SMS_APP_bStopPhonetics
*
*******************************************************************************/
static BOOLEAN SMS_APP_bStopPhonetics (
    void *pvAssetObjectData
        )
{
    DATASERVICE_MGR_OBJECT *phService =
        (DATASERVICE_MGR_OBJECT *)pvAssetObjectData;

    PHONETICS.vStop(*phService);
    *phService = DATASERVICE_MGR_INVALID_OBJECT;

    return TRUE;
}

/*******************************************************************************
*
*   PHONETICS_vEventCallback
*
*******************************************************************************/
static void PHONETICS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    if (0 != (tEventMask & DATASERVICE_EVENT_STATE))
    {
        DATASERVICE_STATE_ENUM eState;

        eState = PHONETICS.eState(hDataService);

        SMS_APP_vGenericStateCallback(PHONETICS_ASSET, (UN32)eState);
    }

    return;
}

/*******************************************************************************
*
*   main
*
*******************************************************************************/
int main( int argc, char *argv[] )
{
    BOOLEAN bOsalStarted;
    int iRetval = EXIT_FAILURE;
    N32 idx = 1;

    // Initialize serial device as empty
    gpacSerialDevice = "";

    for ( idx = 1; idx < argc; idx++ )
    {
        if ( strcmp(argv[idx], "-port") == 0 )
        {
            if ( idx+1 >= argc )
            {
                puts( "You're missing an argument to -port!" );
                return EXIT_FAILURE;
            }

            gpacSerialDevice = argv[idx+1];
            idx+=1;
        }
        else if ( strcmp(argv[idx], "-cycles") == 0 )
        {
            if (0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_START_ALL) ||
                0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_CYCLE_FOREVER))
            {
                puts( "ERROR: -cycles can not be used"
                    " with -startall or -contcycle!" );
                return EXIT_FAILURE;
            }

            ++idx;

            if ( idx >= argc )
            {
                puts( "You're missing an argument to -cycles!" );
                return EXIT_FAILURE;
            }

            gsAppData.un32MaxCycles = atoi( argv[idx] );
        }
        else if ( strcmp(argv[idx], "-contcycle" ) == 0)
        {
            if (gsAppData.un32MaxCycles > 0 ||
                0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_START_ALL))
            {
                puts( "ERROR: -contcycle can not be used with -startall or -cycles!" );
                return EXIT_FAILURE;
            }

            gsAppData.tProps |= SMS_APP_CYCLER_PROP_CYCLE_FOREVER;
        }
        else if ( strcmp(argv[idx], "-random" ) == 0)
        {
            gsAppData.tProps |= SMS_APP_CYCLER_PROP_RANDOM_TEST;
        }
        else if ( strcmp(argv[idx], "-optime" ) == 0)
        {
            ++idx;

            if ( idx >= argc )
            {
                puts( "You're missing an argument to -optime!" );
                return EXIT_FAILURE;
            }

            gsAppData.sCycleInfo.un32OperationalTime = atoi( argv[idx] );
        }
        else if ( strcmp(argv[idx], "-nowait" ) == 0)
        {
            ++idx;

            if ( idx >= argc )
            {
                puts( "You're missing an argument to -nowait!" );
                return EXIT_FAILURE;
            }

            if ( strcmp(argv[idx], "start") == 0 )
            {
                gsAppData.tProps |= SMS_APP_CYCLER_PROP_NO_START_WAIT;
            }
            else if ( strcmp(argv[idx], "stop") == 0 )
            {
                gsAppData.tProps |= SMS_APP_CYCLER_PROP_NO_STOP_WAIT;
            }
            else if ( strcmp(argv[idx], "all") == 0 )
            {
                gsAppData.tProps |= SMS_APP_CYCLER_PROP_NO_START_WAIT;
                gsAppData.tProps |= SMS_APP_CYCLER_PROP_NO_STOP_WAIT;
            }
            else
            {
                puts( "Invalid argument to -nowait!" );
                return EXIT_FAILURE;
            }
        }
        else if ( strcmp(argv[idx], "-stoponerror" ) == 0)
        {
            gsAppData.tProps |= SMS_APP_CYCLER_PROP_STOP_ON_ERROR;
        }
        else if ( strcmp(argv[idx], "-cfgpath" ) == 0)
        {
            ++idx;

            if ( idx >= argc )
            {
                puts( "You're missing an argument to -cfgpath!" );
                return EXIT_FAILURE;
            }

            // Reassign SMS Configuration directory
            gpacDefaultSmsCfgPath = argv[idx];
        }
        else if ( strcmp(argv[idx], "-clipath" ) == 0)
        {
            ++idx;

            if ( idx >= argc )
            {
                puts( "You're missing an argument to -clipath!" );
                return EXIT_FAILURE;
            }

            // Reassign CLI Resources directory
            gpacDefaultCliResourcesDir = argv[idx];

        }
        else if ( strcmp(argv[idx], "-tmppath" ) == 0)
        {
            ++idx;

            if ( idx >= argc )
            {
                puts( "You're missing an argument to -tmppath!" );
                return EXIT_FAILURE;
            }

            // Reassign temp directory
            gpacTempDirPath = argv[idx];
        }
        else if ( strcmp(argv[idx], "-startall" ) == 0)
        {
            if (gsAppData.un32MaxCycles > 0 ||
                0 != (gsAppData.tProps & SMS_APP_CYCLER_PROP_CYCLE_FOREVER))
            {
                puts( "ERROR: -startall can not be used with -cycles or -contcycle!" );
                return EXIT_FAILURE;
            }

            gsAppData.tProps |= SMS_APP_CYCLER_PROP_START_ALL;
        }
#ifdef SUPPORT_CUNIT
        else if ( strcmp(argv[idx], "-cunit") == 0 )
        {
            gsAppData.tProps |= SMS_APP_CYCLER_PROP_RUN_CUNIT;

            ++idx;

            if ( idx >= argc )
            {
                gsAppData.tCUnitSuite = SMS_CUNIT_TEST_SUITE_FEATURE;
            }
            else if ( strcmp(argv[idx], "feature") == 0 )
            {
                gsAppData.tCUnitSuite = SMS_CUNIT_TEST_SUITE_FEATURE;
            }
            else if ( strcmp(argv[idx], "all") == 0 )
            {
                gsAppData.tCUnitSuite = SMS_CUNIT_TEST_SUITE_ALL;
            }
            else if ( strcmp(argv[idx], "basic") == 0 )
            {
                gsAppData.tCUnitSuite = SMS_CUNIT_TEST_SUITE_SMS_BASICS;
            }
            else if ( strcmp(argv[idx], "core") == 0 )
            {
                gsAppData.tCUnitSuite = SMS_CUNIT_TEST_SUITE_SMS_CORE;
            }
            else if ( strcmp(argv[idx], "ds") == 0 )
            {
                gsAppData.tCUnitSuite = SMS_CUNIT_TEST_SUITE_SMS_DS;
            }
            else if ( strcmp(argv[idx], "osal") == 0 )
            {
                gsAppData.tCUnitSuite = SMS_CUNIT_TEST_SUITE_OSAL;
            }
            else
            {
                puts( "ERROR: Invalid CUnit suite name!" );
                return EXIT_FAILURE;
            }
        }
#endif
#if OSAL_DEBUG == 1
        else if ( strcmp(argv[idx], "-comtest" ) == 0)
        {
            gsAppData.tProps |= SMS_APP_CYCLER_PROP_COM_TEST;
        }
        else if ( strcmp(argv[idx], "-priority") == 0 )
        {
            N32 n32Priority;

            ++idx;

            if ( idx >= argc )
            {
                puts( "You're missing an argument to -priority!" );
                return EXIT_FAILURE;
            }

            n32Priority = atoi( argv[idx] );

            if ( ( n32Priority > 3) ||
                 ( n32Priority < 1) )
            {
                puts( "Priority must be between 1 and 3!" );
                return EXIT_FAILURE;
            }

            gsAppData.eAppTaskPriority = (OSAL_TASK_PRIORITY_ENUM)n32Priority;
        }
#endif
        else
        {
            puts( "Usage: SmsCliApp_hb_sxi_uart "
                    "[-help] [-port xxx] "
                    "[-cfgpath xxx] [-clipath xxx] [-tmppath xxx] "
                    "[-cycles yyy ] [-contcycle] [-comtest] [-statall] "
#ifdef SUPPORT_CUNIT
                    "[-cunit [xxx]]"
#endif
                    "\n");
            puts( "-help         Prints this help message." );
            puts( "-port xxx     Selects port XXX for communications.\n"
                  "              Must be a string similar to 'com1:' or '/dev/ttyS0'.");
            printf(
                  "-cfgpath      Select SMS Configuration path [%s].\n",
                  gpacDefaultSmsCfgPath);
            printf(
                  "-clipath      Select SMS CLI resources path[%s]\n",
                  gpacDefaultCliResourcesDir);
            printf(
                  "-tmppath      Select TEMP path [%s].\n",
                  gpacTempDirPath);
            puts( "-cycles yyy   Limits the number of cycles to yyy, a positive integer." );
            puts( "-contcycle    Cycles continuously until the program is terminated." );
            puts( "-random       Random cycler test." );
            puts( "-optime mmm     SMS operational time (between start and stop) in milliseconds." );
            puts( "-nowait [start] [stop] [all]   Disables wait for READY or(and) STOPPED states." );
            puts( "-stoponerror  The cycler is stopped immediately if any error occurs." );
            puts( "-startall     Starts decoder and all data services, but does not cycle." );

#if OSAL_DEBUG == 1
            puts( "-comtest      Runs a communications check in the background." );
            puts( "-priority     Runs the SMS app at a given priority; 1=low, 2=medium, 3=high." );
#endif

#ifdef SUPPORT_CUNIT
            puts( "-cunit [xxx]  Executes CUnit suite. If suite name is omitted,"
                            "\"feature\" suite is assumed.\n\t"
                            "      Possible suite names are "
                            "\"feature\", \"all\", \"osal\", \"basic\", \"core\", \"ds\"" );
#endif
            puts("");
            return EXIT_FAILURE;
        } // if (strcmp(argv[idx], "-port") == 0)
    }

    // Setting window title for CLITEST support
#ifdef OS_WIN32
    system("title SMS CLI");
#endif

    bOsalStarted = OSAL.bStart( &gsStartHandlers );
    if( bOsalStarted == TRUE )
    {
        iRetval = EXIT_SUCCESS;
    }

    return iRetval;
}
