////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfd_receiver_config.h
///
/// @brief	Preprocessor definitions for configuration of the RFD Receiver.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_RECEIVER_CONFIG_H
#define RFD_RECEIVER_CONFIG_H

#include "rfd_platform_port.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

///////////////////////////////////////////////////////////////////////////////
// RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
// Enable full power loss resilience. No loss of data  on abrupt power-down
// (or abrupt system reset or other abrupt process exit).
// For example, no loss of collected block data or partial decode solution.
//
// With RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE disabled (not defined),
// abrupt power-down may result in loss of data and the RFD file
// collection/reconstruction process will be restarted.
// RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE disabled seletion is intended for
// platforms for which a graceful shut-down of the platoform and RFDR Client
// is guaranteed with acceptibly high probability.
#if 1
#define RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
#endif

///////////////////////////////////////////////////////////////////////////////
// RFDR_DISABLE_DIRECTORY_AND_FILE_SYNC
// Define this macro to DISABLE RFD Receiver sync of directories and files.
// On a platform for which RFD_FileSync() and/or RFD_DirectorySync() functions
// implement syncing (not just stubs), RFDR_DISABLE_DIRECTORY_AND_FILE_SYNC
// can be defined to explicitly disable 'syncing' for the RFD Receiver.
//
// CAUTION: Directory/File Sync'ing (forcing the commit of dir and file
// data immediately to Disk/Flash media) is critical to ensure power-loss resilient
// operation of the RFD Receiver. The developer should disable syncing ONLY
// for specific platforms or RFD applications for which directories and files
// are guaranteed to be sync'ed to disk/flash before any normal system
// shut-down/power-down and for which abrubt or unexpected power downs or
// system lockups/crashes are not possible.
#if 0
#define RFDR_DISABLE_DIRECTORY_AND_FILE_SYNC
#endif

///////////////////////////////////////////////////////////////////////////////
// RFDR_DISABLE_CLEANUP_FS_CORRUPTED_DIR_ENTRIES
// Defining the macro DISABLES the following functionality:
// On RFDR startup, for RFDR related directories, try to cleanup anomalous 
// corrupted file system directory entries. Such corruption may be caused for example
// on some platforms by sudden power-loss with file system data not stored to NVM 
// in consistent state and not successfully corrected by the file system after restarting.
// Such corrupted/invalid files may show as directory entries, but opening the file or
// trying to get attributes of the file may fail. Trying to delete these files
// through regular file system calls may also fail. Typical RFDR application 
// storage initialization procedures including file validation checking/restore/cleanup
// in presence of these types of files may fail to successfully restore state or cleanup 
// to initial state as expected. The "cleanup of invalid file system directory entries"
// functionality tries to scan for and cleanup (delete) any such invalid files, before
// continuing with regular RFDR initialization and storage validation checking.
#if 0
#define RFDR_DISABLE_CLEANUP_FS_CORRUPTED_DIR_ENTRIES
#endif

///////////////////////////////////////////////////////////////////////////////
// RFDR_AVAILABLE_DECODE_MEMORY_BYTES
// The memory available for decoding operation, allocated from the Heap.
#define RFDR_MAX_DECODE_MEMORY_BYTES (1024*1024) // 1 Megabyte

///////////////////////////////////////////////////////////////////////////////
// ENABLE_ACCUMULATED_DECODER_TEST_STATS
// For collecting and printing decoder statistics accumulated over multiple
// processed files.
#if 1
#define ENABLE_ACCUMULATED_DECODER_TEST_STATS
#endif

///////////////////////////////////////////////////////////////////////////////
// RFDR_NUM_CLIENTS
// Specifies the Number of Clients supported by the RFD Receiver.
#define RFDR_NUM_CLIENTS 10

///////////////////////////////////////////////////////////////////////////////
// RFDR Client Directories Base Path Name,
extern char GacBasePathName[RFD_MAX_PATH_LEN];
#define RFDR_BASE_PATH      TEXT((const char *)&GacBasePathName[0])

///////////////////////////////////////////////////////////////////////////////
// File Consumer directories Base Path Name,
extern char GacConsumerBasePathName[RFD_MAX_PATH_LEN];
#define RFD_FILE_CONSUMER_BASE_PATH    TEXT((const char *)&GacConsumerBasePathName[0])

///////////////////////////////////////////////////////////////////////////////
// Message Configuration parameters for each supported Client
///////////////////////////////////////////////////////////////////////////////
#define RFDR_CLIENT_0_ID   0
#define RFDR_CLIENT_0_MSG_PROTOCOL_VERSION_NUM			1    // = 1 for Fuel Prices
#define RFDR_CLIENT_0_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	3    // = 3 for Fuel Prices
#define RFDR_CLIENT_0_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	4    // = 4 for Fuel Prices (could functionally be 3, but spec. states 4)
#define RFDR_CLIENT_0_FILE_SET_MSG_CAROUSEL_ID			3	 // = 3 for Fuel Prices
#define RFDR_CLIENT_0_MAX_BLOCK_LEN_BYTES				8192 // = 8 KB for Fuel Prices
#define RFDR_CLIENT_0_MAX_FILE_LEN_BYTES 				(16*1024*1024) // = 16 MB for Fuel Prices
#define RFD_CLIENT_0_NAME TEXT("Fuel")
#define RFDR_CLIENT_0_DIR_NAME   TEXT("fuel")
#define RFD_FILE_CONSUMER_0_DIR_NAME   TEXT("fuel")

#define RFDR_CLIENT_1_ID   1
#define RFDR_CLIENT_1_MSG_PROTOCOL_VERSION_NUM			1    // = 1 for Movies
#define RFDR_CLIENT_1_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	3    // = 3 for Movies
#define RFDR_CLIENT_1_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	4    // = 4 for Movies
#define RFDR_CLIENT_1_FILE_SET_MSG_CAROUSEL_ID			3	 // = 3 for Movies
#define RFDR_CLIENT_1_MAX_BLOCK_LEN_BYTES				5120 // = for Movies
#define RFDR_CLIENT_1_MAX_FILE_LEN_BYTES 				(4*1024*1024) // = 4 MB for Movies, per Leslie email on 8/23/10
#define RFD_CLIENT_1_NAME TEXT("Movies")
#define RFDR_CLIENT_1_DIR_NAME   TEXT("movies")
#define RFD_FILE_CONSUMER_1_DIR_NAME   TEXT("movies")

#define RFDR_CLIENT_2_ID   2
#define RFDR_CLIENT_2_MSG_PROTOCOL_VERSION_NUM			1    // = 1 for Weather
#define RFDR_CLIENT_2_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	3    // = 3 for Weather
#define RFDR_CLIENT_2_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	4    // = 4 for Weather
#define RFDR_CLIENT_2_FILE_SET_MSG_CAROUSEL_ID			3	 // = 3 for Weather
#define RFDR_CLIENT_2_MAX_BLOCK_LEN_BYTES				5120 // = for Weather
#define RFDR_CLIENT_2_MAX_FILE_LEN_BYTES 				(128*1024) // = 128K for weather.
#define RFD_CLIENT_2_NAME TEXT("TabWeather")
#define RFDR_CLIENT_2_DIR_NAME   TEXT("weather")
#define RFD_FILE_CONSUMER_2_DIR_NAME   TEXT("weather")

#define RFDR_CLIENT_3_ID   3
#define RFDR_CLIENT_3_MSG_PROTOCOL_VERSION_NUM			1    // = 1 for Safety Camera
#define RFDR_CLIENT_3_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	3    // = 3 for Safety Camera
#define RFDR_CLIENT_3_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	4    // = 4 for Safety Camera
#define RFDR_CLIENT_3_FILE_SET_MSG_CAROUSEL_ID			2	 // = 2 for Safety Camera
#define RFDR_CLIENT_3_MAX_BLOCK_LEN_BYTES				5120 // TODO value is TBD in spec
#define RFDR_CLIENT_3_MAX_FILE_LEN_BYTES 				(4*1024*1024) // TODO value is TBD in spec
#define RFD_CLIENT_3_NAME TEXT("SafeCamSTU")
#define RFDR_CLIENT_3_DIR_NAME   TEXT("safecam_stu")
#define RFD_FILE_CONSUMER_3_DIR_NAME   TEXT("safecam_stu")

#define RFDR_CLIENT_4_ID   4
#define RFDR_CLIENT_4_MSG_PROTOCOL_VERSION_NUM			1    // = 1 for Safety Camera
#define RFDR_CLIENT_4_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	3    // = 3 for Safety Camera
#define RFDR_CLIENT_4_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	4    // = 4 for Safety Camera
#define RFDR_CLIENT_4_FILE_SET_MSG_CAROUSEL_ID			2	 // = 2 for Safety Camera
#define RFDR_CLIENT_4_MAX_BLOCK_LEN_BYTES				5120 // TODO value is TBD in spec
#define RFDR_CLIENT_4_MAX_FILE_LEN_BYTES 				(4*1024*1024) // TODO value is TBD in spec
#define RFD_CLIENT_4_NAME TEXT("SafeCamLTU")
#define RFDR_CLIENT_4_DIR_NAME   TEXT("safecam_ltu")
#define RFD_FILE_CONSUMER_4_DIR_NAME   TEXT("safecam_ltu")

//TODO: check
#define RFDR_CLIENT_5_ID   5
#define RFDR_CLIENT_5_MSG_PROTOCOL_VERSION_NUM			1    // = 1 for WSAlerts
#define RFDR_CLIENT_5_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	3    // = 3 for WSAlerts
#define RFDR_CLIENT_5_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	4    // = 4 for WSAlerts
#define RFDR_CLIENT_5_FILE_SET_MSG_CAROUSEL_ID			3	 // = 3 for WSAlerts
#define RFDR_CLIENT_5_MAX_BLOCK_LEN_BYTES				5120 // = for WSAlerts
#define RFDR_CLIENT_5_MAX_FILE_LEN_BYTES 				(4*1024*1024)
#define RFD_CLIENT_5_NAME TEXT("WSAlerts")
#define RFDR_CLIENT_5_DIR_NAME   TEXT("ws_alerts")
#define RFD_FILE_CONSUMER_5_DIR_NAME   TEXT("ws_alerts")

#define RFDR_CLIENT_6_ID   6
#define RFDR_CLIENT_6_MSG_PROTOCOL_VERSION_NUM			1    // = 1 for EV Charging
#define RFDR_CLIENT_6_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	3    // = 3 for EV Charging
#define RFDR_CLIENT_6_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	4    // = 4 for EV Charging (could functionally be 3, but spec. states 4)
#define RFDR_CLIENT_6_FILE_SET_MSG_CAROUSEL_ID			3	 // = 3 for EV Charging
#define RFDR_CLIENT_6_MAX_BLOCK_LEN_BYTES				8192 // = 8 KB for EV Charging
#define RFDR_CLIENT_6_MAX_FILE_LEN_BYTES 				(16*1024*1024) // = 16 MB for EV Charging
#define RFD_CLIENT_6_NAME TEXT("EV")
#define RFDR_CLIENT_6_DIR_NAME   TEXT("evcharging")
#define RFD_FILE_CONSUMER_6_DIR_NAME   TEXT("evcharging")

//  Non-nav MAPs Client 1
#define RFDR_CLIENT_7_ID   7
#define RFDR_CLIENT_7_MSG_PROTOCOL_VERSION_NUM          1    // = 1 for Maps
#define RFDR_CLIENT_7_MSG_CAROUSEL_ID_FIELD_SIZE_BITS   4    // = 4 for Maps
#define RFDR_CLIENT_7_FILE_SET_MSG_CAR_FIELD_SIZE_BITS  4    // = 4 for Maps
#define RFDR_CLIENT_7_FILE_SET_MSG_CAROUSEL_ID          0    // = 0 for Maps
#define RFDR_CLIENT_7_MAX_BLOCK_LEN_BYTES               5120 // = 5 KB for Maps
#define RFDR_CLIENT_7_MAX_FILE_LEN_BYTES                (16*1024*1024) // = 16 MB for Maps
#define RFD_CLIENT_7_NAME   TEXT("Maps1")
#define RFDR_CLIENT_7_DIR_NAME   TEXT("maps1")
#define RFD_FILE_CONSUMER_7_DIR_NAME   TEXT("maps1")

//  Non-nav MAPs Client 2
#define RFDR_CLIENT_8_ID   8
#define RFDR_CLIENT_8_MSG_PROTOCOL_VERSION_NUM			RFDR_CLIENT_7_MSG_PROTOCOL_VERSION_NUM
#define RFDR_CLIENT_8_MSG_CAROUSEL_ID_FIELD_SIZE_BITS	RFDR_CLIENT_7_MSG_CAROUSEL_ID_FIELD_SIZE_BITS
#define RFDR_CLIENT_8_FILE_SET_MSG_CAR_FIELD_SIZE_BITS	RFDR_CLIENT_7_FILE_SET_MSG_CAR_FIELD_SIZE_BITS
#define RFDR_CLIENT_8_FILE_SET_MSG_CAROUSEL_ID			RFDR_CLIENT_7_FILE_SET_MSG_CAROUSEL_ID
#define RFDR_CLIENT_8_MAX_BLOCK_LEN_BYTES				RFDR_CLIENT_7_MAX_BLOCK_LEN_BYTES
#define RFDR_CLIENT_8_MAX_FILE_LEN_BYTES 				RFDR_CLIENT_7_MAX_FILE_LEN_BYTES
#define RFD_CLIENT_8_NAME TEXT("Maps2")
#define RFDR_CLIENT_8_DIR_NAME   TEXT("maps2")
#define RFD_FILE_CONSUMER_8_DIR_NAME   TEXT("maps2")

//  Canadian Fuel
#define RFDR_CLIENT_9_ID   9
#define RFDR_CLIENT_9_MSG_PROTOCOL_VERSION_NUM          1    // = 1 for Fuel Prices
#define RFDR_CLIENT_9_MSG_CAROUSEL_ID_FIELD_SIZE_BITS   3    // = 3 for Fuel Prices
#define RFDR_CLIENT_9_FILE_SET_MSG_CAR_FIELD_SIZE_BITS  4    // = 4 for Fuel Prices (could functionally be 3, but spec. states 4)
#define RFDR_CLIENT_9_FILE_SET_MSG_CAROUSEL_ID          3    // = 3 for Fuel Prices
#define RFDR_CLIENT_9_MAX_BLOCK_LEN_BYTES               8192 // = 8 KB for Fuel Prices
#define RFDR_CLIENT_9_MAX_FILE_LEN_BYTES                (16*1024*1024) // = 16 MB for Fuel Prices
#define RFD_CLIENT_9_NAME TEXT("CanFuel")
#define RFDR_CLIENT_9_DIR_NAME   TEXT("canfuel")
#define RFD_FILE_CONSUMER_9_DIR_NAME   TEXT("canfuel")


#ifdef __cplusplus
}
#endif

#endif // RFD_CONFIG_H
