////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfd_file_consumer.h
///
/// @brief	Declares the rfd file consumer class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_FILE_CONSUMER_H
#define RFD_FILE_CONSUMER_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include "rfd_config.h"
#include "rfd_receiver.h"

///////////////////////////////////////////////////////////////////////////////
#define RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT TEXT(".dat")
#define RFDR_CONSUMER_FID_TYPE_INFO_FILE_NAME_EXT TEXT(".inf")
#define RFDR_CONSUMER_FID_TYPE_TRANS_IN_PROGRESS_FILE_NAME_EXT TEXT(".tip")
#define RFDR_CONSUMER_FID_TYPE_HANDOVER_IN_PROGRESS_FILE_NAME_EXT TEXT(".hip")

#define RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_BASE_LEN 5 // the file id number, zero extended e.g. 00321, 94233
#define RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_EXT_LEN 4 // includes the '.' e.g. .dat, .inf, .tip, .hip
#define RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_LEN (RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_BASE_LEN + RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_EXT_LEN)

// Name of the file that serves as persistent flag for file set (file collection) reset request.
// It persists an unfinished file set (file collection) reset request across power-cycles.
#define RFD_CONSUMER_FSET_RESET_REQ_FNAME       TEXT("reset.req")

typedef enum {
	RFD_CONSUMER_ITEM_UNDEFINED_FILE,
	RFD_CONSUMER_ITEM_FID_TYPE_DATA_FILE,
	RFD_CONSUMER_ITEM_FID_TYPE_INFO_FILE,
	RFD_CONSUMER_ITEM_FID_TYPE_TIP_FILE,
	RFD_CONSUMER_ITEM_FID_TYPE_HIP_FILE,
	RFD_CONSUMER_ITEM_FSET_RESET_REQUEST_FILE

} RFD_CONSUMER_ITEM_FILE_TYPE;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrOutputApi
///
/// @brief
/// Consumer File Information Structure.
///
/// Contains Application (Consumer) relevant metadata about an RFD output file.
///
/// An instance of this structure is passed to the application provided callback function,
/// consumerFileTransferCallback, for an RFD file transfer operation using
/// RFD_ConsumerTransferNewFileExt().
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {
	/// Size of the file.
	UINT32 fileSize;
	/// Lifetime of the file (RFD metadata Lifetime field)
	INT32 lifeTime;
	/// Name of the file (RFD metadata Name field)
	TCHAR name[RFD_FSET_NAME_MAX_LEN];

} RFD_CONSUMER_FILE_INFO_STRUCT;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrOutputApi
///
/// @brief
/// Function prototype of the Consumer File-Transfer Callback.
///
/// Prototype for the application implemented callback function which is called by
/// RFD_ConsumerTransferNewFileExt() to complete ownership transfer of a new file to the
/// application. The application implements a function compliant to this prototype and passes this
/// function pointer in parameter \e consumerFileTransferCallback of the RFDR API function
/// RFD_ConsumerTransferNewFileExt().
///
/// @param  newFileName   Full path and name string identifying location of the new file being
///                       transferred.
/// @param  fileInfoPtr   Pointer to a file info structure describing the new file.
/// @param  callbackArg   The argument provided by the application to associate with this callback
///                       function. The application passes this argument in parameter
///                       \e consumerFileTransferCallbackArg of the function
///                       RFD_ConsumerTransferNewFileExt(). The application callback function will
///                       typically recast this variable from a void data type to some application
///                       specific data type per application design.
///
////////////////////////////////////////////////////////////////////////////////////////////////////
typedef void (*RFD_CONSUMER_FILE_TRANSFER_CALLBACK)(
    const TCHAR newFileName[],
    RFD_CONSUMER_FILE_INFO_STRUCT * newFileInfo,
    void * callbackArg
    );

///////////////////////////////////////////////////////////////////////////////
// Function Prototypes for API
///////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ConsumerWaitForNewFileEvent(
    RFD_CONSUMER_THREAD_DATA_HANDLE handle,
    INT32 timeoutMilliseconds
    );

RFD_STATUS RFD_ConsumerTransferNewFileExt(
    RFD_CONSUMER_THREAD_DATA_HANDLE handle,
    RFD_CONSUMER_FILE_TRANSFER_CALLBACK consumerFileTransferCallback,
    void *consumerFileTransferCallbackArg
    );

RFD_STATUS RFD_ConsumerTransferNewFile_Deprecated(
    RFD_CONSUMER_THREAD_DATA_HANDLE handle,
    BOOL * isNewFileAvailablePtr,
    TCHAR newFileName[], int newFileNameBufMaxLen,
    RFD_CONSUMER_FILE_INFO_STRUCT * newFileInfo
    );

RFD_STATUS RFD_ConsumerInitializeStorage(RFD_CONSUMER_THREAD_DATA_HANDLE handle);

RFD_STATUS RFD_ConsumerResetFileCollectionRequest(RFD_CONSUMER_THREAD_DATA_HANDLE handle);

///////////////////////////////////////////////////////////////////////////////
// Internal (private) Function Prototypes
////////////////////////////////////////////////////////////////////////////////////////////////////
#if 0
static RFD_STATUS RFD_ReadConsumerFileInfo(
    TCHAR * consumerPathBuf,
    UINT16 fileNamefileId,
    BOOL doDataFileCrossValidation,
    RFD_CONSUMER_FILE_INFO_STRUCT * fileInfoPtr
    );

static RFD_STATUS RFD_ConsumerTransferNewFileBegin(
    RFD_CONSUMER_THREAD_DATA_HANDLE handle,
    BOOL * isNewFileAvailablePtr,
    RFD_CONSUMER_FILE_INFO_STRUCT * newFileInfoPtr,
    UINT16 * newFileNamefileIdPtr,
    BOOL markFileHandoverInProgress
    );

static RFD_STATUS RFD_ConsumerCleanupStorage(
     TCHAR pathBuf[],
     BOOL * foundUnfinishedFileHandoverPtr,
     UINT16 * unfinishedFileHandoverFileIdPtr,
     BOOL * foundFileSetResetRequestFile
     );
#endif

#ifdef __cplusplus
}
#endif

#endif // RFD_RECEIVER_H

