////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfd_dec_xcc2.c
///
/// @brief	rfd decoder extended ecc 2 class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_codec_xcc.h"
#include "rfd_gf4_tables.h"

///////////////////////////////////////////////////////////////////////////////////////////////
#ifdef RFD_XCC1_CODE
	#undef RFD_XCC1_CODE
#endif

#define RFD_XCC2_CODE
#define XCC_BITS_PER_ELEMENT 2
#define XCC_ELEMENT_BIT_MASK 0x03
#define XCC_IDENTITY_ELEM 1
#define XCC_NUM_ELEMENTS (1<<XCC_BITS_PER_ELEMENT)
#define XCC_SCALE_ROW_NUM_BUFS  (XCC_NUM_ELEMENTS - 2)
#define RFD_DECODE_XCC					RFD_DecodeXcc2Code
#define XCC_MULTIPLICATIVE_INVERSE_LUT	xcc2MultiplicativeInverseLut
#define XCC_BYTE_PACKED_MULT_TABLES		xcc2BytePackedMultTables


///////////////////////////////////////////////////////////////////////////////
typedef struct {
	DWORD * coeffStart[XCC_NUM_ELEMENTS]; // (0'th entry not used)
	DWORD * blockStart[XCC_NUM_ELEMENTS]; // (0'th entry not used)
	DWORD * coeffRowBufs[XCC_SCALE_ROW_NUM_BUFS];
	DWORD * blockRowBufs[XCC_SCALE_ROW_NUM_BUFS];
	int iNextAvailableBuf;
	DWORD * srcBlockRow;
	int srcCoeffElement;
	int blkLenDWords;
	DWORD * srcCoeffRow;
	int coeffRowLenDWords;
	int adjustedCoeffRowLenDWords;
	int coeffStartDWordOffset;
	XCC_ROW_INFO_BLOCK_TYPE_ENUM rowType; // src or check

} XCC_ROW_SCALE_INFO;

// xcc2MultiplicativeInverseLut:
// Find the multiplicative inverse of an element (index)
// XCC2 elements are from the GF(2^2) field.
// index x ? = 1
// =====   =   =
//     0 undefined (just a placeholder in the table)
//	   1 x 1 = 1
//	   2 x 3 = 1
//	   3 x 2 = 1
const UCHAR xcc2MultiplicativeInverseLut[XCC_NUM_ELEMENTS] = { 0,   // 0 undefined
															   1,   // 1 x 1 = 1
															   3,	// 2 x 3 = 1
															   2 };	// 3 x 2 = 1

// example: what number when multiplied by value a results in value b
// a x ? = b
// ? = b x 1/a
// where 1/a is the multiplicative inverse of a.

const UCHAR * xcc2BytePackedMultTables[XCC_NUM_ELEMENTS] = { NULL,                   // a x 0 = 0 (no LUT needed)
															 NULL,                   // a x 1 = a (no LUT needed)
															 gf4Times2ByteWideLut,   // a x 2 = (use this LUT)
															 gf4Times3ByteWideLut }; // a x 3 = (use this LUT)

///////////////////////////////////////////////////////////////////////////////
static void _CopyBackScaledRow(XCC_ROW_SCALE_INFO * rowScaleInfo, int srcCoeffElement)
{
	register DWORD * pSrc;
	register DWORD * pDst;
	register int i;
	register int nWordsToCopy;

	// This function is used to copy a generated/scaled row back
	// to the original row. This is done typically to copy a generated
	// 'Identity Element Pivot' row that serves as an implemented Pivot row
	// back to the original row that is part of the transformation matrix.
	//
	// note: caller is responsible for ensuring that the scaled row
	// being copied-back has already been generated.
	//
	//if(scalePivotInfo->coeffStart[srcCoeffElement] == NULL) {
	//	return;
	//}

	////////////////////////////
	// Copy-back the Coefficient Row.
	////////////////////////////

	// writing back to the orig src.
	pDst = rowScaleInfo->srcCoeffRow + rowScaleInfo->coeffStartDWordOffset;
	pSrc = rowScaleInfo->coeffStart[srcCoeffElement];

	if(rowScaleInfo->rowType == XCC_ROW_INFO_CHECK_TYPE_BLOCK) {
		nWordsToCopy = rowScaleInfo->adjustedCoeffRowLenDWords;
	}
	else {
		nWordsToCopy = 1;
	}

	for(i=0;i<nWordsToCopy;i++) {
		*pDst = *pSrc;
		pDst++; pSrc++;
	}

	////////////////////////////
	// Copy-back the Block Row.
	////////////////////////////

	pDst = rowScaleInfo->srcBlockRow;
	pSrc = rowScaleInfo->blockStart[srcCoeffElement];

	nWordsToCopy = rowScaleInfo->blkLenDWords;;
	for(i=0;i<nWordsToCopy;i++) {
		*pDst = *pSrc;
		pDst++; pSrc++;
	}
}

///////////////////////////////////////////////////////////////////////////////
static void _ScaleRow(XCC_ROW_SCALE_INFO * rowScaleInfo, int dstCoeffElement)
{
	register DWORD srcDWord;
	register DWORD scaledDWord;
	register const UCHAR * packedMultTable;
	register DWORD * pScaled;
	register DWORD * pSrc;
#if 0
	register int j;
#endif
	register int i;
	register int nWordsToScale;
	int scalar, srcMultInverse;

	// note: this function should never be called if either the src or dst element
	// is 0; this case is not error checked and can result in bad pointers.

	if(dstCoeffElement == rowScaleInfo->srcCoeffElement) {
		// src and destination elements are already equal, no scaling necessary.
		// Just update the appropriate pointer in the array to point to the
		// actual src data.
		rowScaleInfo->coeffStart[dstCoeffElement] = rowScaleInfo->srcCoeffRow +
													rowScaleInfo->coeffStartDWordOffset;
		rowScaleInfo->blockStart[dstCoeffElement] = rowScaleInfo->srcBlockRow;
		return;
	}

	////////////////////////////
	// Calculate the scalar value, that, when multiplied by the src pivot element value,
	// will give a resultant value equal to the dst pivot element dstCoeffElement.
	// Then setup the multiplication table that implements the multiplication by
	// this calculated scalar value.
	////////////////////////////

	// calculate the multiplicative inverse of the source element.
	srcMultInverse = XCC_MULTIPLICATIVE_INVERSE_LUT[rowScaleInfo->srcCoeffElement];

	if(dstCoeffElement == XCC_IDENTITY_ELEM) {
		// no multiplication table available, or needed, for x 1.
		// scalar = srcMultInverse * 1 = srcMultInverse
		scalar = srcMultInverse;
	}
	else {
		packedMultTable = XCC_BYTE_PACKED_MULT_TABLES[dstCoeffElement];
		scalar = packedMultTable[srcMultInverse];
	}
	// scalar is the value that the src row must be multiplied by such that
	// the resultant for the src pivot position is equal to the
	// dst pivot elemement value ().
	packedMultTable = XCC_BYTE_PACKED_MULT_TABLES[scalar];

	//if(rowScaleInfo->iNextAvailableBuf >= XCC_SCALE_ROW_NUM_BUFS) {
	//	// unexpected error, the fcn was called more times that necessary.
	//	return;
	//}

	////////////////////////////
	// Scale the Block Row.
	////////////////////////////

	pScaled = rowScaleInfo->blockRowBufs[rowScaleInfo->iNextAvailableBuf];
	rowScaleInfo->blockStart[dstCoeffElement] = pScaled;

	pSrc = rowScaleInfo->srcBlockRow;
	nWordsToScale = rowScaleInfo->blkLenDWords;

	for(i=0;i<nWordsToScale;i++) {

		srcDWord = *pSrc++;
		scaledDWord = 0;
#if 0
		for(j=0;j<(sizeof(DWORD)/sizeof(UCHAR));j++) {
			scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << (j*8) ) ;
			srcDWord = srcDWord >> 8;
		}
#else // unroll for speed
		scaledDWord =  (((UINT32) packedMultTable[srcDWord & 0xff])      ) ;
		srcDWord = srcDWord >> 8;

		scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << 8 ) ;
		srcDWord = srcDWord >> 8;

		scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << 16 ) ;
		srcDWord = srcDWord >> 8;

		scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << 24 ) ;
#endif
		*pScaled++ = scaledDWord;
	}

	////////////////////////////
	// Scale the Coefficient Row.
	////////////////////////////

	pScaled = rowScaleInfo->coeffRowBufs[rowScaleInfo->iNextAvailableBuf] +
			  rowScaleInfo->coeffStartDWordOffset;
	rowScaleInfo->coeffStart[dstCoeffElement] = pScaled;
	rowScaleInfo->iNextAvailableBuf++;

	pSrc = rowScaleInfo->srcCoeffRow + rowScaleInfo->coeffStartDWordOffset;

	if(rowScaleInfo->rowType == XCC_ROW_INFO_CHECK_TYPE_BLOCK) {
		nWordsToScale = rowScaleInfo->adjustedCoeffRowLenDWords;
	}
	else {
		// else the  row is a Source-Type row, which has all
		// zero coefficients except the pivot postion coefficient.
		// So no need to scale any other coefficient postions besides
		// the pivot position.
		nWordsToScale = 1;
	}

	for(i=0;i<nWordsToScale;i++) {

		srcDWord = *pSrc++;
		scaledDWord = 0;

#if 0
		for(j=0;j<(sizeof(DWORD)/sizeof(UCHAR));j++) {
			scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << (j*8) ) ;
			srcDWord = srcDWord >> 8;
		}
#else // unroll for speed
		scaledDWord =  (((UINT32) packedMultTable[srcDWord & 0xff])      ) ;
		srcDWord = srcDWord >> 8;

		scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << 8 ) ;
		srcDWord = srcDWord >> 8;

		scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << 16 ) ;
		srcDWord = srcDWord >> 8;

		scaledDWord |= (((UINT32) packedMultTable[srcDWord & 0xff]) << 24 ) ;
#endif
		*pScaled++ = scaledDWord;
	}

	return;
}

///////////////////////////////////////////////////////////////////////////////
static void _CleanupRowScaleBufs(void * rowScaleInfoVoidType)
{
	XCC_ROW_SCALE_INFO * rowScaleInfo = (XCC_ROW_SCALE_INFO *) rowScaleInfoVoidType;
	int i;

	if(rowScaleInfo == NULL) {
		return;
	}

	for(i=0;i<(XCC_SCALE_ROW_NUM_BUFS);i++) {

		if(rowScaleInfo->coeffRowBufs[i] != NULL) {
			RFD_FREE(rowScaleInfo->coeffRowBufs[i]);
			rowScaleInfo->coeffRowBufs[i] = NULL;
		}

		if(rowScaleInfo->blockRowBufs[i] != NULL) {
			RFD_FREE(rowScaleInfo->blockRowBufs[i]);
			rowScaleInfo->blockRowBufs[i] = NULL;
		}
	}

	RFD_FREE(rowScaleInfo);

	return;
}

///////////////////////////////////////////////////////////////////////////////
static XCC_ROW_SCALE_INFO * _AllocateRowScaleBufs(XCC_DEC_DIMENSION_INFO * dimInfo)
{
	XCC_ROW_SCALE_INFO * rowScaleInfo = NULL;
	int i;

	rowScaleInfo = RFD_MALLOC(sizeof(XCC_ROW_SCALE_INFO));
	if(rowScaleInfo == NULL) {
		return NULL;
	}

	for(i=0;i<(XCC_SCALE_ROW_NUM_BUFS);i++) {
		rowScaleInfo->coeffRowBufs[i] = NULL;
		rowScaleInfo->blockRowBufs[i] = NULL;
	}

	for(i=0;i<(XCC_SCALE_ROW_NUM_BUFS);i++) {

		rowScaleInfo->coeffRowBufs[i] = RFD_MALLOC(dimInfo->coeffRowLenDWords * sizeof(DWORD));
		if(rowScaleInfo->coeffRowBufs[i] == NULL) {
			_CleanupRowScaleBufs(rowScaleInfo);
			return NULL;
		}

		rowScaleInfo->blockRowBufs[i] = RFD_MALLOC(dimInfo->blkLenDWords * sizeof(DWORD));
		if(rowScaleInfo->blockRowBufs[i] == NULL) {
			_CleanupRowScaleBufs(rowScaleInfo);
			return NULL;
		}
	}

	return rowScaleInfo;
}


/*********************************************************************
*
*       #include rfd_dec_xcc1 code
*
**********************************************************************
*
* The XCC2 decoder code is in an other "C" file, shared with ECC1 code.
* This is done to avoid duplication of code.
*/
#include "rfd_dec_xcc1.c"
