////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_utils.c
///
/// @brief	rfd utils class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"

///////////////////////////////////////////////////////////////////////////////
void RFD_CopyCharacterRangeToString(
            TCHAR * dstStringPtr,
            const TCHAR * srcStringPtr,
            UINT16 srcStartCharOffset,
            UINT16 numChars )
{
    UINT16 i;

    for(i=srcStartCharOffset; i<srcStartCharOffset+numChars; i++) {
        *dstStringPtr++ =  srcStringPtr[i];
    }
    // add string termination char
    *dstStringPtr = '\0';

    return;
}

//////////////////////////////////////////////////////////
int RFD_FindConsoleInputString(int argStartNo, int argEndNo, TCHAR * argv[], const TCHAR findString[])
{
	int i;

	for(i=argStartNo;i<=argEndNo;i++) {
		if(!RFD_STRNCMP(argv[i],findString, RFD_CMDLINE_STR__MAX_OPT_STRING_LEN)) {
			// String found, return argument number.
			return i;
		}
	}

	// string not found.
	return -1;
}

///////////////////////////////////////////////////////////////////////////////
void RFD_ReportError(RFD_ERROR_LEVEL_ENUM errLevel, const TCHAR errInfoString[])
{
	RFD_DPrint(TEXT("%s\n"), errInfoString);


	if(errLevel == RFD_ERR_LEVEL_FATAL) {
		#if 0
		// Could use this funtion abort on fatal error, depending on the platform/os
		// e.g. ExitProcess() or reset device.
		//ExitProcess(1);
		#endif
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsMatchingFileNameExtension(TCHAR fileName[], TCHAR fileNameExtension[])
{
	int fileNameLen, extNameLen;

	// cal and validate the length of the file name.
	fileNameLen = (int) RFD_STRNLEN(fileName, RFD_MAX_PATH_LEN);

	if(fileNameLen == 0) {
		// invalid file name length
		return FALSE;
	}

	// calc length of extension name.
	extNameLen = (int) RFD_STRNLEN(fileNameExtension, RFD_MAX_PATH_LEN);

    if(extNameLen >= fileNameLen) {
        // ext len can't be longer than the file name len.
        return FALSE;
    }

	if(RFD_STRCMP(fileName + fileNameLen - extNameLen, fileNameExtension)) {
        // does not match
		return FALSE;
	}

    // file extension name matches the expected ext name, return match true.
    return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
int RFD_StripFileNameExtension(TCHAR fileName[], TCHAR fileNameExtension[])
{
	int fileNameLen, extNameLen;

	// cal and validate the length of the file name.
	fileNameLen = (int) RFD_STRNLEN(fileName, RFD_MAX_PATH_LEN);

	if(fileNameLen == 0) {
		// invalid file name length
		return -1;
	}

	// calc length of extension name.
	extNameLen = (int) RFD_STRNLEN(fileNameExtension, RFD_MAX_PATH_LEN);

    if(extNameLen >= fileNameLen) {
        // ext len can't be longer than the file name len.
        return -1;
    }

    // verify fileName has the expected filename extension
	if(RFD_STRCMP(fileName + fileNameLen - extNameLen, fileNameExtension)) {
        // does not match
		return -1;
	}

    // strip off the extension by inserting string termination
    // character where extension begins.
    fileName[fileNameLen - extNameLen] = '\0';

    // file extension name was stripped, return success.
    return 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void RFD_PrintFileSetInfo( RFD_FSET_MDATA_STRUCT * fileSetInfo )
{
    int i;
    RFD_FILE_INFO_STRUCT * fileInfo;

    if(fileSetInfo->numFiles == 0) {
        RFD_DPrint( TEXT("FSetInfo: numFiles 0 (reset state)\n"));
        return;
    }

    if(fileSetInfo->numFiles > RFD_FSET_MAX_NUM_FILES) {
        RFD_DPrint( TEXT("FSetInfo: error, numFiles %d > max %d\n"),
            fileSetInfo->numFiles, RFD_FSET_MAX_NUM_FILES);
        return;
    }

	RFD_DPrint( TEXT("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n"));
	RFD_DPrint( TEXT("FSetInfo: Name           : %s\n"), fileSetInfo->name);
	RFD_DPrint( TEXT("FSetInfo: LifeTime       : %d\n"), fileSetInfo->lifeTime);
	RFD_DPrint( TEXT("FSetInfo: CodeType       : %d\n"), fileSetInfo->codeType);
	RFD_DPrint( TEXT("FSetInfo: CompType       : %d\n"), fileSetInfo->compType);
	RFD_DPrint( TEXT("FSetInfo: BlkCarouselId  : %d\n"), fileSetInfo->blockMsgCarouselId);
	RFD_DPrint( TEXT("FSetInfo: ExtLen (bytes) : %d\n"), fileSetInfo->extensionLenBytes);
	RFD_DPrint( TEXT("FSetInfo: BlkMsg DMI     : %d\n"), fileSetInfo->blockMsgDmi);
	RFD_DPrint( TEXT("FSetInfo: Num (Sub)Files : %d\n"), fileSetInfo->numFiles);

    for(i=0;i<fileSetInfo->numFiles;i++) {

        fileInfo = &fileSetInfo->fileInfo[i];

	    RFD_DPrint( TEXT("  ===\n"));
        RFD_DPrint( TEXT("  FileInfo: Index:           : %d\n"), i);
	    RFD_DPrint( TEXT("  FileInfo: FileId           : %d\n"), fileInfo->id);
	    RFD_DPrint( TEXT("  FileInfo: FileSize (bytes) : %d\n"), fileInfo->size);
	    RFD_DPrint( TEXT("  FileInfo: CRC              : 0x%04x\n"), fileInfo->crc);
	    RFD_DPrint( TEXT("  FileInfo: BlkLen (bytes)   : %d\n"), fileInfo->blkLenBytes);
	    RFD_DPrint( TEXT("  ===\n"));
    }

	RFD_DPrint( TEXT("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n"));

    return;
}