/////////////////////
/// @file	rfd_common\rfd_platform_port.c
///
/// @brief	rfd platform port class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2012 Sirius XM Satellite Radio. All rights reserved.
/////////////////////

#include "osal.h"
#include "rfd.h"
#include "rfd_config.h"

#if defined(RFD_OSAL)
// Private include file for local prototypes and data structures
#include "_rfd_platform_port.h"
#endif

// doxygen group start symbol, don't remove.
/// @addtogroup RfdPlatformAdaptation RFD Platform Porting Layer
/// @{

#if !defined(RFD_ALT_SINGLE_THREAD_MODE_ENABLE)

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Set the priority value for the specified thread.
///
/// The Priority value can be one of the following values:
///
/// RFD_THREAD_PRIORITY_HIGHEST         - higher than _ABOVE_NORMAL
///
/// RFD_THREAD_PRIORITY_ABOVE_NORMAL    - above _NORMAL
///
/// RFD_THREAD_PRIORITY_NORMAL          - normal priority
///
/// RFD_THREAD_PRIORITY_BELOW_NORMAL    - below _NORMAL
///
/// RFD_THREAD_PRIORITY_LOWEST          - lower than _BELOW_NORMAL.
///
/// @param  threadHandle    Handle of the thread whose priority value is to be set.
/// @param  nPriority       The priority.
///
/// @return true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_SetThreadPriority(RFD_THREAD_HANDLE threadHandle, int nPriority)
{
#if defined(RFD_WIN32)
	HANDLE win32threadHandle;
	int win32nPriority;
	BOOL isOk;

	win32threadHandle = threadHandle;
	win32nPriority = nPriority;
	isOk = SetThreadPriority(win32threadHandle, win32nPriority);
	if(isOk) {
		return TRUE;
	}
	else {
		// error
		return FALSE;
	}
#elif defined(RFD_OSAL)
    OSAL_RETURN_CODE_ENUM eReturnCode;
    PORT_OSAL_THREAD_STRUCT *psThread =
        (PORT_OSAL_THREAD_STRUCT *)threadHandle;

    if (psThread == NULL)
    {
        return FALSE;
    }

    eReturnCode = OSAL.eTaskChangePriority(
        psThread->hTask, (OSAL_TASK_PRIORITY_ENUM)nPriority);


    if (eReturnCode == OSAL_SUCCESS)
    {
        return TRUE;
    }
    else
    {
        // error
        return FALSE;
    }

#else
#error undefined option
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create thread.
///
/// The thread is created with a thread priority of RFD_THREAD_PRIORITY_NORMAL. Use the
/// RFD_SetThreadPriority() function to set the priority value of a thread. Indication that the
/// thread has terminated may be determined by a call to the function RFD_WaitForThreadExit().
/// The thread object remains in the system until the thread has terminated and all handles to it
/// have been closed through a call to RFD_CloseThread(). The typical thread completion sequence
/// is to wait for the thread to terminate by calling RFD_WaitForThreadExit(); then calling
/// RFD_CloseThread() to release resources corresponding to the thread.
///
/// @param  stackSize       Size of the stack in bytes to reserve for use by the thread.
/// @param  startAddress    Pointer to the application-defined function to be executed by the
///                         thread and represents the starting address of the thread.
/// @param  parameterPtr    Pointer to a variable to be passed to the thread.
/// @param  threadIdPtr     (output) Pointer to a variable that receives the thread identifier.
///                         If this parameter is NULL, the thread identifier is not returned.
///
/// @return handle to the created thread. If NULL, thread creation failed.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_THREAD_HANDLE RFD_CreateThread( SIZE_T stackSize,
									const TCHAR * threadName,
									RFD_THREAD_START_ROUTINE_PTR startAddress,
									VOID * parameterPtr,
									DWORD * threadIdPtr )
{
	RFD_THREAD_HANDLE threadHandle;
// platform/OS dependent
#if defined(RFD_WIN32)
	HANDLE win32threadHandle;
	win32threadHandle = CreateThread( NULL,           // default security attributes
									  stackSize,      // stack size  (0 for default stack size)
									  startAddress,   // thread function name
									  parameterPtr,   // argument to thread function
									  0,              // use default creation flags
									  threadIdPtr);   // returns the thread identifier
	// For clarity, explicitly check return handle.
	// Other platforms may have a different error return value
	// other than NULL.
	if(win32threadHandle == NULL) {
		threadHandle = NULL;
	}
	else {
		threadHandle = win32threadHandle;
	}
#elif defined(RFD_OSAL)
    PORT_OSAL_THREAD_STRUCT *psThread;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    static UN32 un32TaskInstanceID = 0;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];

    // Create our local structure that is used for managing an RFD thread
    sprintf(&acName[0], RFDLIB_NAME":ThreadStruct:%u", un32TaskInstanceID);
    psThread = (PORT_OSAL_THREAD_STRUCT *)
        OSAL.pvMemoryAllocate(&acName[0], sizeof(PORT_OSAL_THREAD_STRUCT), TRUE);

    if (psThread == NULL)
    {
        // ERROR
        return NULL;
    }

    // initilize our task management structure
    psThread->dwTaskHandler = startAddress;
    psThread->pvTaskArg = parameterPtr;

    // Consturct our thread exit semaphore
    sprintf(&acName[0], RFDLIB_NAME":TaskSem:%u", un32TaskInstanceID);
    eReturnCode = OSAL.eSemCreate(
        &psThread->hSemThreadExit, &acName[0], 0, 1,
        OSAL_SEM_OPTION_NONE);

    if (eReturnCode != OSAL_SUCCESS)
    {
        // ERROR!
        RFD_CloseThread(psThread);
        return NULL;
    }

    // Construct name for the task -- make sure its unique
    sprintf(&acName[0], RFDLIB_NAME": %s:%u", threadName, un32TaskInstanceID);

    eReturnCode = OSAL.eTaskCreate(
        &psThread->hTask, &acName[0], n32LocalTaskHandler, psThread,
        stackSize,  OSAL_TASK_PRIORITY_LOW, OSAL_TASK_OPTION_NONE);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // set thread ID pointer
        *threadIdPtr = un32TaskInstanceID;

        // increment task instance id
        un32TaskInstanceID++;

        // set the handle to our task
        threadHandle = psThread;
    }
    else
    {
        // Error, return NULL for our thread handle
        RFD_CloseThread(psThread);
        threadHandle = NULL;
    }
#else
#error undefined option
#endif
	return threadHandle;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Close the specified thread. See RFD_CreateThread() for additional info.
///
/// @param	threadHandle	Handle of the thread.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseThread(RFD_THREAD_HANDLE threadHandle)
{
#if defined(RFD_WIN32)
	HANDLE win32threadHandle;
	BOOL isOk;

	win32threadHandle = threadHandle;
	isOk = CloseHandle(win32threadHandle);
	if(isOk) {
		return TRUE;
	}
	else {
		// error
		return FALSE;
	}
#elif defined(RFD_OSAL)
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = FALSE;
    PORT_OSAL_THREAD_STRUCT *psThread =
        (PORT_OSAL_THREAD_STRUCT *)threadHandle;

    if (psThread == NULL)
    {
        return FALSE;
    }

    // Delete the task, we don't care about the
    // error code since OSAL may have already deleted the task for us
    if (psThread->hTask != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eTaskDelete(psThread->hTask);
    }

    // Delete the semaphore
    eReturnCode = OSAL.eSemDelete(psThread->hSemThreadExit);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.vMemoryFree(psThread);
        bSuccess = TRUE;
    }


    return bSuccess;
#else
#error undefined option
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Wait for the specified thread to exit (terminate).
///
/// If the thread has not terminated within the number of milliseconds specified by the
/// timeoutMilliseconds parameter, then RFD_WaitForThreadExit() will return with
/// RFD_STATUS_OBJECT_TIMEOUT status. RFD_WaitForThreadExit() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE. See RFD_CreateThread() for additional info.
///
/// @param  threadHandle        Handle of the thread.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if thread has exited normally, RFD_STATUS_OBJECT_TIMEOUT if thread has not
/// termated withing the specified timeout period, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForThreadExit(RFD_THREAD_HANDLE threadHandle, INT32 timeoutMilliseconds)
{
	RFD_STATUS status;

// platform/OS dependent
#if defined(RFD_WIN32)
	DWORD dwWaitResult;
	DWORD dwMilliseconds = timeoutMilliseconds;

	dwWaitResult = WaitForSingleObject(
		threadHandle,		// handle to thread
		dwMilliseconds);	// no time-out interval

	switch (dwWaitResult)
	{
		///////////
		// The event occurred
		///////////
		case WAIT_OBJECT_0:

			status = RFD_STATUS_OK;

		break;

		case WAIT_TIMEOUT:

			status = RFD_STATUS_OBJECT_TIMEOUT;
			break;

		//////////
		// The thread got ownership of an abandoned event
		// Treat this as error.
		//////////
		case WAIT_ABANDONED:

			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Thread Object Abandoned Error\n"));
			status = RFD_STATUS_ERROR_OBJECT_ABANDONED;

		break;

		default:

			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Thread Unexpected wait error\n"));
			status = RFD_STATUS_ERROR_GENERAL;

			break;
	}
#elif defined(RFD_OSAL)
{
    PORT_OSAL_THREAD_STRUCT *psThread =
      (PORT_OSAL_THREAD_STRUCT *)threadHandle;

    if (psThread == NULL)
    {
        return RFD_STATUS_ERROR_OBJECT_ABANDONED;
    }

    // Wait on the exit semaphore
    // Using RFD_WaitForSemaphore function so that status code is conditioned
    status = RFD_WaitForSemaphore(psThread->hSemThreadExit, timeoutMilliseconds);
}
#else
#error undefined option
#endif

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a mutex object.
///
/// Mutex objects are used to protect a shared resource from simultaneous access by multiple
/// threads. Each thread must wait for ownership of the mutex before it can execute the code that
/// accesses the shared resource. For example, if several threads share access to a database, the
/// threads can use a mutex object to permit only one thread at a time to write to the database.
///
/// A mutex is first created using RFD_CreateMutex(). To obtain ownership of a mutex, a thread
/// calls RFD_AcquireMutex(). If another thread still has ownership, then the RFD_AcquireMutex()
/// blocks (suspends the calling thread) until the other thread has released ownership or until
/// RFD_AcquireMutex() times-out. Once a thread has ownership of the mutex, the shared resource
/// may be safely evaluated and/or modified by the owning thread. The thread may then call
/// RFD_ReleaseMutex() to release ownership of the mutex and thus allow other threads access to
/// the shared resource. If the mutex is no longer in use by any threads (e.g. closing the
/// application or shutting down) the mutex reasources are cleaned up by calling RFD_CloseMutex().
///
/// @return the handle of the created mutex. NULL if creation failed.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_MUTEX_HANDLE RFD_CreateMutex(void)
{
	RFD_MUTEX_HANDLE mutexHandle;
// platform/OS dependent
#if defined(RFD_WIN32)
	HANDLE win32mutexHandle;
	win32mutexHandle = CreateMutex(
							NULL,    // default security attributes
							FALSE,   // initially not owned
							NULL);   // unnamed mutex
	// For clarity, explicitly check return handle.
	// Other platforms may have a different error return value
	// other than NULL.
	if(win32mutexHandle == NULL) {
		mutexHandle = NULL;
	}
	else {
		mutexHandle = win32mutexHandle;
	}
#elif defined(RFD_OSAL)

    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hMutex;
    static UN32 un32MutexInstanceID = 0;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];

    // construct name for mutex
    sprintf(&acName[0], RFDLIB_NAME":Mutex:%u", un32MutexInstanceID);

    // Create a binay semaphore to act as a mutex
    // We could just use RFD_CreateSemaphore if we don't care
    // about the OSAL Name.
    eReturnCode = OSAL.eSemCreate(
        &hMutex, &acName[0], 1, 1, OSAL_SEM_OPTION_NONE);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // increment mutex instance id
        un32MutexInstanceID++;

        // set the handle to our mutex
        mutexHandle = hMutex;

        RFD_ReleaseMutex(mutexHandle);
    }
    else
    {
        // Error, return NULL for our mutex handle
        mutexHandle = NULL;
    }
#else
#error undefined option
#endif
	return mutexHandle;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Close the specified mutex object. See RFD_CreateMutex() for additional info.
///
/// @param	mutexHandle	Handle of the mutex.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseMutex(RFD_MUTEX_HANDLE mutexHandle)
{
#if defined(RFD_WIN32)
	HANDLE win32mutexHandle;
	BOOL isOk;

	win32mutexHandle = mutexHandle;
	isOk = CloseHandle(win32mutexHandle);
	if(isOk) {
		return TRUE;
	}
	else {
		// error
		return FALSE;
	}
#elif defined(RFD_OSAL)

    // Our mutex is just a binary semaphore, use semaphore code.
    return RFD_CloseSemaphore(mutexHandle);
#else
#error undefined option
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Acquire the specified mutex object.
///
/// If the mutex has not been acquired within the number of milliseconds specified by the
/// timeoutMilliseconds parameter, then RFD_AcquireMutex() will return with
/// RFD_STATUS_OBJECT_TIMEOUT status. RFD_AcquireMutex() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE. See RFD_CreateMutex() for additional info.
///
/// @param  mutexHandle         The mutex handle.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if the mutex was sucessfully acquired, RFD_STATUS_OBJECT_TIMEOUT if the mutex
/// was not acquired and a timeout has occurred instead, other RFD_STATUS enumeration value if
/// failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_AcquireMutex(RFD_MUTEX_HANDLE mutexHandle, INT32 timeoutMilliseconds)
{
	RFD_STATUS status;

// platform/OS dependent
#if defined(RFD_WIN32)
	DWORD dwWaitResult;
	DWORD dwMilliseconds = timeoutMilliseconds;

	dwWaitResult = WaitForSingleObject(
		mutexHandle,		// handle to mutex
		dwMilliseconds);	// time-out interval

	switch (dwWaitResult)
	{
		///////////
		// The thread got ownership of the mutex
		///////////
		case WAIT_OBJECT_0:

			//RFD_DPrint( TEXT("Mutex Aquired\n"));
			status = RFD_STATUS_OK;

		break;

		case WAIT_TIMEOUT:

			//RFD_DPrint( TEXT("Mutex Timed-Out\n"));
			status = RFD_STATUS_OBJECT_TIMEOUT;

		//////////
		// The thread got ownership of an abandoned mutex
		// Treat this as error. e.g. shared data may be in indeterminate state if another
		// tread did not properly release the mutex.
		//////////
		case WAIT_ABANDONED:

			//RFD_DPrint( TEXT("Mutex Abandoned Error\n"));
			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Mutex Abandoned Error\n"));
			status = RFD_STATUS_ERROR_OBJECT_ABANDONED;

		break;

		default:

			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Mutex RFD_AcquireMutex() error\n"));
			status = RFD_STATUS_ERROR_GENERAL;

			break;
	}
#elif defined(RFD_OSAL)

    // Our mutex is just a binary semaphore, use semaphore code.
    // RFD_DPrint( TEXT("Acquiring mutext - 0x%p, timeout: %u\n"), mutexHandle, timeoutMilliseconds);
    status = RFD_WaitForSemaphore(mutexHandle, timeoutMilliseconds);
#else
#error undefined option
#endif

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Release the specified mutex object.
///
/// See RFD_CreateMutex() for additional info.
///
/// @param  mutexHandle The mutex handle.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ReleaseMutex(RFD_MUTEX_HANDLE mutexHandle)
{
	RFD_STATUS status;

// platform/OS dependent
#if defined(RFD_WIN32)
	if (ReleaseMutex(mutexHandle))
	{
		//RFD_DPrint( TEXT("Mutex Released\n"));
		status = RFD_STATUS_OK;
	}
	else
	{
		//  error.
		RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Error Releasing Mutex\n"));
		status = RFD_STATUS_ERROR_MUTEX_RELEASE;
	}
#elif defined(RFD_OSAL)

    // Our mutex is just a binary semaphore, use semaphore code.
    // RFD_DPrint( TEXT("Releasing mutext - 0x%p\n"), mutexHandle);
    status = RFD_SignalSemaphore(mutexHandle);
#else
#error undefined option
#endif
	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create an event object.
///
/// An event object is a type of synchronization object. The purpose of an event object is to
/// enable one or multiple threads to wait for a particular event to occur. The waiting threads
/// can be kept suspended until the event is set by another thread. The event is set (to signaled
/// state) by calling RFD_SetEvent(). Threads wait on the event to be signaled by calling
/// RFD_WaitForEvent(). An event in the signaled state is changed to the non-signaled state when
/// a single waiting thread is released. If no threads are waiting, the event object's state
/// remains signaled. If more than one thread is waiting, a waiting thread is selected. Do not
/// assume a particular wait order.
///
/// An event object is created by RFD_CreateEvent(). When created, the event object is initially
/// in the non-signaled state. If the event object is no longer in use by any threads (e.g.
/// closing the application or shutting down) the event object resources are cleaned up by
/// calling RFD_CloseEvent().
///
/// @return the handle of the created event. NULL if creation failed.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_EVENT_HANDLE RFD_CreateEvent(void)
{
	RFD_EVENT_HANDLE eventHandle;
// platform/OS dependent
#if defined(RFD_WIN32)
	HANDLE win32eventHandle;
	win32eventHandle = CreateEvent(
						NULL,   // default security attributes
						FALSE,  // auto-reset event
						FALSE,  // initial state is nonsignaled
						NULL	// unnamed event
						);
	// For clarity, explicitly check return handle.
	// Other platforms may have a different error return value
	// other than NULL.
	if(win32eventHandle == NULL) {
		eventHandle = NULL;
	}
	else {
		eventHandle = win32eventHandle;
	}
#elif defined(RFD_OSAL)
    OSAL_RETURN_CODE_ENUM eReturnCode;
    static UN32 un32EventInstanceID = 0;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];
    OSAL_OBJECT_HDL hMutex;

    // construct name for mutex
    sprintf(&acName[0], RFDLIB_NAME":Event:Sem:%u", un32EventInstanceID);

    // Create our mutex
    eReturnCode = OSAL.eSemCreate(
        &hMutex, &acName[0], 0, 1,
        OSAL_SEM_OPTION_NONE);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // increment event instance id
        un32EventInstanceID++;
        eventHandle = (RFD_EVENT_HANDLE)hMutex;
    }
    else
    {
        // Error, return NULL for our event handle
        eventHandle = NULL;
    }
#else
#error undefined option
#endif
	return eventHandle;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Close the specified event object.
///
/// See RFD_CreateEvent() for additional info.
///
/// @param	eventHandle	Handle of the event.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseEvent(RFD_EVENT_HANDLE eventHandle)
{
#if defined(RFD_WIN32)
	HANDLE win32eventHandle;
	BOOL isOk;

	win32eventHandle = eventHandle;
	isOk = CloseHandle(win32eventHandle);
	if(isOk) {
		return TRUE;
	}
	else {
		// error
		return FALSE;
	}
#elif defined(RFD_OSAL)
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hMutex =
        (OSAL_OBJECT_HDL)eventHandle;

    // Delete the mutex
    eReturnCode = OSAL.eSemDelete(hMutex);

    if (eReturnCode == OSAL_SUCCESS)
    {
        bSuccess = TRUE;
    }

    return bSuccess;
#else
#error undefined option
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Set the specified event object to the signaled state.
///
/// See RFD_CreateEvent() for additional info.
///
/// @param  eventHandle Handle of the event.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_SetEvent(RFD_EVENT_HANDLE eventHandle)
{
// platform/OS dependent
#if defined(RFD_WIN32)
	if(!SetEvent(eventHandle)) {
		RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("RFD_SetEvent() failure\n"));
		return RFD_STATUS_ERROR_GENERAL;
	}

	return RFD_STATUS_OK;
#elif defined(RFD_OSAL)
    RFD_STATUS status = RFD_STATUS_OK;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hMutex =
        (OSAL_OBJECT_HDL)eventHandle;

    eReturnCode = OSAL.eSemGive(hMutex);

    if (eReturnCode != OSAL_SUCCESS)
    {
        status = RFD_STATUS_ERROR_GENERAL;
    }

    RFD_DPrint("Gave Event(0x%p)\n", hMutex);

    return status;
#else
#error undefined option
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Waits for an event and suspends the calling thread for a specified time as long as the event
/// is not signaled.
///
/// If the specified event object is already set to the signaled state, the calling task resets
/// the event to the non-signaled state and continues program execution. If the specified event
/// object is not set to the signaled state, the calling task is suspended until the event object
/// becomes signaled or the timeout time has expired. RFD_WaitForEvent() will wait indefinitely
/// if timeoutMilliseconds = RFD_INFINITE. See RFD_WaitForEvent() for additional info.
///
/// @param  eventHandle         Handle of the event.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if event object was detected as signaled and was reset to non-signaled,
/// RFD_STATUS_OBJECT_TIMEOUT if the event object was not signaled and a timeout has occurred
/// instead, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForEvent(RFD_EVENT_HANDLE eventHandle, INT32 timeoutMilliseconds)
{
	RFD_STATUS status;

// platform/OS dependent
#if defined(RFD_WIN32)
	DWORD dwWaitResult;
	DWORD dwMilliseconds = timeoutMilliseconds;

	dwWaitResult = WaitForSingleObject(
		eventHandle,				// handle to event
		dwMilliseconds);	// no time-out interval

	switch (dwWaitResult)
	{
		///////////
		// The event occurred
		///////////
		case WAIT_OBJECT_0:

			//RFD_DPrint( TEXT("Event Occurred\n"));
			status = RFD_STATUS_OK;

		break;

		case WAIT_TIMEOUT:

			//RFD_DPrint( TEXT("Event Timed-Out\n"));
			status = RFD_STATUS_OBJECT_TIMEOUT;
			break;

		//////////
		// The thread got ownership of an abandoned event
		// Treat this as error.
		//////////
		case WAIT_ABANDONED:

			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Event Object Abandoned Error\n"));
			status = RFD_STATUS_ERROR_OBJECT_ABANDONED;

		break;

		default:

			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Event Unexpected wait error\n"));
			status = RFD_STATUS_ERROR_GENERAL;

			break;
	}
#elif defined(RFD_OSAL)

    // Our event is just a crazy semaphore, use RFD semaphore code
    // since it already sanatizes the RFD status code value
    status = RFD_WaitForSemaphore(
        (RFD_SEMAPHORE_HANDLE)eventHandle, timeoutMilliseconds);
#else
#error undefined option
#endif

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a multi wait event object.
///
/// A multi wait event object holds a collection of event objects. See RFD_WaitForEvent()
/// for a description of event objects. The multi wait event object enables a thread to
/// simultaneosly wait for any of the collection of event objects to be signaled (set to signaled
/// state). The RFD_WaitForOneOfMultiEvents() function is used for this purpose.
///
/// A multi wait event object is created using RFD_CreateMultiWaitEventObject(). The number of
/// events in the collection is specified in the call to this function. Individual event objects
/// are then added to the multi wait event object using RFD_AddEventToMultiWaitEventObject(). The
/// individual event objects are previously created using the event object function
/// RFD_CreateEvent(). An individual event of the multi wait event object is signaled (set to
/// signaled state) using the event object function RFD_SetEvent(). The
/// RFD_WaitForOneOfMultiEvents() function allows a thread to wait for any of the events of the
/// multi wait event object to be signaled. The first event to be detected as signaled is then
/// reset to the non-signaled state and program execution continues for the calling thread. If
/// the multi wait even object is no longer in use by any threads (e.g. closing the application
/// or shutting down) the object resources are cleaned up by calling
/// RFD_DeleteMultiWaitEventObject(). This does not clean up the individual event objects that
/// were added to the multi wait event object. The individual event objects are cleaned up by the
/// application using RFD_CloseEvent() for each individual event object.
///
/// @param  nWaitEventEntries   The number of event objects that the multi wait event object is
///                             to contain.
///
/// @return multi-wait-event structure. NULL if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_MULTI_WAIT_EVENT_STRUCT * RFD_CreateMultiWaitEventObject( int nWaitEventEntries)
{
	RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle;
	int i;

	// Allocate the overall structure.
	multiWaitEventHandle = (RFD_MULTI_WAIT_EVENT_STRUCT *) RFD_MALLOC(sizeof(RFD_MULTI_WAIT_EVENT_STRUCT));
	if(multiWaitEventHandle == NULL) {
		return NULL;
	}

	// Allocate the array of event objects within the main structure.
	multiWaitEventHandle->waitEventArray = (RFD_EVENT_HANDLE *) RFD_MALLOC(nWaitEventEntries * sizeof(RFD_EVENT_HANDLE));
	if(multiWaitEventHandle->waitEventArray == NULL) {
		RFD_FREE(multiWaitEventHandle);
		return NULL;
	}

	for(i=0;i<nWaitEventEntries;i++) {
		multiWaitEventHandle->waitEventArray[i] = NULL;
	}

	multiWaitEventHandle->nWaitEventEntries = nWaitEventEntries;

	return multiWaitEventHandle;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Delete a multi wait event object to clean up resources of the multi wait event object.
///
/// The individual event objects that were added to the multi wait event object are not cleaned
/// up. See RFD_CreateMultiWaitEventObject() for addition info.
///
/// @param  multiWaitEventHandle    The multi wait event object.
////////////////////////////////////////////////////////////////////////////////////////////////////

void RFD_DeleteMultiWaitEventObject( RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle)
{
	if(multiWaitEventHandle != NULL) {
		RFD_FREE(multiWaitEventHandle->waitEventArray);
	}
	RFD_FREE(multiWaitEventHandle);
	return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Add an individual event object to a multi wait event object.
///
/// Each event object that is added must have a unique event index, specified by the eventIndex
/// parameter, that is used to identify which event is signaled in the call to
/// RFD_WaitForOneOfMultiEvents(). The number of event objects added to the multi wait event
/// object must not exceed the maximum number of event objects configured in the call to
/// RFD_CreateMultiWaitEventObject. As such the eventIndex parameter must be less than this
/// maximum number. See RFD_CreateMultiWaitEventObject() for addition info.
///
/// @param  multiWaitEventHandle    The multi wait event handle.
/// @param  eventHandle             The event object handle being added.
/// @param  eventIndex              Zero-based index of the event object being added.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_AddEventToMultiWaitEventObject(RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle,
											  RFD_EVENT_HANDLE eventHandle,
											  int eventIndex)
{
	if(eventIndex < multiWaitEventHandle->nWaitEventEntries) {
		multiWaitEventHandle->waitEventArray[eventIndex] = eventHandle;
		return RFD_STATUS_OK;
	}
	else {
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Wait for one of the event objects of the multi wait event object to be signaled.
///
/// The RFD_WaitForOneOfMultiEvents() function allows a thread to simultaneously wait for any of
/// the events of the multi wait event object to be set to the signaled state. The first event to
/// be detected as signaled is reset to the non-signaled state and program execution continues
/// for the calling thread. If no event object is signaled within the timeout period specified by
/// timeoutMilliseconds, the program execution continues but returns a value of
/// RFD_STATUS_OBJECT_TIMEOUT. RFD_WaitForOneOfMultiEvents() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE. See RFD_CreateMultiWaitEventObject() for addition info.
///
/// @param  multiWaitEventHandle    The multi wait event handle.
/// @param  timeoutMilliseconds     The timeout in milliseconds. No timeout if
///                                 timeoutMilliseconds = RFD_INFINITE.
/// @param  ocurredEventIndexPtr    (output) The zero-relative index of the event object that was
///                                 signaled.
///
/// @return
/// RFD_STATUS_OK if an event object was detected as signaled and was reset to non- signaled,
/// RFD_STATUS_OBJECT_TIMEOUT if no event objects were signaled and a timeout has occurred
/// instead, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForOneOfMultiEvents(RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle,
									   INT32 timeoutMilliseconds,
								       int * ocurredEventIndexPtr)

{
    RFD_STATUS status;

// platform/OS dependent
#if defined(RFD_WIN32)
	DWORD dwEvent;

	*ocurredEventIndexPtr = 0;

	dwEvent = WaitForMultipleObjects(
					multiWaitEventHandle->nWaitEventEntries,    // number of objects in array
					multiWaitEventHandle->waitEventArray,		// array of objects
					FALSE,				// wait for any object
					timeoutMilliseconds);       // wait time

	if(dwEvent >= WAIT_OBJECT_0 &&
		dwEvent < (WAIT_OBJECT_0 + multiWaitEventHandle->nWaitEventEntries) ) {
		// An event occurred.
		*ocurredEventIndexPtr = dwEvent - WAIT_OBJECT_0;
		status = RFD_STATUS_OK;
	}
	else if(dwEvent == WAIT_TIMEOUT) {
		// A Timeout
		status = RFD_STATUS_OBJECT_TIMEOUT;
	}
	else {
		// Error
		RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Multi-Event Wait error\n"));
		status = RFD_STATUS_ERROR_GENERAL;
	}

#elif defined(RFD_OSAL)
    RFD_EVENT_HANDLE hEvent;
    UN8 un8Index;
    *ocurredEventIndexPtr = 0;

    // init the status variable
    status = RFD_STATUS_ERROR_GENERAL;

    // Is this an infinite wait?
    if (RFD_INFINITE == timeoutMilliseconds)
    {
        // This is just not supported here
        return RFD_STATUS_UNSUPPORTED_FEATURE;
    }

    // Iterate all of our semaphores so we can poll each one
    for (un8Index = 0; un8Index < multiWaitEventHandle->nWaitEventEntries; un8Index++)
    {
        // Grab the current event handle
        hEvent = multiWaitEventHandle->waitEventArray[un8Index];

        // Perform a "try wait" on this event.  We don't want to do anything fancy
        // like put in a wait here based on timeoutMilliseconds (something like
        // (timeoutMilliseconds / multiWaitEventHandle->nWaitEventEntries) because
        // it won't scale well when the number of entries increases or if the
        // wait time is small or if both of those things happen in tandem.
        status = RFD_WaitForEvent(hEvent, 0);

        // Continue looping only if we experienced a timeout
        if (RFD_STATUS_OBJECT_TIMEOUT != status)
        {
            break;
        }
    }

    // Did all of the semaphores time out?
    if (RFD_STATUS_OBJECT_TIMEOUT == status)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Yes, induce the delay requested
        eReturnCode = OSAL.eTaskDelay((UN32)timeoutMilliseconds);

        // Did that work?
        if (OSAL_SUCCESS != eReturnCode)
        {
            // No, tell RFD because we don't
            // want this thread running without
            // a working sleep. It'll eat
            // the CPU.
            status = RFD_STATUS_ERROR_GENERAL;
        }
    }
    // Did we find a waiting event?
    else if (RFD_STATUS_OK == status)
    {
        // we have a winner
        *ocurredEventIndexPtr = un8Index;
    }
#else
#error undefined option
#endif

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a counting semaphore.
///
/// Semaphores are used for synchronizing threads and to manage resources. Counting semaphores
/// are counters that are managed by the Kernel/OS. They are used in situations where a thread
/// needs to wait for something that can be signaled (e.g. by another thread) one or more times.
///
/// A semaphore is first created using RFD_CreateSemaphore(). Each call to RFD_SignalSemaphore()
/// will increment the semaphore's counter. If the counter is not 0, a call to
/// RFD_WaitForSemaphore() decrements the counter by one and execution continues for the thread
/// making the call. If the count is 0, RFD_WaitForSemaphore()
/// waits until the counter is incremented by another thread that calls RFD_SignalSemaphore().
/// Multiple threads can wait on a semaphore. The thread with the highest priority will continue
/// program execution, while other threads will continue waiting if the semaphore counter is 0.
/// If the semaphore is no longer in use by any threads (e.g. closing the application or shutting
/// down) the semaphore resources are cleaned up by calling RFD_CloseSemaphore().
///
/// @param  initialCount    Initial count of the semaphore. This value must be greater than or
///                         equal to zero and less than or equal to maximumCount.
/// @param  maximumCount    Maximum count of the semaphore. This value must be greater than zero.
///
/// @return handle of the created semaphore. NULL if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_SEMAPHORE_HANDLE RFD_CreateSemaphore(int initialCount, int maximumCount)
{
	RFD_SEMAPHORE_HANDLE semaphoreHandle;
// platform/OS dependent
#if defined(RFD_WIN32)
	HANDLE win32semaphoreHandle;
	win32semaphoreHandle = CreateSemaphore(
						NULL,			// default security attributes
						initialCount,	// initial count
						maximumCount,	// maximum count
						NULL			// unnamed semaphore
						);
	// For clarity, explicitly check return handle.
	// Other platforms may have a different error return value
	// other than NULL.
	if(win32semaphoreHandle == NULL) {
		semaphoreHandle = NULL;
	}
	else {
		semaphoreHandle = win32semaphoreHandle;
	}
#elif defined(RFD_OSAL)
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hSem;
    static UN32 un32SemInstanceID = 0;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];

    // construct name for semaphore
    sprintf(&acName[0], RFDLIB_NAME":Sem:%u", un32SemInstanceID);

    eReturnCode = OSAL.eSemCreate(
        &hSem, &acName[0], (UN32)initialCount, (UN32)maximumCount,
        OSAL_SEM_OPTION_NONE);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // increment semaphore instance id
        un32SemInstanceID++;

        // set the handle to our semaphore
        semaphoreHandle = hSem;
    }
    else
    {
        // Error, return NULL for our semaphore handle
        semaphoreHandle = NULL;
    }
#else
#error undefined option
#endif
	return semaphoreHandle;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Close a specified semaphore.
///
/// See RFD_CreateSemaphore() for additional info.
///
/// @param  semaphoreHandle Handle of the semaphore.
///
/// @return true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle)
{
#if defined(RFD_WIN32)
	HANDLE win32semaphoreHandle;
	BOOL isOk;

	win32semaphoreHandle = semaphoreHandle;
	isOk = CloseHandle(win32semaphoreHandle);
	if(isOk) {
		return TRUE;
	}
	else {
		// error
		return FALSE;
	}
#elif defined(RFD_OSAL)
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eSemDelete((OSAL_OBJECT_HDL)semaphoreHandle);

    if (eReturnCode == OSAL_SUCCESS)
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
#else
#error undefined option
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Signal a semaphore by incrementing the semaphore's counter.
///
/// The counter is not incremented if the semaphore's current counter is at the maximum value as
/// configured by RFD_CreateSemaphore(). See RFD_CreateSemaphore() for additional info.
///
/// @param  semaphoreHandle Handle of the semaphore.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_SignalSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle)
{
// platform/OS dependent
#if defined(RFD_WIN32)
	if(!ReleaseSemaphore(semaphoreHandle, 1, NULL)) {
		RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("RFD_SignalSemaphore() failure\n"));
		return RFD_STATUS_ERROR_GENERAL;
	}

	return RFD_STATUS_OK;
#elif defined(RFD_OSAL)
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eSemGive((OSAL_OBJECT_HDL)semaphoreHandle);

    if (eReturnCode == OSAL_SUCCESS)
    {
        return RFD_STATUS_OK;
    }
    else
    {
        return RFD_STATUS_ERROR_GENERAL;
    }
#else
#error undefined option
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Waits for a specified semaphore to be signaled.
///
/// If the counter of the semaphore is not 0, the counter is decremented and program execution
/// continues for the calling thread. If the counter is 0, RFD_WaitForSemaphore() waits until the
/// counter is incremented by another thread. The counter is then decremented and program
/// execution continues. If the semaphore was not signaled within the timeout period specified by
/// timeoutMilliseconds, the program execution continues but returns a value of
/// RFD_STATUS_OBJECT_TIMEOUT. RFD_WaitForSemaphore() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE.
///
/// @param  semaphoreHandle     Handle of the semaphore.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if semaphore is signaled, RFD_STATUS_OBJECT_TIMEOUT if the semaphore is not
/// signaled and a timeout has occurred instead, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle, INT32 timeoutMilliseconds)
{
	RFD_STATUS status;

// platform/OS dependent
#if defined(RFD_WIN32)
	DWORD dwWaitResult;
	DWORD dwMilliseconds = timeoutMilliseconds;

	dwWaitResult = WaitForSingleObject(
		semaphoreHandle,	// handle to semaphore
		dwMilliseconds);	// no time-out interval

	switch (dwWaitResult)
	{
		///////////
		// The event occurred
		///////////
		case WAIT_OBJECT_0:

			//RFD_DPrint( TEXT("Semaphore Decrement Occurred\n"));
			status = RFD_STATUS_OK;

		break;

		case WAIT_TIMEOUT:

			//RFD_DPrint( TEXT("Semaphore Wait Timed-Out\n"));
			status = RFD_STATUS_OBJECT_TIMEOUT;
			break;

		//////////
		// The thread got ownership of an abandoned semaphore.
		// Treat this as error.
		//////////
		case WAIT_ABANDONED:

			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Semaphore Object Abandoned Error\n"));
			status = RFD_STATUS_ERROR_OBJECT_ABANDONED;

		break;

		default:

			RFD_ReportError(RFD_ERR_LEVEL_FATAL, TEXT("Semaphore Unexpected wait error\n"));
			status = RFD_STATUS_ERROR_GENERAL;

			break;
	}
#elif defined(RFD_OSAL)
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eSemTake((OSAL_OBJECT_HDL)semaphoreHandle, timeoutMilliseconds);

    if (eReturnCode == OSAL_SUCCESS)
    {
        status = RFD_STATUS_OK;
    }
    else if (eReturnCode == OSAL_TIMEOUT || eReturnCode == OSAL_SEM_NOT_AVAILABLE)
    {
        status = RFD_STATUS_OBJECT_TIMEOUT;
    }
    else if (eReturnCode == OSAL_ERROR_INVALID_HANDLE)
    {
        status = RFD_STATUS_ERROR_OBJECT_ABANDONED;
    }
    else
    {
        status = RFD_STATUS_ERROR_GENERAL;
    }
#else
#error undefined option
#endif

	return status;
}

#endif // RFD_ALT_SINGLE_THREAD_MODE_ENABLE

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Find the first file entry in a directory.
///
/// RFD_FindFirstFile() is used in combination with the RFD_FindNextFile() call, which returns
/// the remaining file entries in a directory sequentially. The order in which entries are
/// returned is not specified. RFD_FindFirstFile() creates and returns a handle that can be used
/// by sequential calls to RFD_FindNextFile(). A NULL handle is returned with output parameter
/// *isError = FALSE if the directory is empty. The handle must be closed by an RFD_FindClose()
/// call when RFD_FindFirstFile() / RFD_FindNextFile()
/// operation is complete. A file entry refers to either a file or a (sub)directory. A
/// RFD_FIND_FILE_DATA_INFO structure describing each file entry found is copied to the user
/// allocated structure pointed to by the input parameter findFileDataInfoPtr. To get information
/// about the file entry, the caller may call the following functions using this
/// findFileDataInfoPtr pointer : RFD_FindDataGetFileNamePtr(), RFD_FindDataIsItemADirectory(),
/// RFD_FindDataIsItemReadOnly(), RFD_FindDataGetFileSize().
///
/// Some implementations of RFD_FindFirstFile() and  RFD_FindNextFile() may output file entries
/// corresponding to reserved File System file entries. These are file entries which are visible
/// to the user application, but modifiable only by the File System. In FAT file systems, such
/// file entries are the "." and ".." files. The user application must therefore verify that a
/// returned file entry is \e not a reserved file system entry before performing any operation on
/// the file. The function RFD_IsReservedFileSystemFileName() is called by the application for
/// this purpose.
///
/// @param  pathBuf             The full path and name string. For example, the file path "c:/
///                             test" is used to find all file entries in the directory "c:/
///                             test".
/// @param  findFileDataInfoPtr (output) The find file data information structure describing the
///                             found file entry. The structure is allocated by the calling
///                             funtion and filled by this funcion.
/// @param  isError             (output) Pointer to boolen error indicator variable that is set
///                             to true if error, false if not error.
///
/// @return
/// Find File Handle that can be used in subsequent calls to RFD_FindNextFile(), and used in call
/// to RFD_FindClose() when finished.
///
/// Additional return info:
/// This function is modeled after the Win32 function FindFirstFile(). Stating the return value
/// information is terms of the corresponding Win32 documentation:
/// If the function succeeds, the return value is a search handle used in a subsequent call to 
/// RFD_FindNextFile or RFD_FindClose, and the findFileDataInfoPtr parameter contains information 
/// about the first file or directory found.
/// If the function fails or fails to locate files from the pathBuf directory, the return value is NULL 
/// and the contents of findFileDataInfoPtr are indeterminate. 
/// If the function fails because no files can be found, the *isError output value is FALSE. Otherwise,
/// if the function fails because of an error, *isError output value is TRUE.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_FIND_FILE_HANDLE  RFD_FindFirstFile(const TCHAR pathBuf[],
										RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr,
										BOOL * isError)
{
#if defined(RFD_WIN32)
	HANDLE findFileHandle;
	TCHAR * findFileNamePath;
	BOOL doSearch = TRUE;

	*isError = FALSE;

	findFileNamePath = (TCHAR *) RFD_MALLOC(  sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(findFileNamePath == NULL) {
		*isError = TRUE;
		return NULL;
	}

	if (pathBuf == NULL) {
        RFD_FREE(findFileNamePath);
		*isError = TRUE;
		return NULL;
	}

    if (RFD_IsDirectory(pathBuf) == FALSE) {
        // path name is not a directory, return error indication.
        RFD_FREE(findFileNamePath);
		*isError = TRUE;
		return NULL;
    }

	// For WIN32 FindFirstFile() usage, append the wildcard "*" to directory path name.
	// to specifiy finding all files in the directory.
	RFD_STRCPY_S(findFileNamePath, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(findFileNamePath, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(findFileNamePath, RFD_MAX_PATH_LEN, TEXT("*"));    // searching all files of the directory

	findFileHandle = FindFirstFile(findFileNamePath, findFileDataInfoPtr);
	if(findFileHandle == INVALID_HANDLE_VALUE) {
		RFD_FREE(findFileNamePath);
		// Note: with WIN32, a FindFirstFile() with a valid "directory_path\\*" as argument always finds
		// at least the system directory entries ( '.' and '..') if there is no error.
		// So an empty directory (empty of non-system directory entries) will always result in
		// a valid findFileHandle when there is no error.
		*isError = TRUE;
		return NULL;
	}

	RFD_FREE(findFileNamePath);

	*isError = FALSE;
	return findFileHandle;
#elif defined(RFD_OSAL)
    PORT_OSAL_FIND_FILE_DATA_STRUCT *psFindFile = NULL;
    static UN32 un32FindFileInstanceID = 0;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    *isError = FALSE;

    do
    {
        // Check input
        if (pathBuf == NULL)
        {
            break;
        }

        // Create our local structure that is used for managing a FindFile op
        sprintf(&acName[0], RFDLIB_NAME":FindFile:%u", un32FindFileInstanceID++);
        psFindFile = (PORT_OSAL_FIND_FILE_DATA_STRUCT *)
            OSAL.pvMemoryAllocate(&acName[0],
                sizeof(PORT_OSAL_FIND_FILE_DATA_STRUCT), TRUE);

        if (psFindFile == NULL)
        {
			*isError = TRUE;
            break;
        }

        // Save the path we are searching
        strncpy(&psFindFile->acPathToSearch[0],
            pathBuf, sizeof(psFindFile->acPathToSearch));

        psFindFile->acPathToSearch[sizeof(psFindFile->acPathToSearch)-1] = '\0';

        // Get our list of items in this directory
        eReturnCode = OSAL.eFileSystemGetDirItems(
            &psFindFile->hDirItems,
            &psFindFile->acPathToSearch[0],
            NULL,
            NULL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            char * pcCurItemFileName = NULL;
            BOOLEAN bOk;

            // Return the first file that we found and
            // get the file name directly
            psFindFile->hCurItem =
                OSAL.hLinkedListFirst(psFindFile->hDirItems,
                (void **)&pcCurItemFileName);

            bOk = bSetFindFileDataInfo(&psFindFile->sCurrItemFindFileDataInfo,
                pcCurItemFileName, &psFindFile->acPathToSearch[0]);

            if(bOk == TRUE)
            {
                *findFileDataInfoPtr =
                    (RFD_FIND_FILE_DATA_INFO *) &psFindFile->sCurrItemFindFileDataInfo;
            }
            else
            {
				RFD_DPrint( TEXT("RFD_FindFirstFile() Error!: pathBuf = %s, pcCurItemFileName = %s\n"),  
					pathBuf, pcCurItemFileName==NULL?"(NULL val)":pcCurItemFileName);

                *findFileDataInfoPtr = NULL;
				*isError = TRUE;
				break;
            }

            return (RFD_FIND_FILE_HANDLE)psFindFile;
        }
		else if (eReturnCode != OSAL_NO_OBJECTS)
		{
			// If the operation was a success, but the directory was empty
			// then OSAL_NO_OBJECTS would have been returned.
			// Otherwise it looks like we had an error.
			*isError = TRUE;
			break;
		}

    } while (FALSE);

    // Either an error or No items were found,
    // so free any memory and return NULLs for both
    // find file data structures
    RFD_FindClose(psFindFile);
    psFindFile = NULL;
    *findFileDataInfoPtr = NULL;

    return (RFD_FIND_FILE_HANDLE)psFindFile;
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Find the next file entry in a directory. For a functional description, refer to
/// RFD_FindFirstFile().
///
/// @param  findFileHandle      The Handle created by prior call to RFD_FindFirstFile().
/// @param  findFileDataInfoPtr (output) The find file data information structure describing the
///                             found file entry. The structure is allocated by the calling
///                             funtion and filled by this funcion.
/// @param  isError             (output) Pointer to boolen error indicator variable that is set
///                             to true if error, false if not error.
///
/// @return
/// true if file is found, false if file not found. If file not found, it is due to no more files
/// found if isError = false, otherwise it is due to error (isError = true).
///
/// Additional return info:
/// This function is modeled after the Win32 function FindNextFile(). Stating the return value
/// information is terms of the corresponding Win32 documentation:
/// If the function succeeds, the return value is TRUE and the findFileDataInfoPtr parameter contains 
/// information about the next file or directory found.
/// If the function fails, the return value is FALSE and the contents of findFileDataInfoPtr are 
/// indeterminate.
/// If the function fails because no more files can be found, *isError output value
/// is FALSE. Otherwise, if the function fails because of an error, *isError is TRUE.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL  RFD_FindNextFile(RFD_FIND_FILE_HANDLE findFileHandle,
									   RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr,
									   BOOL * isError)
{
#if defined(RFD_WIN32)
	BOOL foundItem;

	*isError = FALSE;

	foundItem = FindNextFile(findFileHandle, findFileDataInfoPtr);
	if(!foundItem) {
		// Item (file/directory) not found
		// either due to error or no more files
		if(GetLastError() == ERROR_NO_MORE_FILES) {
			*isError = FALSE;
		}
		else {
			*isError = TRUE;
		}
	}

	return foundItem;
#elif defined(RFD_OSAL)
    PORT_OSAL_FIND_FILE_DATA_STRUCT *psFindFile =
        (PORT_OSAL_FIND_FILE_DATA_STRUCT *) findFileHandle;
    BOOLEAN bFileFound = FALSE;
    OSAL_LINKED_LIST_ENTRY hNextItem;
    char * pcCurItemFileName = NULL;

    if (psFindFile == NULL)
    {
        *isError = TRUE;
        return FALSE;
    }

    // Get next item in our list
    hNextItem = OSAL.hLinkedListNext(
        psFindFile->hCurItem, (void **)&pcCurItemFileName);

    if (hNextItem != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        BOOLEAN bOk;

        // Return the first file that we found
        psFindFile->hCurItem = hNextItem;

        bOk = bSetFindFileDataInfo(&psFindFile->sCurrItemFindFileDataInfo,
                pcCurItemFileName, &psFindFile->acPathToSearch[0]);

        if(bOk == TRUE)
        {
            *findFileDataInfoPtr =
                (RFD_FIND_FILE_DATA_INFO *) &psFindFile->sCurrItemFindFileDataInfo;
	        *isError = FALSE;
			bFileFound = TRUE;
        }
        else
        {
			RFD_DPrint( TEXT("RFD_FindNextFile() Error!: pcCurItemFileName = %s\n"),  
				pcCurItemFileName==NULL?"(NULL val)":pcCurItemFileName);

            *findFileDataInfoPtr = NULL;
			*isError = TRUE;
        }
    }
    else
    {
        *isError = FALSE;
    }

    return bFileFound;
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Close a find-file handle previously opened by RFD_FindFirst().
///
/// RFD_FindClose() is called after operations using RFD_FindFirst()/RFD_FindNextFile()
/// are complete. Refer to RFD_FindFirstFile() for additional info.
///
/// @param  findFileHandle  The Handle created by prior call to RFD_FindFirstFile().
///
/// @return true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL  RFD_FindClose(RFD_FIND_FILE_HANDLE findFileHandle)
{
#if defined(RFD_WIN32)
	BOOL isOk;
	isOk = FindClose(findFileHandle);
	return isOk;
#elif defined(RFD_OSAL)
    PORT_OSAL_FIND_FILE_DATA_STRUCT *psFindFile =
        (PORT_OSAL_FIND_FILE_DATA_STRUCT *) findFileHandle;

    if (psFindFile == NULL)
    {
        return FALSE;
    }

    // Free our dir items list
    if (psFindFile->hDirItems != OSAL_INVALID_OBJECT_HDL)
    {
        // Doesn't really matter what happens here...
        OSAL.eFileSystemReleaseDirItems(psFindFile->hDirItems);

        psFindFile->hDirItems = OSAL_INVALID_OBJECT_HDL;
    }

    // Free our memory structure
    OSAL.vMemoryFree(psFindFile);

    return TRUE;
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Get the name string of a file entry (file or directory) referred to by a find file info
/// structure.
///
/// The find data info structure is filled out by a prior call to either RFD_FindFirstFile() or
/// RFD_FindNextFile(). See RFD_FindFirstFile() for addional info.
///
/// @param  findFileDataInfoPtr Pointer to a find file info structure.
///
/// @return null if it fails, else points to file name string.
////////////////////////////////////////////////////////////////////////////////////////////////////

TCHAR * RFD_FindDataGetFileNamePtr( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr)
{
#if defined(RFD_WIN32)
	return findFileDataInfoPtr->cFileName;
#elif defined(RFD_OSAL)
    PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *psFindFileDataInfo;

    if (findFileDataInfoPtr == NULL || *findFileDataInfoPtr == NULL)
    {
        return NULL;
    }

    psFindFileDataInfo =
        (PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *) *findFileDataInfoPtr;

    return psFindFileDataInfo->pcFileName;
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Determine if a found file entry (file or directory) is a directory.
///
/// Information about the file entry is in the file info structure pointed to by
/// findFileDataInfoPtr. This structure is filled out by a prior call to either
/// RFD_FindFirstFile() or RFD_FindNextFile(). See RFD_FindFirstFile() for addional info.
///
/// @param  findFileDataInfoPtr Pointer to a find file info structure.
///
/// @return true if file entry is a directory, return false if file entry is a file.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_FindDataIsItemADirectory( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr)
{
#if defined(RFD_WIN32)
	if(findFileDataInfoPtr->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
		return TRUE;
	}
	else {
		return FALSE;
	}
#elif defined(RFD_OSAL)
    PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *psFindFileDataInfo;

    if (findFileDataInfoPtr == NULL || *findFileDataInfoPtr == NULL)
    {
        return FALSE;
    }

    psFindFileDataInfo =
        (PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *) *findFileDataInfoPtr;

    if ((psFindFileDataInfo->un8Attr & OSAL_FILE_ATTR_DIRECTORY) == OSAL_FILE_ATTR_DIRECTORY)
    {
        // Is directory
        return TRUE;
    }
    else
    {
        return FALSE;
    }
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Determine if a found file entry is read-only.
///
/// Information about the file entry is in the file info structure pointed to by
/// findFileDataInfoPtr. This structure is filled out by a prior call to either
/// RFD_FindFirstFile() or RFD_FindNextFile(). See RFD_FindFirstFile() for addional info.
///
/// @param  findFileDataInfoPtr Pointer to a find file info structure.
///
/// @return true if file is read-only, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_FindDataIsItemReadOnly( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr)
{
#if defined(RFD_WIN32)
	if(findFileDataInfoPtr->dwFileAttributes & FILE_ATTRIBUTE_READONLY) {
		return TRUE;
	}
	else {
		return FALSE;
	}
#elif defined(RFD_OSAL)
    PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *psFindFileDataInfo;

    if (findFileDataInfoPtr == NULL || *findFileDataInfoPtr == NULL)
    {
        return FALSE;
    }

    psFindFileDataInfo =
        (PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *) *findFileDataInfoPtr;

    if (
        (psFindFileDataInfo->un8Attr & OSAL_FILE_ATTR_READ) == OSAL_FILE_ATTR_READ &&
        (psFindFileDataInfo->un8Attr & OSAL_FILE_ATTR_WRITE) != OSAL_FILE_ATTR_WRITE
       )
    {
        // It is read-only,
        return TRUE;
    }
    else
    {
        // It isn't read-only
        return FALSE;
    }
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Get file size of a file entry referred to by a find file info structure.
///
/// Information about the file entry is in the file info structure pointed to by
/// findFileDataInfoPtr. This structure is filled out by a prior call to either
/// RFD_FindFirstFile() or RFD_FindNextFile(). See RFD_FindFirstFile() for addional info.
///
/// @param  findFileDataInfoPtr The find file data information pointer.
///
/// @return the size of the file.
////////////////////////////////////////////////////////////////////////////////////////////////////

DWORD RFD_FindDataGetFileSize( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr)
{
#if defined(RFD_WIN32)
	return findFileDataInfoPtr->nFileSizeLow;
#elif defined(RFD_OSAL)
    PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *psFindFileDataInfo;

    if (findFileDataInfoPtr == NULL)
    {
        return FALSE;
    }

    psFindFileDataInfo =
        (PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT *) *findFileDataInfoPtr;

    return (DWORD) psFindFileDataInfo->tFileSize;
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Determine if a specified file name is a special-case file name reserved for file system use.
///
/// RFD_IsReservedFileSystemFileName() is called by the application for the following purpose:
/// Some implementations of RFD_FindFirstFile() and  RFD_FindNextFile() may output file entries
/// corresponding to reserved File System file entries. These are file entries which are visible
/// to the user application, but modifiable only by the File System. In FAT file systems, such
/// file entries are the "." and ".." files. The user application must therefore verify that a
/// returned file entry is \e not a reserved file system entry before performing any operation on
/// the file.
///
/// @param  fileName    File name string. This is the name of a file only; it does not include a
///                     path.
///
/// @return true if file name is a special case name reserved for file system use, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_IsReservedFileSystemFileName(const TCHAR* fileName) {
#if defined(RFD_WIN32)

	if( fileName == NULL ) {
        RFD_DPrint(TEXT("RFD_IsReservedFileSystemFileName():Error! NULL argument\n"), lastErrorCode);
		return TRUE;
	}

	if(RFD_STRCMP(fileName,TEXT(".")) && RFD_STRCMP(fileName,TEXT(".."))) {
		// String is neither of the the special WIN32 file names reserved for file system use.
		return FALSE;
	}
	else {
		// String is one of the special file names reserved for file system use.
		return TRUE;
	}
#elif defined(RFD_OSAL)

	if( fileName == NULL ) {
        RFD_DPrint(TEXT("RFD_IsReservedFileSystemFileName():Error! NULL argument\n"));
		return TRUE;
	}

    // TODO: Make some sort of OSAL function to look for reserved file names?
    if(RFD_STRCMP(fileName,TEXT(".")) && RFD_STRCMP(fileName,TEXT(".."))) {
        return FALSE;
    }
    else
    {
        return TRUE;
    }
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief  Determine if a named file object is a directory.
///
/// @param  dirName Full path and name string of the file object.
///
/// @return
/// true if named file object is a directory, otherwise false if the file object does not exist,
/// the file object is a file, or some error.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_IsDirectory(const TCHAR dirName[])
{
#if defined(RFD_WIN32)
	DWORD dwAttrs;
	dwAttrs = GetFileAttributes(dirName);
	if ((dwAttrs != INVALID_FILE_ATTRIBUTES) && (dwAttrs & FILE_ATTRIBUTE_DIRECTORY) ) {
		return TRUE;
	}
	else {
		return FALSE;
	}
#elif defined(RFD_OSAL)
    BOOLEAN bOk;
    UN8 un8Attr;

    bOk = OSAL.bFileSystemGetFileAttributes(dirName, &un8Attr);

    if (bOk == TRUE)
    {
        if ((un8Attr & OSAL_FILE_ATTR_DIRECTORY) == OSAL_FILE_ATTR_DIRECTORY)
        {
            // It is a directory,
            return TRUE;
        }
        else
        {
            // It isn't a directory
            return FALSE;
        }
    }
    else
    {
        // Error, we don't know what it is
        return FALSE;
    }
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief  Determine if a named file object is a file (as opposed to a directory).
///
/// @param  fileName    Full path and name string of the file object.
///
/// @return
/// true if named file object is a file, otherwise false if the file object does not exist, the
/// file object is a directory, or some error.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_IsFile(const TCHAR fileName[])
{
#if defined(RFD_WIN32)
	DWORD dwAttrs;
	dwAttrs = GetFileAttributes(fileName);
	if ((dwAttrs != INVALID_FILE_ATTRIBUTES) && !(dwAttrs & FILE_ATTRIBUTE_DIRECTORY) ) {
		return TRUE;
	}
	else {
		return FALSE;
	}
#elif defined(RFD_OSAL)
    BOOLEAN bOk;
    UN8 un8Attr;

    // With OSAL, we just have an attribute stating if a path is a directory,
    // so we can use that function and return the opposite of what it says
    bOk = OSAL.bFileSystemGetFileAttributes(fileName, &un8Attr);
    if (bOk == TRUE)
    {
        if ((un8Attr & OSAL_FILE_ATTR_DIRECTORY) == OSAL_FILE_ATTR_DIRECTORY)
        {
            // It is a directory, so return false
            return FALSE;
        }
        else
        {
            // It isn't a directory
            return TRUE;
        }
    }
    else
    {
        // Error, we don't know what it is
        return FALSE;
    }
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Get the size of the file referred to by a file name.
///
/// @param	fileName	Full path name string of the file.
/// @param	pFileSize	(output) Size of the file in bytes.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS  RFD_GetFileSize( const TCHAR fileName[], UINT32 * pFileSize)
{
#if defined(RFD_WIN32)
	HANDLE findFileHandle;
	WIN32_FIND_DATA findFileDataInfo;

	*pFileSize = 0;

	findFileHandle = FindFirstFile(fileName, &findFileDataInfo);
	if(findFileHandle == INVALID_HANDLE_VALUE) {
		return RFD_STATUS_ERROR_FILE_FIND;
	}
	else if(findFileDataInfo.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
		// file is a directory type, return error.
		RFD_FindClose(findFileHandle);
		return RFD_STATUS_ERROR_FILE_FIND;
	}

	RFD_FindClose(findFileHandle);

	// Return file size
	*pFileSize = findFileDataInfo.nFileSizeLow;
	return RFD_STATUS_OK;
#elif defined(RFD_OSAL)
    BOOLEAN bSuccess = FALSE;
    FILE *psFile;
    size_t tFileSize;

    *pFileSize = 0;

    psFile = fopen(fileName, "r");

    if (psFile == NULL)
    {
        return RFD_STATUS_ERROR_FILE_OPEN;
    }

    bSuccess = OSAL.bFileSystemGetFileSize(psFile, &tFileSize);
    fclose(psFile);

    if (bSuccess != TRUE)
    {
        return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
    }

    *pFileSize = (UINT32)tFileSize;
    return RFD_STATUS_OK;
#else
#error undefined platform
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Sync the File to disk/flash device (force file data to persistent storage media).
///
/// Perform a 'file sync' type operation to cause data of a specified file, the data possibly
/// residing in an intermediate RAM cache, to be commited immediately (synchronously) to the
/// persistent media (e.g. to disk or flash). As such, upon return from this function, the file
/// data is safely stored, even upon a power loss or system crash at the point of function
/// return. On the Windows platform, FlushFileBuffers() provides this functionality. On linux
/// platforms, fdatasync() or fsync() provides this functionality.
///
/// This file sync functionality is critical to the RFD Receiver to ensure power loss resilient
/// operation (ensure the ability to properly recover after sudden power loss or other system
/// abort events). This functionality is important for either
/// RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE (mode of the RFD Recevier) defined or undefined.
///
/// On systems which otherwise guarantee data of a file is synchronously written directly to
/// persistent media at the point of writing data to the file (upon RFD_FWRITE / fwrite) or at
/// point of closing a file (upon RFD_FCLOSE / fclose), the developer may simply implement
/// RFD_FileSync() function as: RFD_STATUS  RFD_FileSync(const TCHAR fileName[]) {return
/// RFD_STATUS_OK;}
///
/// Further justification and description of file sync'ing using fsync/fdatasync or
/// FlushFileBuffers to ensure reliable and uncorrupted data storage is given in SQLite
/// documentation, e.g.: http://www.sqlite.org/atomiccommit.html
/// http://www.sqlite.org/lockingv3.html.
///
/// @param  hFile    File Handle, obtained from fopen()
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS  RFD_FileSyncByHandle(RFD_FILE *hFile)
{
#if defined(RFD_WIN32)
    BOOLEAN bOk;
    HANDLE hFile;

    // Check inputs
    if( hFile == NULL )
    {
        // Error!
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }
    
    hFile = (HANDLE)_get_osfhandle(_fileno((struct _iobuf *)psFile->pvHdl));
    
    bOk = FlushFileBuffers(hFile);
    if(bOk == FALSE)
    {
        printf("Error! OS_bFileSystemSyncFile");
        return RFD_STATUS_ERROR_FILE_SYNC;
    }

    return RFD_STATUS_OK;

#elif defined(RFD_OSAL)
	BOOLEAN bOk;
	RFD_STATUS rfdStatus = RFD_STATUS_OK;

	if(hFile == NULL)
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	bOk = OSAL.bFileSystemSyncFile(hFile);
	if(!bOk)
	{
		rfdStatus = RFD_STATUS_ERROR_FILE_SYNC;
		RFD_DPrint(TEXT("RFD_FileSyncByHandle() file sync error\n"));
	}
	
	return rfdStatus;

#else
#error undefined platform

#if 0 // Example implementation for unix/linux based system.
    int fd, rc;
    RFD_STATUS rfdStatus = RFD_STATUS_OK;

    // Open file
    hFile = fopen(fileName, "rb");

    if (hFile == NULL) {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    // Return the file descriptor for a stream
    fd = fileno(hFile);
    if(fd == -1) {
        RFD_DPrint(TEXT("RFD_FileSync():Error! fileno(), errno %d\n"), errno);
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    ////////////////////
    // Sync file to disk
    ////////////////////
#ifdef NO_FDATASYNC
    // use fsync() if fdatasync() is not available on the platform.
    rc = fsync(fd);
#else
    // Use fdatasync() if it is available on the platform.
    // fdatasync() is more efficient vs. fsync() which syncs both file data and metadata.
    // RFD does not care about syncing file metadata that is not needed in order to
    // allow a subsequent data retrieval to be correctly handled (e.g. st_atime or st_mtime).
    rc = fdatasync(fd);
#endif
    if(rc == -1) {
        RFD_DPrint(TEXT("RFD_FileSync():Error! fdatasync(), errno %d)\n"), errno);
        return RFD_STATUS_ERROR_FILE_SYNC;
    }

    return RFD_STATUS_OK;
#endif

#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Sync the directory to disk/flash device (force directory file data to persistent storage
/// media).
///
/// The purpose of this function is the same as that of RFD_FileSync(), except the data stored is
/// the data relating to a specified directory instead of a specified regular file.
/// Refer to RFD_FileSync() documentation for additional information.
///
/// A user calls this function to ensure all critical information that defines the directory,
/// including the list of all the files and sub-directories that the directory contains, is
/// stored immediately (synchronously) to persistent storage media. As such, even after a power
/// loss or system abort type event at point of this function return, all information defining
/// the directory is safely preserved.
///
/// An example where this 'directory sync' functionality is useful, is to protect against the
/// case whereby a power loss occurs shortly after the contents of the directory has been changed
/// (e.g. a file in this directory is created for the first time, or renamed, or deleted).
///
/// On some platforms/file-systems, a directory is a special type of file (a directory file)
/// whose data content is a list of files and (sub)directories (directory entries) contained in
/// the directory. Typical linux platforms are like this, and the same functions used to sync the
/// data contents of files are used to sync the data contents of directories (e.g. fdatasync() or
/// fsyn() functions). Some platforms/file-systems don't require the use of 'directory sync', for
/// example, for the Windows platform FlushFileBuffers() syncs the contents of files, but there
/// is no equivalent function to sync the contents of directories.
///
/// This directory sync functionality is critical to the RFD Receiver to ensure power loss
/// resilient operation (ensure the ability to properly recover after sudden power loss or other
/// system abort events). This functionality is important for either
/// RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE (mode of the RFD Recevier) defined or undefined.
///
/// On systems which otherwise guarantee data of a directory is synchronously written directly to
/// persistent media at the point of modifying the directory contents (e.g. creating, deleting,
/// or renaming a containing file or (sub)directory), the developer may simply implement
/// RFD_DirectorySync() function as: RFD_STATUS  RFD_DirectorySync(const TCHAR fileName[])
/// {return RFD_STATUS_OK;}
///
/// @param  dirName Full path name string of the directory (no trailing \ or /).
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS  RFD_DirectorySync(const TCHAR dirName[])
{
#if defined(RFD_WIN32)
    // Treat Directory Sync as a NOP for Windows. This follows the SQLite example.
    return RFD_STATUS_OK;
#elif defined(RFD_OSAL)
    FILE *psFile;
	UN8 un8Attr;
	BOOLEAN bOk;
	RFD_STATUS rfdStatus;

	if(dirName == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	// Get attributes to verify this is a directory
	bOk = OSAL.bFileSystemGetFileAttributes(dirName, &un8Attr );

	if(!bOk) {
		// Error getting attributes.
		rfdStatus = RFD_STATUS_ERROR_GENERAL;
		RFD_DPrint(TEXT("RFD_DirectorySync() error getting dir attibutes: %s\n"), dirName);
	}
	else if((un8Attr & OSAL_FILE_ATTR_DIRECTORY) == OSAL_FILE_ATTR_DIRECTORY)
	{
		// Entry is a directory.

		// Open directory in read mode. For some OS's like Linux, read mode is the
		// only mode allowed for directories.
		psFile = fopen(dirName, "rb");
		if(psFile != (FILE *)NULL)
		{
			// Open succeeded,
			// try to sync the directory.
			bOk = OSAL.bFileSystemSyncFile(psFile);

			fclose(psFile);

			if(bOk) {
				// successful directory sync
				rfdStatus = RFD_STATUS_OK;
			}
			else {
				// directory sync error
				rfdStatus = RFD_STATUS_ERROR_FILE_SYNC;
				RFD_DPrint(TEXT("RFD_DirectorySync() sync error: %s\n"), dirName);
			}
		}
		else {
			// Treat open failure as a case where the OS does not support
			// opening and syncing directories (e.g. WIN32).
			// Report as ok.
			rfdStatus = RFD_STATUS_OK;
			RFD_DPrint(TEXT("RFD_DirectorySync() unable to fopen: %s\n"), dirName);
		}
	}
	else {
		rfdStatus = RFD_STATUS_ERROR_INVALID_MODE;
		RFD_DPrint(TEXT("RFD_DirectorySync() item not a directory: %s\n"), dirName);
	}

	return rfdStatus;

#else
#error undefined platform

#if 0 // Example implementation for unix/linux based system.
    // Sync for a directory can be done like file sync for file,
    // so just call RFD_FileSync().
    return RFD_FileSync(dirName);
#endif

#endif
}


////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Retrieve the last error code value of the calling thread.
///
/// Each thread maintains it's own last error code.
///
/// @return The calling thread's last error code value.
////////////////////////////////////////////////////////////////////////////////////////////////////

DWORD RFD_GetLastError(VOID)
{
#if defined(RFD_WIN32)
	DWORD lastErrorCode;
	lastErrorCode = GetLastError();
	return lastErrorCode;
#elif defined(RFD_OSAL)
    UN32 un32LastErrorCode;
    un32LastErrorCode = errno;
    return un32LastErrorCode;
#else
#error undefined platform
#endif
}


#ifdef DEBUG_PRINT_ENABLE
////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Print info to the console in support of test and debug.
///
/// @param	format_string	Format control (as in standard printf()).
/// @param	...				Optional arguments (as in standard printf()).
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_DPrint( const TCHAR * format_string, ...)
{
#if defined(RFD_WIN32)

	va_list ap; /*will point to each unnamed argument in turn*/
    size_t cchStringSize;
    DWORD dwChars;
	TCHAR * msgBuf;
	DWORD msgBufSize = RFD_PRINT_MAX_BUF_SIZE;

	if(NULL == (msgBuf = (TCHAR *) RFD_MALLOC(sizeof(TCHAR)*RFD_PRINT_MAX_BUF_SIZE)))
	{
		return FALSE;
	}

	va_start(ap,format_string);

	// to calc string length e.g. if dst_string buffer is allocated in this function.
	//cchStringSize = _vscwprintf( format_string, ap ) + 1 ;

	cchStringSize = vswprintf_s(msgBuf, msgBufSize, format_string, ap );
	va_end(ap);

    WriteConsole(GetStdHandle(STD_OUTPUT_HANDLE), msgBuf, (DWORD)cchStringSize, &dwChars, NULL);

	RFD_FREE(msgBuf);
	return TRUE;
#elif defined(RFD_OSAL)
    va_list tList; // variable arguments list
    TCHAR * msgBuf;

    if(NULL == (msgBuf = (TCHAR *) RFD_MALLOC(sizeof(TCHAR)*RFD_PRINT_MAX_BUF_SIZE)))
    {
        return FALSE;
    }

    // grab variable arguments (pop)
    va_start(tList, format_string);

    vsprintf(msgBuf, format_string, tList);

    // use vfprintf to print out to stderr
    fprintf(stderr, "RFD_DEBUG: %s", msgBuf);

    // restore stack (push)
    va_end(tList);

    RFD_FREE(msgBuf);
    return TRUE;
#else
#error undefined platform
#endif
}
#else
////////////////
// RFD_DPrint() stub when DEBUG_PRINT_ENABLE is not defined.
////////////////
BOOL RFD_DPrint( const TCHAR * format_string, ...)
{
	return TRUE;
}
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief Scan the specified directory to find and delete corrupted directory entries.
///        
/// Corrupted directory entries are entries that appear in the directory listing and are provided
/// by name by the file system directory read functions, but for which additional access to the file
/// fails (e.g. stats(), fopen()).
/// The function by be called recursively by setting doRecursive TRUE, to also process all subdirectories
/// under dirPathName.
/// The caller may specify an "ancilliary" file, by filename pAncillaryDeleteFileName that the function
/// will also delete if any corrupted directory entries are found and deleted. The ancillary file 
/// of that name will be deleted from the same directory in which one or more corrupted directory entries
/// are found and deleted from. This ancillary file delete feature is intended to support the RFD transaction
/// protection scheme in which the RFD transaction state file name is specified as the ancilliary file name.
/// The ancillary file, if specified (pAncillaryDeleteFileName not NULL) will be deleted first, before
/// the first corrupted directory entry is deleted. If the caller specifies bDoDirSyncAfterAncillaryDelete = TRUE
/// then the function performs a Directory Sync after deleting the ancilliary file.
/// 
/// @param  dirPathName							Path name of directory to process
/// @param  pNumCorruptedEntriesFound			(output) Number of corrupted directory entries found.
///												Caller may set to NULL to disable output of this value.
/// @param  pAncillaryDeleteFileName			The name (not including path) of an extra file in dirPathName directory
///												to delete for the case of finding one or more corrupdated directory entries.
///												Set to NULL to disable ancillary file delete.
/// @param  bDoDirSyncAfterAncillaryDelete		Set to TRUE to perform a Directory Sync operation after deleting a specified
///												ancilliary file.
/// @param  bDoDirSyncAfterCorruptedEntryDeletes Set to TRUE to perform a final Directory Sync operation after finding and deleting 
///												 one or more corrupted directory entries.
/// @param  doRecursive							Set to TRUE to perform recursively to process all subdirectories below dirPathName.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
///			On failure return, *pNumCorruptedEntriesFound will still indicate the number of
///         directory entries found/deleted.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS  RFD_DeleteFileSysCorruptedDirectoryEntries(
				const TCHAR *dirPathName, 
				UN32 *pNumCorruptedEntriesFound, 
				const TCHAR *pAncillaryDeleteFileName, 
				BOOL bDoDirSyncAfterAncillaryDelete, 
				BOOL bDoDirSyncAfterCorruptedEntryDeletes, 
				BOOL doRecursive)
{
#if defined(RFD_WIN32)
	// Not supported
	RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries(), not supported for RFD_WIN32\n"));
	return RFD_STATUS_OK;
#elif defined(RFD_OSAL)
    static UN32 un32ThisFcnCallInstanceID = 0;
    OSAL_RETURN_CODE_ENUM eOSALReturnCode;
    OSAL_OBJECT_HDL hDirItems = OSAL_INVALID_OBJECT_HDL;
	char *pcFilePathName = NULL;
    size_t tPathLen;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];
	UN8 un8Attr;
	RFD_STATUS eReturnStatus = RFD_STATUS_OK;
	int iStringLen;

	if( pNumCorruptedEntriesFound != NULL )
	{
		*pNumCorruptedEntriesFound = 0;
	}

	// check input
	if( dirPathName == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

    // Create a buffer for a file path name string
	tPathLen = RFD_MAX_PATH_LEN;
	sprintf(acName, RFDLIB_NAME":DelCorruptDirEntries:%u", un32ThisFcnCallInstanceID++);
	pcFilePathName = (char *)
		OSAL.pvMemoryAllocate(acName, tPathLen, TRUE);
    if (pcFilePathName == NULL)
    {
        return RFD_STATUS_ERROR_MEM_ALLOC;
    }

    // Get the list of all the items (directory entries) in the directory
    eOSALReturnCode = OSAL.eFileSystemGetDirItems(
        &hDirItems, dirPathName,
        NULL, NULL);
    if (eOSALReturnCode == OSAL_SUCCESS)
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        const char *pcEntry;
		BOOL bFoundCorruptedEntryInThisDir = FALSE;

		//////////////////////////////////////////////////////
        // Iterate all directory entries, 
		// Evaluating for corrupted entries and deleting them.
		//////////////////////////////////////////////////////

        hEntry = OSAL.hLinkedListFirst(hDirItems, (void **)&pcEntry);

        while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
               (pcEntry != NULL))
        {
			if(!RFD_IsReservedFileSystemFileName(pcEntry)) 
			{
				// This is a regular directory entry, not reserved, so continue processing it.

				if( !( bFoundCorruptedEntryInThisDir && 
					   pAncillaryDeleteFileName != NULL && 
					   strcmp(pAncillaryDeleteFileName, pcEntry) == 0 ) )
				{
					// This entry is NOT the ancilliary file that would have already been deleted
					// upon a first corrupted entry already found (as indicated by bFoundCorruptedEntryInThisDir).
					// Since it would have already been deleted, it would evaluate here as 
					// a corrupted file, so that's why it's skipped.

					// Construct the full file path name string for this directory entry.
					iStringLen = snprintf( pcFilePathName, tPathLen,
						"%s/%s", dirPathName, pcEntry );

					if (iStringLen > 0 && (size_t) iStringLen < tPathLen)
					{
						///////////////////////////////////
						// Evaluating the Directory Entry stage
						///////////////////////////////////

						BOOL bIsCurrEntryCorrupted = FALSE;
						BOOL bOk;

						RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries(), evaluating item: %s\n"),  pcFilePathName);

						// Get file attributes
						bOk = OSAL.bFileSystemGetFileAttributes(pcFilePathName, &un8Attr );

						if(!bOk)
						{
							// Indication of corrupted directory entry.
							bIsCurrEntryCorrupted = TRUE;
						}
						else if((un8Attr & OSAL_FILE_ATTR_DIRECTORY) != OSAL_FILE_ATTR_DIRECTORY)
						{
							// Directory entry is a file (not a directory).
							// Try to open file and get file size 
							// as a further directory entry validation step.
				
							FILE *psFile = fopen(pcFilePathName, "r");

							if (psFile == NULL)
							{
								// Indication of corrupted directory entry,
								bIsCurrEntryCorrupted = TRUE;
							}
							else 
							{
								// Was able to open the file, now try to get file size as yet another validation step.

								size_t tFileSize;
								bOk = OSAL.bFileSystemGetFileSize(psFile, &tFileSize);

								if(bOk == FALSE)
								{
									// Indication of corrupted directory entry.
									bIsCurrEntryCorrupted = TRUE;
								}

								fclose(psFile);
							}
						}
						else 
						{
							// Directory entry is a directory

							if( doRecursive )
							{
								RFD_STATUS eRecursiveStatus;
								UN32 un32ResursiveCallNumCorruptedEntriesFound = 0;

								eRecursiveStatus = RFD_DeleteFileSysCorruptedDirectoryEntries(
														pcFilePathName, 
														&un32ResursiveCallNumCorruptedEntriesFound, 
														pAncillaryDeleteFileName, 
														bDoDirSyncAfterAncillaryDelete, 
														bDoDirSyncAfterCorruptedEntryDeletes, 
														doRecursive );

								if( eRecursiveStatus != RFD_STATUS_OK )
								{
									// Continue even if error reported,
									// but maintain this error for the final return status 
									eReturnStatus = eRecursiveStatus;
								}

								// This function is defined to support output of corrupted entry found count
								// with and without error return.
								if( pNumCorruptedEntriesFound != NULL )
								{
									*pNumCorruptedEntriesFound += un32ResursiveCallNumCorruptedEntriesFound;
								}
							}
						}
					
						//////////////////////////////////////////////
						// Deleting Corrupted Directory Entry stage
						//////////////////////////////////////////////

						if( bIsCurrEntryCorrupted )
						{
							BOOLEAN bUnlinkStatus;

							RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries() Found corrupted Dir Entry, file path name = %s\n"),  
								pcFilePathName);

							// Found a corrupted directory entry - specifically in this directory, not in a subdirectory.
							//
							// Ensure that any named ancillary file is deleted first
							// and that the directory is "sync'ed" right afterwards, if directory sync is 
							// requested by the caller. 
							// The intent of the ancillary file delete feature is the the ancillary file
							// is a "transaction state" file used by the caller and it's non-existence
							// (i.e. if it is deleted here) will indicate an incomplete transaction, 
							// which is the desired state the application wants to recognize if a
							// corrupted directory entry is deleted here.
							// By deleting the ancillary "transaction state" file and syncing the directory first, 
							// before the corrupted dir entry is deleted, this protects for the case of power-loss
							// withing this function.

							if( !bFoundCorruptedEntryInThisDir && pAncillaryDeleteFileName != NULL)
							{
								// This is the first found corrupted dir entry for this directory and
								// the caller requested deletion of a named ancillary file if a corrupted entry was found,
								// so delete the ancillary file now.
								// Note that current corrupted entry found may or may not be this ancillary file

								// Construct the full file path name for the ancillary file.
								iStringLen = snprintf( pcFilePathName, tPathLen,
									"%s/%s", dirPathName, pAncillaryDeleteFileName );

								if (iStringLen > 0 && (size_t) iStringLen < tPathLen)
								{
									// delete the requested ancillary file.
									// Use the unlink function (not remove function) here also, because it is possible
									// that the ancilliary file was itself the corrupted entry that was found.
									bUnlinkStatus = OSAL.bFileSystemUnconditionalFileUnlink(pcFilePathName);

									RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries() attempted delete (unlink) of requested ancilliary file,"
											" status = %d, file path name = %s\n"),  
											bUnlinkStatus, pcFilePathName);

									if( bDoDirSyncAfterAncillaryDelete )
									{
										RFD_DirectorySync( dirPathName );
									}
								}
								else 
								{
									RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries() , snprintf error,  iStringLen = %d, tPathLen = %d"
											" dirPathName = %s, pAncillaryDeleteFileName = %s\n"),  
											iStringLen, tPathLen, dirPathName, pAncillaryDeleteFileName);
								}
							}
							
							// check if the current corrupted entry is the named ancillary file that would have already been deleted
							if( !( pAncillaryDeleteFileName != NULL && strcmp(pAncillaryDeleteFileName, pcEntry) == 0 ) )
							{
								// This corrupted entry is not the named ancillary file that would have already been deleted,
								// so delete this found corrupted entry now. 

								// Construct the full file path name for this directory entry.
								iStringLen = snprintf( pcFilePathName, tPathLen,
									"%s/%s", dirPathName, pcEntry );

								if (iStringLen > 0 && (size_t) iStringLen < tPathLen)
								{
									// delete the current corrupted directory entry.
									// Use unlink() as it is more likely to successfully delete,
									// versus remove() as the latter may do pre-checks including
									// check for file or directory type entry and then return
									// with error without deleting the entry.
									bUnlinkStatus = OSAL.bFileSystemUnconditionalFileUnlink(pcFilePathName);

									RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries() attempted delete (unlink) of corrupted dir entry,"
													 " status = %d, file path name = %s\n"),  
													bUnlinkStatus, pcFilePathName);
								}
								else 
								{
									RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries() , snprintf error,  iStringLen = %d, tPathLen = %d"
											" dirPathName = %s, pcEntry = %s\n"),  
											iStringLen, tPathLen, dirPathName, pcEntry);
								}
							}
							// else this corrupted entry was the named ancillary file that would have already been deleted 
							// on the first found corrupted entry  

							// Flag that a corrupted entry is found
							// and increment the found corrupted entry count.
							bFoundCorruptedEntryInThisDir = TRUE;
							if( pNumCorruptedEntriesFound != NULL ) 
							{
								*pNumCorruptedEntriesFound += 1;
							}
						}
					}
					else
					{
						RFD_DPrint( TEXT("RFD_DeleteFileSysCorruptedDirectoryEntries() , snprintf error,  iStringLen = %d, tPathLen = %d"
							" dirPathName = %s, pcEntry = %s\n"),  
							iStringLen, tPathLen, dirPathName, pcEntry);
					}
				}
			}

            // Move to the next entry
            hEntry = OSAL.hLinkedListNext(hEntry, (void **)&pcEntry);
        }

        // Release our dir items list
        OSAL.eFileSystemReleaseDirItems(hDirItems);

		if(bDoDirSyncAfterCorruptedEntryDeletes == TRUE && bFoundCorruptedEntryInThisDir == TRUE)
		{
			RFD_DirectorySync( dirPathName );
		}
    }

	OSAL.vMemoryFree(pcFilePathName);

    return eReturnStatus;
#else
#error undefined platform
#endif
}

///
/// @} // doxygen group end symbol, don't remove.
///

#if defined(RFD_OSAL)
/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   PORT_fopen
*
*   Called when the RFD_FOPEN macro is called.  Wrapper function needed so
*   that the method signature matches the fopen in the WIN32 APIs.
*
*****************************************************************************/
INT32 PORT_fopen (
    RFD_FILE **ptFile,
    const char* pcFileName,
    const char* pcMode
        )
{
    *ptFile = fopen(pcFileName, pcMode);

    if (*ptFile != NULL)
    {
        // Success!!, return value indicating it is all good
        return 0;
    }
    else
    {
        // Error!
        return errno;
    }
}

/*****************************************************************************
*
*   PORT_remove
*
*   Called when the RFD_FOPEN macro is called.  Wrapper function needed so
*   that the method signature matches the fopen in the WIN32 APIs.
*
*****************************************************************************/
INT32 PORT_remove (
    const char* pcFileName
        )
{
    INT32 iRetVal = remove(pcFileName);

    if (iRetVal > 0)
    {
        // Return a value of zero if there was a problem
        return (INT32)FALSE;
    }
    else
    {
        // Return a non-zero value if all is good
        return (INT32)TRUE;
    }
}

/*****************************************************************************
*
*   PORT_mkdir
*
*   Called when the RFD_CREATE_DIR macro is called.  Wrapper function needed so
*   that the method signature matches the mkdir standard C call.
*
*   http://www.opengroup.org/onlinepubs/000095399/functions/mkdir.html
*
*****************************************************************************/
INT32 PORT_mkdir (
    const char* pcDirName
        )
{
    BOOLEAN bSuccess = OSAL.bFileSystemMakeDir(pcDirName);

    if (bSuccess == TRUE)
    {
        return 0;
    }
    else
    {
        // ERROR
        return -1;
    }
}

/*****************************************************************************
*
*   PORT_rmdir
*
*   Called when the RFD_REMOVE_DIR macro is called.  Wrapper function needed so
*   that the method signature matches the rmdir standard C call.
*
*   http://opengroup.org/onlinepubs/007908775/xsh/rmdir.html
*
*****************************************************************************/
INT32 PORT_rmdir (
    const char* pcDirName
        )
{
    BOOLEAN bSuccess = OSAL.bFileSystemRemoveDir(pcDirName);

    if (bSuccess == TRUE)
    {
        return 0;
    }
    else
    {
        // ERROR
        return -1;
    }
}

/*****************************************************************************
*
*   PORT_sprintf_s
*
*   Called when the RFD_SPRINTF_S macro is called.  Wrapper function needed so
*   that the method signature matches the "safe" sprintf in the WIN32 APIs.
*
*****************************************************************************/
INT32 PORT_sprintf_s (
    TCHAR *buffer,
    size_t sizeOfBuffer,
    const TCHAR *format,
    ...
        )
{
    va_list tList; // variable arguments list
    INT32 i32ReturnValue;
    // grab variable arguments (pop)
    va_start(tList, format);

    // use vsprintf to print out
    i32ReturnValue = vsprintf(buffer, format, tList );

    // restore stack (push)
    va_end(tList);

    return i32ReturnValue;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n32LocalTaskHandler
*
*   Local task handler function for all tasks created with RFD_CreateThread.
*   Provides a method to implement the RFD_WaitForThreadExit function through
*   the use of a semaphore.
*
*****************************************************************************/
static N32 n32LocalTaskHandler (
    void *pvArg
        )
{
    N32 n32ReturnValue = N32_MIN;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    PORT_OSAL_THREAD_STRUCT *psThread =
      (PORT_OSAL_THREAD_STRUCT *)pvArg;

    if (psThread == NULL)
    {
        return N32_MIN;
    }

#ifdef DEBUG_PRINT_ENABLE
    OSAL.vControlOutputThisTask(TRUE);
#endif

    /* Register the Task */
    eReturnCode = OSAL.eTaskRegister( 0, /* No Reporting */
        NULL, NULL, NULL, NULL);

    if(eReturnCode != OSAL_SUCCESS)
    {
        RFD_DPrint("[RFD_PORT] Error! Could not register task: %s\r\n",
               OSAL.pacGetReturnCodeName(eReturnCode));
        n32ReturnValue = -1;
    }

    // Call the specified task handler
    psThread->dwTaskHandler(psThread->pvTaskArg);

    psThread->hTask = OSAL_INVALID_OBJECT_HDL;

    // When task returns, signal our semaphore
    OSAL.eSemGive(psThread->hSemThreadExit);

    // Unregister from the monitor.
    eReturnCode = OSAL.eTaskUnregister();

    if (eReturnCode == OSAL_SUCCESS)
    {
        // Indicate the task exited correctly
        n32ReturnValue = OSAL_TASK_REPORT_NO_ERROR;
    }

    return n32ReturnValue;
}

/*****************************************************************************
*
*   bSetFindFileDataInfo
*
*
*****************************************************************************/
static BOOLEAN bSetFindFileDataInfo (
    PORT_OSAL_RFD_FIND_FILE_DATA_INFO_STRUCT * psFindFileDataInfo,
    char * pcFileName, char * pcPathName
        )
{
   BOOLEAN bOk = FALSE;
   FILE *psFile = NULL;
   UN8 un8Attr = UN8_MIN;
   size_t tFileSize = UN32_MIN;

   do
   {
       if(psFindFileDataInfo == NULL || pcFileName == NULL || pcPathName == NULL)
       {
           break;
       }

       
       // Build the whole file name and path string.
       strncpy(psFindFileDataInfo->acFullFilePath,
                      pcPathName, sizeof(psFindFileDataInfo->acFullFilePath));

       psFindFileDataInfo->acFullFilePath[sizeof(psFindFileDataInfo->acFullFilePath)-1] = '\0';
       
       if (strlen(psFindFileDataInfo->acFullFilePath) < sizeof(psFindFileDataInfo->acFullFilePath) - 1 )
          strcat(psFindFileDataInfo->acFullFilePath, "/");

       if (strlen(psFindFileDataInfo->acFullFilePath) < sizeof(psFindFileDataInfo->acFullFilePath) - strlen(pcFileName))
          strcat(psFindFileDataInfo->acFullFilePath, pcFileName);

       // Get file attributes
       bOk = OSAL.bFileSystemGetFileAttributes(
           psFindFileDataInfo->acFullFilePath, &un8Attr);

       if(bOk == FALSE)
       {
           break;
       }

       // Get file size if entry is not a directory
       if((un8Attr & OSAL_FILE_ATTR_DIRECTORY) != OSAL_FILE_ATTR_DIRECTORY)
       {
           psFile = fopen(psFindFileDataInfo->acFullFilePath, "r");

           if (psFile == NULL)
           {
               break;
           }

           bOk = OSAL.bFileSystemGetFileSize(psFile, &tFileSize);

           if(bOk == FALSE)
           {
               break;
           }

           // Close our file
           fclose(psFile);
        }

        bOk = TRUE;

   } while(FALSE);

   if ( psFindFileDataInfo != NULL )
   {
      psFindFileDataInfo->pcFileName = pcFileName;
      psFindFileDataInfo->tFileSize = tFileSize;
      psFindFileDataInfo->un8Attr = un8Attr;
   }

   return bOk;
}

#endif
