////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_message_generate.c
///
/// @brief	rfd message generate class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_bitstream_io.h"

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_GenerateFileSetMetadataExtensionRecord(
                RFD_FSET_MDATA_STRUCT * fileSetMetadataMsg)
{
	UINT32 dWord;
    INT32 currExtRecordLenBytes;
    INT32 maxExtensionLenBits;
	RFD_BITSTREAM_INFO streamInfo;

	////////////////////////////////////
    //
    // To support testing, on input fileSetMetadataMsg->extension may contain fill data set by application,
    // with fileSetMetadataMsg->extensionLenBytes also an input specifying the length of fill.
    //
    // This function will update the fileSetMetadataMsg->extension buffer based on other filds of
    // fileSetMetadataMsg that belong in the Extension Record (written over the original fill data).
    //
    // This function will update fileSetMetadataMsg->extensionLenBytes if the input value
    // is less than the resulting extension length after updates in this function to the extension record.
    //
	////////////////////////////////////


    if(fileSetMetadataMsg->blockMsgDmi == RFD_FSET_MDATA_BLK_MSG_DMI_DEFAULT_VALUE) {
        // Default DMI value. No need to communicate default DMI value in the extension.
        return RFD_STATUS_OK;
    }

    // Check for out of range / invalid DMI
    if(fileSetMetadataMsg->blockMsgDmi > RFD_FSET_MDATA_BLK_MSG_DMI_MAX_VALUE) {
        // error, requested DMI value is out of range.
        return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
    }

	////////////////////////////////////
	// Initialize Bitstream processing structure to the extension record buffer.
	////////////////////////////////////
    maxExtensionLenBits = RFD_FSET_MSG_MAX_EXT_FIELD_LEN_BYTES * RFD_BITS_PER_BYTE;

	if(RFD_BitstreamInit(&streamInfo,
						 fileSetMetadataMsg->extension,
						 maxExtensionLenBits)) {
		return RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR;
	}

	////////////////////////////////////
	// Write DMI
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_UPDATE_FILES_DMI_FIELD_NUM_BITS, fileSetMetadataMsg->blockMsgDmi)) {
	    return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Write DMI RFU
	////////////////////////////////////
    dWord = 0;
	if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_UPDATE_FILES_DMIRFU_FIELD_NUM_BITS, dWord)) {
	    return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Align to the next byte boundary (should be already, no change if already byte aligned).
	////////////////////////////////////
	if(RFD_BitstreamSeek(&streamInfo, 0, RFD_BSTRM_SEEK_OS_NEXT_BYTE_ALIGN)) {
	    return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

	////////////////////////////////////
    // Get current extension record length.
    // Already byte aligned (from seek above), so no need to accound for bit offset adjust.
	////////////////////////////////////
    currExtRecordLenBytes = RFD_BitstreamGetCurrBitPosition(&streamInfo)/RFD_BITS_PER_BYTE;

	////////////////////////////////////
    // fileSetMetadataMsg->extensionLenBytes is both an input and output parameter.
    //
    // If the current byte position after insterting user requested fields (DMI)
    // is less than the input (user specified) extension length (i.e. user could have
    // inserted fill pattern into extension record, for testing) then
    // adjust the user specified extension length to the current byte position.
	////////////////////////////////////
    if(currExtRecordLenBytes > fileSetMetadataMsg->extensionLenBytes) {

        fileSetMetadataMsg->extensionLenBytes = currExtRecordLenBytes;
    }

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_GenerateBlockMessage(RFD_BLOCK_MSG_STRUCT * blockMsg,
											  UCHAR * bitstreamBuf,
											  UINT32 messageBitstreamLenBytes,
											  RFD_MESSAGE_CONFIG_INFO * configInfo,
											  UINT32 * messageLenBitsPtr)
{
	RFD_BITSTREAM_INFO streamInfo;
	int nBitsBlockIndexField = 0;
	UINT32 messageCrc;
	UINT32 messageLenBytesSemCrc;

	////////////////////////////////////
	// Clear the bitstream buffer so that any skipped fields will be zero initialized.
	////////////////////////////////////
	RFD_MEMSET(bitstreamBuf, 0, messageBitstreamLenBytes);

	////////////////////////////////////
	// Initialize Bitstream processing structure.
	////////////////////////////////////
	if(RFD_BitstreamInit(&streamInfo,
						 bitstreamBuf,
						 messageBitstreamLenBytes * RFD_BITS_PER_BYTE)) {
		return RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR;
	}


	////////////////////////////////////
	// Write PVN
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_PVN_FIELD_NUM_BITS, configInfo->protocolVersionNo)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Write CARID
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, configInfo->nBitsCarouselIdField, configInfo->blockMsgCarouselId)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}


	////////////////////////////////////
	// Write File ID
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_FILE_ID_FIELD_NUM_BITS, blockMsg->fileId)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Write Block Type
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_BLK_TYPE_FIELD_NUM_BITS, blockMsg->blkType)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	// Write Block Index, size and presence dependent on Block Type.
	switch(blockMsg->blkType) {

		case BLK_TYPE_SOURCE:
			// Block Index filed size for Source type block.
			nBitsBlockIndexField = RFD_MSG_SRC_TYPE_BLK_INDEX_FIELD_NUM_BITS;
			break;

		case BLK_TYPE_ECC1_CHECK:
			// Block Index filed size for ECC1 Check type block.
			nBitsBlockIndexField = RFD_MSG_ECC1_CHECK_TYPE_BLK_INDEX_FIELD_NUM_BITS;
			break;

		case BLK_TYPE_ECC2_CHECK:
			// Block Index filed size for ECC2 Check type block.
			nBitsBlockIndexField = RFD_MSG_ECC2_CHECK_TYPE_BLK_INDEX_FIELD_NUM_BITS;
			break;

		case BLK_TYPE_SIMPLE_PARITY:
			// Simple Parity Block does not have a Block Index.
			// Just set to 0.
			nBitsBlockIndexField = 0;
			break;

		default:
			// Unsupported Block Type.
			return RFD_STATUS_UNSUPPORTED_BLOCK_TYPE;
	}

	if(nBitsBlockIndexField != 0) {
		////////////////////////////////////
		// Write the Block Index Field
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, nBitsBlockIndexField, blockMsg->blkIndex)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}
	}
	// else Block Index not implemented for Simple Parity type block.


	////////////////////////////////////
	// Skip PADDING field.
	// Adjust bitstream pointer to the next byte aligned bit postion
	// (no change if already byte aligned).
	////////////////////////////////////
	if(RFD_BitstreamSeek(&streamInfo, 0, RFD_BSTRM_SEEK_OS_NEXT_BYTE_ALIGN)) {
		return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

	////////////////////////////////////
	// Write Extension Field Size.
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_BLK_EXT_FIELD_NUM_BITS, blockMsg->extensionLenBytes)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Write Extension Filed Data.
	////////////////////////////////////
	if(blockMsg->extensionLenBytes != 0) {
		// Write the Extension Field.
		if(RFD_BitstreamPutByteArray(&streamInfo, blockMsg->extensionLenBytes, blockMsg->extension)) {
			return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
		}
	}

	// Check block length before copying block to buffer.
	if(blockMsg->blkLenBytes > configInfo->maxBlockLenBytes) {
		return RFD_STATUS_ERROR_INVALID_BLOCK_LENGTH;
	}

	////////////////////////////////////
	// Write the Block Data into the block buffer.
	////////////////////////////////////
	if(RFD_BitstreamPutByteArray(&streamInfo, blockMsg->blkLenBytes, blockMsg->block)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Calculate the Message CRC
	////////////////////////////////////

	messageLenBytesSemCrc = RFD_BitstreamGetCurrBitPosition(&streamInfo)/RFD_BITS_PER_BYTE;

	RFD_ResetCrc(&messageCrc);
	RFD_CalculateCrc(&messageCrc,
					 bitstreamBuf,
					 messageLenBytesSemCrc);

	////////////////////////////////////
	// Write the Message CRC
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_CRC_FIELD_NUM_BITS, messageCrc)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Output the Length of the generated message
	////////////////////////////////////
	if(messageLenBitsPtr != NULL) {
		*messageLenBitsPtr = RFD_BitstreamGetCurrBitPosition(&streamInfo);
	}

	return RFD_STATUS_OK;
}


///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_GenerateFileSetMetadataMessage(RFD_FSET_MDATA_STRUCT * fileSetMetadataMsg,
											  UCHAR * bitstreamBuf,
											  UINT32 messageBitstreamLenBytes,
											  RFD_MESSAGE_CONFIG_INFO * configInfo,
											  UINT32 * messageLenBitsPtr)
{
	RFD_BITSTREAM_INFO streamInfo;
	UINT32 messageLenBytesSemCrc;
	UINT32 messageCrc;
	int i;
	int nFilesField;
	RFD_STATUS status;

	////////////////////////////////////
	// Clear the bitstream buffer so that any skipped fields will be zero initialized.
	////////////////////////////////////
	RFD_MEMSET(bitstreamBuf, 0, messageBitstreamLenBytes);

	////////////////////////////////////
	// Initialize Bitstream processing structure.
	////////////////////////////////////
	if(RFD_BitstreamInit(&streamInfo,
						 bitstreamBuf,
						 messageBitstreamLenBytes * RFD_BITS_PER_BYTE)) {
		return RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR;
	}

	////////////////////////////////////
	// Write PVN
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_PVN_FIELD_NUM_BITS, configInfo->protocolVersionNo)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Write CARID
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, configInfo->nBitsCarouselIdField, configInfo->fileSetMdataMsgCarouselId)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	//////////////////////////
	// Write the FSET NAME field.
	// The regularly-formatted string stored in fileSetMetadataMsg->name is converted to Baudot format
	// and written to the bitstream.
	//////////////////////////
	status = RFD_ConvertStringToBaudotBitStream( fileSetMetadataMsg->name, RFD_FSET_NAME_MAX_LEN,
									    &streamInfo, RFD_FSET_NAME_MAX_NUM_BAUDOT_SYMBOLS );
	if(status != RFD_STATUS_OK) {
		return status;
	}

	////////////////////////////////////
	// Write FSET SIZE field
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_FSET_SIZE_FIELD_NUM_BITS, fileSetMetadataMsg->totalSize)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

#ifdef DYNAMIC_BLOCK_MSG_CARID_ENABLE
	////////////////////////////////////
	// Write CAR field (Carousel on which Block Messages are received)
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, configInfo->nBitsFileSetUpdateCarField, fileSetMetadataMsg->blockMsgCarouselId)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}
#endif

	////////////////////////////////////
	// Write ALG, the Coding Algorithm.
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_CODING_ALG_FIELD_NUM_BITS, fileSetMetadataMsg->codeType)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	switch(fileSetMetadataMsg->codeType) {

		case RFD_CODE_TYPE_SIMPLE_BLK:
		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:
		// These code types are supported so continue parsing.
			break;

		default:
		// Any other code type is not supported, stop generating.
			return RFD_STATUS_UNSUPPORTED_CODE_TYPE;
	}

	////////////////////////////////////
	// Write COMP, the Compression Algorithm.
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_COMPRESSION_ALG_FIELD_NUM_BITS, fileSetMetadataMsg->compType)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	switch(fileSetMetadataMsg->compType) {

		case RFD_COMP_TYPE_NONE:
		case RFD_COMP_TYPE_GZIP:
		// These compression types are supported, continue parsing.
			break;

		default:
		// Any other compression type is not supported, stop parsing.
			return RFD_STATUS_UNSUPPORTED_COMPRESSION_TYPE;
	}

	////////////////////////////////////
	// Write the LIFE field
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_LIFE_TIME_FIELD_NUM_BITS, fileSetMetadataMsg->lifeTime)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Write the NUM FILES field
	////////////////////////////////////
	nFilesField = fileSetMetadataMsg->numFiles-1;
	if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_NUM_FILES_FIELD_NUM_BITS, nFilesField)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	if(fileSetMetadataMsg->numFiles > RFD_FSET_MAX_NUM_FILES) {
		return RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES;
	}

	for(i=0;i<fileSetMetadataMsg->numFiles;i++) {

		////////////////////////////////////
		// Write the File Id field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_FILE_ID_FIELD_NUM_BITS, fileSetMetadataMsg->fileInfo[i].id)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}

		////////////////////////////////////
		// Write the Block Length field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_BLK_LEN_FIELD_NUM_BITS, fileSetMetadataMsg->fileInfo[i].blkLenBytes)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}

		if(fileSetMetadataMsg->fileInfo[i].blkLenBytes > configInfo->maxBlockLenBytes) {
			return RFD_STATUS_ERROR_INVALID_BLOCK_LENGTH;
		}

#ifdef RFD_FSET_MESSAGE_INCLUDE_FILE_SIZES_ENABLE
		////////////////////////////////////
		// Write the File Size field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_FILE_SIZE_FIELD_NUM_BITS, fileSetMetadataMsg->fileInfo[i].size)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}
#endif
		if(fileSetMetadataMsg->fileInfo[i].size > configInfo->maxFileLenBytes) {
			return RFD_STATUS_ERROR_INVALID_FILE_SIZE;
		}


		////////////////////////////////////
		// Write the File CRC field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_FILE_CRC_FIELD_NUM_BITS, fileSetMetadataMsg->fileInfo[i].crc)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}
	}

	////////////////////////////////////
	// Generate (update) the extension record.
	////////////////////////////////////
    RFD_GenerateFileSetMetadataExtensionRecord(fileSetMetadataMsg);

	////////////////////////////////////
	// Write the F+EXTL field
	////////////////////////////////////
	if(fileSetMetadataMsg->extensionLenBytes == 0) {
		// No Extension data

		////////////////////////////////////
		// Write the F+ bit = 0
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_PRESENCE_FLAG_FIELD_NUM_BITS, RFD_MGS_FIELD_IS_PRESENT_FALSE)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}
	}
	else { // Extension data is present

		////////////////////////////////////
		// Write the F+ bit = 1
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_PRESENCE_FLAG_FIELD_NUM_BITS, RFD_MGS_FIELD_IS_PRESENT_TRUE)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}

		////////////////////////////////////
		// Write the EXT Length
		////////////////////////////////////
		if(RFD_BitstreamPutU32(&streamInfo, RFD_FSET_MSG_EXT_LENGTH_FIELD_NUM_BITS, fileSetMetadataMsg->extensionLenBytes-1)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}

		////////////////////////////////////
		// Write the EXT Data
		////////////////////////////////////
		if(RFD_BitstreamPutByteArray(&streamInfo, fileSetMetadataMsg->extensionLenBytes, fileSetMetadataMsg->extension)) {
			return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
		}
	}

	////////////////////////////////////
	// Skip Padding/Bit Stuffing field.
	// Adjust bitstream pointer to the next byte aligned bit postion
	// (no change if already byte aligned).
	////////////////////////////////////
	if(RFD_BitstreamSeek(&streamInfo, 0, RFD_BSTRM_SEEK_OS_NEXT_BYTE_ALIGN)) {
		return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

	////////////////////////////////////
	// Calculate the Message CRC
	////////////////////////////////////

	messageLenBytesSemCrc = RFD_BitstreamGetCurrBitPosition(&streamInfo)/RFD_BITS_PER_BYTE;

	RFD_ResetCrc(&messageCrc);
	RFD_CalculateCrc(&messageCrc,
					 bitstreamBuf,
					 messageLenBytesSemCrc);

	////////////////////////////////////
	// Write the Message CRC
	////////////////////////////////////
	if(RFD_BitstreamPutU32(&streamInfo, RFD_MSG_CRC_FIELD_NUM_BITS, messageCrc)) {
		return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
	}

	////////////////////////////////////
	// Output the Length of the generated message
	////////////////////////////////////
	if(messageLenBitsPtr != NULL) {
		*messageLenBitsPtr = RFD_BitstreamGetCurrBitPosition(&streamInfo);
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_GenerateMessage(RFD_MESSAGE_INFO * messageInfo, UINT32 * messageLenBitsPtr)
{
	RFD_STATUS status;

	switch(messageInfo->messageType) {

		case MSG_TYPE_FSET_MDATA:
			status = RFD_GenerateFileSetMetadataMessage(&messageInfo->message.fileSetInfoMsg,
														messageInfo->bitstreamBuf,
														messageInfo->messageBitstreamLenBytes,
														&messageInfo->configInfo,
														messageLenBitsPtr);
			break;

		case MSG_TYPE_BLOCK:
			status = RFD_GenerateBlockMessage(&messageInfo->message.blockMsg,
											  messageInfo->bitstreamBuf,
											  messageInfo->messageBitstreamLenBytes,
										 	  &messageInfo->configInfo,
											  messageLenBitsPtr);
			break;

		default:
			status = RFD_STATUS_ERROR_INVALID_MESSAGE_TYPE;
	}

	return status;
}