////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_file_io.c
///
/// @brief	rfd file i/o class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"

///////////////////////////////////////////////////////////////////////////////
// version that calls RFD_DeleteDirectoryContentsNonRecursive()
// and then deletes the requested directory.
BOOL RFD_DeleteDirectoryNonRecursive(const TCHAR* sPath)
{
	if(sPath == NULL) {
		return FALSE;
	}

	if(RFD_DeleteDirectoryContentsNonRecursive(sPath, TRUE /* includeSubDirRemoval */)) {
		if(RFD_REMOVE_DIR(sPath)) { // remove the empty directory
			// successful remove directory.
			return TRUE;
		}
		else {
			// remove directory failed.
			return FALSE;
		}
	}
	else {
		return FALSE; //error
	}
}

///////////////////////////////////////////////////////////////////////////////
BOOL RFD_DeleteDirectoryContentsNonRecursive(const TCHAR* sPath, BOOL includeSubDirRemoval)
{
	RFD_FIND_FILE_DATA_INFO fileData;
	RFD_FIND_FILE_HANDLE hSearch;
	TCHAR * findDataFileName;
	BOOL isError;
	BOOL doSearch = TRUE;
	TCHAR * fileName;

	if(sPath == NULL) {
		return FALSE;
	}

	if(NULL == (fileName = (TCHAR *) RFD_MALLOC(  sizeof(TCHAR) * RFD_MAX_PATH_LEN ))) {
		return FALSE;
	}

	// find the first file entry in the directory.
	hSearch = RFD_FindFirstFile(sPath, &fileData, &isError);
	if(hSearch == NULL) {
		RFD_FREE(fileName);

		if(isError) {
			return FALSE;
		}
		else {
			// search handle is null due to empty directory, not error,
			// return success indication.
			return TRUE;
		}
	}

	while(doSearch) { // until no entry found

		findDataFileName = RFD_FindDataGetFileNamePtr(&fileData);

		if(findDataFileName != NULL && !RFD_IsReservedFileSystemFileName(findDataFileName)) {

			RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, sPath);
			RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, findDataFileName);

			if(RFD_FindDataIsItemADirectory(&fileData)) {

				if(includeSubDirRemoval) {
#if 0 // remove recursive component
					// we have found a directory, recurse
					if(RFD_REMOVE_DIR(fileName)) {
						RFD_FREE(fileName);
						RFD_FindClose(hSearch);
						return FALSE;    // directory couldn't be deleted
					}
#endif
					// try to remove directory (may fail if not empty, and WIN32 may abort process)
					RFD_REMOVE_DIR(fileName);
				}
			}
			else {
#if 0 // remove check for read-only
				if(RFD_FindDataIsItemReadOnly(&fileData)
					// change read-only file mode
					_chmod(fileName, _S_IWRITE);
#endif
				// delete the file
				if(RFD_DELETE_FILE(fileName))
				{
					// File Delete error. Don't stop, keep trying to delete other files.
				}
			}
		}

		if(!RFD_FindNextFile(hSearch,&fileData, &isError)) {
			// no entry found. Either no more entries or error.

			// Close the search.
			RFD_FindClose(hSearch);

			if(!isError) {
				// Not an error, no more entries, so stop searching.
				doSearch = FALSE;
			}
			else {
				// some error occurred; cleanup and return FALSE
				RFD_FREE(fileName);
				return FALSE;
			}
		}
	}

	// free memory
	RFD_FREE(fileName);

	return TRUE;
}

///////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Delete directory recursively.
///
/// @param	sPath					Full pathname of the directory.
/// @param	deleteDirContentsOnly	true to delete the dir contents only, not the directory itsef.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_DeleteDirectoryRecursive(const TCHAR* sPath, BOOL deleteDirContentsOnly )
{
	RFD_FIND_FILE_DATA_INFO fileData;
	RFD_FIND_FILE_HANDLE hSearch;
	TCHAR * findDataFileName;
	BOOL isError;
	BOOL doSearch = TRUE;
	TCHAR * fileName;

	if(sPath == NULL) {
		return FALSE;
	}

	fileName = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

	if(fileName == NULL) {
		return FALSE;
	}

	// find the first file
	hSearch = RFD_FindFirstFile(sPath, &fileData, &isError);

	if(hSearch == NULL) {
		RFD_FREE(fileName);
		if(isError) {
			return FALSE;
		}
		else {
			// search handle is null due to empty directory, not error,
			// return success indication.
			return TRUE;
		}
	}

	while(doSearch) { // until no entry found

		findDataFileName = RFD_FindDataGetFileNamePtr(&fileData);

		if(findDataFileName != NULL && !RFD_IsReservedFileSystemFileName(findDataFileName)) {
			// regular directory entry, so try to delete.

			RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, sPath);
			RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, findDataFileName);

			if(RFD_FindDataIsItemADirectory(&fileData)) {

				// found a directory, recurse
				if(!RFD_DeleteDirectoryRecursive(fileName, FALSE /* delete this sub-dir, not just contents */ )) {
					RFD_FREE(fileName);
					RFD_FindClose(hSearch);
					return FALSE;    // directory couldn't be deleted
				}
			}
			else
			{
#if 0 // remove check for read-only
				if(RFD_FindDataIsItemReadOnly(&fileData)
					// change read-only file mode
					_chmod(fileName, _S_IWRITE);
#endif
				// delete the file
				if(RFD_DELETE_FILE(fileName)) {
					// File Delete error. Don't stop, keep trying to delete other files.
				}
			}
		}
		// else this a special-case directory entry.

		// Find next entry.
		if(!RFD_FindNextFile(hSearch,&fileData, &isError)) {

			// no entry found. Either no more entries or error.

			// Close the search.
			RFD_FindClose(hSearch);

			if(!isError) {
				// Not an error, no more entries, so remove the direcotory.

				doSearch = FALSE;
				if(!deleteDirContentsOnly) {
					// Delete the 'base' directory also.
					if(RFD_REMOVE_DIR(sPath)) {
						RFD_FREE(fileName);
						return FALSE;    // directory couldn't be deleted
					}
				}
				// else do not delete the 'base' dir (only the contents of the base directory).
			}
			else {
				// some error occurred; cleanup and return FALSE
				RFD_FREE(fileName);
				return FALSE;
			}
		}
		// else entry found, continue processing.
	}

	// close the file handle and free memory
	RFD_FREE(fileName);

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Copy files from one directory to another, skipping the copy of a specified single
/// 		file.
///
/// @param	dstPathBuf		destination directory path name string.
/// @param	srcPathBuf		source directory path name string.
/// @param	skipFileName	Filename string for the specified file to skip copy of. NULL value
/// 						indicates no specified file to skip.
/// @param	doFileSync		Perform file sync for each copied (destination) file.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CopyDirectoryFilesSkipSingleFile(const TCHAR dstPathBuf[], const TCHAR srcPathBuf[], const TCHAR skipFileName[], BOOL doFileSync)
{
	RFD_FIND_FILE_DATA_INFO fileData;
	RFD_FIND_FILE_HANDLE hSearch;
	TCHAR * findDataFileName;
	BOOL isError;
	BOOL doSearch = TRUE;
	TCHAR * srcFileName;
	TCHAR * dstFileName;
	RFD_STATUS status;

	if(dstPathBuf == NULL || srcPathBuf == NULL || skipFileName == NULL) {
		return FALSE;
	}

	if(NULL == (srcFileName = (TCHAR *) RFD_MALLOC(  sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return FALSE;
	}

	if(NULL == (dstFileName = (TCHAR *) RFD_MALLOC(  sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		RFD_FREE(srcFileName);
		return FALSE;
	}

	// find the first file
	hSearch = RFD_FindFirstFile(srcPathBuf, &fileData, &isError);
	if(hSearch == NULL) {
		RFD_FREE(srcFileName);
		RFD_FREE(dstFileName);
		if(isError) {
			return FALSE;
		}
		else {
			// search handle is null due to empty directory, not error,
			// return success indication.
			return TRUE;
		}
	}

	while(doSearch) {

		findDataFileName = RFD_FindDataGetFileNamePtr(&fileData);

		if(findDataFileName != NULL && !RFD_IsReservedFileSystemFileName(findDataFileName)) {

			if(!RFD_FindDataIsItemADirectory(&fileData)) {
				// found object is not a directory, so it is a file.

				if(skipFileName != NULL && RFD_STRCMP(skipFileName, findDataFileName)) {

					// The found file does not match the specified file to skip,
					// so copy the found file.

					// Construct the full source file name with path
					RFD_STRCPY_S(srcFileName, RFD_MAX_PATH_LEN, srcPathBuf);
					RFD_STRCAT_S(srcFileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					RFD_STRCAT_S(srcFileName, RFD_MAX_PATH_LEN, findDataFileName);

					// Construct the full destination file name with path
					RFD_STRCPY_S(dstFileName, RFD_MAX_PATH_LEN, dstPathBuf);
					RFD_STRCAT_S(dstFileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					RFD_STRCAT_S(dstFileName, RFD_MAX_PATH_LEN, findDataFileName);

					// Copy the file
					status = RFD_CopyFile(dstFileName, srcFileName, doFileSync);
					if(status != RFD_STATUS_OK) {
						RFD_FindClose(hSearch);
						RFD_FREE(srcFileName);
						RFD_FREE(dstFileName);
						return FALSE;
					}
				}
			}
		}

		if(!RFD_FindNextFile(hSearch,&fileData, &isError)) {

			// no more files

			if(!isError) {
				doSearch = FALSE;
			}
			else {
				// some error occurred; close the handle and return FALSE
				RFD_FindClose(hSearch);
				RFD_FREE(srcFileName);
				RFD_FREE(dstFileName);
				return FALSE;
			}
		}
	}

	// close the file handle and free memory
	RFD_FindClose(hSearch);
	RFD_FREE(srcFileName);
	RFD_FREE(dstFileName);

	return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
int RFD_GetNumFilesInDirectory(const TCHAR* sPath, BOOL includeDirs)
{
	RFD_FIND_FILE_DATA_INFO fileData;
	RFD_FIND_FILE_HANDLE hSearch;
	TCHAR * findDataFileName;
	BOOL isError;
	int nFiles = 0;
	BOOL doSearch = TRUE;

	if(sPath == NULL) {
		return -1;
	}

	// find the first file
	hSearch = RFD_FindFirstFile(sPath, &fileData, &isError);

	if(hSearch == NULL) {
		if(isError) {
			return -1;
		}
		else {
			// search handle is null due to empty directory, not error,
			// return number-of-files = 0.
			return 0;
		}
	}

	while(doSearch) {

		findDataFileName = RFD_FindDataGetFileNamePtr(&fileData);

		if(findDataFileName != NULL && !RFD_IsReservedFileSystemFileName(findDataFileName)) {

			if(RFD_FindDataIsItemADirectory(&fileData)) {

				if(includeDirs) {
					// include directories in the count.
					nFiles++;
				}
			}
			else {
				nFiles++;
			}
		}

		if(!RFD_FindNextFile(hSearch,&fileData, &isError)) {
			// no more files

			if(!isError) {
				doSearch = FALSE;
			}
			else {
				// some error occurred; close the handle and return error indication (-1)
				RFD_FindClose(hSearch);
				return -1;
			}
		}
	}

	// close the file handle and return
	RFD_FindClose(hSearch);

	return nFiles;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ReadFile(const TCHAR pathBuf[], void * pData, INT32 DataSize, INT32 * pActualReadDataSize, BOOL CheckFileSize)
{
	RFD_FILE *hFile;
	*pActualReadDataSize = 0;

	if(pathBuf == NULL || pData == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	if(RFD_FOPEN_S(&hFile, pathBuf, TEXT("rb")) || hFile == NULL)
	{
		//RFD_DPrint( TEXT("Error Creating File: %s \n"),pathBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Read file
	*pActualReadDataSize = (INT32) RFD_FREAD(pData, 1, DataSize, hFile);
	if(*pActualReadDataSize != DataSize)
	{
		//RFD_DPrint( TEXT("Error Reading File: %s \n"),pathBuf);
		RFD_FCLOSE(hFile);
		return RFD_STATUS_ERROR_FILE_READ;
	}

	if(CheckFileSize)
	{
		// Check if file file size matches the requested data size.
		if(RFD_FSEEK(hFile, 0, SEEK_END))
		{
			RFD_FCLOSE(hFile);
			return RFD_STATUS_ERROR_FILE_SEEK;
		}

		if(RFD_FTELL(hFile) != DataSize)
		{
			RFD_FCLOSE(hFile);
			return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
		}
	}

	RFD_FCLOSE(hFile);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CreateFile(TCHAR pathBuf[], BOOL doFileSync)
{
	RFD_FILE *hFile;

	if(pathBuf == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	////////////////////////////////////
	// Open then close file to create the file.
	////////////////////////////////////

	if(RFD_FOPEN_S(&hFile, pathBuf, TEXT("wb")) || hFile == NULL)
	{
		//RFD_DPrint( TEXT("Error Creating File: %s \n"),pathBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	if(doFileSync) {
		int r;
		RFD_STATUS fsyncStatus;

		// Must fflush first to flush upper layer buffers, before file sync
		r = RFD_FFLUSH(hFile);
		if(r) {
			// print error condition, but continue
			RFD_DPrint( TEXT("RFD File I/O: File Flush Error (%d, %s)\n"), r, pathBuf);
		}

		fsyncStatus = RFD_FileSyncByHandle(hFile);
		if(fsyncStatus != RFD_STATUS_OK) {
			// print error condition, but continue
			RFD_DPrint( TEXT("RFD File I/O: File Sync Error (%d, %s)\n"), fsyncStatus, pathBuf);
		}
	}

	RFD_FCLOSE(hFile);

	return RFD_STATUS_OK;
}


///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_WriteFile(const TCHAR pathBuf[], void * pData, INT32 DataSize, BOOL doFileSync)
{
	RFD_FILE *hFile;
	INT32 ActualWriteDataSize = 0;

	if(pathBuf == NULL || pData == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	if(RFD_FOPEN_S(&hFile, pathBuf, TEXT("wb")) || hFile == NULL )
	{
		//RFD_DPrint( TEXT("Error Creating File: %s \n"),pathBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Write file
	ActualWriteDataSize = (INT32) RFD_FWRITE(pData, 1, DataSize, hFile);
	if(ActualWriteDataSize != DataSize)
	{
		//RFD_DPrint( TEXT("Error Writing File: %s \n"),pathBuf);
		RFD_FCLOSE(hFile);
		return RFD_STATUS_ERROR_FILE_WRITE;
	}

	if(doFileSync) {
		int r;
		RFD_STATUS fsyncStatus;

		// Must fflush first to flush upper layer buffers, before file sync
		r = RFD_FFLUSH(hFile);
		if(r) {
			// print error condition, but continue
			RFD_DPrint( TEXT("RFD File I/O: File Flush Error (%d, %s)\n"), r, pathBuf);
		}

		fsyncStatus = RFD_FileSyncByHandle(hFile);
		if(fsyncStatus != RFD_STATUS_OK) {
			// print error condition, but continue
			RFD_DPrint( TEXT("RFD File I/O: File Sync Error (%d, %s)\n"), fsyncStatus, pathBuf);
		}
	}

	RFD_FCLOSE(hFile);

	return RFD_STATUS_OK;
}


///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_AppendDataToFile(TCHAR pathBuf[], void * pData, INT32 DataSize, BOOL doFileSync)
{
	RFD_FILE *hFile;
	INT32 ActualWriteDataSize = 0;

	if(pathBuf == NULL || pData == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	if(RFD_FOPEN_S(&hFile, pathBuf, TEXT("ab")) || hFile == NULL)
	{
		//RFD_DPrint( TEXT("Error Opening File: %s \n"),pathBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Write file
	ActualWriteDataSize = (INT32) RFD_FWRITE(pData, 1, DataSize, hFile);
	if(ActualWriteDataSize != DataSize)
	{
		//RFD_DPrint( TEXT("Error Writing File: %s \n"),pathBuf);
		RFD_FCLOSE(hFile);
		return RFD_STATUS_ERROR_FILE_WRITE;
	}

	if(doFileSync) {
		int r;
		RFD_STATUS fsyncStatus;

		// Must fflush first to flush upper layer buffers, before file sync
		r = RFD_FFLUSH(hFile);
		if(r) {
			// print error condition, but continue
			RFD_DPrint( TEXT("RFD File I/O: File Flush Error (%d, %s)\n"), r, pathBuf);
		}

		fsyncStatus = RFD_FileSyncByHandle(hFile);
		if(fsyncStatus != RFD_STATUS_OK) {
			// print error condition, but continue
			RFD_DPrint( TEXT("RFD File I/O: File Sync Error (%d, %s)\n"), fsyncStatus, pathBuf);
		}
	}

	RFD_FCLOSE(hFile);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_AppendFiles(TCHAR srcFileNameBuf[], TCHAR dstFileNameBuf[], BOOL doFileSync)
{
	RFD_FILE * dstFileHandle = NULL;
	RFD_FILE * srcFileHandle = NULL;
	UCHAR * dataBuf = NULL;
	INT32 actualReadDataSize;
	UINT32 actualWriteDataSize;
	INT32 dataBufSize;
	BOOL isFileAppendComplete;
	BOOL isFileWriteError;
	RFD_STATUS status;

	if(dstFileNameBuf == NULL || srcFileNameBuf == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	dataBufSize = RFD_APPEND_FILES_BUF_LEN;

	dataBuf = (UCHAR *) RFD_MALLOC( sizeof(UCHAR) * dataBufSize );
	if(dataBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Open the source file for read.
	if(RFD_FOPEN_S(&srcFileHandle,srcFileNameBuf, TEXT("rb")) || srcFileHandle == NULL) {
		// Error.
		RFD_FREE(dataBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Open the destination file for append.
	if(RFD_FOPEN_S(&dstFileHandle, dstFileNameBuf, TEXT("ab")) || dstFileHandle == NULL) {
		// Error.
		RFD_FCLOSE(srcFileHandle);
		RFD_FREE(dataBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Read from source file to buffer and
	// write from buffer to destination file.
	// until end of source file is reached.

	isFileWriteError = FALSE;
	isFileAppendComplete = FALSE;

	while(!isFileAppendComplete) {

		// Read next section of the source file into buffer
		actualReadDataSize = (INT32) RFD_FREAD(dataBuf, 1, dataBufSize, srcFileHandle);

		if(actualReadDataSize > 0) {
			// Write the buffer to destination file.
			actualWriteDataSize = (UINT32) RFD_FWRITE(dataBuf, 1, actualReadDataSize, dstFileHandle);

			// Check for file write error.
			if(actualWriteDataSize != (UINT32)actualReadDataSize) {
				isFileWriteError = TRUE;
				isFileAppendComplete = TRUE;
			}
		}

		// Check for end of read file (or read error)
		if(actualReadDataSize < dataBufSize) {
			isFileAppendComplete = TRUE;
		}
	}

	// Check for errors.
	if(isFileWriteError) {
		status = RFD_STATUS_ERROR_FILE_WRITE;
	}
	else if(!RFD_FEOF(srcFileHandle)) {
		// Error is read file is not at EOF.
		status = RFD_STATUS_ERROR_FILE_READ;
	}
	else {
		status = RFD_STATUS_OK;

		// Successful operation,
		// now do File Sync of destination file if sync requested.
		if(doFileSync) {
			int r;
			RFD_STATUS fsyncStatus;

			// Must fflush first to flush upper layer buffers, before file sync
			r = RFD_FFLUSH(dstFileHandle);
			if(r) {
				// print error condition, but continue
				RFD_DPrint( TEXT("RFD File I/O: File Flush Error (%d, %s)\n"), r, dstFileNameBuf);
			}

			fsyncStatus = RFD_FileSyncByHandle(dstFileHandle);
			if(fsyncStatus != RFD_STATUS_OK) {
				// print error condition, but continue
				RFD_DPrint( TEXT("RFD File I/O: File Sync Error (%d, %s)\n"), fsyncStatus, dstFileNameBuf);
			}
		}
	}

	// Close the source file.
	RFD_FCLOSE(srcFileHandle);
	// Close the destination file.
	RFD_FCLOSE(dstFileHandle);
	// Free the data transfer buffer
	RFD_FREE(dataBuf);

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CopyFile(const TCHAR dstFileNameBuf[], const TCHAR srcFileNameBuf[], BOOL doFileSync)
{
	RFD_FILE * dstFileHandle = NULL;
	RFD_FILE * srcFileHandle = NULL;
	UCHAR * dataBuf = NULL;
	INT32 actualReadDataSize;
	UINT32 actualWriteDataSize;
	INT32 dataBufSize;
	BOOL isFileCopyComplete;
	BOOL isFileWriteError;
	RFD_STATUS status;

	if(dstFileNameBuf == NULL || srcFileNameBuf == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	dataBufSize = RFD_COPY_FILES_BUF_LEN;

	dataBuf = (UCHAR *) RFD_MALLOC( sizeof(UCHAR) * dataBufSize );
	if(dataBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Open the source file for read.
	if(RFD_FOPEN_S(&srcFileHandle,srcFileNameBuf, TEXT("rb")) || srcFileHandle == NULL) {
		// Error.
		RFD_FREE(dataBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Open the destination file for write.
	if(RFD_FOPEN_S(&dstFileHandle, dstFileNameBuf, TEXT("wb")) || dstFileHandle == NULL) {
		// Error.
		RFD_FCLOSE(srcFileHandle);
		RFD_FREE(dataBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Read from source file to buffer and
	// write from buffer to destination file.
	// until end of source file is reached.

	isFileWriteError = FALSE;
	isFileCopyComplete = FALSE;

	while(!isFileCopyComplete) {

		// Read next section of the source file into buffer
		actualReadDataSize = (INT32) RFD_FREAD(dataBuf, 1, dataBufSize, srcFileHandle);

		if(actualReadDataSize > 0) {
			// Write the buffer to destination file.
			actualWriteDataSize = (UINT32) RFD_FWRITE(dataBuf, 1, actualReadDataSize, dstFileHandle);

			// Check for file write error.
			if(actualWriteDataSize != (UINT32)actualReadDataSize) {
				isFileWriteError = TRUE;
				isFileCopyComplete = TRUE;
			}
		}

		// Check for end of read file (or read error)
		if(actualReadDataSize < dataBufSize) {
			isFileCopyComplete = TRUE;
		}
	}

	// Check for errors.
	if(isFileWriteError) {
		status = RFD_STATUS_ERROR_FILE_WRITE;
	}
	else if(!RFD_FEOF(srcFileHandle)) {
		// Error is read file is not at EOF.
		status = RFD_STATUS_ERROR_FILE_READ;
	}
	else {
		status = RFD_STATUS_OK;

		// Successful operation,
		// now do File Sync of destination file if sync requested.
		if(doFileSync) {
			int r;
			RFD_STATUS fsyncStatus;

			// Must fflush first to flush upper layer buffers, before file sync
			r = RFD_FFLUSH(dstFileHandle);
			if(r) {
				// print error condition, but continue
				RFD_DPrint( TEXT("RFD File I/O: File Flush Error (%d, %s)\n"), r, dstFileNameBuf);
			}

			fsyncStatus = RFD_FileSyncByHandle(dstFileHandle);
			if(fsyncStatus != RFD_STATUS_OK) {
				// print error condition, but continue
				RFD_DPrint( TEXT("RFD File I/O: File Sync Error (%d, %s)\n"), fsyncStatus, dstFileNameBuf);
			}
		}
	}

	// Close the source file.
	RFD_FCLOSE(srcFileHandle);
	// Close the destination file.
	RFD_FCLOSE(dstFileHandle);
	// Free the data transfer buffer
	RFD_FREE(dataBuf);

	return status;
}
