////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_bitstream_io.c
///
/// @brief	rfd bitstream i/o class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_bitstream_io.h"


///////////////////////////////////////////////////////////////////////////////
int RFD_BitstreamInit(RFD_BITSTREAM_INFO * streamInfo, UCHAR streamBuf[], INT32 streamBitLen)
{
	streamInfo->currBitOffset = 0;
	streamInfo->currByteOffset = 0;
	streamInfo->streamBitLen = streamBitLen;
	streamInfo->streamBuf = streamBuf;

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
int RFD_BitstreamSeek(RFD_BITSTREAM_INFO * streamInfo, INT32 nBits, RFD_BITSTREAM_SEEK_OFFSET_TYPE seekType)
{
	INT32 currByteOffset;
	int   currBitOffset;
	int byteOffsetIncrement;

	switch(seekType) {

	case RFD_BSTRM_SEEK_OS_BEGINING:

		byteOffsetIncrement = (0 + nBits) / RFD_BITS_PER_BYTE;
		currByteOffset = 0 + byteOffsetIncrement;
		currBitOffset = (0 + nBits) % RFD_BITS_PER_BYTE;
		break;

	case RFD_BSTRM_SEEK_OS_END:

		byteOffsetIncrement = (streamInfo->streamBitLen + nBits) / RFD_BITS_PER_BYTE;
		currByteOffset = 0 + byteOffsetIncrement;
		currBitOffset = (streamInfo->streamBitLen + nBits) % RFD_BITS_PER_BYTE;
		break;

	case RFD_BSTRM_SEEK_OS_CURRENT:

		byteOffsetIncrement = (streamInfo->currBitOffset + nBits) / RFD_BITS_PER_BYTE;
		currByteOffset = streamInfo->currByteOffset + byteOffsetIncrement;
		currBitOffset = (streamInfo->currBitOffset + nBits) % RFD_BITS_PER_BYTE;
		break;

	case RFD_BSTRM_SEEK_OS_NEXT_BYTE_ALIGN:

		// First byte align the current offsets.
		// If offset is currently byte aligned then offset is not changed.
		currByteOffset = streamInfo->currByteOffset + (streamInfo->currBitOffset ? 1:0);
		currBitOffset = 0;

		// Next adjust offset for requested nBits
		byteOffsetIncrement = (currBitOffset + nBits) / RFD_BITS_PER_BYTE;
		currByteOffset += byteOffsetIncrement;
		currBitOffset = (currBitOffset + nBits) % RFD_BITS_PER_BYTE;
		break;

	default:
		// Error, undefined seek offset type.
		return -1;
	}

	// Convert bit offset from negative to positive.
	if(currBitOffset < 0) {
		currByteOffset -= 1;
		currBitOffset += RFD_BITS_PER_BYTE;
	}

	// Error check offsets.
	if( (currBitOffset +
		 currByteOffset * RFD_BITS_PER_BYTE) > streamInfo->streamBitLen ) {
		// Error, seek advances past end of buffer.
		return -1;
	}
	else if( (currBitOffset +
		      currByteOffset * RFD_BITS_PER_BYTE) < 0 ) {
		// Error, seek advances past begining of buffer.
		return -1;
	}

	// Update offsets and return.
	streamInfo->currByteOffset = currByteOffset;
	streamInfo->currBitOffset = currBitOffset;

	return 0;
}


///////////////////////////////////////////////////////////////////////////////
int RFD_BitstreamPutByteArray(RFD_BITSTREAM_INFO * streamInfo, int nBytes, UCHAR byteBuf[])
{
	int i;
	int   firstBitOffset, firstBitOffsetInverse;
	int   lastBitOffset, lastBitOffsetInverse;
	int nBytesDst, nBytesSrc;
	UCHAR * pSrc, * pDst;
	UCHAR src1, src2, dst;

	if( (nBytes * RFD_BITS_PER_BYTE +
		 streamInfo->currBitOffset +
		 streamInfo->currByteOffset * RFD_BITS_PER_BYTE) > streamInfo->streamBitLen ) {
		// Error, write advances past end of buffer.
		return -1;
	}

	// Note: byteBuf[nBytes-1] corresponds to the last bit of the bitstream segment being written
	// byteBuf[0] corresponds to the first byte of the bitstream segment.

	// Calculate number of bytes to read (src) and write (dst).
	nBytesSrc = nBytes;
	nBytesDst = nBytes + (streamInfo->currBitOffset ? 1:0);

	// Calculate bitstream bit offset relative to the first bit of the
	// bitstream segment being written.
	firstBitOffset = streamInfo->currBitOffset;
	firstBitOffsetInverse = RFD_BITS_PER_BYTE - firstBitOffset;

	// Calculate bitstream bit offset relative to the last bit of the
	// bitstream segment being written.
	lastBitOffset = firstBitOffset;
	lastBitOffsetInverse = firstBitOffsetInverse;

	// Initialize poiners to source and destination.
	pDst = streamInfo->streamBuf + streamInfo->currByteOffset;
	pSrc = byteBuf;

	if(lastBitOffset != 0) {
		// Last bit of bitstream segment is Not aligned to buffer byte boundary.
		// Need to combine shifted versions of source bytes to compose ouput bytes.

		src1 = 0;

		for(i=0;i<nBytesDst;i++) {

			if(i != nBytesDst-1) {
				src2 = *pSrc++;
			}
			else {
				// Last byte, only src1 contributes to the last destination byte.
				src2 = 0;
			}

			// clear any previous non-zero bits in the bitstream segment being written.
			if( (i != 0) && (i != nBytesDst-1) ) {
				// one of the 'middle' bytes (not first of last byte)
				dst = 0;
			}
			else {
				// Read current content of destination byte and clear bits
				// corresponding to the bitstream segment being written.
				dst = *pDst;
				if( (i == 0) && (i == nBytesDst-1) ) {
					// first and last byte (nBytesDst = 1)
					dst &= ((0xff << firstBitOffsetInverse) | (0xff >> lastBitOffset));
				}
				else if(i == 0) {
					// first byte only
					dst &= (0xff << firstBitOffsetInverse);
				}
				else { //if(i == nBytesDst-1)
					// last byte only
					dst &= (0xff >> lastBitOffset);
				}
			}

			dst |= (src1 << lastBitOffsetInverse) + (src2 >> lastBitOffset);
			*pDst++ = dst;
			src1 = src2;
		}
	}
	else {
		// Last bit is byte aligned,
		// do simpler/faster copy without bit shifting.

		for(i=0;i<nBytesDst;i++) {
			*pDst++ = *pSrc++;
		}
	}

	// Update offsets.
	// streamInfo->currBitOffset is unchachanged.
	streamInfo->currByteOffset += nBytes;

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
int RFD_BitstreamGetByteArray(RFD_BITSTREAM_INFO * streamInfo, int nBytes, UCHAR byteBuf[])
{
	int i;
	int   bitOffset, bitOffsetInverse;
	int nBytesDst;
	UCHAR * pSrc, * pDst;
	UCHAR src1, src2;

	if( (nBytes * RFD_BITS_PER_BYTE +
		 streamInfo->currBitOffset +
		 streamInfo->currByteOffset * RFD_BITS_PER_BYTE) > streamInfo->streamBitLen ) {
		// Error, read advances past end of buffer.
		return -1;
	}

	// Calculate bitstream bit offset relative to the last bit of the
	// bitstream segment being read.
	bitOffset = streamInfo->currBitOffset;
	bitOffsetInverse = RFD_BITS_PER_BYTE - bitOffset;

	nBytesDst = nBytes;
	pDst = byteBuf;
    pSrc = streamInfo->streamBuf + streamInfo->currByteOffset;

	if(bitOffset != 0) {

		src1 = *pSrc++;

		for(i=0;i<nBytesDst;i++) {
			src2 = *pSrc++;
			*pDst++ = (src1 << bitOffset) + (src2 >> bitOffsetInverse);
			src1 = src2;
		}
	}
	else {
		// last bit is byte aligned,
		// do simpler/faster copy without bit shifting.
		for(i=0;i<nBytesDst;i++) {
			*pDst++ = *pSrc++;
		}
	}

	// Note: byteBuf[nBytesDst-1] contains the last bit of the bitstream segment being read
	// (last bit aligned to the lsb of byteBuf[nBytesDst-1])
	// byteBuf[0] contains the first bit of the bitstream segment.

	// Update offsets.
	// streamInfo->currBitOffset is unchachanged.
	streamInfo->currByteOffset += nBytes;

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
int RFD_BitstreamPutU32(RFD_BITSTREAM_INFO * streamInfo, int nBits, UINT32 dWord)
{

	int i;
	int   firstBitOffset, firstBitOffsetInverse;
	int   lastBitOffset, lastBitOffsetInverse;
	UCHAR byteBuf[RFD_BYTES_PER_DWORD];
	int nBytesDst, nBytesSrc;
	UCHAR * pSrc, * pDst;
	UCHAR src1, src2, dst;
	int byteOffsetIncrement;

	if(nBits > RFD_BITS_PER_DWORD) {
		// Error, requested read size larger than dword.
		return -1;
	}

	if( (nBits + streamInfo->currBitOffset +
		streamInfo->currByteOffset * RFD_BITS_PER_BYTE) > streamInfo->streamBitLen ) {
		// Error, write advances past end of buffer.
		return -1;
	}

	// Clear byteBuf[].
	for(i=0;i<RFD_BYTES_PER_DWORD;i++) {
		byteBuf[i] = 0;
	}

	// Note: byteBuf[3] corresponds to the last bit of the bitstream segment being written
	// (last bit aligned to the lsb of byteBuf[3])
	// (when nBytesSrc = 4, msb of byteBuf[0] will contain the bit corresponding to the
	// first bit of the bitstream segment).

	// Note: This implementation using bit shifts on the native dWord is endianism independent.
	byteBuf[0] =  dWord >> 24;
	byteBuf[1] = (dWord >> 16) & 0xff;
	byteBuf[2] = (dWord >>  8) & 0xff;
	byteBuf[3] = (dWord      ) & 0xff;

	// Calculate number of bytes to read (src) and write (dst).
	nBytesSrc = nBits/RFD_BITS_PER_BYTE + ((nBits % RFD_BITS_PER_BYTE) ? 1:0);
	nBytesDst = (streamInfo->currBitOffset + nBits)/RFD_BITS_PER_BYTE +
				(((streamInfo->currBitOffset + nBits) % RFD_BITS_PER_BYTE) ? 1:0);

	// Calculate bitstream bit offset relative to the last bit of the
	// bitstream segment being written.
	lastBitOffset = (streamInfo->currBitOffset + nBits) % RFD_BITS_PER_BYTE;
	lastBitOffsetInverse = RFD_BITS_PER_BYTE - lastBitOffset;

	// Calculate bitstream bit offset relative to the first bit of the
	// bitstream segment being written.
	firstBitOffset = streamInfo->currBitOffset % RFD_BITS_PER_BYTE;
	firstBitOffsetInverse = RFD_BITS_PER_BYTE - firstBitOffset;

	// Initialize poiners to source and destination.
	pDst = streamInfo->streamBuf + streamInfo->currByteOffset;
	pSrc = &byteBuf[RFD_BYTES_PER_DWORD - nBytesSrc];

	if(lastBitOffset != 0) {
		// Last bit of bitstream segment is Not aligned to buffer byte boundary.
		// Need to combine shifted versions of source bytes to compose ouput bytes.

		if(nBytesDst > nBytesSrc) {
			src1 = 0;
			// No pSrc increment.
		}
		else {
			src1 = *pSrc++;
		}

		for(i=0;i<nBytesDst;i++) {

			if(i != nBytesDst-1) {
				src2 = *pSrc++;
			}
			else {
				// Last byte, only src1 contributes to the last destination byte.
				src2 = 0;
			}

			// clear any previous non-zero bits in the bitstream segment being written.
			if( (i != 0) && (i != nBytesDst-1) ) {
				// one of the 'middle' bytes (not first or last byte)
				dst = 0;
			}
			else {
				// Read current content of destination byte and clear bits
				// corresponding to the bitstream segment being written.
				dst = *pDst;
				if( (i == 0) && (i == nBytesDst-1) ) {
					// first and last byte (nBytesDst = 1)
					dst &= ((0xff << firstBitOffsetInverse) | (0xff >> lastBitOffset));
				}
				else if(i == 0) {
					// first byte only
					dst &= (0xff << firstBitOffsetInverse);
				}
				else { //if(i == nBytesDst-1)
					// last byte only
					dst &= (0xff >> lastBitOffset);
				}
			}

			dst |= (src1 << lastBitOffsetInverse) + (src2 >> lastBitOffset);
			*pDst++ = dst;
			src1 = src2;
		}
	}
	else {
		// Last bit is byte aligned,
		// do simpler/faster copy without bit shifting.

		for(i=0;i<nBytesDst;i++) {
			// Check for first byte.
			// No need to check and mask for last byte since the
			// last bit offset is byte aligned in this case.
			if(i == 0) {
				// first byte
				dst = *pDst;
				dst &= (0xff << firstBitOffsetInverse);
			}
			else {
				dst = 0;
			}
			dst |= *pSrc++;
			*pDst++ = dst;
		}
	}

	// Update offsets.
	byteOffsetIncrement = (streamInfo->currBitOffset + nBits) / RFD_BITS_PER_BYTE;
	streamInfo->currByteOffset += byteOffsetIncrement;
	streamInfo->currBitOffset = (streamInfo->currBitOffset + nBits) % RFD_BITS_PER_BYTE;

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
int RFD_BitstreamGetU32(RFD_BITSTREAM_INFO * streamInfo, int nBits, UINT32 * dWord)
{
	int err;
	int byteOffsetIncrement;

	err = RFD_BitstreamPeakU32(streamInfo, nBits, dWord);
	if(!err) {
		byteOffsetIncrement = (streamInfo->currBitOffset + nBits) / RFD_BITS_PER_BYTE;
		streamInfo->currByteOffset += byteOffsetIncrement;
		streamInfo->currBitOffset = (streamInfo->currBitOffset + nBits) % RFD_BITS_PER_BYTE;
	}
	// else error, don't advance offset.

	return err;
}

///////////////////////////////////////////////////////////////////////////////
int RFD_BitstreamPeakU32(RFD_BITSTREAM_INFO * streamInfo, int nBits, UINT32 * dWord)
{
	int i;
	int   bitOffset, bitOffsetInverse;
	UCHAR byteBuf[RFD_BYTES_PER_DWORD];
	int nBytesDst, nBytesSrc;
	UCHAR * pSrc, * pDst;
	UCHAR src1, src2;

	if(nBits > RFD_BITS_PER_DWORD) {
		// Error, requested read size larger than dword.
		return -1;
	}

	if( (nBits + streamInfo->currBitOffset +
		streamInfo->currByteOffset * RFD_BITS_PER_BYTE) > streamInfo->streamBitLen ) {
		// Error, read advances past end of buffer.
		return -1;
	}

	// Clear byteBuf[].
	for(i=0;i<RFD_BYTES_PER_DWORD;i++) {
		byteBuf[i] = 0;
	}

	// Calculate number of bytes to read.
	nBytesDst = nBits/RFD_BITS_PER_BYTE + ((nBits % RFD_BITS_PER_BYTE) ? 1:0);
	nBytesSrc = (streamInfo->currBitOffset + nBits)/RFD_BITS_PER_BYTE +
				(((streamInfo->currBitOffset + nBits) % RFD_BITS_PER_BYTE) ? 1:0);

	// Calculate bitstream bit offset relative to the last bit of the
	// bitstream segment being read.
	bitOffset = (streamInfo->currBitOffset + nBits) % RFD_BITS_PER_BYTE;
	bitOffsetInverse = RFD_BITS_PER_BYTE - bitOffset;

	pDst = &byteBuf[RFD_BYTES_PER_DWORD - nBytesDst];

	if(bitOffset != 0) {

		pSrc = streamInfo->streamBuf + streamInfo->currByteOffset;

		if(nBytesSrc > nBytesDst) {
			src1 = *pSrc++;
		}
		else {
			src1 = 0;
			// no increment of pSrc.
		}

		for(i=0;i<nBytesDst;i++) {
			src2 = *pSrc++;
			*pDst++ = (src1 << bitOffset) + (src2 >> bitOffsetInverse);
			src1 = src2;
		}
	}
	else {
		// last bit is byte aligned,
		// do simpler/faster copy without bit shifting.
		pSrc = streamInfo->streamBuf + streamInfo->currByteOffset;
		for(i=0;i<nBytesDst;i++) {
			*pDst++ = *pSrc++;
		}
	}

	if(nBits % RFD_BITS_PER_BYTE) {
		// clear the extraneous bits (before the requested bitstream segment) that were read.
		byteBuf[RFD_BYTES_PER_DWORD - nBytesDst] &= (0xff >> (RFD_BITS_PER_BYTE - (nBits % RFD_BITS_PER_BYTE)));
	}

	// Note: byteBuf[3] will contain the last bit of the bitstream segment being read
	// (last bit aligned to the lsb of byteBuf[3])
	// (when nBytesDst = 4, byteBuf[0] will contain the first bit of the bitstream segment).

	// Note: This implementation using bit shifts on the native dWord is endianism independent.
	*dWord =      (((UINT32) byteBuf[0]) << 24)
				+ (((UINT32) byteBuf[1]) << 16)
				+ (((UINT32) byteBuf[2]) <<  8)
				+  ((UINT32) byteBuf[3]);

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
INT32 RFD_BitstreamGetCurrBitPosition(RFD_BITSTREAM_INFO * streamInfo)
{
	INT32 bitPosition;

	bitPosition = streamInfo->currByteOffset * RFD_BITS_PER_BYTE +
				     streamInfo->currBitOffset;

	return bitPosition;
}

///////////////////////////////////////////////////////////////////////////////
INT32 RFD_BitstreamGetStreamLengthBits(RFD_BITSTREAM_INFO * streamInfo)
{
	return streamInfo->streamBitLen;
}