////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	multifile_manager\multifile_manager.c
///
/// @brief	multifile manager class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_receiver.h"
#include "rfd_msg_collector.h"
#include "rfd_linked_list.h"
#include "rfd_time.h"
#include "rfd_file_consumer.h"
#include "mfm_consumer.h"
#include "multifile_manager.h"

////////////////////////////////////////////////////////////////////////////
static void _PrintActiveDmiInfo(
        MFM_ACTIVE_DMI_INFO_HANDLE hActiveDmiInfo,
        const TCHAR * headerName1,
        const TCHAR * headerName2,
        const TCHAR * headerName3 )
{
    const TCHAR * noName = TEXT("");

    if(headerName1 == NULL) {
        headerName1 = noName;
    }
    if(headerName2 == NULL) {
        headerName2 = noName;
    }
    if(headerName3 == NULL) {
        headerName3 = noName;
    }

    RFD_DPrint(
        TEXT("%s %s %s : ")
        TEXT("hActiveDmiInfo, DMI (%d), In-Use Count (%d)\n"),
        headerName1,
        headerName2,
        headerName3,
        hActiveDmiInfo->dmi,
        hActiveDmiInfo->inUseCount );

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintActiveDmiList(
                       RFD_LLIST_HANDLE activeDmiList,
                       int * itemCountPtr,
                       const TCHAR * headerName1,
                       const TCHAR * headerName2,
                       const TCHAR * headerName3 )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_ACTIVE_DMI_INFO_HANDLE hActiveDmiInfo = NULL;
    int itemCount = 0;

    if(itemCountPtr != NULL) {
        *itemCountPtr = 0;
    }

	/////////////////////////////////////////
    // For each MFM RFD Client in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    activeDmiList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        hActiveDmiInfo = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        // Print the MFM RFD Client
        _PrintActiveDmiInfo(
            hActiveDmiInfo,
            headerName1,
            headerName2,
            headerName3 );

        // Increment the item count
        itemCount++;

        status = RFD_LLIST_Next(
                        activeDmiList,
                        &hListNode );

    }  // End of List iteration

    // Output item count if caller provided pointer.
    if(itemCountPtr != NULL) {
        *itemCountPtr = itemCount;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintFramedActiveDmiList( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_STATUS status = RFD_STATUS_OK;
    int itemCount = 0;

    do { // once

        ///////////////////////////////////////////////////////////////////////////////
        // Print the Active DMI List
        ///////////////////////////////////////////////////////////////////////////////

        RFD_DPrint(
            TEXT("MFM:**************************************\n")
            TEXT("MFM (%s, %d): Active DMI List\n")
            TEXT("\n"),
            hManager->mfmClientName,
            hManager->mfmClientId );

        status = _PrintActiveDmiList(
            hManager->activeDmiList,
            &itemCount,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") Active DMI ") );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        RFD_DPrint( TEXT("MFM (%s, %d): Active DMI List has %d DMIs.\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            itemCount );

        RFD_DPrint(
            TEXT("MFM:**************************************\n\n"));

    } while(FALSE);

    if(status != RFD_STATUS_OK) {

        RFD_DPrint( TEXT("MFM (%s, %d): _PrintFramedActiveDmiList() errror, status (%d).\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            status );
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////
static void _PrintMfmRfdClient(
        MFM_RFD_CLIENT_HANDLE hMfmRfdClient,
        const TCHAR * headerName1,
        const TCHAR * headerName2,
        const TCHAR * headerName3 )
{
    const TCHAR * noName = TEXT("");
    DWORD rfdClientId, rfdClientIndex;

    if(headerName1 == NULL) {
        headerName1 = noName;
    }
    if(headerName2 == NULL) {
        headerName2 = noName;
    }
    if(headerName3 == NULL) {
        headerName3 = noName;
    }

    if(hMfmRfdClient->rfdMessageCollectorHandle != NULL) {
        rfdClientId =
            hMfmRfdClient->rfdMessageCollectorHandle->clientInfo->clientIdNo;
        rfdClientIndex =
            hMfmRfdClient->rfdMessageCollectorHandle->clientInfo->clientIndex;
    }
    else {
        rfdClientId = 0;
        rfdClientIndex = 0;
    }

    RFD_DPrint(
        TEXT("%s %s %s : ")
        TEXT("hMfmRfdClient (0x%x), hRfdCollector (0x%x), hRfdConsumer (0x%x), ")
        TEXT("RFD Client Id (%d), RFD Client Index (%d)\n"),
        headerName1,
        headerName2,
        headerName3,
        hMfmRfdClient,
        hMfmRfdClient->rfdMessageCollectorHandle,
        hMfmRfdClient->rfdFileConsumerHandle,
        rfdClientId,
        rfdClientIndex );

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintMfmRfdClientList(
                       RFD_LLIST_HANDLE mfmRfdClientList,
                       int * itemCountPtr,
                       const TCHAR * headerName1,
                       const TCHAR * headerName2,
                       const TCHAR * headerName3 )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_RFD_CLIENT_HANDLE hMfmRfdClient = NULL;
    int itemCount = 0;

    if(itemCountPtr != NULL) {
        *itemCountPtr = 0;
    }

	/////////////////////////////////////////
    // For each MFM RFD Client in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    mfmRfdClientList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        hMfmRfdClient = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        // Print the MFM RFD Client
        _PrintMfmRfdClient(
            hMfmRfdClient,
            headerName1,
            headerName2,
            headerName3 );

        // Increment the item count
        itemCount++;

        status = RFD_LLIST_Next(
                        mfmRfdClientList,
                        &hListNode );

    }  // End of List iteration

    // Output item count if caller provided pointer.
    if(itemCountPtr != NULL) {
        *itemCountPtr = itemCount;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintAllMfmRfdClientLists( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_STATUS status = RFD_STATUS_OK;
    int itemCount = 0;

    do { // once

        ///////////////////////////////////////////////////////////////////////////////
        // Print the Busy MFM-RFD Client List
        ///////////////////////////////////////////////////////////////////////////////

        RFD_DPrint(
            TEXT("MFM:**************************************\n")
            TEXT("MFM (%s, %d): Busy MFM-RFD Client List\n")
            TEXT("\n"),
            hManager->mfmClientName,
            hManager->mfmClientId );

        status = _PrintMfmRfdClientList(
            hManager->rfdClientResourceInfo.busyMfmRfdClientList,
            &itemCount,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") MFM RFD Client ") );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        RFD_DPrint( TEXT("MFM (%s, %d): Busy MFM RFD Client List has %d Clients.\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            itemCount );

        RFD_DPrint(
            TEXT("MFM:**************************************\n\n"));

        ///////////////////////////////////////////////////////////////////////////////
        // Print the Free MFM-RFD Client List
        ///////////////////////////////////////////////////////////////////////////////

         RFD_DPrint(
             TEXT("MFM:**************************************\n")
            TEXT("MFM (%s, %d): Free MFM-RFD Client List\n")
            TEXT("\n"),
            hManager->mfmClientName,
            hManager->mfmClientId );

        status = _PrintMfmRfdClientList(
            hManager->rfdClientResourceInfo.freeMfmRfdClientList,
            &itemCount,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") MFM RFD Client ") );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        RFD_DPrint( TEXT("MFM (%s, %d): Free MFM RFD Client List has %d Clients.\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            itemCount );

        RFD_DPrint(
            TEXT("MFM:**************************************\n\n"));

    } while(FALSE);

    if(status != RFD_STATUS_OK) {

        RFD_DPrint( TEXT("MFM (%s, %d): _PrintAllMfmRfdClientLists() errror, status (%d).\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            status );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _PrintFileItem(
        MFM_FILE_ITEM_HANDLE hFileItem,
        const TCHAR * headerName1,
        const TCHAR * headerName2,
        const TCHAR * headerName3 )
{
    const TCHAR * noName = TEXT("");
    MFM_FILE_IDENTITY_HANDLE hFileIdentity = &hFileItem->fileIdentityInfo;
    DWORD rfdClientId;

    if(headerName1 == NULL) {
        headerName1 = noName;
    }
    if(headerName2 == NULL) {
        headerName2 = noName;
    }
    if(headerName3 == NULL) {
        headerName3 = noName;
    }

    if(hFileItem->hMfmRfdClient != NULL) {
        rfdClientId =
            hFileItem->hMfmRfdClient->rfdMessageCollectorHandle->clientInfo->clientIdNo;
    }
    else {
        rfdClientId = 0;
    }

    RFD_DPrint(
        TEXT("%s %s %s: ")
        TEXT("Subname (%s), Version From (%d), Version To (%03d) : ")
        TEXT("hMfmRfdClient (0x%x), RfdClientId (%d) : ")
        TEXT("Mdata LifeDays (%d), Blk DMI (%d), Msg Buf (0x%x), Size (%d), Count (%d)\n"),
        headerName1,
        headerName2,
        headerName3,
        hFileIdentity->subname,
        hFileIdentity->versionFrom,
        hFileIdentity->versionTo,
        hFileItem->hMfmRfdClient,
        rfdClientId,
        hFileItem->rfdLifeTime,
        hFileItem->rfdBlockMsgDmi,
        hFileItem->rfdMetadataMsgBuf,
        hFileItem->rfdMetadataMsgSize,
        hFileItem->rfdMetadataMsgRxCount );

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintFileItemList(
                       RFD_LLIST_HANDLE fileItemList,
                       int * itemCountPtr,
                       const TCHAR * headerName1,
                       const TCHAR * headerName2,
                       const TCHAR * headerName3 )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;
    int itemCount = 0;

    if(itemCountPtr != NULL) {
        *itemCountPtr = 0;
    }

	/////////////////////////////////////////
    // For each File Item in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    fileItemList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        hFileItem = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        // Print the File Item
        _PrintFileItem(
            hFileItem,
            headerName1,
            headerName2,
            headerName3 );

        // Increment the item count
        itemCount++;

        status = RFD_LLIST_Next(
                        fileItemList,
                        &hListNode );

    }  // End of List iteration

    // Output item count if caller provided pointer.
    if(itemCountPtr != NULL) {
        *itemCountPtr = itemCount;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintFileItemListList(
                       RFD_LLIST_HANDLE fileItemListList,
                       int * itemCountPtr,
                       const TCHAR * headerName1,
                       const TCHAR * headerName2,
                       const TCHAR * headerName3 )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    RFD_LLIST_HANDLE hFileItemList = NULL;
    int totalItemCount = 0;
    int singleListItemCount = 0;

    if(itemCountPtr != NULL) {
        *itemCountPtr = 0;
    }

	/////////////////////////////////////////
    // For each File Item List in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    fileItemListList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        hFileItemList = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

	    /////////////////////////////////////////
        // Print the File Item List
	    /////////////////////////////////////////
        status = _PrintFileItemList(
                        hFileItemList,
                        &singleListItemCount,
                        headerName1,
                        headerName2,
                        headerName3 );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // Increment the item count
        totalItemCount += singleListItemCount;

        status = RFD_LLIST_Next(
                        fileItemListList,
                        &hListNode );

    }  // End of List iteration

    // Output item count if caller provided pointer.
    if(itemCountPtr != NULL) {
        *itemCountPtr = totalItemCount;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintActiveFileItemList( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_STATUS status = RFD_STATUS_OK;
    int itemCount = 0;
    RFD_LLIST_HANDLE hFileItemList = hManager->fileManagementLists.activeFileItemList;

    do { // once

        ///////////////////////////////////////////////////////////////////////////////
        // Print the Active File Item List
        ///////////////////////////////////////////////////////////////////////////////

        RFD_DPrint(
            TEXT("MFM:**************************************\n")
            TEXT("MFM (%s, %d): Active File Item List\n")
            TEXT("\n"),
            hManager->mfmClientName,
            hManager->mfmClientId );

        status = _PrintFileItemList(
            hFileItemList,
            &itemCount,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") File Item ") );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        RFD_DPrint( TEXT("MFM (%s, %d): Active File Item List has %d File Items.\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            itemCount );

        RFD_DPrint(
            TEXT("MFM:**************************************\n\n"));

    } while(FALSE);

    if(status != RFD_STATUS_OK) {

        RFD_DPrint( TEXT("MFM (%s, %d): _PrintActiveFileItemList() errror, status (%d).\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            status );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintWaitingFileItemListList( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_STATUS status = RFD_STATUS_OK;
    int itemCount = 0;
    RFD_LLIST_HANDLE hFileItemListList =
        hManager->fileManagementLists.waitingSubnameTitledFileItemListList;

    do { // once

        RFD_DPrint(
            TEXT("MFM:**************************************\n")
            TEXT("MFM (%s, %d): Waiting File Item List\n")
            TEXT("\n"),
            hManager->mfmClientName,
            hManager->mfmClientId  );

        status = _PrintFileItemListList(
            hFileItemListList,
            &itemCount,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") File Item ") );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        RFD_DPrint( TEXT("MFM (%s, %d): Waiting File Item List has %d File Items.\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            itemCount );

        RFD_DPrint( TEXT("MFM (%s, %d): Waiting File Item List %s Sorted.\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            hManager->fileManagementLists.isWaitingListCurrentlySorted ? TEXT("is"):TEXT("is not") );

        RFD_DPrint(
            TEXT("MFM:**************************************\n\n"));

    } while(FALSE);

    if(status != RFD_STATUS_OK) {

        RFD_DPrint( TEXT("MFM (%s, %d): _PrintWaitingFileItemListList() errror, status (%d).\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            status );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _PrintAllFileItemLists( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        ///////////////////////////////////////////////////////////////////////////////
        // Print the Active File Item List
        ///////////////////////////////////////////////////////////////////////////////

        status = _PrintActiveFileItemList( hManager );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////////////////////////////////////////
        // Print the Waiting File Item List-List
        ///////////////////////////////////////////////////////////////////////////////

        status = _PrintWaitingFileItemListList( hManager );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

    } while(FALSE);

    if(status != RFD_STATUS_OK) {

        RFD_DPrint( TEXT("MFM (%s, %d): _PrintAllFileItemLists() errror, status (%d).\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            status );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static const TCHAR * _FileManagementUpdatePolicyText(
    MFM_FILE_MANAGEMENT_POLICY_UPDATE_TYPE_ENUM updatePolicy )
{
    const TCHAR *returnString;

    switch (updatePolicy)
    {
    case MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES:
        returnString = TEXT("MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES");
        break;

    case MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES:
        returnString = TEXT("MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES");
        break;

    default:
        returnString = TEXT("unknown");
        break;
    }

    return returnString;
}

///////////////////////////////////////////////////////////////////////////////
static void _PrintMfmFileManagementPolicyInfo( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_DPrint(
        TEXT("MFM:**************************************\n")
        TEXT("MFM (%s, %d): File Managemement Policy Configuration\n")
        TEXT("\n"),
        hManager->mfmClientName,
        hManager->mfmClientId  );

        RFD_DPrint( TEXT("MFM (%s, %d): File Management Policy: Update Type (%s).\n"),
        hManager->mfmClientName,
        hManager->mfmClientId,
        _FileManagementUpdatePolicyText(hManager->fileManagementPolicyInfo.updateType) );

    RFD_DPrint( TEXT("MFM (%s, %d): File Management Policy: File Item Priority Sorting (%s).\n"),
        hManager->mfmClientName,
        hManager->mfmClientId,
        hManager->fileManagementPolicyInfo.isFilePrioritySortingEnabled ? TEXT("ENABLED"):TEXT("DISABLED"));

    RFD_DPrint(
        TEXT("MFM:**************************************\n\n"));

    return;
}

///////////////////////////////////////////////////////////////////////////////
static void _PrintAll( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        RFD_DPrint(
            TEXT("MFM: Printing info for MFM with Client Name (%s), Client Id (%d).\n"),
            hManager->mfmClientName,
            hManager->mfmClientId );

        ///////////////////////////////////////////////////////////////////////////////
        // Print all the File Item Lists
        ///////////////////////////////////////////////////////////////////////////////
        status = _PrintAllFileItemLists( hManager );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////////////////////////////////////////
        // Print all the MFM-RFD Client Lists
        ///////////////////////////////////////////////////////////////////////////////
        status = _PrintAllMfmRfdClientLists( hManager );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////////////////////////////////////////
        // Print the Active DMI List
        ///////////////////////////////////////////////////////////////////////////////
        status = _PrintFramedActiveDmiList( hManager );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////////////////////////////////////////
        // Print MFM File Managemement Policy config
        ///////////////////////////////////////////////////////////////////////////////
        _PrintMfmFileManagementPolicyInfo( hManager );

    } while(FALSE);

    if(status != RFD_STATUS_OK) {

        RFD_DPrint( TEXT("MFM (%s, %d): _PrintAll() errror, status (%d).\n"),
            hManager->mfmClientName,
            hManager->mfmClientId,
            status );
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static BOOL _GetCurrentSxiTime(
                MULTIFILE_MANAGER_HANDLE hManager,
                RFD_SXI_TIME_INFO_STRUCT * sxiTimePtr )
{
    BOOL isTimeAvailable;

    MFM_APP_GET_TIME_CALLBACK_FCN appGetTimeCallbackFcn =
        hManager->appCallbackInfo.appGetTimeCallbackFcn;

    void * appGetTimeCallbackArg =
        hManager->appCallbackInfo.appGetTimeCallbackArg;

    ///////////////////////////////////////////////////////////
    // Get current time using app provided callback.
    ///////////////////////////////////////////////////////////
    isTimeAvailable = appGetTimeCallbackFcn(
                            sxiTimePtr,
                            appGetTimeCallbackArg );

    if( isTimeAvailable == TRUE ) {

        BOOL isValidSxiTimeFormat;
        // Validate the SXI Time received externally from the app provided callback function.
        isValidSxiTimeFormat = RFD_TIME_IsSxiTimeFormatValid( sxiTimePtr );

        if( isValidSxiTimeFormat == FALSE ) {
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _GetCurrentSxiTime() warning, SXI Time from app callback has invalid format,\n")
                TEXT("MFM... min %d, hour %d, day %d, month %d, year %d\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                sxiTimePtr->minute, sxiTimePtr->hour , sxiTimePtr->day , sxiTimePtr->month, sxiTimePtr->yearSince2k );

            // Treat an invalid SXI Time Format as Time Not Available condition as MFM continues functioning
            // if time is not available / invalid for some subset of operating time (versus treating this as a hard error).
            isTimeAvailable = FALSE;
        }
    }

    return isTimeAvailable;
}

///////////////////////////////////////////////////////////////////////////////
static BOOL _GetCurrentSimpleCalendarTime(
                MULTIFILE_MANAGER_HANDLE hManager,
                time_t * currentSimpleCalendarTimePtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isTimeAvailable;
    RFD_SXI_TIME_INFO_STRUCT currentSxiTime;

    // Initialize output
    *currentSimpleCalendarTimePtr = (time_t) -1;

    ///////////////////////////////////////////////////////////
    // Get the current SXI time
    ///////////////////////////////////////////////////////////

    isTimeAvailable = _GetCurrentSxiTime( hManager, &currentSxiTime );

    if( isTimeAvailable == TRUE ) {

        ///////////////////////////////////////////////////////////
        // Convert sxi time to the c standard lib " simple calendar" time
        ///////////////////////////////////////////////////////////

        status = RFD_TIME_SxiTime2StdCalendarTime( currentSimpleCalendarTimePtr, &currentSxiTime);

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _GetCurrentSimpleCalendarTime() error in RFD_TIME_SxiTime2StdCalendarTime(), %d\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status );

            // Treat error in conversion from SXI to Simple Calendar Time as Time Not Available condition.
            isTimeAvailable = FALSE;
        }
    }

    return isTimeAvailable;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _IsRfdLifeExpired(
                        MULTIFILE_MANAGER_HANDLE hManager,
                        INT32 rfdLifeTime,
                        time_t currentSimpleCalendarTime,
                        BOOL isCurrentTimeAvailable,
                        BOOL * isLifeExpiredPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isLifeExpired;
    double timeToLiveSeconds;
    time_t rfdLifeSimpleCalendarTime;

    // Initialize output
    *isLifeExpiredPtr = FALSE;
    isLifeExpired = FALSE;

    do { // once

        if( isCurrentTimeAvailable == TRUE ) {

	        /////////////////////////////////////////
            // Convert LifeTime (in days) to standard simple calendar time
	        /////////////////////////////////////////

             status = RFD_TIME_RfdLifeDays2StdCalendarTime(
                            &rfdLifeSimpleCalendarTime,
                            rfdLifeTime );

            if(status != RFD_STATUS_OK) {
                break;
            }

            timeToLiveSeconds = RFD_TIME_DiffSimpleCalendarTime( rfdLifeSimpleCalendarTime, currentSimpleCalendarTime );

            // Use less-than, not less-than-or-equal comparison so as to error on the side of
            // not expiring items too early due to discrepancies in UTC time versus local time
            // setting of the current time. Result is erroring to waiting 1 day longer
            // than necessary before expiring File Items.

            if( timeToLiveSeconds < 0 ) {

                // RFD Life is expired.
                isLifeExpired = TRUE;

                RFD_DPrint(
                    TEXT("MFM (%s, %d): _IsRfdLifeExpired() Life (%d), Expired by %lf days\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    rfdLifeTime,
                    -timeToLiveSeconds/(60*60*24) );
            }
            else {

                isLifeExpired = FALSE;

                RFD_DPrint(
                    TEXT("MFM (%s, %d): _IsRfdLifeExpired() Life (%d), has %lf days remaining\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    rfdLifeTime,
                    timeToLiveSeconds/(60*60*24) );
            }
        }
        else {
            // Current Time is not available.
            // Treat RFD Life as Not Expired for this unknown current time condition. There may
            // be periods during normal MFM/RFD operation in which the current (SXM broadcast derived) time is unknown
            // (e.g. during No Signal conditions).
            isLifeExpired = FALSE;
        }

    } while(FALSE);

    // update output
    if(status == RFD_STATUS_OK) {
        *isLifeExpiredPtr = isLifeExpired;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteActiveDmiInfo( MFM_ACTIVE_DMI_INFO_HANDLE hActiveDmiInfo )
{
    if(hActiveDmiInfo != NULL) {

        RFD_FREE( hActiveDmiInfo );
        hActiveDmiInfo = NULL;
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static MFM_ACTIVE_DMI_INFO_HANDLE _CreateActiveDmiInfo(
                        UINT16 dmi,
                        INT32 initialCount )
{
    RFD_STATUS status = RFD_STATUS_OK;
    MFM_ACTIVE_DMI_INFO_HANDLE hActiveDmiInfo = NULL;

    do { // once

        // allocate
        hActiveDmiInfo = (MFM_ACTIVE_DMI_INFO_HANDLE) RFD_CALLOC(sizeof(MFM_ACTIVE_DMI_INFO_STRUCT));

        if(hActiveDmiInfo == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        // initialize info
        hActiveDmiInfo->dmi = dmi;
        hActiveDmiInfo->inUseCount = initialCount;

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup
        _DeleteActiveDmiInfo(hActiveDmiInfo);
        hActiveDmiInfo = NULL;
    }

    return hActiveDmiInfo;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS _FindActiveDmiInList(
                RFD_LLIST_HANDLE activeDmiList,
                UINT16 referenceDmi,
                RFD_LLIST_NODE_HANDLE * hFoundListNodePtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_ACTIVE_DMI_INFO_HANDLE hActiveDmiInfo = NULL;

    // Initialize output to NULL to indicate no match.
    *hFoundListNodePtr = NULL;

	/////////////////////////////////////////
    // For each DMI Info handle in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    activeDmiList,
                    &hListNode );

    while(hListNode != NULL) {

        hActiveDmiInfo = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        if(hActiveDmiInfo->dmi == referenceDmi) {

	        /////////////////////////////////////////
            // match found
	        /////////////////////////////////////////

            // Output the found List Node.
            *hFoundListNodePtr = hListNode;

            // Stop searching since an item was found.
            break;
        }

        status = RFD_LLIST_Next(
                        activeDmiList,
                        &hListNode );
        if(status != RFD_STATUS_OK) {
            break;
        }

    }  // End of List iteration

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _HandleDmiDeactivation(
                MULTIFILE_MANAGER_HANDLE hManager,
                UINT16 dmi )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hFoundListNode = NULL;
    MFM_APP_RFD_BLK_MSG_DMI_STATUS_CALLBACK_FCN appBlkMsgDmiStatusCallbackFcn;
    void * appBlkMsgDmiStatusCallbackArg;
    MFM_ACTIVE_DMI_INFO_HANDLE hActiveDmiInfo = NULL;
    RFD_LLIST_HANDLE activeDmiList = hManager->activeDmiList;

    do { // once

        // search active dmi list for an existing dmi
        status = _FindActiveDmiInList(
                        activeDmiList,
                        dmi,
                        &hFoundListNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // if found, decrement count
        if( hFoundListNode != NULL ) {

            // Found existing Active Dmi.

            hActiveDmiInfo = RFD_LLIST_GetUserData(hFoundListNode, &status);
            if(status != RFD_STATUS_OK) {
                break;
            }

            // Decrement the count.
            hActiveDmiInfo->inUseCount--;

            if( hActiveDmiInfo->inUseCount <= 0 ) {

                // First log warning if count is less that zero,
                // but continue normally in effort to recover from this condition.
                if(hActiveDmiInfo->inUseCount < 0) {

                    RFD_DPrint(
                        TEXT("MFM (%s, %d): _ProcessActiveDmiOnFileItemDeactivation() error, DMI (%d) count is negative (%d) \n"),
                        hManager->mfmClientName,
                        hManager->mfmClientId,
                        dmi,
                        hActiveDmiInfo->inUseCount );
                }

                // This Deactivated File Item was the last user of this DMI.
                // Remove DMI from the Active DMI list, delete it, and invoke DMI status callback
                // to inform app the DMI is no longer in use.

                // Remove the List Node
                status = RFD_LLIST_RemoveNode( activeDmiList, hFoundListNode );

                if(status != RFD_STATUS_OK) {
                    break;
                }

                // Delete the contained Active DMI Info.
                _DeleteActiveDmiInfo( hActiveDmiInfo );
                hActiveDmiInfo = NULL;

                // Do DMI status callback only if DMI is valid DMI value.
                // The invalid/unused DMI value may be set for data services not using the Block Message DMI field
                // and is also the initial setting used out of powerup instead of any initial RFD Client DMI state info
                // which may be stale.

                if( dmi != RFD_DMI_UNUSED_VALUE ) {

                    // Do DMI status Callback if enabled.
                    appBlkMsgDmiStatusCallbackFcn = hManager->appCallbackInfo.appBlkMsgDmiStatusCallbackFcn;
                    appBlkMsgDmiStatusCallbackArg = hManager->appCallbackInfo.appBlkMsgDmiStatusCallbackArg;

                    if(appBlkMsgDmiStatusCallbackFcn != NULL) {
                        // App is using dmi callback,
                        // Invoke the callback with In-Use = FALSE.
                        appBlkMsgDmiStatusCallbackFcn( dmi, FALSE, appBlkMsgDmiStatusCallbackArg );
                    }
                }

                RFD_DPrint(
                    TEXT("MFM (%s, %d): Active DMI (%d) Removed from List\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    dmi );
            }
            else {
                // Just print that Active DMI Count decremented.
                RFD_DPrint(
                    TEXT("MFM (%s, %d): Active DMI (%d) Use Count decremented to (%d)\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    dmi,
                    hActiveDmiInfo->inUseCount );
            }
        }
        else {

            // Active DMI not found.
            // This is an error condition since every File Item Deactivation should have been
            // preceeded by a corresponding File Item Activation (which increments the count).

            RFD_DPrint(
                TEXT("MFM (%s, %d): _ProcessActiveDmiOnFileItemDeactivation() error, Active DMI (%d) not found on File Item Deactivation \n"),
                hManager->mfmClientName,
                hManager->mfmClientId,
                dmi );

            status = RFD_STATUS_ERROR_INVALID_STATE;
            break;
        }

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _HandleDmiActivation(
                MULTIFILE_MANAGER_HANDLE hManager,
                UINT16 dmi )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hFoundListNode = NULL;
    MFM_ACTIVE_DMI_INFO_HANDLE hNewActiveDmiInfo = NULL;
    MFM_APP_RFD_BLK_MSG_DMI_STATUS_CALLBACK_FCN appBlkMsgDmiStatusCallbackFcn;
    void * appBlkMsgDmiStatusCallbackArg;
    // Set ref count to 0 before any creation.
    int newActiveDmiInfoRefCount = 0;
    RFD_LLIST_HANDLE activeDmiList = hManager->activeDmiList;

    do { // once

        // search active dmi list of existing dmi
        status = _FindActiveDmiInList(
                        activeDmiList,
                        dmi,
                        &hFoundListNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // if found, increment count
        if( hFoundListNode != NULL ) {

            // Use separate info handle for existing handle vs. newly created.
            // This is in case of error exit we won't try to cleanup something
            // thats already referenced in the list.
            MFM_ACTIVE_DMI_INFO_HANDLE hExistingActiveDmiInfo = NULL;

            // Found existing Active Dmi.

            hExistingActiveDmiInfo = RFD_LLIST_GetUserData(hFoundListNode, &status);
            if(status != RFD_STATUS_OK) {
                break;
            }

            // Just increment the count.
            hExistingActiveDmiInfo->inUseCount++;

            RFD_DPrint(
                TEXT("MFM (%s, %d): Active DMI (%d) Use Count incremented to (%d)\n"),
                hManager->mfmClientName,
                hManager->mfmClientId,
                dmi,
                hExistingActiveDmiInfo->inUseCount );
        }
        else {

            // Active DMI not found.

            // Create new instance, with initial count of 1.
            hNewActiveDmiInfo = _CreateActiveDmiInfo( dmi, 1 );
            // Set ref count to 1 on creation.
            newActiveDmiInfoRefCount = 1;

            if(hNewActiveDmiInfo == NULL) {
                // error
                status = RFD_STATUS_ERROR_MEM_ALLOC;
                break;
            }

            // insert to list
            status = RFD_LLIST_Insert( activeDmiList, hNewActiveDmiInfo );

            if(status != RFD_STATUS_OK) {
                break;
            }
            // Incremement ref count since it's now referenced in the List.
            // This is for proper cleanup handling on error exit.
            newActiveDmiInfoRefCount++;

            // Do DMI status callback only if DMI is valid DMI value.
            // The invalid/unused DMI value may be set for data services not using the Block Message DMI field
            // and is also the initial setting used out of powerup instead of any initial RFD Client DMI state info
            // which may be stale.

            if( dmi != RFD_DMI_UNUSED_VALUE ) {

                appBlkMsgDmiStatusCallbackFcn = hManager->appCallbackInfo.appBlkMsgDmiStatusCallbackFcn;
                appBlkMsgDmiStatusCallbackArg = hManager->appCallbackInfo.appBlkMsgDmiStatusCallbackArg;

                if(appBlkMsgDmiStatusCallbackFcn != NULL) {
                    // App is using dmi callback,
                    // Invoke the callback with In-Use = TRUE.
                    appBlkMsgDmiStatusCallbackFcn( dmi, TRUE, appBlkMsgDmiStatusCallbackArg );
                }
            }

            RFD_DPrint(
                TEXT("MFM (%s, %d): Active DMI (%d) Added to List, Use Count (%d)\n"),
                hManager->mfmClientName,
                hManager->mfmClientId,
                dmi,
                hNewActiveDmiInfo->inUseCount );
        }

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // On error condition, cleanup only if the corresponding ref count = 1
        // (it's been created only, which sets the count to 1).
        // This is so we don't try to clean it up twice
        // i.e. when its also referenced in the List.
        if(newActiveDmiInfoRefCount == 1) {
            _DeleteActiveDmiInfo( hNewActiveDmiInfo );
            hNewActiveDmiInfo = NULL;
        }
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void _SetExpirationProcessInfoLifeUpdated( MULTIFILE_MANAGER_HANDLE hManager )
{
    hManager->fileItemRfdLifeExpirationProcessInfo.isLifeUpdatedSinceLastExpProcessTime = TRUE;

    return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InitializeExpirationProcessInfo( MULTIFILE_MANAGER_STRUCT * hManager )
{
    // Just set to begining of SXI Epoc Time.
    // The first File Item Expiration Check will observe a different with this Epoc time
    // and the current sampled SXI time, which is the purpose of this variable.

    RFD_TIME_SetSxiEpocTime( &hManager->fileItemRfdLifeExpirationProcessInfo.lastExpProcessSxiTime );

    hManager->fileItemRfdLifeExpirationProcessInfo.isLifeUpdatedSinceLastExpProcessTime = FALSE;

    return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _GetNumFreeMfmRfdClients(
                MULTIFILE_MANAGER_HANDLE hManager,
                size_t * numFreeMfmRfdClientsPtr )
{
    RFD_STATUS status;

    RFD_LLIST_HANDLE mfmRfdClientList =
        hManager->rfdClientResourceInfo.freeMfmRfdClientList;

    // Get number of Stored File Items.
    status = RFD_LLIST_GetNumNodes( mfmRfdClientList, numFreeMfmRfdClientsPtr );

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _GetNumWaitingFileItems(
                MULTIFILE_MANAGER_HANDLE hManager,
                size_t * numWaitingFileItemsPtr )
{
    RFD_STATUS status;

    RFD_LLIST_HANDLE fileItemListList =
        hManager->fileManagementLists.waitingSubnameTitledFileItemListList;

    // Get number of Waiting File Items.
    status = RFD_LLIST_2DGetNumNodes( fileItemListList, numWaitingFileItemsPtr );

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static BOOL _IsMetadataCarouselReceptionComplete( MULTIFILE_MANAGER_HANDLE hManager )
{
    return hManager->mfmRfdMessageProcessInfo.isRfdUpdateFileMetadataCarouselComplete;
}

///////////////////////////////////////////////////////////////////////////////
static void _OnFileItemMessageCountInc(
    MULTIFILE_MANAGER_HANDLE hManager,
    MFM_FILE_ITEM_HANDLE hFileItem,
    MFM_FILE_ITEM_LIST_TYPE_ENUM fileItemListType )
{
    ///////////////////////////////////////////////////////////////////////////////
    // Process Carousel Reception Complete status
    // Check for Message Reception Count >= defined threshold for any File Item
    // in any of the List types (Active, Waiting or Stored).
    ///////////////////////////////////////////////////////////////////////////////

    if(hManager->mfmRfdMessageProcessInfo.isRfdUpdateFileMetadataCarouselComplete == FALSE) {

        if( hFileItem->rfdMetadataMsgRxCount >= MFM_UPDATE_FILE_METADATA_CAROUSEL_COMPLETE_MSG_COUNT_THRESH ) {

            hManager->mfmRfdMessageProcessInfo.isRfdUpdateFileMetadataCarouselComplete = TRUE;

            RFD_DPrint( TEXT("MFM (%s, %d): Update File Metadata Carousel reception is Complete due to a File Item message count = %d\n"),
                hManager->mfmClientName,
                hManager->mfmClientId,
                hFileItem->rfdMetadataMsgRxCount );

            MFM_UTIL_PrintFileItemIdentity(
                &hFileItem->fileIdentityInfo,
                TEXT("MFM:"),
                hManager->mfmClientName,
                TEXT("File Identity") );
        }
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _FindMfmRfdClientInList(
                       RFD_LLIST_HANDLE mfmRfdClientList,
                        MFM_RFD_CLIENT_HANDLE hReferenceMfmRfdClient,
                        RFD_LLIST_NODE_HANDLE * hFoundListNodePtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_RFD_CLIENT_HANDLE hMfmRfdclient;

    // Initialize output to NULL to indicate no match.
    *hFoundListNodePtr = NULL;

	/////////////////////////////////////////
    // For each  File Item in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    mfmRfdClientList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        hMfmRfdclient = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        if(hMfmRfdclient == hReferenceMfmRfdClient) {

	        /////////////////////////////////////////
            // match found
	        /////////////////////////////////////////

            // Output the found List Node.
            *hFoundListNodePtr = hListNode;

            // Stop searching since an item was found.
            break;
        }

        status = RFD_LLIST_Next(
                        mfmRfdClientList,
                        &hListNode );

    }  // End of List iteration

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _SetDefaultFileIdentityInfo( MFM_FILE_IDENTITY_STRUCT * fileIdentityInfo )
{
    fileIdentityInfo->subname[0] = '\0';
    fileIdentityInfo->versionFrom = 0;
    fileIdentityInfo->versionTo = 0;

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _UpdateFileItemMessage(
            MFM_FILE_ITEM_HANDLE hFileItem,
            UCHAR * messageBufToCopy,
            UINT32 messageSize )
{
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        // First cleanup any exiting message allocation.
        if(hFileItem->rfdMetadataMsgBuf != NULL) {

            RFD_FREE(hFileItem->rfdMetadataMsgBuf);
            hFileItem->rfdMetadataMsgBuf = NULL;
            hFileItem->rfdMetadataMsgSize = 0;
        }

        if(messageBufToCopy != NULL) {

            // allocate the rfd metadata message buffer owned by this File Item,
            // and then copy in from the input buffer.

            hFileItem->rfdMetadataMsgBuf = (UCHAR *) RFD_MALLOC(messageSize);

            if(hFileItem->rfdMetadataMsgBuf == NULL) {

                status = RFD_STATUS_ERROR_MEM_ALLOC;
                break;
            }

            RFD_MEMCPY(
                hFileItem->rfdMetadataMsgBuf,
                messageBufToCopy,
                messageSize );

            // set the message size
            hFileItem->rfdMetadataMsgSize = messageSize;
        }
        else {
            // input source buffer was null,
            // just initialize file item message to null.
            hFileItem->rfdMetadataMsgBuf = NULL;
            hFileItem->rfdMetadataMsgSize = 0;
        }

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup
        if(hFileItem->rfdMetadataMsgBuf != NULL) {
            RFD_FREE(hFileItem->rfdMetadataMsgBuf);
            hFileItem->rfdMetadataMsgBuf = NULL;
        }
        hFileItem->rfdMetadataMsgSize = 0;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteFileItem(MFM_FILE_ITEM_HANDLE hFileItem)
{
    if(hFileItem != NULL) {

        // free the rfd metada message buffer if not already free.
        if(hFileItem->rfdMetadataMsgBuf != NULL)
        {
            RFD_FREE(hFileItem->rfdMetadataMsgBuf);

            hFileItem->rfdMetadataMsgBuf = NULL;
        }

        hFileItem->rfdMetadataMsgSize = 0;

        // hFileItem->hMfmRfdClient cleanup is not responsibiltiy here,
        // just set to null.
        hFileItem->hMfmRfdClient = NULL;

        // finally, free the File Item handle.
        RFD_FREE(hFileItem);
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static MFM_FILE_ITEM_HANDLE _CreateFileItem(
                        MFM_FILE_IDENTITY_STRUCT * fileIdentityInfo,
                        MFM_RFD_CLIENT_HANDLE hMfmRfdClient,
                        UCHAR * rfdMetadataMsgBufToCopy,
                        UINT32 rfdMetadataMsgSize,
                        UINT32 rfdMetadataMsgRxCount,
                        RFD_FSET_MDATA_STRUCT * rfdMetadataInfo )
{
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        // allocate the client info structure
        hFileItem = (MFM_FILE_ITEM_HANDLE) RFD_CALLOC(sizeof(MFM_FILE_ITEM_STRUCT));

        if(hFileItem == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        // copy the file identity info structure
        if(fileIdentityInfo != NULL) {
            hFileItem->fileIdentityInfo = *fileIdentityInfo;
        }
        else {
            _SetDefaultFileIdentityInfo( &hFileItem->fileIdentityInfo );
        }

        // set the mfm rfd client handle
        hFileItem->hMfmRfdClient = hMfmRfdClient;

        // set the rfd metadata message reception count.
        hFileItem->rfdMetadataMsgRxCount = rfdMetadataMsgRxCount;

        if(rfdMetadataMsgBufToCopy != NULL) {

            // allocate the rfd metadata message buffer owned by this File Item,
            // and then copy in from the input buffer.
            //
            // Use the _UpdateFileItemMessage() function to do this.
            // but make sure initial messageBuf is null before call so that
            // the update function doesn't try to free it.

            hFileItem->rfdMetadataMsgBuf = NULL;
            hFileItem->rfdMetadataMsgSize = 0;

            status = _UpdateFileItemMessage(
                            hFileItem,
                            rfdMetadataMsgBufToCopy,
                            rfdMetadataMsgSize );

            if(status != RFD_STATUS_OK) {
                break;
            }
        }
        else {
            hFileItem->rfdMetadataMsgBuf = NULL;
            hFileItem->rfdMetadataMsgSize = 0;
        }

        // set the rfd metadata Life Time and Block Message DMI.
        if(rfdMetadataInfo != NULL) {
            hFileItem->rfdLifeTime = rfdMetadataInfo->lifeTime;
            hFileItem->rfdBlockMsgDmi = rfdMetadataInfo->blockMsgDmi;
        }
        else {
            hFileItem->rfdLifeTime = 0;
            hFileItem->rfdBlockMsgDmi = RFD_DMI_UNUSED_VALUE;
        }

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup
        _DeleteFileItem(hFileItem);
        hFileItem = NULL;
    }

    return hFileItem;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteSubnameTitledFileItemList( RFD_LLIST_HANDLE hSubnameTitledFileItemList )
{
    // A Subname Titled File Item List is just File Item List, with the Subname
    // title determined by the first Subname of the first File Item in the list.
    //
    // Delete it like any other list, including the File Item Delete callback
    // to cleanup each individual File Item node in the list.

    if (hSubnameTitledFileItemList != NULL ) {

        RFD_LLIST_DeleteList(
            hSubnameTitledFileItemList ,
            (RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN)_DeleteFileItem );

        hSubnameTitledFileItemList = NULL;
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_LLIST_HANDLE _CreateNullSubnameTitledFileItemList( void )
{
    RFD_LLIST_HANDLE hFileItemList = NULL;
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        //////////////////////////////////////////
        // Create the File Item List (initially empty)
        //////////////////////////////////////////

        hFileItemList = RFD_LLIST_CreateList();

        if(hFileItemList == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        //////////////////////////////////////////
        // The Subname title of this File Item List is defined by the Subname
        // of the first (List Node) File Item in the List. So this File Item List
        // initially does not have a Subname title, until the first
        // File Item is inserted. By implementation, all File Items
        // inserted to this "Subnam-Titled File Item List" must have the same Subname.
        //////////////////////////////////////////

    } while(FALSE);

    return hFileItemList;
}

///////////////////////////////////////////////////////////////////////////////
// _CleanupFileItemListListOnFileItemRemoved() Description:
//
// If a previous removed File Item (and corresponding File Item Node)
// resulted in a empty Inner File Item List,
// then remove this Inner List from the (Outer) List of Lists by
// Deleting the Inner List and Removing the (outer list) Node that contained this removed Inner List.
//
// The ListList (2D List) Structure example:
//
///////////////////
//  Outer List (
//    Outer List Node -> Inner List x ( Inner List Node -> File Item xa
//                                      Inner List Node -> File Item xb
//                                      Inner List Node -> File Item xc )
//
//    Outer List Node -> Inner List y ( empty )
//
//    Outer List Node -> Inner List z ( Inner List Node -> File Item za
//                                      Inner List Node -> File Item zb )
//  )
///////////////////
//
// In this example, on entry to _CleanupFileItemListListOnFileItemRemoved(),
// "Inner List y" is empty, perhaps because the last File Item and its
// corresponding Inner List Node were just removed from "Inner List y".
//
// On completion of _CleanupFileItemListListOnFileItemRemoved(),
// Inner List y and its corresponding Outer List Node will be removed, resulting
// in the below overall List structure:
//
///////////////////
//  Outer List (
//    Outer List Node -> Inner List x ( Inner List Node -> File Item xa
//                                      Inner List Node -> File Item xb
//                                      Inner List Node -> File Item xc )
//
//    Outer List Node -> Inner List z ( Inner List Node -> File Item za
//                                      Inner List Node -> File Item zb )
//  )
//
// Params: For this example above,
//  hFileItemListList  corresponds to "Outer List"
//  hListListNode      corresponds to "Outer List Node" containing "Inner List y"
//  hInnerFileItemList corresponds to "Inner List y"
//
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS _CleanupFileItemListListOnFileItemRemoved(
                RFD_LLIST_HANDLE hFileItemListList,
                RFD_LLIST_NODE_HANDLE hListListNode,
                RFD_LLIST_HANDLE hInnerFileItemList )
{
    RFD_STATUS status = RFD_STATUS_OK;
    size_t numInnerListNodes = 0;

    do { // once

        status = RFD_LLIST_GetNumNodes( hInnerFileItemList, &numInnerListNodes );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(numInnerListNodes == 0) {

            _DeleteSubnameTitledFileItemList( hInnerFileItemList );
            //same as RFD_LLIST_DeleteList( hInnerFileItemList, _DeleteFileItem );

            // inner list is deleted, remove the outer list node that contained it.
            status = RFD_LLIST_RemoveNode( hFileItemListList, hListListNode );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }
        }

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _FindFileItemInFileItemList(
                       RFD_LLIST_HANDLE fileItemList,
                       MFM_FILE_IDENTITY_HANDLE hReferenceFileIdentity,
                       MFM_FILE_IDENTITY_MASK fileIdentityCompareMask,
                       BOOL doListItemIdentityUpdate,
                       RFD_LLIST_NODE_HANDLE * hFoundListNodePtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;
    MFM_FILE_IDENTITY_HANDLE hFileIdentity;
    int compareResult;

    // Initialize output to NULL to indicate no match.
    *hFoundListNodePtr = NULL;

	/////////////////////////////////////////
    // For each  File Item in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    fileItemList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        hFileItem = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

       	hFileIdentity = &hFileItem->fileIdentityInfo;

        compareResult = MFM_UTIL_CompareFileIdentity(
                                hReferenceFileIdentity,
                                hFileIdentity,
                                fileIdentityCompareMask );

        if(!compareResult) {

	        /////////////////////////////////////////
            // match found
	        /////////////////////////////////////////

            // Output the found List Node.
            *hFoundListNodePtr = hListNode;

            if(doListItemIdentityUpdate == TRUE) {

	            /////////////////////////////////////////
                // The caller wants to also update the
                // Identity Item found in the list to be exactly
                // equal to the reference Identity Item.
                //
                // Invert the compare mask to generate the Update Mask
                // as there's no need to update the members that we know are
                // alread equal.
	            /////////////////////////////////////////

                 MFM_FILE_IDENTITY_MASK fileIdentityUpdateMask =
                                                    ~fileIdentityCompareMask;
                // Do the Identity Update.
                MFM_UTIL_UpdateFileIdentity(
                        hFileIdentity, // dst in list
                        hReferenceFileIdentity, // src
                        fileIdentityUpdateMask);
            }

            // Stop searching since an item was found.
            break;
        }

        status = RFD_LLIST_Next(
                        fileItemList,
                        &hListNode );

    }  // End of List iteration

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InsertFileItemToFileItemList(
                       RFD_LLIST_HANDLE hFileItemList,
                       MFM_FILE_ITEM_HANDLE hFileItem,
                       MFM_FILE_IDENTITY_MASK fileIdentityCompareMask,
                       BOOL doReplaceForMatchingFileIdentity )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isReplaced = FALSE;

    do { // once

        if(doReplaceForMatchingFileIdentity == TRUE) {

            // Caller requests replacing any existing item if
            // existing item identity matches criteria.

            RFD_LLIST_NODE_HANDLE hFoundListNode;

            status = _FindFileItemInFileItemList(
                            hFileItemList,
                            &hFileItem->fileIdentityInfo,
                            fileIdentityCompareMask,
                            FALSE, //doListItemIdentityUpdate = FALSE
                            &hFoundListNode );

            if(status != RFD_STATUS_OK) {
                break;
            }

            if(hFoundListNode != NULL) {

                // Found an existing file item in list with matching File Identity.
                // Delete it and replace with new file item.

                MFM_FILE_ITEM_HANDLE hExistingFileItem;

                hExistingFileItem = RFD_LLIST_GetUserData(hFoundListNode, &status);
                if(status != RFD_STATUS_OK) {
                    break;
                }

                _DeleteFileItem(hExistingFileItem);

                RFD_LLIST_SetUserData(hFoundListNode, hFileItem);

                // done
                isReplaced = TRUE;
            }
        }

        if(isReplaced == FALSE) {

            // a Replace of existing File Item did not occur,
            // do the insert.

            status = RFD_LLIST_Insert( hFileItemList, hFileItem );

            if(status != RFD_STATUS_OK) {
                break;
            }
        }

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static const TCHAR * _GetSubnameTitledFileItemListSubname( RFD_LLIST_HANDLE hFileItemList )
{
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_FILE_ITEM_HANDLE hFileItem;
    const TCHAR * subname = NULL;
    RFD_STATUS status = RFD_STATUS_OK;

    //////////////////////////////////////////////////
    // Get the Subname for a Subname-Grouped list of File Items
    // (i.e. all File Items of the list have the same Subname).
    // Get the Subname from the File Identity field of the
    // First File Item of the LIst.
    //////////////////////////////////////////////////

    do { // once

        status = RFD_LLIST_First(
            hFileItemList,
            &hListNode );

        if(status != RFD_STATUS_OK) {
            break;
        }

        if(hListNode == NULL) {
            // An empty list in the list of lists is an invalid scenario,
            // indicative of a memory leak due to list not being deleted
            // after a last node of the list was removed.
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        hFileItem = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        // success
        subname = hFileItem->fileIdentityInfo.subname;

    } while(FALSE);

    return subname;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _FindSubnameTitledFileItemList(
                       RFD_LLIST_HANDLE fileItemListList,
                       const TCHAR * referenceSubname,
                       RFD_LLIST_NODE_HANDLE * hFoundListNodePtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;

    //MFM_SUBNAME_TITLED_FILE_ITEMS_HANDLE hSubnameTitledFileItemInfo;
    RFD_LLIST_HANDLE hFileItemList;

    const TCHAR * subname;

    // Initialize output to NULL to indicate no match.
    *hFoundListNodePtr = NULL;

	/////////////////////////////////////////
    // For each  File Item List in the List of File Item Lists:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    fileItemListList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        hFileItemList = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        // Get the Subname for this (Subname-Grouped) File Item List.
        subname = _GetSubnameTitledFileItemListSubname( hFileItemList );

        if(subname == NULL) {
            // Error. Any Subname-Grouped File Item List must have a subname.
            // This is indicative of possbily an empty list which should
            // have been cleaned up after the last File Item node was removed.
            // (alternative action here would be to ignore error condition and keep iterating?).
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        if(!RFD_STRNCMP(subname, referenceSubname, MFM_SUBNAME_MAX_LEN)) {

            // Subname strings are equal, found match.
            // Set output and break from list iterate loop to return.
            *hFoundListNodePtr = hListNode;
            break;
        }

        status = RFD_LLIST_Next(
                        fileItemListList,
                        &hListNode );

    }  // End of List iteration

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _FindFileItemInFileItemListList(
                       RFD_LLIST_HANDLE fileItemListList,
                       MFM_FILE_IDENTITY_HANDLE hReferenceFileIdentity,
                       MFM_FILE_IDENTITY_MASK fileIdentityCompareMask,
                       BOOL doListItemIdentityUpdate,
                       RFD_LLIST_NODE_HANDLE * hFoundListNodePtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_HANDLE hFileItemList;

    // Initialize output to NULL to indicate no match.
    *hFoundListNodePtr = NULL;

    do { // once

        /////////////////////////////////////////////////
        // Look for Subname matching File Item List in the
        // list of Subname-Titled File Item Lists.
        /////////////////////////////////////////////////

        status = _FindSubnameTitledFileItemList(
                           fileItemListList,
                           hReferenceFileIdentity->subname,
                           hFoundListNodePtr );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(*hFoundListNodePtr == NULL) {
            // did not find Subname in SubnameTitledListsList
            break;
        }

        /////////////////////////////////////////////////
        // Found a matching Subname-Titled File Item List in the
        // list of "Subname-Titled File Item Lists".
        //
        // Try to find the corresponding Identity in this found File Item List.
        /////////////////////////////////////////////////

        hFileItemList = RFD_LLIST_GetUserData(*hFoundListNodePtr, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        // initialize to NULL again, for final use below.
        *hFoundListNodePtr = NULL;

        status = _FindFileItemInFileItemList(
                        hFileItemList,
                        hReferenceFileIdentity,
                        fileIdentityCompareMask,
                        doListItemIdentityUpdate,
                        hFoundListNodePtr );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InsertFileItemToFileItemListList(
                       RFD_LLIST_HANDLE hFileItemListList,
                       MFM_FILE_ITEM_HANDLE hFileItem,
                       MFM_FILE_IDENTITY_MASK fileIdentityCompareMask,
                       BOOL doReplaceForMatchingFileIdentity )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_HANDLE hFileItemList;
    RFD_LLIST_NODE_HANDLE hFoundListNode;

    do { // once

        /////////////////////////////////////////////////
        // First find the Subname Titled List item where File Item should
        // be inserted.
        // Look for Subname match in the subname lists.
        /////////////////////////////////////////////////

        status = _FindSubnameTitledFileItemList(
                           hFileItemListList,
                           hFileItem->fileIdentityInfo.subname,
                           &hFoundListNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(hFoundListNode != NULL) {
            /////////////////////////////////////////////////
            // found existing Subname Titled list,
            // get File Item List from it.
            /////////////////////////////////////////////////

            hFileItemList = RFD_LLIST_GetUserData(hFoundListNode, &status);
            if(status != RFD_STATUS_OK) {
                break;
            }
        }
        else {
            /////////////////////////////////////////////////
            // Did not find Subname in SubnameTitledListsList
            // Create and insert a new Subname Titled list.
            /////////////////////////////////////////////////

            // Create Subname Titled File Item List.
            // It is temporarily empty until the first insert is performed below.
            // A File Item insert into this list must be performed as this is what gives the
            // Subname Titled File Item List a Subname Title.
            hFileItemList = _CreateNullSubnameTitledFileItemList();

            if(hFileItemList == NULL) {
                status = RFD_STATUS_ERROR_MEM_ALLOC;
                break;
            }

            // Insert Subname Titled File Item List into the list of Subname Titled File Item List.
            status = RFD_LLIST_Insert( hFileItemListList, hFileItemList );

            if(status != RFD_STATUS_OK) {
                // error.
                // In this case, the hFileItemList was locally allocated here
                // and since the insert failed, there's no other reference to hFileItemList
                // that would allow it to be cleaned up.
                // So cleanup hFileItemList here.
                _DeleteSubnameTitledFileItemList( hFileItemList );
                break;
            }
        }

        /////////////////////////////////////////////////////////////////
        // We now have a File Item List that was either
        // found in the the existing list of File Item Lists, or that was created above.
        // Insert the caller input File Item to this list, using the caller requested
        // find/replace configuration.
        // Note: in the case of this File Item List first created above,
        // this File Item instert is what gives the File Item List it's Subname.
        /////////////////////////////////////////////////////////////////

        status = _InsertFileItemToFileItemList(
                           hFileItemList,
                           hFileItem,
                           fileIdentityCompareMask,
                           doReplaceForMatchingFileIdentity );

        if(status != RFD_STATUS_OK) {
            break;
        }

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _OnWaitingFileItemsIdentityInfoUpdate( MULTIFILE_MANAGER_HANDLE hManager )
{
    ////////////////////////////////////////////////////////////////////////////
    // A change in File Identity Info on Waiting File Items has been done, so do
    // all actions needed upon this occurrence.
    ////////////////////////////////////////////////////////////////////////////

    // Indicate that the Waiting List is not currently sorted.
    hManager->fileManagementLists.isWaitingListCurrentlySorted = FALSE;

    return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _TryActiveFileItemUpdateOnRfdMetadata(
                        MULTIFILE_MANAGER_HANDLE hManager,
                        UCHAR * messageBuf,
                        UINT32 messageSize,
                        RFD_FSET_MDATA_STRUCT * messageMetadataInfo,
                        MFM_FILE_IDENTITY_HANDLE hMessageFileIdentity,
                        BOOL * isActiveFileItemPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_HANDLE fileItemList;
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;
    MFM_FILE_IDENTITY_MASK fileIdentityCompareMask;
    RFD_LLIST_NODE_HANDLE hFoundListNode = NULL;
    BOOL doListItemIdentityUpdate;
    BOOL compareResult;
    BOOL isFileItemFound, isFileIdentityUpdated;
    MFM_FILE_MANAGEMENT_POLICY_UPDATE_TYPE_ENUM updateTypePolicy =
        hManager->fileManagementPolicyInfo.updateType;

    // initialize output to false.
    *isActiveFileItemPtr = FALSE;

    isFileItemFound = FALSE;
    isFileIdentityUpdated = FALSE;

    do { // once

        if(updateTypePolicy == MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES) {

            /////////////////////////////////////////////////
            // Look for Identitiy match in the Active list.
            //
            // The following described action is valid for MFM Absolute Update Policy, not Differential Update:
            // In case there was just an OTA change in version info for this Subname,
            // update the version info for this file item.
            // This action is ok for item in the active list.
            // Make sure to put/post this update file metadata message
            // to the RFD Client, and client will detect change in the
            // Metadata Name field (from which version info is derived) and RFD Client
            // will automatically reset RFD collection and processing for the new file.
            //
            // Set doListIdentityUpdate = FALSE to disable identity updating by _FindFileItemInFileItemList().
            // This is because we are also checking for full identity match so as to increment the rfdMetadataMsgRxCount.
            // Any file identity update is then done afterwards.
            /////////////////////////////////////////////////

            fileItemList = hManager->fileManagementLists.activeFileItemList;

            // MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES
            // - compare just Subname first to dermine match/partial-match
            fileIdentityCompareMask = MFM_FILE_IDENTITY_SUBNAME;
            doListItemIdentityUpdate = FALSE;
            hFoundListNode = NULL;

            status = _FindFileItemInFileItemList(
                            fileItemList,
                            hMessageFileIdentity,
                            fileIdentityCompareMask,
                            doListItemIdentityUpdate,
                            &hFoundListNode );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if(hFoundListNode != NULL) {

                /////////////////////////////////////////////////////////////////////////////////////
                // Identity (Subname only) match found in Active List
                /////////////////////////////////////////////////////////////////////////////////////

                isFileItemFound = TRUE;

                hFileItem = RFD_LLIST_GetUserData(hFoundListNode, &status);
                if(status != RFD_STATUS_OK) {
                    break;
                }

                /////////////////////////////////////////////////////////////////////////////////////
                // Check for version info match
                /////////////////////////////////////////////////////////////////////////////////////

                // Compare file identity version info
                compareResult = MFM_UTIL_CompareFileIdentity(
                    hMessageFileIdentity,
                    &hFileItem->fileIdentityInfo,
                    MFM_FILE_IDENTITY_VERSION_INFO );

                if(!compareResult) {

                    isFileIdentityUpdated = FALSE;
                }
                else {
                    // Not an exact match.

                    isFileIdentityUpdated = TRUE;

                    // Absolute Policy specific
                    // Since the _Find in list performed above specified match by Subname only, the version info
                    // must have been different to cause mismatch. Update the File Item Identity so that
                    // the Subname and version will match (the subname already matches, so no need to update it again).

                    RFD_DPrint(
                        TEXT("MFM (%s, %d): Version Info Update of Update File Metadata Message detected for Active MFM File Item, updating File Identity Info.\n"),
                        hManager->mfmClientName,
                        hManager->mfmClientId );

                    // print "From"  File Identity
                    MFM_UTIL_PrintFileItemIdentity(
                        &hFileItem->fileIdentityInfo,
                        TEXT("MFM:"),
                        hManager->mfmClientName,
                        TEXT(" From File Identity") );

                    // print "To" File Identity
                    MFM_UTIL_PrintFileItemIdentity(
                        hMessageFileIdentity,
                        TEXT("MFM:"),
                        hManager->mfmClientName,
                        TEXT(" To File Identity") );

                    /////////////////////////////////////////////////////////////////////////////////////
                    // Perform the File Identity Update
                    /////////////////////////////////////////////////////////////////////////////////////
                    MFM_UTIL_UpdateFileIdentity(
                        &hFileItem->fileIdentityInfo,
                        hMessageFileIdentity,
                        MFM_FILE_IDENTITY_VERSION_INFO );
                }
            }
        }
        else if(updateTypePolicy == MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES) {

            /////////////////////////////////////////////////
            // Look for full Identitiy match in the Active list.
            /////////////////////////////////////////////////

            fileItemList = hManager->fileManagementLists.activeFileItemList;

            // MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES
            // - Do Full Compare (Subname and Version Info).
            fileIdentityCompareMask = MFM_FILE_IDENTITY_ALL;
            doListItemIdentityUpdate = FALSE;
            hFoundListNode = NULL;

            status = _FindFileItemInFileItemList(
                            fileItemList,
                            hMessageFileIdentity,
                            fileIdentityCompareMask,
                            doListItemIdentityUpdate,
                            &hFoundListNode );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if(hFoundListNode != NULL) {

                /////////////////////////////////////////////////////////////////////////////////////
                // Identity (Full - Subname and Version Info) match found in Active List
                /////////////////////////////////////////////////////////////////////////////////////

                isFileItemFound = TRUE;
                isFileIdentityUpdated = FALSE;

                hFileItem = RFD_LLIST_GetUserData(hFoundListNode, &status);
                if(status != RFD_STATUS_OK) {
                    break;
                }
            }
        }
        else {
            // error, unsupported policy
            status = RFD_STATUS_ERROR_INVALID_MODE;
            break;
        }

        /////////////////////////////////////////////////////////////////////////////////////
        // Do Common operations for both policy update types
        /////////////////////////////////////////////////////////////////////////////////////
        if(isFileItemFound == TRUE) {

            /////////////////////////////////////////////////////////////////////////////////////
            // Found matching File Item in the Active List.
            // Put message to the found Active File Item's RFD Client
            /////////////////////////////////////////////////////////////////////////////////////

            status = RFD_CollectorPutMessage(
                hFileItem->hMfmRfdClient->rfdMessageCollectorHandle,
                messageBuf,
                messageSize);

            if(status != RFD_STATUS_OK) {

                // Put Message Buffer may be full,
                // don't treat this as a hard failure, continue normal flow.

                RFD_DPrint( TEXT("MFM (%s, %d): _TryActiveFileItemUpdateOnRfdMetadata() message put failed, status %d\n"),
                    hManager->mfmClientName, hManager->mfmClientId, status);
            }

            /////////////////////////////////////////////////////////////////////////////////////
            // Update File Item Lifetime if it changed
            /////////////////////////////////////////////////////////////////////////////////////
            if( hFileItem->rfdLifeTime != messageMetadataInfo->lifeTime ) {

                // File Item Lifetime changed, update File Item.

                RFD_DPrint( TEXT("MFM (%s, %d): _TryActiveFileItemUpdateOnRfdMetadata(), RFD Lifetime updated from %d to %d\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    hFileItem->rfdLifeTime,
                    messageMetadataInfo->lifeTime );

                hFileItem->rfdLifeTime = messageMetadataInfo->lifeTime;

                // Set Life Updated flag to "File Item Expiration Processor", so that it knows
                // to perform Expiration Processing over the lists, on the next opportunity,
                // rather than wait for current-time change to perform the processing.
                _SetExpirationProcessInfoLifeUpdated( hManager );
            }

            /////////////////////////////////////////////////////////////////////////////////////
            // Update File Item Block Messages DMI if it changed, and take corresponding actions
            // for changed DMI value on Active File Item.
            /////////////////////////////////////////////////////////////////////////////////////
            if( hFileItem->rfdBlockMsgDmi != messageMetadataInfo->blockMsgDmi ) {

                // Block Messages DMI changed, update File Item and perform actions releated to change in Active DMI.

                RFD_DPrint(
                    TEXT("MFM (%s, %d): _TryActiveFileItemUpdateOnRfdMetadata(), RFD Block Msg DMI updated from %d to %d\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    hFileItem->rfdBlockMsgDmi,
                    messageMetadataInfo->blockMsgDmi );

                /////////////////////////////////////////////////////////////////////////
                // Do actions related to Deactivation of the current DMI for this Active File Item
                // (i.e. sends DMI status callback if this File Item was the last File Item
                // using this current DMI value, etc.).
                /////////////////////////////////////////////////////////////////////////

                status = _HandleDmiDeactivation( hManager, hFileItem->rfdBlockMsgDmi );
                if(status != RFD_STATUS_OK) {
                    break;
                }

                /////////////////////////////////////////////////////////////////////////
                // Next do actions related to Activation of the new DMI for this Active File Item
                // (i.e. sends DMI status callback if this File Item is the first File Item
                // using this current DMI value, etc.).
                /////////////////////////////////////////////////////////////////////////

                status = _HandleDmiActivation( hManager, messageMetadataInfo->blockMsgDmi );
                if(status != RFD_STATUS_OK) {
                    break;
                }

                // Finally, Update the File Item DMI.
                hFileItem->rfdBlockMsgDmi = messageMetadataInfo->blockMsgDmi;
            }

            /////////////////////////////////////////////////////////////////////////////////////
            // Update Message counts appropriately, depending on if File Item Identity was modified/updated or not.
            /////////////////////////////////////////////////////////////////////////////////////

            if(isFileIdentityUpdated == FALSE) {

                // File Item Identity not modified - Exact match of active List File Item Identity and received message
                // File Item Identity.
                // Increment the Update File Metadata Message Count. This is for Carousel Complete determination.
                hFileItem->rfdMetadataMsgRxCount++;

                // Handle File Item Message Count Increment (do Carousel Completion determination).
                _OnFileItemMessageCountInc(
                    hManager,
                    hFileItem,
                    MFM_FILE_ITEM_LIST_TYPE_ACTIVE );
            }
            else {
                // File Item Identity was modified (Not an exact match).

                // We are effectively replacing existing file item with new item (new version info),
                // so reset the message count to 1.
                hFileItem->rfdMetadataMsgRxCount = 1;
            }
        }

    } while(FALSE);

    // update output
    if(isFileItemFound == TRUE) {
        *isActiveFileItemPtr = TRUE;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _TryWaitingFileItemUpdateOnRfdMetadata(
                        MULTIFILE_MANAGER_HANDLE hManager,
                        UCHAR * messageBuf,
                        UINT32 messageSize,
                        RFD_FSET_MDATA_STRUCT * messageMetadataInfo,
                        MFM_FILE_IDENTITY_HANDLE hMessageFileIdentity,
                        BOOL * isWaitingFileItemPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_HANDLE fileItemListList;
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;
    MFM_FILE_IDENTITY_MASK fileIdentityCompareMask;
    RFD_LLIST_NODE_HANDLE hFoundListNode = NULL;
    BOOL doListItemIdentityUpdate;
    BOOL compareResult;
    BOOL isFileItemFound, isFileIdentityUpdated;
    MFM_FILE_MANAGEMENT_POLICY_UPDATE_TYPE_ENUM updateTypePolicy =
        hManager->fileManagementPolicyInfo.updateType;

    // initialize output to false.
    *isWaitingFileItemPtr = FALSE;

    isFileItemFound = FALSE;
    isFileIdentityUpdated = FALSE;

    do { // once

        if(updateTypePolicy == MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES) {

            /////////////////////////////////////////////////
            // Look for Identitiy match in the Waiting list.
            //
            // The following described action is valid for MFM Absolute Update Policy, not Differential Update:
            // In case there was just an OTA change in version info for this Subname,
            // update the version info for this file item.
            // This action is ok for item in the waiting list.
            // Make sure to update the buffered pdate file metadata message for for this waiting file item.
            //
            // Set doListIdentityUpdate = FALSE to disable identity updating by _FindFileItemInFileItemListList().
            // This is because we are also checking for full identity match so as to increment the rfdMetadataMsgRxCount.
            // Any file identity update is then done afterwards.
            /////////////////////////////////////////////////

            fileItemListList = hManager->fileManagementLists.waitingSubnameTitledFileItemListList;

            // MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES
            // - compare just Subname first to dermine match/partial-match
            fileIdentityCompareMask = MFM_FILE_IDENTITY_SUBNAME;
            doListItemIdentityUpdate = FALSE;
            hFoundListNode = NULL;

            status = _FindFileItemInFileItemListList(
                            fileItemListList,
                            hMessageFileIdentity,
                            fileIdentityCompareMask,
                            doListItemIdentityUpdate,
                            &hFoundListNode );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if(hFoundListNode != NULL) {

                /////////////////////////////////////////////////////////////////////////////////////
                // Identity (Subname only) match found in Waiting List
                /////////////////////////////////////////////////////////////////////////////////////

                isFileItemFound = TRUE;

                hFileItem = RFD_LLIST_GetUserData(hFoundListNode, &status);
                if(status != RFD_STATUS_OK) {
                    break;
                }

                /////////////////////////////////////////////////////////////////////////////////////
                // Check for version info match
                /////////////////////////////////////////////////////////////////////////////////////

                // Compare file identity version info
                compareResult = MFM_UTIL_CompareFileIdentity(
                    hMessageFileIdentity,
                    &hFileItem->fileIdentityInfo,
                    MFM_FILE_IDENTITY_VERSION_INFO );

                if(!compareResult) {

                    isFileIdentityUpdated = FALSE;
                }
                else {
                    // Not an exact match.

                    isFileIdentityUpdated = TRUE;

                    // Absolute Policy specific
                    // Since the _Find in list performed above specified match by Subname only, the version info
                    // must have been different to cause mismatch. Update the File Item Identity so that
                    // the Subname and version will match (the subname already matches, so no need to update it again).

                    RFD_DPrint(
                        TEXT("MFM (%s, %d): Version Info Update of Update File Metadata Message detected for Waiting MFM File Item, updating File Identity Info.\n"),
                        hManager->mfmClientName,
                        hManager->mfmClientId );

                    // print "From"  File Identity
                    MFM_UTIL_PrintFileItemIdentity(
                        &hFileItem->fileIdentityInfo,
                        TEXT("MFM:"),
                        hManager->mfmClientName,
                        TEXT(" From File Identity") );

                    // print "To" File Identity
                    MFM_UTIL_PrintFileItemIdentity(
                        hMessageFileIdentity,
                        TEXT("MFM:"),
                        hManager->mfmClientName,
                        TEXT(" To File Identity") );

                    /////////////////////////////////////////////////////////////////////////////////////
                    // Perform the File Identity Update
                    /////////////////////////////////////////////////////////////////////////////////////
                    MFM_UTIL_UpdateFileIdentity(
                        &hFileItem->fileIdentityInfo,
                        hMessageFileIdentity,
                        MFM_FILE_IDENTITY_VERSION_INFO );

                    // A change in File Identity Info on Waiting File Items has been done, so do
                    // all actions needed upon this occurrence.
                    status = _OnWaitingFileItemsIdentityInfoUpdate( hManager );

                    if(status != RFD_STATUS_OK) {
                        break;
                    }
                }
            }
        }
        else if(updateTypePolicy == MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES) {

            /////////////////////////////////////////////////
            // Look for full Identitiy match in the Active list.
            /////////////////////////////////////////////////

            fileItemListList = hManager->fileManagementLists.waitingSubnameTitledFileItemListList;

            // MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES
            // - Do Full Compare (Subname and Version Info).
            fileIdentityCompareMask = MFM_FILE_IDENTITY_ALL;
            doListItemIdentityUpdate = FALSE;
            hFoundListNode = NULL;

            status = _FindFileItemInFileItemListList(
                            fileItemListList,
                            hMessageFileIdentity,
                            fileIdentityCompareMask,
                            doListItemIdentityUpdate,
                            &hFoundListNode );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if(hFoundListNode != NULL) {

                /////////////////////////////////////////////////////////////////////////////////////
                // Identity (Full - Subname and Version Info) match found in Active List
                /////////////////////////////////////////////////////////////////////////////////////

                isFileItemFound = TRUE;
                isFileIdentityUpdated = FALSE;

                hFileItem = RFD_LLIST_GetUserData(hFoundListNode, &status);
                if(status != RFD_STATUS_OK) {
                    break;
                }
            }
        }
        else {
            // error, unsupported policy
            status = RFD_STATUS_ERROR_INVALID_MODE;
            break;
        }

        /////////////////////////////////////////////////////////////////////////////////////
        // Do Common operations for all policy update types
        /////////////////////////////////////////////////////////////////////////////////////
        if(isFileItemFound == TRUE) {

            /////////////////////////////////////////////////////////////////////////////////////
            // Found matching File Item in the Waiting List.
            // Update the RFD Message in case new message content is in any way different for this Waiting File Item.
            // Update function deletes any existing message and creates and copies in new message.
            // Do this message replacement even for exact identity match in case any field of this new message changed
            // versus previous messages for this item.
            /////////////////////////////////////////////////////////////////////////////////////
            status = _UpdateFileItemMessage(
                            hFileItem,
                            messageBuf,
                            messageSize );

            if(status != RFD_STATUS_OK) {
                break;
            }

            /////////////////////////////////////////////////////////////////////////////////////
            // Update File Item Lifetime if it changed
            /////////////////////////////////////////////////////////////////////////////////////
            if( hFileItem->rfdLifeTime != messageMetadataInfo->lifeTime ) {

                // File Item Lifetime changed, update File Item.

                RFD_DPrint( TEXT("MFM (%s, %d): _TryWaitingFileItemUpdateOnRfdMetadata(), RFD Lifetime updated from %d to %d\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    hFileItem->rfdLifeTime,
                    messageMetadataInfo->lifeTime );

                hFileItem->rfdLifeTime = messageMetadataInfo->lifeTime;

                // Set Life Updated flag to "File Item Expiration Processor", so that it knows
                // to perform Expiration Processing over the lists, on the next opportunity,
                // rather than wait for current-time change to perform the processing.
                _SetExpirationProcessInfoLifeUpdated( hManager );
            }

            /////////////////////////////////////////////////////////////////////////////////////
            // Update File Item Block Messages DMI if it changed
            /////////////////////////////////////////////////////////////////////////////////////
            if( hFileItem->rfdBlockMsgDmi != messageMetadataInfo->blockMsgDmi ) {

                // Block Messages DMI changed, update File Item.

                RFD_DPrint(
                    TEXT("MFM (%s, %d): _TryWaitingFileItemUpdateOnRfdMetadata(), RFD Block Msg DMI updated from %d to %d\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    hFileItem->rfdBlockMsgDmi,
                    messageMetadataInfo->blockMsgDmi );

                // In the case of Waiting File Items, no DMI information is reported to app.
                // (i.e. Active DMI processing and DMI status callbacks are only done for Active File Items)

                hFileItem->rfdBlockMsgDmi = messageMetadataInfo->blockMsgDmi;
            }

            /////////////////////////////////////////////////////////////////////////////////////
            // Update Message counts appropriately, depending on if File Item Identity was modified/updated or not.
            /////////////////////////////////////////////////////////////////////////////////////

            if(isFileIdentityUpdated == FALSE) {

                // File Item Identity not modified - Exact match of Waiting List File Item Identity and received message
                // File Item Identity.
                // Increment the Update File Metadata Message Count. This is for Carousel Complete determination.
                hFileItem->rfdMetadataMsgRxCount++;

                // Handle File Item Message Count Increment (do Carousel Completion determination).
                _OnFileItemMessageCountInc(
                    hManager,
                    hFileItem,
                    MFM_FILE_ITEM_LIST_TYPE_WAITING );
            }
            else {
                // File Item Identity was modified (Not an exact match).

                // We are effectively replacing existing file item with new item (new version info),
                // so reset the message count to 1.
                hFileItem->rfdMetadataMsgRxCount = 1;
            }
        }

    } while(FALSE);

    // update output
    if(isFileItemFound == TRUE) {
        *isWaitingFileItemPtr = TRUE;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _TryNewFileItemInsertToWaitingList(
                        MULTIFILE_MANAGER_HANDLE hManager,
                        UCHAR * messageBuf,
                        UINT32 messageSize,
                        RFD_FSET_MDATA_STRUCT * messageMetadataInfo,
                        MFM_FILE_IDENTITY_HANDLE hMessageFileIdentity,
                        BOOL * isWaitingListSizeIncrementedPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_HANDLE fileItemListList;
    MFM_FILE_ITEM_HANDLE hFileItem;
    MFM_FILE_IDENTITY_MASK fileIdentityCompareMask;
    BOOL doReplaceForMatchingFileIdentity;
    BOOL isWaitingListSizeIncremented;
    MFM_FILE_MANAGEMENT_POLICY_UPDATE_TYPE_ENUM updateTypePolicy =
        hManager->fileManagementPolicyInfo.updateType;

    // initialize output to false.
    *isWaitingListSizeIncrementedPtr = FALSE;

    isWaitingListSizeIncremented = FALSE;

    do { // once

        if(updateTypePolicy == MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES) {

            RFD_DPrint(
                TEXT(" Inserting new File Item to the Waiting List.\n"),
                hManager->mfmClientName,
                hManager->mfmClientId );

            // print File Identity
            MFM_UTIL_PrintFileItemIdentity(
                hMessageFileIdentity,
                TEXT("MFM:"),
                hManager->mfmClientName,
                TEXT(" File Identity") );

            /////////////////////////////////////////////////
            // Create File Item.
            /////////////////////////////////////////////////

            hFileItem = _CreateFileItem(
                            hMessageFileIdentity,
                            NULL, // hMfmRfdClient not assigned in waiting list.
                            messageBuf, // messageBuf is assigned for file item in waiting list.
                            messageSize,
                            1, /* initial count of 1 for this message */
                            messageMetadataInfo );

            if(hFileItem == NULL) {
                // error
                status = RFD_STATUS_ERROR_MEM_ALLOC;
                break;
            }

            /////////////////////////////////////////////////
            // Insert File Item to the Waiting List.
            /////////////////////////////////////////////////

            fileItemListList = hManager->fileManagementLists.waitingSubnameTitledFileItemListList;
            fileIdentityCompareMask = MFM_FILE_IDENTITY_SUBNAME; // _SUBNAME is for Absolute Policy.
            doReplaceForMatchingFileIdentity = TRUE; // Not necessary since we've already searched, but it's ok to set TRUE.

            status = _InsertFileItemToFileItemListList(
                fileItemListList,
                hFileItem,
                fileIdentityCompareMask,
                doReplaceForMatchingFileIdentity );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            // Indicate File Item was added to the WaitingList(List).
            isWaitingListSizeIncremented = TRUE;

            // A change in File Identity Info on Waiting File Items has been done, so do
            // all actions needed upon this occurrence.
            status = _OnWaitingFileItemsIdentityInfoUpdate( hManager );

            if(status != RFD_STATUS_OK) {
                break;
            }
        }
        else if(updateTypePolicy == MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES) {
            // currently unsupported.
            // note: need to know the file item version number the application is currently at.
            //       if this file item version number is less than application current version number,
            //       then don't insert to waiting list.
            status = RFD_STATUS_ERROR_INVALID_MODE;
            break;
        }
        else {
            // unsupported mode:
            status = RFD_STATUS_ERROR_INVALID_MODE;
            break;
        }

    } while(FALSE);

    // update output
    *isWaitingListSizeIncrementedPtr = isWaitingListSizeIncremented;

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ProcessAppInServiceFileItem(
                        MULTIFILE_MANAGER_HANDLE hManager,
                        UCHAR * messageBuf,
                        UINT32 messageSize,
                        RFD_FSET_MDATA_STRUCT * messageMetadataInfo,
                        MFM_FILE_IDENTITY_HANDLE hMessageFileIdentity,
                        BOOL isRfdLifeExpired,
                        BOOL * isWaitingListSizeIncrementedPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isFileItemFound;

    // initialize output to false.
    *isWaitingListSizeIncrementedPtr = FALSE;

    do { // once

        /////////////////////////////////////////////////
        // Try to *Update* an existing File Item even if RFD Life is expired.
        // This is in case the Life field value of the existing File Item
        // is updated by this Metadata, from value not expired to value expired.
        // In this case, we want to update the Life field so that the File Item
        // will get removed on the next processing of File Item RFD Life Expirations.
        /////////////////////////////////////////////////

        /////////////////////////////////////////////////
        // First look for Identitiy match in the Active list.
        /////////////////////////////////////////////////

        isFileItemFound = FALSE;

        status = _TryActiveFileItemUpdateOnRfdMetadata(
                        hManager,
                        messageBuf,
                        messageSize,
                        messageMetadataInfo,
                        hMessageFileIdentity,
                        &isFileItemFound );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(isFileItemFound == TRUE) {
            // match found and processed for existing Active File Item, work complete.
            break;
        }

        /////////////////////////////////////////////////
        // Identity Not found in the Active list.
        /////////////////////////////////////////////////

        /////////////////////////////////////////////////
        // Next look for Identitiy match in the Waiting list.
        /////////////////////////////////////////////////

        isFileItemFound = FALSE;

        status = _TryWaitingFileItemUpdateOnRfdMetadata(
                        hManager,
                        messageBuf,
                        messageSize,
                        messageMetadataInfo,
                        hMessageFileIdentity,
                        &isFileItemFound );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(isFileItemFound == TRUE) {
            // match found and processed for existing Active File Item, work complete.
            break;
        }

        /////////////////////////////////////////////////
        // Identity Not found in the Waiting list or Active list
        /////////////////////////////////////////////////

        // Try new file item insert to waiting list only if RFD Life is not expired for this item.
        if( isRfdLifeExpired == FALSE ) {

            /////////////////////////////////////////////////
            // Try insert of this File Item into the Waiting list
            // This is done only if the RFD Life is not expired for this File Item
            // This saves unneeded cycle of adding waiting item, possible activating it,
            // then subsequently deactivating it when the expire handling proccessing
            // is eventually done.
            /////////////////////////////////////////////////

            status = _TryNewFileItemInsertToWaitingList(
                            hManager,
                            messageBuf,
                            messageSize,
                            messageMetadataInfo,
                            hMessageFileIdentity,
                            isWaitingListSizeIncrementedPtr );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }
        }

        /////////////////////////////////////////////////
        // "Activation" attempt of Waiting file items is not done here.
        // Its done at a higher level.
        /////////////////////////////////////////////////

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ManagerPostRfdFileTransferInfo(
                MULTIFILE_MANAGER_HANDLE hManager,
                RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle,
                RFD_CONSUMER_FILE_INFO_STRUCT * consumerFileInfo )
{
    RFD_STATUS status;
    MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT * postInfoStructPtr;
    UINT32 postInfoStructSize;
    RFD_MSG_QUEUE_HANDLE messageQueueHandle =
        hManager->fileConsumerInfo.hMfmFileTransferMessageQueue;

    ///////////////////////////////
    // Allocate a single structure that will be posted.
    ///////////////////////////////
    postInfoStructSize = sizeof(MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT);

    postInfoStructPtr = (MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT *) RFD_MALLOC(postInfoStructSize);

    if(postInfoStructPtr == NULL) {
        return RFD_STATUS_ERROR_MEM_ALLOC;
    }

    ///////////////////////////////
    // populate info to the allocated structure.
    ///////////////////////////////
    postInfoStructPtr->rfdFileConsumerHandle = rfdFileConsumerHandle;
    postInfoStructPtr->rfdConsumerFileInfo = *consumerFileInfo;

    ///////////////////////////////
    // post.
    ///////////////////////////////
    status = RFD_MSG_QUEUE_PutMessage(
                    messageQueueHandle,
                    (UCHAR *) postInfoStructPtr,
                    postInfoStructSize );

    if(status != RFD_STATUS_OK) {
        // error.

        // This implementation RFD_MSG_QUEUE_PutMessage() does not support blocking indefinately
        // in the case of message que buffer full. The buffer size and system design should be so
        // as to ensure that this buffer full condition does not happen.
        RFD_DPrint( TEXT("MFM (%s, %d): _ManagerPostRfdFileTransferInfo(), RFD_MSG_QUEUE_PutMessage(), status %d\n"),
            hManager->mfmClientName, hManager->mfmClientId, status );
    }

    ///////////////////////////////
    // free the allocated structure.
    ///////////////////////////////
    if(postInfoStructPtr != NULL) {
        RFD_FREE(postInfoStructPtr);
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
// _MfmConsumerFileTransferCallback()
// The MFM callback function called by the MFM Consumer upon a newly available
// RFD Update File.
// This function in turn calls the Application provided callback function,
// hManager->appCallbackInfo.appRfdFileTransferCallbackFcn, which completes
// the transfer of control of the Update File to the application.
// The application callback function is responsible for deleting the Update File.
// If the application returns FALSE, indicating error in transferring control of
// the Update File, then RFD_ConsumerResetFileCollectionRequest() is
// called here to reset RFD collection and processing of the Update File to thus
// attempt reprocessing of the Update File.
///////////////////////////////////////////////////////////////////////////////
static void _MfmConsumerFileTransferCallback (
        RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle,
        const TCHAR storedFileName[],
        RFD_CONSUMER_FILE_INFO_STRUCT * consumerFileInfo,
        void * mfmManagerFileTransferCallbackArg )
{
    MULTIFILE_MANAGER_HANDLE hManager;
    MFM_APP_RFD_FILE_TRANSFER_CALLBACK_FCN appRfdFileTransferCallbackFcn;
    void * appRfdFileTransferCallbackArg;
    BOOL isOk;
    RFD_STATUS status;

    hManager = (MULTIFILE_MANAGER_HANDLE) mfmManagerFileTransferCallbackArg;

    RFD_DPrint(
        TEXT("MFM (%s, %d) _MfmConsumerFileTransferCallback() called on RFD Client ID (%d)\n"),
        hManager->mfmClientName, hManager->mfmClientId,
        rfdFileConsumerHandle->clientInfo->clientIdNo);

    appRfdFileTransferCallbackFcn =
        hManager->appCallbackInfo.appRfdFileTransferCallbackFcn;

    appRfdFileTransferCallbackArg =
        hManager->appCallbackInfo.appRfdFileTransferCallbackArg;

    isOk = appRfdFileTransferCallbackFcn(
                storedFileName,
                consumerFileInfo,
                appRfdFileTransferCallbackArg );

    if(isOk == TRUE) {

        // We are running in the context of the MFM Consumer thread.
        // Post the File Transfer info to queue, to be handled by the
        // MFM Manager in context of the "Producer" thread thats receiving
        // RFD Messages. The posting instead of synchronous updating
        // Manager structures with mutex locking is done to avoid
        // potential interactive deadlock issues in case of lower level
        // callbacks to application functions (e.g. for sorting priority)
        // that may also have their own lock.

        status = _ManagerPostRfdFileTransferInfo(
                        hManager,
                        rfdFileConsumerHandle,
                        consumerFileInfo );

        if(status != RFD_STATUS_OK) {
            // unexpected error.
            // This post should never fail, but as failsafe action, reset
            // the rfd file collection/processing to restart collection of the same
            // file. This will result in the application eventually receiving the same
            // file just received.

            RFD_DPrint(
                TEXT("MFM (%s, %d): _MfmConsumerFileTransferCallback(), _ManagerPostRfdFileTransferInfo() failed to post, status (%d),\n")
                TEXT("... calling RFD_ConsumerResetFileCollectionRequest(), RFD Client ID (%d)\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status,
                rfdFileConsumerHandle->clientInfo->clientIdNo );

            RFD_ConsumerResetFileCollectionRequest( rfdFileConsumerHandle );
        }
    }
    else {
        // Application indicates that the RFD File was not successfully transferred,
        // and that the rfd collection/processing process should be reset to try again.

        RFD_DPrint(
            TEXT("MFM (%s, %d): _MfmConsumerFileTransferCallback(), appRfdFileTransferCallbackFcn() indicates transfer fail,\n")
            TEXT("... calling RFD_ConsumerResetFileCollectionRequest(), RFD Client ID (%d)\n"),
            hManager->mfmClientName, hManager->mfmClientId,
            rfdFileConsumerHandle->clientInfo->clientIdNo);

        // Reset the RFD file collection/processing for this file.
        RFD_ConsumerResetFileCollectionRequest( rfdFileConsumerHandle );
    }

    // Delete the stored RFD File in case the application did not delete it
    // or did not move it to application owned storage as required.
    RFD_DELETE_FILE( storedFileName );

    return;
}

///////////////////////////////////////////////////////////////////////////////
static void _UninitializeRfdMessageProcessingInfo(
        MFM_RFD_MESSAGE_PROCESS_INFO_STRUCT * mfmRfdMessageProcessInfo )
{
    // rfdDefaultMessageConfigInfo: is not allocated/freed by MFM,
    // just set to null.
    mfmRfdMessageProcessInfo->rfdDefaultMessageConfigInfo = NULL;

    // rfdUpdateFileMetadataInfo: is allocated/freed by MFM.
    if(mfmRfdMessageProcessInfo->rfdUpdateFileMetadataInfo != NULL) {
        RFD_FREE(mfmRfdMessageProcessInfo->rfdUpdateFileMetadataInfo);
        mfmRfdMessageProcessInfo->rfdUpdateFileMetadataInfo = NULL;
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InitializeRfdMessageProcessingInfo(
                MFM_RFD_MESSAGE_PROCESS_INFO_STRUCT * mfmRfdMessageProcessInfo,
                RFD_COLLECTOR_THREAD_DATA_HANDLE hDefaultRfdMessageCollector )
{

    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        ///////////////////////////////////////////
        // On initialization, indicate that the complete Carousel of rfd update file metadata messages
        // has not yet been received by MFM.
        ///////////////////////////////////////////
        mfmRfdMessageProcessInfo->isRfdUpdateFileMetadataCarouselComplete = FALSE;

        ///////////////////////////////////////////
        // Initialize the pointer to the Default RFD Message Config Info.
        //
        // This is used for Message Parsing
        // - Parsing configuration of all the available RFD Clients must be the same.
        //
        // NOTE: any future enhancement to make the RFD Clients dynamically created/deleted
        //   during runtime needs to be aware that this Default RFD Message Config Info
        //   belongs to the RFD Client from which it is obtained, and can therefore be
        //   deleted if that RFD Client is deleted. This Default RFD Message Config Info
        //   therefore needs to be made more permantent/independent of RFD Clients
        //   for any 'dynamic' RFD Client implementation.
        ///////////////////////////////////////////

        status = RFD_CollectorGetDefaultMessageConfig(
            hDefaultRfdMessageCollector,
            &mfmRfdMessageProcessInfo->rfdDefaultMessageConfigInfo);

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Allocate memory for RFD_FSET_MDATA_STRUCT structure.
        // This is used in parsing of each received
        // RFD Update File Metadata message.
        ///////////////////////////////////////////

        mfmRfdMessageProcessInfo->rfdUpdateFileMetadataInfo =
            (RFD_FSET_MDATA_STRUCT *) RFD_CALLOC(sizeof(RFD_FSET_MDATA_STRUCT));

        if(mfmRfdMessageProcessInfo->rfdUpdateFileMetadataInfo == NULL) {
            // error
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup.
        _UninitializeRfdMessageProcessingInfo( mfmRfdMessageProcessInfo );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteMfdRfdClientHandle(MFM_RFD_CLIENT_HANDLE mfdRfdClientHandle)
{
    if(mfdRfdClientHandle != NULL) {
        RFD_FREE(mfdRfdClientHandle);
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteRfdClientResourceLists(
         MFM_RFD_CLIENT_RESOURCE_INFO_STRUCT * rfdClientResourceInfo )
{
    // delete busy client list
    RFD_LLIST_DeleteList(
        rfdClientResourceInfo->busyMfmRfdClientList,
        (RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN)_DeleteMfdRfdClientHandle );

    rfdClientResourceInfo->busyMfmRfdClientList = NULL;

    // delete free client list
    RFD_LLIST_DeleteList(
        rfdClientResourceInfo->freeMfmRfdClientList,
        (RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN)_DeleteMfdRfdClientHandle );

    rfdClientResourceInfo->freeMfmRfdClientList = NULL;

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _CreateRfdClientResourceLists(
                MFM_RFD_CLIENT_RESOURCE_INFO_STRUCT * rfdClientResourceInfo )
{
    RFD_STATUS status = RFD_STATUS_OK;

    rfdClientResourceInfo->busyMfmRfdClientList = NULL;
    rfdClientResourceInfo->freeMfmRfdClientList = NULL;

    do { // once

        //////////////////////////////////////////
        // Create the Free client list, initially empty
        //////////////////////////////////////////

        rfdClientResourceInfo->freeMfmRfdClientList = RFD_LLIST_CreateList();

        if(rfdClientResourceInfo->freeMfmRfdClientList == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        //////////////////////////////////////////
        // Create the Busy client list, initially empty.
        //////////////////////////////////////////

        rfdClientResourceInfo->busyMfmRfdClientList = RFD_LLIST_CreateList();

        if(rfdClientResourceInfo->busyMfmRfdClientList == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        // success
        status = RFD_STATUS_OK;

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        _DeleteRfdClientResourceLists(rfdClientResourceInfo);
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InitializeFromRfdClient(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_RFD_CLIENT_HANDLE mfmRfdClientHandle )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_HANDLE mfmRfdClientList;
    RFDR_CLIENT_STATE rfdClientState;
    RFD_FSET_MDATA_STRUCT * rfdUpdateFileMetadataInfo = NULL;
    MFM_FILE_IDENTITY_PARSE_CALLBACK_FCN mfmFileIdentityParseCallbackFcn;
    void * mfmFileIdentityParseCallbackArg;
    MFM_FILE_IDENTITY_HANDLE hFileIdentity = NULL;
    BOOL isFileItemActivated = FALSE;
    BOOL isOk;
    RFD_LLIST_HANDLE fileItemList;
    MFM_FILE_ITEM_HANDLE hFileItem;
    MFM_FILE_IDENTITY_MASK fileIdentityCompareMask;
    RFD_LLIST_NODE_HANDLE hFoundStoredListNode = NULL;
    RFD_LLIST_NODE_HANDLE hFoundActiveListNode = NULL;
    BOOL doListItemIdentityUpdate, doReplaceForMatchingFileIdentity;
    BOOL isLifeExpired = FALSE;
    BOOL isCurrentTimeAvailable;
    time_t currentSimpleCalendarTime;

    /////////////////////////////////////////////////////////////////////////////////////
    // Do initialization related to the initial state of the input RFD Client.
    // In summary:
    // Get the state info of the RFD Client, which include File Identity if state is in-progress.
    // If the RFD Client is in the In-Progress state, then:
    //   - Add the RFD Client the Busy RFD Client List
    //   - Create a Corresponding File Item from the RFD Client handle and File Identity info.
    //   - Add the File Item to the Active File Item List.
    // Else RFD Client is in the Reset or Complete state, then:
    //   - just add the RFD Client to the Free RFD Client List.
    //
    // This function is common for both Absolute and Delta MFM Policy.
    /////////////////////////////////////////////////////////////////////////////////////

    do { // once

        // Reuse common pre-allocated buffer to hold metadata structure.
        rfdUpdateFileMetadataInfo =
            hManager->mfmRfdMessageProcessInfo.rfdUpdateFileMetadataInfo;

        /////////////////////////////////////////////////////////////////////////////////////
        // Get State info from RFD
        // if state is in progress
        //      if Life Not Expired
        //          generate file identity info
        //          do validity check with stored list
        //          if not in stored list (expected)
        //              insert client to busy list
        //              create file item from identity
        //              insert file item to active list list
        //          else (in stored list with exact match)
        //              Reset RFD Client
        //              insert client to free list
        //          end
        //      end
        //      if file item Not Activated
        //          Reset Client Collection (cleanup resources)
        //      end
        // else state complete or reset
        //      insert client to free list
        // end
        /////////////////////////////////////////////////////////////////////////////////////

        // Get State info for this RFD client
        status = RFD_CollectorGetClientStateInfo(
                            mfmRfdClientHandle->rfdMessageCollectorHandle,
                            &rfdClientState,
                            rfdUpdateFileMetadataInfo);

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d): _InitializeFromRfdClient(), RFD_CollectorGetClientStateInfo() error. status (%d)\n"),
                hManager->mfmClientName, hManager->mfmClientId, status);
            break;
        }

        // Initialize to false
        isFileItemActivated = FALSE;

        if( rfdClientState == RFDR_CLIENT_STATE_IN_PROGRESS ) {

            // Allocate the MFM File Identity structure
            hFileIdentity = (MFM_FILE_IDENTITY_HANDLE) RFD_MALLOC(sizeof(MFM_FILE_IDENTITY_STRUCT));

            if(hFileIdentity == NULL) {
                // error
                status = RFD_STATUS_ERROR_MEM_ALLOC;
                break;
            }

            /////////////////////////////////////////////////////////////////////////////////////
            // Get the current time and check if the Life of this Update File currently in progress
            // is expired.
            /////////////////////////////////////////////////////////////////////////////////////

            isCurrentTimeAvailable = _GetCurrentSimpleCalendarTime( hManager, &currentSimpleCalendarTime );

            status = _IsRfdLifeExpired(
                            hManager,
                            rfdUpdateFileMetadataInfo->lifeTime,
                            currentSimpleCalendarTime,
                            isCurrentTimeAvailable,
                            &isLifeExpired );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if( isLifeExpired == FALSE ) {

                // Life not expired, continue intitialization.

                /////////////////////////////////////////////////////////////////////////////////////
                // Call the user provided callback funcion to
                // parse the RFD Metadata Name field into the MFM File Identity structure
                /////////////////////////////////////////////////////////////////////////////////////

                mfmFileIdentityParseCallbackFcn =
                    hManager->appCallbackInfo.mfmFileIdentityParseCallbackFcn;
                mfmFileIdentityParseCallbackArg =
                    hManager->appCallbackInfo.mfmFileIdentityParseCallbackArg;

                isOk = mfmFileIdentityParseCallbackFcn(
                                rfdUpdateFileMetadataInfo->name,
                                hFileIdentity, // input, called fcn. fills out the data members of struct.
                                mfmFileIdentityParseCallbackArg );

                if(isOk == TRUE) {
                    // Successful parse

                    /////////////////////////////////////////////////
                    // Check if this File Identity Subname is In-Service for the app.
                    // (i.e. check if this is a Subname that the app wants MFM to process.)
                    /////////////////////////////////////////////////

                    BOOL isSubnameInService;
                    UINT32 appStoredFileIdentifyVersionTo = MFM_FILE_IDENTITY_VERSION_NUM_UNUSED;
                    MFM_APP_IS_SUBNAME_IN_SERVICE_CALLBACK_FCN mfmIsSubnameInServiceFcn;
                    void * mfmIsSubnameInServiceArg;

                    mfmIsSubnameInServiceFcn =
                        hManager->appCallbackInfo.appIsSubnameInServiceCallbackFcn;
                    mfmIsSubnameInServiceArg =
                        hManager->appCallbackInfo.appIsSubnameInServiceCallbackArg;

                    isSubnameInService = mfmIsSubnameInServiceFcn(
                                                hFileIdentity->subname,
                                                &appStoredFileIdentifyVersionTo,
                                                mfmIsSubnameInServiceArg );

                    if( isSubnameInService == TRUE ) {

                        /////////////////////////////////////////////////
                        // This File Identity Subname is In-Service for the application.
                        /////////////////////////////////////////////////

                        if(hManager->fileManagementPolicyInfo.updateType
                            == MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES) {

                            /////////////////////////////////////////////////
                            // Absolute Update model, follow version comparison rules specific to this mode.
                            /////////////////////////////////////////////////

                            if( hFileIdentity->versionTo != appStoredFileIdentifyVersionTo ) {

                                /////////////////////////////////////////////////
                                // This File Identity Subname is In-Service for the application (checked above),
                                // and the File Identity VersionTo is not the same value already Stored by the App.
                                // So this File Item should continue being processed by this RFD Client,
                                // and be added to the Active File Item list.
                                /////////////////////////////////////////////////

                                /////////////////////////////////////////////////
                                // Also check that the Identity is not already in the Active list
                                // e.g. if this is a duplicate entry processing the same File Item as a previously
                                // initialized RFD Client / File Item. for a File Item
                                /////////////////////////////////////////////////

                                fileItemList = hManager->fileManagementLists.activeFileItemList;
                                // Do full identity compare (subname and version info).
                                fileIdentityCompareMask = MFM_FILE_IDENTITY_ALL;
                                // don't update identity item in list.
                                doListItemIdentityUpdate = FALSE;
                                hFoundActiveListNode = NULL;

                                status = _FindFileItemInFileItemList(
                                                fileItemList,
                                                hFileIdentity,
                                                fileIdentityCompareMask,
                                                doListItemIdentityUpdate,
                                                &hFoundActiveListNode );

                                if(status != RFD_STATUS_OK) {
                                    // error
                                    break;
                                }

                                if( hFoundActiveListNode == NULL ) {

                                    // Complete Identitity Match (Subname and Version) not found in the Active List.
                                    // This is the expected result.

                                    /////////////////////////////////////////////////
                                    // Create File Item from File Identity, and instert to the Active List.
                                    /////////////////////////////////////////////////

                                    RFD_DPrint(
                                        TEXT("MFM (%s, %d): _InitializeFromRfdClient(): Found RFD Client in In-Progress state (RFD Client ID %d).")
                                        TEXT("            : Initializing MFM with corresponding Active File Item\n\n"),
                                        hManager->mfmClientName,
                                        hManager->mfmClientId,
                                        mfmRfdClientHandle->rfdMessageCollectorHandle->clientInfo->clientIdNo );

                                    // Create File Item from the File Identity and RFD Client info (there is no correponding message).

                                    // First overwrite the Metadata Info Block Msg DMI with RFD_DMI_UNUSED_VALUE (which is used in
                                    // initializing the hFileItem DMI).
                                    // This avoids generating a DMI status callback to the app from this function
                                    // (called from MFM_Create()), which might not be desired.
                                    // It also avoids sending potentially stale/old DMI info to app in the callback
                                    // (e.g. powered down for long time).
                                    // On receiving the first RFD Update File Metadata for this File Item, a change in DMI value
                                    // will be detected, triggering an update to the DMI, a change in the Active DMI list and
                                    // a DMI status callback will then be generated with the correct DMI info.

                                    rfdUpdateFileMetadataInfo->blockMsgDmi = RFD_DMI_UNUSED_VALUE;

                                    hFileItem = _CreateFileItem(
                                        hFileIdentity,
                                        mfmRfdClientHandle,
                                        NULL, // messageBuf is NULL.
                                        0,    // message size is 0.
                                        0,     // initial msg count of 0 since there's no corresponding message in this case
                                        rfdUpdateFileMetadataInfo );

                                    if(hFileItem == NULL) {
                                        // error
                                        status = RFD_STATUS_ERROR_MEM_ALLOC;
                                        break;
                                    }

                                    // print File Item
                                    _PrintFileItem(
                                        hFileItem,
                                        TEXT("MFM:"),
                                        hManager->mfmClientName,
                                        TEXT(" File Item") );

                                    // Insert File Item to Active List.

                                    fileItemList = hManager->fileManagementLists.activeFileItemList;
                                    fileIdentityCompareMask = MFM_FILE_IDENTITY_ALL;
                                    // Not necessary to do replace since we've already searched, but it's ok to set TRUE.
                                    doReplaceForMatchingFileIdentity = TRUE;

                                    status = _InsertFileItemToFileItemList(
                                        fileItemList,
                                        hFileItem,
                                        fileIdentityCompareMask,
                                        doReplaceForMatchingFileIdentity );

                                    if(status != RFD_STATUS_OK) {
                                        // error
                                        break;
                                    }

                                    // Do processing/updating for Active DMI on this File Item Activation.
                                    // Note: for this special case Activation on initialization, the hFileItem->rfdBlockMsgDmi
                                    // should be overwritten to RFD_DMI_UNUSED_VALUE as noted above on File Item Creation.
                                    status = _HandleDmiActivation( hManager, hFileItem->rfdBlockMsgDmi );

                                    if(status != RFD_STATUS_OK) {
                                        // error
                                        break;
                                    }

                                    /////////////////////////////////////////////////////////////////////////////////////
                                    // Set File Item Activated flag.
                                    /////////////////////////////////////////////////////////////////////////////////////
                                    isFileItemActivated = TRUE;
                                }
                                else {

                                    /////////////////////////////////////////////////////////////////////////////////////
                                    // Unexpectedly found exact Identity in the Stored list or the Active list.
                                    //
                                    // (there may be a very low probability case where Identity could be recorded as Stored
                                    // by the application and the RFD Client is still In-Progress - i.e. if power down
                                    // at very end of an RFD File Transfer / Handoff to the application, before return to
                                    // RFD to advance the RFD processing state.)
                                    //
                                    // Reset RFD Client to clear client procesing info. The reset will be done at common point
                                    // below for all cases where a Client State is IN-PROGRESS but was not Activated.
                                    /////////////////////////////////////////////////////////////////////////////////////

                                    if( hFoundStoredListNode == NULL ) {

                                        RFD_DPrint( TEXT("MFM (%s, %d): _InitializeFromRfdClient() found In-Progress RFD Client with equal MFM File Identity in MFM Stored List\n"),
                                            hManager->mfmClientName,
                                            hManager->mfmClientId );
                                    }
                                    else { // hFoundActiveListNode == NULL
                                        RFD_DPrint( TEXT("MFM (%s, %d): _InitializeFromRfdClient() - warning, found In-Progress RFD Client with equal MFM File Identity already in MFM Active List\n"),
                                            hManager->mfmClientName,
                                            hManager->mfmClientId );
                                    }
                                }
                            }
                        }
                        else {
                            // error, unsupported mode.
                            // For supporting Delta mode of operation in future, probably need to
                            // check for hFileIdentity->versionTo >= appStoredFileIdentifyVersionTo
                            // instead of just inequality.
                            status = RFD_STATUS_UNSUPPORTED_FEATURE;
                            break;
                        }
                    }
                }
                else {
                    /////////////////////////////////////////////////////////////////////////////////////
                    // File Identity Parser failed to parse the Metadata Name field provided from the RFD Client.
                    // It's possible that the content of the name is somehow incompatible with the
                    // structure expected for this Parser (e.g. maybe some part of the Name is required
                    // to contain some expected string, or the version info field contained non-numeric
                    // characters. This could imply this RFD Client was utilized for a different
                    // Data Service with different Name format on the previous session. Therefore
                    // request Reset the RFD Client to cleanup file system resources for previous session
                    // and treat this RFD Client as now Free. The reset will be done at common point
                    // below for all cases where a Client State is IN-PROGRESS but was not Activated.
                    //
                    // NOTE: This mechanism of detetecting possible reassignment of RFD Clients across
                    // different Data Services should not be relied on, because in some cases, the
                    // format of the Name field across the different Data Services may be compatible
                    // and the reassignment will not be detected. This could then lead to the RFD Client
                    // being permanently assigned to processing a non-existent File Update on this
                    // MFM managed Data Service - no RFD messages will arrive to complete a File Update
                    // and this RFD Client resource will be completely wasted as it never becomes free.
                    // If RFD Clients are every reassigned across Data Services in this manner, the
                    // Application should reset the RFD Client by calling RFD_ConsumerResetFileCollectionRequest().
                    // This should only be done by the application when there is an RFD Client reassignment;
                    // that is, the reset must NOT be done on every power up as this would erase all
                    // previous RFD block collection and decoding still in progress from a previous power-on
                    // session.
                    /////////////////////////////////////////////////////////////////////////////////////

                    RFD_DPrint( TEXT("MFM (%s, %d): _InitializeFromRfdClient() File Identity Parser failed to parse Metadata Name \"%s\", RFD Client reset was requested.\n"),
                        hManager->mfmClientName, hManager->mfmClientId, rfdUpdateFileMetadataInfo->name );
                }

                // Finished with File Identity, clean it up.
                RFD_FREE(hFileIdentity);
                hFileIdentity = NULL;

            }
            else {
                // Life is expired
                RFD_DPrint( TEXT("MFM (%s, %d): _InitializeFromRfdClient() Found expired Update File for Client Id %d with Metadata Name \"%s\".\n"),
                        hManager->mfmClientName,
                        hManager->mfmClientId,
                        mfmRfdClientHandle->rfdFileConsumerHandle->clientInfo->clientIdNo,
                        rfdUpdateFileMetadataInfo->name );
            }

            if( isFileItemActivated == FALSE ) {
                /////////////////////////////////////////////////////////////////////////////////////
                // RFD Client State is IN-PROGRESS, but File Item was not Activated for one of the reasons determined above.
                // Reset the File Collection for this client, to cleanup file collection
                // processing resources (file system, etc.)
                /////////////////////////////////////////////////////////////////////////////////////

                RFD_DPrint(
                    TEXT("MFM (%s, %d): _InitializeFromRfdClient() Resetting Update File collection for Client Id: %d, Update File Name \"%s\".\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    mfmRfdClientHandle->rfdFileConsumerHandle->clientInfo->clientIdNo,
                    rfdUpdateFileMetadataInfo->name );

                RFD_ConsumerResetFileCollectionRequest( mfmRfdClientHandle->rfdFileConsumerHandle );
            }

        } // end Client State IN-PROGRESS

        if( isFileItemActivated == TRUE ) {

            // RFD client state is either RFDR_CLIENT_STATE_RESET or RFDR_CLIENT_STATE_COMPLETE,
            // or the RFD client state was RFDR_CLIENT_STATE_IN_PROGRESS but was not activated for some other reason.

            mfmRfdClientList = hManager->rfdClientResourceInfo.busyMfmRfdClientList;
        }
        else {
            // File Item was not activated.

            // This RFD client belongs in the Free RFD Client List.
            mfmRfdClientList = hManager->rfdClientResourceInfo.freeMfmRfdClientList;
        }

        // Insert the RFD client info handle into RFD client list seleted above.
        status = RFD_LLIST_Insert(
                        mfmRfdClientList,
                        mfmRfdClientHandle );

        if(status != RFD_STATUS_OK) {
            break;
        }

    } while(FALSE);

    if(hFileIdentity != NULL) {
        // Cleanup File Identity in case of error case exit.
        RFD_FREE(hFileIdentity);
        hFileIdentity = NULL;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InitializeFromRfdInitialState(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_RFD_CLIENT_STRUCT mfmRfdClientArray[],
                UINT16 numRfdClients )
{
    UINT16 i;
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        for(i=0;i<numRfdClients;i++) {

            MFM_RFD_CLIENT_HANDLE mfmRfdClientHandle;

            // allocate the client info structure
            mfmRfdClientHandle = (MFM_RFD_CLIENT_HANDLE) RFD_MALLOC(sizeof(MFM_RFD_CLIENT_STRUCT));

            if(mfmRfdClientHandle == NULL) {
                status = RFD_STATUS_ERROR_MEM_ALLOC;
                break;
            }

            // copy the client info structure
            *mfmRfdClientHandle = mfmRfdClientArray[i];

            status = _InitializeFromRfdClient(
                            hManager,
                            mfmRfdClientHandle );

            if(status != RFD_STATUS_OK) {
                break;
            }
        }

        if(status != RFD_STATUS_OK) {
            break;
        }

        // success
        status = RFD_STATUS_OK;

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteActiveDmiList( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_LLIST_DeleteList(
        hManager->activeDmiList,
        (RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN)_DeleteActiveDmiInfo );

    hManager->activeDmiList = NULL;

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _CreateActiveDmiList( MULTIFILE_MANAGER_HANDLE hManager )
{
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        //////////////////////////////////////////
        // Create the Active DMI list, initially empty
        //////////////////////////////////////////

        hManager->activeDmiList = NULL;

        hManager->activeDmiList = RFD_LLIST_CreateList();

        if(hManager->activeDmiList == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup.
        _DeleteActiveDmiList( hManager );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteFileManagementLists(
         MFM_FILE_MANAGEMENT_LISTS_STRUCT *fileManagementLists )
{
    // active list
    RFD_LLIST_DeleteList(
        fileManagementLists->activeFileItemList,
        (RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN)_DeleteFileItem );

    fileManagementLists->activeFileItemList = NULL;

    // waiting list (this is a list of subname titled lists).
    RFD_LLIST_DeleteList(
        fileManagementLists->waitingSubnameTitledFileItemListList,
        (RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN)_DeleteSubnameTitledFileItemList );

    fileManagementLists->waitingSubnameTitledFileItemListList = NULL;

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _CreateFileManagementLists(
                MFM_FILE_MANAGEMENT_LISTS_STRUCT *fileManagementLists )
{
    RFD_STATUS status = RFD_STATUS_OK;

    fileManagementLists->activeFileItemList = NULL;
    fileManagementLists->waitingSubnameTitledFileItemListList = NULL;

    do { // once

        //////////////////////////////////////////
        // Create the Active list, initially empty
        //////////////////////////////////////////

        fileManagementLists->activeFileItemList = RFD_LLIST_CreateList();

        if(fileManagementLists->activeFileItemList == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        //////////////////////////////////////////
        // Create the Waiting list, initially empty
        //////////////////////////////////////////

        fileManagementLists->waitingSubnameTitledFileItemListList = RFD_LLIST_CreateList();

        if(fileManagementLists->waitingSubnameTitledFileItemListList == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        // Also associated with the Waiting List is the isWaitingListSorted flag.
        // Initialize it to FALSE, list is not sorted (even though list is initially empty here).
        fileManagementLists->isWaitingListCurrentlySorted = FALSE;

        // success
        status = RFD_STATUS_OK;

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup.
        _DeleteFileManagementLists( fileManagementLists );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _DeleteFileConsumer( MULTIFILE_MANAGER_HANDLE hManager )
{
    if(hManager->fileConsumerInfo.hMfmFileConsumer != NULL) {

        MFM_DeleteFileConsumer( hManager->fileConsumerInfo.hMfmFileConsumer );
    }

    // delete queue after deleting the File Consumer Thread.
    if(hManager->fileConsumerInfo.hMfmFileTransferMessageQueue != NULL) {

        RFD_MSG_QUEUE_Delete( hManager->fileConsumerInfo.hMfmFileTransferMessageQueue );
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _CreateFileConsumer(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_RFD_CLIENT_STRUCT mfmRfdClientArray[],
                UINT16 numRfdClients,
                INT32 fileReadyPollIntervalMillisec,
                BOOL isAltSingleThreadMode )
{
    RFD_CONSUMER_THREAD_DATA_HANDLE * rfdFileConsumerHandleArray = NULL;
    UINT16 i;
    UINT32 messageOverheadSize, messageQueueBufSize;
    RFD_STATUS status = RFD_STATUS_OK;

    hManager->fileConsumerInfo.hMfmFileConsumer = NULL;
    hManager->fileConsumerInfo.hMfmFileTransferMessageQueue = NULL;

    if(numRfdClients == 0) {
        return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
    }

    do { // once

        ////////////////////////////////////
        // Create the Message Queue used to receive File Transfer messages
        // from the File Consumer Thread.
        // Do this first, before creating and starting the thread.
        ////////////////////////////////////

        // Calculate the Message Queue buffer size.
        // Allow for numRfdClients number of simultaneous messages (one for each consumer).

        messageOverheadSize = RFD_MSG_QUEUE_GetMessageOverheadSize();
        messageQueueBufSize =
            numRfdClients * (sizeof(MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT) + messageOverheadSize);

        status = RFD_MSG_QUEUE_Create(
                        &hManager->fileConsumerInfo.hMfmFileTransferMessageQueue,
                        messageQueueBufSize );

        if(status != RFD_STATUS_OK) {
            hManager->fileConsumerInfo.hMfmFileTransferMessageQueue = NULL; // just in case.
            break;
        }

        // Build array of rfd consumer handles - format that's input to the mfm consumer create fcn.

        rfdFileConsumerHandleArray
            = (RFD_CONSUMER_THREAD_DATA_HANDLE *) RFD_MALLOC( numRfdClients * sizeof(RFD_CONSUMER_THREAD_DATA_HANDLE) );

        if(rfdFileConsumerHandleArray == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        for(i=0;i<numRfdClients;i++) {
            rfdFileConsumerHandleArray[i] = mfmRfdClientArray[i].rfdFileConsumerHandle;
        }

        // Call MFM Consumer create function - Create and Start the Consumer Thread.

        status = MFM_CreateFileConsumerExt(
                        &hManager->fileConsumerInfo.hMfmFileConsumer,
                        rfdFileConsumerHandleArray,
                        numRfdClients,
                        _MfmConsumerFileTransferCallback,
                        hManager,
                        fileReadyPollIntervalMillisec,
                        isAltSingleThreadMode  );

        if(status != RFD_STATUS_OK) {
            hManager->fileConsumerInfo.hMfmFileConsumer = NULL; // just in case.
            break;
        }

    } while(FALSE);

    // Cleanup local/temp allocations
    if(rfdFileConsumerHandleArray != NULL) {
        RFD_FREE(rfdFileConsumerHandleArray);
        rfdFileConsumerHandleArray = NULL;
    }

    if(status != RFD_STATUS_OK) {
        // error, cleanup FileConsumer.
        _DeleteFileConsumer( hManager );
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS  _GetFileConsumerMessage(
                    MULTIFILE_MANAGER_HANDLE hManager,
                    MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT * mfmFileTransferInfo,
                    INT32 timeOutMilliseconds  )
{
	RFD_STATUS status;
    void * messageBuf = mfmFileTransferInfo;
    UINT32 messageBufSize = sizeof(MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT);
    UINT32 actualMessageSize = 0;
    RFD_MSG_QUEUE_HANDLE messageQueue =
                            hManager->fileConsumerInfo.hMfmFileTransferMessageQueue;

    do { // once

        status = RFD_MSG_QUEUE_WaitForMessage(messageQueue, timeOutMilliseconds);

        if(status != RFD_STATUS_OK) {
            // error or timeout, no message available.
            // Note: caller should expect timeout as one of possible return status codes.
            break;
        }

        status = RFD_MSG_QUEUE_GetMessage(
                        messageQueue,
                        messageBuf,
                        messageBufSize,
                        &actualMessageSize );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // do basic message size check.
        if(actualMessageSize != messageBufSize) {
            status = RFD_STATUS_ERROR_INVALID_MESSAGE_LENGTH;
            break;
        }

    } while(FALSE);

	return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _TransferMfmRfdClientAcrossLists(
                MFM_RFD_CLIENT_HANDLE hMfmRfdClient,
                RFD_LLIST_HANDLE hDstMfmRfdClientList,
                RFD_LLIST_HANDLE hSrcMfmRfdClientList )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hMfmRfdClientListNode = NULL;

    do { // once

        // first find the client in the src list.
        status = _FindMfmRfdClientInList(
            hSrcMfmRfdClientList,
            hMfmRfdClient,
            &hMfmRfdClientListNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(hMfmRfdClientListNode == NULL) {
            // error, the requested Client is not in the source client list
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        // Move the mfm rfd client from source list to destination list.
        // This is done here by transferring the
        // containing node from the source list to the destination list.

        status = RFD_LLIST_TransferNode(
            hDstMfmRfdClientList,
            hSrcMfmRfdClientList,
            hMfmRfdClientListNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _HandleFileItemDeactivation(
                 MULTIFILE_MANAGER_HANDLE hManager,
                 RFD_LLIST_NODE_HANDLE hActiveFileItemListNode,
                 BOOL isDeleteFileItemRequested )
{
    RFD_STATUS status = RFD_STATUS_OK;
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;
    MFM_RFD_CLIENT_HANDLE hMfmRfdClient = NULL;
    RFD_LLIST_HANDLE hActiveFileItemList =
                        hManager->fileManagementLists.activeFileItemList;

    do { // once

        hFileItem = RFD_LLIST_GetUserData(hActiveFileItemListNode, &status);

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        /////////////////////////////////////////////////////////////////////////
        // Do actions related to Deactivation of the DMI for this File Item
        // (i.e. send DMI status callback if this File Item was the last File Item
        // using this DMI value, etc.).
        /////////////////////////////////////////////////////////////////////////

        _HandleDmiDeactivation( hManager, hFileItem->rfdBlockMsgDmi );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        /////////////////////////////////////////////////////////////////////////
        // Before removing and cleaning up the File Item from Active list,
        // Get reference to the MFM RFD Client which is needed to update
        // the MFM RFD Client Resource lists (move from busy to free).
        /////////////////////////////////////////////////////////////////////////

        // Get reference to the MFM RFD Client before destroying containing parent File Item
        hMfmRfdClient = hFileItem->hMfmRfdClient;

        // Remove the corresponding node from active list, this cleans up the list node but not the
        // user data (file item) of the node.
        status = RFD_LLIST_RemoveNode(hActiveFileItemList, hActiveFileItemListNode);

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }
        hActiveFileItemListNode = NULL;

        // Print File Item that was removed from the active list (before deleting it)
        _PrintFileItem(
            hFileItem,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") On MFM _HandleFileItemDeactivation(), completed File Item removal from the MFM Active List ...")
            TEXT("\nMFM ... File Item ") );

        if(isDeleteFileItemRequested == TRUE) {
            // Cleanup the File Item, as caller requested
            _DeleteFileItem(hFileItem);
            hFileItem = NULL;
        }

        /////////////////////////////////////////////////////////////////////////
        // Move the MFM RFD Client used by this File Item from Busy RFD Client List
        // to the Free RFD Client List.
        /////////////////////////////////////////////////////////////////////////

        status = _TransferMfmRfdClientAcrossLists(
                        hMfmRfdClient,
                        hManager->rfdClientResourceInfo.freeMfmRfdClientList,
                        hManager->rfdClientResourceInfo.busyMfmRfdClientList );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // Print the MFM-RFD Client that was moved
        _PrintMfmRfdClient(
            hMfmRfdClient,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") On MFM File Consumer Update, MFM-RFD Client moved from Busy to Free Client List...")
            TEXT("\nMFM ... MFM RFD Client ") );

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ActivateFileItem(
                   MULTIFILE_MANAGER_HANDLE hManager,
                   RFD_LLIST_NODE_HANDLE hFoundFileItemListNode,
                   RFD_LLIST_NODE_HANDLE hFoundFileItemNode,
                   RFD_LLIST_NODE_HANDLE hFoundClientNode )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_HANDLE waitingFileItemList, activeFileItemList, freeClientList, busyClientList;
    RFD_LLIST_HANDLE waitingFileItemListList;
    MFM_RFD_CLIENT_HANDLE hRfdClient;
    MFM_FILE_ITEM_HANDLE hFileItem;

    waitingFileItemListList = hManager->fileManagementLists.waitingSubnameTitledFileItemListList;
    activeFileItemList = hManager->fileManagementLists.activeFileItemList;
    freeClientList = hManager->rfdClientResourceInfo.freeMfmRfdClientList;
    busyClientList = hManager->rfdClientResourceInfo.busyMfmRfdClientList;

    do { // once

        // get the RFD Client handle
        hRfdClient = RFD_LLIST_GetUserData(hFoundClientNode, &status);
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // get the Waiting File Item Handle
        hFileItem = RFD_LLIST_GetUserData(hFoundFileItemNode, &status);
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(hFileItem->hMfmRfdClient != NULL) {
            // error, a file item in waiting list should NOT have an assigned RFD Client.
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        if(hFileItem->rfdMetadataMsgBuf == NULL || hFileItem->rfdMetadataMsgSize == 0) {
            // error, a file item in waiting list should have an assigned RFD Metadata Message.
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        // Assign the Free RFD Client to the File Item.
        hFileItem->hMfmRfdClient = hRfdClient;

        RFD_DPrint( TEXT("MFM (%s, %d): Activating new File Item\n"),
            hManager->mfmClientName, hManager->mfmClientId );
        _PrintFileItem(
            hFileItem,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") File Item ") );

        // Put the File Item's RFD Metadata Message to the RFD Client
        status = RFD_CollectorPutMessage(
            hFileItem->hMfmRfdClient->rfdMessageCollectorHandle,
            hFileItem->rfdMetadataMsgBuf,
            hFileItem->rfdMetadataMsgSize );

        if(status != RFD_STATUS_OK) {

            // Put Message Buffer may be full.
            RFD_DPrint( TEXT("MFM (%s, %d): _TryFileItemActivations() message put failed, status %d\n"),
                hManager->mfmClientName, hManager->mfmClientId, status);
        }

        // Free the Message Buffer after message is Put to RFD; it's no longer used after file Item is activated.
        // Do this by call to _UpdateFileItemMessage() with NULL message param.
        status = _UpdateFileItemMessage(hFileItem, NULL, 0);

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        waitingFileItemList = RFD_LLIST_GetUserData(hFoundFileItemListNode, &status);
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // Transfer the File Item Node from Waiting List to Active List
        status = RFD_LLIST_TransferNode(activeFileItemList, waitingFileItemList, hFoundFileItemNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // Since a File Item and its corresponding List Node was removed from this Inner File Item List,
        // perform necessary cleanup at the Outer List (ListList) level
        // ( i.e. if the Inner File Item List is new empty, remove that list and the outer ListList Node
        // that contains the now empty inner list).

        status = _CleanupFileItemListListOnFileItemRemoved(
                        waitingFileItemListList,
                        hFoundFileItemListNode,
                        waitingFileItemList );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // Transfer the RFD Client Node from the Free Client List to the Busy Client List.
        status = RFD_LLIST_TransferNode(busyClientList, freeClientList, hFoundClientNode );

        /////////////////////////////////////////////////////////////////////////
        // Do actions related to Activation of the DMI for this File Item
        // (i.e. send DMI status callback if this File Item is the first Active File Item
        // using this DMI value, etc.).
        /////////////////////////////////////////////////////////////////////////

        status = _HandleDmiActivation( hManager, hFileItem->rfdBlockMsgDmi );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

    } while(FALSE);

    return status;

}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ProcessFileConsumerUpdate(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_RFD_FILE_TRANSFER_INFO_POST_HANDLE hMfmConsumerFileTransferInfo,
                BOOL * isActiveListDecrementedPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    MFM_FILE_IDENTITY_PARSE_CALLBACK_FCN mfmFileIdentityParseCallbackFcn;
    void * mfmFileIdentityParseCallbackArg;
    MFM_FILE_IDENTITY_HANDLE hFileIdentity = NULL;
    BOOL isOk;
    RFD_LLIST_HANDLE hFileItemList;
    MFM_FILE_IDENTITY_MASK fileIdentityCompareMask;
    BOOL doListItemIdentityUpdate;
    RFD_LLIST_NODE_HANDLE hFoundListNode = NULL;

    /////////////////////////////////////////////////////////////////////////////////////
    // In summary,
    // - Convert File Consumer RFD File Metadata Name to MFM File Identity; this identifies
    //   the File Item that has been sucessfully handed over to app/data service.
    // - Remove this File Item from the MFM Active List (if it's still in the list).
    // - Move the MFM-RFD Client Handle used for this File Item from the
    //   RFD Client Busy List to the RFD Client Free List (it's available for use
    //   by a Waiting File Item).
    /////////////////////////////////////////////////////////////////////////////////////

    // initialize output
    *isActiveListDecrementedPtr = FALSE;

    do { // once

        /////////////////////////////////////////////////////////////////////////////////////
        // Allocate the MFM File Identity structure
        /////////////////////////////////////////////////////////////////////////////////////
        hFileIdentity = (MFM_FILE_IDENTITY_HANDLE) RFD_MALLOC(sizeof(MFM_FILE_IDENTITY_STRUCT));

        if(hFileIdentity == NULL) {
            // error
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        /////////////////////////////////////////////////////////////////////////////////////
        // Call the user provided callback funcion to
        // parse the RFD Metadata Name field into the MFM File Identity structure
        /////////////////////////////////////////////////////////////////////////////////////

        mfmFileIdentityParseCallbackFcn =
            hManager->appCallbackInfo.mfmFileIdentityParseCallbackFcn;
        mfmFileIdentityParseCallbackArg =
            hManager->appCallbackInfo.mfmFileIdentityParseCallbackArg;

        // Create File Identity struct
        isOk = mfmFileIdentityParseCallbackFcn(
                        hMfmConsumerFileTransferInfo->rfdConsumerFileInfo.name,
                        hFileIdentity, // input, called fcn. fills out the data members of struct.
                        mfmFileIdentityParseCallbackArg );

        if(isOk == FALSE) {
            // error, file identity parser failed.
            // There's a possible corner case in which this could possibly happen:
            // This corresponding RFD Client was reassigned from a different data service to this MFM instance,
            // the RFD Client had already completed reconstuction of an update file but transfer to app was incomplete.
            // On powerup and assignement to this MFM session, the RFD Client then completes the file transfer of
            // the already reconstucted file.
            // The Name field format for this other data service is incompatible with this MFM name format, so the
            // name parser fails.
            // Note that this reassignment should not be done without the Application first resetting
            // the RFD Client. However, since this reset is also an asynchronous operation, there may be some case
            // where this could still happen.
            RFD_DPrint( TEXT("MFM (%s, %d): _ProcessFileConsumerUpdate(), Warning, File Identity Parser failed to parse Metadata Name \"%s\"\n"),
                hManager->mfmClientName, hManager->mfmClientId, hMfmConsumerFileTransferInfo->rfdConsumerFileInfo.name );
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        // Print identity of File Item
        MFM_UTIL_PrintFileItemIdentity(
            hFileIdentity,
            TEXT("MFM ("),
            hManager->mfmClientName,
            TEXT(") On MFM File Consumer Update...")
            TEXT("\nMFM ... File Identity ") );

        /////////////////////////////////////////////////////////////////////////
        // Find the corresponding File Item from Active list, in preparation
        // for removing it from this list.
        /////////////////////////////////////////////////////////////////////////

        hFileItemList = hManager->fileManagementLists.activeFileItemList;
        // Match the full identity: Subname and version.
        // This is important in case the active list was just updated (e.g. to another version) and
        // we don't want to remove the updated versions.
        fileIdentityCompareMask = MFM_FILE_IDENTITY_ALL;
        doListItemIdentityUpdate = FALSE;

        status = _FindFileItemInFileItemList(
                        hFileItemList,
                        hFileIdentity,
                        fileIdentityCompareMask,
                        doListItemIdentityUpdate,
                        &hFoundListNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(hFoundListNode != NULL) {

            // Found the File Item in the Active List, as normally expected.

            /////////////////////////////////////////////////////////////////////////
            // Do File Item Deactivation, i.e. perform all actions assosciated with
            // Deactivating a File Item like:
            //  -remove item node from active list,
            //  - move corresponding RFD Client from busy to free client list,
            //  - delete the file item itself if signalled by caller (sometimes caller
            //    may want to preserve the file item for additional actions associated
            //    with some deactivations, like handling expiration of file item.
            /////////////////////////////////////////////////////////////////////////

            RFD_DPrint( TEXT("MFM (%s, %d): On MFM File Consumer Update, begin actions for File Item Deactivation...\n"),
                hManager->mfmClientName, hManager->mfmClientId);

            status = _HandleFileItemDeactivation(
                            hManager,
                            hFoundListNode,
                            TRUE );

            if(status != RFD_STATUS_OK) {
                // error
                RFD_DPrint( TEXT("MFM (%s, %d): On MFM File Consumer Update, error in _HandleFileItemDeactivation() (%d)\n"),
                    hManager->mfmClientName, hManager->mfmClientId, status);
                break;
            }

            RFD_DPrint( TEXT("MFM (%s, %d): On MFM File Consumer Update, actions completed for File Item Deactivation.\n"),
                hManager->mfmClientName, hManager->mfmClientId);

            /////////////////////////////////////////////////////////////////////////
            // The Active File Item List has been decremented (number of list members decremented),
            // and the mfm-rfd client busy/free resource lists have also been successfully updated.
            // Output this info to caller.
            /////////////////////////////////////////////////////////////////////////
            *isActiveListDecrementedPtr = TRUE;
        }
        else { // hFoundListNode is null
            // This is not necessarily an error: due to asynchronous operation,
            // it's possible that the active File Item was removed due to OTA replacement of the previous
            // RFD File for this same Subname, and this happened to occur just as the previous RFD file
            // RFD processing was completed and reported asynchronously from the MFM File Consumer.
            //
            // Another potential for this is that a implementation of MFM may allow for changing the active File Items
            // based on priority changes.
            RFD_DPrint( TEXT("MFM (%s, %d): _HandleFileConsumerUpdates(), attempt to remove Active File Item, but not File Item not found\n"),
                hManager->mfmClientName, hManager->mfmClientId);

            // print File Identity
            MFM_UTIL_PrintFileItemIdentity(
                hFileIdentity,
                TEXT("MFM ("),
                hManager->mfmClientName,
                TEXT("): File Identity") );
        }

    } while(FALSE);

    ////////////////////////////////////////////
    // cleanup
    ////////////////////////////////////////////

    if(hFileIdentity != NULL) {
        RFD_FREE(hFileIdentity);
        hFileIdentity = NULL;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static int _SubNameTitledFileItemListPriorityCompareShim(
            RFD_LLIST_HANDLE subNameTitledFileItemListA,
            RFD_LLIST_HANDLE subNameTitledFileItemListB,
            MULTIFILE_MANAGER_HANDLE hManager )
{
    const TCHAR * subNameA;
    const TCHAR * subNameB;

    MFM_FILE_PRIORITY_COMPARE_CALLBACK_FCN filePriorityCompareFcn =
        hManager->appCallbackInfo.mfmFilePriorityCallbackFcns.filePriorityCompareFcn;

    void * filePriorityCallbackArg =
        hManager->appCallbackInfo.mfmFilePriorityCallbackArg;

    // initialize compare result to 0 (A = B indication). This default value is returned from this
    // callback in case of any errors or unexpected values encountered.
    int cmpResult = 0;

    do { // once

        subNameA = _GetSubnameTitledFileItemListSubname( subNameTitledFileItemListA );
        subNameB = _GetSubnameTitledFileItemListSubname( subNameTitledFileItemListB );

        // Check for NULL which indicates subName (titles) not available for the lists.
        // This could occur if the subname titled lists are empty. While this is unexpected by
        // design (the waiting list-of-lists is cleaned up to remove empty lists), as a precaution
        // just sort the lists so that empty lists are sorted to lower priority (end of list).
        if(subNameA == NULL) {
            // if A is null (or A and B are null), treat A as lower priority.
            cmpResult = -1;
            break;
        }
        else if(subNameB == NULL) {
            // if B is null, treat B as lower.
            cmpResult = 1;
            break;
        }
        // else both A and B are NOT NULL.

        // Call the user priority compare callback.
        // Keep the same ordering of subNameA and subnameB in this final callback to application.
        cmpResult = filePriorityCompareFcn( subNameA, subNameB, filePriorityCallbackArg );

    } while(FALSE);

    return cmpResult;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _TrySortWaitingFileItemsSubNamePriority(
                MULTIFILE_MANAGER_HANDLE hManager,
                BOOL * isUserReadyPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;

    BOOL isUserReady = FALSE;

    MFM_FILE_PRIORITY_SORT_BEGIN_CALLBACK_FCN filePrioritySortBeginFcn =
        hManager->appCallbackInfo.mfmFilePriorityCallbackFcns.filePrioritySortBeginFcn;

    MFM_FILE_PRIORITY_SORT_END_CALLBACK_FCN filePrioritySortEndFcn =
        hManager->appCallbackInfo.mfmFilePriorityCallbackFcns.filePrioritySortEndFcn;

    void * filePriorityCallbackArg =
        hManager->appCallbackInfo.mfmFilePriorityCallbackArg;

    RFD_LLIST_HANDLE waitingFileItemListList =
        hManager->fileManagementLists.waitingSubnameTitledFileItemListList;

    // initialize output to false.
    *isUserReadyPtr = FALSE;

    do { // once

        if(filePrioritySortBeginFcn != NULL) {
            // Call Begin Sort Callback
            isUserReady = filePrioritySortBeginFcn( filePriorityCallbackArg );
        }
        else {
            isUserReady = TRUE;
        }

        if(isUserReady == TRUE) {

            RFD_DPrint(
                TEXT("MFM (%s, %d):  Starting Priority Sort of the Waiting File Item List.\n"),
                hManager->mfmClientName, hManager->mfmClientId);

            // Do the Priority Sort.
            status = RFD_LLIST_Sort(
                waitingFileItemListList,
                (RFD_LLIST_SORT_COMPARE_CALLBACK_FCN)_SubNameTitledFileItemListPriorityCompareShim,
                hManager );

            RFD_DPrint(
                TEXT("MFM (%s, %d):  Priority Sort completed.\n"),
                hManager->mfmClientName, hManager->mfmClientId);

            if(filePrioritySortEndFcn != NULL) {
                // Call End of Sort Callback
                filePrioritySortEndFcn( filePriorityCallbackArg );
            }

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }
        }
        else {
            // app/user not ready to do priority sorting.
            RFD_DPrint(
                TEXT("MFM (%s, %d):  Skipping Priority Sort of the Waiting File Item List, user not ready.\n"),
                hManager->mfmClientName, hManager->mfmClientId);

            status = RFD_STATUS_OK;
        }

    } while(FALSE);

    // update output to caller.
    *isUserReadyPtr = isUserReady;

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _TryFileItemActivations(
                   MULTIFILE_MANAGER_HANDLE hManager,
                   BOOL * isFileItemActivatedPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hFoundClientNode = NULL, hFoundFileItemNode = NULL, hFoundFileItemListNode = NULL;
    RFD_LLIST_HANDLE activeFileItemList, freeClientList;
    RFD_LLIST_HANDLE waitingFileItemListList;
    size_t numFreeRfdClients, numWaitingFileItems;
    BOOL isSorterReady = FALSE;

    // initialize output
    *isFileItemActivatedPtr = FALSE;

    waitingFileItemListList = hManager->fileManagementLists.waitingSubnameTitledFileItemListList;
    activeFileItemList = hManager->fileManagementLists.activeFileItemList;
    freeClientList = hManager->rfdClientResourceInfo.freeMfmRfdClientList;

    do { // once

        // Get number of Free RFD Clients
        status = _GetNumFreeMfmRfdClients( hManager, &numFreeRfdClients );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(numFreeRfdClients == 0) {
            // no RFD Client resources available, so no activation is possible.
            status = RFD_STATUS_OK;
            break;
        }

        // Get number of Waiting File Items.
        status = _GetNumWaitingFileItems( hManager, &numWaitingFileItems );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if(numWaitingFileItems == 0) {

            // no File Items are waiting, so there's no activation work to do.
            status = RFD_STATUS_OK;
            break;
        }

        // Try Priority Sorting if File Priority Sorting is Enabled.
        // Try even if isWaitingListCurrentlySorted is TRUE, because the applications
        // "Priority Sort Criteria" may have changed since the last time the list was Sorted
        // (e.g. user's current location changed for location/distance based Prioroity Sorting).

        if( hManager->fileManagementPolicyInfo.isFilePrioritySortingEnabled == TRUE ) {

            // Priority Sorting is enabled by application, and the File Items are currently unsorted.
            // Try to do the Sort.
            status = _TrySortWaitingFileItemsSubNamePriority( hManager, &isSorterReady );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if( isSorterReady == TRUE ) {
                // Sorter was ready (Sorting was successful).
                // Set the currently sorted flag.
                hManager->fileManagementLists.isWaitingListCurrentlySorted = TRUE;
            }
            else if( numWaitingFileItems > numFreeRfdClients ) {

                // Sorter not ready but there are more waiting file items than free rfd clients.
                // Just print warning here.
                //
                // Could take action not to do the activations (i.e. to avoid tying up
                // RFD Clients with possibly low priority file items),
                // but this would introduce complexity and perhaps this is undesired action
                // for some applications (i.e. better to just go ahead and do activations rather than
                // wait for Sorter to become ready). Another MFM configuration parameter could be added so
                // application could control the behavior for this case, but this is still
                // additional complexity.

                RFD_DPrint(
                    TEXT("MFM (%s, %d): _TryFileItemActivations() warning. Sorter reported not ready,\n")
                    TEXT("...MFM but Activations being issued with more pending updates than free RFD Clients,\n")
                    TEXT("...MFM numWaitingFileItems (%d), numFreeRfdClients (%d)\n"),
                    hManager->mfmClientName,
                    hManager->mfmClientId,
                    numWaitingFileItems,
                    numFreeRfdClients);
            }
        }

        if(hManager->fileManagementPolicyInfo.updateType == MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES) {

            // For Delta Update Mode/Policy, perform ordering of File Items based on File Identity Version Info.
            //
            // Keep "Version Ordering" and "Sorting" naming separate to avoid confusion between
            // sorting of Subname info and version ordering of Version Info.
            //
            // todo for delta type update support
            // _VersionOrderWaitingFileItems( hManager );
            // For now, while feature is unsupported, just return error.

            status = RFD_STATUS_ERROR_INVALID_MODE;
            break;
        }

        // Find the First Free Rfd Client
        status = RFD_LLIST_First(freeClientList, &hFoundClientNode);

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // Find the First Waiting File Item. Do this after any sorting.
        status = RFD_LLIST_2DFirst(
                        waitingFileItemListList,
                        &hFoundFileItemListNode,
                        &hFoundFileItemNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // While there are more waiting file items and free RFD Clients
        // move waiting file items to active file items (activate the waiting file items).

        while(hFoundFileItemNode != NULL && hFoundClientNode != NULL) {

            status = _ActivateFileItem(
                            hManager,
                            hFoundFileItemListNode,
                            hFoundFileItemNode,
                            hFoundClientNode );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            // Succesful activation of a file time.
            // Set File Item Activated output flag to true.
            *isFileItemActivatedPtr = TRUE;

            // Find the next Free RFD Client
            status = RFD_LLIST_Next(freeClientList, &hFoundClientNode);

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            // Find the next Waiting File Item
            status = RFD_LLIST_2DNext(
                            waitingFileItemListList,
                            &hFoundFileItemListNode,
                            &hFoundFileItemNode );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

        } // while (hFoundFileItemNode != NULL && hFoundClientNode != NULL)

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _HandleFileConsumerUpdates(
                MULTIFILE_MANAGER_HANDLE hManager,
                BOOL * isActiveListDecrementedPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    MFM_RFD_FILE_TRANSFER_INFO_POST_HANDLE hMfmConsumerFileTransferInfo = NULL;

    BOOL isActiveListDecrementedOnSingleLoop = FALSE;
    BOOL isMessageReceived = FALSE;

    // initialize output
    *isActiveListDecrementedPtr = FALSE;

    do { // once

        /////////////////////////////////////////////////////////////////////////////////////
        // Allocate buffer to contain message, this is the MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT.
        // (note: could allocate this buffer just once as part hManager instead of
        // allocating/freeing each time)
        /////////////////////////////////////////////////////////////////////////////////////
        hMfmConsumerFileTransferInfo =
            (MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT *) RFD_CALLOC(sizeof(MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT));

        if(hMfmConsumerFileTransferInfo == NULL) {
            // error
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        do { // do until no messages from file consumer are available

            /////////////////////////////////////////////////////////////////////////////////////
            // Try to get File Consumer Message
            /////////////////////////////////////////////////////////////////////////////////////
            status = _GetFileConsumerMessage(
                hManager,
                hMfmConsumerFileTransferInfo,
                0 /* timeout immediately */ );

            if(status == RFD_STATUS_OK) {

                /////////////////////////////////////////////////////////////////////////////////////
                // Message successfully received containing hMfmConsumerFileTransferInfo, process it.
                /////////////////////////////////////////////////////////////////////////////////////

                status = _ProcessFileConsumerUpdate(
                    hManager,
                    hMfmConsumerFileTransferInfo,
                    &isActiveListDecrementedOnSingleLoop );

                if(status != RFD_STATUS_OK) {
                    // error
                    break;
                }

                // OR the output result from each call to _ProcessFileConsumerUpdate()
                // so that the flag is set if any of the calls resulted in true indication.
                *isActiveListDecrementedPtr |= isActiveListDecrementedOnSingleLoop;

                // continue checking for more messages.
                isMessageReceived = TRUE;
            }
            else if(status == RFD_STATUS_OBJECT_TIMEOUT)  {
                // not an error, but no message available.
                // override timeout status code back to RFD_STATUS_OK
                // for return to caller.
                status = RFD_STATUS_OK;
                // set exit condition
                isMessageReceived = FALSE;
            }
            else {
                // error, with status indicating error (not _OK and not _TIMEOUT)
                isMessageReceived = FALSE;
                break;
            }

        } while(isMessageReceived == TRUE);

    } while(FALSE);

    ////////////////////////////////////////////
    // cleanup
    ////////////////////////////////////////////

    if(hMfmConsumerFileTransferInfo != NULL) {
        RFD_FREE(hMfmConsumerFileTransferInfo);
        hMfmConsumerFileTransferInfo = NULL;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _HandleFileItemExpiration(
                MULTIFILE_MANAGER_HANDLE hManager,
                RFD_LLIST_HANDLE hFileItemList,
                RFD_LLIST_NODE_HANDLE hListNode,
                MFM_FILE_ITEM_LIST_TYPE_ENUM fileItemListType,
                BOOL * isFileItemListIteratedPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;
    BOOL isFileItemListIterated = FALSE;

    // initialize output.
    *isFileItemListIteratedPtr = FALSE;

    do { // once

        hFileItem = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        if(fileItemListType == MFM_FILE_ITEM_LIST_TYPE_ACTIVE) {

            // Print File Item that is being expired
            _PrintFileItem(
                hFileItem,
                TEXT("MFM ("),
                hManager->mfmClientName,
                TEXT(") On MFM _HandleFileItemExpiration(), expiring an Active File Item ...")
                TEXT("\nMFM ... File Item ") );

            // Reset the associated RFD Client to clear any intermediate processing (delete working files etc.)
            // for this expired file item.
            RFD_ConsumerResetFileCollectionRequest( hFileItem->hMfmRfdClient->rfdFileConsumerHandle );

            // Perform all common actions for deactivating any file item
            // (remove file item from active list, move rfd client to free list, ...).
            // Specifify not to delete the file item, we'll do it later in this function
            // after we're finished with it.

            status = _HandleFileItemDeactivation(
                            hManager,
                            hListNode,
                            FALSE );

            if(status != RFD_STATUS_OK) {
                break;
            }
            // deactivate cleans-up the List Node, but not the contained File Item Node
            // because we requested not to.
            hListNode = NULL;

	        /////////////////////////////////////////
            // For Active File Item expiration, take related actions based on the specific Update Mode.
	        /////////////////////////////////////////

            if(hManager->fileManagementPolicyInfo.updateType == MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES) {
                // nothing more to do for expiring an active file item that is specific for ABSOLUTE type update.

                // No nested iterations were done, no need to set restart true (default is false).
            }
            else if(hManager->fileManagementPolicyInfo.updateType == MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES) {
                // currently unsupported mode
                status = RFD_STATUS_ERROR_INVALID_MODE;
                break;

                ///////////////////////////////
                //// Note: keep these comments, relevant to possible future support of DELTA UPDATE mode.
                ///////////////////////////////
                //// Deactivate all other File Items in the Active list that have
                //// versionTo numbers above the VersionTo number of this File Item, and with the same subname.
                //// This is because the deactivated file item will break the version-chain of updates,
                //// if any higher version numbers are in the chain.
                ////
                //// The _DeactivateAllFileItemVersionsAbove() does something like:
                ////     Iterate Active List,
                ////          if subname == ref_subname && versionTo > ref_versionTo,
                ////              _HandleFileItemDeactivation( , , TRUE);
                ////
                //// Probably always true, but put output isActiveFileItemListIterated param in fcn signature to
                //// make it clear to function caller that list will be iterated by the function
                //// (so the caller knows to reset it's own iteration of the same list,
                //// i.e. nested iteration not suported).
                //
                //status = _DeactivateAllFileItemVersionsAbove(
                //                    hManager,
                //                    &hFileItem->fileIdentityInfo, // the reference subname and version info
                //                    &isFileItemListIterated );
                //
                //if(status != RFD_STATUS_OK) {
                //    break;
                //}
            }
            else {
                // error, unknown update mode
                status = RFD_STATUS_ERROR_INVALID_MODE;
                break;
            }
        }
        else if(fileItemListType == MFM_FILE_ITEM_LIST_TYPE_WAITING) {

            // Print File Item that is being expired
            _PrintFileItem(
                hFileItem,
                TEXT("MFM ("),
                hManager->mfmClientName,
                TEXT(") On MFM _HandleFileItemExpiration(), expiring Waiting File Item ...")
                TEXT("\nMFM ... File Item ") );

            // Remove the corresponding node from waiting list, this cleans up the list node but not the
            // user data (file item) of the node (delete the File Item further below).
            status = RFD_LLIST_RemoveNode(hFileItemList, hListNode);

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }
            hListNode = NULL;
        }
        else {
            // error, handling file itme expirations not relevant on this list type,
            // or unknown list type.
            status = RFD_STATUS_ERROR_INVALID_MODE;
            break;
        }

        // Cleanup the File Item, we defered the cleanup to hear so as to do the cleanup
        // in one common place for all the different cases.

        if(hFileItem != NULL) {
            _DeleteFileItem(hFileItem);
            hFileItem = NULL;
        }

    } while(FALSE);

    // Tell caller if List was iterated/browsed by this function.
    *isFileItemListIteratedPtr = isFileItemListIterated;

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ProcessItemExpirationsInFileItemList(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_FILE_ITEM_EXPIRATION_CRITERION_FCN expirationCriterionFcn,
                void * expirationCriterionArg,
                RFD_LLIST_HANDLE hFileItemList,
                MFM_FILE_ITEM_LIST_TYPE_ENUM fileItemListType,
                BOOL * isFileItemExpiredPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isFileItemExpired = FALSE;
    BOOL isFileItemListIterated;
    BOOL isRestartListIterationRequired;
    RFD_LLIST_NODE_HANDLE hListNode = NULL;
    MFM_FILE_ITEM_HANDLE hFileItem = NULL;

    // initializize output
    *isFileItemExpiredPtr = FALSE;

    isFileItemExpired = FALSE;

	/////////////////////////////////////////
    // For each  File Item in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    hFileItemList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        // set restart iteration to false by default at begining of each iteration step.
        isRestartListIterationRequired = FALSE;

        hFileItem = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

	    /////////////////////////////////////////
        // Check if this File Item is expired
	    /////////////////////////////////////////

        // Call the caller provided expiration criterion function that dermined if File Item is expired.
        status = expirationCriterionFcn(
                    hManager,
                    hFileItem,
                    &isFileItemExpired,
                    expirationCriterionArg );

        if(status != RFD_STATUS_OK) {
            break;
        }

        if( isFileItemExpired == TRUE ) {

            // Found an expired file item.

            RFD_DPrint(
                TEXT("MFM (%s, %d): _ProcessItemExpirationsInFileItemList() Found Expired File Item\n"),
                hManager->mfmClientName,
                hManager->mfmClientId );

	        /////////////////////////////////////////
            // Handle everything to do with Expiring a File Item
            // (cleanup Node in list, cleanup File Item, any related callbacks, etc.).
	        /////////////////////////////////////////

            status = _HandleFileItemExpiration(
                            hManager,
                            hFileItemList,
                            hListNode,
                            fileItemListType,
                            &isFileItemListIterated );

            if(status != RFD_STATUS_OK) {
                break;
            }

            // _HandleFileItemExpiration() is responsible for cleaning up
            // the List Node and contained List Item.
            hListNode = NULL;
            hFileItem = NULL;

            if(isFileItemListIterated == TRUE) {
                // Called function iterated/browsed the list in handling the expiration.
                // Since nested list iterations are not supported, restart the iterating
                // (start from First Node again)
                isRestartListIterationRequired = TRUE;
            }

            // Set File Item expired flag; an expiration was detected and successfully handled
            // for at least one File Item.
            isFileItemExpired = TRUE;
        }

        if(isRestartListIterationRequired == TRUE) {
           // Select First Node to restart list iteration.
           status = RFD_LLIST_First( hFileItemList, &hListNode );
        }
        else {
            // Select Next Node to continue the current list iteration, as no restart required.
            status = RFD_LLIST_Next( hFileItemList, &hListNode );
        }
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

    }  // End of List iteration

    // update output
    if(status == RFD_STATUS_OK) {
        *isFileItemExpiredPtr = isFileItemExpired;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ProcessItemExpirationsInWaitingFileItemListList(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_FILE_ITEM_EXPIRATION_CRITERION_FCN expirationCriterionFcn,
                void * expirationCriterionArg,
                BOOL * isFileItemExpiredPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isAnyFileItemExpired, isFileItemExpiredInCurList;
    RFD_LLIST_NODE_HANDLE hListListNode = NULL;
    RFD_LLIST_HANDLE hInnerFileItemList = NULL;
    RFD_LLIST_HANDLE hFileItemListList =
        hManager->fileManagementLists.waitingSubnameTitledFileItemListList;

    // initializize output
    *isFileItemExpiredPtr = FALSE;

    isAnyFileItemExpired = FALSE;

	/////////////////////////////////////////
    // For each File Item List in the List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    hFileItemListList,
                    &hListListNode );

    while(hListListNode != NULL) {

        hInnerFileItemList = RFD_LLIST_GetUserData(hListListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

        ///////////////////////////////////////////////////////////
        // Process the currently iterated File Item List in the Waiting List of Lists
        ///////////////////////////////////////////////////////////

        isFileItemExpiredInCurList = FALSE;

        status = _ProcessItemExpirationsInFileItemList(
                        hManager,
                        expirationCriterionFcn,
                        expirationCriterionArg,
                        hInnerFileItemList,
                        MFM_FILE_ITEM_LIST_TYPE_WAITING,
                        &isFileItemExpiredInCurList );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _ProcessItemExpirationsInWaitingFileItemListList() error in _ProcessItemExpirationsInFileItemList(), %d\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status );
            break;
        }

        if(isFileItemExpiredInCurList == TRUE) {

            // At least one File Item was expired and thus removed from the current Inner File Item List.

            // Set the overall flag that a file item was expired.
            isAnyFileItemExpired = TRUE;

            // Since at least one File Item (and its list node) was removed for the current Inner File Item List,
            // perform necessary cleanup at the Outer List (ListList) level
            // ( i.e. if the Inner File Item List is new empty, remove that list and the outer ListList Node
            // that contains the now empty inner list).
            status = _CleanupFileItemListListOnFileItemRemoved(
                            hFileItemListList,
                            hListListNode,
                            hInnerFileItemList );

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }
        }

        status = RFD_LLIST_Next(
                        hFileItemListList,
                        &hListListNode );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

    }  // End of List iteration

    // update output
    if(status == RFD_STATUS_OK) {
        *isFileItemExpiredPtr = isAnyFileItemExpired;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS _ExpirationCriterionFcn_IsLifeExpired(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_FILE_ITEM_HANDLE hFileItem,
                BOOL * isExpiredPtr,
                void * expirationCriterionArg )
{
    RFD_STATUS status = RFD_STATUS_OK;
    FILE_ITEM_LIFE_EXPIRATION_CRITERION_INFO * lifeExpirationCriterionInfo =
        (FILE_ITEM_LIFE_EXPIRATION_CRITERION_INFO *) expirationCriterionArg;

    status = _IsRfdLifeExpired(
                    hManager,
                    hFileItem->rfdLifeTime,
                    lifeExpirationCriterionInfo->currentSimpleCalendarTime,
                    lifeExpirationCriterionInfo->isCurrentTimeAvailable,
                    isExpiredPtr );

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS _ExpirationCriterionFcn_IsFileItemAlreadyStoredByApp(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_FILE_ITEM_HANDLE hFileItem,
                BOOL * isExpiredPtr,
                void * expirationCriterionArg )
{
    int compareResult;
    MFM_FILE_IDENTITY_HANDLE appExpiredFileItemFileIdentity =
        (MFM_FILE_IDENTITY_HANDLE) expirationCriterionArg;

    // Compare file identity version info
    compareResult = MFM_UTIL_CompareFileIdentity(
        appExpiredFileItemFileIdentity,
        &hFileItem->fileIdentityInfo,
        MFM_FILE_IDENTITY_SUBNAME | MFM_FILE_IDENTITY_VERSION_TO );

    if(!compareResult) {
        // File Item Subnames and VersionTo numbers matches. This is the Criteria for File Item Already Stored By App
        *isExpiredPtr = TRUE;
    }
    else {
        *isExpiredPtr = FALSE;
    }

    return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS _ExpirationCriterionFcn_IsFileItemExpiredByApp(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_FILE_ITEM_HANDLE hFileItem,
                BOOL * isExpiredPtr,
                void * expirationCriterionArg )
{
    TCHAR * AppExpiredFileItemSubname =
        (TCHAR *) expirationCriterionArg;

    if(!RFD_STRNCMP( AppExpiredFileItemSubname, hFileItem->fileIdentityInfo.subname, MFM_SUBNAME_MAX_LEN )) {

        // File Item Subname strings match. This is the Criteria for File Item Expired By App
        *isExpiredPtr = TRUE;
    }
    else {
        *isExpiredPtr = FALSE;
    }

    return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ProcessExpiredFileItems(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_FILE_ITEM_EXPIRATION_CRITERION_FCN expirationCriterionFcn,
                void * expirationCriterionArg,
                BOOL * isFileItemExpiredPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isActiveFileItemExpired, isWaitingFileItemExpired;

    // initialize output
    *isFileItemExpiredPtr = FALSE;

    isActiveFileItemExpired = FALSE;
    isWaitingFileItemExpired = FALSE;

    do { // once

        ///////////////////////////////////////////////////////////
        // Process the Active list for expired file items
        ///////////////////////////////////////////////////////////

        status = _ProcessItemExpirationsInFileItemList(
                        hManager,
                        expirationCriterionFcn,
                        expirationCriterionArg,
                        hManager->fileManagementLists.activeFileItemList,
                        MFM_FILE_ITEM_LIST_TYPE_ACTIVE,
                        &isActiveFileItemExpired );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _ProcessExpiredFileItems() error in _ProcessItemExpirationsInFileItemList(), %d\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status );
            break;
        }

        ///////////////////////////////////////////////////////////
        // Process the Waiting list for expired file items
        ///////////////////////////////////////////////////////////

        status = _ProcessItemExpirationsInWaitingFileItemListList(
                        hManager,
                        expirationCriterionFcn,
                        expirationCriterionArg,
                        &isWaitingFileItemExpired );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _ProcessExpiredFileItems() error in _ProcessItemExpirationsInWaitingFileItemListList(), %d\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status );
            break;
        }

    } while(FALSE);

    // update output
    if(status == RFD_STATUS_OK) {
        *isFileItemExpiredPtr = isActiveFileItemExpired | isWaitingFileItemExpired;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _ProcessRfdLifeExpiredFileItems(
                MULTIFILE_MANAGER_HANDLE hManager,
                BOOL * isFileItemExpiredPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isFileItemExpired;
    BOOL isCurrentTimeAvailable, isEqualDay;
    RFD_SXI_TIME_INFO_STRUCT currentSxiTime;
    time_t currentSimpleCalendarTime;
    FILE_ITEM_RFD_FILE_EXPIRATION_PROCESS_INFO_STRUCT * hRfdLifeExpirationProcessInfo =
        &hManager->fileItemRfdLifeExpirationProcessInfo;
    FILE_ITEM_LIFE_EXPIRATION_CRITERION_INFO lifeExpirationCriterionInfo;

    // initializize output
    *isFileItemExpiredPtr = FALSE;

    isFileItemExpired = FALSE;

    do { // once

        ///////////////////////////////////////////////////////////
        // Get current time
        ///////////////////////////////////////////////////////////
        isCurrentTimeAvailable = _GetCurrentSxiTime( hManager, &currentSxiTime );

        if( isCurrentTimeAvailable == TRUE ) {

            // Time is available

            ///////////////////////////////////////////////////////////
            // Check if time is has changed, for Day resolution, which is the resolution of
            // the RFD Metadata Life (no need to check if time is same to within 1 day resolution).
            ///////////////////////////////////////////////////////////

            isEqualDay = RFD_TIME_IsEqualSxiTime(
                            &currentSxiTime,
                            &hRfdLifeExpirationProcessInfo->lastExpProcessSxiTime,
                            RFD_SXI_TIME_DAY_RES );

            if( isEqualDay == FALSE  ||
                hRfdLifeExpirationProcessInfo->isLifeUpdatedSinceLastExpProcessTime == TRUE ) {

                // The SXI time has changed by 1 or more days since last time,
                // or a File Item Life field has been updated since the last File Item Expirations Processing.

                // Set new time and continue processing File Item Expirations.

                ///////////////////////////////////////////////////////////
                // Convert sxi time to the c standard lib "calendar" time which is
                // a format that facilitates time comparison operations for expiration checking.
                ///////////////////////////////////////////////////////////

                status = RFD_TIME_SxiTime2StdCalendarTime( &currentSimpleCalendarTime, &currentSxiTime);

                if(status != RFD_STATUS_OK) {
                    // error
                    RFD_DPrint(
                        TEXT("MFM (%s, %d):: _ProcessRfdLifeExpiredFileItems() error in RFD_TIME_SxiTime2StdCalendarTime(), %d\n"),
                        hManager->mfmClientName, hManager->mfmClientId,
                        status );
                    break;
                }

                ///////////////////////////////////////////////////////////
                // Call common function to process expired file items, passing specialized criterion function and arg
                // to specifically do the <RFD Life Time> type expiration processing.
                ///////////////////////////////////////////////////////////

                lifeExpirationCriterionInfo.currentSimpleCalendarTime = currentSimpleCalendarTime;
                lifeExpirationCriterionInfo.isCurrentTimeAvailable = isCurrentTimeAvailable;

                status = _ProcessExpiredFileItems(
                                hManager,
                                _ExpirationCriterionFcn_IsLifeExpired,
                                &lifeExpirationCriterionInfo,
                                &isFileItemExpired );

                if(status != RFD_STATUS_OK) {
                    // error
                    RFD_DPrint(
                        TEXT("MFM (%s, %d):: _ProcessRfdLifeExpiredFileItems() error in _ProcessExpiredFileItems(), %d\n"),
                        hManager->mfmClientName, hManager->mfmClientId,
                        status );
                    break;
                }

                ///////////////////////////////////////////////////////////
                // Copy current time to last processed time,
                // and clear the File Item Life Updated flag.
                ///////////////////////////////////////////////////////////
                hRfdLifeExpirationProcessInfo->lastExpProcessSxiTime = currentSxiTime;
                hRfdLifeExpirationProcessInfo->isLifeUpdatedSinceLastExpProcessTime = FALSE;
            }
            // Else no need to process if current time has not changed since previous processing.
        }
        else {
            // Print warning, time not available
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _ProcessRfdLifeExpiredFileItems(), App Callback appGetTimeCallbackFcn() indicates Time Info not available.\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status );
        }

    } while(FALSE);

    // update output
    if(status == RFD_STATUS_OK) {
        *isFileItemExpiredPtr = isFileItemExpired;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _CleanupAppAlreadyStoredFileItem(
                MULTIFILE_MANAGER_HANDLE hManager,
                MFM_FILE_IDENTITY_HANDLE hAlreadyStoredFileIdendity,
                BOOL * isFileItemExpiredPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isFileItemExpired;

    // initialize output
    *isFileItemExpiredPtr = FALSE;

    isFileItemExpired = FALSE;

    do { // once

        status = _ProcessExpiredFileItems(
                        hManager,
                        _ExpirationCriterionFcn_IsFileItemAlreadyStoredByApp,   // expriationCriterionFcn
                        hAlreadyStoredFileIdendity,                             // expriationCriterionArg
                        &isFileItemExpired );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _CleanupAppAlreadyStoredFileItem() error in _ProcessExpiredFileItems(), %d\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status );
            break;
        }

    } while(FALSE);

    // update output
    if(status == RFD_STATUS_OK) {
        *isFileItemExpiredPtr = isFileItemExpired;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _CleanupAppOutOfServiceFileItems(
                MULTIFILE_MANAGER_HANDLE hManager,
                TCHAR * appExpiredFileItemSubname,
                BOOL * isFileItemExpiredPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    BOOL isFileItemExpired;

    // initialize output
    *isFileItemExpiredPtr = FALSE;

    isFileItemExpired = FALSE;

    do { // once

        status = _ProcessExpiredFileItems(
                        hManager,
                        _ExpirationCriterionFcn_IsFileItemExpiredByApp,     // expriationCriterionFcn
                        appExpiredFileItemSubname,                          // expriationCriterionArg
                        &isFileItemExpired );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _CleanupAppOutOfServiceFileItems() error in _ProcessExpiredFileItems(), %d\n"),
                hManager->mfmClientName, hManager->mfmClientId,
                status );
            break;
        }

    } while(FALSE);

    // update output
    if(status == RFD_STATUS_OK) {
        *isFileItemExpiredPtr = isFileItemExpired;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static void _UninitializeClientIdentity( MULTIFILE_MANAGER_HANDLE mfmHandle )
{
    if( mfmHandle->mfmClientName != NULL ) {
        // deallocate the MFM owned Client Name buffer.
        RFD_FREE(mfmHandle->mfmClientName);
        mfmHandle->mfmClientName = NULL;
    }

    mfmHandle->mfmClientId = 0;

    return;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InitializeClientIdentity(
                            MULTIFILE_MANAGER_HANDLE mfmHandle,
                            UINT16 inClientId,
                            const TCHAR * inClientName )
{
    RFD_STATUS status = RFD_STATUS_OK;
    size_t nameStringLength, nameBufLength, nameBufSize;
    const TCHAR * clientName = NULL;

    do { // once

        // Initialize name pointer to NULL
        mfmHandle->mfmClientName = NULL;

        // Assign MFM Client Id
        mfmHandle->mfmClientId = inClientId;

        if(inClientName != NULL) {
            // Caller provided Name, use it.
            clientName = inClientName;
        }
        else {
            // Caller did provide name, use default.
            clientName = MFM_DEFAULT_CLIENT_NAME;
        }

        // nameLength is in number of TCHARs, not including string terminator.
        nameStringLength = RFD_STRNLEN( clientName, RFD_MAX_PATH_LEN );

        if(nameStringLength == RFD_MAX_PATH_LEN) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        // Allocate string buffer, include extra character for string terminator char.

        // buf length in TCHARs
        nameBufLength = nameStringLength + 1;

        // buf length in bytes
        nameBufSize = nameBufLength * sizeof(TCHAR);

        // allocate bytes
        mfmHandle->mfmClientName = RFD_CALLOC(nameBufSize);

        if(mfmHandle->mfmClientName == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        // finally, copy the name string to MFM-owned string buffer.
        // Specify length of buffer in TCHARs for STRCPY().
        RFD_STRCPY_S( mfmHandle->mfmClientName,  nameBufLength, clientName );

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // cleanup on error
        if(mfmHandle->mfmClientName != NULL) {
            RFD_FREE(mfmHandle->mfmClientName);
            mfmHandle->mfmClientName = NULL;
        }
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _InitializeCallbackInfo(
                MULTIFILE_MANAGER_HANDLE mfmHandle,
                const MFM_APP_CALLBACK_INFO_STRUCT * appCallbackInfo,
                const MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT * fileManagementPolicyInfo )
{
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        if(appCallbackInfo == NULL) {
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        if(appCallbackInfo->appGetTimeCallbackFcn == NULL) {

            // error, the Get Time function must always be provided.
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _InitializeCallbackInfo() error - appCallbackInfo->appGetTimeCallbackFcn is NULL\n"),
                mfmHandle->mfmClientName, mfmHandle->mfmClientId,
                status );

            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        if(fileManagementPolicyInfo->isFilePrioritySortingEnabled == TRUE &&
            appCallbackInfo->mfmFilePriorityCallbackFcns.filePriorityCompareFcn == NULL) {

            // error, a File Priority Compare function must be provided if
            // Priority Sorting is enabled.
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _InitializeCallbackInfo() error - filePriorityCompareFcn is NULL with Priority Sorting Enabled\n"),
                mfmHandle->mfmClientName, mfmHandle->mfmClientId,
                status );

            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        if(appCallbackInfo->appIsSubnameInServiceCallbackFcn == NULL) {

            // error, the Is Subname In Service function must always be provided.
            RFD_DPrint(
                TEXT("MFM (%s, %d):: _InitializeCallbackInfo() error - appCallbackInfo->appIsSubnameInServiceCallbackFcn is NULL\n"),
                mfmHandle->mfmClientName, mfmHandle->mfmClientId,
                status );

            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        // Copy structure contents
        mfmHandle->appCallbackInfo = *appCallbackInfo;

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _HandleRfdUpdateFileMetadataMessage(
                MULTIFILE_MANAGER_HANDLE hManager,
                UCHAR * messageBuf,
                UINT32 messageSize,
                BOOL * isWaitingListSizeIncrementedPtr )
{
    RFD_STATUS status;
    RFD_FSET_MDATA_STRUCT * rfdUpdateFileMetadataInfo;
    MFM_FILE_IDENTITY_PARSE_CALLBACK_FCN mfmFileIdentityParseCallbackFcn;
    void * mfmFileIdentityParseCallbackArg;
    MFM_FILE_IDENTITY_HANDLE hFileIdentity = NULL;
    BOOL isParseOk;
    BOOL isRfdLifeExpired = FALSE;
    BOOL isFileItemRemoved = FALSE;
    time_t currentSimpleCalendarTime;
    BOOL isCurrentTimeAvailable;

    do { // once

        /////////////////////////////////////////////////////////////////////////////////////
        // Allocate the MFM File Identity structure
        /////////////////////////////////////////////////////////////////////////////////////
        hFileIdentity = (MFM_FILE_IDENTITY_HANDLE) RFD_MALLOC(sizeof(MFM_FILE_IDENTITY_STRUCT));

        if(hFileIdentity == NULL) {
            // error
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        /////////////////////////////////////////////////////////////////////////////////////
        // Parse the RFD Update File Metadata Message
        // to extract the Metadata Name field
        /////////////////////////////////////////////////////////////////////////////////////

        rfdUpdateFileMetadataInfo =
            hManager->mfmRfdMessageProcessInfo.rfdUpdateFileMetadataInfo;

        status = RFD_CollectorParseUpdateFileMetadataMsg(
                        hManager->mfmRfdMessageProcessInfo.rfdDefaultMessageConfigInfo,
                        messageBuf,
                        messageSize,
                        rfdUpdateFileMetadataInfo );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        /////////////////////////////////////////////////////////////////////////////////////
        // Check if this Update File is expired. Would not expect corresponding Metadata Messages
        // for Expired Update Files, but maybe there are is some residual transmission, particualarly
        // near the expiration time.
        /////////////////////////////////////////////////////////////////////////////////////

        isCurrentTimeAvailable = _GetCurrentSimpleCalendarTime(
                                        hManager,
                                        &currentSimpleCalendarTime );
        status = _IsRfdLifeExpired(
                     hManager,
                     rfdUpdateFileMetadataInfo->lifeTime,
                     currentSimpleCalendarTime,
                     isCurrentTimeAvailable,
                     &isRfdLifeExpired );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        if( isRfdLifeExpired == TRUE ) {
            // RFD Update File Metadata Message detected as Expired.
            RFD_DPrint(
                TEXT("MFM (%s, %d): _HandleRfdUpdateFileMetadataMessage(), Update File Metadata Message was detected as RFD Life Expired (Metadata Name \"%s\")\n"),
                hManager->mfmClientName, hManager->mfmClientId, rfdUpdateFileMetadataInfo->name );
        }

        /////////////////////////////////////////////////////////////////////////////////////
        // Call the user provided callback function to
        // parse the RFD Metadata Name field into the MFM File Identity structure
        /////////////////////////////////////////////////////////////////////////////////////

        mfmFileIdentityParseCallbackFcn =
            hManager->appCallbackInfo.mfmFileIdentityParseCallbackFcn;
        mfmFileIdentityParseCallbackArg =
            hManager->appCallbackInfo.mfmFileIdentityParseCallbackArg;

        isParseOk = mfmFileIdentityParseCallbackFcn(
                        rfdUpdateFileMetadataInfo->name,
                        hFileIdentity, // input, called fcn. fills out the data members of struct.
                        mfmFileIdentityParseCallbackArg );

        if(isParseOk == TRUE) {
            // successful parse

            /////////////////////////////////////////////////////////////////////////////////////
            // Call the user provided callback function to
            // determine if File Items of this Subname are In-Service (does the application want
            // File Items of this Subname).
            // If the application indicates the Subname is In-Service, then it also provides
            // the VersionTo number of the File Item it already has in
            /////////////////////////////////////////////////////////////////////////////////////

            BOOL isSubnameInService;
            UINT32 appStoredFileIdentifyVersionTo = MFM_FILE_IDENTITY_VERSION_NUM_UNUSED;
            MFM_APP_IS_SUBNAME_IN_SERVICE_CALLBACK_FCN mfmIsSubnameInServiceFcn;
            void * mfmIsSubnameInServiceArg;

            mfmIsSubnameInServiceFcn =
                hManager->appCallbackInfo.appIsSubnameInServiceCallbackFcn;
            mfmIsSubnameInServiceArg =
                hManager->appCallbackInfo.appIsSubnameInServiceCallbackArg;

            isSubnameInService = mfmIsSubnameInServiceFcn(
                                        hFileIdentity->subname,
                                        &appStoredFileIdentifyVersionTo,
                                        mfmIsSubnameInServiceArg );

            if( isSubnameInService == TRUE ) {

                // The application wants MFM to process file items with this subname,

                if(hManager->fileManagementPolicyInfo.updateType
                    == MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES) {

                    // Check first that the app doesn't already have this particular file item version in storage.

                    if(hFileIdentity->versionTo != appStoredFileIdentifyVersionTo) {

                        status = _ProcessAppInServiceFileItem(
                                        hManager,
                                        messageBuf,
                                        messageSize,
                                        rfdUpdateFileMetadataInfo,
                                        hFileIdentity,
                                        isRfdLifeExpired,
                                        isWaitingListSizeIncrementedPtr );

                        if(status != RFD_STATUS_OK) {
                            // error
                            break;
                        }

                    }
                    else {
                        // The application already has the file item with this subname and versionTo value.
                        // Cleanup this particular file item if it's currently in the active or waiting list.
                        // e.g. Maybe some data services allow the application to receive file items from other
                        // data services or the corresponding app baseline dbase was updated dynamically.

                        status = _CleanupAppAlreadyStoredFileItem( hManager, hFileIdentity, &isFileItemRemoved );

                        if(status != RFD_STATUS_OK) {
                            // error
                            RFD_DPrint(
                                TEXT("MFM (%s, %d):: _CleanupAppAlreadyStoredFileItem() error, status %d\n"),
                                hManager->mfmClientName, hManager->mfmClientId, status);
                            break;
                        }

                        if(isFileItemRemoved == TRUE) {
                            RFD_DPrint(
                                TEXT("MFM (%s, %d):  An existing File Item was expired (removed) due to File Item indicated as already in storage by Application.\n"),
                                hManager->mfmClientName, hManager->mfmClientId);
                        }
                    }
                }
                else {
                    // currently unsupported mode
                    status = RFD_STATUS_UNSUPPORTED_FEATURE;
                    break;
                }
            }
            else {
                // The application has no interest in MFM processing the File Item with this subname (Out-Of-Service).
                // In case the application has just changed the subname from In-Service to Out-Of-Service,
                // it's possible that the corresponding File Item is currently in the Waiting or Active lists,
                // so try to cleanup/remove any File Items with this subname. Otherwise, the File Item would
                // remain and possibly block other In-Service File Items from being processed/activated.

                status = _CleanupAppOutOfServiceFileItems( hManager, hFileIdentity->subname, &isFileItemRemoved );

                if(status != RFD_STATUS_OK) {
                    // error
                    RFD_DPrint(
                        TEXT("MFM (%s, %d):: _CleanupAppOutOfServiceFileItems() error, status %d\n"),
                        hManager->mfmClientName, hManager->mfmClientId, status);
                    break;
                }

                if(isFileItemRemoved == TRUE) {
                    RFD_DPrint(
                        TEXT("MFM (%s, %d):  One or more File Items expired (removed) due to Subname 'out-of-service' indication by Application.\n"),
                        hManager->mfmClientName, hManager->mfmClientId);
                }
            }
        }
        else {
            // The file identity parser failed to parse the Metadata Name field.
            // This could happen if the Name field of the message was corrupted;
            // but this is a very low probability due to 32-bit CRC check.
            // It's also possible if the Metadata message carousel contains
            // Metadata messages targeted to other services and these services use
            // incompatible Metadata Name fields. For this case, we want to
            // just drop processing of this Metadata message and continue normal
            // operation. Note that if a single Metadata carousel was to be multiplexed
            // for multiple services in this way, the uplink would still have to maintain
            // unique RFD File IDs for the different Update Files across all services.

            RFD_DPrint( TEXT("MFM (%s, %d): _HandleRfdUpdateFileMetadataMessage(), Warning, File Identity Parser failed to parse Metadata Name \"%s\"\n"),
                hManager->mfmClientName, hManager->mfmClientId, rfdUpdateFileMetadataInfo->name );
        }

    } while(FALSE);

    if(hFileIdentity != NULL) {
        // (anyone else using this must use their own copy)
        RFD_FREE(hFileIdentity);
        hFileIdentity = NULL;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _HandleRfdBlockMessage(
    MULTIFILE_MANAGER_HANDLE hManager,
    UCHAR * messageBuf,
    UINT32 messageSize,
    UINT16 blkMessageSourceDmi )
{
    RFD_STATUS status;
    RFD_LLIST_NODE_HANDLE hListNode;
    RFD_LLIST_HANDLE busyMfmRfdClientList;
    MFM_RFD_CLIENT_HANDLE busyMfmRfdClientHandle;
    RFD_COLLECTOR_THREAD_DATA_HANDLE rfdMessageCollectorHandle = NULL;
    BOOL foundTargetRfdClient = FALSE;
    BOOL doFilterBlkMsgDmi;

    if( blkMessageSourceDmi == RFD_DMI_UNUSED_VALUE )  {
        doFilterBlkMsgDmi = FALSE;
    }
    else {
        doFilterBlkMsgDmi = TRUE;
    }

    busyMfmRfdClientList = hManager->rfdClientResourceInfo.busyMfmRfdClientList;

	//////////////////////////////////////////
    // Search the list of Busy (Active) RFD Clients for an RFD Client
    // to which the input RFD Block Message is to be routed.
    // If one is found, then Put/Post the message to that target RFD Client.
	/////////////////////////////////////////

	/////////////////////////////////////////
    // For each MFM RFD Client in the Busy MFM RFD Client List:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    busyMfmRfdClientList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        busyMfmRfdClientHandle = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

       	rfdMessageCollectorHandle = busyMfmRfdClientHandle->rfdMessageCollectorHandle;

        /////////////////////////////////////////////////////////////////////////////////////
        // Determine if this RFD Block message should be passed to this RFD Collector instance.
        /////////////////////////////////////////////////////////////////////////////////////
        status = RFD_CollectorIsMessageCollectableExt(
            rfdMessageCollectorHandle,
            messageBuf,
            messageSize,
            TRUE, // Enable filtering based on Block Message File IDs
            doFilterBlkMsgDmi,
            blkMessageSourceDmi );

        if(status == RFD_STATUS_OK) {

            // Found an RFD Client to which this RFD Block can be routed to.
            // Break out of the list iterate loop.
            foundTargetRfdClient = TRUE;
            break;
        }
        // else this RFD Block does not 'belong' to this RFD Client,
        // continue checking the other RFD Clients in the list.

        status = RFD_LLIST_Next(
                        busyMfmRfdClientList,
                        &hListNode );

    }  // End of List iteration

    if(foundTargetRfdClient == TRUE) {

        /////////////////////////////////////////////////////////////////////////////////////
        // Put/Post the RFD Block message to the found target RFD Client.
        /////////////////////////////////////////////////////////////////////////////////////
        status = RFD_CollectorPutMessage(
                        rfdMessageCollectorHandle,
                        messageBuf,
                        messageSize);

        if(status != RFD_STATUS_OK) {

            // Put Message Buffer may be full.
            RFD_DPrint( TEXT("MFM (%s, %d): _HandleRfdBlockMessage() message put failed, status %d\n"),
                hManager->mfmClientName, hManager->mfmClientId, status);
        }
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Public Functions
////////////////////////////////////////////////////////////////////////////////////////////////////

// doxygen group start symbol, don't remove.
///////////////////////////////////////////////////////////////////////////////////////////////////
/// @addtogroup MfmAPI RFD Multifile Manager (MFM) API
/// @{
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Use of the MFM_Run() function is only relevant for applications running in the
/// "Alt Single Threaded Mode" (enalbled with macro RFD_ALT_SINGLE_THREAD_MODE_ENABLE),
/// which is a non-standard mode of operation (contact Sirius XM for additional info).
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS MFM_Run(
            MULTIFILE_MANAGER_HANDLE mfmHandle,
            BOOL *isThreadExitRequestedPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_STATUS statusLL = RFD_STATUS_OK;
    BOOL isThreadExitRequested = FALSE;
    MFM_FILE_CONSUMER_HANDLE hMfmFileConsumer = NULL;;

    do { // once

        if(mfmHandle == NULL) {
            // error
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        hMfmFileConsumer = mfmHandle->fileConsumerInfo.hMfmFileConsumer;

        if(hMfmFileConsumer == NULL) {
            // error
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

         if(hMfmFileConsumer->isRunning == FALSE) {
		    RFD_DPrint( TEXT("MFM File Consumer: Error, MFM File Consumer is not in run state\n"));
            return RFD_STATUS_ERROR_INVALID_STATE;
        }

        if(hMfmFileConsumer->isAltSingleThreadMode == FALSE) {
		    RFD_DPrint( TEXT("MFM File Consumer: Error, MFM_Run() cannot be called in standard Multi-Threaded mode\n"));
            return RFD_STATUS_ERROR_INVALID_MODE;
        }

        // MFM File Consumer is the only "active" component of MFM, run it.
        status = MFM_FileConsumerRunLL(
                        mfmHandle->fileConsumerInfo.hMfmFileConsumer,
                        &statusLL,
                        &isThreadExitRequested );

    } while(FALSE);

    *isThreadExitRequestedPtr = isThreadExitRequested;

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Delete the specified Multifile Manager (MFM) instance.
///
/// All resources owned by the Multifile Manager instance are cleaned-up (de-allocated). The
/// caller must ensure that no other MFM API functions are called concurrently (e.g.
/// MFM_PutRfdMessage()).
///
/// @param  mfmHandle   The handle of the Multifile Manager instance that is to be deleted. This
///                     is the handle obtained using MFM_Create().
////////////////////////////////////////////////////////////////////////////////////////////////////

void MFM_Delete( MULTIFILE_MANAGER_HANDLE mfmHandle )
{
    if(mfmHandle != NULL) {

        // cleanup the MFM File Consumer first
        _DeleteFileConsumer( mfmHandle );

        // cleanup the Active DMI list
        _DeleteActiveDmiList( mfmHandle );

        // cleanup rfd resource lists
        _DeleteRfdClientResourceLists(
            &mfmHandle->rfdClientResourceInfo );

        // cleanup file management lists
        _DeleteFileManagementLists(
            &mfmHandle->fileManagementLists );

        // Cleanup RFD Message Processing Info
        _UninitializeRfdMessageProcessingInfo(
            &mfmHandle->mfmRfdMessageProcessInfo );

        // cleanup mutex
        if(mfmHandle->hMutex != NULL) {
            RFD_CloseMutex(mfmHandle->hMutex);
            mfmHandle->hMutex = NULL;
        }

        // Cleanup client identity (deallocate client name)
        _UninitializeClientIdentity( mfmHandle );

        // cleanup mfmHandle last
        RFD_FREE(mfmHandle);
        mfmHandle = NULL;
    }

    return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a Multifile Manager (MFM) instance.
///
/// MFM_Create() must be called only after the RFD Receiver has been *started* using
/// RFD_StartReceiver(). This is because on create, MFM gets and uses state information for its
/// assigned RFD Clients, and this state info is only initialized on completion of
/// RFD_StartReceiver().
///
/// @param  mfmHandlePtr                (output) Pointer to the handle of the created Multifile
///                                     Manager, on successful return.
/// @param  mfmClientId                 An id number the caller can assign to the Multifile
///                                     Manager instance being created.
/// @param  mfmClientName               A name string the caller can assign to the Multifile
///                                     Manager instance being created. Caller sets value to NULL
///                                     if no name is assigned.
/// @param  mfmRfdClientArray           The array of MFM RFD Client structures that specify the RFD
///                                     Clients used by the Multifile Manager.
/// @param  numRfdClients               Number of RFD Clients in the array
///                                     mfmRfdClientArray.
/// @param  appCallbackInfo             Pointer to the that specifies the application callback
///                                     functions and arguments used by MFM.
/// @param  fileManagementPolicyInfo    Pointer to structure that specifies the MFM file
///                                     management policies (i.e. rules governing how MFM
///                                     operates in processing Update Files)
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS MFM_Create(
                MULTIFILE_MANAGER_HANDLE * mfmHandlePtr,
                UINT16 mfmClientId,
                const TCHAR * mfmClientName,
                MFM_RFD_CLIENT_STRUCT mfmRfdClientArray[],
                UINT16 numRfdClients,
                MFM_APP_CALLBACK_INFO_STRUCT * appCallbackInfo,
                MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT * fileManagementPolicyInfo )
{
    // Select the standard multi-threaded mode of operation.
    BOOL isAltSingleThreadMode = FALSE;

    // Select the default File Ready Poll Interval.
    INT32 fileReadyPollIntervalMillisec = MFM_DEFAULT_FILE_READY_POLL_INTERVAL_MSEC;

    RFD_STATUS status;

    status = MFM_CreateExt(
                mfmHandlePtr,
                mfmClientId,
                mfmClientName,
                mfmRfdClientArray,
                numRfdClients,
                appCallbackInfo,
                fileManagementPolicyInfo,
                fileReadyPollIntervalMillisec,
                isAltSingleThreadMode );

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a Multifile Manager (MFM) instance.
///
/// MFM_CreateExt() is an *extended* version of MFM_Create() that supports the Alternate
/// Single Thread mode of operation using the isAltSingleThreadMode parameter, and supports the
/// fileReadyPollIntervalMillisec input parameter.
///
/// MFM_CreateExt() must be called only after the RFD Receiver has been *started* using
/// RFD_StartReceiver(). This is because on create, MFM gets and uses state information for its
/// assigned RFD Clients, and this state info is only initialized on completion of
/// RFD_StartReceiver().
///
/// @param  mfmHandlePtr                    (output) Pointer to the handle of the created
///                                         Multifile Manager, on successful return.
/// @param  mfmClientId                     An id number the caller can assign to the Multifile
///                                         Manager instance being created.
/// @param  mfmClientName                   A name string the caller can assign to the Multifile
///                                         Manager instance being created. Caller sets value to
///                                         NULL if no name is assigned.
/// @param  mfmRfdClientArray               The array of MFM RFD Client structures that specify the RFD
///                                         Clients used by the Multifile Manager.
/// @param  numRfdClients                   Number of RFD Clients in the array
///                                         mfmRfdClientArray.
/// @param  appCallbackInfo                 Pointer to the that specifies the application
///                                         callback functions and arguments used by MFM.
/// @param  fileManagementPolicyInfo        Pointer to structure that specifies the MFM file
///                                         management policies (i.e. rules governing how MFM
///                                         operates in processing Update Files)
/// @param  fileReadyPollIntervalMillisec   The interval in milliseconds that the MFM File
///                                         Consumer thread waits between each iteration of
///                                         polling the assigned RFD Clients for New File Ready
///                                         events. Set to negative value or
///                                         MFM_DEFAULT_FILE_READY_POLL_INTERVAL_MSEC to
///                                         select the default poll iterval. The default interval
///                                         provides good responsiveness to New File Ready events
///                                         while minimizing the cpu utilization due to polling
///                                         to very low levels (e.g. fraction of 1%). Some
///                                         applications may choose to customize the poll time
///                                         for fine tuning responsiveness vs. cpu utilization,
///                                         but this is generally not necessary.
/// @param  isAltSingleThreadMode           TRUE to select the alternate single-thread mode of
///                                         MFM operation, FALSE to select the standard multi-
///                                         threaded mode of MFM operation.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS MFM_CreateExt(
                MULTIFILE_MANAGER_HANDLE * mfmHandlePtr,
                UINT16 mfmClientId,
                const TCHAR * mfmClientName,
                MFM_RFD_CLIENT_STRUCT mfmRfdClientArray[],
                UINT16 numRfdClients,
                MFM_APP_CALLBACK_INFO_STRUCT * appCallbackInfo,
                MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT * fileManagementPolicyInfo,
                INT32 fileReadyPollIntervalMillisec,
                BOOL isAltSingleThreadMode )
{
    RFD_STATUS status = RFD_STATUS_OK;
    MULTIFILE_MANAGER_STRUCT * mfmHandle = NULL;
    RFD_COLLECTOR_THREAD_DATA_HANDLE hDefaultRfdMessageCollector;

    if(mfmHandlePtr == NULL) {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    do { // once

        ///////////////////////////////////////////
        // Check the fileReadyPollIntervalMillisec input parameter for special case value.
        // A negative specified Poll Interval is indication to use the Default Poll Interval.
        ///////////////////////////////////////////
        if(fileReadyPollIntervalMillisec < 0) {
            fileReadyPollIntervalMillisec = MFM_DEFAULT_FILE_READY_POLL_INTERVAL_MSEC;
        }

        ///////////////////////////////////////////
        // Create the main structure
        //
        // Use CALLOC() to clear values initially (NULL values) in case
        // of early cleanup.
        ///////////////////////////////////////////
        mfmHandle = RFD_CALLOC(sizeof(MULTIFILE_MANAGER_STRUCT));

        if(mfmHandle == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        ///////////////////////////////////////////
        // Initialize the MFM Client Name and Id
        ///////////////////////////////////////////

        status = _InitializeClientIdentity( mfmHandle, mfmClientId, mfmClientName );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        RFD_DPrint(
            TEXT("MFM: MFM_CreateExt() Creating Multifile Manager instance. MFM Handle (0x%x), Name (%s), Id (%d)\n"),
            mfmHandle,
            mfmHandle->mfmClientName,
            mfmHandle->mfmClientId );

        ///////////////////////////////////////////
        // Create MFM Mutex
        ///////////////////////////////////////////
        mfmHandle->hMutex = RFD_CreateMutex();

        if (mfmHandle->hMutex == NULL) {
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        ///////////////////////////////////////////
        // Initialize the MFM File Management Policy
        ///////////////////////////////////////////
        mfmHandle->fileManagementPolicyInfo = *fileManagementPolicyInfo;

        if( mfmHandle->fileManagementPolicyInfo.updateType !=
            MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES ) {

            // error, unsupported mode
            // DELTA type updates currently not supported.
            status = RFD_STATUS_UNSUPPORTED_FEATURE;
            break;
        }

        ///////////////////////////////////////////
        // Initialize MFM Callback Info with callback info structure from the app
        ///////////////////////////////////////////

        status = _InitializeCallbackInfo(
                        mfmHandle,
                        appCallbackInfo,
                        &mfmHandle->fileManagementPolicyInfo );
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Initialize the RFD Message Processing Info
        ///////////////////////////////////////////

        // Use fist RFD Client of input array as default handle
        // (used to get the rfd default message config info, which must
        // be the same for all RFD Clients processed by an MFM instance.
        hDefaultRfdMessageCollector =
            mfmRfdClientArray[0].rfdMessageCollectorHandle;

        status = _InitializeRfdMessageProcessingInfo(
                    &mfmHandle->mfmRfdMessageProcessInfo,
                    hDefaultRfdMessageCollector );
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Create the RFD Client Resource Lists
        ///////////////////////////////////////////
        status = _CreateRfdClientResourceLists(
                    &mfmHandle->rfdClientResourceInfo );
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Create the File Item Management Lists
        ///////////////////////////////////////////
        status = _CreateFileManagementLists(
                    &mfmHandle->fileManagementLists );
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Create the Active DMI List
        ///////////////////////////////////////////
        status = _CreateActiveDmiList(
                    mfmHandle );
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Initialize the Active File Items
        // - Check the initial state of the passed in RFD Clients.
        //   For each RFD Client that is currently processing an assigned update,
        //   make that RFD Client busy and setup a corresponding Active File Item.
        //   Otherwise add RFD Client to the RFD Client free list.
        //
        // Call only after all MFM (list) creation is complete,
        // after the initialization of Stored File Items,
        // and after initialization of app provided callbacks (_InitializeCallbackInfo()),
        //(for use of the File Idenendity Parser).
        ///////////////////////////////////////////
        status = _InitializeFromRfdInitialState(
                    mfmHandle,
                    mfmRfdClientArray,
                    numRfdClients );
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Create the MFM File Consumer Thread
        ///////////////////////////////////////////
        status = _CreateFileConsumer(
                        mfmHandle,
                        mfmRfdClientArray,
                        numRfdClients,
                        fileReadyPollIntervalMillisec,
                        isAltSingleThreadMode );
        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Initialize File Item Expiration Processing Info
        // (includes Last File Item Expiration Check SXI Time).
        ///////////////////////////////////////////
        status = _InitializeExpirationProcessInfo( mfmHandle );

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        ///////////////////////////////////////////
        // Create is complete, now print out the current state.
        ///////////////////////////////////////////
        _PrintAll( mfmHandle );

        ///////////////////////////////////////////
        // successful create/init
        ///////////////////////////////////////////
        status = RFD_STATUS_OK;

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup.
        MFM_Delete(mfmHandle);
        mfmHandle = NULL;

        RFD_DPrint(
            TEXT("MFM (%s, %d):: MFM_CreateExt() error - creation failed status (%d)\n"),
            mfmHandle->mfmClientName, mfmHandle->mfmClientId,
            status );
    }
    else {
        RFD_DPrint(
            TEXT("MFM (%s, %d):: MFM_CreateExt() success\n"),
            mfmHandle->mfmClientName, mfmHandle->mfmClientId );
    }

    *mfmHandlePtr = mfmHandle;
    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief  Put an RFD Message to a Multifile Manager (MFM) instance.
///
/// @param  mfmHandle           Handle of the Multifile Manager instance.
/// @param  messageBuf          Buffer containing the RFD message data.
/// @param  messageSize         Size of the RFD message.
/// @param  blkMessageSourceDmi The DMI source of a passed in Block type RFD Message. MFM uses
///                             this value to determine the RFD Update File to which this Block
///                             message corresponds, to then route the Block message to the
///                             correct RFD Client that is processing this Update File. If no
///                             matching Update File is found based on blkMessageSourceDmi (and
///                             also the File Unique Identifer that is part of the Block message),
///                             the Block message is not processed by MFM. MFM also ignores the
///                             blkMessageSourceDmi parameter for SXM Data Services and
///                             corresponding Update Files that do not implement RFD based DMI
///                             specification (services that do not utilize the Update File
///                             Metadata Message DMI field to specify DMI source of RFD Block
///                             messages). The caller may also set blkMessageSourceDmi to
///                             RFD_DMI_UNUSED_VALUE to disable MFM use this parameter.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS MFM_PutRfdMessage(
    MULTIFILE_MANAGER_HANDLE mfmHandle,
    UCHAR * messageBuf,
    UINT32 messageSize,
    UINT16 blkMessageSourceDmi )
{
    RFD_STATUS status;
    BOOL isUpdateFileMetadataMessage = FALSE;
    BOOL isWaitingListIncremented = FALSE;
    BOOL isActiveListDecremented = FALSE;
    BOOL isFileItemActivated = FALSE;
    BOOL isFileItemExpired = FALSE;
    BOOL isFilePrioritySortingEnabled = FALSE;
    BOOL isMetadataCarouselReceptionComplete = FALSE;
    BOOL isMutexAquired = FALSE;

    do { // once

        /////////////////////////////////////////////////////////////////////////////////////
        // Aquire Mutex
        // This is required if application calls MFM API concurrently from
        // different threads. E.g. if MFM_PutRfdMessage() is called from
        // different threads maybe due to application feeding RFD messages to
        // the same MFM from different DMI's handled by different messaging threads?
        //
        // Note: Application needs to be careful if it aquired it's own Mutex locks
        // in any callbacks to it from MFM e.g. the sorting callbacks. In this case,
        // there could be the potential for interactive deadlock due to the MFM mutex
        // and the appliction mutex being locked in oposite ordering from different threads.
        /////////////////////////////////////////////////////////////////////////////////////

        status = RFD_AcquireMutex(mfmHandle->hMutex, RFD_INFINITE);

        if(status == RFD_STATUS_OK) {
            // mutex sucessfully aquired.
            isMutexAquired = TRUE;
        }
        else {
            // error
            break;
        }

        /////////////////////////////////////////////////////////////////////////////////////
        // Check if this message is an RFD Update File Metadata Message
        /////////////////////////////////////////////////////////////////////////////////////
        status = RFD_CollectorIsUpdateFileMetadataMessage(
                        mfmHandle->mfmRfdMessageProcessInfo.rfdDefaultMessageConfigInfo,
                        messageBuf,
                        messageSize,
                        &isUpdateFileMetadataMessage);

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d):: RFD_CollectorIsUpdateFileMetadataMessage() error, status %d\n"),
                mfmHandle->mfmClientName, mfmHandle->mfmClientId, status);
            break;
        }

        if(isUpdateFileMetadataMessage == TRUE) {

            ///////////////////////////////////////////////////////////////////////
            // Message is an Update File Metatdata message.
            ///////////////////////////////////////////////////////////////////////

            ///////////////////////////////////////////////////////////////////////
            // Process this Update File Metadata Message
            ///////////////////////////////////////////////////////////////////////

            status = _HandleRfdUpdateFileMetadataMessage(
                            mfmHandle,
                            messageBuf,
                            messageSize,
                            &isWaitingListIncremented );

            if(status != RFD_STATUS_OK) {
                // error
                RFD_DPrint(
                    TEXT("MFM (%s, %d):: _HandleRfdUpdateFileMetadataMessage() error, status %d\n"),
                    mfmHandle->mfmClientName, mfmHandle->mfmClientId, status);
                break;
            }

            if(isWaitingListIncremented == TRUE) {
                RFD_DPrint(
                    TEXT("MFM (%s, %d):  One or more File Items added to waiting list.\n"),
                    mfmHandle->mfmClientName, mfmHandle->mfmClientId);
            }

            ///////////////////////////////////////////////////////////////////////
            // Check for Lifetime expiration of Waiting and Active File Items.
            //
            // Do this on Update File Metadata Message reception only; performing
            // the check also on every RFD Block message would be excessive.
            // This Expiration Process function avoids checking all file items if current time hasn't
            // changed and no file item life fields were updated; but even so, it's still not necessary
            // to call it as frequently as every Block Message (the function at least has to get the
            // the current time and do a little work).
            //
            // Call this after _HandleRfdUpdateFileMetadataMessage() as the Life field of an existing
            // File Item may get updated there (maybe resulting in File Item becomming expired
            // if the Life value was updated to an earlier time). By calling right after, it just
            // means that such an expired File Item would be removed immediately instead of
            // on the next RFD Update File Metadata message received (may be more helpful for testing
            // when ordered this way.
            ///////////////////////////////////////////////////////////////////////

            status = _ProcessRfdLifeExpiredFileItems(
                            mfmHandle,
                            &isFileItemExpired );

            if(status != RFD_STATUS_OK) {
                // error
                RFD_DPrint(
                    TEXT("MFM (%s, %d):: _ProcessRfdLifeExpiredFileItems() error, status %d\n"),
                    mfmHandle->mfmClientName, mfmHandle->mfmClientId, status);
                break;
            }

            if(isFileItemExpired == TRUE) {
                RFD_DPrint(
                    TEXT("MFM (%s, %d):  One or more File Items expired due to RFD Life.\n"),
                    mfmHandle->mfmClientName, mfmHandle->mfmClientId);
            }

        }
        else {
            // assume message is RFD Block message

            status = _HandleRfdBlockMessage(
                            mfmHandle,
                            messageBuf,
                            messageSize,
                            blkMessageSourceDmi );

            if(status != RFD_STATUS_OK) {
                // error
                RFD_DPrint(
                    TEXT("MFM (%s, %d):: _HandleRfdBlockMessage() error, status %d\n"),
                    mfmHandle->mfmClientName, mfmHandle->mfmClientId, status);
                break;
            }
        }

        ///////////////////////////////////////////////////////////////////////
        // Check for File Completion updates from the MFM File Consumer.
        ///////////////////////////////////////////////////////////////////////

        status = _HandleFileConsumerUpdates(
                        mfmHandle,
                        &isActiveListDecremented );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(
                TEXT("MFM (%s, %d): _HandleFileConsumerUpdates() error, status %d\n"),
                mfmHandle->mfmClientName, mfmHandle->mfmClientId, status);
            break;
        }

        if(isActiveListDecremented == TRUE) {
            RFD_DPrint(
                TEXT("MFM (%s, %d):  One or more File Items removed from active list.\n"),
                mfmHandle->mfmClientName, mfmHandle->mfmClientId);
        }

        ///////////////////////////////////////////////////////////////////////
        //
        // Only try File Item Activations after
        // the RFD Update File Metatdata Carousel is received/completed, unless
        // File Priority Sorting is disabled (not enabled).
        //
        // Try "activating" File Items if the above conditions are met.
        //
        // Note: The logic to call Try File Item Activation only if the
        // Waiting List was just incremented or the Active List was just decremented
        // was removed. This was done to avoid possible bug introduction in case there
        // is ever an implementation change that does postponement of File Item Activations
        // when Sorting feature is enabled but the Application Sorter is not ready.
        ///////////////////////////////////////////////////////////////////////

        isMetadataCarouselReceptionComplete =
            _IsMetadataCarouselReceptionComplete( mfmHandle );

        isFilePrioritySortingEnabled =
            mfmHandle->fileManagementPolicyInfo.isFilePrioritySortingEnabled;

        if( isFilePrioritySortingEnabled == FALSE || isMetadataCarouselReceptionComplete == TRUE) {

            status = _TryFileItemActivations(
                            mfmHandle,
                            &isFileItemActivated );

            if(status != RFD_STATUS_OK) {
                // error
                RFD_DPrint(
                    TEXT("MFM (%s, %d): _TryFileItemActivations() error, status %d\n"),
                    mfmHandle->mfmClientName, mfmHandle->mfmClientId, status);
                break;
            }

            if(isFileItemActivated == TRUE) {
                RFD_DPrint(
                    TEXT("MFM (%s, %d):  One or more File Items were activated.\n"),
                    mfmHandle->mfmClientName, mfmHandle->mfmClientId);
            }
        }

        if( isFileItemActivated == TRUE ||
            isWaitingListIncremented == TRUE ||
            isActiveListDecremented == TRUE ||
            isFileItemExpired == TRUE ) {

            ///////////////////////////////////////////////////////////////////////
            // Print some of the lists that may have changed.
            ///////////////////////////////////////////////////////////////////////

            _PrintFramedActiveDmiList( mfmHandle );
            _PrintActiveFileItemList( mfmHandle );
            _PrintWaitingFileItemListList( mfmHandle );
            _PrintAllMfmRfdClientLists( mfmHandle );
        }

    } while(FALSE);

    // Release Mutex if it's aquired.
    if(isMutexAquired == TRUE) {
	    RFD_ReleaseMutex(mfmHandle->hMutex);
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Update the MFM File Priority (Sort) Callback Argument.
///
/// This is the appCallbackInfo->mfmFilePriorityCallbackArg parameter that the application
/// initially sets through MFM_Create(). An example use case is when the File Priority Sorting
/// criteria changes over time. The argument may then represent the changing sort criteria, for
/// example, the current location (e.g. for the Non-Nav Traffic Data Service). Using the
/// MFM_UpdateFilePriorityCallbackArg(), the application can update the sort criteria when it
/// changes (e.g. change in current location)
///
/// The old argument value that is being replaced is output by this function. On function
/// return, this old argument is no longer accessed by MFM or by MFM File Priority Sort
/// callbacks. The caller is therefore free to deallocate it, etc.
///
/// The MFM_UpdateFilePriorityCallbackArg() is mutex protected by MFM so that the File Priority
/// Sort Argument can not be updated during invocation of any of the File Priority Sort Callback
/// functions (no risk of concurrent access).
///
/// @param  mfmHandle   Handle of the Multifile Manager instance.
/// @param  newArg      The new argument.
/// @param  oldArgPtr   (output) Pointer to the old argument. On successful return, MFM outputs
///                     the old File Priority Sort Argument that has now been replaced by the
///                     new argument.
/// @return
/// RFD_STATUS_OK if there is no error, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS MFM_UpdateFilePriorityCallbackArg(
                MULTIFILE_MANAGER_HANDLE mfmHandle,
                void * newArg,
                void ** oldArgPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;

    // initialize mutex aquired to FALSE
    BOOL isMutexAquired = FALSE;
    void * oldArg = NULL;

    // check input parameters.
    if( mfmHandle == NULL || oldArgPtr == NULL ) {
        // error
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    // initialize output value to NULL
    *oldArgPtr = NULL;

    do { // once

        status = RFD_AcquireMutex(mfmHandle->hMutex, RFD_INFINITE);

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }
        // mutex sucessfully aquired.
        isMutexAquired = TRUE;

        if( mfmHandle->fileManagementPolicyInfo.isFilePrioritySortingEnabled == FALSE ) {
            // Should not update the callback if MFM was created with Priority Sorting disabled.
            RFD_DPrint(
                TEXT("MFM (%s, %d):  MFM_UpdateFilePriorityCallbackArg(), ")
                TEXT("No action because File Priority Sorting was disabled on MFM Create.\n"),
                mfmHandle->mfmClientName,
                mfmHandle->mfmClientId );
            status = RFD_STATUS_ERROR_INVALID_MODE;
            break;
        }

        // preserve the old argument, to return to caller.
        oldArg = mfmHandle->appCallbackInfo.mfmFilePriorityCallbackArg;

        // update the File Priority Sort callback argument
        mfmHandle->appCallbackInfo.mfmFilePriorityCallbackArg = newArg;

        RFD_DPrint(
            TEXT("MFM (%s, %d):  MFM_UpdateFilePriorityCallbackArg(), Callback Argument updated from 0x%x to 0x%x.\n"),
            mfmHandle->mfmClientName,
            mfmHandle->mfmClientId,
            oldArg,
            newArg );

    } while(FALSE);

    // Release Mutex if it's aquired.
    if(isMutexAquired == TRUE) {
	    RFD_ReleaseMutex(mfmHandle->hMutex);
    }

    // set output
    *oldArgPtr = oldArg;

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Check for an error reported by a Multifile Manager instance.
///
/// The function will block, waiting for an error event indication, for timeOutMilliSeconds
/// milliseconds.
///
/// @param  mfmHandle           Handle of the Mulitifile Mangager instance.
/// @param  timeOutMilliSeconds The time-out in milliseconds. Value RFD_INFINITE to wait
///                             indefinitely. Value 0 to return immediately.
///
/// @return
/// RFD_STATUS_OK if there is no error, RFD_STATUS_ERROR_THREAD_EXIT if there is an Multifile
/// Mangager error, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS MFM_ErrorCheck(MULTIFILE_MANAGER_HANDLE mfmHandle, INT32 timeOutMilliSeconds)
{
    RFD_STATUS status;

    status = MFM_FileConsumerErrorCheck(
                    mfmHandle->fileConsumerInfo.hMfmFileConsumer,
                    timeOutMilliSeconds );

    return status;
}

///////////////////////////////////////////////////////////////////////////////
void MFM_UTIL_PrintFileItemIdentity(
        MFM_FILE_IDENTITY_HANDLE hFileIdentity,
        const TCHAR * headerName1,
        const TCHAR * headerName2,
        const TCHAR * headerName3 )
{
    const TCHAR * noName = TEXT("");

    if(headerName1 == NULL) {
        headerName1 = noName;
    }
    if(headerName2 == NULL) {
        headerName2 = noName;
    }
    if(headerName3 == NULL) {
        headerName3 = noName;
    }

    RFD_DPrint( TEXT("%s %s %s : Subname (%s), Version From (%d), Version To (%d)\n"),
        headerName1,
        headerName2,
        headerName3,
        hFileIdentity->subname,
        hFileIdentity->versionFrom,
        hFileIdentity->versionTo);

    return;
}

///////////////////////////////////////////////////////////////////////////////
int MFM_UTIL_CompareFileIdentity(
               MFM_FILE_IDENTITY_HANDLE hFileIdentityA,
               MFM_FILE_IDENTITY_HANDLE hFileIdentityB,
               MFM_FILE_IDENTITY_MASK compareMask )
{
    int result = 0;

    // Subname
    if(compareMask & MFM_FILE_IDENTITY_SUBNAME) {
        if(RFD_STRNCMP(hFileIdentityA->subname, hFileIdentityB->subname, MFM_SUBNAME_MAX_LEN)) {
            // not equal
            result += 1;
        }
    }

    // Version From
    if(compareMask & MFM_FILE_IDENTITY_VERSION_FROM) {
        if(hFileIdentityA->versionFrom != hFileIdentityB->versionFrom) {
            // not equal
            result += 1;
        }
    }

    // Version To
    if(compareMask & MFM_FILE_IDENTITY_VERSION_TO) {
        if(hFileIdentityA->versionTo != hFileIdentityB->versionTo) {
            // not equal
            result += 1;
        }
    }

    // return 0 if equal, !=0 otherwise.
    return result;
}

///////////////////////////////////////////////////////////////////////////////
void MFM_UTIL_UpdateFileIdentity(
           MFM_FILE_IDENTITY_HANDLE hFileIdentityDst,
           MFM_FILE_IDENTITY_HANDLE hFileIdentitySrc,
           MFM_FILE_IDENTITY_MASK updateMask )
{
    // Subname
    if(updateMask & MFM_FILE_IDENTITY_SUBNAME) {

        RFD_STRCPY_S(hFileIdentityDst->subname, MFM_SUBNAME_MAX_LEN, hFileIdentitySrc->subname);
    }

    // Version From
    if(updateMask & MFM_FILE_IDENTITY_VERSION_FROM) {

        hFileIdentityDst->versionFrom = hFileIdentitySrc->versionFrom;
    }

    // Version To
    if(updateMask & MFM_FILE_IDENTITY_VERSION_TO) {

        hFileIdentityDst->versionTo = hFileIdentitySrc->versionTo;
    }

    return;
}

///////////////////////////////////////////////////////////////////////////////
void MFM_UTIL_SetFileIdentity(
           MFM_FILE_IDENTITY_HANDLE hFileIdentity,
           const TCHAR * subname,
           UINT32 versionFrom,
           UINT32 versionTo )
{
    // Subname
    RFD_STRCPY_S(hFileIdentity->subname, MFM_SUBNAME_MAX_LEN, subname);

    // Version From
    hFileIdentity->versionFrom = versionFrom;

    // Version To
    hFileIdentity->versionTo = versionTo;

    return;
}

// doxygen group end symbol, don't remove.
/////////////////////////////////////////////////////////////////////////////////////////////
/// @}  doxygen group end symbol, don't remove.
////////////////////////////////////////////////////////////////////////////////////////////////

