////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	multifile_manager\mfm_consumer_setup.c
///
/// @brief	multifile manager file consumer setup class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_receiver.h"
#include "rfd_linked_list.h"
#include "rfd_file_consumer.h"
#include "mfm_consumer.h"

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_CreateFileConsumerExt(
    MFM_FILE_CONSUMER_HANDLE * mfmFileConsumerHandlePtr,
    RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle[],
    UINT16 numRfdFileConsumerHandles,
    MFM_MANAGER_FILE_TRANSFER_CALLBACK_FCN mfmManagerFileTransferCallbackFcn,
    void * mfmManagerFileTransferCallbackArg,
    INT32 fileReadyPollIntervalMillisec,
    BOOL isAltSingleThreadMode
    )
{
    RFD_STATUS status = RFD_STATUS_OK;
    MFM_FILE_CONSUMER_HANDLE hMfmFileConsumer;
    UINT16 i;

	// Initialize output handle to null.
	*mfmFileConsumerHandlePtr = NULL;

    hMfmFileConsumer = NULL;

    do { // once

        ///////////////////////////////////////////////////
        // Open the MFM File Consumer
        ///////////////////////////////////////////////////
        status = MFM_OpenFileConsumerExt(
                        &hMfmFileConsumer,
                        isAltSingleThreadMode );

        if(status != RFD_STATUS_OK) {
            break;
	    }

        ///////////////////////////////////////////////////
        // Initialialize required parameters
        // after successful Open(), but befor Start()
        ///////////////////////////////////////////////////

        // MFM File Transfer Callback Function
        hMfmFileConsumer->mfmManagerFileTransferCallbackFcn =
            mfmManagerFileTransferCallbackFcn;

        // MFM File Transfer Callback argument
        hMfmFileConsumer->mfmManagerFileTransferCallbackArg =
            mfmManagerFileTransferCallbackArg;

        // MFM File Consumer Thread Polling Interval for New File Ready events.
        hMfmFileConsumer->fileReadyPollIntervalMillisec =
            fileReadyPollIntervalMillisec;

        // Insert each RFD File Consumer Handle to the List
        for(i=0;i<numRfdFileConsumerHandles;i++) {

            status = RFD_LLIST_InsertAtHead(
                            hMfmFileConsumer->rfdFileConsumerHandleList,
                            rfdFileConsumerHandle[i] );

            if(status != RFD_STATUS_OK) {
                break; // from for loop
            }
        }

        if(status != RFD_STATUS_OK) {
            break;
        }

        ///////////////////////////////////////////////////
        // Start the MFM File Consumer
        ///////////////////////////////////////////////////
        status = MFM_StartFileConsumer( hMfmFileConsumer );

        if(status != RFD_STATUS_OK) {
            break;
        }

    } while(FALSE);

    if(status != RFD_STATUS_OK) {
        // error, cleanup.
        if( hMfmFileConsumer != NULL ) {
            MFM_DeleteFileConsumer( hMfmFileConsumer );
            hMfmFileConsumer = NULL;
        }
    }

    // Update output
    *mfmFileConsumerHandlePtr = hMfmFileConsumer;

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_DeleteFileConsumer(
    MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle )
{
    RFD_STATUS status = RFD_STATUS_OK;

    if(mfmFileConsumerHandle == NULL) {
        return status;
    }

    MFM_StopFileConsumer( mfmFileConsumerHandle );

    // There's nothing to cleanup in the "app data" of each
    // node of the RFD File Consumer Handles list as the "app data"
    // is just the handle which is a pointer. Cleanup of these
    // RFD Handles is the responsibility outside of MFM consumer
    // (but these handles must be cleaned up only after stopping
    // the MFM Consumer which uses the handles).

    MFM_CloseFileConsumer( mfmFileConsumerHandle );

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_OpenFileConsumerExt(
    MFM_FILE_CONSUMER_HANDLE * mfmFileConsumerHandlePtr,
    BOOL isAltSingleThreadMode
    )
{
	MFM_FILE_CONSUMER_DATA_STRUCT *	hConsumer;

	// Initialize output handle to null.
	*mfmFileConsumerHandlePtr = NULL;

	/////////////////////////////////////////////////////
    // Allocate memory for mfm file consumer handle data, and clear structure (calloc).
	/////////////////////////////////////////////////////
	hConsumer = (MFM_FILE_CONSUMER_HANDLE) RFD_CALLOC(sizeof(MFM_FILE_CONSUMER_DATA_STRUCT));
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
    }

    // Set the requested RFDR operating mode
    if(isAltSingleThreadMode) {
        // alternate single-threaded mode
        hConsumer->isAltSingleThreadMode = TRUE;
    }
    else {
        // standard multi-threaded mode
        hConsumer->isAltSingleThreadMode = FALSE;
    }

    // Initialize run state
    hConsumer->isRunning = FALSE;

	/////////////////////////////////////////////////////
	// Initialize the file transfer callback fcn and arg to NULL.
	// It must be initializated after MFM_OpenFileConsumer(),
	// but before MFM_StartFileConsumer().
	/////////////////////////////////////////////////////
    hConsumer->mfmManagerFileTransferCallbackFcn = NULL;
    hConsumer->mfmManagerFileTransferCallbackArg = NULL;

	/////////////////////////////////////////////////////
	// Create the RFD Consumer Handles Linked List object.
	// The RFD Consumer Handles must be inserted to the list
    // after MFM_OpenFileConsumer(), but before MFM_StartFileConsumer().
	/////////////////////////////////////////////////////
	hConsumer->rfdFileConsumerHandleList = RFD_LLIST_CreateList();

	if (hConsumer->rfdFileConsumerHandleList == NULL) {
		MFM_CloseFileConsumer(hConsumer);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////////////////////////////////////////////////
	// Create Error Exit Event object.
	// The event is set by the thread to indicate
	// thread is exiting due to unexpected error condition.
	// The event is monitored here by the 'mfmlication manager' thread.
	/////////////////////////////////////////////////////
	hConsumer->errorExitEvent = RFD_CreateEvent();

	if (hConsumer->errorExitEvent == NULL) {
		MFM_CloseFileConsumer(hConsumer);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////////////////////////////////////////////////
	// Create Exit Request Event object.
	// The event is set by the mfmlication manager
	// and monitored by this consumer thread
	// to command this consumer thread to exit.
	/////////////////////////////////////////////////////
	hConsumer->exitRequestEvent = RFD_CreateEvent();

	if (hConsumer->exitRequestEvent == NULL) {
		MFM_CloseFileConsumer(hConsumer);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	*mfmFileConsumerHandlePtr = hConsumer;
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_OpenFileConsumer(MFM_FILE_CONSUMER_HANDLE * mfmFileConsumerHandlePtr)
{
    RFD_STATUS status;

    status = MFM_OpenFileConsumerExt(mfmFileConsumerHandlePtr, FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_CloseFileConsumer(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;

	hConsumer = mfmFileConsumerHandle;

    if( hConsumer != NULL ) {

		/////////////////////////////////////////////////////
		// Close all thread handles, event handles and free memory allocations.
		/////////////////////////////////////////////////////

		// Consumer Thread
        if(hConsumer->isAltSingleThreadMode == FALSE) {
            // Close thread only in standard mode, not the alt-single-thread mode
		    if(hConsumer->consumerThread != NULL) {
			    RFD_CloseThread(hConsumer->consumerThread);
			    hConsumer->consumerThread = NULL;
		    }
        }

		// exitRequestEvent
		if(hConsumer->exitRequestEvent != NULL) {
			RFD_CloseEvent(hConsumer->exitRequestEvent);
			hConsumer->exitRequestEvent = NULL;
		}

		// errorExitEvent
		if(hConsumer->errorExitEvent != NULL) {
			RFD_CloseEvent(hConsumer->errorExitEvent);
			hConsumer->errorExitEvent = NULL;
		}

        // rfdFileConsumerHandleList
        if(hConsumer->rfdFileConsumerHandleList != NULL) {
            // free the list (including nodes), but not the actual RFD Consumer Handles
            // (mfm data) contained in the list.
            RFD_LLIST_DeleteList(hConsumer->rfdFileConsumerHandleList, NULL);
        }

		// Free the the mfm file consumer Handle last
		RFD_FREE(hConsumer);
		hConsumer = NULL;
	}

	RFD_DPrint( TEXT("MFM File Consumer Cleanup Complete\n"));

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_StartFileConsumerAltSingleThreadMode(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;
	RFD_STATUS status;
    BOOL isThreadExitRequested;

	hConsumer = mfmFileConsumerHandle;
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	/////////////////////////////////////////////////////
	// Create MFM File Consumer run object
	/////////////////////////////////////////////////////

    MFM_FileConsumerOpenLL(
        hConsumer,
        &status,
        &isThreadExitRequested
            );
	RFD_DPrint( TEXT("MFM File Consumer Open\n"));

	if (isThreadExitRequested) {
	   RFD_DPrint( TEXT("Error on MFM File Consumer Open\n"));
	   MFM_StopFileConsumer(hConsumer);
	   return RFD_STATUS_ERROR_CREATE_THREAD;
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_StartFileConsumerStandardMultiThreadMode(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;

	hConsumer = mfmFileConsumerHandle;
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	/////////////////////////////////////////////////////
	// Create RFDR Client File Consumer Threads now that the Processor thread is ready
	/////////////////////////////////////////////////////
	hConsumer->consumerThread = RFD_CreateThread(
											MFM_FILE_CONSUMER_THREAD_STACK_SIZE,
											"MFM File Consumer",
											MFM_FileConsumer,   // thread function name
											hConsumer,			// argument to thread function
											&hConsumer->consumerThreadId);     // returns the thread identifier

	// Check the return value for success.
	// If RFD_CreateThread fails, terminate execution.
	// This will automatically clean up threads and memory.

	if (hConsumer->consumerThread == NULL) {
		MFM_StopFileConsumer(hConsumer);
		return RFD_STATUS_ERROR_CREATE_THREAD;
	}

	/////////////////////////////////////////////////////
	// Set File Consumer Thread Priority
	/////////////////////////////////////////////////////
	if(!RFD_SetThreadPriority(hConsumer->consumerThread, MFM_FILE_CONSUMER_THREAD_PRIORITY)) {
		MFM_StopFileConsumer(hConsumer);
		return RFD_STATUS_ERROR_SET_THREAD_PRIORITY;
	}

	return RFD_STATUS_OK;
}


///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_StartFileConsumer(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;
    RFD_STATUS status;

	hConsumer = mfmFileConsumerHandle;
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if(hConsumer->isRunning == TRUE) {
	    RFD_DPrint( TEXT("MFM File Consumer: Error, already started\n"));
        return RFD_STATUS_ERROR_INVALID_STATE;
    }

    if(hConsumer->isAltSingleThreadMode) {
        status = MFM_StartFileConsumerAltSingleThreadMode(hConsumer);
    }
    else {
        status = MFM_StartFileConsumerStandardMultiThreadMode(hConsumer);
    }

    if(status == RFD_STATUS_OK) {
        // update run state
        hConsumer->isRunning = TRUE;
    }

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_StopFileConsumerAltSingleThreadMode(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;
    RFD_STATUS status;

	hConsumer = mfmFileConsumerHandle;
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	/////////////////////////////////////////////////////
	// Stop (close) the MFM File Consumer run object
	/////////////////////////////////////////////////////

    MFM_FileConsumerCloseLL(
        hConsumer,
        &status
            );
	RFD_DPrint( TEXT("MFM File Consumer Close\n"));

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_StopFileConsumerStandardMultiThreadMode(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;

	hConsumer = mfmFileConsumerHandle;
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	/////////////////////////////////////////////////////
	// Request Shutdown of Thread
	/////////////////////////////////////////////////////

	// Request shutdown of RFDR Processor Thread
	RFD_SetEvent(hConsumer->exitRequestEvent);

	RFD_DPrint( TEXT("Set File Consumer Shutdown Request Event\n"));

	////////////////////////////////////////////////////
	// Wait until thread hase terminated.
	////////////////////////////////////////////////////

	// Wait for processor thread
	if(hConsumer->consumerThread != NULL) {
		RFD_WaitForThreadExit(hConsumer->consumerThread, RFD_INFINITE);
	}
	RFD_DPrint( TEXT("File Consumer Shutdown Request Complete\n"));

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_StopFileConsumer(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;
    RFD_STATUS status;

	hConsumer = mfmFileConsumerHandle;
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if(hConsumer->isRunning == FALSE) {
	    RFD_DPrint( TEXT("MFM File Consumer: Stop not performed because not started\n"));
        return RFD_STATUS_ERROR_INVALID_STATE;
    }

    if(hConsumer->isAltSingleThreadMode) {
        status = MFM_StopFileConsumerAltSingleThreadMode(hConsumer);
    }
    else {
        status = MFM_StopFileConsumerStandardMultiThreadMode(hConsumer);
    }

    // update run state
    hConsumer->isRunning = FALSE;

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_FileConsumerErrorCheck(MFM_FILE_CONSUMER_HANDLE mfmFileConsumerHandle, INT32 timeOutMilliSeconds)
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;
	RFD_STATUS status;
	RFD_STATUS rfdStatus;

	hConsumer = mfmFileConsumerHandle;
    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	rfdStatus = RFD_WaitForEvent(
                        hConsumer->errorExitEvent,
                        timeOutMilliSeconds );

	if(rfdStatus == RFD_STATUS_OK) {
		RFD_DPrint( TEXT("MFM: File Consumer thread signalled an error exit\n"));
		status = RFD_STATUS_ERROR_THREAD_EXIT;
	}
	else {
		status = RFD_STATUS_OK;
	}

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_RunFileConsumer(
    MFM_FILE_CONSUMER_HANDLE hConsumer,
    BOOL *isThreadExitRequestedPtr
    )
{
    RFD_STATUS status;
    RFD_STATUS statusLL;
    BOOL isThreadExitRequested;

    if( hConsumer == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if(hConsumer->isRunning == FALSE) {
		RFD_DPrint( TEXT("MFM File Consumer: Error, File Consumer is not in run state\n"));
        return RFD_STATUS_ERROR_INVALID_STATE;
    }

    if(!hConsumer->isAltSingleThreadMode) {
		RFD_DPrint( TEXT("MFM File Consumer: Error, MFM_RunFileConsumer() cannot be called in standard Multi-Threaded mode\n"));
        return RFD_STATUS_ERROR_INVALID_MODE;
    }

    status = MFM_FileConsumerRunLL(
                hConsumer,
                &statusLL,
                &isThreadExitRequested );

    *isThreadExitRequestedPtr = isThreadExitRequested;

    return status;
}