/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*   GLOBAL HEADER
*******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef OS_TASK_H_
#define OS_TASK_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <sched.h>

#if (defined LINUX)
// Required for syscall() in OS_GetTid()
#include <unistd.h>
#include <sys/syscall.h>
#endif

#include "standard.h"
#include "osal.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

// Definition for the general scheduling policy selected.
#if (defined LINUX) || (defined ANDROID)
// For user (nice) priorities
#define OS_SCHEDULING_POLICY SCHED_OTHER
#else
// For root (real-time) priorities.
#define OS_SCHEDULING_POLICY SCHED_RR
#endif

#if (defined LINUX)
/*
 * Setting nice values within [0,19] as defined for setpriority() function
 * call. Using only upper part of the range so all threads priorities are
 * lower than master thread priority (for PC version)
 */
// Bosch ID#3: Priority settings for bosch system
#define OS_TASK_LOWEST_PRIORITY       (19)
#define OS_TASK_LOWEST_LOW_PRIORITY   (0) // DSM, CM, RFD-Consumer, RFD-Processor,
#define OS_TASK_HIGHEST_LOW_PRIORITY  (0)
#define OS_TASK_LOWEST_MED_PRIORITY   (-5) // SRM, MODULE, DECODER, RFD-Collector
#define OS_TASK_HIGHEST_MED_PRIORITY  (-5)
#define OS_TASK_LOWEST_HIGH_PRIORITY  (-10) // Rx, RxTx
#define OS_TASK_HIGHEST_HIGH_PRIORITY (-10)
#define OS_TASK_HIGHEST_PRIORITY      (-12) // timer

#define OS_TASK_NUMBER_OF_PRIORITIES \
      ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
        OS_TASK_LOWEST_PRIORITY - OS_TASK_HIGHEST_PRIORITY + 1 : \
        OS_TASK_HIGHEST_PRIORITY - OS_TASK_LOWEST_PRIORITY + 1 )

#define OS_TASK_PRIORITY_GROUP_SIZE ( (OS_TASK_NUMBER_OF_PRIORITIES - 2) / 3)

#elif defined (ANDROID)
/*
 * Setting nice values within [-20,19] as defined for setpriority() function
 * call. Android allows parent thread to assign higher priority to the child,
 * so using the full range
 */
#define OS_TASK_LOWEST_PRIORITY       (19)
#define OS_TASK_LOWEST_LOW_PRIORITY   (10)
#define OS_TASK_HIGHEST_LOW_PRIORITY  (5)
#define OS_TASK_LOWEST_MED_PRIORITY   (0)
#define OS_TASK_HIGHEST_MED_PRIORITY  (5)
#define OS_TASK_LOWEST_HIGH_PRIORITY  (-10)
#define OS_TASK_HIGHEST_HIGH_PRIORITY (-15)
#define OS_TASK_HIGHEST_PRIORITY      (-20)

#define OS_TASK_NUMBER_OF_PRIORITIES \
      ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
        OS_TASK_LOWEST_PRIORITY - OS_TASK_HIGHEST_PRIORITY + 1 : \
        OS_TASK_HIGHEST_PRIORITY - OS_TASK_LOWEST_PRIORITY + 1 )

#define OS_TASK_PRIORITY_GROUP_SIZE ( (OS_TASK_NUMBER_OF_PRIORITIES - 2) / 3)

#elif defined (__QNX__) && defined (NUTRO)
/*
 * Setting priority values appropriate for Nutro board.
 */
#define OS_TASK_LOWEST_PRIORITY       (2)
#define OS_TASK_LOWEST_LOW_PRIORITY   (3)
#define OS_TASK_HIGHEST_LOW_PRIORITY  (6)
#define OS_TASK_LOWEST_MED_PRIORITY   (12)
#define OS_TASK_HIGHEST_MED_PRIORITY  (12)
#define OS_TASK_LOWEST_HIGH_PRIORITY  (14)
#define OS_TASK_HIGHEST_HIGH_PRIORITY (14)
#define OS_TASK_HIGHEST_PRIORITY      (16)

#define OS_TASK_NUMBER_OF_PRIORITIES \
      ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
        OS_TASK_LOWEST_PRIORITY - OS_TASK_HIGHEST_PRIORITY + 1 : \
        OS_TASK_HIGHEST_PRIORITY - OS_TASK_LOWEST_PRIORITY + 1 )

#define OS_TASK_PRIORITY_GROUP_SIZE ( (OS_TASK_NUMBER_OF_PRIORITIES - 2) / 3)

#else
/*
 * We can obtain the minimum and maximum priority available
 * for the system using POSIX APIs. However we only need 5
 * unique priorities to represent all the levels we need.
 */
#define OS_TASK_LOWEST_PRIORITY sched_get_priority_min(OS_SCHEDULING_POLICY)
#if (defined __QNX__)
#define OS_TASK_HIGHEST_PRIORITY (sched_get_priority_max(OS_SCHEDULING_POLICY)/2)
#elif (defined __INTEGRITY)
/*
 * Highest priority is determined by __PosixServerPriority constant of the VAS
 * or by __DEFAULT_POSIX_SERVER_PRIORITY macro of the BSP. It is 127 by default.
 */
#define OS_TASK_HIGHEST_PRIORITY sched_get_priority_max(OS_SCHEDULING_POLICY)
#else
#define OS_TASK_HIGHEST_PRIORITY (sched_get_priority_max(OS_SCHEDULING_POLICY)/4)
#endif

// The following are generic priority definitions
// which are based upon OS-specific priority numbers.
#define OS_TASK_NUMBER_OF_PRIORITIES \
      ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
        OS_TASK_LOWEST_PRIORITY - OS_TASK_HIGHEST_PRIORITY + 1 : \
        OS_TASK_HIGHEST_PRIORITY - OS_TASK_LOWEST_PRIORITY + 1 )

#define OS_TASK_PRIORITY_GROUP_SIZE ( (OS_TASK_NUMBER_OF_PRIORITIES - 2) / 3)

#define OS_TASK_LOWEST_LOW_PRIORITY \
       ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
         OS_TASK_LOWEST_PRIORITY - 1  : \
         OS_TASK_LOWEST_PRIORITY + 1 )

#define OS_TASK_HIGHEST_LOW_PRIORITY \
       ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
         OS_TASK_LOWEST_LOW_PRIORITY - OS_TASK_PRIORITY_GROUP_SIZE + 1 : \
         OS_TASK_LOWEST_LOW_PRIORITY + OS_TASK_PRIORITY_GROUP_SIZE - 1 )

#define OS_TASK_LOWEST_MED_PRIORITY \
       ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
         OS_TASK_HIGHEST_LOW_PRIORITY - 1  : \
         OS_TASK_HIGHEST_LOW_PRIORITY + 1 )

#define OS_TASK_HIGHEST_MED_PRIORITY \
       ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
         OS_TASK_LOWEST_MED_PRIORITY - OS_TASK_PRIORITY_GROUP_SIZE + 1 : \
         OS_TASK_LOWEST_MED_PRIORITY + OS_TASK_PRIORITY_GROUP_SIZE - 1 )

#define OS_TASK_LOWEST_HIGH_PRIORITY \
       ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
         OS_TASK_HIGHEST_MED_PRIORITY - 1  : \
         OS_TASK_HIGHEST_MED_PRIORITY + 1 )

#define OS_TASK_HIGHEST_HIGH_PRIORITY \
       ( OS_TASK_LOWEST_PRIORITY > OS_TASK_HIGHEST_PRIORITY ? \
         OS_TASK_LOWEST_HIGH_PRIORITY - OS_TASK_PRIORITY_GROUP_SIZE + 1  : \
         OS_TASK_LOWEST_HIGH_PRIORITY + OS_TASK_PRIORITY_GROUP_SIZE - 1 )
#endif

#define SMS_STACK_MIN      PTHREAD_STACK_MIN
#define SMS_STACK_GUARD    4096

// Macro definition for Thread ID getting function.
#if (defined ANDROID)
#define OS_GetTid() gettid()
#elif (defined LINUX)
#define OS_GetTid() ((pid_t)syscall(SYS_gettid))
#else
#define OS_GetTid() ((pid_t)pthread_self())
#endif



  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

OSAL_RETURN_CODE_ENUM OS_eTaskCreate (
    OSAL_OBJECT_HDL *phTaskObj,
    const char *pacName,
    OSAL_TASK_HANDLER tTaskHandler,
    void *pvTaskHandlerArgument,
    UN32 un32StackSize,
    OSAL_TASK_PRIORITY_ENUM ePriority,
    UN32 un32Options
        );

OSAL_RETURN_CODE_ENUM OS_eTaskDelete (
    OSAL_OBJECT_HDL hTaskObj
        );

OSAL_RETURN_CODE_ENUM OS_eTaskSuspend (
    OSAL_OBJECT_HDL hTaskObj
        );

OSAL_RETURN_CODE_ENUM OS_eTaskResume (
    OSAL_OBJECT_HDL hTaskObj
        );

OSAL_RETURN_CODE_ENUM OS_eTaskDelay (
    UN32 un32Milliseconds
        );

void OS_vTaskYield ( void );

OSAL_RETURN_CODE_ENUM OS_eTaskChangePriority (
    OSAL_OBJECT_HDL hTaskObj,
    OSAL_TASK_PRIORITY_ENUM eNewPriority
        );

OSAL_OBJECT_HDL OS_hTaskGetHandle ( void );

OSAL_RETURN_CODE_ENUM OS_eTaskGetInfo  (
    OSAL_OBJECT_HDL hTaskObj,
    OSAL_TASK_INFO_STRUCT *psTaskInfo
        );

BOOLEAN OS_bTaskGetId( OSAL_TASK_ID *ptTaskId );

OSAL_RETURN_CODE_ENUM OS_eTaskList ( void );

void OS_vThreadSpecificKeyDestructor( void *pvKeyValue );

  /*****************************/
 /** GLOBAL INLINE FUNCTIONS **/
/*****************************/

#endif	// OS_TASK_H_
