/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
*       This module will contain all the OSAL task APIs.
* The implementations in this module are nearly 100% POSIX compliant making
* it suitable for any POSIX.1b (POSIX 1003.1b-1993) compliant OS.
*
*******************************************************************************/

#define _GNU_SOURCE
#include <pthread.h>
#include <sched.h>
#ifndef __INTEGRITY
#include <sys/resource.h>
#endif

#include <errno.h>
#include <time.h>

// POSIX and standard includes
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifndef ANDROID
#include <signal.h>
#else
#include <unistd.h>
#include <asm/page.h>
#include "os_android.h"
#endif

#if (GPROF_ENABLED == 1)
#include <sys/time.h>
#endif

#if !(defined __QNX__) && !(defined __INTEGRITY)
#ifndef HAVE_PRCTL
#define HAVE_PRCTL
#endif

#ifdef HAVE_PRCTL
#include <sys/prctl.h>
#endif
#endif

#ifdef __INTEGRITY
#include <pthread.h>
#endif

#define OSAL_TRACE_ENABLE 0
#include "osal_trace.h"

#include "standard.h"
#include "osal.h"
#include "osal_general.h"       //for OSAL_pacGetReturnCodeName() definition
#include "os_version.h"
#include "os_general.h"
#include "os.h"
#include "os_task_.h"
#include "_os_task_.h"
#ifdef ANDROID
#include "os_android.h"
#endif

#if OSAL_DEBUG==1
#include "osal_debug.h"
#endif

// Bosch ID#1: Function to increase prio of DSM and CM during startup-phase
extern void fc_sxm_vSetPatchedPrio(char const *szThreadName, OSAL_OBJECT_HDL tThreadId, OSAL_TASK_PRIORITY_ENUM *piPrioVal);

/*******************************************************************************
*
*   OS_eTaskCreate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskCreate (
    OSAL_OBJECT_HDL *phTaskObj,
    const char *pacName,
    OSAL_TASK_HANDLER tTaskHandler,
    void *pvTaskHandlerArgument,
    UN32 un32StackSize,
    OSAL_TASK_PRIORITY_ENUM ePriority,
    UN32 un32Options
        )
{
    OS_TASK_OBJECT_STRUCT *psTaskObj;
    OSAL_OBJECT_HDL hParentObject = *phTaskObj;
   char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
#ifdef __QNX__
    struct sched_param sSchedParams;
#else
    OSAL_RETURN_CODE_ENUM eRetval;
#endif
    pthread_attr_t tThdAttr;
    int iRetval, iPriority;
    BOOLEAN bPriorityOk;

    TRACE_START();
    // Make sure thread stack size is at least PTHREAD_STACK_MIN
    if(un32StackSize < SMS_STACK_MIN)
    {
      // Make sure stack size is at least the minimum.
        un32StackSize = SMS_STACK_MIN;
    }
    un32StackSize += SMS_STACK_GUARD;

    // Always align size request with the native
    // underlying machine alignment.
    ALIGN_SIZE(un32StackSize);

    // Get the Thread Priority mapped from OSAL priority
    bPriorityOk = OS_bMapPriority( ePriority, &iPriority );
    if(bPriorityOk == FALSE)
    {
        // Invalid priority
        TRACE_END();
        return OSAL_ERROR_INVALID_PRIORITY;
    }

    /*
     * Initialize a thread-attribute object
     */

    // The pthread_attr_init() function initializes the thread attributes
    // in the thread attribute object tThdAttr to their default values.
    iRetval = pthread_attr_init( &tThdAttr );
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_attr_init()");
        TRACE_END();
        return OSAL_ERROR;
    }

    // from this point on, if we exit this function, we need to
    // release the tThdAttr object with pthread_attr_destroy().

#ifdef __QNX__
    // The pthread_attr_getschedparam() function gets the thread scheduling
    // parameters attribute from the thread attribute object tThdAttr and
    //returns it in sSchedParams
    iRetval = pthread_attr_getschedparam(&tThdAttr, &sSchedParams);
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_getschedparam()");
        TRACE_END();
        return OSAL_ERROR;
    }

    // At this point we have now an initialized thread attribute structure
    // and a scheduling parameters structure which we can use
    // to initialized our desired parameters for this (and every) thread.

    // Explicitly configure the priority as determined from the call above.
    sSchedParams.sched_priority = iPriority;

    // The pthread_attr_setinheritsched() function sets the thread inherit
    // scheduling attribute in the attribute object tThdAttr to inheritsched.
    // This option specifies using the scheduling policy specified in tThdAttr
    // for the thread instead of inheriting the parameters.
    iRetval = pthread_attr_setinheritsched(&tThdAttr, PTHREAD_EXPLICIT_SCHED);
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_setinheritsched()");
        TRACE_END();
        return OSAL_ERROR;
    }

    // The pthread_attr_setschedpolicy() function sets the thread scheduling
    // policy attribute in the thread attribute object attr to policy. The
    // policy attribute is used only if you've set the thread
    // inherit-scheduling attribute to PTHREAD_EXPLICIT_SCHED by
    // calling pthread_attr_setinheritsched().
    iRetval = pthread_attr_setschedpolicy(&tThdAttr, OS_SCHEDULING_POLICY);
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_setschedpolicy()");
        TRACE_END();
        return OSAL_ERROR;
    }

    // The pthread_attr_setschedparam() function sets the thread scheduling
    // parameters attribute in the thread attribute object attr to param.
    // The thread scheduling parameters are used only if you've set the thread
    // inherit scheduling attribute to PTHREAD_EXPLICIT_SCHED by calling
    // pthread_attr_setinheritsched(). By default, a thread inherits its
    // parent's priority.
    iRetval = pthread_attr_setschedparam(&tThdAttr, &sSchedParams);
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_setschedparam()");
        TRACE_END();
        return OSAL_ERROR;
    }
#endif
    // The pthread_attr_setdetachstate() function sets the thread detach
    // state attribute in the thread attribute object tThdAttr to detachstate.
    // This means that when the thread terminates any resources it used can
    // immediately be reclaimed by the system.
    iRetval = pthread_attr_setdetachstate(&tThdAttr, PTHREAD_CREATE_DETACHED);
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_setdetachstate()");
        TRACE_END();
        return OSAL_ERROR;
    }

#ifdef __INTEGRITY
    iRetval = pthread_attr_setthreadname(&tThdAttr, pacName);
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_setthreadname()");
        TRACE_END();
        return OSAL_ERROR;
    }
#endif

     // The pthread_attr_setstacksize() function sets the thread stack size
     // attribute in the thread attribute object tThdAttr to stacksize.
     // This is the 'initial' stack size for the thread.
     // Since the stack is defined above as dynamic the stack will be allowed
     // to grow if needed.
     // For Linux stack size is set to 2 pages

     iRetval = pthread_attr_setstacksize(&tThdAttr, un32StackSize);
     if(iRetval != EOK)
     {
         // Error!
         pthread_attr_destroy(&tThdAttr);
         print_err(iRetval, "Error! pthread_attr_setstacksize()");
         TRACE_END();
         return OSAL_ERROR;
     }

    // The pthread_attr_setguardsize() function sets the guard area
    // immediately after the stack that the thread can't write to.
    // If it does (meaning that the stack was about to overflow), the thread will get hit with
    // a SIGSEGV.
    // Also, note that the guard page doesn't take up any physical memory...
    // it's done as a virtual address (MMU) "trick".
    iRetval = pthread_attr_setguardsize( &tThdAttr, SMS_STACK_GUARD );
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_setguardsize()");
        TRACE_END();
        return OSAL_ERROR;
    }

    // Now we are ready to create the os-layer thread object which
    // OSAL will use to manage the new thread.


    // Construct an appropriate name for memory to be allocated
    snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL, "%-*s",
            OSAL_MAX_OBJECT_NAME_LENGTH,
            OSAL_NAME_PREFIX"ThreadCtrlBlk");

    // Before we create a task, we need to create the memory needed
    // for this task's object..
    psTaskObj = (OS_TASK_OBJECT_STRUCT *)
                OSALC_pvMemoryAllocate(acName,
                             sizeof(OS_TASK_OBJECT_STRUCT), TRUE);
    if(psTaskObj == NULL)
    {
        pthread_attr_destroy(&tThdAttr);
        TRACE_END();
        return OSAL_ERROR_OUT_OF_MEMORY;
    }

    psTaskObj->pacName = ( const char * )
                OSALC_pvMemoryAllocate( OSAL_NAME_PREFIX"TaskName",
                             strlen( pacName ) + 1, TRUE );
    if(psTaskObj->pacName == NULL)
    {
        // Free memory
        pthread_attr_destroy(&tThdAttr);
        OSALC_vMemoryFree(psTaskObj);
        TRACE_END();
        return OSAL_ERROR_OUT_OF_MEMORY;
    }

    // Populate task structure with inputs provided by the caller
    strcpy((char *)psTaskObj->pacName, pacName);
    psTaskObj->tTaskHandler = tTaskHandler;
    psTaskObj->pvTaskHandlerArgument = pvTaskHandlerArgument;
    psTaskObj->un32StackSize = un32StackSize;
    psTaskObj->hParentObject = hParentObject;
#ifndef __QNX__
    psTaskObj->iPriority = iPriority;
    psTaskObj->tThreadPid = 0;
    psTaskObj->bPrioritySet = FALSE;
#endif

    do
    {
#ifndef __QNX__
       /*
        * Initializing mutexes to synchronize setting thread priority
        *
        * First, we need to create the thread and acquire its TID (thread ID).
        * Using tThreadPidMutex we are locking creator thread until new thread
        * would get ID and save it into info structure. If this is not done, wrong
        * PID (TID) is used while calling priority setting functions
        *
        * After that, we are locking newly created thread (by tThreadPriorityMutex
        * until creator would set correct priority. Otherwise race conditions occurs
        * because new thread runs with the same priority as parent for some amount
        * of time (Queue Put unit test fails)
        *
        */

       // Initializing PID mutex
       iRetval = pthread_mutex_init( &(psTaskObj->tThreadPidMutex), NULL);
       if(iRetval != EOK)
       {
          // Error!
          print_err(iRetval, "Error! tThreadPidMutex mutex init()");
          break;
       }

       //Initializing PID condition valiable
       iRetval = pthread_cond_init( &(psTaskObj->tThreadPidCond), NULL);
       if(iRetval != EOK)
       {
          // Error!
          print_err(iRetval, "Error! tThreadPidCond cond_var init()");
          break;
       }

       // Initializing Priority mutex
       iRetval = pthread_mutex_init( &(psTaskObj->tThreadPriorityMutex), NULL);
       if(iRetval != EOK)
       {
          // Error!
          print_err(iRetval, "Error! tThreadPriorityMutex mutex init()");
          break;
       }

       //Initializing Priority condition valiable
       iRetval = pthread_cond_init( &(psTaskObj->tThreadPriorityCond), NULL);
       if(iRetval != EOK)
       {
          // Error!
          print_err(iRetval, "Error! tThreadPriorityCond cond_var init()");
          break;
       }
#endif

       // The pthread_create() function creates a new thread, with the attributes
       // specified in the thread attribute object tThdAttr. The created thread
       // inherits the signal mask of the parent thread, and its set of pending
       // signals is empty.
#if (GPROF_ENABLED == 1)
       iRetval = pthread_create_wrapper(&psTaskObj->tThreadId, &tThdAttr, OS_vTaskShell, psTaskObj);
#else
       iRetval = pthread_create(&psTaskObj->tThreadId, &tThdAttr, OS_vTaskShell, psTaskObj);
#endif
       if( iRetval != EOK)
       {
          // Error! Thread cannot be created
          print_err(iRetval, "Error! pthread_create()");
          break;
       }

       // The returned handle will now be the the OS-handle
       *phTaskObj = (OSAL_OBJECT_HDL)psTaskObj;

#ifndef __QNX__
       // Letting thread function to acquire PID of just created process
       iRetval = pthread_mutex_lock( &psTaskObj->tThreadPidMutex );
       if( iRetval != EOK)
       {
          print_err(iRetval, "Error! PID mutex lock - creator task");
          break;
       }

       while( psTaskObj->tThreadPid == 0 )
       {
          iRetval = pthread_cond_wait( &psTaskObj->tThreadPidCond, &psTaskObj->tThreadPidMutex );
          if( iRetval != EOK )
          {
              print_err(iRetval, "Error: PID condition wait");
              pthread_mutex_unlock( &psTaskObj->tThreadPidMutex );
              break;
          }
       }
       // Bosch ID#1: call the function to increase prio of DSM and CM during startup-phase
       fc_sxm_vSetPatchedPrio(pacName, *phTaskObj, &ePriority);

       iRetval = pthread_mutex_unlock( &psTaskObj->tThreadPidMutex );
       if( iRetval != EOK)
       {
          print_err(iRetval, "Error! PID mutex unlock - creator task");
          break;
       }

       // Locking Priority mutex to prevent created thread from further execution until
       // priority is set
       iRetval = pthread_mutex_lock( &psTaskObj->tThreadPriorityMutex );
       if( iRetval != EOK)
       {
          print_err(iRetval, "Error! PRIORITY mutex lock - creator task");
          break;
       }

       // Setting process priority
       eRetval = OS_eTaskChangePriority((OSAL_OBJECT_HDL)psTaskObj, ePriority);
       if( eRetval != OSAL_SUCCESS)
       {
          printf("Error! OS_eTaskChangePriority() : %s\n", OSAL_pacGetReturnCodeName(eRetval));
       }
       else
       {
          psTaskObj->bPrioritySet = TRUE;
       }

       //Signaling priority assignment
       iRetval = pthread_cond_signal( &psTaskObj->tThreadPriorityCond );
       if( iRetval != EOK)
       {
          print_err(iRetval, "Error! PRIORITY condition signal - creator task");
          break;
       }

       // Unlocking Priority mutex, letting new thread to proceed
       iRetval = pthread_mutex_unlock( &psTaskObj->tThreadPriorityMutex );
       if( iRetval != EOK)
       {
          print_err(iRetval, "Error! PRIORITY mutex unlock - creator task");
          break;
       }
#endif

       // Successful exit
       // we don't need the thread attr object anymore
       pthread_attr_destroy(&tThdAttr);

       TRACE_END();
       return OSAL_SUCCESS;
    } while(FALSE);

    // "break" from a block above means error condition.
    // Need to reclaim all allocated memory and return error code
    pthread_attr_destroy(&tThdAttr);
    OSALC_vMemoryFree((void*)psTaskObj->pacName);
    OSALC_vMemoryFree(psTaskObj);
    TRACE_END();
    return OSAL_ERROR;
}

/*******************************************************************************
*
*   OS_eTaskDelete
*
*   This function is always called from the context of the task which is
*   being deleted. As far as the OS is concerned however, the task is still
*   running just fine. There is nothing to do in this function.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskDelete (
    OSAL_OBJECT_HDL hTaskObj
        )
{
    OS_TASK_OBJECT_STRUCT *psTaskObj
        = (OS_TASK_OBJECT_STRUCT *)hTaskObj;

    TRACE_START();
    if(psTaskObj != NULL)
    {
        // Invalidate the parent object from the os-object
        // representing this thread.
        psTaskObj->hParentObject = OSAL_INVALID_OBJECT_HDL;
    }

    // pthreads will clean-up (free the memory) via
    // OS_vThreadSpecificKeyDestructor() once the function
    // (OS_vTaskShell) returns back to the OS normally.
    TRACE_END();
    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OS_eTaskSuspend
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskSuspend (
    OSAL_OBJECT_HDL hTaskObj
        )
{
    OS_TASK_OBJECT_STRUCT *psTaskObj
        = (OS_TASK_OBJECT_STRUCT *)hTaskObj;
    int iRetval;

    TRACE_START();
    // Check that provided task object is not NULL
    if(hTaskObj == OSAL_INVALID_OBJECT_HDL)
    {
        TRACE_END();
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // The pthread_kill() function sends a signal to the thread tThreadId.
    // The target thread and calling thread must be in the same process.
    // If the signal to be sent is zero, error checking is performed but
    // no signal is sent.
    iRetval = pthread_kill(psTaskObj->tThreadId, OS_TASK_SUSPEND_SIGNAL);
    if(iRetval != EOK)
    {
        // Error! Thread cannot be suspended
        print_err(iRetval, "Error! pthread_kill()");
        TRACE_END();
        return OSAL_ERROR;
    }

    TRACE_END();
    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OS_eTaskResume
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskResume (
    OSAL_OBJECT_HDL hTaskObj
        )
{
    OS_TASK_OBJECT_STRUCT *psTaskObj
        = (OS_TASK_OBJECT_STRUCT *)hTaskObj;
    int iRetval;

    TRACE_START();
    // Check that provided task object is not NULL
    if(hTaskObj == OSAL_INVALID_OBJECT_HDL)
    {
        TRACE_END();
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // The pthread_kill() function sends a signal to the thread tThreadId.
    // The target thread and calling thread must be in the same process.
    // If the signal to be sent is zero, error checking is performed but
    // no signal is sent.
    iRetval = pthread_kill(psTaskObj->tThreadId, OS_TASK_RESUME_SIGNAL);
    if(iRetval != EOK)
    {
        // Error! Thread cannot be resumed
        print_err(iRetval, "Error! pthread_kill()");
        TRACE_END();
        return OSAL_ERROR;
    }

    TRACE_END();
    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OS_eTaskDelay
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskDelay (
    UN32 un32Milliseconds
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    struct timespec time_out = {0,0};
#ifndef __QNX__
    struct timespec rem_time = {0,0};
#endif
    int iRetval;

    TRACE_START();

    // nanosleep() function does not accept numbers greater than 999999999
    if (un32Milliseconds > 999999999)
    {
       un32Milliseconds = 999999999;
    }

    time_out.tv_sec = un32Milliseconds / 1000; // seconds
    time_out.tv_nsec = (un32Milliseconds % 1000) * (1000000); // nsec

    // Sleep for a spell...
#ifdef __QNX__
    // Use clock_nanosleep for a High resolution sleep with specifiable clock
    iRetval = clock_nanosleep(CLOCK_MONOTONIC, 0, &time_out, NULL);
#else
    iRetval = nanosleep( &time_out, &rem_time );
#endif
    if(iRetval != EOK)
    {
#ifdef __QNX__
       print_err(iRetval, "Error! nanosleep()");
#else
        printf("Error! nanosleep():%s \n\tun32Milliseconds = %d\n\ttime_out.tv_sec = %d\n\ttime_out.tv_nsec = %d\n\trem_time.tv_sec = %d\n\trem_time.tv_nsec = %d\n",
                 strerror(iRetval),
                 un32Milliseconds,
                 time_out.tv_sec,
                 time_out.tv_nsec,
                 rem_time.tv_sec,
                 rem_time.tv_nsec
                 );
#endif
        eReturnCode = OSAL_ERROR;
    }

    TRACE_END();
    return eReturnCode;
}

/*******************************************************************************
*
*   OS_vTaskYield
*
*******************************************************************************/
void OS_vTaskYield ( void )
{
    TRACE_START();
    // Yield to other ready threads at the same priority.
    // The sched_yield() function checks to see if other threads, at the
    // same priority as that of the calling thread, are READY to run. If so,
    // the calling thread yields to them and places itself at the end of the
    // READY thread queue. The sched_yield() function never yields to a lower
    // priority thread. This function always succeeds and returns zero.
     sched_yield();
    TRACE_END();
    return;
}

/*******************************************************************************
*
*   OS_eTaskChangePriority
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskChangePriority (
    OSAL_OBJECT_HDL hTaskObj,
    OSAL_TASK_PRIORITY_ENUM eNewPriority
        )
{
    int iRetval, iPriority;
#if (defined __QNX__) || (defined __INTEGRITY)
    int iSchedule;
    struct sched_param sSchedParams;
#endif
    OS_TASK_OBJECT_STRUCT *psTaskObj = (OS_TASK_OBJECT_STRUCT *)hTaskObj;
    BOOLEAN bPriorityOk;

    TRACE_START();
    // Check that provided task object is not NULL
    if(hTaskObj == OSAL_INVALID_OBJECT_HDL)
    {
        TRACE_END();
        return OSAL_ERROR_INVALID_HANDLE;
    }

    bPriorityOk = OS_bMapPriority(eNewPriority, &iPriority);
    if(bPriorityOk == FALSE)
    {
        TRACE_END();
        return OSAL_ERROR_INVALID_PRIORITY;
    }

#if !(defined __QNX__) && !(defined __INTEGRITY)
    /*
     * This section is for Linux only utilizing setpriority/getpriority functions
     */

    psTaskObj->iPriority = iPriority;

    if(psTaskObj->tThreadPid == 0)
    {
        TRACE_END();
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Clearing errno
    errno = EOK;

    iRetval = setpriority(PRIO_PROCESS, psTaskObj->tThreadPid, psTaskObj->iPriority);
    if(iRetval != EOK)
    {
        switch (errno)
        {
            case EINVAL:
            {
                printf("Error! setpriority(): invalid parameter\n");
            }
            break;

            case ESRCH:
            {
                printf("Error! setpriority(): thread %d not found\n", psTaskObj->tThreadPid);
            }
            break;

            case EPERM:
            {
                printf("Error! setpriority(): not owner (%d)\n", errno);
            }
            break;

            case EACCES:
            {
                printf("Error! setpriority(): insufficient permissions (%d)\n", errno);

                // this doesn't have to be a fatal error, so over-ride iRetval here
                // and allow things to continue.
                iRetval = EOK;
            }
            break;

            default:
            {
                printf("Error! setpriority(): error %d\n", errno);
            }
            break;
        }

        if ( iRetval != EOK )
        {
            TRACE_END();
            return OSAL_ERROR;
        }
    }
#else
    /*
     * This section is non-Linux implementatoin utilizing pthread sheduling
     * parameters control functions
     */

    // Get a thread's scheduling parameters
    iRetval = pthread_getschedparam(psTaskObj->tThreadId, &iSchedule, &sSchedParams);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_getschedparam()");
        TRACE_END();
        return OSAL_ERROR;
    }

    // Modify the thread's priority to the new priority
    sSchedParams.sched_priority = iPriority;

    // Set thread scheduling parameters
    iRetval = pthread_setschedparam(
        psTaskObj->tThreadId, iSchedule,&sSchedParams);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_setschedparam()");
        TRACE_END();
        return OSAL_ERROR;
    }
#endif

    TRACE_END();
    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OS_hTaskGetHandle
*
*******************************************************************************/
OSAL_OBJECT_HDL OS_hTaskGetHandle ( void )
{
    OSAL_OBJECT_HDL hObjectHdl = OSAL_INVALID_OBJECT_HDL;
    OS_TASK_OBJECT_STRUCT *psTaskObj;

    TRACE_START();
    // Retrieve this thread's specific key value which contains the
    // os-object representing this thread.
    psTaskObj = (OS_TASK_OBJECT_STRUCT *)
        pthread_getspecific(GsCtrl.tThreadSpecificObjectKey);
    if(psTaskObj != NULL)
    {
        // Extract parent object from the os-object representing this thread.
        hObjectHdl = psTaskObj->hParentObject;
    }

    TRACE_END();
    return hObjectHdl;
}

/*******************************************************************************
*
*   OS_eTaskGetInfo
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskGetInfo  (
    OSAL_OBJECT_HDL hTaskObj,
    OSAL_TASK_INFO_STRUCT *psTaskInfo
        )
{
#if !(defined __QNX__) && !(defined __INTEGRITY)
    OS_TASK_OBJECT_STRUCT *psTaskObj =
            (OS_TASK_OBJECT_STRUCT *)hTaskObj;
#endif

    pthread_attr_t tThdAttr;
    int iRetval;
    size_t tStackSize = 0;
    void *pvStackBase = 0;
    UN8 un8StackPercentUsed = 0;
    size_t tStackBytesUsed = 0;
    size_t tStackBytesFree = 0;

   TRACE_START();

   // Checking input parameters for validity
   if( hTaskObj == OSAL_INVALID_OBJECT_HDL ||
            psTaskInfo == (OSAL_TASK_INFO_STRUCT *)NULL )
   {
      // Error!
      printf("Error: %s: Invalid input", __FUNCTION__);
      TRACE_END();
      return OSAL_ERROR_INVALID_INPUT;
   }
   //  All we need to get at this level is the stack utilization.
   //  Everything else is known by OSAL.

    // Getting attributes from task ID
#if (defined __QNX__) || (defined __INTEGRITY)
    iRetval = pthread_attr_init(&tThdAttr);
#else
    iRetval = pthread_getattr_np( psTaskObj->tThreadId, &tThdAttr );
#endif
    if(iRetval != EOK)
    {
        // Error!
#ifdef __QNX__
        print_err(iRetval, "Error! pthread_attr_init()");
#else
        print_err(iRetval, "Error! pthread_getattr_np()");
#endif
        TRACE_END();
        return OSAL_ERROR;
    }

    // Getting stack base address and size
    iRetval = pthread_attr_getstack( &tThdAttr, &pvStackBase, &tStackSize );
    if(iRetval != EOK)
    {
        // Error!
        pthread_attr_destroy(&tThdAttr);
        print_err(iRetval, "Error! pthread_attr_getstack()");
        TRACE_END();
        return OSAL_ERROR;
    }

    // Check that we have a valid stack size
    if(tStackSize != 0)
    {
        // "Bytes used" are calculated taking address of the top stack variable
        // "Bytes free" is delta between "used" and "total"
        if(tStackSize >= tStackBytesUsed) // This only works when...
        {
            tStackBytesFree = tStackSize - tStackBytesUsed;
        }
        // "Percentage" is easy now...
        un8StackPercentUsed = tStackBytesUsed * 100UL / tStackSize;
    }


    // Copy in computed values
    psTaskInfo->un32StackSize =  tStackSize;
    psTaskInfo->un8StackPercentUsed = un8StackPercentUsed;
    psTaskInfo->un32StackBytesUsed = tStackBytesUsed;
    psTaskInfo->un32StackBytesFree = tStackBytesFree;

    pthread_attr_destroy(&tThdAttr);

    TRACE_END();
    return OSAL_SUCCESS;
}


/*******************************************************************************
*
*   OS_bTaskGetId
*
*******************************************************************************/
BOOLEAN OS_bTaskGetId( OSAL_TASK_ID *ptTaskId )
{
    pthread_t tThreadId;

    TRACE_START();
    // Determine this thread's id.
    tThreadId = pthread_self();

    if(ptTaskId !=  NULL)
    {
        *ptTaskId = (OSAL_TASK_ID)tThreadId;
    }

    TRACE_END();
    return TRUE;
}

/*******************************************************************************
*
*   OS_eTaskList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eTaskList ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    TRACE_START();
    printf("\nOS: My priority levels are:\n");
    printf("\t%s = %d and \n\t%s = %d\n",
          MACRO_TO_STRING(OS_TASK_NUMBER_OF_PRIORITIES),
          OS_TASK_NUMBER_OF_PRIORITIES,
          MACRO_TO_STRING(OS_TASK_PRIORITY_GROUP_SIZE),
          OS_TASK_PRIORITY_GROUP_SIZE);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_LOWEST_PRIORITY),
          OS_TASK_LOWEST_PRIORITY);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_LOWEST_LOW_PRIORITY),
         OS_TASK_LOWEST_LOW_PRIORITY);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_HIGHEST_LOW_PRIORITY),
         OS_TASK_HIGHEST_LOW_PRIORITY);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_LOWEST_MED_PRIORITY),
         OS_TASK_LOWEST_MED_PRIORITY);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_HIGHEST_MED_PRIORITY),
         OS_TASK_HIGHEST_MED_PRIORITY);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_LOWEST_HIGH_PRIORITY),
         OS_TASK_LOWEST_HIGH_PRIORITY);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_HIGHEST_HIGH_PRIORITY),
         OS_TASK_HIGHEST_HIGH_PRIORITY);
    printf("\t%s = %d\n", MACRO_TO_STRING(OS_TASK_HIGHEST_PRIORITY),
        OS_TASK_HIGHEST_PRIORITY);

    TRACE_END();
    return eReturnCode;
}

/*******************************************************************************
*
*   OS_vThreadSpecificKeyDestructor
*
*******************************************************************************/
void OS_vThreadSpecificKeyDestructor(void *pvKeyValue)
{
    OS_TASK_OBJECT_STRUCT *psTask = (OS_TASK_OBJECT_STRUCT *)pvKeyValue;

    TRACE_START();
    // This function runs when a thread finally exits. It is used
    // as a destructor for any thread specific key data. In our
    // case this is the os-object which defines the thread.
    // It is this destructor which allows us to free the os-layer
    // thread object once the thread has finally exited.

    // Checking input parameters for validity
    if( psTask != (OS_TASK_OBJECT_STRUCT *)NULL )
    {
        if( psTask->pacName != NULL )
        {
            OSALC_vMemoryFree((void*)psTask->pacName);
        }
        // Free os-layer thread object memory
        OSALC_vMemoryFree(psTask);
    }

    TRACE_END();
    return;
}

/*****************************************************************************
*
*       OS_vThreadSetName
*
*****************************************************************************/
static int OS_vThreadSetName(const char *pcThreadName)
{
    int iReturn = EINVAL;
    TRACE_START();

    if( pcThreadName == (const char*)NULL )
    {
       pcThreadName = "Unnamed OSAL Thread";
    }

#if defined(HAVE_PTHREAD_SETNAME_NP)
    iReturn = pthread_setname_np(pthread_self(), pcThreadName, 0);
#elif defined(__QNX__)
    iReturn = pthread_setname_np(pthread_self(), pcThreadName);
#elif defined(HAVE_PRCTL)
    {
        int saved_errno = errno;

        iReturn = prctl(PR_SET_NAME, (unsigned long) pcThreadName, 0, 0, 0) ?
            errno : EOK;
        errno = saved_errno;
    }
#else
    iReturn = EOK;
#endif
    TRACE_END();
    return iReturn;
}

/*****************************************************************************
*
*       OS_vTaskShell
*
*****************************************************************************/
static void *OS_vTaskShell( void *pvArg )
{
    OS_TASK_OBJECT_STRUCT *psTask = (OS_TASK_OBJECT_STRUCT *)pvArg;

    TRACE_START();
    // Verify input is valid
    if(psTask != NULL)
    {
        N32 n32ReturnCode;
        int iRetval;

        // Set this thread's specific data using the object key. This key
        // will contain the thread object pointer. This way we can retrieve
        // this when we need it.
        iRetval = pthread_setspecific(
            GsCtrl.tThreadSpecificObjectKey, (const void *)psTask);
        if(iRetval != EOK)
        {
            // Error!
            print_err(iRetval, "Error! pthread_setspecific()");
            TRACE_END();
            return NULL;
        }

#ifndef __QNX__
        do
        {
        // Locking PID mutex
        iRetval = pthread_mutex_lock( &psTask->tThreadPidMutex );

           if( iRetval != EOK )
           {
              print_err(iRetval, "Error: PID mutex lock");
              break;
           }
        // Get thread PID for setting priority
        psTask->tThreadPid = OS_GetTid();

           if(psTask->tThreadPid != 0)
           {
        //Signalling PID assignment
              iRetval = pthread_cond_signal( &psTask->tThreadPidCond );
              if( iRetval != EOK )
              {
                 print_err(iRetval, "Error: PID condition signal");
                 pthread_mutex_unlock( &psTask->tThreadPidMutex );
                 break;
              }
           }
        // Unlocking PID mutex
        iRetval = pthread_mutex_unlock( &psTask->tThreadPidMutex );
           if( iRetval != EOK )
           {
              print_err(iRetval, "Error: PID mutex unlock");
              break;
           }

        // Waiting until parent thread would set priority
           iRetval = pthread_mutex_lock( &psTask->tThreadPriorityMutex );
           if( iRetval != EOK )
           {
              print_err(iRetval, "Error: PRIORITY mutex lock");
              break;
           }

        // If priority is not set yet - waiting for condition
           while( psTask->bPrioritySet == FALSE)
        {
              iRetval = pthread_cond_wait( &psTask->tThreadPriorityCond, &psTask->tThreadPriorityMutex );
              if( iRetval != EOK )
              {
                 print_err(iRetval, "Error: PRIORITY condition wait");
                 pthread_mutex_unlock( &psTask->tThreadPriorityMutex );
                 break;
              }
        }

        //Releasing mutex. We are done now
           iRetval = pthread_mutex_unlock( &psTask->tThreadPriorityMutex );
           if( iRetval != EOK )
           {
              print_err(iRetval, "Error: PRIORITY mutex unlock");
              break;
           }

        } while (FALSE);
        if (iRetval != EOK)
        {
           return NULL;
        }
#endif

        // Set this thread's name
        //iRetval = pthread_setname_np(0, psTask->pacName);
        iRetval = OS_vThreadSetName(psTask->pacName);
        if(iRetval != EOK)
        {
            // Error!
            print_err(iRetval, "Error! pthread_setname_np()");
            TRACE_END();
            return NULL;
        }

#if !(defined __QNX__) && !(defined __INTEGRITY)
        printf("### Task %s has id=%x, TID=%d, priority=%d\n",
                             psTask->pacName,
                             psTask->hParentObject,
                             psTask->tThreadPid,
                             getpriority(PRIO_PROCESS, psTask->tThreadPid));
#endif

        // Run their task, when it returns the task is through
        // and the task must clean up and delete itself
        n32ReturnCode =
            psTask->tTaskHandler(psTask->pvTaskHandlerArgument);
#ifdef ANDROID
        // Detach thread from JVM if it is attached.
        OS_eVmDetachCurrentThread();
#endif
        if(n32ReturnCode != OSAL_TASK_REPORT_NO_ERROR)
        {
            // Handle any required action here.
            printf("OS Error: Task returned non-zero result (%d).\n",
                   n32ReturnCode);
        }
    }

    TRACE_END();
    return NULL;
}

/*******************************************************************************
*
*   OS_bMapPriority
*
*******************************************************************************/
static BOOLEAN OS_bMapPriority(
    OSAL_TASK_PRIORITY_ENUM ePriority,
    int *piPriority
        )
{
    int iPriority;

    TRACE_START();
    // Based on input priority map this one to an os-dependent
    // priority level. Since this system uses POSIX pthreads and
    // pthreads allow multiple threads at the same priority there
    // is really only the need for 5 distinct priority levels;
    // LOWEST, LOW, MEDIUM, HIGH and HIGHEST.
    // Therefore we will just use the lowest number within each group.
    switch(ePriority)
    {
        case OSAL_TASK_PRIORITY_LOWEST:
            iPriority = OS_TASK_LOWEST_PRIORITY;
        break;

        case OSAL_TASK_PRIORITY_LOW:
            iPriority = OS_TASK_LOWEST_LOW_PRIORITY;
        break;

        case OSAL_TASK_PRIORITY_MEDIUM:
            iPriority = OS_TASK_LOWEST_MED_PRIORITY;
        break;

        case OSAL_TASK_PRIORITY_HIGH:
            iPriority = OS_TASK_LOWEST_HIGH_PRIORITY;
        break;

        case OSAL_TASK_PRIORITY_HIGHEST:
            iPriority = OS_TASK_HIGHEST_PRIORITY;
        break;

        case OSAL_TASK_PRIORITY_INVALID:
        default:
            // Invalid priority input
            TRACE_END();
            return FALSE;
    }

    *piPriority = iPriority;
    TRACE_END();
    return TRUE;
}

#if (GPROF_ENABLED == 1)
/*******************************************************************************
*
*   wrapper_routine
*
*******************************************************************************/
/* The wrapper function in charge for setting the itimer value */
static void *wrapper_routine(void *data)
{
    /* Put user data in thread-local variables */
    void * (*start_routine)(void *) = ((wrapper_t*)data)->start_routine;
    void *arg = ((wrapper_t*)data)->arg;

    /* Set the profile timer value */
    setitimer(ITIMER_PROF, &((wrapper_t*)data)->itimer, NULL);

    /* Tell the calling thread that we don't need its data anymore */
    pthread_mutex_lock(&((wrapper_t*)data)->lock);
    pthread_cond_signal(&((wrapper_t*)data)->wait);
    pthread_mutex_unlock(&((wrapper_t*)data)->lock);

    /* Call the real function */
    return start_routine(arg);
}

/*******************************************************************************
*
*   pthread_create_wrapper
*
*******************************************************************************/
/* Our wrapper function for the real pthread_create() */
static int pthread_create_wrapper(
    pthread_t *__restrict thread,
    __const pthread_attr_t *__restrict attr,
    void * (*start_routine)(void *),
    void *__restrict arg
        )
{
    wrapper_t wrapper_data;
    int i_return;

    /* Initialize the wrapper structure */
    wrapper_data.start_routine = start_routine;
    wrapper_data.arg = arg;
    getitimer(ITIMER_PROF, &wrapper_data.itimer);
    pthread_cond_init(&wrapper_data.wait, NULL);
    pthread_mutex_init(&wrapper_data.lock, NULL);
    pthread_mutex_lock(&wrapper_data.lock);

    /* The real pthread_create call */
    i_return = pthread_create(thread,
                              attr,
                              &wrapper_routine,
                              &wrapper_data);

    /* If the thread was successfully spawned, wait for the data
     * to be released */
    if(i_return == 0)
    {
        pthread_cond_wait(&wrapper_data.wait, &wrapper_data.lock);
    }

    pthread_mutex_unlock(&wrapper_data.lock);
    pthread_mutex_destroy(&wrapper_data.lock);
    pthread_cond_destroy(&wrapper_data.wait);

    return i_return;
}
#endif // GPROF_ENABLED == 1
