/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This module will contain all the OS Implementations of
 * ANSI-C File System APIs.
 * The implementations in this module are 100% POSIX complient making
 * it suitable for any POSIX.1b (POSIX 1003.1b-1993) compliant OS.
 *
 *******************************************************************************/

// POSIX and standard includes
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include <errno.h>

// Bosch ID#13: needed for calls inside OS_Rename below
#include <unistd.h>
#include <sys/stat.h> 
#include <fcntl.h>

#ifdef ANDROID
#define LOG_TAG      "SIRIUS"
#include <utils/Log.h>
#endif

#if OSAL_REMOTE_CONSOLE == 1
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#endif /* OSAL_REMOTE_CONSOLE == 1 */

#define OSAL_TRACE_ENABLE 0
#include "osal_trace.h"

#include "standard.h"
#include "os_version.h"
#include "os_stdio.h"
#include "_os_stdio.h"

// NOTE: You must not include osal.h or any other header which includes
// osal.h as these functions must map directly to the compiler stdio
// functions and NOT the macro replaced functions from osal.h
#ifdef OSAL_H_
#error Error! You cannot include any header which includes osal.h!!!
#endif


/*******************************************************************************
 *
 *   OS_FOpen
 *
 *******************************************************************************/
FILE *OS_FOpen(const char *pFileName, const char *pMode)
{
    FILE *pFileDspt;

    TRACE_START();
    pFileDspt = fopen(pFileName, pMode);

    TRACE_END();
    return pFileDspt;
}

/*******************************************************************************
 *
 *   OS_FClose
 *
 *******************************************************************************/
int OS_FClose(FILE *pFile)
{
    int iRetval = EOK;

    TRACE_START();

#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        iRetval = EOK;
        errno = EOK;
    }
    else
#endif
    {
        iRetval = fclose(pFile);
    }

    TRACE_END();
    return iRetval;
}

/*******************************************************************************
 *
 *   OS_FRead
 *
 *******************************************************************************/
size_t OS_FRead(void *pData, size_t tSize, size_t tNumber, FILE *pFile)
{
    size_t tNumBytes = 0;

    TRACE_START();

#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN )
    {
        char *pcData = (char *)pData;

        if (gSockfd < 0)
        {
            struct sockaddr_in cli_addr;
            socklen_t clilen;

            clilen = sizeof(cli_addr);
            gSockfd = accept( gConsoleSockfd, ( struct sockaddr * )&cli_addr,
                &clilen );
            if (gSockfd >= 0)
            {
                printf("REM_CONSOLE: Connection from remote client "
                    "has been accepted\n");
            }
            else
            {
                printf("REM_CONSOLE: ERROR on accept\n");
            }
        }
        else
        {
            int n;
            n = read(gSockfd, pcData, tSize * tNumber);
            if( n > 0 )
            {
                tNumBytes = n;
                if (memchr(pcData, '!', n) != NULL)
                {
                    printf ("REM_CONSOLE: Closing remote connection\n");
                    close(gSockfd);
                    gSockfd = -1;
                    tNumBytes = 0;
                }
            }
            else
            {
                printf( "REM_CONSOLE: Closing remote connection\n" );
                close( gSockfd );
                gSockfd = -1;
                tNumBytes = 0;
            }

        }
        fwrite(pcData, tNumBytes, 1, stdout);
    }
    else
    {
        tNumBytes = fread(pData , tSize , tNumber , pFile);
    }
#else
    tNumBytes = fread(pData , tSize , tNumber , pFile);
#endif

    TRACE_END();
    return tNumBytes;
}

/*******************************************************************************
 *
 *   OS_FWrite
 *
 *******************************************************************************/
size_t OS_FWrite(const void *pData, size_t tSize, size_t tNumber, FILE *pFile)
{
    size_t tNumBytes;

    TRACE_START();

#ifdef ANDROID
    if( pFile == FAKE_STDOUT )
    {
        tNumBytes = tSize * tNumber;
        LOGD( (char*)pData );
    }
    else if( pFile == FAKE_STDERR )
    {
        tNumBytes = tSize * tNumber;
        LOGE( (char*)pData );
    }
    else
    {
        tNumBytes = fwrite(pData , tSize , tNumber , pFile);

        // Since BIONIC implementation for some cases doesn't set error
        // flag we set it here as it expected by OSAL based on errno value
        // which is set in bad case by fwrite
        if (errno != 0)
        {
            pFile->_flags |= __SERR;
        }
    }
#else
    tNumBytes = fwrite( pData, tSize, tNumber, pFile );
#endif
    TRACE_END();
    return tNumBytes;
}

/*******************************************************************************
 *
 *   OS_FTell
 *
 *******************************************************************************/
long OS_FTell( FILE *pFile)
{
    long lOffset;

    TRACE_START();

#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        lOffset = 0;
        errno = EOK;
    }
    else
#endif
    {
        lOffset = ftell(pFile);
    }

    TRACE_END();
    return lOffset;
}

/*******************************************************************************
 *
 *   OS_Ioctl
 *
 *******************************************************************************/
int OS_Ioctl( void *psFile, int iCmd, va_list *ptList )
{
    int iRetval = -1;

    TRACE_START();
    TRACE_END();

    return iRetval;
}

/*******************************************************************************
 *
 *   OS_FSeek
 *
 *******************************************************************************/
int OS_FSeek(FILE *pFile, long lOffset, int iMode)
{
    int iRetval;

    TRACE_START();

#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        iRetval = EOK;
        errno = EOK;
    }
    else
#endif
    {
        iRetval = fseek(pFile , lOffset, iMode);
    }

    TRACE_END();
    return iRetval;
}

/*******************************************************************************
 *
 *   OS_Remove
 *
 *******************************************************************************/
int OS_Remove(const char *sFileName)
{
    int iRetval;

    TRACE_START();
    iRetval = remove(sFileName);

    TRACE_END();
    return iRetval;
}

/*******************************************************************************
 *
 *   OS_Rename
 *
 *******************************************************************************/
int OS_Rename(const char *sOldName, const char *sNewName)
{
    N32 iRetval;

    TRACE_START();
    iRetval = rename(sOldName , sNewName);

	// Bosch ID#13: fsync is needed, while renaming sms.cfg
    int fd = open(sNewName,  O_RDWR);
    if (fd==-1) {
        printf("OS_Rename: open failed on %s\n", sNewName);
    }
    if (fsync(fd)) {
        printf("OS_Rename: fsync failed on %s\n", sNewName);

    }
    if (close(fd)) {
        printf("OS_Rename: close failed on %s\n", sNewName);
    }

    TRACE_END();
    return iRetval;
}


/*******************************************************************************
 *
 *   OS_FFlush
 *
 *******************************************************************************/
int OS_FFlush(FILE *pFile)
{
    N32 n32RtnCode;

    TRACE_START();

#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        n32RtnCode = 0;
        errno = EOK;
    }
    else
#endif
    {
        n32RtnCode = (N32)fflush(pFile);
    }

    TRACE_END();
    return n32RtnCode;
}

/*******************************************************************************
 *
 *   OS_FError
 *
 *******************************************************************************/
int OS_FError(FILE *pFile)
{
    N32 n32RtnCode;

    TRACE_START();

#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        n32RtnCode = 0;
        errno = EOK;
    }
    else
#endif
    {
        n32RtnCode = (N32)ferror(pFile);
    }

    TRACE_END();
    return n32RtnCode;
}

/*******************************************************************************
 *
 *   OS_FEof
 *
 *******************************************************************************/
int OS_FEof(FILE *pFile)
{
    N32 n32RtnCode;

    TRACE_START();
#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        n32RtnCode = 0;
        errno = EOK;
    }
    else
#endif
    {
        n32RtnCode = (N32)feof(pFile);
    }

    TRACE_END();
    return n32RtnCode;
}

/*******************************************************************************
 *
 *   OS_Clearerr
 *
 *******************************************************************************/
void OS_Clearerr(FILE *pFile)
{
    TRACE_START();

#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        errno = EOK;
    }
    else
#endif
    {
        clearerr(pFile);
    }

    TRACE_END();
}

/*******************************************************************************
*
*   OS_Fileno
*
*******************************************************************************/
int OS_Fileno(FILE *pFile)
{
    N32 n32RtnCode;

    TRACE_START();

    if (pFile == NULL)
    {
    	n32RtnCode = -1;
    }
    else
#if OSAL_REMOTE_CONSOLE == 1
    if( pFile == FAKE_STDIN || pFile == FAKE_STDOUT || pFile == FAKE_STDERR )
    {
        n32RtnCode = 0;
        errno = EOK;
    }
    else
#endif
    {
        n32RtnCode = (N32)fileno(pFile);
    }

    TRACE_END();
    return n32RtnCode;
}
