/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*DESCRIPTION
*
*       This module will contain all the OSAL File System APIs.
* The implementations in this module are 100% POSIX complient making
* it suitable for any POSIX.1b (POSIX 1003.1b-1993) compliant OS.
*
*******************************************************************************/

// POSIX and standard includes
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stddef.h>
#include <dirent.h>
#include <errno.h>

#if defined(__QNX__)
#include <sys/statvfs.h>
#include <process.h>
#elif defined(__INTEGRITY)
#include <sys/statvfs.h>
#else
#include <sys/statfs.h>
#endif

#define OSAL_TRACE_ENABLE   0
#include "osal_trace.h"

#include "standard.h"
#include "osal.h"
#include "os_version.h"
#include "os.h"
#include "os_fs.h"
#include "_os_fs.h"
#include "os_stdio.h"

#if OSAL_FILE_SYSTEM == 1

/*******************************************************************************
*
*   OS_bFileSystemStart
*
*******************************************************************************/
BOOLEAN OS_bFileSystemStart ( void )
{
    TRACE_START();
    // Nothing to implement.
    TRACE_END();
	return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemStop
*
*******************************************************************************/
BOOLEAN OS_bFileSystemStop ( void )
{
    TRACE_START();
    // The sync() function queues all the modified block buffers for writing,
    // and returns; it doesn't wait for the actual I/O to take place. Use
    // this function -- or fsync() for a single file -- to ensure consistency
    // of the entire on-disk filesystem with the contents of the in-memory
    // buffer cache.

    // Note! This does not complete any pending I/O operations. It only
    // queues them. The rest of the system must make sure that any pending
    // data is written before pulling the plug.
    sync();

    TRACE_END();
    return TRUE;
}

/*******************************************************************************
*
*   OS_vFileSystemVolumes
*
*******************************************************************************/
void OS_vFileSystemVolumes ( FILE *psFile )
{
    TRACE_START();
    // Check input
    if(psFile == NULL)
    {
        // Error! Bad input
        TRACE_END();
        return;
    }

    // TODO: Need to implement the capability to list the available
    // mounted volumes to an output file descriptor. For now,
    // just tell them we cannot do this.
    fprintf(psFile, "This API is currently not supported.\n");

    TRACE_END();
    return;
}

/*******************************************************************************
*
*   OS_bFileSystemVolumeInfo
*
*******************************************************************************/
BOOLEAN OS_bFileSystemVolumeInfo (
    FILE *psFile,
    const char *pcVolName
        )
{
#ifdef __QNX__
    struct statvfs sBuf;
#else
    struct statfs sBuf;
#endif
    int iRetval;

    TRACE_START();
    // Check inputs
    if( (psFile == NULL) || (pcVolName == NULL) )
    {
        // Error! Bad input(s)
        TRACE_END();
        return FALSE;
    }

    // Get filesystem information, given a path
#ifdef __QNX__
    iRetval = statvfs(pcVolName , &sBuf);
#else
    iRetval = statfs(pcVolName , &sBuf);
#endif
    if(iRetval == -1)
    {
        print_errno("Error! statvfs()");
        TRACE_END();
        return FALSE;
    }

    // Dump results to provided output file descriptor
    fprintf(psFile, "\nFile system information for '%s'\n",
        pcVolName);
#ifdef __QNX__
    fprintf(psFile, "f_bsize = %u\n",(unsigned int)sBuf.f_bsize);
    fprintf(psFile, "f_frsize = %u\n",(unsigned int)sBuf.f_frsize);
#else
    fprintf(psFile, "f_type = %u\n",sBuf.f_type);
    fprintf(psFile, "f_bsize = %u\n",sBuf.f_bsize);
#endif

    fprintf(psFile, "f_blocks = %u\n",sBuf.f_blocks);
    fprintf(psFile, "f_bfree = %u\n",sBuf.f_bfree);
    fprintf(psFile, "f_bavail = %u\n",sBuf.f_bavail);
    fprintf(psFile, "f_files = %u\n",sBuf.f_files);
    fprintf(psFile, "f_ffree = %u\n",sBuf.f_ffree);

#if defined(__QNX__)
    fprintf(psFile, "f_favail = %u\n",sBuf.f_favail);
    fprintf(psFile, "f_fsid = %u\n",(unsigned int)sBuf.f_fsid);
    fprintf(psFile, "f_flag = %#0x (%s%s%s%s%s%s)\n",
        (unsigned int)sBuf.f_flag,
        (((sBuf.f_flag & ST_NOATIME) == ST_NOATIME) ?
            MACRO_TO_STRING(ST_NOATIME)"|" : ""),
        (((sBuf.f_flag & ST_OFF32) == ST_OFF32)     ?
            MACRO_TO_STRING(ST_OFF32)"|" : ""),
        (((sBuf.f_flag & ST_NOCREAT) == ST_NOCREAT) ?
            MACRO_TO_STRING(ST_NOCREAT)"|" : ""),
        (((sBuf.f_flag & ST_NOSUID) == ST_NOSUID)   ?
            MACRO_TO_STRING(ST_NOSUID)"|" : ""),
        (((sBuf.f_flag & ST_NOEXEC) == ST_NOEXEC)   ?
            MACRO_TO_STRING(ST_NOEXEC)"|" : ""),
        (((sBuf.f_flag & ST_RDONLY) == ST_RDONLY)   ?
            MACRO_TO_STRING(ST_RDONLY) : "")
            );
    fprintf(psFile, "f_namemax = %u\n",(unsigned int)sBuf.f_namemax);
    fprintf(psFile, "f_basetype = %s\n",sBuf.f_basetype);
#elif defined(__INTEGRITY)
    // Nothing here
#else
    fprintf(psFile, "f_namelen = %u\n",(unsigned int)sBuf.f_namelen);
    fprintf(psFile, "f_frsize = %u\n",sBuf.f_frsize);
#endif

    // Summary information
    fputs("", psFile);
    fprintf(psFile, "%u bytes total.\n",
        sBuf.f_bsize * sBuf.f_blocks);
    fprintf(psFile, "%u bytes available.\n",
        sBuf.f_bsize * sBuf.f_bfree);

    TRACE_END();
    return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemDirectory
*
*******************************************************************************/
BOOLEAN OS_bFileSystemDirectory (
    FILE *psFile,
    const char *pcPathName
        )
{
    DIR *dirp;
    struct dirent *direntp;

    TRACE_START();
    // Check inputs
    if( (psFile == NULL) || (pcPathName == NULL) )
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Open a directory for reading
    dirp = opendir( pcPathName );
    if( dirp != NULL )
    {
        for(;;)
        {
            // Read directory entries until no more left
            direntp = readdir(dirp);
            if( direntp == NULL )
            {
                break;
            }
            fprintf(psFile, "%s\n", direntp->d_name );
        }

        // Close directory access
        closedir( dirp );
    }
    else
    {
        // Error! Cannot open directory

        // Only print the error when it
        // isn't ENOENT.  ENOENT just means
        // we tried to query a non-existent
        // directory, which isn't all that
        // big of a problem, and also may
        // be a simple test that a caller
        // may want to execute.
        if (errno != ENOENT)
        {
            print_errno("Error! opendir()");
        }

        TRACE_END();
        return FALSE;
    }

    TRACE_END();
    return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemMakeDir
*
*******************************************************************************/
BOOLEAN OS_bFileSystemMakeDir (
    const char *pcDirName
        )
{
	int iRetval;
	mode_t tMode = ((S_IRWXU) | (S_IRWXG) | (S_IRWXO));

    TRACE_START();
    // Check input
    if(pcDirName == NULL)
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Create a sub-directory
	iRetval = mkdir(pcDirName , tMode);
    if(iRetval == -1)
    {
        print_errno("Error! mkdir()");
        TRACE_END();
        return FALSE;
    }

    TRACE_END();
	return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemRemoveDir
*
*******************************************************************************/
BOOLEAN OS_bFileSystemRemoveDir (
    const char *pcDirName
        )
{
	int iRetval;

    TRACE_START();
    // Check input
    if(pcDirName == NULL)
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Delete an empty directory
	iRetval = rmdir(pcDirName);
    if(iRetval == -1)
    {
        // Only print the error when it
        // isn't ENOENT.  ENOENT just means
        // we tried to remove a non-existent
        // directory, which isn't all that
        // big of a problem, and also may
        // be a simple test that a caller
        // may want to execute.
        if (errno != ENOENT)
        {
            print_errno("Error! rmdir()");
        }

        TRACE_END();
        return FALSE;
    }

    TRACE_END();
	return TRUE;
}

/*******************************************************************************
*
*   OS_eFileSystemGetDirItems
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eFileSystemGetDirItems (
    OSAL_OBJECT_HDL *phDirItems, const char *pcPathName,
    OSAL_FS_FILE_QUALIFIER bFileQualifierFunc, void *pvQualiferArg)
{
    DIR *dirp;
    struct dirent *direntp;
    BOOLEAN bAddItem = TRUE, bListEmpty = TRUE;
    char *pcDirItem;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    TRACE_START();
    // Check inputs
    if( (phDirItems == NULL) || (pcPathName == NULL) )
    {
        // Error!
        TRACE_END();
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Open a directory for reading
    dirp = opendir( pcPathName );
    if( dirp != NULL )
    {
        for(;;)
        {
            // Read directory entries until no more left
            direntp = readdir(dirp);
            if( direntp == NULL )
            {
                break;
            }

            if (bFileQualifierFunc != NULL)
            {
                bAddItem = bFileQualifierFunc(
                    direntp->d_name, pvQualiferArg);
            }

            if (bAddItem == TRUE)
            {
                // Allocate memory for the name
                pcDirItem = OSALC_pvMemoryAllocate("FS:DirItem",
                    strlen(direntp->d_name)+1, FALSE);

                if (pcDirItem == NULL)
                {
                    eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
                    break;
                }

                // Copy the string to our pointer
                strcpy(pcDirItem, direntp->d_name);

                eReturnCode = OSAL.eLinkedListAdd(*phDirItems,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void*)pcDirItem);

                if (eReturnCode != OSAL_SUCCESS)
                {
                    // Error
                    break;
                }

                // We've got at least one item, so clear the flag
                if (bListEmpty == TRUE)
                {
                    bListEmpty = FALSE;
                }
            }
        }

        // Close directory access
        closedir( dirp );
    }
    else
    {
        // Error! Cannot open directory

        // Only print the error when it
        // isn't ENOENT.  ENOENT just means
        // we tried to query a non-existent
        // directory, which isn't all that
        // big of a problem, and also may
        // be a simple test that a caller
        // may want to execute.
        if (errno != ENOENT)
        {
            print_errno("Error! opendir()");
        }

        eReturnCode = OSAL_ERROR_UNKNOWN;
    }

    if (eReturnCode == OSAL_SUCCESS && bListEmpty == TRUE)
    {
        eReturnCode = OSAL_NO_OBJECTS;
    }

    TRACE_END();
    return eReturnCode;
}

/*******************************************************************************
*
*   OS_bFileSystemTruncateFile
*
*******************************************************************************/
BOOLEAN OS_bFileSystemTruncateFile (
    FILE *psFile,
    size_t tNewSize
        )
{
	int iRetval, ifd;

    TRACE_START();
    // Check input
    if(psFile == NULL)
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Return the file descriptor for a stream
	ifd = OS_Fileno(psFile->pvHdl);
    if(ifd == -1)
    {
        print_errno("Error! fileno()");
        TRACE_END();
        return FALSE;
    }

    // Truncate a file
	iRetval = ftruncate(ifd , (off_t)tNewSize);
    if(iRetval == -1)
    {
        print_errno("Error! ftruncate()");
        TRACE_END();
        return FALSE;
    }

    // Synchronize the file state. The fsync() function forces all queued I/O
    // operations for the file specified by the 'ifd' file descriptor to
    // finish, synchronizing the file's state.
    iRetval = fsync(ifd);
    if(iRetval == -1)
    {
        print_errno("Error! fsync()");
        TRACE_END();
        return FALSE;
    }

    TRACE_END();
	return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemGetFileSize
*
*******************************************************************************/
BOOLEAN OS_bFileSystemGetFileSize (
    FILE *psFile,
    size_t *ptFileSize
        )
{
	struct stat sBuf;
	int iRetval, ifd;

    TRACE_START();
    // Check inputs
    if( (psFile == NULL) || (ptFileSize == NULL) )
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Return the file descriptor for a stream
    ifd = OS_Fileno(psFile->pvHdl);
    if(ifd == -1)
    {
        print_errno("Error! fileno()");
        TRACE_END();
        return FALSE;
    }

    // Get file information, given a file description
	iRetval = fstat(ifd , &sBuf);
	if(iRetval == -1)
	{
        print_errno("Error! fstat()");
        TRACE_END();
        return FALSE;
	}

   	*ptFileSize = (size_t)sBuf.st_size;

    TRACE_END();
	return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemGetFileAttributes
*
*******************************************************************************/
BOOLEAN OS_bFileSystemGetFileAttributes (
    const char *pcPath,
    UN8 *pun8FileAttributes
        )
{
	int iRetval;
	struct stat sBuf;
	N8 n8SystemFlag;
    char *pcSlashPos, *pcDotPos;
    const char *pcStartDotSearchPos;

    TRACE_START();
    // Check inputs
    if(pcPath == NULL)
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Initialize attributes mask
    // If NULL pointer is passed for attributes storage, just return
    // TRUE or FALSE depending on path existence (ENOENT)
    if(pun8FileAttributes != NULL)
    {
       *pun8FileAttributes = 0;
    }

    // Get information about a file or directory, given a path
	iRetval = stat( pcPath , &sBuf );
    if(iRetval == -1)
    {
        // Only print the error when it
        // isn't ENOENT.  ENOENT just means
        // we tried to query a non-existent
        // file, which isn't all that
        // big of a problem, and also may
        // be a simple test that a caller
        // may want to execute.
        if (errno != ENOENT)
        {
            print_errno("stat()");
        }

        TRACE_END();
        return FALSE;
    }

    if(pun8FileAttributes != NULL)
    {
       // Parse attributes...
       if(sBuf.st_mode & S_IRUSR)
       {
          *pun8FileAttributes |= OSAL_FILE_ATTR_READ;
       }

       if(sBuf.st_mode & S_IWUSR)
       {
          *pun8FileAttributes |= OSAL_FILE_ATTR_WRITE;
       }

       if(S_ISDIR(sBuf.st_mode) != 0)
       {
          *pun8FileAttributes |= OSAL_FILE_ATTR_DIRECTORY;
       }

       n8SystemFlag = (S_ISBLK(sBuf.st_mode) || S_ISCHR(sBuf.st_mode)
                || S_ISFIFO(sBuf.st_mode) || S_ISLNK(sBuf.st_mode)
#ifdef __QNX__
                || S_TYPEISMQ(&sBuf) || S_TYPEISSEM(&sBuf)
					|| S_TYPEISSHM(&sBuf)
#endif
		);

       if(n8SystemFlag)
       {
          *pun8FileAttributes |= OSAL_FILE_ATTR_SYSTEM;
       }

       // File is hidden if "." is a first character in linux file name.
       // Get the last slash position - is is beginning of the file name
       pcSlashPos = strrchr(pcPath, '/');
       pcStartDotSearchPos = (pcSlashPos == NULL) ? pcPath  : (pcSlashPos + 1);
       pcDotPos = strchr( pcStartDotSearchPos, '.');

       // file name length should be greater than 1 if file is hidden even if we have a dot inside.
       if ((pcDotPos != NULL) && (strlen(pcDotPos) > 1))
       {
           if (pcDotPos == pcStartDotSearchPos)
           {
               *pun8FileAttributes |= OSAL_FILE_ATTR_HIDDEN;
           }
       }
    }

    TRACE_END();
	return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemSetFileAttributes
*
*******************************************************************************/
BOOLEAN OS_bFileSystemSetFileAttributes (
    const char *pcPath,
    UN8 un8FileAttributes
        )
{
	int iRetval;
    struct stat sBuf;

    TRACE_START();
    // Check input
    if(pcPath == NULL)
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Set operation for HIDDEN and SYSTEM file attributes is not supported
    if (un8FileAttributes & (OSAL_FILE_ATTR_HIDDEN | OSAL_FILE_ATTR_SYSTEM))
    {
    	printf("Error! Set is not supported for hidden and system file attributes\n");
    	TRACE_END();
    	return FALSE;
    }

    // Extract current attributes from the path provided
    iRetval = stat( pcPath , &sBuf );
    if(iRetval == -1)
    {
        print_errno("Error! stat()");
        TRACE_END();
        return FALSE;
    }

    // Clear both read and write permission groups
    sBuf.st_mode &=
        ~( S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR | S_IWGRP | S_IWOTH );

    // Parse attriibutes and set appropriately
	if(un8FileAttributes & OSAL_FILE_ATTR_READ)
    {
		sBuf.st_mode |= (S_IRUSR | S_IRGRP | S_IROTH);
    }

	if(un8FileAttributes & OSAL_FILE_ATTR_WRITE)
    {
		sBuf.st_mode |= (S_IWUSR | S_IWGRP | S_IWOTH);
    }

    // Change the permissions for a file based on parsed parameters
	iRetval = chmod (pcPath , sBuf.st_mode);
    if(iRetval == -1)
    {
        print_errno("Error! chmod()");
        TRACE_END();
        return FALSE;
    }

    TRACE_END();
	return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemSyncFile
*
*******************************************************************************/
BOOLEAN OS_bFileSystemSyncFile (
    FILE *psFile
        )
{
    int iRetval, ifd;

    TRACE_START();
    // Check inputs
    if( psFile == NULL )
    {
        // Error!
        TRACE_END();
        return FALSE;
    }

    // Return the file descriptor for a stream
    ifd = fileno(psFile->pvHdl);
    if(ifd == -1)
    {
        print_errno("Error! fileno()");
        TRACE_END();
        return FALSE;
    }

    // Synchronize the file state. The fsync() function forces all queued I/O
    // operations for the file specified by the 'ifd' file descriptor to
    // finish, synchronizing the file's state.
    iRetval = fsync(ifd);
    if(iRetval == -1)
    {
        if ((errno != ENOSYS) && (errno != EROFS))
        {
            print_errno("Error! fsync()");
            TRACE_END();
            return FALSE;
        }
    }

    TRACE_END();
    return TRUE;
}

/*******************************************************************************
*
*   OS_bFileSystemUnconditionalFileUnlink
*
*   This function is intended for forced file removal.
*   As opposite to OS_iRemove(), this should remove files even in case of
*   corrupted file system when some extra system-level checks being done by
*   remove() function call would fail.
*
*******************************************************************************/
BOOLEAN OS_bFileSystemUnconditionalFileUnlink (
    const char *pcPath
        )
{
    BOOLEAN bReturn = FALSE;
    int iReturn;

    iReturn = unlink(pcPath);
    if (iReturn == EOK)
    {
        // Success
        bReturn = TRUE;
    }

    return bReturn;
}

#endif /* OSAL_FILE_SYSTEM == 1  */
