/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.                      */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio Inc.              */
/*                        Proprietary & Confidential                          */
/******************************************************************************/
/*******************************************************************************
 *
 *
 *
 *
 *
 *DESCRIPTION
 *
 *       This module will contain all the custom stdio
 *       functionality.
 *
 *******************************************************************************/

#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>

#include "standard.h"

#include "osal.h"
#include "osal_debughandler.h"
#include "osal_core.h"

#include "os_intf.h"

#include "osal_stdio.h"
#include "_osal_stdio.h"

/*******************************************************************************
 *
 *   OSAL_psFopen
 *
 *******************************************************************************/
OSAL_FILE_STRUCT *OSAL_psFopen( const char *pcFileName, const char *pcMode )
{
   OSAL_FILE_STRUCT *psFile = NULL;
   char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

   // Check input, pcMode may be NULL
   if( pcFileName == NULL )
   {
      return NULL;
   }

   // Construct an appropriate name for memory to be allocated
   snprintf( &acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
             OSAL_NAME_PREFIX"%s", pcFileName );

   // Allocate a file handle
   psFile = OSALC_pvMemoryAllocate( &acName[0], sizeof(OSAL_FILE_STRUCT), FALSE );
   if( psFile == NULL )
   {
      return NULL;
   }

   // Ensure the protection attribute is initialized correctly
   psFile->pvUsageProtection = OSAL_INVALID_OBJECT_HDL;

   // Check if its something in the device driver framework (DEVICE) first
   if( GsDeviceIntf.psFopen != NULL )
   {
      psFile->pvHdl = GsDeviceIntf.psFopen( pcFileName, pcMode );
      if( psFile->pvHdl != DEV_INVALID_HDL )
      {
         // Initialize file type as 'DEVICE'
         psFile->psIntf = &GsDeviceIntf;
         return psFile;
      }
   }

   // Must be directed to the file system (FS)
   if( GsFSIntf.psFopen != NULL )
   {
      BOOLEAN bModeOk;

      // Validate the mode going to the fopen call
      bModeOk = bVerifyFileMode( pcMode );

      if (bModeOk == TRUE)
      {
          // Call FS API
          psFile->pvHdl = GsFSIntf.psFopen( pcFileName, pcMode );
          if( psFile->pvHdl != NULL )
          {
             // Initialize file type as 'FS'
             psFile->psIntf = &GsFSIntf;
             return psFile;
          }
      }
   }

   OSALC_vMemoryFree( psFile );
   return NULL;
}

/*******************************************************************************
 *
 *   OSAL_iFclose
 *
 *******************************************************************************/
int OSAL_iFclose( OSAL_FILE_STRUCT *psFile )
{
   OSAL_OBJECT_HDL hUsageSem;
   int iRetval = EOF;

   // Check input
   if( psFile == NULL )
   {
      return EOF;
   }

   // If this file has usage protection enabled, verify
   // it is no longer being used (accomplished via a
   // semaphore pend)
   hUsageSem = ( OSAL_OBJECT_HDL )psFile->pvUsageProtection;
   if( hUsageSem != OSAL_INVALID_OBJECT_HDL )
   {
      OSAL_RETURN_CODE_ENUM eReturnCode;

      // Pend on the usage semaphore
        eReturnCode = OSAL.eSemTake(
            hUsageSem,
            OSAL_OBJ_TIMEOUT_INFINITE );

      if( eReturnCode == OSAL_SUCCESS )
      {
         // Delete the semaphore and clear the handle
         OSAL.eSemDelete( hUsageSem );
         psFile->pvUsageProtection = OSAL_INVALID_OBJECT_HDL;

         // No error
         iRetval = 0;
      }
      else
      {
         // Unable to access the usage semaphore!
         return EOF;
      }
   }

   // Call API
   if( psFile->psIntf != NULL )
   {
	   // Bosch ID#28: Additional check added to check psFile->pvHdl's validity
#if 0
      if( psFile->psIntf->iFclose != NULL )
#else
      if(( psFile->psIntf->iFclose != NULL ) && ( psFile->pvHdl != NULL ))
#endif
      {
         iRetval = psFile->psIntf->iFclose( psFile->pvHdl );
         if( iRetval == 0 )
         {
            // Nullify elements;
            psFile->psIntf = NULL;
            psFile->pvHdl = NULL;
            psFile->pvUsageProtection = NULL;
            psFile->tProtectionNesting = 0;

         }
         else
         {
            iRetval = EOF;
         }

         // Release memory
         OSALC_vMemoryFree( psFile );
         psFile = NULL;
      }
   }

   return iRetval;
}

/*******************************************************************************
 *
 *   OSAL_tFread
 *
 *******************************************************************************/
size_t OSAL_tFread( void *pvDst, size_t tSize, size_t tNumObj,
                    OSAL_FILE_STRUCT *psFile )
{
   size_t tNum = 0; // Number of objects read

   // Check input
   if( psFile == NULL )
   {
      return 0;
   }

   // Call API
   if( psFile->psIntf != NULL )
   {
      if( psFile->psIntf->tFread != NULL )
      {
            tNum = psFile->psIntf->tFread(
                pvDst, tSize, tNumObj, psFile->pvHdl);
      }
   }

   return tNum;
}

/*******************************************************************************
 *
 *   OSAL_tFwrite
 *
 *******************************************************************************/
size_t OSAL_tFwrite( const void *pvSrc, size_t tSize, size_t tNumObj,
                     OSAL_FILE_STRUCT *psFile )
{
   size_t tNum = 0; // Number of objects written

   // Check input
   if( psFile == NULL )
   {
      return 0;
   }

   // Call API
   if( psFile->psIntf != NULL )
   {
      if( psFile->psIntf->tFwrite != NULL )
      {
            tNum = psFile->psIntf->tFwrite(
                pvSrc, tSize, tNumObj, psFile->pvHdl);
      }
   }

   return tNum;
}

/*******************************************************************************
 *
 *   OSAL_tFtell
 *
 *******************************************************************************/
long OSAL_lFtell( OSAL_FILE_STRUCT *psFile )
{
   long lOffset = 0;

   // Check input
   if( psFile == NULL )
   {
      return 0;
   }

   // Call API
   if( psFile->psIntf != NULL )
   {
      if( psFile->psIntf->lFtell != NULL )
      {
         lOffset = psFile->psIntf->lFtell( psFile->pvHdl );
      }
   }

   return lOffset;
}

/*******************************************************************************
 *
 *   OSAL_iIoctl
 *
 *******************************************************************************/
int OSAL_iIoctl( OSAL_FILE_STRUCT *psFile, int iCmd, ... )
{
   int err = EOF; // return value
   va_list tArgList; // variable arguments list

   // Check input
   if( psFile == NULL )
   {
      return EOF;
   }

   // grab variable arguments from the stack
   va_start(tArgList, iCmd);

   // Determine if command is for us...
   switch( iCmd )
   {
      // Enable and increment the semaphore protecting
      // this file
      case FILE_IOCTL_USAGE_PROTECT:
      {
            OSAL_RETURN_CODE_ENUM eReturnCode =
                OSAL_SUCCESS;
         OSAL_OBJECT_HDL hLocalSem = OSAL_INVALID_OBJECT_HDL;
         OSAL_OBJECT_HDL *phUsageSem;
         char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

         // Construct an appropriate name for memory to be allocated,
         // but do it atomically
         snprintf( &acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                OSAL_NAME_PREFIX"FileUsage:%u",
                (size_t)psFile->pvHdl
                    );

         // Create a semaphore in order to protect
         // this file from an fclose
            eReturnCode =
                OSAL.eSemCreate(
                    &hLocalSem, // Semaphore handle
                    &acName[0],
                    0,          // Initial value
                    1, // Semaphore max count
                    OSAL_SEM_OPTION_NONE );

            phUsageSem = (OSAL_OBJECT_HDL *)
                &psFile->pvUsageProtection;

         /*****************************************************************/
         OSAL.eEnterTaskSafeSection();
         // If the file protection semaphore doesn't exist,
         // determine if we were able to create one
         if( *phUsageSem == OSAL_INVALID_OBJECT_HDL )
         {
            // We were able to create a file protection semaphore
                if ((eReturnCode == OSAL_SUCCESS)
                    && (hLocalSem != OSAL_INVALID_OBJECT_HDL))
            {
               psFile->tProtectionNesting = 0;
               psFile->pvUsageProtection = ( void * )hLocalSem;
               hLocalSem = OSAL_INVALID_OBJECT_HDL;
               err = 0;
            }
         }
         else
         {
            // Adjust nesting count
            psFile->tProtectionNesting++;
            err = 0;
         }
         OSAL.eExitTaskSafeSection();
         /*****************************************************************/

         // Clean up anything we left behind...
         if( hLocalSem != OSAL_INVALID_OBJECT_HDL )
         {
            OSAL.eSemDelete( hLocalSem );
         }
      }
         break;

         // Decrement the semaphore protecting this file
         // if it is present
      case FILE_IOCTL_USAGE_UNPROTECT:
      {
            OSAL_RETURN_CODE_ENUM eReturnCode =
                OSAL_SUCCESS;
         OSAL_OBJECT_HDL hUsageSem;
         BOOLEAN bGive = FALSE;

         /*****************************************************************/
         OSAL.eEnterTaskSafeSection();
         hUsageSem = ( OSAL_OBJECT_HDL )psFile->pvUsageProtection;

         if( hUsageSem != OSAL_INVALID_OBJECT_HDL )
         {
            if( psFile->tProtectionNesting > 0 )
            {
               psFile->tProtectionNesting--;
            }

            if( psFile->tProtectionNesting == 0 )
            {
               bGive = TRUE;
            }
         }
         OSAL.eExitTaskSafeSection();
         /*****************************************************************/

         if( bGive == TRUE )
         {
            // Give to the semaphore --
            eReturnCode = OSAL.eSemGive( hUsageSem );
         }

         // Ensure all has gone well...
         if( eReturnCode == OSAL_SUCCESS )
         {
            err = 0;
         }
      }
         break;

      default: // Pass the ioctl to the next processor...
      {

         // Call API
         if( psFile->psIntf != NULL )
         {
            if( psFile->psIntf->iIoctl != NULL )
            {
                    err = psFile->psIntf->iIoctl(
                        psFile->pvHdl, iCmd, &tArgList);
            }
         }

         break;
      }
   }

   // restore stack
   va_end(tArgList);

   return err;
}

/*******************************************************************************
 *
 *   OSAL_iFseek
 *
 *******************************************************************************/
int OSAL_iFseek( OSAL_FILE_STRUCT *psFile, long lOffset, int iMode )
{
   int iRetval;

   // Check input
   if( psFile == NULL )
   {
      return EOF;
   }

   // Call API
   if( psFile->psIntf != NULL )
   {
      if( psFile->psIntf->iFseek != NULL )
      {
         iRetval = psFile->psIntf->iFseek( psFile->pvHdl, lOffset, iMode );
         return iRetval;
      }
   }

   return EOF;
}

/*******************************************************************************
 *
 *   OSAL_iRemove
 *
 *******************************************************************************/
int OSAL_iRemove( const char *pcFileName )
{
    int iRetval = 0;
    UN8 un8Attributes = 0;
    BOOLEAN bOk = FALSE;

    // Check input
    if( pcFileName == NULL )
    {
        return EOF;
    }

    bOk = OSAL.bFileSystemGetFileAttributes(pcFileName, &un8Attributes);
    if((bOk == TRUE) &&
       ((un8Attributes & OSAL_FILE_ATTR_WRITE) == OSAL_FILE_ATTR_WRITE) &&
       (GsFSIntf.iRemove != NULL))
    {
        iRetval = GsFSIntf.iRemove( pcFileName );

        if (iRetval == 0)
        {
            // Remove request successed. Just need to be sure that
            // the file was really removed
            UN8 un8Attempts = OSAL_FS_REMOVE_ATTEMPTS;
            UN32 un32DelayBetweenAttempts = OSAL_FS_REMOVE_DELAY; /* ms */

            do
            {
                // Get attributes
                bOk = OSAL.bFileSystemGetFileAttributes(pcFileName, &un8Attributes);
                if (bOk == FALSE)
                {
                    // File does not exist. Exiting
                    iRetval = 0;
                    break;
                }
                // The file still exists. Need to wait until it is removed
                iRetval = EOF;
                OSAL.eTaskDelay(un32DelayBetweenAttempts);
            } while (--un8Attempts > 0);
        }
    }

    return iRetval;
}

/*******************************************************************************
 *
 *   OSAL_iCopyFile
 *
 *******************************************************************************/
int OSAL_iCopyFile( const char *pcSrcFileName, const char *pcDstFileName, BOOLEAN bForce )
{
   int iRetval = EOF;
   OSAL_FILE_STRUCT *psOldFile = NULL;
   OSAL_FILE_STRUCT *psNewFile = NULL;
   size_t tFileSize = 0;
   size_t tReadBytes = 0, tWrittenBytes = 0;
   BOOLEAN bOk;

   do
   {
       // Check inputs
       if( ( pcSrcFileName == NULL ) || ( pcDstFileName == NULL ) )
       {
           break;
       }

       psOldFile = OSAL_psFopen(pcSrcFileName, "rb");
       if(psOldFile == NULL)
       {
           // Cannot open old file
           break;
       }

       bOk = OSAL.bFileSystemGetFileSize(psOldFile, &tFileSize);
       if(bOk != TRUE)
       {
           // Error reading file size
           break;
       }

       if(bForce == FALSE)
       {
           UN8 un8Attr;

           bOk = OSAL.bFileSystemGetFileAttributes(pcDstFileName, &un8Attr);
           if(bOk == TRUE)
           {
               // File exist. Cannot copy, because bForce flag is not set
               break;
           }
       }

       psNewFile = OSAL_psFopen(pcDstFileName, "wb");
       if(psNewFile == NULL)
       {
           // Cannot create new file
           break;
       }
       // Now copying file by blocks
       do
       {
           static char acTempBuf[STDIO_FILE_CLUSTER_SIZE];

           tReadBytes = OSAL_tFread(acTempBuf, 1, STDIO_FILE_CLUSTER_SIZE, psOldFile);
           tWrittenBytes = OSAL_tFwrite(acTempBuf, 1, tReadBytes, psNewFile);
           if(tReadBytes != tWrittenBytes)
           {
               // Error writing file
               break;
           }

           if(tWrittenBytes > tFileSize)
           {
               // Strange... we've read more than expected
               break;
           }
           else if(tWrittenBytes == tFileSize)
           {
               // EOF. Normal exit
               iRetval = 0;
               break;
           }
           else
           {
               // Need to copy next block
               tFileSize -= tWrittenBytes;
           }
       } while(TRUE);
   } while (FALSE);

   if(psNewFile != NULL)
   {
       OSAL_iFclose(psNewFile);
   }

   if(psOldFile != NULL)
   {
       OSAL_iFclose(psOldFile);
   }

   return iRetval;
}

/*******************************************************************************
 *
 *   OSAL_iRename
 *
 *******************************************************************************/
int OSAL_iRename( const char *pcOldFileName, const char *pcNewFileName )
{
   int iRetval;

   // Check inputs
   if( ( pcOldFileName == NULL ) || ( pcNewFileName == NULL ) )
   {
      return EOF;
   }

   // Call API
   if( GsFSIntf.iRename != NULL )
   {
      iRetval = GsFSIntf.iRename( pcOldFileName, pcNewFileName );
      return iRetval;
   }

   return EOF;
}

/*******************************************************************************
 *
 *   OSAL_iFflush
 *
 *******************************************************************************/
int OSAL_iFflush( OSAL_FILE_STRUCT *psFile )
{
   int iReturn = EOF;

   if( psFile == NULL )
   {
      // If the file pointer is NULL, we need to flush all open
      // files.  But, we can't use the file pointer to call the
      // actual fflush function.  So, we will make a direct
      // OS call
      iReturn = GsFSIntf.iFflush( NULL );
   }
   else
   {
      // Call API
      if( psFile->psIntf != NULL )
      {
         if( psFile->psIntf->iFflush != NULL )
         {
            iReturn = psFile->psIntf->iFflush( psFile->pvHdl );
         }
      }
   }

   return iReturn;
}

/*******************************************************************************
 *
 *   OSAL_iFerror
 *
 *******************************************************************************/
int OSAL_iFerror( OSAL_FILE_STRUCT *psFile )
{
   int iReturn = EOF;

   if( psFile != NULL )
   {
      // Call API
      if( psFile->psIntf != NULL )
      {
         if( psFile->psIntf->iFerror != NULL )
         {
            iReturn = psFile->psIntf->iFerror( psFile->pvHdl );
         }
      }
   }

   return iReturn;
}

/*******************************************************************************
 *
 *   OSAL_iFeof
 *
 *******************************************************************************/
int OSAL_iFeof( OSAL_FILE_STRUCT *psFile )
{
   int iReturn = EOF;

   if( psFile != NULL )
   {
      // Call API
      if( psFile->psIntf != NULL )
      {
         if( psFile->psIntf->iFeof != NULL )
         {
            iReturn = psFile->psIntf->iFeof( psFile->pvHdl );
         }
      }
   }

   return iReturn;
}

/*******************************************************************************
 *
 *   OSAL_vClearerr
 *
 *******************************************************************************/
void OSAL_vClearerr( OSAL_FILE_STRUCT *psFile )
{
   if( psFile == NULL )
   {
      return;
   }

   // Call API
   if( psFile->psIntf != NULL )
   {
      if( psFile->psIntf->vClearerror != NULL )
      {
         psFile->psIntf->vClearerror( psFile->pvHdl );
      }
   }

   return;
}

/*******************************************************************************
 *
 *   OSAL_iFgetc
 *
 *******************************************************************************/
int OSAL_iFgetc( OSAL_FILE_STRUCT *psFile )
{
   int iReturn;
   size_t tNum = 0;
   char cChar;

   // Read one character from the file
   tNum = OSAL_tFread( &cChar, sizeof(char), 1, psFile );
   if( tNum == 1 )
   {
      // Return byte read
      iReturn = ( int )cChar;
   }
   else
   {
      // Return error
      iReturn = EOF;
   }

   return iReturn;
}

/*******************************************************************************
 *
 *   OSAL_pcFgets
 *
 *******************************************************************************/
char *OSAL_pcFgets( char *pcDest, int iNum, OSAL_FILE_STRUCT *psFile )
{
   register int iChar = EOF;
   register char *pcReturn;

   BOOLEAN bDone = FALSE;

   pcReturn = pcDest;

   if( pcDest != NULL )
   {
      while( ( bDone == FALSE ) && ( iNum > 1 ) )
      {
         iNum--;

         iChar = fgetc( psFile );
         if( iChar == EOF )
         {
            bDone = TRUE;
         }
         else
         {
            *pcReturn++ = iChar;
            if( iChar == '\n' )
            {
               bDone = TRUE;
            }
         }
      }
      *pcReturn = '\0';

      if( ( iChar == EOF ) && ( pcReturn == pcDest ) )
      {
         return NULL;
      }
   }

   return pcDest;
}

/*******************************************************************************
 *
 *   OSAL_iFputc
 *
 *******************************************************************************/
int OSAL_iFputc( int iChar, OSAL_FILE_STRUCT *psFile )
{
   int iResult;
   unsigned char ucChar = ( unsigned char )iChar;

   // Check input
   if( psFile == NULL )
   {
      return EOF;
   }

   // Check if this is for stdout and output disabled.
   // If so, do nothing since output is disabled.
   if( ( psFile == stdout )
#if OSAL_DEBUG == 1
            && ( OSAL.bOutputEnabledThisTask() == FALSE )
#endif
   )
   {
      return 0;
   }

   iResult = OSAL_tFwrite( &ucChar, 1, 1, psFile );
   if( iResult == 1 )
   {
      return ( int )ucChar;
   }

   return EOF;
}

/*******************************************************************************
 *
 *   OSAL_iFputs
 *
 *******************************************************************************/
int OSAL_iFputs( const char *pcSrc, OSAL_FILE_STRUCT *psFile )
{
   int iRetval;

   // Check inputs
   if( ( pcSrc == NULL ) || ( psFile == NULL ) )
   {
      return EOF;
   }

   // Check if this is for stdout and output disabled.
   // If so, do nothing since output is disabled.
   if( ( psFile == stdout )
#if OSAL_DEBUG == 1
            && ( OSAL.bOutputEnabledThisTask() == FALSE )
#endif
   )
   {
      return 0;
   }
   iRetval = OSAL_tFwrite( pcSrc, 1, strlen( pcSrc ), psFile );

   return iRetval;
}

/*******************************************************************************
 *
 *   OSAL_iFPrintf
 *
 *******************************************************************************/
int OSAL_iFPrintf( OSAL_FILE_STRUCT *psFile, const char *pcFormat, ... )
{
   int num = 0; // number of characters generated
   va_list tArgList; // variable arguments list

   // Check input
   if( ( psFile == NULL ) || ( pcFormat == NULL ) )
   {
      return EOF;
   }

   // Check if this is for stdout and output disabled.
   // If so, do nothing since output is disabled.
   if( ( psFile == stdout )
#if OSAL_DEBUG == 1
            && ( OSAL.bOutputEnabledThisTask() == FALSE )
#endif
   )
   {
      return 0;
   }

   // grab variable arguments from the stack
   va_start(tArgList, pcFormat);

   // Send formatted output to stream
   num = _outputf( psFile, pcFormat, &tArgList );

   // restore stack
   va_end(tArgList);

   return num;
}

/*******************************************************************************
 *
 * FUNCTION
 *
 *	OSAL_vDump
 *
 * DESCRIPTION
 *
 *	Note! For this function to work properly you must make sure that the
 *   project options "General Options - Library Options Tab has selected
 *   OSAL_iPrintf formatting as "small" or greater. "tiny" will not work.
 *
 * INPUTS
 *
 *	nArg1		- Describe purpose of argument.
 *				- List of valid inputs
 *
 * RETURNS
 *
 *	int		- Describe return values and meaning
 *				- List of valid values
 *
 *******************************************************************************/
void OSAL_vDump( OSAL_FILE_STRUCT *psFile, UN16 *pun16Row, const void *pvData,
                 UN32 un32Length )
{

#if OSAL_DEBUG == 1

   /*
    Print output similar to as follows:

    00000001: xx xx xx xx xx xx xx xx   xx xx xx xx xx xx xx xx   ..abc..t xyz..ty
    00000002: xx xx xx xx xx xx xx xx   xx xx xx xx xx xx xx xx   ..abc..t xyz..ty
    00000003: xx xx xx xx xx xx xx xx   xx xx xx xx xx xx xx xx   ..abc..t xyz..ty
    */
   UN8 un8ColumnIndex, un8Columns;
   UN16 un16RowIndex, un16Rows, un16RowStart = 0;
   UN32 un32Index = 0;
   const UCHAR *pucData = (const UCHAR *)pvData;
   char acRow[8 + 1 + 1];
   char acHex[2][( 3 * ( STDIO_DUMP_WIDTH / 2 ) ) + 1];
   char acASCII[STDIO_DUMP_WIDTH + 1];
   BOOLEAN bAddresss = FALSE;

   // Ignore if output disabled to stdout
   if (psFile == stdout)
   {
       BOOLEAN bOutputEnabled;

       bOutputEnabled = OSAL.bOutputEnabledThisTask();
       if (bOutputEnabled == FALSE)
       {
           return;
       }
   }

   if( un32Length )
   {
      // First check if they are providing a ROW pointer if not,
      // use our own row.
      if( pun16Row != NULL )
      {
         // Start with this row
         un16RowStart = *pun16Row;
      }
      else
      {
         // Don't use row numbers, instead use an address
         bAddresss = TRUE;

         // Print mast header
         fprintf( psFile, "Hex/ASCII Dump: %d bytes\n", un32Length );
      }

      // Determine the number of rows to print

      // Check if it's an even number of rows
      if( un32Length % STDIO_DUMP_WIDTH )
      {
         // it's not even so divide and add 1
         un16Rows = ( un32Length / STDIO_DUMP_WIDTH ) + 1;
      }
      else
      {
         // it's even so just divide
         un16Rows = un32Length / STDIO_DUMP_WIDTH;
      }

      // For each row...
      for( un16RowIndex = 1; un16RowIndex <= un16Rows; un16RowIndex++ )
      {
         // Report current row back to caller.
         if( pun16Row != NULL )
         {
            // Calculate current row
            *pun16Row = un16RowIndex + un16RowStart;
         }

         // record row(line) number
         if(bAddresss == FALSE)
         {
             // Print row number
             snprintf(acRow, sizeof(acRow), "%8.8d:",
                 un16RowIndex + un16RowStart - 1);
         }
         else
         {
             // Print address
             snprintf(acRow, sizeof(acRow), "%p:",
                 (void *)(pucData + un32Index));
         }

         // Init the number of columns to print based on the remaining length
         if( ( un32Length - un32Index ) < STDIO_DUMP_WIDTH )
         {
            // init with remaining number
            un8Columns = un32Length - un32Index;
         }
         else
         {
            // init with maximum in a row
            un8Columns = STDIO_DUMP_WIDTH;
         }

         // Clear out previous hex and ASCII info by setting the strings
         // to NULL
         acHex[0][0] = acHex[1][0] = acASCII[0] = '\0';

         // Print hex notation in columns for each one
         // also build a string containing the data for ASCII display
            for(un8ColumnIndex = 0; un8ColumnIndex < un8Columns;
                un8ColumnIndex++)
         {
            // Check if this is in the first or second half
            if( un8ColumnIndex < ( STDIO_DUMP_WIDTH / 2 ) )
            {
               // First half
               // Print hex representation
               snprintf( &acHex[0][0] + ( un8ColumnIndex * 3 ),
                            sizeof(acHex[0]) - (un8ColumnIndex * 3),
                            " %2.2x", *(pucData + un32Index));
            }
            else
            {
               // Second half
               // Print hex representation
                    snprintf(
                        &acHex[1][0] + ((un8ColumnIndex -
                            (STDIO_DUMP_WIDTH / 2)) * 3),
                        sizeof(acHex[1]) - ((un8ColumnIndex -
                            (STDIO_DUMP_WIDTH / 2)) * 3),
                        " %2.2x", *(pucData + un32Index));
            }

            // Store string representation if it is printable, otherwise
            // we just store a '.'
            if( isgraph(*(pucData+un32Index)) )
            {
                   snprintf(acASCII + un8ColumnIndex,
                           sizeof(acASCII) - un8ColumnIndex,
                           "%c", *(pucData + un32Index));
            }
            else
            {
                    snprintf(acASCII + un8ColumnIndex,
                        sizeof(acASCII) - un8ColumnIndex, ".");
            }

            // move on to the next one!
            un32Index++;
         }

         // We have now completed a row, so let's print it
         fprintf( psFile, "%9s%-24s  %-24s  %-16s\n", acRow,
                        &acHex[0][0], &acHex[1][0], acASCII );
      }

      // Blank line (if not a successive call)
      if( pun16Row == NULL )
      {
         fprintf( psFile, "\n" );
      }
   }

#endif

   return;
}


/*****************************************************************************
 *
 *       OSAL_iPrintf
 *
 *****************************************************************************/
int OSAL_iPrintf( const char *pcFormat, ... )
{
   int num = 0; // number of characters generated

#if OSAL_DEBUG == 1 && !defined(OSAL_USE_NATIVE_STDIO)
   va_list tArgList; // variable arguments list

   if( pcFormat == NULL )
   {
      return EOF;
   }

   // Check if output enabled
   if( OSAL.bOutputEnabledThisTask() == FALSE )
   {
      return 0;
   }

   // grab variable arguments from the stack
   va_start(tArgList, pcFormat);

   // Send formatted output to stream
   num = _outputf( stdout, pcFormat, &tArgList );

   // restore stack
   va_end(tArgList);
#endif

   return num;
}

/*****************************************************************************
 *
 *       OSAL_iPutchar
 *
 *****************************************************************************/
int OSAL_iPutchar( int iChar )
{

#if OSAL_DEBUG == 1 && !defined(OSAL_USE_NATIVE_STDIO)

    int iResult, iWriteLen;
    char acChars[3];

    // Check if output enabled
    if( OSAL.bOutputEnabledThisTask() == FALSE )
    {
        return iChar;
    }

    // Translate as necessary
    iWriteLen = iTranslateChar((char)iChar, acChars);

    // Write it
    iResult = OSAL_iFputs(acChars, stdout);

    // Check if write succeeded as expected
    if( iResult == iWriteLen )
    {
        return ( int )iChar;
    }

#endif

    return EOF;
}

/*****************************************************************************
 *
 *       OSAL_iPuts
 *
 *****************************************************************************/
int OSAL_iPuts( const char *pcSrc )
{
    int iWritten = 0;

#if OSAL_DEBUG == 1 && !defined(OSAL_USE_NATIVE_STDIO)
    // Check if output enabled
    if( OSAL.bOutputEnabledThisTask() == TRUE )
    {
        int iResult, iWriteLen;
        char acChars[3];

        if (pcSrc == NULL)
        {
            return EOF;
        }

        while(*pcSrc != '\0')
        {
            // Translate as necessary
            iWriteLen = iTranslateChar(*pcSrc++, acChars);

            // Write it
            iResult = OSAL_iFputs(acChars, stdout);

            // Check if write succeeded as expected
            if( iResult == iWriteLen )
            {
                iWritten += iResult;
            }
        };

        // Translate as necessary
        iWriteLen = iTranslateChar('\n', acChars);

        // Write it
        iResult = OSAL_iFputs(acChars, stdout);

        // Check if write succeeded as expected
        if( iResult == iWriteLen )
        {
            iWritten += iResult;
        }
    }
#endif

    return iWritten;
 }

/*******************************************************************************
 *
 *   OSAL_iGetchar
 *
 *******************************************************************************/
int OSAL_iGetchar( void )
{
   int iErr = EOF;

#if OSAL_DEBUG == 1 && !defined(OSAL_USE_NATIVE_STDIO)

   do
   {
      // Attempt to grab a character from the 'stdin' device
      iErr = fgetc( stdin );
      if( iErr == EOF )
      {
         // A timeout expiration means the task needs to simply
         // check-in with the task monitor and continue doing
         // it's business.
         OSAL.eTaskReport( OSAL_TASK_REPORT_NO_ERROR );
      }

    } while(iErr == EOF);

#endif

   return iErr;
}

/*******************************************************************************
 *
 *   OSAL_pcGets
 *
 *******************************************************************************/
char *OSAL_pcGets( char *pcDest )
{
   char *pcReturn = NULL;

#if OSAL_DEBUG == 1 && !defined(OSAL_USE_NATIVE_STDIO)

   char *pcStart = pcDest;
   int iChar;

   if( pcDest == NULL )
   {
      return NULL;
   }

   for( ;; )
   {
      // Grab a char
      iChar = fgetc( stdin );

      // Check for <CR> or EOF
      if( iChar == EOF )
      {
         // A timeout expiration (EOF) means the task needs to simply
         // check-in with the task monitor and continue doing
         // it's business.
         OSAL.eTaskReport( OSAL_TASK_REPORT_NO_ERROR );
      }
      else if( iChar == '\n' )
      {
         // Terminate the string
         *pcDest = '\0';

         // We're done one way or another...
         break;
      }
      else if( iChar == '\b' ) // backspace
      {
         // Remove char from buffer...
         if( pcDest > pcStart )
         {
            pcDest--;
         }
         *pcDest = '\0';
      }
      else if( isprint(iChar) ) // anything printable
      {
         // Write char into buffer...
         *pcDest++ = ( char )iChar;
      }
      else
      {
         // drop
      }
   }

   // Now decide if we return success
   if( iChar != EOF )
   {
      pcReturn = pcStart;
   }

#endif

   return pcReturn;
}


/*****************************************************************************
 *
 *       _outputf
 *
 *****************************************************************************/
static int _outputf( OSAL_FILE_STRUCT *psFile, const char *pcFormat,
                     va_list *ptArgList )
{
    // Declare a static buffer for storing formatted strings (double banked)
    static char _acBuffer[2][_BUFSIZ];

    // Semaphore for protecting static data
    static OSAL_OBJECT_HDL hSemaphoreHdl = OSAL_INVALID_OBJECT_HDL;

    int num = 0; // number of characters generated

    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check if the print semaphore has already been created
    if( hSemaphoreHdl == NULL )
    {
        // Create the semaphore (first time!)
        eReturnCode = OSAL.eSemCreate( &hSemaphoreHdl,
            OSAL_NAME_PREFIX"OSAL_iPrintf",
            1, 1, OSAL_SEM_OPTION_NONE);
        if( eReturnCode != OSAL_SUCCESS )
        {
            // error
            return EOF;
        }
    }

    // Acquire the semaphore
    eReturnCode = OSAL.eSemTake( hSemaphoreHdl, OSAL_OBJ_TIMEOUT_INFINITE );
    if( eReturnCode != OSAL_SUCCESS )
    {
        // error
        return EOF;
    }

    // printing semaphore has been acquired

    // process provided format and variable arguments into the buffer
    // not to exceed the size remaining in the buffer.
    num = vsnprintf( _acBuffer[0], sizeof( _acBuffer[0] ), pcFormat, *ptArgList );

    if ((num > 0) && (num < sizeof(_acBuffer[0])))
    {
        // Call the debug print fxn
        if( psFile != stdout )
        {
            // Use file formatted output
            num = fwrite(_acBuffer[0], sizeof(char), num, psFile );
        }
#if OSAL_DEBUG == 1
        else
        {
            char *pcSrc = _acBuffer[0], *pcDest = _acBuffer[1],
                    *pcStop = _acBuffer[1] + sizeof(_acBuffer[1]) - 2;

            // Use stdout formatted output, with any potential character
            // translation required.
            while( (*pcSrc != '\0') && (pcDest < pcStop) )
            {
                // Translate/expand as necessary
                pcDest += iTranslateChar( *pcSrc, pcDest );
                pcSrc++;
            };

            // Null terminate destination
            *pcDest = '\0';

            // Write translated/expanded buffer
            num = fwrite(_acBuffer[1], sizeof(char), pcDest - _acBuffer[1], psFile);
        }
#endif /* OSAL_DEBUG == 1 */
    }
    else
    {
        num = 0;
    }

    // give up the print semaphore now that we are done with the buffer
    OSAL.eSemGive( hSemaphoreHdl );

    // What to do if the semaphore cannot be returned???

    return num;
}

/*****************************************************************************
 *
 *       bVerifyFileMode
 *
 *       OSAL only allows these modes:
 *
 *       r, r+, rb, rb+
 *       w, w+, wb, wb+
 *       a, a+, ab, ab+
 *
 *****************************************************************************/
static __INLINE__ BOOLEAN bVerifyFileMode (const char *pcMode)
{
    if (pcMode == NULL)
    {
        return FALSE;
    }

    // Verify first character
    if ( (pcMode[0] != 'r') &&
         (pcMode[0] != 'w') &&
         (pcMode[0] != 'a') )
    {
        return FALSE;
    }

    // Verify next character (if necessary)
    if (pcMode[1] != '\0')
    {
        if ( (pcMode[1] != '+') &&
             (pcMode[1] != 'b') )
        {
            return FALSE;
        }

        // Verify next character (if necessary)
        if (pcMode[2] != '\0')
        {
            // 'b' character must be the second
            // character of a three-character mode
            if (pcMode[1] != 'b')
            {
                return FALSE;
            }

            // '+' is the only acceptable
            // third character
            if (pcMode[2] != '+')
            {
                return FALSE;
            }
        }
    }

    return TRUE;
}

#if OSAL_DEBUG == 1
/*****************************************************************************
 *
 *       iTranslateChar
 *
 *****************************************************************************/
static __INLINE__ int iTranslateChar( char cChar, char *pacChars )
{
    int iLen = 0;

    if( cChar == '\n' )
    {
        // Translate
        iLen = 2;
        *pacChars++ = '\r';
        *pacChars++ = '\n';
        *pacChars++ = '\0';
    }
    else if( cChar != '\r' )
    {
        // Pass Through
        iLen = 1;
        *pacChars++ = cChar;
        *pacChars++ = '\0';
    }

    return iLen;
}
#endif
