/******************************************************************************/
/*                     Copyright (c) Sirius XM Radio Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio Inc.           */
/*                           Proprietary & Confidential	                      */
/******************************************************************************/
/*******************************************************************************
*
*  DESCRIPTION
*
*       This module will contain all the OSAL sem APIs.
*
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"
#include "osal_core.h"

#include "os_intf.h"

#include "_osal_sem.h"

/*******************************************************************************
*
*   OSAL_eSemCreate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eSemCreate (
    OSAL_OBJECT_HDL *phSemObj,
    const char *pacName,
    UN32 un32InitialValue,
    UN32 un32Resources,
    UN32 un32Options
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;
    OS_OBJECT_HDL hOS;
    OSAL_SEM_INFO_STRUCT *psSem;
    OSAL_OBJECT_HDL hSemObj;

    // Check input
    if(phSemObj == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phSemObj = OSAL_INVALID_OBJECT_HDL;

    // Verify they have provided valid options
    if( (un32Options & ~OSAL_SEM_OPTION_ALL) != OSAL_SEM_OPTION_NONE)
    {
        return OSAL_ERROR_UNSUPPORTED_OPTION;
    }

    // Make sure they are protecting at least one resource and the initial
    // value does not exceed the the number of resources it's protecting
    if( (un32Resources == 0) || (un32InitialValue > un32Resources) )
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Create the object...
    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_SEM,
        0,
        pacName,
        &hSemObj,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psSem
            );
    if((eReturnCode == OSAL_SUCCESS) && (psSem != NULL))
    {
        // Now we're actually ready to create the object

        // Call OS-Specific API
        eReturnCode = OS.eSemCreate((OSAL_OBJECT_HDL*)&hOS, pacName, un32InitialValue,
                                    un32Resources, un32Options);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Remove and destroy the object
            OSALC_eRemoveObject(hSemObj);
            return eReturnCode;
        }

        // populate the object info
        psSem->un32Resources = un32Resources;
        psSem->un32Options = un32Options;
        psSem->un32Count = un32InitialValue;
        psSem->un32MinCount = un32InitialValue;
        psSem->un32MaxCount = un32InitialValue;
        psSem->un32TakeAttemptCount = 0;

        // Populate the returned OS-Specific object handle
        psObj->hOS = hOS;

        // Populate the object
        *phSemObj = hSemObj;

        // Semaphore created successfully
        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eSemDelete
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eSemDelete (
    OSAL_OBJECT_HDL hSemObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hSemObj, OSAL_OBJECT_TYPE_SEM);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Call OS specific API
    eReturnCode = OS.eSemDelete((OSAL_OBJECT_HDL)psObj->hOS);
    if(eReturnCode == OSAL_SUCCESS)
    {
        psObj->hOS = OS_INVALID_OBJECT_HDL;

        // Remove and destroy the object
        eReturnCode = OSALC_eRemoveObject(hSemObj);
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eSemGive
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eSemGive (
    OSAL_OBJECT_HDL hSemObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    OSAL_OBJECT_STRUCT *psObj;
    BOOLEAN bGive = FALSE;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hSemObj, OSAL_OBJECT_TYPE_SEM);
    if(psObj != NULL)
    {
        OSAL_SEM_INFO_STRUCT *psSem;

        // Extract semaphore info
        psSem = &psObj->puInfo->sSem;

        /* Enter Critical Section */
        OSAL.vEnterCriticalSection();

        // Check if we can really give the semaphore
        if((psSem->un32Resources - psSem->un32Count) > 0)
        {
            // Yep, increment count
            psSem->un32Count++;

            // Update statistics as necessary
            if(psSem->un32Count > psSem->un32MaxCount)
            {
                psSem->un32MaxCount = psSem->un32Count;
            }
            if(psSem->un32Count < psSem->un32MinCount)
            {
                psSem->un32MinCount = psSem->un32Count;
            }

            // Give-up semaphore
            bGive = TRUE;
        }
        else
        {
            // Can't give any more than you took
            eReturnCode = OSAL_SUCCESS;
        }

        /* Exit Critical Section */
        OSAL.vExitCriticalSection();

        // Call OS specific API
        if(bGive == TRUE)
        {
            eReturnCode = OS.eSemGive((OSAL_OBJECT_HDL)psObj->hOS);
        }
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eSemTake
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eSemTake (
    OSAL_OBJECT_HDL hSemObj,
    N32 n32Timeout
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hSemObj, OSAL_OBJECT_TYPE_SEM);
    if(psObj != NULL)
    {
        OSAL_SEM_INFO_STRUCT *psSem;

        // Extract semaphore info
        if(psObj->puInfo != NULL)
        {
            psSem = &psObj->puInfo->sSem;
            if(psSem == NULL)
            {
                return OSAL_ERROR_INVALID_POINTER;
            }
        }
        else
        {
            return OSAL_ERROR_INVALID_POINTER;
        }

        // Call OS specific API
        eReturnCode = OS.eSemTake((OSAL_OBJECT_HDL)psObj->hOS, n32Timeout);

        /* Enter Critical Section */
        OSAL.vEnterCriticalSection();

        // Count the number of times this API has been invoked
        psSem->un32TakeAttemptCount++;

        // Was pend successful?
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Yep, decrement count
            psSem->un32Count--;

            // Update statistics as necessary
            if(psSem->un32Count > psSem->un32MaxCount)
            {
                psSem->un32MaxCount = psSem->un32Count;
            }
            if(psSem->un32Count < psSem->un32MinCount)
            {
                psSem->un32MinCount = psSem->un32Count;
            }
        }
        else if(eReturnCode == OSAL_SEM_NOT_AVAILABLE)
        {
            psSem->un32NotAvailableCount++;
        }
        else if(eReturnCode == OSAL_TIMEOUT)
        {
            psSem->un32TimeoutCount++;
        }

        /* Exit Critical Section */
        OSAL.vExitCriticalSection();
    }
    else
    {
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_hSemGetHandleByName
*
*******************************************************************************/
OSAL_OBJECT_HDL OSAL_hSemGetHandleByName (
    const char *pacName
        )
{
    OSAL_OBJECT_HDL hObjectHdl = OSAL_INVALID_OBJECT_HDL;
    OSAL_OBJECT_STRUCT *psObj;

    // Search for the object by name
    psObj = OSALC_psFindObjectFromName(OSAL_OBJECT_TYPE_SEM, pacName);
    if(psObj != NULL)
    {
        hObjectHdl = (OSAL_OBJECT_HDL)psObj;
    }

    return hObjectHdl;
}

/*******************************************************************************
*
*   OSAL_eSemGetInfo
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eSemGetInfo (
    OSAL_OBJECT_HDL hSemObj,
    OSAL_SEM_INFO_STRUCT *psSemInfo
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Check provided structure pointer
    if(psSemInfo == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // List is now locked, cocked and ready to rock

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hSemObj, OSAL_OBJECT_TYPE_SEM);
    if(psObj != NULL)
    {
        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        OSAL.vEnterCriticalSection();
        *psSemInfo = psObj->puInfo->sSem;
        OSAL.vExitCriticalSection();

        eReturnCode = OSAL_SUCCESS;
    }
    else
    {
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eSemList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eSemList ( void )
{
#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hSemList;
    UN32 un32Semaphores = 0, un32SemaphoreNumber = 0;
    OSAL_STATISTICS_STRUCT sStatistics;

    // Iterate the list of semaphores and list each entry

    // Grab the semaphore linked list
    hSemList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_SEM);
    if(hSemList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of semaphores in the list
    eReturnCode = OSAL.eLinkedListItems(hSemList, &un32Semaphores);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_SEM, &sStatistics);

    // Check that one or more elements exist in the list
    if(un32Semaphores > 0)
    {
        // Print title
        printf("=======================\n");
        printf("| OSAL Semaphore List |\n");
        printf("=======================\n");
    }

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hSemList,
        OSALS_bPrintSemaphore, &un32SemaphoreNumber);
    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else
    {
        // Print summary
        printf("%u Semaphores(s) currently allocated by the system.\n",
                    un32Semaphores);

        // Print statistics
        printf("Current = %u, Minimum = %u, Maximum = %u\n",
               sStatistics.un32Current,
               sStatistics.un32Minimum,
               sStatistics.un32Maximum );

    }

    return OSAL_SUCCESS;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

#if (OSAL_DEBUG == 1)
/*******************************************************************************
*
*   OSALS_bPrintSemaphore
*
*******************************************************************************/
BOOLEAN OSALS_bPrintSemaphore ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData,
        *psCreatorObj;
    UN32 *pun32SemaphoreNumber = (UN32 *)pvArg;
    const char *pacCreatorObjName = "Unknown";

    // Check if argument is valid
    if(pun32SemaphoreNumber == NULL)
    {
        return FALSE;
    }

    // Prepare semaphore number
    (*pun32SemaphoreNumber)++;

    // Check if object is valid
    if(psObj != NULL)
    {
        // Point to this object's info
        OSAL_SEM_INFO_STRUCT sSem;

        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        sSem = psObj->puInfo->sSem;

        // Extract the creator task's name
        psCreatorObj = (OSAL_OBJECT_STRUCT *)psObj->hCreatorObj;
        if(psCreatorObj != NULL)
        {
            // Extract name of creator
#if OSAL_OBJECT_TRACKING == 1
            pacCreatorObjName = psCreatorObj->pacName;
#else
            pacCreatorObjName = "Unknown";
#endif
        }

        printf("Semaphore #%u\n", *pun32SemaphoreNumber);
        printf("Name = '%s' (hdl = 0x%p)\n",
#if OSAL_OBJECT_TRACKING == 1
               psObj->pacName,
#else
               "Unknown",
#endif
               psObj
                   );
        printf("\tCreator = %s\n", pacCreatorObjName);
        printf("\tOptions = %#010x\n",
               sSem.un32Options);
        printf("\tTotal Resources = %u, Consumed = %u\n",
               sSem.un32Resources, sSem.un32Count);
        printf("\tConsumed: Minimum = %u, Maximum = %u\n",
               sSem.un32MinCount, sSem.un32MaxCount);
        printf("\tTake unavailable = %u, Take timeouts = %u\n",
            sSem.un32NotAvailableCount, sSem.un32TimeoutCount);
        puts("");
    }

    // Keep iterating
    return TRUE;
}

#endif /* (OSAL_DEBUG == 1) */
