/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"

#include "osal_core.h"
#include "osal_ll.h"
#include "_osal_queue.h"

/*******************************************************************************
*
*   OSAL_eQueueCreate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eQueueCreate (
    OSAL_OBJECT_HDL *phQueueObj,
    const char *pacName,
    UN32 un32QueueDepth, UN32 un32MessageSize,
    UN32 un32Options
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hQueueObj;
    OSAL_QUEUE_PRIVATE_INFO_STRUCT *psQueue;
    OSAL_QUEUE_MSG_STRUCT *psQueueMsg;
    size_t tMessageObjectSize, tAlignedMessageSize, tLLExtraSize;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    UN32 un32Msg;

    // Check input
    if (phQueueObj == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phQueueObj = OSAL_INVALID_OBJECT_HDL;

    // Verify they have provided valid options
    if ( (un32Options & ~OSAL_QUEUE_OPTION_ALL) != OSAL_QUEUE_OPTION_NONE )
    {
        return OSAL_ERROR_UNSUPPORTED_OPTION;
    }

    // Either Fixed or Variable sized queues must be selected. If neither
    // was selected we default to fixed size according to the documentation
    if ((un32Options & (OSAL_QUEUE_OPTION_FIXED_SIZE |
                        OSAL_QUEUE_OPTION_VARIABLE_SIZE)) == 0)
    {
        un32Options |= OSAL_QUEUE_OPTION_FIXED_SIZE;
    }

    // Make sure they have provided a request for a queue of some size
    // and some depth.
    if ((un32QueueDepth == 0) || (un32MessageSize == 0))
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Always align size request with the native
    // underlying machine alignment. Makes for cleaner even boundaries
    // allocations.
    tAlignedMessageSize = un32MessageSize;
    ALIGN_SIZE(tAlignedMessageSize);

    // Request size required to keep list of the entries
    tLLExtraSize = OSAL_tLinkedListEntrySize();

    // Compute the actual size of each message we need. This includes
    // both the requested message size, plus the structure (header)
    // for each message.
    tMessageObjectSize = tLLExtraSize +
                         sizeof(OSAL_QUEUE_MSG_STRUCT) +
                         tAlignedMessageSize;

    // Create the object to hold this OS-object and manage it
    // Also ask for additional memory to hold the messages in the
    // queue as well. This is all one big hunk of memory which we will
    // divide up as needed.
    // This method always zero initializes the memory.
    // Create the object...
    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_QUEUE,
        tMessageObjectSize * un32QueueDepth,
        pacName,
        &hQueueObj,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psQueue
            );
    if ((eReturnCode == OSAL_SUCCESS) && (psQueue != NULL))
    {
        OSAL_QUEUE_PRIVATE_INFO_STRUCT *psPrivate =
            &psObj->puInfo->sQueue;

        // No OS-specific implementation required. Yea!
        psObj->hOS = OS_INVALID_OBJECT_HDL;

        // Initialize object structure...
        psQueue->sPublicInfo.un32QueueDepth = un32QueueDepth;
        psQueue->sPublicInfo.un32MessageSize = un32MessageSize;
        psQueue->sPublicInfo.un32Options = un32Options;
        psQueue->sPublicInfo.un32MessagesAvailable = un32QueueDepth;
        psQueue->sPublicInfo.un32MessagesQueued = 0;
        psQueue->sPublicInfo.un32MessagesTaken = 0;
        psQueue->sPublicInfo.un32FailedAllocations = 0;

        // Determine if this queue will be used in an interrupt
        if ((un32Options & OSAL_QUEUE_OPTION_INTERRUPT) ==
             OSAL_QUEUE_OPTION_INTERRUPT)
        {
            // Since this queue is intended to be used in an interrupt
            // we need to choose Interrupt for entering/exiting exclusive access.
            psPrivate->bEnterExclusiveAccess =
                bEnterExclusiveAccessInterrupt;
            psPrivate->vExitExclusiveAccess =
                vExitExclusiveAccessInterrupt;

            // Queue does not need a mutex
            psPrivate->hMutex =
                OSAL_INVALID_OBJECT_HDL;
        }
        else
        {
            // Since this queue will not be used in an interrupt
            // we need to choose TaskOnly for entering/exiting exclusive access.
            psPrivate->bEnterExclusiveAccess =
                bEnterExclusiveAccessTaskOnly;
            psPrivate->vExitExclusiveAccess =
                vExitExclusiveAccessTaskOnly;

            // Construct an appropriate name for semaphore to be created
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     OSAL_NAME_PREFIX"%s:QueueMutex", pacName);

            // Queue needs a mutex
            eReturnCode = OSAL.eSemCreate(
                &psPrivate->hMutex,
                &acName[0], 1, 1, OSAL_SEM_OPTION_NONE);
            if (eReturnCode != OSAL_SUCCESS)
            {
                OSAL_eQueueDelete(hQueueObj);
                return OSAL_ERROR;
            }
        }

        // Construct an appropriate name for the message alloc semaphore
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 OSAL_NAME_PREFIX"%s:QAlloc", pacName);

        // Allocate a semaphore which is used to allow/limit message
        // allocations. This counting semaphore is the same size as the
        // depth of the queue requested. We initialize message queue
        // with all messages available (initial value == resources)
        eReturnCode = OSAL.eSemCreate(
            &psPrivate->hMessageAllocSem,
            &acName[0],
            un32QueueDepth,
            un32QueueDepth,
            OSAL_SEM_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Delete queue
            OSAL_eQueueDelete(hQueueObj);
            // Return problem with creating the semaphore
            return eReturnCode;
        }

        // Construct an appropriate name for the queue pend semaphore
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 OSAL_NAME_PREFIX"%s:QPend", pacName);

        // Allocate a semaphore which is used to allow a caller to pend
        // on a message becoming available (or posted) onto the queue.
        // Initialize message queue with no messages posted (initial value == 0)
        // The maximum number of messages which can be on this queue is the
        // requested queue depth.
        eReturnCode = OSAL.eSemCreate(
            &psPrivate->hMessagePendSem,
            &acName[0],
            0,
            un32QueueDepth,
            OSAL_SEM_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Delete queue
            OSAL_eQueueDelete(hQueueObj);
            // Return problem with creating the semaphore
            return eReturnCode;
        }

        // Construct an appropriate name for the message list. This
        // is the list of all used messages in the queue. This list
        // is kept as a sorted list.
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 OSAL_NAME_PREFIX"%s:MsgQ", pacName);

        // Create a linked list which will hold a pointer to each
        // of the allocated and queued messages in the queue.
        eReturnCode = OSAL.eLinkedListCreate(
            &psPrivate->hQueuedMessageList,
            &acName[0],
            NULL,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Delete queue
            OSAL_eQueueDelete(hQueueObj);
            // Return problem with creating the linked list
            return eReturnCode;
        }

        // Construct an appropriate name for the free-to-use message list..
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            OSAL_NAME_PREFIX"%s:MsgF", pacName);

        // Create a linked list which will hold a pointer to each
        // of the free-to-use messages available in the queue.
        eReturnCode = OSAL.eLinkedListCreate(
            &psPrivate->hFreeMessageList,
            &acName[0],
            NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Delete queue
            OSAL_eQueueDelete(hQueueObj);
            // Return problem with creating the linked list
            return eReturnCode;
        }

        // Create a linked list which will hold a pointer to each
        // of the allocated but not queued messages in the queue.
        eReturnCode = OSAL.eLinkedListCreate(
            &psPrivate->hAllocatedMessageList,
            &acName[0],
            NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Delete queue
            OSAL_eQueueDelete(hQueueObj);
            // Return problem with creating the linked list
            return eReturnCode;
        }

        // Point to the start of the message memory (just after
        // the queue structure itself). This is where all the messages
        // begin, including both the header and payload of each message.
        psQueueMsg = (OSAL_QUEUE_MSG_STRUCT *)((UN8*)(psQueue + 1) + tLLExtraSize);

        // Populate the linked list of messages with the pointer for
        // each message. Initialize each message at the same time.
        for(un32Msg = 0; un32Msg < un32QueueDepth; un32Msg++)
        {
            // Initialize Message Header (all as available)
            *psQueueMsg = GsQueueMsgDefault;
            psQueueMsg->hQueueObj = hQueueObj; // Queue it belongs to

            // Add this message pointer to the linked list of messages
            eReturnCode = OSAL.eLinkedListAdd(
                psPrivate->hFreeMessageList,
                &psQueueMsg->hThisEntry,
                psQueueMsg);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // Delete queue
                OSAL_eQueueDelete(hQueueObj);
                // Return problem with populating the linked list
                return eReturnCode;
            }

            // Compute next message pointer
            psQueueMsg = (OSAL_QUEUE_MSG_STRUCT *)
                ((UN8 *)psQueueMsg + tMessageObjectSize);
        }

        // Populate the object
        *phQueueObj = hQueueObj;

        // Object created successfully
        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eQueueDelete
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eQueueDelete (
    OSAL_OBJECT_HDL hQueueObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_QUEUE_PRIVATE_INFO_STRUCT *psPrivate;
    void (*vExitExclusiveAccess)(OSAL_OBJECT_HDL hMutex);
    BOOLEAN bOk;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hQueueObj, OSAL_OBJECT_TYPE_QUEUE);
    if (psObj == NULL)
    {
        // Error!
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract the info pointers for the object
    psPrivate = &psObj->puInfo->sQueue;

    /***************************/
    bOk = psPrivate->bEnterExclusiveAccess(psPrivate->hMutex);
    /***************************/
    // Trying to enter exclusive access mode. If we fail, it could mean
    // that the queue is already deleted from another thread, so bail.
    if (bOk == FALSE)
    {
        return OSAL_ERROR_CANNOT_DELETE_OBJECT;
    }

    // Delete resource (if it exists)
    if (psPrivate->hQueuedMessageList != OSAL_INVALID_OBJECT_HDL)
    {
        // Re-initialize message header in list first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psPrivate->hQueuedMessageList,
            vInitializeMessageHeader );
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Delete the object
            eReturnCode = OSAL.eLinkedListDelete(
                                psPrivate->hQueuedMessageList);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate object handle
                psPrivate->hQueuedMessageList = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Delete resource (if it exists)
    if (psPrivate->hFreeMessageList != OSAL_INVALID_OBJECT_HDL)
    {
        // Re-initialize message header in list first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psPrivate->hFreeMessageList,
            vInitializeMessageHeader );
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Delete the object
            eReturnCode = OSAL.eLinkedListDelete(
                            psPrivate->hFreeMessageList);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate object handle
                psPrivate->hFreeMessageList = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Delete resource (if it exists)
    if (psPrivate->hAllocatedMessageList != OSAL_INVALID_OBJECT_HDL)
    {
        // Re-initialize message header in list first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psPrivate->hAllocatedMessageList,
            vInitializeMessageHeader );
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Delete the object
            eReturnCode = OSAL.eLinkedListDelete(
                            psPrivate->hAllocatedMessageList);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate object handle
                psPrivate->hAllocatedMessageList = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Delete resource (if it exists)
    if (psPrivate->hMessagePendSem != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete the object
        eReturnCode = OSAL.eSemDelete(psPrivate->hMessagePendSem);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate object handle
            psPrivate->hMessagePendSem = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Delete resource (if it exists)
    if (psPrivate->hMessageAllocSem != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete the object
        eReturnCode = OSAL.eSemDelete(psPrivate->hMessageAllocSem);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate object handle
            psPrivate->hMessageAllocSem = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // If the mutex hasn't been destroyed yet, destroy it now
    if (psPrivate->hMutex != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(psPrivate->hMutex);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate mutex handle
            psPrivate->hMutex = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Safe exit exclusive access function
    vExitExclusiveAccess = psPrivate->vExitExclusiveAccess;

    // Invalidate appropriate field
    psPrivate->vExitExclusiveAccess = NULL;

    // Remove and destroy the object
    eReturnCode = OSALC_eRemoveObject(hQueueObj);

    /*************************************/
    /* Exit the Exclusive Access section */
    if (vExitExclusiveAccess != NULL)
    {
        vExitExclusiveAccess(OSAL_INVALID_OBJECT_HDL);
    }
    /*************************************/

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eQueueGet
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eQueueGet (
    OSAL_OBJECT_HDL hQueueObj,
    void **ppvMessage, UN32 *pun32MessageSize,
    N32 n32Timeout
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_QUEUE_MSG_STRUCT *psQueueMsg;
    OSAL_QUEUE_PRIVATE_INFO_STRUCT *psPrivate;

    // Check input parameters
    if ((ppvMessage == NULL) || (pun32MessageSize == NULL))
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hQueueObj, OSAL_OBJECT_TYPE_QUEUE);
    if (psObj == NULL)
    {
        // Error!
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract the info pointers for the object
    psPrivate = &psObj->puInfo->sQueue;

    // Wait on queue for timeout provided
    eReturnCode = OSAL.eSemTake(psPrivate->hMessagePendSem, n32Timeout);
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        BOOLEAN bOk;

        // Enter an exclusive access section to de-queue the next available
        // message from the list.
        bOk = psPrivate->bEnterExclusiveAccess(psPrivate->hMutex);
        if (bOk == FALSE)
        {
            // Cannot enter exclusive access section. Aborting.
            OSAL.eSemGive(psPrivate->hMessagePendSem);
            return OSAL_ERROR;
        }

        // A message is available, go get it now. The first message
        // in the list is always the next available and most recent urgent
        // message to retrieve.
        hEntry = OSAL.hLinkedListFirst(
            psPrivate->hQueuedMessageList, (void**)&psQueueMsg);
        if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Check that message has the right flags. The message must
            // be both ALLOCATED and QUEUED
            if ((psQueueMsg->un8Flags &
                 (OSAL_MSG_FLAG_ALLOCATED | OSAL_MSG_FLAG_QUEUED)) ==
               (OSAL_MSG_FLAG_ALLOCATED | OSAL_MSG_FLAG_QUEUED))
            {
                // Clear QUEUED flag
                psQueueMsg->un8Flags &= ~(OSAL_MSG_FLAG_QUEUED);

                // Move this message as simply allocated one...
                eReturnCode = eMoveMsgToList(psQueueMsg,
                                psPrivate->hAllocatedMessageList,
                                FALSE);
                if (eReturnCode == OSAL_SUCCESS)
                {
                    // Extract the size of this message and
                    // return to the caller
                    *pun32MessageSize = psQueueMsg->tSize;

                    // The actual pointer we return back to the caller is one
                    // which points to the payload area, not the object itself.
                    // So we now calculate the proper return value.
                    *ppvMessage = (psQueueMsg + 1);
                }
            }
            else
            {
                // Something is wrong with OSAL
                eReturnCode = OSAL_ERROR;
            }
        }

        // Exit exclusive access section now
        psPrivate->vExitExclusiveAccess(psPrivate->hMutex);
    }
    else if (eReturnCode == OSAL_SEM_NOT_AVAILABLE)
    {
        // Re-process return to something more meaningful
        eReturnCode = OSAL_QUEUE_EMPTY;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eQueuePut
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eQueuePut (
    const void *pvMessage,
    UN32 un32MessageSize
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_QUEUE_MSG_STRUCT *psQueueMsg;
    OSAL_OBJECT_HDL hQueueObj;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_QUEUE_INFO_STRUCT *psQueue;
    OSAL_QUEUE_PRIVATE_INFO_STRUCT *psPrivate;

    // Check that input is valid
    if (pvMessage == NULL)
    {
        // Error!
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract from the provided message pointer the actual
    // message object pointer.
    psQueueMsg = ((OSAL_QUEUE_MSG_STRUCT *)pvMessage - 1);

    // Now that we have a pointer to the message object we need to extract
    // the queue handle from which this object belongs
    hQueueObj = psQueueMsg->hQueueObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hQueueObj, OSAL_OBJECT_TYPE_QUEUE);
    if (psObj != NULL)
    {
        // Extract the info pointers for the object
        psPrivate = &psObj->puInfo->sQueue;
        psQueue = &psPrivate->sPublicInfo;

        // Verify the size provided does not exceed the embedded message
        // size provided when it was allocated. Also if the 'fixed' size
        // flag was set when the queue was created, the posted message
        // must equal the allocated message size regardless of what the
        // caller provides.
        if (un32MessageSize <= psQueue->un32MessageSize)
        {
            // Verify the message to be posted has the appropriate flags set.

            // It must be ALLOCATED
            if ((psQueueMsg->un8Flags & OSAL_MSG_FLAG_ALLOCATED) ==
                OSAL_MSG_FLAG_ALLOCATED)
            {
                // It must NOT be QUEUED
                if ((psQueueMsg->un8Flags & OSAL_MSG_FLAG_QUEUED) ==
                    OSAL_MSG_FLAG_NONE)
                {
                    BOOLEAN bOk;

                    // Populate message size field
                    if ((psQueue->un32Options & OSAL_QUEUE_OPTION_FIXED_SIZE) ==
                         OSAL_QUEUE_OPTION_FIXED_SIZE)
                    {
                        // Set the length of this message to be posted based on
                        // the allocated size at queue creation
                        psQueueMsg->tSize = psQueue->un32MessageSize;
                    }
                    else
                    {
                        // Set the length of this message to be posted
                        // based on what the caller provided
                        psQueueMsg->tSize = un32MessageSize;
                    }

                    // Set QUEUED flag
                    psQueueMsg->un8Flags |= OSAL_MSG_FLAG_QUEUED;

                    // Enter an exclusive access section to en-queue this
                    // message onto the list.
                    bOk = psPrivate->bEnterExclusiveAccess(psPrivate->hMutex);
                    if (bOk == TRUE)
                    {
                        // Enqueue message (or at least replace it in the list anyway)
                        eReturnCode = eMoveMsgToList(psQueueMsg,
                            psPrivate->hQueuedMessageList,
                            TRUE);

                        // Exit exclusive access section now
                        psPrivate->vExitExclusiveAccess(psPrivate->hMutex);
                    }

                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        // Post a semaphore to indicate to the queue something is
                        // available for the taking.
                        eReturnCode = OSAL.eSemGive(psPrivate->hMessagePendSem);
                        if (eReturnCode != OSAL_SUCCESS)
                        {
                            // Can't put any more than you took. This probably can't
                            // happen, but might is someone tries to queue something
                            // bogus.
                            eReturnCode = OSAL_ERROR_INVALID_INPUT;
                        }
                    }
                    else
                    {
                        // Something is wrong with OSAL
                        eReturnCode = OSAL_ERROR;
                    }
                }
                else
                {
                    // Error! Message has already been queued.
                    eReturnCode = OSAL_QUEUE_MESSAGE_ALREADY_QUEUED;
                }
            }
            else
            {
                // Error! Message has not been allocated.
                eReturnCode = OSAL_ERROR_MESSAGE_NOT_ALLOCATED;
            }
        }
        else
        {
            // Incorrect message size was provided or caller provided something
            // which was NOT an allocated message to post. Something is wrong.
            eReturnCode = OSAL_ERROR_INVALID_INPUT;
        }
    }
    else
    {
        // Invalid object handle was provided
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_hQueueGetHandleByName
*
*******************************************************************************/
OSAL_OBJECT_HDL OSAL_hQueueGetHandleByName (
    const char *pacName
        )
{
    OSAL_OBJECT_HDL hObjectHdl = OSAL_INVALID_OBJECT_HDL;
    OSAL_OBJECT_STRUCT *psObj;

    psObj = OSALC_psFindObjectFromName(OSAL_OBJECT_TYPE_QUEUE, pacName);
    if (psObj != NULL)
    {
        hObjectHdl = (OSAL_OBJECT_HDL)psObj;
    }

    return hObjectHdl;
}

/*******************************************************************************
*
*   OSAL_eQueueGetInfo
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eQueueGetInfo (
    OSAL_OBJECT_HDL hQueueObj,
    OSAL_QUEUE_INFO_STRUCT *psQueueInfo
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode =
        OSAL_ERROR_INVALID_HANDLE;
    OSAL_OBJECT_STRUCT *psObj;

    // Check provided structure pointer
    if (psQueueInfo == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hQueueObj, OSAL_OBJECT_TYPE_QUEUE);
    if (psObj != NULL)
    {
        OSAL_QUEUE_PRIVATE_INFO_STRUCT *psPrivate;
        BOOLEAN bOk;

        // Extract the info pointers for the object
        psPrivate = &psObj->puInfo->sQueue;

        // Initialize return code
        eReturnCode = OSAL_ERROR;

        // Allow exclusive access of the queue structure
        bOk = psPrivate->bEnterExclusiveAccess(psPrivate->hMutex);
        if (bOk == TRUE)
        {
            // Grab semaphore info
            OSAL.eSemGetInfo(psPrivate->hMessageAllocSem, &psQueueInfo->sAllocSemInfo);
            OSAL.eSemGetInfo(psPrivate->hMessagePendSem, &psQueueInfo->sPendSemInfo);

            // Copy queue information locally
            psQueueInfo->un32QueueDepth = psPrivate->sPublicInfo.un32QueueDepth;
            psQueueInfo->un32MessageSize = psPrivate->sPublicInfo.un32MessageSize;
            psQueueInfo->un32Options = psPrivate->sPublicInfo.un32Options;

            // Exit exclusive access section now
            psPrivate->vExitExclusiveAccess(psPrivate->hMutex);

            // Make necessary calculations
            psQueueInfo->un32MessagesQueued =
                psQueueInfo->sPendSemInfo.un32Count;
            psQueueInfo->un32MessagesAvailable =
                psQueueInfo->sAllocSemInfo.un32Count;
            psQueueInfo->un32MessagesTaken =
                psQueueInfo->sAllocSemInfo.un32Resources -
                    psQueueInfo->sAllocSemInfo.un32Count;
            psQueueInfo->un32FailedAllocations =
                psQueueInfo->sAllocSemInfo.un32NotAvailableCount +
                psQueueInfo->sAllocSemInfo.un32TimeoutCount;

            eReturnCode = OSAL_SUCCESS;
        }
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eQueueList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eQueueList ( void )
{
#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hQueueList;
    UN32 un32Queues = 0, un32QueueNumber = 0;
    OSAL_STATISTICS_STRUCT sStatistics;

    // Iterate the list of queues and list each entry

    // Grab the queue linked list
    hQueueList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_QUEUE);
    if (hQueueList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of queues in the list
    eReturnCode = OSAL.eLinkedListItems(hQueueList, &un32Queues);
    if (eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_QUEUE, &sStatistics);

    // Check that one or more elements exist in the list
    if (un32Queues > 0)
    {
        // Print title
        puts("===================");
        puts("| OSAL Queue List |");
        puts("===================\n");
    }

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hQueueList,
        OSALQ_bPrintQueue, &un32QueueNumber);
    if (eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return eReturnCode;
    }
    else if (eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else
    {
        // Print summary
        printf("%u Queue(s) currently allocated by the system.\n",
                    un32Queues);

        // Print statistics
        printf("Current = %u, Minimum = %u, Maximum = %u\n",
               sStatistics.un32Current,
               sStatistics.un32Minimum,
               sStatistics.un32Maximum );

    }

    return OSAL_SUCCESS;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

#if OSAL_DEBUG == 1

/*******************************************************************************
*
*   OSALQ_bPrintQueue
*
*******************************************************************************/
BOOLEAN OSALQ_bPrintQueue ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData,
        *psCreatorObj;
    UN32 *pun32QueueNumber = (UN32 *)pvArg, un32QueueNumber = 0;
    const char *pacCreatorObjName = "Unknown";

    // Check if argument is valid, if so increment it
    if (pun32QueueNumber != NULL)
    {
        // Prepare queue number
        un32QueueNumber = (*pun32QueueNumber)++;
    }

    // Check if object is valid
    if (psObj != NULL)
    {
        // Point to this object's info
        OSAL_QUEUE_INFO_STRUCT sQueue;

        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        OSAL.eQueueGetInfo((OSAL_OBJECT_HDL)psObj, &sQueue);

        // Extract the creator queue's name
        psCreatorObj = (OSAL_OBJECT_STRUCT *)psObj->hCreatorObj;
        if (psCreatorObj != NULL)
        {
            // Extract name of creator
#if OSAL_OBJECT_TRACKING == 1
            pacCreatorObjName = psCreatorObj->pacName;
#else
            pacCreatorObjName = "Unknown";
#endif
        }

        // extract information we are interested in, and print
        printf("Queue #%u\n", un32QueueNumber);
        printf("Name = '%s' (hdl = 0x%p)\n",
#if OSAL_OBJECT_TRACKING == 1
               psObj->pacName,
#else
               "Unknown",
#endif
               psObj
                   );
        printf("\tCreator = %s\n", pacCreatorObjName);
        printf("\tOptions = %#010x\n",
               sQueue.un32Options);
        printf("\tDepth = %u, Size = %u, Queued = %u\n",
               sQueue.un32QueueDepth, sQueue.un32MessageSize,
               sQueue.un32MessagesQueued);
        printf("\tAvailable = %u, Taken = %u\n",
               sQueue.un32MessagesAvailable, sQueue.un32MessagesTaken);
        printf("\tMaximum number of messages allocated = %u\n",
            sQueue.sAllocSemInfo.un32Resources -
            sQueue.sAllocSemInfo.un32MinCount);
        printf("\tMaximum number of messages queued = %u\n",
            sQueue.sPendSemInfo.un32MaxCount);
        printf("\tNumber of failed allocation attempts = %u\n",
            sQueue.un32FailedAllocations);
        printf("\tMessage Memory = 0x%p\n", &sQueue + 1);
        puts("");
    }

    // Keep iterating
    return TRUE;
}

#endif /* (OSAL_DEBUG == 1)  */

/*******************************************************************************
*
*   OSAL_eMessageAllocate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eMessageAllocate (
    OSAL_OBJECT_HDL hQueueObj,
    void **ppvMessage,
    UN32 un32Flags
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Check input parameter
    if (ppvMessage == NULL)
    {
        // Error!
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Verify they have provided valid flags
    if ((un32Flags & ~OSAL_QUEUE_FLAG_ALL) != OSAL_QUEUE_FLAG_NONE)
    {
        // Error!
        return OSAL_ERROR_UNSUPPORTED_FLAG;
    }

    // Verify that either the BLOCK or NONBLOCK flag is set. If neither
    // is set, then make sure the default (OSAL_QUEUE_FLAG_NONBLOCK) is set
    if ((un32Flags & (OSAL_QUEUE_FLAG_BLOCK | OSAL_QUEUE_FLAG_NONBLOCK))
        == OSAL_QUEUE_FLAG_NONE)
    {
        // Make sure default case is set
        un32Flags |= OSAL_QUEUE_FLAG_NONBLOCK;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hQueueObj, OSAL_OBJECT_TYPE_QUEUE);
    if (psObj != NULL)
    {
        OSAL_QUEUE_PRIVATE_INFO_STRUCT *psPrivate;
        N32 n32Timeout;

        // Check if this allocation is to be blocked or not
        if ((un32Flags & OSAL_QUEUE_FLAG_BLOCK) == OSAL_QUEUE_FLAG_BLOCK)
        {
            // Block if none available
            n32Timeout = OSAL_OBJ_TIMEOUT_INFINITE;
        }
        else
        {
            // apply default case
            n32Timeout = OSAL_OBJ_TIMEOUT_NONE;
        }

        // Extract the info pointers for the object
        psPrivate = &psObj->puInfo->sQueue;

        // Wait for an available message to allocate with the determined
        // timeout value required based on provided flags.
        eReturnCode = OSAL.eSemTake(psPrivate->hMessageAllocSem, n32Timeout);
        if (eReturnCode == OSAL_SUCCESS)
        {
            BOOLEAN bOk;

            // Enter an exclusive access section to retrieve the next available
            // message for allocation from the list.
            bOk = psPrivate->bEnterExclusiveAccess(psPrivate->hMutex);
            // If we are able to enter exclusive access section, then proceed
            if (bOk == TRUE)
            {
                OSAL_LINKED_LIST_ENTRY hEntry;
                OSAL_QUEUE_MSG_STRUCT *psQueueMsg;

                // A message is available, go get it now.
                hEntry =
                    OSAL.hLinkedListFirst(psPrivate->hFreeMessageList,
                        (void**)&psQueueMsg);
                if ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
                    (hEntry == psQueueMsg->hThisEntry))
                {
                    // Verify message found has the right flags. The message
                    // must not be ALLOCATED or QUEUED
                    if ((psQueueMsg->un8Flags &
                         (OSAL_MSG_FLAG_QUEUED | OSAL_MSG_FLAG_ALLOCATED)) ==
                            OSAL_MSG_FLAG_NONE)
                    {
                        // Set ALLOCATED flag
                        psQueueMsg->un8Flags |= OSAL_MSG_FLAG_ALLOCATED;

                        // Check if URGENT flag needs to be set as well
                        if (un32Flags & OSAL_QUEUE_FLAG_URGENT)
                        {
                            psQueueMsg->un8Flags |= OSAL_MSG_FLAG_URGENT;
                        }

                        // Check if DEFERRED flag needs to be set as well
                        if (un32Flags & OSAL_QUEUE_FLAG_DEFERRED)
                        {
                            psQueueMsg->un8Flags |= OSAL_MSG_FLAG_DEFERRED;
                        }

                        // Allocate message in list (i.e. add as used message)
                        eReturnCode = eMoveMsgToList(psQueueMsg,
                            psPrivate->hAllocatedMessageList,
                            FALSE
                                );
                        if (eReturnCode == OSAL_SUCCESS)
                        {
                            // The actual pointer we return back to the caller is
                            // one which points to the payload area, not the object
                            // itself. So we now calculate the proper return value.
                            *ppvMessage = (psQueueMsg + 1);
                        }
                    }
                    else
                    {
                        // Something is wrong with OSAL
                        eReturnCode = OSAL_ERROR;
                    }
                }
                else
                {
                    // Something bad happened in OSAL
                    eReturnCode = OSAL_OBJECT_NOT_FOUND;
                }

                // Exit exclusive access section now
                psPrivate->vExitExclusiveAccess(psPrivate->hMutex);
            }
            else
            {
                // Cannot enter exclusive access section. Fail.
                eReturnCode = OSAL_ERROR_MESSAGE_NOT_ALLOCATED;
            }
        }
        else if ((eReturnCode == OSAL_TIMEOUT) ||
                 (eReturnCode == OSAL_SEM_NOT_AVAILABLE))
        {
            // Error! Recondition the return code to something meaningful
            eReturnCode = OSAL_MESSAGE_NOT_AVAILABLE;
        }
    }
    else
    {
        // An invalid handle was provided
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eMessageFree
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eMessageFree (
    const void *pvMessage
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_QUEUE_MSG_STRUCT *psQueueMsg;
    OSAL_OBJECT_HDL hQueueObj;
    OSAL_OBJECT_STRUCT *psObj;

    // Check that input is valid
    if (pvMessage == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract from the provided message pointer the actual
    // message object pointer.
    psQueueMsg = ((OSAL_QUEUE_MSG_STRUCT *)pvMessage - 1);

    // Now that we have a pointer to the message object we need to extract
    // the queue handle from which this object belongs
    hQueueObj = psQueueMsg->hQueueObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hQueueObj, OSAL_OBJECT_TYPE_QUEUE);
    if (psObj != NULL)
    {
        OSAL_QUEUE_PRIVATE_INFO_STRUCT *psPrivate;

        // Extract the info pointers for the object
        psPrivate = &psObj->puInfo->sQueue;

        // Check if message flags are correct. The message must be
        // ALLOCATED and NOT QUEUED
        if ((psQueueMsg->un8Flags & (OSAL_MSG_FLAG_QUEUED | OSAL_MSG_FLAG_ALLOCATED)) ==
               OSAL_MSG_FLAG_ALLOCATED)
        {
            BOOLEAN bOk;

            // Enter an exclusive access section to en-queue this
            // message onto the list.
            bOk = psPrivate->bEnterExclusiveAccess(psPrivate->hMutex);

            if (bOk == TRUE)
            {

#if OSAL_DEBUG ==1
                // Clear out the memory for the message
                // We do this in OSAL_DEBUG in order to catch issues where
                // we keep using a queue message's memory even after it has been
                // freed.
                // In release, we don't want to do this since we are wasting time.
                OSAL.bMemSet((psQueueMsg + 1), 0, psQueueMsg->tSize);
#endif

                // Make size of message empty
                psQueueMsg->tSize = 0;

                // Clear up flags
                psQueueMsg->un8Flags = OSAL_MSG_FLAG_NONE;

                // Remove the entry from its current position
                eReturnCode =
                    eMoveMsgToList(psQueueMsg, psPrivate->hFreeMessageList, FALSE);
                if (eReturnCode == OSAL_SUCCESS)
                {
                    // Post a semaphore to indicate that something is
                    // available for allocation when needed.
                    eReturnCode = OSAL.eSemGive(psPrivate->hMessageAllocSem);
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        // Can't put any more than you took. This probably can't
                        // happen, but might is someone tries to free something
                        // bogus.
                        eReturnCode = OSAL_ERROR_INVALID_INPUT;
                    }
                }

                // Exit exclusive access section now
                psPrivate->vExitExclusiveAccess(psPrivate->hMutex);
            }
            else
            {
                // Cannot enter exclusive access section. Aborting.
                eReturnCode = OSAL_ERROR;
            }
        }
        else
        {
            // Something is wrong with OSAL
            eReturnCode = OSAL_ERROR;
        }
    }
    else
    {
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eMoveMsgToList
*
*****************************************************************************/
static OSAL_RETURN_CODE_ENUM eMoveMsgToList (
    OSAL_QUEUE_MSG_STRUCT *psQueueMsg,
    OSAL_OBJECT_HDL hList,
    BOOLEAN bUsePriotity
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Remove entry from its current list
        if (psQueueMsg->hThisEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eReturnCode = OSAL.eLinkedListRemove(psQueueMsg->hThisEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
            psQueueMsg->hThisEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        // Put entry back to the list
        if (bUsePriotity == FALSE)
        {
            eReturnCode =
                OSAL.eLinkedListAdd(hList, &psQueueMsg->hThisEntry,
                    (void*)psQueueMsg);
        }
        else
        {
            OSAL_LINKED_LIST_ENTRY hAnchorEntry;
            BOOLEAN bIsUrgent;
            BOOLEAN bIsDeferred;

            bIsUrgent = ((psQueueMsg->un8Flags & OSAL_MSG_FLAG_URGENT)
                        == OSAL_MSG_FLAG_URGENT) ? TRUE : FALSE;
            bIsDeferred = ((psQueueMsg->un8Flags & OSAL_MSG_FLAG_DEFERRED)
                        == OSAL_MSG_FLAG_DEFERRED) ? TRUE : FALSE;

            // Choose the anchor list entry to place the message
            if (bIsUrgent == TRUE)
            {
                // Put message to the top of the target list
                hAnchorEntry = OSAL.hLinkedListFirst(hList, NULL);
            }
            else
            {
                // Put message at the bottom
                hAnchorEntry = OSAL.hLinkedListLast(hList, NULL);
            }

            if (hAnchorEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                // Add entry as-is!
                eReturnCode = OSAL.eLinkedListAdd(
                                hList, &hAnchorEntry, (void*)psQueueMsg);
            }
            else if (bIsUrgent == TRUE)
            {
                eReturnCode = OSAL.eLinkedListAddBeforeEntry(
                                hList, &hAnchorEntry, (void*)psQueueMsg);
            }
            else if (bIsDeferred == TRUE)
            {
                eReturnCode = OSAL.eLinkedListAddAfterEntry(
                                hList, &hAnchorEntry, (void*)psQueueMsg);
            }
            else
            {
                OSAL_LINKED_LIST_ENTRY hEntry;
                OSAL_QUEUE_MSG_STRUCT *psMsg;

                // Each message should be added before deferred message(s)
                hEntry = hAnchorEntry;
                // Get entry object
                psMsg = (OSAL_QUEUE_MSG_STRUCT *)
                    OSAL.pvLinkedListThis(hEntry);
                do
                {
                    // Check if this message is deferred
                    bIsDeferred = ((psMsg->un8Flags & OSAL_MSG_FLAG_DEFERRED)
                                    == OSAL_MSG_FLAG_DEFERRED) ? TRUE : FALSE;
                    if (bIsDeferred == FALSE)
                    {
                        hAnchorEntry = hEntry;
                        break;
                    }

                    hAnchorEntry = hEntry;

                    hEntry = OSAL.hLinkedListPrev(hAnchorEntry, (void**)&psMsg);

                } while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY);

                if (bIsDeferred == TRUE)
                {
                    eReturnCode = OSAL.eLinkedListAddBeforeEntry(
                                    hList, &hAnchorEntry, (void*)psQueueMsg);
                }
                else
                {
                    eReturnCode = OSAL.eLinkedListAddAfterEntry(
                                    hList, &hAnchorEntry, (void*)psQueueMsg);
                }
            }

            // Refer the LL entry from the message
            psQueueMsg->hThisEntry = hAnchorEntry;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   vInitializeMessageHeader
*
*****************************************************************************/
static void vInitializeMessageHeader ( void *pvData )
{
    OSAL_QUEUE_MSG_STRUCT *psMsg =
        (OSAL_QUEUE_MSG_STRUCT *)pvData;

    if (psMsg != NULL)
    {
        // Initialize Message Header (as available)
        *psMsg = GsQueueMsgDefault;
    }

    return;
}

/*******************************************************************************
*
*   bEnterExclusiveAccessTaskOnly
*
*   This is a local-helper function which is used to enter an exclusive access
*   section by simply preventing any other task from accessing these same
*   internal structures. Interrupts may still occur since we were told that
*   the caller does not use this object from an ISR.
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static BOOLEAN bEnterExclusiveAccessTaskOnly(OSAL_OBJECT_HDL hMutex)
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = FALSE;

    eReturnCode = OSAL.eSemTake(hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
    if (eReturnCode == OSAL_SUCCESS)
    {
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*******************************************************************************
*
*   vExitExclusiveAccessTaskOnly
*
*   This is a local-helper function which is used to exit an exclusive access
*   section which was entered by the call to bEnterExclusiveAccessTaskOnly().
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static void vExitExclusiveAccessTaskOnly(OSAL_OBJECT_HDL hMutex)
{
    OSAL.eSemGive(hMutex);
    return;
}
/*******************************************************************************
*
*   bEnterExclusiveAccessInterrupt
*
*   This is a local-helper function which is used to enter an exclusive access
*   section by disabling interrupts as these APIs for this object might
*   be used by an ISR as well as between tasks. So this method disables
*   interrupts to protect the shared data. This is much more struct than
*   just blocking tasks as in TaskOnly.
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static BOOLEAN bEnterExclusiveAccessInterrupt(OSAL_OBJECT_HDL hMutex)
{
    OSAL.vEnterCriticalSection();

    // This function always returns TRUE, because it is not possible to destroy
    // OSAL control structure handling Critical Section semaphore before queues
    return TRUE;
}

/*******************************************************************************
*
*   vExitExclusiveAccessInterrupt
*
*   This is a local-helper function which is used to exit an exclusive access
*   section which was entered by the call to bEnterExclusiveAccessInterrupt().
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static void vExitExclusiveAccessInterrupt(OSAL_OBJECT_HDL hMutex)
{
    OSAL.vExitCriticalSection();
    return;
}


#ifdef SUPPORT_CUNIT
#include <osal_queue.cunit>
#endif
