/******************************************************************************/
/*                     Copyright (c) Sirius XM Radio Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio Inc.           */
/*                           Proprietary & Confidential                       */
/******************************************************************************/

#include <string.h>
#include <stdarg.h>
#include <stdio.h>

#include "standard.h"

#include "osal_version.h"

#include "osal_core.h"
#include "_osal_log.h"

#include "osal.h"

/*******************************************************************************
*
*   OSAL_eLogCreate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLogCreate(
    OSAL_OBJECT_HDL *phLog,
    const char *pacName,
    N32 n32Size,
    UN32 un32Options
        )
{
#if OSAL_LOG == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hLog;
    OSAL_LOG_INFO_STRUCT *psLog;

    // Check input
    if(phLog == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phLog = OSAL_INVALID_OBJECT_HDL;

    // Verify they have provided valid options
    if( (un32Options & ~OSAL_LOG_OPTION_ALL) != OSAL_LOG_OPTION_NONE )
    {
        return OSAL_ERROR_UNSUPPORTED_OPTION;
    }

    // Make sure they have provided a request for a log of some size
    if(n32Size <= 0)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Create the object to hold this OS-object and manage it
    // We also specify we would like a number of 'additional' bytes
    // which we will use for the log 'buffer'
    // Add one byte for a terminating null from 'vsnprintf'
    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_LOG,
        n32Size + 1,
        pacName,
        &hLog,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psLog
            );
    if((eReturnCode == OSAL_SUCCESS) && (psLog != NULL))
    {
        // populate the object info
        // Buffer space is last part of object + info memory
        psLog->pcBuffer = (char *)((OSAL_LOG_INFO_STRUCT*)(psObj + 1) + 1);
        psLog->n32Size = n32Size;
        psLog->un32Options = un32Options;

        // Initialize index
        psLog->n32Index = 0;

        // Initialize log
        psLog->bLocked = FALSE;
        psLog->bEnabled = TRUE;
        psLog->bPersist = FALSE;

        // Populate populate the object
        *phLog = hLog;

        // Object created successfully
        eReturnCode = OSAL_SUCCESS;
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_eLogDelete
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLogDelete(
    OSAL_OBJECT_HDL hLog
        )
{
#if OSAL_LOG == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    BOOLEAN bPersist = FALSE;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    /*************************************************************************/
    // Extract current log flags (as an atomic operation)
    OSAL.vEnterCriticalSection();

    // Lock this log unconditionally
    psObj->puInfo->sLog.bLocked = TRUE;

    // Grab persistence
    bPersist = psObj->puInfo->sLog.bPersist;

    OSAL.vExitCriticalSection();
    /*************************************************************************/

    // Remove and destroy the object if not persisted
    if(bPersist == FALSE)
    {
        eReturnCode = OSALC_eRemoveObject(hLog);
    }
    else
    {
        // Nevermind, we are not getting rid of it, but let someone
        // know we tried.
        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_n32LogWrite
*
*******************************************************************************/
N32 OSAL_n32LogWrite(
    OSAL_OBJECT_HDL hLog,
    const char *pcFormat,
    ...
        )
{
#if OSAL_LOG == 1
    N32 n32Num = EOF;
    va_list tArgList;

    // grab variable arguments from the stack
    va_start(tArgList, pcFormat);

    n32Num = OSAL_n32VLogWrite(hLog, pcFormat, &tArgList);

    // restore stack
    va_end(tArgList);

    return n32Num;
#else
    return EOF;
#endif
}

/*******************************************************************************
*
*   OSAL_n32VLogWrite
*
* This function is equivalent  to  the  functions  OSAL_n32LogWrite, except
* that it is called with a va_list instead  of  a variable  number  of
* arguments. This function does not call the va_end macro. Consequently,
* the value of ptArgList is undefined after the  call.  The
* application should call va_end(ptArgList) itself afterwards.
*
*******************************************************************************/
N32 OSAL_n32VLogWrite(
    OSAL_OBJECT_HDL hLog,
    const char *pcFormat,
    va_list *ptArgList
        )
{
#if OSAL_LOG == 1
    N32 n32Num = 0, n32Remaining;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LOG_INFO_STRUCT *psLog;
    BOOLEAN bLocked, bEnabled;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj == NULL)
    {
        // Error! Object does not exist
        return -1;
    }

    // Extract log info from object
    psLog = &psObj->puInfo->sLog;

    /*************************************************************************/
    // Extract current log flags (as an atomic operation)
    OSAL.vEnterCriticalSection();

    // Grab locked value and lock it if unlocked
    if((bLocked = psLog->bLocked) == FALSE)
    {
        // Lock this log
        psLog->bLocked = TRUE;
    }

    // Grab enabled flag
    bEnabled = psLog->bEnabled;

    OSAL.vExitCriticalSection();
    /*************************************************************************/

    // Check if log is locked or disabled, if so bail!
    if((bLocked == TRUE) || (bEnabled != TRUE))
    {
        // Bail! Log is locked by another resource
        return -1;
    }

    // Compute the amount of space left in the buffer
    n32Remaining = psLog->n32Size - psLog->n32Index;

    // There must be at least enough room for at least one byte.
    // If there is not enough room, just simply return and indicate
    // that no bytes were written
    if(n32Remaining < 1)
    {
        // Unlock log
        psLog->bLocked = FALSE;

        // Bail! Log has no more room
        return 0;
    }

    // Based on this log's options use a formatter or not
    if((psLog->un32Options & OSAL_LOG_OPTION_FORMAT) == OSAL_LOG_OPTION_FORMAT)
    {
        // process provided format and variable arguments into the buffer
        // not to exceed the size remaining in the buffer, but include
        // the NULL terminator.
        n32Num = vsnprintf (&psLog->pcBuffer[psLog->n32Index],
                            n32Remaining + 1, pcFormat, *ptArgList);
    }
    else
    {
        // Just copy the provided format/const parameter, no var args
        strncpy(&psLog->pcBuffer[psLog->n32Index],
                 pcFormat, n32Remaining + 1);

        // Number it would have (or did) write
        n32Num = strlen(pcFormat);
    }

    // Check how much was written, if more than the size of the buffer remaining
    // is returned this means it was truncated, so we should reflect the
    // truncated amount to the caller and also use it to update the index
    n32Num = (n32Num >= n32Remaining + 1) ?
        n32Remaining : // truncated (do not include written NULL)
        n32Num; // not truncated

    // Update buffer index with the number of bytes written
    psLog->n32Index += n32Num;

    /*************************************************************************/
    // Extract current log flags (as an atomic operation)
    OSAL.vEnterCriticalSection();

    // Unlock log
    psLog->bLocked = FALSE;

    OSAL.vExitCriticalSection();
    /*************************************************************************/
    return n32Num;
#else
    return EOF;
#endif
}

/*******************************************************************************
*
*   OSAL_n32LogRead
*
*******************************************************************************/
N32 OSAL_n32LogRead(
    OSAL_OBJECT_HDL hLog,
    char *pcDest,
    N32 n32Position,
    N32 n32Length
        )
{
#if OSAL_LOG == 1
    N32 n32Num = EOF;
    N32 n32Index;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LOG_INFO_STRUCT *psLog;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj == NULL)
    {
        // Error! Object does not exist
        return -1;
    }

    // Extract log info from object
    psLog = &psObj->puInfo->sLog;

    /*************************************************************************/
    // Extract current index (as an atomic operation)
    OSAL.vEnterCriticalSection();

    n32Index = psLog->n32Index;

    OSAL.vExitCriticalSection();
    /*************************************************************************/

    // Verify requested position is within the size of the log buffer
    // and also less than the current index position
    if( (n32Position >= psLog->n32Size) || (n32Position > n32Index) )
    {
        // Indicate nothing can be read
        return 0;
    }

    // Set the number of bytes to read from the current position to the
    // end of the available data in the buffer, truncate it if necessary
    n32Num = (n32Length > n32Index - n32Position) ?
        n32Index - n32Position : // truncate to max
        n32Length; // ok

    // Read bytes
    OSAL.bMemCpy(pcDest, &psLog->pcBuffer[n32Position], n32Num);
    return n32Num;
#else
    return EOF;
#endif
}

/*******************************************************************************
*
*   OSAL_eLogClear
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLogClear( OSAL_OBJECT_HDL hLog )
{
#if OSAL_LOG == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LOG_INFO_STRUCT *psLog;

    // List is now locked, cocked and ready to rock

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj != NULL)
    {
        // Extract log info
        psLog = &psObj->puInfo->sLog;

        /*************************************************************************/
        // Manipulate index to effectively clear the log
        // (as an atomic operation)
        OSAL.vEnterCriticalSection();

        psLog->n32Index = 0;

        OSAL.vExitCriticalSection();
        /*************************************************************************/
    }
    else
    {
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_vLogDisable
*
*******************************************************************************/
void OSAL_vLogDisable( OSAL_OBJECT_HDL hLog )
{
#if OSAL_LOG == 1
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj == NULL)
    {
        // Error! Object does not exist
        return;
    }

    /*************************************************************************/
    // Disable logging, next time this is checked it will take effect
    OSAL.vEnterCriticalSection();

    psObj->puInfo->sLog.bEnabled = FALSE;

    OSAL.vExitCriticalSection();
    /*************************************************************************/
#endif

    return;
}

/*******************************************************************************
*
*   OSAL_vLogEnable
*
*******************************************************************************/
void OSAL_vLogEnable( OSAL_OBJECT_HDL hLog )
{
#if OSAL_LOG == 1
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj == NULL)
    {
        // Error! Object does not exist
        return;
    }

    /*************************************************************************/
    // Enable logging, next time this is checked it will take effect
    OSAL.vEnterCriticalSection();

    psObj->puInfo->sLog.bEnabled = TRUE;

    OSAL.vExitCriticalSection();
    /*************************************************************************/
#endif

    return;
}

/*******************************************************************************
*
*   OSAL_vLogPersist
*
*******************************************************************************/
void OSAL_vLogPersist( OSAL_OBJECT_HDL hLog, BOOLEAN bPersist )
{
#if OSAL_LOG == 1
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj == NULL)
    {
        // Error! Object does not exist
        return;
    }

    // Make sure we have the pointer we need
    if(psObj->puInfo != NULL)
    {
        /***********************************************************************/
        // Modify log persistence, next time this is checked it will take effect
        OSAL.vEnterCriticalSection();

        psObj->puInfo->sLog.bPersist = bPersist;

        OSAL.vExitCriticalSection();
        /***********************************************************************/
    }

#endif

    return;
}

/*******************************************************************************
*
*   OSAL_hLogGetHandleByName
*
*******************************************************************************/
OSAL_OBJECT_HDL OSAL_hLogGetHandleByName (
    const char *pacName
        )
{
#if OSAL_LOG == 1
    OSAL_OBJECT_HDL hObjectHdl = OSAL_INVALID_OBJECT_HDL;
    OSAL_OBJECT_STRUCT *psObj;

    // Call OS specific API
    psObj = OSALC_psFindObjectFromName(OSAL_OBJECT_TYPE_LOG, pacName);
    if(psObj != NULL)
    {
        hObjectHdl = (OSAL_OBJECT_HDL)psObj;
    }
    return hObjectHdl;
#else
    return OSAL_INVALID_OBJECT_HDL;
#endif
}

/*******************************************************************************
*
*   OSAL_eLogGetInfo
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLogGetInfo (
    OSAL_OBJECT_HDL hLog,
    OSAL_LOG_INFO_STRUCT *psLogInfo
        )
{
#if OSAL_LOG == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OSAL_OBJECT_STRUCT *psObj;

    // Check provided structure pointer
    if(psLogInfo == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // List is now locked, cocked and ready to rock

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLog, OSAL_OBJECT_TYPE_LOG);
    if(psObj != NULL)
    {
        /*************************************************************************/
        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        OSAL.vEnterCriticalSection();

        *psLogInfo = psObj->puInfo->sLog;

        OSAL.vExitCriticalSection();
        /*************************************************************************/
    }
    else
    {
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_eLogList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLogList  ( BOOLEAN bVerbose )
{
#if (OSAL_LOG == 1) && (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hLogList;
    OSALL_PRINT_STRUCT sPrint;
    UN32 un32Logs = 0;
    OSAL_STATISTICS_STRUCT sStatistics;

    // Iterate the list of logs and list each entry

    // Grab the log linked list
    hLogList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_LOG);
    if(hLogList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of logs in the list
    eReturnCode = OSAL.eLinkedListItems(hLogList, &un32Logs);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_LOG, &sStatistics);

    // Check that one or more elements exist in the list
    if((un32Logs > 0) && (bVerbose == TRUE))
    {
        // Print title
        puts("=================");
        puts("| OSAL Log List |");
        puts("=================");
    }

    // Initialize iterator argument
    sPrint.bVerbose = bVerbose;
    sPrint.un32LogNumber = 0;

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hLogList,
        OSALL_bPrintLog, &sPrint);
    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        puts("Log list is empty!");
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else if(bVerbose == TRUE)
    {
        // Print summary
        printf("%u Log(s) currently allocated by the system.\n",
                    un32Logs);

        // Print statistics
        printf("Current = %u, Minimum = %u, Maximum = %u\n",
               sStatistics.un32Current,
               sStatistics.un32Minimum,
               sStatistics.un32Maximum );

    }

    return OSAL_SUCCESS;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

#if (OSAL_LOG == 1) && (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

/*******************************************************************************
*
*   OSALL_bPrintLog
*
*******************************************************************************/
static BOOLEAN OSALL_bPrintLog ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData,
        *psCreatorObj;
    OSALL_PRINT_STRUCT *psPrint = (OSALL_PRINT_STRUCT *)pvArg;
    const char *pacCreatorObjName = "Unknown";

    // Check if argument is valid
    if(psPrint == NULL)
    {
        return FALSE;
    }

    // Prepare log number
    (psPrint->un32LogNumber)++;

    // Check if object is valid
    if(psObj != NULL)
    {
        // Point to this object's info
        OSAL_LOG_INFO_STRUCT sLog;

        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        sLog = psObj->puInfo->sLog;

        // Check verbosity
        if(psPrint->bVerbose == TRUE)
        {
            // Extract the creator log's name
            psCreatorObj = (OSAL_OBJECT_STRUCT *)psObj->hCreatorObj;
            if(psCreatorObj != NULL)
            {
                // Extract name of creator
                pacCreatorObjName = psCreatorObj->pacName;
            }

            // extract information we are interested, and print
            printf("Log #%u\n", psPrint->un32LogNumber);
            printf("Name = '%s' (hdl = 0x%p)\n",
                psObj->pacName,
                   psObj
                       );
            printf("\tCreator = %s\n", pacCreatorObjName);
            printf("\tOptions = %#010x\n", sLog.un32Options);
            printf("\tBuffer = 0x%p\n", sLog.pcBuffer);
            printf("\tSize = %d\n", sLog.n32Size);
            printf("\tIndex = %d\n", sLog.n32Index);
            printf("\t%s\n", sLog.bLocked == TRUE ? "Locked" : "Unlocked");
            printf("\t%s\n", sLog.bEnabled == TRUE ? "Enabled" : "Disabled");
            puts("");
        }
        else // Terse
        {
            // extract information we are interested, and print it
            printf("#%u '%s' %d/%d(%u%%)\n",
                psPrint->un32LogNumber,
                psObj->pacName,
                sLog.n32Index,
                sLog.n32Size,
                (sLog.n32Index * 100) / sLog.n32Size
                    );
        }
    }

    // Keep iterating
    return TRUE;
}

/*******************************************************************************
*
*   OSALL_bIterateLog
*
*******************************************************************************/
static BOOLEAN OSALL_bIterateLog ( void *pvData, void *pvArg )
{
    BOOLEAN bContinue = FALSE;
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData;
    OSALL_ITERATOR_STRUCT *psIterator =
        (OSALL_ITERATOR_STRUCT *)pvArg;

    // Check if object and iterator is valid
    if((psObj != NULL) && (psIterator != NULL))
    {
        // Prepare log number
        (psIterator->un32LogNumber)++;

        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        psIterator->sLog = psObj->puInfo->sLog;

        // Call them
        if(psIterator->pbIterator != NULL)
        {
            bContinue = psIterator->pbIterator(
                (OSAL_OBJECT_HDL)pvData, psObj->pacName,
                psIterator->pvIteratorArg);
        }
    }

    return bContinue;
}

#endif /* (OSAL_LOG == 1) && (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */

/*******************************************************************************
*
*   OSAL_eLogIterate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLogIterate  (
    OSAL_LOG_ITERATOR_HANDLER pbIterator,
    void *pvIteratorArg
        )
{
#if (OSAL_LOG == 1) && (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hLogList;
    OSALL_ITERATOR_STRUCT sIterator;

    // Iterate the list of logs...

    // Grab the log linked list
    hLogList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_LOG);
    if(hLogList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(
        OSAL_OBJECT_TYPE_LOG, &sIterator.sStatistics);

    // Initialize iterator argument
    sIterator.un32LogNumber = 0;

    // Use caller's callback
    sIterator.pbIterator = pbIterator;
    sIterator.pvIteratorArg = pvIteratorArg;

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hLogList,
        OSALL_bIterateLog, &sIterator);
    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    return OSAL_SUCCESS;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}
