/******************************************************************************/
/*                      Copyright (c) Sirius XM Radio Inc.                    */
/*                            All Rights Reserved                             */
/*             Licensed Materials - Property of Sirius XM Radio Inc.          */
/*                         Proprietary & Confidential                         */
/******************************************************************************/
/*******************************************************************************
*
*  DESCRIPTION
*
*       This module will contain all the OSAL Linked List
*       functionality.
*
*******************************************************************************/

#include <string.h>
#include <stdio.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"
#include "osal_core.h"
#include "os_intf.h"

#include "osal_ll.h"
#include "_osal_ll.h"

/*****************************************************************************
*
*       OSAL_eLinkedListCreate
*
*        This is the function used to create a new linked list and return
*       a handle to this list. Upon successful creation the linked list
*       will contain no elements.
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListCreate (
    OSAL_OBJECT_HDL *phLL,
    const char *pacName,
    OSAL_LL_COMPARE_HANDLER tn16CompareFunction,
    UN32 un32Options
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode, eReturnCode2;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psLL;
    OSAL_OBJECT_HDL hLL;

    // Check input
    if(phLL == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phLL = OSAL_INVALID_OBJECT_HDL;

    // Verify they have provided only valid options
    if( ((un32Options & ~OSAL_LL_OPTION_ALL) != OSAL_LL_OPTION_NONE ) ||
        ((un32Options & (OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_LINEAR)) ==
                        (OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_LINEAR)))
    {
        return OSAL_ERROR_UNSUPPORTED_OPTION;
    }

    // Verify they have selected either circular or linear
    // One must be selected, but if not, default to CIRCULAR
    if( (un32Options &
            (OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_RBTREE)) ==
                OSAL_LL_OPTION_NONE )
    {
        // Select circular by default
        un32Options |= OSAL_LL_OPTION_CIRCULAR;
    }

    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_LINKED_LIST,
        0,
        pacName,
        &hLL,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psLL
            );
    if((eReturnCode == OSAL_SUCCESS) && (psLL != NULL))
    {
        // Populate LL info from caller provided info
        psLL->hCreatorObj = OSAL.hTaskGetHandle();
        psLL->un32Options = un32Options;

        if ((un32Options & OSAL_LL_OPTION_RBTREE) == OSAL_LL_OPTION_RBTREE)
        {
            // By default the RBtree implementation provides
            // unique item stored inside the tree nodes
            psLL->un32Options |= OSAL_LL_OPTION_UNIQUE;
            psLL->uList.hRBTree = NULL;
            eReturnCode = OSAL_eRBTreeCreate(
                    &psLL->uList.hRBTree, pacName, hLL,
                    tn16CompareFunction
                        );
            if (eReturnCode != OSAL_SUCCESS)
            {
                // Return object allocated and return error code to caller
                // Remove and destroy the object
                eReturnCode2 = OSALC_eRemoveObject(hLL);
                if(eReturnCode2 != OSAL_SUCCESS)
                {
                    eReturnCode = eReturnCode2;
                }
                return eReturnCode;
            }
        }
        else
        {
            psLL->uList.sList.hTopOfListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psLL->uList.sList.hBottomOfListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psLL->uList.sList.tn16CompareFunction = tn16CompareFunction;
            psLL->uList.sList.un32Count = 0;
        }

        // Check if they want this list to be protected
        if((psLL->un32Options & OSAL_LL_OPTION_PROTECT) ==
           OSAL_LL_OPTION_PROTECT)
        {
            // Protect list, by creating a semaphore for it
            eReturnCode = OSAL.eLinkedListProtect(hLL);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Return object allocated and return error code to caller
                // Remove and destroy the object
                eReturnCode2 = OSALC_eRemoveObject(hLL);
                if(eReturnCode2 != OSAL_SUCCESS)
                {
                    eReturnCode = eReturnCode2;
                }
                return eReturnCode;
            }
        }
        else
        {
            // Do not protect list, not semaphore object required
            psObj->hOS = OS_INVALID_OBJECT_HDL;
        }

        // Populate the object
        *phLL = hLL;

        // We finished creating the linked list successfully
        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListDelete
*
*        This is the function used to delete a linked list object
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListDelete(
    OSAL_OBJECT_HDL hLL
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode, eLockReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psLL;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract linked list info from object
    psLL = &psObj->puInfo->sLL;

    // You can only destroy a linked list if all elements have
    // been removed first.
    if ((psLL->un32Options & OSAL_LL_OPTION_RBTREE)
            == OSAL_LL_OPTION_RBTREE)
    {
        eReturnCode = OSAL_eRBTreeDelete(psLL->uList.hRBTree);
        if (eReturnCode == OSAL_ERROR_LIST_NOT_EMPTY)
        {
            return eReturnCode;
        }
    }
    else if(psLL->uList.sList.hTopOfListEntry !=
       OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return OSAL_ERROR_LIST_NOT_EMPTY;
    }
    else
    {
        eReturnCode = OSAL_SUCCESS;
    }

    // Delete list semaphore (if list is protected)
    if(psObj->hOS != OS_INVALID_OBJECT_HDL)
    {
        // Delete protection semaphore
        eReturnCode =
            OSAL.eSemDelete( (OSAL_OBJECT_HDL)psObj->hOS );
        if(eReturnCode != OSAL_SUCCESS)
        {
            return eReturnCode;
        }

        // Invalidate semaphore now that it is deleted
        psObj->hOS = OS_INVALID_OBJECT_HDL;
    }

    // Return object allocated for the linked list object
    // Remove and destroy the object
    eLockReturnCode = OSALC_eRemoveObject(hLL);
    if (eLockReturnCode != OSAL_SUCCESS)
    {
        eReturnCode = eLockReturnCode;
    }

    // We finished deleting successfully

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListAdd
*
*        This is the function used to add an new element to the linked list
*       object provided.
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListAdd (
    OSAL_OBJECT_HDL hLL,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;

    // If the caller has not specified any specific insertion options
    if ((phEntry == OSAL_INVALID_LINKED_LIST_ENTRY_PTR) ||
        ((phEntry != OSAL_INVALID_LINKED_LIST_ENTRY_PTR) &&
            (*phEntry == OSAL_INVALID_LINKED_LIST_ENTRY)))
    {
        eReturnCode = OSALLL_eAllocateAndAdd(
            OSAL_LL_INSERT_HERE,
            hLL, phEntry, pvData);
    }

    // Invalid input parameters
    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListReplaceEntry
*
*        This is the function used to replace an element which exists in the
*       linked list object provided.
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListReplaceEntry(
    OSAL_OBJECT_HDL hLL,
    OSAL_LINKED_LIST_ENTRY hEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;
    OSAL_RETURN_CODE_ENUM eLockReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;

    // Note: It is possible the caller wishes to add an object
    // which has value of zero. So testing pvData for NULL is not desired.

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return eReturnCode;
    }

    // Extract linked-list info from new entry object
    psInfo = &psObj->puInfo->sLL;

    // hEntry must be a valid handle
    if ( hEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
    {
           // Lock list that this element is going to be added to
        eLockReturnCode = OSALLL_eLockDirect(psObj);
        if(eLockReturnCode != OSAL_SUCCESS)
        {
            return eLockReturnCode;
        }

        if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE) == OSAL_LL_OPTION_RBTREE)
        {
            eReturnCode = OSAL_eRBTreeReplace(psInfo->uList.hRBTree,
                                    (OSAL_RB_TREE_ENTRY)hEntry, pvData
                                        );
        }
        else
        {
            OSAL_LL_ELEMENT_STRUCT *psNewElement;

            // The new element pointer is the old one
            psNewElement = (OSAL_LL_ELEMENT_STRUCT *)hEntry;

            // Insert the element into the list based on how the linked list
            // was created. If a compare function was provided (non-NULL)
            // then they want the insertion to be 'sorted'. Otherwise
            // they just want to add it to the end of the list
            //
            // If the list is created as a set, force it to use the
            // Sorted insert feature to verify only unique elements
            // are added

            // This will perform a replace in the same position as
            // before in the list.
            if(psInfo->uList.sList.tn16CompareFunction == NULL &&
               (psInfo->un32Options & OSAL_LL_OPTION_UNIQUE) !=
               OSAL_LL_OPTION_UNIQUE )
            {
                psNewElement->pvData = pvData;

                eReturnCode = OSAL_SUCCESS;

                // Do not increment the count here because we did not do a remove
                // and then insert, we just changed pvData for an existing entry
            }

            // This will perform a Sorted Insert
            else
            {
                BOOLEAN bLinear;

                // Based on list options, set the linear list flag
                bLinear = ( (psInfo->un32Options & OSAL_LL_OPTION_LINEAR)
                    == OSAL_LL_OPTION_LINEAR ) ? TRUE : FALSE;

                // Remove the object
                OSALLL_vRemove( psNewElement );

                // Remove was successful, decrement count
                psInfo->uList.sList.un32Count--;

                // Insert new entry into obj list
                eReturnCode =
                    OSALLL_eSortedInsert(hLL,
                        psInfo,
                        (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hTopOfListEntry,
                        &psNewElement, pvData,
                        psInfo->uList.sList.tn16CompareFunction,
                        bLinear
                            );

                // Check if element insert was successful
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Insert was successful, increment count
                    psInfo->uList.sList.un32Count++;
                }
            }
        }

        // Unlock list
        eLockReturnCode = OSALLL_eUnlockDirect(psObj);
        if(eLockReturnCode != OSAL_SUCCESS)
        {
            eReturnCode = eLockReturnCode;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListAddBeforeEntry
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListAddBeforeEntry(
    OSAL_OBJECT_HDL hLL,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;

    // If we want to add the specified item in pvData before phEntry(reference),
    // we better have a valid value in phEntry
    if ( ( phEntry != OSAL_INVALID_LINKED_LIST_ENTRY_PTR ) &&
         ( *phEntry != OSAL_INVALID_LINKED_LIST_ENTRY ) )
    {
        eReturnCode = OSALLL_eAllocateAndAdd(
            OSAL_LL_INSERT_BEFORE,
            hLL, phEntry, pvData);
    }

    // Invalid input parameters
    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListAddAfterEntry
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListAddAfterEntry(
    OSAL_OBJECT_HDL hLL,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;

    // If we want to add the specified item in pvData before phEntry(reference),
    // we better have a valid value in phEntry
    if ( ( phEntry != OSAL_INVALID_LINKED_LIST_ENTRY_PTR ) &&
         ( *phEntry != OSAL_INVALID_LINKED_LIST_ENTRY ) )
    {
        eReturnCode = OSALLL_eAllocateAndAdd(
            OSAL_LL_INSERT_AFTER,
            hLL, phEntry, pvData);
    }

    // Invalid input parameters
    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListRemove
*
*        This is the function used to remove an element from a list
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListRemove(
    OSAL_LINKED_LIST_ENTRY hEntry
        )
{
    OSAL_LL_ELEMENT_STRUCT *psEntry =
        (OSAL_LL_ELEMENT_STRUCT *)hEntry;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hLL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    OSAL_RETURN_CODE_ENUM eLockReturnCode;

    // Verify a valid handle was provided
    if(hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return eReturnCode;
    }

    // Extract the linked list handle
    hLL = psEntry->hLL;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return eReturnCode;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL_LL_INFO_STRUCT *psInfo;

        // Extract linked-list info from new entry object
        psInfo = &psObj->puInfo->sLL;

        // Perform removal
        eReturnCode = OSALLL_eRemoveDirect(psInfo, hEntry);

        // Unlock list
        eLockReturnCode = OSALLL_eUnlockDirect(psObj);
        if(eLockReturnCode != OSAL_SUCCESS)
        {
            eReturnCode = eLockReturnCode;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListSearch
*
*        This is the function used to find an element in a list
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListSearch(
    OSAL_OBJECT_HDL hLL,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode, eLockReturnCode;
    OSAL_OBJECT_STRUCT *psObj = NULL;
    OSAL_LL_ELEMENT_STRUCT *psEntry = NULL;
    OSAL_LL_INFO_STRUCT *psInfo = NULL;
    OSAL_LINKED_LIST_ENTRY hTopOfListEntry;

    // Note: It is possible the caller wishes to search for an object
    // which has value of zero. So testing pvData for NULL is not desired.

    // Check input
    if(phEntry == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE) == OSAL_LL_OPTION_RBTREE)
    {
        eReturnCode = OSAL_eRBTreeSearch(psInfo->uList.hRBTree,
            (OSAL_RB_TREE_ENTRY *)&psEntry, NULL, pvData);
    }
    else
    {
        // Initialize where to start from the entry pointer provided.
        // If the entry provided is invalid, we start from the top of the list
        // otherwise we start from the position(entry) provided.
        if(*phEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Start from the top of the list
            hTopOfListEntry = psInfo->uList.sList.hTopOfListEntry;
        }
        else
        {
            // Start from the provided entry's next element
            hTopOfListEntry = OSAL.hLinkedListNext(*phEntry, (void **)NULL);
        }

        if(psInfo->uList.sList.tn16CompareFunction == NULL)
        {
            // List does not have one when created

            // The list is 'non-sorted'
            // Find the element (first one) which matches the given data.
            psEntry = OSALLL_psSearch((OSAL_LL_ELEMENT_STRUCT *)
                                      hTopOfListEntry, pvData,
                                      OSALLL_n16ComparePayload);
        }
        else if ((psInfo->un32Options & OSAL_LL_OPTION_BINARY_SEARCH) == 
            OSAL_LL_OPTION_BINARY_SEARCH)
        {
            // The list is 'sorted'
            // Find the element which matches the given data via binary search
            psEntry = OSALLL_psSortedBinarySearch(psInfo, pvData, 
                psInfo->uList.sList.tn16CompareFunction);
        }
        else
        {
            // The list is 'sorted'
            // Find the element (first one) which matches the given data.
            psEntry = OSALLL_psSortedSearch((OSAL_LL_ELEMENT_STRUCT *)
                hTopOfListEntry, pvData, 
                psInfo->uList.sList.tn16CompareFunction);
        }

        // At this point the appropriate compare function has been determined
        // and the search was made.
    }

    // Unlock list
    eLockReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eLockReturnCode != OSAL_SUCCESS)
    {
        eReturnCode = eLockReturnCode;
    }

    // Populate results if everything is OK
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Check if we found something
        if(psEntry == NULL)
        {
            eReturnCode = OSAL_OBJECT_NOT_FOUND;
        }

        // Return the entry (as a handle)
        *phEntry = (OSAL_LINKED_LIST_ENTRY) psEntry;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListLinearSearch
*
*       This is the function used to find an element in a list 
*       using non-sorted search
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListLinearSearch(
    OSAL_OBJECT_HDL hLL,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    OSAL_LL_COMPARE_HANDLER tn16CompareFunction,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode, eLockReturnCode;
    OSAL_OBJECT_STRUCT *psObj = NULL;
    OSAL_LL_ELEMENT_STRUCT *psEntry = NULL;
    OSAL_LL_INFO_STRUCT *psInfo = NULL;
    OSAL_LINKED_LIST_ENTRY hTopOfListEntry;

    // Note: It is possible the caller wishes to search for an object
    // which has value of zero. So testing pvData for NULL is not desired.

    // Check input
    if(phEntry == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    if (tn16CompareFunction == NULL)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE) == OSAL_LL_OPTION_RBTREE)
    {
        eReturnCode = OSAL_eRBTreeSearch(psInfo->uList.hRBTree,
                            (OSAL_RB_TREE_ENTRY*)&psEntry,
                            tn16CompareFunction, pvData
                                );
    }
    else
    {
        // Initialize where to start from the entry pointer provided.
        // If the entry provided is invalid, we start from the top of the list
        // otherwise we start from the position(entry) provided.
        if(*phEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Start from the top of the list
            hTopOfListEntry = psInfo->uList.sList.hTopOfListEntry;
        }
        else
        {
            // Start from the provided entry's next element
            hTopOfListEntry = OSAL.hLinkedListNext(*phEntry, (void **)NULL);
        }

        // Find the element (first one) which matches the given data.
        psEntry = OSALLL_psSearch((OSAL_LL_ELEMENT_STRUCT *)hTopOfListEntry,
            pvData, tn16CompareFunction);
    }

    // Unlock list
    eLockReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eLockReturnCode != OSAL_SUCCESS)
    {
        eReturnCode = eLockReturnCode;
    }

    // Populate results if everything is OK
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Check if we found something
        if(psEntry == NULL)
        {
            eReturnCode = OSAL_OBJECT_NOT_FOUND;
        }

        // Return the entry (as a handle)
        *phEntry = (OSAL_LINKED_LIST_ENTRY)psEntry;
    }

    return eReturnCode;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSAL_eLinkedListSort
*
* DESCRIPTION
*
*    Sort the linked list.
*
* INPUTS
*
*    hLL            - the linked list object to be re-sorted.
*
*    tn16Compare    - pointer to the function used to compare data.
*
* OUTPUTS
*
*    OSAL_RETURN_CODE_ENUM
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListSort(
    OSAL_OBJECT_HDL hLL,
    OSAL_LL_COMPARE_HANDLER tn16Compare
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;
    OSAL_LL_ELEMENT_STRUCT *psStart, *psEnd, *psList, *psItem;
    OSAL_LL_ELEMENT_SORT_STACK_ITEM_STRUCT *psStackItem;
    OSAL_LL_ELEMENT_SORT_STACK_ITEM_STRUCT asStack[OSAL_LL_ELEMENT_SORT_STACK_SIZE];
    UN8 un8StackPos;

    // Verify input
    if(tn16Compare == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    // Check type of the list - for RBTree sorting are not required
    // since all nodes inside are sorted by provided during
    // creation function
    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
            == OSAL_LL_OPTION_RBTREE)
    {
        return OSAL_ERROR_UNSUPPORTED_API;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    // Replace the current sort function with the one provided
    psInfo->uList.sList.tn16CompareFunction = tn16Compare;

    // If there are less than 2 items then there's nothing to sort.
    if( psInfo->uList.sList.un32Count < 2 )
    {
        eReturnCode = OSALLL_eUnlockDirect(psObj);
        return eReturnCode;
    }

    // Set-up start entry and next entry to kick things off
    psStart = (OSAL_LL_ELEMENT_STRUCT *) psInfo->uList.sList.hTopOfListEntry;

    // If this is a circular list, make it linear for now
    if ( (psInfo->un32Options & OSAL_LL_OPTION_CIRCULAR) == OSAL_LL_OPTION_CIRCULAR )
    {
        psEnd = psStart->psPrev;
        psEnd->psNext = NULL;
        psStart->psPrev = NULL;
    }

    ////////////////////////////////////////////////
    /// Sorting... [START]
    un8StackPos = 0;
    psItem = (OSAL_LL_ELEMENT_STRUCT*)psInfo->uList.sList.hTopOfListEntry;

    while (psItem != NULL)
    {
        // Initialize
        psStackItem = &asStack[un8StackPos];
        psStackItem->un32Level = 1;
        psStackItem->psEntry = psItem;

        // Move next
        psItem = psItem->psNext;
        psStackItem->psEntry->psNext = NULL;
        ++un8StackPos;

        // Merge loop
        while ((un8StackPos > 1) &&
               (asStack[un8StackPos - 2].un32Level == asStack[un8StackPos - 1].un32Level))
        {
            psStackItem = &asStack[un8StackPos - 2];
            psStackItem->psEntry =
                OSALLL_psSortedMerge(psStackItem->psEntry,
                                     asStack[un8StackPos - 1].psEntry, tn16Compare);
            ++(psStackItem->un32Level);
            --un8StackPos;
        }
    }

    // Final merge loop
    while (un8StackPos > 1)
    {
        psStackItem = &asStack[un8StackPos - 2];
        psStackItem->psEntry =
            OSALLL_psSortedMerge(psStackItem->psEntry,
                                 asStack[un8StackPos - 1].psEntry,
                                 tn16Compare);
        --un8StackPos;
    }

    // Update the reference to the head
    psStart = (*asStack).psEntry;
    psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY) psStart;
    /// Sorting... [END]
    ////////////////////////////////////////////////

    // Restore double linked list - prev references
    psList = psStart;
    psList->psPrev = NULL;
    while (psList->psNext != NULL)
    {
        psList->psNext->psPrev = psList;
        psList = psList->psNext;
    }

    // Update the reference to the end
    psEnd = psList;
    psInfo->uList.sList.hBottomOfListEntry = (OSAL_LINKED_LIST_ENTRY)psEnd;

    // Restore this as a circular list if necessary
    if ( (psInfo->un32Options & OSAL_LL_OPTION_CIRCULAR) == OSAL_LL_OPTION_CIRCULAR )
    {
        psEnd->psNext = psStart;
        psStart->psPrev = psEnd;
    }

    // Unlock list
    eReturnCode = OSALLL_eUnlockDirect(psObj);

    return eReturnCode;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSAL_eLinkedListRemoveAll
*
* DESCRIPTION
*
*    Remove all items from a linked list.
*
* INPUTS
*
*   psList - pointer to the main list element
*    psLink    - pointer to the list item to be removed.
*
* OUTPUTS
*
*    OSAL_RETURN_CODE_ENUM
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListRemoveAll(
    OSAL_OBJECT_HDL hLL,
    OSAL_LL_RELEASE_HANDLER tvReleaseFunction
        )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;
    OSAL_LINKED_LIST_ENTRY hEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_RETURN_CODE_ENUM eLockReturnCode = OSAL_ERROR;
    void *pvData;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock the LIST object
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
            == OSAL_LL_OPTION_RBTREE)
    {
        eReturnCode = OSAL_eRBTreeRemoveAll(
                            psInfo->uList.hRBTree,
                            tvReleaseFunction
                                );
    }
    else
    {
        // Let's pretend that we are good for now.
        eReturnCode = OSAL_SUCCESS;
        for(;;)
        {
            // Extract top of list from object
            hEntry = psInfo->uList.sList.hTopOfListEntry;
            if(hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                OSAL_LL_ELEMENT_STRUCT sEntry;

                // Saving some pointers for future use
                sEntry = *((OSAL_LL_ELEMENT_STRUCT *)hEntry);

                if (tvReleaseFunction != NULL)
                {
                    // Extract the user data from the entry
                    pvData = OSAL_pvLinkedListThis(hEntry);
                    // Call the user provided 'release' function
                    tvReleaseFunction(pvData);
                }

                // Per SX-0945-0167, caller must use
                // OSAL.pvLinkedListMemoryAllocate() and
                // OSAL.vLinkedListMemoryFree() when the list was created with
                // OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS option.

                // In this case, release function shall be, or contain inside
                // OSAL.vLinkedListMemoryFree. After calling this function,
                // entry's memory will be eventually freed, so the call to
                // OSAL_eLinkedListRemove(hEntry) will always fail

                // Checking if the list had pre-allocated elements
                if((psInfo->un32Options & OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS)
                    == OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS)
                {
                    // hEntry memory was already deallocated by user callback.
                    // So we need to re-link remaining entries

                    // Fixing items count
                    --psInfo->uList.sList.un32Count;

                    // Fixing top of the list. Don't care about bottom - the list
                    // will be cleaned up anyway
                    if (psInfo->uList.sList.hTopOfListEntry !=
                        (OSAL_LINKED_LIST_ENTRY)sEntry.psNext)
                    {
                        // Since hEntry is always fetched from the top, we just
                        // need to make sure it is not the only remaining entry
                        // in circular list
                        psInfo->uList.sList.hTopOfListEntry =
                            (OSAL_LINKED_LIST_ENTRY)sEntry.psNext;
                    }
                    else
                    {
                        psInfo->uList.sList.hTopOfListEntry =
                            OSAL_INVALID_LINKED_LIST_ENTRY;
                    }

                    if (psInfo->uList.sList.un32Count > 0)
                    {
                        // Fixing next/previous links of neighbors
                        OSALLL_vRemove(&sEntry);
                    }
                }
                else
                {
                    // Remove the top-of-list entry
                    eReturnCode = OSALLL_eRemoveDirect(psInfo, hEntry);
                }
            }
            else
            {
                // No entry or non-left
                break;
            }
        }

        psInfo->uList.sList.hBottomOfListEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Release LIST object
    eLockReturnCode = OSALLL_eUnlockDirect(psObj);
    if (eLockReturnCode != OSAL_SUCCESS)
    {
        eReturnCode = eLockReturnCode;
    }

    return eReturnCode;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSAL_pvLinkedListThis
*
* DESCRIPTION
*
*    Returns the data in this entry.
*
* INPUTS
*
*    hEntry    - pointer to the list item to be used.
*
* OUTPUTS
*
*    void *    - pointer to the data.
*
*******************************************************************************/
void *OSAL_pvLinkedListThis( OSAL_LINKED_LIST_ENTRY hEntry )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_ELEMENT_STRUCT *psEntry =
        (OSAL_LL_ELEMENT_STRUCT *)hEntry;
	void *pvResult;

    // Verify a valid handle was provided
    if(hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return NULL;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(psEntry->hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return NULL;
    }

    if ((psObj->puInfo->sLL.un32Options & OSAL_LL_OPTION_RBTREE)
                == OSAL_LL_OPTION_RBTREE)
    {
        pvResult = OSAL_pvRBTreeThis((OSAL_RB_TREE_ENTRY) hEntry);
    }
    else
    {
        pvResult = psEntry->pvData;
    }

	return pvResult;
}

/*****************************************************************************
*
*       OSAL_hLinkedListFirst
*
*       This is the function used to get the first element in a linked list
*
*****************************************************************************/
OSAL_LINKED_LIST_ENTRY OSAL_hLinkedListFirst(
    OSAL_OBJECT_HDL hLL,
    void **ppvData
         )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LL_ELEMENT_STRUCT *psEntry;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
            == OSAL_LL_OPTION_RBTREE)
    {
        OSAL_RB_TREE_ENTRY hResult;

        hResult = OSAL_hRBTreeFirst(psInfo->uList.hRBTree, ppvData);

        psEntry = (OSAL_LL_ELEMENT_STRUCT*)hResult;
    }
    else
    {
        // Extract the first item
        psEntry = (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hTopOfListEntry;

        // If they provided a pointer return the data
        if(ppvData != NULL)
        {
            if(psEntry != NULL)
            {
                *ppvData = psEntry->pvData;
            }
            else
            {
                *ppvData = NULL;
            }
        }
    }

    // Unlock list
    eReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    return (OSAL_LINKED_LIST_ENTRY)psEntry;
}

/*****************************************************************************
*
*       OSAL_hLinkedListLast
*
*       This is the function used to get the last element in a linked list
*
*****************************************************************************/
OSAL_LINKED_LIST_ENTRY OSAL_hLinkedListLast(
    OSAL_OBJECT_HDL hLL,
    void **ppvData )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LL_ELEMENT_STRUCT *psEntry;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE) == OSAL_LL_OPTION_RBTREE)
    {
        OSAL_RB_TREE_ENTRY hResult;

        hResult = OSAL_hRBTreeLast(psInfo->uList.hRBTree, ppvData);

        psEntry = (OSAL_LL_ELEMENT_STRUCT*)hResult;
    }
    else
    {
        // Extract the last item
        psEntry = (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hBottomOfListEntry;

        // If they provided a pointer return the data
        if(ppvData != NULL)
        {
            if(psEntry != NULL)
            {
                *ppvData = psEntry->pvData;
            }
            else
            {
                *ppvData = NULL;
            }
        }
    }

    // Unlock list
    eReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    return (OSAL_LINKED_LIST_ENTRY)psEntry;
}


/*******************************************************************************
*
* FUNCTION
*
*    OSAL_hLinkedListNext
*
* DESCRIPTION
*
*    Get the next member of the list.
*
* INPUTS
*
*    hCurrent        - pointer to an item in a linked list.
*
*    ppvData        - pointer to a pointer for the returned data.
*
* OUTPUTS
*
*    OSAL_LINKED_LIST_ENTRY    - pointer to the control structure of the next item
*                                in the linked list.
*                            - NULL = invalid list
*
*    ppvData    - pointer to the data in the next item.
*
*******************************************************************************/
OSAL_LINKED_LIST_ENTRY OSAL_hLinkedListNext(
    OSAL_LINKED_LIST_ENTRY hCurrent,
    void **ppvData )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;
    OSAL_LL_ELEMENT_STRUCT *psEntry =
        (OSAL_LL_ELEMENT_STRUCT *)hCurrent;
    OSAL_LINKED_LIST_ENTRY hEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify a valid handle was provided
    if(hCurrent == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(psEntry->hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
            == OSAL_LL_OPTION_RBTREE)
    {
        OSAL_RB_TREE_ENTRY hRBTreeEntry = (OSAL_RB_TREE_ENTRY)hCurrent;
        OSAL_RB_TREE_ENTRY hRBTreeNextEntry;

        hRBTreeNextEntry = OSAL_hRBTreeNext(hRBTreeEntry, ppvData);

        hEntry = (OSAL_LINKED_LIST_ENTRY)hRBTreeNextEntry;
    }
    else
    {
        // If they provided a pointer return the data
        if(ppvData != NULL)
        {
            if(psEntry->psNext != NULL)
            {
                *ppvData = psEntry->psNext->pvData;
            }
            else
            {
                *ppvData = NULL;
            }
        }

        // Grab next entry
        hEntry = (OSAL_LINKED_LIST_ENTRY)psEntry->psNext;
    }

    // Unlock list
    eReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    return hEntry;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSAL_hLinkedListPrev
*
* DESCRIPTION
*
*    Get the previous member of the list.
*
* INPUTS
*
*    hCurrent        - pointer to an item in a linked list.
*
*    ppvData        - pointer to a pointer for the returned data.
*
* OUTPUTS
*
*    OSAL_LINKED_LIST_ENTRY    - pointer to the control structure of the next item
*                                in the linked list.
*                            - NULL = invalid list
*
*    ppvData    - pointer to the data in the next item.
*
*******************************************************************************/
OSAL_LINKED_LIST_ENTRY OSAL_hLinkedListPrev(
    OSAL_LINKED_LIST_ENTRY hCurrent,
    void **ppvData )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;
    OSAL_LL_ELEMENT_STRUCT *psEntry =
        (OSAL_LL_ELEMENT_STRUCT *)hCurrent;
    OSAL_LINKED_LIST_ENTRY hEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify a valid handle was provided
    if(hCurrent == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(psEntry->hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
            == OSAL_LL_OPTION_RBTREE)
    {
        OSAL_RB_TREE_ENTRY hRBTreeEntry = (OSAL_RB_TREE_ENTRY)hCurrent;
        OSAL_RB_TREE_ENTRY hRBTreePrevEntry;

        hRBTreePrevEntry = OSAL_hRBTreePrev(hRBTreeEntry, ppvData);

        hEntry = (OSAL_LINKED_LIST_ENTRY)hRBTreePrevEntry;
    }
    else
    {
        // If they provided a pointer return the data
        if(ppvData != NULL)
        {
            if(psEntry->psPrev != NULL)
            {
                *ppvData = psEntry->psPrev->pvData;
            }
            else
            {
                *ppvData = NULL;
            }
        }

        // Grab prev entry
        hEntry = (OSAL_LINKED_LIST_ENTRY)psEntry->psPrev;
    }

    // Unlock list
    eReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    return hEntry;
}

/*****************************************************************************
*
*       OSAL_eLinkedListIterate
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListIterate (
    OSAL_OBJECT_HDL hLL,
    OSAL_LL_ITERATOR_HANDLER tbIterator,
    void *pvArg)
{
    OSAL_RETURN_CODE_ENUM eReturnCode, eLockReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;
    OSAL_LL_ELEMENT_STRUCT *psThis, *psFirst, *psNext;
    OSAL_LINKED_LIST_ENTRY hTopEntry;
    BOOLEAN bRetVal;

    // Check if function is NULL
    if(tbIterator == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
            == OSAL_LL_OPTION_RBTREE)
    {
        eReturnCode = OSAL_eRBTreeIterate(
                            psObj->puInfo->sLL.uList.hRBTree,
                            tbIterator, pvArg
                                );
    }
    else
    {
        // Extract top-of-list element from list
        hTopEntry = psInfo->uList.sList.hTopOfListEntry;

        // Check if list is empty
        if(hTopEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Unlock list
            OSALLL_eUnlockDirect(psObj);
            return OSAL_NO_OBJECTS;
        }

        // Make the first element the current one
        psThis = (OSAL_LL_ELEMENT_STRUCT *)hTopEntry;

        do
        {
            // as mentioned above, the callback could delete the item from the list
            // if it delete's the first one we need to learn the new top so we
            // can stop when we wrap around
            psFirst = (OSAL_LL_ELEMENT_STRUCT *) psInfo->uList.sList.hTopOfListEntry;

            // Go get the next memory block
            // We do this first because the callback might end-up deleting
            // itself which would make iteration impossible.
            psNext = psThis->psNext;

            // Call the provided function for this element
            bRetVal = tbIterator( psThis->pvData, pvArg );

            // Make this the next one
            psThis = psNext;

        // Go again as long as we have not reached the start again
        } while ( (psThis != NULL) && (psThis != psFirst) && (bRetVal == TRUE) );
    }

    // Unlock list
    eLockReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eLockReturnCode != OSAL_SUCCESS)
    {
        eReturnCode = eLockReturnCode;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListItems
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListItems (
    OSAL_OBJECT_HDL hLL,
    UN32 *pun32Items)
{
    OSAL_RETURN_CODE_ENUM eReturnCode, eLockReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_LL_INFO_STRUCT *psInfo;

    // Verify a valid pointer was provided
    if(pun32Items == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock list
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    // Extract linked-list info from entry object
    psInfo = &psObj->puInfo->sLL;

    // Retrieve the number of items in the list
    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
                    == OSAL_LL_OPTION_RBTREE)
    {
        eReturnCode = OSAL_eRBTreeItems(
                            psObj->puInfo->sLL.uList.hRBTree,
                            pun32Items
                                );
    }
    else
    {
        *pun32Items = psInfo->uList.sList.un32Count;

        eReturnCode = OSAL_SUCCESS;
    }

    // Unlock list
    eLockReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eLockReturnCode != OSAL_SUCCESS)
    {
        eReturnCode = eLockReturnCode;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListProtect
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListProtect (
    OSAL_OBJECT_HDL hLL
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OS_OBJECT_HDL hOS;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Grab object OS handle
    hOS = psObj->hOS;

    // Check if this is a protected list already, if so don't protect it
    if(hOS != OS_INVALID_OBJECT_HDL)
    {
        // List is already protected
        eReturnCode = OSAL_SUCCESS;
    }
    else
    {
        const char *pacName;

#if OSAL_OBJECT_TRACKING == 1
        pacName = psObj->pacName;
#else
        pacName = "Unknown";
#endif

        // Create a unique name for the semaphore we need
        snprintf( acName, OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            "%s%.*s", pacName,
            (N32)(OSAL_MAX_OBJECT_NAME_LENGTH -
                            strlen(pacName)),
            "_Semaphore" );

        // Call OSAL API to protect list
        eReturnCode =
            OSAL.eSemCreate( (OSAL_OBJECT_HDL*)&hOS, acName, 1, 1,
                      OSAL_SEM_OPTION_NONE);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Store the mutex into the LL structure
            psObj->hOS = hOS;

            // Mark as protected
            psObj->puInfo->sLL.un32Options |= OSAL_LL_OPTION_PROTECT;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListUnprotect
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListUnprotect (
    OSAL_OBJECT_HDL hLL
        )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OS_OBJECT_HDL hOS;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Grab object OS handle
    hOS = psObj->hOS;

    // Check if this is not protected list, if it isn't don't unprotect it
    if(hOS == OS_INVALID_OBJECT_HDL)
    {
        // List is not protected
        return OSAL_ERROR;
    }

    // Unprotect list
    eReturnCode =
        OSAL.eSemDelete( (OSAL_OBJECT_HDL)hOS );
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Invalidate OS handle
        psObj->hOS = OS_INVALID_OBJECT_HDL;

        // Mark as unprotected
        psObj->puInfo->sLL.un32Options &= ~OSAL_LL_OPTION_PROTECT;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_eLinkedListLock
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListLock (
    OSAL_OBJECT_HDL hLL
        )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_HANDLE;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj != NULL)
    {
        // Call lock function
        eReturnCode = OSALLL_eLockDirect(psObj);
    }

    return eReturnCode;
}


/*****************************************************************************
*
*       OSAL_eLinkedListUnlock
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eLinkedListUnlock (
    OSAL_OBJECT_HDL hLL
        )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_HANDLE;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj != NULL)
    {
        // Call unlock function
        eReturnCode = OSALLL_eUnlockDirect(psObj);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSAL_pvLinkedListMemoryAllocate
*
*****************************************************************************/
void *OSAL_pvLinkedListMemoryAllocate(
    const char *pacName,
    size_t tSize,
    BOOLEAN bZeroInitialize
        )
{
    void *pvObject = NULL;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Initialize local name
    acName[0] = '\0';

    // Create a unique name for the semaphore we need
    snprintf( acName, OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
        OSAL_NAME_PREFIX"LL:Object:%s", (pacName == NULL ? "Unknown" : pacName)
            );

    // Call OS-Specific API to allocate memory required by the object itself,
    // plus the linked list element which will be required to add this
    // object to a linked list (support for pre-allocated elements).
    tSize += OSAL_LL_ELEMENT_STRUCT_SIZE;

    pvObject = OSALC_pvMemoryAllocate(acName, tSize, bZeroInitialize);
    if(pvObject != NULL)
    {
        // Establish the start of this object. Move past the pre-allocated
        // linked list element structure if one is required. This is where
        // the actual object begins.
        pvObject = ((char *)pvObject) + OSAL_LL_ELEMENT_STRUCT_SIZE;
    }

    return pvObject;
}

/*******************************************************************************
*
*   OSAL_vLinkedListMemoryFree
*
*******************************************************************************/
void OSAL_vLinkedListMemoryFree(
      void *pvObject
          )
{
    if (pvObject != NULL)
    {
        // De-reference object element to find actual memory pointer
        pvObject = ((char *)pvObject) - OSAL_LL_ELEMENT_STRUCT_SIZE;
        OSALC_vMemoryFree(pvObject);
    }

    return;
}

/*******************************************************************************
*
*   OSAL_tLinkedListEntrySize
*
*******************************************************************************/
size_t OSAL_tLinkedListEntrySize(
    void
        )
{
    return OSAL_LL_ELEMENT_STRUCT_SIZE;
}

/*****************************************************************************
*
*       OSALLL_n16ComparePayload
*
*        This is the function used to compare two objects for searching
*
*       Inputs:
*               pvObj1, pvObj2 - Objects to compare
*
*       Outputs:
*               0   - Objects have the same value (equal)
*               > 0 - Object1 is not equal to Object2
*
*****************************************************************************/
static N16 OSALLL_n16ComparePayload(
    void *pvObj1,
    void *pvObj2
        )
{
    N16 n16Result;

    n16Result = ((pvObj1 == pvObj2) ? 0 : 1);

    return n16Result;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_eInsert
*
* DESCRIPTION
*
*    Inserts a new item into a linked list.
*
* INPUTS
*    hLL           - the OSAL LL object itself
*
*    psInfo        - the OSAL LL info structure in order to avoid extra operation
*                    to gather it directly from the hLL
*
*    eInsertType   - before or after the item in psPosition
*
*    psPosition    - pointer to any member of an existing linked list.  The new
*                        item will be inserted following this member.
*                  - NULL = start a new list with this item.
*
*    ppsNewLink    - pointer to the control structure for the new item.
*
*    pvData        - pointer to the data to be added to the list.
*
*    bLinear       - TRUE if the list has a linear nature and FALSE otherwise
*
* OUTPUTS
*
*    OSAL_RETURN_CODE_ENUM - An OSAL return code
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALLL_eInsert(
    OSAL_OBJECT_HDL hLL,
    OSAL_LL_INFO_STRUCT *psInfo,
    OSAL_LL_INSERT_ENUM eInsertType,
    OSAL_LL_ELEMENT_STRUCT *psPosition,
    OSAL_LL_ELEMENT_STRUCT **ppsNewLink,
    void *pvData,
    BOOLEAN bLinear
        )
{
    // Don't even bother continuing if no link is given or if the insertion type
    // is out of the supported range
    if ( ( psInfo == NULL ) || ( ppsNewLink == NULL ) ||
         !( eInsertType < OSAL_LL_INSERT_MAX ) )
    {
        return OSAL_ERROR_CANNOT_ADD_OBJECT;
    }

    // If the reference position is NULL, this item is the first item in the
    // linked list, it is now the top item.  eInsertType does not matter in this
    // case because if there is nothing to insert after or before, we just put
    // the item in the list
    if( psPosition == NULL )
    {
        // Allocate memory for a new element in the linked list if the
        // linked-list options do not indicate pre-allocation of the element
        // was performed.
        *ppsNewLink = OSALLL_psAllocateEntry(psInfo, pvData);

        // Verify we have a valid pointer to work with
        if(*ppsNewLink == NULL)
        {
            return OSAL_ERROR_OUT_OF_MEMORY;
        }

        // Make this item the top
        psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);

        if(bLinear == TRUE)
        {
            // Linear lists do not wrap, so point to NULL
            (*ppsNewLink)->psNext = (struct osal_ll_element_struct *)NULL;
            (*ppsNewLink)->psPrev = (struct osal_ll_element_struct *)NULL;
        }
        else
        {
            // Non-linear lists wrap around to themselves.
            (*ppsNewLink)->psNext = (*ppsNewLink);
            (*ppsNewLink)->psPrev = (*ppsNewLink);
        }

        // Store the data address.
        (*ppsNewLink)->pvData = pvData;

        // Link this element to the linked list for which it belongs
        (*ppsNewLink)->hLL = hLL;

        // Either way, this element is the bottom item in the list
        psInfo->uList.sList.hBottomOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);

        return OSAL_SUCCESS;
    }

    // We are now adding an item to a list that already has items in it.
    // If we are adding the item to a list of unique items, make sure
    // this item doesn't exist
    if ((psInfo->un32Options & OSAL_LL_OPTION_UNIQUE) ==
        OSAL_LL_OPTION_UNIQUE)
    {
        OSAL_LL_ELEMENT_STRUCT *psEntry;

        // Do a search on this item we are adding
        if(psInfo->uList.sList.tn16CompareFunction == NULL)
        {
            psEntry = OSALLL_psSearch((OSAL_LL_ELEMENT_STRUCT *)
                                      psInfo->uList.sList.hTopOfListEntry, pvData,
                                      OSALLL_n16ComparePayload);
        }
        else
        {
            psEntry = OSALLL_psSortedSearch((OSAL_LL_ELEMENT_STRUCT *)
                                            psInfo->uList.sList.hTopOfListEntry, pvData,
                                            psInfo->uList.sList.tn16CompareFunction);
        }

        if (psEntry != NULL)
        {
            // We found a match
            // Bail out here and let the caller know this item
            // is already in the list. Return a handle to the existing
            // list entry for easy reference (if needed).
            (*ppsNewLink) = psEntry;
            return OSAL_ERROR_LIST_ITEM_NOT_UNIQUE;
        }
    }

    // Allocate memory for a new element in the linked list if the
    // linked-list options do not indicate pre-allocation of the element
    // was performed.
    *ppsNewLink = OSALLL_psAllocateEntry(psInfo, pvData);

    // Verify we have a valid pointer to work with
    if(*ppsNewLink == NULL)
    {
        return OSAL_ERROR_OUT_OF_MEMORY;
    }

    // If we plan on inserting the provided (*ppsNewLink) after psPosition and
    // psPosition is non-NULL
    if ( eInsertType == OSAL_LL_INSERT_AFTER )
    {
        // Insert after the given item.
        (*ppsNewLink)->psNext = psPosition->psNext;
        (*ppsNewLink)->psPrev = psPosition;
        if((*ppsNewLink)->psNext != NULL)
        {
            (*ppsNewLink)->psNext->psPrev = (*ppsNewLink);
        }

        psPosition->psNext = (*ppsNewLink);

        // Store the data address.
        (*ppsNewLink)->pvData = pvData;

        // Link this element to the linked list for which it belongs
        (*ppsNewLink)->hLL = hLL;

        // If the reference entry was at the end of the list, store a new end to the
        // list
        if ( psPosition == (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hBottomOfListEntry )
        {
            // this element is the new bottom item in the list
            psInfo->uList.sList.hBottomOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);
        }

        return OSAL_SUCCESS;
    }

    // If we plan on inserting the provided (*ppsNewLink) before psPosition and
    // psPosition is non-NULL
    if ( eInsertType == OSAL_LL_INSERT_BEFORE )
    {
        // Insert before the given item
        (*ppsNewLink)->psPrev = psPosition->psPrev;
        (*ppsNewLink)->psNext = psPosition;

        if ( (*ppsNewLink)->psPrev != NULL )
        {
            (*ppsNewLink)->psPrev->psNext = (*ppsNewLink);
        }

        psPosition->psPrev = (*ppsNewLink);

        // Store the data address.
        (*ppsNewLink)->pvData = pvData;

        // Link this element to the linked list for which it belongs
        (*ppsNewLink)->hLL = hLL;

        // If the reference entry was at the beginning of the list, store a new
        // beginning of the list
        if ( psPosition == (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hTopOfListEntry )
        {
            // this element is the new top item in the list
            psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);
        }

        return OSAL_SUCCESS;
    }

    return OSAL_ERROR_CANNOT_ADD_OBJECT;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_eSortedInsert
*
* DESCRIPTION
*
*    Inserts a new item into a sorted linked list.
*
* INPUTS
*    hLL           - the OSAL LL object itself
*
*    psInfo        - the OSAL LL info structure in order to avoid extra operation
*                    to gather it directly from the hLL
*
*    psList        - pointer to any member of an existing sorted linked list.
*                    - NULL = start a new list with this item.
*
*    ppsNewLink    - pointer to the control structure for the new item.
*
*    pvData        - pointer to the data to be added to the list.
*
*    pn16Compare   - pointer to a function used to compare data.
*
*    bLinear       - TRUE if the list has a linear nature and FALSE otherwise
*
* OUTPUTS
*
*    OSAL_RETURN_CODE_ENUM - An OSAL return code
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALLL_eSortedInsert(
    OSAL_OBJECT_HDL hLL,
    OSAL_LL_INFO_STRUCT *psInfo,
    OSAL_LL_ELEMENT_STRUCT *psCurrent,
    OSAL_LL_ELEMENT_STRUCT **ppsNewLink,
    void *pvData,
    OSAL_LL_COMPARE_HANDLER vCompare,
    BOOLEAN bLinear
        )
{
    OSAL_LL_ELEMENT_STRUCT *psStart, *psLastCompare;
    BOOLEAN bForward;
    N16 n16Result;

    if( ( ppsNewLink == NULL ) || ( psInfo == NULL ) )
    {
        // Don't even bother continuing if no link is given.
        return OSAL_ERROR_CANNOT_ADD_OBJECT;
    }

    // Check if it is the first one in the list
    if( psCurrent == NULL )
    {
        // Verify that we already have allocated memory
        if (*ppsNewLink == NULL)
        {
            // Allocate memory for a new element in the linked list if the
            // linked-list options do not indicate pre-allocation of the element
            // was performed.
            *ppsNewLink = OSALLL_psAllocateEntry(psInfo, pvData);

            // Verify we have a valid pointer to work with
            if(*ppsNewLink == NULL)
            {
                return OSAL_ERROR_OUT_OF_MEMORY;
            }
        }

        // This is the first one in the list
        if(bLinear == TRUE)
        {
            // Since it's the first (of a linear list) we simply
            // wrap around back to NULL.
            (*ppsNewLink)->psNext = (struct osal_ll_element_struct *)NULL;
            (*ppsNewLink)->psPrev = (struct osal_ll_element_struct *)NULL;
        }
        else
        {
            // Since it's the first (of a circular list) we simply
            // wrap around back to ourself.
            (*ppsNewLink)->psNext = (*ppsNewLink);
            (*ppsNewLink)->psPrev = (*ppsNewLink);
        }

        // Since this is the first one in the list, it IS the list
        // so this is the top and bottom list pointer.

        // Store the data address into this entry
        (*ppsNewLink)->pvData = pvData;

        // Link this element to the linked list for which it belongs
        (*ppsNewLink)->hLL = hLL;

        // Set the top of list entry
        psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);

        // Set the bottom of list entry
        psInfo->uList.sList.hBottomOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);
    }
    else if ((OSAL_LINKED_LIST_ENTRY)psCurrent == psInfo->uList.sList.hTopOfListEntry)
    {
        BOOLEAN bFound = FALSE;
        OSAL_LL_ELEMENT_STRUCT *psMid;
        size_t tLowIndex, tMidIndex, tHighIndex, tLastMidIndex, tIndexDiff;

        // If we were given a NULL compare handler, use
        // our default one
        if (vCompare == NULL)
        {
            vCompare = &OSALLL_n16ComparePayload;
        }

        // Initialize our search criteria
        bForward = TRUE;
        psMid = (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hTopOfListEntry;
        tLastMidIndex = tLowIndex = 0;
        tHighIndex = ((size_t)psInfo->uList.sList.un32Count) - 1;

        while (tLowIndex <= tHighIndex)
        {
            // What is the real index we'll be visiting next?
            tMidIndex = (tLowIndex + tHighIndex) / 2;

            // Okay, now find the distance from our last index
            if (bForward == TRUE)
            {
                tIndexDiff = (tMidIndex - tLastMidIndex);
            }
            else
            {
                tIndexDiff = (tLastMidIndex - tMidIndex);
            }

            tLastMidIndex = tMidIndex;

            // Get the node at that point
            psMid = OSALLL_psNextSearchPoint(psMid, bForward, tIndexDiff);
            if (psMid == NULL)
            {
                break;
            }

            // Okay, now find out which way to go
            n16Result = vCompare( psMid->pvData, pvData );
            if( n16Result < 0 )
            {
                // Move forward
                bForward = TRUE;

                // Look at the part of the list after this
                tLowIndex = tMidIndex + 1;
            }
            else if( n16Result > 0 )
            {
                // Move backward
                bForward = FALSE;

                // Look at the part of the list before this
                if (tMidIndex == 0)
                {
                    // Do not let it go before the beginning
                    break;
                }
                tHighIndex = tMidIndex - 1;
            }
            else
            {
                if ((psInfo->un32Options & OSAL_LL_OPTION_UNIQUE) ==
                    OSAL_LL_OPTION_UNIQUE)
                {
                    // Return a handle to the existing
                    // list entry for easy reference (if needed).
                    (*ppsNewLink) = psMid;
                    return OSAL_ERROR_LIST_ITEM_NOT_UNIQUE;
                }
                bFound = TRUE;
                break;
            }
        }

        if (psMid != NULL)
        {
            // Verify that we already have allocated memory
            if (*ppsNewLink == NULL)
            {
                // Allocate memory for a new element in the linked list if the
                // linked-list options do not indicate pre-allocation of the element
                // was performed.
                *ppsNewLink = OSALLL_psAllocateEntry(psInfo, pvData);

                // Verify we have a valid pointer to work with
                if(*ppsNewLink == NULL)
                {
                    return OSAL_ERROR_OUT_OF_MEMORY;
                }
            }

            // Store the data address into this entry
            (*ppsNewLink)->pvData = pvData;

            // Link this element to the linked list for which it belongs
            (*ppsNewLink)->hLL = hLL;

            // We're done with the search
            // Insert the object now
            if ((bFound == TRUE) || (bForward == TRUE))
            {
                // Insert after the last position compared (current)
                INSERT_ENTRY_AFTER(ppsNewLink, psMid);

                if(((*ppsNewLink)->psNext == NULL) || ((OSAL_LINKED_LIST_ENTRY)((*ppsNewLink)->psNext) == psInfo->uList.sList.hTopOfListEntry))
                {
                    psInfo->uList.sList.hBottomOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);
                }
            }
            else
            {
                // Insert before the last position compared (current)
                INSERT_ENTRY_BEFORE(ppsNewLink, psMid);

                if(((*ppsNewLink)->psPrev == NULL) || ((OSAL_LINKED_LIST_ENTRY)((*ppsNewLink)->psPrev) == psInfo->uList.sList.hBottomOfListEntry))
                {
                    psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);
                }
            }

        }
    }
    else
    {
        // There are others in the list already
        BOOLEAN bUpdate = FALSE;

        // If we were given a NULL compare handler, use
        // our default one
        if (vCompare == NULL)
        {
            vCompare = &OSALLL_n16ComparePayload;
        }

        // Set the search start point to the element provided as psCurrent
        psStart = psCurrent;

        // Initialize the previously compared entry which will
        // be so 'after' this comparison.
        psLastCompare = psCurrent;

        // Compare the current entry to the new entry
        n16Result = vCompare( psCurrent->pvData, pvData );

        if( n16Result > 0 )
        {
            // The current object is bigger than the new one
            // Since the new data comes before the current one
            // start searching backward
            bForward = FALSE;
            psCurrent = psCurrent->psPrev;

            // Since the link being inserted is 'less than'
            // the one provided as 'start' we return the new one
            // which is the new 'top of the list'
            bUpdate = TRUE; //psInfo->uList.sList.hTopOfListEntry = (*ppsNewLink);
        }
        else if ( n16Result == 0 &&
                 (psInfo->un32Options & OSAL_LL_OPTION_UNIQUE) ==
                  OSAL_LL_OPTION_UNIQUE)
        {
            // Return a handle to the existing
            // list entry for easy reference (if needed).
            (*ppsNewLink) = psCurrent;
            return OSAL_ERROR_LIST_ITEM_NOT_UNIQUE;
        }
        else
        {
            // If the new data comes after the current position or it is
            // equal to it, start searching forward.
            bForward = TRUE;
            psCurrent = psCurrent->psNext;

            // Since the link being inserted is 'greater than'
            // the one provided as 'start' we return the start item
            // which is the 'top of the list'
            psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY)psStart;
        }

        // While the list has not wrapped back to the
        // beginning or at the end of the list
        while( (psCurrent != NULL) && (psCurrent != psStart) )
        {
            // Initialize the previously compared entry which will
            // be so 'after' this comparison.
            psLastCompare = psCurrent;

            // Compare the current entry to the new entry
            n16Result = vCompare( psCurrent->pvData, pvData );

            if( bForward ) // Forward
            {
                if( n16Result <= 0 )
                {
                    if ( n16Result == 0 &&
                         (psInfo->un32Options & OSAL_LL_OPTION_UNIQUE) ==
                          OSAL_LL_OPTION_UNIQUE)
                    {
                        // Return a handle to the existing
                        // list entry for easy reference (if needed).
                        (*ppsNewLink) = psCurrent;
                        return OSAL_ERROR_LIST_ITEM_NOT_UNIQUE;
                    }
                    else
                    {
                        // The current object is less than or equal to
                        // the new one so we continue searching...
                        psCurrent = psCurrent->psNext;
                    }
                }
                else
                {
                    // The current object is bigger than the new one
                    // so the new link will be inserted before this one.
                    break;
                }
            }
            else // Reverse
            {
                if( n16Result <  0 )
                {
                    // The current object is bigger than the new one
                    // so we continue searching...
                    psCurrent = psCurrent->psPrev;
                }
                else if ( n16Result == 0 &&
                         (psInfo->un32Options & OSAL_LL_OPTION_UNIQUE) ==
                          OSAL_LL_OPTION_UNIQUE)
                {
                    // Return a handle to the existing
                    // list entry for easy reference (if needed).
                    (*ppsNewLink) = psCurrent;
                    return OSAL_ERROR_LIST_ITEM_NOT_UNIQUE;
                }
                else
                {
                    // The current object is less than or equal to
                    // the new one, so the new link will be inserted after
                    // this one.
                    break;
                }
            }
        }

        // If the current position is NULL then we are either at the
        // end of the list or moved past the beginning (in a linear
        // list). If the current list is at the start then we have
        // wrapped around. Either way, We need to recall the last
        // entry we had.
        if( (psCurrent == NULL) || (psCurrent == psStart) )
        {
            // Recall the last entry we compared to
            psCurrent = psLastCompare;

            // Reverse direction flag
            // This is because if we are going forward and we found the end
            // of the list before we found something larger then we need to
            // tack it onto the end of the list, which means 'after' the
            // current one (see below).
            // Similarly if we are going in reverse and we found the end
            // of the list (beginning actually), we need to tack it onto the
            // beginning of the list which means 'before' the current one
            // (see below).
            bForward = (bForward == TRUE) ? FALSE : TRUE;
        }

        // Verify that we already have allocated memory
        if (*ppsNewLink == NULL)
        {
            // Allocate memory for a new element in the linked list if the
            // linked-list options do not indicate pre-allocation of the element
            // was performed.
            *ppsNewLink = OSALLL_psAllocateEntry(psInfo, pvData);

            // Verify we have a valid pointer to work with
            if(*ppsNewLink == NULL)
            {
                return OSAL_ERROR_OUT_OF_MEMORY;
            }
        }

        // Store the data address into this entry
        (*ppsNewLink)->pvData = pvData;

        // Link this element to the linked list for which it belongs
        (*ppsNewLink)->hLL = hLL;

        // We're done with the search
        // Insert the object now
        if(bForward == TRUE) // Forward
        {
            // Insert before the last position compared (current)
            INSERT_ENTRY_BEFORE(ppsNewLink, psCurrent);
        }
        else
        {
            // Insert after the last position compared (current)
            INSERT_ENTRY_AFTER(ppsNewLink, psCurrent);

            // Set the bottom of list entry
            psInfo->uList.sList.hBottomOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);
        }

        if (bUpdate == TRUE)
        {
            psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY)(*ppsNewLink);
        }
    }

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_psRemove
*
* DESCRIPTION
*
*    Remove an item from a linked list.
*
* INPUTS
*
*    psLink    - pointer to the list item to be removed.
*
* OUTPUTS
*
*    None
*
*******************************************************************************/
static void OSALLL_vRemove(
    OSAL_LL_ELEMENT_STRUCT *psLink
        )
{
    OSAL_LL_INFO_STRUCT *psInfo;

    if(psLink == NULL)
    {
        // Don't even bother continuing if the pointers aren't right.
        return;
    }

    // Point the surrounding items around the removed item.
    if(psLink->psNext != NULL)
    {
        psLink->psNext->psPrev = psLink->psPrev;
    }

    if(psLink->psPrev != NULL)
    {
        psLink->psPrev->psNext = psLink->psNext;
    }

    // Extract linked-list info from object
    psInfo = &((OSAL_OBJECT_STRUCT *)psLink->hLL)->puInfo->sLL;

    // Check if the one I removed is my top of the list
    if(psInfo->uList.sList.hTopOfListEntry == (OSAL_LINKED_LIST_ENTRY)psLink)
    {
        // I removed the top one, so the new top is the one
        // I removed's next one in the list. Determine if there is a next one.
        if( (psLink->psNext == NULL) || (psLink->psNext == psLink) )
        {
            // No more items
            psInfo->uList.sList.hTopOfListEntry  = NULL;
        }
        else
        {
            // There is another item which is next, so it is now on top
            psInfo->uList.sList.hTopOfListEntry = (OSAL_LINKED_LIST_ENTRY)psLink->psNext;
        }
    }

    // Check if the one I removed is my bottom of the list
    if(psInfo->uList.sList.hBottomOfListEntry == (OSAL_LINKED_LIST_ENTRY)psLink)
    {
        // I removed the bottom one, so the new bottom is the one
        // I removed's previous one in the list. Determine if there is
        // a previous one.
        if( (psLink->psPrev == NULL) || (psLink->psPrev == psLink) )
        {
            // No more items
            psInfo->uList.sList.hBottomOfListEntry  = NULL;
        }
        else
        {
            // There is another item which is previous, so it is now
            // on the bottom.
            psInfo->uList.sList.hBottomOfListEntry = (OSAL_LINKED_LIST_ENTRY)psLink->psPrev;
        }
    }

    // Clear out this item's links.
    psLink->psPrev = (struct osal_ll_element_struct *)NULL;
    psLink->psNext = (struct osal_ll_element_struct *)NULL;
    psLink->pvData = NULL;

    return;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_eRemoveDirect
*
* DESCRIPTION
*
*    Remove an item from a linked list including memory release.
*
* INPUTS
*
*    psInfo    - pointer to the list information structure.
*    hEntry    - pointer to the list item to be removed.
*
* OUTPUTS
*
*    None
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSALLL_eRemoveDirect(
    OSAL_LL_INFO_STRUCT *psInfo,
    OSAL_LINKED_LIST_ENTRY hEntry
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
                == OSAL_LL_OPTION_RBTREE)
    {
        eReturnCode = OSAL_eRBTreeRemove((OSAL_RB_TREE_ENTRY)hEntry);
    }
    else
    {
        OSAL_LL_ELEMENT_STRUCT *psEntry =
            (OSAL_LL_ELEMENT_STRUCT *)hEntry;

        // Remove the object
        OSALLL_vRemove( psEntry );

        // Remove was successful, decrement count
        --psInfo->uList.sList.un32Count;

        // Return allocated memory if we locally allocated it in
        // the first place. Otherwise leave it up to the caller.
        if( (psInfo->un32Options &
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS) !=
                OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS )
        {
            OSALC_vMemoryFree(psEntry);
        }

        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_psSearch
*
* DESCRIPTION
*
*    Search the list for a matching item.
*
* INPUTS
*
*    psList        - pointer to any item in the list to be searched.
*
*    pvData        - pointer to the data to be found.
*
*    tn16Compare    - pointer to the function used to compare data.
*
* OUTPUTS
*
*    OSAL_LL_ELEMENT_STRUCT *    - pointer to the in the matching link.
*                - NULL = no match found
*
*******************************************************************************/
static OSAL_LL_ELEMENT_STRUCT *OSALLL_psSearch(
    OSAL_LL_ELEMENT_STRUCT *psList,
    void *pvData,
    OSAL_LL_COMPARE_HANDLER tn16Compare
        )
{
    OSAL_LL_ELEMENT_STRUCT *psStart, *psFound = (OSAL_LL_ELEMENT_STRUCT *)NULL;

    if(psList == NULL)
    {
        // Don't even bother continuing if the pointers aren't right.
        return (OSAL_LL_ELEMENT_STRUCT *)NULL;
    }

    // Set the starting position.
    psStart = psList;

    do
    {
        // If a match was found then exit the loop.
        if( tn16Compare( psList->pvData, pvData ) == 0 )
        {
            psFound = psList;
            break;
        }

        psList = psList->psNext;

    }    while( (psList != NULL) && (psList != psStart) );

    return psFound;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_psSortedSearch
*
* DESCRIPTION
*
*    Search a sorted list for an item.
*
* INPUTS
*
*    psList        - pointer to any item in the list to be searched.
*
*    pvData        - pointer to the data to be found.
*
*    tn16Compare    - pointer to the function used to compare data.
*
* OUTPUTS
*
*    OSAL_LL_ELEMENT_STRUCT *    - pointer to the data in the matching link.
*                - NULL = no match found
*
*******************************************************************************/
static OSAL_LL_ELEMENT_STRUCT *OSALLL_psSortedSearch(
    OSAL_LL_ELEMENT_STRUCT *psList,
    void *pvData,
    OSAL_LL_COMPARE_HANDLER tn16Compare
        )
{
    N16 n16Result;
    BOOLEAN bForward = FALSE;
    OSAL_LL_ELEMENT_STRUCT *psStart, *psFound = (OSAL_LL_ELEMENT_STRUCT *)NULL;

    if(psList == NULL)
    {
        // Don't even bother continuing if the pointers aren't right.
        return (OSAL_LL_ELEMENT_STRUCT *)NULL;
    }

    // Set the search start point.
    psStart = psList;

    n16Result = tn16Compare( psList->pvData, pvData );

    if( n16Result < 0 )
    {
        // If the data comes after the current position
        // then start searching forward.
        bForward = TRUE;
        psList = psList->psNext;
    }
    else if( n16Result > 0 )
    {
        // Search backwards.
        bForward = FALSE;
        psList = psList->psPrev;
    }
    else
    {
        // Match already found, do no more.
        psFound = psList;
    }

    while( (psList != NULL) && (psList != psStart) )
    {
        n16Result = tn16Compare( psList->pvData, pvData );

        if( n16Result == 0 )
        {
            // Match found.
            psFound = psList;

            // Exit loop.
            break;
        }

        if( bForward == TRUE )
        {
            // Searching forward...
            if( n16Result < 0 )
            {
                // The data is still after the
                // current position, continue searching...
                psList = psList->psNext;
            }
            else
            {
                // The data comes before the current link
                // so there's no point in continuing the search.
                break;
            }
        }
        else
        {
            // Searching backward...
            if( n16Result > 0 )
            {
                // The data is still before the
                // current position, continue searching...
                psList = psList->psPrev;
            }
            else
            {
                // The new data comes after the current link
                // so there's no point in continuing the search.
                break;
            }
        }
    }

    return psFound;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_psSortedBinarySearch
*
* DESCRIPTION
*
*    Search a sorted list for an item using a binary search algorithm.
*
* INPUTS
*
*    psInfo        - pointer to the list info structure to search on
*
*    pvData        - pointer to the data to be found.
*
*    tn16Compare    - pointer to the function used to compare data.
*
* OUTPUTS
*
*    OSAL_LL_ELEMENT_STRUCT *    - pointer to the data in the matching link.
*                - NULL = no match found
*
*******************************************************************************/
static OSAL_LL_ELEMENT_STRUCT *OSALLL_psSortedBinarySearch(
    OSAL_LL_INFO_STRUCT *psInfo,
    void *pvData,
    OSAL_LL_COMPARE_HANDLER tn16Compare
        )
{
    N16 n16Result;
    BOOLEAN bForward = TRUE;
    OSAL_LL_ELEMENT_STRUCT *psMid;
    size_t tLowIndex, tMidIndex, tHighIndex, tLastMidIndex, tIndexDiff;

    if(psInfo == NULL)
    {
        // Don't even bother continuing if the pointers aren't right.
        return (OSAL_LL_ELEMENT_STRUCT *)NULL;
    }

    // Initialize our search criteria
    psMid = (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hTopOfListEntry;
    tLastMidIndex = tLowIndex = 0;
    tHighIndex = ((size_t)psInfo->uList.sList.un32Count) - 1;

    if (psMid == NULL)
    {
        // Nothing to search for
        return (OSAL_LL_ELEMENT_STRUCT *)NULL;
    }

    while (tLowIndex <= tHighIndex)
    {
        // What is the real index we'll be visiting next?
        tMidIndex = (tLowIndex + tHighIndex) / 2;

        // Okay, now find the distance from our last index
        if (bForward == TRUE)
        {
            tIndexDiff = (tMidIndex - tLastMidIndex);
        }
        else
        {
            tIndexDiff = (tLastMidIndex - tMidIndex);
        }

        tLastMidIndex = tMidIndex;

        // Get the node at that point
        psMid = OSALLL_psNextSearchPoint(psMid, bForward, tIndexDiff);
        if (psMid == NULL)
        {
            break;
        }

        // Okay, now find out which way to go
        n16Result = tn16Compare( psMid->pvData, pvData );
        if( n16Result < 0 )
        {
            // Move forward
            bForward = TRUE;

            // Look at the part of the list after this
            tLowIndex = tMidIndex + 1;
        }
        else if( n16Result > 0 )
        {
            // Move backward
            bForward = FALSE;

            // Look at the part of the list before this
            if (tMidIndex == 0)
            {
                // Do not let it go before the beginning
                break;
            }
            tHighIndex = tMidIndex - 1;
        }
        else
        {
            return psMid;
        }
    }

    return (OSAL_LL_ELEMENT_STRUCT *)NULL;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_psNextSearchPoint
*
* DESCRIPTION
*
*    Get the next node for the binary search algorithm
*
* INPUTS
*
*    psStart   - the point from which to start
*
*    bForward  - flag indicating which direction to go from psStart
*
*    tCount    - Number of nodes from the start point to move
*
* OUTPUTS
*
*    OSAL_LL_ELEMENT_STRUCT *    - pointer to the node at the requested location
*                - NULL = no node found
*
*******************************************************************************/
static OSAL_LL_ELEMENT_STRUCT *OSALLL_psNextSearchPoint (
    OSAL_LL_ELEMENT_STRUCT *psStart,
    BOOLEAN bForward,
    size_t tCount
        )
{
    OSAL_LL_ELEMENT_STRUCT *psCur = psStart;
    size_t tIndex;

    if (bForward == TRUE)
    {
        for (tIndex = 0; (tIndex < tCount) && (psCur != NULL); tIndex++)
        {
            psCur = psCur->psNext;
        }
    }
    else
    {
        for (tIndex = 0; (tIndex < tCount) && (psCur != NULL); tIndex++)
        {
            psCur = psCur->psPrev;
        }
    }

    return psCur;
}

/*******************************************************************************
*
* FUNCTION
*
*    OSALLL_psSortedMerge
*
* DESCRIPTION
*
*    Merge two lists represented by their head references in sorted order.
*
* INPUTS
*
*    psEntry1      - pointer to the list #1 head
*
*    psEntry2      - pointer to the list #2 head
*
*    tn16Compare   - pointer to the function used to compare data.
*
* OUTPUTS
*
*    Head of the merged list
*
*******************************************************************************/
static OSAL_LL_ELEMENT_STRUCT *OSALLL_psSortedMerge(
    OSAL_LL_ELEMENT_STRUCT *psEntry1,
    OSAL_LL_ELEMENT_STRUCT *psEntry2,
    OSAL_LL_COMPARE_HANDLER tn16Compare
        )
{
    OSAL_LL_ELEMENT_STRUCT *psCurEntry;
    OSAL_LL_ELEMENT_STRUCT *psResult;
    N16 n16Result;

    // Select the merged list head entry
    n16Result = tn16Compare(psEntry1->pvData, psEntry2->pvData);
    if (n16Result <= 0)
    {
        psCurEntry = psEntry1;
        psEntry1 = psEntry1->psNext;
    }
    else
    {
        psCurEntry = psEntry2;
        psEntry2 = psEntry2->psNext;
    }

    // From here result is the same as we selected
    psResult = psCurEntry;

    // Merge rest pars of the lists
    while ((psEntry1 != NULL) && (psEntry2 != NULL))
    {
        // Compare entries
        n16Result = tn16Compare(psEntry1->pvData, psEntry2->pvData);
        if (n16Result <= 0)
        {
            // The first is our candidate
            psCurEntry->psNext = psEntry1;
            psCurEntry = psEntry1;
            psEntry1 = psEntry1->psNext;
        }
        else
        {
            // The second is our candidate
            psCurEntry->psNext = psEntry2;
            psCurEntry = psEntry2;
            psEntry2 = psEntry2->psNext;
        }
    }

    // Finalize merging
    psCurEntry->psNext = (psEntry1 != NULL) ? psEntry1 : psEntry2;

    return psResult;
}

/*******************************************************************************
*
*   OSALLL_eAllocateAndAddDirect
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALLL_eAllocateAndAddDirect (
    OSAL_LL_INSERT_ENUM eHow,
    OSAL_OBJECT_HDL hLL,
    OSAL_LL_INFO_STRUCT *psInfo,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LL_ELEMENT_STRUCT *psNewElement = NULL, *psHere = NULL;
    BOOLEAN bLinear = FALSE, bSortedInsert = FALSE;

    // Note: It is possible the caller wishes to add an object
    // which has value of zero. So testing pvData for NULL is not desired.

    // Based on list options, set the linear list flag
    bLinear = ( (psInfo->un32Options & OSAL_LL_OPTION_LINEAR)
        == OSAL_LL_OPTION_LINEAR ) ?
        TRUE : FALSE;

    // Determine how to handle this element
    switch(eHow)
    {
        case OSAL_LL_INSERT_BEFORE:
        case OSAL_LL_INSERT_AFTER:
        {
            // Direct inserts
            psHere = (OSAL_LL_ELEMENT_STRUCT *)*phEntry;
        }
        break;

        case OSAL_LL_INSERT_HERE:
        {
            // Sorted or directed insert?
            if(psInfo->uList.sList.tn16CompareFunction == NULL)
            {
                // Directed (bottom)
                eHow = OSAL_LL_INSERT_AFTER;
                psHere = (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hBottomOfListEntry;
            }
            else
            {
                // Sorted
                psHere = (OSAL_LL_ELEMENT_STRUCT *)psInfo->uList.sList.hTopOfListEntry;
                bSortedInsert = TRUE;
            }
        }
        break;

        default:
        {
            // Error!
            return OSAL_ERROR_INVALID_INPUT;
        }
    }

	// Insert based on flag
    if(bSortedInsert == FALSE)
    {
        eReturnCode =
            OSALLL_eInsert(
                hLL,
                psInfo,
                eHow,
                // Ptr to any member of LL, insert happens after/before
                // this element
                psHere,
                // Ptr to recently allocated ctrl struct for the item
                &psNewElement,
                // Data to be added to the list
                pvData,
                // Linear flag
                bLinear
                    );
    }
    else
    {
        eReturnCode =
            OSALLL_eSortedInsert(
                hLL,
                psInfo,
                // Ptr to any member of LL, insert happens after
                // this element
                psHere,
                // Ptr to any member of LL, insert happens after this element
                &psNewElement,
                // Data to be added to the list
                pvData,
                // Compare method
                psInfo->uList.sList.tn16CompareFunction,
                // Linear flag
                bLinear
                    );
    }

    // Check if element insert was successful
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Insert was successful, increment count
        psInfo->uList.sList.un32Count++;
        if(phEntry != NULL)
        {
            // Return the entry (as a handle)
            *phEntry = (OSAL_LINKED_LIST_ENTRY)psNewElement;
        }
    }
    else
    {
        // Check if the element to be added was not unique. If so,
        // then do the caller a favor by returning the element which
        // exists already.
        if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
        {
            // Return existing element
            // (if they provided a pointer for it)
            if(phEntry != NULL)
            {
                // Return the entry (as a handle)
                *phEntry = (OSAL_LINKED_LIST_ENTRY)psNewElement;
            }
        }
        // Return NULL element (if they provided a pointer for it)
        else if(phEntry != NULL)
        {
            // Return the entry (as a handle)
            *phEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALLL_eAllocateAndAdd
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALLL_eAllocateAndAdd (
    OSAL_LL_INSERT_ENUM eHow,
    OSAL_OBJECT_HDL hLL,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode, eLockReturnCode;
    OSAL_OBJECT_STRUCT *psObj = NULL;
    OSAL_LL_INFO_STRUCT *psInfo = NULL;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hLL, OSAL_OBJECT_TYPE_LINKED_LIST);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock list that this element is going to be added to
    eReturnCode = OSALLL_eLockDirect(psObj);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }

    // Extract linked-list info from new entry object
    psInfo = &psObj->puInfo->sLL;

    if ((psInfo->un32Options & OSAL_LL_OPTION_RBTREE)
                == OSAL_LL_OPTION_RBTREE)
    {
        if (eHow == OSAL_LL_INSERT_HERE)
        {
            eReturnCode = OSAL_eRBTreeAdd(
                psInfo->uList.hRBTree,
                (OSAL_RB_TREE_ENTRY*)phEntry,
                pvData);
        }
        else
        {
            eReturnCode = OSAL_ERROR_UNSUPPORTED_API;
        }
    }
    else
    {
        eReturnCode = OSALLL_eAllocateAndAddDirect(
                                eHow, hLL, psInfo,
                                phEntry, pvData
                                    );
    }

    // Unlock list
    eLockReturnCode = OSALLL_eUnlockDirect(psObj);
    if(eLockReturnCode != OSAL_SUCCESS)
    {
        eReturnCode = eLockReturnCode;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALLL_psAllocateEntry
*
*******************************************************************************/
static OSAL_LL_ELEMENT_STRUCT *OSALLL_psAllocateEntry(
    OSAL_LL_INFO_STRUCT *psInfo,
    void *pvData
        )
{
    OSAL_LL_ELEMENT_STRUCT *psNewElement;

    // Allocate memory for a new element in the linked list if the
    // linked-list options do not indicate pre-allocation of the element
    // was performed.
    if( (psInfo->un32Options & OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS) ==
        OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS )
    {
        psNewElement = (OSAL_LL_ELEMENT_STRUCT *)pvData - 1;
    }
    else
    {
        psNewElement = (OSAL_LL_ELEMENT_STRUCT *)
            OSALC_pvMemoryAllocate( OSAL_NAME_PREFIX"LL Element",
            OSAL_LL_ELEMENT_STRUCT_SIZE, FALSE );
    }

    return psNewElement;
}

/*******************************************************************************
*
*   OSALLL_eLockDirect
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALLL_eLockDirect(
    OSAL_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OS_OBJECT_HDL hOS;

    // Obtain OS-depended object handle
    hOS = psObj->hOS;
    if (hOS != OS_INVALID_OBJECT_HDL)
    {
        // Lock the list
        eReturnCode = OSAL.eSemTake((OSAL_OBJECT_HDL)hOS, OSAL_OBJ_TIMEOUT_INFINITE);
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALLL_eUnlockDirect
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALLL_eUnlockDirect(
    OSAL_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OS_OBJECT_HDL hOS;

    // Obtain OS-depended object handle
    hOS = psObj->hOS;

    if (hOS != OS_INVALID_OBJECT_HDL)
    {
        // Unlock the list
        eReturnCode = OSAL.eSemGive((OSAL_OBJECT_HDL)hOS);
    }

    return eReturnCode;
}

#ifdef SUPPORT_CUNIT
#include <osal_ll.cunit>
#endif
