/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*DESCRIPTION
*
*       This module will contain all the OSAL general APIs.
*
*******************************************************************************/

#include <stdlib.h>

#include "standard.h"

#include "osal.h"
#include "osal_version.h"
#include "osal_core.h"

#include "os_intf.h"

#include "_osal_general.h"

/*******************************************************************************
*
* OSAL_tConfigInfo
*
*******************************************************************************/
OSAL_CONFIG_INFO_MASK OSAL_tConfigInfo( void )
{
    OSAL_CONFIG_INFO_MASK tCurrentConfig = OSAL_CONFIG_NONE;

#if OSAL_FILE_SYSTEM == 1
    tCurrentConfig |= OSAL_CONFIG_FILE_SYSTEM;
#endif

#if OSAL_LOG == 1
    tCurrentConfig |= OSAL_CONFIG_LOG;
#endif

#if OSAL_BUFFER == 1
    tCurrentConfig |= OSAL_CONFIG_BUFFER;
#endif

#if OSAL_CRC == 1
    tCurrentConfig |= OSAL_CONFIG_CRC;
#endif

#if OSAL_INTERRUPT == 1
    tCurrentConfig |= OSAL_CONFIG_INTERRUPT;
#endif

#if OSAL_DEVICES == 1
    tCurrentConfig |= OSAL_CONFIG_DEVICES;
#endif

#if OSAL_TEST == 1
    tCurrentConfig |= OSAL_CONFIG_TEST;
#endif

#if OSAL_FIXED_MATH == 1
    tCurrentConfig |= OSAL_CONFIG_FIXED_MATH;
#endif

#ifdef OSAL_USE_NATIVE_STDIO
    tCurrentConfig |= OSAL_CONFIG_NATIVE_STDIO;
#endif

    return tCurrentConfig;
}

/*******************************************************************************
*
*   OSAL_bStart
*
*******************************************************************************/
BOOLEAN OSAL_bStart (
    const OSAL_START_HANDLER_STRUCT *psStartHandlers
        )
{
    BOOLEAN bRetval;

    // Call OS-Specific API since we would like the underlying OS to
    // initialize(start) itself, then we can do our thing.
    bRetval = OS.bStart(OSALC_bBootstrap, psStartHandlers);

    return bRetval;
}

/*******************************************************************************
*
*   OSAL_bSleep
*
*******************************************************************************/
BOOLEAN OSAL_bSleep ( void )
{
    OSAL_OBJECT_HDL hTaskList;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Iterate the list of tasks and call each one's sleep callback

    // Grab the task linked list
    hTaskList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TASK);
    if(hTaskList == OSAL_INVALID_OBJECT_HDL)
    {
        return FALSE;
    }

    // Iterate the list calling the sleep callback
    eReturnCode = OSAL.eLinkedListIterate(hTaskList,
        OSALG_bNotifyTask, (void *)OSALG_NOTIFY_SLEEP );
    if( (eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS) )
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   OSAL_bWakeup
*
*******************************************************************************/
BOOLEAN OSAL_bWakeup ( void )
{
    OSAL_OBJECT_HDL hTaskList;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Iterate the list of tasks and call each one's wake-up callback

    // Grab the task linked list
    hTaskList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TASK);
    if(hTaskList == OSAL_INVALID_OBJECT_HDL)
    {
        return FALSE;
    }

    // Iterate the list calling the wake-up callback
    eReturnCode = OSAL.eLinkedListIterate(hTaskList,
        OSALG_bNotifyTask, (void *)OSALG_NOTIFY_WAKEUP );
    if( (eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS) )
    {
        return FALSE;
    }

    return TRUE;
}


/*******************************************************************************
*
*   OSAL_bShutdown
*
*******************************************************************************/
BOOLEAN OSAL_bShutdown ( void )
{
    OSAL_OBJECT_HDL hTaskList;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Iterate the list of tasks and call each one's shutdown callback

    // Grab the task linked list
    hTaskList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TASK);
    if(hTaskList == OSAL_INVALID_OBJECT_HDL)
    {
        return FALSE;
    }

    // Iterate the list calling the shutdown callback
    eReturnCode = OSAL.eLinkedListIterate(hTaskList,
        OSALG_bNotifyTask, (void *)OSALG_NOTIFY_SHUTDOWN );
    if( (eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS) )
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   OSAL_vEnterCriticalSection
*
*******************************************************************************/
void OSAL_vEnterCriticalSection ( void )
{
   // Call OS-Specific API
   OS.vEnterCriticalSection();
#ifdef OSAL_TEST_CRITICAL_SECTION
#if OSAL_DEBUG == 1
#if __ICCARM__ != 1 // We will allow nested calls for uCOS-II (for now)
   vTestEnterSection(&gsCritical);
#endif
#endif
#endif

   return;
}

/*******************************************************************************
*
*   OSAL_vExitCriticalSection
*
*******************************************************************************/
void OSAL_vExitCriticalSection ( void )
{
#ifdef OSAL_TEST_CRITICAL_SECTION
#if OSAL_DEBUG == 1
#if __ICCARM__ != 1 // We will allow nested calls for uCOS-II (for now)
    vTestExitSection(&gsCritical);
#endif
#endif
#endif

    // Call OS-Specific API
    OS.vExitCriticalSection();

    return;
}

/*******************************************************************************
*
*   OSAL_eEnterTaskSafeSection
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eEnterTaskSafeSection ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bLocked;

    bLocked = OSALC_bLockCore();
    if(bLocked == TRUE)
    {
        eReturnCode = OSAL_SUCCESS;
    }
    else
    {
        eReturnCode = OSAL_ERROR;
    }

#ifdef OSAL_TEST_CRITICAL_SECTION
#if OSAL_DEBUG == 1
    vTestEnterSection(&gsTaskSafe);
#endif
#endif

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eExitTaskSafeSection
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eExitTaskSafeSection ( void )
{

#ifdef OSAL_TEST_CRITICAL_SECTION
#if OSAL_DEBUG == 1
    vTestExitSection(&gsTaskSafe);
#endif
#endif

    OSALC_vUnlockCore();
    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSAL_eGetUtilizationCPU
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eGetUtilizationCPU(
    UN8 *pun8Current,
    UN8 *pun8Max,
    UN8 *pun8Min
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;

    // Check for sensible inputs
    if(
        (pun8Current != NULL) ||
        (pun8Max != NULL) ||
        (pun8Min != NULL)
        )
    {
        // Call OS-Specific API
        eReturnCode = OS.eGetUtilizationCPU(pun8Current, pun8Max, pun8Min);
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_vResetUtilizationCPU
*
*******************************************************************************/
void OSAL_vResetUtilizationCPU ( void )
{
    // Call OS-Specific API
    OS.vResetUtilizationCPU();

    return;
}

/*******************************************************************************
*
*   OSAL_pacGetReturnCodeName
*
*******************************************************************************/
const char *OSAL_pacGetReturnCodeName (
    OSAL_RETURN_CODE_ENUM eReturnCode
        )
{
    const char *pacReturnCodeName = "Unknown";
    UN8 un8Index = 0;

    // Translate enum to a text map
    do
    {
        if(eReturnCode == gsReturnCodeMap[un8Index].eReturnCode)
        {
            break;
        }

    } while(gsReturnCodeMap[++un8Index].eReturnCode !=
            OSAL_ERROR_UNKNOWN);

    // Set the current text
    pacReturnCodeName =
        gsReturnCodeMap[un8Index].pacText;

    return pacReturnCodeName;
}

/*******************************************************************************
*
*   OSAL_pacGetVersion
*
*******************************************************************************/
const char *OSAL_pacGetVersion (
    UN8 *pun8MajorVersion,
    UN8 *pun8MinorVersion,
    UN8 *pun8SubVersion
    )
{
    const char *pacVersion = PROJECT_NAME" (v" MAJOR_VERSION "."
        MINOR_VERSION "." SUBVERSION ") " DATE;

    if(pun8MajorVersion != NULL)
    {
        *pun8MajorVersion = (UN8)atoi(MAJOR_VERSION);
    }
    if(pun8MinorVersion != NULL)
    {
        *pun8MinorVersion = (UN8)atoi(MINOR_VERSION);
    }
    if(pun8SubVersion != NULL)
    {
        *pun8SubVersion = (UN8)atoi(SUBVERSION);
    }

    return pacVersion;
}

/*******************************************************************************
*
*   OSAL_pacGetObjectName
*
*******************************************************************************/
const char *OSAL_pacGetObjectName (
    OSAL_OBJECT_HDL hObj
        )
{
    const char *pacObjectName = "Unknown";
    OSAL_OBJECT_STRUCT *psObj = NULL;

#if OSAL_OBJECT_TRACKING == 1

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hObj, OSAL_OBJECT_TYPE_UNKNOWN);
    if(psObj != NULL)
    {
        // Extract name
        pacObjectName = psObj->pacName;
    }
#else
    psObj = psObj;
#endif

    return pacObjectName;
}

/*******************************************************************************
*
*   OSALG_bNotifyTask
*
*******************************************************************************/
static BOOLEAN OSALG_bNotifyTask ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData;
    OSALG_NOTIFY_TYPE_ENUM eNotifyType =
        (OSALG_NOTIFY_TYPE_ENUM)(UN32)pvArg;
    OSAL_TASK_INFO_STRUCT *psTaskInfo;

    // Check if data is valid
    if(psObj == NULL)
    {
        // Stop iterating
        return FALSE;
    }

    // Point to this object's info
    psTaskInfo = &psObj->puInfo->sTask;

    // Call apropriate notification function based on type provided
    switch(eNotifyType)
    {
        case OSALG_NOTIFY_SLEEP:

            // Call sleep handler...
            if(psTaskInfo->tSleepHandler != NULL)
            {
                psTaskInfo->tSleepHandler(
                    psTaskInfo->pvHandlerArgument);
            }

        break;

        case OSALG_NOTIFY_WAKEUP:

            // Call wakeup handler...
            if(psTaskInfo->tWakeupHandler != NULL)
            {
                psTaskInfo->tWakeupHandler(
                    psTaskInfo->pvHandlerArgument);
            }

        break;

        case OSALG_NOTIFY_SHUTDOWN:

            // Call shutdown handler...
            if(psTaskInfo->tShutdownHandler != NULL)
            {
                psTaskInfo->tShutdownHandler(
                    psTaskInfo->pvHandlerArgument);
            }

        break;

        default:
            // Do nothing...
        break;
    }

    // Keep iterating...
    return TRUE;
}

#ifdef OSAL_TEST_CRITICAL_SECTION
#if OSAL_DEBUG == 1
/*******************************************************************************
*
*   vTestEnterSection
*
*******************************************************************************/
static void vTestEnterSection (OSALG_SECTION_ASSOCIATION_STRUCT volatile *psSection)
{
    // Check our section structure for sanity.
    if(psSection->tCounter != 0)
    {
        // Error! Something went wrong. This should never be anything
        // but zero at this point since no one should be able to enter
        // this section unless it has been unlocked.
        // TODO: BREAKPOINT
        printf("Error! Enter section where tCounter = %u.\n",
            psSection->tCounter);
        // Segmentation fault to check backtrace.
        *((char*)0) = 1;
    }
    else
    {
        OSAL_OBJECT_STRUCT *psObj;

        // Record task info
        psSection->hTask = OSAL.hTaskGetHandle();
        psObj = (OSAL_OBJECT_STRUCT *)psSection->hTask;
        if(psObj != NULL)
        {
            psSection->pacTaskName = psObj->pacName;
        }
        else
        {
            psSection->pacTaskName = NULL;
        }

        // Increment counter
        psSection->tCounter++;
    }

    return;
}

/*******************************************************************************
*
*   vTestExitSection
*
*******************************************************************************/
static void vTestExitSection (OSALG_SECTION_ASSOCIATION_STRUCT volatile *psSection)
{
    // Check our section structure for sanity.
    if(psSection->tCounter != 1)
    {
        // Error! Something went wrong. This should never be anything
        // but one at this point since should already have entered
        // this section.
        // TODO: BREAKPOINT
        printf("Error! Exit Section where tCounter = %u.\n",
            psSection->tCounter);
        // Segmentation fault to check backtrace.
        *((char*)0) = 1;
    }
    else
    {
        // Clear task info
        psSection->hTask = OSAL_INVALID_OBJECT_HDL;
        psSection->pacTaskName = "None";

        // Decrement counter
        psSection->tCounter--;
    }

    return;
}

#endif
#endif
