/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the FIXED Object implementation for OSAL
 *
 ******************************************************************************/
#include <string.h>
#define _USE_MATH_DEFINES
#include <math.h>

#include "standard.h"
#include "osal.h"
#include "os_intf.h"
#include "osal_fixed.h"
#include "osal_core.h"

#if OSAL_FIXED_MATH == 1
#include "_osal_fixed.h"

// TODO: Fix these two functions for bug #4809
static float fFixedToFloat (
    N64 n64WholeValue,
    UN8 un8FractionalBits
        );

static float fFixedObjToFloat (
    FIXED_OBJECT_STRUCT *psObj
        );
#endif

/*****************************************************************************
*
*   FIXED_hCreate
*
* This function creates a OSAL_FIXED_OBJECT using a value for the whole part of
* the number and a value for the fractional part of the number
*
* Inputs:
*
*   n32WholeValue - The whole part of the number
*   n32FractionalValue - The fractional part of the number
*
* Outputs:
*
*   A valid OSAL_FIXED_OBJECT handle on success
*   OSAL_FIXED_INVALID_OBJECT on error.
*
*****************************************************************************/
OSAL_FIXED_OBJECT FIXED_hCreate(
    N32 n32WholeValue,
    N32 n32FractionalValue,
    UN8 un8Pow10
        )
{
#if OSAL_FIXED_MATH == 1
    UN8 un8NumWholeBits = 0,
        un8NumFracBitsAvailable = 0,
        un8NumFracBitsNeeded = 0;
    N32 n32FixedValue = 0;
    N64 n64FractionalValue = 0;
    UN8 un8Index = 0;
    N32 n32Computed = 1;

    // Determine the number of bits for the fractional value
    un8NumWholeBits = un8NumWholeBitsNeeded(n32WholeValue);
    un8NumFracBitsAvailable = TOTAL_NUM_BITS - un8NumWholeBits;

    // Convert our base-10 fractional value to a base-2 fixed value
    // Divide our fractional part by 10^un8Ppower to get our fractional
    // value as a real live fraction
    if (un8Pow10 > 0)
    {
        for (un8Index = 0;
             un8Index < un8Pow10;
             un8Index++)
        {
            n32Computed = n32Computed * 10;
        }

        n64FractionalValue =
            n64FixedDivide(n32FractionalValue, 0,
                           n32Computed, 0,
                           &un8NumFracBitsNeeded);

        if (un8NumFracBitsNeeded > un8NumFracBitsAvailable)
        {
            // shift the fractional value to match the bits we have available
            // we could also error out here, but I'm not sure if that is useful
            // to anyone.
            n64FractionalValue >>=
                (un8NumFracBitsNeeded - un8NumFracBitsAvailable);

        }
    }

    // Add our two fixed values together
    n32WholeValue <<= un8NumFracBitsAvailable;
    if (((n32WholeValue < 0) && (n32FractionalValue > 0)) ||
        ((n32WholeValue > 0) && (n32FractionalValue < 0)))
    {
        // signs are different so we need to subtract
        n32FixedValue = n32WholeValue - (N32)n64FractionalValue;
        if ((n32WholeValue > 0) && (n32FractionalValue < 0))
        {
            // need to invert
            n32FixedValue = -1 * n32FixedValue;
        }
    }
    else
    {
        // signs are the same so we need to add.
        n32FixedValue = n32WholeValue + (N32)n64FractionalValue;
    }

    // Create our object
    return FIXED_hCreateFromFixed(n32FixedValue, un8NumFracBitsAvailable);
#else
    return OSAL_FIXED_INVALID_OBJECT;
#endif
}

/*****************************************************************************
*
*   FIXED_hCreateFromFixed
*
* This function creates a OSAL_FIXED_OBJECT using a given integer
* (in a fixed format) and a given number of fractional bits that
* describe the fixed format.
*
* Inputs:
*
*   n32Value - The value in a fixed format
*   un8NumFractionalBits - The fixed format to use
*
* Outputs:
*
*   A valid OSAL_FIXED_OBJECT handle on success
*   OSAL_FIXED_INVALID_OBJECT on error.
*
*****************************************************************************/
OSAL_FIXED_OBJECT FIXED_hCreateFromFixed (
    N32 n32Value,
    UN8 un8NumFractionalBits
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj;


    // Allocate memory for the structure
    psObj = (FIXED_OBJECT_STRUCT*)
                OSALC_pvMemoryAllocate(FIXED_OBJECT_NAME,
                                    sizeof(FIXED_OBJECT_STRUCT),
                                    FALSE);
    if (psObj == NULL)
    {
        return OSAL_FIXED_INVALID_OBJECT;
    }

    // Set the member variables
    psObj->n32Value = n32Value;
    psObj->un8NumFractionalBits = un8NumFractionalBits;

    if (psObj->un8NumFractionalBits > TOTAL_NUM_BITS)
    {
        psObj->n32Value = n32ScaleFixed(psObj, TOTAL_NUM_BITS);
        psObj->un8NumFractionalBits = TOTAL_NUM_BITS;
    }

    return (OSAL_FIXED_OBJECT)psObj;
#else
    return OSAL_FIXED_INVALID_OBJECT;
#endif
}


/*****************************************************************************
*
*   FIXED_hDuplicate
*
* This object interface method is used to access, then do a "deep-copy" of a
* OSAL_FIXED.
*
* Inputs:
*
*   hFixed - The OSAL_FIXED_OBJECT object to duplicate
*
* Outputs:
*
*   A FIXED that is the duplicate of hFixed on success
*   OSAL_FIXED_INVALID_OBJECT on error.
*
*****************************************************************************/
OSAL_FIXED_OBJECT FIXED_hDuplicate (
    OSAL_FIXED_OBJECT hFixed
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;

    if (hFixed == OSAL_FIXED_INVALID_OBJECT)
    {
        return OSAL_FIXED_INVALID_OBJECT;

    }

    return FIXED_hCreateFromFixed(psObj->n32Value,
                                  psObj->un8NumFractionalBits);
#else
    return OSAL_FIXED_INVALID_OBJECT;
#endif
}

/*****************************************************************************
*
*   FIXED_eEval
*
*  Description:
*
*  The function performs complex mathematical calculations using both OSAL_FIXED
*  and N32 values as well as atomic operations and several function. The number of
*  operand allowed in the expression restricted by the buffer size provided by
*  caller, however, each binary operation decreases number of utilized blocks
*  and expression can be rather big.
*
*  The expression should be written in RPN which will be calculated utilizing stack.
*  As additional feature there is a set of 10 registers for storing some intermediate
*  values for further usage in order to prevent doing the same actions under
*  the same values several times.
*
*  There are a few classes of commands:
*   1) Operand command - puts new value of the stack top except the '<' command
*                        which removes the items on the top and moves it to the
*                        registers.
*   2) binary operations - the operation pops two most top elements of the stack
*                          to perform the operations. And put result back on the
*                          stack top. The operation is performed with the elements
*                          by the rule "<top-1><operation><top>".
*   3) unary operation - the operation pops top most element to perform the operations
*                        and puts result back on the stack top.
*
*  The following symbols can be used in the expression:
*
*   Operands operations:
*       %   - place the OSAL_FIXED object on the stack
*       d   - place the N32 value on the stack
*       <   - move the stack top operand to the register.
*             The user must know the sequence number of that operation to use
*             the value later. Please note, once the value is put it will be
*             there till the end of calculations.
*       0-9 - restore variable from the register w/o removing it from there.
*
*   Operation:
*       Unary operations pop the top stack item to perform calculation and place the result
*       back on top of the stack.
*       Binary operations pop two top stack items to perform the calculation in the following manner:
*           <top - 1><operation><top>
*
*   Here the set supported operations:
*     +------------------------------------------------------------+
*     |  Alias | Type  | Math meaning | OSAL Function              |
*     +--------+-------+--------------+----------------------------+
*     |   s    | una   |     sin      | OSAL_FIXED.eSin            |
*     |   c    | una   |     cos      | OSAL_FIXED.eCos            |
*     |   a    | una   |     abs      | OSAL_FIXED.eAbs            |
*     |   t    | una   |     atan     | OSAL_FIXED.eATan           |
*     |   e    | una   |     asin     | OSAL_FIXED.eASin           |
*     |   f    | una   |     acos     | OSAL_FIXED.eACos           |
*     |   q    | una   |     sqrt     | OSAL_FIXED.eSqrt           |
*     |   r    | una   | degree-2-rad | OSAL_FIXED.eDegreeToRadian |
*     |   R    | una   | rad-2-degree | OSAL_FIXED.eRadianToDegree |
*     |   +    | bin   |     add      | OSAL_FIXED.eAdd            |
*     |   -    | bin   |     sub      | OSAL_FIXED.eSubtract       |
*     |   *    | bin   |     mul      | OSAL_FIXED.eMultiply       |
*     |   /    | bin   |     div      | OSAL_FIXED.eDivide         |
*     |   u    | bin   |    atan2     | OSAL_FIXED.eATan2          |
*     |   ^    | bin   |     pow      | OSAL_FIXED.ePow
*     +--------+-------+--------------+----------------------------+
*
* Memory Map:
*
*   The memory provided by the caller is shared between operands "stack" and "registers".
*
*   pvData + tDataSize -> +-----+
*                         | N-1 | <- Reg start point (grows from the N-1 down to 0)
*                         +-----+
*                         | N-2 |
*                         +-----+
*                           ...
*                         +-----+
*                         |  3  |
*                         +-----+
*                         |  2  |
*                         +-----+
*                         |  1  |
*                         +-----+
*                         |  0  | <- Stack start point
*               pvData -> +-----+
*
*  Stack memory grows from the item 0 up to N-1, or up to the registers memory
*  Registers memory grows from the N-1 down to 0, or down to the stack memory
*
* Input:
*    hResult - OSAL_FIXED object for evaluation result in case of success
*    pvData - reference to the memory which is used for internal "stack" and "registers"
*             needs
*    tDataSize - size of the memory addressed by pvData in bytes. Make sure that this size
*                should be multiple of OSAL_FIXED_OBJECT_SIZE * sizeof(OSAL_FIXED_OBJECT_DATA)
*    pacExpr - expression written in RPN.
*    ... - set of arguments corresponded to the expression.
*
* Output:
*   OSAL_SUCCESS if evaluation succeeded
*   OSAL_ERROR_INVALID_INPUT is passed arguments considered as invalid at some point.
*   OSAL_ERROR_OUT_OF_MEMORY - if passed data array is not enough to evaluate expression.
*   OSAL_ERROR - in some general error case
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eEval(
    OSAL_FIXED_OBJECT hResult,
    void *pvData,
    size_t tDataSize,
    const char *pacExpr,
    ...
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    va_list vOperands;
    const char *pacCmd = pacExpr;
    const size_t tBlockSize  = OSAL_FIXED_OBJECT_SIZE * sizeof(OSAL_FIXED_OBJECT_DATA);
    FIXED_OBJECT_STRUCT *psStackBottom = NULL;
    FIXED_OBJECT_STRUCT *psStackCur = NULL;
    FIXED_OBJECT_STRUCT *psRegTop = NULL;
    FIXED_OBJECT_STRUCT *psRegCur = NULL;

    // Start processing arguments
    va_start(vOperands, pacExpr);

    do
    {
        if ((pacExpr == NULL) ||(*pacExpr == '\0') ||
            (hResult == OSAL_FIXED_INVALID_OBJECT) ||
            (tDataSize == 0) || ((tDataSize / tBlockSize) == 0))
        {
            eReturnCode = OSAL_ERROR_INVALID_INPUT;
            break;
        }

        // Gets the stack start point
        psStackBottom = (FIXED_OBJECT_STRUCT*)pvData;
        // Keeps reference to the item after current
        psStackCur = psStackBottom;
        // Gets the regs start point
        psRegTop = (FIXED_OBJECT_STRUCT*)pvData + (tDataSize / tBlockSize - 1);
        // Keeps reference to the item after current
        psRegCur = psRegTop;

        // Let's pretend we are success
        eReturnCode = OSAL_SUCCESS;

        while (*pacCmd != '\0')
        {
            OSAL_FIXED_MATH_UNARY_FUNC eUnaryFunc = NULL;
            OSAL_FIXED_MATH_BINARY_FUNC eBinaryFunc = NULL;

            switch (*pacCmd)
            {
                ///////////////////////////////////////////////
                // Process operands
                case FIXED_EVAL_OPERAND_FIXED_CMD:
                {
                    if (psStackCur > psRegCur)
                    {
                        // So, the stack is full to add one more operand
                        eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
                    }
                    else
                    {
                        OSAL_FIXED_OBJECT hValue;
                        hValue = (OSAL_FIXED_OBJECT)va_arg(vOperands, void*);
                        if (hValue != OSAL_FIXED_INVALID_OBJECT)
                        {
                            // Put value on the stack and move the cursor to the next
                            // item
                            *psStackCur++ = *((FIXED_OBJECT_STRUCT*)hValue);
                        }
                        else
                        {
                            eReturnCode = OSAL_ERROR_INVALID_INPUT;
                        }
                    }
                }
                break;
                case FIXED_EVAL_OPERAND_N32_CMD:
                {
                    if (psStackCur > psRegCur)
                    {
                        // So, the stack is full to add one more operand
                        eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
                    }
                    else
                    {
                        OSAL_FIXED_OBJECT hValue;
                        // Create OSAL_FIXED object for the N32 value.
                        N32 n32Value = va_arg(vOperands, N32);
                        hValue = FIXED_hCreateInMemory(n32Value, 0, (void*)psStackCur);
                        if (hValue != OSAL_FIXED_INVALID_OBJECT)
                        {
                            // Just move the stack top since we already utilize
                            // current top
                            ++psStackCur;
                        }
                        else
                        {
                            eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
                        }
                    }
                }
                break;
                case FIXED_EVAL_OPERAND_TOREG_CMD:
                {
                    if (
                        // At least one item in the stack
                        (psStackBottom < psStackCur) &&
                        // There are less utilized registers than it is
                        // theoretically possible
                        ((psRegTop - psRegCur) < FIXED_EVAL_REGISTERS_COUNT) &&
                        // The regs memory allow to have one more new value
                        ((psRegCur >= psStackCur) ||
                        // [OR] the next reg item lays right in the same place
                        // where the top stack operand does (the pointers
                        // might be just moved w/o data copying)
                         (psRegCur == (psStackCur - 1)))
                       )
                    {
                        // Move the operand from the stack into
                        // the next reg cell.
                        *psRegCur-- = *(--psStackCur);
                    }
                    else
                    {
                        eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
                    }
                }
                break;

                ///////////////////////////////////////////////
                // Process binary operations
                case FIXED_EVAL_ADD_CMD:
                {
                    eBinaryFunc = FIXED_eAdd;
                }
                break;
                case FIXED_EVAL_SUB_CMD:
                {
                    eBinaryFunc = FIXED_eSubtract;
                }
                break;
                case FIXED_EVAL_MUL_CMD:
                {
                    eBinaryFunc = FIXED_eMultiply;
                }
                break;
                case FIXED_EVAL_DIV_CMD:
                {
                    eBinaryFunc = FIXED_eDivide;
                }
                break;
                case FIXED_EVAL_ATAN2_CMD:
                {
                    eBinaryFunc = FIXED_eATan2;
                }
                break;
                case FIXED_EVAL_POW_CMD:
                {
                    eBinaryFunc = ePowForEval;
                }
                break;

                ///////////////////////////////////////////////
                // Process unary operations
                case FIXED_EVAL_SIN_CMD:
                {
                    eUnaryFunc = FIXED_eSin;
                }
                break;
                case FIXED_EVAL_COS_CMD:
                {
                    eUnaryFunc = FIXED_eCos;
                }
                break;
                case FIXED_EVAL_ABS_CMD:
                {
                    eUnaryFunc = FIXED_eAbs;
                }
                break;
                case FIXED_EVAL_ATAN_CMD:
                {
                    eUnaryFunc = FIXED_eATan;
                }
                break;
                case FIXED_EVAL_ASIN_CMD:
                {
                    eUnaryFunc = FIXED_eASin;
                }
                break;
                case FIXED_EVAL_ACOS_CMD:
                {
                    eUnaryFunc = FIXED_eACos;
                }
                break;
                case FIXED_EVAL_SQRT_CMD:
                {
                    eUnaryFunc = FIXED_eSqrt;
                }
                break;
                case FIXED_EVAL_DEG2RAD_CMD:
                {
                    eUnaryFunc = FIXED_eDegreeToRadian;
                }
                break;
                case FIXED_EVAL_RAD2DEG_CMD:
                {
                    eUnaryFunc = FIXED_eRadianToDegree;
                }
                break;
                default:
                {
                    // if the passed symbol is a digit interpret it as
                    // requests to restore value from one of the registers
                    if ((*pacCmd >= FIXED_EVAL_REG_FIRST) &&
                        (*pacCmd <= FIXED_EVAL_REG_LAST))
                    {
                        const size_t tRegIndex = (*pacCmd - FIXED_EVAL_REG_FIRST);

                        // The index addresses the existing register cell
                        if ((psRegTop - tRegIndex) <= psRegCur)
                        {
                            eReturnCode = OSAL_ERROR_INVALID_INPUT;
                        }
                        // The stack has no space for new item
                        else if (psStackCur > psRegCur)
                        {
                            eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
                        }
                        else
                        {
                            // Put the reg value on the stack top
                            *psStackCur++ = *(psRegTop - tRegIndex);
                        }
                    }
                    else
                    {
                        // Some unknown symbol found.
                        eReturnCode = OSAL_ERROR_INVALID_INPUT;
                    }
                }
                break;
            }

            // Is the unary operation requested?
            if (eUnaryFunc != NULL)
            {
                FIXED_OBJECT_STRUCT sTempValue;
                // At least one operand must be in the stack
                if (psStackBottom + 1 <= psStackCur)
                {
                    FIXED_OBJECT_STRUCT *psOperand = psStackCur - 1;
                    eReturnCode = eUnaryFunc((OSAL_FIXED_OBJECT)psOperand,
                                             (OSAL_FIXED_OBJECT)&sTempValue);
                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        // Update stack
                        *psOperand = sTempValue;
                    }
                }
                else
                {
                    eReturnCode = OSAL_ERROR;
                }
            }
            // Is the binary operation requested?
            else if (eBinaryFunc != NULL)
            {
                FIXED_OBJECT_STRUCT sTempValue;

                // At least two operands must be in the stack
                if (psStackBottom + 2 <= psStackCur)
                {
                    FIXED_OBJECT_STRUCT *psFirst = psStackCur - 2;
                    FIXED_OBJECT_STRUCT *psSecond = psStackCur - 1;
                    eReturnCode = eBinaryFunc(
                                        (OSAL_FIXED_OBJECT)psFirst,
                                        (OSAL_FIXED_OBJECT)psSecond,
                                        (OSAL_FIXED_OBJECT)&sTempValue);
                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        // Update stack
                        psStackCur = psSecond;
                        *psFirst = sTempValue;
                    }
                }
                else
                {
                    eReturnCode = OSAL_ERROR;
                }
            }

            // Check the result of recent operation including the set of
            // operand related commands.
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            // Get next symbol
            ++pacCmd;
        }
    } while (FALSE);

    // Finalize va_arg
    va_end(vOperands);

    if ((eReturnCode == OSAL_SUCCESS) && (hResult != OSAL_FIXED_INVALID_OBJECT))
    {
        // At the end in properly composed expression exactly one item in the
        // stack must remain - otherwise the computation shall be considered as invalid.
        if (psStackCur != psStackBottom + 1)
        {
            eReturnCode = OSAL_ERROR_INVALID_INPUT;
        }
        else
        {
            BOOLEAN bOk;
            // Get value from the stack's bottom
            bOk = FIXED_bCopyToMemory((OSAL_FIXED_OBJECT)psStackBottom, (void*)hResult);
            if (bOk == FALSE)
            {
                eReturnCode = OSAL_ERROR;
            }
        }
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_vDestroy
*
* This function destroys a OSAL_FIXED_OBJECT -- releasing the memory to
* the system
*
* Inputs:
*
*   hFixed - The OSAL_FIXED_OBJECT to send back to its maker
*
* Outputs:
*
*   nichts
*
*****************************************************************************/
void FIXED_vDestroy (
    OSAL_FIXED_OBJECT hFixed
        )
{
#if OSAL_FIXED_MATH == 1
    if (hFixed != OSAL_FIXED_INVALID_OBJECT)
    {
        OSALC_vMemoryFree(hFixed);
    }
#endif
    return;
}

/*****************************************************************************
*
*   FIXED_n32Value
*
* This function returns the value of a given OSAL_FIXED_OBJECT as a fixed-point
* value
*
* Inputs:
*
*   hFixed - The OSAL_FIXED_OBJECT to get the number of fixed-point value for
*
* Outputs:
*
*   The value in the fixed format.  A value of zero would be returned if
*   there is an error.
*
*****************************************************************************/
N32 FIXED_n32Value (
    OSAL_FIXED_OBJECT hFixed
        )
{
#if OSAL_FIXED_MATH == 1
    N32 n32ReturnValue = 0;

    if (hFixed != OSAL_FIXED_INVALID_OBJECT)
    {
        FIXED_OBJECT_STRUCT *psObj =
            (FIXED_OBJECT_STRUCT *)hFixed;

        n32ReturnValue = psObj->n32Value;
    }
    return n32ReturnValue;
#else
    return 0;
#endif
}

/*****************************************************************************
*
*   FIXED_un8NumFractionalBits
*
* This function returns the number of fractional bits used to represent this
* OSAL_FIXED_OBJECT.
*
* Inputs:
*
*   hFixed - The OSAL_FIXED_OBJECT to get the number of fractional bits used
*
* Outputs:
*
*   The number of bits used.  A value of zero would be returned if there is an
*   error.
*
*****************************************************************************/
UN8 FIXED_un8NumFractionalBits (
    OSAL_FIXED_OBJECT hFixed
        )
{
#if OSAL_FIXED_MATH == 1
    UN8 un8ReturnValue = 0;

    if (hFixed != OSAL_FIXED_INVALID_OBJECT)
    {
        FIXED_OBJECT_STRUCT *psObj =
            (FIXED_OBJECT_STRUCT *)hFixed;

        un8ReturnValue = psObj->un8NumFractionalBits;
    }
    return un8ReturnValue;
#else
    return 0;
#endif
}

/*****************************************************************************
*
*   FIXED_n32ScaledValue
*
* This function returns the value of a given OSAL_FIXED_OBJECT in the
* specified fixed point value.
*
* Inputs:
*
*   hFixed - The OSAL_FIXED_OBJECT to get the new fixed-point value for
*   un8NumFractionalBits - The number of fractional bits to use for
*       the returned fixed value
*
* Outputs:
*
*   The value in the new fixed format.  A value of zero would be returned if
*   there is an error.
*
*****************************************************************************/
N32 FIXED_n32ScaledValue (
    OSAL_FIXED_OBJECT hFixed,
    UN8 un8NumFractionalBits
        )
{
#if OSAL_FIXED_MATH == 1
    N32 n32ReturnValue = 0;

    if ((hFixed != OSAL_FIXED_INVALID_OBJECT) &&
        (un8NumFractionalBits <= TOTAL_NUM_BITS))
    {
        FIXED_OBJECT_STRUCT *psObj =
            (FIXED_OBJECT_STRUCT *)hFixed;

        n32ReturnValue = n32ScaleFixed(psObj, un8NumFractionalBits);
    }
    return n32ReturnValue;
#else
    return 0;
#endif
}

/*****************************************************************************
*
*   FIXED_n32Ceiling
*
* This function returns the next highest whole value of a given
* OSAL_FIXED_OBJECT as an N32.
*
* Inputs:
*
*   hFixed - The OSAL_FIXED_OBJECT to get a ceiling value for
*
* Outputs:
*
*   The next highest whole value for this OSAL_FIXED_OBJECT.
*   A value of N32_MIN would be returned if there is an error.
*
*****************************************************************************/
N32 FIXED_n32Ceiling (
    OSAL_FIXED_OBJECT hFixed
        )
{
#if OSAL_FIXED_MATH == 1
    N32 n32ReturnValue = N32_MIN;

    if (hFixed != OSAL_FIXED_INVALID_OBJECT)
    {
        FIXED_OBJECT_STRUCT *psObj =
            (FIXED_OBJECT_STRUCT *)hFixed;

        // Scale the value so that there is no fractional component
        n32ReturnValue = n32ScaleFixed(psObj, 0);

        // See if there was a fractional component
        if ((psObj->n32Value & (BIN_POINT_VALUE(psObj->un8NumFractionalBits)-1)) != 0)
        {
            // Round to next whole number
            n32ReturnValue++;
        }
    }
    return n32ReturnValue;
#else
    return N32_MIN;
#endif
}

/*****************************************************************************
*
*   FIXED_n32Floor
*
* This function returns the next lowest whole value of a given
* OSAL_FIXED_OBJECT as an N32.
*
* Inputs:
*
*   hFixed - The OSAL_FIXED_OBJECT to get a floor value for
*
* Outputs:
*
*   The next lowest whole value for this OSAL_FIXED_OBJECT.
*   A value of N32_MAX would be returned if there is an error.
*   N32_MAX would never be returned by a floor function, so it
*   is a better error value than N32_MIN.
*
*   TODO: fix OSAL API document, which says 0 is returned if there's
*         an error, which was already different from the function
*
*****************************************************************************/
N32 FIXED_n32Floor (
    OSAL_FIXED_OBJECT hFixed
        )
{
#if OSAL_FIXED_MATH == 1
    N32 n32ReturnValue = N32_MAX;

    if (hFixed != OSAL_FIXED_INVALID_OBJECT)
    {
        FIXED_OBJECT_STRUCT *psObj =
            (FIXED_OBJECT_STRUCT *)hFixed;

        // Scale the value so that there is no fractional component
        n32ReturnValue = n32ScaleFixed(psObj, 0);
    }
    return n32ReturnValue;
#else
    return N32_MAX;
#endif
}

/*****************************************************************************
*
*   FIXED_eAdd
*
* This function adds two FIXED_OBJECTs together and returns a new
* OSAL_FIXED_OBJECT representing the sum (as the third parameter).
*
* Inputs:
*
*   hFixed1 - The OSAL_FIXED_OBJECT to use as an operand
*   hFixed2 - The OSAL_FIXED_OBJECT to use as an operand
*   hFixedResult - The OSAL_FIXED_OBJECT where the result will be

*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eAdd (
    OSAL_FIXED_OBJECT hFixed1,
    OSAL_FIXED_OBJECT hFixed2,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FIXED_OBJECT_STRUCT *psObj1 =
        (FIXED_OBJECT_STRUCT *)hFixed1;
    FIXED_OBJECT_STRUCT *psObj2 =
        (FIXED_OBJECT_STRUCT *)hFixed2;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    N64 n64Result = 0;
    UN8 un8ResultBits;

#if DEBUG_MATH == 1
    float fAdd = fFixedObjToFloat(psObj1) + fFixedObjToFloat(psObj2);
#endif

    if ( (hFixed1 == OSAL_FIXED_INVALID_OBJECT) ||
         (hFixed2 == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    printf(FIXED_OBJECT_NAME": %f plus %f is ...",
        fFixedToFloat(psObj1->n32Value, psObj1->un8NumFractionalBits),
        fFixedToFloat(psObj2->n32Value, psObj2->un8NumFractionalBits));

    // Figure out which format is the largest.  Use that one.
    if ( psObj1->un8NumFractionalBits <
         psObj2->un8NumFractionalBits)
     {
        un8ResultBits = psObj2->un8NumFractionalBits;
     }
     else
     {
        un8ResultBits = psObj1->un8NumFractionalBits;
     }

    // Throw the inital value on the N64
    n64Result = n64ScaleFixed(psObj1, un8ResultBits);

    // Do the addition
    n64Result += n64ScaleFixed(psObj2, un8ResultBits);

    // Adjust the result
    eReturnCode = eAdjustResult(psObjResult, n64Result, un8ResultBits);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(" %f\n",
            fFixedToFloat(psObjResult->n32Value, psObjResult->un8NumFractionalBits));

#if DEBUG_MATH == 1
        vValidateResult(fAdd, psObjResult, "addition");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error adding %f and %f "
            "Error %s.\n",
            fFixedObjToFloat(psObj1), fFixedObjToFloat(psObj2),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eSubtract
*
* This function subtracts two OSAL_FIXED_OBJECTs together and returns a new
* OSAL_FIXED_OBJECT representing the output via the third parameter.
*
* Inputs:
*
*   hFixed1 - A OSAL_FIXED_OBJECT to use as an operand
*   hFixed2 - A OSAL_FIXED_OBJECT to use as an operand
*   hResult - The OSAL_FIXED_OBJECT where the result will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eSubtract (
    OSAL_FIXED_OBJECT hFixed1,
    OSAL_FIXED_OBJECT hFixed2,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;

    FIXED_OBJECT_STRUCT *psObj1 =
        (FIXED_OBJECT_STRUCT *)hFixed1;
    FIXED_OBJECT_STRUCT *psObj2 =
        (FIXED_OBJECT_STRUCT *)hFixed2;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;

    N64 n64Result = 0;
    UN8 un8ResultBits;

#if DEBUG_MATH == 1
    float fSubtract = fFixedObjToFloat(psObj1) - fFixedObjToFloat(psObj2);
#endif

    if ( (hFixed1 == OSAL_FIXED_INVALID_OBJECT) ||
         (hFixed2 == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Figure out which format is the largest.  Use that one.
    if ( psObj1->un8NumFractionalBits < psObj2->un8NumFractionalBits)
    {
        un8ResultBits = psObj2->un8NumFractionalBits;
    }
    else
    {
        un8ResultBits = psObj1->un8NumFractionalBits;
    }

    // Throw the inital value on the N64
    n64Result = n64ScaleFixed(psObj1, un8ResultBits);

    // Do the subtraction
    n64Result -= n64ScaleFixed(psObj2, un8ResultBits);

    // Adjust the result
    eReturnCode = eAdjustResult(psObjResult, n64Result, un8ResultBits);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME": %f minus %f is %f\n",
            fFixedToFloat(psObj1->n32Value, psObj1->un8NumFractionalBits),
            fFixedToFloat(psObj2->n32Value, psObj2->un8NumFractionalBits),
            fFixedToFloat(psObjResult->n32Value,
                psObjResult->un8NumFractionalBits));

#if DEBUG_MATH == 1
        vValidateResult(fSubtract, psObjResult, "subtraction");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error subtracting %f from %f "
            "Error %s.\n",
            fFixedObjToFloat(psObj2), fFixedObjToFloat(psObj1),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eMultiply
*
* This function multiplies two OSAL_FIXED_OBJECTs together and returns a new
* OSAL_FIXED_OBJECT representing the output as the third parameter.
*
* Inputs:
*
*   hFixed1 - A OSAL_FIXED_OBJECT to use as an operand
*   hFixed2 - A OSAL_FIXED_OBJECT to use as an operand
*   hResult - The OSAL_FIXED_OBJECT where the result will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eMultiply (
    OSAL_FIXED_OBJECT hFixed1,
    OSAL_FIXED_OBJECT hFixed2,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1

    OSAL_RETURN_CODE_ENUM eReturnCode;
    FIXED_OBJECT_STRUCT *psObj1 =
        (FIXED_OBJECT_STRUCT *)hFixed1;
    FIXED_OBJECT_STRUCT *psObj2 =
        (FIXED_OBJECT_STRUCT *)hFixed2;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    N64 n64Result = 0;
    UN8 un8ResultBits = 0;

#if DEBUG_MATH == 1
    float fMultiply = fFixedObjToFloat(psObj1) * fFixedObjToFloat(psObj2);
#endif

    if ( (hFixed1 == OSAL_FIXED_INVALID_OBJECT) ||
         (hFixed2 == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    printf(FIXED_OBJECT_NAME": %f times %f is ... \n",
        fFixedToFloat(psObj1->n32Value, psObj1->un8NumFractionalBits),
        fFixedToFloat(psObj2->n32Value, psObj2->un8NumFractionalBits));

    // Call our private multiplication function
    n64Result = n64FixedMultiply (psObj1->n32Value,
                                  psObj1->un8NumFractionalBits,
                                  psObj2->n32Value,
                                  psObj2->un8NumFractionalBits,
                                  &un8ResultBits);

    // Adjust the result
    eReturnCode = eAdjustResult(psObjResult, n64Result, un8ResultBits);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(" %f\n",
            fFixedToFloat(psObjResult->n32Value, psObjResult->un8NumFractionalBits));

#if DEBUG_MATH == 1
        vValidateResult(fMultiply, psObjResult, "multiplication");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error multiplying %f and %f "
            "Error %s.\n",
            fFixedObjToFloat(psObj1), fFixedObjToFloat(psObj2),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else /* OSAL_FIXED_MATH == 1 */
    return OSAL_ERROR_UNSUPPORTED_API;
#endif

}

/*****************************************************************************
*
*   FIXED_eDivide
*
* This function divides two OSAL_FIXED_OBJECTs together and returns a new
* OSAL_FIXED_OBJECT representing the output.
*
* Inputs:
*
*   hFixedNumerator - A OSAL_FIXED_OBJECT to use as the numerator
*   hFixedDenominator - A OSAL_FIXED_OBJECT to use as an denominator
*   hResult - The OSAL_FIXED_OBJECT where the result will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eDivide (
    OSAL_FIXED_OBJECT hFixedNumerator,
    OSAL_FIXED_OBJECT hFixedDenominator,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;

    FIXED_OBJECT_STRUCT *psObj1 =
        (FIXED_OBJECT_STRUCT *)hFixedNumerator;
    FIXED_OBJECT_STRUCT *psObj2 =
        (FIXED_OBJECT_STRUCT *)hFixedDenominator;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    N64 n64Result = 0;
    UN8 un8ResultBits = 0;

#if DEBUG_MATH == 1
    float fDivide = fFixedObjToFloat(psObj1) / fFixedObjToFloat(psObj2);
#endif


    if ( (hFixedNumerator == OSAL_FIXED_INVALID_OBJECT) ||
         (hFixedDenominator == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    if (psObj2->n32Value == 0)
    {
        return OSAL_ERROR_MATH_DIVIDE_BY_ZERO;
    }

    // Call our private multiplication function
    n64Result = n64FixedDivide (psObj1->n32Value,
                                psObj1->un8NumFractionalBits,
                                psObj2->n32Value,
                                psObj2->un8NumFractionalBits,
                                &un8ResultBits);

    // Adjust the result
    eReturnCode = eAdjustResult(psObjResult, n64Result, un8ResultBits);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME": %f divided by %f is %f\n",
            fFixedToFloat(psObj1->n32Value, psObj1->un8NumFractionalBits),
            fFixedToFloat(psObj2->n32Value, psObj2->un8NumFractionalBits),
            fFixedToFloat(psObjResult->n32Value, psObjResult->un8NumFractionalBits));

#if DEBUG_MATH == 1
        vValidateResult(fDivide, psObjResult, "divison");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error dividing %f by %f "
            "Error %s.\n",
            fFixedObjToFloat(psObj1), fFixedObjToFloat(psObj2),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eDegreeToRadian
*
* This function converts a OSAL_FIXED_OBJECT from degrees to radians,
* returning a OSAL_FIXED_OBJECT representing the output.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT that represents degrees
*   hResult - The OSAL_FIXED_OBJECT where the result will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eDegreeToRadian (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;

    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    N64 n64Result;
    UN8 un8ResultBits;

#if DEBUG_MATH == 1
    float fDegreeToRadian = (float)((fFixedObjToFloat(psObj) * M_PI) / 180.0);
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Multiply by Pi over 180
    // Call our private multiplication function    
    n64Result = n64FixedMultiply (psObj->n32Value,
                                  psObj->un8NumFractionalBits,
                                  FX3B29_PI_OVER_180,
                                  PI_FRACTIONAL_BITS + 1,
                                  &un8ResultBits);

    // Adjust the result
    eReturnCode = eAdjustResult(psObjResult, n64Result, un8ResultBits);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME":%f degrees as radian is %f\n",
            fFixedToFloat(psObj->n32Value,
                psObj->un8NumFractionalBits),
            fFixedToFloat(psObjResult->n32Value,
                psObjResult->un8NumFractionalBits));

#if DEBUG_MATH == 1
        vValidateResult(fDegreeToRadian, psObjResult, "degree to radian");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error calculating radians of %f "
            "Error %s.\n",
            fFixedObjToFloat(psObj),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}


/*****************************************************************************
*
*   FIXED_eRadianToDegree
*
* This function converts a OSAL_FIXED_OBJECT from radians to degrees,
* returning a OSAL_FIXED_OBJECT representing the output.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT that represents a value in radians
*   hResult - The OSAL_FIXED_OBJECT where the result in degrees will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eRadianToDegree (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;

    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    N64 n64Result;
    UN8 un8ResultBits;

#if DEBUG_MATH == 1
    float fRadianToDegree = (float)((fFixedObjToFloat(psObj) * 180.0)/M_PI);
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Divide by Pi over 180
    // Call our private division function    
    n64Result = n64FixedDivide (psObj->n32Value,
                                psObj->un8NumFractionalBits,
                                FX3B29_PI_OVER_180,
                                PI_FRACTIONAL_BITS + 1,
                                &un8ResultBits);

    // Adjust the result
    eReturnCode = eAdjustResult(psObjResult, n64Result, un8ResultBits);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME":%f radians as degrees is %f\n",
            fFixedToFloat(psObj->n32Value,
                psObj->un8NumFractionalBits),
            fFixedToFloat(psObjResult->n32Value,
                psObjResult->un8NumFractionalBits));

#if DEBUG_MATH == 1
        vValidateResult(fRadianToDegree, psObjResult, "radian to degree");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error calculating degrees of %f "
            "Error %s.\n",
            fFixedObjToFloat(psObj),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eSin
*
* This function calculates the sin of a FIXED object and returns it via the
* hResult parameter.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT in radians
*   hResult - The OSAL_FIXED_OBJECT where the result will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eSin (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT sValue, sAbsValue;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    UN8 un8NumSubtractions = 0;
    N32 n32RadValue;
    N16 n16CompareResult;
    BOOLEAN bInvert = FALSE;

#if DEBUG_MATH == 1
    float fSin = (float)sin(fFixedObjToFloat(psObj));
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    sValue.n32Value = psObj->n32Value;
    sValue.un8NumFractionalBits = psObj->un8NumFractionalBits;

    printf(FIXED_OBJECT_NAME": sin of %f is ...\n", fFixedObjToFloat(&sValue));

    do
    {
        // if | angle | > pi
        eReturnCode = FIXED_eAbs((OSAL_FIXED_OBJECT)&sValue, (OSAL_FIXED_OBJECT)&sAbsValue);
        n16CompareResult = FIXED_n16Compare(
            (OSAL_FIXED_OBJECT)&sAbsValue, (OSAL_FIXED_OBJECT)&gsPi);
        while (n16CompareResult >=0 && eReturnCode == OSAL_SUCCESS)
        {
            if (sValue.n32Value > 0)
            {
                eReturnCode = FIXED_eSubtract(
                    (OSAL_FIXED_OBJECT)&sValue,
                    (OSAL_FIXED_OBJECT)&gsPi,
                    (OSAL_FIXED_OBJECT)&sValue);
            }
            else
            {
                eReturnCode = FIXED_eAdd(
                    (OSAL_FIXED_OBJECT)&sValue,
                    (OSAL_FIXED_OBJECT)&gsPi,
                    (OSAL_FIXED_OBJECT)&sValue);
            }

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = FIXED_eAbs((OSAL_FIXED_OBJECT)&sValue,
                                         (OSAL_FIXED_OBJECT)&sAbsValue);
                n16CompareResult = FIXED_n16Compare(
                    (OSAL_FIXED_OBJECT)&sAbsValue, (OSAL_FIXED_OBJECT)&gsPi);

                un8NumSubtractions++;
            }
        }

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // so now we know that | angle | <= pi

        // we need angle in the range of -pi/2 to pi/2

        // is angle greater than pi/2?
        n16CompareResult = FIXED_n16Compare((OSAL_FIXED_OBJECT)&sValue, (OSAL_FIXED_OBJECT)&gsPiOverTwo);
        if (n16CompareResult > 0)
        {
            // yes, we need to adjust it
            eReturnCode = FIXED_eSubtract(
                    (OSAL_FIXED_OBJECT)&sValue,
                    (OSAL_FIXED_OBJECT)&gsPi,
                    (OSAL_FIXED_OBJECT)&sValue);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
            // we adjusted the angle, but may need to invert the answer
            bInvert = TRUE;
        }

        // is angle less than -pi/2?
        n16CompareResult = FIXED_n16Compare((OSAL_FIXED_OBJECT)&sValue, (OSAL_FIXED_OBJECT)&gsNegPiOverTwo);
        if (n16CompareResult < 0)
        {
            // yes, we need to adjust it
            eReturnCode = FIXED_eAdd(
                    (OSAL_FIXED_OBJECT)&gsPi,
                    (OSAL_FIXED_OBJECT)&sValue,
                    (OSAL_FIXED_OBJECT)&sValue);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
            // we adjusted the angle, but may need to invert the answer
            bInvert = TRUE;
        }

        // angle is in range

        // Shift the value to the "trig" format
        n32RadValue = n32ScaleFixed(&sValue, FRACTIONAL_BITS_FOR_TRIG);

        // Use the CORDIC function to get our sin value
        psObjResult->n32Value =
            n32CordicEval(n32RadValue, FIXED_OBJ_CORDIC_EVAL_TYPE_SIN);
        psObjResult->un8NumFractionalBits = FRACTIONAL_BITS_FOR_TRIG;

        // do we need to invert the answer (based on the number of times we subtracted pi
        // and if the angle was > pi/2 or  angle < -pi/2
        if (((bInvert == TRUE) && ((un8NumSubtractions % 2) == 0)) ||
             ((bInvert == FALSE) && ((un8NumSubtractions % 2) != 0)) )
        {
            // 2nd or third quadrant so need to invert
            eReturnCode = FIXED_eMultiply(
                (OSAL_FIXED_OBJECT)psObjResult,
                (OSAL_FIXED_OBJECT)&gsNegOne,
                (OSAL_FIXED_OBJECT)psObjResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        eReturnCode = OSAL_SUCCESS;

#if DEBUG_MATH == 1
        vValidateResult(fSin, psObjResult, "sin");
#endif
    } while (0);

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}
/*****************************************************************************
*
*   FIXED_eASin
*
* This function calculates the inverse sin of a FIXED object and returns it
* via the hResult parameter.
* The range of hFixed must be between -pi/2 and pi/2
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT in radians
*   hResult - The OSAL_FIXED_OBJECT where the result will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eASin (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;

    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    FIXED_OBJECT_STRUCT sXSquared, sOneMinusXSquared,
                        sSqrtOfOneMinusXSquared,
                        sXOverTheSqrtOfOneMinusXSquared,
                        sOnePlusSqrtOfOneMinusXSquared,
                        sATan;
    N16 n16CompareResult;

#if DEBUG_MATH == 1
    float fASin = (float)asin(fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hFixed));
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Make sure it is not greater than pi/2 and
    n16CompareResult = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&gsPiOverTwo);
    if (n16CompareResult > 0)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Make sure it is not less than -pi/2
    n16CompareResult = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&gsNegPiOverTwo);
    if (n16CompareResult < 0)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    do
    {
        // If x is 1, then the result is pi/2
        n16CompareResult = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&gsOne);
        if (n16CompareResult == 0)
        {
            psObjResult->n32Value = gsPiOverTwo.n32Value;
            psObjResult->un8NumFractionalBits = gsPiOverTwo.un8NumFractionalBits;

            eReturnCode = OSAL_SUCCESS;
            break;
        }

        // If x is -1, then the result is -pi/2
        n16CompareResult = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&gsNegOne);
        if (n16CompareResult == 0)
        {
            psObjResult->n32Value = gsNegPiOverTwo.n32Value;
            psObjResult->un8NumFractionalBits = gsNegPiOverTwo.un8NumFractionalBits;

            eReturnCode = OSAL_SUCCESS;
            break;
        }

        // x^2
        eReturnCode = FIXED_ePow(hFixed, 2, (OSAL_FIXED_OBJECT)&sXSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //      1-x^2
        eReturnCode = FIXED_eSubtract(
            (OSAL_FIXED_OBJECT)&gsOne,
            (OSAL_FIXED_OBJECT)&sXSquared,
            (OSAL_FIXED_OBJECT)&sOneMinusXSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //        _______
        //      \/(1-x^2)
        eReturnCode = FIXED_eSqrt(
            (OSAL_FIXED_OBJECT)&sOneMinusXSquared,
            (OSAL_FIXED_OBJECT)&sSqrtOfOneMinusXSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //           _______
        //      1+ \/(1-x^2)
        eReturnCode = FIXED_eAdd(
            (OSAL_FIXED_OBJECT)&sSqrtOfOneMinusXSquared,
            (OSAL_FIXED_OBJECT)&gsOne,
            (OSAL_FIXED_OBJECT)&sOnePlusSqrtOfOneMinusXSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //            x
        //      ------------
        //           _______
        //      1+ \/(1-x^2)
        eReturnCode = FIXED_eDivide(
            hFixed,
            (OSAL_FIXED_OBJECT)&sOnePlusSqrtOfOneMinusXSquared,
            (OSAL_FIXED_OBJECT)&sXOverTheSqrtOfOneMinusXSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // ATan
        eReturnCode = FIXED_eATan((OSAL_FIXED_OBJECT)&sXOverTheSqrtOfOneMinusXSquared, (OSAL_FIXED_OBJECT)&sATan);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // 2 * ATan (add to itself is same as multply by 2)
        eReturnCode = FIXED_eAdd(
            (OSAL_FIXED_OBJECT)&sATan,
            (OSAL_FIXED_OBJECT)&sATan,
            hResult);

    } while (FALSE);

#if DEBUG_MATH == 1
    vValidateResult(fASin, psObjResult, "asin");
#endif
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eCOS
*
* This function calculates the cos of a FIXED object and returns it via the
* hResult parameter.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT in radians
*   hResult - The OSAL_FIXED_OBJECT where the result will be stored
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eCos (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT sValue, sAbsValue;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    UN8 un8NumSubtractions = 0;
    N32 n32RadValue;
    N16 n16CompareResult;
    BOOLEAN bInvert = FALSE;

#if DEBUG_MATH == 1
    float fCos = (float)cos(fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hFixed));
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    do
    {
            sValue.n32Value = psObj->n32Value;
            sValue.un8NumFractionalBits = psObj->un8NumFractionalBits;

            // if | angle | > pi
            eReturnCode = FIXED_eAbs((OSAL_FIXED_OBJECT)&sValue, (OSAL_FIXED_OBJECT)&sAbsValue);
            n16CompareResult = FIXED_n16Compare(
                (OSAL_FIXED_OBJECT)&sAbsValue, (OSAL_FIXED_OBJECT)&gsPi);
            while (n16CompareResult >=0 && eReturnCode == OSAL_SUCCESS)
            {
                if (sValue.n32Value > 0)
                {
                    eReturnCode = FIXED_eSubtract(
                        (OSAL_FIXED_OBJECT)&sValue,
                        (OSAL_FIXED_OBJECT)&gsPi,
                        (OSAL_FIXED_OBJECT)&sValue);
                }
                else
                {
                    eReturnCode = FIXED_eAdd(
                        (OSAL_FIXED_OBJECT)&sValue,
                        (OSAL_FIXED_OBJECT)&gsPi,
                        (OSAL_FIXED_OBJECT)&sValue);
                }

                if (eReturnCode == OSAL_SUCCESS)
                {
                    eReturnCode = FIXED_eAbs((OSAL_FIXED_OBJECT)&sValue,
                                             (OSAL_FIXED_OBJECT)&sAbsValue);
                    n16CompareResult = FIXED_n16Compare(
                        (OSAL_FIXED_OBJECT)&sAbsValue, (OSAL_FIXED_OBJECT)&gsPi);

                    un8NumSubtractions++;
                }
            }

            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            // so now we know that | angle | <= pi

            // we need in the range of -pi/2 to pi/2

            // is angle greater than pi/2?
            n16CompareResult = FIXED_n16Compare((OSAL_FIXED_OBJECT)&sValue, (OSAL_FIXED_OBJECT)&gsPiOverTwo);
            if (n16CompareResult > 0)
            {
                // yes, we need to adjust it
                eReturnCode = FIXED_eSubtract(
                        (OSAL_FIXED_OBJECT)&sValue,
                        (OSAL_FIXED_OBJECT)&gsPi,
                        (OSAL_FIXED_OBJECT)&sValue);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    break;
                }

                // we adjusted the angle, but may need to invert the answer
                bInvert = TRUE;
            }

            // is angle less than -pi/2?
            n16CompareResult = FIXED_n16Compare((OSAL_FIXED_OBJECT)&sValue, (OSAL_FIXED_OBJECT)&gsNegPiOverTwo);
            if (n16CompareResult < 0)
            {
                // yes, we need to adjust it
                eReturnCode = FIXED_eAdd(
                        (OSAL_FIXED_OBJECT)&gsPi,
                        (OSAL_FIXED_OBJECT)&sValue,
                        (OSAL_FIXED_OBJECT)&sValue);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    break;
                }
                // we adjusted the angle, but may need to invert the answer
                bInvert = TRUE;
            }

            // now the angle is in range

            // Shift the value to the "trig" format
            n32RadValue = n32ScaleFixed(&sValue, FRACTIONAL_BITS_FOR_TRIG);

            // Use the CORDIC function to get our cos value
            psObjResult->n32Value =
                n32CordicEval(n32RadValue, FIXED_OBJ_CORDIC_EVAL_TYPE_COS);
            psObjResult->un8NumFractionalBits = FRACTIONAL_BITS_FOR_TRIG;

            // do we need to invert the answer (based on the number of times we subtracted pi
            // and if the angle was > pi/2 or  angle < -pi/2
            if ( ((bInvert == TRUE) && ((un8NumSubtractions % 2) == 0)) ||
                 ((bInvert == FALSE) && ((un8NumSubtractions % 2) != 0)) )
            {
                // 2nd or third quadrant so need to invert
                eReturnCode = FIXED_eMultiply(
                    (OSAL_FIXED_OBJECT)psObjResult,
                    (OSAL_FIXED_OBJECT)&gsNegOne,
                    (OSAL_FIXED_OBJECT)psObjResult);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    break;
                }
            }

            eReturnCode = OSAL_SUCCESS;

        #if DEBUG_MATH == 1
            vValidateResult(fCos, psObjResult, "cos");
        #endif
    }while (0);

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eACos
*
* This calculates the inverse cosine of a FIXED object and places the result
* in the specified result parameter.  The range of hFixed must be between
* -1 and 1.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT to get the arccos of
*   hResult - A OSAL_FIXED_OBJECT where the inverse cosine while be store.
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eACos (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT sFinalValue, sFixedSquared, sOneMinusFixedSquared, sSqrtOfOneMinusFixedSquared,
                        sOnePlusX, sATan;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    N16 n16CompareResult;

#if DEBUG_MATH == 1
    float fACos = (float)acos(fFixedObjToFloat(psObj));
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Make sure it is not greater than 1
    n16CompareResult = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&gsOne);
    if (n16CompareResult > 0)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Make sure it is not less than -1
    n16CompareResult = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&gsNegOne);
    if (n16CompareResult < 0)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }
    else if (n16CompareResult == 0)
    {
        FIXED_OBJECT_STRUCT *psObjResult =
            (FIXED_OBJECT_STRUCT *)hResult;

        // The formula we use to compute acos is only valid
        // if -1 < x <= 1
        // so since in this case x = -1 we jsut return the answer (pi)
        psObjResult->n32Value = FX3B29_PI;
        psObjResult->un8NumFractionalBits = PI_FRACTIONAL_BITS;
        return OSAL_SUCCESS;
    }

    // Make sure it isn't a value of zero.  If so, return pi/2
    if (psObj->n32Value == 0)
    {
        FIXED_OBJECT_STRUCT *psObjResult =
            (FIXED_OBJECT_STRUCT *)hResult;

        // Just return pi/2
        psObjResult->n32Value = FX3B29_PI_OVER_2;
        psObjResult->un8NumFractionalBits = PI_FRACTIONAL_BITS;
        return OSAL_SUCCESS;
    }

    do
    {
        //      x^2
        eReturnCode = FIXED_ePow(hFixed, 2,
            (OSAL_FIXED_OBJECT)&sFixedSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //      (1-x^2)
        eReturnCode = FIXED_eSubtract(
            (OSAL_FIXED_OBJECT)&gsOne,
            (OSAL_FIXED_OBJECT)&sFixedSquared,
            (OSAL_FIXED_OBJECT)&sOneMinusFixedSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //        _______
        //      \/(1-x^2)
        eReturnCode = FIXED_eSqrt(
            (OSAL_FIXED_OBJECT)&sOneMinusFixedSquared,
            (OSAL_FIXED_OBJECT)&sSqrtOfOneMinusFixedSquared);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //      (1 + x)
        eReturnCode = FIXED_eAdd(
            (OSAL_FIXED_OBJECT)&gsOne,
            hFixed,
            (OSAL_FIXED_OBJECT)&sOnePlusX);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //        _______
        //      \/(1-x^2)
        //      ----------
        //        (1 + x)
        eReturnCode = FIXED_eDivide(
            (OSAL_FIXED_OBJECT)&sSqrtOfOneMinusFixedSquared,
            (OSAL_FIXED_OBJECT)&sOnePlusX,
            (OSAL_FIXED_OBJECT)&sFinalValue);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = FIXED_eATan((OSAL_FIXED_OBJECT)&sFinalValue, (OSAL_FIXED_OBJECT)&sATan);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // 2 * ATan (add to itself is same as multply by 2)
        eReturnCode = FIXED_eAdd(
            (OSAL_FIXED_OBJECT)&sATan,
            (OSAL_FIXED_OBJECT)&sATan,
            hResult);

    } while (FALSE);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME": acos of %f is ... %f\n",
            fFixedObjToFloat(psObj), fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hResult));

#if DEBUG_MATH == 1
        vValidateResult(fACos, (FIXED_OBJECT_STRUCT *)hResult, "acos");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error calculating acos of %f "
            "Error %s.\n",
            fFixedObjToFloat(psObj),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eATan
*
* This calculates the inverse tangent of a FIXED object and places the result
* in the specified result parameter.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT to get the inverse tangent of
*   hResult - A OSAL_FIXED_OBJECT where the inverse cosine while be store.
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
****************************************`*************************************/
OSAL_RETURN_CODE_ENUM FIXED_eATan (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    N32 n32RadValue;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    BOOLEAN bValid;

#if DEBUG_MATH == 1
    float fATan = (float)atan(fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hFixed));
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    bValid = bIsValidCORDICATan(hFixed);

    if (bValid == FALSE)
    {
        eReturnCode = eATan2(hFixed, (OSAL_FIXED_OBJECT)&gsOne, hResult);
    }
    else
    {
        // Shift the value to the "trig" format
        n32RadValue = n32ScaleFixed((FIXED_OBJECT_STRUCT *)hFixed, FRACTIONAL_BITS_FOR_TRIG);

        // Use the CORDIC atan function to get our cos value
        psObjResult->n32Value =
            n32FixedATan(n32RadValue);
        psObjResult->un8NumFractionalBits = FRACTIONAL_BITS_FOR_TRIG;

        eReturnCode = OSAL_SUCCESS;
    }

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME": atan is %f\n", fFixedObjToFloat(psObjResult));

#if DEBUG_MATH == 1
        vValidateResult(fATan, psObjResult, "atan");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error calculating atan of %f. "
            "Error %s.\n",
            fFixedObjToFloat((FIXED_OBJECT_STRUCT*)hFixed),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}


/*****************************************************************************
*
*   FIXED_eATan2
*
* This calculates the inverse tangent using y and x and places the result
* in the hResult handle
*
* The following approximation is used:
*    if ( x == 0.0f )
*    {
*        if ( y > 0.0f ) return PIBY2_FLOAT;
*        if ( y == 0.0f ) return 0.0f;
*        return -PIBY2_FLOAT;
*    }
*    float atan;
*    float z = y/x;
*    if ( fabsf( z ) < 1.0f )
*    {
*        atan = z/(1.0f + 0.28f*z*z);
*        if ( x < 0.0f )
*        {
*            if ( y < 0.0f ) return atan - PI_FLOAT;
*            return atan + PI_FLOAT;
*        }
*    }
*    else
*    {
*        atan = PIBY2_FLOAT - z/(z*z + 0.28f);
*        if ( y < 0.0f ) return atan - PI_FLOAT;
*    }
*
*   Note that it doesn't look like this approximation is very good when X == 1.
*   For those cases, we just use our FIXED_eATan function to handle it.
*
*
* Inputs:
*
*   hFixedY - An OSAL_FIXED_OBJECT that is the y componet to get the
*       inverse tangent of
*   hFixedX - An OSAL_FIXED_OBJECT that is the x componet to get the
*       inverse tangent of
*   hResult - An OSAL_FIXED_OBJECT where the inverse tangent will be stored.
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eATan2 (
    OSAL_FIXED_OBJECT hFixedY,
    OSAL_FIXED_OBJECT hFixedX,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;

#if DEBUG_MATH == 1
    float fATan2 = (float)atan2(fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hFixedY),
                                fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hFixedX));
#endif

    do
    {
        if ( (hFixedY == OSAL_FIXED_INVALID_OBJECT) ||
             (hFixedX == OSAL_FIXED_INVALID_OBJECT) ||
             (hResult == OSAL_FIXED_INVALID_OBJECT) )
        {
            eReturnCode = OSAL_ERROR_INVALID_HANDLE;
            break;
        }

        eReturnCode = eATan2(hFixedY, hFixedX, hResult);

    } while (FALSE);

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME": atan2 is %f\n",
            fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hResult));

#if DEBUG_MATH == 1
        vValidateResult(fATan2, (FIXED_OBJECT_STRUCT *)hResult, "atan2");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error calculating atan2. Error %s.\n",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eAbs
*
* This calculates the absolute value of a FIXED_OBJECT. The precision of the
* result hResult will be the same as the precision of the input hFixed.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT to get the abs value of
*   hResult - A OSAL_FIXED_OBJECT where the abs value while be store.
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eAbs (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT *psResult =
        (FIXED_OBJECT_STRUCT *)hResult;

#if DEBUG_MATH == 1
    float fAbs = fabsf(fFixedObjToFloat(psObj));
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    if (psObj->n32Value < 0)
    {
        psResult->n32Value = psObj->n32Value * -1;
    }
    else
    {
        psResult->n32Value = psObj->n32Value;
    }

    psResult->un8NumFractionalBits = psObj->un8NumFractionalBits;

    printf(FIXED_OBJECT_NAME": abs value of %f is ... %f\n",
        fFixedObjToFloat(psObj), fFixedObjToFloat((FIXED_OBJECT_STRUCT *)hResult));

#if DEBUG_MATH == 1
    vValidateResult(fAbs, (FIXED_OBJECT_STRUCT *)hResult, "abs");
#endif

    return OSAL_SUCCESS;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_eSqrt
*
* This function calculates the square root of a OSAL_FIXED_OBJECT and returns
* it as a new OSAL_FIXED_OBJECT referenced by hResult
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT to get the square root for
*   hResult - A OSAL_FIXED_OBJECT that is the result of the square root of
*             hFixed
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_eSqrt (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    UN32 un32Result;
    N32 n32Value;
    UN8 un8FractionalBits;

#if DEBUG_MATH == 1
    float fSqrt = (float)sqrt(fFixedObjToFloat(psObj));
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }
    // Lets not deal with imaginary numbers
    if (psObj->n32Value < 0)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Shift the value to the "sqrt" format
    // if we are of a greater percision
    if (psObj->un8NumFractionalBits > MAX_FRACTIONAL_BITS_FOR_SQRT)
    {
        n32Value = psObj->n32Value >> (psObj->un8NumFractionalBits - MAX_FRACTIONAL_BITS_FOR_SQRT);
        un8FractionalBits = MAX_FRACTIONAL_BITS_FOR_SQRT;
    }
    else
    {
        n32Value = psObj->n32Value;
        un8FractionalBits = psObj->un8NumFractionalBits;
    }

    // Pass the work off to our private function
    un32Result = un32FixedSqrt((UN32)n32Value, un8FractionalBits);

    psObjResult->n32Value = (N32)un32Result;
    psObjResult->un8NumFractionalBits = un8FractionalBits;

    if (un32Result > N32_MAX)
    {
        eReturnCode = OSAL_ERROR_MATH_OVERFLOW;
    }

    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(FIXED_OBJECT_NAME": square root of %f is %f\n",
            fFixedObjToFloat(psObj), fFixedObjToFloat(psObjResult));

#if DEBUG_MATH == 1
        vValidateResult(fSqrt, psObjResult, "square-root");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error calculating square root of %f. "
            "Error %s.\n",
            fFixedObjToFloat(psObj),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   FIXED_ePow
*
* This function raises the value of a OSAL_FIXED_OBJECT to a particulr power
* and returns it as a new OSAL_FIXED_OBJECT referenced by hResult
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT to get raise to the un8Power
*   un8Power - The power to raise hFixed to
*   hResult - A OSAL_FIXED_OBJECT that is the result
*
* Outputs:
*
*   OSAL_SUCCESS when there are no problems, an OSAL error code on error
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM FIXED_ePow (
    OSAL_FIXED_OBJECT hFixed,
    UN8 un8Power,
    OSAL_FIXED_OBJECT hResult
        )
{
#if OSAL_FIXED_MATH == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;
    FIXED_OBJECT_STRUCT sTempResult;
    UN8 un8ResultBits;
    N64 n64Result;

#if DEBUG_MATH == 1
    float fPow = (float)pow(fFixedObjToFloat(psObj), un8Power);
#endif

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) ||
         (hResult == OSAL_FIXED_INVALID_OBJECT) )
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Setup the initial values
    sTempResult.n32Value = 1;
    sTempResult.un8NumFractionalBits = 0;

    // Get to work
    while (un8Power > 0 && eReturnCode == OSAL_SUCCESS)
    {
        // Call our private multiplication function
        n64Result = n64FixedMultiply (psObj->n32Value,
                                      psObj->un8NumFractionalBits,
                                      sTempResult.n32Value,
                                      sTempResult.un8NumFractionalBits,
                                      &un8ResultBits);

        // Adjust the result
        eReturnCode = eAdjustResult(&sTempResult, n64Result, un8ResultBits);

        un8Power--;
    }

    if (eReturnCode == OSAL_SUCCESS)
    {
        // Since it all worked out
        // Copy over our temp value to the result
        psObjResult->n32Value = sTempResult.n32Value;
        psObjResult->un8NumFractionalBits = sTempResult.un8NumFractionalBits;

        printf(FIXED_OBJECT_NAME": %f raised to the %d power is %f\n",
            fFixedObjToFloat(psObj), un8Power,
            fFixedObjToFloat(psObjResult));

#if DEBUG_MATH == 1
        vValidateResult(fPow, psObjResult, "pow");
#endif
    }
    else
    {
        printf(FIXED_OBJECT_NAME": error calculating %d power of %f. Error $s\n",
            un8Power, fFixedObjToFloat(psObj),
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   n16Compare
*
* This method is used to compare two FIXED objects.  This function compares
* the values of the function, regardless of the fixed format.  If the fixed
* formats don't match, the values are converted to the most precise format
* and compared.
*
* Inputs:
*
*   hFixed1 - A handle to a valid FIXED object
*   hFixed2 - A handle to a valid FIXED object
*
* Outputs:
*               0   - FIXEDs have the same value (equal)
*               > 0 - hFixed1 is greater than (after) hFixed2
*               < 0 - hFixed1 is less than (before) hFixed2
*               N16_MIN - error
*
* TODO: fix api doc for this function to include error value.
*
*****************************************************************************/
N16 FIXED_n16Compare (
    OSAL_FIXED_OBJECT hFixed1,
    OSAL_FIXED_OBJECT hFixed2
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj1 =
        (FIXED_OBJECT_STRUCT *)hFixed1;
    FIXED_OBJECT_STRUCT *psObj2 =
        (FIXED_OBJECT_STRUCT *)hFixed2;
    N64 n64Value1, n64Value2;
    N16 n16Result = N16_MIN;

    if ( (hFixed1 == OSAL_FIXED_INVALID_OBJECT) ||
         (hFixed2 == OSAL_FIXED_INVALID_OBJECT) )
    {
        // ERROR!
        return N16_MIN;
    }

    n64Value1 = psObj1->n32Value;
    n64Value2 = psObj2->n32Value;

    if (psObj1->un8NumFractionalBits < psObj2->un8NumFractionalBits)
    {
        // shift n64Value1
        n64Value1 <<=
            ((UN32)(psObj2->un8NumFractionalBits - psObj1->un8NumFractionalBits));
    }
    else
    {
        // shift n64Value2
        n64Value2 <<=
            ((UN32)(psObj1->un8NumFractionalBits - psObj2->un8NumFractionalBits));
    }

    // compare the values now that we know their formats match
    if (n64Value1 == n64Value2)
    {
        n16Result = 0;
    }
    else if (n64Value1 < n64Value2)
    {
        n16Result = -1;
    }
    else if (n64Value1 > n64Value2)
    {
        n16Result = 1;
    }

    return n16Result;
#else
    return N16_MIN;
#endif
}

/*****************************************************************************
*
*   FIXED_n32FWrite
*
* This function serializes a OSAL_FIXED_OBJECT to a file
*
* Inputs:
*
*   hFloat - A OSAL_FIXED_OBJECT to write out
*   psFile - An open, valid file pointer to write the OSAL_FIXED_OBJECT to
*
* Outputs:
*
*   The number of bytes written to the file, or EOF on error
*
*****************************************************************************/
N32 FIXED_n32FWrite (
    OSAL_FIXED_OBJECT hFloat,
    FILE *psFile
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
            (FIXED_OBJECT_STRUCT *)hFloat;
    UN8 un8ByteToWrite;
    N32 n32Return = 0;

    // Verify inputs
    if( (hFloat == OSAL_FIXED_INVALID_OBJECT) || (psFile == NULL) )
    {
        // Error!
        return EOF;
    }

    // Write FLOAT to file...

    // Write n32Value
    un8ByteToWrite = BYTE0(psObj->n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(psObj->n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(psObj->n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(psObj->n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);


    // Write un8NumFractionalBits
    n32Return += (N32)fwrite(&psObj->un8NumFractionalBits,
        sizeof(psObj->un8NumFractionalBits), 1, psFile);

    return n32Return;
#else
    return EOF;
#endif
}

/*****************************************************************************
*
*   FIXED_hFRead
*
* This function reads a OSAL_FIXED_OBJECT from a file that has previously been
* written to by FIXED_n32FWrite.
*
* Inputs:
*
*   psFile - An open, valid file pointer to read the OSAL_FIXED_OBJECT from
*
* Outputs:
*
*   The valid OSAL_FIXED_OBJECT on success.
*   A OSAL_FIXED_INVALID_OBJECT on error
*
*****************************************************************************/
OSAL_FIXED_OBJECT FIXED_hFRead (
    FILE *psFile
        )
{
#if OSAL_FIXED_MATH == 1
    N32 n32Value;
    UN8 un8NumFractionalBits;
    size_t tTotal, tLen;

    // Verify input
    if(psFile == NULL)
    {
        return OSAL_FIXED_INVALID_OBJECT;
    }

    // Read FIXED from file...

    // Read n32Value
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&n32Value) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(N32)));

    if(tTotal != sizeof(N32))
    {
        // End of file or error occurred
        return OSAL_FIXED_INVALID_OBJECT;
    }

    // Read un8NumFractionalBits
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&un8NumFractionalBits) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(UN8)));

    if(tTotal != sizeof(UN8))
    {
        // End of file or error occurred
        return OSAL_FIXED_INVALID_OBJECT;
    }

    // Create an instance of the FIXED object
    return FIXED_hCreateFromFixed(n32Value,
                                  un8NumFractionalBits);
#else
    return OSAL_FIXED_INVALID_OBJECT;
#endif
}

/*****************************************************************************
*
*   FIXED_n32FPrintf
*
* This function prints a OSAL_FIXED_OBJECT to a file in a human-readable
* format.  Useful for debugging.
*
* Inputs:
*
*   hFixed - A OSAL_FIXED_OBJECT to write out
*   psFile - An open, valid file pointer to printf the OSAL_FIXED_OBJECT to
*   bIsTerse - If TRUE, then just the value as a float will be printed
*              If FALSE, then all members are written out
*
* Outputs:
*
*   The number of bytes written to the file, or EOF on error
*
*****************************************************************************/
N32 FIXED_n32FPrintf (
    OSAL_FIXED_OBJECT hFixed,
    FILE *psFile,
    BOOLEAN bIsTerse
        )
{
#if OSAL_FIXED_MATH == 1
    N32 n32Return = 0;
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)hFixed;

    if ( (hFixed == OSAL_FIXED_INVALID_OBJECT) || (psFile == NULL) )
    {
        // Error!
        return EOF;
    }

    // Print FIXED information header
    n32Return += fprintf(psFile, "FIXED: hFixed = 0x%p\n",
        psObj);

    if (bIsTerse == FALSE)
    {
        // Print members
        n32Return += fprintf(psFile, "\tFixedValue = %d\n",
            psObj->n32Value);
        n32Return += fprintf(psFile, "\tFractionalBits = %d\n",
            psObj->un8NumFractionalBits);
#if DEBUG_OBJECT == 1
        n32Return += fprintf(psFile, "\tFloat value = %f\n",
            fFixedToFloat(psObj->n32Value, psObj->un8NumFractionalBits));
#endif
    }

    return n32Return;
#else
    return EOF;
#endif

}

/*****************************************************************************
*
*   FIXED_hCreateInMemory
*
*****************************************************************************/
OSAL_FIXED_OBJECT FIXED_hCreateInMemory (
    N32 n32FixedValue,
    UN8 un8NumFractionalBits,
    void *pvMemory
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)pvMemory;

    // do a range check
    if (un8NumFractionalBits > TOTAL_NUM_BITS)
    {
        return OSAL_FIXED_INVALID_OBJECT;
    }

    // Set the member variables
    psObj->n32Value = n32FixedValue;
    psObj->un8NumFractionalBits = un8NumFractionalBits;

    return (OSAL_FIXED_OBJECT)psObj;
#else
    return OSAL_FIXED_INVALID_OBJECT;
#endif
}

/*****************************************************************************
*
*   FIXED_hDuplicateInMemory
*
*****************************************************************************/
OSAL_FIXED_OBJECT FIXED_hDuplicateInMemory (
    OSAL_FIXED_OBJECT hFixed,
    void *pvMemory
    )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psObj =
        (FIXED_OBJECT_STRUCT *)pvMemory;
    FIXED_OBJECT_STRUCT *psSource =
        (FIXED_OBJECT_STRUCT *)hFixed;

    if (psObj != NULL && psSource != NULL)
    {
        // Set the value
        *psObj = *psSource;
    }

    return (OSAL_FIXED_OBJECT)psObj;
#else
    return OSAL_FIXED_INVALID_OBJECT;
#endif
}

/*****************************************************************************
*
*   FIXED_bCopyToMemory
*
*****************************************************************************/
BOOLEAN FIXED_bCopyToMemory (
    OSAL_FIXED_OBJECT hFixed,
    void *pvMemory
        )
{
#if OSAL_FIXED_MATH == 1
    FIXED_OBJECT_STRUCT *psOrigObj =
        (FIXED_OBJECT_STRUCT *)hFixed;
    FIXED_OBJECT_STRUCT *psDestObj =
        (FIXED_OBJECT_STRUCT*)pvMemory;

    if (hFixed == OSAL_FIXED_INVALID_OBJECT)
    {
        return FALSE;
    }

    psDestObj->n32Value = psOrigObj->n32Value;
    psDestObj->un8NumFractionalBits = psOrigObj->un8NumFractionalBits;

    return TRUE;
#else
    return FALSE;
#endif
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
#if OSAL_FIXED_MATH == 1
/*****************************************************************************
*
*   n64FixedMultiply
*
* A private function to multiply fixed point values without having to work
* with the FIXED_OBJECT_STRUCT.  This was done to speed up operations that
* may use multiplication as part of a formula.
*
* Inputs:
*
*   n32Value1 - A fixed-point value
*   un8NumFractionalBits1 - The fixed-point format for n32Value1
*   n32Value2 - A fixed-point value
*   un8NumFractionalBits2 - The fixed-point format for n32Value2
*   pun8NumResultFractionalBits - The fixed-point format for the result
*
* Outputs:
*
*   A value in the fixed-point format specified by pun8NumResultFractionalBits
*
*****************************************************************************/
static N64 n64FixedMultiply (
    N32 n32Value1,
    UN8 un8NumFractionalBits1,
    N32 n32Value2,
    UN8 un8NumFractionalBits2,
    UN8 *pun8NumResultFractionalBits
        )
{
    N64 n64Result = 0;

    // Do the multiplication
    n64Result = (N64)n32Value1 * (N64)n32Value2;
    *pun8NumResultFractionalBits =
        un8NumFractionalBits1 + un8NumFractionalBits2;

    printf(FIXED_OBJECT_NAME": %f (XXX)times(XXX) %f is %f\n",
        fFixedToFloat(n32Value1, un8NumFractionalBits1),
        fFixedToFloat(n32Value2, un8NumFractionalBits2),
        fFixedToFloat(n64Result, *pun8NumResultFractionalBits));

    return n64Result;
}

/*****************************************************************************
*
*   n64FixedDivide
*
* A private function to divide fixed point values without having to work
* with the FIXED_OBJECT_STRUCT.  This was done to speed up operations that
* may use division as part of a formula.
*
* Inputs:
*
*   n32Numerator - A fixed-point value
*   un8NumFractionalNumeratorBits - The fixed-point format for n32Numerator
*   n32Denominator - A fixed-point value
*   un8NumFractionalDenominatorBits - The fixed-point format for
*                                     un8NumFractionalDenominatorBits
*   pun8NumResultFractionalBits - The fixed-point format for the result
*
* Outputs:
*
*   A value in the fixed-point format specified by pun8NumResultFractionalBits
*
*****************************************************************************/
static N64 n64FixedDivide (
    N32 n32Numerator,
    UN8 un8NumFractionalNumeratorBits,
    N32 n32Denominator,
    UN8 un8NumFractionalDenominatorBits,
    UN8 *pun8NumResultFractionalBits
        )
{
    N64 n64Result = 0;
    UN8 un8WholeBitsNeeded = 0;
    UN8 un8FractionalBitsNeeded = 0;

    if (n32Denominator == 0)
    {
        // Divide by zero ERROR!
        return 0;
    }

    // Adjust fractional bits for correct calculation
    un8WholeBitsNeeded =
        un8NumWholeBitsNeeded( n32Numerator >> un8NumFractionalNumeratorBits );
    un8FractionalBitsNeeded = TOTAL_NUM_BITS - un8WholeBitsNeeded;

    if (un8FractionalBitsNeeded > un8NumFractionalNumeratorBits)
    {
        n32Numerator <<=
            (un8FractionalBitsNeeded - un8NumFractionalNumeratorBits);
    }
    else
    {
        n32Numerator >>=
            (un8NumFractionalNumeratorBits - un8FractionalBitsNeeded);
    }
    un8NumFractionalNumeratorBits = un8FractionalBitsNeeded;

    // Throw the inital value on the N64
    n64Result = n32Numerator;

    // Shift the numerator so that it has maximum fractional percision
    n64Result <<= DIV_SHIFT_NUM_BITS;

    // Do the division
    n64Result = n64Result / n32Denominator;

    // Using the total num bits since that is what our numerator is at
    // after shifting
   *pun8NumResultFractionalBits = (DIV_SHIFT_NUM_BITS + un8NumFractionalNumeratorBits) - un8NumFractionalDenominatorBits;

   printf(FIXED_OBJECT_NAME": %f (XXX)divided by(XXX) %f is %f\n",
        fFixedToFloat(n32Numerator, un8NumFractionalNumeratorBits),
        fFixedToFloat(n32Denominator, un8NumFractionalDenominatorBits),
        fFixedToFloat(n64Result, *pun8NumResultFractionalBits));

    return n64Result;
}

/*****************************************************************************
*
*   un32FixedSqrt
*
* A private function to get the square-root of fixed point values without
* having to work with the FIXED_OBJECT_STRUCT.  This was done to speed up
* operations that may use square-roots as part of a formula.
*
* Inputs:
*
*   un32Value - A fixed-point value
*   un8NumFractionalBits - The fixed-point format for un32Value
*
* Outputs:
*
*   A value in the fixed-point format specified by un8NumFractionalBits
*
*****************************************************************************/
static UN32 un32FixedSqrt (
    UN32 un32Value,
    UN8 un8NumFractionalBits
        )
{
    UN32 un32Root = 0;
    UN32 un32RemHigh = 0;
    UN32 un32RemLow = un32Value;
    UN32 un32TestDiv = 0;
    UN32 un32Count = un8NumFractionalBits;

    do
    {
        un32RemHigh =
            (un32RemHigh<<((UN32)(32-un8NumFractionalBits)))
            |
            (un32RemLow>>un8NumFractionalBits);

        un32RemLow <<= 2; /* get 2 bits of arg */
        un32Root <<= 1; /* Get ready for the next bit in the root */
        un32TestDiv = (un32Root << 1) + 1; /* Test radical */

        if (un32RemHigh >= un32TestDiv)
        {
            un32RemHigh -= un32TestDiv;
            un32Root++;
        }

    } while (un32Count-- != 0);


    return un32Root;
}

/*****************************************************************************
*
*   eAtanImpl_fixed
*
* A private function implementing alternative formula to ATAN calculation.
* taken from "Algorithmes et developpements pour le 8051" by Jean-Louis Vern
*
* Inputs:
*
*   hFixedX - A fixed-point input value
*   hResult - A fixed point result value
*
* Outputs:
*
*   OSAL return code specifying success or error
*
*****************************************************************************/
static OSAL_RETURN_CODE_ENUM eAtanImpl_fixed (
    OSAL_FIXED_OBJECT hFixedX,
    OSAL_FIXED_OBJECT hResult
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;

    FIXED_OBJECT_STRUCT sFixedX55Mul, sFixedXSquared, sPlus105,
        s90Mul, sXSqSq, sXSqSq9Mul, sNumerator, sDenominator;
    N16 n16CompareLinear;


    // Check if our ratio is between FIXED_ATAN_LINEAR_START and 1.
    // In this case we will use linear approximation from the table
    n16CompareLinear = FIXED_n16Compare(
        hFixedX,
        (OSAL_FIXED_OBJECT)&gsAtanLinearStart);

    do
    {

        if (n16CompareLinear > 0)
        {
            FIXED_OBJECT_STRUCT sIndex;
            UN32 un32Index;

            // Multiply by FIXED_ATAN_LINEAR_MULTIPLIER...
            eReturnCode = FIXED_eMultiply(hFixedX,
                (OSAL_FIXED_OBJECT)&gsAtanLinearMultiplier,
                (OSAL_FIXED_OBJECT)&sIndex);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            // ...find an index in a table...
            un32Index = FIXED_n32ScaledValue((OSAL_FIXED_OBJECT)&sIndex, 0) -
                            (FIXED_ATAN2_LINEAR_MULTIPLIER - FIXED_ATAN2_LINEAR_NUM_ELEMENTS);
            // ...and take the value from the table
            psObjResult->n32Value = gan32AtanTable[un32Index];
            psObjResult->un8NumFractionalBits = FRACTIONAL_BITS_FOR_TRIG;

            // All done. Exiting.
            break;
        }

        //x^2
        eReturnCode = FIXED_eMultiply(
            hFixedX,
            hFixedX,
            (OSAL_FIXED_OBJECT)&sFixedXSquared);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //55*x^2
        eReturnCode = FIXED_eMultiply(
            (OSAL_FIXED_OBJECT)&sFixedXSquared,
            (OSAL_FIXED_OBJECT)&gs55,
            (OSAL_FIXED_OBJECT)&sFixedX55Mul);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //105+55*x^2
        eReturnCode = FIXED_eAdd(
            (OSAL_FIXED_OBJECT)&sFixedX55Mul,
            (OSAL_FIXED_OBJECT)&gs105,
            (OSAL_FIXED_OBJECT)&sPlus105);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //x*(105+55*x^2)
        eReturnCode = FIXED_eMultiply(
            (OSAL_FIXED_OBJECT)&sPlus105,
            hFixedX,
            (OSAL_FIXED_OBJECT)&sNumerator);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //90*x^2
        eReturnCode = FIXED_eMultiply(
            (OSAL_FIXED_OBJECT)&sFixedXSquared,
            (OSAL_FIXED_OBJECT)&gs90,
            (OSAL_FIXED_OBJECT)&s90Mul);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //x^4
        eReturnCode = FIXED_eMultiply(
            (OSAL_FIXED_OBJECT)&sFixedXSquared,
            (OSAL_FIXED_OBJECT)&sFixedXSquared,
            (OSAL_FIXED_OBJECT)&sXSqSq);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //9*x^4
        eReturnCode = FIXED_eMultiply(
            (OSAL_FIXED_OBJECT)&sXSqSq,
            (OSAL_FIXED_OBJECT)&gs9,
            (OSAL_FIXED_OBJECT)&sXSqSq9Mul);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //105+90*x^2+9*x^4
        eReturnCode = FIXED_eAdd(
            (OSAL_FIXED_OBJECT)&gs105,
            (OSAL_FIXED_OBJECT)&s90Mul,
            (OSAL_FIXED_OBJECT)&s90Mul);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = FIXED_eAdd(
            (OSAL_FIXED_OBJECT)&s90Mul,
            (OSAL_FIXED_OBJECT)&sXSqSq9Mul,
            (OSAL_FIXED_OBJECT)&sDenominator);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        //x*(105+55*x*x)/(105+90*x*x+9*x*x*x*x)
        eReturnCode = FIXED_eDivide(
            (OSAL_FIXED_OBJECT)&sNumerator,
            (OSAL_FIXED_OBJECT)&sDenominator,
            (OSAL_FIXED_OBJECT)psObjResult);
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eATan2
*
*****************************************************************************/
static OSAL_RETURN_CODE_ENUM eATan2 (
    OSAL_FIXED_OBJECT hFixedY,
    OSAL_FIXED_OBJECT hFixedX,
    OSAL_FIXED_OBJECT hResult
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    FIXED_OBJECT_STRUCT *psObjResult =
        (FIXED_OBJECT_STRUCT *)hResult;

    FIXED_OBJECT_STRUCT sFixedZ, sFixedYAbs, sFixedXAbs;
    N16 n16CompareX, n16CompareY, n16Compare;

    do
    {
        eReturnCode = FIXED_eAbs(hFixedX, (OSAL_FIXED_OBJECT)&sFixedXAbs);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = FIXED_eAbs(hFixedY, (OSAL_FIXED_OBJECT)&sFixedYAbs);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // All comparisons are done with specific precision. Since we do
        // know that formula returns result within 0.0001 threshold, we
        // assume that values different by less than this value are equal

        //Y ? 0
        n16CompareY = FIXED_n16Compare(
            hFixedY,
            (OSAL_FIXED_OBJECT)&gsZero);
        //X ? 0
        n16CompareX = FIXED_n16Compare(
            hFixedX,
            (OSAL_FIXED_OBJECT)&gsZero);
        //X ? Y
        n16Compare = FIXED_n16Compare(
            (OSAL_FIXED_OBJECT)&sFixedXAbs,
            (OSAL_FIXED_OBJECT)&sFixedYAbs);

        // check if X is zero
        if (n16CompareX == 0)
        {
            if (n16CompareY == 0)
            {
                // If Y is  zero, return 0
                psObjResult->n32Value = 0;
                psObjResult->un8NumFractionalBits = 0;
            }
            else if (n16CompareY > 0)
            {
                // If Y is greater than zero, return Pi/2
                psObjResult->n32Value = gsPiOverTwo.n32Value;
                psObjResult->un8NumFractionalBits = gsPiOverTwo.un8NumFractionalBits;
            }
            else
            {
                // If Y < zero, return -Pi/2
                psObjResult->n32Value = gsNegPiOverTwo.n32Value;
                psObjResult->un8NumFractionalBits = gsNegPiOverTwo.un8NumFractionalBits;
            }

            eReturnCode = OSAL_SUCCESS;
            break;
        }

        // From here down X is definitely not 0

        // Y is zero
        if (n16CompareY == 0)
        {
            if (n16CompareX > 0)
            {
                // X > 0, return 0
                psObjResult->n32Value = 0;
                psObjResult->un8NumFractionalBits = 0;
            }
            else
            {
                // X < 0, return PI
                psObjResult->n32Value = gsPi.n32Value;
                psObjResult->un8NumFractionalBits = gsPi.un8NumFractionalBits;
            }

            eReturnCode = OSAL_SUCCESS;
            break;
        }

        // So, neither X nor Y are zeros down here.

        // Making sure we always calculate 0 to PI/4 (X>Y>0)
        if (n16Compare > 0)
        {
            eReturnCode = FIXED_eDivide(
                (OSAL_FIXED_OBJECT)&sFixedYAbs,
                (OSAL_FIXED_OBJECT)&sFixedXAbs,
                (OSAL_FIXED_OBJECT)&sFixedZ);
        }
        else
        {
            eReturnCode = FIXED_eDivide(
                (OSAL_FIXED_OBJECT)&sFixedXAbs,
                (OSAL_FIXED_OBJECT)&sFixedYAbs,
                (OSAL_FIXED_OBJECT)&sFixedZ);
        }

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // atan(y/x) implementation
        eReturnCode = eAtanImpl_fixed((OSAL_FIXED_OBJECT)&sFixedZ, (OSAL_FIXED_OBJECT)psObjResult);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Returning angle back to original
        if (n16CompareX < 0)
        {
            if (n16CompareY < 0)
            {
                if (n16Compare < 0)
                {
                    eReturnCode = FIXED_eSubtract((OSAL_FIXED_OBJECT)&gsPiOverTwo,
                                        (OSAL_FIXED_OBJECT)psObjResult, (OSAL_FIXED_OBJECT)psObjResult);
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        break;
                    }
                }

                eReturnCode = FIXED_eSubtract((OSAL_FIXED_OBJECT)psObjResult,
                                    (OSAL_FIXED_OBJECT)&gsPi, (OSAL_FIXED_OBJECT)psObjResult);
            }
            else
            {
                if (n16Compare > 0)
                {
                    eReturnCode = FIXED_eSubtract((OSAL_FIXED_OBJECT)&gsPiOverTwo,
                                        (OSAL_FIXED_OBJECT)psObjResult, (OSAL_FIXED_OBJECT)psObjResult);
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        break;
                    }
                }

                eReturnCode = FIXED_eAdd((OSAL_FIXED_OBJECT)psObjResult,
                                    (OSAL_FIXED_OBJECT)&gsPiOverTwo, (OSAL_FIXED_OBJECT)psObjResult);
            }
        }
        // X > 0
        else
        {
            if (n16CompareY < 0)
            {
                if (n16Compare > 0)
                {
                    eReturnCode = FIXED_eSubtract((OSAL_FIXED_OBJECT)&gsPiOverTwo,
                                        (OSAL_FIXED_OBJECT)psObjResult, (OSAL_FIXED_OBJECT)psObjResult);
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        break;
                    }
                }

                eReturnCode = FIXED_eSubtract((OSAL_FIXED_OBJECT)psObjResult,
                                    (OSAL_FIXED_OBJECT)&gsPiOverTwo, (OSAL_FIXED_OBJECT)psObjResult);
            }
            else
            {
                if (n16Compare < 0)
                {
                    eReturnCode = FIXED_eSubtract((OSAL_FIXED_OBJECT)&gsPiOverTwo,
                                        (OSAL_FIXED_OBJECT)psObjResult, (OSAL_FIXED_OBJECT)psObjResult);
                }
            }
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bIsValidCORDICATan
*
*****************************************************************************/
static BOOLEAN bIsValidCORDICATan(
    OSAL_FIXED_OBJECT hFixed
        )
{
    FIXED_OBJECT_STRUCT sValidLow, sValidHigh;
    N16 n16CompareHigh, n16CompareLow;
    BOOLEAN bValid = FALSE;

    sValidLow.n32Value = FIXED_ATAN_VALID_LOW;
    sValidLow.un8NumFractionalBits = FRACTIONAL_BITS_FOR_TRIG;

    sValidHigh.n32Value = FIXED_ATAN_VALID_HIGH;
    sValidHigh.un8NumFractionalBits = FRACTIONAL_BITS_FOR_TRIG;

    // If out of range of our valid values, we can't use the
    // CORIDC ATan
    n16CompareLow = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&sValidLow);
    if (n16CompareLow > 0)
    {
        n16CompareHigh = FIXED_n16Compare(hFixed, (OSAL_FIXED_OBJECT)&sValidHigh);

        if (n16CompareHigh < 0)
        {
            bValid = TRUE;
        }
    }

    return bValid;
}

/*****************************************************************************
*
*   n32FixedATan
*
* A private function to get the arctan of fixed point values without
* having to work with the FIXED_OBJECT_STRUCT.  This was done to speed up
* operations that may use arctan as part of a formula.
*
* Inputs:
*
*   n32Value - A fixed-point value in the FRACTIONAL_BITS_FOR_TRIG format
*
* Outputs:
*
*   A value in the FRACTIONAL_BITS_FOR_TRIG fixed-point format
*
*****************************************************************************/
static N32 n32FixedATan (
    N32 n32Value
        )
{
    N32 n32Count, n32tX, n32tY, n32tZ;
    N32 n32X = 1 << FRACTIONAL_BITS_FOR_TRIG;
    N32 n32Y = n32Value;
    N32 n32Z = 0;

    if (n32Value == 0)
    {
        return 0;
    }

    for (n32Count = 0; n32Count < NUM_CORDIC_TABLE_ENTRIES; ++n32Count)
    {
        n32tX = n32X >> n32Count;
        n32tY = n32Y >> n32Count;
        n32tZ = gan32CoridcTable[n32Count];

        n32X -= ((-n32Y)>=0) ? (n32tY) : -(n32tY);
        n32Z -= ((-n32Y)>=0) ? (n32tZ) : -(n32tZ);
        n32Y += ((-n32Y)>=0) ? (n32tX) : -(n32tX);
    }

    return n32Z;
}

/*****************************************************************************
*
*   n32CordicEval
*
* This function uses the CORDIC method to determine the sin and cos values for
* a given angle.
*
* This function was discussed at:
* http://www.dcs.gla.ac.uk/~jhw/cordic/
* S:\EmbeddedSoftware\SMS\CORDIC Resources
*
* Inputs:
*
*   n32Radian - The radian to compute the sin or cos of
*   eCordicEvalType - This enumeration specifies which portion of the CORDIC
*                     evaluation the requester is looking for.
*
* Outputs:
*
*   Either the sin or cos values as a N32 in a fixed-point format using
*   FRACTIONAL_BITS_FOR_TRIG.  This format is used since that is
*****************************************************************************/
static N32 n32CordicEval (
    N32 n32Radian,
    FIXED_OBJ_CORDIC_EVAL_TYPE_ENUM eCordicEvalType
        )
{
    N32 n32Count, n32Sign, n32tX, n32tY, n32tZ;
    N32 n32X = CORDIC_1K;
    N32 n32Y = 0;
    N32 n32Z = n32Radian;

    for (n32Count = 0; n32Count < NUM_CORDIC_TABLE_ENTRIES; ++n32Count)
    {
        n32Sign = n32Z >= 0 ? 0 : -1;
        n32tX = n32X - (((n32Y>>n32Count) ^ n32Sign) - n32Sign);
        n32tY = n32Y + (((n32X>>n32Count) ^ n32Sign) - n32Sign);
        n32tZ = n32Z - ((gan32CoridcTable[n32Count] ^ n32Sign) - n32Sign);

        n32X = n32tX;
        n32Y = n32tY;
        n32Z = n32tZ;
    }

    if (eCordicEvalType == FIXED_OBJ_CORDIC_EVAL_TYPE_SIN)
    {
        printf(FIXED_OBJECT_NAME": sin of %f is %f\n",
            fFixedToFloat(n32Radian, FRACTIONAL_BITS_FOR_TRIG),
            fFixedToFloat(n32Y, FRACTIONAL_BITS_FOR_TRIG));
        return n32Y;
    }
    else
    {
        printf(FIXED_OBJECT_NAME": cos of %f is %f\n",
            fFixedToFloat(n32Radian, FRACTIONAL_BITS_FOR_TRIG),
            fFixedToFloat(n32X, FRACTIONAL_BITS_FOR_TRIG));
        return n32X;
    }
}

/*****************************************************************************
*
*   eAdjustResult
*
*   An internal function to take a FIXED_OBJECT_STRUCT and adjust it to match
*   the values specified by n64Result and un8ResultBits.
*****************************************************************************/
static OSAL_RETURN_CODE_ENUM eAdjustResult (
    FIXED_OBJECT_STRUCT *psObj,
    N64 n64Result,
    UN8 un8ResultBits
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    UN8 un8WholeBitsNeeded, un8FractionalBitsNeeded;

    // Determine how many bits are needed for the whole value
    un8WholeBitsNeeded =
        un8NumWholeBitsNeeded((N32) (n64Result >> un8ResultBits));
    un8FractionalBitsNeeded = TOTAL_NUM_BITS - un8WholeBitsNeeded;

    printf(FIXED_OBJECT_NAME": %d_%d needed for %d\n",
        un8WholeBitsNeeded,
        un8FractionalBitsNeeded,
        (N32) (n64Result >> un8ResultBits));

    if (un8FractionalBitsNeeded > un8ResultBits)
    {
        psObj->n32Value =
            (N32)(n64Result << (un8FractionalBitsNeeded - un8ResultBits));
    }
    else
    {
        psObj->n32Value =
            (N32)(n64Result >> (un8ResultBits - un8FractionalBitsNeeded));
    }
    psObj->un8NumFractionalBits = un8FractionalBitsNeeded;

    printf(FIXED_OBJECT_NAME": adjusted from %d bits to %d bits -- now %f\n",
        un8ResultBits,
        psObj->un8NumFractionalBits,
        fFixedToFloat(psObj->n32Value, psObj->un8NumFractionalBits));

    return eReturnCode;
}

/*****************************************************************************
*
*   un8NumWholeBitsNeeded
*
*  This function calculates the number of whole bits needed to contain the
*  whole value specified by n32WholeValue
*****************************************************************************/
static UN8 un8NumWholeBitsNeeded (
    N32 n32WholeValue
        )
{
    UN8 un8NumBits = 0;
    UN32 un32MaxValue = 0;

    n32WholeValue = abs(n32WholeValue);

    for ( un8NumBits = 0;
         (un8NumBits < TOTAL_NUM_BITS) && (n32WholeValue != 0);
          un8NumBits++)
    {
        // the max value that can be held with un8NumBits is the sum of
        // the max value of this and all the previous ones
        // for example 3 bits can hold a max value of 7 (4+2+1)
        un32MaxValue = un32MaxValue + BIN_POINT_VALUE(un8NumBits);

        // Wait until our whole value is less the max value that can be held
        // with this bin point
        if ( ((UN32)n32WholeValue) < un32MaxValue)
        {
            break;
        }
    }
    return un8NumBits;
}

/*****************************************************************************
*
*   n32ScaleFixed
*
*****************************************************************************/
static N32 n32ScaleFixed(
    FIXED_OBJECT_STRUCT *psObj,
    UN8 un8ScaleBits
        )
{
    N32 n32ScaledValue = 0;

    if (psObj != NULL)
    {
        if (psObj->un8NumFractionalBits < un8ScaleBits)
        {
            n32ScaledValue = ((N32)
                psObj->n32Value << ((UN32)(un8ScaleBits - psObj->un8NumFractionalBits)));
        }
        else
        {
            n32ScaledValue = ((N32)
                psObj->n32Value >> ((UN32)(psObj->un8NumFractionalBits - un8ScaleBits)));
        }
    }

    return n32ScaledValue;
}

/*****************************************************************************
*
*   n64ScaleFixed
*
*****************************************************************************/
static N64 n64ScaleFixed(
    FIXED_OBJECT_STRUCT *psObj,
    UN8 un8ScaleBits
        )
{
    N64 n64ScaledValue = 0;

    if (psObj != NULL)
    {
        if (psObj->un8NumFractionalBits < un8ScaleBits)
        {
            n64ScaledValue = ((N64)
                psObj->n32Value << ((UN32)(un8ScaleBits - psObj->un8NumFractionalBits)));
        }
        else
        {
            n64ScaledValue = ((N64)
                psObj->n32Value >> ((UN32)(psObj->un8NumFractionalBits - un8ScaleBits)));
        }
    }

    return n64ScaledValue;
}


/*****************************************************************************
*
*   fFixedToFloat
*
*  A debug function used mostly for debug print out.  Only available
*  with DEBUG_OBJECT or DEBUG_MATH enabled
*****************************************************************************/
static float fFixedToFloat (
    N64 n64WholeValue,
    UN8 un8FractionalBits
        )
{
// TODO: Fix this for bug #4809
#if (OSAL_DEBUG == 0)
    return 0;
}
#else
    float fResult;

    fResult = (float)n64WholeValue / BIN_POINT_VALUE(un8FractionalBits);
    return fResult;
}
#endif //(OSAL_DEBUG == 0)

/*****************************************************************************
*
*   fFixedObjToFloat
*
*  A debug function used mostly for debug print out.  Only available
*  with DEBUG_OBJECT or DEBUG_MATH enabled
*****************************************************************************/
static float fFixedObjToFloat (
    FIXED_OBJECT_STRUCT *psObj
        )
{
// TODO: Fix this for bug #4809
#if (OSAL_DEBUG == 0)
    return 0;
}
#else
    float fResult;

    fResult = (float)psObj->n32Value / BIN_POINT_VALUE(psObj->un8NumFractionalBits);
    return fResult;
}
#endif //(OSAL_DEBUG == 0)

/*****************************************************************************
*
*   ePowForEval
*
*   Wraps the OSAL_FIXED.ePow() function for OSAL_FIXED.eEval() internal
*   needs.
*
*****************************************************************************/
static OSAL_RETURN_CODE_ENUM ePowForEval (
    OSAL_FIXED_OBJECT hFixed,
    OSAL_FIXED_OBJECT hPower,
    OSAL_FIXED_OBJECT hResult
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;

    if (hFixed != OSAL_FIXED_INVALID_OBJECT)
    {
        // Extract the power value from the OSAL_FIXED object
        N32 n32Power = FIXED_n32Floor(hPower);
        if ((n32Power >= 0) && (n32Power <= UN8_MAX))
        {
            UN8 un8Power = (UN8)(n32Power % (UN8_MAX + 1));

            // Make a call
            eReturnCode = FIXED_ePow(hFixed, un8Power, hResult);
        }
    }

    return eReturnCode;
}

#if (DEBUG_MATH == 1)

/*****************************************************************************
*
*   vValidateResult
*
*****************************************************************************/
static void vValidateResult(
    float fValue,
    FIXED_OBJECT_STRUCT *psObjResult,
    const char *pcFunctionName)
{
    float fDiff = fValue - fFixedObjToFloat(psObjResult);
    float fAbsDiff = fabsf(fDiff);
    if ( fAbsDiff > DEBUG_MATH_ERROR_THRESHOLD)
    {
        fprintf(stderr, FIXED_OBJECT_NAME": Error --  %s does not match standard math library (%f != %f)\n",
            pcFunctionName, fValue, fFixedObjToFloat(psObjResult));
    }

    return;
}

#endif

#ifdef SUPPORT_CUNIT
#include <osal_fixed.cunit>
#endif

#endif /* OSAL_FIXED_MATH == 1 */
